/**
 * Copyright (C) 2019 MongoDB, Inc.  All Rights Reserved.
 */

#include "mongo/platform/basic.h"

#include "encryption_schema_tree.h"

#include "fle_test_fixture.h"
#include "mongo/bson/bsonobj.h"
#include "mongo/bson/json.h"
#include "mongo/db/bson/bson_helper.h"
#include "mongo/db/matcher/schema/encrypt_schema_gen.h"
#include "mongo/unittest/death_test.h"
#include "mongo/unittest/unittest.h"
#include "mongo/util/uuid.h"

namespace mongo {
namespace {

/**
 * Parses 'schema' into an encryption schema tree and returns the EncryptionMetadata for
 * 'path'.
 */
ResolvedEncryptionInfo extractMetadata(BSONObj schema, std::string path) {
    auto result = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    auto metadata = result->getEncryptionMetadataForPath(FieldRef(path));
    ASSERT(metadata);
    return metadata.get();
}

static const uint8_t uuidBytes[] = {0, 0, 0, 0, 0, 0, 0x40, 0, 0x80, 0, 0, 0, 0, 0, 0, 0};
const BSONObj encryptObj =
    BSON("encrypt" << BSON("algorithm"
                           << "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic"
                           << "keyId" << BSON_ARRAY(BSONBinData(uuidBytes, 16, newUUID))
                           << "bsonType"
                           << "string"));
const BSONObj encryptObjNumber =
    BSON("encrypt" << BSON("algorithm"
                           << "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic"
                           << "keyId" << BSON_ARRAY(BSONBinData(uuidBytes, 16, newUUID))
                           << "bsonType"
                           << "int"));
/**
 * Parses 'schema' into an encryption schema tree and verifies that 'path' is not encrypted.
 */
void assertNotEncrypted(BSONObj schema, std::string path) {
    auto result = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT_FALSE(result->getEncryptionMetadataForPath(FieldRef(path)));
}

class EncryptionSchemaTreeTest : public FLETestFixture {};

TEST(EncryptionSchemaTreeTest, MarksTopLevelFieldAsEncrypted) {
    const auto uuid = UUID::gen();
    ResolvedEncryptionInfo metadata{
        EncryptSchemaKeyId{std::vector<UUID>{uuid}}, FleAlgorithmEnum::kRandom, boost::none};

    BSONObj schema = fromjson(
        R"({
            type: "object",
            properties: {
                ssn: {
                    encrypt: {
                        algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                        keyId: [)" +
        uuid.toBSON().getField("uuid").jsonString(
            JsonStringFormat::ExtendedCanonicalV2_0_0, false, false) +
        R"(]
                    }
                },
                name: {
                    type: "string"
                }
            }
        })");
    ASSERT(extractMetadata(schema, "ssn") == metadata);
    assertNotEncrypted(schema, "name");
}

TEST(EncryptionSchemaTreeTest, MarksNestedFieldsAsEncrypted) {
    const auto uuid = UUID::gen();
    ResolvedEncryptionInfo metadata{
        EncryptSchemaKeyId{std::vector<UUID>{uuid}}, FleAlgorithmEnum::kRandom, boost::none};

    BSONObj schema = fromjson(
        R"({
        type: "object",
        properties: {
            user: {
                type: "object",
                properties: {
                    ssn: {
                        encrypt: {
                            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                            keyId: [)" +
        uuid.toBSON().getField("uuid").jsonString(
            JsonStringFormat::ExtendedCanonicalV2_0_0, false, false) +
        R"(]
                        }
                    }
                }
            }
        }})");
    auto result = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT(extractMetadata(schema, "user.ssn") == metadata);
    assertNotEncrypted(schema, "user");
    assertNotEncrypted(schema, "user.name");
}

TEST(EncryptionSchemaTreeTest, MarksNumericFieldNameAsEncrypted) {
    const auto uuid = UUID::gen();
    ResolvedEncryptionInfo metadata{
        EncryptSchemaKeyId{std::vector<UUID>{uuid}}, FleAlgorithmEnum::kRandom, boost::none};
    BSONObj encryptObj = fromjson(
        R"({
        algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
        keyId: [)" +
        uuid.toBSON().getField("uuid").jsonString(
            JsonStringFormat::ExtendedCanonicalV2_0_0, false, false) +
        R"(]})");

    BSONObj schema = BSON("type"
                          << "object"
                          << "properties" << BSON("0" << BSON("encrypt" << encryptObj)));
    ASSERT(extractMetadata(schema, "0") == metadata);

    schema = BSON(
        "type"
        << "object"
        << "properties"
        << BSON("nested" << BSON("type"
                                 << "object"
                                 << "properties" << BSON("0" << BSON("encrypt" << encryptObj)))));
    ASSERT(extractMetadata(schema, "nested.0") == metadata);
}

TEST(EncryptionSchemaTreeTest, MarksMultipleFieldsAsEncrypted) {
    const auto uuid = UUID::gen();
    const auto uuidStr = uuid.toBSON().getField("uuid").jsonString(
        JsonStringFormat::ExtendedCanonicalV2_0_0, false, false);
    ResolvedEncryptionInfo metadata{
        EncryptSchemaKeyId{std::vector<UUID>{uuid}}, FleAlgorithmEnum::kRandom, boost::none};

    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            ssn: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [)" +
                              uuidStr + R"(]
                }
            },
            accountNumber: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [)" +
                              uuidStr + R"(]
                }
            },
            super: {
                type: "object",
                properties: {
                    secret: {
                        encrypt: {
                            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                            keyId: [)" +
                              uuidStr + R"(]
                        }
                    }
                }
            }
        }})");
    ASSERT(extractMetadata(schema, "ssn") == metadata);
    ASSERT(extractMetadata(schema, "accountNumber") == metadata);
    ASSERT(extractMetadata(schema, "super.secret") == metadata);
    assertNotEncrypted(schema, "super");
}

TEST(EncryptionSchemaTreeTest, TopLevelEncryptMarksEmptyPathAsEncrypted) {
    const auto uuid = UUID::gen();
    const auto uuidStr = uuid.toBSON().getField("uuid").jsonString(
        JsonStringFormat::ExtendedCanonicalV2_0_0, false, false);
    ResolvedEncryptionInfo metadata{
        EncryptSchemaKeyId{std::vector<UUID>{uuid}}, FleAlgorithmEnum::kRandom, boost::none};

    BSONObj schema = fromjson(R"({
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [)" +
                              uuidStr + R"(]
                }})");

    ASSERT(extractMetadata(schema, "") == metadata);
}

TEST(EncryptionSchemaTreeTest, ExtractsCorrectMetadataOptions) {
    const auto uuid = UUID::gen();
    ResolvedEncryptionInfo ssnMetadata{EncryptSchemaKeyId{std::vector<UUID>{uuid}},
                                       FleAlgorithmEnum::kDeterministic,
                                       MatcherTypeSet{BSONType::String}};

    const IDLParserErrorContext encryptCtxt("encrypt");
    auto encryptObj = BSON("algorithm"
                           << "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic"
                           << "bsonType"
                           << "string"
                           << "keyId" << BSON_ARRAY(uuid));
    BSONObj schema = BSON("type"
                          << "object"
                          << "properties" << BSON("ssn" << BSON("encrypt" << encryptObj)));
    ASSERT(extractMetadata(schema, "ssn") == ssnMetadata);
}

TEST(EncryptionSchemaTreeTest, ThrowsAnErrorIfPathContainsEncryptedPrefix) {
    BSONObj schema = fromjson(R"({type: "object", properties: {ssn: {encrypt: {}}}})");
    ASSERT_THROWS_CODE(extractMetadata(schema, "ssn.nonexistent"), AssertionException, 51099);
}

TEST(EncryptionSchemaTreeTest, ReturnsNotEncryptedForPathWithNonEncryptedPrefix) {
    BSONObj schema = fromjson(R"({type: "object", properties: {ssn: {}}})");
    assertNotEncrypted(schema, "ssn.nonexistent");

    schema = fromjson(R"({type: "object", properties: {blah: {}}, additionalProperties: {}})");
    assertNotEncrypted(schema, "additional.prop.path");
}

TEST(EncryptionSchemaTreeTest, ThrowsAnErrorIfPathContainsPrefixToEncryptedAdditionalProperties) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            blah: {}
        },
        additionalProperties: {
            encrypt: {
                algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic",
                keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}],
                bsonType: "int"
            }
        }
    })");
    ASSERT_THROWS_CODE(extractMetadata(schema, "path.extends.encrypt"), AssertionException, 51102);
}

TEST(EncryptionSchemaTreeTest,
     ThrowsAnErrorIfPathContainsPrefixToNestedEncryptedAdditionalProperties) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            blah: {
                type: "object",
                properties: {
                    foo: {}
                },
                additionalProperties: {encrypt: {}}
            }
        }
    })");
    ASSERT_THROWS_CODE(extractMetadata(schema, "blah.not.foo"), AssertionException, 51099);
}

TEST(EncryptionSchemaTreeTest, FailsToParseEncryptAlongsideAnotherTypeKeyword) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            ssn: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
                },
                type: "object"
            }
        }})");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       ErrorCodes::FailedToParse);

    schema = fromjson(R"({
        type: "object",
        properties: {
            ssn: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
                },
                bsonType: "BinData"
            }
        }})");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       ErrorCodes::FailedToParse);
}

TEST(EncryptionSchemaTreeTest, FailsToParseEncryptWithSiblingKeywords) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            ssn: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
                },
                properties: {invalid: {}}
            }
        }})");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51078);

    schema = fromjson(R"({
        type: "object",
        properties: {
            ssn: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
                },
                items: {}
            }
        }})");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51078);

    schema = fromjson(R"({
            type: "object",
            properties: {
                ssn: {
                    encrypt: {
                        algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                        keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
                    },
                    minItems: 1
                }
            }})");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kRemote),
                       AssertionException,
                       51078);
}

TEST(EncryptionSchemaTreeTest, FailsToParseIfConflictingEncryptKeywords) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            ssn: {
                encrypt: {},
                properties: {
                    invalid: {
                        encrypt: {
                            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                            keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
                        }
                    }
                },
                items: {
                    encrypt: {
                        algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                        keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
                    }
                }
            }
        }})");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51077);
}

TEST(EncryptionSchemaTreeTest, FailsToParseIfEncryptWithinItems) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            ssn: {
                type: "array",
                items: {
                    encrypt: {
                        algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                        keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
                    }
                }
            }
        }})");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51077);

    schema = fromjson(R"({
        type: "object",
        properties: {
            user: {
                type: "object",
                properties: {
                    ssn: {
                        type: "array",
                        items: {
                            encrypt: {
                                algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                                keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
                            }
                        }
                    }
                }
            }
        }})");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51077);
}

TEST(EncryptionSchemaTreeTest, FailsToParseIfEncryptMetadataWithinItems) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            ssn: {
                type: "array",
                items: {
                    encryptMetadata: {
                        algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random"
                    }
                }
            }
        }})");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       31077);
}

TEST(EncryptionSchemaTreeTest, FailsToParseIfEncryptWithinAdditionalItems) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            ssn: {
                type: "array",
                items: {},
                additionalItems: {
                    encrypt: {
                        algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                        keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
                    }
                }
            }
        }})");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51077);

    schema = fromjson(R"({
        type: "object",
        properties: {
            ssn: {
                additionalItems: {
                    encrypt: {
                        algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                        keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
                    }
                }
            }
        }})");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51077);
}

TEST(EncryptionSchemaTreeTest, FailsToParseIfEncryptMetadataWithinAdditionalItems) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            ssn: {
                type: "array",
                items: {},
                additionalItems: {
                    encryptMetadata: {
                        algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random"
                    }
                }
            }
        }})");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       31077);
}

TEST(EncryptionSchemaTreeTest, FailsToParseIfEncryptParentIsNotTypeRestricted) {
    BSONObj schema = fromjson(R"({
        properties: {
            ssn: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
                }
            }
        }})");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51077);

    schema = fromjson(R"({
        type: "object",
        properties: {
            user: {
                properties: {
                    ssn: {
                        encrypt: {
                            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                            keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
                        }
                    }
                }
            }
        }})");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51077);
}

TEST(EncryptionSchemaTreeTest, FailsToParseIfEncryptParentIsTypeRestrictedWithMultipleTypes) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            user: {
                type: ["object", "array"],
                properties: {
                    ssn: {
                        encrypt: {
                            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                            keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
                        }
                    }
                }
            }
        }})");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51077);
}

TEST(EncryptionSchemaTreeTest, FailsToParseIfPropertyHasDot) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            "dotted.field" : {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
                }
            }
        }})");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51077);
}

TEST(EncryptionSchemaTreeTest, FailsToParseIfParentPropertyHasDot) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            "dotted.parent.field" : {
                type: "object",
                properties: {
                    secret: {
                        encrypt: {
                            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                            keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
                        }
                    }
                }
            }
        }})");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51077);
}

TEST(EncryptionSchemaTreeTest, FailsToParseIfDottedPropertyNestedInPatternProperties) {
    BSONObj schema = fromjson(R"({
        type: "object",
        patternProperties: {
            "abc" : {
                type: "object",
                properties: {
                    "d.e" : {
                        encrypt: {
                            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                            keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
                        }
                    }
                }
            }
        }})");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51077);
}

TEST(EncryptionSchemaTreeTest, FailsToParseIfDottedPropertyNestedInAdditionalProperties) {
    BSONObj schema = fromjson(R"({
        type: "object",
        additionalProperties: {
            type: "object",
            properties: {
                "a.c" : {
                    encrypt: {
                        algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                        keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
                    }
                }
            }
        }})");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51077);
}

TEST(EncryptionSchemaTreeTest, AllowDottedNonEncryptProperties) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            "dotted.non.encrypt" : {
                type: "object"
            },
            secret: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
                }
            }
        }})");
    auto result = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    assertNotEncrypted(schema, "dotted.non.encrypt");
    ASSERT(result->getEncryptionMetadataForPath(FieldRef{"secret"}));
}

TEST(EncryptionSchemaTreeTest, DottedPropertiesNotEncryptedWithMatchingPatternProperties) {
    BSONObj schema = fromjson(R"({
        type: "object",
        patternProperties: {
            "a.c" : {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
                }
            }
        }})");
    auto result = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    assertNotEncrypted(schema, "a.c");
    ASSERT(result->getEncryptionMetadataForPath(FieldRef{"abc"}));
}

TEST(EncryptionSchemaTreeTest, FailsToParseInvalidSchema) {
    BSONObj schema = fromjson(R"({properties: "invalid"})");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       ErrorCodes::TypeMismatch);

    schema = fromjson(R"({
        type: "object",
        properties: {
            ssn: {
                encrypt: "invalid"
            }
        }})");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       ErrorCodes::TypeMismatch);
}

TEST(EncryptionSchemaTreeTest, FailsToParseUnknownFieldInEncrypt) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            ssn: {
                encrypt: {unknownField: {}}
            }
        }})");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       40415);
}

TEST(EncryptionSchemaTreeTest, FailsToParseInvalidAlgorithm) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            ssn: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-SomeInvalidAlgo",
                    keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
                }
            }
        }})");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       ErrorCodes::BadValue);
}

TEST(EncryptionSchemaTreeTest, FailsToParseInvalidKeyIdUUID) {
    BSONObj schema =
        BSON("type"
             << "object"
             << "properties"
             << BSON("ssn" << BSON("encrypt" << BSON("keyId" << BSON_ARRAY(BSONBinData(
                                                         nullptr, 0, BinDataType::MD5Type))))));
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51084);
}

TEST(EncryptionSchemaTreeTest, FailsToParseIfEncryptBelowAdditionalPropertiesWithoutTypeObject) {
    BSONObj schema = fromjson(R"({
        additionalProperties: {
            encrypt: {
                algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
            }
        }})");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51077);
}

TEST(EncryptionSchemaTreeTest, FailsToParseIfIllegalSubschemaUnderAdditionalProperties) {
    BSONObj schema = fromjson(R"({
        type: "object",
        additionalProperties: {
            encrypt: {
                algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
            },
            illegal: 1
        }})");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       ErrorCodes::FailedToParse);
}

TEST(EncryptionSchemaTreeTest, FailsToParseIfAdditionalPropertiesIsWrongType) {
    BSONObj schema = fromjson("{additionalProperties: [{type: 'string'}]}");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       ErrorCodes::TypeMismatch);
}

TEST(EncryptionSchemaTreeTest, FailsToParseIfAdditionalPropertiesWithEncryptInsideItems) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            arr: {
                type: "array",
                items: {
                    type: "object",
                    additionalProperties: {
                        encrypt: {
                            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                            keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
                        }
                    }
                }
            }
        }})");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51077);
}

TEST(EncryptionSchemaTreeTest, AdditionalPropertiesAllPropertiesCorrectlyReportedAsEncrypted) {
    BSONObj schema = fromjson(R"({
        type: "object",
        additionalProperties: {
            encrypt: {
                algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic",
                keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}],
                bsonType: "int"
            }
        }
    })");
    auto encryptionTree = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foo"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"bar"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"baz"}));
}

TEST(EncryptionSchemaTreeTest,
     NestedAdditionalPropertiesAllPropertiesCorrectlyReportedAsEncrypted) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            obj: {
                type: "object",
                additionalProperties: {
                    encrypt: {
                        algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                        keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
                    }
                }
            }
        }})");
    auto encryptionTree = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"obj.foo"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"obj.bar"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"obj.baz"}));
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"other"}));
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"other.foo"}));
}

TEST(EncryptionSchemaTreeTest, AdditionalPropertiesOnlyAppliesToFieldsNotNamedByProperties) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            obj: {
                type: "object",
                properties: {
                    a: {type: "string"},
                    b: {type: "object"},
                    c: {
                        encrypt: {
                            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                            keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
                        }
                    }
                },
                additionalProperties: {
                    encrypt: {
                        algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                        keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
                    }
                }
            }
        }})");
    auto encryptionTree = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"obj.foo"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"obj.bar"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"obj.baz"}));
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"obj.a"}));
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"obj.b"}));
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"obj.b.c"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"obj.c"}));
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"other"}));
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"other.foo"}));
}

TEST(EncryptionSchemaTreeTest, AdditionalPropertiesWorksWithNestedPropertiesSubschema) {
    BSONObj schema = fromjson(R"({
        type: "object",
        additionalProperties: {
            type: "object",
            properties: {
                a: {type: "string"},
                b: {
                    encrypt: {
                        algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic",
                        keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}],
                        bsonType: "int"
                    }
                }
            }
        }})");
    auto encryptionTree = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foo.b"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"bar.b"}));
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foo.a"}));
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"bar.a"}));
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"bar.c"}));
}

TEST(EncryptionSchemaTreeTest, AdditionalPropertiesWorksWithNestedAdditionalPropertiesSubschema) {
    BSONObj schema = fromjson(R"({
        type: "object",
        additionalProperties: {
            type: "object",
            additionalProperties: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic",
                    keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}],
                    bsonType: "int"
                }
            }
        }})");
    auto encryptionTree = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foo"}));
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"bar"}));
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"baz"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foo.bar"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foo.baz"}));
}

TEST(EncryptionSchemaTreeTest, CanSuccessfullyParseAdditionalItemsWhenBoolean) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            arr: {
                type: "array",
                additionalItems: true
            }
        }})");
    auto encryptionTree = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"arr"}));
}

TEST(EncryptionSchemaTreeTest, CanSuccessfullyParseAdditionalPropertiesWhenBoolean) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            obj: {
                type: "object",
                additionalProperties: false
            }
        }})");
    auto encryptionTree = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"obj"}));
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"obj.foo"}));
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"obj.bar"}));
}

TEST(EncryptionSchemaTreeTest, InheritEncryptMetadataWithOverriding) {
    const auto uuid = UUID::gen();

    ResolvedEncryptionInfo secretMetadata{
        EncryptSchemaKeyId{std::vector<UUID>{uuid}}, FleAlgorithmEnum::kRandom, boost::none};

    ResolvedEncryptionInfo ssnMetadata{EncryptSchemaKeyId{{uuid}},
                                       FleAlgorithmEnum::kDeterministic,
                                       MatcherTypeSet{BSONType::String}};

    BSONObj schema = fromjson(
        R"({
        encryptMetadata: {
            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
            keyId: [)" +
        uuid.toBSON().getField("uuid").jsonString(
            JsonStringFormat::ExtendedCanonicalV2_0_0, false, false) +
        R"(]
        },
        type: "object",
        properties: {
            ssn: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic",
                    bsonType: "string"
                }
            },
            super: {
                type: "object",
                properties: {
                    secret: {encrypt: {}}
                }
            }
        }})");
    ASSERT(extractMetadata(schema, "ssn") == ssnMetadata);
    ASSERT(extractMetadata(schema, "super.secret") == secretMetadata);
}

TEST(EncryptionSchemaTreeTest, InheritEncryptMetadataMultipleLevels) {
    const auto uuid1 = UUID::gen();
    const auto uuid2 = UUID::gen();

    ResolvedEncryptionInfo secretMetadata{EncryptSchemaKeyId{{uuid1}},
                                          FleAlgorithmEnum::kDeterministic,
                                          MatcherTypeSet{BSONType::String}};

    ResolvedEncryptionInfo mysteryMetadata{
        EncryptSchemaKeyId{{uuid2}}, FleAlgorithmEnum::kRandom, boost::none};

    BSONObj schema = fromjson(
        R"({
        encryptMetadata: {
            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random"
        },
        type: "object",
        properties: {
            super: {
                encryptMetadata: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic",
                    keyId: [)" +
        uuid1.toBSON().getField("uuid").jsonString(
            JsonStringFormat::ExtendedCanonicalV2_0_0, false, false) +
        R"(]
                },
                type: "object",
                properties: {
                    secret: {encrypt: {bsonType: "string"}}
                }
            },
            duper: {
                type: "object",
                properties: {
                    mystery: {
                        encrypt: {
                            keyId: [)" +
        uuid2.toBSON().getField("uuid").jsonString(
            JsonStringFormat::ExtendedCanonicalV2_0_0, false, false) +
        R"(]
                        }
                    }
                }
            }
        }})");
    ASSERT(extractMetadata(schema, "super.secret") == secretMetadata);
    ASSERT(extractMetadata(schema, "duper.mystery") == mysteryMetadata);
}

TEST(EncryptionSchemaTreeTest, InheritEncryptMetadataMissingAlgorithm) {
    const auto uuid = UUID::gen();

    BSONObj schema = fromjson(
        R"({
        encryptMetadata: {
            keyId: [)" +
        uuid.toBSON().getField("uuid").jsonString(
            JsonStringFormat::ExtendedCanonicalV2_0_0, false, false) +
        R"(]
        },
        type: "object",
        properties: {
            super: {
                type: "object",
                properties: {
                    secret: {encrypt: {}}
                }
            }
        }})");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51099);
}

TEST(EncryptionSchemaTreeTest, InheritEncryptMetadataMissingKeyId) {
    BSONObj schema = fromjson(R"({
        encryptMetadata: {
            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic"
        },
        type: "object",
        properties: {
            super: {
                type: "object",
                properties: {
                    secret: {encrypt: {}}
                }
            }
        }})");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51097);
}

TEST(EncryptionSchemaTreeTest, FailsToParseIfPatternPropertiesIsNotAnObject) {
    BSONObj schema = fromjson(R"({
        type: "object",
        patternProperties: true
    })");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       ErrorCodes::TypeMismatch);
}

TEST(EncryptionSchemaTreeTest, FailsToParseIfPatternPropertiesHasNonObjectProperty) {
    BSONObj schema = fromjson(R"({
        type: "object",
        patternProperties: {
            foo: true
        }
    })");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       ErrorCodes::TypeMismatch);
}

TEST(EncryptionSchemaTreeTest, FailsToParseIfPatternPropertiesHasAnIllFormedRegex) {
    BSONObj schema = fromjson(R"({
        type: "object",
        patternProperties: {
            "(": {}
        }
    })");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       ErrorCodes::BadValue);
}

TEST(EncryptionSchemaTreeTest, LegalPatternPropertiesParsesSuccessfully) {
    BSONObj schema = fromjson(R"({
        type: "object",
        patternProperties: {
            foo: {},
            bar: {}
        }
    })");

    auto encryptionTree = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foo"}));
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"bar"}));
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"baz"}));
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foo.quux"}));
}

TEST(EncryptionSchemaTreeTest, FieldNamesMatchingPatternPropertiesReportEncryptedMetadata) {
    BSONObj schema = fromjson(R"({
        type: "object",
        patternProperties: {
            foo: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                }
            }
        }
    })");

    auto encryptionTree = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foo"}));
    ASSERT_THROWS_CODE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foo.quux"}),
                       AssertionException,
                       51102);
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"bar"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"a_foo_b"}));
    ASSERT_THROWS_CODE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"a_foo_b.quux"}),
                       AssertionException,
                       51102);
}

TEST(EncryptionSchemaTreeTest,
     PatternPropertiesWorksTogetherWithPropertiesAndAdditionalProperties) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            a: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                }
            },
            b: {type: "number"}
        },
        patternProperties: {
            foo: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                }
            },
            bar: {type: "number"}
        },
        additionalProperties: {
            encrypt: {
                algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic",
                keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}],
                bsonType: "int"
            }
        }
    })");

    auto encryptionTree = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"a"}));
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"b"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foo"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foo2"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"3foo"}));
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"bar"}));
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"bar2"}));
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"3bar"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"x"}));
    ASSERT_THROWS_CODE(
        encryptionTree->getEncryptionMetadataForPath(FieldRef{"y.z.w"}), AssertionException, 51102);
}

TEST(EncryptionSchemaTreeTest, PatternPropertiesErrorIfMultiplePatternsMatchButOnlyOneEncrypted) {
    BSONObj schema = fromjson(R"({
        type: "object",
        patternProperties: {
            foo: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                }
            },
            bar: {type: "number"}
        }
    })");

    auto encryptionTree = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT_THROWS_CODE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foobar"}),
                       AssertionException,
                       51142);
}

TEST(EncryptionSchemaTreeTest,
     PatternPropertiesErrorIfMultiplePatternsMatchWithDifferentEncryptionOptions) {
    BSONObj schema = fromjson(R"({
        type: "object",
        patternProperties: {
            foo: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                }
            },
            bar: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{"$binary": "0PCfrZlWRWeIe4ZbJ1IODQ==", "$type": "04"}]
                }
            }
        }
    })");

    auto encryptionTree = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foo"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"bar"}));
    ASSERT_THROWS_CODE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foobar"}),
                       AssertionException,
                       51142);
}

TEST(EncryptionSchemaTreeTest,
     PatternPropertiesEncryptIfMultiplePatternsMatchWithSameEncryptionOptions) {
    BSONObj schema = fromjson(R"({
        type: "object",
        patternProperties: {
            foo: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                }
            },
            bar: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                }
            }
        }
    })");

    auto encryptionTree = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foo"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"bar"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foobar"}));
}

TEST(EncryptionSchemaTreeTest, PatternPropertiesErrorIfInconsistentWithPropertiesOnlyOneEncrypted) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            foobar: {type: "string"}
        },
        patternProperties: {
            foo: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                }
            }
        }
    })");

    auto encryptionTree = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT_THROWS_CODE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foobar"}),
                       AssertionException,
                       51142);
}

TEST(EncryptionSchemaTreeTest,
     PatternPropertiesErrorIfInconsistentWithPropertiesDifferentEncryptionOptions) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            foobar: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{"$binary": "0PCfrZlWRWeIe4ZbJ1IODQ==", "$type": "04"}]
                }
            }
        },
        patternProperties: {
            foo: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                }
            }
        }
    })");

    auto encryptionTree = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT_THROWS_CODE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foobar"}),
                       AssertionException,
                       51142);
}

TEST(EncryptionSchemaTreeTest, PatternPropertiesSuccessIfAlsoInPropertiesButSameEncryptionOptions) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            foobar: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                }
            }
        },
        patternProperties: {
            foo: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                }
            }
        }
    })");

    auto encryptionTree = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foobar"}));
}

TEST(EncryptionSchemaTreeTest, PatternPropertiesFailsIfTypeObjectNotSpecified) {
    BSONObj schema = fromjson(R"({
        patternProperties: {
            foo: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                }
            }
        }
    })");

    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51077);
}

TEST(EncryptionSchemaTreeTest, NestedPatternPropertiesFailsIfTypeObjectNotSpecified) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            foo: {
                patternProperties: {
                    bar: {
                        encrypt: {
                            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                            keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                        }
                    }
                }
            }
        }
    })");

    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51077);
}

TEST(EncryptionSchemaTreeTest, PatternPropertiesNestedBelowProperties) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            foo: {
                type: "object",
                patternProperties: {
                    bar: {
                        encrypt: {
                            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                            keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                        }
                    }
                }
            }
        }
    })");

    auto encryptionTree = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foo"}));
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"baz"}));
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"baz.x.y"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foo.bar"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foo.foobar"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foo.bar2"}));
}

TEST(EncryptionSchemaTreeTest, PatternPropertiesNestedBelowPatternProperties) {
    BSONObj schema = fromjson(R"({
        type: "object",
        patternProperties: {
            foo: {
                type: "object",
                patternProperties: {
                    bar: {
                        encrypt: {
                            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                            keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                        }
                    }
                }
            }
        }
    })");

    auto encryptionTree = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foo"}));
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foobar"}));
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"baz.x.y"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foo.bar"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foo.foobar"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foo.bar2"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foobar.foobar"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"foo2.bar3"}));
}

TEST(EncryptionSchemaTreeTest, FourMatchingEncryptSpecifiersSucceedsIfAllEncryptOptionsSame) {
    BSONObj schema = fromjson(R"({
        type: "object",
        patternProperties: {
            a: {
                type: "object",
                patternProperties: {
                    x: {
                        encrypt: {
                            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                            keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                        }
                    },
                    y: {
                        encrypt: {
                            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                            keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                        }
                    }
                }
            },
            b: {
                type: "object",
                patternProperties: {
                    w: {
                        encrypt: {
                            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                            keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                        }
                    },
                    z: {
                        encrypt: {
                            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                            keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                        }
                    }
                }
            }
        }
    })");

    auto encryptionTree = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"aa.xx"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"aa.yy"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"aa.xy"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"bb.ww"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"bb.zz"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"ab.xyzw"}));
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"i.j"}));
}

TEST(EncryptionSchemaTreeTest, FourMatchingEncryptSpecifiersFailsIfOneEncryptOptionsDiffers) {
    BSONObj schema = fromjson(R"({
        type: "object",
        patternProperties: {
            a: {
                type: "object",
                patternProperties: {
                    x: {
                        encrypt: {
                            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                            keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                        }
                    },
                    y: {
                        encrypt: {
                            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                            keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                        }
                    }
                }
            },
            b: {
                type: "object",
                patternProperties: {
                    w: {
                        encrypt: {
                            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                            keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                        }
                    },
                    z: {
                        encrypt: {
                            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                            keyId: [{"$binary": "0PCfrZlWRWeIe4ZbJ1IODQ==", "$type": "04"}]
                        }
                    }
                }
            }
        }
    })");

    auto encryptionTree = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"aa.xx"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"aa.yy"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"aa.xy"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"bb.ww"}));
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"bb.zz"}));
    ASSERT_FALSE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"i.j"}));
    ASSERT_THROWS_CODE(encryptionTree->getEncryptionMetadataForPath(FieldRef{"ab.xyzw"}),
                       AssertionException,
                       51142);
}

TEST(EncryptionSchemaTreeTest,
     AdditionalPropertiesWithInconsistentEncryptOptionsNotConsideredIfPatternMatches) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            a: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                }
            }
        },
        patternProperties: {
            b: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                }
            }
        },
        additionalProperties: {type: "string"}
    })");

    auto encryptionTree = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT(encryptionTree->getEncryptionMetadataForPath(FieldRef{"ab"}));
}

TEST(EncryptionSchemaTreeTest, PatternPropertiesInheritsParentEncryptMetadata) {
    const auto uuid = UUID::gen();
    ResolvedEncryptionInfo expectedMetadata{
        EncryptSchemaKeyId{std::vector<UUID>{uuid}}, FleAlgorithmEnum::kRandom, boost::none};

    auto uuidJsonStr = uuid.toBSON().getField("uuid").jsonString(
        JsonStringFormat::ExtendedCanonicalV2_0_0, false, false);
    BSONObj schema = fromjson(R"({
        type: "object",
        encryptMetadata: {
            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
            keyId: [)" + uuidJsonStr +
                              R"(]
        },
        properties: {
            a: {
                type: "object",
                patternProperties: {
                    b: {encrypt: {}}
                }
            }
        }
    })");

    auto encryptionTree = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    auto foundMetadata = encryptionTree->getEncryptionMetadataForPath(FieldRef{"a.bb"});
    ASSERT(foundMetadata == expectedMetadata);
}

TEST(EncryptionSchemaTreeTest, ContainsEncryptReturnsTrueIfPropertyContainsEncryptedNode) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            a: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                }
            }
        }
    })");

    ASSERT(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal)
               ->mayContainEncryptedNode());
}

TEST(EncryptionSchemaTreeTest, ContainsEncryptReturnsTrueIfNestedPropertyContainsEncryptedNode) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            a: {
                type: "object",
                properties: {
                    b: {
                        encrypt: {
                            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                            keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                        }
                    }
                }
            }
        }
    })");

    ASSERT(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal)
               ->mayContainEncryptedNode());
}

TEST(EncryptionSchemaTreeTest, ContainsEncryptReturnsFalseIfPropertyDoesNotContainEncryptedNode) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            a: {type: "string"}
        }
    })");

    ASSERT_FALSE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal)
                     ->mayContainEncryptedNode());
}


TEST(EncryptionSchemaTreeTest,
     ContainsEncryptReturnsFalseIfPropertyWithNestedPropertyDoesNotContainEncryptedNode) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            a: {
                type: "object", 
                properties: {
                    b: {}
                }
            }
        }
    })");

    ASSERT_FALSE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal)
                     ->mayContainEncryptedNode());
}

TEST(EncryptionSchemaTreeTest, ContainsEncryptReturnsTrueIfAdditionalPropertiesHasEncryptNode) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            a: {type: "string"}
        },
        additionalProperties: {
            encrypt: {
                algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic",
                keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}],
                bsonType: "int"
            }
        }
    })");

    ASSERT_TRUE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal)
                    ->mayContainEncryptedNode());
}

TEST(EncryptionSchemaTreeTest,
     ContainsEncryptReturnsFalseIfAdditionalPropertiesDoesNotHaveEncryptNode) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            a: {type: "string"}
        },
        additionalProperties: {
            type: "number"
        }
    })");

    ASSERT_FALSE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal)
                     ->mayContainEncryptedNode());
}

TEST(EncryptionSchemaTreeTest, ContainsEncryptReturnsTrueIfPatternPropertiesHasEncryptNode) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            a: {
                description: "Must be a string",
                type: "string"
            }
        },
        patternProperties: {
            "^b": {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                }
            }
        }
    })");

    ASSERT_TRUE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal)
                    ->mayContainEncryptedNode());
}

TEST(EncryptionSchemaTreeTest, EncryptedBelowPrefixReturnsTrueOnShortPrefix) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            a: {
                type: "object",
                properties: {
                    b: {
                        encrypt: {
                            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                            keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                        }}
                    }
            }
        }
    })");
    ASSERT_TRUE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal)
                    ->mayContainEncryptedNodeBelowPrefix(FieldRef("a")));
}

TEST(EncryptionSchemaTreeTest, EncryptedBelowPrefixReturnsFalseOnMissingPath) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            a: {
                type: "object",
                properties: {
                    b: {
                        encrypt: {
                            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                            keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                        }}
                    }
            }
        }
    })");
    ASSERT_FALSE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal)
                     ->mayContainEncryptedNodeBelowPrefix(FieldRef("c")));
}


DEATH_TEST(EncryptionSchemaTreeTest, EncryptedBelowPrefixInvariantOnEncryptedPath, "invariant") {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            foo: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                }
            }
        }
    })");

    EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal)
        ->mayContainEncryptedNodeBelowPrefix(FieldRef("foo"));
}

TEST(EncryptionSchemaTreeTest,
     ContainsEncryptReturnsFalseIfPatternPropertiesDoesNotHaveEncryptNode) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            a: {type: "string"}
        },
        patternProperties: {
            "^b": {
                type: "number"
            }
        }
    })");

    ASSERT_FALSE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal)
                     ->mayContainEncryptedNode());
}

TEST(EncryptionSchemaTreeTest, EncryptedBelowPrefixReturnsTrueOnLongPrefix) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            a: {
                type: "object",
                properties: {
                    b: {
                        type: "object",
                        properties: {
                            c: {
                                type: "object",
                                properties: {
                                    d: {
                                        encrypt: {
                                        algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                                        keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    })");
    auto schemaTree = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT_TRUE(schemaTree->mayContainEncryptedNodeBelowPrefix(FieldRef("a")));
    ASSERT_TRUE(schemaTree->mayContainEncryptedNodeBelowPrefix(FieldRef("a.b")));
    ASSERT_TRUE(schemaTree->mayContainEncryptedNodeBelowPrefix(FieldRef("a.b.c")));
}

TEST(EncryptionSchemaTreeTest, ParseFailsIfDeterministicAndNoBSONType) {
    const auto uuid = UUID::gen();
    auto encryptObj = BSON("encrypt" << BSON("algorithm"
                                             << "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic"
                                             << "keyId" << BSON_ARRAY(uuid)));
    auto schema = BSON("type"
                       << "object"
                       << "properties" << BSON("foo" << encryptObj));
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       31051);
}

TEST(EncryptionSchemaTreeTest, ParseFailsIfDeterministicAndMultipleBSONType) {
    const auto uuid = UUID::gen();
    auto encryptObj = BSON("encrypt" << BSON("algorithm"
                                             << "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic"
                                             << "bsonType"
                                             << BSON_ARRAY("string"
                                                           << "int")
                                             << "keyId" << BSON_ARRAY(uuid)));
    auto schema = BSON("type"
                       << "object"
                       << "properties" << BSON("foo" << encryptObj));
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       31051);
}

TEST(EncryptionSchemaTreeTest, ParseFailsIfDeterministicAndPointerKey) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            ssn: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic",
                    bsonType: "string",
                    keyId: "/customKey"
                }
            },
            customKey: {bsonType: "string"}
        }
    })");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       31169);
}

TEST(EncryptionSchemaTreeTest, ParseFailsIfInheritedDeterministicAndPointerKey) {
    BSONObj schema = fromjson(R"({
        type: "object",
        encryptMetadata: {
            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic"
        },
        properties: {
             ssn: {
                encrypt: {
                    bsonType: "string",
                    keyId: "/customKey"
                }
            },
            customKey: {bsonType: "string"}
        }
    })");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       31169);
}

TEST(EncryptionSchemaTreeTest, ParseSucceedsIfRandomAndPointerKey) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            ssn: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: "/customKey"
                }
            },
            customKey: {bsonType: "string"}
        }
    })");
    ASSERT(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal));
}

TEST(EncryptionSchemaTreeTest, ParseSucceedsIfLengthOneTypeArray) {
    const auto uuid = UUID::gen();
    auto encryptObj = BSON("encrypt" << BSON("algorithm"
                                             << "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic"
                                             << "bsonType" << BSON_ARRAY("string") << "keyId"
                                             << BSON_ARRAY(uuid)));
    auto schema = BSON("type"
                       << "object"
                       << "properties" << BSON("foo" << encryptObj));
    // Just making sure the parse succeeds, don't care about the result.
    EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
}

TEST(EncryptionSchemaTreeTest, AllOfKeywordSubschemaWithoutEncryptSucceeds) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            foo: {
                allOf: [
                    {
                        type: "object",
                        properties: {
                            bar: {type: "string"}
                        }
                    },
                    {
                        type: "object",
                        properties: {
                            baz: {bsonType: "int"}
                        }
                    }
                ]
            }
        }
    })");
    auto schemaTree = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT_FALSE(schemaTree->getEncryptionMetadataForPath(FieldRef("foo")));
    ASSERT_FALSE(schemaTree->getEncryptionMetadataForPath(FieldRef("foo.bar")));
    ASSERT_FALSE(schemaTree->getEncryptionMetadataForPath(FieldRef("foo.baz")));
}

TEST(EncryptionSchemaTreeTest, AllOfKeywordSubschemaWithEncryptFails) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            foo: {
                allOf: [
                    {
                        type: "object",
                        properties: {
                            bar: {type: "string"}
                        }
                    },
                    {
                        type: "object",
                        properties: {
                            bar: {
                                encrypt: {
                                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                                    keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                                }
                            }
                        }
                    }
                ]
            }
        }
    })");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51077);
}

TEST(EncryptionSchemaTreeTest, AllOfKeywordSubschemaWithEncryptMetadataFails) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            foo: {
                allOf: [
                    {
                        type: "object",
                        properties: {
                            bar: {type: "string"}
                        }
                    },
                    {
                        type: "object",
                        properties: {
                            bar: {
                                encryptMetadata: {
                                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random"
                                }
                            }
                        }
                    }
                ]
            }
        }
    })");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       31077);
}

TEST(EncryptionSchemaTreeTest, AnyOfKeywordSubschemaWithoutEncryptSucceeds) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            foo: {
                anyOf: [
                    {
                        type: "object",
                        properties: {
                            bar: {type: "string"}
                        }
                    },
                    {
                        type: "object",
                        properties: {
                            baz: {bsonType: "int"}
                        }
                    }
                ]
            }
        }
    })");
    auto schemaTree = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT_FALSE(schemaTree->getEncryptionMetadataForPath(FieldRef("foo")));
    ASSERT_FALSE(schemaTree->getEncryptionMetadataForPath(FieldRef("foo.bar")));
    ASSERT_FALSE(schemaTree->getEncryptionMetadataForPath(FieldRef("foo.baz")));
}

TEST(EncryptionSchemaTreeTest, AnyOfKeywordSubschemaWithEncryptFails) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            foo: {
                anyOf: [
                    {
                        type: "object",
                        properties: {
                            bar: {type: "string"}
                        }
                    },
                    {
                        type: "object",
                        properties: {
                            bar: {
                                encrypt: {
                                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                                    keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                                }
                            }
                        }
                    }
                ]
            }
        }
    })");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51077);
}

TEST(EncryptionSchemaTreeTest, AnyOfKeywordSubschemaWithEncryptMetadataFails) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            foo: {
                anyOf: [
                    {
                        type: "object",
                        properties: {
                            bar: {type: "string"}
                        }
                    },
                    {
                        type: "object",
                        properties: {
                            bar: {
                                encryptMetadata: {
                                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random"
                                }
                            }
                        }
                    }
                ]
            }
        }
    })");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       31077);
}

TEST(EncryptionSchemaTreeTest, OneOfKeywordSubschemaWithoutEncryptSucceeds) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            foo: {
                oneOf: [
                    {
                        type: "object",
                        properties: {
                            bar: {type: "string"}
                        }
                    },
                    {
                        type: "object",
                        properties: {
                            baz: {bsonType: "int"}
                        }
                    }
                ]
            }
        }
    })");
    auto schemaTree = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT_FALSE(schemaTree->getEncryptionMetadataForPath(FieldRef("foo")));
    ASSERT_FALSE(schemaTree->getEncryptionMetadataForPath(FieldRef("foo.bar")));
    ASSERT_FALSE(schemaTree->getEncryptionMetadataForPath(FieldRef("foo.baz")));
}

TEST(EncryptionSchemaTreeTest, OneOfKeywordSubschemaWithEncryptFails) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            foo: {
                oneOf: [
                    {
                        type: "object",
                        properties: {
                            bar: {type: "string"}
                        }
                    },
                    {
                        type: "object",
                        properties: {
                            bar: {
                                encrypt: {
                                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                                    keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                                }
                            }
                        }
                    }
                ]
            }
        }
    })");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51077);
}

TEST(EncryptionSchemaTreeTest, OneOfKeywordSubschemaWithEncryptMetadataFails) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            foo: {
                oneOf: [
                    {
                        type: "object",
                        properties: {
                            bar: {type: "string"}
                        }
                    },
                    {
                        type: "object",
                        properties: {
                            bar: {
                                encryptMetadata: {
                                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random"
                                }
                            }
                        }
                    }
                ]
            }
        }
    })");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       31077);
}

TEST(EncryptionSchemaTreeTest, NotKeywordSubschemaWithoutEncryptSucceeds) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            foo: {
                not: {
                    type: "object",
                    properties: {
                        bar: {type: "string"}
                    }
                }
            }
        }
    })");
    auto schemaTree = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT_FALSE(schemaTree->getEncryptionMetadataForPath(FieldRef("foo")));
    ASSERT_FALSE(schemaTree->getEncryptionMetadataForPath(FieldRef("foo.bar")));
    ASSERT_FALSE(schemaTree->getEncryptionMetadataForPath(FieldRef("foo.baz")));
}

TEST(EncryptionSchemaTreeTest, NotKeywordSubschemaWithEncryptFails) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            foo: {
                not: {
                    type: "object",
                    properties: {
                        bar: {
                            encrypt: {
                                algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                                keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                            }
                        }
                    }
                }
            }
        }
    })");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51077);
}

TEST(EncryptionSchemaTreeTest, NotKeywordSubschemaWithEncryptMetadataFails) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            foo: {
                not: {
                    type: "object",
                    properties: {
                        bar: {
                            encryptMetadata: {
                                algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random"
                            }
                        }
                    }
                }
            }
        }
    })");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       31077);
}

TEST(EncryptionSchemaTreeTest, EncryptMetadataWithoutExplicitTypeObjectSpecificationFails) {
    BSONObj schema = fromjson(R"({
        properties: {
            foo: {
                encryptMetadata: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random"
                }
            }
        }
    })");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       31077);
}

void verifySchemaKeywordWorksOnlyForRemoteSchema(StringData schema, bool hasEncryptedNode) {
    ASSERT_THROWS_CODE(
        EncryptionSchemaTreeNode::parse(fromjson(schema.rawData()), EncryptionSchemaType::kLocal),
        AssertionException,
        31068);
    ASSERT_EQ(
        EncryptionSchemaTreeNode::parse(fromjson(schema.rawData()), EncryptionSchemaType::kRemote)
            ->mayContainEncryptedNode(),
        hasEncryptedNode);
};

TEST(EncryptionSchemaTreeTest, VerifyRemoteSchemaKeywordsAreAllowed) {
    verifySchemaKeywordWorksOnlyForRemoteSchema("{properties: {foo: {enum: [1, 2, 3]}}}", false);
    verifySchemaKeywordWorksOnlyForRemoteSchema(
        "{properties: {foo: {exclusiveMaximum: true, maximum: 1}}}", false);
    verifySchemaKeywordWorksOnlyForRemoteSchema(
        "{properties: {foo: {exclusiveMinimum: true, minimum: 1}}}", false);
    verifySchemaKeywordWorksOnlyForRemoteSchema("{properties: {foo: {maxItems: 1}}}", false);
    verifySchemaKeywordWorksOnlyForRemoteSchema("{properties: {foo: {minItems: 1}}}", false);
    verifySchemaKeywordWorksOnlyForRemoteSchema("{properties: {foo: {maxLength: 1}}}", false);
    verifySchemaKeywordWorksOnlyForRemoteSchema("{properties: {foo: {minLength: 1}}}", false);
    verifySchemaKeywordWorksOnlyForRemoteSchema("{maxProperties: 1}", false);
    verifySchemaKeywordWorksOnlyForRemoteSchema("{minProperties: 1}", false);
    verifySchemaKeywordWorksOnlyForRemoteSchema("{properties: {foo: {maximum: 1}}}", false);
    verifySchemaKeywordWorksOnlyForRemoteSchema("{properties: {foo: {minimum: 1}}}", false);
    verifySchemaKeywordWorksOnlyForRemoteSchema("{properties: {foo: {multipleOf: 2}}}", false);
    verifySchemaKeywordWorksOnlyForRemoteSchema("{properties: {foo: {pattern: '^bar'}}}", false);
    verifySchemaKeywordWorksOnlyForRemoteSchema("{required: ['a', 'b']}", false);
    verifySchemaKeywordWorksOnlyForRemoteSchema("{properties: {foo: {uniqueItems: true}}}", false);
}

TEST(EncryptionSchemaTreeTest, VerifyRemoteSchemaKeywordsInNestedObjectAreAllowed) {
    StringData schemaWithKeywordsInNestedObject = R"({
            type: "object",
            title: "title",
            minProperties: 4,
            maxProperties: 3,
            properties: {
                user: {
                    type: "object",
                    minProperties: 4,
                    maxProperties: 3,
                    properties: {}
                },
                a: {type: "string", minLength: 1}
            },
            patternProperties: {
                "^b": {
                    encrypt: {
                        algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                        keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                    }
                }
            }
        })";
    verifySchemaKeywordWorksOnlyForRemoteSchema(schemaWithKeywordsInNestedObject, true);
}

TEST(EncryptionSchemaTreeTest, VerifyRemoteSchemaKeywordsInsideArrayAreAllowed) {
    StringData schemaWithKeywordsInArray = R"({
            type: "object",
            properties: {
                user: {
                    type: "object",
                    properties: {
                        arr: {
                            type: "array",
                            "minItems": 2,
                            "maxItems": 3,
                            items: {
                            }
                        }, 
                        ssn : {
                            encrypt: {
                                algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                                keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                            }
                        }
                    }
                }

            }
            })";
    verifySchemaKeywordWorksOnlyForRemoteSchema(schemaWithKeywordsInArray, true);
}

TEST(EncryptionSchemaTreeTest, VerifyTitleAndDescriptionKeywordsAlwaysPermitted) {
    const auto verifySchemaKeywordWorks = [](StringData schema, bool hasEncryptedNode) {
        ASSERT_EQ(EncryptionSchemaTreeNode::parse(fromjson(schema.rawData()),
                                                  EncryptionSchemaType::kLocal)
                      ->mayContainEncryptedNode(),
                  hasEncryptedNode);
        ASSERT_EQ(EncryptionSchemaTreeNode::parse(fromjson(schema.rawData()),
                                                  EncryptionSchemaType::kRemote)
                      ->mayContainEncryptedNode(),
                  hasEncryptedNode);
    };
    verifySchemaKeywordWorks("{description: 'description'}", false);
    verifySchemaKeywordWorks("{title: 'title'}", false);

    StringData schema = R"({
                type: "object",
                title: "title",
                description: "description",
                properties: {
                    a: {type: "string", title: "title"}
                },
                patternProperties: {
                    "^b": {
                        encrypt: {
                            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                            keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                        }
                    }
                }
            })";
    verifySchemaKeywordWorks(schema, true);
}

TEST(EncryptionSchemaTreeTest, VerifyIllegalSchemaRejected) {
    const auto verifyIllegalSchemaRejected = [](StringData schema, int errorCode) {
        ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(fromjson(schema.rawData()),
                                                           EncryptionSchemaType::kLocal),
                           AssertionException,
                           errorCode);
        ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(fromjson(schema.rawData()),
                                                           EncryptionSchemaType::kRemote),
                           AssertionException,
                           errorCode);
    };
    verifyIllegalSchemaRejected("{unsupportedKeyword: {foo: {pattern: '(bar[bar'}}}",
                                ErrorCodes::FailedToParse);

    // Negative number of items.
    verifyIllegalSchemaRejected("{properties: {foo: {maxItems: -10}}}", ErrorCodes::FailedToParse);
    verifyIllegalSchemaRejected("{properties: {foo: {minItems: -10}}}", ErrorCodes::FailedToParse);

    // Type mismatch.
    verifyIllegalSchemaRejected("{properties: {foo: {enum: {invalid: 'field'}}}}",
                                ErrorCodes::TypeMismatch);
    verifyIllegalSchemaRejected("{properties: {foo: {uniqueItems: 1}}}", ErrorCodes::TypeMismatch);

    // Invalid regex.
    verifyIllegalSchemaRejected("{properties: {foo: {pattern: '(bar[bar'}}}", 51091);

    // mongocryptd currently doesn't support 'dependencies'. This can be removed when SERVER-41288
    // is completed.
    verifyIllegalSchemaRejected("{dependencies: {foo: {properties: {bar: {type: 'string'}}}}}",
                                31126);
}

TEST(EncryptionSchemaTreeTest, SchemaTreeHoldsBsonTypeSetWithMultipleTypes) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            foo: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}],
                    bsonType: ["string", "int", "long"]
                }
            }
        }
    })");

    auto schemaTree = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    auto resolvedMetadata = schemaTree->getEncryptionMetadataForPath(FieldRef{"foo"});
    ASSERT(resolvedMetadata);
    MatcherTypeSet typeSet;
    typeSet.bsonTypes = {BSONType::String, BSONType::NumberInt, BSONType::NumberLong};
    ASSERT(resolvedMetadata->bsonTypeSet == typeSet);
}

TEST(EncryptionSchemaTreeTest, FailsToParseWithNoBSONTypeInDeterministicEncrypt) {
    auto uuid = UUID::gen();
    BSONObj schema = BSON("encrypt" << BSON("algorithm"
                                            << "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic"
                                            << "keyId" << BSON_ARRAY(uuid)));
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       31051);
}

TEST(EncryptionSchemaTreeTest, FailsToParseWithBSONTypeObjectInDeterministicEncrypt) {
    auto uuid = UUID::gen();
    BSONObj schema = BSON("encrypt" << BSON("algorithm"
                                            << "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic"
                                            << "keyId" << BSON_ARRAY(uuid) << "bsonType"
                                            << "object"));
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       31122);
}

TEST(EncryptionSchemaTreeTest, FailsToParseWithEmptyArrayBSONTypeInDeterministicEncrypt) {
    auto uuid = UUID::gen();
    BSONObj schema =
        BSON("encrypt" << BSON("algorithm"
                               << "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic"
                               << "keyId" << BSON_ARRAY(uuid) << "bsonType" << BSONArray()));
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       31051);
}

TEST(EncryptionSchemaTreeTest, FailsToParseWithMultipleElementArrayBSONTypeInDeterministicEncrypt) {
    auto uuid = UUID::gen();
    BSONObj schema = BSON("encrypt" << BSON("algorithm"
                                            << "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic"
                                            << "keyId" << BSON_ARRAY(uuid) << "bsonType"
                                            << BSON_ARRAY("int"
                                                          << "string")));
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       31051);
}

TEST(EncryptionSchemaTreeTest, FailsToParseWithObjectInArrayBSONTypeInDeterministicEncrypt) {
    auto uuid = UUID::gen();
    BSONObj schema = BSON("encrypt" << BSON("algorithm"
                                            << "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic"
                                            << "keyId" << BSON_ARRAY(uuid) << "bsonType"
                                            << BSON_ARRAY("object")));
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       31122);
}

TEST(EncryptionSchemaTreeTest, FailsToParseWithSingleValueBSONTypeInEncryptObject) {
    auto uuid = UUID::gen();
    // Test MinKey
    BSONObj encrypt = BSON("encrypt" << BSON("algorithm"
                                             << "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic"
                                             << "bsonType"
                                             << "minKey"
                                             << "keyId" << BSON_ARRAY(uuid)));
    BSONObj schema = BSON("type"
                          << "object"
                          << "properties" << BSON("foo" << encrypt));
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       31041);
    // Test MaxKey
    encrypt = BSON("encrypt" << BSON("algorithm"
                                     << "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic"
                                     << "bsonType"
                                     << "maxKey"
                                     << "keyId" << BSON_ARRAY(uuid)));
    schema = BSON("type"
                  << "object"
                  << "properties" << BSON("foo" << encrypt));
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       31041);
    // Test Undefined
    encrypt = BSON("encrypt" << BSON("algorithm"
                                     << "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic"
                                     << "bsonType"
                                     << "undefined"
                                     << "keyId" << BSON_ARRAY(uuid)));
    schema = BSON("type"
                  << "object"
                  << "properties" << BSON("foo" << encrypt));
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       31041);
    // Test Null
    encrypt = BSON("encrypt" << BSON("algorithm"
                                     << "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic"
                                     << "bsonType"
                                     << "null"
                                     << "keyId" << BSON_ARRAY(uuid)));
    schema = BSON("type"
                  << "object"
                  << "properties" << BSON("foo" << encrypt));
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       31041);
}

TEST(EncryptionSchemaTreeTest, IdEncryptedWithDeterministicAlgorithmSucceeds) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            _id: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic",
                    keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}],
                    bsonType: "string"
                }
            }
        }
    })");
    EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
}

TEST(EncryptionSchemaTreeTest, NestedIdEncryptedWithRandomAlgorithmSucceeds) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            foo: {
                type: "object",
                properties: {
                    _id: {
                        encrypt: {
                            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                            keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                        }
                    }
                }
            }
        }
    })");
    EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
}

TEST(EncryptionSchemaTreeTest, IdEncryptedWithRandomAlgorithmFails) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            _id: {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                }
            }
        }
    })");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51194);
}

TEST(EncryptionSchemaTreeTest, IdDescendantEncryptedWithRandomAlgorithmFails) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            _id: {
                type: "object",
                properties: {
                    b: {
                        encrypt: {
                            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                            keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                        }}
                    }
            }
        }
    })");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51194);
}

TEST(EncryptionSchemaTreeTest,
     TopLevelAdditionalPropertiesEncryptedWithRandomAlgorithmExcludingIdSucceeds) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            _id: {
                type: "string"
            }
        },
        additionalProperties: {
            encrypt: {
                algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
            }
        }
    })");
    EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
}

TEST(EncryptionSchemaTreeTest,
     TopLevelAdditionalPropertiesEncryptedWithRandomAlgorithmIncludingIdFails) {
    BSONObj schema = fromjson(R"({
        type: "object",
        additionalProperties: {
            encrypt: {
                algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}]
            }
        }
    })");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51194);
}

TEST(EncryptionSchemaTreeTest,
     NestedPatternPropertiesMatchingIdEncryptedWithRandomAlgorithmSucceeds) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            foo: {
                type: "object",
                patternProperties: {
                    "^_": {
                        encrypt: {
                            algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                            keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                        }
                    }
                }
            }
        }
    })");
    EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
}

TEST(EncryptionSchemaTreeTest,
     TopLevelPatternPropertiesMatchingIdEncryptedWithDeterministicAlgorithmSucceeds) {
    BSONObj schema = fromjson(R"({
        type: "object",
        patternProperties: {
            "^_": {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic",
                    keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}],
                    bsonType: "string"
                }
            }
        }
    })");
    EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
}

TEST(EncryptionSchemaTreeTest,
     TopLevelPatternPropertiesMatchingIdEncryptedWithRandomAlgorithmFails) {
    BSONObj schema = fromjson(R"({
        type: "object",
        patternProperties: {
            "^_": {
                encrypt: {
                    algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                    keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}]
                }
            }
        }
    })");
    ASSERT_THROWS_CODE(EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal),
                       AssertionException,
                       51194);
}

TEST(EncryptionSchemaTreeTest, SingleUnencryptedPropertySchemasAreEqual) {
    BSONObj BSONSchema = BSON("type"
                              << "object"
                              << "properties"
                              << BSON("user" << BSON("type"
                                                     << "string")));
    auto firstSchema = EncryptionSchemaTreeNode::parse(BSONSchema, EncryptionSchemaType::kLocal);
    auto secondSchema = EncryptionSchemaTreeNode::parse(BSONSchema, EncryptionSchemaType::kLocal);
    ASSERT_TRUE(*firstSchema == *secondSchema);
}

TEST(EncryptionSchemaTreeTest, SingleEncryptedPropertySchemasAreEqual) {
    BSONObj BSONSchema = BSON("type"
                              << "object"
                              << "properties" << BSON("user" << encryptObj));
    auto firstSchema = EncryptionSchemaTreeNode::parse(BSONSchema, EncryptionSchemaType::kLocal);
    auto secondSchema = EncryptionSchemaTreeNode::parse(BSONSchema, EncryptionSchemaType::kLocal);
    ASSERT_TRUE(*firstSchema == *secondSchema);
}

TEST(EncryptionSchemaTreeTest, OutOfOrderSchemasAreEqual) {
    BSONObj firstBSON = BSON("type"
                             << "object"
                             << "properties"
                             << BSON("user" << encryptObj << "person" << encryptObj));
    BSONObj secondBSON = BSON("type"
                              << "object"
                              << "properties"
                              << BSON("person" << encryptObj << "user" << encryptObj));
    auto firstSchema = EncryptionSchemaTreeNode::parse(firstBSON, EncryptionSchemaType::kLocal);
    auto secondSchema = EncryptionSchemaTreeNode::parse(secondBSON, EncryptionSchemaType::kLocal);
    ASSERT_TRUE(*firstSchema == *secondSchema);
}

TEST(EncryptionSchemaTreeTest, DifferentEncryptionMetadataAreNotEqual) {
    BSONObj firstBSON = BSON("type"
                             << "object"
                             << "properties" << BSON("user" << encryptObj));
    BSONObj secondBSON = BSON("type"
                              << "object"
                              << "properties" << BSON("user" << encryptObjNumber));
    auto firstSchema = EncryptionSchemaTreeNode::parse(firstBSON, EncryptionSchemaType::kLocal);
    auto secondSchema = EncryptionSchemaTreeNode::parse(secondBSON, EncryptionSchemaType::kLocal);
    ASSERT_FALSE(*firstSchema == *secondSchema);
}

TEST(EncryptionSchemaTreeTest, DifferentFieldNamesAreNotEqual) {
    BSONObj firstBSON = BSON("type"
                             << "object"
                             << "properties" << BSON("user" << encryptObj));
    BSONObj secondBSON = BSON("type"
                              << "object"
                              << "properties" << BSON("person" << encryptObj));
    auto firstSchema = EncryptionSchemaTreeNode::parse(firstBSON, EncryptionSchemaType::kLocal);
    auto secondSchema = EncryptionSchemaTreeNode::parse(secondBSON, EncryptionSchemaType::kLocal);
    ASSERT_FALSE(*firstSchema == *secondSchema);
}

TEST(EncryptionSchemaTreeTest, ExtraUnencryptedLeafNodeAreNotEqual) {
    BSONObj firstBSON = BSON("type"
                             << "object"
                             << "properties" << BSON("user" << encryptObj));
    BSONObj secondBSON = BSON("type"
                              << "object"
                              << "properties"
                              << BSON("user" << encryptObj << "extra"
                                             << BSON("type"
                                                     << "string")));
    auto firstSchema = EncryptionSchemaTreeNode::parse(firstBSON, EncryptionSchemaType::kLocal);
    auto secondSchema = EncryptionSchemaTreeNode::parse(secondBSON, EncryptionSchemaType::kLocal);
    ASSERT_FALSE(*firstSchema == *secondSchema);
}

TEST(EncryptionSchemaTreeTest, IdenticalNestedSchemaAreEqual) {
    const BSONObj BSONSchema = BSON(
        "type"
        << "object"
        << "properties"
        << BSON("person" << encryptObj << "user"
                         << BSON("type"
                                 << "object"
                                 << "properties"
                                 << BSON("ssn"
                                         << encryptObj << "address" << encryptObj << "accounts"
                                         << BSON("type"
                                                 << "object"
                                                 << "properties" << BSON("bank" << encryptObj))))));
    auto firstSchema = EncryptionSchemaTreeNode::parse(BSONSchema, EncryptionSchemaType::kLocal);
    auto secondSchema = EncryptionSchemaTreeNode::parse(BSONSchema, EncryptionSchemaType::kLocal);
    ASSERT_TRUE(*firstSchema == *secondSchema);
}

TEST(EncryptionSchemaTreeTest, DifferentMetadataNestedSchemaAreNotEqual) {
    const BSONObj firstBSON = BSON(
        "type"
        << "object"
        << "properties"
        << BSON("person" << encryptObj << "user"
                         << BSON("type"
                                 << "object"
                                 << "properties"
                                 << BSON("ssn"
                                         << encryptObj << "address" << encryptObj << "accounts"
                                         << BSON("type"
                                                 << "object"
                                                 << "properties" << BSON("bank" << encryptObj))))));
    const BSONObj secondBSON =
        BSON("type"
             << "object"
             << "properties"
             << BSON("person" << encryptObj << "user"
                              << BSON("type"
                                      << "object"
                                      << "properties"
                                      << BSON("ssn" << encryptObjNumber << "address" << encryptObj
                                                    << "accounts"
                                                    << BSON("type"
                                                            << "object"
                                                            << "properties"
                                                            << BSON("bank" << encryptObj))))));
    auto firstSchema = EncryptionSchemaTreeNode::parse(firstBSON, EncryptionSchemaType::kLocal);
    auto secondSchema = EncryptionSchemaTreeNode::parse(secondBSON, EncryptionSchemaType::kLocal);
    ASSERT_FALSE(*firstSchema == *secondSchema);
}

TEST(EncryptionSchemaTreeTest, DifferentPathNestedSchemaAreNotEqual) {
    const BSONObj firstBSON = BSON(
        "type"
        << "object"
        << "properties"
        << BSON("person" << encryptObj << "user"
                         << BSON("type"
                                 << "object"
                                 << "properties"
                                 << BSON("ssn"
                                         << encryptObj << "address" << encryptObj << "accounts"
                                         << BSON("type"
                                                 << "object"
                                                 << "properties" << BSON("bank" << encryptObj))))));
    const BSONObj secondBSON = BSON(
        "type"
        << "object"
        << "properties"
        << BSON("person" << encryptObj << "user"
                         << BSON("type"
                                 << "object"
                                 << "properties"
                                 << BSON("ssn"
                                         << encryptObj << "homeLocation" << encryptObj << "accounts"
                                         << BSON("type"
                                                 << "object"
                                                 << "properties" << BSON("bank" << encryptObj))))));
    auto firstSchema = EncryptionSchemaTreeNode::parse(firstBSON, EncryptionSchemaType::kLocal);
    auto secondSchema = EncryptionSchemaTreeNode::parse(secondBSON, EncryptionSchemaType::kLocal);
    ASSERT_FALSE(*firstSchema == *secondSchema);
}

TEST(EncryptionSchemaTreeTest, ExtraNestedAdditionalPropertiesAreNotEqual) {
    const BSONObj firstBSON = BSON("type"
                                   << "object"
                                   << "properties"
                                   << BSON("person" << encryptObj << "user"
                                                    << BSON("type"
                                                            << "object"
                                                            << "properties"
                                                            << BSON("ssn" << encryptObj << "address"
                                                                          << encryptObj))));
    const BSONObj secondBSON =
        BSON("type"
             << "object"
             << "properties"
             << BSON("person" << encryptObj << "user"
                              << BSON("type"
                                      << "object"
                                      << "properties"
                                      << BSON("ssn" << encryptObj << "address" << encryptObj)
                                      << "additionalProperties" << encryptObjNumber)));
    auto firstSchema = EncryptionSchemaTreeNode::parse(firstBSON, EncryptionSchemaType::kLocal);
    auto secondSchema = EncryptionSchemaTreeNode::parse(secondBSON, EncryptionSchemaType::kLocal);
    ASSERT_FALSE(*firstSchema == *secondSchema);
}

TEST(EncryptionSchemaTreeTest, DifferentNestedAdditionalPropertiesAreNotEqual) {
    const BSONObj firstBSON =
        BSON("type"
             << "object"
             << "properties"
             << BSON("person" << encryptObj << "user"
                              << BSON("type"
                                      << "object"
                                      << "properties"
                                      << BSON("ssn" << encryptObj << "address" << encryptObj)
                                      << "additionalProperties" << encryptObj)));
    const BSONObj secondBSON =
        BSON("type"
             << "object"
             << "properties"
             << BSON("person" << encryptObj << "user"
                              << BSON("type"
                                      << "object"
                                      << "properties"
                                      << BSON("ssn" << encryptObj << "address" << encryptObj)
                                      << "additionalProperties" << encryptObjNumber)));
    auto firstSchema = EncryptionSchemaTreeNode::parse(firstBSON, EncryptionSchemaType::kLocal);
    auto secondSchema = EncryptionSchemaTreeNode::parse(secondBSON, EncryptionSchemaType::kLocal);
    ASSERT_FALSE(*firstSchema == *secondSchema);
}

TEST(EncryptionSchemaTreeTest, IdenticalNestedAdditionalPropertiesAreEqual) {
    const BSONObj BSONSchema =
        BSON("type"
             << "object"
             << "properties"
             << BSON("person" << encryptObj << "user"
                              << BSON("type"
                                      << "object"
                                      << "properties"
                                      << BSON("ssn" << encryptObj << "address" << encryptObj)
                                      << "additionalProperties" << encryptObj)));
    auto firstSchema = EncryptionSchemaTreeNode::parse(BSONSchema, EncryptionSchemaType::kLocal);
    auto secondSchema = EncryptionSchemaTreeNode::parse(BSONSchema, EncryptionSchemaType::kLocal);
    ASSERT_TRUE(*firstSchema == *secondSchema);
}

TEST(EncryptionSchemaTreeTest, OutOfOrderPatternSchemasAreEqual) {
    BSONObj firstBSON = BSON("type"
                             << "object"
                             << "patternProperties"
                             << BSON("user" << encryptObj << "person" << encryptObj));
    BSONObj secondBSON = BSON("type"
                              << "object"
                              << "patternProperties"
                              << BSON("person" << encryptObj << "user" << encryptObj));
    auto firstSchema = EncryptionSchemaTreeNode::parse(firstBSON, EncryptionSchemaType::kLocal);
    auto secondSchema = EncryptionSchemaTreeNode::parse(secondBSON, EncryptionSchemaType::kLocal);
    ASSERT_TRUE(*firstSchema == *secondSchema);
}
TEST(EncryptionSchemaTreeTest, IdenticalNestedPatternPropertiesAreEqual) {
    const BSONObj BSONSchema =
        BSON("type"
             << "object"
             << "properties"
             << BSON("person" << encryptObj << "user"
                              << BSON("type"
                                      << "object"
                                      << "properties"
                                      << BSON("ssn" << encryptObj << "address" << encryptObj)
                                      << "patternProperties" << BSON("foo" << encryptObj))));
    auto firstSchema = EncryptionSchemaTreeNode::parse(BSONSchema, EncryptionSchemaType::kLocal);
    auto secondSchema = EncryptionSchemaTreeNode::parse(BSONSchema, EncryptionSchemaType::kLocal);
    ASSERT_TRUE(*firstSchema == *secondSchema);
}

TEST(EncryptionSchemaTreeTest, DifferentNameNestedPatternPropertiesAreNotEqual) {
    const BSONObj firstBSON =
        BSON("type"
             << "object"
             << "properties"
             << BSON("person" << encryptObj << "user"
                              << BSON("type"
                                      << "object"
                                      << "properties"
                                      << BSON("ssn" << encryptObj << "address" << encryptObj)
                                      << "patternProperties" << BSON("foo" << encryptObj))));
    const BSONObj secondBSON =
        BSON("type"
             << "object"
             << "properties"
             << BSON("person" << encryptObj << "user"
                              << BSON("type"
                                      << "object"
                                      << "properties"
                                      << BSON("ssn" << encryptObj << "address" << encryptObj)
                                      << "patternProperties" << BSON("bar" << encryptObj))));
    auto firstSchema = EncryptionSchemaTreeNode::parse(firstBSON, EncryptionSchemaType::kLocal);
    auto secondSchema = EncryptionSchemaTreeNode::parse(secondBSON, EncryptionSchemaType::kLocal);
    ASSERT_FALSE(*firstSchema == *secondSchema);
}

TEST(EncryptionSchemaTreeTest, DifferentMetadataNestedPatternPropertiesAreNotEqual) {
    const BSONObj firstBSON =
        BSON("type"
             << "object"
             << "properties"
             << BSON("person" << encryptObj << "user"
                              << BSON("type"
                                      << "object"
                                      << "properties"
                                      << BSON("ssn" << encryptObj << "address" << encryptObj)
                                      << "patternProperties" << BSON("foo" << encryptObj))));
    const BSONObj secondBSON =
        BSON("type"
             << "object"
             << "properties"
             << BSON("person" << encryptObj << "user"
                              << BSON("type"
                                      << "object"
                                      << "properties"
                                      << BSON("ssn" << encryptObj << "address" << encryptObj)
                                      << "patternProperties" << BSON("foo" << encryptObjNumber))));
    auto firstSchema = EncryptionSchemaTreeNode::parse(firstBSON, EncryptionSchemaType::kLocal);
    auto secondSchema = EncryptionSchemaTreeNode::parse(secondBSON, EncryptionSchemaType::kLocal);
    ASSERT_FALSE(*firstSchema == *secondSchema);
}

TEST(EncryptionSchemaTreeTest, MissingNestedPatternPropertiesAreNotEqual) {
    const BSONObj firstBSON =
        BSON("type"
             << "object"
             << "properties"
             << BSON("person" << encryptObj << "user"
                              << BSON("type"
                                      << "object"
                                      << "properties"
                                      << BSON("ssn" << encryptObj << "address" << encryptObj)
                                      << "patternProperties" << BSON("foo" << encryptObj))));
    const BSONObj secondBSON =
        BSON("type"
             << "object"
             << "properties"
             << BSON("person" << encryptObj << "user"
                              << BSON("type"
                                      << "object"
                                      << "properties"
                                      << BSON("ssn" << encryptObj << "address" << encryptObj))));
    auto firstSchema = EncryptionSchemaTreeNode::parse(firstBSON, EncryptionSchemaType::kLocal);
    auto secondSchema = EncryptionSchemaTreeNode::parse(secondBSON, EncryptionSchemaType::kLocal);
    ASSERT_FALSE(*firstSchema == *secondSchema);
}

TEST(EncryptionSchemaTreeTest, StateMixedNodeGetEncryptionMetadataFails) {
    EncryptionSchemaStateMixedNode node{};
    ASSERT_THROWS_CODE(node.getEncryptionMetadata(), AssertionException, 31133);
}

TEST(EncryptionSchemaTreeTest, StateMixedNodeContainsEncryptedNodeReturnsTrue) {
    EncryptionSchemaStateMixedNode node{};
    ASSERT_TRUE(node.mayContainEncryptedNode());
}

TEST(EncryptionSchemaTreeTest, StateMixedNodeContainsRandomEncryptedNodeReturnsTrue) {
    EncryptionSchemaStateMixedNode node{};
    ASSERT_TRUE(node.mayContainRandomlyEncryptedNode());
}

TEST(EncryptionSchemaTreeTest, GetEncryptionMetadataFailsIfPathHitsStateMixedNode) {
    BSONObj schema = fromjson(R"({
            type: "object",
            properties: {
                ssn: {
                    encrypt: {
                        algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                        keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}],
                        bsonType: ["string", "int", "long"]
                    }
                
                },
                name: {
                    type: "string"
                }
            }
        })");
    auto root = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    root->addChild(FieldRef("ticker"), std::make_unique<EncryptionSchemaStateMixedNode>());
    ASSERT_THROWS_CODE(
        root->getEncryptionMetadataForPath(FieldRef("ticker")), AssertionException, 31133);
    ASSERT_TRUE(root->mayContainEncryptedNode());
    ASSERT(root->getEncryptionMetadataForPath(FieldRef("ssn")));
    ASSERT_FALSE(root->getEncryptionMetadataForPath(FieldRef("name")));
}

TEST(EncryptionSchemaTreeTest, StateMixedNodeGetMetadataForStateMixedNodeNestedCorrectlyFails) {
    auto root = std::make_unique<EncryptionSchemaNotEncryptedNode>();
    root->addChild(FieldRef("name.ticker"), std::make_unique<EncryptionSchemaStateMixedNode>());
    ASSERT_THROWS_CODE(
        root->getEncryptionMetadataForPath(FieldRef("name.ticker")), AssertionException, 31133);
}

TEST(EncryptionSchemaTreeTest, StateMixedNodeInAdditionalPropertiesFailsOnUndefinedFieldPath) {
    BSONObj schema = fromjson(R"({
            type: "object",
            properties: {
                ssn: {
                    encrypt: {
                        algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                        keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}],
                        bsonType: ["string", "int", "long"]
                    }
                
                },
                name: {
                    type: "string"
                }
            }
        })");
    auto root = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    root->addAdditionalPropertiesChild(std::make_unique<EncryptionSchemaStateMixedNode>());
    ASSERT(root->getEncryptionMetadataForPath(FieldRef("ssn")));
    ASSERT_FALSE(root->getEncryptionMetadataForPath(FieldRef("name")));
    ASSERT_TRUE(root->mayContainEncryptedNode());
    ASSERT_THROWS_CODE(
        root->getEncryptionMetadataForPath(FieldRef("undefinedField")), AssertionException, 31133);
}

TEST(EncryptionSchemaTreeTest, StateMixedNodeInPatternPropertiesMetadataFailOnMatchedProperty) {
    auto root = std::make_unique<EncryptionSchemaNotEncryptedNode>();
    root->addPatternPropertiesChild(R"(tickers+)",
                                    std::make_unique<EncryptionSchemaStateMixedNode>());
    ASSERT_FALSE(root->getEncryptionMetadataForPath(FieldRef("some.path")));
    ASSERT_TRUE(root->mayContainEncryptedNode());
    ASSERT_THROWS_CODE(
        root->getEncryptionMetadataForPath(FieldRef("tickerssss")), AssertionException, 31133);
}

TEST(EncryptionSchemaTreeTest, AddChildThrowsIfAddingToEncryptedNode) {
    BSONObj schema = fromjson(R"({
            type: "object",
            properties: {
                ssn: {
                    encrypt: {
                        algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Random",
                        keyId: [{$binary: "fkJwjwbZSiS/AtxiedXLNQ==", $type: "04"}],
                        bsonType: ["string", "int", "long"]
                    }
                },
                name: {
                    type: "string"
                }
            }
        })");
    auto root = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT_THROWS_CODE(
        root->addChild(FieldRef{"ssn.test"}, std::make_unique<EncryptionSchemaNotEncryptedNode>()),
        AssertionException,
        51096);
}

TEST_F(EncryptionSchemaTreeTest, CanAffixLiteralsToEncryptedNodesButNotToNotEncryptedNodes) {
    BSONObj schema = fromjson(R"({
        type: "object",
        properties: {
            a: {
                    encrypt: {
                        algorithm: "AEAD_AES_256_CBC_HMAC_SHA_512-Deterministic",
                        keyId: [{$binary: "ASNFZ4mrze/ty6mHZUMhAQ==", $type: "04"}],
                        bsonType: "int"
                    }
                }
            }
        }
    })");
    // EncryptionSchemaNotEncryptedNodes have no space for literals.
    auto root = EncryptionSchemaTreeNode::parse(schema, EncryptionSchemaType::kLocal);
    ASSERT(root->literals() == boost::none);
    // EncryptionSchemaEncryptedNodes allow access to mutable space for literals.
    auto* leaf = root->getNode(FieldRef{"a"});
    ASSERT(leaf->literals() != boost::none);
    auto literal = ExpressionConstant::create(getExpCtx(), Value{23});
    leaf->literals()->push_back(*literal);
    ASSERT_FALSE(leaf->literals()->empty());
}

}  // namespace
}  // namespace mongo
