/**
 *    Copyright (C) 2015 MongoDB Inc.
 */

#include "kmip_response.h"
#include "mongo/unittest/unittest.h"

namespace mongo {
namespace {

using namespace mongo::kmip;

const uint8_t getSymmetricKey[] = {
    /*respMessage*/ 0x42,
    0x00,
    0x7B,
    0x01,
    0x00,
    0x00,
    0x01,
    0x28,
    /*respHeader*/ 0x42,
    0x00,
    0x7A,
    0x01,
    0x00,
    0x00,
    0x00,
    0x48,
    /*protocolVer*/ 0x42,
    0x00,
    0x69,
    0x01,
    0x00,
    0x00,
    0x00,
    0x20,
    /*major*/ 0x42,
    0x00,
    0x6A,
    0x02,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x01,
    0x00,
    0x00,
    0x00,
    0x00,
    /*minor*/ 0x42,
    0x00,
    0x6B,
    0x02,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    /*timeStamp*/ 0x42,
    0x00,
    0x92,
    0x09,
    0x00,
    0x00,
    0x00,
    0x08,
    0x00,
    0x00,
    0x00,
    0x00,
    0x4A,
    0xFB,
    0xE7,
    0xC8,
    /*batchCount*/ 0x42,
    0x00,
    0x0D,
    0x02,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x01,
    0x00,
    0x00,
    0x00,
    0x00,
    /*batchItem*/ 0x42,
    0x00,
    0x0F,
    0x01,
    0x00,
    0x00,
    0x00,
    0xD0,
    0x42,
    0x00,
    0x5C,
    0x05,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x0A,
    0x00,
    0x00,
    0x00,
    0x00,
    /*resStatus*/ 0x42,
    0x00,
    0x7F,
    0x05,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    /*respPayload*/ 0x42,
    0x00,
    0x7C,
    0x01,
    0x00,
    0x00,
    0x00,
    0xA8,
    0x42,
    0x00,
    0x57,
    0x05,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x02,
    0x00,
    0x00,
    0x00,
    0x00,
    /*UID*/ 0x42,
    0x00,
    0x94,
    0x07,
    0x00,
    0x00,
    0x00,
    0x24,
    0x31,
    0x65,
    0x64,
    0x32,
    0x38,
    0x65,
    0x61,
    0x35,
    0x2D,
    0x32,
    0x62,
    0x33,
    0x31,
    0x2D,
    0x34,
    0x31,
    0x34,
    0x35,
    0x2D,
    0x62,
    0x63,
    0x66,
    0x32,
    0x2D,
    0x33,
    0x36,
    0x64,
    0x30,
    0x37,
    0x35,
    0x36,
    0x64,
    0x33,
    0x38,
    0x39,
    0x30,
    0x00,
    0x00,
    0x00,
    0x00,
    /*key*/ 0x42,
    0x00,
    0x8F,
    0x01,
    0x00,
    0x00,
    0x00,
    0x60,
    /*block*/ 0x42,
    0x00,
    0x40,
    0x01,
    0x00,
    0x00,
    0x00,
    0x58,
    /*type*/ 0x42,
    0x00,
    0x42,
    0x05,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x01,
    0x00,
    0x00,
    0x00,
    0x00,
    /*value*/ 0x42,
    0x00,
    0x45,
    0x01,
    0x00,
    0x00,
    0x00,
    0x20,
    /*mat*/ 0x42,
    0x00,
    0x43,
    0x08,
    0x00,
    0x00,
    0x00,
    0x18,
    0xC8,
    0xE5,
    0x15,
    0x23,
    0xF7,
    0x3D,
    0x6E,
    0xE9,
    0xF4,
    0x0E,
    0xAB,
    0x7C,
    0xD0,
    0x68,
    0x25,
    0x49,
    0x9D,
    0x8C,
    0x0B,
    0xD0,
    0x73,
    0x9E,
    0x10,
    0x46,
    /*algo*/ 0x42,
    0x00,
    0x28,
    0x05,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x02,
    0x00,
    0x00,
    0x00,
    0x00,
    /*length*/ 0x42,
    0x00,
    0x2A,
    0x02,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0xA8,
    0x00,
    0x00,
    0x00,
    0x00};

const uint8_t destroyKey[] = {
    /*respMessage*/ 0x42,
    0x00,
    0x7B,
    0x01,
    0x00,
    0x00,
    0x00,
    0xC0,
    /*respHeader*/ 0x42,
    0x00,
    0x7A,
    0x01,
    0x00,
    0x00,
    0x00,
    0x48,
    /*protocolVer*/ 0x42,
    0x00,
    0x69,
    0x01,
    0x00,
    0x00,
    0x00,
    0x20,
    /*major*/ 0x42,
    0x00,
    0x6A,
    0x02,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x01,
    0x00,
    0x00,
    0x00,
    0x00,
    /*minor*/ 0x42,
    0x00,
    0x6B,
    0x02,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    /*timeStamp*/ 0x42,
    0x00,
    0x92,
    0x09,
    0x00,
    0x00,
    0x00,
    0x08,
    0x00,
    0x00,
    0x00,
    0x00,
    0x4A,
    0xFB,
    0xE7,
    0xC2,
    /*batchCount*/ 0x42,
    0x00,
    0x0D,
    0x02,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x01,
    0x00,
    0x00,
    0x00,
    0x00,
    /*batchItem*/ 0x42,
    0x00,
    0x0F,
    0x01,
    0x00,
    0x00,
    0x00,
    0x68,
    /*operation*/ 0x42,
    0x00,
    0x5C,
    0x05,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x01,
    0x00,
    0x00,
    0x00,
    0x00,
    /*resultStatus*/ 0x42,
    0x00,
    0x7F,
    0x05,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    /*respPayload*/ 0x42,
    0x00,
    0x7C,
    0x01,
    0x00,
    0x00,
    0x00,
    0x40,
    /*objectType*/ 0x42,
    0x00,
    0x57,
    0x05,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x02,
    0x00,
    0x00,
    0x00,
    0x00,
    /*UID*/ 0x42,
    0x00,
    0x94,
    0x07,
    0x00,
    0x00,
    0x00,
    0x24,
    0x66,
    0x63,
    0x38,
    0x38,
    0x33,
    0x33,
    0x64,
    0x65,
    0x2D,
    0x37,
    0x30,
    0x64,
    0x32,
    0x2D,
    0x34,
    0x65,
    0x63,
    0x65,
    0x2D,
    0x62,
    0x30,
    0x36,
    0x33,
    0x2D,
    0x66,
    0x65,
    0x64,
    0x65,
    0x33,
    0x61,
    0x33,
    0x63,
    0x35,
    0x39,
    0x66,
    0x65,
    0x00,
    0x00,
    0x00,
    0x00};

const uint8_t extraTags[] = {
    /*respMessage*/ 0x42,
    0x00,
    0x7B,
    0x01,
    0x00,
    0x00,
    0x00,
    0xC0,
    /*respHeader*/ 0x42,
    0x00,
    0x7A,
    0x01,
    0x00,
    0x00,
    0x00,
    0x48,
    /*protocolVer*/ 0x42,
    0x00,
    0x69,
    0x01,
    0x00,
    0x00,
    0x00,
    0x20,
    /*major*/ 0x42,
    0x00,
    0x6A,
    0x02,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x01,
    0x00,
    0x00,
    0x00,
    0x00,
    /*minor*/ 0x42,
    0x00,
    0x6B,
    0x02,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    /*integer1*/ 0x42,
    0xFF,
    0xFF,
    0x02,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    /*integer2*/ 0x42,
    0xFF,
    0xFF,
    0x02,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    /*string1*/ 0x42,
    0xFF,
    0xFF,
    0x07,
    0x00,
    0x00,
    0x00,
    0x11,
    0xA1,
    0xA2,
    0xA3,
    0xA4,
    0xA5,
    0xA6,
    0xA7,
    0xA8,
    0xA9,
    0xAA,
    0xAB,
    0xAC,
    0xAD,
    0xAE,
    0xAF,
    0xB0,
    0xB1,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    /*timeStamp*/ 0x42,
    0x00,
    0x92,
    0x09,
    0x00,
    0x00,
    0x00,
    0x08,
    0x00,
    0x00,
    0x00,
    0x00,
    0x4A,
    0xFB,
    0xE7,
    0xC2,
    /*batchCount*/ 0x42,
    0x00,
    0x0D,
    0x02,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x01,
    0x00,
    0x00,
    0x00,
    0x00,
    /*batchItem*/ 0x42,
    0x00,
    0x0F,
    0x01,
    0x00,
    0x00,
    0x00,
    0x68,
    /*operation*/ 0x42,
    0x00,
    0x5C,
    0x05,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x01,
    0x00,
    0x00,
    0x00,
    0x00,
    /*resStatus*/ 0x42,
    0x00,
    0x7F,
    0x05,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    /*respPayload*/ 0x42,
    0x00,
    0x7C,
    0x01,
    0x00,
    0x00,
    0x00,
    0x40,
    /*enum1*/ 0x42,
    0xFF,
    0xFF,
    0x05,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    /*objectType*/ 0x42,
    0x00,
    0x57,
    0x05,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x02,
    0x00,
    0x00,
    0x00,
    0x00,
    /*UID*/ 0x42,
    0x00,
    0x94,
    0x07,
    0x00,
    0x00,
    0x00,
    0x24,
    0x66,
    0x63,
    0x38,
    0x38,
    0x33,
    0x33,
    0x64,
    0x65,
    0x2D,
    0x37,
    0x30,
    0x64,
    0x32,
    0x2D,
    0x34,
    0x65,
    0x63,
    0x65,
    0x2D,
    0x62,
    0x30,
    0x36,
    0x33,
    0x2D,
    0x66,
    0x65,
    0x64,
    0x65,
    0x33,
    0x61,
    0x33,
    0x63,
    0x35,
    0x39,
    0x66,
    0x65,
    0x00,
    0x00,
    0x00,
    0x00};

const uint8_t keyNotFound[] = {
    0x42, 0x00, 0x7b, 0x01, 0x00, 0x00, 0x00, 0xa0, 0x42, 0x00, 0x7a, 0x01, 0x00, 0x00, 0x00, 0x48,
    0x42, 0x00, 0x69, 0x01, 0x00, 0x00, 0x00, 0x20, 0x42, 0x00, 0x6a, 0x02, 0x00, 0x00, 0x00, 0x04,
    0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x42, 0x00, 0x6b, 0x02, 0x00, 0x00, 0x00, 0x04,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x42, 0x00, 0x92, 0x09, 0x00, 0x00, 0x00, 0x08,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x0f, 0xf6, 0xc5, 0x42, 0x00, 0x0d, 0x02, 0x00, 0x00, 0x00, 0x04,
    0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x42, 0x00, 0x0f, 0x01, 0x00, 0x00, 0x00, 0x48,
    0x42, 0x00, 0x5c, 0x05, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x00,
    0x42, 0x00, 0x7f, 0x05, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x42, 0x00, 0x7e, 0x05, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x42, 0x00, 0x7d, 0x07, 0x00, 0x00, 0x00, 0x09, 0x4e, 0x4f, 0x54, 0x5f, 0x46, 0x4f, 0x55, 0x4e,
    0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

const uint8_t getAttributes[] = {
    /*respMessage*/ 0x42,
    0x00,
    0x7B,
    0x01,
    0x00,
    0x00,
    0x01,
    0xB0,
    /*respHeader*/ 0x42,
    0x00,
    0x7A,
    0x01,
    0x00,
    0x00,
    0x00,
    0x48,
    /*protocolVer*/ 0x42,
    0x00,
    0x69,
    0x01,
    0x00,
    0x00,
    0x00,
    0x20,
    /*major*/ 0x42,
    0x00,
    0x6A,
    0x02,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x01,
    0x00,
    0x00,
    0x00,
    0x00,
    /*minor*/ 0x42,
    0x00,
    0x6B,
    0x02,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    /*timeStamp*/ 0x42,
    0x00,
    0x92,
    0x09,
    0x00,
    0x00,
    0x00,
    0x08,
    0x00,
    0x00,
    0x00,
    0x00,
    0x4A,
    0xFB,
    0xE7,
    0xC6,
    /*batchCount*/ 0x42,
    0x00,
    0x0D,
    0x02,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x01,
    0x00,
    0x00,
    0x00,
    0x00,
    /*batchItem*/ 0x42,
    0x00,
    0x0F,
    0x01,
    0x00,
    0x00,
    0x01,
    0x58,
    /*operation*/ 0x42,
    0x00,
    0x5C,
    0x05,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x0B,
    0x00,
    0x00,
    0x00,
    0x00,
    /*resStatus*/ 0x42,
    0x00,
    0x7F,
    0x05,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    /*respPayload*/ 0x42,
    0x00,
    0x7C,
    0x01,
    0x00,
    0x00,
    0x01,
    0x30,
    /*UID*/ 0x42,
    0x00,
    0x94,
    0x07,
    0x00,
    0x00,
    0x00,
    0x24,
    0x36,
    0x31,
    0x62,
    0x31,
    0x30,
    0x36,
    0x31,
    0x34,
    0x2D,
    0x64,
    0x38,
    0x62,
    0x35,
    0x2D,
    0x34,
    0x36,
    0x66,
    0x39,
    0x2D,
    0x38,
    0x64,
    0x31,
    0x37,
    0x2D,
    0x32,
    0x66,
    0x61,
    0x36,
    0x65,
    0x61,
    0x31,
    0x64,
    0x37,
    0x34,
    0x37,
    0x61,
    0x00,
    0x00,
    0x00,
    0x00,
    /*Attr*/ 0x42,
    0x00,
    0x08,
    0x01,
    0x00,
    0x00,
    0x00,
    0x28,
    /*Name*/ 0x42,
    0x00,
    0x0A,
    0x07,
    0x00,
    0x00,
    0x00,
    0x0C,
    0x4F,
    0x62,
    0x6A,
    0x65,
    0x63,
    0x74,
    0x20,
    0x47,
    0x72,
    0x6F,
    0x75,
    0x70,
    0x00,
    0x00,
    0x00,
    0x00,
    /*Value*/ 0x42,
    0x00,
    0x0B,
    0x07,
    0x00,
    0x00,
    0x00,
    0x06,
    0x47,
    0x72,
    0x6F,
    0x75,
    0x70,
    0x31,
    0x00,
    0x00,
    /*Attr*/ 0x42,
    0x00,
    0x08,
    0x01,
    0x00,
    0x00,
    0x00,
    0x58,
    /*Name*/ 0x42,
    0x00,
    0x0A,
    0x07,
    0x00,
    0x00,
    0x00,
    0x20,
    0x41,
    0x70,
    0x70,
    0x6C,
    0x69,
    0x63,
    0x61,
    0x74,
    0x69,
    0x6F,
    0x6E,
    0x20,
    0x53,
    0x70,
    0x65,
    0x63,
    0x69,
    0x66,
    0x69,
    0x63,
    0x20,
    0x49,
    0x6E,
    0x66,
    0x6F,
    0x72,
    0x6D,
    0x61,
    0x74,
    0x69,
    0x6F,
    0x6E,
    /*Value*/ 0x42,
    0x00,
    0x0B,
    0x01,
    0x00,
    0x00,
    0x00,
    0x28,
    /*NS*/ 0x42,
    0x00,
    0x03,
    0x07,
    0x00,
    0x00,
    0x00,
    0x03,
    0x73,
    0x73,
    0x6C,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    /*Data*/ 0x42,
    0x00,
    0x02,
    0x07,
    0x00,
    0x00,
    0x00,
    0x0F,
    0x77,
    0x77,
    0x77,
    0x2E,
    0x65,
    0x78,
    0x61,
    0x6D,
    0x70,
    0x6C,
    0x65,
    0x2E,
    0x63,
    0x6F,
    0x6D,
    0x00,
    /*Attr*/ 0x42,
    0x00,
    0x08,
    0x01,
    0x00,
    0x00,
    0x00,
    0x30,
    /*Name*/ 0x42,
    0x00,
    0x0A,
    0x07,
    0x00,
    0x00,
    0x00,
    0x13,
    0x43,
    0x6F,
    0x6E,
    0x74,
    0x61,
    0x63,
    0x74,
    0x20,
    0x49,
    0x6E,
    0x66,
    0x6F,
    0x72,
    0x6D,
    0x61,
    0x74,
    0x69,
    0x6F,
    0x6E,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    /*Value*/ 0x42,
    0x00,
    0x0B,
    0x07,
    0x00,
    0x00,
    0x00,
    0x03,
    0x4A,
    0x6F,
    0x65,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00};

const uint8_t gibberish[] = {0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17};

const uint8_t incomplete[] = {
    /*respMessage*/ 0x42,
    0x00,
    0x7B,
    0x01,
    0x00,
    0x00,
    0x01,
    0xB0,
    /*respHeader*/ 0x42,
    0x00,
    0x7A,
    0x01,
    0x00,
    0x00,
    0x00,
    0x48,
    /*protocolVer*/ 0x42,
    0x00,
    0x69,
    0x01,
    0x00,
    0x00,
    0x00,
    0x20,
    /*major*/ 0x42,
    0x00,
    0x6A,
    0x02,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x01,
    0x00,
    0x00,
    0x00,
    0x00,
    /*minor*/ 0x42,
    0x00,
    0x6B,
    0x02,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    0x00,
    /*timeStamp*/ 0x42,
    0x00,
    0x92,
    0x09,
    0x00,
    0x00,
    0x00,
    0x08,
    0x00,
    0x00,
    0x00,
    0x00,
    0x4A,
    0xFB,
    0xE7,
    0xC6,
    /*batchCount*/ 0x42,
    0x00,
    0x0D,
    0x02,
    0x00,
    0x00,
    0x00,
    0x04,
    0x00,
    0x00,
    0x00,
    0x01,
    0x00,
    0x00,
    0x00,
    0x00};

TEST(KMIPResponseParsing, GetSymmetricKey) {
    // Get symmetric key response
    StatusWith<KMIPResponse> swKeyResponse = KMIPResponse::create(
        reinterpret_cast<const char*>(getSymmetricKey), sizeof(getSymmetricKey));
    ASSERT(swKeyResponse.isOK()) << "Failed to parse getSymmetricKey "
                                 << swKeyResponse.getStatus().toString();
    KMIPResponse r = std::move(swKeyResponse.getValue());

    uint32_t major = std::get<0>(r.getProtocolVersion());
    uint32_t minor = std::get<1>(r.getProtocolVersion());
    ASSERT(major == 1 && minor == 0) << "Unexpected protocol version" << major << "." << minor;

    ASSERT(r.getOpType() == static_cast<uint8_t>(OperationType::get))
        << "Unexpected operation type: " << r.getOpType();

    const std::string expectedUID = "1ed28ea5-2b31-4145-bcf2-36d0756d3890";
    ASSERT(expectedUID == r.getUID()) << "UID mismatch " << expectedUID << " != " << r.getUID();

    const uint8_t expectedKey[] = {0xC8, 0xE5, 0x15, 0x23, 0xF7, 0x3D, 0x6E, 0xE9,
                                   0xF4, 0x0E, 0xAB, 0x7C, 0xD0, 0x68, 0x25, 0x49,
                                   0x9D, 0x8C, 0x0B, 0xD0, 0x73, 0x9E, 0x10, 0x46};
    std::unique_ptr<SymmetricKey> key = r.getSymmetricKey();
    ASSERT(memcmp(expectedKey, key.get()->getKey(), key.get()->getKeySize()) == 0)
        << "Symmetric key mismatch";
    ASSERT(key.get()->getAlgorithm() == 0x02)
        << "Algorithm mismatch, expected 0x02, found " << key.get()->getAlgorithm();
    ASSERT(key.get()->getKeySize() == 21)
        << "Key size mismatch, expexted 21, found " << key.get()->getKeySize();
}

TEST(KMIPResponseParsing, DestroyKey) {
    // Destroy key response
    StatusWith<KMIPResponse> swDestroyResponse =
        KMIPResponse::create(reinterpret_cast<const char*>(destroyKey), sizeof(destroyKey));
    ASSERT(swDestroyResponse.isOK())
        << "Failed to parse destroy response" << swDestroyResponse.getStatus().toString();

    KMIPResponse r = std::move(swDestroyResponse.getValue());
    const std::string expectedUID = "fc8833de-70d2-4ece-b063-fede3a3c59fe";
    ASSERT(expectedUID == r.getUID()) << "UID mismatch " << expectedUID << " != " << r.getUID();
}

TEST(KMIPResponseParsing, ExtraTags) {
    // Extra tags
    StatusWith<KMIPResponse> swExtraTags =
        KMIPResponse::create(reinterpret_cast<const char*>(extraTags), sizeof(extraTags));
    ASSERT(swExtraTags.isOK()) << "Failed to parse extraTags response"
                               << swExtraTags.getStatus().toString();
}

TEST(KMIPResponseParsing, KeyNotFound) {
    StatusWith<KMIPResponse> swResponse =
        KMIPResponse::create(reinterpret_cast<const char*>(keyNotFound), sizeof(keyNotFound));
    ASSERT(swResponse.isOK()) << "Failed to parse keyNotFound response"
                              << swResponse.getStatus().toString();

    KMIPResponse r = std::move(swResponse.getValue());
    ASSERT(r.getResultStatus() != kmip::statusSuccess) << "Expected command to fail";
    ASSERT(r.getResultReason() == 0x01 /* Item not found */)
        << "Expected error code 0x01 (Item not found), found: " << r.getResultReason();
    ASSERT(r.getResultMsg() == "NOT_FOUND")
        << "Expected result msg \"NOT_FOUND\", found: " << r.getResultMsg();
}

TEST(KMIPResponseParsing, NegativeTests) {
    StatusWith<KMIPResponse> swResponse =
        KMIPResponse::create(reinterpret_cast<const char*>(getAttributes), sizeof(getAttributes));

    ASSERT(swResponse.getStatus().reason() ==
           "Response message was malformed: unknown operation type 11")
        << "Invalid error message for unsupported response getAttributes";

    swResponse = KMIPResponse::create(reinterpret_cast<const char*>(gibberish), sizeof(gibberish));
    ASSERT(!swResponse.isOK()) << "Parsing gibberish did not produce an invalid Status";

    swResponse =
        KMIPResponse::create(reinterpret_cast<const char*>(incomplete), sizeof(incomplete));
    ASSERT(!swResponse.isOK())
        << "Parsing an incomplete response did not produce an invalid Status";
}
}  // namespace
}  // namespace mongo
