/**
 *  Copyright (C) 2015 MongoDB Inc.
 */

#include "kmip_request.h"
#include "mongo/unittest/unittest.h"

namespace mongo {
namespace {

TEST(KMIPRequestGeneration, CreateSymmetricKey) {
    std::vector<uint8_t> algorithm(std::begin(mongo::kmip::aesCryptoAlgorithm),
                                   std::end(mongo::kmip::aesCryptoAlgorithm));
    std::vector<uint8_t> length = mongo::kmip::convertIntToBigEndianArray(128);
    std::vector<uint8_t> usageMask;
    usageMask.push_back(0x00);
    usageMask.push_back(0x00);
    usageMask.push_back(0x00);
    usageMask.push_back(0x0C);

    mongo::kmip::CreateKMIPRequestParameters createParams(algorithm, length, usageMask);
    std::vector<uint8_t> createOutput = mongo::kmip::encodeKMIPRequest(createParams);

    std::vector<uint8_t> expectedOutput = {
        0x42, 0x00, 0x78, 0x01, 0x00, 0x00, 0x01, 0x20, 0x42, 0x00, 0x77, 0x01, 0x00, 0x00, 0x00,
        0x38, 0x42, 0x00, 0x69, 0x01, 0x00, 0x00, 0x00, 0x20, 0x42, 0x00, 0x6a, 0x02, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x42, 0x00, 0x6b, 0x02, 0x00,
        0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x42, 0x00, 0x0d, 0x02,
        0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x42, 0x00, 0x0f,
        0x01, 0x00, 0x00, 0x00, 0xd8, 0x42, 0x00, 0x5c, 0x05, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
        0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x42, 0x00, 0x79, 0x01, 0x00, 0x00, 0x00, 0xc0, 0x42,
        0x00, 0x57, 0x05, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00,
        0x42, 0x00, 0x91, 0x01, 0x00, 0x00, 0x00, 0xa8, 0x42, 0x00, 0x08, 0x01, 0x00, 0x00, 0x00,
        0x30, 0x42, 0x00, 0x0a, 0x07, 0x00, 0x00, 0x00, 0x17, 0x43, 0x72, 0x79, 0x70, 0x74, 0x6f,
        0x67, 0x72, 0x61, 0x70, 0x68, 0x69, 0x63, 0x20, 0x41, 0x6c, 0x67, 0x6f, 0x72, 0x69, 0x74,
        0x68, 0x6d, 0x00, 0x42, 0x00, 0x0b, 0x05, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x03,
        0x00, 0x00, 0x00, 0x00, 0x42, 0x00, 0x08, 0x01, 0x00, 0x00, 0x00, 0x30, 0x42, 0x00, 0x0a,
        0x07, 0x00, 0x00, 0x00, 0x14, 0x43, 0x72, 0x79, 0x70, 0x74, 0x6f, 0x67, 0x72, 0x61, 0x70,
        0x68, 0x69, 0x63, 0x20, 0x4c, 0x65, 0x6e, 0x67, 0x74, 0x68, 0x00, 0x00, 0x00, 0x00, 0x42,
        0x00, 0x0b, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00,
        0x42, 0x00, 0x08, 0x01, 0x00, 0x00, 0x00, 0x30, 0x42, 0x00, 0x0a, 0x07, 0x00, 0x00, 0x00,
        0x18, 0x43, 0x72, 0x79, 0x70, 0x74, 0x6f, 0x67, 0x72, 0x61, 0x70, 0x68, 0x69, 0x63, 0x20,
        0x55, 0x73, 0x61, 0x67, 0x65, 0x20, 0x4d, 0x61, 0x73, 0x6b, 0x42, 0x00, 0x0b, 0x02, 0x00,
        0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00};

    ASSERT(createOutput == expectedOutput);
}

TEST(KMIPRequestGeneration, GetSymmetricKey) {
    const uint8_t uidx[] = {0x34, 0x39, 0x61, 0x31, 0x63, 0x61, 0x38, 0x38, 0x2D, 0x36, 0x62, 0x65,
                            0x61, 0x2D, 0x34, 0x66, 0x62, 0x32, 0x2D, 0x62, 0x34, 0x35, 0x30, 0x2D,
                            0x37, 0x65, 0x35, 0x38, 0x38, 0x30, 0x32, 0x63, 0x33, 0x30, 0x33, 0x38};
    std::vector<uint8_t> uid(std::begin(uidx), std::end(uidx));

    mongo::kmip::GetKMIPRequestParameters getParams(uid);
    std::vector<uint8_t> getOutput = mongo::kmip::encodeKMIPRequest(getParams);

    std::vector<uint8_t> expectedOutput = {
        0x42, 0x00, 0x78, 0x01, 0x00, 0x00, 0x00, 0x90, 0x42, 0x00, 0x77, 0x01, 0x00, 0x00,
        0x00, 0x38, 0x42, 0x00, 0x69, 0x01, 0x00, 0x00, 0x00, 0x20, 0x42, 0x00, 0x6A, 0x02,
        0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x42, 0x00,
        0x6B, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x42, 0x00, 0x0D, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
        0x00, 0x00, 0x42, 0x00, 0x0F, 0x01, 0x00, 0x00, 0x00, 0x48, 0x42, 0x00, 0x5C, 0x05,
        0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x0A, 0x00, 0x00, 0x00, 0x00, 0x42, 0x00,
        0x79, 0x01, 0x00, 0x00, 0x00, 0x30, 0x42, 0x00, 0x94, 0x07, 0x00, 0x00, 0x00, 0x24,
        0x34, 0x39, 0x61, 0x31, 0x63, 0x61, 0x38, 0x38, 0x2D, 0x36, 0x62, 0x65, 0x61, 0x2D,
        0x34, 0x66, 0x62, 0x32, 0x2D, 0x62, 0x34, 0x35, 0x30, 0x2D, 0x37, 0x65, 0x35, 0x38,
        0x38, 0x30, 0x32, 0x63, 0x33, 0x30, 0x33, 0x38, 0x00, 0x00, 0x00, 0x00};

    ASSERT(getOutput == expectedOutput);
}

TEST(KMIPRequestGeneration, DiscoverVersions) {
    mongo::kmip::DiscoverVersionsKMIPRequestParameters discoverParams;
    std::vector<uint8_t> discoverOutput = mongo::kmip::encodeKMIPRequest(discoverParams);

    std::vector<uint8_t> expectedOutput = {
        0x42, 0x00, 0x78, 0x01, 0x00, 0x00, 0x00, 0x60, 0x42, 0x00, 0x77, 0x01, 0x00, 0x00, 0x00,
        0x38, 0x42, 0x00, 0x69, 0x01, 0x00, 0x00, 0x00, 0x20, 0x42, 0x00, 0x6A, 0x02, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x42, 0x00, 0x6B, 0x02, 0x00,
        0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x42, 0x00, 0x0D, 0x02,
        0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x42, 0x00, 0x0F,
        0x01, 0x00, 0x00, 0x00, 0x18, 0x42, 0x00, 0x5C, 0x05, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
        0x00, 0x1E, 0x00, 0x00, 0x00, 0x00, 0x42, 0x00, 0x79, 0x01, 0x00, 0x00, 0x00, 0x00};

    ASSERT(discoverOutput == expectedOutput);
}

}  // namespace
}  // namespace mongo
