/**
 *    Copyright (C) 2013 10gen Inc.
 */

#define MONGO_LOGV2_DEFAULT_COMPONENT ::mongo::logv2::LogComponent::kAccessControl

#include "mongo/platform/basic.h"

#include "audit_event.h"
#include "audit_log.h"
#include "audit_manager_global.h"
#include "audit_private.h"
#include "mongo/base/status.h"
#include "mongo/base/string_data.h"
#include "mongo/db/audit.h"
#include "mongo/db/auth/action_type.h"
#include "mongo/db/auth/authorization_session.h"
#include "mongo/db/auth/user_name.h"
#include "mongo/db/client.h"
#include "mongo/util/assert_util.h"

namespace mongo {

namespace audit {
namespace {

constexpr auto kMechanism = "mechanism"_sd;
constexpr auto kUser = "user"_sd;
constexpr auto kDatabase = "db"_sd;

/**
 * Event representing the result of an authentication activity.
 */
class AuthenticationEvent : public AuditEvent {
public:
    AuthenticationEvent(const AuditEventEnvelope& envelope, const AuthenticateEvent& authEvent)
        : AuditEvent(envelope), _authEvent{authEvent} {}

private:
    BSONObjBuilder& putParamsBSON(BSONObjBuilder& builder) const final {
        _authEvent.appendExtraInfo(&builder);
        builder.append(kUser, _authEvent.getUser());
        builder.append(kDatabase, _authEvent.getDatabase());
        builder.append(kMechanism, _authEvent.getMechanism());
        return builder;
    }

    audit::AuthenticateEvent _authEvent;
};

}  // namespace
}  // namespace audit

void audit::logAuthentication(Client* client, const AuthenticateEvent& authEvent) {
    if (!getGlobalAuditManager()->enabled) {
        return;
    }

    AuthenticationEvent event(makeEnvelope(client, ActionType::authenticate, authEvent.getResult()),
                              authEvent);
    if (getGlobalAuditManager()->auditFilter->matches(&event)) {
        logEvent(event);
    }
}

}  // namespace mongo
