/**
 * Test that search queries with sorts fail against sharded and unsharded collections in
 * sharded clusters in v5.0 and below.
 */
(function() {
"use strict";

load("src/mongo/db/modules/enterprise/jstests/search/lib/mongotmock.js");
load("src/mongo/db/modules/enterprise/jstests/search/lib/shardingtest_with_mongotmock.js");

const dbName = "test";
const shardedCollName = jsTestName() + "_sharded";
const unshardedCollName = jsTestName() + "_unsharded";

const stWithMock = new ShardingTestWithMongotMock({
    name: "sharded_search",
    shards: {
        rs0: {nodes: 1},
        rs1: {nodes: 1},
    },
    mongos: 1,
    other: {
        rsOptions: {setParameter: {enableTestCommands: 1}},
    }
});
stWithMock.start();
const st = stWithMock.st;

const mongos = st.s;

const testDB = mongos.getDB(dbName);
const shardedColl = testDB.getCollection(shardedCollName);
const mongostmock = stWithMock.getMockConnectedToHost(mongos);

shardedColl.drop();
assert.commandWorked(shardedColl.insert([
    {_id: 1, x: "ow"},
    {_id: 2, x: "now", y: "lorem"},
    {_id: 3, x: "brown", y: "ipsum"},
    {_id: 4, x: "cow", y: "lorem ipsum"},
    {_id: 11, x: "brown", y: "ipsum"},
    {_id: 12, x: "cow", y: "lorem ipsum"},
    {_id: 13, x: "brown", y: "ipsum"},
    {_id: 14, x: "cow", y: "lorem ipsum"}
]));

// Shard the test collection, split it at {_id: 10}, and move the higher chunk to shard1.
assert.commandWorked(mongos.getDB("admin").runCommand({enableSharding: dbName}));
st.ensurePrimaryShard(dbName, st.shard0.name);
st.shardColl(shardedColl, {_id: 1}, {_id: 10}, {_id: 10 + 1});

/**
 * Mock a failed planShardedSearch command.
 * @param {MongotMock} mock
 * @param {String} collName
 * @param {Object} mongotQuery - search query for which we will mock a failed response.
 */
function mockPlanShardedSearchFailed(mock, collName, mongotQuery) {
    mock.setMockResponses([{
                              expectedCommand: {
                                  planShardedSearch: collName,
                                  query: mongotQuery,
                                  $db: dbName,
                                  searchFeatures: {},
                              },
                              response: {ok: 0},  // Failed status
                          }],
                          1423);
}

(function testShardedCollSort() {
    const mongotQuery = {
        sort: {x: 1},
    };
    // Mock PSS to fail.
    mockPlanShardedSearchFailed(mongostmock, shardedCollName, mongotQuery);
    assert.commandFailed(testDB.runCommand(
        {aggregate: shardedCollName, pipeline: [{$search: mongotQuery}], cursor: {}}));
})();

(function testViewWithShardedSearchSort() {
    const mongotQuery = {
        sort: {x: 1},
    };
    const viewName = shardedCollName + "_view";
    assert.commandWorked(testDB.createView(viewName, shardedCollName, [{$search: mongotQuery}]));

    // Ensure that planShardedSearch is invoked twice, once per query against the view.
    mockPlanShardedSearchFailed(mongostmock, shardedCollName, mongotQuery);
    assert.commandFailed(testDB.runCommand({find: viewName}));
    mockPlanShardedSearchFailed(mongostmock, shardedCollName, mongotQuery);
    assert.commandFailed(testDB.runCommand({find: viewName}));
})();

const unshardedColl = testDB.getCollection(unshardedCollName);

unshardedColl.drop();
assert.commandWorked(unshardedColl.insert([
    {_id: 1, a: 5},
    {_id: 2, a: 10},
    {_id: 3, a: 0},
]));

let shard0Conn = st.rs0.getPrimary();

(function testUnshardedCollSortInShardedCluster() {
    const mongotQuery = {
        sort: {x: 1},
    };

    // Mongos will forward the the query to the primary shard which will invoke PSS because the
    // cluster is in a sharded environment. Here, we mock PSS to fail.
    const s0Mongot = stWithMock.getMockConnectedToHost(shard0Conn);
    mockPlanShardedSearchFailed(s0Mongot, unshardedCollName, mongotQuery);

    assert.commandFailed(testDB.runCommand(
        {aggregate: unshardedCollName, pipeline: [{$search: mongotQuery}], cursor: {}}));
})();

stWithMock.stop();
})();