/**
 * Test that a $search in a view is desugared on each query against the view.
 */
(function() {
"use strict";

load("jstests/libs/uuid_util.js");  // For getUUIDFromListCollections.
load("src/mongo/db/modules/enterprise/jstests/search/lib/mongotmock.js");
load("src/mongo/db/modules/enterprise/jstests/search/lib/shardingtest_with_mongotmock.js");

const dbName = "test";
const collName = jsTestName();

const stWithMock = new ShardingTestWithMongotMock({
    name: "sharded_search",
    shards: {
        rs0: {nodes: 1},
        rs1: {nodes: 1},
    },
    mongos: 1,
    other: {
        rsOptions: {setParameter: {enableTestCommands: 1}},
    }
});
stWithMock.start();
const st = stWithMock.st;

const testDB = st.s.getDB(dbName);
const coll = testDB.getCollection(collName);
coll.drop();

assert.commandWorked(coll.insert([
    {_id: 1, a: 10},
    {_id: 2, a: 0},
    {_id: 3, a: 5},
]));

const viewName = collName + "_search_view";

// Enable sharding and make shard0 primary shard, but do not shard the collection.
assert.commandWorked(st.s.getDB("admin").runCommand({enableSharding: dbName}));
st.ensurePrimaryShard(dbName, st.shard0.name);

const mongotQuery = {
    text: "foo"
};
assert.commandWorked(testDB.createView(viewName, collName, [{$search: mongotQuery}]));
const searchView = testDB.getCollection(viewName);

const mockMongots = function() {
    // Mock PSS on primary shard
    mockPlanShardSearchOk(
        stWithMock.getMockConnectedToHost(st.shard0), dbName, collName, mongotQuery);
    // Mock responses on primary shard
    const responseBatch = [
        {_id: 1, $searchScore: 1.0},
        {_id: 2, $searchScore: 0.8},
        {_id: 3, $searchScore: 0.6},
    ];
    const collUUID = getUUIDFromListCollections(testDB, collName);
    stWithMock.getMockConnectedToHost(st.shard0).setMockResponses(
        [{
            expectedCommand: mongotCommandForQuery(mongotQuery, collName, dbName, collUUID),
            response: mongotResponseForBatch(responseBatch, NumberLong(0), coll.getFullName(), 1),
        }],
        123);
};

// Ensure that $search is desugared twice for two queries against the view. Each desugaring phase
// will cause the primary shard to invoke planShardedSearch.
mockMongots();
assert.eq([{_id: 1, a: 10}], searchView.find().limit(1).toArray());
mockMongots();
assert.eq([{_id: 1, a: 10}], searchView.find().limit(1).toArray());

// Helper function which gets the pipeline associated with a view defintion.
function getPipelineFromViewDef(viewName) {
    return testDB.system.views.findOne({_id: viewName}).pipeline;
}

// Check explicitly that the $search pipeline was not desugared in the view.
assert.eq([{$search: mongotQuery}], getPipelineFromViewDef(searchView.getFullName()));

// Ensure that $searchMeta does not desugar in view definition parsing.
{
    const searchMetaViewName = collName + "_search_meta_view";
    assert.commandWorked(
        testDB.createView(searchMetaViewName, collName, [{$searchMeta: mongotQuery}]));
    const searchMetaView = testDB.getCollection(searchMetaViewName);
    assert.eq([{$searchMeta: mongotQuery}], getPipelineFromViewDef(searchMetaView.getFullName()));
}

stWithMock.stop();
})();