/**
 * Test that a mixed version cluster with a search query without $$SEARCH_META succeeds.
 * @tags: [multiversion_sanity_check]
 */

(function() {
"use strict";

load('jstests/libs/uuid_util.js');  // For getUUIDFromListCollections.
load("src/mongo/db/modules/enterprise/jstests/search/lib/mongotmock.js");

// Set up mongotmock and point the mongod to it.
const mongotmock = new MongotMock();
mongotmock.start();
const mongotConn = mongotmock.getConnection();

const conn = MongoRunner.runMongod({setParameter: {mongotHost: mongotConn.host}});
const db = conn.getDB("test");
const coll = db.search;
coll.drop();
assert.commandWorked(coll.insert({"_id": 1, "title": "cakes"}));

const collUUID = getUUIDFromListCollections(db, coll.getName());
const searchQuery = {
    query: "cakes",
    path: "title"
};
const searchCmd = {
    search: coll.getName(),
    collectionUUID: collUUID,
    query: searchQuery,
    $db: "test"
};

function resetHistory(cursorNum) {
    {
        const cursorId = NumberLong(cursorNum);
        const history = [
            {
                expectedCommand: searchCmd,
                response: {
                    cursor: {
                        id: NumberLong(0),
                        ns: coll.getFullName(),
                        nextBatch: [{_id: 1, score: 0.321}]
                    },
                    ok: 1
                }
            },
        ];

        assert.commandWorked(
            mongotConn.adminCommand({setMockResponses: 1, cursorId: cursorId, history: history}));
    }
}

function assertResults(resultCursor) {
    assert.eq(resultCursor.cursor.firstBatch, [{"_id": 1, "title": "cakes"}]);
}

// Downgrade to 4.2
assert.commandWorked(db.adminCommand({setFeatureCompatibilityVersion: "4.2"}));

// Give mongotmock some stuff to return.
resetHistory(123);

// Search query should succeed.
let result = assert.commandWorked(
    db.runCommand({aggregate: coll.getName(), pipeline: [{$search: searchQuery}], cursor: {}}));
assertResults(result);

// Upgrade to 4.4.
assert.commandWorked(db.adminCommand({setFeatureCompatibilityVersion: "4.4"}));

// Query still works
resetHistory(125);
result = assert.commandWorked(
    db.runCommand({aggregate: coll.getName(), pipeline: [{$search: searchQuery}], cursor: {}}));
assertResults(result);

MongoRunner.stopMongod(conn);
mongotmock.stop();
})();
