// Copyright (c) 2015-present Mattermost, Inc. All Rights Reserved.
// See LICENSE.txt for license information.

package app

import (
	"fmt"
	"sort"
	"sync/atomic"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"

	"github.com/mattermost/mattermost/server/public/model"
	"github.com/mattermost/mattermost/server/v8/channels/store/storetest/mocks"
)

/* TODO: Temporarily comment out until MM-11108
func TestAppRace(t *testing.T) {
	for i := 0; i < 10; i++ {
		a, err := New()
		require.NoError(t, err)
		a.UpdateConfig(func(cfg *model.Config) { *cfg.ServiceSettings.ListenAddress = "localhost:0" })
		serverErr := a.StartServer()
		require.NoError(t, serverErr)
		a.Srv().Shutdown()
	}
}
*/

var allPermissionIDs []string

func init() {
	for _, perm := range model.AllPermissions {
		allPermissionIDs = append(allPermissionIDs, perm.Id)
	}
}

func TestUnitUpdateConfig(t *testing.T) {
	mainHelper.Parallel(t)
	th := SetupWithStoreMock(t)

	mockStore := th.App.Srv().Store().(*mocks.Store)
	mockUserStore := mocks.UserStore{}
	mockUserStore.On("Count", mock.Anything).Return(int64(10), nil)
	mockPostStore := mocks.PostStore{}
	mockPostStore.On("GetMaxPostSize").Return(65535, nil)
	mockSystemStore := mocks.SystemStore{}
	mockSystemStore.On("GetByName", "UpgradedFromTE").Return(&model.System{Name: "UpgradedFromTE", Value: "false"}, nil)
	mockSystemStore.On("GetByName", "InstallationDate").Return(&model.System{Name: "InstallationDate", Value: "10"}, nil)
	mockSystemStore.On("GetByName", "FirstServerRunTimestamp").Return(&model.System{Name: "FirstServerRunTimestamp", Value: "10"}, nil)
	mockLicenseStore := mocks.LicenseStore{}
	mockLicenseStore.On("Get", "").Return(&model.LicenseRecord{}, nil)
	mockStore.On("User").Return(&mockUserStore)
	mockStore.On("Post").Return(&mockPostStore)
	mockStore.On("System").Return(&mockSystemStore)
	mockStore.On("License").Return(&mockLicenseStore)
	mockStore.On("GetDBSchemaVersion").Return(1, nil)

	prev := *th.App.Config().ServiceSettings.SiteURL

	require.False(t, th.App.IsConfigReadOnly())

	var called int32
	th.App.AddConfigListener(func(old, current *model.Config) {
		atomic.AddInt32(&called, 1)
		assert.Equal(t, prev, *old.ServiceSettings.SiteURL)
		assert.Equal(t, "http://foo.com", *current.ServiceSettings.SiteURL)
	})

	th.App.UpdateConfig(func(cfg *model.Config) {
		*cfg.ServiceSettings.SiteURL = "http://foo.com"
	})

	// callback should be called once
	assert.Equal(t, int32(1), atomic.LoadInt32(&called))
}

func TestDoAdvancedPermissionsMigration(t *testing.T) {
	th := Setup(t)

	th.ResetRoleMigration(t)

	err := th.App.DoAdvancedPermissionsMigration()
	require.NoError(t, err)

	roleNames := []string{
		"system_user",
		"system_admin",
		"team_user",
		"team_admin",
		"channel_user",
		"channel_admin",
		"system_post_all",
		"system_post_all_public",
		"system_user_access_token",
		"team_post_all",
		"team_post_all_public",
		"playbook_admin",
		"playbook_member",
		"run_admin",
		"run_member",
	}

	roles1, err1 := th.App.GetRolesByNames(roleNames)
	assert.Nil(t, err1)
	assert.Equal(t, len(roles1), len(roleNames))

	expected1 := map[string][]string{
		"channel_user": {
			model.PermissionReadChannel.Id,
			model.PermissionReadChannelContent.Id,
			model.PermissionAddReaction.Id,
			model.PermissionRemoveReaction.Id,
			model.PermissionManagePublicChannelMembers.Id,
			model.PermissionUploadFile.Id,
			model.PermissionGetPublicLink.Id,
			model.PermissionCreatePost.Id,
			model.PermissionUseChannelMentions.Id,
			model.PermissionManagePublicChannelProperties.Id,
			model.PermissionDeletePublicChannel.Id,
			model.PermissionManagePrivateChannelProperties.Id,
			model.PermissionDeletePrivateChannel.Id,
			model.PermissionManagePrivateChannelMembers.Id,
			model.PermissionDeletePost.Id,
			model.PermissionEditPost.Id,
			model.PermissionAddBookmarkPublicChannel.Id,
			model.PermissionEditBookmarkPublicChannel.Id,
			model.PermissionDeleteBookmarkPublicChannel.Id,
			model.PermissionOrderBookmarkPublicChannel.Id,
			model.PermissionAddBookmarkPrivateChannel.Id,
			model.PermissionEditBookmarkPrivateChannel.Id,
			model.PermissionDeleteBookmarkPrivateChannel.Id,
			model.PermissionOrderBookmarkPrivateChannel.Id,
		},
		"channel_admin": {
			model.PermissionManageChannelRoles.Id,
			model.PermissionUseGroupMentions.Id,
			model.PermissionAddBookmarkPublicChannel.Id,
			model.PermissionEditBookmarkPublicChannel.Id,
			model.PermissionDeleteBookmarkPublicChannel.Id,
			model.PermissionOrderBookmarkPublicChannel.Id,
			model.PermissionAddBookmarkPrivateChannel.Id,
			model.PermissionEditBookmarkPrivateChannel.Id,
			model.PermissionDeleteBookmarkPrivateChannel.Id,
			model.PermissionOrderBookmarkPrivateChannel.Id,
			model.PermissionManagePublicChannelBanner.Id,
			model.PermissionManagePrivateChannelBanner.Id,
			model.PermissionManageChannelAccessRules.Id,
		},
		"team_user": {
			model.PermissionListTeamChannels.Id,
			model.PermissionJoinPublicChannels.Id,
			model.PermissionReadPublicChannel.Id,
			model.PermissionViewTeam.Id,
			model.PermissionCreatePublicChannel.Id,
			model.PermissionCreatePrivateChannel.Id,
			model.PermissionInviteUser.Id,
			model.PermissionAddUserToTeam.Id,
		},
		"team_post_all": {
			model.PermissionCreatePost.Id,
			model.PermissionUseChannelMentions.Id,
		},
		"team_post_all_public": {
			model.PermissionCreatePostPublic.Id,
			model.PermissionUseChannelMentions.Id,
		},
		"team_admin": {
			model.PermissionRemoveUserFromTeam.Id,
			model.PermissionManageTeam.Id,
			model.PermissionImportTeam.Id,
			model.PermissionManageTeamRoles.Id,
			model.PermissionManageChannelRoles.Id,
			model.PermissionManageOwnIncomingWebhooks.Id,
			model.PermissionManageOthersIncomingWebhooks.Id,
			model.PermissionManageOwnOutgoingWebhooks.Id,
			model.PermissionManageOthersOutgoingWebhooks.Id,
			model.PermissionManageOwnSlashCommands.Id,
			model.PermissionManageOthersSlashCommands.Id,
			model.PermissionBypassIncomingWebhookChannelLock.Id,
			model.PermissionConvertPublicChannelToPrivate.Id,
			model.PermissionConvertPrivateChannelToPublic.Id,
			model.PermissionDeletePost.Id,
			model.PermissionDeleteOthersPosts.Id,
			model.PermissionAddBookmarkPublicChannel.Id,
			model.PermissionEditBookmarkPublicChannel.Id,
			model.PermissionDeleteBookmarkPublicChannel.Id,
			model.PermissionOrderBookmarkPublicChannel.Id,
			model.PermissionAddBookmarkPrivateChannel.Id,
			model.PermissionEditBookmarkPrivateChannel.Id,
			model.PermissionDeleteBookmarkPrivateChannel.Id,
			model.PermissionOrderBookmarkPrivateChannel.Id,
			model.PermissionManagePublicChannelBanner.Id,
			model.PermissionManagePrivateChannelBanner.Id,
			model.PermissionManageChannelAccessRules.Id,
		},
		"system_user": {
			model.PermissionListPublicTeams.Id,
			model.PermissionJoinPublicTeams.Id,
			model.PermissionCreateDirectChannel.Id,
			model.PermissionCreateGroupChannel.Id,
			model.PermissionViewMembers.Id,
			model.PermissionCreateTeam.Id,
			model.PermissionCreateCustomGroup.Id,
			model.PermissionEditCustomGroup.Id,
			model.PermissionDeleteCustomGroup.Id,
			model.PermissionRestoreCustomGroup.Id,
			model.PermissionManageCustomGroupMembers.Id,
		},
		"system_post_all": {
			model.PermissionCreatePost.Id,
			model.PermissionUseChannelMentions.Id,
		},
		"system_post_all_public": {
			model.PermissionCreatePostPublic.Id,
			model.PermissionUseChannelMentions.Id,
		},
		"system_user_access_token": {
			model.PermissionCreateUserAccessToken.Id,
			model.PermissionReadUserAccessToken.Id,
			model.PermissionRevokeUserAccessToken.Id,
		},
		"system_admin": allPermissionIDs,
	}
	assert.Contains(t, allPermissionIDs, model.PermissionManageSharedChannels.Id, "manage_shared_channels permission not found")
	assert.Contains(t, allPermissionIDs, model.PermissionManageSecureConnections.Id, "manage_secure_connections permission not found")

	// Check the migration matches what's expected.
	for name, permissions := range expected1 {
		role, err := th.App.GetRoleByName(th.Context, name)
		assert.Nil(t, err)
		assert.Equal(t, role.Permissions, permissions, fmt.Sprintf("role %q didn't match", name))
	}

	th.App.Srv().SetLicense(model.NewTestLicense())

	// Check the migration doesn't change anything if run again.
	err = th.App.DoAdvancedPermissionsMigration()
	require.NoError(t, err)

	roles2, err2 := th.App.GetRolesByNames(roleNames)
	assert.Nil(t, err2)
	assert.Equal(t, len(roles2), len(roleNames))

	for name, permissions := range expected1 {
		role, err := th.App.GetRoleByName(th.Context, name)
		assert.Nil(t, err)
		assert.Equal(t, permissions, role.Permissions)
	}
}

func TestDoEmojisPermissionsMigration(t *testing.T) {
	th := SetupWithoutPreloadMigrations(t)

	expectedSystemAdmin := allPermissionIDs
	sort.Strings(expectedSystemAdmin)

	th.ResetEmojisMigration(t)
	err := th.App.DoEmojisPermissionsMigration()
	require.NoError(t, err)

	role3, err3 := th.App.GetRoleByName(th.Context, model.SystemUserRoleId)
	assert.Nil(t, err3)
	expected3 := []string{
		model.PermissionCreateCustomGroup.Id,
		model.PermissionEditCustomGroup.Id,
		model.PermissionDeleteCustomGroup.Id,
		model.PermissionManageCustomGroupMembers.Id,
		model.PermissionRestoreCustomGroup.Id,
		model.PermissionListPublicTeams.Id,
		model.PermissionJoinPublicTeams.Id,
		model.PermissionCreateDirectChannel.Id,
		model.PermissionCreateGroupChannel.Id,
		model.PermissionCreateTeam.Id,
		model.PermissionCreateEmojis.Id,
		model.PermissionDeleteEmojis.Id,
		model.PermissionViewMembers.Id,
	}
	sort.Strings(expected3)
	sort.Strings(role3.Permissions)
	assert.Equal(t, expected3, role3.Permissions, fmt.Sprintf("'%v' did not have expected permissions", model.SystemUserRoleId))

	systemAdmin2, systemAdminErr2 := th.App.GetRoleByName(th.Context, model.SystemAdminRoleId)
	assert.Nil(t, systemAdminErr2)
	sort.Strings(systemAdmin2.Permissions)
	assert.Equal(t, expectedSystemAdmin, systemAdmin2.Permissions, fmt.Sprintf("'%v' did not have expected permissions", model.SystemAdminRoleId))
}

func TestDBHealthCheckWriteAndDelete(t *testing.T) {
	mainHelper.Parallel(t)
	th := Setup(t)

	expectedKey := "health_check_" + th.App.GetClusterId()
	assert.Equal(t, expectedKey, th.App.dbHealthCheckKey())

	_, err := th.App.Srv().Store().System().GetByName(expectedKey)
	assert.Error(t, err)

	err = th.App.DBHealthCheckWrite()
	assert.NoError(t, err)

	systemVal, err := th.App.Srv().Store().System().GetByName(expectedKey)
	assert.NoError(t, err)
	assert.NotNil(t, systemVal)

	err = th.App.DBHealthCheckDelete()
	assert.NoError(t, err)

	_, err = th.App.Srv().Store().System().GetByName(expectedKey)
	assert.Error(t, err)
}
