// Licensed to Elasticsearch B.V. under one or more contributor
// license agreements. See the NOTICE file distributed with
// this work for additional information regarding copyright
// ownership. Elasticsearch B.V. licenses this file to you under
// the Apache License, Version 2.0 (the "License"); you may
// not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

// Code generated from the elasticsearch-specification DO NOT EDIT.
// https://github.com/elastic/elasticsearch-specification/tree/470b4b9aaaa25cae633ec690e54b725c6fc939c7

// Get global checkpoints.
//
// Get the current global checkpoints for an index.
// This API is designed for internal use by the Fleet server project.
package globalcheckpoints

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"github.com/elastic/elastic-transport-go/v8/elastictransport"
	"github.com/elastic/go-elasticsearch/v8/typedapi/types"
)

const (
	indexMask = iota + 1
)

// ErrBuildPath is returned in case of missing parameters within the build of the request.
var ErrBuildPath = errors.New("cannot build path, check for missing path parameters")

type GlobalCheckpoints struct {
	transport elastictransport.Interface

	headers http.Header
	values  url.Values
	path    url.URL

	raw io.Reader

	paramSet int

	index string

	spanStarted bool

	instrument elastictransport.Instrumentation
}

// NewGlobalCheckpoints type alias for index.
type NewGlobalCheckpoints func(index string) *GlobalCheckpoints

// NewGlobalCheckpointsFunc returns a new instance of GlobalCheckpoints with the provided transport.
// Used in the index of the library this allows to retrieve every apis in once place.
func NewGlobalCheckpointsFunc(tp elastictransport.Interface) NewGlobalCheckpoints {
	return func(index string) *GlobalCheckpoints {
		n := New(tp)

		n._index(index)

		return n
	}
}

// Get global checkpoints.
//
// Get the current global checkpoints for an index.
// This API is designed for internal use by the Fleet server project.
//
// https://www.elastic.co/guide/en/elasticsearch/reference/current/get-global-checkpoints.html
func New(tp elastictransport.Interface) *GlobalCheckpoints {
	r := &GlobalCheckpoints{
		transport: tp,
		values:    make(url.Values),
		headers:   make(http.Header),
	}

	if instrumented, ok := r.transport.(elastictransport.Instrumented); ok {
		if instrument := instrumented.InstrumentationEnabled(); instrument != nil {
			r.instrument = instrument
		}
	}

	return r
}

// HttpRequest returns the http.Request object built from the
// given parameters.
func (r *GlobalCheckpoints) HttpRequest(ctx context.Context) (*http.Request, error) {
	var path strings.Builder
	var method string
	var req *http.Request

	var err error

	r.path.Scheme = "http"

	switch {
	case r.paramSet == indexMask:
		path.WriteString("/")

		if instrument, ok := r.instrument.(elastictransport.Instrumentation); ok {
			instrument.RecordPathPart(ctx, "index", r.index)
		}
		path.WriteString(r.index)
		path.WriteString("/")
		path.WriteString("_fleet")
		path.WriteString("/")
		path.WriteString("global_checkpoints")

		method = http.MethodGet
	}

	r.path.Path = path.String()
	r.path.RawQuery = r.values.Encode()

	if r.path.Path == "" {
		return nil, ErrBuildPath
	}

	if ctx != nil {
		req, err = http.NewRequestWithContext(ctx, method, r.path.String(), r.raw)
	} else {
		req, err = http.NewRequest(method, r.path.String(), r.raw)
	}

	req.Header = r.headers.Clone()

	if req.Header.Get("Content-Type") == "" {
		if r.raw != nil {
			req.Header.Set("Content-Type", "application/vnd.elasticsearch+json;compatible-with=8")
		}
	}

	if req.Header.Get("Accept") == "" {
		req.Header.Set("Accept", "application/vnd.elasticsearch+json;compatible-with=8")
	}

	if err != nil {
		return req, fmt.Errorf("could not build http.Request: %w", err)
	}

	return req, nil
}

// Perform runs the http.Request through the provided transport and returns an http.Response.
func (r GlobalCheckpoints) Perform(providedCtx context.Context) (*http.Response, error) {
	var ctx context.Context
	if instrument, ok := r.instrument.(elastictransport.Instrumentation); ok {
		if r.spanStarted == false {
			ctx := instrument.Start(providedCtx, "fleet.global_checkpoints")
			defer instrument.Close(ctx)
		}
	}
	if ctx == nil {
		ctx = providedCtx
	}

	req, err := r.HttpRequest(ctx)
	if err != nil {
		if instrument, ok := r.instrument.(elastictransport.Instrumentation); ok {
			instrument.RecordError(ctx, err)
		}
		return nil, err
	}

	if instrument, ok := r.instrument.(elastictransport.Instrumentation); ok {
		instrument.BeforeRequest(req, "fleet.global_checkpoints")
		if reader := instrument.RecordRequestBody(ctx, "fleet.global_checkpoints", r.raw); reader != nil {
			req.Body = reader
		}
	}
	res, err := r.transport.Perform(req)
	if instrument, ok := r.instrument.(elastictransport.Instrumentation); ok {
		instrument.AfterRequest(req, "elasticsearch", "fleet.global_checkpoints")
	}
	if err != nil {
		localErr := fmt.Errorf("an error happened during the GlobalCheckpoints query execution: %w", err)
		if instrument, ok := r.instrument.(elastictransport.Instrumentation); ok {
			instrument.RecordError(ctx, localErr)
		}
		return nil, localErr
	}

	return res, nil
}

// Do runs the request through the transport, handle the response and returns a globalcheckpoints.Response
func (r GlobalCheckpoints) Do(providedCtx context.Context) (*Response, error) {
	var ctx context.Context
	r.spanStarted = true
	if instrument, ok := r.instrument.(elastictransport.Instrumentation); ok {
		ctx = instrument.Start(providedCtx, "fleet.global_checkpoints")
		defer instrument.Close(ctx)
	}
	if ctx == nil {
		ctx = providedCtx
	}

	response := NewResponse()

	res, err := r.Perform(ctx)
	if err != nil {
		if instrument, ok := r.instrument.(elastictransport.Instrumentation); ok {
			instrument.RecordError(ctx, err)
		}
		return nil, err
	}
	defer res.Body.Close()

	if res.StatusCode < 299 {
		err = json.NewDecoder(res.Body).Decode(response)
		if err != nil {
			if instrument, ok := r.instrument.(elastictransport.Instrumentation); ok {
				instrument.RecordError(ctx, err)
			}
			return nil, err
		}

		return response, nil
	}

	errorResponse := types.NewElasticsearchError()
	err = json.NewDecoder(res.Body).Decode(errorResponse)
	if err != nil {
		if instrument, ok := r.instrument.(elastictransport.Instrumentation); ok {
			instrument.RecordError(ctx, err)
		}
		return nil, err
	}

	if errorResponse.Status == 0 {
		errorResponse.Status = res.StatusCode
	}

	if instrument, ok := r.instrument.(elastictransport.Instrumentation); ok {
		instrument.RecordError(ctx, errorResponse)
	}
	return nil, errorResponse
}

// IsSuccess allows to run a query with a context and retrieve the result as a boolean.
// This only exists for endpoints without a request payload and allows for quick control flow.
func (r GlobalCheckpoints) IsSuccess(providedCtx context.Context) (bool, error) {
	var ctx context.Context
	r.spanStarted = true
	if instrument, ok := r.instrument.(elastictransport.Instrumentation); ok {
		ctx = instrument.Start(providedCtx, "fleet.global_checkpoints")
		defer instrument.Close(ctx)
	}
	if ctx == nil {
		ctx = providedCtx
	}

	res, err := r.Perform(ctx)

	if err != nil {
		return false, err
	}
	io.Copy(io.Discard, res.Body)
	err = res.Body.Close()
	if err != nil {
		return false, err
	}

	if res.StatusCode >= 200 && res.StatusCode < 300 {
		return true, nil
	}

	if res.StatusCode != 404 {
		err := fmt.Errorf("an error happened during the GlobalCheckpoints query execution, status code: %d", res.StatusCode)
		if instrument, ok := r.instrument.(elastictransport.Instrumentation); ok {
			instrument.RecordError(ctx, err)
		}
		return false, err
	}

	return false, nil
}

// Header set a key, value pair in the GlobalCheckpoints headers map.
func (r *GlobalCheckpoints) Header(key, value string) *GlobalCheckpoints {
	r.headers.Set(key, value)

	return r
}

// Index A single index or index alias that resolves to a single index.
// API Name: index
func (r *GlobalCheckpoints) _index(index string) *GlobalCheckpoints {
	r.paramSet |= indexMask
	r.index = index

	return r
}

// WaitForAdvance A boolean value which controls whether to wait (until the timeout) for the
// global checkpoints
// to advance past the provided `checkpoints`.
// API name: wait_for_advance
func (r *GlobalCheckpoints) WaitForAdvance(waitforadvance bool) *GlobalCheckpoints {
	r.values.Set("wait_for_advance", strconv.FormatBool(waitforadvance))

	return r
}

// WaitForIndex A boolean value which controls whether to wait (until the timeout) for the
// target index to exist
// and all primary shards be active. Can only be true when `wait_for_advance` is
// true.
// API name: wait_for_index
func (r *GlobalCheckpoints) WaitForIndex(waitforindex bool) *GlobalCheckpoints {
	r.values.Set("wait_for_index", strconv.FormatBool(waitforindex))

	return r
}

// Checkpoints A comma separated list of previous global checkpoints. When used in
// combination with `wait_for_advance`,
// the API will only return once the global checkpoints advances past the
// checkpoints. Providing an empty list
// will cause Elasticsearch to immediately return the current global
// checkpoints.
// API name: checkpoints
func (r *GlobalCheckpoints) Checkpoints(checkpoints ...int64) *GlobalCheckpoints {
	tmp := []string{}
	for _, item := range checkpoints {
		tmp = append(tmp, fmt.Sprintf("%v", item))
	}
	r.values.Set("checkpoints", strings.Join(tmp, ","))

	return r
}

// Timeout Period to wait for a global checkpoints to advance past `checkpoints`.
// API name: timeout
func (r *GlobalCheckpoints) Timeout(duration string) *GlobalCheckpoints {
	r.values.Set("timeout", duration)

	return r
}

// ErrorTrace When set to `true` Elasticsearch will include the full stack trace of errors
// when they occur.
// API name: error_trace
func (r *GlobalCheckpoints) ErrorTrace(errortrace bool) *GlobalCheckpoints {
	r.values.Set("error_trace", strconv.FormatBool(errortrace))

	return r
}

// FilterPath Comma-separated list of filters in dot notation which reduce the response
// returned by Elasticsearch.
// API name: filter_path
func (r *GlobalCheckpoints) FilterPath(filterpaths ...string) *GlobalCheckpoints {
	tmp := []string{}
	for _, item := range filterpaths {
		tmp = append(tmp, fmt.Sprintf("%v", item))
	}
	r.values.Set("filter_path", strings.Join(tmp, ","))

	return r
}

// Human When set to `true` will return statistics in a format suitable for humans.
// For example `"exists_time": "1h"` for humans and
// `"eixsts_time_in_millis": 3600000` for computers. When disabled the human
// readable values will be omitted. This makes sense for responses being
// consumed
// only by machines.
// API name: human
func (r *GlobalCheckpoints) Human(human bool) *GlobalCheckpoints {
	r.values.Set("human", strconv.FormatBool(human))

	return r
}

// Pretty If set to `true` the returned JSON will be "pretty-formatted". Only use
// this option for debugging only.
// API name: pretty
func (r *GlobalCheckpoints) Pretty(pretty bool) *GlobalCheckpoints {
	r.values.Set("pretty", strconv.FormatBool(pretty))

	return r
}
