// Copyright (c) 2015-present Mattermost, Inc. All Rights Reserved.
// See LICENSE.enterprise for license information.

package global_relay_export

import (
	"archive/zip"
	"bytes"
	"encoding/base64"
	"fmt"
	"io"
	"os"
	"path"
	"strings"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"

	"github.com/mattermost/mattermost/server/public/model"
	"github.com/mattermost/mattermost/server/public/plugin/plugintest/mock"
	"github.com/mattermost/mattermost/server/public/shared/request"
	"github.com/mattermost/mattermost/server/v8/channels/store/storetest"
	"github.com/mattermost/mattermost/server/v8/channels/utils/fileutils"
	"github.com/mattermost/mattermost/server/v8/enterprise/message_export/shared"
	"github.com/mattermost/mattermost/server/v8/platform/shared/filestore"
	"github.com/mattermost/mattermost/server/v8/platform/shared/templates"
)

func TestGlobalRelayExport(t *testing.T) {
	templatesDir, ok := fileutils.FindDir("templates")
	require.True(t, ok)

	templatesContainer, err := templates.New(templatesDir)
	require.NotNil(t, templatesContainer)
	require.NoError(t, err)

	tempDir, err := os.MkdirTemp("", "")
	require.NoError(t, err)
	t.Cleanup(func() {
		err = os.RemoveAll(tempDir)
		assert.NoError(t, err)
	})

	rctx := request.TestContext(t)

	config := filestore.FileBackendSettings{
		DriverName: model.ImageDriverLocal,
		Directory:  tempDir,
	}

	fileBackend, err := filestore.NewFileBackend(config)
	assert.NoError(t, err)
	exportBackend := fileBackend

	chanTypeDirect := model.ChannelTypeDirect
	grExportTests := []struct {
		name                      string
		cmhs                      map[string][]*model.ChannelMemberHistoryResult
		metadata                  map[string]*shared.MetadataChannel
		startTime                 int64
		endTime                   int64
		posts                     []*model.MessageExport
		attachments               map[string][]*model.FileInfo
		attachmentsContent        map[string]string
		expectedAttachmentContent [][]string
		maxEmailBytes             int64
		numExpectedEmls           int
		expectedHeaders           [][]string
		expectedTexts             [][]string
		notExpectedTexts          [][]string
		expectedHTMLs             [][]string
		expectedWarnings          int
		empty                     bool
	}{
		{
			name:             "empty",
			cmhs:             map[string][]*model.ChannelMemberHistoryResult{},
			posts:            []*model.MessageExport{},
			attachments:      map[string][]*model.FileInfo{},
			expectedWarnings: 0,
			empty:            true,
		},
		{
			name: "posts",
			cmhs: map[string][]*model.ChannelMemberHistoryResult{
				"channel-id": {
					{
						JoinTime: 0, UserId: "id-test1", UserEmail: "test1@test.com", Username: "test1", LeaveTime: model.NewPointer(int64(100_000)),
					},
					{
						JoinTime: 8, UserId: "id-test2", UserEmail: "test2@test.com", Username: "test2", LeaveTime: model.NewPointer(int64(100_000)),
					},
					{
						JoinTime: 400, UserId: "id-test3", UserEmail: "test3@test.com", Username: "test3",
					},
				},
			},
			metadata: map[string]*shared.MetadataChannel{
				"channel-id": {
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          "channel-id",
					ChannelName:        "channel-name",
					ChannelDisplayName: "channel-display-name",
					ChannelType:        chanTypeDirect,
					RoomId:             "direct - channel-id",
					StartTime:          1,
					EndTime:            100,
				},
			},
			startTime: 1,
			endTime:   100_000,
			posts: []*model.MessageExport{
				{
					PostId:             model.NewPointer("post-id1"),
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          model.NewPointer("channel-id"),
					ChannelName:        model.NewPointer("channel-name"),
					ChannelDisplayName: model.NewPointer("channel-display-name"),
					PostUpdateAt:       model.NewPointer(int64(1)),
					PostCreateAt:       model.NewPointer(int64(1)),
					PostMessage:        model.NewPointer("message 1"),
					PostProps:          model.NewPointer("{}"),
					PostType:           model.NewPointer(""),
					UserEmail:          model.NewPointer("test1@test.com"),
					UserId:             model.NewPointer("id-test1"),
					Username:           model.NewPointer("test1"),
					ChannelType:        &chanTypeDirect,
					PostFileIds:        []string{},
				},
				{
					PostId:             model.NewPointer("post-id2"),
					PostRootId:         model.NewPointer("post-root-id"),
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          model.NewPointer("channel-id"),
					ChannelName:        model.NewPointer("channel-name"),
					ChannelDisplayName: model.NewPointer("channel-display-name"),
					PostUpdateAt:       model.NewPointer(int64(100_000)),
					PostCreateAt:       model.NewPointer(int64(100_000)),
					PostMessage:        model.NewPointer("message 2"),
					PostProps:          model.NewPointer("{}"),
					PostType:           model.NewPointer(""),
					UserEmail:          model.NewPointer("test1@test.com"),
					UserId:             model.NewPointer("id-test1"),
					Username:           model.NewPointer("test1"),
					ChannelType:        &chanTypeDirect,
					PostFileIds:        []string{},
				},
			},
			attachments: map[string][]*model.FileInfo{},
			expectedHeaders: [][]string{{
				"MIME-Version: 1.0",
				"X-Mattermost-ChannelType: direct",
				"Content-Transfer-Encoding: 8bit",
				"Precedence: bulk",
				"X-GlobalRelay-MsgType: Mattermost",
				"X-Mattermost-ChannelID: channel-id",
				"X-Mattermost-ChannelName: channel-display-name",
				"Auto-Submitted: auto-generated",
				"Date: Thu, 01 Jan 1970 00:01:40 +0000",
				"From: test1@test.com",
				"To: test1@test.com,test2@test.com",
				"Subject: Mattermost Compliance Export: channel-display-name",
			}},

			expectedTexts: [][]string{{
				strings.Join([]string{
					"* TeamId: team-id",
					"* TeamName: team-name",
					"* TeamDisplayName: team-display-name",
					"* ChannelId: channel-id",
					"* ChannelName: channel-name",
					"* ChannelDisplayName: channel-display-name",
					"* Started: 1970-01-01T00:00:00Z",
					"* Ended: 1970-01-01T00:01:40Z",
					"* Duration: 2 minutes",
				}, "\r\n"),
				strings.Join([]string{
					"--------",
					"Messages",
					"--------",
					"",
					"* post-id1 1970-01-01T00:00:00Z @test1 id-test1 @test1 user (test1@test.com=",
					") message 1",
					"* post-id2 1970-01-01T00:01:40Z @test1 id-test1 @test1 user (test1@test.com=",
					") message 2",
				}, "\r\n"),
			}},

			expectedHTMLs: [][]string{{
				strings.Join([]string{
					"    <ul>",
					"        <li><span class=3D\"bold\">TeamId:</span>team-id</li>",
					"        <li><span class=3D\"bold\">TeamName:</span>team-name</li>",
					"        <li><span class=3D\"bold\">TeamDisplayName:</span>team-display-name</=",
					"li>",
					"        <li><span class=3D\"bold\">ChannelId:</span>channel-id</li>",
					"        <li><span class=3D\"bold\">ChannelName:</span>channel-name</li>",
					"        <li><span class=3D\"bold\">ChannelDisplayName:</span>channel-display-=",
					"name</li>",
					"        <li><span class=3D\"bold\">Started:</span>1970-01-01T00:00:00Z</li>",
					"        <li><span class=3D\"bold\">Ended:</span>1970-01-01T00:01:40Z</li>",
					"        <li><span class=3D\"bold\">Duration:</span>2 minutes</li>",
					"    </ul>",
				}, "\r\n"),
				strings.Join([]string{
					"<tr>",
					"    <td class=3D\"userid\">id-test1</td>",
					"    <td class=3D\"username\">@test1</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test1@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">2</td>",
					"</tr>",
					"",
					"<tr>",
					"    <td class=3D\"userid\">id-test2</td>",
					"    <td class=3D\"username\">@test2</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test2@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">0</td>",
					"</tr>",
					"",
					"<tr>",
					"    <td class=3D\"userid\">id-test3</td>",
					"    <td class=3D\"username\">@test3</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test3@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">0</td>",
					"</tr>",
				}, "\r\n"),

				strings.Join([]string{
					"<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id1</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:00:00Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">id-test1</span>",
					"    <span class=3D\"postusername\">@test1</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">message 1</span>",
					"</li>",
					"",
					"<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id2</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:01:40Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">id-test1</span>",
					"    <span class=3D\"postusername\">@test1</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">message 2</span>",
					"</li>",
				}, "\r\n"),
			}},
			expectedWarnings: 0,
			numExpectedEmls:  1,
		},

		{
			name: "posts with attachments, size ok",
			cmhs: map[string][]*model.ChannelMemberHistoryResult{
				"channel-id": {
					{
						JoinTime: 0, UserId: "test1", UserEmail: "test1@test.com", Username: "test", LeaveTime: model.NewPointer(int64(100_000)),
					},
					{
						JoinTime: 8, UserId: "test2", UserEmail: "test2@test.com", Username: "test2", LeaveTime: model.NewPointer(int64(100_000)),
					},
					{
						JoinTime: 400, UserId: "test3", UserEmail: "test3@test.com", Username: "test3",
					},
				},
			},
			metadata: map[string]*shared.MetadataChannel{
				"channel-id": {
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          "channel-id",
					ChannelName:        "channel-name",
					ChannelDisplayName: "channel-display-name",
					ChannelType:        chanTypeDirect,
					RoomId:             "direct - channel-id",
					StartTime:          1,
					EndTime:            100,
				},
			},
			startTime: 1,
			endTime:   100_000,
			posts: []*model.MessageExport{
				{
					PostId:             model.NewPointer("post-id-1"),
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          model.NewPointer("channel-id"),
					ChannelName:        model.NewPointer("channel-name"),
					ChannelDisplayName: model.NewPointer("channel-display-name"),
					PostUpdateAt:       model.NewPointer(int64(1)),
					PostCreateAt:       model.NewPointer(int64(1)),
					PostMessage:        model.NewPointer("message1"),
					PostProps:          model.NewPointer("{}"),
					PostType:           model.NewPointer(""),
					UserEmail:          model.NewPointer("test1@test.com"),
					UserId:             model.NewPointer("test1"),
					Username:           model.NewPointer("test1"),
					ChannelType:        &chanTypeDirect,
					PostFileIds:        []string{"test1"},
				},
				{
					PostId:             model.NewPointer("post-id-2"),
					PostRootId:         model.NewPointer("post-id-1"),
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          model.NewPointer("channel-id"),
					ChannelName:        model.NewPointer("channel-name"),
					ChannelDisplayName: model.NewPointer("channel-display-name"),
					PostUpdateAt:       model.NewPointer(int64(100_000)),
					PostCreateAt:       model.NewPointer(int64(100_000)),
					PostMessage:        model.NewPointer("message2"),
					PostProps:          model.NewPointer("{}"),
					PostType:           model.NewPointer(""),
					UserEmail:          model.NewPointer("test1@test.com"),
					UserId:             model.NewPointer("test1"),
					Username:           model.NewPointer("test1"),
					ChannelType:        &chanTypeDirect,
					PostFileIds:        []string{"test2"},
				},
			},
			attachments: map[string][]*model.FileInfo{
				"post-id-1": {
					{
						Name:     "test1-attachment",
						Id:       "test1-attachment",
						Path:     "test1-attachment",
						CreateAt: 1,
					},
				},
				"post-id-2": {
					{
						Name:     "test2-attachment",
						Id:       "test2-attachment",
						Path:     "test2-attachment",
						CreateAt: 1,
					},
				},
			},
			attachmentsContent: map[string]string{
				"test1-attachment": "this is the attachment content",
				"test2-attachment": "this is the second attachment content",
			},
			expectedAttachmentContent: [][]string{
				{base64.StdEncoding.EncodeToString([]byte("this is the attachment content"))},
				{base64.StdEncoding.EncodeToString([]byte("this is the second attachment content"))},
			},
			expectedHeaders: [][]string{{
				"MIME-Version: 1.0",
				"X-Mattermost-ChannelType: direct",
				"Content-Transfer-Encoding: 8bit",
				"Precedence: bulk",
				"X-GlobalRelay-MsgType: Mattermost",
				"X-Mattermost-ChannelID: channel-id",
				"X-Mattermost-ChannelName: channel-display-name",
				"Auto-Submitted: auto-generated",
				"Date: Thu, 01 Jan 1970 00:01:40 +0000",
				"From: test1@test.com",
				"To: test1@test.com,test2@test.com",
				"Subject: Mattermost Compliance Export: channel-display-name",
			}},

			expectedTexts: [][]string{{
				strings.Join([]string{
					"* TeamId: team-id",
					"* TeamName: team-name",
					"* TeamDisplayName: team-display-name",
					"* ChannelId: channel-id",
					"* ChannelName: channel-name",
					"* ChannelDisplayName: channel-display-name",
					"* Started: 1970-01-01T00:00:00Z",
					"* Ended: 1970-01-01T00:01:40Z",
					"* Duration: 2 minutes",
				}, "\r\n"),
				strings.Join([]string{
					"--------",
					"Messages",
					"--------",
					"",
					"* post-id-1 1970-01-01T00:00:00Z @test1 test1 @test1 user (test1@test.com) =",
					"message1",
					"* post-id-1 1970-01-01T00:00:00Z @test1 test1 @test1 user (test1@test.com) =",
					"Uploaded file test1-attachment",
					"* post-id-2 1970-01-01T00:01:40Z @test1 test1 @test1 user (test1@test.com) =",
					"message2",
				}, "\r\n"),
				strings.Join([]string{
					"Content-Disposition: attachment; filename=\"test1-attachment\"",
					"Content-Transfer-Encoding: base64",
					"Content-Type: application/octet-stream; name=\"test1-attachment\"",
				}, "\r\n"),
			}},

			expectedHTMLs: [][]string{{
				strings.Join([]string{
					"    <ul>",
					"        <li><span class=3D\"bold\">TeamId:</span>team-id</li>",
					"        <li><span class=3D\"bold\">TeamName:</span>team-name</li>",
					"        <li><span class=3D\"bold\">TeamDisplayName:</span>team-display-name</=",
					"li>",
					"        <li><span class=3D\"bold\">ChannelId:</span>channel-id</li>",
					"        <li><span class=3D\"bold\">ChannelName:</span>channel-name</li>",
					"        <li><span class=3D\"bold\">ChannelDisplayName:</span>channel-display-=",
					"name</li>",
					"        <li><span class=3D\"bold\">Started:</span>1970-01-01T00:00:00Z</li>",
					"        <li><span class=3D\"bold\">Ended:</span>1970-01-01T00:01:40Z</li>",
					"        <li><span class=3D\"bold\">Duration:</span>2 minutes</li>",
					"    </ul>",
				}, "\r\n"),
				strings.Join([]string{
					"<tr>",
					"    <td class=3D\"userid\">test1</td>",
					"    <td class=3D\"username\">@test</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test1@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">2</td>",
					"</tr>",
					"",
					"<tr>",
					"    <td class=3D\"userid\">test2</td>",
					"    <td class=3D\"username\">@test2</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test2@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">0</td>",
					"</tr>",
					"",
					"<tr>",
					"    <td class=3D\"userid\">test3</td>",
					"    <td class=3D\"username\">@test3</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test3@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">0</td>",
					"</tr>",
				}, "\r\n"),

				strings.Join([]string{
					"<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id-1</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:00:00Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">test1</span>",
					"    <span class=3D\"postusername\">@test1</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">message1</span>",
					"</li>",
					"",
					"<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id-1</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:00:00Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">test1</span>",
					"    <span class=3D\"postusername\">@test1</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">Uploaded file test1-attachment</span>",
					"</li>",
					"",
					"<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id-2</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:01:40Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">test1</span>",
					"    <span class=3D\"postusername\">@test1</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">message2</span>",
					"</li>",
				}, "\r\n"),
			}},
			expectedWarnings: 0,
			numExpectedEmls:  1,
		},

		{
			name: "posts with attachments, size too large, new channel export needed",
			cmhs: map[string][]*model.ChannelMemberHistoryResult{
				"channel-id": {
					{
						JoinTime: 0, UserId: "test1", UserEmail: "test1@test.com", Username: "test", LeaveTime: model.NewPointer(int64(100_000)),
					},
					{
						JoinTime: 8, UserId: "test2", UserEmail: "test2@test.com", Username: "test2", LeaveTime: model.NewPointer(int64(100_000)),
					},
					{
						JoinTime: 400, UserId: "test3", UserEmail: "test3@test.com", Username: "test3",
					},
				},
			},
			metadata: map[string]*shared.MetadataChannel{
				"channel-id": {
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          "channel-id",
					ChannelName:        "channel-name",
					ChannelDisplayName: "channel-display-name",
					ChannelType:        chanTypeDirect,
					RoomId:             "direct - channel-id",
					StartTime:          1,
					EndTime:            100,
				},
			},
			startTime: 1,
			endTime:   100_000,
			posts: []*model.MessageExport{
				{
					PostId:             model.NewPointer("post-id-1"),
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          model.NewPointer("channel-id"),
					ChannelName:        model.NewPointer("channel-name"),
					ChannelDisplayName: model.NewPointer("channel-display-name"),
					PostUpdateAt:       model.NewPointer(int64(1)),
					PostCreateAt:       model.NewPointer(int64(1)),
					PostMessage:        model.NewPointer("message1"),
					PostProps:          model.NewPointer("{}"),
					PostType:           model.NewPointer(""),
					UserEmail:          model.NewPointer("test1@test.com"),
					UserId:             model.NewPointer("test1"),
					Username:           model.NewPointer("test1"),
					ChannelType:        &chanTypeDirect,
					PostFileIds:        []string{"test1"},
				},
				{
					PostId:             model.NewPointer("post-id-2"),
					PostRootId:         model.NewPointer("post-id-1"),
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          model.NewPointer("channel-id"),
					ChannelName:        model.NewPointer("channel-name"),
					ChannelDisplayName: model.NewPointer("channel-display-name"),
					PostUpdateAt:       model.NewPointer(int64(100_000)),
					PostCreateAt:       model.NewPointer(int64(100_000)),
					PostMessage:        model.NewPointer("message2"),
					PostProps:          model.NewPointer("{}"),
					PostType:           model.NewPointer(""),
					UserEmail:          model.NewPointer("test1@test.com"),
					UserId:             model.NewPointer("test1"),
					Username:           model.NewPointer("test1"),
					ChannelType:        &chanTypeDirect,
					PostFileIds:        []string{"test2"},
				},
			},
			attachments: map[string][]*model.FileInfo{
				"post-id-1": {
					{
						Name: "test1-attachment",
						Id:   "test1-attachment",
						Path: "test1-attachment",
						Size: 90, // 90 + 8 for message = message of 98 bytes. next one will be too big.
					},
				},
				"post-id-2": {
					{
						Name: "test2-attachment",
						Id:   "test2-attachment",
						Path: "test2-attachment",
						Size: 56,
					},
				},
			},
			maxEmailBytes:   100,
			numExpectedEmls: 2,
			attachmentsContent: map[string]string{
				"test1-attachment": "this is the attachment content",
				"test2-attachment": "this is the second attachment content",
			},
			expectedAttachmentContent: [][]string{
				{base64.StdEncoding.EncodeToString([]byte("this is the attachment content"))},
				{base64.StdEncoding.EncodeToString([]byte("this is the second attachment content"))},
			},
			expectedHeaders: [][]string{
				// eml 0
				{
					"MIME-Version: 1.0",
					"X-Mattermost-ChannelType: direct",
					"Content-Transfer-Encoding: 8bit",
					"Precedence: bulk",
					"X-GlobalRelay-MsgType: Mattermost",
					"X-Mattermost-ChannelID: channel-id",
					"X-Mattermost-ChannelName: channel-display-name",
					"Auto-Submitted: auto-generated",
					"Date: Thu, 01 Jan 1970 00:01:40 +0000",
					"From: test1@test.com",
					"To: test1@test.com,test2@test.com",
					"Subject: Mattermost Compliance Export: channel-display-name",
				},
				// eml 1
				{
					"MIME-Version: 1.0",
					"X-Mattermost-ChannelType: direct",
					"Content-Transfer-Encoding: 8bit",
					"Precedence: bulk",
					"X-GlobalRelay-MsgType: Mattermost",
					"X-Mattermost-ChannelID: channel-id",
					"X-Mattermost-ChannelName: channel-display-name",
					"Auto-Submitted: auto-generated",
					"Date: Thu, 01 Jan 1970 00:01:40 +0000",
					"From: test1@test.com",
					"To: test1@test.com,test2@test.com",
					"Subject: Mattermost Compliance Export: channel-display-name",
				},
			},

			expectedTexts: [][]string{
				// eml 0
				{
					strings.Join([]string{
						"* TeamId: team-id",
						"* TeamName: team-name",
						"* TeamDisplayName: team-display-name",
						"* ChannelId: channel-id",
						"* ChannelName: channel-name",
						"* ChannelDisplayName: channel-display-name",
						"* Started: 1970-01-01T00:00:00Z",
						"* Ended: 1970-01-01T00:01:40Z",
						"* Duration: 2 minutes",
					}, "\r\n"),
					strings.Join([]string{
						"--------",
						"Messages",
						"--------",
						"",
						"* post-id-1 1970-01-01T00:00:00Z @test1 test1 @test1 user (test1@test.com) =",
						"message1",
						"* post-id-1 1970-01-01T00:00:00Z @test1 test1 @test1 user (test1@test.com) =",
						"Uploaded file test1-attachment",
					}, "\r\n"),
					strings.Join([]string{
						"Content-Disposition: attachment; filename=\"test1-attachment\"",
						"Content-Transfer-Encoding: base64",
						"Content-Type: application/octet-stream; name=\"test1-attachment\"",
					}, "\r\n"),
				},
				// eml 1
				{
					strings.Join([]string{
						"* TeamId: team-id",
						"* TeamName: team-name",
						"* TeamDisplayName: team-display-name",
						"* ChannelId: channel-id",
						"* ChannelName: channel-name",
						"* ChannelDisplayName: channel-display-name",
						"* Started: 1970-01-01T00:00:00Z",
						"* Ended: 1970-01-01T00:01:40Z",
						"* Duration: 2 minutes",
					}, "\r\n"),
					strings.Join([]string{
						"--------",
						"Messages",
						"--------",
						"",
						"* post-id-2 1970-01-01T00:01:40Z @test1 test1 @test1 user (test1@test.com) =",
						"message2",
						"* post-id-2 1970-01-01T00:01:40Z @test1 test1 @test1 user (test1@test.com) =",
						"Uploaded file test2-attachment",
					}, "\r\n"),
					strings.Join([]string{
						"Content-Disposition: attachment; filename=\"test2-attachment\"",
						"Content-Transfer-Encoding: base64",
						"Content-Type: application/octet-stream; name=\"test2-attachment\"",
					}, "\r\n"),
				},
			},

			expectedHTMLs: [][]string{
				// eml 0
				{
					strings.Join([]string{
						"    <ul>",
						"        <li><span class=3D\"bold\">TeamId:</span>team-id</li>",
						"        <li><span class=3D\"bold\">TeamName:</span>team-name</li>",
						"        <li><span class=3D\"bold\">TeamDisplayName:</span>team-display-name</=",
						"li>",
						"        <li><span class=3D\"bold\">ChannelId:</span>channel-id</li>",
						"        <li><span class=3D\"bold\">ChannelName:</span>channel-name</li>",
						"        <li><span class=3D\"bold\">ChannelDisplayName:</span>channel-display-=",
						"name</li>",
						"        <li><span class=3D\"bold\">Started:</span>1970-01-01T00:00:00Z</li>",
						"        <li><span class=3D\"bold\">Ended:</span>1970-01-01T00:01:40Z</li>",
						"        <li><span class=3D\"bold\">Duration:</span>2 minutes</li>",
						"    </ul>",
					}, "\r\n"),
					strings.Join([]string{
						"<tr>",
						"    <td class=3D\"userid\">test1</td>",
						"    <td class=3D\"username\">@test</td>",
						"    <td class=3D\"usertype\">user</td>",
						"    <td class=3D\"email\">test1@test.com</td>",
						"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
						"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
						"    <td class=3D\"duration\">2 minutes</td>",
						"    <td class=3D\"messages\">1</td>",
						"</tr>",
						"",
						"<tr>",
						"    <td class=3D\"userid\">test2</td>",
						"    <td class=3D\"username\">@test2</td>",
						"    <td class=3D\"usertype\">user</td>",
						"    <td class=3D\"email\">test2@test.com</td>",
						"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
						"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
						"    <td class=3D\"duration\">2 minutes</td>",
						"    <td class=3D\"messages\">0</td>",
						"</tr>",
						"",
						"<tr>",
						"    <td class=3D\"userid\">test3</td>",
						"    <td class=3D\"username\">@test3</td>",
						"    <td class=3D\"usertype\">user</td>",
						"    <td class=3D\"email\">test3@test.com</td>",
						"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
						"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
						"    <td class=3D\"duration\">2 minutes</td>",
						"    <td class=3D\"messages\">0</td>",
						"</tr>",
					}, "\r\n"),
					strings.Join([]string{
						"<li class=3D\"message\">",
						"    <span class=3D\"post_id\">post-id-1</span>",
						"    <span class=3D\"sent_time\">1970-01-01T00:00:00Z</span>",
						"    <span class=3D\"username\">@test1</span>",
						"    <span class=3D\"userid\">test1</span>",
						"    <span class=3D\"postusername\">@test1</span>",
						"    <span class=3D\"usertype\">user</span>",
						"    <span class=3D\"email\">(test1@test.com)</span>",
						"    <span class=3D\"message\">message1</span>",
						"</li>",
						"",
						"<li class=3D\"message\">",
						"    <span class=3D\"post_id\">post-id-1</span>",
						"    <span class=3D\"sent_time\">1970-01-01T00:00:00Z</span>",
						"    <span class=3D\"username\">@test1</span>",
						"    <span class=3D\"userid\">test1</span>",
						"    <span class=3D\"postusername\">@test1</span>",
						"    <span class=3D\"usertype\">user</span>",
						"    <span class=3D\"email\">(test1@test.com)</span>",
						"    <span class=3D\"message\">Uploaded file test1-attachment</span>",
						"</li>",
					}, "\r\n"),
				},
				// eml 1
				{
					strings.Join([]string{
						"    <ul>",
						"        <li><span class=3D\"bold\">TeamId:</span>team-id</li>",
						"        <li><span class=3D\"bold\">TeamName:</span>team-name</li>",
						"        <li><span class=3D\"bold\">TeamDisplayName:</span>team-display-name</=",
						"li>",
						"        <li><span class=3D\"bold\">ChannelId:</span>channel-id</li>",
						"        <li><span class=3D\"bold\">ChannelName:</span>channel-name</li>",
						"        <li><span class=3D\"bold\">ChannelDisplayName:</span>channel-display-=",
						"name</li>",
						"        <li><span class=3D\"bold\">Started:</span>1970-01-01T00:00:00Z</li>",
						"        <li><span class=3D\"bold\">Ended:</span>1970-01-01T00:01:40Z</li>",
						"        <li><span class=3D\"bold\">Duration:</span>2 minutes</li>",
						"    </ul>",
					}, "\r\n"),
					strings.Join([]string{
						"<tr>",
						"    <td class=3D\"userid\">test1</td>",
						"    <td class=3D\"username\">@test</td>",
						"    <td class=3D\"usertype\">user</td>",
						"    <td class=3D\"email\">test1@test.com</td>",
						"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
						"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
						"    <td class=3D\"duration\">2 minutes</td>",
						"    <td class=3D\"messages\">1</td>",
						"</tr>",
						"",
						"<tr>",
						"    <td class=3D\"userid\">test2</td>",
						"    <td class=3D\"username\">@test2</td>",
						"    <td class=3D\"usertype\">user</td>",
						"    <td class=3D\"email\">test2@test.com</td>",
						"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
						"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
						"    <td class=3D\"duration\">2 minutes</td>",
						"    <td class=3D\"messages\">0</td>",
						"</tr>",
						"",
						"<tr>",
						"    <td class=3D\"userid\">test3</td>",
						"    <td class=3D\"username\">@test3</td>",
						"    <td class=3D\"usertype\">user</td>",
						"    <td class=3D\"email\">test3@test.com</td>",
						"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
						"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
						"    <td class=3D\"duration\">2 minutes</td>",
						"    <td class=3D\"messages\">0</td>",
						"</tr>",
					}, "\r\n"),
					strings.Join([]string{
						"<li class=3D\"message\">",
						"    <span class=3D\"post_id\">post-id-2</span>",
						"    <span class=3D\"sent_time\">1970-01-01T00:01:40Z</span>",
						"    <span class=3D\"username\">@test1</span>",
						"    <span class=3D\"userid\">test1</span>",
						"    <span class=3D\"postusername\">@test1</span>",
						"    <span class=3D\"usertype\">user</span>",
						"    <span class=3D\"email\">(test1@test.com)</span>",
						"    <span class=3D\"message\">message2</span>",
						"</li>",
						"",
						"<li class=3D\"message\">",
						"    <span class=3D\"post_id\">post-id-2</span>",
						"    <span class=3D\"sent_time\">1970-01-01T00:01:40Z</span>",
						"    <span class=3D\"username\">@test1</span>",
						"    <span class=3D\"userid\">test1</span>",
						"    <span class=3D\"postusername\">@test1</span>",
						"    <span class=3D\"usertype\">user</span>",
						"    <span class=3D\"email\">(test1@test.com)</span>",
						"    <span class=3D\"message\">Uploaded file test2-attachment</span>",
						"</li>",
					}, "\r\n"),
				},
			},
			expectedWarnings: 0,
		},

		{
			name: "posts with attachments, attachment too large, remove it but only one eml needed",
			cmhs: map[string][]*model.ChannelMemberHistoryResult{
				"channel-id": {
					{
						JoinTime: 0, UserId: "test1", UserEmail: "test1@test.com", Username: "test", LeaveTime: model.NewPointer(int64(100_000)),
					},
					{
						JoinTime: 8, UserId: "test2", UserEmail: "test2@test.com", Username: "test2", LeaveTime: model.NewPointer(int64(100_000)),
					},
					{
						JoinTime: 400, UserId: "test3", UserEmail: "test3@test.com", Username: "test3",
					},
				},
			},
			metadata: map[string]*shared.MetadataChannel{
				"channel-id": {
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          "channel-id",
					ChannelName:        "channel-name",
					ChannelDisplayName: "channel-display-name",
					ChannelType:        chanTypeDirect,
					RoomId:             "direct - channel-id",
					StartTime:          1,
					EndTime:            100,
				},
			},
			startTime: 1,
			endTime:   100_000,
			posts: []*model.MessageExport{
				{
					PostId:             model.NewPointer("post-id-1"),
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          model.NewPointer("channel-id"),
					ChannelName:        model.NewPointer("channel-name"),
					ChannelDisplayName: model.NewPointer("channel-display-name"),
					PostUpdateAt:       model.NewPointer(int64(1)),
					PostCreateAt:       model.NewPointer(int64(1)),
					PostMessage:        model.NewPointer("message1"),
					PostProps:          model.NewPointer("{}"),
					PostType:           model.NewPointer(""),
					UserEmail:          model.NewPointer("test1@test.com"),
					UserId:             model.NewPointer("test1"),
					Username:           model.NewPointer("test1"),
					ChannelType:        &chanTypeDirect,
					PostFileIds:        []string{"test1"},
				},
				{
					PostId:             model.NewPointer("post-id-2"),
					PostRootId:         model.NewPointer("post-id-1"),
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          model.NewPointer("channel-id"),
					ChannelName:        model.NewPointer("channel-name"),
					ChannelDisplayName: model.NewPointer("channel-display-name"),
					PostUpdateAt:       model.NewPointer(int64(100_000)),
					PostCreateAt:       model.NewPointer(int64(100_000)),
					PostMessage:        model.NewPointer("message2"),
					PostProps:          model.NewPointer("{}"),
					PostType:           model.NewPointer(""),
					UserEmail:          model.NewPointer("test1@test.com"),
					UserId:             model.NewPointer("test1"),
					Username:           model.NewPointer("test1"),
					ChannelType:        &chanTypeDirect,
				},
			},
			attachments: map[string][]*model.FileInfo{
				"post-id-1": {
					{
						Name:     "test1-attachment",
						Id:       "test1-attachment",
						Path:     "test1-attachment",
						CreateAt: 1,
						Size:     101,
					},
				},
			},
			attachmentsContent: map[string]string{
				"test1-attachment": "this is the attachment content",
			},
			expectedAttachmentContent: [][]string{
				{""},
			},
			maxEmailBytes:   100,
			numExpectedEmls: 1,
			expectedHeaders: [][]string{{
				"MIME-Version: 1.0",
				"X-Mattermost-ChannelType: direct",
				"Content-Transfer-Encoding: 8bit",
				"Precedence: bulk",
				"X-GlobalRelay-MsgType: Mattermost",
				"X-Mattermost-ChannelID: channel-id",
				"X-Mattermost-ChannelName: channel-display-name",
				"Auto-Submitted: auto-generated",
				"Date: Thu, 01 Jan 1970 00:01:40 +0000",
				"From: test1@test.com",
				"To: test1@test.com,test2@test.com",
				"Subject: Mattermost Compliance Export: channel-display-name",
			}},

			expectedTexts: [][]string{{
				strings.Join([]string{
					"* TeamId: team-id",
					"* TeamName: team-name",
					"* TeamDisplayName: team-display-name",
					"* ChannelId: channel-id",
					"* ChannelName: channel-name",
					"* ChannelDisplayName: channel-display-name",
					"* Started: 1970-01-01T00:00:00Z",
					"* Ended: 1970-01-01T00:01:40Z",
					"* Duration: 2 minutes",
				}, "\r\n"),
				strings.Join([]string{
					"--------",
					"Messages",
					"--------",
					"",
					"* post-id-1 1970-01-01T00:00:00Z @test1 test1 @test1 user (test1@test.com) =",
					"message1",
					"* post-id-1 1970-01-01T00:00:00Z @test1 test1 @test1 user (test1@test.com) =",
					"Uploaded file \"test1-attachment\" (id 'test1-attachment') was removed becaus=",
					"e it was too large to send.",
					"* post-id-2 1970-01-01T00:01:40Z @test1 test1 @test1 user (test1@test.com) =",
					"message2",
				}, "\r\n"),
			}},

			notExpectedTexts: [][]string{{
				"Content-Disposition: attachment; filename=\"test1-attachment\"",
			}},
			expectedHTMLs: [][]string{{
				strings.Join([]string{
					"    <ul>",
					"        <li><span class=3D\"bold\">TeamId:</span>team-id</li>",
					"        <li><span class=3D\"bold\">TeamName:</span>team-name</li>",
					"        <li><span class=3D\"bold\">TeamDisplayName:</span>team-display-name</=",
					"li>",
					"        <li><span class=3D\"bold\">ChannelId:</span>channel-id</li>",
					"        <li><span class=3D\"bold\">ChannelName:</span>channel-name</li>",
					"        <li><span class=3D\"bold\">ChannelDisplayName:</span>channel-display-=",
					"name</li>",
					"        <li><span class=3D\"bold\">Started:</span>1970-01-01T00:00:00Z</li>",
					"        <li><span class=3D\"bold\">Ended:</span>1970-01-01T00:01:40Z</li>",
					"        <li><span class=3D\"bold\">Duration:</span>2 minutes</li>",
					"    </ul>",
				}, "\r\n"),
				strings.Join([]string{
					"<tr>",
					"    <td class=3D\"userid\">test1</td>",
					"    <td class=3D\"username\">@test</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test1@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">2</td>",
					"</tr>",
					"",
					"<tr>",
					"    <td class=3D\"userid\">test2</td>",
					"    <td class=3D\"username\">@test2</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test2@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">0</td>",
					"</tr>",
					"",
					"<tr>",
					"    <td class=3D\"userid\">test3</td>",
					"    <td class=3D\"username\">@test3</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test3@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">0</td>",
					"</tr>",
				}, "\r\n"),

				strings.Join([]string{
					"<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id-1</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:00:00Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">test1</span>",
					"    <span class=3D\"postusername\">@test1</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">message1</span>",
					"</li>",
					"",
					"<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id-1</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:00:00Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">test1</span>",
					"    <span class=3D\"postusername\">@test1</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">Uploaded file &#34;test1-attachment&#34; (id &#=",
					"39;test1-attachment&#39;) was removed because it was too large to send.</sp=",
					"an>",
					"</li>",
					"",
					"<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id-2</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:01:40Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">test1</span>",
					"    <span class=3D\"postusername\">@test1</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">message2</span>",
					"</li>",
				}, "\r\n"),
			}},
			expectedWarnings: 0,
		},

		{
			name: "posts with attachments, post size and attachment too large, new channel export needed and attachment removed",
			cmhs: map[string][]*model.ChannelMemberHistoryResult{
				"channel-id": {
					{
						JoinTime: 0, UserId: "test1", UserEmail: "test1@test.com", Username: "test", LeaveTime: model.NewPointer(int64(100_000)),
					},
					{
						JoinTime: 8, UserId: "test2", UserEmail: "test2@test.com", Username: "test2", LeaveTime: model.NewPointer(int64(100_000)),
					},
					{
						JoinTime: 400, UserId: "test3", UserEmail: "test3@test.com", Username: "test3",
					},
				},
			},
			metadata: map[string]*shared.MetadataChannel{
				"channel-id": {
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          "channel-id",
					ChannelName:        "channel-name",
					ChannelDisplayName: "channel-display-name",
					ChannelType:        chanTypeDirect,
					RoomId:             "direct - channel-id",
					StartTime:          1,
					EndTime:            100,
				},
			},
			startTime: 1,
			endTime:   100_000,
			posts: []*model.MessageExport{
				{
					PostId:             model.NewPointer("post-id-1"),
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          model.NewPointer("channel-id"),
					ChannelName:        model.NewPointer("channel-name"),
					ChannelDisplayName: model.NewPointer("channel-display-name"),
					PostUpdateAt:       model.NewPointer(int64(1)),
					PostCreateAt:       model.NewPointer(int64(1)),
					PostMessage:        model.NewPointer("message1"),
					PostProps:          model.NewPointer("{}"),
					PostType:           model.NewPointer(""),
					UserEmail:          model.NewPointer("test1@test.com"),
					UserId:             model.NewPointer("test1"),
					Username:           model.NewPointer("test1"),
					ChannelType:        &chanTypeDirect,
					PostFileIds:        []string{"test1"},
				},
				{
					PostId:             model.NewPointer("post-id-2"),
					PostRootId:         model.NewPointer("post-id-1"),
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          model.NewPointer("channel-id"),
					ChannelName:        model.NewPointer("channel-name"),
					ChannelDisplayName: model.NewPointer("channel-display-name"),
					PostUpdateAt:       model.NewPointer(int64(100_000)),
					PostCreateAt:       model.NewPointer(int64(100_000)),
					PostMessage:        model.NewPointer("message2"),
					PostProps:          model.NewPointer("{}"),
					PostType:           model.NewPointer(""),
					UserEmail:          model.NewPointer("test1@test.com"),
					UserId:             model.NewPointer("test1"),
					Username:           model.NewPointer("test1"),
					ChannelType:        &chanTypeDirect,
					PostFileIds:        []string{"test2"},
				},
			},
			attachments: map[string][]*model.FileInfo{
				"post-id-1": {
					{
						Name: "test1-attachment",
						Id:   "test1-attachment",
						Path: "test1-attachment",
						Size: 20, // 20 + 8 for message = message of 28 bytes. next one will be too big.
					},
				},
				"post-id-2": {
					{
						Name: "test2-attachment",
						Id:   "test2-attachment",
						Path: "test2-attachment",
						Size: 31, // too big, will be deleted
					},
				},
			},
			maxEmailBytes:   30,
			numExpectedEmls: 2,
			attachmentsContent: map[string]string{
				"test1-attachment": "this is the attachment content",
				"test2-attachment": "this is the second attachment content",
			},
			expectedAttachmentContent: [][]string{
				{base64.StdEncoding.EncodeToString([]byte("this is the attachment content"))},
				{""},
			},
			notExpectedTexts: [][]string{
				// eml 0
				{
					"Content-Disposition: attachment; filename=\"test2-attachment\"",
				},
				// eml 1
				{
					"Content-Disposition: attachment; filename=\"test1-attachment\"",
					"Content-Disposition: attachment; filename=\"test2-attachment\"",
				},
			},

			expectedHeaders: [][]string{
				// eml 0
				{
					"MIME-Version: 1.0",
					"X-Mattermost-ChannelType: direct",
					"Content-Transfer-Encoding: 8bit",
					"Precedence: bulk",
					"X-GlobalRelay-MsgType: Mattermost",
					"X-Mattermost-ChannelID: channel-id",
					"X-Mattermost-ChannelName: channel-display-name",
					"Auto-Submitted: auto-generated",
					"Date: Thu, 01 Jan 1970 00:01:40 +0000",
					"From: test1@test.com",
					"To: test1@test.com,test2@test.com",
					"Subject: Mattermost Compliance Export: channel-display-name",
				},
				// eml 1
				{
					"MIME-Version: 1.0",
					"X-Mattermost-ChannelType: direct",
					"Content-Transfer-Encoding: 8bit",
					"Precedence: bulk",
					"X-GlobalRelay-MsgType: Mattermost",
					"X-Mattermost-ChannelID: channel-id",
					"X-Mattermost-ChannelName: channel-display-name",
					"Auto-Submitted: auto-generated",
					"Date: Thu, 01 Jan 1970 00:01:40 +0000",
					"From: test1@test.com",
					"To: test1@test.com,test2@test.com",
					"Subject: Mattermost Compliance Export: channel-display-name",
				},
			},

			expectedTexts: [][]string{
				// eml 0
				{
					strings.Join([]string{
						"* TeamId: team-id",
						"* TeamName: team-name",
						"* TeamDisplayName: team-display-name",
						"* ChannelId: channel-id",
						"* ChannelName: channel-name",
						"* ChannelDisplayName: channel-display-name",
						"* Started: 1970-01-01T00:00:00Z",
						"* Ended: 1970-01-01T00:01:40Z",
						"* Duration: 2 minutes",
					}, "\r\n"),
					strings.Join([]string{
						"--------",
						"Messages",
						"--------",
						"",
						"* post-id-1 1970-01-01T00:00:00Z @test1 test1 @test1 user (test1@test.com) =",
						"message1",
						"* post-id-1 1970-01-01T00:00:00Z @test1 test1 @test1 user (test1@test.com) =",
						"Uploaded file test1-attachment",
					}, "\r\n"),
					strings.Join([]string{
						"Content-Disposition: attachment; filename=\"test1-attachment\"",
						"Content-Transfer-Encoding: base64",
						"Content-Type: application/octet-stream; name=\"test1-attachment\"",
					}, "\r\n"),
				},
				// eml 1
				{
					strings.Join([]string{
						"* TeamId: team-id",
						"* TeamName: team-name",
						"* TeamDisplayName: team-display-name",
						"* ChannelId: channel-id",
						"* ChannelName: channel-name",
						"* ChannelDisplayName: channel-display-name",
						"* Started: 1970-01-01T00:00:00Z",
						"* Ended: 1970-01-01T00:01:40Z",
						"* Duration: 2 minutes",
					}, "\r\n"),
					strings.Join([]string{
						"--------",
						"Messages",
						"--------",
						"",
						"* post-id-2 1970-01-01T00:01:40Z @test1 test1 @test1 user (test1@test.com) =",
						"message2",
						"* post-id-2 1970-01-01T00:01:40Z @test1 test1 @test1 user (test1@test.com) =",
						"Uploaded file \"test2-attachment\" (id 'test2-attachment') was removed becaus=",
						"e it was too large to send.",
					}, "\r\n"),
				},
			},

			expectedHTMLs: [][]string{
				// eml 0
				{
					strings.Join([]string{
						"    <ul>",
						"        <li><span class=3D\"bold\">TeamId:</span>team-id</li>",
						"        <li><span class=3D\"bold\">TeamName:</span>team-name</li>",
						"        <li><span class=3D\"bold\">TeamDisplayName:</span>team-display-name</=",
						"li>",
						"        <li><span class=3D\"bold\">ChannelId:</span>channel-id</li>",
						"        <li><span class=3D\"bold\">ChannelName:</span>channel-name</li>",
						"        <li><span class=3D\"bold\">ChannelDisplayName:</span>channel-display-=",
						"name</li>",
						"        <li><span class=3D\"bold\">Started:</span>1970-01-01T00:00:00Z</li>",
						"        <li><span class=3D\"bold\">Ended:</span>1970-01-01T00:01:40Z</li>",
						"        <li><span class=3D\"bold\">Duration:</span>2 minutes</li>",
						"    </ul>",
					}, "\r\n"),
					strings.Join([]string{
						"<tr>",
						"    <td class=3D\"userid\">test1</td>",
						"    <td class=3D\"username\">@test</td>",
						"    <td class=3D\"usertype\">user</td>",
						"    <td class=3D\"email\">test1@test.com</td>",
						"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
						"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
						"    <td class=3D\"duration\">2 minutes</td>",
						"    <td class=3D\"messages\">1</td>",
						"</tr>",
						"",
						"<tr>",
						"    <td class=3D\"userid\">test2</td>",
						"    <td class=3D\"username\">@test2</td>",
						"    <td class=3D\"usertype\">user</td>",
						"    <td class=3D\"email\">test2@test.com</td>",
						"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
						"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
						"    <td class=3D\"duration\">2 minutes</td>",
						"    <td class=3D\"messages\">0</td>",
						"</tr>",
						"",
						"<tr>",
						"    <td class=3D\"userid\">test3</td>",
						"    <td class=3D\"username\">@test3</td>",
						"    <td class=3D\"usertype\">user</td>",
						"    <td class=3D\"email\">test3@test.com</td>",
						"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
						"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
						"    <td class=3D\"duration\">2 minutes</td>",
						"    <td class=3D\"messages\">0</td>",
						"</tr>",
					}, "\r\n"),
					strings.Join([]string{
						"<li class=3D\"message\">",
						"    <span class=3D\"post_id\">post-id-1</span>",
						"    <span class=3D\"sent_time\">1970-01-01T00:00:00Z</span>",
						"    <span class=3D\"username\">@test1</span>",
						"    <span class=3D\"userid\">test1</span>",
						"    <span class=3D\"postusername\">@test1</span>",
						"    <span class=3D\"usertype\">user</span>",
						"    <span class=3D\"email\">(test1@test.com)</span>",
						"    <span class=3D\"message\">message1</span>",
						"</li>",
						"",
						"<li class=3D\"message\">",
						"    <span class=3D\"post_id\">post-id-1</span>",
						"    <span class=3D\"sent_time\">1970-01-01T00:00:00Z</span>",
						"    <span class=3D\"username\">@test1</span>",
						"    <span class=3D\"userid\">test1</span>",
						"    <span class=3D\"postusername\">@test1</span>",
						"    <span class=3D\"usertype\">user</span>",
						"    <span class=3D\"email\">(test1@test.com)</span>",
						"    <span class=3D\"message\">Uploaded file test1-attachment</span>",
						"</li>",
					}, "\r\n"),
				},
				// eml 1
				{
					strings.Join([]string{
						"    <ul>",
						"        <li><span class=3D\"bold\">TeamId:</span>team-id</li>",
						"        <li><span class=3D\"bold\">TeamName:</span>team-name</li>",
						"        <li><span class=3D\"bold\">TeamDisplayName:</span>team-display-name</=",
						"li>",
						"        <li><span class=3D\"bold\">ChannelId:</span>channel-id</li>",
						"        <li><span class=3D\"bold\">ChannelName:</span>channel-name</li>",
						"        <li><span class=3D\"bold\">ChannelDisplayName:</span>channel-display-=",
						"name</li>",
						"        <li><span class=3D\"bold\">Started:</span>1970-01-01T00:00:00Z</li>",
						"        <li><span class=3D\"bold\">Ended:</span>1970-01-01T00:01:40Z</li>",
						"        <li><span class=3D\"bold\">Duration:</span>2 minutes</li>",
						"    </ul>",
					}, "\r\n"),
					strings.Join([]string{
						"<tr>",
						"    <td class=3D\"userid\">test1</td>",
						"    <td class=3D\"username\">@test</td>",
						"    <td class=3D\"usertype\">user</td>",
						"    <td class=3D\"email\">test1@test.com</td>",
						"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
						"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
						"    <td class=3D\"duration\">2 minutes</td>",
						"    <td class=3D\"messages\">1</td>",
						"</tr>",
						"",
						"<tr>",
						"    <td class=3D\"userid\">test2</td>",
						"    <td class=3D\"username\">@test2</td>",
						"    <td class=3D\"usertype\">user</td>",
						"    <td class=3D\"email\">test2@test.com</td>",
						"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
						"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
						"    <td class=3D\"duration\">2 minutes</td>",
						"    <td class=3D\"messages\">0</td>",
						"</tr>",
						"",
						"<tr>",
						"    <td class=3D\"userid\">test3</td>",
						"    <td class=3D\"username\">@test3</td>",
						"    <td class=3D\"usertype\">user</td>",
						"    <td class=3D\"email\">test3@test.com</td>",
						"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
						"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
						"    <td class=3D\"duration\">2 minutes</td>",
						"    <td class=3D\"messages\">0</td>",
						"</tr>",
					}, "\r\n"),
					strings.Join([]string{
						"<li class=3D\"message\">",
						"    <span class=3D\"post_id\">post-id-2</span>",
						"    <span class=3D\"sent_time\">1970-01-01T00:01:40Z</span>",
						"    <span class=3D\"username\">@test1</span>",
						"    <span class=3D\"userid\">test1</span>",
						"    <span class=3D\"postusername\">@test1</span>",
						"    <span class=3D\"usertype\">user</span>",
						"    <span class=3D\"email\">(test1@test.com)</span>",
						"    <span class=3D\"message\">message2</span>",
						"</li>",
						"",
						"<li class=3D\"message\">",
						"    <span class=3D\"post_id\">post-id-2</span>",
						"    <span class=3D\"sent_time\">1970-01-01T00:01:40Z</span>",
						"    <span class=3D\"username\">@test1</span>",
						"    <span class=3D\"userid\">test1</span>",
						"    <span class=3D\"postusername\">@test1</span>",
						"    <span class=3D\"usertype\">user</span>",
						"    <span class=3D\"email\">(test1@test.com)</span>",
						"    <span class=3D\"message\">Uploaded file &#34;test2-attachment&#34; (id &#=",
						"39;test2-attachment&#39;) was removed because it was too large to send.</sp=",
						"an>",
						"</li>",
					}, "\r\n"),
				},
			},
			expectedWarnings: 0,
		},

		{
			name: "posts with multiple attachments, size ok",
			cmhs: map[string][]*model.ChannelMemberHistoryResult{
				"channel-id": {
					{
						JoinTime: 0, UserId: "test1", UserEmail: "test1@test.com", Username: "test", LeaveTime: model.NewPointer(int64(100_000)),
					},
					{
						JoinTime: 8, UserId: "test2", UserEmail: "test2@test.com", Username: "test2", LeaveTime: model.NewPointer(int64(100_000)),
					},
					{
						JoinTime: 400, UserId: "test3", UserEmail: "test3@test.com", Username: "test3",
					},
				},
			},
			metadata: map[string]*shared.MetadataChannel{
				"channel-id": {
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          "channel-id",
					ChannelName:        "channel-name",
					ChannelDisplayName: "channel-display-name",
					ChannelType:        chanTypeDirect,
					RoomId:             "direct - channel-id",
					StartTime:          1,
					EndTime:            100,
				},
			},
			startTime: 1,
			endTime:   100_000,
			posts: []*model.MessageExport{
				{
					PostId:             model.NewPointer("post-id-1"),
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          model.NewPointer("channel-id"),
					ChannelName:        model.NewPointer("channel-name"),
					ChannelDisplayName: model.NewPointer("channel-display-name"),
					PostUpdateAt:       model.NewPointer(int64(1)),
					PostCreateAt:       model.NewPointer(int64(1)),
					PostMessage:        model.NewPointer("message1"),
					PostProps:          model.NewPointer("{}"),
					PostType:           model.NewPointer(""),
					UserEmail:          model.NewPointer("test1@test.com"),
					UserId:             model.NewPointer("test1"),
					Username:           model.NewPointer("test1"),
					ChannelType:        &chanTypeDirect,
					PostFileIds:        []string{"test1", "test1-2"},
				},
				{
					PostId:             model.NewPointer("post-id-2"),
					PostRootId:         model.NewPointer("post-id-1"),
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          model.NewPointer("channel-id"),
					ChannelName:        model.NewPointer("channel-name"),
					ChannelDisplayName: model.NewPointer("channel-display-name"),
					PostUpdateAt:       model.NewPointer(int64(100_000)),
					PostCreateAt:       model.NewPointer(int64(100_000)),
					PostMessage:        model.NewPointer("message2"),
					PostProps:          model.NewPointer("{}"),
					PostType:           model.NewPointer(""),
					UserEmail:          model.NewPointer("test1@test.com"),
					UserId:             model.NewPointer("test1"),
					Username:           model.NewPointer("test1"),
					ChannelType:        &chanTypeDirect,
				},
			},
			attachments: map[string][]*model.FileInfo{
				"post-id-1": {
					{
						Name:     "test1-attachment",
						Id:       "test1-attachment",
						Path:     "test1-attachment",
						CreateAt: 1,
					},
					{
						Name:     "test1-attachment-2",
						Id:       "test1-attachment-2",
						Path:     "test1-attachment-2",
						CreateAt: 1,
					},
				},
			},
			attachmentsContent: map[string]string{
				"test1-attachment":   "this is the attachment content",
				"test1-attachment-2": "this is the second attachment content",
			},
			expectedAttachmentContent: [][]string{
				{base64.StdEncoding.EncodeToString([]byte("this is the attachment content"))},
				{base64.StdEncoding.EncodeToString([]byte("this is the second attachment content"))},
			},

			expectedHeaders: [][]string{{
				"MIME-Version: 1.0",
				"X-Mattermost-ChannelType: direct",
				"Content-Transfer-Encoding: 8bit",
				"Precedence: bulk",
				"X-GlobalRelay-MsgType: Mattermost",
				"X-Mattermost-ChannelID: channel-id",
				"X-Mattermost-ChannelName: channel-display-name",
				"Auto-Submitted: auto-generated",
				"Date: Thu, 01 Jan 1970 00:01:40 +0000",
				"From: test1@test.com",
				"To: test1@test.com,test2@test.com",
				"Subject: Mattermost Compliance Export: channel-display-name",
			}},

			expectedTexts: [][]string{{
				strings.Join([]string{
					"* TeamId: team-id",
					"* TeamName: team-name",
					"* TeamDisplayName: team-display-name",
					"* ChannelId: channel-id",
					"* ChannelName: channel-name",
					"* ChannelDisplayName: channel-display-name",
					"* Started: 1970-01-01T00:00:00Z",
					"* Ended: 1970-01-01T00:01:40Z",
					"* Duration: 2 minutes",
				}, "\r\n"),
				strings.Join([]string{
					"--------",
					"Messages",
					"--------",
					"",
					"* post-id-1 1970-01-01T00:00:00Z @test1 test1 @test1 user (test1@test.com) =",
					"message1",
					"* post-id-1 1970-01-01T00:00:00Z @test1 test1 @test1 user (test1@test.com) =",
					"Uploaded file test1-attachment",
					"* post-id-1 1970-01-01T00:00:00Z @test1 test1 @test1 user (test1@test.com) =",
					"Uploaded file test1-attachment-2",
					"* post-id-2 1970-01-01T00:01:40Z @test1 test1 @test1 user (test1@test.com) =",
					"message2",
				}, "\r\n"),
				strings.Join([]string{
					"Content-Disposition: attachment; filename=\"test1-attachment\"",
					"Content-Transfer-Encoding: base64",
					"Content-Type: application/octet-stream; name=\"test1-attachment\"",
				}, "\r\n"),
				strings.Join([]string{
					"Content-Disposition: attachment; filename=\"test1-attachment-2\"",
					"Content-Transfer-Encoding: base64",
					"Content-Type: application/octet-stream; name=\"test1-attachment-2\"",
				}, "\r\n"),
			}},

			expectedHTMLs: [][]string{{
				strings.Join([]string{
					"    <ul>",
					"        <li><span class=3D\"bold\">TeamId:</span>team-id</li>",
					"        <li><span class=3D\"bold\">TeamName:</span>team-name</li>",
					"        <li><span class=3D\"bold\">TeamDisplayName:</span>team-display-name</=",
					"li>",
					"        <li><span class=3D\"bold\">ChannelId:</span>channel-id</li>",
					"        <li><span class=3D\"bold\">ChannelName:</span>channel-name</li>",
					"        <li><span class=3D\"bold\">ChannelDisplayName:</span>channel-display-=",
					"name</li>",
					"        <li><span class=3D\"bold\">Started:</span>1970-01-01T00:00:00Z</li>",
					"        <li><span class=3D\"bold\">Ended:</span>1970-01-01T00:01:40Z</li>",
					"        <li><span class=3D\"bold\">Duration:</span>2 minutes</li>",
					"    </ul>",
				}, "\r\n"),
				strings.Join([]string{
					"<tr>",
					"    <td class=3D\"userid\">test1</td>",
					"    <td class=3D\"username\">@test</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test1@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">2</td>",
					"</tr>",
					"",
					"<tr>",
					"    <td class=3D\"userid\">test2</td>",
					"    <td class=3D\"username\">@test2</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test2@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">0</td>",
					"</tr>",
					"",
					"<tr>",
					"    <td class=3D\"userid\">test3</td>",
					"    <td class=3D\"username\">@test3</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test3@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">0</td>",
					"</tr>",
				}, "\r\n"),

				strings.Join([]string{
					"<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id-1</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:00:00Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">test1</span>",
					"    <span class=3D\"postusername\">@test1</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">message1</span>",
					"</li>",
					"",
					"<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id-1</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:00:00Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">test1</span>",
					"    <span class=3D\"postusername\">@test1</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">Uploaded file test1-attachment</span>",
					"</li>",
					"",
					"<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id-1</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:00:00Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">test1</span>",
					"    <span class=3D\"postusername\">@test1</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">Uploaded file test1-attachment-2</span>",
					"</li>",
					"",
					"<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id-2</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:01:40Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">test1</span>",
					"    <span class=3D\"postusername\">@test1</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">message2</span>",
					"</li>",
				}, "\r\n"),
			}},
			expectedWarnings: 0,
			numExpectedEmls:  1,
		},

		{
			name: "posts with deleted post and deleted attachments",
			cmhs: map[string][]*model.ChannelMemberHistoryResult{
				"channel-id": {
					{
						JoinTime: 0, UserId: "test1", UserEmail: "test1@test.com", Username: "test", LeaveTime: model.NewPointer(int64(100_000)),
					},
					{
						JoinTime: 8, UserId: "test2", UserEmail: "test2@test.com", Username: "test2", LeaveTime: model.NewPointer(int64(100_000)),
					},
					{
						JoinTime: 400, UserId: "test3", UserEmail: "test3@test.com", Username: "test3",
					},
				},
			},
			metadata: map[string]*shared.MetadataChannel{
				"channel-id": {
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          "channel-id",
					ChannelName:        "channel-name",
					ChannelDisplayName: "channel-display-name",
					ChannelType:        chanTypeDirect,
					RoomId:             "direct - channel-id",
					StartTime:          1,
					EndTime:            100,
				},
			},
			startTime: 1,
			endTime:   200_000,
			posts: []*model.MessageExport{
				{
					PostId:             model.NewPointer("post-id-1"),
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          model.NewPointer("channel-id"),
					ChannelName:        model.NewPointer("channel-name"),
					ChannelDisplayName: model.NewPointer("channel-display-name"),
					PostUpdateAt:       model.NewPointer(int64(1)),
					PostCreateAt:       model.NewPointer(int64(1)),
					PostMessage:        model.NewPointer("message"),
					PostProps:          model.NewPointer("{}"),
					PostType:           model.NewPointer(""),
					UserEmail:          model.NewPointer("test1@test.com"),
					UserId:             model.NewPointer("test1"),
					Username:           model.NewPointer("test1"),
					ChannelType:        &chanTypeDirect,
					PostFileIds:        []string{"test1"},
				},
				{
					PostId:             model.NewPointer("post-id-2"),
					PostRootId:         model.NewPointer("post-id-1"),
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          model.NewPointer("channel-id"),
					ChannelName:        model.NewPointer("channel-name"),
					ChannelDisplayName: model.NewPointer("channel-display-name"),
					PostCreateAt:       model.NewPointer(int64(100_000)),
					PostUpdateAt:       model.NewPointer(int64(200_000)),
					PostDeleteAt:       model.NewPointer(int64(200_000)),
					PostMessage:        model.NewPointer("message"),
					PostProps:          model.NewPointer("{\"deleteBy\":\"fy8j97gwii84bk4zxprbpc9d9w\"}"),
					PostType:           model.NewPointer(""),
					UserEmail:          model.NewPointer("test1@test.com"),
					UserId:             model.NewPointer("test1"),
					Username:           model.NewPointer("test1"),
					ChannelType:        &chanTypeDirect,
					PostFileIds:        []string{},
				},
			},
			attachments: map[string][]*model.FileInfo{
				"post-id-1": {
					{
						Name:     "test1-attachment",
						Id:       "test1-attachment",
						Path:     "test1-attachment",
						CreateAt: 1,
						UpdateAt: 200_000,
						DeleteAt: 200_000,
					},
				},
			},
			attachmentsContent: map[string]string{
				"test1-attachment": "this is the attachment content",
			},
			expectedAttachmentContent: [][]string{
				{base64.StdEncoding.EncodeToString([]byte("this is the attachment content"))},
			},
			expectedHeaders: [][]string{{
				"MIME-Version: 1.0",
				"X-Mattermost-ChannelType: direct",
				"Content-Transfer-Encoding: 8bit",
				"Precedence: bulk",
				"X-GlobalRelay-MsgType: Mattermost",
				"X-Mattermost-ChannelID: channel-id",
				"X-Mattermost-ChannelName: channel-display-name",
				"Auto-Submitted: auto-generated",
				"Date: Thu, 01 Jan 1970 00:03:20 +0000",
				"From: test1@test.com",
				"To: test1@test.com,test2@test.com",
				"Subject: Mattermost Compliance Export: channel-display-name",
			}},

			expectedTexts: [][]string{{
				strings.Join([]string{
					"* TeamId: team-id",
					"* TeamName: team-name",
					"* TeamDisplayName: team-display-name",
					"* ChannelId: channel-id",
					"* ChannelName: channel-name",
					"* ChannelDisplayName: channel-display-name",
					"* Started: 1970-01-01T00:00:00Z",
					"* Ended: 1970-01-01T00:03:20Z",
					"* Duration: 3 minutes",
				}, "\r\n"),
				strings.Join([]string{
					"--------",
					"Messages",
					"--------",
					"",
					"* post-id-1 1970-01-01T00:00:00Z @test1 test1 @test1 user (test1@test.com) =",
					"message",
					"* post-id-1 1970-01-01T00:00:00Z @test1 test1 @test1 user (test1@test.com) =",
					"Uploaded file test1-attachment",
					"* post-id-1 1970-01-01T00:00:00Z @test1 test1 @test1 user (test1@test.com) =",
					"Deleted file test1-attachment FileDeleted 1970-01-01T00:03:20Z",
					"* post-id-2 1970-01-01T00:01:40Z @test1 test1 @test1 user (test1@test.com) =",
					"message",
					"* post-id-2 1970-01-01T00:01:40Z @test1 test1 @test1 user (test1@test.com) =",
					"delete message Deleted 1970-01-01T00:03:20Z",
				}, "\r\n"),
				strings.Join([]string{
					"Content-Disposition: attachment; filename=\"test1-attachment\"",
					"Content-Transfer-Encoding: base64",
					"Content-Type: application/octet-stream; name=\"test1-attachment\"",
				}, "\r\n"),
			}},

			expectedHTMLs: [][]string{{
				strings.Join([]string{
					"    <ul>",
					"        <li><span class=3D\"bold\">TeamId:</span>team-id</li>",
					"        <li><span class=3D\"bold\">TeamName:</span>team-name</li>",
					"        <li><span class=3D\"bold\">TeamDisplayName:</span>team-display-name</=",
					"li>",
					"        <li><span class=3D\"bold\">ChannelId:</span>channel-id</li>",
					"        <li><span class=3D\"bold\">ChannelName:</span>channel-name</li>",
					"        <li><span class=3D\"bold\">ChannelDisplayName:</span>channel-display-=",
					"name</li>",
					"        <li><span class=3D\"bold\">Started:</span>1970-01-01T00:00:00Z</li>",
					"        <li><span class=3D\"bold\">Ended:</span>1970-01-01T00:03:20Z</li>",
					"        <li><span class=3D\"bold\">Duration:</span>3 minutes</li>",
					"    </ul>",
				}, "\r\n"),
				strings.Join([]string{
					"<tr>",
					"    <td class=3D\"userid\">test1</td>",
					"    <td class=3D\"username\">@test</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test1@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">3</td>",
					"</tr>",
					"", "<tr>",
					"    <td class=3D\"userid\">test2</td>",
					"    <td class=3D\"username\">@test2</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test2@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">0</td>",
					"</tr>",
					"", "<tr>",
					"    <td class=3D\"userid\">test3</td>",
					"    <td class=3D\"username\">@test3</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test3@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:03:20Z</td>",
					"    <td class=3D\"duration\">3 minutes</td>",
					"    <td class=3D\"messages\">0</td>",
					"</tr>",
				}, "\r\n"),

				strings.Join([]string{
					"<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id-1</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:00:00Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">test1</span>",
					"    <span class=3D\"postusername\">@test1</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">message</span>",
					"</li>",
					"", "<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id-1</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:00:00Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">test1</span>",
					"    <span class=3D\"postusername\">@test1</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">Uploaded file test1-attachment</span>",
					"</li>",
					"", "<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id-1</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:00:00Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">test1</span>",
					"    <span class=3D\"postusername\">@test1</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">Deleted file test1-attachment</span>",
					"    <span class=3D\"update_type\">FileDeleted</span>",
					"    <span class=3D\"update_time\">1970-01-01T00:03:20Z</span>",
					"    <span class=3D\"edited_new_msg_id\"></span>",
					"</li>",
					"", "<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id-2</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:01:40Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">test1</span>",
					"    <span class=3D\"postusername\">@test1</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">message</span>",
					"</li>",
					"", "<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id-2</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:01:40Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">test1</span>",
					"    <span class=3D\"postusername\">@test1</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">delete message</span>",
					"    <span class=3D\"update_type\">Deleted</span>",
					"    <span class=3D\"update_time\">1970-01-01T00:03:20Z</span>",
					"    <span class=3D\"edited_new_msg_id\"></span>",
					"</li>",
				}, "\r\n"),
			}},
			expectedWarnings: 0,
			numExpectedEmls:  1,
		},

		{
			name: "posts with deleted attachments, no deleted post, and at different time from original post",
			cmhs: map[string][]*model.ChannelMemberHistoryResult{
				"channel-id": {
					{
						JoinTime: 0, UserId: "test1", UserEmail: "test1@test.com", Username: "test", LeaveTime: model.NewPointer(int64(100_000)),
					},
					{
						JoinTime: 8, UserId: "test2", UserEmail: "test2@test.com", Username: "test2", LeaveTime: model.NewPointer(int64(100_000)),
					},
					{
						JoinTime: 400, UserId: "test3", UserEmail: "test3@test.com", Username: "test3",
					},
				},
			},
			metadata: map[string]*shared.MetadataChannel{
				"channel-id": {
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          "channel-id",
					ChannelName:        "channel-name",
					ChannelDisplayName: "channel-display-name",
					ChannelType:        chanTypeDirect,
					RoomId:             "direct - channel-id",
					StartTime:          1,
					EndTime:            100,
				},
			},
			startTime: 1,
			endTime:   100_000,
			posts: []*model.MessageExport{
				{
					PostId:             model.NewPointer("post-id-1"),
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          model.NewPointer("channel-id"),
					ChannelName:        model.NewPointer("channel-name"),
					ChannelDisplayName: model.NewPointer("channel-display-name"),
					PostUpdateAt:       model.NewPointer(int64(1)),
					PostCreateAt:       model.NewPointer(int64(1)),
					PostMessage:        model.NewPointer("message"),
					PostProps:          model.NewPointer("{}"),
					PostType:           model.NewPointer(""),
					UserEmail:          model.NewPointer("test1@test.com"),
					UserId:             model.NewPointer("test1"),
					Username:           model.NewPointer("test1"),
					ChannelType:        &chanTypeDirect,
					PostFileIds:        []string{"test1"},
				},
				{
					PostId:             model.NewPointer("post-id-2"),
					PostRootId:         model.NewPointer("post-id-1"),
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          model.NewPointer("channel-id"),
					ChannelName:        model.NewPointer("channel-name"),
					ChannelDisplayName: model.NewPointer("channel-display-name"),
					PostUpdateAt:       model.NewPointer(int64(100_000)),
					PostCreateAt:       model.NewPointer(int64(100_000)),
					PostMessage:        model.NewPointer("message"),
					PostProps:          model.NewPointer("{}"),
					PostType:           model.NewPointer(""),
					UserEmail:          model.NewPointer("test1@test.com"),
					UserId:             model.NewPointer("test1"),
					Username:           model.NewPointer("test1"),
					ChannelType:        &chanTypeDirect,
					PostFileIds:        []string{},
				},
			},
			attachments: map[string][]*model.FileInfo{
				"post-id-1": {
					{
						Name:     "test1-attachment",
						Id:       "test1-attachment",
						Path:     "test1-attachment",
						CreateAt: 1,
						UpdateAt: 200_000,
						DeleteAt: 200_000,
					},
				},
			},
			attachmentsContent: map[string]string{
				"test1-attachment": "this is the attachment content",
			},
			expectedAttachmentContent: [][]string{
				{base64.StdEncoding.EncodeToString([]byte("this is the attachment content"))},
			},
			expectedHeaders: [][]string{{
				"MIME-Version: 1.0",
				"X-Mattermost-ChannelType: direct",
				"Content-Transfer-Encoding: 8bit",
				"Precedence: bulk",
				"X-GlobalRelay-MsgType: Mattermost",
				"X-Mattermost-ChannelID: channel-id",
				"X-Mattermost-ChannelName: channel-display-name",
				"Auto-Submitted: auto-generated",
				"Date: Thu, 01 Jan 1970 00:01:40 +0000",
				"From: test1@test.com",
				"To: test1@test.com,test2@test.com",
				"Subject: Mattermost Compliance Export: channel-display-name",
			}},

			expectedTexts: [][]string{{
				strings.Join([]string{
					"* TeamId: team-id",
					"* TeamName: team-name",
					"* TeamDisplayName: team-display-name",
					"* ChannelId: channel-id",
					"* ChannelName: channel-name",
					"* ChannelDisplayName: channel-display-name",
					"* Started: 1970-01-01T00:00:00Z",
					"* Ended: 1970-01-01T00:01:40Z",
					"* Duration: 2 minutes",
				}, "\r\n"),
				strings.Join([]string{
					"--------",
					"Messages",
					"--------",
					"",
					"* post-id-1 1970-01-01T00:00:00Z @test1 test1 @test1 user (test1@test.com) =",
					"message",
					"* post-id-1 1970-01-01T00:00:00Z @test1 test1 @test1 user (test1@test.com) =",
					"Uploaded file test1-attachment",
					"* post-id-1 1970-01-01T00:00:00Z @test1 test1 @test1 user (test1@test.com) =",
					"Deleted file test1-attachment FileDeleted 1970-01-01T00:03:20Z",
					"* post-id-2 1970-01-01T00:01:40Z @test1 test1 @test1 user (test1@test.com) =",
					"message",
				}, "\r\n"),
				strings.Join([]string{
					"Content-Disposition: attachment; filename=\"test1-attachment\"",
					"Content-Transfer-Encoding: base64",
					"Content-Type: application/octet-stream; name=\"test1-attachment\"",
				}, "\r\n"),
			}},

			expectedHTMLs: [][]string{{
				strings.Join([]string{
					"    <ul>",
					"        <li><span class=3D\"bold\">TeamId:</span>team-id</li>",
					"        <li><span class=3D\"bold\">TeamName:</span>team-name</li>",
					"        <li><span class=3D\"bold\">TeamDisplayName:</span>team-display-name</=",
					"li>",
					"        <li><span class=3D\"bold\">ChannelId:</span>channel-id</li>",
					"        <li><span class=3D\"bold\">ChannelName:</span>channel-name</li>",
					"        <li><span class=3D\"bold\">ChannelDisplayName:</span>channel-display-=",
					"name</li>",
					"        <li><span class=3D\"bold\">Started:</span>1970-01-01T00:00:00Z</li>",
					"        <li><span class=3D\"bold\">Ended:</span>1970-01-01T00:01:40Z</li>",
					"        <li><span class=3D\"bold\">Duration:</span>2 minutes</li>",
					"    </ul>",
				}, "\r\n"),
				strings.Join([]string{
					"<tr>",
					"    <td class=3D\"userid\">test1</td>",
					"    <td class=3D\"username\">@test</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test1@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">2</td>",
					"</tr>",
					"",
					"<tr>",
					"    <td class=3D\"userid\">test2</td>",
					"    <td class=3D\"username\">@test2</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test2@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">0</td>",
					"</tr>",
					"",
					"<tr>",
					"    <td class=3D\"userid\">test3</td>",
					"    <td class=3D\"username\">@test3</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test3@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">0</td>",
					"</tr>",
				}, "\r\n"),

				strings.Join([]string{
					"<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id-1</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:00:00Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">test1</span>",
					"    <span class=3D\"postusername\">@test1</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">message</span>",
					"</li>",
					"",
					"<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id-1</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:00:00Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">test1</span>",
					"    <span class=3D\"postusername\">@test1</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">Uploaded file test1-attachment</span>",
					"</li>",
					"",
					"<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id-1</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:00:00Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">test1</span>",
					"    <span class=3D\"postusername\">@test1</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">Deleted file test1-attachment</span>",
					"    <span class=3D\"update_type\">FileDeleted</span>",
					"    <span class=3D\"update_time\">1970-01-01T00:03:20Z</span>",
					"    <span class=3D\"edited_new_msg_id\"></span>",
					"</li>",
					"",
					"<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id-2</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:01:40Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">test1</span>",
					"    <span class=3D\"postusername\">@test1</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">message</span>",
					"</li>",
				}, "\r\n"),
			}},
			expectedWarnings: 0,
			numExpectedEmls:  1,
		},

		{
			name: "posts with missing attachments",
			cmhs: map[string][]*model.ChannelMemberHistoryResult{
				"channel-id": {
					{
						JoinTime: 0, UserId: "test1", UserEmail: "test1@test.com", Username: "test", LeaveTime: model.NewPointer(int64(100_000)),
					},
					{
						JoinTime: 8, UserId: "test2", UserEmail: "test2@test.com", Username: "test2", LeaveTime: model.NewPointer(int64(100_000)),
					},
					{
						JoinTime: 400, UserId: "test3", UserEmail: "test3@test.com", Username: "test3",
					},
				},
			},
			metadata: map[string]*shared.MetadataChannel{
				"channel-id": {
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          "channel-id",
					ChannelName:        "channel-name",
					ChannelDisplayName: "channel-display-name",
					ChannelType:        chanTypeDirect,
					RoomId:             "direct - channel-id",
					StartTime:          1,
					EndTime:            100,
				},
			},
			startTime: 1,
			endTime:   100_000,
			posts: []*model.MessageExport{
				{
					PostId:             model.NewPointer("post-id-1"),
					PostOriginalId:     model.NewPointer("post-original-id"),
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          model.NewPointer("channel-id"),
					ChannelName:        model.NewPointer("channel-name"),
					ChannelDisplayName: model.NewPointer("channel-display-name"),
					PostUpdateAt:       model.NewPointer(int64(1)),
					PostCreateAt:       model.NewPointer(int64(1)),
					PostMessage:        model.NewPointer("message"),
					PostProps:          model.NewPointer("{}"),
					PostType:           model.NewPointer(""),
					UserEmail:          model.NewPointer("test1@test.com"),
					UserId:             model.NewPointer("test1"),
					Username:           model.NewPointer("test1"),
					ChannelType:        &chanTypeDirect,
					PostFileIds:        []string{"test1"},
				},
				{
					PostId:             model.NewPointer("post-id-2"),
					PostOriginalId:     model.NewPointer("post-original-id"),
					PostRootId:         model.NewPointer("post-id-1"),
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          model.NewPointer("channel-id"),
					ChannelName:        model.NewPointer("channel-name"),
					ChannelDisplayName: model.NewPointer("channel-display-name"),
					PostUpdateAt:       model.NewPointer(int64(100_000)),
					PostCreateAt:       model.NewPointer(int64(100_000)),
					PostMessage:        model.NewPointer("message"),
					PostProps:          model.NewPointer("{}"),
					PostType:           model.NewPointer(""),
					UserEmail:          model.NewPointer("test1@test.com"),
					UserId:             model.NewPointer("test1"),
					Username:           model.NewPointer("test1"),
					ChannelType:        &chanTypeDirect,
					PostFileIds:        []string{},
				},
			},
			attachments: map[string][]*model.FileInfo{
				"post-id-1": {
					{
						Name:     "test1-attachment",
						Id:       "test1-attachment",
						Path:     "test1-attachment",
						CreateAt: 1,
					},
				},
			},
			expectedHeaders: [][]string{{
				"MIME-Version: 1.0",
				"X-Mattermost-ChannelType: direct",
				"Content-Transfer-Encoding: 8bit",
				"Precedence: bulk",
				"X-GlobalRelay-MsgType: Mattermost",
				"X-Mattermost-ChannelID: channel-id",
				"X-Mattermost-ChannelName: channel-display-name",
				"Auto-Submitted: auto-generated",
				"Date: Thu, 01 Jan 1970 00:01:40 +0000",
				"From: test1@test.com",
				"To: test1@test.com,test2@test.com",
				"Subject: Mattermost Compliance Export: channel-display-name",
			}},

			expectedTexts: [][]string{{
				strings.Join([]string{
					"* TeamId: team-id",
					"* TeamName: team-name",
					"* TeamDisplayName: team-display-name",
					"* ChannelId: channel-id",
					"* ChannelName: channel-name",
					"* ChannelDisplayName: channel-display-name",
					"* Started: 1970-01-01T00:00:00Z",
					"* Ended: 1970-01-01T00:01:40Z",
					"* Duration: 2 minutes",
				}, "\r\n"),
				strings.Join([]string{
					"--------",
					"Messages",
					"--------",
					"",
					"* post-id-1 1970-01-01T00:00:00Z @test1 test1 @test1 user (test1@test.com) =",
					"message",
					"* post-id-1 1970-01-01T00:00:00Z @test1 test1 @test1 user (test1@test.com) =",
					"Uploaded file test1-attachment",
					"* post-id-2 1970-01-01T00:01:40Z @test1 test1 @test1 user (test1@test.com) =",
					"message",
				}, "\r\n"),
				strings.Join([]string{
					"Content-Disposition: attachment; filename=\"test1-attachment\"",
					"Content-Transfer-Encoding: base64",
					"Content-Type: application/octet-stream; name=\"test1-attachment\"",
				}, "\r\n"),
			}},

			expectedHTMLs: [][]string{{
				strings.Join([]string{
					"    <ul>",
					"        <li><span class=3D\"bold\">TeamId:</span>team-id</li>",
					"        <li><span class=3D\"bold\">TeamName:</span>team-name</li>",
					"        <li><span class=3D\"bold\">TeamDisplayName:</span>team-display-name</=",
					"li>",
					"        <li><span class=3D\"bold\">ChannelId:</span>channel-id</li>",
					"        <li><span class=3D\"bold\">ChannelName:</span>channel-name</li>",
					"        <li><span class=3D\"bold\">ChannelDisplayName:</span>channel-display-=",
					"name</li>",
					"        <li><span class=3D\"bold\">Started:</span>1970-01-01T00:00:00Z</li>",
					"        <li><span class=3D\"bold\">Ended:</span>1970-01-01T00:01:40Z</li>",
					"        <li><span class=3D\"bold\">Duration:</span>2 minutes</li>",
					"    </ul>",
				}, "\r\n"),
				strings.Join([]string{
					"<tr>",
					"    <td class=3D\"userid\">test1</td>",
					"    <td class=3D\"username\">@test</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test1@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">2</td>",
					"</tr>",
					"",
					"<tr>",
					"    <td class=3D\"userid\">test2</td>",
					"    <td class=3D\"username\">@test2</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test2@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">0</td>",
					"</tr>",
					"",
					"<tr>",
					"    <td class=3D\"userid\">test3</td>",
					"    <td class=3D\"username\">@test3</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test3@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">0</td>",
					"</tr>",
				}, "\r\n"),

				strings.Join([]string{
					"<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id-1</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:00:00Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">test1</span>",
					"    <span class=3D\"postusername\">@test1</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">message</span>",
					"</li>",
					"",
					"<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id-1</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:00:00Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">test1</span>",
					"    <span class=3D\"postusername\">@test1</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">Uploaded file test1-attachment</span>",
					"</li>",
					"",
					"<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id-2</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:01:40Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">test1</span>",
					"    <span class=3D\"postusername\">@test1</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">message</span>",
					"</li>",
				}, "\r\n"),
			}},
			expectedWarnings: 1,
			numExpectedEmls:  1,
		},

		{
			name: "posts with override_username property",
			cmhs: map[string][]*model.ChannelMemberHistoryResult{
				"channel-id": {
					{
						JoinTime: 0, UserId: "test1", UserEmail: "test1@test.com", Username: "test", LeaveTime: model.NewPointer(int64(100_000)),
					},
					{
						JoinTime: 8, UserId: "test2", UserEmail: "test2@test.com", Username: "test2", LeaveTime: model.NewPointer(int64(100_000)),
					},
					{
						JoinTime: 400, UserId: "test3", UserEmail: "test3@test.com", Username: "test3",
					},
				},
			},
			metadata: map[string]*shared.MetadataChannel{
				"channel-id": {
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          "channel-id",
					ChannelName:        "channel-name",
					ChannelDisplayName: "channel-display-name",
					ChannelType:        chanTypeDirect,
					RoomId:             "direct - channel-id",
					StartTime:          1,
					EndTime:            100,
				},
			},
			startTime: 1,
			endTime:   100_000,
			posts: []*model.MessageExport{
				{
					PostId:             model.NewPointer("post-id"),
					PostOriginalId:     model.NewPointer("post-original-id"),
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          model.NewPointer("channel-id"),
					ChannelName:        model.NewPointer("channel-name"),
					ChannelDisplayName: model.NewPointer("channel-display-name"),
					PostUpdateAt:       model.NewPointer(int64(1)),
					PostCreateAt:       model.NewPointer(int64(1)),
					PostMessage:        model.NewPointer("message"),
					PostProps:          model.NewPointer("{}"),
					PostType:           model.NewPointer(""),
					UserEmail:          model.NewPointer("test1@test.com"),
					UserId:             model.NewPointer("test1"),
					Username:           model.NewPointer("test1"),
					ChannelType:        &chanTypeDirect,
					PostFileIds:        []string{},
				},
				{
					PostId:             model.NewPointer("post-id"),
					PostOriginalId:     model.NewPointer("post-original-id"),
					PostRootId:         model.NewPointer("post-root-id"),
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          model.NewPointer("channel-id"),
					ChannelName:        model.NewPointer("channel-name"),
					ChannelDisplayName: model.NewPointer("channel-display-name"),
					PostUpdateAt:       model.NewPointer(int64(100_000)),
					PostCreateAt:       model.NewPointer(int64(100_000)),
					PostMessage:        model.NewPointer("message"),
					PostProps:          model.NewPointer("{\"from_webhook\":\"true\",\"html\":\"<b>Test HTML</b>\",\"override_username\":\"test_username_override\",\"webhook_display_name\":\"Test Bot\"}"),
					PostType:           model.NewPointer(""),
					UserEmail:          model.NewPointer("test1@test.com"),
					UserId:             model.NewPointer("test1"),
					Username:           model.NewPointer("test1"),
					ChannelType:        &chanTypeDirect,
					PostFileIds:        []string{},
				},
			},
			attachments: map[string][]*model.FileInfo{},
			expectedHeaders: [][]string{{
				"MIME-Version: 1.0",
				"X-Mattermost-ChannelType: direct",
				"Content-Transfer-Encoding: 8bit",
				"Precedence: bulk",
				"X-GlobalRelay-MsgType: Mattermost",
				"X-Mattermost-ChannelID: channel-id",
				"X-Mattermost-ChannelName: channel-display-name",
				"Auto-Submitted: auto-generated",
				"Date: Thu, 01 Jan 1970 00:01:40 +0000",
				"From: test1@test.com",
				"To: test1@test.com,test2@test.com",
				"Subject: Mattermost Compliance Export: channel-display-name",
			}},
			expectedTexts: [][]string{{
				strings.Join([]string{
					"* TeamId: team-id",
					"* TeamName: team-name",
					"* TeamDisplayName: team-display-name",
					"* ChannelId: channel-id",
					"* ChannelName: channel-name",
					"* ChannelDisplayName: channel-display-name",
					"* Started: 1970-01-01T00:00:00Z",
					"* Ended: 1970-01-01T00:01:40Z",
					"* Duration: 2 minutes",
				}, "\r\n"),
				strings.Join([]string{
					"--------",
					"Messages",
					"--------",
					"",
					"* post-id 1970-01-01T00:00:00Z @test1 test1 @test1 user (test1@test.com) me=",
					"ssage",
					"* post-id 1970-01-01T00:01:40Z @test1 test1 @test_username_override user (t=",
					"est1@test.com) message",
				}, "\r\n"),
			}},

			expectedHTMLs: [][]string{{
				strings.Join([]string{
					"    <ul>",
					"        <li><span class=3D\"bold\">TeamId:</span>team-id</li>",
					"        <li><span class=3D\"bold\">TeamName:</span>team-name</li>",
					"        <li><span class=3D\"bold\">TeamDisplayName:</span>team-display-name</=",
					"li>",
					"        <li><span class=3D\"bold\">ChannelId:</span>channel-id</li>",
					"        <li><span class=3D\"bold\">ChannelName:</span>channel-name</li>",
					"        <li><span class=3D\"bold\">ChannelDisplayName:</span>channel-display-=",
					"name</li>",
					"        <li><span class=3D\"bold\">Started:</span>1970-01-01T00:00:00Z</li>",
					"        <li><span class=3D\"bold\">Ended:</span>1970-01-01T00:01:40Z</li>",
					"        <li><span class=3D\"bold\">Duration:</span>2 minutes</li>",
					"    </ul>",
				}, "\r\n"),
				strings.Join([]string{
					"<tr>",
					"    <td class=3D\"userid\">test1</td>",
					"    <td class=3D\"username\">@test</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test1@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">2</td>",
					"</tr>",
					"",
					"<tr>",
					"    <td class=3D\"userid\">test2</td>",
					"    <td class=3D\"username\">@test2</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test2@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">0</td>",
					"</tr>",
					"",
					"<tr>",
					"    <td class=3D\"userid\">test3</td>",
					"    <td class=3D\"username\">@test3</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test3@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">0</td>",
					"</tr>",
				}, "\r\n"),

				strings.Join([]string{
					"<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:00:00Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">test1</span>",
					"    <span class=3D\"postusername\">@test1</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">message</span>",
					"</li>",
					"",
					"<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:01:40Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">test1</span>",
					"    <span class=3D\"postusername\">@test_username_override</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">message</span>",
					"</li>",
				}, "\r\n"),
			}},
			numExpectedEmls: 1,
		},

		{
			name: "posts with webhook_display_name property",
			cmhs: map[string][]*model.ChannelMemberHistoryResult{
				"channel-id": {
					{
						JoinTime: 0, UserId: "test1", UserEmail: "test1@test.com", Username: "test", LeaveTime: model.NewPointer(int64(100_000)),
					},
					{
						JoinTime: 8, UserId: "test2", UserEmail: "test2@test.com", Username: "test2", LeaveTime: model.NewPointer(int64(100_000)),
					},
					{
						JoinTime: 400, UserId: "test3", UserEmail: "test3@test.com", Username: "test3",
					},
				},
			},
			metadata: map[string]*shared.MetadataChannel{
				"channel-id": {
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          "channel-id",
					ChannelName:        "channel-name",
					ChannelDisplayName: "channel-display-name",
					ChannelType:        chanTypeDirect,
					RoomId:             "direct - channel-id",
					StartTime:          1,
					EndTime:            100,
				},
			},
			startTime: 1,
			endTime:   100_000,
			posts: []*model.MessageExport{
				{
					PostId:             model.NewPointer("post-id"),
					PostOriginalId:     model.NewPointer("post-original-id"),
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          model.NewPointer("channel-id"),
					ChannelName:        model.NewPointer("channel-name"),
					ChannelDisplayName: model.NewPointer("channel-display-name"),
					PostUpdateAt:       model.NewPointer(int64(1)),
					PostCreateAt:       model.NewPointer(int64(1)),
					PostMessage:        model.NewPointer("message"),
					PostProps:          model.NewPointer("{}"),
					PostType:           model.NewPointer(""),
					UserEmail:          model.NewPointer("test1@test.com"),
					UserId:             model.NewPointer("test1"),
					Username:           model.NewPointer("test1"),
					ChannelType:        &chanTypeDirect,
					PostFileIds:        []string{},
				},
				{
					PostId:             model.NewPointer("post-id"),
					PostOriginalId:     model.NewPointer("post-original-id"),
					PostRootId:         model.NewPointer("post-root-id"),
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          model.NewPointer("channel-id"),
					ChannelName:        model.NewPointer("channel-name"),
					ChannelDisplayName: model.NewPointer("channel-display-name"),
					PostUpdateAt:       model.NewPointer(int64(100_000)),
					PostCreateAt:       model.NewPointer(int64(100_000)),
					PostMessage:        model.NewPointer("message"),
					PostProps:          model.NewPointer("{\"from_webhook\":\"true\",\"webhook_display_name\":\"Test Bot\"}"),
					PostType:           model.NewPointer(""),
					UserEmail:          model.NewPointer("test1@test.com"),
					UserId:             model.NewPointer("test1"),
					Username:           model.NewPointer("test1"),
					ChannelType:        &chanTypeDirect,
					PostFileIds:        []string{},
				},
			},
			attachments: map[string][]*model.FileInfo{},
			expectedHeaders: [][]string{{
				"MIME-Version: 1.0",
				"X-Mattermost-ChannelType: direct",
				"Content-Transfer-Encoding: 8bit",
				"Precedence: bulk",
				"X-GlobalRelay-MsgType: Mattermost",
				"X-Mattermost-ChannelID: channel-id",
				"X-Mattermost-ChannelName: channel-display-name",
				"Auto-Submitted: auto-generated",
				"Date: Thu, 01 Jan 1970 00:01:40 +0000",
				"From: test1@test.com",
				"To: test1@test.com,test2@test.com",
				"Subject: Mattermost Compliance Export: channel-display-name",
			}},

			expectedTexts: [][]string{{
				strings.Join([]string{
					"* TeamId: team-id",
					"* TeamName: team-name",
					"* TeamDisplayName: team-display-name",
					"* ChannelId: channel-id",
					"* ChannelName: channel-name",
					"* ChannelDisplayName: channel-display-name",
					"* Started: 1970-01-01T00:00:00Z",
					"* Ended: 1970-01-01T00:01:40Z",
					"* Duration: 2 minutes",
				}, "\r\n"),
				strings.Join([]string{
					"--------",
					"Messages",
					"--------",
					"",
					"* post-id 1970-01-01T00:00:00Z @test1 test1 @test1 user (test1@test.com) me=",
					"ssage",
					"* post-id 1970-01-01T00:01:40Z @test1 test1 @Test Bot user (test1@test.com)=",
					" message",
				}, "\r\n"),
			}},

			expectedHTMLs: [][]string{{
				strings.Join([]string{
					"    <ul>",
					"        <li><span class=3D\"bold\">TeamId:</span>team-id</li>",
					"        <li><span class=3D\"bold\">TeamName:</span>team-name</li>",
					"        <li><span class=3D\"bold\">TeamDisplayName:</span>team-display-name</=",
					"li>",
					"        <li><span class=3D\"bold\">ChannelId:</span>channel-id</li>",
					"        <li><span class=3D\"bold\">ChannelName:</span>channel-name</li>",
					"        <li><span class=3D\"bold\">ChannelDisplayName:</span>channel-display-=",
					"name</li>",
					"        <li><span class=3D\"bold\">Started:</span>1970-01-01T00:00:00Z</li>",
					"        <li><span class=3D\"bold\">Ended:</span>1970-01-01T00:01:40Z</li>",
					"        <li><span class=3D\"bold\">Duration:</span>2 minutes</li>",
					"    </ul>",
				}, "\r\n"),
				strings.Join([]string{
					"<tr>",
					"    <td class=3D\"userid\">test1</td>",
					"    <td class=3D\"username\">@test</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test1@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">2</td>",
					"</tr>",
					"",
					"<tr>",
					"    <td class=3D\"userid\">test2</td>",
					"    <td class=3D\"username\">@test2</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test2@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">0</td>",
					"</tr>",
					"",
					"<tr>",
					"    <td class=3D\"userid\">test3</td>",
					"    <td class=3D\"username\">@test3</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test3@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">0</td>",
					"</tr>",
				}, "\r\n"),

				strings.Join([]string{
					"<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:00:00Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">test1</span>",
					"    <span class=3D\"postusername\">@test1</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">message</span>",
					"</li>",
					"",
					"<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:01:40Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">test1</span>",
					"    <span class=3D\"postusername\">@Test Bot</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">message</span>",
					"</li>",
				}, "\r\n"),
			}},
			numExpectedEmls: 1,
		},

		{
			name: "post with permalink preview",
			cmhs: map[string][]*model.ChannelMemberHistoryResult{
				"channel-id": {
					{
						JoinTime: 0, UserId: "test1", UserEmail: "test1@test.com", Username: "test", LeaveTime: model.NewPointer(int64(100_000)),
					},
					{
						JoinTime: 8, UserId: "test2", UserEmail: "test2@test.com", Username: "test2", LeaveTime: model.NewPointer(int64(100_000)),
					},
					{
						JoinTime: 400, UserId: "test3", UserEmail: "test3@test.com", Username: "test3",
					},
				},
			},
			metadata: map[string]*shared.MetadataChannel{
				"channel-id": {
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          "channel-id",
					ChannelName:        "channel-name",
					ChannelDisplayName: "channel-display-name",
					ChannelType:        chanTypeDirect,
					RoomId:             "direct - channel-id",
					StartTime:          1,
					EndTime:            100,
				},
			},
			startTime: 1,
			endTime:   100_000,
			posts: []*model.MessageExport{
				{
					PostId:             model.NewPointer("post-id"),
					PostOriginalId:     model.NewPointer("post-original-id"),
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          model.NewPointer("channel-id"),
					ChannelName:        model.NewPointer("channel-name"),
					ChannelDisplayName: model.NewPointer("channel-display-name"),
					PostUpdateAt:       model.NewPointer(int64(1)),
					PostCreateAt:       model.NewPointer(int64(1)),
					PostMessage:        model.NewPointer("message"),
					PostProps:          model.NewPointer(`{"previewed_post":"o4w39mc1ff8y5fite4b8hacy1x"}`),
					PostType:           model.NewPointer(""),
					UserEmail:          model.NewPointer("test1@test.com"),
					UserId:             model.NewPointer("test1"),
					Username:           model.NewPointer("test1"),
					ChannelType:        &chanTypeDirect,
					PostFileIds:        []string{},
				},
				{
					PostId:             model.NewPointer("post-id"),
					PostOriginalId:     model.NewPointer("post-original-id"),
					PostRootId:         model.NewPointer("post-root-id"),
					TeamId:             model.NewPointer("team-id"),
					TeamName:           model.NewPointer("team-name"),
					TeamDisplayName:    model.NewPointer("team-display-name"),
					ChannelId:          model.NewPointer("channel-id"),
					ChannelName:        model.NewPointer("channel-name"),
					ChannelDisplayName: model.NewPointer("channel-display-name"),
					PostUpdateAt:       model.NewPointer(int64(100_000)),
					PostCreateAt:       model.NewPointer(int64(100_000)),
					PostMessage:        model.NewPointer("message"),
					PostProps:          model.NewPointer("{}"),
					PostType:           model.NewPointer(""),
					UserEmail:          model.NewPointer("test1@test.com"),
					UserId:             model.NewPointer("test1"),
					Username:           model.NewPointer("test1"),
					ChannelType:        &chanTypeDirect,
					PostFileIds:        []string{},
				},
			},
			attachments: map[string][]*model.FileInfo{},
			expectedHeaders: [][]string{{
				"MIME-Version: 1.0",
				"X-Mattermost-ChannelType: direct",
				"Content-Transfer-Encoding: 8bit",
				"Precedence: bulk",
				"X-GlobalRelay-MsgType: Mattermost",
				"X-Mattermost-ChannelID: channel-id",
				"X-Mattermost-ChannelName: channel-display-name",
				"Auto-Submitted: auto-generated",
				"Date: Thu, 01 Jan 1970 00:01:40 +0000",
				"From: test1@test.com",
				"To: test1@test.com,test2@test.com",
				"Subject: Mattermost Compliance Export: channel-display-name",
			}},

			expectedTexts: [][]string{{
				strings.Join([]string{
					"* TeamId: team-id",
					"* TeamName: team-name",
					"* TeamDisplayName: team-display-name",
					"* ChannelId: channel-id",
					"* ChannelName: channel-name",
					"* ChannelDisplayName: channel-display-name",
					"* Started: 1970-01-01T00:00:00Z",
					"* Ended: 1970-01-01T00:01:40Z",
					"* Duration: 2 minutes",
				}, "\r\n"),
				strings.Join([]string{
					"--------",
					"Messages",
					"--------",
					"",
					"* post-id 1970-01-01T00:00:00Z @test1 test1 @test1 user (test1@test.com) me=",
					"ssage o4w39mc1ff8y5fite4b8hacy1x",
					"* post-id 1970-01-01T00:01:40Z @test1 test1 @test1 user (test1@test.com) me=",
					"ssage",
				}, "\r\n"),
			}},

			expectedHTMLs: [][]string{{
				strings.Join([]string{
					"    <ul>",
					"        <li><span class=3D\"bold\">TeamId:</span>team-id</li>",
					"        <li><span class=3D\"bold\">TeamName:</span>team-name</li>",
					"        <li><span class=3D\"bold\">TeamDisplayName:</span>team-display-name</=",
					"li>",
					"        <li><span class=3D\"bold\">ChannelId:</span>channel-id</li>",
					"        <li><span class=3D\"bold\">ChannelName:</span>channel-name</li>",
					"        <li><span class=3D\"bold\">ChannelDisplayName:</span>channel-display-=",
					"name</li>",
					"        <li><span class=3D\"bold\">Started:</span>1970-01-01T00:00:00Z</li>",
					"        <li><span class=3D\"bold\">Ended:</span>1970-01-01T00:01:40Z</li>",
					"        <li><span class=3D\"bold\">Duration:</span>2 minutes</li>",
					"    </ul>",
				}, "\r\n"),
				strings.Join([]string{
					"<tr>",
					"    <td class=3D\"userid\">test1</td>",
					"    <td class=3D\"username\">@test</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test1@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">2</td>",
					"</tr>",
					"",
					"<tr>",
					"    <td class=3D\"userid\">test2</td>",
					"    <td class=3D\"username\">@test2</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test2@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">0</td>",
					"</tr>",
					"",
					"<tr>",
					"    <td class=3D\"userid\">test3</td>",
					"    <td class=3D\"username\">@test3</td>",
					"    <td class=3D\"usertype\">user</td>",
					"    <td class=3D\"email\">test3@test.com</td>",
					"    <td class=3D\"joined\">1970-01-01T00:00:00Z</td>",
					"    <td class=3D\"left\">1970-01-01T00:01:40Z</td>",
					"    <td class=3D\"duration\">2 minutes</td>",
					"    <td class=3D\"messages\">0</td>",
					"</tr>",
				}, "\r\n"),

				strings.Join([]string{
					"<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:00:00Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">test1</span>",
					"    <span class=3D\"postusername\">@test1</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">message</span>",
					"    <span class=3D\"previews_post\">o4w39mc1ff8y5fite4b8hacy1x</span>",
					"</li>",
					"",
					"<li class=3D\"message\">",
					"    <span class=3D\"post_id\">post-id</span>",
					"    <span class=3D\"sent_time\">1970-01-01T00:01:40Z</span>",
					"    <span class=3D\"username\">@test1</span>",
					"    <span class=3D\"userid\">test1</span>",
					"    <span class=3D\"postusername\">@test1</span>",
					"    <span class=3D\"usertype\">user</span>",
					"    <span class=3D\"email\">(test1@test.com)</span>",
					"    <span class=3D\"message\">message</span>",
					"</li>",
				}, "\r\n"),
			}},
			expectedWarnings: 0,
			numExpectedEmls:  1,
		},
	}

	for _, tt := range grExportTests {
		t.Run(tt.name, func(t *testing.T) {
			mockStore := &storetest.Store{}
			defer mockStore.AssertExpectations(t)

			if len(tt.attachments) > 0 {
				for postId, attachments := range tt.attachments {
					call := mockStore.FileInfoStore.On("GetForPost", postId, true, true, false)
					call.Run(func(args mock.Arguments) {
						call.Return(attachments, nil)
					})
					for _, attachment := range attachments {
						if tt.expectedWarnings == 0 {
							content, ok := tt.attachmentsContent[attachment.Id]
							require.True(t, ok, "attachment not found for id: %s", attachment.Id)
							_, err = fileBackend.WriteFile(strings.NewReader(content), attachment.Path)
							require.NoError(t, err)

							t.Cleanup(func() {
								err = fileBackend.RemoveFile(attachment.Path)
								assert.NoError(t, err)
							})
						}
					}
				}
			}

			if tt.maxEmailBytes > 0 {
				origMaxEmailBytes := MaxEmailBytes
				MaxEmailBytes = tt.maxEmailBytes
				t.Cleanup(func() {
					MaxEmailBytes = origMaxEmailBytes
				})
			}

			exportFileName := path.Join("export", "jobName", "jobName-batch001.zip")
			results, err := GlobalRelayExport(rctx, shared.ExportParams{
				ExportType:             model.ComplianceExportTypeGlobalrelayZip,
				ChannelMetadata:        tt.metadata,
				Posts:                  tt.posts,
				ChannelMemberHistories: tt.cmhs,
				BatchPath:              exportFileName,
				BatchStartTime:         tt.startTime,
				BatchEndTime:           tt.endTime,
				Db:                     shared.NewMessageExportStore(mockStore),
				FileAttachmentBackend:  fileBackend,
				ExportBackend:          exportBackend,
				Templates:              templatesContainer,
			})
			assert.NoError(t, err)
			assert.Equal(t, tt.expectedWarnings, results.NumWarnings)

			// channel-name - (channel-id) - 0.eml
			if !tt.empty {
				if tt.numExpectedEmls == 0 {
					require.True(t, false, "need numExpectedEmls to be at least 1")
				}
				openZipAndAssertNumEmls(t, exportBackend, exportFileName, tt.numExpectedEmls)

				for batchNum := 0; batchNum < tt.numExpectedEmls; batchNum++ {
					metadata := tt.metadata["channel-id"]
					emlName := fmt.Sprintf("%s - (%s) - %d.eml", metadata.ChannelName,
						metadata.ChannelId, batchNum)
					eml := openZipAndReadFileStartingWith(t, exportBackend, exportFileName, emlName)

					// Note: for debugging, better keep this in case we need it again.
					//t.Logf("<><> batchNum %d actual: \n%s", batchNum, eml)

					t.Run("headers", func(t *testing.T) {
						for _, expectedHeader := range tt.expectedHeaders[batchNum] {
							assert.Contains(t, eml, expectedHeader, "batchNum %d, expected: %s", batchNum, expectedHeader)
						}
					})

					t.Run("text-version", func(t *testing.T) {
						for _, expectedText := range tt.expectedTexts[batchNum] {
							assert.Contains(t, eml, expectedText, "batchNum %d, expected: %s", batchNum, expectedText)
						}
						if len(tt.notExpectedTexts) > 0 {
							for _, notExpectedText := range tt.notExpectedTexts[batchNum] {
								assert.NotContains(t, eml, notExpectedText, "batchNum %d, expected: %s", batchNum, notExpectedText)
							}
						}
					})

					t.Run("html-version", func(t *testing.T) {
						for _, expectedHTML := range tt.expectedHTMLs[batchNum] {
							assert.Contains(t, eml, expectedHTML, "batchNum %d, \nexpected:\n %s\n\nactual:\n %s\n", batchNum, expectedHTML, eml)
						}
					})

					assert.Len(t, tt.expectedAttachmentContent, len(tt.attachmentsContent), "every attachmentsContent should have an expectedAttachmentContent")

					if len(tt.expectedAttachmentContent) > 0 {
						t.Run("file encoding", func(t *testing.T) {
							for _, expected := range tt.expectedAttachmentContent[batchNum] {
								assert.Contains(t, eml, expected, "batchNum %d, \nexpected:\n %s\n\nactual:\n %s\n", batchNum, expected, eml)
							}
						})
					}
				}
			}
		})
	}
}

func openZipAndReadFileStartingWith(t *testing.T, backend filestore.FileBackend, path string, startsWith string) string {
	zipBytes, err := backend.ReadFile(path)
	require.NoError(t, err)

	zipReader, err := zip.NewReader(bytes.NewReader(zipBytes), int64(len(zipBytes)))
	require.NoError(t, err)

	var names []string
	for _, f := range zipReader.File {
		if strings.HasPrefix(f.Name, startsWith) {
			file, err := f.Open()
			require.NoError(t, err)
			contents, err := io.ReadAll(file)
			require.NoError(t, err)
			err = file.Close()
			require.NoError(t, err)

			return string(contents)
		}
		names = append(names, f.Name)
	}

	require.True(t, false, "called openZipAndReadFileStartingWith but didn't file file starting with: %s. Found: %v", startsWith, names)
	return ""
}

func openZipAndAssertNumEmls(t *testing.T, backend filestore.FileBackend, path string, numEmls int) {
	zipBytes, err := backend.ReadFile(path)
	require.NoError(t, err)

	zipReader, err := zip.NewReader(bytes.NewReader(zipBytes), int64(len(zipBytes)))
	require.NoError(t, err)

	require.Len(t, zipReader.File, numEmls, "numEmls wrong")
}
