#include "parser-stmt.h"

#include "config.h"

#include "libks/arena.h"

#include "clang.h"
#include "doc.h"
#include "expr.h"
#include "lexer.h"
#include "options.h"
#include "parser-decl.h"
#include "parser-expr.h"
#include "parser-priv.h"
#include "parser-stmt-asm.h"
#include "parser-stmt-expr.h"
#include "simple-stmt-empty-loop.h"
#include "simple-stmt-switch.h"
#include "simple-stmt.h"
#include "simple.h"
#include "style.h"
#include "token.h"

#define PARSER_STMT_EXPR_DOWHILE		0x00000001u
#define PARSER_STMT_EXPR_ELSEIF			0x00000002u

static int	parser_stmt(struct parser *, struct doc *);
static int	parser_stmt1(struct parser *, struct doc *);
static int	parser_stmt_if(struct parser *, struct doc *);
static int	parser_stmt_for(struct parser *, struct doc *);
static int	parser_stmt_dowhile(struct parser *, struct doc *);
static int	parser_stmt_kw_expr(struct parser *, struct doc *, int,
    unsigned int);
static int	parser_stmt_label(struct parser *, struct doc *);
static int	parser_stmt_case(struct parser *, struct doc *);
static int	parser_stmt_goto(struct parser *, struct doc *);
static int	parser_stmt_switch(struct parser *, struct doc *);
static int	parser_stmt_while(struct parser *, struct doc *);
static int	parser_stmt_break(struct parser *, struct doc *);
static int	parser_stmt_continue(struct parser *, struct doc *);
static int	parser_stmt_return(struct parser *, struct doc *);
static int	parser_stmt_semi(struct parser *, struct doc *);
static int	parser_stmt_cpp(struct parser *, struct doc *);

static int		 parser_simple_stmt_enter(struct parser *,
    struct simple_cookie *);
static struct doc	*parser_simple_stmt_no_braces_enter(struct parser *,
    struct doc *, void **);
static void		 parser_simple_stmt_no_braces_leave(struct parser *,
    void *);
static int		 is_loop_stmt(const struct token *);
static int		 is_simple_stmt(const struct token *);
static int		 peek_simple_stmt(struct parser *);

int
parser_stmt_peek(struct parser *pr)
{
	struct doc *dc;
	int error, simple;

	arena_scope(pr->pr_arena.doc, doc_scope);
	parser_arena_scope(&pr->pr_arena_scope.doc, &doc_scope, cookie);

	dc = doc_root(&doc_scope);
	simple = simple_disable(pr->pr_si);
	error = parser_stmt1(pr, dc);
	simple_enable(pr->pr_si, simple);
	return error & GOOD;
}

static int
parser_stmt(struct parser *pr, struct doc *dc)
{
	int error;

	simple_cookie(simple);
	if (peek_simple_stmt(pr)) {
		error = parser_simple_stmt_enter(pr, &simple);
		if (error & HALT)
			return error;
	}
	error = parser_stmt1(pr, dc);
	return error;
}

static int
parser_stmt1(struct parser *pr, struct doc *dc)
{
	struct parser_stmt_block_arg ps = {
		.head	= dc,
		.tail	= dc,
		.flags	= PARSER_STMT_BLOCK_TRIM,
	};

	/*
	 * Most likely statement comes first with some crucial exceptions:
	 *
	 *     1. Detect blocks before expressions as the expression parser can
	 *        also detect blocks through recovery using the parser.
	 *
	 *     2. Detect expressions before declarations as functions calls
	 *        would otherwise be treated as declarations. This in turn is
	 *        caused by parser_decl() being able to detect declarations
	 *        making use of preprocessor directives such as the ones
	 *        provided by queue(3).
	 */
	if ((parser_stmt_block(pr, &ps) & GOOD) ||
	    (parser_stmt_expr(pr, dc) & GOOD) ||
	    (parser_stmt_if(pr, dc) & GOOD) ||
	    (parser_stmt_return(pr, dc) & GOOD) ||
	    (parser_decl(pr, dc,
	     PARSER_DECL_BREAK | PARSER_DECL_SIMPLE |
	     PARSER_DECL_TRIM_SEMI) & GOOD) ||
	    (parser_stmt_case(pr, dc) & GOOD) ||
	    (parser_stmt_break(pr, dc) & GOOD) ||
	    (parser_stmt_goto(pr, dc) & GOOD) ||
	    (parser_stmt_for(pr, dc) & GOOD) ||
	    (parser_stmt_while(pr, dc) & GOOD) ||
	    (parser_stmt_label(pr, dc) & GOOD) ||
	    (parser_stmt_switch(pr, dc) & GOOD) ||
	    (parser_stmt_continue(pr, dc) & GOOD) ||
	    (parser_stmt_asm(pr, dc) & GOOD) ||
	    (parser_stmt_dowhile(pr, dc) & GOOD) ||
	    (parser_stmt_semi(pr, dc) & GOOD) ||
	    (parser_stmt_cpp(pr, dc) & GOOD))
		return parser_good(pr);
	return parser_none(pr);
}

static int
is_case_on_same_line(struct parser *pr, const struct token *pv)
{
	struct lexer *lx = pr->pr_lx;
	struct token *nx;

	if (pv->tk_type != TOKEN_CASE)
		return 0;
	if (!lexer_peek_if(lx, TOKEN_CASE, &nx))
		return 0;
	return token_cmp(pv, nx) == 0;
}

/*
 * Parse a block statement wrapped in braces.
 */
int
parser_stmt_block(struct parser *pr, struct parser_stmt_block_arg *arg)
{
	struct doc *dc = arg->tail;
	struct doc *concat, *indent, *line;
	struct lexer *lx = pr->pr_lx;
	struct token *lbrace, *nx, *pv, *rbrace, *tk;
	int isswitch = arg->flags & PARSER_STMT_BLOCK_SWITCH;
	int doindent = !isswitch && !is_simple_enabled(pr->pr_si, SIMPLE_STMT);
	int nstmt = 0;
	int error;

	if (!lexer_peek_if_pair(lx, TOKEN_LBRACE, TOKEN_RBRACE, &lbrace,
	    &rbrace))
		return parser_none(pr);

	/*
	 * Remove semi before emitting the right brace in order to honor
	 * optional lines.
	 */
	nx = token_next(rbrace);
	if (nx != NULL && nx->tk_type == TOKEN_SEMI)
		lexer_remove(lx, nx);

	if (doindent)
		pr->pr_stmt.depth++;

	if ((arg->flags & PARSER_STMT_BLOCK_EXPR_GNU) == 0 &&
	    is_simple_enabled(pr->pr_si, SIMPLE_STMT)) {
		dc = simple_stmt_braces_enter(pr->pr_simple.stmt, dc, lbrace,
		    rbrace, pr->pr_stmt.depth * style(pr->pr_st, IndentWidth));
	}

	parser_token_trim_before(pr, rbrace);

	if (!lexer_expect(lx, TOKEN_LBRACE, &lbrace))
		return parser_fail(pr);
	/*
	 * Optionally remove empty lines after the opening left brace.
	 * An empty line is however allowed in the beginning of a
	 * function implementation, a convention used when the function lacks
	 * local variables. But discard it if the following line is a
	 * declaration.
	 */
	if ((arg->flags & PARSER_STMT_BLOCK_TRIM) ||
	    (token_has_line(lbrace, 2) && parser_decl_peek(pr)))
		parser_token_trim_after(pr, lbrace);
	parser_doc_token(pr, lbrace, arg->head);

	if (isswitch)
		indent = dc;
	else
		indent = doc_indent(style(pr->pr_st, IndentWidth), dc);
	if ((arg->flags & PARSER_STMT_BLOCK_EXPR_GNU) == 0)
		line = doc_alloc(DOC_HARDLINE, indent);
	else
		line = doc_literal(" ", indent);
	if (!lexer_peek(lx, &pv))
		return parser_fail(pr);
	while ((error = parser_stmt(pr, indent)) & GOOD) {
		nstmt++;
		if (lexer_peek(lx, &tk) && tk == rbrace)
			break;
		if (isswitch && is_case_on_same_line(pr, pv))
			doc_literal(" ", indent);
		else
			doc_alloc(DOC_HARDLINE, indent);
		if (!lexer_peek(lx, &pv))
			return parser_fail(pr);
	}
	/* Do not keep the hard line if the statement block is empty. */
	if (nstmt == 0 && (error & BRCH) == 0)
		doc_remove(line, indent);

	if ((arg->flags & PARSER_STMT_BLOCK_EXPR_GNU) == 0)
		doc_alloc(DOC_HARDLINE, arg->tail);
	else
		doc_literal(" ", arg->tail);

	/*
	 * The right brace and any following statement is expected to fit on a
	 * single line.
	 */
	concat = doc_alloc(DOC_CONCAT, doc_alloc(DOC_GROUP, arg->tail));
	if (lexer_expect(lx, TOKEN_RBRACE, &tk)) {
		if (lexer_peek_if(lx, TOKEN_ELSE, NULL))
			parser_token_trim_after(pr, tk);
		parser_doc_token(pr, tk, concat);
	}
	if (lexer_peek_if(lx, TOKEN_SEMI, NULL))
		parser_semi(pr, concat);
	arg->rbrace = concat;

	if (doindent)
		pr->pr_stmt.depth--;

	return parser_good(pr);
}

static int
peek_else_if(struct parser *pr)
{
	struct lexer_state s;
	struct lexer *lx = pr->pr_lx;
	struct token *tkelse, *tkif;
	int peek;

	lexer_peek_enter(lx, &s);
	peek = lexer_if(lx, TOKEN_ELSE, &tkelse) &&
	    lexer_peek_if(lx, TOKEN_IF, &tkif) &&
	    token_cmp(tkelse, tkif) == 0;
	lexer_peek_leave(lx, &s);
	return peek;
}

static int
has_if_stmt_braces(struct parser *pr, int elseif)
{
	struct lexer_state ls;
	struct lexer *lx = pr->pr_lx;
	int peek = 0;

	lexer_peek_enter(lx, &ls);
	if ((elseif ? lexer_if(lx, TOKEN_ELSE, NULL) : 1) &&
	    lexer_if(lx, TOKEN_IF, NULL) &&
	    lexer_if_pair(lx, TOKEN_LPAREN, TOKEN_RPAREN, NULL, NULL) &&
	    lexer_if(lx, TOKEN_LBRACE, NULL))
		peek = 1;
	lexer_peek_leave(lx, &ls);
	return peek;
}

static int
parser_stmt_if(struct parser *pr, struct doc *dc)
{
	struct lexer *lx = pr->pr_lx;
	int has_braces;

	if (!lexer_peek_if(lx, TOKEN_IF, NULL))
		return parser_none(pr);

	has_braces = has_if_stmt_braces(pr, 0);
	if (parser_stmt_kw_expr(pr, dc, TOKEN_IF, 0) & (FAIL | NONE))
		return parser_fail(pr);

	while (lexer_peek_if(lx, TOKEN_ELSE, NULL)) {
		int error;

		if (has_braces)
			doc_literal(" ", dc);
		else
			doc_alloc(DOC_HARDLINE, dc);

		if (peek_else_if(pr)) {
			has_braces = has_if_stmt_braces(pr, 1);
			error = parser_stmt_kw_expr(pr, dc, TOKEN_IF,
			    PARSER_STMT_EXPR_ELSEIF);
			if (error & HALT)
				return parser_fail(pr);
		} else {
			struct token *tkelse;

			if (!lexer_expect(lx, TOKEN_ELSE, &tkelse))
				return parser_fail(pr);
			parser_doc_token(pr, tkelse, dc);
			doc_literal(" ", dc);

			if (lexer_peek_if(lx, TOKEN_LBRACE, NULL)) {
				error = parser_stmt(pr, dc);
				if (error & FAIL)
					return parser_fail(pr);
			} else {
				void *simple = NULL;

				dc = doc_indent(style(pr->pr_st, IndentWidth),
				    dc);
				doc_alloc(DOC_HARDLINE, dc);

				dc = parser_simple_stmt_no_braces_enter(pr, dc,
				    &simple);
				error = parser_stmt(pr, dc);
				parser_simple_stmt_no_braces_leave(pr, simple);
				if (error & (FAIL | NONE))
					return parser_fail(pr);
			}

			/* Terminate if/else chain. */
			break;
		}
	}

	return parser_good(pr);
}

static int
parser_stmt_for(struct parser *pr, struct doc *dc)
{
	struct lexer *lx = pr->pr_lx;
	struct doc *expr = NULL;
	struct doc *indent, *loop;
	struct token *semi = NULL;
	struct token *tk;
	void *simple = NULL;
	unsigned int w;
	int error;

	if (!lexer_if(lx, TOKEN_FOR, &tk))
		return parser_none(pr);

	loop = doc_alloc(DOC_CONCAT, doc_alloc(DOC_GROUP, dc));
	parser_doc_token(pr, tk, loop);
	doc_literal(" ", loop);

	if (lexer_expect(lx, TOKEN_LPAREN, &tk))
		parser_doc_token(pr, tk, loop);

	if (style(pr->pr_st, AlignOperands) == Align)
		w = parser_width(pr, dc);
	else
		w = style(pr->pr_st, ContinuationIndentWidth);

	/* Declarations are allowed in the first expression. */
	if (parser_decl(pr, loop, 0) & GOOD) {
		expr = loop;
	} else {
		error = parser_expr(pr, &expr, &(struct parser_expr_arg){
		    .dc		= loop,
		    .indent	= w,
		});
		if (error & (FAIL | BRCH))
			return parser_fail(pr);
		/* Let the semicolon hang of the expression unless empty. */
		if (error & NONE)
			expr = loop;
		if (lexer_expect(lx, TOKEN_SEMI, &semi))
			parser_doc_token(pr, semi, expr);
	}

	/* If the expression does not fit, break after the semicolon. */
	error = parser_expr(pr, &expr, &(struct parser_expr_arg){
	    .dc		= loop,
	    .indent	= w,
	    .flags	= EXPR_EXEC_LINE,
	});
	if (error & (FAIL | BRCH))
		return parser_fail(pr);
	if (error & NONE)
		expr = loop;
	if (!lexer_expect(lx, TOKEN_SEMI, &semi))
		return parser_fail(pr);
	parser_doc_token(pr, semi, expr);

	/* If the expression does not fit, break after the semicolon. */
	error = parser_expr(pr, &expr, &(struct parser_expr_arg){
	    .dc		= loop,
	    .indent	= w,
	    .flags	= EXPR_EXEC_LINE,
	});
	if (error & (FAIL | BRCH))
		return parser_fail(pr);
	if (error & NONE)
		expr = loop;
	if (lexer_expect(lx, TOKEN_RPAREN, &tk)) {
		parser_token_trim_after(pr, tk);
		parser_doc_token(pr, tk, expr);
	}

	simple_cookie(cookie);
	if (simple_enter(pr->pr_si, SIMPLE_STMT_EMPTY_LOOP, 0, &cookie)) {
		if (lexer_peek_if(lx, TOKEN_LBRACE, NULL))
			simple_stmt_empty_loop_braces(lx);
		else
			simple_stmt_empty_loop_no_braces(lx);
	}

	if (lexer_peek_if(lx, TOKEN_LBRACE, NULL)) {
		doc_literal(" ", expr);
		return parser_stmt(pr, dc);
	}

	indent = doc_indent(style(pr->pr_st, IndentWidth), dc);
	doc_alloc(DOC_HARDLINE, indent);
	indent = parser_simple_stmt_no_braces_enter(pr, indent, &simple);
	error = parser_stmt(pr, indent);
	parser_simple_stmt_no_braces_leave(pr, simple);
	if (error & (FAIL | NONE))
		return parser_fail(pr);
	return parser_good(pr);
}

static int
parser_stmt_dowhile(struct parser *pr, struct doc *dc)
{
	struct parser_stmt_block_arg ps = {
		.head	= dc,
		.tail	= dc,
		.flags	= PARSER_STMT_BLOCK_TRIM,
	};
	struct lexer *lx = pr->pr_lx;
	struct doc *concat = dc;
	struct token *tk;
	int error;

	if (!lexer_if(lx, TOKEN_DO, &tk))
		return parser_none(pr);

	parser_doc_token(pr, tk, concat);
	if (lexer_peek_if(lx, TOKEN_LBRACE, NULL)) {
		doc_literal(" ", concat);
		error = parser_stmt_block(pr, &ps);
		if (error & HALT)
			return parser_fail(pr);

		/*
		 * The following while statement is intended to fit on the same
		 * line as the right brace.
		 */
		concat = ps.rbrace;
		doc_literal(" ", concat);
	} else {
		struct doc *indent;

		indent = doc_indent(style(pr->pr_st, IndentWidth), concat);
		doc_alloc(DOC_HARDLINE, indent);
		error = parser_stmt(pr, indent);
		doc_alloc(DOC_HARDLINE, concat);
	}
	if (error & HALT)
		return parser_fail(pr);

	if (lexer_peek_if(lx, TOKEN_WHILE, NULL)) {
		return parser_stmt_kw_expr(pr, concat, TOKEN_WHILE,
		    PARSER_STMT_EXPR_DOWHILE);
	}
	return parser_fail(pr);
}

/*
 * Parse a statement consisting of a keyword, expression wrapped in parenthesis
 * and followed by additional nested statement(s).
 */
static int
parser_stmt_kw_expr(struct parser *pr, struct doc *dc, int token_type,
    unsigned int flags)
{
	struct doc *expr = NULL;
	struct doc *stmt;
	struct lexer *lx = pr->pr_lx;
	struct token *kw, *lparen, *prefix, *rparen;
	unsigned int w;
	int error;

	if (flags & PARSER_STMT_EXPR_ELSEIF) {
		if (!lexer_expect(lx, TOKEN_ELSE, &prefix))
			return parser_fail(pr);
	}

	if (!lexer_expect(lx, token_type, &kw) ||
	    !lexer_peek_if_pair(lx, TOKEN_LPAREN, TOKEN_RPAREN, &lparen,
	    &rparen))
		return parser_fail(pr);
	parser_token_trim_before(pr, rparen);
	parser_token_trim_after(pr, rparen);

	stmt = doc_alloc(DOC_CONCAT, doc_alloc(DOC_GROUP, dc));
	if (flags & PARSER_STMT_EXPR_ELSEIF) {
		parser_doc_token(pr, prefix, stmt);
		doc_literal(" ", stmt);
	}
	parser_doc_token(pr, kw, stmt);
	if (token_type != TOKEN_IDENT)
		doc_literal(" ", stmt);

	if (style(pr->pr_st, BasedOnStyle) == OpenBSD) {
		w = 0;
	} else {
		/*
		 * Usage of non-default style in which clang-format indentation
		 * is assumed causing the statement expression to be aligned
		 * with the left parenthesis.
		 *
		 * Note, must take note of the width before emitting the left
		 * parenthesis as it could be followed by comments which should
		 * not affect alignment.
		 */
		w = parser_width(pr, dc) + 1;
	}

	if (lexer_expect(lx, TOKEN_LPAREN, NULL)) {
		struct doc *optional = stmt;

		if (token_has_suffix(lparen, TOKEN_COMMENT)) {
			optional = doc_alloc(DOC_CONCAT,
			    doc_alloc(DOC_OPTIONAL, stmt));
		}
		parser_doc_token(pr, lparen, optional);
	}

	/*
	 * The tokens after the expression must be nested underneath the same
	 * expression since we want to fit everything until the following
	 * statement on a single line.
	 */
	error = parser_expr(pr, &expr, &(struct parser_expr_arg){
	    .dc		= stmt,
	    .stop	= rparen,
	    .indent	= style(pr->pr_st, ContinuationIndentWidth),
	    .align	= w,
	});
	if (error & (FAIL | BRCH))
		return parser_fail(pr);
	if (error & NONE)
		expr = stmt;
	if (lexer_expect(lx, TOKEN_RPAREN, &rparen)) {
		struct token *lbrace;

		/* Move suffixes if the left brace is about to move. */
		if (lexer_peek_if(lx, TOKEN_LBRACE, &lbrace) &&
		    token_cmp(rparen, lbrace) < 0)
			token_move_suffixes(rparen, lbrace);
		parser_doc_token(pr, rparen, expr);
	}

	if (flags & PARSER_STMT_EXPR_DOWHILE)
		return parser_semi(pr, expr);

	simple_cookie(cookie);
	if (is_loop_stmt(kw) &&
	    simple_enter(pr->pr_si, SIMPLE_STMT_EMPTY_LOOP, 0, &cookie)) {
		if (lexer_peek_if(lx, TOKEN_LBRACE, NULL))
			simple_stmt_empty_loop_braces(lx);
		else
			simple_stmt_empty_loop_no_braces(lx);
	}

	if (lexer_peek_if(lx, TOKEN_LBRACE, NULL)) {
		unsigned int parser_stmt_flags = 0;

		doc_literal(" ", expr);

		parser_stmt_flags |= PARSER_STMT_BLOCK_TRIM;
		if (token_type == TOKEN_SWITCH)
			parser_stmt_flags |= PARSER_STMT_BLOCK_SWITCH;
		return parser_stmt_block(pr, &(struct parser_stmt_block_arg){
		    .head	= expr,
		    .tail	= dc,
		    .flags	= parser_stmt_flags,
		});
	} else {
		struct doc *indent;
		void *simple = NULL;

		indent = doc_indent(style(pr->pr_st, IndentWidth), dc);
		doc_alloc(DOC_HARDLINE, indent);
		if (is_simple_stmt(kw)) {
			indent = parser_simple_stmt_no_braces_enter(pr, indent,
			    &simple);
		}
		error = parser_stmt(pr, indent);
		if (is_simple_stmt(kw))
			parser_simple_stmt_no_braces_leave(pr, simple);
		return error;
	}
}

static int
parser_stmt_label(struct parser *pr, struct doc *dc)
{
	struct lexer_state s;
	struct doc *noindent;
	struct lexer *lx = pr->pr_lx;
	struct token *colon = NULL;
	struct token *ident;
	int peek = 0;

	lexer_peek_enter(lx, &s);
	if (lexer_if(lx, TOKEN_IDENT, NULL) &&
	    lexer_if(lx, TOKEN_COLON, NULL))
		peek = 1;
	lexer_peek_leave(lx, &s);
	if (!peek)
		return parser_none(pr);

	noindent = doc_alloc(DOC_CONCAT, doc_alloc(DOC_NOINDENT, dc));
	if (lexer_expect(lx, TOKEN_IDENT, &ident)) {
		struct doc *label;

		label = parser_doc_token(pr, ident, noindent);
		/*
		 * Honor indentation before label but make sure to emit it right
		 * before the label. Necessary when the label is prefixed with
		 * comment(s).
		 */
		if (token_has_indent(ident)) {
			doc_move_before(doc_literal(" ", noindent), label,
			    noindent);
		}
	}

	if (lexer_expect(lx, TOKEN_COLON, &colon)) {
		struct token *nx;

		parser_doc_token(pr, colon, noindent);

		/*
		 * A label is not necessarily followed by a hard line, there
		 * could be another statement on the same line.
		 */
		if (lexer_peek(lx, &nx) && token_cmp(colon, nx) == 0) {
			struct doc *indent;

			indent = doc_indent(DOC_INDENT_FORCE, dc);
			return parser_stmt(pr, indent);
		}
	}

	return parser_good(pr);
}

static int
parser_stmt_case(struct parser *pr, struct doc *dc)
{
	struct doc *indent, *lhs;
	struct lexer *lx = pr->pr_lx;
	struct token *kw, *tk;

	if (!lexer_if(lx, TOKEN_CASE, &kw) && !lexer_if(lx, TOKEN_DEFAULT, &kw))
		return parser_none(pr);

	lhs = doc_alloc(DOC_CONCAT, doc_alloc(DOC_GROUP, dc));
	parser_doc_token(pr, kw, lhs);
	if (!lexer_peek_until(lx, TOKEN_COLON, NULL))
		return parser_fail(pr);
	if (kw->tk_type == TOKEN_CASE) {
		int error;

		doc_alloc(DOC_LINE, lhs);
		error = parser_expr(pr, NULL, &(struct parser_expr_arg){
		    .dc	= lhs,
		});
		if (error & HALT)
			return parser_fail(pr);
	}
	if (!lexer_expect(lx, TOKEN_COLON, &tk))
		return parser_fail(pr);
	parser_token_trim_after(pr, tk);
	parser_doc_token(pr, tk, lhs);

	if (lexer_peek_if(lx, TOKEN_LBRACE, NULL)) {
		doc_alloc(DOC_LINE, lhs);
		if (parser_stmt(pr, dc) & FAIL)
			return parser_fail(pr);
	}

	simple_cookie(cookie);
	if (simple_enter(pr->pr_si, SIMPLE_STMT_SWITCH, 0, &cookie))
		simple_stmt_switch(pr->pr_lx, kw);

	pr->pr_stmt.depth++;
	indent = doc_indent(style(pr->pr_st, IndentWidth), dc);
	for (;;) {
		struct doc *line;
		struct token *nx;

		if (lexer_peek_if(lx, TOKEN_CASE, NULL) ||
		    lexer_peek_if(lx, TOKEN_DEFAULT, NULL) ||
		    !lexer_peek(lx, &nx))
			break;

		/*
		 * Allow following statement(s) to be placed on the same line as
		 * the case/default keyword.
		 */
		if (token_cmp(kw, nx) == 0)
			line = doc_literal(" ", indent);
		else
			line = doc_alloc(DOC_HARDLINE, indent);

		if (parser_stmt(pr, indent) & HALT) {
			/* No statement, remove the line. */
			doc_remove(line, indent);
			break;
		}
	}
	pr->pr_stmt.depth--;

	return parser_good(pr);
}

static int
parser_stmt_goto(struct parser *pr, struct doc *dc)
{
	struct lexer *lx = pr->pr_lx;
	struct doc *concat;
	struct token *tk;

	if (!lexer_if(lx, TOKEN_GOTO, &tk))
		return parser_none(pr);

	concat = doc_alloc(DOC_CONCAT, doc_alloc(DOC_GROUP, dc));
	parser_doc_token(pr, tk, concat);
	doc_alloc(DOC_LINE, concat);
	if (lexer_expect(lx, TOKEN_IDENT, &tk))
		parser_doc_token(pr, tk, concat);
	return parser_semi(pr, concat);
}

static int
parser_stmt_switch(struct parser *pr, struct doc *dc)
{
	struct lexer *lx = pr->pr_lx;

	if (!lexer_peek_if(lx, TOKEN_SWITCH, NULL))
		return parser_none(pr);
	return parser_stmt_kw_expr(pr, dc, TOKEN_SWITCH, 0);
}

static int
parser_stmt_while(struct parser *pr, struct doc *dc)
{
	struct lexer *lx = pr->pr_lx;

	if (!lexer_peek_if(lx, TOKEN_WHILE, NULL))
		return parser_none(pr);
	return parser_stmt_kw_expr(pr, dc, TOKEN_WHILE, 0);
}

static int
parser_stmt_break(struct parser *pr, struct doc *dc)
{
	struct lexer *lx = pr->pr_lx;
	struct token *tk;

	if (!lexer_peek_if(lx, TOKEN_BREAK, NULL))
		return parser_none(pr);

	if (lexer_expect(lx, TOKEN_BREAK, &tk))
		parser_doc_token(pr, tk, dc);
	return parser_semi(pr, dc);
}

static int
parser_stmt_continue(struct parser *pr, struct doc *dc)
{
	struct lexer *lx = pr->pr_lx;
	struct token *tk;

	if (!lexer_peek_if(lx, TOKEN_CONTINUE, &tk))
		return parser_none(pr);

	if (lexer_expect(lx, TOKEN_CONTINUE, &tk))
		parser_doc_token(pr, tk, dc);
	return parser_semi(pr, dc);
}

static int
parser_stmt_return(struct parser *pr, struct doc *dc)
{
	struct lexer *lx = pr->pr_lx;
	struct doc *concat, *expr;
	struct token *tk;

	if (!lexer_if(lx, TOKEN_RETURN, &tk))
		return parser_none(pr);

	concat = doc_alloc(DOC_CONCAT, doc_alloc(DOC_GROUP, dc));
	parser_token_trim_after(pr, tk);
	parser_doc_token(pr, tk, concat);
	if (!lexer_peek_if(lx, TOKEN_SEMI, NULL)) {
		int error;

		doc_literal(" ", concat);
		error = parser_expr(pr, &expr, &(struct parser_expr_arg){
		    .dc		= concat,
		    .indent	= style(pr->pr_st, ContinuationIndentWidth),
		});
		if (error & HALT)
			return parser_fail(pr);
	} else {
		expr = concat;
	}
	return parser_semi(pr, expr);
}

static int
parser_stmt_semi(struct parser *pr, struct doc *dc)
{
	struct lexer *lx = pr->pr_lx;

	if (!lexer_peek_if(lx, TOKEN_SEMI, NULL))
		return parser_none(pr);
	return parser_semi(pr, dc);
}

static int
parser_stmt_cpp(struct parser *pr, struct doc *dc)
{
	struct lexer_state s;
	struct lexer *lx = pr->pr_lx;
	struct token *ident;
	int peek = 0;

	/* Statement hidden behind cpp, such as loop construct from queue(3). */
	lexer_peek_enter(lx, &s);
	if (lexer_if(lx, TOKEN_IDENT, &ident) &&
	    lexer_if_pair(lx, TOKEN_LPAREN, TOKEN_RPAREN, NULL, NULL) &&
	    !lexer_if(lx, TOKEN_SEMI, NULL))
		peek = 1;
	lexer_peek_leave(lx, &s);
	if (peek)
		return parser_stmt_kw_expr(pr, dc, TOKEN_IDENT, 0);

	/* Swith case fallthrough macro w/o semicolon. */
	if (lexer_peek_if(lx, TOKEN_IDENT, &ident) &&
	    clang_token_type(ident) == CLANG_TOKEN_FALLTHROUGH &&
	    lexer_expect(lx, TOKEN_IDENT, &ident)) {
		parser_doc_token(pr, ident, dc);
		return parser_good(pr);
	}

	return parser_none(pr);
}

/*
 * Called while entering a section of the source code with one or many
 * statements potentially wrapped in curly braces ahead. The statements
 * will silently be formatted in order to determine if each statement fits on a
 * single line, making the curly braces redundant and thus removed. Otherwise,
 * curly braces will be added around all covered statements for consistency.
 * Once this routine returns, parsing continues as usual.
 */
static int
parser_simple_stmt_enter(struct parser *pr, struct simple_cookie *simple)
{
	struct lexer_state s;
	struct doc *dc;
	struct lexer *lx = pr->pr_lx;
	int error;

	if (!simple_enter(pr->pr_si, SIMPLE_STMT, 0, simple))
		return parser_good(pr);

	arena_scope(pr->pr_arena.scratch, scratch_scope);

	arena_scope(pr->pr_arena.doc, doc_scope);
	parser_arena_scope(&pr->pr_arena_scope.doc, &doc_scope, cookie);

	pr->pr_simple.stmt = simple_stmt_enter(lx, pr->pr_st,
	    &scratch_scope, pr->pr_arena.scratch,
	    pr->pr_arena.buffer, pr->pr_op);
	dc = doc_root(&doc_scope);
	lexer_peek_enter(lx, &s);
	error = parser_stmt1(pr, dc);
	lexer_peek_leave(lx, &s);
	if (error & GOOD)
		simple_stmt_leave(pr->pr_simple.stmt);
	simple_stmt_free(pr->pr_simple.stmt);
	pr->pr_simple.stmt = NULL;
	simple_leave(simple);

	return parser_good(pr);
}

static struct doc *
parser_simple_stmt_no_braces_enter(struct parser *pr, struct doc *dc,
    void **cookie)
{
	struct lexer *lx = pr->pr_lx;
	struct token *lbrace;
	unsigned int indent;

	if (!is_simple_enabled(pr->pr_si, SIMPLE_STMT) ||
	    !lexer_peek(lx, &lbrace))
		return dc;

	indent = (pr->pr_stmt.depth + 1) * style(pr->pr_st, IndentWidth);
	return simple_stmt_no_braces_enter(pr->pr_simple.stmt, dc, lbrace,
	    indent, cookie);
}

static void
parser_simple_stmt_no_braces_leave(struct parser *pr, void *cookie)
{
	struct lexer *lx = pr->pr_lx;
	struct token *rbrace;

	if (cookie == NULL || !lexer_peek(lx, &rbrace))
		return;
	simple_stmt_no_braces_leave(pr->pr_simple.stmt, rbrace, cookie);
}

static int
is_loop_stmt(const struct token *tk)
{
	switch (tk->tk_type) {
	case TOKEN_WHILE:
	case TOKEN_IDENT:
		return 1;
	default:
		return 0;
	}
}

static int
is_simple_stmt(const struct token *tk)
{
	switch (tk->tk_type) {
	case TOKEN_IF:
	case TOKEN_FOR:
	case TOKEN_WHILE:
	case TOKEN_IDENT:
		return 1;
	default:
		return 0;
	}
}

static int
peek_simple_stmt(struct parser *pr)
{
	struct lexer_state s;
	struct lexer *lx = pr->pr_lx;
	struct token *tk;
	int peek = 0;

	if (!pr->pr_op->simple)
		return 0;

	lexer_peek_enter(lx, &s);
	if (lexer_pop(lx, &tk) && is_simple_stmt(tk) &&
	    lexer_if(lx, TOKEN_LPAREN, NULL))
		peek = 1;
	lexer_peek_leave(lx, &s);

	return peek;
}
