module Exfetch
  module Resource
    extend self

    # Class variables
    @@platform : String? = nil
    @@release : String? = nil
    # # Cache to use later
    @@uptime_info : String? = nil
    @@swap_info : Tuple(Int64, Int64)? = nil
    @@memory_info : Tuple(Int64, Int64)? = nil

    def get_platform
      @@platform ||=
        begin
          pretty =
            if File.exists?("/etc/os-release")
              Manip.scrape_file("PRETTY_NAME", "/etc/os-release").strip.presence
            end
          sysname =
            begin
              {% if flag?(:linux) %}
                "Linux"
              {% elsif flag?(:freebsd) %}
                "FreeBSD"
              {% elsif flag?(:openbsd) %}
                "OpenBSD"
              {% elsif flag?(:netbsd) %}
                "NetBSD"
              {% end %}
            end
          pretty ? "#{pretty} (#{sysname})" : sysname
        end
    end

    def get_release
      @@release ||=
        begin
          kernel =
            if get_platform.includes?("BSD") && Manip.has_command?("sysctl")
              Manip.run_command("sysctl", ["-n", "kern.osrelease"])
            elsif Manip.has_command?("uname")
              Manip.run_command("uname", ["-r"])
            else
              "unknown"
            end
          version = File.exists?("/etc/os-release") ? Manip.scrape_file("VERSION_ID", "/etc/os-release") : ""
          "#{kernel.strip} #{version}".strip
        end
    end

    private def get_load_avg_info
      if get_uptime_info =~ /load averages?: (\d+\.\d+), (\d+\.\d+), (\d+\.\d+)/
        [$1.to_f, $2.to_f, $3.to_f]
      else
        ["ERR", "ERR", "ERR"]
      end
    rescue
      ["ERR", "ERR", "ERR"]
    end

    def get_load_avg
      get_load_avg_info
        .zip(["1m", "5m", "15m"])
        .map { |(avg, lbl)| "#{lbl}: #{avg.to_s.ljust(4, '0')}" }
        .join(", ")
    end

    def get_user
      ENV["USER"]? || "unknown"
    end

    def get_host
      System.hostname
    end

    def get_shell
      File.basename(ENV["SHELL"]? || "unknown")
    end

    def get_terminal
      ENV["TERM"]? || "unknown"
    end

    def get_de : String?
      ENV["XDG_CURRENT_DESKTOP"]? || ENV["DESKTOP_SESSION"]? || nil # we can't "unknown" fallback here if there is no DE detected
    end

    WM_PROCESS_PATTERNS = {
      /catwm/     => "catwm",
      /fvwm/      => "fvwm",
      /dwm/       => "dwm",
      /2bwm/      => "2bwm",
      /monsterwm/ => "monsterwm",
      /wmaker/    => "Window Maker",
      /sowm/      => "sowm",
      /penrose/   => "penrose",
      /cluncwm/   => "cluncwm",
    }

    def get_wm : String?
      # 1st path: ps scan
      if Manip.has_command?("ps")
        begin
          Manip.run_command("ps", ["x"])
            .each_line
            .find { |line| WM_PROCESS_PATTERNS.keys.any? { |re| line =~ re } }
            .try { |line|
              WM_PROCESS_PATTERNS.find { |re, _| line =~ re }.try &.[1]
            }
            .tap { |wm| return wm if wm }
        rescue
          # Continue to xprop fallback if ps fails
        end
      end

      # fallback
      return nil unless Manip.has_command?("xprop") && ENV.has_key?("DISPLAY")

      id = begin
        Manip.run_command("xprop", ["-root", "-notype", "_NET_SUPPORTING_WM_CHECK"])
          .strip
          .split
          .last?
          .try { |v| v =~ /^0x[0-9a-fA-F]+$/ ? v : nil }
      rescue
        nil
      end

      return nil unless id

      begin
        Manip.run_command(
          "xprop",
          [
            "-id", id,
            "-notype",
            "-len", "25",
            "-f", "_NET_WM_NAME", "8t",
          ]
        ).try { |output| output =~ /_NET_WM_NAME.*= "(.*?)"/ ? $1 : nil }
      rescue
        nil
      end
    end

    def get_session
      de = get_de.presence
      wm = get_wm.presence

      if de && wm && de.downcase != wm.downcase
        "#{de} (#{wm})"
      elsif de
        de
      elsif wm
        wm
      else
        "unknown"
      end
    end

    private def get_swap_info
      @@swap_info ||= begin
        case get_platform
        when .includes?("Linux")
          meminfo =
            File.read_lines("/proc/meminfo")
              .map { |l| k, v = l.split(":", 2); {k.strip, v.strip.split.first.to_i64 * 1024} }
              .to_h

          total = meminfo["SwapTotal"]? || 0_i64
          free = meminfo["SwapFree"]? || 0_i64
          {total, total - free}
        else
          swap =
            if Manip.has_command?("swapctl")
              Manip.run_command("swapctl", ["-lk"])
                .each_line
                .reject(&.starts_with?("Device"))
                .map(&.split)
                .select { |p| p.size >= 3 }
                .map { |p| {p[1].to_i64, p[2].to_i64} }
                .reduce({0_i64, 0_i64}) { |(t, u), (tt, uu)| {t + tt, u + uu} }
            else
              {0_i64, 0_i64}
            end

          {swap[0] * 1024_i64, swap[1] * 1024_i64}
        end
      rescue
        {0_i64, 0_i64}
      end
    end

    def get_swap
      total, used = get_swap_info
      return "ERR" if (total == 0) && (used == 0)
      Manip.bytes_to_mebibytes(total)
    end

    def get_swap_usage
      total, used = get_swap_info
      return "ERR" if (total == 0) && (used == 0)
      Manip.bytes_to_mebibytes(used)
    end

    private def get_memory_info
      @@memory_info ||=
        begin
          case get_platform
          when .includes?("Linux")
            meminfo =
              File.read_lines("/proc/meminfo")
                .map { |l| k, v = l.split(":", 2); {k.strip, v.strip.split.first.to_i64 * 1024} }
                .to_h

            total = meminfo["MemTotal"]? || 0_i64
            free = meminfo["MemFree"]? || 0_i64
            buffers = meminfo["Buffers"]? || 0_i64
            cached = meminfo["Cached"]? || 0_i64

            used = total - free - buffers - cached
            {total, used}
          else
            pagesize =
              if Manip.has_command?("sysctl")
                Manip.run_command("sysctl", ["-n", "hw.pagesize"]).to_i64? || 4096_i64
              else
                4096_i64
              end

            total =
              if Manip.has_command?("sysctl")
                Manip.run_command("sysctl", ["-n", "hw.physmem"]).to_i64? || 0_i64
              else
                0_i64
              end

            active_pages =
              if Manip.has_command?("vmstat")
                Manip.run_command("vmstat", ["-s"])
                  .each_line
                  .find(&.includes?("pages active"))
                  .try { |l| l.split.first.to_i64 } || 0_i64
              else
                0_i64
              end

            {total, active_pages * pagesize}
          end
        rescue
          {0_i64, 0_i64}
        end
    end

    def get_memory
      total, used = get_memory_info
      return "ERR" if (total == 0) && (used == 0)
      Manip.bytes_to_mebibytes(total)
    end

    def get_memory_usage
      total, used = get_memory_info
      return "ERR" if (total == 0) && (used == 0)
      Manip.bytes_to_mebibytes(used)
    end

    def get_cpu : String
      model =
        begin
          if get_platform.includes?("Linux")
            File.read_lines("/proc/cpuinfo")
              .find(&.starts_with?("model name"))
              .try(&.split(":", 2).last.strip)
          elsif Manip.has_command?("sysctl")
            Manip.run_command("sysctl", ["-n", "hw.model"]).strip.presence
          else
            nil
          end
        rescue
          nil
        end

      (model || "unknown").gsub(/\s*\(.*?\)/, "")
    end

    private def get_gpu_info
      extract_value = ->(line : String) do
        line.split(":", 2)[1]?.try(&.strip).presence
      end

      gpu_lines = {
        "nvidia-smi" => {["-q"], /Product Name/},
        "glxinfo"    => {["-B"], /OpenGL renderer string/},
      }.flat_map do |cmd, (args, key_regex)|
        next [] of String unless Manip.has_command?(cmd)
        Manip.run_command(cmd, args)
          .lines
          .select(&.match(key_regex))
          .compact_map { |l| extract_value.call(l) }
      end

      gpu_lines
        .map { |g| g.gsub("Mesa", "").gsub(/\s*\([^)]*\)/, "").strip }
        .uniq
    end

    def get_gpu
      gpus = get_gpu_info
      gpus.empty? ? "unknown" : gpus.join(", ")
    end

    private def get_uptime_info
      @@uptime_info ||=
        if Manip.has_command?("uptime")
          Manip.run_command("uptime").strip
        else
          "unknown"
        end
    end

    def get_uptime
      if get_uptime_info =~ /\bup\b\s+(.+?),\s+\d+\s+users?/
        $1.strip
      else
        "unknown"
      end
    end

    def get_pkgs
      case get_platform
      when .includes?("Linux") then get_linux_pkgs.to_s
      when .includes?("BSD")   then get_bsd_pkgs.to_s
      else                          "unsupported"
      end
    end

    private def get_linux_pkgs
      used = false

      base =
        {
          {"dpkg-query", ["-f", ".\n", "-W"]},
          {"pacman", ["-Qq"]},
          {"rpm", ["-qa"]},
          {"apk", ["info"]},
          {"xbps-query", ["-l"]},
          {"brew", ["list"]},
        }.sum do |(cmd, args)|
          if Manip.has_command?(cmd)
            used = true
            Manip.command_lines_len(cmd, args)
          else
            0
          end
        end

      emerge =
        if Manip.has_command?("emerge")
          used = true
          Manip.dir_size("/var/db/pkg/", recursive: true)
        else
          0
        end

      nix =
        if Manip.has_command?("nix-store")
          used = true
          %w[
            /run/current-system/sw
            #{ENV["HOME"]}/.nix-profile
          ].sum do |path|
            Manip.command_lines_len("nix-store", ["-q", "--requisites", path])
          end
        else
          0
        end

      used ? (base + emerge + nix).to_s : "unsupported"
    end

    private def get_bsd_pkgs
      case get_platform
      when .includes?("FreeBSD")
        Manip.command_lines_len("pkg", ["info"])
      when .includes?("OpenBSD")
        Manip.dir_size("/var/db/pkg/", recursive: false)
      when .includes?("NetBSD")
        Manip.command_lines_len("pkg_info")
      else
        "unsupported"
      end
    end
  end
end
