require "option_parser"

module Exfetch
  class Options
    property lowercase = false
    property nerd_icons = false
    property right_justify = false
    property order : Array(Symbol) = Exfetch::CLI::SYMBOLS.dup
    property color = 4_u8 # Blue
    property ascii = "Tear"
    property separator = " -> "
    property padding = 1

    # Map to ANSI 16-color codes (0-15) supported by most terminals
    COLOR_MAP = {
      "black" => 0_u8, "red" => 1_u8, "green" => 2_u8, "yellow" => 3_u8,
      "blue" => 4_u8, "magenta" => 5_u8, "cyan" => 6_u8, "white" => 7_u8,
      "bright_black" => 8_u8, "bright_red" => 9_u8, "bright_green" => 10_u8,
      "bright_yellow" => 11_u8, "bright_blue" => 12_u8, "bright_magenta" => 13_u8,
      "bright_cyan" => 14_u8, "bright_white" => 15_u8,
    }

    def self.parse(args)
      options = new
      parser = OptionParser.new do |p|
        p.banner = "Usage: exfetch [options]"

        p.on("-l", "--lowercase", "Use lowercase labels \n\tDefault: uppercase") { options.lowercase = true }
        p.on("-n", "--nerd-icons", "Use Nerd Font icons \n\tDefault: not set") { options.nerd_icons = true }
        p.on("-r", "--right-justify", "Right justify labels \n\tDefault: left justified") { options.right_justify = true }
        p.on("-s SEP", "--separator=SEP", "Separator between labels and values \n\tDefault: \" -> \"") { |s| options.separator = s }

        p.on("-o ORDER", "--order=ORDER", "Specify label order (comma-separated)") do |o|
          symbol_map = Exfetch::CLI::SYMBOLS.each_with_object({} of String => Symbol) do |sym, hash|
            hash[sym.to_s] = sym
          end

          symbols = o.split(",").map(&.strip).map do |s|
            symbol_map.fetch(s) { STDERR.puts "Invalid label: #{s}"; exit 1 }
          end
          invalid = symbols.reject { |s| Exfetch::CLI::SYMBOLS.includes?(s) }
          if invalid.any?
            STDERR.puts "Invalid labels: #{invalid.join(", ")}"
            STDERR.puts "Available labels: #{Exfetch::CLI::SYMBOLS.join(", ")}"
            exit 1
          end
          options.order = symbols
        end
        p.on("-O", "--list-labels", "List available labels for ordering") do
          puts "Available labels: #{Exfetch::CLI::SYMBOLS.join(", ")}"
          exit
        end

        p.on("-c COLOR", "--color=COLOR", "Text color name or code (0-15) \n\tDefault: 4 (blue)") do |c|
          resolve_color(c, options)
        end
        p.on("-C", "--list-colors", "List available colors") { list_colors }

        p.on("-a ASCII", "--ascii=ASCII", "ASCII art name or file path \n\tDefault: \"Tear\"") do |a|
          options.ascii = resolve_ascii(a)
        end
        p.on("-A", "--list-ascii", "List built-in ASCII art") { list_ascii_art }
        p.on("-P NAME", "--preview-ascii=NAME", "Preview ASCII art") do |name|
          preview_ascii(name, options)
        end

        p.on("-p PADDING", "--padding=PADDING", "Padding around ASCII art (left and right) (only last digit used) \n\tDefault: 1") do |p|
          options.padding = (p.match(/(\d)\z/).try &.[1].to_i) || 1
        end

        p.on("-h", "--help", "Show help") { puts p; exit }

        p.invalid_option do |flag|
          STDERR.puts "Invalid option: #{flag}"
          STDERR.puts "See exfetch(1) or --help for usage."
          exit 1
        end

        p.missing_option do |flag|
          STDERR.puts "Missing argument for #{flag}"
          STDERR.puts "See exfetch(1) or --help for usage."
          exit 1
        end
      end

      parser.parse(args)
      options
    end

    private def self.resolve_color(c, options)
      options.color =
        begin
          if COLOR_MAP[c.downcase]?
            COLOR_MAP[c.downcase]
          elsif (code = c.to_u8? rescue nil) && (0..15).includes?(code)
            code
          else
            STDERR.puts "Invalid color: #{c}"
            list_colors
            exit 1
          end
        end
    end

    private def self.list_colors
      puts "Valid colors:"
      COLOR_MAP.each do |name, code|
        colored_name = CLI.colorize(name.ljust(12), code)
        colored_code = CLI.colorize(code.to_s.rjust(2), code)
        puts "  #{colored_code}\t#{colored_name}"
      end
      exit
    end

    private def self.resolve_ascii(a)
      File.exists?(a) ? a : CLI::ASCII_ART.has_key?(a) ? a : "Tear"
    rescue
      "Tear"
    end

    private def self.list_ascii_art
      puts "Built-in ASCII art:"
      CLI::ASCII_ART.each_key do |name|
        puts "  #{name}"
      end
      puts "\nUse --preview-ascii=NAME to preview built-in asciis"
      puts "Use --ascii=NAME to select one"
      puts "You can also pass a file"
      exit
    end

    def self.preview_ascii(name, options)
      ascii =
        CLI::ASCII_ART[name]? ||
          begin
            File.read_lines(name) if File.exists?(name)
          rescue error
            STDERR.puts "Error reading #{name}: #{error.message}"
            nil
          end

      unless ascii
        STDERR.puts "ASCII art '#{name}' not found (built-in or file)"
        list_ascii_art
        exit 1
      end

      preview_color = options.color
      puts "\n#{name} preview:"
      puts CLI.colorize(ascii.join("\n"), preview_color)
      exit
    end
  end
end
