require "baked_file_system" # to "bake" in the ascii data

module Exfetch
  class ASCIIBlob
    extend BakedFileSystem

    bake_folder "#{__DIR__}/ascii"
  end

  class CLI
    private record Info,
      symbol : Symbol,        # How should it be referred to in the code?
      nerd_icon : String,     # What nerd icon should it show?
      text_label : String,    # What label will it show in the fetch?
      resource : Proc(String) # What will it do?

    INFOS = [
      Info.new(:user, "", "USER", -> { "#{Resource.get_user}@#{Resource.get_host}" }),
      Info.new(:os, "", "OS", -> { Resource.get_platform }),
      Info.new(:ver, "", "VER", -> { Resource.get_release }),
      Info.new(:uptime, "", "UPTIME", -> { Resource.get_uptime }),
      Info.new(:dewm, "", "DE/WM", -> { Resource.get_session }),
      Info.new(:term, "", "TERM", -> { Resource.get_terminal }),
      Info.new(:shell, "", "SHELL", -> { Resource.get_shell }),
      Info.new(:cpu, "", "CPU", -> { Resource.get_cpu }),
      Info.new(:gpu, "", "GPU", -> { Resource.get_gpu }),
      Info.new(:pkgs, "", "PKGS", -> { Resource.get_pkgs }),
      Info.new(:load, "", "LOAD", -> { Resource.get_load_avg }),
      Info.new(:swap, "", "SWAP", -> { "#{Resource.get_swap_usage} / #{Resource.get_swap} MiB" }),
      Info.new(:mem, "", "MEM", -> { "#{Resource.get_memory_usage} / #{Resource.get_memory} MiB" }),
    ]

    NERD_ICONS = INFOS.each_with_object({} of Symbol => String) do |info, hash|
      hash[info.symbol] = info.nerd_icon
    end

    TEXT_LABELS = INFOS.each_with_object({} of Symbol => String) do |info, hash|
      hash[info.symbol] = info.text_label
    end

    SYMBOLS = INFOS.map(&.symbol)

    {% if files = env("ASCII_FILES") %}
      ASCII_FILE_NAMES = {{ files.split('\n').map(&.strip) }}
    {% else %}
      ASCII_FILE_NAMES = [] of String
    {% end %}

    ASCII_ART =
      begin
        ascii = {} of String => Array(String)
        {% for name in ASCII_FILE_NAMES %}
          {% key = name.stringify[1..-2].gsub(/\.ascii$/, "") %}
          ascii[{{key}}] = ASCIIBlob.get("{{name.id}}").gets_to_end.lines
        {% end %}
        ascii
      rescue
        {"Tear" => [
          "   ,   ",
          "  / \\  ",
          " /   \\ ",
          "|     |",
          " \\___/ ",
        ]}
      end

    def self.run(options : Options)
      infos =
        options.order.compact_map do |sym|
          INFOS.find(&.symbol.==(sym))
        end

      channel = Channel(Tuple(Int32, String)).new(infos.size)

      infos.each_with_index do |info, i|
        spawn do
          begin
            channel.send({i, info.resource.call})
          rescue ex
            channel.send({i, "Error: #{ex.message}"})
          end
        end
      end

      results = Array(String).new(infos.size, "")
      infos.size.times do
        i, value = channel.receive
        results[i] = value
      end

      labels =
        (options.nerd_icons ? infos.map(&.nerd_icon) : infos.map(&.text_label))
          .tap { |ls| ls.map!(&.downcase) if options.lowercase && !options.nerd_icons }

      ascii = load_ascii(options.ascii)
      max_label = labels.max_of(&.size)

      # render it out
      puts render(ascii, labels, results, options, max_label)
    end

    def self.render(
      ascii : Array(String),
      labels : Array(String),
      results : Array(String),
      options : Options,
      max_label : Int32,
    ) : String
      pad = " " * options.padding
      ascii_w = ascii.any? ? ascii.max_of(&.size) + options.padding * 2 : 0
      lines = {ascii.size, options.order.size}.max

      String.build do |io|
        io << '\n'

        lines.times do |i|
          ascii_line =
            (pad +
              (ascii[i]? || "") +
              pad
              ).ljust(ascii_w)

          info_line =
            if key = options.order[i]?
              label =
                options.right_justify ? labels[i].rjust(max_label) : labels[i].ljust(max_label)

              label_color = options.color | 8

              colorize(label, label_color) +
                options.separator +
                results[i]
            else
              ""
            end

          io <<
            colorize(ascii_line, options.color) <<
            info_line <<
            '\n'
        end

        io << '\n'
      end
    end

    def self.colorize(text : String, color : UInt8) : String
      base_code = 30
      color_index = color % 8
      is_bright = color >= 8

      intensity = is_bright ? "1;" : ""
      code = base_code + color_index

      "\e[#{intensity}#{code}m#{text}\e[0m"
    end

    private def self.load_ascii(source) : Array(String)
      if File.exists?(source)
        return File.read_lines(source)
      end

      ASCII_ART.fetch(source) { ASCII_ART["Tear"] }
    rescue
      ASCII_ART["Tear"]
    end
  end
end
