require "ecr/macros" # used to template the output
require "baked_file_system" # to "bake" in the ascii data

module Exfetch
  class ASCIIBlob
    extend BakedFileSystem

    bake_folder "#{__DIR__}/ascii"
  end

  class RenderTemplate
    @ascii : Array(String)
    @labels : Array(String)
    @results : Hash(Symbol, String)
    @options : Options
    @max_label : Int32

    def initialize(@ascii, @labels, @results, @options, @max_label)
    end

    ECR.def_to_s "src/exfetch/templates/default.ecr"
  end

  class CLI
    private record Info,
                   symbol : Symbol,        # How should it be referred to in the code?
                   nerd_icon : String,     # What nerd icon should it show?
                   text_label : String,    # What label will it show in the fetch?
                   resource : Proc(String) # What will it do?

    INFOS = [
      Info.new(:user, "", "USER", ->{ "#{Resource.get_user}@#{Resource.get_host}" }),
      Info.new(:os, "", "OS", ->{ Resource.get_platform }),
      Info.new(:ver, "", "VER", ->{ Resource.get_release }),
      Info.new(:uptime, "", "UPTIME", ->{ Resource.get_uptime }),
      Info.new(:dewm, "", "DE/WM", ->{ Resource.get_session }),
      Info.new(:term, "", "TERM", ->{ Resource.get_terminal }),
      Info.new(:shell, "", "SHELL", ->{ Resource.get_shell }),
      Info.new(:cpu, "", "CPU", ->{ Resource.get_cpu }),
      Info.new(:gpu, "", "GPU", ->{ Resource.get_gpu }),
      Info.new(:pkgs, "", "PKGS", ->{ Resource.get_pkgs }),
      Info.new(:load, "", "LOAD", ->{ Resource.get_load_avg }),
      Info.new(:swap, "", "SWAP", ->{ "#{Resource.get_swap_usage} / #{Resource.get_swap} MiB" }),
      Info.new(:mem, "", "MEM", ->{ "#{Resource.get_memory_usage} / #{Resource.get_memory} MiB" }),
    ]

    NERD_ICONS = INFOS.each_with_object({} of Symbol => String) do |info, hash|
      hash[info.symbol] = info.nerd_icon
    end

    TEXT_LABELS = INFOS.each_with_object({} of Symbol => String) do |info, hash|
      hash[info.symbol] = info.text_label
    end

    SYMBOLS = INFOS.map(&.symbol)

    {% if files = env("ASCII_FILES") %}
      ASCII_FILE_NAMES = {{ files.split('\n').map(&.strip) }}
    {% else %}
      ASCII_FILE_NAMES = [] of String
    {% end %}

    ASCII_ART =
      begin
        ascii = {} of String => Array(String)
        {% for name in ASCII_FILE_NAMES %}
          {% key = name.stringify[1..-2].gsub(/\.ascii$/, "") %}
          ascii[{{key}}] = ASCIIBlob.get("{{name.id}}").gets_to_end.lines
        {% end %}
        ascii
      rescue
        {"Tear" => [
           "   ,   ",
           "  / \\  ",
           " /   \\ ",
           "|     |",
           " \\___/ ",
         ]
        }
      end

    def self.run(options : Options)
      info_map = INFOS.each_with_object({} of Symbol => Info) do |info, hash|
        hash[info.symbol] = info
      end

      resources = options.order.compact_map do |sym|
        if info = info_map[sym]?
          {sym, info.resource}
        end
      end.to_h

      # Concurrently fetch all resources using spawn/channels
      channel = Channel(Tuple(Symbol, String)).new(resources.size)
      resources.each do |key, proc|
        spawn do
          begin
            channel.send({key, proc.call})
          rescue ex
            channel.send({key, "Error: #{ex.message}"})
          end
        end
      end

      results = {} of Symbol => String
      resources.size.times do
        key, value = channel.receive
        results[key] = value
      end

      labels =
        if options.nerd_icons
          options.order.map { |sym| NERD_ICONS[sym] }
        else
          options.order.map { |sym| TEXT_LABELS[sym] }
        end
      labels = labels.map(&.downcase) if options.lowercase && !options.nerd_icons
      max_label = labels.max_of(&.size)

      ascii = load_ascii(options.ascii)
      ascii_width = ascii.any? ? (ascii.max_of(&.size) + options.padding) * 2 : 0

      # render it out
      puts RenderTemplate.new(ascii, labels, results, options, max_label).to_s
    end

    private def self.load_ascii(source) : Array(String)
      if File.exists?(source)
        return File.read_lines(source)
      end

      ASCII_ART.fetch(source) { ASCII_ART["Tear"] }
    rescue
      ASCII_ART["Tear"]
    end

    def self.colorize(text : String, color : UInt8) : String
      base_code = 30
      color_index = color % 8
      is_bright = color >= 8

      intensity = is_bright ? "1;" : ""
      code = base_code + color_index

      "\e[#{intensity}#{code}m#{text}\e[0m"
    end

    def self.pad_ascii(line : String, width : Int32, padding : Int32) : String
      padded = " " * padding + line + " " * padding
      padded.ljust(width)
    end

    def self.bright_label(label : String, width : Int32, right_justify : Bool, color : UInt8) : String
      # If right_justify truthy, rjust width. otherwise, assume ljust
      justified = right_justify ? label.rjust(width) : label.ljust(width)
      # ANSI bright & bold: 90–97 (bright) and 1 (bold). For 0–7, add 90 to get bright. For 8–15, modulo 8 and add 90 again
      bright_base = 90 + (color % 8)
      "\e[1;#{bright_base}m#{justified}\e[0m"
    end
  end
end
