-- autogenerated from https://www.unicode.org/Public/17.0.0/ucd/UnicodeData.txt
-- |
-- Module      : Unicode.Internal.Char.UnicodeData.GeneralCategory
-- Copyright   : (c) 2020 Composewell Technologies and Contributors
-- License     : Apache-2.0
-- Maintainer  : streamly@composewell.com
-- Stability   : experimental

{-# OPTIONS_HADDOCK hide #-}
{-# LANGUAGE PatternSynonyms #-}

module Unicode.Internal.Char.UnicodeData.GeneralCategory
( -- * Lookup functions
  generalCategory
, generalCategoryPlanes0To3

  -- * General categories
, pattern UppercaseLetter
, pattern LowercaseLetter
, pattern TitlecaseLetter
, pattern ModifierLetter
, pattern OtherLetter
, pattern NonSpacingMark
, pattern SpacingCombiningMark
, pattern EnclosingMark
, pattern DecimalNumber
, pattern LetterNumber
, pattern OtherNumber
, pattern ConnectorPunctuation
, pattern DashPunctuation
, pattern OpenPunctuation
, pattern ClosePunctuation
, pattern InitialQuote
, pattern FinalQuote
, pattern OtherPunctuation
, pattern MathSymbol
, pattern CurrencySymbol
, pattern ModifierSymbol
, pattern OtherSymbol
, pattern Space
, pattern LineSeparator
, pattern ParagraphSeparator
, pattern Control
, pattern Format
, pattern Surrogate
, pattern PrivateUse
, pattern NotAssigned

  -- * Characters bounds for predicates
, pattern MaxIsLetter
, pattern MaxIsAlphaNum
, pattern MaxIsLower
, pattern MaxIsUpper
, pattern MaxIsNumber
, pattern MaxIsSpace
, pattern MaxIsSeparator
) where

import Data.Bits (Bits(..))
import Data.Char (ord)
import Data.Int (Int8)
import Data.Word (Word16)
import GHC.Exts (Ptr(..))
import Unicode.Internal.Bits (lookupWord16AsInt, lookupWord8AsInt)

--------------------------------------------------------------------------------
-- General category patterns
--------------------------------------------------------------------------------

-- | General category Lu
pattern UppercaseLetter :: Int
pattern UppercaseLetter = 0

-- | General category Ll
pattern LowercaseLetter :: Int
pattern LowercaseLetter = 1

-- | General category Lt
pattern TitlecaseLetter :: Int
pattern TitlecaseLetter = 2

-- | General category Lm
pattern ModifierLetter :: Int
pattern ModifierLetter = 3

-- | General category Lo
pattern OtherLetter :: Int
pattern OtherLetter = 4

-- | General category Mn
pattern NonSpacingMark :: Int
pattern NonSpacingMark = 5

-- | General category Mc
pattern SpacingCombiningMark :: Int
pattern SpacingCombiningMark = 6

-- | General category Me
pattern EnclosingMark :: Int
pattern EnclosingMark = 7

-- | General category Nd
pattern DecimalNumber :: Int
pattern DecimalNumber = 8

-- | General category Nl
pattern LetterNumber :: Int
pattern LetterNumber = 9

-- | General category No
pattern OtherNumber :: Int
pattern OtherNumber = 10

-- | General category Pc
pattern ConnectorPunctuation :: Int
pattern ConnectorPunctuation = 11

-- | General category Pd
pattern DashPunctuation :: Int
pattern DashPunctuation = 12

-- | General category Ps
pattern OpenPunctuation :: Int
pattern OpenPunctuation = 13

-- | General category Pe
pattern ClosePunctuation :: Int
pattern ClosePunctuation = 14

-- | General category Pi
pattern InitialQuote :: Int
pattern InitialQuote = 15

-- | General category Pf
pattern FinalQuote :: Int
pattern FinalQuote = 16

-- | General category Po
pattern OtherPunctuation :: Int
pattern OtherPunctuation = 17

-- | General category Sm
pattern MathSymbol :: Int
pattern MathSymbol = 18

-- | General category Sc
pattern CurrencySymbol :: Int
pattern CurrencySymbol = 19

-- | General category Sk
pattern ModifierSymbol :: Int
pattern ModifierSymbol = 20

-- | General category So
pattern OtherSymbol :: Int
pattern OtherSymbol = 21

-- | General category Zs
pattern Space :: Int
pattern Space = 22

-- | General category Zl
pattern LineSeparator :: Int
pattern LineSeparator = 23

-- | General category Zp
pattern ParagraphSeparator :: Int
pattern ParagraphSeparator = 24

-- | General category Cc
pattern Control :: Int
pattern Control = 25

-- | General category Cf
pattern Format :: Int
pattern Format = 26

-- | General category Cs
pattern Surrogate :: Int
pattern Surrogate = 27

-- | General category Co
pattern PrivateUse :: Int
pattern PrivateUse = 28

-- | General category Cn
pattern NotAssigned :: Int
pattern NotAssigned = 29

--------------------------------------------------------------------------------
-- Characters bounds for predicates
--------------------------------------------------------------------------------

-- | Maximum codepoint satisfying @isLetter@
pattern MaxIsLetter :: Int
pattern MaxIsLetter = 0x33479

-- | Maximum codepoint satisfying @isAlphaNum@
pattern MaxIsAlphaNum :: Int
pattern MaxIsAlphaNum = 0x33479

-- | Maximum codepoint satisfying @isLower@
pattern MaxIsLower :: Int
pattern MaxIsLower = 0x1E943

-- | Maximum codepoint satisfying @isUpper@
pattern MaxIsUpper :: Int
pattern MaxIsUpper = 0x1E921

-- | Maximum codepoint satisfying @isNumber@
pattern MaxIsNumber :: Int
pattern MaxIsNumber = 0x1FBF9

-- | Maximum codepoint satisfying @isSpace@
pattern MaxIsSpace :: Int
pattern MaxIsSpace = 0x3000

-- | Maximum codepoint satisfying @isSeparator@
pattern MaxIsSeparator :: Int
pattern MaxIsSeparator = 0x3000

--------------------------------------------------------------------------------
-- Lookup functions
--------------------------------------------------------------------------------

-- | Return the general category of a code point in planes 0 to 3
--
-- The caller of this function must ensure its parameter is \< @0x40000@.
{-# INLINE generalCategoryPlanes0To3 #-}
generalCategoryPlanes0To3 :: Int -> Int
generalCategoryPlanes0To3 = lookupGeneralCategoryBitMap

-- | Return the general category of a character
{-# INLINE generalCategory #-}
generalCategory :: Char -> Int
generalCategory c
    -- Planes 0-3
    | cp < 0x3347A = lookupGeneralCategoryBitMap cp
    -- Planes 4-13: Cn
    | cp < 0xE0000 = NotAssigned
    -- Plane 14
    | cp < 0xE01F0 = lookupGeneralCategoryBitMap (cp - 0xACB86)
    -- Plane 14: Cn
    | cp < 0xF0000 = NotAssigned
    -- Plane 15: Co
    | cp < 0xFFFFE = PrivateUse
    -- Plane 15: Cn
    | cp < 0x100000 = NotAssigned
    -- Plane 16: Co
    | cp < 0x10FFFE = PrivateUse
    -- Default: Cn
    | otherwise = NotAssigned
    where
    cp = ord c

{-# INLINE lookupGeneralCategoryBitMap #-}
lookupGeneralCategoryBitMap :: Int -> Int
lookupGeneralCategoryBitMap n =
    lookupWord8AsInt data# (
        lookupWord16AsInt offsets1# (
            lookupWord16AsInt offsets2# (
                n `shiftR` 8
            ) + ((n `shiftR` 3) .&. maskOffsets)
        ) + (n .&. maskData)
    )
    where
    maskData = (1 `shiftL` 3) - 1
    maskOffsets = (1 `shiftL` 5) - 1
    !(Ptr data#) = generalCategoryDataBitMap
    !(Ptr offsets1#) = generalCategoryOffsets1BitMap
    !(Ptr offsets2#) = generalCategoryOffsets2BitMap

generalCategoryDataBitMap :: Ptr Int8
generalCategoryDataBitMap = Ptr
    "\17\15\16\17\15\16\17\17\15\16\15\16\17\17\15\16\17\15\16\17\17\17\17\17\17\17\12\12\17\17\17\17\17\17\29\3\18\17\17\19\17\17\21\21\10\10\21\21\21\29\
    \\29\29\29\21\21\21\29\29\29\21\21\21\21\20\20\20\20\20\21\4\15\18\26\21\20\10\4\16\10\10\10\17\19\17\17\29\29\29\29\5\5\5\5\5\3\3\5\5\5\5\
    \\3\3\21\17\0\0\0\0\0\0\0\0\18\1\1\1\1\1\1\29\29\1\1\1\29\1\1\29\29\29\4\4\4\4\4\6\5\6\4\4\4\4\4\4\9\9\9\9\9\9\
    \\9\29\20\1\1\1\1\1\1\1\1\18\1\1\1\1\4\4\1\1\21\18\21\21\1\21\18\18\18\18\21\21\29\8\8\17\17\29\17\5\4\4\4\4\4\5\3\4\1\21\
    \\21\1\1\0\0\1\1\0\0\0\1\0\1\0\1\0\1\0\0\0\0\1\4\4\4\4\0\2\1\0\1\29\29\29\29\29\5\29\5\5\29\5\5\29\5\5\5\5\29\29\
    \\29\29\29\7\7\0\1\0\1\0\1\1\1\0\1\0\1\0\1\0\1\1\1\1\1\3\3\0\0\0\0\2\20\1\20\0\0\0\0\0\29\0\29\0\0\0\0\29\0\0\
    \\0\0\29\29\0\0\0\0\0\0\0\29\0\0\0\0\0\0\29\29\29\29\29\29\8\8\6\6\6\5\21\21\0\21\21\21\21\0\1\0\1\0\29\29\29\12\12\12\12\12\
    \\12\17\17\10\10\10\10\10\10\10\10\4\4\5\5\6\17\17\29\5\6\4\5\5\5\5\17\17\17\3\4\4\4\4\4\4\4\3\5\5\5\5\5\5\5\6\5\5\5\4\
    \\4\4\4\5\4\4\6\6\29\29\5\5\6\5\4\5\5\5\4\4\5\5\4\29\29\29\29\29\12\17\13\17\17\17\17\17\12\12\11\11\13\14\13\14\18\18\18\18\18\18\
    \\18\18\13\14\18\18\21\21\21\21\21\21\10\10\10\10\21\21\4\6\5\6\4\4\5\5\17\17\8\8\10\29\29\29\21\21\12\13\14\13\14\13\14\17\29\29\29\29\29\29\
    \\29\3\1\1\1\17\0\23\24\26\26\26\26\26\22\17\19\19\19\19\21\17\17\29\29\29\29\29\29\29\5\5\6\6\6\6\29\5\5\6\29\4\4\4\29\4\4\4\17\17\
    \\29\29\29\29\29\4\5\4\4\4\4\4\5\5\6\5\5\6\5\5\29\6\6\5\6\5\17\17\17\17\17\17\5\4\5\4\17\17\29\17\17\17\17\29\29\29\4\4\29\29\
    \\5\4\5\5\21\21\21\21\21\21\5\21\21\17\0\29\29\1\1\1\1\1\3\3\3\3\3\3\3\3\20\20\20\20\20\20\20\20\3\20\3\20\20\0\1\0\1\4\4\5\
    \\5\29\29\29\29\22\22\22\22\22\22\22\22\26\26\26\26\26\29\29\29\29\29\5\5\5\5\5\5\5\12\5\5\5\5\5\4\29\29\4\4\29\5\5\4\6\6\17\17\17\
    \\17\17\17\17\17\29\26\26\29\29\29\29\29\29\4\4\29\29\5\4\6\6\5\5\5\5\29\29\6\5\5\5\5\5\5\5\3\5\5\5\5\5\29\29\5\5\5\29\29\4\
    \\4\4\3\4\4\29\29\29\29\29\17\0\1\0\1\3\20\0\1\0\1\0\1\4\5\6\6\29\4\4\4\5\6\6\5\5\5\29\29\29\29\5\4\29\4\4\4\4\29\4\
    \\4\29\4\4\4\4\4\4\4\4\29\4\4\4\4\29\29\29\29\4\4\29\4\29\4\4\29\4\4\29\4\29\29\4\4\4\4\3\17\17\17\5\21\5\5\5\5\4\4\4\
    \\4\5\5\5\5\17\17\17\4\29\29\29\4\29\29\4\4\4\4\4\4\29\29\29\29\29\29\29\29\6\6\29\6\6\6\5\29\29\5\5\5\5\5\5\4\5\4\4\5\5\
    \\5\5\4\4\4\4\4\4\5\6\5\5\5\5\5\5\5\5\29\29\29\29\19\21\29\29\29\29\29\29\1\1\29\1\29\1\1\1\17\3\1\1\1\20\3\3\3\3\1\1\
    \\1\1\1\1\1\0\1\18\0\0\0\0\0\21\21\26\26\26\26\26\26\18\18\18\18\18\21\21\18\21\21\18\21\18\21\21\21\21\21\21\21\18\21\21\21\21\21\6\6\5\
    \\5\5\6\5\4\17\4\6\29\29\29\15\16\13\15\15\16\13\15\16\13\14\13\14\13\14\13\14\13\14\10\10\18\18\18\13\14\29\5\5\6\6\29\29\29\29\3\17\17\17\
    \\17\17\17\21\21\21\21\21\21\21\21\29\29\29\29\29\29\29\17\17\17\17\17\17\17\3\29\29\5\19\19\29\29\29\29\29\29\2\2\2\2\2\2\2\2\1\0\2\1\0\
    \\1\0\0\0\0\0\1\0\1\0\1\1\0\1\1\0\1\0\0\0\1\1\0\0\0\1\1\1\13\18\14\18\13\14\17\17\17\17\17\3\3\3\3\17\21\29\29\18\18\18\
    \\18\18\21\21\21\4\4\4\4\4\4\14\13\14\13\14\17\17\13\3\3\3\3\29\3\3\29\3\3\3\3\3\3\29\3\3\3\3\0\1\4\4\4\4\5\6\29\29\1\18\
    \\1\1\1\1\1\1\0\0\0\0\0\0\0\1\1\1\1\1\1\29\1\1\1\1\1\0\29\0\0\0\0\29\20\20\20\4\4\9\29\29\29\29\29\4\5\5\5\5\5\5\
    \\5\21\21\21\21\0\21\0\21\0\0\0\0\21\1\3\20\20\29\29\29\29\4\5\5\5\29\5\5\29\4\29\4\29\4\29\4\4\4\4\29\4\29\4\29\29\29\29\20\20\
    \\0\17\8\8\10\10\17\17\17\21\18\10\10\20\1\17\17\18\12\18\18\18\29\29\29\29\29\29\21\21\4\5\5\5\5\5\5\29\8\8\29\10\10\10\10\10\10\8\8\21\
    \\21\21\29\29\29\21\21\21\21\21\21\21\18\18\18\18\18\0\1\1\1\1\0\0\29\0\29\0\29\0\29\0\4\4\4\5\17\17\4\29\4\29\29\29\4\4\29\29\29\29\
    \\6\6\17\29\29\29\29\29\29\18\18\18\18\21\21\21\21\12\4\4\4\4\4\4\4\21\17\4\4\4\4\4\4\29\17\22\4\4\4\4\4\4\4\21\17\4\4\4\4\4\
    \\10\10\17\5\6\6\6\5\4\4\4\29\4\4\29\29\5\4\6\5\5\5\5\6\6\5\6\5\5\5\6\5\6\6\6\6\5\5\5\5\5\6\6\6\6\5\6\6\6\6\
    \\6\29\5\6\6\29\4\4\4\4\17\4\5\5\5\29\29\17\29\3\3\1\4\4\4\4\4\29\29\26\26\26\26\26\26\26\26\29\26\26\26\21\21\29\29\29\17\17\8\8\
    \\4\4\29\29\29\29\18\18\18\18\18\18\13\14\13\14\13\14\12\29\29\20\20\0\1\0\1\0\1\0\0\1\0\1\0\0\0\0\0\20\20\20\4\5\6\6\17\4\4\4\
    \\5\4\4\5\4\4\4\5\4\4\4\4\4\4\4\5\5\5\5\5\5\6\6\5\5\6\6\5\5\4\4\4\5\17\29\4\5\4\29\29\29\29\29\29\1\1\1\1\1\1\
    \\1\29\29\29\29\29\5\5\29\29\29\29\29\0\29\29\0\0\29\4\18\4\4\4\4\4\4\29\4\29\29\4\29\29\29\29\4\4\4\6\6\5\5\6\6\5\5\29\29\29\
    \\29\5\29\29\29\29\6\6\3\3\9\9\9\29\4\4\4\4\4\29\29\10\21\4\4\4\4\4\4\4\10\10\10\10\29\29\29\17\8\8\29\29\29\29\4\21\4\4\4\4\
    \\4\4\4\5\5\29\29\8\8\29\29\17\17\17\17\4\6\6\4\6\5\17\17\17\29\17\17\17\17\4\3\3\6\5\29\6\29\29\6\29\6\6\5\5\29\29\5\5\5\17\
    \\26\17\17\17\5\29\29\29\29\29\26\29\29\8\8\29\29\29\29\29\19\21\21\21\21\21\21\21\19\19\19\19\19\19\19\19\18\20\21\19\19\29\8\8\4\4\4\4\4\29\
    \\29\4\29\29\29\29\29\29\6\8\8\29\29\4\4\4\4\29\29\5\5\5\21\21\5\5\5\5\21\21\21\6\6\6\6\6\6\6\6\4\4\4\4\17\17\17\19\4\5\29\
    \\29\29\29\29\29\5\5\5\5\5\29\21\21\1\0\0\0\1\1\4\0\1\1\1\0\0\1\0\0\1\0\0\0\0\2\20\20\29\4\4\29\5\5\12\29\29\4\4\6\29\
    \\29\29\29\4\4\4\29\29\29\4\4\3\17\17\8\8\10\10\10\10\10\10\21\5\21\5\10\10\10\10\17\17\17\8\8\4\4\4\4\4\4\3\4\4\4\4\4\5\5\4\
    \\6\6\6\4\4\6\6\6\6\6\29\6\6\5\6\29\6\6\5\29\29\5\6\6\5\29\29\17\17\18\17\11\17\17\17\18\13\14\17\18\17\12\17\17\0\0\0\1\21\0\
    \\21\21\10\10\10\10\10\10\21\21\21\21\21\21\21\29\21\21\21\21\21\17\17\17\13\14\13\14\21\21\21\21\13\14\21\21\13\14\13\14\12\13\14\14\17\17\17\17\11\11\
    \\11\7\7\7\17\5\5\5\5\5\5\17\3\9\9\5\5\5\5\6\6\4\4\4\4\4\4\17\17\17\17\17\29\29\29\29\10\10\4\4\29\29\5\29\6\6\6\6\6\5\
    \\4\6\8\8\4\4\4\21\21\4\4\4\4\4\4\19\21\21\21\5\10\10\10\10\10\10\10\29\29\29\29\29\29\29\4\3\8\8\4\17\4\17\17\17\5\5\17\5\5\17\
    \\5\5\5\5\17\6\5\5\5\5\5\29\5\29\29\29\29\29\29\29\10\10\10\10\10\10\10\21\10\10\10\10\10\10\10\17\5\6\6\6\6\29\29\6\6\6\29\29\6\6\
    \\5\4\29\5\5\29\29\29\29\5\5\4\6\4\4\4\4\6\6\5\5\5\29\5\29\0\0\0\0\0\0\0\0\13\17\14\20\11\0\0\0\0\0\29\1\1\1\1\1\1\
    \\1\1\13\18\14\18\25\25\25\25\25\25\25\25\0\1\21\5\5\5\5\5\5\5\17\17\17\17\17\17\17\5\5\5\26\5\5\5\5\5\6\4\4\4\4\4\29\3\29\5\
    \\6\6\29\29\29\6\6\6\6\6\6\29\6\6\6\5\4\21\21\21\17\17\17\17\21\17\21\21\21\22\4\4\4\4\4\4\4\13\14\29\29\29\22\17\17\17\21\3\4\9\
    \\4\4\4\4\4\4\4\17\17\17\9\9\9\9\9\29\29\8\8\29\29\29\4\4\4\5\6\5\6\6\5\5\5\5\5\6\6\6\5\6\5\5\4\4\17\4\17\29\29\29\
    \\29\29\29\20\4\4\4\4\4\4\4\6\6\5\6\6\4\4\4\4\4\29\17\17\17\17\10\17\17\3\3\3\3\3\3\17\17\18\17\11\17\17\17\17\17\17\17\22\5\5\
    \\5\5\5\5\7\5\7\7\7\5\5\5\5\5\7\7\7\21\9\9\9\9\9\9\9\9\0\1\9\9\9\3\4\17\21\21\0\1\0\1\1\21\21\21\0\1\0\1\5\8\
    \\8\17\17\17\17\17\17\17\12\17\15\16\17\17\17\17\11\17\19\29\29\29\29\29\29\4\4\4\17\3\3\3\4\4\4\4\4\4\21\4\4\20\20\20\20\20\20\21\21\21\
    \\21\21\5\29\29\29\9\9\9\9\9\10\10\10\10\29\29\29\21\3\29\3\3\3\3\3\3\29\29\29\29\29\1\29\29\5\5\4\6\4\4\4\4\5\5\6\6\17\6\5\
    \\29\29\5\5\5\5\5\12\3\3\3\3\3\21\21\22\17\17\17\19\17\17\17\4\17\4\4\5\6\6\5\5\6\6\29\4\4\4\6\5\5\6\6\6\5\5\5\5\5\17\
    \\29\29\1\17\12\29\29\21\21\19\21\29\29\29\29\29\4\4\4\4\29\29\4\29\29\8\8\17\17\17\17\4\4\4\4\4\4\6\6\6\6\26\26\26\26\26\5\5\5\5\
    \\5\26\21\5\26\26\26\5\5\5\5\5\21\21\5\5\5\21\29\29\21\13\14\21\21\21\21\21\18\18\21\21\21\21\5\5\5\5\5\21\21\21\21\21\5\21\21\4\8\8\
    \\8\8\8\8\8\29\29\29\29\17\17\17\19\17\17\17\17\17\17\29\29\29\29\29\1\1\4\1\1\1\1\1\5\5\5\5\3\29\29\29\29\29\29\29\21\9\10\21\21\29\
    \\29\29\29\21\21\21\21\21\29\29\21\21\8\8\8\8\8\8\8\8\17\17\18\18\18\17\3\3\20\20\20\20\3\3\20\20\20\20\20\20\10\10\29\4\4\4\4\4\6\6\
    \\6\5\5\5\5\6\4\4\4\4\5\6\5\4\6\6\6\6\6\6\6\5\6\6\6\29\29\29\29\4\4\19\19\10\10\10\10\21\19\4\17\5\29\21\4\10\10\10\10\10\
    \\10\21\5\21\5\13\14\13\14\6\6\6\6\5\5\29\29\9\4\4\4\4\4\4\4\5\5\5\6\29\29\6\6\5\29\29\0\0\0\0\2\20\20\20\1\1\1\29\1\1\
    \\10\3\29\29\10\10\10\10\18\18\18\13\14\3\0\1\0\1\3\3\5\5\5\5\5\5\17\3\14\17\13\14\17\17\4\4\4\4\4\4\3\3\17\17\17\17\17\17\17\13\
    \\4\4\4\29\29\5\6\5\5\5\5\29\6\6\6\6\5\5\6\5\5\17\17\26\17\17\5\29\29\29\29\29\4\4\4\4\6\6\5\5\5\29\29\29\6\6\5\5\5\5\
    \\6\5\5\17\29\29\29\29\5\5\5\5\5\17\5\5\5\5\3\3\3\3\3\17\3\5\29\29\29\8\8\17\17\29\29\29\29\4\4\4\4\29\29\21\5\5\17\17\17\4\
    \\17\17\1\18\1\1\1\1\1\1\0\1\29\29\8\8\4\4\4\4\3\4\4\4\4\4\4\5\29\29\8\8\29\29\29\29\29\29\21\21\21\20\20\20\20\20\8\8\17\17\
    \\18\18\18\17\0\0\0\0\0\0\0\29\0\0\29\1\21\21\21\0\1\0\1\5\5\0\1\29\29\29\29\4\4\4\29\29\29\29\5\5\20\20\3\3\4\4\4\17\3\3\
    \\3\4\4\29\26\29\29\29\29\27\27\27\27\27\27\27\27\28\28\28\28\28\28\28\28\8\8\5\29\29\29\29\29\18\29\29\29\29\29\29\29\19\10\10\10\10\29\29\29\8\
    \\8\21\29\29\29\29\29"#

generalCategoryOffsets1BitMap :: Ptr Word16
generalCategoryOffsets1BitMap = Ptr
    "\10\10\34\3\34\3\34\3\34\3\34\3\33\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\35\3\204\1\29\4\34\3\34\3\34\3\34\3\29\4\29\4\29\4\29\4\31\4\21\11\34\3\34\3\34\3\97\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\
    \\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\103\3\103\3\103\3\103\3\34\3\34\3\194\7\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\29\4\29\4\
    \\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\31\4\103\3\103\3\103\3\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\
    \\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\46\11\67\5\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\41\3\35\3\41\3\34\3\34\3\34\3\34\3\34\3\41\3\34\3\34\3\34\3\34\3\41\3\35\3\41\3\34\3\35\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\
    \\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\97\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\34\3\34\3\
    \\80\10\237\1\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\99\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\155\4\145\4\34\3\34\3\34\3\97\3\34\3\
    \\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\97\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\101\3\103\3\103\3\103\3\103\3\34\3\34\3\34\3\34\3\
    \\34\3\34\3\249\5\249\5\133\3\142\3\144\3\103\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\100\3\34\3\98\3\34\3\102\3\34\3\67\12\152\2\103\3\103\3\103\3\
    \\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\34\3\34\3\34\3\34\3\34\3\150\9\208\2\246\11\19\9\185\2\185\2\213\6\103\3\103\3\103\3\103\3\29\4\29\4\29\4\30\4\97\1\8\8\29\4\29\4\29\4\97\1\179\8\97\1\29\4\29\4\29\4\29\4\97\1\8\8\29\4\29\4\29\4\
    \\29\4\179\8\97\1\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\216\3\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\216\3\29\4\214\3\29\4\29\4\29\4\29\4\29\4\29\4\188\1\2\9\
    \\2\9\2\9\2\9\105\10\244\7\46\11\52\11\103\0\104\0\104\0\232\8\152\0\153\0\153\0\251\8\2\9\2\9\2\9\2\9\2\2\67\0\29\5\74\0\104\0\104\0\105\0\182\4\153\0\153\0\154\0\153\0\8\11\153\0\108\6\103\6\112\6\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\
    \\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\35\2\103\3\34\3\34\3\34\3\35\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\34\3\34\3\
    \\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\100\3\103\3\57\7\103\3\103\3\103\3\54\7\103\3\152\10\103\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\
    \\34\3\34\3\34\3\99\3\103\3\34\3\34\3\35\3\127\0\34\3\34\3\34\3\34\3\34\3\99\3\34\3\34\3\34\3\34\3\34\3\98\3\217\2\61\10\46\11\202\6\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\34\3\34\3\34\3\
    \\99\12\103\3\103\3\34\3\34\3\34\3\34\3\34\3\75\6\46\11\21\7\34\3\34\3\194\7\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\98\3\29\4\29\4\29\4\29\4\29\4\29\4\
    \\30\4\103\3\34\3\34\3\34\3\34\3\34\3\175\9\214\9\214\9\214\9\214\9\214\9\214\9\214\9\214\9\214\9\214\9\214\9\214\9\214\9\144\0\79\8\103\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\101\3\103\3\103\3\103\3\103\3\103\3\103\3\
    \\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\34\3\34\3\34\3\98\3\34\3\34\3\34\3\34\3\34\3\34\3\102\3\103\3\121\8\97\1\97\1\125\8\34\3\38\3\34\3\34\3\35\3\34\3\34\3\27\3\34\3\97\3\34\3\97\3\103\3\103\3\103\3\103\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\
    \\34\3\34\3\34\3\34\3\34\3\34\3\34\3\100\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\179\12\103\3\103\3\103\3\247\5\249\5\249\5\249\5\249\5\249\5\249\5\249\5\249\5\249\5\249\5\249\5\194\2\104\0\104\0\104\0\
    \\104\0\187\4\153\0\153\0\153\0\12\11\17\11\46\11\243\10\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\171\8\97\1\97\1\97\1\97\1\97\1\97\1\175\8\219\12\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\
    \\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\34\3\34\3\34\3\128\1\46\11\108\12\171\8\97\1\97\1\97\1\97\1\173\8\97\1\123\8\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\
    \\103\3\103\3\171\8\97\1\97\1\97\1\97\1\97\1\97\1\175\8\219\12\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\34\3\34\3\34\3\73\6\237\10\242\10\137\5\34\3\34\3\34\3\34\3\34\3\34\3\
    \\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\140\5\34\3\34\3\155\5\34\3\34\3\80\9\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\100\9\95\9\102\3\34\3\34\3\34\3\35\3\208\5\105\11\
    \\213\5\15\3\6\6\46\11\34\3\34\3\34\3\97\3\98\3\103\3\34\3\34\3\34\3\34\3\34\3\99\3\34\3\34\3\34\3\101\3\46\11\222\1\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\216\3\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\
    \\29\4\29\4\29\4\29\4\254\3\0\4\97\1\97\1\97\1\6\8\29\4\29\4\29\4\29\4\29\4\191\1\188\1\188\1\188\1\190\1\254\3\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\
    \\188\1\188\1\188\1\23\6\253\3\253\3\187\1\188\1\188\1\188\1\188\1\188\1\188\1\188\1\184\1\188\1\188\1\188\1\192\1\29\4\29\4\29\4\29\4\29\4\29\4\188\1\188\1\203\3\123\4\29\4\29\4\29\4\29\4\38\11\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\
    \\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\254\3\0\4\97\1\97\1\97\1\6\8\29\4\29\4\29\4\29\4\29\4\191\1\188\1\188\1\188\1\190\1\254\3\188\1\188\1\88\5\43\1\50\1\240\8\153\0\153\0\153\0\39\1\29\1\46\1\244\8\153\0\153\0\178\4\104\0\
    \\104\0\183\4\153\0\153\0\153\0\104\0\104\0\104\0\187\4\153\0\153\0\178\4\104\0\104\0\183\4\153\0\153\0\153\0\104\0\104\0\104\0\187\4\153\0\153\0\113\0\104\0\104\0\104\0\111\0\153\0\153\0\158\0\180\4\104\0\104\0\109\0\34\3\34\3\34\3\234\11\22\12\146\3\46\11\22\5\35\3\103\3\103\3\103\3\103\3\103\3\
    \\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\159\2\142\3\142\3\142\3\142\3\142\3\142\3\142\3\142\3\142\3\142\3\142\3\142\3\142\3\142\3\142\3\142\3\238\0\34\3\34\3\34\3\34\3\75\6\103\3\46\11\108\12\46\11\92\12\104\0\104\0\56\1\91\10\
    \\153\0\153\0\113\0\123\5\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\97\1\97\1\97\1\122\8\34\3\34\3\34\3\34\3\34\3\146\7\101\3\103\3\235\2\103\3\28\4\36\4\103\3\103\3\103\3\159\2\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\
    \\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\97\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\102\3\103\3\34\3\34\3\73\6\222\2\220\2\224\2\185\2\186\2\34\3\34\3\
    \\34\3\231\5\34\3\100\3\34\3\34\3\34\3\149\9\154\2\142\3\34\3\34\3\34\3\34\3\34\3\130\1\142\3\142\3\28\9\142\3\142\3\142\3\142\3\142\3\142\3\142\3\142\3\142\3\142\3\142\3\142\3\142\3\142\3\249\2\243\1\14\5\31\1\181\4\104\0\53\1\185\4\153\0\153\0\153\0\180\3\96\4\209\0\209\0\209\0\183\3\
    \\202\0\214\9\214\9\214\9\214\9\214\9\214\9\214\9\214\9\214\9\214\9\214\9\214\9\214\9\144\0\79\8\103\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\99\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\
    \\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\34\3\34\3\34\3\35\3\61\6\73\6\100\12\130\8\34\3\34\3\34\3\34\3\34\3\34\3\13\9\25\4\115\4\103\3\46\11\58\5\74\11\34\3\34\3\10\10\34\3\34\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\19\10\34\3\
    \\34\3\34\3\34\3\221\11\46\11\108\12\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\35\3\26\3\34\3\35\3\93\3\26\3\34\3\34\3\34\3\51\9\151\1\231\6\103\3\46\11\108\12\103\3\
    \\103\3\103\3\103\3\103\3\103\3\103\3\103\3\34\3\95\3\34\3\34\3\34\3\34\3\81\11\16\2\235\3\103\3\103\3\103\3\34\3\34\3\34\3\34\3\34\3\34\3\35\3\103\3\34\3\34\3\97\3\103\3\34\3\103\3\103\3\103\3\153\4\110\2\110\2\110\2\110\2\110\2\151\4\64\10\103\3\103\3\103\3\103\3\103\3\103\3\103\3\
    \\103\3\19\10\34\3\34\3\34\3\34\3\221\11\46\11\108\12\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\104\0\104\0\104\0\104\0\153\0\153\0\153\0\153\0\97\1\97\1\180\8\0\11\104\0\104\0\104\0\102\2\153\0\153\0\111\6\103\3\103\3\103\3\103\3\103\3\104\0\
    \\104\0\104\0\104\0\104\0\104\0\153\0\153\0\153\0\153\0\153\0\153\0\209\0\209\0\209\0\209\0\10\9\253\0\209\0\209\0\209\0\209\0\209\0\209\0\208\0\8\1\209\0\209\0\209\0\209\0\209\0\209\0\47\1\104\0\104\0\104\0\48\1\21\4\153\0\153\0\153\0\153\0\153\0\142\10\160\2\142\3\142\3\142\3\142\3\162\2\145\8\
    \\103\3\34\3\34\3\34\3\44\3\33\2\103\3\108\6\103\3\105\6\38\2\34\3\133\6\35\3\136\6\28\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\29\10\118\2\34\10\29\4\128\4\34\3\34\3\34\3\34\3\34\3\34\3\34\3\131\4\29\4\29\4\34\3\34\3\34\3\34\3\34\3\34\3\
    \\34\3\34\3\129\4\34\3\34\3\34\3\34\3\34\3\34\3\29\4\103\3\103\3\103\3\103\3\34\3\113\8\29\4\26\8\29\4\29\4\198\1\210\10\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\219\3\29\4\29\4\29\4\79\5\188\1\188\1\129\5\29\4\29\4\29\4\29\4\78\5\198\1\29\4\29\4\29\4\29\4\29\4\
    \\35\4\103\3\103\3\29\4\34\4\103\3\103\3\97\1\97\1\97\1\97\1\97\1\97\1\97\1\6\8\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\2\8\97\1\97\1\34\3\34\3\188\6\194\6\34\3\34\3\34\3\241\2\34\3\34\3\34\3\101\3\103\3\103\3\103\3\103\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\
    \\84\8\97\1\97\1\170\8\97\1\97\1\97\1\97\1\97\1\5\8\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\31\4\103\3\103\3\103\3\103\3\103\3\103\3\42\5\29\4\29\4\29\4\29\4\29\4\33\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\
    \\29\4\29\4\29\4\29\4\29\4\33\4\103\3\29\4\30\4\74\5\29\4\74\5\29\4\74\5\29\4\29\4\29\4\31\4\103\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\100\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\
    \\34\3\97\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\137\5\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\
    \\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\41\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\72\7\185\2\96\1\97\1\124\8\34\3\34\3\29\4\35\4\104\0\104\0\104\0\104\0\104\0\104\0\104\0\104\0\104\0\104\0\56\1\113\0\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\
    \\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\102\3\103\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\97\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\
    \\103\3\103\3\103\3\103\3\209\0\209\0\209\0\209\0\209\0\209\0\209\0\209\0\209\0\209\0\209\0\209\0\209\0\209\0\209\0\209\0\209\0\209\0\9\1\181\3\209\0\209\0\209\0\209\0\209\0\209\0\209\0\209\0\209\0\209\0\209\0\209\0\210\0\210\0\5\1\209\0\209\0\209\0\209\0\209\0\80\4\42\6\92\4\132\7\128\7\211\0\
    \\87\4\206\0\124\7\222\0\70\4\210\0\82\4\209\0\209\0\71\4\209\0\209\0\209\0\209\0\209\0\209\0\11\1\130\7\214\0\209\0\153\0\153\0\153\0\153\0\153\0\153\0\153\0\153\0\162\0\153\0\153\0\153\0\110\2\110\2\60\11\110\2\116\2\118\2\113\2\122\2\118\2\118\2\119\2\110\2\35\6\209\0\5\1\209\0\209\0\209\0\
    \\209\0\209\0\209\0\209\0\179\3\38\6\209\0\128\2\1\1\209\0\209\0\75\4\77\4\209\0\213\0\40\6\209\0\81\1\103\3\103\3\159\4\239\5\34\3\34\3\34\3\189\6\98\1\103\3\34\3\34\3\73\6\142\3\183\7\9\2\103\3\103\3\34\3\34\3\81\3\9\2\103\3\103\3\103\3\103\3\34\3\34\3\189\6\125\8\103\3\103\3\
    \\34\3\34\3\35\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\97\1\97\1\97\1\122\8\34\3\34\3\34\3\34\3\34\3\146\7\101\3\103\3\235\2\103\3\28\4\36\4\103\3\103\3\103\3\159\2\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\159\12\97\11\89\7\89\7\89\7\89\7\
    \\89\7\89\7\11\2\43\12\110\2\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\49\12\103\3\170\6\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\104\0\104\0\104\0\104\0\153\0\153\0\153\0\153\0\97\1\97\1\180\8\0\11\104\0\104\0\104\0\102\2\153\0\153\0\111\6\103\3\103\3\103\3\103\3\103\3\197\3\36\0\
    \\142\3\4\7\34\3\34\3\34\3\34\3\123\1\76\6\142\3\142\3\46\11\165\10\71\6\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\226\5\186\10\89\0\73\3\46\11\104\8\185\2\187\2\70\6\34\3\34\3\34\3\142\3\142\3\142\3\232\2\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\
    \\73\6\142\3\100\6\103\3\46\11\191\7\34\3\34\3\34\3\76\6\96\0\49\4\33\9\34\3\34\3\34\3\34\3\34\3\34\3\92\11\141\3\210\5\219\4\34\3\216\1\46\11\122\1\34\3\5\3\52\7\94\3\34\3\34\3\32\3\108\5\202\2\209\2\197\8\104\3\47\3\215\6\46\11\113\11\119\11\24\2\44\3\94\3\34\3\34\3\32\3\
    \\54\3\90\8\158\6\229\2\163\8\5\5\62\1\46\11\91\6\103\3\24\2\36\3\31\3\34\3\34\3\32\3\29\3\202\2\157\8\221\7\102\3\103\3\215\6\46\11\7\10\218\4\221\5\52\7\94\3\34\3\34\3\32\3\29\3\182\5\209\2\162\11\13\2\47\3\215\6\46\11\127\11\103\3\21\3\161\7\41\3\50\3\111\5\161\7\34\3\114\5\
    \\45\9\112\3\59\7\103\3\62\1\46\11\7\8\148\10\173\5\37\3\32\3\34\3\34\3\32\3\34\3\84\2\19\2\242\0\115\6\178\5\215\6\46\11\37\4\172\8\56\6\37\3\32\3\34\3\34\3\32\3\38\3\182\5\216\5\252\6\106\3\157\7\215\6\46\11\153\7\103\3\69\8\37\3\32\3\34\3\34\3\34\3\34\3\202\7\240\11\56\9\
    \\146\6\98\1\215\6\46\11\97\1\185\6\221\5\34\3\35\3\95\3\34\3\34\3\31\3\137\6\35\3\163\6\219\8\89\7\62\1\46\11\118\5\103\3\33\3\34\3\34\3\34\3\34\3\34\3\126\3\147\3\125\1\13\9\46\11\57\12\103\3\103\3\103\3\103\3\50\3\39\3\34\3\34\3\6\5\34\3\126\3\169\2\37\9\143\3\46\11\67\7\
    \\103\3\103\3\103\3\103\3\62\9\185\2\67\9\90\2\46\11\172\7\176\7\137\11\34\3\33\3\34\3\34\3\34\3\98\3\160\2\132\1\36\12\74\6\142\3\160\2\142\3\142\3\142\3\113\7\92\2\12\8\24\4\8\2\103\3\103\3\103\3\103\3\34\3\34\3\34\3\34\3\34\3\216\8\140\3\86\6\46\11\245\9\151\9\77\3\206\7\91\7\
    \\129\3\34\3\115\10\96\8\46\11\68\1\104\0\104\0\104\0\104\0\49\1\122\6\153\0\153\0\153\0\153\0\153\0\167\3\34\3\34\3\157\11\129\8\34\3\34\3\105\1\103\3\34\3\34\3\135\2\103\3\34\3\37\3\203\8\103\3\34\3\34\3\34\3\34\3\34\3\34\3\46\2\79\6\99\11\141\3\116\1\101\7\46\11\108\12\97\1\127\8\
    \\248\9\24\9\46\11\108\12\34\3\34\3\34\3\34\3\196\7\34\3\34\3\34\3\34\3\34\3\34\3\102\3\200\7\34\3\34\3\34\3\34\3\99\6\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\97\3\103\3\34\3\34\3\72\6\229\7\34\3\34\3\34\3\34\3\34\3\34\3\130\0\143\3\121\9\206\5\203\5\225\2\46\11\108\12\
    \\46\11\108\12\44\4\253\10\142\3\194\9\142\3\142\3\142\3\144\3\142\3\146\3\103\3\103\3\32\9\34\3\34\3\34\3\34\3\34\3\136\3\197\5\157\9\162\9\46\11\245\9\28\4\221\10\223\4\18\8\34\9\34\3\34\3\34\3\187\5\137\1\46\11\191\7\34\3\34\3\34\3\34\3\134\3\129\9\14\4\40\4\34\3\34\3\34\3\34\3\
    \\173\10\202\5\81\6\41\4\46\11\113\9\46\11\191\7\34\3\34\3\34\3\175\9\153\0\228\0\104\0\104\0\104\0\104\0\104\0\43\1\185\2\103\3\38\12\142\3\140\3\142\1\143\1\164\1\153\0\153\0\153\0\153\0\153\0\106\2\110\2\110\2\110\2\110\2\110\2\110\2\110\2\177\3\153\0\179\3\153\0\153\0\153\0\107\2\110\2\110\2\
    \\110\2\110\2\142\3\142\3\142\3\142\3\142\3\142\3\142\3\142\3\153\0\104\0\113\0\56\1\153\0\104\0\153\0\104\0\113\0\56\1\153\0\94\5\153\0\104\0\153\0\113\0\153\0\63\4\153\0\63\4\153\0\63\4\190\4\21\1\176\11\170\11\115\0\203\4\153\0\48\6\119\0\138\7\143\2\148\2\89\1\243\3\185\2\251\1\185\2\255\9\
    \\239\7\185\2\235\7\186\9\252\5\249\5\184\11\190\11\97\1\6\4\110\2\62\10\36\7\36\7\36\7\36\7\55\4\103\3\142\3\205\9\200\9\142\3\164\8\103\3\74\1\119\7\252\7\189\3\227\4\233\4\217\0\197\0\82\5\168\0\97\1\97\1\214\9\214\9\214\9\214\9\219\9\29\11\123\4\216\10\207\3\217\3\29\4\29\4\29\4\76\5\
    \\211\3\29\4\29\4\29\4\78\5\188\1\104\0\104\0\104\0\104\0\104\0\104\0\153\0\153\0\153\0\153\0\153\0\153\0\91\4\212\0\83\4\15\1\209\0\209\0\209\0\209\0\209\0\209\0\209\0\209\0\209\0\209\0\209\0\209\0\232\9\237\9\151\12\167\9\153\0\153\0\153\0\153\0\189\4\67\10\34\3\34\3\34\3\34\3\34\3\34\3\
    \\34\3\238\1\237\1\11\2\34\3\34\3\35\3\103\3\35\3\35\3\35\3\35\3\35\3\35\3\35\3\35\3\142\3\142\3\142\3\142\3\6\0\0\0\19\0\252\9\250\3\108\4\185\2\24\0\172\1\185\2\21\8\27\6\103\3\103\3\103\3\103\3\29\4\29\4\29\4\15\8\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\
    \\33\4\103\3\88\9\254\3\34\8\38\8\213\9\65\8\97\10\224\9\33\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\35\3\165\12\137\5\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\15\10\34\3\65\3\34\3\34\3\46\11\12\6\103\3\103\3\209\0\209\0\209\0\209\0\209\0\255\2\53\8\57\8\
    \\209\0\209\0\209\0\198\11\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\138\0\214\9\18\9\103\3\65\6\68\6\34\3\34\3\150\6\38\10\4\8\154\3\34\3\34\3\34\3\34\3\34\3\34\3\75\8\103\3\71\8\34\3\34\3\34\3\34\3\34\3\173\10\89\7\143\11\38\4\46\11\108\12\142\3\142\3\132\3\110\10\46\11\191\7\
    \\34\3\34\3\73\6\14\9\34\3\34\3\72\6\142\3\14\4\37\4\34\3\34\3\34\3\98\3\33\9\34\3\34\3\34\3\34\3\34\3\10\3\83\6\184\2\28\0\46\11\243\10\190\0\34\3\46\11\50\7\34\3\34\3\34\3\34\3\34\3\78\6\154\6\103\3\68\6\166\4\46\11\221\6\34\3\34\3\21\10\210\1\34\3\34\3\34\3\34\3\
    \\34\3\34\3\157\1\44\2\99\6\103\3\103\3\164\7\34\3\125\10\241\6\103\3\96\3\96\3\96\3\103\3\35\3\35\3\153\0\153\0\153\0\153\0\153\0\172\3\153\0\240\4\153\0\153\0\153\0\153\0\153\0\153\0\153\0\153\0\153\0\153\0\34\3\34\3\34\3\34\3\154\9\84\10\46\11\108\12\187\12\187\12\187\12\187\12\187\12\187\12\
    \\187\12\187\12\187\12\187\12\187\12\187\12\187\12\187\12\187\12\187\12\187\12\187\12\187\12\187\12\187\12\187\12\187\12\187\12\187\12\187\12\187\12\187\12\187\12\187\12\187\12\187\12\195\12\195\12\195\12\195\12\195\12\195\12\195\12\195\12\195\12\195\12\195\12\195\12\195\12\195\12\195\12\195\12\195\12\195\12\195\12\195\12\195\12\195\12\195\12\195\12\
    \\195\12\195\12\195\12\195\12\195\12\195\12\195\12\195\12\142\3\142\3\226\11\236\1\142\3\142\3\179\1\253\3\137\4\45\8\235\6\230\1\35\5\81\0\37\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\242\5\248\10\244\7\46\11\52\11\103\0\104\0\104\0\232\8\152\0\153\0\
    \\153\0\101\4\212\11\34\3\123\1\34\3\34\3\34\3\34\3\218\11\34\3\34\3\34\3\35\3\95\3\95\3\95\3\159\7\42\7\175\0\103\3\1\6\81\8\97\1\97\1\97\1\97\1\97\1\51\10\29\4\214\9\214\9\214\9\214\9\214\9\214\9\46\10\9\8\29\4\42\0\29\4\32\4\36\4\103\3\103\3\103\3\103\3\103\3\29\4\29\4\
    \\29\4\29\4\29\4\37\10\34\3\34\3\34\3\34\3\125\8\10\10\34\3\34\3\94\9\211\4\34\3\34\3\34\3\34\3\73\6\245\0\34\3\34\3\34\3\147\5\34\3\34\3\34\3\34\3\99\3\34\3\105\9\103\3\103\3\103\3\103\3\103\3\104\0\104\0\104\0\104\0\104\0\153\0\153\0\153\0\153\0\153\0\34\3\34\3\34\3\34\3\
    \\34\3\34\3\34\3\34\3\34\3\97\3\46\11\108\12\104\0\104\0\104\0\104\0\58\1\153\0\153\0\153\0\153\0\111\6\34\3\34\3\34\3\34\3\34\3\103\3\34\3\34\3\34\3\34\3\34\3\34\3\99\3\37\4\104\0\52\1\104\0\52\1\136\12\153\0\193\4\153\0\193\4\122\0\34\3\34\3\34\3\34\3\34\3\34\3\99\3\103\3\
    \\97\3\32\3\34\3\34\3\34\3\34\3\36\3\90\3\34\3\34\3\147\5\97\1\34\3\34\3\138\5\179\8\34\3\34\3\34\3\35\3\165\8\97\1\103\3\103\3\103\3\103\3\103\3\103\3\34\3\34\3\178\5\169\8\248\4\157\2\38\3\33\3\34\3\34\3\97\3\16\3\97\1\128\8\185\2\237\1\34\3\34\3\34\3\165\5\34\3\34\3\
    \\34\3\189\6\103\3\103\3\103\3\103\3\34\3\186\6\34\3\34\3\212\6\169\8\186\2\103\3\34\3\34\3\34\3\34\3\34\3\34\3\97\3\43\4\34\3\34\3\97\3\97\1\34\3\34\3\100\3\97\1\34\3\34\3\101\3\76\2\103\3\171\8\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\34\3\34\3\34\3\34\3\
    \\34\3\34\3\34\3\34\3\34\3\102\3\103\3\103\3\103\3\103\3\103\3\103\3\104\0\104\0\104\0\104\0\104\0\104\0\59\1\103\3\153\0\153\0\153\0\153\0\153\0\153\0\112\6\170\8\135\0\34\3\34\3\34\3\34\3\34\3\34\3\142\3\13\9\253\10\170\8\97\1\61\5\46\11\161\1\11\2\34\9\34\3\34\3\34\3\34\3\34\3\
    \\82\11\252\11\10\7\13\7\34\3\34\3\34\3\102\3\46\11\108\12\131\3\34\3\34\3\34\3\72\6\135\1\52\5\46\11\225\6\103\3\34\3\34\3\34\3\34\3\102\5\103\3\34\9\34\3\34\3\34\3\34\3\34\3\79\11\132\1\96\7\151\8\46\11\138\8\171\8\97\1\124\8\103\3\34\3\34\3\31\3\34\3\34\3\78\11\191\5\64\2\
    \\105\7\103\3\103\3\103\3\103\3\103\3\103\3\103\3\35\3\24\3\34\3\36\3\34\3\141\9\34\3\34\3\34\3\34\3\34\3\72\6\203\5\245\0\46\11\108\12\120\10\52\7\94\3\34\3\34\3\32\3\29\3\177\2\188\8\192\8\59\7\10\10\148\1\145\3\145\3\103\3\34\3\139\6\34\3\34\3\34\3\34\3\36\3\203\5\247\6\215\7\
    \\70\2\237\1\119\6\103\3\103\3\103\3\34\3\34\3\34\3\34\3\34\3\34\3\172\10\142\3\229\3\76\8\46\11\183\0\101\3\103\3\103\3\103\3\34\3\34\3\34\3\34\3\34\3\34\3\203\5\199\5\134\9\103\3\46\11\108\12\103\3\103\3\103\3\103\3\34\3\34\3\34\3\34\3\34\3\34\3\203\5\190\5\86\3\103\3\46\11\108\12\
    \\185\2\79\8\103\3\103\3\34\3\34\3\34\3\34\3\34\3\118\9\133\1\141\9\46\11\108\12\46\11\46\11\241\10\103\3\103\3\103\3\34\3\34\3\34\3\34\3\34\3\78\11\142\3\28\12\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\104\0\104\0\104\0\104\0\153\0\153\0\153\0\153\0\46\11\172\7\
    \\126\8\129\8\219\4\131\3\34\3\34\3\34\3\34\3\76\6\113\1\20\9\103\3\78\6\139\1\34\3\34\3\34\3\34\3\34\3\77\6\132\1\72\12\0\11\103\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\102\3\185\2\9\2\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\195\5\103\3\103\3\103\3\
    \\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\34\3\34\3\34\3\34\3\141\9\103\3\46\11\108\12\34\3\32\3\34\3\34\3\34\3\150\9\143\3\133\1\78\8\103\3\46\11\172\7\97\1\124\8\169\10\34\3\34\3\34\3\159\2\142\3\142\3\215\2\51\2\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\35\3\31\3\
    \\34\3\34\3\34\3\34\3\50\5\234\0\120\3\103\3\46\11\108\12\36\3\32\3\34\3\34\3\34\3\211\7\56\2\102\3\46\11\108\12\34\3\34\3\34\3\34\3\34\3\239\2\46\11\108\12\103\3\103\3\211\8\34\3\32\3\34\3\34\3\34\3\215\8\16\12\61\2\185\2\46\11\203\12\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\
    \\103\3\103\3\102\3\103\3\97\1\97\1\5\8\31\7\28\7\29\4\35\4\37\4\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\35\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\34\3\34\3\34\3\73\6\142\3\129\10\
    \\46\11\108\12\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\102\3\34\3\34\3\34\3\35\3\46\11\243\10\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\35\3\
    \\46\11\108\12\34\3\34\3\34\3\97\3\134\10\103\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\97\3\103\3\103\3\103\3\103\3\129\8\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\
    \\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\33\4\73\5\29\4\29\4\36\4\103\3\29\4\29\4\211\12\103\3\142\3\142\3\142\3\142\3\142\3\144\3\142\3\142\3\143\3\103\3\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\33\4\103\3\103\3\103\3\
    \\103\3\103\3\103\3\103\3\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\31\4\103\3\29\4\29\4\29\4\29\4\30\4\74\5\29\4\29\4\29\4\29\4\29\4\29\4\224\3\84\7\
    \\178\10\183\10\77\7\223\4\29\4\29\4\29\4\80\7\29\4\29\4\29\4\29\4\29\4\29\4\29\4\34\4\103\3\103\3\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\202\10\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\97\1\97\1\125\8\103\3\97\1\97\1\125\8\103\3\
    \\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\30\4\103\3\97\1\97\1\97\1\128\8\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\104\0\104\0\104\0\187\4\153\0\153\0\178\4\104\0\104\0\183\4\190\4\153\0\153\0\104\0\104\0\104\0\187\4\153\0\
    \\153\0\197\4\125\6\36\1\183\4\162\3\191\4\153\0\104\0\104\0\104\0\187\4\153\0\153\0\153\0\153\0\156\0\178\4\104\0\104\0\107\0\153\0\153\0\154\0\176\4\104\0\104\0\105\0\153\0\153\0\153\0\174\4\104\0\104\0\104\0\111\0\153\0\153\0\158\0\86\12\46\11\46\11\46\11\46\11\46\11\46\11\142\3\142\3\142\3\142\3\
    \\142\3\142\3\220\4\221\10\142\3\142\3\142\3\142\3\142\3\222\4\93\2\29\4\94\2\80\8\103\3\158\2\160\2\142\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\143\3\142\3\142\3\89\10\238\0\245\0\110\2\110\2\110\2\110\2\110\2\110\2\110\2\61\10\103\3\103\3\103\3\11\2\103\3\103\3\103\3\103\3\
    \\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\34\3\178\6\97\1\143\3\103\3\103\3\103\3\103\3\103\3\38\3\34\3\34\3\34\3\58\3\33\3\39\3\10\5\
    \\143\6\1\5\58\3\255\4\58\3\39\3\39\3\5\5\34\3\31\3\34\3\99\3\28\2\31\3\34\3\99\3\103\3\103\3\103\3\103\3\103\3\103\3\40\5\103\3\34\4\103\3\29\4\29\4\29\4\29\4\29\4\33\4\29\4\36\4\35\4\103\3\31\4\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\103\3\
    \\103\3\103\3\103\3\103\3\103\3\103\3\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\60\0\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\
    \\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\55\0\29\4\32\4\29\4\32\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\35\4\29\4\33\4\
    \\36\4\103\3\29\4\33\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\103\3\29\4\35\4\29\4\29\4\29\4\29\4\29\4\103\3\29\4\29\4\29\4\31\4\29\4\33\4\35\4\103\3\188\1\211\12\103\3\103\3\103\3\103\3\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\103\3\29\4\31\4\29\4\32\4\
    \\29\4\53\0\29\4\29\4\29\4\29\4\29\4\29\4\30\4\48\0\29\4\37\11\29\4\46\0\29\4\36\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\14\8\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\29\4\46\11\227\12"#

generalCategoryOffsets2BitMap :: Ptr Word16
generalCategoryOffsets2BitMap = Ptr
    "\243\1\231\8\6\9\239\5\7\7\33\7\194\9\226\9\210\5\2\10\34\10\66\10\98\10\130\10\162\10\194\10\226\10\187\0\160\0\112\8\19\4\187\0\38\4\2\11\34\11\70\4\66\11\98\11\130\11\162\11\206\8\194\11\226\11\2\12\147\4\124\7\153\7\211\1\197\1\117\4\210\4\163\4\147\4\195\4\34\12\66\12\98\12\98\0\130\12\0\0\
    \\173\1\210\4\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\74\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\
    \\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\
    \\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\66\0\187\0\187\0\187\0\214\2\187\0\162\12\36\9\194\12\226\12\2\13\34\13\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\
    \\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\121\2\66\13\66\13\66\13\66\13\66\13\66\13\66\13\66\13\98\13\98\13\98\13\98\13\98\13\98\13\98\13\98\13\98\13\98\13\98\13\98\13\98\13\98\13\98\13\98\13\98\13\98\13\98\13\98\13\98\13\98\13\98\13\98\13\98\13\187\0\
    \\58\1\65\7\187\0\92\7\130\13\162\13\58\3\194\13\26\3\226\13\2\14\34\14\187\0\183\6\66\14\185\7\98\14\130\14\162\14\106\5\126\5\68\9\194\14\226\14\2\15\34\15\66\15\141\1\98\15\56\5\130\15\151\6\162\15\194\15\226\15\2\16\220\0\34\16\187\0\187\0\187\0\6\3\246\2\252\0\5\1\5\1\5\1\5\1\5\1\5\1\
    \\5\1\5\1\5\1\38\2\187\0\187\0\187\0\187\0\90\1\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\102\1\187\0\187\0\66\16\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\
    \\5\1\5\1\5\1\98\16\5\1\5\1\5\1\5\1\5\1\5\1\187\0\187\0\130\16\80\6\5\1\104\6\231\6\131\9\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\162\16\70\2\5\1\5\1\
    \\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\7\1\187\0\96\2\110\2\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\120\1\5\1\5\1\5\1\5\1\5\1\
    \\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\128\0\210\4\194\16\226\16\2\17\34\17\66\17\98\17\130\17\5\5\24\5\162\17\210\4\210\4\194\17\5\1\5\1\5\1\5\1\18\2\226\17\153\2\167\2\5\1\170\3\243\3\48\6\120\6\2\18\129\3\5\1\5\1\147\3\202\3\34\18\5\1\245\7\216\7\66\18\98\18\
    \\210\4\210\4\130\18\162\18\194\18\210\4\226\18\2\19\5\1\5\1\5\1\5\1\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\
    \\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\
    \\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\
    \\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\34\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\30\0\187\0\187\0\187\0\
    \\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\50\1\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\180\5\187\0\187\0\
    \\208\0\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\5\1\187\0\187\0\216\0\5\1\5\1\5\1\5\1\5\1\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\80\3\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\
    \\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\187\0\97\3\73\5\92\5"#

