# Internationalization

Now that we have Meson set up, we can add internationalization to our To-Do app.
We want users to be able to use the app in their own language.
With GTK, [gettext](https://www.gnu.org/software/gettext/) is the standard system for this.

## How Gettext Works

Gettext uses message catalogs to store translations.
The workflow looks like this:

1. **Mark strings** in source code and UI files as translatable
2. **Extract strings** into a template file (`.pot`)
3. **Translate strings** into language-specific files (`.po`)
4. **Compile translations** into binary format (`.mo`) during build

GTK already handles translations for strings marked with `translatable="yes"` in UI templates.
However, strings defined in Rust code (like our dialog messages) require explicit gettext calls.
Let's set this up!

## Project Structure

We extend our project with a `po/` directory for translation files:

```
├── po/
│   ├── LINGUAS           # List of supported languages
│   ├── POTFILES.in       # Files containing translatable strings
│   ├── meson.build       # Meson i18n integration
│   └── de.po             # German translation
```

## Setting Up Translations

### The po Directory

We create `po/meson.build` which tells Meson to handle translations using the glib preset.
The preset knows how to extract strings from `.ui` files and Rust source code.

Filename: <a class=file-link href="https://github.com/gtk-rs/gtk4-rs/blob/main/book/listings/todo/10/po/meson.build">listings/todo/10/po/meson.build</a>

```meson
{{#include ../listings/todo/10/po/meson.build}}
```

`POTFILES.in` lists all files that contain translatable strings.

Filename: <a class=file-link href="https://github.com/gtk-rs/gtk4-rs/blob/main/book/listings/todo/10/po/POTFILES.in">listings/todo/10/po/POTFILES.in</a>

```
{{#include ../listings/todo/10/po/POTFILES.in}}
```

`LINGUAS` lists the language codes of available translations.
In our case, we'll only add German (de), the mother tongue of the author of this book.

Filename: <a class=file-link href="https://github.com/gtk-rs/gtk4-rs/blob/main/book/listings/todo/10/po/LINGUAS">listings/todo/10/po/LINGUAS</a>

```
{{#include ../listings/todo/10/po/LINGUAS}}
```

### Updating meson.build

We import the `i18n` module and add the `po` subdirectory.

Filename: <a class=file-link href="https://github.com/gtk-rs/gtk4-rs/blob/main/book/listings/todo/10/meson.build">listings/todo/10/meson.build</a>

```meson
{{#include ../listings/todo/10/meson.build}}
```

We also pass `LOCALEDIR` and `GETTEXT_PACKAGE` to the Rust build, so our code knows where to find translations.

Filename: <a class=file-link href="https://github.com/gtk-rs/gtk4-rs/blob/main/book/listings/todo/10/src/meson.build">listings/todo/10/src/meson.build</a>

```meson
{{#include ../listings/todo/10/src/meson.build}}
```


## Translations in Rust Code

So far so good.
The build system is ready, but we haven't touched the Rust code yet.
How do we actually translate strings that are defined in code rather than in `.ui` files?

### Adding the Dependency

We add `gettext-rs` to our dependencies.

```
cargo add gettext-rs
```

### The Config Module

We extend `config.rs` to provide the locale directory and gettext package name:

Filename: <a class=file-link href="https://github.com/gtk-rs/gtk4-rs/blob/main/book/listings/todo/10/src/config.rs">listings/todo/10/src/config.rs</a>

```rust
{{#include ../listings/todo/10/src/config.rs}}
```

### Initializing Gettext

We create an `i18n` module to initialize the translation system.
This must happen early, before any translatable strings are used.

Filename: <a class=file-link href="https://github.com/gtk-rs/gtk4-rs/blob/main/book/listings/todo/10/src/i18n.rs">listings/todo/10/src/i18n.rs</a>

```rust,no_run,noplayground
{{#rustdoc_include ../listings/todo/10/src/i18n.rs:init}}
```

We call `i18n::init()` in `main.rs` before loading resources:

Filename: <a class=file-link href="https://github.com/gtk-rs/gtk4-rs/blob/main/book/listings/todo/10/src/main.rs">listings/todo/10/src/main.rs</a>

```rust,no_run,noplayground
{{#rustdoc_include ../listings/todo/10/src/main.rs:i18n_init}}
```

### Marking Strings for Translation

Now we can use `gettext()` to mark strings in our Rust code.
We wrap each user-visible string in the "New Collection" dialog:

Filename: <a class=file-link href="https://github.com/gtk-rs/gtk4-rs/blob/main/book/listings/todo/10/src/window/mod.rs">listings/todo/10/src/window/mod.rs</a>

```rust,no_run,noplayground
{{#rustdoc_include ../listings/todo/10/src/window/mod.rs:new_collection}}
```

The `gettext()` function looks up the translation at runtime.
If no translation exists, it just returns the original string.

## Desktop File Translation

Desktop files also support translations.
We prefix translatable keys with an underscore.
Also since it is now templated with both Meson and gettext, we change the file extension from `org.gtk_rs.Todo10.desktop.in` to `org.gtk_rs.Todo10.desktop.in.in`.

Filename: <a class=file-link href="https://github.com/gtk-rs/gtk4-rs/blob/main/book/listings/todo/10/data/org.gtk_rs.Todo10.desktop.in.in">listings/todo/10/data/org.gtk_rs.Todo10.desktop.in.in</a>

```ini
{{#include ../listings/todo/10/data/org.gtk_rs.Todo10.desktop.in.in}}
```

Meson's `i18n.merge_file()` processes `_Comment` and merges in the translations.

## Creating a Translation

### The Translation File

Translation files use the `.po` format.
This is how a small section of the file looks like:

Filename: <a class=file-link href="https://github.com/gtk-rs/gtk4-rs/blob/main/book/listings/todo/10/po/de.po">listings/todo/10/po/de.po</a>

```po
#: data/resources/window.ui:134
msgid "Main Menu"
msgstr "Hauptmenü"

#: data/resources/window.ui:153
msgid "Enter a Task…"
msgstr "Aufgabe eingeben…"
```

Each entry has:
- A comment showing where the string appears
- `msgid`: the original string (must match exactly)
- `msgstr`: the translated string

### Generating the Template

To create a `.pot` template file with all translatable strings:

```
meson compile -C builddir todo-10-pot
```

Translators can use this template to create new `.po` files.
Tools like [Poedit](https://poedit.net/) or [GNOME Translation Editor](https://wiki.gnome.org/Apps/Gtranslator) make this easier.


## Building and Testing

Build and install as usual:

```
meson setup builddir -Dprofile=development --prefix=~/.local
meson install -C builddir
```

To test a specific language, set the `LANGUAGE` environment variable:

```
LANGUAGE=de todo-10
```

<div style="text-align:center"><img src="img/todo_10_german.png" alt="To-Do app in German"/></div>

That was it!
Our To-Do app can now speak German.
For projects with many contributors, translation platforms like [Weblate](https://weblate.org/) or [Damned Lies](https://l10n.gnome.org/) help coordinate the work.
