// SPDX-License-Identifier: GPL-2.0-only WITH linking exception
// SPDX-FileCopyrightText: 2021–2024 grommunio GmbH
// This file is part of Gromox.
#include <condition_variable>
#include <csignal>
#include <cstdio>
#include <cstring>
#include <memory>
#include <mutex>
#include <pthread.h>
#include <string>
#include <unistd.h>
#include <unordered_map>
#include <vector>
#include <libHX/defs.h>
#include <libHX/string.h>
#include <gromox/atomic.hpp>
#include <gromox/defs.h>
#include <gromox/double_list.hpp>
#include <gromox/proc_common.h>
#include <gromox/process.hpp>
#include <gromox/util.hpp>
#include "asyncemsmdb_interface.hpp"
#include "common_util.hpp"
#include "emsmdb_interface.hpp"
#include "emsmdb_ndr.hpp"
#define WAITING_INTERVAL						300

#define FLAG_NOTIFICATION_PENDING				0x00000001

using namespace gromox;

namespace {

struct ASYNC_WAIT {
	DOUBLE_LIST_NODE node;
	time_t wait_time;
	char username[UADDR_SIZE];
	uint16_t cxr;
	uint32_t async_id;
	union {
		ECDOASYNCWAITEX_OUT *pout;
		int context_id; /* when async_id is 0 */
	} out_payload;
};

}

static constexpr size_t TAG_SIZE = UADDR_SIZE + 1 + HXSIZEOF_Z32;
static unsigned int g_threads_num;
static pthread_t g_scan_id;
static std::vector<pthread_t> g_thread_ids;
static gromox::atomic_bool g_notify_stop{true};
static DOUBLE_LIST g_wakeup_list;
static std::unordered_map<std::string, ASYNC_WAIT *> g_tag_hash;
static size_t g_tag_hash_max;
static std::mutex g_list_lock; /* protects g_wakeup_list */
static std::mutex g_async_lock; /* protects g_tag_hash & g_async_hash */
static std::condition_variable g_waken_cond;
static std::unordered_map<int, ASYNC_WAIT *> g_async_hash;

static void *aemsi_scanwork(void *);
static void *aemsi_thrwork(void *);

static void (*active_hpm_context)(int context_id, BOOL b_pending);

/* called by moh_emsmdb module */
void asyncemsmdb_interface_register_active(void *pproc)
{
	active_hpm_context = reinterpret_cast<decltype(active_hpm_context)>(pproc);
}

void asyncemsmdb_interface_init(unsigned int threads_num)
{
	g_threads_num = threads_num;
	g_thread_ids.reserve(threads_num);
	double_list_init(&g_wakeup_list);
}

int asyncemsmdb_interface_run()
{
	int context_num;
	
	context_num = get_context_num();
	g_tag_hash_max = context_num;
	g_notify_stop = false;
	auto ret = pthread_create4(&g_scan_id, nullptr, aemsi_scanwork, nullptr);
	if (ret != 0) {
		mlog(LV_ERR, "emsmdb: failed to create scanning thread "
		       "for asyncemsmdb: %s", strerror(ret));
		g_notify_stop = true;
		return -5;
	}
	pthread_setname_np(g_scan_id, "asyncems/scan");
	for (unsigned int i = 0; i < g_threads_num; ++i) {
		pthread_t tid;
		ret = pthread_create4(&tid, nullptr, aemsi_thrwork, nullptr);
		if (ret != 0) {
			mlog(LV_ERR, "emsmdb: failed to create wake up "
			       "thread for asyncemsmdb: %s", strerror(ret));
			asyncemsmdb_interface_stop();
			return -6;
		}
		char buf[32];
		snprintf(buf, sizeof(buf), "asyncems/%u", i);
		pthread_setname_np(tid, buf);
		g_thread_ids.push_back(tid);
	}
	return 0;
}

void asyncemsmdb_interface_stop()
{
	if (!g_notify_stop) {
		g_notify_stop = true;
		g_waken_cond.notify_all();
		if (!pthread_equal(g_scan_id, {})) {
			pthread_kill(g_scan_id, SIGALRM);
			pthread_join(g_scan_id, NULL);
		}
		for (auto tid : g_thread_ids) {
			pthread_kill(tid, SIGALRM);
			pthread_join(tid, nullptr);
		}
	}
	g_thread_ids.clear();
	{ /* silence cov-scan, take locks even in single-thread scenarios */
		std::lock_guard lk(g_async_lock);
		g_tag_hash.clear();
		g_async_hash.clear();
	}
}

void asyncemsmdb_interface_free()
{
	double_list_free(&g_wakeup_list);
}

int asyncemsmdb_interface_async_wait(uint32_t async_id,
	ECDOASYNCWAITEX_IN *pin, ECDOASYNCWAITEX_OUT *pout)
{
	char tmp_tag[TAG_SIZE];
	
	auto pwait = new ASYNC_WAIT();
	if (NULL == pwait) {
		pout->flags_out = 0;
		pout->result = ecRejected;
		return DISPATCH_SUCCESS;
	}
	auto rpc_info = get_rpc_info();
	if (!emsmdb_interface_check_acxh(&pin->acxh, pwait->username, &pwait->cxr, TRUE) ||
		0 != strcasecmp(rpc_info.username, pwait->username)) {
		delete pwait;
		pout->flags_out = 0;
		pout->result = ecRejected;
		return DISPATCH_SUCCESS;
	}
	if (emsmdb_interface_notifications_pending(pin->acxh)) {
		delete pwait;
		pout->flags_out = FLAG_NOTIFICATION_PENDING;
		pout->result = ecSuccess;
		return DISPATCH_SUCCESS;
	}
	pwait->node.pdata = pwait;
	pwait->async_id = async_id;
	HX_strlower(pwait->username);
	pwait->wait_time = time(nullptr);
	if (async_id == 0)
		pwait->out_payload.context_id = pout->flags_out;
	else
		pwait->out_payload.pout = pout;
	snprintf(tmp_tag, std::size(tmp_tag), "%s:%d", pwait->username,
	         static_cast<int>(pwait->cxr));
	HX_strlower(tmp_tag);
	std::unique_lock as_hold(g_async_lock);
	if (async_id != 0) try {
		if (g_async_hash.size() >= 2 * get_context_num())
			throw std::bad_alloc();
		auto pair = g_async_hash.emplace(async_id, pwait);
		if (!pair.second)
			throw std::bad_alloc();
	} catch (const std::bad_alloc &) {
		as_hold.unlock();
		delete pwait;
		pout->flags_out = 0;
		pout->result = ecRejected;
		return DISPATCH_SUCCESS;
	}
	try {
		if (g_tag_hash.size() < g_tag_hash_max &&
		    g_tag_hash.emplace(tmp_tag, pwait).second)
			return DISPATCH_PENDING;
	} catch (const std::bad_alloc &) {
		mlog(LV_WARN, "W-1540: ENOMEM");
	}
	if (async_id != 0)
		g_async_hash.erase(async_id);
	as_hold.unlock();
	delete pwait;
	pout->flags_out = 0;
	pout->result = ecRejected;
	return DISPATCH_SUCCESS;
}

void asyncemsmdb_interface_reclaim(uint32_t async_id)
{
	char tmp_tag[TAG_SIZE];
	
	std::unique_lock as_hold(g_async_lock);
	auto iter = g_async_hash.find(async_id);
	if (iter == g_async_hash.end())
		return;
	auto pwait = iter->second;
	snprintf(tmp_tag, std::size(tmp_tag), "%s:%d", pwait->username,
	         static_cast<int>(pwait->cxr));
	HX_strlower(tmp_tag);
	g_tag_hash.erase(tmp_tag);
	g_async_hash.erase(async_id);
	as_hold.unlock();
	delete pwait;
}

/* called by moh_emsmdb module */
void asyncemsmdb_interface_remove(ACXH *pacxh)
{
	uint16_t cxr;
	char tmp_tag[TAG_SIZE];
	char username[UADDR_SIZE];

	if (!emsmdb_interface_check_acxh(pacxh, username, &cxr, false))
		return;
	snprintf(tmp_tag, std::size(tmp_tag), "%s:%d", username, cxr);
	HX_strlower(tmp_tag);
	std::unique_lock as_hold(g_async_lock);
	auto iter = g_tag_hash.find(tmp_tag);
	if (iter == g_tag_hash.cend())
		return;
	auto pwait = iter->second;
	if (pwait->async_id != 0)
		g_async_hash.erase(pwait->async_id);
	g_tag_hash.erase(iter);
	as_hold.unlock();
	delete pwait;
}

static void asyncemsmdb_interface_activate(
	ASYNC_WAIT *pwait, BOOL b_pending)
{
	if (0 == pwait->async_id) {
		active_hpm_context(pwait->out_payload.context_id, b_pending);
	} else if (rpc_build_environment(pwait->async_id)) {
		pwait->out_payload.pout->result = ecSuccess;
		pwait->out_payload.pout->flags_out = b_pending ? FLAG_NOTIFICATION_PENDING : 0;
		async_reply(pwait->async_id, pwait->out_payload.pout);
	}
	delete pwait;
}

void asyncemsmdb_interface_wakeup(const char *username, uint16_t cxr)
{
	char tmp_tag[TAG_SIZE];
	
	snprintf(tmp_tag, std::size(tmp_tag), "%s:%d",
	         username, static_cast<int>(cxr));
	HX_strlower(tmp_tag);
	std::unique_lock as_hold(g_async_lock);
	auto iter = g_tag_hash.find(tmp_tag);
	if (iter == g_tag_hash.cend())
		return;
	auto pwait = iter->second;
	g_tag_hash.erase(iter);
	if (pwait->async_id != 0)
		g_async_hash.erase(pwait->async_id);
	as_hold.unlock();
	std::unique_lock ll_hold(g_list_lock);
	double_list_append_as_tail(&g_wakeup_list, &pwait->node);
	ll_hold.unlock();
	g_waken_cond.notify_one();
}

static void *aemsi_thrwork(void *param)
{
	DOUBLE_LIST_NODE *pnode;
	std::mutex g_cond_mutex;

	while (true) {
		{
			std::unique_lock<std::mutex> cm_hold(g_cond_mutex);
			g_waken_cond.wait(cm_hold, [] {
				return g_notify_stop || double_list_get_nodes_num(&g_wakeup_list) > 0;
			});
			if (g_notify_stop)
				break;
		}
		{
			std::unique_lock<std::mutex> ll_hold(g_list_lock);
			pnode = double_list_pop_front(&g_wakeup_list);
		}
		if (pnode == nullptr)
			continue;
		asyncemsmdb_interface_activate(static_cast<ASYNC_WAIT *>(pnode->pdata), TRUE);
	}
	return nullptr;
}

static void *aemsi_scanwork(void *param)
{
	DOUBLE_LIST temp_list;
	DOUBLE_LIST_NODE *pnode;
	
	double_list_init(&temp_list);
	while (!g_notify_stop) {
		sleep(1);
		auto cur_time = time(nullptr);
		std::unique_lock as_hold(g_async_lock);
		for (auto iter = g_tag_hash.cbegin(); iter != g_tag_hash.end(); ){
			auto pwait = iter->second;
			if (cur_time - pwait->wait_time <= WAITING_INTERVAL - 3) {
				++iter;
				continue;
			}
			iter = g_tag_hash.erase(iter);
			if (pwait->async_id != 0)
				g_async_hash.erase(pwait->async_id);
			double_list_append_as_tail(&temp_list, &pwait->node);
		}
		as_hold.unlock();
		while ((pnode = double_list_pop_front(&temp_list)) != nullptr)
			asyncemsmdb_interface_activate(static_cast<ASYNC_WAIT *>(pnode->pdata), FALSE);
	}
	double_list_free(&temp_list);
	return nullptr;
}
