// Copyright 2023 Dolthub, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package types

type SpatialRef struct {
	Name          string
	ID            uint32
	Organization  interface{}
	OrgCoordsysId interface{}
	Definition    string
	Description   interface{}
}

var SupportedSRIDs = map[uint32]SpatialRef{
	uint32(0):     {``, uint32(0), nil, nil, ``, nil},
	uint32(2000):  {`Anguilla 1957 / British West Indies Grid`, uint32(2000), `EPSG`, uint32(2000), `PROJCS["Anguilla 1957 / British West Indies Grid",GEOGCS["Anguilla 1957",DATUM["Anguilla 1957",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465,AUTHORITY["EPSG","7012"]],AUTHORITY["EPSG","6600"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4600"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-62,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9995,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2000"]]`, nil},
	uint32(2001):  {`Antigua 1943 / British West Indies Grid`, uint32(2001), `EPSG`, uint32(2001), `PROJCS["Antigua 1943 / British West Indies Grid",GEOGCS["Antigua 1943",DATUM["Antigua 1943",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465,AUTHORITY["EPSG","7012"]],TOWGS84[-255,-15,71,0,0,0,0],AUTHORITY["EPSG","6601"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4601"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-62,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9995,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2001"]]`, nil},
	uint32(2002):  {`Dominica 1945 / British West Indies Grid`, uint32(2002), `EPSG`, uint32(2002), `PROJCS["Dominica 1945 / British West Indies Grid",GEOGCS["Dominica 1945",DATUM["Dominica 1945",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465,AUTHORITY["EPSG","7012"]],TOWGS84[725,685,536,0,0,0,0],AUTHORITY["EPSG","6602"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4602"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-62,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9995,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2002"]]`, nil},
	uint32(2003):  {`Grenada 1953 / British West Indies Grid`, uint32(2003), `EPSG`, uint32(2003), `PROJCS["Grenada 1953 / British West Indies Grid",GEOGCS["Grenada 1953",DATUM["Grenada 1953",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465,AUTHORITY["EPSG","7012"]],TOWGS84[72,213.7,93,0,0,0,0],AUTHORITY["EPSG","6603"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4603"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-62,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9995,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2003"]]`, nil},
	uint32(2004):  {`Montserrat 1958 / British West Indies Grid`, uint32(2004), `EPSG`, uint32(2004), `PROJCS["Montserrat 1958 / British West Indies Grid",GEOGCS["Montserrat 1958",DATUM["Montserrat 1958",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465,AUTHORITY["EPSG","7012"]],TOWGS84[174,359,365,0,0,0,0],AUTHORITY["EPSG","6604"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4604"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-62,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9995,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2004"]]`, nil},
	uint32(2005):  {`St. Kitts 1955 / British West Indies Grid`, uint32(2005), `EPSG`, uint32(2005), `PROJCS["St. Kitts 1955 / British West Indies Grid",GEOGCS["St. Kitts 1955",DATUM["St. Kitts 1955",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465,AUTHORITY["EPSG","7012"]],TOWGS84[9,183,236,0,0,0,0],AUTHORITY["EPSG","6605"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4605"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-62,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9995,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2005"]]`, nil},
	uint32(2006):  {`St. Lucia 1955 / British West Indies Grid`, uint32(2006), `EPSG`, uint32(2006), `PROJCS["St. Lucia 1955 / British West Indies Grid",GEOGCS["St. Lucia 1955",DATUM["St. Lucia 1955",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465,AUTHORITY["EPSG","7012"]],TOWGS84[-153,153,307,0,0,0,0],AUTHORITY["EPSG","6606"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4606"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-62,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9995,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2006"]]`, nil},
	uint32(2007):  {`St. Vincent 45 / British West Indies Grid`, uint32(2007), `EPSG`, uint32(2007), `PROJCS["St. Vincent 45 / British West Indies Grid",GEOGCS["St. Vincent 1945",DATUM["St. Vincent 1945",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465,AUTHORITY["EPSG","7012"]],TOWGS84[195.671,332.517,274.607,0,0,0,0],AUTHORITY["EPSG","6607"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4607"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-62,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9995,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2007"]]`, nil},
	uint32(2009):  {`NAD27(CGQ77) / SCoPQ zone 3`, uint32(2009), `EPSG`, uint32(2009), `PROJCS["NAD27(CGQ77) / SCoPQ zone 3",GEOGCS["NAD27(CGQ77)",DATUM["North American Datum 1927 (CGQ77)",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],AUTHORITY["EPSG","6609"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4609"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-58.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",304800,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2009"]]`, nil},
	uint32(2010):  {`NAD27(CGQ77) / SCoPQ zone 4`, uint32(2010), `EPSG`, uint32(2010), `PROJCS["NAD27(CGQ77) / SCoPQ zone 4",GEOGCS["NAD27(CGQ77)",DATUM["North American Datum 1927 (CGQ77)",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],AUTHORITY["EPSG","6609"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4609"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-61.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",304800,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2010"]]`, nil},
	uint32(2011):  {`NAD27(CGQ77) / SCoPQ zone 5`, uint32(2011), `EPSG`, uint32(2011), `PROJCS["NAD27(CGQ77) / SCoPQ zone 5",GEOGCS["NAD27(CGQ77)",DATUM["North American Datum 1927 (CGQ77)",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],AUTHORITY["EPSG","6609"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4609"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-64.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",304800,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2011"]]`, nil},
	uint32(2012):  {`NAD27(CGQ77) / SCoPQ zone 6`, uint32(2012), `EPSG`, uint32(2012), `PROJCS["NAD27(CGQ77) / SCoPQ zone 6",GEOGCS["NAD27(CGQ77)",DATUM["North American Datum 1927 (CGQ77)",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],AUTHORITY["EPSG","6609"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4609"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-67.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",304800,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2012"]]`, nil},
	uint32(2013):  {`NAD27(CGQ77) / SCoPQ zone 7`, uint32(2013), `EPSG`, uint32(2013), `PROJCS["NAD27(CGQ77) / SCoPQ zone 7",GEOGCS["NAD27(CGQ77)",DATUM["North American Datum 1927 (CGQ77)",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],AUTHORITY["EPSG","6609"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4609"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-70.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",304800,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2013"]]`, nil},
	uint32(2014):  {`NAD27(CGQ77) / SCoPQ zone 8`, uint32(2014), `EPSG`, uint32(2014), `PROJCS["NAD27(CGQ77) / SCoPQ zone 8",GEOGCS["NAD27(CGQ77)",DATUM["North American Datum 1927 (CGQ77)",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],AUTHORITY["EPSG","6609"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4609"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-73.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",304800,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2014"]]`, nil},
	uint32(2015):  {`NAD27(CGQ77) / SCoPQ zone 9`, uint32(2015), `EPSG`, uint32(2015), `PROJCS["NAD27(CGQ77) / SCoPQ zone 9",GEOGCS["NAD27(CGQ77)",DATUM["North American Datum 1927 (CGQ77)",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],AUTHORITY["EPSG","6609"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4609"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-76.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",304800,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2015"]]`, nil},
	uint32(2016):  {`NAD27(CGQ77) / SCoPQ zone 10`, uint32(2016), `EPSG`, uint32(2016), `PROJCS["NAD27(CGQ77) / SCoPQ zone 10",GEOGCS["NAD27(CGQ77)",DATUM["North American Datum 1927 (CGQ77)",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],AUTHORITY["EPSG","6609"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4609"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-79.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",304800,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2016"]]`, nil},
	uint32(2017):  {`NAD27(76) / MTM zone 8`, uint32(2017), `EPSG`, uint32(2017), `PROJCS["NAD27(76) / MTM zone 8",GEOGCS["NAD27(76)",DATUM["North American Datum 1927 (1976)",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],AUTHORITY["EPSG","6608"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4608"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-73.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",304800,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2017"]]`, nil},
	uint32(2018):  {`NAD27(76) / MTM zone 9`, uint32(2018), `EPSG`, uint32(2018), `PROJCS["NAD27(76) / MTM zone 9",GEOGCS["NAD27(76)",DATUM["North American Datum 1927 (1976)",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],AUTHORITY["EPSG","6608"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4608"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-76.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",304800,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2018"]]`, nil},
	uint32(2019):  {`NAD27(76) / MTM zone 10`, uint32(2019), `EPSG`, uint32(2019), `PROJCS["NAD27(76) / MTM zone 10",GEOGCS["NAD27(76)",DATUM["North American Datum 1927 (1976)",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],AUTHORITY["EPSG","6608"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4608"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-79.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",304800,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2019"]]`, nil},
	uint32(2020):  {`NAD27(76) / MTM zone 11`, uint32(2020), `EPSG`, uint32(2020), `PROJCS["NAD27(76) / MTM zone 11",GEOGCS["NAD27(76)",DATUM["North American Datum 1927 (1976)",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],AUTHORITY["EPSG","6608"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4608"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-82.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",304800,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2020"]]`, nil},
	uint32(2021):  {`NAD27(76) / MTM zone 12`, uint32(2021), `EPSG`, uint32(2021), `PROJCS["NAD27(76) / MTM zone 12",GEOGCS["NAD27(76)",DATUM["North American Datum 1927 (1976)",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],AUTHORITY["EPSG","6608"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4608"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-81,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",304800,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2021"]]`, nil},
	uint32(2022):  {`NAD27(76) / MTM zone 13`, uint32(2022), `EPSG`, uint32(2022), `PROJCS["NAD27(76) / MTM zone 13",GEOGCS["NAD27(76)",DATUM["North American Datum 1927 (1976)",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],AUTHORITY["EPSG","6608"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4608"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-84,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",304800,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2022"]]`, nil},
	uint32(2023):  {`NAD27(76) / MTM zone 14`, uint32(2023), `EPSG`, uint32(2023), `PROJCS["NAD27(76) / MTM zone 14",GEOGCS["NAD27(76)",DATUM["North American Datum 1927 (1976)",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],AUTHORITY["EPSG","6608"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4608"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-87,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",304800,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2023"]]`, nil},
	uint32(2024):  {`NAD27(76) / MTM zone 15`, uint32(2024), `EPSG`, uint32(2024), `PROJCS["NAD27(76) / MTM zone 15",GEOGCS["NAD27(76)",DATUM["North American Datum 1927 (1976)",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],AUTHORITY["EPSG","6608"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4608"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-90,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",304800,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2024"]]`, nil},
	uint32(2025):  {`NAD27(76) / MTM zone 16`, uint32(2025), `EPSG`, uint32(2025), `PROJCS["NAD27(76) / MTM zone 16",GEOGCS["NAD27(76)",DATUM["North American Datum 1927 (1976)",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],AUTHORITY["EPSG","6608"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4608"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-93,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",304800,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2025"]]`, nil},
	uint32(2026):  {`NAD27(76) / MTM zone 17`, uint32(2026), `EPSG`, uint32(2026), `PROJCS["NAD27(76) / MTM zone 17",GEOGCS["NAD27(76)",DATUM["North American Datum 1927 (1976)",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],AUTHORITY["EPSG","6608"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4608"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-96,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",304800,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2026"]]`, nil},
	uint32(2027):  {`NAD27(76) / UTM zone 15N`, uint32(2027), `EPSG`, uint32(2027), `PROJCS["NAD27(76) / UTM zone 15N",GEOGCS["NAD27(76)",DATUM["North American Datum 1927 (1976)",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],AUTHORITY["EPSG","6608"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4608"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-93,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2027"]]`, nil},
	uint32(2028):  {`NAD27(76) / UTM zone 16N`, uint32(2028), `EPSG`, uint32(2028), `PROJCS["NAD27(76) / UTM zone 16N",GEOGCS["NAD27(76)",DATUM["North American Datum 1927 (1976)",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],AUTHORITY["EPSG","6608"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4608"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-87,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2028"]]`, nil},
	uint32(2029):  {`NAD27(76) / UTM zone 17N`, uint32(2029), `EPSG`, uint32(2029), `PROJCS["NAD27(76) / UTM zone 17N",GEOGCS["NAD27(76)",DATUM["North American Datum 1927 (1976)",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],AUTHORITY["EPSG","6608"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4608"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-81,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2029"]]`, nil},
	uint32(2030):  {`NAD27(76) / UTM zone 18N`, uint32(2030), `EPSG`, uint32(2030), `PROJCS["NAD27(76) / UTM zone 18N",GEOGCS["NAD27(76)",DATUM["North American Datum 1927 (1976)",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],AUTHORITY["EPSG","6608"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4608"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-75,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2030"]]`, nil},
	uint32(2031):  {`NAD27(CGQ77) / UTM zone 17N`, uint32(2031), `EPSG`, uint32(2031), `PROJCS["NAD27(CGQ77) / UTM zone 17N",GEOGCS["NAD27(CGQ77)",DATUM["North American Datum 1927 (CGQ77)",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],AUTHORITY["EPSG","6609"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4609"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-81,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2031"]]`, nil},
	uint32(2032):  {`NAD27(CGQ77) / UTM zone 18N`, uint32(2032), `EPSG`, uint32(2032), `PROJCS["NAD27(CGQ77) / UTM zone 18N",GEOGCS["NAD27(CGQ77)",DATUM["North American Datum 1927 (CGQ77)",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],AUTHORITY["EPSG","6609"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4609"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-75,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2032"]]`, nil},
	uint32(2033):  {`NAD27(CGQ77) / UTM zone 19N`, uint32(2033), `EPSG`, uint32(2033), `PROJCS["NAD27(CGQ77) / UTM zone 19N",GEOGCS["NAD27(CGQ77)",DATUM["North American Datum 1927 (CGQ77)",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],AUTHORITY["EPSG","6609"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4609"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-69,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2033"]]`, nil},
	uint32(2034):  {`NAD27(CGQ77) / UTM zone 20N`, uint32(2034), `EPSG`, uint32(2034), `PROJCS["NAD27(CGQ77) / UTM zone 20N",GEOGCS["NAD27(CGQ77)",DATUM["North American Datum 1927 (CGQ77)",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],AUTHORITY["EPSG","6609"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4609"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-63,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2034"]]`, nil},
	uint32(2035):  {`NAD27(CGQ77) / UTM zone 21N`, uint32(2035), `EPSG`, uint32(2035), `PROJCS["NAD27(CGQ77) / UTM zone 21N",GEOGCS["NAD27(CGQ77)",DATUM["North American Datum 1927 (CGQ77)",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],AUTHORITY["EPSG","6609"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4609"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-57,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2035"]]`, nil},
	uint32(2039):  {`Israel 1993 / Israeli TM Grid`, uint32(2039), `EPSG`, uint32(2039), `PROJCS["Israel 1993 / Israeli TM Grid",GEOGCS["Israel 1993",DATUM["Israel 1993",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[-48,55,52,0,0,0,0],AUTHORITY["EPSG","6141"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4141"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",31.7343936111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",35.2045169444444,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1.0000067,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",219529.584,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",626907.39,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2039"]]`, nil},
	uint32(2040):  {`Locodjo 1965 / UTM zone 30N`, uint32(2040), `EPSG`, uint32(2040), `PROJCS["Locodjo 1965 / UTM zone 30N",GEOGCS["Locodjo 1965",DATUM["Locodjo 1965",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465,AUTHORITY["EPSG","7012"]],TOWGS84[-125,53,467,0,0,0,0],AUTHORITY["EPSG","6142"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4142"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-3,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2040"]]`, nil},
	uint32(2041):  {`Abidjan 1987 / UTM zone 30N`, uint32(2041), `EPSG`, uint32(2041), `PROJCS["Abidjan 1987 / UTM zone 30N",GEOGCS["Abidjan 1987",DATUM["Abidjan 1987",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465,AUTHORITY["EPSG","7012"]],TOWGS84[-124.76,53,466.79,0,0,0,0],AUTHORITY["EPSG","6143"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4143"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-3,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2041"]]`, nil},
	uint32(2042):  {`Locodjo 1965 / UTM zone 29N`, uint32(2042), `EPSG`, uint32(2042), `PROJCS["Locodjo 1965 / UTM zone 29N",GEOGCS["Locodjo 1965",DATUM["Locodjo 1965",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465,AUTHORITY["EPSG","7012"]],TOWGS84[-125,53,467,0,0,0,0],AUTHORITY["EPSG","6142"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4142"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-9,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2042"]]`, nil},
	uint32(2043):  {`Abidjan 1987 / UTM zone 29N`, uint32(2043), `EPSG`, uint32(2043), `PROJCS["Abidjan 1987 / UTM zone 29N",GEOGCS["Abidjan 1987",DATUM["Abidjan 1987",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465,AUTHORITY["EPSG","7012"]],TOWGS84[-124.76,53,466.79,0,0,0,0],AUTHORITY["EPSG","6143"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4143"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-9,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2043"]]`, nil},
	uint32(2044):  {`Hanoi 1972 / Gauss-Kruger zone 18`, uint32(2044), `EPSG`, uint32(2044), `PROJCS["Hanoi 1972 / Gauss-Kruger zone 18",GEOGCS["Hanoi 1972",DATUM["Hanoi 1972",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[-17.51,-108.32,-62.39,0,0,0,0],AUTHORITY["EPSG","6147"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4147"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",105,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",18500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2044"]]`, nil},
	uint32(2045):  {`Hanoi 1972 / Gauss-Kruger zone 19`, uint32(2045), `EPSG`, uint32(2045), `PROJCS["Hanoi 1972 / Gauss-Kruger zone 19",GEOGCS["Hanoi 1972",DATUM["Hanoi 1972",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[-17.51,-108.32,-62.39,0,0,0,0],AUTHORITY["EPSG","6147"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4147"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",19500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2045"]]`, nil},
	uint32(2046):  {`Hartebeesthoek94 / Lo15`, uint32(2046), `EPSG`, uint32(2046), `PROJCS["Hartebeesthoek94 / Lo15",GEOGCS["Hartebeesthoek94",DATUM["Hartebeesthoek94",SPHEROID["WGS 84",6378137,298.257223563,AUTHORITY["EPSG","7030"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6148"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4148"]],PROJECTION["Transverse Mercator (South Orientated)",AUTHORITY["EPSG","9808"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",15,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["Y",WEST],AXIS["X",SOUTH],AUTHORITY["EPSG","2046"]]`, nil},
	uint32(2047):  {`Hartebeesthoek94 / Lo17`, uint32(2047), `EPSG`, uint32(2047), `PROJCS["Hartebeesthoek94 / Lo17",GEOGCS["Hartebeesthoek94",DATUM["Hartebeesthoek94",SPHEROID["WGS 84",6378137,298.257223563,AUTHORITY["EPSG","7030"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6148"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4148"]],PROJECTION["Transverse Mercator (South Orientated)",AUTHORITY["EPSG","9808"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",17,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["Y",WEST],AXIS["X",SOUTH],AUTHORITY["EPSG","2047"]]`, nil},
	uint32(2048):  {`Hartebeesthoek94 / Lo19`, uint32(2048), `EPSG`, uint32(2048), `PROJCS["Hartebeesthoek94 / Lo19",GEOGCS["Hartebeesthoek94",DATUM["Hartebeesthoek94",SPHEROID["WGS 84",6378137,298.257223563,AUTHORITY["EPSG","7030"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6148"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4148"]],PROJECTION["Transverse Mercator (South Orientated)",AUTHORITY["EPSG","9808"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",19,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["Y",WEST],AXIS["X",SOUTH],AUTHORITY["EPSG","2048"]]`, nil},
	uint32(2049):  {`Hartebeesthoek94 / Lo21`, uint32(2049), `EPSG`, uint32(2049), `PROJCS["Hartebeesthoek94 / Lo21",GEOGCS["Hartebeesthoek94",DATUM["Hartebeesthoek94",SPHEROID["WGS 84",6378137,298.257223563,AUTHORITY["EPSG","7030"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6148"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4148"]],PROJECTION["Transverse Mercator (South Orientated)",AUTHORITY["EPSG","9808"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",21,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["Y",WEST],AXIS["X",SOUTH],AUTHORITY["EPSG","2049"]]`, nil},
	uint32(2050):  {`Hartebeesthoek94 / Lo23`, uint32(2050), `EPSG`, uint32(2050), `PROJCS["Hartebeesthoek94 / Lo23",GEOGCS["Hartebeesthoek94",DATUM["Hartebeesthoek94",SPHEROID["WGS 84",6378137,298.257223563,AUTHORITY["EPSG","7030"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6148"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4148"]],PROJECTION["Transverse Mercator (South Orientated)",AUTHORITY["EPSG","9808"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",23,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["Y",WEST],AXIS["X",SOUTH],AUTHORITY["EPSG","2050"]]`, nil},
	uint32(2051):  {`Hartebeesthoek94 / Lo25`, uint32(2051), `EPSG`, uint32(2051), `PROJCS["Hartebeesthoek94 / Lo25",GEOGCS["Hartebeesthoek94",DATUM["Hartebeesthoek94",SPHEROID["WGS 84",6378137,298.257223563,AUTHORITY["EPSG","7030"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6148"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4148"]],PROJECTION["Transverse Mercator (South Orientated)",AUTHORITY["EPSG","9808"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",25,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["Y",WEST],AXIS["X",SOUTH],AUTHORITY["EPSG","2051"]]`, nil},
	uint32(2052):  {`Hartebeesthoek94 / Lo27`, uint32(2052), `EPSG`, uint32(2052), `PROJCS["Hartebeesthoek94 / Lo27",GEOGCS["Hartebeesthoek94",DATUM["Hartebeesthoek94",SPHEROID["WGS 84",6378137,298.257223563,AUTHORITY["EPSG","7030"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6148"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4148"]],PROJECTION["Transverse Mercator (South Orientated)",AUTHORITY["EPSG","9808"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",27,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["Y",WEST],AXIS["X",SOUTH],AUTHORITY["EPSG","2052"]]`, nil},
	uint32(2053):  {`Hartebeesthoek94 / Lo29`, uint32(2053), `EPSG`, uint32(2053), `PROJCS["Hartebeesthoek94 / Lo29",GEOGCS["Hartebeesthoek94",DATUM["Hartebeesthoek94",SPHEROID["WGS 84",6378137,298.257223563,AUTHORITY["EPSG","7030"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6148"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4148"]],PROJECTION["Transverse Mercator (South Orientated)",AUTHORITY["EPSG","9808"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",29,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["Y",WEST],AXIS["X",SOUTH],AUTHORITY["EPSG","2053"]]`, nil},
	uint32(2054):  {`Hartebeesthoek94 / Lo31`, uint32(2054), `EPSG`, uint32(2054), `PROJCS["Hartebeesthoek94 / Lo31",GEOGCS["Hartebeesthoek94",DATUM["Hartebeesthoek94",SPHEROID["WGS 84",6378137,298.257223563,AUTHORITY["EPSG","7030"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6148"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4148"]],PROJECTION["Transverse Mercator (South Orientated)",AUTHORITY["EPSG","9808"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",31,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["Y",WEST],AXIS["X",SOUTH],AUTHORITY["EPSG","2054"]]`, nil},
	uint32(2055):  {`Hartebeesthoek94 / Lo33`, uint32(2055), `EPSG`, uint32(2055), `PROJCS["Hartebeesthoek94 / Lo33",GEOGCS["Hartebeesthoek94",DATUM["Hartebeesthoek94",SPHEROID["WGS 84",6378137,298.257223563,AUTHORITY["EPSG","7030"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6148"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4148"]],PROJECTION["Transverse Mercator (South Orientated)",AUTHORITY["EPSG","9808"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",33,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["Y",WEST],AXIS["X",SOUTH],AUTHORITY["EPSG","2055"]]`, nil},
	uint32(2056):  {`CH1903+ / LV95`, uint32(2056), `EPSG`, uint32(2056), `PROJCS["CH1903+ / LV95",GEOGCS["CH1903+",DATUM["CH1903+",SPHEROID["Bessel 1841",6377397.155,299.1528128,AUTHORITY["EPSG","7004"]],TOWGS84[674.374,15.056,405.346,0,0,0,0],AUTHORITY["EPSG","6150"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4150"]],PROJECTION["Hotine Oblique Mercator (variant B)",AUTHORITY["EPSG","9815"]],PARAMETER["Latitude of projection centre",46.9524055555556,AUTHORITY["EPSG","8811"]],PARAMETER["Longitude of projection centre",7.43958333333333,AUTHORITY["EPSG","8812"]],PARAMETER["Azimuth of initial line",90,AUTHORITY["EPSG","8813"]],PARAMETER["Angle from Rectified to Skew Grid",90,AUTHORITY["EPSG","8814"]],PARAMETER["Scale factor on initial line",1,AUTHORITY["EPSG","8815"]],PARAMETER["Easting at projection centre",2600000,AUTHORITY["EPSG","8816"]],PARAMETER["Northing at projection centre",1200000,AUTHORITY["EPSG","8817"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2056"]]`, nil},
	uint32(2057):  {`Rassadiran / Nakhl e Taqi`, uint32(2057), `EPSG`, uint32(2057), `PROJCSuint32([")Rassadiran / Nakhl e Taqi",GEOGCS["Rassadiran",DATUM["Rassadiran",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-133.63,-157.5,-158.62,0,0,0,0],AUTHORITY["EPSG","6153"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4153"]],PROJECTION["Hotine Oblique Mercator (variant B)",AUTHORITY["EPSG","9815"]],PARAMETER["Latitude of projection centre",27.5188288055556,AUTHORITY["EPSG","8811"]],PARAMETER["Longitude of projection centre",52.6035391666667,AUTHORITY["EPSG","8812"]],PARAMETER["Azimuth of initial line",0.571661194444444,AUTHORITY["EPSG","8813"]],PARAMETER["Angle from Rectified to Skew Grid",0.571661194444444,AUTHORITY["EPSG","8814"]],PARAMETER["Scale factor on initial line",0.999895934,AUTHORITY["EPSG","8815"]],PARAMETER["Easting at projection centre",658377.437,AUTHORITY["EPSG","8816"]],PARAMETER["Northing at projection centre",3044969.194,AUTHORITY["EPSG","8817"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2057"]]`, nil},
	uint32(2058):  {`ED50(ED77) / UTM zone 38N`, uint32(2058), `EPSG`, uint32(2058), `PROJCS["ED50(ED77) / UTM zone 38N",GEOGCS["ED50(ED77)",DATUM["European Datum 1950(1977)",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-110.33,-97.73,-119.85,0.3423,1.1634,0.2715,0.063],AUTHORITY["EPSG","6154"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4154"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",45,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2058"]]`, nil},
	uint32(2059):  {`ED50(ED77) / UTM zone 39N`, uint32(2059), `EPSG`, uint32(2059), `PROJCS["ED50(ED77) / UTM zone 39N",GEOGCS["ED50(ED77)",DATUM["European Datum 1950(1977)",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-110.33,-97.73,-119.85,0.3423,1.1634,0.2715,0.063],AUTHORITY["EPSG","6154"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4154"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",51,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2059"]]`, nil},
	uint32(2060):  {`ED50(ED77) / UTM zone 40N`, uint32(2060), `EPSG`, uint32(2060), `PROJCS["ED50(ED77) / UTM zone 40N",GEOGCS["ED50(ED77)",DATUM["European Datum 1950(1977)",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-110.33,-97.73,-119.85,0.3423,1.1634,0.2715,0.063],AUTHORITY["EPSG","6154"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4154"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",57,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2060"]]`, nil},
	uint32(2061):  {`ED50(ED77) / UTM zone 41N`, uint32(2061), `EPSG`, uint32(2061), `PROJCS["ED50(ED77) / UTM zone 41N",GEOGCS["ED50(ED77)",DATUM["European Datum 1950(1977)",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-110.33,-97.73,-119.85,0.3423,1.1634,0.2715,0.063],AUTHORITY["EPSG","6154"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4154"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",63,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2061"]]`, nil},
	uint32(2062):  {`Madrid 1870 (Madrid) / Spain`, uint32(2062), `EPSG`, uint32(2062), `PROJCS["Madrid 1870 (Madrid) / Spain",GEOGCS["Madrid 1870 (Madrid)",DATUM["Madrid 1870 (Madrid)",SPHEROID["Struve 1860",6378298.3,294.73,AUTHORITY["EPSG","7028"]],AUTHORITY["EPSG","6903"]],PRIMEM["Madrid",-3.6879388888888895,AUTHORITY["EPSG","8905"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4903"]],PROJECTION["Lambert Conic Conformal (1SP)",AUTHORITY["EPSG","9801"]],PARAMETER["Latitude of natural origin",40,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",0,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9988085293,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",600000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",600000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2062"]]`, nil},
	uint32(2065):  {`S-JTSK (Ferro) / Krovak`, uint32(2065), `EPSG`, uint32(2065), `PROJCS["S-JTSK (Ferro) / Krovak",GEOGCS["S-JTSK (Ferro)",DATUM["System of the Unified Trigonometrical Cadastral Network (Ferro)",SPHEROID["Bessel 1841",6377397.155,299.1528128,AUTHORITY["EPSG","7004"]],AUTHORITY["EPSG","6818"]],PRIMEM["Ferro",-17.677777777777774,AUTHORITY["EPSG","8909"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4818"]],PROJECTION["Krovak",AUTHORITY["EPSG","9819"]],PARAMETER["Latitude of projection centre",49.5111111111111,AUTHORITY["EPSG","8811"]],PARAMETER["Longitude of origin",42.5111111111111,AUTHORITY["EPSG","8833"]],PARAMETER["Co-latitude of cone axis",30.2881397222222,AUTHORITY["EPSG","1036"]],PARAMETER["Latitude of pseudo standard parallel",78.5111111111111,AUTHORITY["EPSG","8818"]],PARAMETER["Scale factor on pseudo standard parallel",0.9999,AUTHORITY["EPSG","8819"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",SOUTH],AXIS["Y",WEST],AUTHORITY["EPSG","2065"]]`, nil},
	uint32(2066):  {`Mount Dillon / Tobago Grid`, uint32(2066), `EPSG`, uint32(2066), `PROJCS["Mount Dillon / Tobago Grid",GEOGCS["Mount Dillon",DATUM["Mount Dillon",SPHEROID["Clarke 1858",6378293.645208759,294.26067636926103,AUTHORITY["EPSG","7007"]],AUTHORITY["EPSG","6157"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4157"]],PROJECTION["Cassini-Soldner",AUTHORITY["EPSG","9806"]],PARAMETER["Latitude of natural origin",11.2521786111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-60.6860088888889,AUTHORITY["EPSG","8802"]],PARAMETER["False easting",187500,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",180000,AUTHORITY["EPSG","8807"]],UNIT["Clarke's link",0.201166195164,AUTHORITY["EPSG","9039"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2066"]]`, nil},
	uint32(2067):  {`Naparima 1955 / UTM zone 20N`, uint32(2067), `EPSG`, uint32(2067), `PROJCS["Naparima 1955 / UTM zone 20N",GEOGCS["Naparima 1955",DATUM["Naparima 1955",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-0.465,372.095,171.736,0,0,0,0],AUTHORITY["EPSG","6158"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4158"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-63,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2067"]]`, nil},
	uint32(2068):  {`ELD79 / Libya zone 5`, uint32(2068), `EPSG`, uint32(2068), `PROJCS["ELD79 / Libya zone 5",GEOGCS["ELD79",DATUM["European Libyan Datum 1979",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-194.513,-63.978,-25.759,-3.4027,3.756,-3.352,-0.9175],AUTHORITY["EPSG","6159"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4159"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",9,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",200000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2068"]]`, nil},
	uint32(2069):  {`ELD79 / Libya zone 6`, uint32(2069), `EPSG`, uint32(2069), `PROJCS["ELD79 / Libya zone 6",GEOGCS["ELD79",DATUM["European Libyan Datum 1979",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-194.513,-63.978,-25.759,-3.4027,3.756,-3.352,-0.9175],AUTHORITY["EPSG","6159"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4159"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",11,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",200000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2069"]]`, nil},
	uint32(2070):  {`ELD79 / Libya zone 7`, uint32(2070), `EPSG`, uint32(2070), `PROJCS["ELD79 / Libya zone 7",GEOGCS["ELD79",DATUM["European Libyan Datum 1979",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-194.513,-63.978,-25.759,-3.4027,3.756,-3.352,-0.9175],AUTHORITY["EPSG","6159"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4159"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",13,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",200000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2070"]]`, nil},
	uint32(2071):  {`ELD79 / Libya zone 8`, uint32(2071), `EPSG`, uint32(2071), `PROJCS["ELD79 / Libya zone 8",GEOGCS["ELD79",DATUM["European Libyan Datum 1979",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-194.513,-63.978,-25.759,-3.4027,3.756,-3.352,-0.9175],AUTHORITY["EPSG","6159"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4159"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",15,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",200000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2071"]]`, nil},
	uint32(2072):  {`ELD79 / Libya zone 9`, uint32(2072), `EPSG`, uint32(2072), `PROJCS["ELD79 / Libya zone 9",GEOGCS["ELD79",DATUM["European Libyan Datum 1979",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-194.513,-63.978,-25.759,-3.4027,3.756,-3.352,-0.9175],AUTHORITY["EPSG","6159"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4159"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",17,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",200000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2072"]]`, nil},
	uint32(2073):  {`ELD79 / Libya zone 10`, uint32(2073), `EPSG`, uint32(2073), `PROJCS["ELD79 / Libya zone 10",GEOGCS["ELD79",DATUM["European Libyan Datum 1979",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-194.513,-63.978,-25.759,-3.4027,3.756,-3.352,-0.9175],AUTHORITY["EPSG","6159"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4159"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",19,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",200000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2073"]]`, nil},
	uint32(2074):  {`ELD79 / Libya zone 11`, uint32(2074), `EPSG`, uint32(2074), `PROJCS["ELD79 / Libya zone 11",GEOGCS["ELD79",DATUM["European Libyan Datum 1979",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-194.513,-63.978,-25.759,-3.4027,3.756,-3.352,-0.9175],AUTHORITY["EPSG","6159"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4159"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",21,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",200000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2074"]]`, nil},
	uint32(2075):  {`ELD79 / Libya zone 12`, uint32(2075), `EPSG`, uint32(2075), `PROJCS["ELD79 / Libya zone 12",GEOGCS["ELD79",DATUM["European Libyan Datum 1979",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-194.513,-63.978,-25.759,-3.4027,3.756,-3.352,-0.9175],AUTHORITY["EPSG","6159"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4159"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",23,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",200000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2075"]]`, nil},
	uint32(2076):  {`ELD79 / Libya zone 13`, uint32(2076), `EPSG`, uint32(2076), `PROJCS["ELD79 / Libya zone 13",GEOGCS["ELD79",DATUM["European Libyan Datum 1979",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-194.513,-63.978,-25.759,-3.4027,3.756,-3.352,-0.9175],AUTHORITY["EPSG","6159"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4159"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",25,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",200000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2076"]]`, nil},
	uint32(2077):  {`ELD79 / UTM zone 32N`, uint32(2077), `EPSG`, uint32(2077), `PROJCS["ELD79 / UTM zone 32N",GEOGCS["ELD79",DATUM["European Libyan Datum 1979",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-194.513,-63.978,-25.759,-3.4027,3.756,-3.352,-0.9175],AUTHORITY["EPSG","6159"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4159"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",9,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2077"]]`, nil},
	uint32(2078):  {`ELD79 / UTM zone 33N`, uint32(2078), `EPSG`, uint32(2078), `PROJCS["ELD79 / UTM zone 33N",GEOGCS["ELD79",DATUM["European Libyan Datum 1979",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-194.513,-63.978,-25.759,-3.4027,3.756,-3.352,-0.9175],AUTHORITY["EPSG","6159"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4159"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",15,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2078"]]`, nil},
	uint32(2079):  {`ELD79 / UTM zone 34N`, uint32(2079), `EPSG`, uint32(2079), `PROJCS["ELD79 / UTM zone 34N",GEOGCS["ELD79",DATUM["European Libyan Datum 1979",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-194.513,-63.978,-25.759,-3.4027,3.756,-3.352,-0.9175],AUTHORITY["EPSG","6159"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4159"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",21,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2079"]]`, nil},
	uint32(2080):  {`ELD79 / UTM zone 35N`, uint32(2080), `EPSG`, uint32(2080), `PROJCS["ELD79 / UTM zone 35N",GEOGCS["ELD79",DATUM["European Libyan Datum 1979",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-194.513,-63.978,-25.759,-3.4027,3.756,-3.352,-0.9175],AUTHORITY["EPSG","6159"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4159"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",27,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2080"]]`, nil},
	uint32(2081):  {`Chos Malal 1914 / Argentina 2`, uint32(2081), `EPSG`, uint32(2081), `PROJCS["Chos Malal 1914 / Argentina 2",GEOGCS["Chos Malal 1914",DATUM["Chos Malal 1914",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],AUTHORITY["EPSG","6160"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4160"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-90,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-69,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",2500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2081"]]`, nil},
	uint32(2082):  {`Pampa del Castillo / Argentina 2`, uint32(2082), `EPSG`, uint32(2082), `PROJCS["Pampa del Castillo / Argentina 2",GEOGCS["Pampa del Castillo",DATUM["Pampa del Castillo",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[27.5,14,186.4,0,0,0,0],AUTHORITY["EPSG","6161"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4161"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-90,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-69,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",2500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2082"]]`, nil},
	uint32(2083):  {`Hito XVIII 1963 / Argentina 2`, uint32(2083), `EPSG`, uint32(2083), `PROJCS["Hito XVIII 1963 / Argentina 2",GEOGCS["Hito XVIII 1963",DATUM["Hito XVIII 1963",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[18.38,192.45,96.82,0.056,-0.142,-0.2,-0.0013],AUTHORITY["EPSG","6254"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4254"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-90,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-69,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",2500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2083"]]`, nil},
	uint32(2084):  {`Hito XVIII 1963 / UTM zone 19S`, uint32(2084), `EPSG`, uint32(2084), `PROJCS["Hito XVIII 1963 / UTM zone 19S",GEOGCS["Hito XVIII 1963",DATUM["Hito XVIII 1963",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[18.38,192.45,96.82,0.056,-0.142,-0.2,-0.0013],AUTHORITY["EPSG","6254"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4254"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-69,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2084"]]`, nil},
	uint32(2087):  {`ELD79 / TM 12 NE`, uint32(2087), `EPSG`, uint32(2087), `PROJCS["ELD79 / TM 12 NE",GEOGCS["ELD79",DATUM["European Libyan Datum 1979",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-194.513,-63.978,-25.759,-3.4027,3.756,-3.352,-0.9175],AUTHORITY["EPSG","6159"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4159"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",12,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2087"]]`, nil},
	uint32(2088):  {`Carthage / TM 11 NE`, uint32(2088), `EPSG`, uint32(2088), `PROJCS["Carthage / TM 11 NE",GEOGCS["Carthage",DATUM["Carthage",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.4660212936269,AUTHORITY["EPSG","7011"]],TOWGS84[-263,6,431,0,0,0,0],AUTHORITY["EPSG","6223"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4223"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",11,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2088"]]`, nil},
	uint32(2089):  {`Yemen NGN96 / UTM zone 38N`, uint32(2089), `EPSG`, uint32(2089), `PROJCS["Yemen NGN96 / UTM zone 38N",GEOGCS["Yemen NGN96",DATUM["Yemen National Geodetic Network 1996",SPHEROID["WGS 84",6378137,298.257223563,AUTHORITY["EPSG","7030"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6163"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4163"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",45,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2089"]]`, nil},
	uint32(2090):  {`Yemen NGN96 / UTM zone 39N`, uint32(2090), `EPSG`, uint32(2090), `PROJCS["Yemen NGN96 / UTM zone 39N",GEOGCS["Yemen NGN96",DATUM["Yemen National Geodetic Network 1996",SPHEROID["WGS 84",6378137,298.257223563,AUTHORITY["EPSG","7030"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6163"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4163"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",51,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2090"]]`, nil},
	uint32(2093):  {`Hanoi 1972 / GK 106 NE`, uint32(2093), `EPSG`, uint32(2093), `PROJCS["Hanoi 1972 / GK 106 NE",GEOGCS["Hanoi 1972",DATUM["Hanoi 1972",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[-17.51,-108.32,-62.39,0,0,0,0],AUTHORITY["EPSG","6147"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4147"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",106,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2093"]]`, nil},
	uint32(2094):  {`WGS 72BE / TM 106 NE`, uint32(2094), `EPSG`, uint32(2094), `PROJCS["WGS 72BE / TM 106 NE",GEOGCS["WGS 72BE",DATUM["WGS 72 Transit Broadcast Ephemeris",SPHEROID["WGS 72",6378135,298.26,AUTHORITY["EPSG","7043"]],TOWGS84[0,0,1.9,0,0,0.814,-0.38],AUTHORITY["EPSG","6324"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4324"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",106,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2094"]]`, nil},
	uint32(2095):  {`Bissau / UTM zone 28N`, uint32(2095), `EPSG`, uint32(2095), `PROJCS["Bissau / UTM zone 28N",GEOGCS["Bissau",DATUM["Bissau",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-173,253,27,0,0,0,0],AUTHORITY["EPSG","6165"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4165"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-15,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2095"]]`, nil},
	uint32(2096):  {`Korean 1985 / East Belt`, uint32(2096), `EPSG`, uint32(2096), `PROJCS["Korean 1985 / East Belt",GEOGCS["Korean 1985",DATUM["Korean Datum 1985",SPHEROID["Bessel 1841",6377397.155,299.1528128,AUTHORITY["EPSG","7004"]],TOWGS84[-145.907,505.034,685.756,1.162,-2.347,-1.592,6.342],AUTHORITY["EPSG","6162"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4162"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",38,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",129,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",200000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",500000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2096"]]`, nil},
	uint32(2097):  {`Korean 1985 / Central Belt`, uint32(2097), `EPSG`, uint32(2097), `PROJCS["Korean 1985 / Central Belt",GEOGCS["Korean 1985",DATUM["Korean Datum 1985",SPHEROID["Bessel 1841",6377397.155,299.1528128,AUTHORITY["EPSG","7004"]],TOWGS84[-145.907,505.034,685.756,1.162,-2.347,-1.592,6.342],AUTHORITY["EPSG","6162"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4162"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",38,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",127,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",200000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",500000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2097"]]`, nil},
	uint32(2098):  {`Korean 1985 / West Belt`, uint32(2098), `EPSG`, uint32(2098), `PROJCS["Korean 1985 / West Belt",GEOGCS["Korean 1985",DATUM["Korean Datum 1985",SPHEROID["Bessel 1841",6377397.155,299.1528128,AUTHORITY["EPSG","7004"]],TOWGS84[-145.907,505.034,685.756,1.162,-2.347,-1.592,6.342],AUTHORITY["EPSG","6162"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4162"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",38,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",125,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",200000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",500000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2098"]]`, nil},
	uint32(2099):  {`Qatar 1948 / Qatar Grid`, uint32(2099), `EPSG`, uint32(2099), `PROJCS["Qatar 1948 / Qatar Grid",GEOGCS["Qatar 1948",DATUM["Qatar 1948",SPHEROID["Helmert 1906",6378200,298.3,AUTHORITY["EPSG","7020"]],AUTHORITY["EPSG","6286"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4286"]],PROJECTION["Cassini-Soldner",AUTHORITY["EPSG","9806"]],PARAMETER["Latitude of natural origin",25.3823611111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",50.7613888888889,AUTHORITY["EPSG","8802"]],PARAMETER["False easting",100000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",100000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2099"]]`, nil},
	uint32(2100):  {`GGRS87 / Greek Grid`, uint32(2100), `EPSG`, uint32(2100), `PROJCS["GGRS87 / Greek Grid",GEOGCS["GGRS87",DATUM["Greek Geodetic Reference System 1987",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[-199.87,74.79,246.62,0,0,0,0],AUTHORITY["EPSG","6121"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4121"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",24,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2100"]]`, nil},
	uint32(2101):  {`Lake / Maracaibo Grid M1`, uint32(2101), `EPSG`, uint32(2101), `PROJCS["Lake / Maracaibo Grid M1",GEOGCS["Lake",DATUM["Lake",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],AUTHORITY["EPSG","6249"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4249"]],PROJECTION["Lambert Conic Conformal (1SP)",AUTHORITY["EPSG","9801"]],PARAMETER["Latitude of natural origin",10.1777777777778,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-71.6056177777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",-52684.972,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2101"]]`, nil},
	uint32(2102):  {`Lake / Maracaibo Grid`, uint32(2102), `EPSG`, uint32(2102), `PROJCS["Lake / Maracaibo Grid",GEOGCS["Lake",DATUM["Lake",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],AUTHORITY["EPSG","6249"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4249"]],PROJECTION["Lambert Conic Conformal (1SP)",AUTHORITY["EPSG","9801"]],PARAMETER["Latitude of natural origin",10.1777777777778,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-71.6056177777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",200000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",147315.028,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2102"]]`, nil},
	uint32(2103):  {`Lake / Maracaibo Grid M3`, uint32(2103), `EPSG`, uint32(2103), `PROJCS["Lake / Maracaibo Grid M3",GEOGCS["Lake",DATUM["Lake",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],AUTHORITY["EPSG","6249"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4249"]],PROJECTION["Lambert Conic Conformal (1SP)",AUTHORITY["EPSG","9801"]],PARAMETER["Latitude of natural origin",10.1777777777778,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-71.6056177777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",447315.028,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2103"]]`, nil},
	uint32(2104):  {`Lake / Maracaibo La Rosa Grid`, uint32(2104), `EPSG`, uint32(2104), `PROJCS["Lake / Maracaibo La Rosa Grid",GEOGCS["Lake",DATUM["Lake",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],AUTHORITY["EPSG","6249"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4249"]],PROJECTION["Lambert Conic Conformal (1SP)",AUTHORITY["EPSG","9801"]],PARAMETER["Latitude of natural origin",10.1777777777778,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-71.6056177777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",-17044,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",-23139.97,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2104"]]`, nil},
	uint32(2105):  {`NZGD2000 / Mount Eden 2000`, uint32(2105), `EPSG`, uint32(2105), `PROJCS["NZGD2000 / Mount Eden 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-36.8797222222222,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",174.764166666667,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2105"]]`, nil},
	uint32(2106):  {`NZGD2000 / Bay of Plenty 2000`, uint32(2106), `EPSG`, uint32(2106), `PROJCS["NZGD2000 / Bay of Plenty 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-37.7611111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",176.466111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2106"]]`, nil},
	uint32(2107):  {`NZGD2000 / Poverty Bay 2000`, uint32(2107), `EPSG`, uint32(2107), `PROJCS["NZGD2000 / Poverty Bay 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-38.6244444444444,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",177.885555555556,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2107"]]`, nil},
	uint32(2108):  {`NZGD2000 / Hawkes Bay 2000`, uint32(2108), `EPSG`, uint32(2108), `PROJCS["NZGD2000 / Hawkes Bay 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-39.6508333333333,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",176.673611111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2108"]]`, nil},
	uint32(2109):  {`NZGD2000 / Taranaki 2000`, uint32(2109), `EPSG`, uint32(2109), `PROJCS["NZGD2000 / Taranaki 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-39.1355555555556,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",174.227777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2109"]]`, nil},
	uint32(2110):  {`NZGD2000 / Tuhirangi 2000`, uint32(2110), `EPSG`, uint32(2110), `PROJCS["NZGD2000 / Tuhirangi 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-39.5122222222222,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",175.64,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2110"]]`, nil},
	uint32(2111):  {`NZGD2000 / Wanganui 2000`, uint32(2111), `EPSG`, uint32(2111), `PROJCS["NZGD2000 / Wanganui 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-40.2419444444444,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",175.488055555556,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2111"]]`, nil},
	uint32(2112):  {`NZGD2000 / Wairarapa 2000`, uint32(2112), `EPSG`, uint32(2112), `PROJCS["NZGD2000 / Wairarapa 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-40.9252777777778,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",175.647222222222,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2112"]]`, nil},
	uint32(2113):  {`NZGD2000 / Wellington 2000`, uint32(2113), `EPSG`, uint32(2113), `PROJCS["NZGD2000 / Wellington 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-41.3011111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",174.776388888889,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2113"]]`, nil},
	uint32(2114):  {`NZGD2000 / Collingwood 2000`, uint32(2114), `EPSG`, uint32(2114), `PROJCS["NZGD2000 / Collingwood 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-40.7147222222222,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",172.671944444444,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2114"]]`, nil},
	uint32(2115):  {`NZGD2000 / Nelson 2000`, uint32(2115), `EPSG`, uint32(2115), `PROJCS["NZGD2000 / Nelson 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-41.2744444444444,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",173.299166666667,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2115"]]`, nil},
	uint32(2116):  {`NZGD2000 / Karamea 2000`, uint32(2116), `EPSG`, uint32(2116), `PROJCS["NZGD2000 / Karamea 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-41.2897222222222,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",172.108888888889,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2116"]]`, nil},
	uint32(2117):  {`NZGD2000 / Buller 2000`, uint32(2117), `EPSG`, uint32(2117), `PROJCS["NZGD2000 / Buller 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-41.8105555555556,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",171.581111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2117"]]`, nil},
	uint32(2118):  {`NZGD2000 / Grey 2000`, uint32(2118), `EPSG`, uint32(2118), `PROJCS["NZGD2000 / Grey 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-42.3336111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",171.549722222222,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2118"]]`, nil},
	uint32(2119):  {`NZGD2000 / Amuri 2000`, uint32(2119), `EPSG`, uint32(2119), `PROJCS["NZGD2000 / Amuri 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-42.6888888888889,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",173.01,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2119"]]`, nil},
	uint32(2120):  {`NZGD2000 / Marlborough 2000`, uint32(2120), `EPSG`, uint32(2120), `PROJCS["NZGD2000 / Marlborough 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-41.5444444444444,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",173.801944444445,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2120"]]`, nil},
	uint32(2121):  {`NZGD2000 / Hokitika 2000`, uint32(2121), `EPSG`, uint32(2121), `PROJCS["NZGD2000 / Hokitika 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-42.8861111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",170.979722222222,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2121"]]`, nil},
	uint32(2122):  {`NZGD2000 / Okarito 2000`, uint32(2122), `EPSG`, uint32(2122), `PROJCS["NZGD2000 / Okarito 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-43.11,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",170.260833333333,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2122"]]`, nil},
	uint32(2123):  {`NZGD2000 / Jacksons Bay 2000`, uint32(2123), `EPSG`, uint32(2123), `PROJCS["NZGD2000 / Jacksons Bay 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-43.9777777777778,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",168.606111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2123"]]`, nil},
	uint32(2124):  {`NZGD2000 / Mount Pleasant 2000`, uint32(2124), `EPSG`, uint32(2124), `PROJCS["NZGD2000 / Mount Pleasant 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-43.5905555555556,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",172.726944444444,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2124"]]`, nil},
	uint32(2125):  {`NZGD2000 / Gawler 2000`, uint32(2125), `EPSG`, uint32(2125), `PROJCS["NZGD2000 / Gawler 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-43.7486111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",171.360555555556,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2125"]]`, nil},
	uint32(2126):  {`NZGD2000 / Timaru 2000`, uint32(2126), `EPSG`, uint32(2126), `PROJCS["NZGD2000 / Timaru 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-44.4019444444444,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",171.057222222222,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2126"]]`, nil},
	uint32(2127):  {`NZGD2000 / Lindis Peak 2000`, uint32(2127), `EPSG`, uint32(2127), `PROJCS["NZGD2000 / Lindis Peak 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-44.735,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",169.4675,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2127"]]`, nil},
	uint32(2128):  {`NZGD2000 / Mount Nicholas 2000`, uint32(2128), `EPSG`, uint32(2128), `PROJCS["NZGD2000 / Mount Nicholas 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-45.1327777777778,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",168.398611111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2128"]]`, nil},
	uint32(2129):  {`NZGD2000 / Mount York 2000`, uint32(2129), `EPSG`, uint32(2129), `PROJCS["NZGD2000 / Mount York 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-45.5636111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",167.738611111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2129"]]`, nil},
	uint32(2130):  {`NZGD2000 / Observation Point 2000`, uint32(2130), `EPSG`, uint32(2130), `PROJCS["NZGD2000 / Observation Point 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-45.8161111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",170.628333333333,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2130"]]`, nil},
	uint32(2131):  {`NZGD2000 / North Taieri 2000`, uint32(2131), `EPSG`, uint32(2131), `PROJCS["NZGD2000 / North Taieri 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-45.8613888888889,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",170.2825,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.99996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2131"]]`, nil},
	uint32(2132):  {`NZGD2000 / Bluff 2000`, uint32(2132), `EPSG`, uint32(2132), `PROJCS["NZGD2000 / Bluff 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",-46.6111111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",168.342777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2132"]]`, nil},
	uint32(2133):  {`NZGD2000 / UTM zone 58S`, uint32(2133), `EPSG`, uint32(2133), `PROJCS["NZGD2000 / UTM zone 58S",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",165,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2133"]]`, nil},
	uint32(2134):  {`NZGD2000 / UTM zone 59S`, uint32(2134), `EPSG`, uint32(2134), `PROJCS["NZGD2000 / UTM zone 59S",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",171,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2134"]]`, nil},
	uint32(2135):  {`NZGD2000 / UTM zone 60S`, uint32(2135), `EPSG`, uint32(2135), `PROJCS["NZGD2000 / UTM zone 60S",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",177,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2135"]]`, nil},
	uint32(2136):  {`Accra / Ghana National Grid`, uint32(2136), `EPSG`, uint32(2136), `PROJCS["Accra / Ghana National Grid",GEOGCS["Accra",DATUM["Accra",SPHEROID["War Office",6378300,296,AUTHORITY["EPSG","7029"]],TOWGS84[-199,32,322,0,0,0,0],AUTHORITY["EPSG","6168"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4168"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",4.66666666666667,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-1,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.99975,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",900000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["Gold Coast foot",0.3047997101815088,AUTHORITY["EPSG","9094"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2136"]]`, nil},
	uint32(2137):  {`Accra / TM 1 NW`, uint32(2137), `EPSG`, uint32(2137), `PROJCS["Accra / TM 1 NW",GEOGCS["Accra",DATUM["Accra",SPHEROID["War Office",6378300,296,AUTHORITY["EPSG","7029"]],TOWGS84[-199,32,322,0,0,0,0],AUTHORITY["EPSG","6168"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4168"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-1,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2137"]]`, nil},
	uint32(2138):  {`NAD27(CGQ77) / Quebec Lambert`, uint32(2138), `EPSG`, uint32(2138), `PROJCS["NAD27(CGQ77) / Quebec Lambert",GEOGCS["NAD27(CGQ77)",DATUM["North American Datum 1927 (CGQ77)",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],AUTHORITY["EPSG","6609"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4609"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",44,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-68.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",60,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",46,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",0,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2138"]]`, nil},
	uint32(2154):  {`RGF93 / Lambert-93`, uint32(2154), `EPSG`, uint32(2154), `PROJCS["RGF93 / Lambert-93",GEOGCS["RGF93",DATUM["Reseau Geodesique Francais 1993",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6171"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4171"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",46.5111111111111,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",3,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",49,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",44,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",700000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",6600000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2154"]]`, nil},
	uint32(2157):  {`IRENET95 / Irish Transverse Mercator`, uint32(2157), `EPSG`, uint32(2157), `PROJCS["IRENET95 / Irish Transverse Mercator",GEOGCS["IRENET95",DATUM["IRENET95",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6173"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4173"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",53.5111111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-8,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.99982,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",600000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",750000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2157"]]`, nil},
	uint32(2158):  {`IRENET95 / UTM zone 29N`, uint32(2158), `EPSG`, uint32(2158), `PROJCS["IRENET95 / UTM zone 29N",GEOGCS["IRENET95",DATUM["IRENET95",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6173"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4173"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-9,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2158"]]`, nil},
	uint32(2159):  {`Sierra Leone 1924 / New Colony Grid`, uint32(2159), `EPSG`, uint32(2159), `PROJCS["Sierra Leone 1924 / New Colony Grid",GEOGCS["Sierra Leone 1924",DATUM["Sierra Leone Colony 1924",SPHEROID["War Office",6378300,296,AUTHORITY["EPSG","7029"]],AUTHORITY["EPSG","6174"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4174"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",6.66666666666667,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-12,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["Gold Coast foot",0.3047997101815088,AUTHORITY["EPSG","9094"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2159"]]`, nil},
	uint32(2160):  {`Sierra Leone 1924 / New War Office Grid`, uint32(2160), `EPSG`, uint32(2160), `PROJCS["Sierra Leone 1924 / New War Office Grid",GEOGCS["Sierra Leone 1924",DATUM["Sierra Leone Colony 1924",SPHEROID["War Office",6378300,296,AUTHORITY["EPSG","7029"]],AUTHORITY["EPSG","6174"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4174"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",6.66666666666667,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-12,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",800000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",600000,AUTHORITY["EPSG","8807"]],UNIT["Gold Coast foot",0.3047997101815088,AUTHORITY["EPSG","9094"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2160"]]`, nil},
	uint32(2161):  {`Sierra Leone 1968 / UTM zone 28N`, uint32(2161), `EPSG`, uint32(2161), `PROJCS["Sierra Leone 1968 / UTM zone 28N",GEOGCS["Sierra Leone 1968",DATUM["Sierra Leone 1968",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465,AUTHORITY["EPSG","7012"]],TOWGS84[-88,4,101,0,0,0,0],AUTHORITY["EPSG","6175"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4175"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-15,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2161"]]`, nil},
	uint32(2162):  {`Sierra Leone 1968 / UTM zone 29N`, uint32(2162), `EPSG`, uint32(2162), `PROJCS["Sierra Leone 1968 / UTM zone 29N",GEOGCS["Sierra Leone 1968",DATUM["Sierra Leone 1968",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465,AUTHORITY["EPSG","7012"]],TOWGS84[-88,4,101,0,0,0,0],AUTHORITY["EPSG","6175"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4175"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-9,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2162"]]`, nil},
	uint32(2163):  {`US National Atlas Equal Area`, uint32(2163), `EPSG`, uint32(2163), `PROJCS["US National Atlas Equal Area",GEOGCS["Unspecified datum based upon the Clarke 1866 Authalic Sphere",DATUM["Not specified (based on Clarke 1866 Authalic Sphere)",SPHEROID["Clarke 1866 Authalic Sphere",6370997,0,AUTHORITY["EPSG","7052"]],AUTHORITY["EPSG","6052"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4052"]],PROJECTION["Lambert Azimuthal Equal Area (Spherical)",AUTHORITY["EPSG","1027"]],PARAMETER["Latitude of natural origin",45,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-100,AUTHORITY["EPSG","8802"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2163"]]`, nil},
	uint32(2164):  {`Locodjo 1965 / TM 5 NW`, uint32(2164), `EPSG`, uint32(2164), `PROJCS["Locodjo 1965 / TM 5 NW",GEOGCS["Locodjo 1965",DATUM["Locodjo 1965",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465,AUTHORITY["EPSG","7012"]],TOWGS84[-125,53,467,0,0,0,0],AUTHORITY["EPSG","6142"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4142"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-5,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2164"]]`, nil},
	uint32(2165):  {`Abidjan 1987 / TM 5 NW`, uint32(2165), `EPSG`, uint32(2165), `PROJCS["Abidjan 1987 / TM 5 NW",GEOGCS["Abidjan 1987",DATUM["Abidjan 1987",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465,AUTHORITY["EPSG","7012"]],TOWGS84[-124.76,53,466.79,0,0,0,0],AUTHORITY["EPSG","6143"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4143"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-5,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2165"]]`, nil},
	uint32(2169):  {`Luxembourg 1930 / Gauss`, uint32(2169), `EPSG`, uint32(2169), `PROJCS["Luxembourg 1930 / Gauss",GEOGCS["Luxembourg 1930",DATUM["Luxembourg 1930",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-189.6806,18.3463,-42.7695,-0.33746,-3.09264,2.53861,0.4598],AUTHORITY["EPSG","6181"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4181"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",49.8333333333333,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",6.17777777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",80000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",100000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2169"]]`, nil},
	uint32(2172):  {`Pulkovo 1942(58) / Poland zone II`, uint32(2172), `EPSG`, uint32(2172), `PROJCS["Pulkovo 1942(58) / Poland zone II",GEOGCS["Pulkovo 1942(58)",DATUM["Pulkovo 1942(58)",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[33.4,-146.6,-76.3,-0.359,-0.053,0.844,-0.84],AUTHORITY["EPSG","6179"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4179"]],PROJECTION["Oblique Stereographic",AUTHORITY["EPSG","9809"]],PARAMETER["Latitude of natural origin",53.0019444444444,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",21.5027777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9998,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",4603000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",5806000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2172"]]`, nil},
	uint32(2173):  {`Pulkovo 1942(58) / Poland zone III`, uint32(2173), `EPSG`, uint32(2173), `PROJCS["Pulkovo 1942(58) / Poland zone III",GEOGCS["Pulkovo 1942(58)",DATUM["Pulkovo 1942(58)",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[33.4,-146.6,-76.3,-0.359,-0.053,0.844,-0.84],AUTHORITY["EPSG","6179"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4179"]],PROJECTION["Oblique Stereographic",AUTHORITY["EPSG","9809"]],PARAMETER["Latitude of natural origin",53.5833333333333,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",17.0083333333333,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9998,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",3501000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",5999000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2173"]]`, nil},
	uint32(2174):  {`Pulkovo 1942(58) / Poland zone IV`, uint32(2174), `EPSG`, uint32(2174), `PROJCS["Pulkovo 1942(58) / Poland zone IV",GEOGCS["Pulkovo 1942(58)",DATUM["Pulkovo 1942(58)",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[33.4,-146.6,-76.3,-0.359,-0.053,0.844,-0.84],AUTHORITY["EPSG","6179"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4179"]],PROJECTION["Oblique Stereographic",AUTHORITY["EPSG","9809"]],PARAMETER["Latitude of natural origin",51.6708333333333,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",16.6722222222222,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9998,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",3703000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",5627000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2174"]]`, nil},
	uint32(2175):  {`Pulkovo 1942(58) / Poland zone V`, uint32(2175), `EPSG`, uint32(2175), `PROJCS["Pulkovo 1942(58) / Poland zone V",GEOGCS["Pulkovo 1942(58)",DATUM["Pulkovo 1942(58)",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[33.4,-146.6,-76.3,-0.359,-0.053,0.844,-0.84],AUTHORITY["EPSG","6179"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4179"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",18.9583333333333,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999983,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",237000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",-4700000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2175"]]`, nil},
	uint32(2176):  {`ETRS89 / Poland CS2000 zone 5`, uint32(2176), `EPSG`, uint32(2176), `PROJCS["ETRS89 / Poland CS2000 zone 5",GEOGCS["ETRS89",DATUM["European Terrestrial Reference System 1989",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6258"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4258"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",15,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999923,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",5500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["x",NORTH],AXIS["y",EAST],AUTHORITY["EPSG","2176"]]`, nil},
	uint32(2177):  {`ETRS89 / Poland CS2000 zone 6`, uint32(2177), `EPSG`, uint32(2177), `PROJCS["ETRS89 / Poland CS2000 zone 6",GEOGCS["ETRS89",DATUM["European Terrestrial Reference System 1989",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6258"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4258"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",18,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999923,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",6500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["x",NORTH],AXIS["y",EAST],AUTHORITY["EPSG","2177"]]`, nil},
	uint32(2178):  {`ETRS89 / Poland CS2000 zone 7`, uint32(2178), `EPSG`, uint32(2178), `PROJCS["ETRS89 / Poland CS2000 zone 7",GEOGCS["ETRS89",DATUM["European Terrestrial Reference System 1989",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6258"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4258"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",21,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999923,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",7500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["x",NORTH],AXIS["y",EAST],AUTHORITY["EPSG","2178"]]`, nil},
	uint32(2179):  {`ETRS89 / Poland CS2000 zone 8`, uint32(2179), `EPSG`, uint32(2179), `PROJCS["ETRS89 / Poland CS2000 zone 8",GEOGCS["ETRS89",DATUM["European Terrestrial Reference System 1989",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6258"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4258"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",24,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999923,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",8500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["x",NORTH],AXIS["y",EAST],AUTHORITY["EPSG","2179"]]`, nil},
	uint32(2180):  {`ETRS89 / Poland CS92`, uint32(2180), `EPSG`, uint32(2180), `PROJCS["ETRS89 / Poland CS92",GEOGCS["ETRS89",DATUM["European Terrestrial Reference System 1989",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6258"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4258"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",19,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9993,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",-5300000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["x",NORTH],AXIS["y",EAST],AUTHORITY["EPSG","2180"]]`, nil},
	uint32(2188):  {`Azores Occidental 1939 / UTM zone 25N`, uint32(2188), `EPSG`, uint32(2188), `PROJCS["Azores Occidental 1939 / UTM zone 25N",GEOGCS["Azores Occidental 1939",DATUM["Azores Occidental Islands 1939",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-422.651,-172.995,84.02,0,0,0,0],AUTHORITY["EPSG","6182"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4182"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-33,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2188"]]`, nil},
	uint32(2189):  {`Azores Central 1948 / UTM zone 26N`, uint32(2189), `EPSG`, uint32(2189), `PROJCS["Azores Central 1948 / UTM zone 26N",GEOGCS["Azores Central 1948",DATUM["Azores Central Islands 1948",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-104,167,-38,0,0,0,0],AUTHORITY["EPSG","6183"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4183"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-27,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2189"]]`, nil},
	uint32(2190):  {`Azores Oriental 1940 / UTM zone 26N`, uint32(2190), `EPSG`, uint32(2190), `PROJCS["Azores Oriental 1940 / UTM zone 26N",GEOGCS["Azores Oriental 1940",DATUM["Azores Oriental Islands 1940",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-203,141,53,0,0,0,0],AUTHORITY["EPSG","6184"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4184"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-27,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2190"]]`, nil},
	uint32(2193):  {`NZGD2000 / New Zealand Transverse Mercator 2000`, uint32(2193), `EPSG`, uint32(2193), `PROJCS["NZGD2000 / New Zealand Transverse Mercator 2000",GEOGCS["NZGD2000",DATUM["New Zealand Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6167"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4167"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",173,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",1600000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2193"]]`, nil},
	uint32(2195):  {`NAD83(HARN) / UTM zone 2S`, uint32(2195), `EPSG`, uint32(2195), `PROJCS["NAD83(HARN) / UTM zone 2S",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-171,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2195"]]`, nil},
	uint32(2196):  {`ETRS89 / Kp2000 Jutland`, uint32(2196), `EPSG`, uint32(2196), `PROJCS["ETRS89 / Kp2000 Jutland",GEOGCS["ETRS89",DATUM["European Terrestrial Reference System 1989",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6258"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4258"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",9.5,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.99995,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",200000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2196"]]`, nil},
	uint32(2197):  {`ETRS89 / Kp2000 Zealand`, uint32(2197), `EPSG`, uint32(2197), `PROJCS["ETRS89 / Kp2000 Zealand",GEOGCS["ETRS89",DATUM["European Terrestrial Reference System 1989",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6258"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4258"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",12,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.99995,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2197"]]`, nil},
	uint32(2198):  {`ETRS89 / Kp2000 Bornholm`, uint32(2198), `EPSG`, uint32(2198), `PROJCS["ETRS89 / Kp2000 Bornholm",GEOGCS["ETRS89",DATUM["European Terrestrial Reference System 1989",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6258"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4258"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",15,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",900000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2198"]]`, nil},
	uint32(2200):  {`ATS77 / New Brunswick Stereographic (ATS77)`, uint32(2200), `EPSG`, uint32(2200), `PROJCS["ATS77 / New Brunswick Stereographic (ATS77)",GEOGCS["ATS77",DATUM["Average Terrestrial System 1977",SPHEROID["Average Terrestrial System 1977",6378135,298.257,AUTHORITY["EPSG","7041"]],AUTHORITY["EPSG","6122"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4122"]],PROJECTION["Oblique Stereographic",AUTHORITY["EPSG","9809"]],PARAMETER["Latitude of natural origin",46.5111111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-66.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999912,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",300000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2200"]]`, nil},
	uint32(2201):  {`REGVEN / UTM zone 18N`, uint32(2201), `EPSG`, uint32(2201), `PROJCS["REGVEN / UTM zone 18N",GEOGCS["REGVEN",DATUM["Red Geodesica Venezolana",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6189"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4189"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-75,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2201"]]`, nil},
	uint32(2202):  {`REGVEN / UTM zone 19N`, uint32(2202), `EPSG`, uint32(2202), `PROJCS["REGVEN / UTM zone 19N",GEOGCS["REGVEN",DATUM["Red Geodesica Venezolana",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6189"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4189"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-69,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2202"]]`, nil},
	uint32(2203):  {`REGVEN / UTM zone 20N`, uint32(2203), `EPSG`, uint32(2203), `PROJCS["REGVEN / UTM zone 20N",GEOGCS["REGVEN",DATUM["Red Geodesica Venezolana",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6189"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4189"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-63,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2203"]]`, nil},
	uint32(2204):  {`NAD27 / Tennessee`, uint32(2204), `EPSG`, uint32(2204), `PROJCS["NAD27 / Tennessee",GEOGCS["NAD27",DATUM["North American Datum 1927",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],TOWGS84[-32.3841359,180.4090461,120.8442577,-2.1545854,-0.1498782,0.5742915,8.1049164],AUTHORITY["EPSG","6267"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4267"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",34.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-86,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",35.2611111111111,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",36.4166666666667,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",2000000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",100000,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2204"]]`, nil},
	uint32(2205):  {`NAD83 / Kentucky North`, uint32(2205), `EPSG`, uint32(2205), `PROJCS["NAD83 / Kentucky North",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",37.5111111111111,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-84.25,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",37.9777777777778,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",38.9777777777778,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",500000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2205"]]`, nil},
	uint32(2206):  {`ED50 / 3-degree Gauss-Kruger zone 9`, uint32(2206), `EPSG`, uint32(2206), `PROJCS["ED50 / 3-degree Gauss-Kruger zone 9",GEOGCS["ED50",DATUM["European Datum 1950",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-157.89,-17.16,-78.41,2.118,2.697,-1.434,-5.38],AUTHORITY["EPSG","6230"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4230"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",27,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",9500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2206"]]`, nil},
	uint32(2207):  {`ED50 / 3-degree Gauss-Kruger zone 10`, uint32(2207), `EPSG`, uint32(2207), `PROJCS["ED50 / 3-degree Gauss-Kruger zone 10",GEOGCS["ED50",DATUM["European Datum 1950",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-157.89,-17.16,-78.41,2.118,2.697,-1.434,-5.38],AUTHORITY["EPSG","6230"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4230"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",30,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",10500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2207"]]`, nil},
	uint32(2208):  {`ED50 / 3-degree Gauss-Kruger zone 11`, uint32(2208), `EPSG`, uint32(2208), `PROJCS["ED50 / 3-degree Gauss-Kruger zone 11",GEOGCS["ED50",DATUM["European Datum 1950",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-157.89,-17.16,-78.41,2.118,2.697,-1.434,-5.38],AUTHORITY["EPSG","6230"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4230"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",33,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",11500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2208"]]`, nil},
	uint32(2209):  {`ED50 / 3-degree Gauss-Kruger zone 12`, uint32(2209), `EPSG`, uint32(2209), `PROJCS["ED50 / 3-degree Gauss-Kruger zone 12",GEOGCS["ED50",DATUM["European Datum 1950",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-157.89,-17.16,-78.41,2.118,2.697,-1.434,-5.38],AUTHORITY["EPSG","6230"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4230"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",36,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",12500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2209"]]`, nil},
	uint32(2210):  {`ED50 / 3-degree Gauss-Kruger zone 13`, uint32(2210), `EPSG`, uint32(2210), `PROJCS["ED50 / 3-degree Gauss-Kruger zone 13",GEOGCS["ED50",DATUM["European Datum 1950",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-157.89,-17.16,-78.41,2.118,2.697,-1.434,-5.38],AUTHORITY["EPSG","6230"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4230"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",39,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",13500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2210"]]`, nil},
	uint32(2211):  {`ED50 / 3-degree Gauss-Kruger zone 14`, uint32(2211), `EPSG`, uint32(2211), `PROJCS["ED50 / 3-degree Gauss-Kruger zone 14",GEOGCS["ED50",DATUM["European Datum 1950",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-157.89,-17.16,-78.41,2.118,2.697,-1.434,-5.38],AUTHORITY["EPSG","6230"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4230"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",42,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",14500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2211"]]`, nil},
	uint32(2212):  {`ED50 / 3-degree Gauss-Kruger zone 15`, uint32(2212), `EPSG`, uint32(2212), `PROJCS["ED50 / 3-degree Gauss-Kruger zone 15",GEOGCS["ED50",DATUM["European Datum 1950",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-157.89,-17.16,-78.41,2.118,2.697,-1.434,-5.38],AUTHORITY["EPSG","6230"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4230"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",45,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",15500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2212"]]`, nil},
	uint32(2213):  {`ETRS89 / TM 30 NE`, uint32(2213), `EPSG`, uint32(2213), `PROJCS["ETRS89 / TM 30 NE",GEOGCS["ETRS89",DATUM["European Terrestrial Reference System 1989",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6258"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4258"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",30,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2213"]]`, nil},
	uint32(2215):  {`Manoca 1962 / UTM zone 32N`, uint32(2215), `EPSG`, uint32(2215), `PROJCS["Manoca 1962 / UTM zone 32N",GEOGCS["Manoca 1962",DATUM["Manoca 1962",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.4660212936269,AUTHORITY["EPSG","7011"]],TOWGS84[-70.9,-151.8,-41.4,0,0,0,0],AUTHORITY["EPSG","6193"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4193"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",9,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2215"]]`, nil},
	uint32(2216):  {`Qornoq 1927 / UTM zone 22N`, uint32(2216), `EPSG`, uint32(2216), `PROJCS["Qornoq 1927 / UTM zone 22N",GEOGCS["Qornoq 1927",DATUM["Qornoq 1927",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[163.511,127.533,-159.789,0,0,0.814,-0.6],AUTHORITY["EPSG","6194"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4194"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-51,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2216"]]`, nil},
	uint32(2217):  {`Qornoq 1927 / UTM zone 23N`, uint32(2217), `EPSG`, uint32(2217), `PROJCS["Qornoq 1927 / UTM zone 23N",GEOGCS["Qornoq 1927",DATUM["Qornoq 1927",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[163.511,127.533,-159.789,0,0,0.814,-0.6],AUTHORITY["EPSG","6194"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4194"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-45,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2217"]]`, nil},
	uint32(2218):  {`Scoresbysund 1952 / Greenland zone 5 east`, uint32(2218), `EPSG`, uint32(2218), `PROJCS["Scoresbysund 1952 / Greenland zone 5 east",GEOGCS["Scoresbysund 1952",DATUM["Scoresbysund 1952",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[105,326,-102.5,0,0,0.814,-0.6],AUTHORITY["EPSG","6195"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4195"]],PROJECTION["Lambert Conic Conformal (West Orientated)",AUTHORITY["EPSG","9826"]],PARAMETER["Latitude of natural origin",70.5111111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-24,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["Y",NORTH],AXIS["X",WEST],AUTHORITY["EPSG","2218"]]`, nil},
	uint32(2219):  {`ATS77 / UTM zone 19N`, uint32(2219), `EPSG`, uint32(2219), `PROJCS["ATS77 / UTM zone 19N",GEOGCS["ATS77",DATUM["Average Terrestrial System 1977",SPHEROID["Average Terrestrial System 1977",6378135,298.257,AUTHORITY["EPSG","7041"]],AUTHORITY["EPSG","6122"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4122"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-69,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2219"]]`, nil},
	uint32(2220):  {`ATS77 / UTM zone 20N`, uint32(2220), `EPSG`, uint32(2220), `PROJCS["ATS77 / UTM zone 20N",GEOGCS["ATS77",DATUM["Average Terrestrial System 1977",SPHEROID["Average Terrestrial System 1977",6378135,298.257,AUTHORITY["EPSG","7041"]],AUTHORITY["EPSG","6122"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4122"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-63,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2220"]]`, nil},
	uint32(2221):  {`Scoresbysund 1952 / Greenland zone 6 east`, uint32(2221), `EPSG`, uint32(2221), `PROJCS["Scoresbysund 1952 / Greenland zone 6 east",GEOGCS["Scoresbysund 1952",DATUM["Scoresbysund 1952",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[105,326,-102.5,0,0,0.814,-0.6],AUTHORITY["EPSG","6195"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4195"]],PROJECTION["Lambert Conic Conformal (West Orientated)",AUTHORITY["EPSG","9826"]],PARAMETER["Latitude of natural origin",67.5111111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-32,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["Y",NORTH],AXIS["X",WEST],AUTHORITY["EPSG","2221"]]`, nil},
	uint32(2222):  {`NAD83 / Arizona East (ft)`, uint32(2222), `EPSG`, uint32(2222), `PROJCS["NAD83 / Arizona East (ft)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",31,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-110.177777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",700000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2222"]]`, nil},
	uint32(2223):  {`NAD83 / Arizona Central (ft)`, uint32(2223), `EPSG`, uint32(2223), `PROJCS["NAD83 / Arizona Central (ft)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",31,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-111.927777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",700000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2223"]]`, nil},
	uint32(2224):  {`NAD83 / Arizona West (ft)`, uint32(2224), `EPSG`, uint32(2224), `PROJCS["NAD83 / Arizona West (ft)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",31,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-113.75,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999933333,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",700000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2224"]]`, nil},
	uint32(2225):  {`NAD83 / California zone 1 (ftUS)`, uint32(2225), `EPSG`, uint32(2225), `PROJCS["NAD83 / California zone 1 (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",39.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-122,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",41.6777777777778,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",40,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",6561666.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",1640416.667,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2225"]]`, nil},
	uint32(2226):  {`NAD83 / California zone 2 (ftUS)`, uint32(2226), `EPSG`, uint32(2226), `PROJCS["NAD83 / California zone 2 (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",37.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-122,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",39.8333333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",38.3333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",6561666.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",1640416.667,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2226"]]`, nil},
	uint32(2227):  {`NAD83 / California zone 3 (ftUS)`, uint32(2227), `EPSG`, uint32(2227), `PROJCS["NAD83 / California zone 3 (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",36.5111111111111,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-120.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",38.4444444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",37.0777777777778,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",6561666.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",1640416.667,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2227"]]`, nil},
	uint32(2228):  {`NAD83 / California zone 4 (ftUS)`, uint32(2228), `EPSG`, uint32(2228), `PROJCS["NAD83 / California zone 4 (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",35.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-119,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",37.2611111111111,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",36,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",6561666.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",1640416.667,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2228"]]`, nil},
	uint32(2229):  {`NAD83 / California zone 5 (ftUS)`, uint32(2229), `EPSG`, uint32(2229), `PROJCS["NAD83 / California zone 5 (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",33.5111111111111,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-118,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",35.4666666666667,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",34.0333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",6561666.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",1640416.667,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2229"]]`, nil},
	uint32(2230):  {`NAD83 / California zone 6 (ftUS)`, uint32(2230), `EPSG`, uint32(2230), `PROJCS["NAD83 / California zone 6 (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",32.1666666666667,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-116.25,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",33.8833333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",32.7944444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",6561666.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",1640416.667,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2230"]]`, nil},
	uint32(2231):  {`NAD83 / Colorado North (ftUS)`, uint32(2231), `EPSG`, uint32(2231), `PROJCS["NAD83 / Colorado North (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",39.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-105.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",40.7944444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",39.7277777777778,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",3000000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",1000000,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2231"]]`, nil},
	uint32(2232):  {`NAD83 / Colorado Central (ftUS)`, uint32(2232), `EPSG`, uint32(2232), `PROJCS["NAD83 / Colorado Central (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",37.8333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-105.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",39.75,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",38.45,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",3000000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",1000000,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2232"]]`, nil},
	uint32(2233):  {`NAD83 / Colorado South (ftUS)`, uint32(2233), `EPSG`, uint32(2233), `PROJCS["NAD83 / Colorado South (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",36.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-105.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",38.4444444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",37.2333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",3000000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",1000000,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2233"]]`, nil},
	uint32(2234):  {`NAD83 / Connecticut (ftUS)`, uint32(2234), `EPSG`, uint32(2234), `PROJCS["NAD83 / Connecticut (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",40.8333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-72.75,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",41.8666666666667,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",41.2111111111111,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1000000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",500000,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2234"]]`, nil},
	uint32(2235):  {`NAD83 / Delaware (ftUS)`, uint32(2235), `EPSG`, uint32(2235), `PROJCS["NAD83 / Delaware (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",38,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-75.4166666666667,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999995,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",656166.667,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2235"]]`, nil},
	uint32(2236):  {`NAD83 / Florida East (ftUS)`, uint32(2236), `EPSG`, uint32(2236), `PROJCS["NAD83 / Florida East (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",24.3444444444444,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-81,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999941177,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",656166.667,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2236"]]`, nil},
	uint32(2237):  {`NAD83 / Florida West (ftUS)`, uint32(2237), `EPSG`, uint32(2237), `PROJCS["NAD83 / Florida West (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",24.3444444444444,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-82,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999941177,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",656166.667,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2237"]]`, nil},
	uint32(2238):  {`NAD83 / Florida North (ftUS)`, uint32(2238), `EPSG`, uint32(2238), `PROJCS["NAD83 / Florida North (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",29,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-84.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",30.7611111111111,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",29.5833333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1968500,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2238"]]`, nil},
	uint32(2239):  {`NAD83 / Georgia East (ftUS)`, uint32(2239), `EPSG`, uint32(2239), `PROJCS["NAD83 / Georgia East (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",30,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-82.1777777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",656166.667,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2239"]]`, nil},
	uint32(2240):  {`NAD83 / Georgia West (ftUS)`, uint32(2240), `EPSG`, uint32(2240), `PROJCS["NAD83 / Georgia West (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",30,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-84.1777777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",2296583.333,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2240"]]`, nil},
	uint32(2241):  {`NAD83 / Idaho East (ftUS)`, uint32(2241), `EPSG`, uint32(2241), `PROJCS["NAD83 / Idaho East (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",41.6777777777778,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-112.177777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999947368,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",656166.667,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2241"]]`, nil},
	uint32(2242):  {`NAD83 / Idaho Central (ftUS)`, uint32(2242), `EPSG`, uint32(2242), `PROJCS["NAD83 / Idaho Central (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",41.6777777777778,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-114,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999947368,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",1640416.667,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2242"]]`, nil},
	uint32(2243):  {`NAD83 / Idaho West (ftUS)`, uint32(2243), `EPSG`, uint32(2243), `PROJCS["NAD83 / Idaho West (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",41.6777777777778,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-115.75,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999933333,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",2624666.667,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2243"]]`, nil},
	uint32(2246):  {`NAD83 / Kentucky North (ftUS)`, uint32(2246), `EPSG`, uint32(2246), `PROJCS["NAD83 / Kentucky North (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",37.5111111111111,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-84.25,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",37.9777777777778,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",38.9777777777778,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1640416.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2246"]]`, nil},
	uint32(2247):  {`NAD83 / Kentucky South (ftUS)`, uint32(2247), `EPSG`, uint32(2247), `PROJCS["NAD83 / Kentucky South (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",36.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-85.75,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",37.9333333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",36.7444444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1640416.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",1640416.667,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2247"]]`, nil},
	uint32(2248):  {`NAD83 / Maryland (ftUS)`, uint32(2248), `EPSG`, uint32(2248), `PROJCS["NAD83 / Maryland (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",37.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-77,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",39.45,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",38.3111111111111,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1312333.333,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2248"]]`, nil},
	uint32(2249):  {`NAD83 / Massachusetts Mainland (ftUS)`, uint32(2249), `EPSG`, uint32(2249), `PROJCS["NAD83 / Massachusetts Mainland (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",41,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-71.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",42.6944444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",41.7277777777778,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",656166.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",2460625,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2249"]]`, nil},
	uint32(2250):  {`NAD83 / Massachusetts Island (ftUS)`, uint32(2250), `EPSG`, uint32(2250), `PROJCS["NAD83 / Massachusetts Island (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",41,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-70.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",41.4944444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",41.2833333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1640416.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2250"]]`, nil},
	uint32(2251):  {`NAD83 / Michigan North (ft)`, uint32(2251), `EPSG`, uint32(2251), `PROJCS["NAD83 / Michigan North (ft)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",44.7944444444444,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-87,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",47.0944444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",45.4944444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",26246719.16,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2251"]]`, nil},
	uint32(2252):  {`NAD83 / Michigan Central (ft)`, uint32(2252), `EPSG`, uint32(2252), `PROJCS["NAD83 / Michigan Central (ft)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",43.3277777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-84.3777777777778,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",45.7,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",44.1944444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",19685039.37,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2252"]]`, nil},
	uint32(2253):  {`NAD83 / Michigan South (ft)`, uint32(2253), `EPSG`, uint32(2253), `PROJCS["NAD83 / Michigan South (ft)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",41.5111111111111,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-84.3777777777778,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",43.6777777777778,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",42.1,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",13123359.58,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2253"]]`, nil},
	uint32(2254):  {`NAD83 / Mississippi East (ftUS)`, uint32(2254), `EPSG`, uint32(2254), `PROJCS["NAD83 / Mississippi East (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",29.5,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-88.8333333333333,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.99995,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",984250,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2254"]]`, nil},
	uint32(2255):  {`NAD83 / Mississippi West (ftUS)`, uint32(2255), `EPSG`, uint32(2255), `PROJCS["NAD83 / Mississippi West (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",29.5,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-90.3333333333333,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.99995,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",2296583.333,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2255"]]`, nil},
	uint32(2256):  {`NAD83 / Montana (ft)`, uint32(2256), `EPSG`, uint32(2256), `PROJCS["NAD83 / Montana (ft)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",44.2611111111111,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-109.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",49,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",45,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1968503.937,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2256"]]`, nil},
	uint32(2257):  {`NAD83 / New Mexico East (ftUS)`, uint32(2257), `EPSG`, uint32(2257), `PROJCS["NAD83 / New Mexico East (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",31,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-104.333333333333,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999909091,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",541337.5,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2257"]]`, nil},
	uint32(2258):  {`NAD83 / New Mexico Central (ftUS)`, uint32(2258), `EPSG`, uint32(2258), `PROJCS["NAD83 / New Mexico Central (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",31,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-106.25,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",1640416.667,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2258"]]`, nil},
	uint32(2259):  {`NAD83 / New Mexico West (ftUS)`, uint32(2259), `EPSG`, uint32(2259), `PROJCS["NAD83 / New Mexico West (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",31,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-107.833333333333,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999916667,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",2723091.667,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2259"]]`, nil},
	uint32(2260):  {`NAD83 / New York East (ftUS)`, uint32(2260), `EPSG`, uint32(2260), `PROJCS["NAD83 / New York East (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",38.8333333333333,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-74.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",492125,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2260"]]`, nil},
	uint32(2261):  {`NAD83 / New York Central (ftUS)`, uint32(2261), `EPSG`, uint32(2261), `PROJCS["NAD83 / New York Central (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",40,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-76.5944444444444,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999375,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",820208.333,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2261"]]`, nil},
	uint32(2262):  {`NAD83 / New York West (ftUS)`, uint32(2262), `EPSG`, uint32(2262), `PROJCS["NAD83 / New York West (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",40,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-78.5944444444444,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999375,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",1148291.667,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2262"]]`, nil},
	uint32(2263):  {`NAD83 / New York Long Island (ftUS)`, uint32(2263), `EPSG`, uint32(2263), `PROJCS["NAD83 / New York Long Island (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",40.1666666666667,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-74,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",41.0333333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",40.6777777777778,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",984250,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2263"]]`, nil},
	uint32(2264):  {`NAD83 / North Carolina (ftUS)`, uint32(2264), `EPSG`, uint32(2264), `PROJCS["NAD83 / North Carolina (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",33.75,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-79,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",36.1666666666667,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",34.3333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",2000000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2264"]]`, nil},
	uint32(2265):  {`NAD83 / North Dakota North (ft)`, uint32(2265), `EPSG`, uint32(2265), `PROJCS["NAD83 / North Dakota North (ft)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",47,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-100.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",48.7444444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",47.4444444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1968503.937,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2265"]]`, nil},
	uint32(2266):  {`NAD83 / North Dakota South (ft)`, uint32(2266), `EPSG`, uint32(2266), `PROJCS["NAD83 / North Dakota South (ft)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",45.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-100.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",47.4944444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",46.1944444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1968503.937,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2266"]]`, nil},
	uint32(2267):  {`NAD83 / Oklahoma North (ftUS)`, uint32(2267), `EPSG`, uint32(2267), `PROJCS["NAD83 / Oklahoma North (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",35,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-98,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",36.7666666666667,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",35.5666666666667,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1968500,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2267"]]`, nil},
	uint32(2268):  {`NAD83 / Oklahoma South (ftUS)`, uint32(2268), `EPSG`, uint32(2268), `PROJCS["NAD83 / Oklahoma South (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",33.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-98,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",35.2333333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",33.9333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1968500,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2268"]]`, nil},
	uint32(2269):  {`NAD83 / Oregon North (ft)`, uint32(2269), `EPSG`, uint32(2269), `PROJCS["NAD83 / Oregon North (ft)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",43.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-120.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",46,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",44.3333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",8202099.738,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2269"]]`, nil},
	uint32(2270):  {`NAD83 / Oregon South (ft)`, uint32(2270), `EPSG`, uint32(2270), `PROJCS["NAD83 / Oregon South (ft)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",41.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-120.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",44,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",42.3333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",4921259.843,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2270"]]`, nil},
	uint32(2271):  {`NAD83 / Pennsylvania North (ftUS)`, uint32(2271), `EPSG`, uint32(2271), `PROJCS["NAD83 / Pennsylvania North (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",40.1666666666667,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-77.75,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",41.95,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",40.8833333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1968500,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2271"]]`, nil},
	uint32(2272):  {`NAD83 / Pennsylvania South (ftUS)`, uint32(2272), `EPSG`, uint32(2272), `PROJCS["NAD83 / Pennsylvania South (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",39.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-77.75,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",40.9777777777778,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",39.9333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1968500,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2272"]]`, nil},
	uint32(2273):  {`NAD83 / South Carolina (ft)`, uint32(2273), `EPSG`, uint32(2273), `PROJCS["NAD83 / South Carolina (ft)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",31.8333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-81,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",34.8333333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",32.5111111111111,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",2000000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2273"]]`, nil},
	uint32(2274):  {`NAD83 / Tennessee (ftUS)`, uint32(2274), `EPSG`, uint32(2274), `PROJCS["NAD83 / Tennessee (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",34.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-86,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",36.4166666666667,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",35.2611111111111,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1968500,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2274"]]`, nil},
	uint32(2275):  {`NAD83 / Texas North (ftUS)`, uint32(2275), `EPSG`, uint32(2275), `PROJCS["NAD83 / Texas North (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",34,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-101.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",36.1944444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",34.65,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",656166.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",3280833.333,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2275"]]`, nil},
	uint32(2276):  {`NAD83 / Texas North Central (ftUS)`, uint32(2276), `EPSG`, uint32(2276), `PROJCS["NAD83 / Texas North Central (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",31.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-98.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",33.9777777777778,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",32.1444444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1968500,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",6561666.667,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2276"]]`, nil},
	uint32(2277):  {`NAD83 / Texas Central (ftUS)`, uint32(2277), `EPSG`, uint32(2277), `PROJCS["NAD83 / Texas Central (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",29.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-100.333333333333,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",31.8833333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",30.1166666666667,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",2296583.333,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",9842500,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2277"]]`, nil},
	uint32(2278):  {`NAD83 / Texas South Central (ftUS)`, uint32(2278), `EPSG`, uint32(2278), `PROJCS["NAD83 / Texas South Central (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",27.8333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-99,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",30.2833333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",28.3833333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1968500,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",13123333.333,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2278"]]`, nil},
	uint32(2279):  {`NAD83 / Texas South (ftUS)`, uint32(2279), `EPSG`, uint32(2279), `PROJCS["NAD83 / Texas South (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",25.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-98.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",27.8333333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",26.1666666666667,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",984250,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",16404166.667,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2279"]]`, nil},
	uint32(2280):  {`NAD83 / Utah North (ft)`, uint32(2280), `EPSG`, uint32(2280), `PROJCS["NAD83 / Utah North (ft)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",40.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-111.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",41.7944444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",40.7277777777778,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1640419.948,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",3280839.895,AUTHORITY["EPSG","8827"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2280"]]`, nil},
	uint32(2281):  {`NAD83 / Utah Central (ft)`, uint32(2281), `EPSG`, uint32(2281), `PROJCS["NAD83 / Utah Central (ft)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",38.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-111.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",40.65,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",39.0277777777778,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1640419.948,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",6561679.79,AUTHORITY["EPSG","8827"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2281"]]`, nil},
	uint32(2282):  {`NAD83 / Utah South (ft)`, uint32(2282), `EPSG`, uint32(2282), `PROJCS["NAD83 / Utah South (ft)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",36.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-111.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",38.35,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",37.2166666666667,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1640419.948,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",9842519.685,AUTHORITY["EPSG","8827"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2282"]]`, nil},
	uint32(2283):  {`NAD83 / Virginia North (ftUS)`, uint32(2283), `EPSG`, uint32(2283), `PROJCS["NAD83 / Virginia North (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",37.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-78.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",39.2111111111111,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",38.0333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",11482916.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",6561666.667,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2283"]]`, nil},
	uint32(2284):  {`NAD83 / Virginia South (ftUS)`, uint32(2284), `EPSG`, uint32(2284), `PROJCS["NAD83 / Virginia South (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",36.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-78.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",37.9777777777778,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",36.7666666666667,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",11482916.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",3280833.333,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2284"]]`, nil},
	uint32(2285):  {`NAD83 / Washington North (ftUS)`, uint32(2285), `EPSG`, uint32(2285), `PROJCS["NAD83 / Washington North (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",47,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-120.833333333333,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",48.7444444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",47.5111111111111,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1640416.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2285"]]`, nil},
	uint32(2286):  {`NAD83 / Washington South (ftUS)`, uint32(2286), `EPSG`, uint32(2286), `PROJCS["NAD83 / Washington South (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",45.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-120.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",47.3333333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",45.8333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1640416.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2286"]]`, nil},
	uint32(2287):  {`NAD83 / Wisconsin North (ftUS)`, uint32(2287), `EPSG`, uint32(2287), `PROJCS["NAD83 / Wisconsin North (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",45.1666666666667,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-90,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",46.7666666666667,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",45.5666666666667,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1968500,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2287"]]`, nil},
	uint32(2288):  {`NAD83 / Wisconsin Central (ftUS)`, uint32(2288), `EPSG`, uint32(2288), `PROJCS["NAD83 / Wisconsin Central (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",43.8333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-90,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",45.5111111111111,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",44.2611111111111,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1968500,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2288"]]`, nil},
	uint32(2289):  {`NAD83 / Wisconsin South (ftUS)`, uint32(2289), `EPSG`, uint32(2289), `PROJCS["NAD83 / Wisconsin South (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",42,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-90,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",44.0777777777778,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",42.7444444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1968500,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2289"]]`, nil},
	uint32(2290):  {`ATS77 / Prince Edward Isl. Stereographic (ATS77)`, uint32(2290), `EPSG`, uint32(2290), `PROJCS["ATS77 / Prince Edward Isl. Stereographic (ATS77)",GEOGCS["ATS77",DATUM["Average Terrestrial System 1977",SPHEROID["Average Terrestrial System 1977",6378135,298.257,AUTHORITY["EPSG","7041"]],AUTHORITY["EPSG","6122"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4122"]],PROJECTION["Oblique Stereographic",AUTHORITY["EPSG","9809"]],PARAMETER["Latitude of natural origin",47.2611111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-63,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999912,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",700000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",400000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E(X)",EAST],AXIS["N(Y)",NORTH],AUTHORITY["EPSG","2290"]]`, nil},
	uint32(2294):  {`ATS77 / MTM Nova Scotia zone 4`, uint32(2294), `EPSG`, uint32(2294), `PROJCS["ATS77 / MTM Nova Scotia zone 4",GEOGCS["ATS77",DATUM["Average Terrestrial System 1977",SPHEROID["Average Terrestrial System 1977",6378135,298.257,AUTHORITY["EPSG","7041"]],AUTHORITY["EPSG","6122"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4122"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-61.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",4500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2294"]]`, nil},
	uint32(2295):  {`ATS77 / MTM Nova Scotia zone 5`, uint32(2295), `EPSG`, uint32(2295), `PROJCS["ATS77 / MTM Nova Scotia zone 5",GEOGCS["ATS77",DATUM["Average Terrestrial System 1977",SPHEROID["Average Terrestrial System 1977",6378135,298.257,AUTHORITY["EPSG","7041"]],AUTHORITY["EPSG","6122"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4122"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-64.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",5500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2295"]]`, nil},
	uint32(2296):  {`Ammassalik 1958 / Greenland zone 7 east`, uint32(2296), `EPSG`, uint32(2296), `PROJCS["Ammassalik 1958 / Greenland zone 7 east",GEOGCS["Ammassalik 1958",DATUM["Ammassalik 1958",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-45,417,-3.5,0,0,0.814,-0.6],AUTHORITY["EPSG","6196"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4196"]],PROJECTION["Lambert Conic Conformal (West Orientated)",AUTHORITY["EPSG","9826"]],PARAMETER["Latitude of natural origin",64.5111111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-40,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["Y",NORTH],AXIS["X",WEST],AUTHORITY["EPSG","2296"]]`, nil},
	uint32(2299):  {`Qornoq 1927 / Greenland zone 2 west`, uint32(2299), `EPSG`, uint32(2299), `PROJCS["Qornoq 1927 / Greenland zone 2 west",GEOGCS["Qornoq 1927",DATUM["Qornoq 1927",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[163.511,127.533,-159.789,0,0,0.814,-0.6],AUTHORITY["EPSG","6194"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4194"]],PROJECTION["Lambert Conic Conformal (West Orientated)",AUTHORITY["EPSG","9826"]],PARAMETER["Latitude of natural origin",79.5111111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-64,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["Y",NORTH],AXIS["X",WEST],AUTHORITY["EPSG","2299"]]`, nil},
	uint32(2301):  {`Qornoq 1927 / Greenland zone 3 west`, uint32(2301), `EPSG`, uint32(2301), `PROJCS["Qornoq 1927 / Greenland zone 3 west",GEOGCS["Qornoq 1927",DATUM["Qornoq 1927",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[163.511,127.533,-159.789,0,0,0.814,-0.6],AUTHORITY["EPSG","6194"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4194"]],PROJECTION["Lambert Conic Conformal (West Orientated)",AUTHORITY["EPSG","9826"]],PARAMETER["Latitude of natural origin",76.5111111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-64,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["Y",NORTH],AXIS["X",WEST],AUTHORITY["EPSG","2301"]]`, nil},
	uint32(2303):  {`Qornoq 1927 / Greenland zone 4 west`, uint32(2303), `EPSG`, uint32(2303), `PROJCS["Qornoq 1927 / Greenland zone 4 west",GEOGCS["Qornoq 1927",DATUM["Qornoq 1927",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[163.511,127.533,-159.789,0,0,0.814,-0.6],AUTHORITY["EPSG","6194"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4194"]],PROJECTION["Lambert Conic Conformal (West Orientated)",AUTHORITY["EPSG","9826"]],PARAMETER["Latitude of natural origin",73.5111111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-52,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["Y",NORTH],AXIS["X",WEST],AUTHORITY["EPSG","2303"]]`, nil},
	uint32(2304):  {`Qornoq 1927 / Greenland zone 5 west`, uint32(2304), `EPSG`, uint32(2304), `PROJCS["Qornoq 1927 / Greenland zone 5 west",GEOGCS["Qornoq 1927",DATUM["Qornoq 1927",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[163.511,127.533,-159.789,0,0,0.814,-0.6],AUTHORITY["EPSG","6194"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4194"]],PROJECTION["Lambert Conic Conformal (West Orientated)",AUTHORITY["EPSG","9826"]],PARAMETER["Latitude of natural origin",70.5111111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-52,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["Y",NORTH],AXIS["X",WEST],AUTHORITY["EPSG","2304"]]`, nil},
	uint32(2305):  {`Qornoq 1927 / Greenland zone 6 west`, uint32(2305), `EPSG`, uint32(2305), `PROJCS["Qornoq 1927 / Greenland zone 6 west",GEOGCS["Qornoq 1927",DATUM["Qornoq 1927",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[163.511,127.533,-159.789,0,0,0.814,-0.6],AUTHORITY["EPSG","6194"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4194"]],PROJECTION["Lambert Conic Conformal (West Orientated)",AUTHORITY["EPSG","9826"]],PARAMETER["Latitude of natural origin",67.5111111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-52,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["Y",NORTH],AXIS["X",WEST],AUTHORITY["EPSG","2305"]]`, nil},
	uint32(2306):  {`Qornoq 1927 / Greenland zone 7 west`, uint32(2306), `EPSG`, uint32(2306), `PROJCS["Qornoq 1927 / Greenland zone 7 west",GEOGCS["Qornoq 1927",DATUM["Qornoq 1927",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[163.511,127.533,-159.789,0,0,0.814,-0.6],AUTHORITY["EPSG","6194"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4194"]],PROJECTION["Lambert Conic Conformal (West Orientated)",AUTHORITY["EPSG","9826"]],PARAMETER["Latitude of natural origin",64.5111111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-52,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["Y",NORTH],AXIS["X",WEST],AUTHORITY["EPSG","2306"]]`, nil},
	uint32(2307):  {`Qornoq 1927 / Greenland zone 8 east`, uint32(2307), `EPSG`, uint32(2307), `PROJCS["Qornoq 1927 / Greenland zone 8 east",GEOGCS["Qornoq 1927",DATUM["Qornoq 1927",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[163.511,127.533,-159.789,0,0,0.814,-0.6],AUTHORITY["EPSG","6194"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4194"]],PROJECTION["Lambert Conic Conformal (West Orientated)",AUTHORITY["EPSG","9826"]],PARAMETER["Latitude of natural origin",61.5111111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-48,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["Y",NORTH],AXIS["X",WEST],AUTHORITY["EPSG","2307"]]`, nil},
	uint32(2308):  {`Batavia / TM 109 SE`, uint32(2308), `EPSG`, uint32(2308), `PROJCS["Batavia / TM 109 SE",GEOGCS["Batavia",DATUM["Batavia",SPHEROID["Bessel 1841",6377397.155,299.1528128,AUTHORITY["EPSG","7004"]],TOWGS84[-377.7,675.1,-52.2,0,0,0,0],AUTHORITY["EPSG","6211"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4211"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",109,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2308"]]`, nil},
	uint32(2309):  {`WGS 84 / TM 116 SE`, uint32(2309), `EPSG`, uint32(2309), `PROJCS["WGS 84 / TM 116 SE",GEOGCS["WGS 84",DATUM["World Geodetic System 1984",SPHEROID["WGS 84",6378137,298.257223563,AUTHORITY["EPSG","7030"]],AUTHORITY["EPSG","6326"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4326"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",116,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2309"]]`, nil},
	uint32(2310):  {`WGS 84 / TM 132 SE`, uint32(2310), `EPSG`, uint32(2310), `PROJCS["WGS 84 / TM 132 SE",GEOGCS["WGS 84",DATUM["World Geodetic System 1984",SPHEROID["WGS 84",6378137,298.257223563,AUTHORITY["EPSG","7030"]],AUTHORITY["EPSG","6326"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4326"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",132,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2310"]]`, nil},
	uint32(2311):  {`WGS 84 / TM 6 NE`, uint32(2311), `EPSG`, uint32(2311), `PROJCS["WGS 84 / TM 6 NE",GEOGCS["WGS 84",DATUM["World Geodetic System 1984",SPHEROID["WGS 84",6378137,298.257223563,AUTHORITY["EPSG","7030"]],AUTHORITY["EPSG","6326"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4326"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",6,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2311"]]`, nil},
	uint32(2312):  {`Garoua / UTM zone 33N`, uint32(2312), `EPSG`, uint32(2312), `PROJCS["Garoua / UTM zone 33N",GEOGCS["Garoua",DATUM["Garoua",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465,AUTHORITY["EPSG","7012"]],AUTHORITY["EPSG","6197"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4197"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",15,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2312"]]`, nil},
	uint32(2313):  {`Kousseri / UTM zone 33N`, uint32(2313), `EPSG`, uint32(2313), `PROJCS["Kousseri / UTM zone 33N",GEOGCS["Kousseri",DATUM["Kousseri",SPHEROID["Clarke 1880 (RGS)",6378249.145,293.465,AUTHORITY["EPSG","7012"]],AUTHORITY["EPSG","6198"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4198"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",15,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2313"]]`, nil},
	uint32(2314):  {`Trinidad 1903 / Trinidad Grid (ftCla)`, uint32(2314), `EPSG`, uint32(2314), `PROJCS["Trinidad 1903 / Trinidad Grid (ftCla)",GEOGCS["Trinidad 1903",DATUM["Trinidad 1903",SPHEROID["Clarke 1858",6378293.645208759,294.26067636926103,AUTHORITY["EPSG","7007"]],TOWGS84[-61.702,284.488,472.052,0,0,0,0],AUTHORITY["EPSG","6302"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4302"]],PROJECTION["Cassini-Soldner",AUTHORITY["EPSG","9806"]],PARAMETER["Latitude of natural origin",10.4416666666667,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-61.3333333333333,AUTHORITY["EPSG","8802"]],PARAMETER["False easting",283800,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",214500,AUTHORITY["EPSG","8807"]],UNIT["Clarke's foot",0.3047972654,AUTHORITY["EPSG","9005"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2314"]]`, nil},
	uint32(2315):  {`Campo Inchauspe / UTM zone 19S`, uint32(2315), `EPSG`, uint32(2315), `PROJCS["Campo Inchauspe / UTM zone 19S",GEOGCS["Campo Inchauspe",DATUM["Campo Inchauspe",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-148,136,90,0,0,0,0],AUTHORITY["EPSG","6221"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4221"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-69,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2315"]]`, nil},
	uint32(2316):  {`Campo Inchauspe / UTM zone 20S`, uint32(2316), `EPSG`, uint32(2316), `PROJCS["Campo Inchauspe / UTM zone 20S",GEOGCS["Campo Inchauspe",DATUM["Campo Inchauspe",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-148,136,90,0,0,0,0],AUTHORITY["EPSG","6221"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4221"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-63,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2316"]]`, nil},
	uint32(2317):  {`PSAD56 / ICN Regional`, uint32(2317), `EPSG`, uint32(2317), `PROJCS["PSAD56 / ICN Regional",GEOGCS["PSAD56",DATUM["Provisional South American Datum 1956",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-60.31,245.935,31.008,12.324,3.755,-7.37,0.447],AUTHORITY["EPSG","6248"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4248"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",6,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-66,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",9,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",3,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1000000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",1000000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2317"]]`, nil},
	uint32(2318):  {`Ain el Abd / Aramco Lambert`, uint32(2318), `EPSG`, uint32(2318), `PROJCS["Ain el Abd / Aramco Lambert",GEOGCS["Ain el Abd",DATUM["Ain el Abd 1970",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-18.944,-379.364,-24.063,0.04,-0.764,6.431,3.657],AUTHORITY["EPSG","6204"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4204"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",25.08951,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",48,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",17,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",33,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",0,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2318"]]`, nil},
	uint32(2319):  {`ED50 / TM27`, uint32(2319), `EPSG`, uint32(2319), `PROJCS["ED50 / TM27",GEOGCS["ED50",DATUM["European Datum 1950",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-157.89,-17.16,-78.41,2.118,2.697,-1.434,-5.38],AUTHORITY["EPSG","6230"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4230"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",27,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2319"]]`, nil},
	uint32(2320):  {`ED50 / TM30`, uint32(2320), `EPSG`, uint32(2320), `PROJCS["ED50 / TM30",GEOGCS["ED50",DATUM["European Datum 1950",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-157.89,-17.16,-78.41,2.118,2.697,-1.434,-5.38],AUTHORITY["EPSG","6230"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4230"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",30,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2320"]]`, nil},
	uint32(2321):  {`ED50 / TM33`, uint32(2321), `EPSG`, uint32(2321), `PROJCS["ED50 / TM33",GEOGCS["ED50",DATUM["European Datum 1950",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-157.89,-17.16,-78.41,2.118,2.697,-1.434,-5.38],AUTHORITY["EPSG","6230"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4230"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",33,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2321"]]`, nil},
	uint32(2322):  {`ED50 / TM36`, uint32(2322), `EPSG`, uint32(2322), `PROJCS["ED50 / TM36",GEOGCS["ED50",DATUM["European Datum 1950",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-157.89,-17.16,-78.41,2.118,2.697,-1.434,-5.38],AUTHORITY["EPSG","6230"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4230"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",36,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2322"]]`, nil},
	uint32(2323):  {`ED50 / TM39`, uint32(2323), `EPSG`, uint32(2323), `PROJCS["ED50 / TM39",GEOGCS["ED50",DATUM["European Datum 1950",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-157.89,-17.16,-78.41,2.118,2.697,-1.434,-5.38],AUTHORITY["EPSG","6230"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4230"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",39,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2323"]]`, nil},
	uint32(2324):  {`ED50 / TM42`, uint32(2324), `EPSG`, uint32(2324), `PROJCS["ED50 / TM42",GEOGCS["ED50",DATUM["European Datum 1950",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-157.89,-17.16,-78.41,2.118,2.697,-1.434,-5.38],AUTHORITY["EPSG","6230"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4230"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",42,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2324"]]`, nil},
	uint32(2325):  {`ED50 / TM45`, uint32(2325), `EPSG`, uint32(2325), `PROJCS["ED50 / TM45",GEOGCS["ED50",DATUM["European Datum 1950",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-157.89,-17.16,-78.41,2.118,2.697,-1.434,-5.38],AUTHORITY["EPSG","6230"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4230"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",45,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2325"]]`, nil},
	uint32(2326):  {`Hong Kong 1980 Grid System`, uint32(2326), `EPSG`, uint32(2326), `PROJCS["Hong Kong 1980 Grid System",GEOGCS["Hong Kong 1980",DATUM["Hong Kong 1980",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-162.619,-276.959,-161.764,0.067753,-2.243649,-1.158827,-1.094246],AUTHORITY["EPSG","6611"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4611"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",22.3121333333333,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",114.178555555556,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",836694.05,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",819069.8,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2326"]]`, nil},
	uint32(2327):  {`Xian 1980 / Gauss-Kruger zone 13`, uint32(2327), `EPSG`, uint32(2327), `PROJCS["Xian 1980 / Gauss-Kruger zone 13",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",75,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",13500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2327"]]`, nil},
	uint32(2328):  {`Xian 1980 / Gauss-Kruger zone 14`, uint32(2328), `EPSG`, uint32(2328), `PROJCS["Xian 1980 / Gauss-Kruger zone 14",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",81,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",14500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2328"]]`, nil},
	uint32(2329):  {`Xian 1980 / Gauss-Kruger zone 15`, uint32(2329), `EPSG`, uint32(2329), `PROJCS["Xian 1980 / Gauss-Kruger zone 15",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",87,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",15500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2329"]]`, nil},
	uint32(2330):  {`Xian 1980 / Gauss-Kruger zone 16`, uint32(2330), `EPSG`, uint32(2330), `PROJCS["Xian 1980 / Gauss-Kruger zone 16",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",93,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",16500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2330"]]`, nil},
	uint32(2331):  {`Xian 1980 / Gauss-Kruger zone 17`, uint32(2331), `EPSG`, uint32(2331), `PROJCS["Xian 1980 / Gauss-Kruger zone 17",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",99,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",17500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2331"]]`, nil},
	uint32(2332):  {`Xian 1980 / Gauss-Kruger zone 18`, uint32(2332), `EPSG`, uint32(2332), `PROJCS["Xian 1980 / Gauss-Kruger zone 18",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",105,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",18500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2332"]]`, nil},
	uint32(2333):  {`Xian 1980 / Gauss-Kruger zone 19`, uint32(2333), `EPSG`, uint32(2333), `PROJCS["Xian 1980 / Gauss-Kruger zone 19",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",19500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2333"]]`, nil},
	uint32(2334):  {`Xian 1980 / Gauss-Kruger zone 20`, uint32(2334), `EPSG`, uint32(2334), `PROJCS["Xian 1980 / Gauss-Kruger zone 20",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",117,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",20500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2334"]]`, nil},
	uint32(2335):  {`Xian 1980 / Gauss-Kruger zone 21`, uint32(2335), `EPSG`, uint32(2335), `PROJCS["Xian 1980 / Gauss-Kruger zone 21",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",123,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",21500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2335"]]`, nil},
	uint32(2336):  {`Xian 1980 / Gauss-Kruger zone 22`, uint32(2336), `EPSG`, uint32(2336), `PROJCS["Xian 1980 / Gauss-Kruger zone 22",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",129,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",22500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2336"]]`, nil},
	uint32(2337):  {`Xian 1980 / Gauss-Kruger zone 23`, uint32(2337), `EPSG`, uint32(2337), `PROJCS["Xian 1980 / Gauss-Kruger zone 23",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",135,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",23500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2337"]]`, nil},
	uint32(2338):  {`Xian 1980 / Gauss-Kruger CM 75E`, uint32(2338), `EPSG`, uint32(2338), `PROJCS["Xian 1980 / Gauss-Kruger CM 75E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",75,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2338"]]`, nil},
	uint32(2339):  {`Xian 1980 / Gauss-Kruger CM 81E`, uint32(2339), `EPSG`, uint32(2339), `PROJCS["Xian 1980 / Gauss-Kruger CM 81E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",81,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2339"]]`, nil},
	uint32(2340):  {`Xian 1980 / Gauss-Kruger CM 87E`, uint32(2340), `EPSG`, uint32(2340), `PROJCS["Xian 1980 / Gauss-Kruger CM 87E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",87,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2340"]]`, nil},
	uint32(2341):  {`Xian 1980 / Gauss-Kruger CM 93E`, uint32(2341), `EPSG`, uint32(2341), `PROJCS["Xian 1980 / Gauss-Kruger CM 93E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",93,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2341"]]`, nil},
	uint32(2342):  {`Xian 1980 / Gauss-Kruger CM 99E`, uint32(2342), `EPSG`, uint32(2342), `PROJCS["Xian 1980 / Gauss-Kruger CM 99E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",99,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2342"]]`, nil},
	uint32(2343):  {`Xian 1980 / Gauss-Kruger CM 105E`, uint32(2343), `EPSG`, uint32(2343), `PROJCS["Xian 1980 / Gauss-Kruger CM 105E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",105,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2343"]]`, nil},
	uint32(2344):  {`Xian 1980 / Gauss-Kruger CM 111E`, uint32(2344), `EPSG`, uint32(2344), `PROJCS["Xian 1980 / Gauss-Kruger CM 111E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2344"]]`, nil},
	uint32(2345):  {`Xian 1980 / Gauss-Kruger CM 117E`, uint32(2345), `EPSG`, uint32(2345), `PROJCS["Xian 1980 / Gauss-Kruger CM 117E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",117,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2345"]]`, nil},
	uint32(2346):  {`Xian 1980 / Gauss-Kruger CM 123E`, uint32(2346), `EPSG`, uint32(2346), `PROJCS["Xian 1980 / Gauss-Kruger CM 123E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",123,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2346"]]`, nil},
	uint32(2347):  {`Xian 1980 / Gauss-Kruger CM 129E`, uint32(2347), `EPSG`, uint32(2347), `PROJCS["Xian 1980 / Gauss-Kruger CM 129E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",129,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2347"]]`, nil},
	uint32(2348):  {`Xian 1980 / Gauss-Kruger CM 135E`, uint32(2348), `EPSG`, uint32(2348), `PROJCS["Xian 1980 / Gauss-Kruger CM 135E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",135,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2348"]]`, nil},
	uint32(2349):  {`Xian 1980 / 3-degree Gauss-Kruger zone 25`, uint32(2349), `EPSG`, uint32(2349), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger zone 25",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",75,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",25500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2349"]]`, nil},
	uint32(2350):  {`Xian 1980 / 3-degree Gauss-Kruger zone 26`, uint32(2350), `EPSG`, uint32(2350), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger zone 26",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",78,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",26500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2350"]]`, nil},
	uint32(2351):  {`Xian 1980 / 3-degree Gauss-Kruger zone 27`, uint32(2351), `EPSG`, uint32(2351), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger zone 27",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",81,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",27500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2351"]]`, nil},
	uint32(2352):  {`Xian 1980 / 3-degree Gauss-Kruger zone 28`, uint32(2352), `EPSG`, uint32(2352), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger zone 28",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",84,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",28500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2352"]]`, nil},
	uint32(2353):  {`Xian 1980 / 3-degree Gauss-Kruger zone 29`, uint32(2353), `EPSG`, uint32(2353), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger zone 29",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",87,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",29500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2353"]]`, nil},
	uint32(2354):  {`Xian 1980 / 3-degree Gauss-Kruger zone 30`, uint32(2354), `EPSG`, uint32(2354), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger zone 30",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",90,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",30500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2354"]]`, nil},
	uint32(2355):  {`Xian 1980 / 3-degree Gauss-Kruger zone 31`, uint32(2355), `EPSG`, uint32(2355), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger zone 31",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",93,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",31500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2355"]]`, nil},
	uint32(2356):  {`Xian 1980 / 3-degree Gauss-Kruger zone 32`, uint32(2356), `EPSG`, uint32(2356), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger zone 32",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",96,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",32500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2356"]]`, nil},
	uint32(2357):  {`Xian 1980 / 3-degree Gauss-Kruger zone 33`, uint32(2357), `EPSG`, uint32(2357), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger zone 33",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",99,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",33500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2357"]]`, nil},
	uint32(2358):  {`Xian 1980 / 3-degree Gauss-Kruger zone 34`, uint32(2358), `EPSG`, uint32(2358), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger zone 34",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",102,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",34500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2358"]]`, nil},
	uint32(2359):  {`Xian 1980 / 3-degree Gauss-Kruger zone 35`, uint32(2359), `EPSG`, uint32(2359), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger zone 35",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",105,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",35500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2359"]]`, nil},
	uint32(2360):  {`Xian 1980 / 3-degree Gauss-Kruger zone 36`, uint32(2360), `EPSG`, uint32(2360), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger zone 36",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",108,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",36500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2360"]]`, nil},
	uint32(2361):  {`Xian 1980 / 3-degree Gauss-Kruger zone 37`, uint32(2361), `EPSG`, uint32(2361), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger zone 37",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",37500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2361"]]`, nil},
	uint32(2362):  {`Xian 1980 / 3-degree Gauss-Kruger zone 38`, uint32(2362), `EPSG`, uint32(2362), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger zone 38",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",114,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",38500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2362"]]`, nil},
	uint32(2363):  {`Xian 1980 / 3-degree Gauss-Kruger zone 39`, uint32(2363), `EPSG`, uint32(2363), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger zone 39",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",117,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",39500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2363"]]`, nil},
	uint32(2364):  {`Xian 1980 / 3-degree Gauss-Kruger zone 40`, uint32(2364), `EPSG`, uint32(2364), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger zone 40",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",120,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",40500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2364"]]`, nil},
	uint32(2365):  {`Xian 1980 / 3-degree Gauss-Kruger zone 41`, uint32(2365), `EPSG`, uint32(2365), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger zone 41",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",123,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",41500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2365"]]`, nil},
	uint32(2366):  {`Xian 1980 / 3-degree Gauss-Kruger zone 42`, uint32(2366), `EPSG`, uint32(2366), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger zone 42",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",126,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",42500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2366"]]`, nil},
	uint32(2367):  {`Xian 1980 / 3-degree Gauss-Kruger zone 43`, uint32(2367), `EPSG`, uint32(2367), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger zone 43",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",129,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",43500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2367"]]`, nil},
	uint32(2368):  {`Xian 1980 / 3-degree Gauss-Kruger zone 44`, uint32(2368), `EPSG`, uint32(2368), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger zone 44",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",132,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",44500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2368"]]`, nil},
	uint32(2369):  {`Xian 1980 / 3-degree Gauss-Kruger zone 45`, uint32(2369), `EPSG`, uint32(2369), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger zone 45",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",135,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",45500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2369"]]`, nil},
	uint32(2370):  {`Xian 1980 / 3-degree Gauss-Kruger CM 75E`, uint32(2370), `EPSG`, uint32(2370), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger CM 75E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",75,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2370"]]`, nil},
	uint32(2371):  {`Xian 1980 / 3-degree Gauss-Kruger CM 78E`, uint32(2371), `EPSG`, uint32(2371), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger CM 78E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",78,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2371"]]`, nil},
	uint32(2372):  {`Xian 1980 / 3-degree Gauss-Kruger CM 81E`, uint32(2372), `EPSG`, uint32(2372), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger CM 81E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",81,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2372"]]`, nil},
	uint32(2373):  {`Xian 1980 / 3-degree Gauss-Kruger CM 84E`, uint32(2373), `EPSG`, uint32(2373), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger CM 84E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",84,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2373"]]`, nil},
	uint32(2374):  {`Xian 1980 / 3-degree Gauss-Kruger CM 87E`, uint32(2374), `EPSG`, uint32(2374), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger CM 87E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",87,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2374"]]`, nil},
	uint32(2375):  {`Xian 1980 / 3-degree Gauss-Kruger CM 90E`, uint32(2375), `EPSG`, uint32(2375), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger CM 90E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",90,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2375"]]`, nil},
	uint32(2376):  {`Xian 1980 / 3-degree Gauss-Kruger CM 93E`, uint32(2376), `EPSG`, uint32(2376), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger CM 93E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",93,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2376"]]`, nil},
	uint32(2377):  {`Xian 1980 / 3-degree Gauss-Kruger CM 96E`, uint32(2377), `EPSG`, uint32(2377), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger CM 96E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",96,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2377"]]`, nil},
	uint32(2378):  {`Xian 1980 / 3-degree Gauss-Kruger CM 99E`, uint32(2378), `EPSG`, uint32(2378), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger CM 99E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",99,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2378"]]`, nil},
	uint32(2379):  {`Xian 1980 / 3-degree Gauss-Kruger CM 102E`, uint32(2379), `EPSG`, uint32(2379), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger CM 102E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",102,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2379"]]`, nil},
	uint32(2380):  {`Xian 1980 / 3-degree Gauss-Kruger CM 105E`, uint32(2380), `EPSG`, uint32(2380), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger CM 105E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",105,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2380"]]`, nil},
	uint32(2381):  {`Xian 1980 / 3-degree Gauss-Kruger CM 108E`, uint32(2381), `EPSG`, uint32(2381), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger CM 108E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",108,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2381"]]`, nil},
	uint32(2382):  {`Xian 1980 / 3-degree Gauss-Kruger CM 111E`, uint32(2382), `EPSG`, uint32(2382), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger CM 111E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2382"]]`, nil},
	uint32(2383):  {`Xian 1980 / 3-degree Gauss-Kruger CM 114E`, uint32(2383), `EPSG`, uint32(2383), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger CM 114E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",114,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2383"]]`, nil},
	uint32(2384):  {`Xian 1980 / 3-degree Gauss-Kruger CM 117E`, uint32(2384), `EPSG`, uint32(2384), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger CM 117E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",117,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2384"]]`, nil},
	uint32(2385):  {`Xian 1980 / 3-degree Gauss-Kruger CM 120E`, uint32(2385), `EPSG`, uint32(2385), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger CM 120E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",120,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2385"]]`, nil},
	uint32(2386):  {`Xian 1980 / 3-degree Gauss-Kruger CM 123E`, uint32(2386), `EPSG`, uint32(2386), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger CM 123E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",123,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2386"]]`, nil},
	uint32(2387):  {`Xian 1980 / 3-degree Gauss-Kruger CM 126E`, uint32(2387), `EPSG`, uint32(2387), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger CM 126E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",126,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2387"]]`, nil},
	uint32(2388):  {`Xian 1980 / 3-degree Gauss-Kruger CM 129E`, uint32(2388), `EPSG`, uint32(2388), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger CM 129E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",129,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2388"]]`, nil},
	uint32(2389):  {`Xian 1980 / 3-degree Gauss-Kruger CM 132E`, uint32(2389), `EPSG`, uint32(2389), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger CM 132E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",132,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2389"]]`, nil},
	uint32(2390):  {`Xian 1980 / 3-degree Gauss-Kruger CM 135E`, uint32(2390), `EPSG`, uint32(2390), `PROJCS["Xian 1980 / 3-degree Gauss-Kruger CM 135E",GEOGCS["Xian 1980",DATUM["Xian 1980",SPHEROID["IAG 1975",6378140,298.257,AUTHORITY["EPSG","7049"]],AUTHORITY["EPSG","6610"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4610"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",135,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2390"]]`, nil},
	uint32(2391):  {`KKJ / Finland zone 1`, uint32(2391), `EPSG`, uint32(2391), `PROJCS["KKJ / Finland zone 1",GEOGCS["KKJ",DATUM["Kartastokoordinaattijarjestelma (1966)",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-96.062,-82.428,-121.753,4.801,0.345,-1.376,1.496],AUTHORITY["EPSG","6123"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4123"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",21,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",1500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2391"]]`, nil},
	uint32(2392):  {`KKJ / Finland zone 2`, uint32(2392), `EPSG`, uint32(2392), `PROJCS["KKJ / Finland zone 2",GEOGCS["KKJ",DATUM["Kartastokoordinaattijarjestelma (1966)",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-96.062,-82.428,-121.753,4.801,0.345,-1.376,1.496],AUTHORITY["EPSG","6123"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4123"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",24,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",2500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2392"]]`, nil},
	uint32(2393):  {`KKJ / Finland Uniform Coordinate System`, uint32(2393), `EPSG`, uint32(2393), `PROJCS["KKJ / Finland Uniform Coordinate System",GEOGCS["KKJ",DATUM["Kartastokoordinaattijarjestelma (1966)",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-96.062,-82.428,-121.753,4.801,0.345,-1.376,1.496],AUTHORITY["EPSG","6123"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4123"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",27,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",3500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2393"]]`, nil},
	uint32(2394):  {`KKJ / Finland zone 4`, uint32(2394), `EPSG`, uint32(2394), `PROJCS["KKJ / Finland zone 4",GEOGCS["KKJ",DATUM["Kartastokoordinaattijarjestelma (1966)",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-96.062,-82.428,-121.753,4.801,0.345,-1.376,1.496],AUTHORITY["EPSG","6123"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4123"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",30,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",4500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2394"]]`, nil},
	uint32(2395):  {`South Yemen / Gauss-Kruger zone 8`, uint32(2395), `EPSG`, uint32(2395), `PROJCS["South Yemen / Gauss-Kruger zone 8",GEOGCS["South Yemen",DATUM["South Yemen",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[-76,-138,67,0,0,0,0],AUTHORITY["EPSG","6164"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4164"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",45,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",8500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2395"]]`, nil},
	uint32(2396):  {`South Yemen / Gauss-Kruger zone 9`, uint32(2396), `EPSG`, uint32(2396), `PROJCS["South Yemen / Gauss-Kruger zone 9",GEOGCS["South Yemen",DATUM["South Yemen",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[-76,-138,67,0,0,0,0],AUTHORITY["EPSG","6164"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4164"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",51,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",9500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2396"]]`, nil},
	uint32(2397):  {`Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 3`, uint32(2397), `EPSG`, uint32(2397), `PROJCS["Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 3",GEOGCS["Pulkovo 1942(83)",DATUM["Pulkovo 1942(83)",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24,-123,-94,0.02,-0.25,-0.13,1.1],AUTHORITY["EPSG","6178"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4178"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",9,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",3500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2397"]]`, nil},
	uint32(2398):  {`Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 4`, uint32(2398), `EPSG`, uint32(2398), `PROJCS["Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 4",GEOGCS["Pulkovo 1942(83)",DATUM["Pulkovo 1942(83)",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24,-123,-94,0.02,-0.25,-0.13,1.1],AUTHORITY["EPSG","6178"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4178"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",12,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",4500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2398"]]`, nil},
	uint32(2399):  {`Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 5`, uint32(2399), `EPSG`, uint32(2399), `PROJCS["Pulkovo 1942(83) / 3-degree Gauss-Kruger zone 5",GEOGCS["Pulkovo 1942(83)",DATUM["Pulkovo 1942(83)",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24,-123,-94,0.02,-0.25,-0.13,1.1],AUTHORITY["EPSG","6178"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4178"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",15,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",5500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2399"]]`, nil},
	uint32(2401):  {`Beijing 1954 / 3-degree Gauss-Kruger zone 25`, uint32(2401), `EPSG`, uint32(2401), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger zone 25",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",75,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",25500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2401"]]`, nil},
	uint32(2402):  {`Beijing 1954 / 3-degree Gauss-Kruger zone 26`, uint32(2402), `EPSG`, uint32(2402), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger zone 26",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",78,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",26500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2402"]]`, nil},
	uint32(2403):  {`Beijing 1954 / 3-degree Gauss-Kruger zone 27`, uint32(2403), `EPSG`, uint32(2403), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger zone 27",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",81,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",27500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2403"]]`, nil},
	uint32(2404):  {`Beijing 1954 / 3-degree Gauss-Kruger zone 28`, uint32(2404), `EPSG`, uint32(2404), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger zone 28",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",84,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",28500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2404"]]`, nil},
	uint32(2405):  {`Beijing 1954 / 3-degree Gauss-Kruger zone 29`, uint32(2405), `EPSG`, uint32(2405), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger zone 29",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",87,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",29500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2405"]]`, nil},
	uint32(2406):  {`Beijing 1954 / 3-degree Gauss-Kruger zone 30`, uint32(2406), `EPSG`, uint32(2406), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger zone 30",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",90,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",30500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2406"]]`, nil},
	uint32(2407):  {`Beijing 1954 / 3-degree Gauss-Kruger zone 31`, uint32(2407), `EPSG`, uint32(2407), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger zone 31",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",93,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",31500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2407"]]`, nil},
	uint32(2408):  {`Beijing 1954 / 3-degree Gauss-Kruger zone 32`, uint32(2408), `EPSG`, uint32(2408), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger zone 32",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",96,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",32500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2408"]]`, nil},
	uint32(2409):  {`Beijing 1954 / 3-degree Gauss-Kruger zone 33`, uint32(2409), `EPSG`, uint32(2409), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger zone 33",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",99,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",33500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2409"]]`, nil},
	uint32(2410):  {`Beijing 1954 / 3-degree Gauss-Kruger zone 34`, uint32(2410), `EPSG`, uint32(2410), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger zone 34",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",102,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",34500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2410"]]`, nil},
	uint32(2411):  {`Beijing 1954 / 3-degree Gauss-Kruger zone 35`, uint32(2411), `EPSG`, uint32(2411), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger zone 35",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",105,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",35500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2411"]]`, nil},
	uint32(2412):  {`Beijing 1954 / 3-degree Gauss-Kruger zone 36`, uint32(2412), `EPSG`, uint32(2412), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger zone 36",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",108,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",36500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2412"]]`, nil},
	uint32(2413):  {`Beijing 1954 / 3-degree Gauss-Kruger zone 37`, uint32(2413), `EPSG`, uint32(2413), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger zone 37",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",37500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2413"]]`, nil},
	uint32(2414):  {`Beijing 1954 / 3-degree Gauss-Kruger zone 38`, uint32(2414), `EPSG`, uint32(2414), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger zone 38",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",114,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",38500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2414"]]`, nil},
	uint32(2415):  {`Beijing 1954 / 3-degree Gauss-Kruger zone 39`, uint32(2415), `EPSG`, uint32(2415), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger zone 39",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",117,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",39500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2415"]]`, nil},
	uint32(2416):  {`Beijing 1954 / 3-degree Gauss-Kruger zone 40`, uint32(2416), `EPSG`, uint32(2416), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger zone 40",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",120,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",40500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2416"]]`, nil},
	uint32(2417):  {`Beijing 1954 / 3-degree Gauss-Kruger zone 41`, uint32(2417), `EPSG`, uint32(2417), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger zone 41",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",123,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",41500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2417"]]`, nil},
	uint32(2418):  {`Beijing 1954 / 3-degree Gauss-Kruger zone 42`, uint32(2418), `EPSG`, uint32(2418), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger zone 42",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",126,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",42500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2418"]]`, nil},
	uint32(2419):  {`Beijing 1954 / 3-degree Gauss-Kruger zone 43`, uint32(2419), `EPSG`, uint32(2419), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger zone 43",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",129,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",43500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2419"]]`, nil},
	uint32(2420):  {`Beijing 1954 / 3-degree Gauss-Kruger zone 44`, uint32(2420), `EPSG`, uint32(2420), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger zone 44",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",132,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",44500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2420"]]`, nil},
	uint32(2421):  {`Beijing 1954 / 3-degree Gauss-Kruger zone 45`, uint32(2421), `EPSG`, uint32(2421), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger zone 45",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",135,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",45500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2421"]]`, nil},
	uint32(2422):  {`Beijing 1954 / 3-degree Gauss-Kruger CM 75E`, uint32(2422), `EPSG`, uint32(2422), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger CM 75E",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",75,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2422"]]`, nil},
	uint32(2423):  {`Beijing 1954 / 3-degree Gauss-Kruger CM 78E`, uint32(2423), `EPSG`, uint32(2423), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger CM 78E",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",78,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2423"]]`, nil},
	uint32(2424):  {`Beijing 1954 / 3-degree Gauss-Kruger CM 81E`, uint32(2424), `EPSG`, uint32(2424), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger CM 81E",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",81,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2424"]]`, nil},
	uint32(2425):  {`Beijing 1954 / 3-degree Gauss-Kruger CM 84E`, uint32(2425), `EPSG`, uint32(2425), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger CM 84E",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",84,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2425"]]`, nil},
	uint32(2426):  {`Beijing 1954 / 3-degree Gauss-Kruger CM 87E`, uint32(2426), `EPSG`, uint32(2426), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger CM 87E",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",87,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2426"]]`, nil},
	uint32(2427):  {`Beijing 1954 / 3-degree Gauss-Kruger CM 90E`, uint32(2427), `EPSG`, uint32(2427), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger CM 90E",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",90,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2427"]]`, nil},
	uint32(2428):  {`Beijing 1954 / 3-degree Gauss-Kruger CM 93E`, uint32(2428), `EPSG`, uint32(2428), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger CM 93E",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",93,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2428"]]`, nil},
	uint32(2429):  {`Beijing 1954 / 3-degree Gauss-Kruger CM 96E`, uint32(2429), `EPSG`, uint32(2429), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger CM 96E",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",96,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2429"]]`, nil},
	uint32(2430):  {`Beijing 1954 / 3-degree Gauss-Kruger CM 99E`, uint32(2430), `EPSG`, uint32(2430), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger CM 99E",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",99,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2430"]]`, nil},
	uint32(2431):  {`Beijing 1954 / 3-degree Gauss-Kruger CM 102E`, uint32(2431), `EPSG`, uint32(2431), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger CM 102E",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",102,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2431"]]`, nil},
	uint32(2432):  {`Beijing 1954 / 3-degree Gauss-Kruger CM 105E`, uint32(2432), `EPSG`, uint32(2432), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger CM 105E",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",105,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2432"]]`, nil},
	uint32(2433):  {`Beijing 1954 / 3-degree Gauss-Kruger CM 108E`, uint32(2433), `EPSG`, uint32(2433), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger CM 108E",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",108,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2433"]]`, nil},
	uint32(2434):  {`Beijing 1954 / 3-degree Gauss-Kruger CM 111E`, uint32(2434), `EPSG`, uint32(2434), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger CM 111E",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2434"]]`, nil},
	uint32(2435):  {`Beijing 1954 / 3-degree Gauss-Kruger CM 114E`, uint32(2435), `EPSG`, uint32(2435), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger CM 114E",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",114,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2435"]]`, nil},
	uint32(2436):  {`Beijing 1954 / 3-degree Gauss-Kruger CM 117E`, uint32(2436), `EPSG`, uint32(2436), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger CM 117E",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",117,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2436"]]`, nil},
	uint32(2437):  {`Beijing 1954 / 3-degree Gauss-Kruger CM 120E`, uint32(2437), `EPSG`, uint32(2437), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger CM 120E",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",120,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2437"]]`, nil},
	uint32(2438):  {`Beijing 1954 / 3-degree Gauss-Kruger CM 123E`, uint32(2438), `EPSG`, uint32(2438), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger CM 123E",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",123,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2438"]]`, nil},
	uint32(2439):  {`Beijing 1954 / 3-degree Gauss-Kruger CM 126E`, uint32(2439), `EPSG`, uint32(2439), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger CM 126E",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",126,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2439"]]`, nil},
	uint32(2440):  {`Beijing 1954 / 3-degree Gauss-Kruger CM 129E`, uint32(2440), `EPSG`, uint32(2440), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger CM 129E",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",129,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2440"]]`, nil},
	uint32(2441):  {`Beijing 1954 / 3-degree Gauss-Kruger CM 132E`, uint32(2441), `EPSG`, uint32(2441), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger CM 132E",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",132,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2441"]]`, nil},
	uint32(2442):  {`Beijing 1954 / 3-degree Gauss-Kruger CM 135E`, uint32(2442), `EPSG`, uint32(2442), `PROJCS["Beijing 1954 / 3-degree Gauss-Kruger CM 135E",GEOGCS["Beijing 1954",DATUM["Beijing 1954",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[18,-136.8,-73.7,0,0,0.814,-0.38],AUTHORITY["EPSG","6214"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4214"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",135,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2442"]]`, nil},
	uint32(2443):  {`JGD2000 / Japan Plane Rectangular CS I`, uint32(2443), `EPSG`, uint32(2443), `PROJCS["JGD2000 / Japan Plane Rectangular CS I",GEOGCS["JGD2000",DATUM["Japanese Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6612"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4612"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",33,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",129.5,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2443"]]`, nil},
	uint32(2444):  {`JGD2000 / Japan Plane Rectangular CS II`, uint32(2444), `EPSG`, uint32(2444), `PROJCS["JGD2000 / Japan Plane Rectangular CS II",GEOGCS["JGD2000",DATUM["Japanese Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6612"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4612"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",33,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",131,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2444"]]`, nil},
	uint32(2445):  {`JGD2000 / Japan Plane Rectangular CS III`, uint32(2445), `EPSG`, uint32(2445), `PROJCS["JGD2000 / Japan Plane Rectangular CS III",GEOGCS["JGD2000",DATUM["Japanese Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6612"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4612"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",36,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",132.177777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2445"]]`, nil},
	uint32(2446):  {`JGD2000 / Japan Plane Rectangular CS IV`, uint32(2446), `EPSG`, uint32(2446), `PROJCS["JGD2000 / Japan Plane Rectangular CS IV",GEOGCS["JGD2000",DATUM["Japanese Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6612"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4612"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",33,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",133.5,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2446"]]`, nil},
	uint32(2447):  {`JGD2000 / Japan Plane Rectangular CS V`, uint32(2447), `EPSG`, uint32(2447), `PROJCS["JGD2000 / Japan Plane Rectangular CS V",GEOGCS["JGD2000",DATUM["Japanese Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6612"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4612"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",36,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",134.344444444444,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2447"]]`, nil},
	uint32(2448):  {`JGD2000 / Japan Plane Rectangular CS VI`, uint32(2448), `EPSG`, uint32(2448), `PROJCS["JGD2000 / Japan Plane Rectangular CS VI",GEOGCS["JGD2000",DATUM["Japanese Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6612"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4612"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",36,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",136,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2448"]]`, nil},
	uint32(2449):  {`JGD2000 / Japan Plane Rectangular CS VII`, uint32(2449), `EPSG`, uint32(2449), `PROJCS["JGD2000 / Japan Plane Rectangular CS VII",GEOGCS["JGD2000",DATUM["Japanese Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6612"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4612"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",36,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",137.177777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2449"]]`, nil},
	uint32(2450):  {`JGD2000 / Japan Plane Rectangular CS VIII`, uint32(2450), `EPSG`, uint32(2450), `PROJCS["JGD2000 / Japan Plane Rectangular CS VIII",GEOGCS["JGD2000",DATUM["Japanese Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6612"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4612"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",36,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",138.5,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2450"]]`, nil},
	uint32(2451):  {`JGD2000 / Japan Plane Rectangular CS IX`, uint32(2451), `EPSG`, uint32(2451), `PROJCS["JGD2000 / Japan Plane Rectangular CS IX",GEOGCS["JGD2000",DATUM["Japanese Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6612"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4612"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",36,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",139.833333333333,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2451"]]`, nil},
	uint32(2452):  {`JGD2000 / Japan Plane Rectangular CS X`, uint32(2452), `EPSG`, uint32(2452), `PROJCS["JGD2000 / Japan Plane Rectangular CS X",GEOGCS["JGD2000",DATUM["Japanese Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6612"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4612"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",40,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",140.833333333333,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2452"]]`, nil},
	uint32(2453):  {`JGD2000 / Japan Plane Rectangular CS XI`, uint32(2453), `EPSG`, uint32(2453), `PROJCS["JGD2000 / Japan Plane Rectangular CS XI",GEOGCS["JGD2000",DATUM["Japanese Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6612"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4612"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",44,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",140.25,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2453"]]`, nil},
	uint32(2454):  {`JGD2000 / Japan Plane Rectangular CS XII`, uint32(2454), `EPSG`, uint32(2454), `PROJCS["JGD2000 / Japan Plane Rectangular CS XII",GEOGCS["JGD2000",DATUM["Japanese Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6612"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4612"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",44,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",142.25,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2454"]]`, nil},
	uint32(2455):  {`JGD2000 / Japan Plane Rectangular CS XIII`, uint32(2455), `EPSG`, uint32(2455), `PROJCS["JGD2000 / Japan Plane Rectangular CS XIII",GEOGCS["JGD2000",DATUM["Japanese Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6612"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4612"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",44,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",144.25,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2455"]]`, nil},
	uint32(2456):  {`JGD2000 / Japan Plane Rectangular CS XIV`, uint32(2456), `EPSG`, uint32(2456), `PROJCS["JGD2000 / Japan Plane Rectangular CS XIV",GEOGCS["JGD2000",DATUM["Japanese Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6612"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4612"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",26,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",142,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2456"]]`, nil},
	uint32(2457):  {`JGD2000 / Japan Plane Rectangular CS XV`, uint32(2457), `EPSG`, uint32(2457), `PROJCS["JGD2000 / Japan Plane Rectangular CS XV",GEOGCS["JGD2000",DATUM["Japanese Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6612"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4612"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",26,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",127.511111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2457"]]`, nil},
	uint32(2458):  {`JGD2000 / Japan Plane Rectangular CS XVI`, uint32(2458), `EPSG`, uint32(2458), `PROJCS["JGD2000 / Japan Plane Rectangular CS XVI",GEOGCS["JGD2000",DATUM["Japanese Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6612"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4612"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",26,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",124,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2458"]]`, nil},
	uint32(2459):  {`JGD2000 / Japan Plane Rectangular CS XVII`, uint32(2459), `EPSG`, uint32(2459), `PROJCS["JGD2000 / Japan Plane Rectangular CS XVII",GEOGCS["JGD2000",DATUM["Japanese Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6612"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4612"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",26,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",131,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2459"]]`, nil},
	uint32(2460):  {`JGD2000 / Japan Plane Rectangular CS XVIII`, uint32(2460), `EPSG`, uint32(2460), `PROJCS["JGD2000 / Japan Plane Rectangular CS XVIII",GEOGCS["JGD2000",DATUM["Japanese Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6612"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4612"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",20,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",136,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2460"]]`, nil},
	uint32(2461):  {`JGD2000 / Japan Plane Rectangular CS XIX`, uint32(2461), `EPSG`, uint32(2461), `PROJCS["JGD2000 / Japan Plane Rectangular CS XIX",GEOGCS["JGD2000",DATUM["Japanese Geodetic Datum 2000",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6612"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4612"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",26,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",154,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2461"]]`, nil},
	uint32(2462):  {`Albanian 1987 / Gauss-Kruger zone 4`, uint32(2462), `EPSG`, uint32(2462), `PROJCS["Albanian 1987 / Gauss-Kruger zone 4",GEOGCS["Albanian 1987",DATUM["Albanian 1987",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[-44.183,-0.58,-38.489,2.3867,2.7072,-3.5196,-8.2703],AUTHORITY["EPSG","6191"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4191"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",21,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",4500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2462"]]`, nil},
	uint32(2463):  {`Pulkovo 1995 / Gauss-Kruger CM 21E`, uint32(2463), `EPSG`, uint32(2463), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 21E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",21,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2463"]]`, nil},
	uint32(2464):  {`Pulkovo 1995 / Gauss-Kruger CM 27E`, uint32(2464), `EPSG`, uint32(2464), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 27E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",27,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2464"]]`, nil},
	uint32(2465):  {`Pulkovo 1995 / Gauss-Kruger CM 33E`, uint32(2465), `EPSG`, uint32(2465), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 33E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",33,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2465"]]`, nil},
	uint32(2466):  {`Pulkovo 1995 / Gauss-Kruger CM 39E`, uint32(2466), `EPSG`, uint32(2466), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 39E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",39,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2466"]]`, nil},
	uint32(2467):  {`Pulkovo 1995 / Gauss-Kruger CM 45E`, uint32(2467), `EPSG`, uint32(2467), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 45E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",45,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2467"]]`, nil},
	uint32(2468):  {`Pulkovo 1995 / Gauss-Kruger CM 51E`, uint32(2468), `EPSG`, uint32(2468), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 51E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",51,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2468"]]`, nil},
	uint32(2469):  {`Pulkovo 1995 / Gauss-Kruger CM 57E`, uint32(2469), `EPSG`, uint32(2469), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 57E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",57,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2469"]]`, nil},
	uint32(2470):  {`Pulkovo 1995 / Gauss-Kruger CM 63E`, uint32(2470), `EPSG`, uint32(2470), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 63E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",63,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2470"]]`, nil},
	uint32(2471):  {`Pulkovo 1995 / Gauss-Kruger CM 69E`, uint32(2471), `EPSG`, uint32(2471), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 69E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",69,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2471"]]`, nil},
	uint32(2472):  {`Pulkovo 1995 / Gauss-Kruger CM 75E`, uint32(2472), `EPSG`, uint32(2472), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 75E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",75,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2472"]]`, nil},
	uint32(2473):  {`Pulkovo 1995 / Gauss-Kruger CM 81E`, uint32(2473), `EPSG`, uint32(2473), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 81E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",81,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2473"]]`, nil},
	uint32(2474):  {`Pulkovo 1995 / Gauss-Kruger CM 87E`, uint32(2474), `EPSG`, uint32(2474), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 87E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",87,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2474"]]`, nil},
	uint32(2475):  {`Pulkovo 1995 / Gauss-Kruger CM 93E`, uint32(2475), `EPSG`, uint32(2475), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 93E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",93,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2475"]]`, nil},
	uint32(2476):  {`Pulkovo 1995 / Gauss-Kruger CM 99E`, uint32(2476), `EPSG`, uint32(2476), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 99E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",99,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2476"]]`, nil},
	uint32(2477):  {`Pulkovo 1995 / Gauss-Kruger CM 105E`, uint32(2477), `EPSG`, uint32(2477), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 105E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",105,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2477"]]`, nil},
	uint32(2478):  {`Pulkovo 1995 / Gauss-Kruger CM 111E`, uint32(2478), `EPSG`, uint32(2478), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 111E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2478"]]`, nil},
	uint32(2479):  {`Pulkovo 1995 / Gauss-Kruger CM 117E`, uint32(2479), `EPSG`, uint32(2479), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 117E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",117,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2479"]]`, nil},
	uint32(2480):  {`Pulkovo 1995 / Gauss-Kruger CM 123E`, uint32(2480), `EPSG`, uint32(2480), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 123E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",123,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2480"]]`, nil},
	uint32(2481):  {`Pulkovo 1995 / Gauss-Kruger CM 129E`, uint32(2481), `EPSG`, uint32(2481), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 129E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",129,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2481"]]`, nil},
	uint32(2482):  {`Pulkovo 1995 / Gauss-Kruger CM 135E`, uint32(2482), `EPSG`, uint32(2482), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 135E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",135,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2482"]]`, nil},
	uint32(2483):  {`Pulkovo 1995 / Gauss-Kruger CM 141E`, uint32(2483), `EPSG`, uint32(2483), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 141E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",141,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2483"]]`, nil},
	uint32(2484):  {`Pulkovo 1995 / Gauss-Kruger CM 147E`, uint32(2484), `EPSG`, uint32(2484), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 147E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",147,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2484"]]`, nil},
	uint32(2485):  {`Pulkovo 1995 / Gauss-Kruger CM 153E`, uint32(2485), `EPSG`, uint32(2485), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 153E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",153,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2485"]]`, nil},
	uint32(2486):  {`Pulkovo 1995 / Gauss-Kruger CM 159E`, uint32(2486), `EPSG`, uint32(2486), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 159E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",159,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2486"]]`, nil},
	uint32(2487):  {`Pulkovo 1995 / Gauss-Kruger CM 165E`, uint32(2487), `EPSG`, uint32(2487), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 165E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",165,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2487"]]`, nil},
	uint32(2488):  {`Pulkovo 1995 / Gauss-Kruger CM 171E`, uint32(2488), `EPSG`, uint32(2488), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 171E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",171,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2488"]]`, nil},
	uint32(2489):  {`Pulkovo 1995 / Gauss-Kruger CM 177E`, uint32(2489), `EPSG`, uint32(2489), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 177E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",177,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2489"]]`, nil},
	uint32(2490):  {`Pulkovo 1995 / Gauss-Kruger CM 177W`, uint32(2490), `EPSG`, uint32(2490), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 177W",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-177,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2490"]]`, nil},
	uint32(2491):  {`Pulkovo 1995 / Gauss-Kruger CM 171W`, uint32(2491), `EPSG`, uint32(2491), `PROJCS["Pulkovo 1995 / Gauss-Kruger CM 171W",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-171,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2491"]]`, nil},
	uint32(2494):  {`Pulkovo 1942 / Gauss-Kruger CM 21E`, uint32(2494), `EPSG`, uint32(2494), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 21E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",21,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2494"]]`, nil},
	uint32(2495):  {`Pulkovo 1942 / Gauss-Kruger CM 27E`, uint32(2495), `EPSG`, uint32(2495), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 27E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",27,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2495"]]`, nil},
	uint32(2496):  {`Pulkovo 1942 / Gauss-Kruger CM 33E`, uint32(2496), `EPSG`, uint32(2496), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 33E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",33,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2496"]]`, nil},
	uint32(2497):  {`Pulkovo 1942 / Gauss-Kruger CM 39E`, uint32(2497), `EPSG`, uint32(2497), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 39E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",39,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2497"]]`, nil},
	uint32(2498):  {`Pulkovo 1942 / Gauss-Kruger CM 45E`, uint32(2498), `EPSG`, uint32(2498), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 45E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",45,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2498"]]`, nil},
	uint32(2499):  {`Pulkovo 1942 / Gauss-Kruger CM 51E`, uint32(2499), `EPSG`, uint32(2499), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 51E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",51,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2499"]]`, nil},
	uint32(2500):  {`Pulkovo 1942 / Gauss-Kruger CM 57E`, uint32(2500), `EPSG`, uint32(2500), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 57E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",57,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2500"]]`, nil},
	uint32(2501):  {`Pulkovo 1942 / Gauss-Kruger CM 63E`, uint32(2501), `EPSG`, uint32(2501), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 63E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",63,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2501"]]`, nil},
	uint32(2502):  {`Pulkovo 1942 / Gauss-Kruger CM 69E`, uint32(2502), `EPSG`, uint32(2502), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 69E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",69,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2502"]]`, nil},
	uint32(2503):  {`Pulkovo 1942 / Gauss-Kruger CM 75E`, uint32(2503), `EPSG`, uint32(2503), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 75E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",75,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2503"]]`, nil},
	uint32(2504):  {`Pulkovo 1942 / Gauss-Kruger CM 81E`, uint32(2504), `EPSG`, uint32(2504), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 81E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",81,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2504"]]`, nil},
	uint32(2505):  {`Pulkovo 1942 / Gauss-Kruger CM 87E`, uint32(2505), `EPSG`, uint32(2505), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 87E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",87,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2505"]]`, nil},
	uint32(2506):  {`Pulkovo 1942 / Gauss-Kruger CM 93E`, uint32(2506), `EPSG`, uint32(2506), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 93E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",93,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2506"]]`, nil},
	uint32(2507):  {`Pulkovo 1942 / Gauss-Kruger CM 99E`, uint32(2507), `EPSG`, uint32(2507), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 99E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",99,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2507"]]`, nil},
	uint32(2508):  {`Pulkovo 1942 / Gauss-Kruger CM 105E`, uint32(2508), `EPSG`, uint32(2508), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 105E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",105,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2508"]]`, nil},
	uint32(2509):  {`Pulkovo 1942 / Gauss-Kruger CM 111E`, uint32(2509), `EPSG`, uint32(2509), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 111E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2509"]]`, nil},
	uint32(2510):  {`Pulkovo 1942 / Gauss-Kruger CM 117E`, uint32(2510), `EPSG`, uint32(2510), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 117E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",117,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2510"]]`, nil},
	uint32(2511):  {`Pulkovo 1942 / Gauss-Kruger CM 123E`, uint32(2511), `EPSG`, uint32(2511), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 123E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",123,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2511"]]`, nil},
	uint32(2512):  {`Pulkovo 1942 / Gauss-Kruger CM 129E`, uint32(2512), `EPSG`, uint32(2512), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 129E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",129,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2512"]]`, nil},
	uint32(2513):  {`Pulkovo 1942 / Gauss-Kruger CM 135E`, uint32(2513), `EPSG`, uint32(2513), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 135E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",135,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2513"]]`, nil},
	uint32(2514):  {`Pulkovo 1942 / Gauss-Kruger CM 141E`, uint32(2514), `EPSG`, uint32(2514), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 141E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",141,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2514"]]`, nil},
	uint32(2515):  {`Pulkovo 1942 / Gauss-Kruger CM 147E`, uint32(2515), `EPSG`, uint32(2515), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 147E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",147,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2515"]]`, nil},
	uint32(2516):  {`Pulkovo 1942 / Gauss-Kruger CM 153E`, uint32(2516), `EPSG`, uint32(2516), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 153E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",153,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2516"]]`, nil},
	uint32(2517):  {`Pulkovo 1942 / Gauss-Kruger CM 159E`, uint32(2517), `EPSG`, uint32(2517), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 159E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",159,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2517"]]`, nil},
	uint32(2518):  {`Pulkovo 1942 / Gauss-Kruger CM 165E`, uint32(2518), `EPSG`, uint32(2518), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 165E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",165,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2518"]]`, nil},
	uint32(2519):  {`Pulkovo 1942 / Gauss-Kruger CM 171E`, uint32(2519), `EPSG`, uint32(2519), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 171E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",171,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2519"]]`, nil},
	uint32(2520):  {`Pulkovo 1942 / Gauss-Kruger CM 177E`, uint32(2520), `EPSG`, uint32(2520), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 177E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",177,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2520"]]`, nil},
	uint32(2521):  {`Pulkovo 1942 / Gauss-Kruger CM 177W`, uint32(2521), `EPSG`, uint32(2521), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 177W",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-177,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2521"]]`, nil},
	uint32(2522):  {`Pulkovo 1942 / Gauss-Kruger CM 171W`, uint32(2522), `EPSG`, uint32(2522), `PROJCS["Pulkovo 1942 / Gauss-Kruger CM 171W",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-171,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2522"]]`, nil},
	uint32(2523):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 7`, uint32(2523), `EPSG`, uint32(2523), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 7",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",21,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",7500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2523"]]`, nil},
	uint32(2524):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 8`, uint32(2524), `EPSG`, uint32(2524), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 8",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",24,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",8500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2524"]]`, nil},
	uint32(2525):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 9`, uint32(2525), `EPSG`, uint32(2525), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 9",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",27,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",9500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2525"]]`, nil},
	uint32(2526):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 10`, uint32(2526), `EPSG`, uint32(2526), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 10",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",30,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",10500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2526"]]`, nil},
	uint32(2527):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 11`, uint32(2527), `EPSG`, uint32(2527), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 11",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",33,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",11500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2527"]]`, nil},
	uint32(2528):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 12`, uint32(2528), `EPSG`, uint32(2528), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 12",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",36,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",12500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2528"]]`, nil},
	uint32(2529):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 13`, uint32(2529), `EPSG`, uint32(2529), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 13",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",39,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",13500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2529"]]`, nil},
	uint32(2530):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 14`, uint32(2530), `EPSG`, uint32(2530), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 14",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",42,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",14500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2530"]]`, nil},
	uint32(2531):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 15`, uint32(2531), `EPSG`, uint32(2531), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 15",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",45,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",15500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2531"]]`, nil},
	uint32(2532):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 16`, uint32(2532), `EPSG`, uint32(2532), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 16",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",48,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",16500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2532"]]`, nil},
	uint32(2533):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 17`, uint32(2533), `EPSG`, uint32(2533), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 17",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",51,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",17500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2533"]]`, nil},
	uint32(2534):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 18`, uint32(2534), `EPSG`, uint32(2534), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 18",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",54,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",18500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2534"]]`, nil},
	uint32(2535):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 19`, uint32(2535), `EPSG`, uint32(2535), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 19",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",57,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",19500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2535"]]`, nil},
	uint32(2536):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 20`, uint32(2536), `EPSG`, uint32(2536), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 20",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",60,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",20500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2536"]]`, nil},
	uint32(2537):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 21`, uint32(2537), `EPSG`, uint32(2537), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 21",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",63,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",21500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2537"]]`, nil},
	uint32(2538):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 22`, uint32(2538), `EPSG`, uint32(2538), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 22",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",66,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",22500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2538"]]`, nil},
	uint32(2539):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 23`, uint32(2539), `EPSG`, uint32(2539), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 23",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",69,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",23500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2539"]]`, nil},
	uint32(2540):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 24`, uint32(2540), `EPSG`, uint32(2540), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 24",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",72,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",24500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2540"]]`, nil},
	uint32(2541):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 25`, uint32(2541), `EPSG`, uint32(2541), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 25",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",75,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",25500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2541"]]`, nil},
	uint32(2542):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 26`, uint32(2542), `EPSG`, uint32(2542), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 26",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",78,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",26500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2542"]]`, nil},
	uint32(2543):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 27`, uint32(2543), `EPSG`, uint32(2543), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 27",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",81,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",27500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2543"]]`, nil},
	uint32(2544):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 28`, uint32(2544), `EPSG`, uint32(2544), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 28",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",84,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",28500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2544"]]`, nil},
	uint32(2545):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 29`, uint32(2545), `EPSG`, uint32(2545), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 29",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",87,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",29500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2545"]]`, nil},
	uint32(2546):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 30`, uint32(2546), `EPSG`, uint32(2546), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 30",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",90,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",30500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2546"]]`, nil},
	uint32(2547):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 31`, uint32(2547), `EPSG`, uint32(2547), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 31",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",93,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",31500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2547"]]`, nil},
	uint32(2548):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 32`, uint32(2548), `EPSG`, uint32(2548), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 32",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",96,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",32500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2548"]]`, nil},
	uint32(2549):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 33`, uint32(2549), `EPSG`, uint32(2549), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 33",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",99,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",33500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2549"]]`, nil},
	uint32(2551):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 34`, uint32(2551), `EPSG`, uint32(2551), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 34",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",102,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",34500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2551"]]`, nil},
	uint32(2552):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 35`, uint32(2552), `EPSG`, uint32(2552), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 35",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",105,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",35500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2552"]]`, nil},
	uint32(2553):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 36`, uint32(2553), `EPSG`, uint32(2553), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 36",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",108,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",36500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2553"]]`, nil},
	uint32(2554):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 37`, uint32(2554), `EPSG`, uint32(2554), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 37",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",37500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2554"]]`, nil},
	uint32(2555):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 38`, uint32(2555), `EPSG`, uint32(2555), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 38",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",114,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",38500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2555"]]`, nil},
	uint32(2556):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 39`, uint32(2556), `EPSG`, uint32(2556), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 39",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",117,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",39500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2556"]]`, nil},
	uint32(2557):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 40`, uint32(2557), `EPSG`, uint32(2557), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 40",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",120,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",40500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2557"]]`, nil},
	uint32(2558):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 41`, uint32(2558), `EPSG`, uint32(2558), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 41",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",123,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",41500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2558"]]`, nil},
	uint32(2559):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 42`, uint32(2559), `EPSG`, uint32(2559), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 42",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",126,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",42500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2559"]]`, nil},
	uint32(2560):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 43`, uint32(2560), `EPSG`, uint32(2560), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 43",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",129,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",43500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2560"]]`, nil},
	uint32(2561):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 44`, uint32(2561), `EPSG`, uint32(2561), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 44",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",132,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",44500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2561"]]`, nil},
	uint32(2562):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 45`, uint32(2562), `EPSG`, uint32(2562), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 45",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",135,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",45500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2562"]]`, nil},
	uint32(2563):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 46`, uint32(2563), `EPSG`, uint32(2563), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 46",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",138,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",46500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2563"]]`, nil},
	uint32(2564):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 47`, uint32(2564), `EPSG`, uint32(2564), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 47",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",141,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",47500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2564"]]`, nil},
	uint32(2565):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 48`, uint32(2565), `EPSG`, uint32(2565), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 48",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",144,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",48500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2565"]]`, nil},
	uint32(2566):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 49`, uint32(2566), `EPSG`, uint32(2566), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 49",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",147,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",49500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2566"]]`, nil},
	uint32(2567):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 50`, uint32(2567), `EPSG`, uint32(2567), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 50",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",150,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",50500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2567"]]`, nil},
	uint32(2568):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 51`, uint32(2568), `EPSG`, uint32(2568), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 51",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",153,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",51500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2568"]]`, nil},
	uint32(2569):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 52`, uint32(2569), `EPSG`, uint32(2569), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 52",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",156,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",52500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2569"]]`, nil},
	uint32(2570):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 53`, uint32(2570), `EPSG`, uint32(2570), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 53",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",159,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",53500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2570"]]`, nil},
	uint32(2571):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 54`, uint32(2571), `EPSG`, uint32(2571), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 54",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",162,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",54500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2571"]]`, nil},
	uint32(2572):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 55`, uint32(2572), `EPSG`, uint32(2572), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 55",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",165,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",55500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2572"]]`, nil},
	uint32(2573):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 56`, uint32(2573), `EPSG`, uint32(2573), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 56",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",168,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",56500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2573"]]`, nil},
	uint32(2574):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 57`, uint32(2574), `EPSG`, uint32(2574), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 57",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",171,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",57500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2574"]]`, nil},
	uint32(2575):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 58`, uint32(2575), `EPSG`, uint32(2575), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 58",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",174,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",58500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2575"]]`, nil},
	uint32(2576):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 59`, uint32(2576), `EPSG`, uint32(2576), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 59",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",177,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",59500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2576"]]`, nil},
	uint32(2578):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 61`, uint32(2578), `EPSG`, uint32(2578), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 61",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-177,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",61500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2578"]]`, nil},
	uint32(2579):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 62`, uint32(2579), `EPSG`, uint32(2579), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 62",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-174,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",62500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2579"]]`, nil},
	uint32(2580):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 63`, uint32(2580), `EPSG`, uint32(2580), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 63",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-171,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",63500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2580"]]`, nil},
	uint32(2581):  {`Pulkovo 1942 / 3-degree Gauss-Kruger zone 64`, uint32(2581), `EPSG`, uint32(2581), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger zone 64",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-168,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",64500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2581"]]`, nil},
	uint32(2582):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 21E`, uint32(2582), `EPSG`, uint32(2582), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 21E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",21,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2582"]]`, nil},
	uint32(2583):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 24E`, uint32(2583), `EPSG`, uint32(2583), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 24E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",24,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2583"]]`, nil},
	uint32(2584):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 27E`, uint32(2584), `EPSG`, uint32(2584), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 27E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",27,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2584"]]`, nil},
	uint32(2585):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 30E`, uint32(2585), `EPSG`, uint32(2585), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 30E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",30,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2585"]]`, nil},
	uint32(2586):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 33E`, uint32(2586), `EPSG`, uint32(2586), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 33E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",33,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2586"]]`, nil},
	uint32(2587):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 36E`, uint32(2587), `EPSG`, uint32(2587), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 36E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",36,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2587"]]`, nil},
	uint32(2588):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 39E`, uint32(2588), `EPSG`, uint32(2588), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 39E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",39,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2588"]]`, nil},
	uint32(2589):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 42E`, uint32(2589), `EPSG`, uint32(2589), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 42E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",42,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2589"]]`, nil},
	uint32(2590):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 45E`, uint32(2590), `EPSG`, uint32(2590), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 45E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",45,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2590"]]`, nil},
	uint32(2591):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 48E`, uint32(2591), `EPSG`, uint32(2591), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 48E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",48,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2591"]]`, nil},
	uint32(2592):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 51E`, uint32(2592), `EPSG`, uint32(2592), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 51E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",51,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2592"]]`, nil},
	uint32(2593):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 54E`, uint32(2593), `EPSG`, uint32(2593), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 54E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",54,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2593"]]`, nil},
	uint32(2594):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 57E`, uint32(2594), `EPSG`, uint32(2594), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 57E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",57,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2594"]]`, nil},
	uint32(2595):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 60E`, uint32(2595), `EPSG`, uint32(2595), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 60E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",60,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2595"]]`, nil},
	uint32(2596):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 63E`, uint32(2596), `EPSG`, uint32(2596), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 63E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",63,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2596"]]`, nil},
	uint32(2597):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 66E`, uint32(2597), `EPSG`, uint32(2597), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 66E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",66,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2597"]]`, nil},
	uint32(2598):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 69E`, uint32(2598), `EPSG`, uint32(2598), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 69E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",69,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2598"]]`, nil},
	uint32(2599):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 72E`, uint32(2599), `EPSG`, uint32(2599), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 72E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",72,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2599"]]`, nil},
	uint32(2601):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 75E`, uint32(2601), `EPSG`, uint32(2601), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 75E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",75,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2601"]]`, nil},
	uint32(2602):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 78E`, uint32(2602), `EPSG`, uint32(2602), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 78E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",78,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2602"]]`, nil},
	uint32(2603):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 81E`, uint32(2603), `EPSG`, uint32(2603), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 81E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",81,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2603"]]`, nil},
	uint32(2604):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 84E`, uint32(2604), `EPSG`, uint32(2604), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 84E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",84,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2604"]]`, nil},
	uint32(2605):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 87E`, uint32(2605), `EPSG`, uint32(2605), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 87E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",87,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2605"]]`, nil},
	uint32(2606):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 90E`, uint32(2606), `EPSG`, uint32(2606), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 90E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",90,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2606"]]`, nil},
	uint32(2607):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 93E`, uint32(2607), `EPSG`, uint32(2607), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 93E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",93,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2607"]]`, nil},
	uint32(2608):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 96E`, uint32(2608), `EPSG`, uint32(2608), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 96E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",96,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2608"]]`, nil},
	uint32(2609):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 99E`, uint32(2609), `EPSG`, uint32(2609), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 99E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",99,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2609"]]`, nil},
	uint32(2610):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 102E`, uint32(2610), `EPSG`, uint32(2610), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 102E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",102,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2610"]]`, nil},
	uint32(2611):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 105E`, uint32(2611), `EPSG`, uint32(2611), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 105E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",105,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2611"]]`, nil},
	uint32(2612):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 108E`, uint32(2612), `EPSG`, uint32(2612), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 108E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",108,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2612"]]`, nil},
	uint32(2613):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 111E`, uint32(2613), `EPSG`, uint32(2613), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 111E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2613"]]`, nil},
	uint32(2614):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 114E`, uint32(2614), `EPSG`, uint32(2614), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 114E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",114,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2614"]]`, nil},
	uint32(2615):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 117E`, uint32(2615), `EPSG`, uint32(2615), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 117E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",117,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2615"]]`, nil},
	uint32(2616):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 120E`, uint32(2616), `EPSG`, uint32(2616), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 120E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",120,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2616"]]`, nil},
	uint32(2617):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 123E`, uint32(2617), `EPSG`, uint32(2617), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 123E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",123,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2617"]]`, nil},
	uint32(2618):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 126E`, uint32(2618), `EPSG`, uint32(2618), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 126E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",126,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2618"]]`, nil},
	uint32(2619):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 129E`, uint32(2619), `EPSG`, uint32(2619), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 129E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",129,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2619"]]`, nil},
	uint32(2620):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 132E`, uint32(2620), `EPSG`, uint32(2620), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 132E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",132,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2620"]]`, nil},
	uint32(2621):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 135E`, uint32(2621), `EPSG`, uint32(2621), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 135E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",135,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2621"]]`, nil},
	uint32(2622):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 138E`, uint32(2622), `EPSG`, uint32(2622), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 138E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",138,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2622"]]`, nil},
	uint32(2623):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 141E`, uint32(2623), `EPSG`, uint32(2623), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 141E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",141,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2623"]]`, nil},
	uint32(2624):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 144E`, uint32(2624), `EPSG`, uint32(2624), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 144E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",144,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2624"]]`, nil},
	uint32(2625):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 147E`, uint32(2625), `EPSG`, uint32(2625), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 147E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",147,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2625"]]`, nil},
	uint32(2626):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 150E`, uint32(2626), `EPSG`, uint32(2626), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 150E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",150,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2626"]]`, nil},
	uint32(2627):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 153E`, uint32(2627), `EPSG`, uint32(2627), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 153E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",153,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2627"]]`, nil},
	uint32(2628):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 156E`, uint32(2628), `EPSG`, uint32(2628), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 156E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",156,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2628"]]`, nil},
	uint32(2629):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 159E`, uint32(2629), `EPSG`, uint32(2629), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 159E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",159,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2629"]]`, nil},
	uint32(2630):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 162E`, uint32(2630), `EPSG`, uint32(2630), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 162E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",162,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2630"]]`, nil},
	uint32(2631):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 165E`, uint32(2631), `EPSG`, uint32(2631), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 165E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",165,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2631"]]`, nil},
	uint32(2632):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 168E`, uint32(2632), `EPSG`, uint32(2632), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 168E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",168,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2632"]]`, nil},
	uint32(2633):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 171E`, uint32(2633), `EPSG`, uint32(2633), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 171E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",171,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2633"]]`, nil},
	uint32(2634):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 174E`, uint32(2634), `EPSG`, uint32(2634), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 174E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",174,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2634"]]`, nil},
	uint32(2635):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 177E`, uint32(2635), `EPSG`, uint32(2635), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 177E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",177,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2635"]]`, nil},
	uint32(2636):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 180E`, uint32(2636), `EPSG`, uint32(2636), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 180E",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",180,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2636"]]`, nil},
	uint32(2637):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 177W`, uint32(2637), `EPSG`, uint32(2637), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 177W",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-177,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2637"]]`, nil},
	uint32(2638):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 174W`, uint32(2638), `EPSG`, uint32(2638), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 174W",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-174,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2638"]]`, nil},
	uint32(2639):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 171W`, uint32(2639), `EPSG`, uint32(2639), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 171W",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-171,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2639"]]`, nil},
	uint32(2640):  {`Pulkovo 1942 / 3-degree Gauss-Kruger CM 168W`, uint32(2640), `EPSG`, uint32(2640), `PROJCS["Pulkovo 1942 / 3-degree Gauss-Kruger CM 168W",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-168,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2640"]]`, nil},
	uint32(2641):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 7`, uint32(2641), `EPSG`, uint32(2641), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 7",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",21,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",7500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2641"]]`, nil},
	uint32(2642):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 8`, uint32(2642), `EPSG`, uint32(2642), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 8",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",24,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",8500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2642"]]`, nil},
	uint32(2643):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 9`, uint32(2643), `EPSG`, uint32(2643), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 9",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",27,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",9500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2643"]]`, nil},
	uint32(2644):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 10`, uint32(2644), `EPSG`, uint32(2644), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 10",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",30,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",10500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2644"]]`, nil},
	uint32(2645):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 11`, uint32(2645), `EPSG`, uint32(2645), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 11",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",33,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",11500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2645"]]`, nil},
	uint32(2646):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 12`, uint32(2646), `EPSG`, uint32(2646), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 12",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",36,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",12500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2646"]]`, nil},
	uint32(2647):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 13`, uint32(2647), `EPSG`, uint32(2647), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 13",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",39,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",13500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2647"]]`, nil},
	uint32(2648):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 14`, uint32(2648), `EPSG`, uint32(2648), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 14",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",42,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",14500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2648"]]`, nil},
	uint32(2649):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 15`, uint32(2649), `EPSG`, uint32(2649), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 15",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",45,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",15500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2649"]]`, nil},
	uint32(2650):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 16`, uint32(2650), `EPSG`, uint32(2650), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 16",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",48,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",16500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2650"]]`, nil},
	uint32(2651):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 17`, uint32(2651), `EPSG`, uint32(2651), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 17",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",51,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",17500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2651"]]`, nil},
	uint32(2652):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 18`, uint32(2652), `EPSG`, uint32(2652), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 18",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",54,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",18500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2652"]]`, nil},
	uint32(2653):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 19`, uint32(2653), `EPSG`, uint32(2653), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 19",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",57,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",19500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2653"]]`, nil},
	uint32(2654):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 20`, uint32(2654), `EPSG`, uint32(2654), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 20",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",60,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",20500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2654"]]`, nil},
	uint32(2655):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 21`, uint32(2655), `EPSG`, uint32(2655), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 21",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",63,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",21500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2655"]]`, nil},
	uint32(2656):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 22`, uint32(2656), `EPSG`, uint32(2656), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 22",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",66,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",22500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2656"]]`, nil},
	uint32(2657):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 23`, uint32(2657), `EPSG`, uint32(2657), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 23",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",69,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",23500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2657"]]`, nil},
	uint32(2658):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 24`, uint32(2658), `EPSG`, uint32(2658), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 24",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",72,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",24500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2658"]]`, nil},
	uint32(2659):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 25`, uint32(2659), `EPSG`, uint32(2659), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 25",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",75,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",25500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2659"]]`, nil},
	uint32(2660):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 26`, uint32(2660), `EPSG`, uint32(2660), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 26",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",78,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",26500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2660"]]`, nil},
	uint32(2661):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 27`, uint32(2661), `EPSG`, uint32(2661), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 27",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",81,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",27500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2661"]]`, nil},
	uint32(2662):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 28`, uint32(2662), `EPSG`, uint32(2662), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 28",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",84,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",28500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2662"]]`, nil},
	uint32(2663):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 29`, uint32(2663), `EPSG`, uint32(2663), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 29",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",87,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",29500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2663"]]`, nil},
	uint32(2664):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 30`, uint32(2664), `EPSG`, uint32(2664), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 30",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",90,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",30500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2664"]]`, nil},
	uint32(2665):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 31`, uint32(2665), `EPSG`, uint32(2665), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 31",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",93,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",31500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2665"]]`, nil},
	uint32(2666):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 32`, uint32(2666), `EPSG`, uint32(2666), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 32",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",96,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",32500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2666"]]`, nil},
	uint32(2667):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 33`, uint32(2667), `EPSG`, uint32(2667), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 33",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",99,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",33500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2667"]]`, nil},
	uint32(2668):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 34`, uint32(2668), `EPSG`, uint32(2668), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 34",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",102,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",34500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2668"]]`, nil},
	uint32(2669):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 35`, uint32(2669), `EPSG`, uint32(2669), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 35",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",105,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",35500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2669"]]`, nil},
	uint32(2670):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 36`, uint32(2670), `EPSG`, uint32(2670), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 36",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",108,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",36500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2670"]]`, nil},
	uint32(2671):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 37`, uint32(2671), `EPSG`, uint32(2671), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 37",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",37500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2671"]]`, nil},
	uint32(2672):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 38`, uint32(2672), `EPSG`, uint32(2672), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 38",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",114,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",38500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2672"]]`, nil},
	uint32(2673):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 39`, uint32(2673), `EPSG`, uint32(2673), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 39",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",117,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",39500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2673"]]`, nil},
	uint32(2674):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 40`, uint32(2674), `EPSG`, uint32(2674), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 40",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",120,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",40500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2674"]]`, nil},
	uint32(2675):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 41`, uint32(2675), `EPSG`, uint32(2675), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 41",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",123,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",41500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2675"]]`, nil},
	uint32(2676):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 42`, uint32(2676), `EPSG`, uint32(2676), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 42",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",126,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",42500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2676"]]`, nil},
	uint32(2677):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 43`, uint32(2677), `EPSG`, uint32(2677), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 43",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",129,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",43500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2677"]]`, nil},
	uint32(2678):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 44`, uint32(2678), `EPSG`, uint32(2678), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 44",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",132,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",44500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2678"]]`, nil},
	uint32(2679):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 45`, uint32(2679), `EPSG`, uint32(2679), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 45",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",135,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",45500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2679"]]`, nil},
	uint32(2680):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 46`, uint32(2680), `EPSG`, uint32(2680), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 46",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",138,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",46500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2680"]]`, nil},
	uint32(2681):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 47`, uint32(2681), `EPSG`, uint32(2681), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 47",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",141,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",47500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2681"]]`, nil},
	uint32(2682):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 48`, uint32(2682), `EPSG`, uint32(2682), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 48",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",144,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",48500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2682"]]`, nil},
	uint32(2683):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 49`, uint32(2683), `EPSG`, uint32(2683), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 49",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",147,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",49500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2683"]]`, nil},
	uint32(2684):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 50`, uint32(2684), `EPSG`, uint32(2684), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 50",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",150,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",50500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2684"]]`, nil},
	uint32(2685):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 51`, uint32(2685), `EPSG`, uint32(2685), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 51",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",153,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",51500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2685"]]`, nil},
	uint32(2686):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 52`, uint32(2686), `EPSG`, uint32(2686), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 52",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",156,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",52500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2686"]]`, nil},
	uint32(2687):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 53`, uint32(2687), `EPSG`, uint32(2687), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 53",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",159,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",53500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2687"]]`, nil},
	uint32(2688):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 54`, uint32(2688), `EPSG`, uint32(2688), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 54",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",162,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",54500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2688"]]`, nil},
	uint32(2689):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 55`, uint32(2689), `EPSG`, uint32(2689), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 55",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",165,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",55500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2689"]]`, nil},
	uint32(2690):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 56`, uint32(2690), `EPSG`, uint32(2690), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 56",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",168,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",56500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2690"]]`, nil},
	uint32(2691):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 57`, uint32(2691), `EPSG`, uint32(2691), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 57",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",171,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",57500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2691"]]`, nil},
	uint32(2692):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 58`, uint32(2692), `EPSG`, uint32(2692), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 58",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",174,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",58500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2692"]]`, nil},
	uint32(2693):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 59`, uint32(2693), `EPSG`, uint32(2693), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 59",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",177,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",59500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2693"]]`, nil},
	uint32(2695):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 61`, uint32(2695), `EPSG`, uint32(2695), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 61",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-177,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",61500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2695"]]`, nil},
	uint32(2696):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 62`, uint32(2696), `EPSG`, uint32(2696), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 62",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-174,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",62500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2696"]]`, nil},
	uint32(2697):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 63`, uint32(2697), `EPSG`, uint32(2697), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 63",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-171,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",63500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2697"]]`, nil},
	uint32(2698):  {`Pulkovo 1995 / 3-degree Gauss-Kruger zone 64`, uint32(2698), `EPSG`, uint32(2698), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger zone 64",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-168,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",64500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2698"]]`, nil},
	uint32(2699):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 21E`, uint32(2699), `EPSG`, uint32(2699), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 21E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",21,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2699"]]`, nil},
	uint32(2700):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 24E`, uint32(2700), `EPSG`, uint32(2700), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 24E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",24,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2700"]]`, nil},
	uint32(2701):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 27E`, uint32(2701), `EPSG`, uint32(2701), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 27E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",27,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2701"]]`, nil},
	uint32(2702):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 30E`, uint32(2702), `EPSG`, uint32(2702), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 30E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",30,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2702"]]`, nil},
	uint32(2703):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 33E`, uint32(2703), `EPSG`, uint32(2703), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 33E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",33,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2703"]]`, nil},
	uint32(2704):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 36E`, uint32(2704), `EPSG`, uint32(2704), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 36E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",36,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2704"]]`, nil},
	uint32(2705):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 39E`, uint32(2705), `EPSG`, uint32(2705), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 39E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",39,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2705"]]`, nil},
	uint32(2706):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 42E`, uint32(2706), `EPSG`, uint32(2706), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 42E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",42,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2706"]]`, nil},
	uint32(2707):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 45E`, uint32(2707), `EPSG`, uint32(2707), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 45E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",45,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2707"]]`, nil},
	uint32(2708):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 48E`, uint32(2708), `EPSG`, uint32(2708), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 48E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",48,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2708"]]`, nil},
	uint32(2709):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 51E`, uint32(2709), `EPSG`, uint32(2709), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 51E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",51,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2709"]]`, nil},
	uint32(2710):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 54E`, uint32(2710), `EPSG`, uint32(2710), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 54E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",54,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2710"]]`, nil},
	uint32(2711):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 57E`, uint32(2711), `EPSG`, uint32(2711), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 57E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",57,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2711"]]`, nil},
	uint32(2712):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 60E`, uint32(2712), `EPSG`, uint32(2712), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 60E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",60,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2712"]]`, nil},
	uint32(2713):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 63E`, uint32(2713), `EPSG`, uint32(2713), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 63E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",63,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2713"]]`, nil},
	uint32(2714):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 66E`, uint32(2714), `EPSG`, uint32(2714), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 66E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",66,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2714"]]`, nil},
	uint32(2715):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 69E`, uint32(2715), `EPSG`, uint32(2715), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 69E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",69,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2715"]]`, nil},
	uint32(2716):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 72E`, uint32(2716), `EPSG`, uint32(2716), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 72E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",72,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2716"]]`, nil},
	uint32(2717):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 75E`, uint32(2717), `EPSG`, uint32(2717), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 75E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",75,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2717"]]`, nil},
	uint32(2718):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 78E`, uint32(2718), `EPSG`, uint32(2718), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 78E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",78,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2718"]]`, nil},
	uint32(2719):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 81E`, uint32(2719), `EPSG`, uint32(2719), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 81E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",81,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2719"]]`, nil},
	uint32(2720):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 84E`, uint32(2720), `EPSG`, uint32(2720), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 84E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",84,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2720"]]`, nil},
	uint32(2721):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 87E`, uint32(2721), `EPSG`, uint32(2721), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 87E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",87,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2721"]]`, nil},
	uint32(2722):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 90E`, uint32(2722), `EPSG`, uint32(2722), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 90E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",90,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2722"]]`, nil},
	uint32(2723):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 93E`, uint32(2723), `EPSG`, uint32(2723), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 93E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",93,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2723"]]`, nil},
	uint32(2724):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 96E`, uint32(2724), `EPSG`, uint32(2724), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 96E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",96,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2724"]]`, nil},
	uint32(2725):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 99E`, uint32(2725), `EPSG`, uint32(2725), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 99E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",99,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2725"]]`, nil},
	uint32(2726):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 102E`, uint32(2726), `EPSG`, uint32(2726), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 102E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",102,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2726"]]`, nil},
	uint32(2727):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 105E`, uint32(2727), `EPSG`, uint32(2727), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 105E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",105,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2727"]]`, nil},
	uint32(2728):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 108E`, uint32(2728), `EPSG`, uint32(2728), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 108E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",108,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2728"]]`, nil},
	uint32(2729):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 111E`, uint32(2729), `EPSG`, uint32(2729), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 111E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2729"]]`, nil},
	uint32(2730):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 114E`, uint32(2730), `EPSG`, uint32(2730), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 114E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",114,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2730"]]`, nil},
	uint32(2731):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 117E`, uint32(2731), `EPSG`, uint32(2731), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 117E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",117,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2731"]]`, nil},
	uint32(2732):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 120E`, uint32(2732), `EPSG`, uint32(2732), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 120E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",120,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2732"]]`, nil},
	uint32(2733):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 123E`, uint32(2733), `EPSG`, uint32(2733), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 123E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",123,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2733"]]`, nil},
	uint32(2734):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 126E`, uint32(2734), `EPSG`, uint32(2734), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 126E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",126,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2734"]]`, nil},
	uint32(2735):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 129E`, uint32(2735), `EPSG`, uint32(2735), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 129E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",129,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2735"]]`, nil},
	uint32(2736):  {`Tete / UTM zone 36S`, uint32(2736), `EPSG`, uint32(2736), `PROJCS["Tete / UTM zone 36S",GEOGCS["Tete",DATUM["Tete",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],TOWGS84[-115.064,-87.39,-101.716,-0.058,4.001,-2.062,9.366],AUTHORITY["EPSG","6127"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4127"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",33,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2736"]]`, nil},
	uint32(2737):  {`Tete / UTM zone 37S`, uint32(2737), `EPSG`, uint32(2737), `PROJCS["Tete / UTM zone 37S",GEOGCS["Tete",DATUM["Tete",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],TOWGS84[-115.064,-87.39,-101.716,-0.058,4.001,-2.062,9.366],AUTHORITY["EPSG","6127"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4127"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",39,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2737"]]`, nil},
	uint32(2738):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 132E`, uint32(2738), `EPSG`, uint32(2738), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 132E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",132,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2738"]]`, nil},
	uint32(2739):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 135E`, uint32(2739), `EPSG`, uint32(2739), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 135E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",135,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2739"]]`, nil},
	uint32(2740):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 138E`, uint32(2740), `EPSG`, uint32(2740), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 138E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",138,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2740"]]`, nil},
	uint32(2741):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 141E`, uint32(2741), `EPSG`, uint32(2741), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 141E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",141,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2741"]]`, nil},
	uint32(2742):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 144E`, uint32(2742), `EPSG`, uint32(2742), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 144E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",144,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2742"]]`, nil},
	uint32(2743):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 147E`, uint32(2743), `EPSG`, uint32(2743), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 147E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",147,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2743"]]`, nil},
	uint32(2744):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 150E`, uint32(2744), `EPSG`, uint32(2744), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 150E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",150,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2744"]]`, nil},
	uint32(2745):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 153E`, uint32(2745), `EPSG`, uint32(2745), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 153E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",153,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2745"]]`, nil},
	uint32(2746):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 156E`, uint32(2746), `EPSG`, uint32(2746), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 156E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",156,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2746"]]`, nil},
	uint32(2747):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 159E`, uint32(2747), `EPSG`, uint32(2747), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 159E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",159,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2747"]]`, nil},
	uint32(2748):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 162E`, uint32(2748), `EPSG`, uint32(2748), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 162E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",162,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2748"]]`, nil},
	uint32(2749):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 165E`, uint32(2749), `EPSG`, uint32(2749), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 165E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",165,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2749"]]`, nil},
	uint32(2750):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 168E`, uint32(2750), `EPSG`, uint32(2750), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 168E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",168,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2750"]]`, nil},
	uint32(2751):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 171E`, uint32(2751), `EPSG`, uint32(2751), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 171E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",171,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2751"]]`, nil},
	uint32(2752):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 174E`, uint32(2752), `EPSG`, uint32(2752), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 174E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",174,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2752"]]`, nil},
	uint32(2753):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 177E`, uint32(2753), `EPSG`, uint32(2753), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 177E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",177,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2753"]]`, nil},
	uint32(2754):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 180E`, uint32(2754), `EPSG`, uint32(2754), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 180E",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",180,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2754"]]`, nil},
	uint32(2755):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 177W`, uint32(2755), `EPSG`, uint32(2755), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 177W",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-177,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2755"]]`, nil},
	uint32(2756):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 174W`, uint32(2756), `EPSG`, uint32(2756), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 174W",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-174,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2756"]]`, nil},
	uint32(2757):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 171W`, uint32(2757), `EPSG`, uint32(2757), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 171W",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-171,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2757"]]`, nil},
	uint32(2758):  {`Pulkovo 1995 / 3-degree Gauss-Kruger CM 168W`, uint32(2758), `EPSG`, uint32(2758), `PROJCS["Pulkovo 1995 / 3-degree Gauss-Kruger CM 168W",GEOGCS["Pulkovo 1995",DATUM["Pulkovo 1995",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[24.47,-130.89,-81.56,0,0,0.13,-0.22],AUTHORITY["EPSG","6200"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4200"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-168,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2758"]]`, nil},
	uint32(2759):  {`NAD83(HARN) / Alabama East`, uint32(2759), `EPSG`, uint32(2759), `PROJCS["NAD83(HARN) / Alabama East",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",30.5,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-85.8333333333333,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.99996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",200000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2759"]]`, nil},
	uint32(2760):  {`NAD83(HARN) / Alabama West`, uint32(2760), `EPSG`, uint32(2760), `PROJCS["NAD83(HARN) / Alabama West",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",30,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-87.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999933333,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",600000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2760"]]`, nil},
	uint32(2761):  {`NAD83(HARN) / Arizona East`, uint32(2761), `EPSG`, uint32(2761), `PROJCS["NAD83(HARN) / Arizona East",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",31,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-110.177777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",213360,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2761"]]`, nil},
	uint32(2762):  {`NAD83(HARN) / Arizona Central`, uint32(2762), `EPSG`, uint32(2762), `PROJCS["NAD83(HARN) / Arizona Central",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",31,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-111.927777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",213360,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2762"]]`, nil},
	uint32(2763):  {`NAD83(HARN) / Arizona West`, uint32(2763), `EPSG`, uint32(2763), `PROJCS["NAD83(HARN) / Arizona West",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",31,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-113.75,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999933333,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",213360,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2763"]]`, nil},
	uint32(2764):  {`NAD83(HARN) / Arkansas North`, uint32(2764), `EPSG`, uint32(2764), `PROJCS["NAD83(HARN) / Arkansas North",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",34.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-92,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",36.2333333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",34.9333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",400000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2764"]]`, nil},
	uint32(2765):  {`NAD83(HARN) / Arkansas South`, uint32(2765), `EPSG`, uint32(2765), `PROJCS["NAD83(HARN) / Arkansas South",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",32.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-92,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",34.7666666666667,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",33.3111111111111,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",400000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",400000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2765"]]`, nil},
	uint32(2766):  {`NAD83(HARN) / California zone 1`, uint32(2766), `EPSG`, uint32(2766), `PROJCS["NAD83(HARN) / California zone 1",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",39.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-122,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",41.6777777777778,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",40,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",2000000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",500000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2766"]]`, nil},
	uint32(2767):  {`NAD83(HARN) / California zone 2`, uint32(2767), `EPSG`, uint32(2767), `PROJCS["NAD83(HARN) / California zone 2",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",37.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-122,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",39.8333333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",38.3333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",2000000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",500000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2767"]]`, nil},
	uint32(2768):  {`NAD83(HARN) / California zone 3`, uint32(2768), `EPSG`, uint32(2768), `PROJCS["NAD83(HARN) / California zone 3",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",36.5111111111111,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-120.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",38.4444444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",37.0777777777778,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",2000000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",500000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2768"]]`, nil},
	uint32(2769):  {`NAD83(HARN) / California zone 4`, uint32(2769), `EPSG`, uint32(2769), `PROJCS["NAD83(HARN) / California zone 4",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",35.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-119,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",37.2611111111111,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",36,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",2000000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",500000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2769"]]`, nil},
	uint32(2770):  {`NAD83(HARN) / California zone 5`, uint32(2770), `EPSG`, uint32(2770), `PROJCS["NAD83(HARN) / California zone 5",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",33.5111111111111,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-118,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",35.4666666666667,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",34.0333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",2000000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",500000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2770"]]`, nil},
	uint32(2771):  {`NAD83(HARN) / California zone 6`, uint32(2771), `EPSG`, uint32(2771), `PROJCS["NAD83(HARN) / California zone 6",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",32.1666666666667,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-116.25,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",33.8833333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",32.7944444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",2000000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",500000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2771"]]`, nil},
	uint32(2772):  {`NAD83(HARN) / Colorado North`, uint32(2772), `EPSG`, uint32(2772), `PROJCS["NAD83(HARN) / Colorado North",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",39.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-105.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",40.7944444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",39.7277777777778,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",914401.8289,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",304800.6096,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2772"]]`, nil},
	uint32(2773):  {`NAD83(HARN) / Colorado Central`, uint32(2773), `EPSG`, uint32(2773), `PROJCS["NAD83(HARN) / Colorado Central",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",37.8333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-105.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",39.75,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",38.45,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",914401.8289,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",304800.6096,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2773"]]`, nil},
	uint32(2774):  {`NAD83(HARN) / Colorado South`, uint32(2774), `EPSG`, uint32(2774), `PROJCS["NAD83(HARN) / Colorado South",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",36.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-105.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",38.4444444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",37.2333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",914401.8289,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",304800.6096,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2774"]]`, nil},
	uint32(2775):  {`NAD83(HARN) / Connecticut`, uint32(2775), `EPSG`, uint32(2775), `PROJCS["NAD83(HARN) / Connecticut",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",40.8333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-72.75,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",41.8666666666667,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",41.2111111111111,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",304800.6096,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",152400.3048,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2775"]]`, nil},
	uint32(2776):  {`NAD83(HARN) / Delaware`, uint32(2776), `EPSG`, uint32(2776), `PROJCS["NAD83(HARN) / Delaware",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",38,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-75.4166666666667,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999995,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",200000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2776"]]`, nil},
	uint32(2777):  {`NAD83(HARN) / Florida East`, uint32(2777), `EPSG`, uint32(2777), `PROJCS["NAD83(HARN) / Florida East",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",24.3444444444444,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-81,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999941177,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",200000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2777"]]`, nil},
	uint32(2778):  {`NAD83(HARN) / Florida West`, uint32(2778), `EPSG`, uint32(2778), `PROJCS["NAD83(HARN) / Florida West",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",24.3444444444444,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-82,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999941177,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",200000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2778"]]`, nil},
	uint32(2779):  {`NAD83(HARN) / Florida North`, uint32(2779), `EPSG`, uint32(2779), `PROJCS["NAD83(HARN) / Florida North",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",29,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-84.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",30.7611111111111,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",29.5833333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",600000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2779"]]`, nil},
	uint32(2780):  {`NAD83(HARN) / Georgia East`, uint32(2780), `EPSG`, uint32(2780), `PROJCS["NAD83(HARN) / Georgia East",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",30,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-82.1777777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",200000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2780"]]`, nil},
	uint32(2781):  {`NAD83(HARN) / Georgia West`, uint32(2781), `EPSG`, uint32(2781), `PROJCS["NAD83(HARN) / Georgia West",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",30,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-84.1777777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",700000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2781"]]`, nil},
	uint32(2782):  {`NAD83(HARN) / Hawaii zone 1`, uint32(2782), `EPSG`, uint32(2782), `PROJCS["NAD83(HARN) / Hawaii zone 1",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",18.8333333333333,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-155.5,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999966667,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2782"]]`, nil},
	uint32(2783):  {`NAD83(HARN) / Hawaii zone 2`, uint32(2783), `EPSG`, uint32(2783), `PROJCS["NAD83(HARN) / Hawaii zone 2",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",20.3444444444444,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-156.666666666667,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999966667,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2783"]]`, nil},
	uint32(2784):  {`NAD83(HARN) / Hawaii zone 3`, uint32(2784), `EPSG`, uint32(2784), `PROJCS["NAD83(HARN) / Hawaii zone 3",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",21.1666666666667,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-158,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.99999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2784"]]`, nil},
	uint32(2785):  {`NAD83(HARN) / Hawaii zone 4`, uint32(2785), `EPSG`, uint32(2785), `PROJCS["NAD83(HARN) / Hawaii zone 4",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",21.8333333333333,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-159.5,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.99999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2785"]]`, nil},
	uint32(2786):  {`NAD83(HARN) / Hawaii zone 5`, uint32(2786), `EPSG`, uint32(2786), `PROJCS["NAD83(HARN) / Hawaii zone 5",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",21.6777777777778,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-160.177777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2786"]]`, nil},
	uint32(2787):  {`NAD83(HARN) / Idaho East`, uint32(2787), `EPSG`, uint32(2787), `PROJCS["NAD83(HARN) / Idaho East",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",41.6777777777778,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-112.177777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999947368,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",200000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2787"]]`, nil},
	uint32(2788):  {`NAD83(HARN) / Idaho Central`, uint32(2788), `EPSG`, uint32(2788), `PROJCS["NAD83(HARN) / Idaho Central",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",41.6777777777778,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-114,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999947368,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2788"]]`, nil},
	uint32(2789):  {`NAD83(HARN) / Idaho West`, uint32(2789), `EPSG`, uint32(2789), `PROJCS["NAD83(HARN) / Idaho West",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",41.6777777777778,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-115.75,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999933333,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",800000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2789"]]`, nil},
	uint32(2790):  {`NAD83(HARN) / Illinois East`, uint32(2790), `EPSG`, uint32(2790), `PROJCS["NAD83(HARN) / Illinois East",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",36.6777777777778,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-88.3333333333333,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999975,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",300000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2790"]]`, nil},
	uint32(2791):  {`NAD83(HARN) / Illinois West`, uint32(2791), `EPSG`, uint32(2791), `PROJCS["NAD83(HARN) / Illinois West",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",36.6777777777778,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-90.1777777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999941177,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",700000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2791"]]`, nil},
	uint32(2792):  {`NAD83(HARN) / Indiana East`, uint32(2792), `EPSG`, uint32(2792), `PROJCS["NAD83(HARN) / Indiana East",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",37.5111111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-85.6666666666667,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999966667,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",100000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",250000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2792"]]`, nil},
	uint32(2793):  {`NAD83(HARN) / Indiana West`, uint32(2793), `EPSG`, uint32(2793), `PROJCS["NAD83(HARN) / Indiana West",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",37.5111111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-87.0944444444444,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999966667,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",900000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",250000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2793"]]`, nil},
	uint32(2794):  {`NAD83(HARN) / Iowa North`, uint32(2794), `EPSG`, uint32(2794), `PROJCS["NAD83(HARN) / Iowa North",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",41.5111111111111,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-93.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",43.2777777777778,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",42.0777777777778,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1500000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",1000000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2794"]]`, nil},
	uint32(2795):  {`NAD83(HARN) / Iowa South`, uint32(2795), `EPSG`, uint32(2795), `PROJCS["NAD83(HARN) / Iowa South",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",40,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-93.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",41.7944444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",40.6277777777778,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",500000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2795"]]`, nil},
	uint32(2796):  {`NAD83(HARN) / Kansas North`, uint32(2796), `EPSG`, uint32(2796), `PROJCS["NAD83(HARN) / Kansas North",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",38.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-98,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",39.7944444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",38.7277777777778,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",400000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2796"]]`, nil},
	uint32(2797):  {`NAD83(HARN) / Kansas South`, uint32(2797), `EPSG`, uint32(2797), `PROJCS["NAD83(HARN) / Kansas South",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",36.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-98.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",38.5666666666667,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",37.2777777777778,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",400000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",400000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2797"]]`, nil},
	uint32(2798):  {`NAD83(HARN) / Kentucky North`, uint32(2798), `EPSG`, uint32(2798), `PROJCS["NAD83(HARN) / Kentucky North",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",37.5111111111111,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-84.25,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",37.9777777777778,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",38.9777777777778,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",500000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2798"]]`, nil},
	uint32(2799):  {`NAD83(HARN) / Kentucky South`, uint32(2799), `EPSG`, uint32(2799), `PROJCS["NAD83(HARN) / Kentucky South",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",36.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-85.75,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",37.9333333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",36.7444444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",500000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",500000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2799"]]`, nil},
	uint32(2800):  {`NAD83(HARN) / Louisiana North`, uint32(2800), `EPSG`, uint32(2800), `PROJCS["NAD83(HARN) / Louisiana North",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",30.5,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-92.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",32.6777777777778,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",31.1666666666667,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1000000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2800"]]`, nil},
	uint32(2801):  {`NAD83(HARN) / Louisiana South`, uint32(2801), `EPSG`, uint32(2801), `PROJCS["NAD83(HARN) / Louisiana South",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",28.5,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-91.3333333333333,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",30.7,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",29.3111111111111,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1000000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2801"]]`, nil},
	uint32(2802):  {`NAD83(HARN) / Maine East`, uint32(2802), `EPSG`, uint32(2802), `PROJCS["NAD83(HARN) / Maine East",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",43.6777777777778,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-68.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",300000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2802"]]`, nil},
	uint32(2803):  {`NAD83(HARN) / Maine West`, uint32(2803), `EPSG`, uint32(2803), `PROJCS["NAD83(HARN) / Maine West",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",42.8333333333333,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-70.1777777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999966667,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",900000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2803"]]`, nil},
	uint32(2804):  {`NAD83(HARN) / Maryland`, uint32(2804), `EPSG`, uint32(2804), `PROJCS["NAD83(HARN) / Maryland",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",37.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-77,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",39.45,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",38.3111111111111,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",400000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2804"]]`, nil},
	uint32(2805):  {`NAD83(HARN) / Massachusetts Mainland`, uint32(2805), `EPSG`, uint32(2805), `PROJCS["NAD83(HARN) / Massachusetts Mainland",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",41,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-71.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",42.6944444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",41.7277777777778,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",200000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",750000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2805"]]`, nil},
	uint32(2806):  {`NAD83(HARN) / Massachusetts Island`, uint32(2806), `EPSG`, uint32(2806), `PROJCS["NAD83(HARN) / Massachusetts Island",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",41,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-70.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",41.4944444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",41.2833333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",500000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2806"]]`, nil},
	uint32(2807):  {`NAD83(HARN) / Michigan North`, uint32(2807), `EPSG`, uint32(2807), `PROJCS["NAD83(HARN) / Michigan North",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",44.7944444444444,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-87,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",47.0944444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",45.4944444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",8000000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2807"]]`, nil},
	uint32(2808):  {`NAD83(HARN) / Michigan Central`, uint32(2808), `EPSG`, uint32(2808), `PROJCS["NAD83(HARN) / Michigan Central",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",43.3277777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-84.3777777777778,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",45.7,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",44.1944444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",6000000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2808"]]`, nil},
	uint32(2809):  {`NAD83(HARN) / Michigan South`, uint32(2809), `EPSG`, uint32(2809), `PROJCS["NAD83(HARN) / Michigan South",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",41.5111111111111,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-84.3777777777778,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",43.6777777777778,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",42.1,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",4000000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2809"]]`, nil},
	uint32(2810):  {`NAD83(HARN) / Minnesota North`, uint32(2810), `EPSG`, uint32(2810), `PROJCS["NAD83(HARN) / Minnesota North",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",46.5111111111111,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-93.1,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",48.6333333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",47.0333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",800000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",100000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2810"]]`, nil},
	uint32(2811):  {`NAD83(HARN) / Minnesota Central`, uint32(2811), `EPSG`, uint32(2811), `PROJCS["NAD83(HARN) / Minnesota Central",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",45,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-94.25,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",47.05,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",45.6277777777778,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",800000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",100000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2811"]]`, nil},
	uint32(2812):  {`NAD83(HARN) / Minnesota South`, uint32(2812), `EPSG`, uint32(2812), `PROJCS["NAD83(HARN) / Minnesota South",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",43,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-94,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",45.2166666666667,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",43.7944444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",800000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",100000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2812"]]`, nil},
	uint32(2813):  {`NAD83(HARN) / Mississippi East`, uint32(2813), `EPSG`, uint32(2813), `PROJCS["NAD83(HARN) / Mississippi East",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",29.5,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-88.8333333333333,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.99995,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",300000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2813"]]`, nil},
	uint32(2814):  {`NAD83(HARN) / Mississippi West`, uint32(2814), `EPSG`, uint32(2814), `PROJCS["NAD83(HARN) / Mississippi West",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",29.5,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-90.3333333333333,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.99995,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",700000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2814"]]`, nil},
	uint32(2815):  {`NAD83(HARN) / Missouri East`, uint32(2815), `EPSG`, uint32(2815), `PROJCS["NAD83(HARN) / Missouri East",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",35.8333333333333,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-90.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999933333,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",250000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2815"]]`, nil},
	uint32(2816):  {`NAD83(HARN) / Missouri Central`, uint32(2816), `EPSG`, uint32(2816), `PROJCS["NAD83(HARN) / Missouri Central",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",35.8333333333333,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-92.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999933333,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2816"]]`, nil},
	uint32(2817):  {`NAD83(HARN) / Missouri West`, uint32(2817), `EPSG`, uint32(2817), `PROJCS["NAD83(HARN) / Missouri West",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",36.1666666666667,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-94.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999941177,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",850000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2817"]]`, nil},
	uint32(2818):  {`NAD83(HARN) / Montana`, uint32(2818), `EPSG`, uint32(2818), `PROJCS["NAD83(HARN) / Montana",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",44.2611111111111,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-109.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",49,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",45,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",600000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2818"]]`, nil},
	uint32(2819):  {`NAD83(HARN) / Nebraska`, uint32(2819), `EPSG`, uint32(2819), `PROJCS["NAD83(HARN) / Nebraska",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",39.8333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-100,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",43,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",40,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",500000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2819"]]`, nil},
	uint32(2820):  {`NAD83(HARN) / Nevada East`, uint32(2820), `EPSG`, uint32(2820), `PROJCS["NAD83(HARN) / Nevada East",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",34.75,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-115.594444444444,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",200000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",8000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2820"]]`, nil},
	uint32(2821):  {`NAD83(HARN) / Nevada Central`, uint32(2821), `EPSG`, uint32(2821), `PROJCS["NAD83(HARN) / Nevada Central",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",34.75,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-116.666666666667,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",6000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2821"]]`, nil},
	uint32(2822):  {`NAD83(HARN) / Nevada West`, uint32(2822), `EPSG`, uint32(2822), `PROJCS["NAD83(HARN) / Nevada West",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",34.75,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-118.594444444444,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",800000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",4000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2822"]]`, nil},
	uint32(2823):  {`NAD83(HARN) / New Hampshire`, uint32(2823), `EPSG`, uint32(2823), `PROJCS["NAD83(HARN) / New Hampshire",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",42.5111111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-71.6666666666667,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999966667,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",300000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2823"]]`, nil},
	uint32(2824):  {`NAD83(HARN) / New Jersey`, uint32(2824), `EPSG`, uint32(2824), `PROJCS["NAD83(HARN) / New Jersey",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",38.8333333333333,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-74.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",150000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2824"]]`, nil},
	uint32(2825):  {`NAD83(HARN) / New Mexico East`, uint32(2825), `EPSG`, uint32(2825), `PROJCS["NAD83(HARN) / New Mexico East",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",31,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-104.333333333333,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999909091,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",165000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2825"]]`, nil},
	uint32(2826):  {`NAD83(HARN) / New Mexico Central`, uint32(2826), `EPSG`, uint32(2826), `PROJCS["NAD83(HARN) / New Mexico Central",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",31,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-106.25,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2826"]]`, nil},
	uint32(2827):  {`NAD83(HARN) / New Mexico West`, uint32(2827), `EPSG`, uint32(2827), `PROJCS["NAD83(HARN) / New Mexico West",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",31,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-107.833333333333,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999916667,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",830000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2827"]]`, nil},
	uint32(2828):  {`NAD83(HARN) / New York East`, uint32(2828), `EPSG`, uint32(2828), `PROJCS["NAD83(HARN) / New York East",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",38.8333333333333,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-74.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",150000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2828"]]`, nil},
	uint32(2829):  {`NAD83(HARN) / New York Central`, uint32(2829), `EPSG`, uint32(2829), `PROJCS["NAD83(HARN) / New York Central",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",40,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-76.5944444444444,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999375,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",250000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2829"]]`, nil},
	uint32(2830):  {`NAD83(HARN) / New York West`, uint32(2830), `EPSG`, uint32(2830), `PROJCS["NAD83(HARN) / New York West",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",40,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-78.5944444444444,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999375,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",350000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2830"]]`, nil},
	uint32(2831):  {`NAD83(HARN) / New York Long Island`, uint32(2831), `EPSG`, uint32(2831), `PROJCS["NAD83(HARN) / New York Long Island",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",40.1666666666667,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-74,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",41.0333333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",40.6777777777778,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",300000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2831"]]`, nil},
	uint32(2832):  {`NAD83(HARN) / North Dakota North`, uint32(2832), `EPSG`, uint32(2832), `PROJCS["NAD83(HARN) / North Dakota North",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",47,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-100.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",48.7444444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",47.4444444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",600000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2832"]]`, nil},
	uint32(2833):  {`NAD83(HARN) / North Dakota South`, uint32(2833), `EPSG`, uint32(2833), `PROJCS["NAD83(HARN) / North Dakota South",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",45.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-100.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",47.4944444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",46.1944444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",600000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2833"]]`, nil},
	uint32(2834):  {`NAD83(HARN) / Ohio North`, uint32(2834), `EPSG`, uint32(2834), `PROJCS["NAD83(HARN) / Ohio North",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",39.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-82.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",41.7,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",40.4444444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",600000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2834"]]`, nil},
	uint32(2835):  {`NAD83(HARN) / Ohio South`, uint32(2835), `EPSG`, uint32(2835), `PROJCS["NAD83(HARN) / Ohio South",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",38,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-82.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",40.0333333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",38.7444444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",600000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2835"]]`, nil},
	uint32(2836):  {`NAD83(HARN) / Oklahoma North`, uint32(2836), `EPSG`, uint32(2836), `PROJCS["NAD83(HARN) / Oklahoma North",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",35,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-98,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",36.7666666666667,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",35.5666666666667,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",600000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2836"]]`, nil},
	uint32(2837):  {`NAD83(HARN) / Oklahoma South`, uint32(2837), `EPSG`, uint32(2837), `PROJCS["NAD83(HARN) / Oklahoma South",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",33.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-98,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",35.2333333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",33.9333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",600000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2837"]]`, nil},
	uint32(2838):  {`NAD83(HARN) / Oregon North`, uint32(2838), `EPSG`, uint32(2838), `PROJCS["NAD83(HARN) / Oregon North",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",43.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-120.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",46,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",44.3333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",2500000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2838"]]`, nil},
	uint32(2839):  {`NAD83(HARN) / Oregon South`, uint32(2839), `EPSG`, uint32(2839), `PROJCS["NAD83(HARN) / Oregon South",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",41.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-120.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",44,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",42.3333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1500000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2839"]]`, nil},
	uint32(2840):  {`NAD83(HARN) / Rhode Island`, uint32(2840), `EPSG`, uint32(2840), `PROJCS["NAD83(HARN) / Rhode Island",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",41.0944444444444,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-71.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.99999375,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",100000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2840"]]`, nil},
	uint32(2841):  {`NAD83(HARN) / South Dakota North`, uint32(2841), `EPSG`, uint32(2841), `PROJCS["NAD83(HARN) / South Dakota North",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",43.8333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-100,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",45.6944444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",44.4166666666667,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",600000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2841"]]`, nil},
	uint32(2842):  {`NAD83(HARN) / South Dakota South`, uint32(2842), `EPSG`, uint32(2842), `PROJCS["NAD83(HARN) / South Dakota South",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",42.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-100.333333333333,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",44.4,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",42.8333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",600000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2842"]]`, nil},
	uint32(2843):  {`NAD83(HARN) / Tennessee`, uint32(2843), `EPSG`, uint32(2843), `PROJCS["NAD83(HARN) / Tennessee",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",34.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-86,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",36.4166666666667,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",35.2611111111111,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",600000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2843"]]`, nil},
	uint32(2844):  {`NAD83(HARN) / Texas North`, uint32(2844), `EPSG`, uint32(2844), `PROJCS["NAD83(HARN) / Texas North",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",34,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-101.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",36.1944444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",34.65,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",200000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",1000000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2844"]]`, nil},
	uint32(2845):  {`NAD83(HARN) / Texas North Central`, uint32(2845), `EPSG`, uint32(2845), `PROJCS["NAD83(HARN) / Texas North Central",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",31.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-98.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",33.9777777777778,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",32.1444444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",600000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",2000000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2845"]]`, nil},
	uint32(2846):  {`NAD83(HARN) / Texas Central`, uint32(2846), `EPSG`, uint32(2846), `PROJCS["NAD83(HARN) / Texas Central",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",29.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-100.333333333333,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",31.8833333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",30.1166666666667,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",700000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",3000000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2846"]]`, nil},
	uint32(2847):  {`NAD83(HARN) / Texas South Central`, uint32(2847), `EPSG`, uint32(2847), `PROJCS["NAD83(HARN) / Texas South Central",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",27.8333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-99,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",30.2833333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",28.3833333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",600000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",4000000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2847"]]`, nil},
	uint32(2848):  {`NAD83(HARN) / Texas South`, uint32(2848), `EPSG`, uint32(2848), `PROJCS["NAD83(HARN) / Texas South",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",25.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-98.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",27.8333333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",26.1666666666667,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",300000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",5000000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2848"]]`, nil},
	uint32(2849):  {`NAD83(HARN) / Utah North`, uint32(2849), `EPSG`, uint32(2849), `PROJCS["NAD83(HARN) / Utah North",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",40.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-111.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",41.7944444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",40.7277777777778,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",500000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",1000000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2849"]]`, nil},
	uint32(2850):  {`NAD83(HARN) / Utah Central`, uint32(2850), `EPSG`, uint32(2850), `PROJCS["NAD83(HARN) / Utah Central",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",38.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-111.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",40.65,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",39.0277777777778,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",500000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",2000000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2850"]]`, nil},
	uint32(2851):  {`NAD83(HARN) / Utah South`, uint32(2851), `EPSG`, uint32(2851), `PROJCS["NAD83(HARN) / Utah South",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",36.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-111.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",38.35,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",37.2166666666667,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",500000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",3000000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2851"]]`, nil},
	uint32(2852):  {`NAD83(HARN) / Vermont`, uint32(2852), `EPSG`, uint32(2852), `PROJCS["NAD83(HARN) / Vermont",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",42.5111111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-72.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999964286,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2852"]]`, nil},
	uint32(2853):  {`NAD83(HARN) / Virginia North`, uint32(2853), `EPSG`, uint32(2853), `PROJCS["NAD83(HARN) / Virginia North",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",37.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-78.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",39.2111111111111,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",38.0333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",3500000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",2000000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2853"]]`, nil},
	uint32(2854):  {`NAD83(HARN) / Virginia South`, uint32(2854), `EPSG`, uint32(2854), `PROJCS["NAD83(HARN) / Virginia South",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",36.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-78.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",37.9777777777778,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",36.7666666666667,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",3500000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",1000000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2854"]]`, nil},
	uint32(2855):  {`NAD83(HARN) / Washington North`, uint32(2855), `EPSG`, uint32(2855), `PROJCS["NAD83(HARN) / Washington North",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",47,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-120.833333333333,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",48.7444444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",47.5111111111111,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",500000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2855"]]`, nil},
	uint32(2856):  {`NAD83(HARN) / Washington South`, uint32(2856), `EPSG`, uint32(2856), `PROJCS["NAD83(HARN) / Washington South",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",45.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-120.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",47.3333333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",45.8333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",500000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2856"]]`, nil},
	uint32(2857):  {`NAD83(HARN) / West Virginia North`, uint32(2857), `EPSG`, uint32(2857), `PROJCS["NAD83(HARN) / West Virginia North",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",38.5111111111111,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-79.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",40.2611111111111,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",39,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",600000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2857"]]`, nil},
	uint32(2858):  {`NAD83(HARN) / West Virginia South`, uint32(2858), `EPSG`, uint32(2858), `PROJCS["NAD83(HARN) / West Virginia South",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",37,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-81,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",38.8833333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",37.4944444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",600000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2858"]]`, nil},
	uint32(2859):  {`NAD83(HARN) / Wisconsin North`, uint32(2859), `EPSG`, uint32(2859), `PROJCS["NAD83(HARN) / Wisconsin North",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",45.1666666666667,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-90,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",46.7666666666667,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",45.5666666666667,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",600000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2859"]]`, nil},
	uint32(2860):  {`NAD83(HARN) / Wisconsin Central`, uint32(2860), `EPSG`, uint32(2860), `PROJCS["NAD83(HARN) / Wisconsin Central",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",43.8333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-90,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",45.5111111111111,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",44.2611111111111,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",600000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2860"]]`, nil},
	uint32(2861):  {`NAD83(HARN) / Wisconsin South`, uint32(2861), `EPSG`, uint32(2861), `PROJCS["NAD83(HARN) / Wisconsin South",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",42,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-90,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",44.0777777777778,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",42.7444444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",600000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2861"]]`, nil},
	uint32(2862):  {`NAD83(HARN) / Wyoming East`, uint32(2862), `EPSG`, uint32(2862), `PROJCS["NAD83(HARN) / Wyoming East",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",40.5111111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-105.177777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999375,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",200000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2862"]]`, nil},
	uint32(2863):  {`NAD83(HARN) / Wyoming East Central`, uint32(2863), `EPSG`, uint32(2863), `PROJCS["NAD83(HARN) / Wyoming East Central",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",40.5111111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-107.333333333333,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999375,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",100000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2863"]]`, nil},
	uint32(2864):  {`NAD83(HARN) / Wyoming West Central`, uint32(2864), `EPSG`, uint32(2864), `PROJCS["NAD83(HARN) / Wyoming West Central",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",40.5111111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-108.75,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999375,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",600000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2864"]]`, nil},
	uint32(2865):  {`NAD83(HARN) / Wyoming West`, uint32(2865), `EPSG`, uint32(2865), `PROJCS["NAD83(HARN) / Wyoming West",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",40.5111111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-110.094444444444,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999375,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",800000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",100000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2865"]]`, nil},
	uint32(2866):  {`NAD83(HARN) / Puerto Rico and Virgin Is.`, uint32(2866), `EPSG`, uint32(2866), `PROJCS["NAD83(HARN) / Puerto Rico and Virgin Is.",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",17.8333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-66.4333333333334,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",18.4333333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",18.0444444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",200000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",200000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2866"]]`, nil},
	uint32(2867):  {`NAD83(HARN) / Arizona East (ft)`, uint32(2867), `EPSG`, uint32(2867), `PROJCS["NAD83(HARN) / Arizona East (ft)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",31,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-110.177777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",700000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2867"]]`, nil},
	uint32(2868):  {`NAD83(HARN) / Arizona Central (ft)`, uint32(2868), `EPSG`, uint32(2868), `PROJCS["NAD83(HARN) / Arizona Central (ft)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",31,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-111.927777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",700000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2868"]]`, nil},
	uint32(2869):  {`NAD83(HARN) / Arizona West (ft)`, uint32(2869), `EPSG`, uint32(2869), `PROJCS["NAD83(HARN) / Arizona West (ft)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",31,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-113.75,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999933333,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",700000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2869"]]`, nil},
	uint32(2870):  {`NAD83(HARN) / California zone 1 (ftUS)`, uint32(2870), `EPSG`, uint32(2870), `PROJCS["NAD83(HARN) / California zone 1 (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",39.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-122,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",41.6777777777778,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",40,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",6561666.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",1640416.667,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2870"]]`, nil},
	uint32(2871):  {`NAD83(HARN) / California zone 2 (ftUS)`, uint32(2871), `EPSG`, uint32(2871), `PROJCS["NAD83(HARN) / California zone 2 (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",37.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-122,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",39.8333333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",38.3333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",6561666.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",1640416.667,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2871"]]`, nil},
	uint32(2872):  {`NAD83(HARN) / California zone 3 (ftUS)`, uint32(2872), `EPSG`, uint32(2872), `PROJCS["NAD83(HARN) / California zone 3 (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",36.5111111111111,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-120.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",38.4444444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",37.0777777777778,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",6561666.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",1640416.667,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2872"]]`, nil},
	uint32(2873):  {`NAD83(HARN) / California zone 4 (ftUS)`, uint32(2873), `EPSG`, uint32(2873), `PROJCS["NAD83(HARN) / California zone 4 (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",35.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-119,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",37.2611111111111,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",36,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",6561666.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",1640416.667,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2873"]]`, nil},
	uint32(2874):  {`NAD83(HARN) / California zone 5 (ftUS)`, uint32(2874), `EPSG`, uint32(2874), `PROJCS["NAD83(HARN) / California zone 5 (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",33.5111111111111,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-118,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",35.4666666666667,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",34.0333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",6561666.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",1640416.667,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2874"]]`, nil},
	uint32(2875):  {`NAD83(HARN) / California zone 6 (ftUS)`, uint32(2875), `EPSG`, uint32(2875), `PROJCS["NAD83(HARN) / California zone 6 (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",32.1666666666667,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-116.25,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",33.8833333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",32.7944444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",6561666.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",1640416.667,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2875"]]`, nil},
	uint32(2876):  {`NAD83(HARN) / Colorado North (ftUS)`, uint32(2876), `EPSG`, uint32(2876), `PROJCS["NAD83(HARN) / Colorado North (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",39.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-105.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",40.7944444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",39.7277777777778,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",3000000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",1000000,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2876"]]`, nil},
	uint32(2877):  {`NAD83(HARN) / Colorado Central (ftUS)`, uint32(2877), `EPSG`, uint32(2877), `PROJCS["NAD83(HARN) / Colorado Central (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",37.8333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-105.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",39.75,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",38.45,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",3000000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",1000000,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2877"]]`, nil},
	uint32(2878):  {`NAD83(HARN) / Colorado South (ftUS)`, uint32(2878), `EPSG`, uint32(2878), `PROJCS["NAD83(HARN) / Colorado South (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",36.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-105.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",38.4444444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",37.2333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",3000000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",1000000,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2878"]]`, nil},
	uint32(2879):  {`NAD83(HARN) / Connecticut (ftUS)`, uint32(2879), `EPSG`, uint32(2879), `PROJCS["NAD83(HARN) / Connecticut (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",40.8333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-72.75,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",41.8666666666667,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",41.2111111111111,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1000000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",500000,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2879"]]`, nil},
	uint32(2880):  {`NAD83(HARN) / Delaware (ftUS)`, uint32(2880), `EPSG`, uint32(2880), `PROJCS["NAD83(HARN) / Delaware (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",38,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-75.4166666666667,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999995,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",656166.667,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2880"]]`, nil},
	uint32(2881):  {`NAD83(HARN) / Florida East (ftUS)`, uint32(2881), `EPSG`, uint32(2881), `PROJCS["NAD83(HARN) / Florida East (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",24.3444444444444,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-81,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999941177,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",656166.667,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2881"]]`, nil},
	uint32(2882):  {`NAD83(HARN) / Florida West (ftUS)`, uint32(2882), `EPSG`, uint32(2882), `PROJCS["NAD83(HARN) / Florida West (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",24.3444444444444,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-82,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999941177,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",656166.667,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2882"]]`, nil},
	uint32(2883):  {`NAD83(HARN) / Florida North (ftUS)`, uint32(2883), `EPSG`, uint32(2883), `PROJCS["NAD83(HARN) / Florida North (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",29,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-84.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",30.7611111111111,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",29.5833333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1968500,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2883"]]`, nil},
	uint32(2884):  {`NAD83(HARN) / Georgia East (ftUS)`, uint32(2884), `EPSG`, uint32(2884), `PROJCS["NAD83(HARN) / Georgia East (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",30,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-82.1777777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",656166.667,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2884"]]`, nil},
	uint32(2885):  {`NAD83(HARN) / Georgia West (ftUS)`, uint32(2885), `EPSG`, uint32(2885), `PROJCS["NAD83(HARN) / Georgia West (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",30,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-84.1777777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",2296583.333,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2885"]]`, nil},
	uint32(2886):  {`NAD83(HARN) / Idaho East (ftUS)`, uint32(2886), `EPSG`, uint32(2886), `PROJCS["NAD83(HARN) / Idaho East (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",41.6777777777778,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-112.177777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999947368,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",656166.667,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2886"]]`, nil},
	uint32(2887):  {`NAD83(HARN) / Idaho Central (ftUS)`, uint32(2887), `EPSG`, uint32(2887), `PROJCS["NAD83(HARN) / Idaho Central (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",41.6777777777778,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-114,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999947368,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",1640416.667,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2887"]]`, nil},
	uint32(2888):  {`NAD83(HARN) / Idaho West (ftUS)`, uint32(2888), `EPSG`, uint32(2888), `PROJCS["NAD83(HARN) / Idaho West (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",41.6777777777778,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-115.75,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999933333,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",2624666.667,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2888"]]`, nil},
	uint32(2891):  {`NAD83(HARN) / Kentucky North (ftUS)`, uint32(2891), `EPSG`, uint32(2891), `PROJCS["NAD83(HARN) / Kentucky North (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",37.5111111111111,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-84.25,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",37.9777777777778,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",38.9777777777778,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1640416.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2891"]]`, nil},
	uint32(2892):  {`NAD83(HARN) / Kentucky South (ftUS)`, uint32(2892), `EPSG`, uint32(2892), `PROJCS["NAD83(HARN) / Kentucky South (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",36.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-85.75,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",37.9333333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",36.7444444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1640416.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",1640416.667,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2892"]]`, nil},
	uint32(2893):  {`NAD83(HARN) / Maryland (ftUS)`, uint32(2893), `EPSG`, uint32(2893), `PROJCS["NAD83(HARN) / Maryland (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",37.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-77,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",39.45,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",38.3111111111111,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1312333.333,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2893"]]`, nil},
	uint32(2894):  {`NAD83(HARN) / Massachusetts Mainland (ftUS)`, uint32(2894), `EPSG`, uint32(2894), `PROJCS["NAD83(HARN) / Massachusetts Mainland (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",41,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-71.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",42.6944444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",41.7277777777778,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",656166.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",2460625,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2894"]]`, nil},
	uint32(2895):  {`NAD83(HARN) / Massachusetts Island (ftUS)`, uint32(2895), `EPSG`, uint32(2895), `PROJCS["NAD83(HARN) / Massachusetts Island (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",41,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-70.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",41.4944444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",41.2833333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1640416.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2895"]]`, nil},
	uint32(2896):  {`NAD83(HARN) / Michigan North (ft)`, uint32(2896), `EPSG`, uint32(2896), `PROJCS["NAD83(HARN) / Michigan North (ft)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",44.7944444444444,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-87,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",47.0944444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",45.4944444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",26246719.16,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2896"]]`, nil},
	uint32(2897):  {`NAD83(HARN) / Michigan Central (ft)`, uint32(2897), `EPSG`, uint32(2897), `PROJCS["NAD83(HARN) / Michigan Central (ft)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",43.3277777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-84.3777777777778,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",45.7,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",44.1944444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",19685039.37,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2897"]]`, nil},
	uint32(2898):  {`NAD83(HARN) / Michigan South (ft)`, uint32(2898), `EPSG`, uint32(2898), `PROJCS["NAD83(HARN) / Michigan South (ft)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",41.5111111111111,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-84.3777777777778,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",43.6777777777778,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",42.1,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",13123359.58,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2898"]]`, nil},
	uint32(2899):  {`NAD83(HARN) / Mississippi East (ftUS)`, uint32(2899), `EPSG`, uint32(2899), `PROJCS["NAD83(HARN) / Mississippi East (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",29.5,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-88.8333333333333,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.99995,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",984250,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2899"]]`, nil},
	uint32(2900):  {`NAD83(HARN) / Mississippi West (ftUS)`, uint32(2900), `EPSG`, uint32(2900), `PROJCS["NAD83(HARN) / Mississippi West (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",29.5,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-90.3333333333333,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.99995,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",2296583.333,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2900"]]`, nil},
	uint32(2901):  {`NAD83(HARN) / Montana (ft)`, uint32(2901), `EPSG`, uint32(2901), `PROJCS["NAD83(HARN) / Montana (ft)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",44.2611111111111,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-109.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",49,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",45,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1968503.937,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2901"]]`, nil},
	uint32(2902):  {`NAD83(HARN) / New Mexico East (ftUS)`, uint32(2902), `EPSG`, uint32(2902), `PROJCS["NAD83(HARN) / New Mexico East (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",31,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-104.333333333333,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999909091,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",541337.5,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2902"]]`, nil},
	uint32(2903):  {`NAD83(HARN) / New Mexico Central (ftUS)`, uint32(2903), `EPSG`, uint32(2903), `PROJCS["NAD83(HARN) / New Mexico Central (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",31,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-106.25,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",1640416.667,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2903"]]`, nil},
	uint32(2904):  {`NAD83(HARN) / New Mexico West (ftUS)`, uint32(2904), `EPSG`, uint32(2904), `PROJCS["NAD83(HARN) / New Mexico West (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",31,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-107.833333333333,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999916667,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",2723091.667,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2904"]]`, nil},
	uint32(2905):  {`NAD83(HARN) / New York East (ftUS)`, uint32(2905), `EPSG`, uint32(2905), `PROJCS["NAD83(HARN) / New York East (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",38.8333333333333,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-74.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",492125,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2905"]]`, nil},
	uint32(2906):  {`NAD83(HARN) / New York Central (ftUS)`, uint32(2906), `EPSG`, uint32(2906), `PROJCS["NAD83(HARN) / New York Central (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",40,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-76.5944444444444,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999375,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",820208.333,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2906"]]`, nil},
	uint32(2907):  {`NAD83(HARN) / New York West (ftUS)`, uint32(2907), `EPSG`, uint32(2907), `PROJCS["NAD83(HARN) / New York West (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",40,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-78.5944444444444,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999375,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",1148291.667,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2907"]]`, nil},
	uint32(2908):  {`NAD83(HARN) / New York Long Island (ftUS)`, uint32(2908), `EPSG`, uint32(2908), `PROJCS["NAD83(HARN) / New York Long Island (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",40.1666666666667,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-74,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",41.0333333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",40.6777777777778,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",984250,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2908"]]`, nil},
	uint32(2909):  {`NAD83(HARN) / North Dakota North (ft)`, uint32(2909), `EPSG`, uint32(2909), `PROJCS["NAD83(HARN) / North Dakota North (ft)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",47,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-100.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",48.7444444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",47.4444444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1968503.937,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2909"]]`, nil},
	uint32(2910):  {`NAD83(HARN) / North Dakota South (ft)`, uint32(2910), `EPSG`, uint32(2910), `PROJCS["NAD83(HARN) / North Dakota South (ft)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",45.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-100.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",47.4944444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",46.1944444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1968503.937,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2910"]]`, nil},
	uint32(2911):  {`NAD83(HARN) / Oklahoma North (ftUS)`, uint32(2911), `EPSG`, uint32(2911), `PROJCS["NAD83(HARN) / Oklahoma North (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",35,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-98,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",36.7666666666667,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",35.5666666666667,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1968500,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2911"]]`, nil},
	uint32(2912):  {`NAD83(HARN) / Oklahoma South (ftUS)`, uint32(2912), `EPSG`, uint32(2912), `PROJCS["NAD83(HARN) / Oklahoma South (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",33.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-98,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",35.2333333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",33.9333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1968500,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2912"]]`, nil},
	uint32(2913):  {`NAD83(HARN) / Oregon North (ft)`, uint32(2913), `EPSG`, uint32(2913), `PROJCS["NAD83(HARN) / Oregon North (ft)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",43.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-120.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",46,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",44.3333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",8202099.738,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2913"]]`, nil},
	uint32(2914):  {`NAD83(HARN) / Oregon South (ft)`, uint32(2914), `EPSG`, uint32(2914), `PROJCS["NAD83(HARN) / Oregon South (ft)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",41.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-120.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",44,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",42.3333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",4921259.843,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2914"]]`, nil},
	uint32(2915):  {`NAD83(HARN) / Tennessee (ftUS)`, uint32(2915), `EPSG`, uint32(2915), `PROJCS["NAD83(HARN) / Tennessee (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",34.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-86,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",36.4166666666667,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",35.2611111111111,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1968500,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2915"]]`, nil},
	uint32(2916):  {`NAD83(HARN) / Texas North (ftUS)`, uint32(2916), `EPSG`, uint32(2916), `PROJCS["NAD83(HARN) / Texas North (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",34,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-101.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",36.1944444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",34.65,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",656166.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",3280833.333,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2916"]]`, nil},
	uint32(2917):  {`NAD83(HARN) / Texas North Central (ftUS)`, uint32(2917), `EPSG`, uint32(2917), `PROJCS["NAD83(HARN) / Texas North Central (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",31.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-98.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",33.9777777777778,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",32.1444444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1968500,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",6561666.667,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2917"]]`, nil},
	uint32(2918):  {`NAD83(HARN) / Texas Central (ftUS)`, uint32(2918), `EPSG`, uint32(2918), `PROJCS["NAD83(HARN) / Texas Central (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",29.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-100.333333333333,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",31.8833333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",30.1166666666667,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",2296583.333,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",9842500,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2918"]]`, nil},
	uint32(2919):  {`NAD83(HARN) / Texas South Central (ftUS)`, uint32(2919), `EPSG`, uint32(2919), `PROJCS["NAD83(HARN) / Texas South Central (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",27.8333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-99,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",30.2833333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",28.3833333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1968500,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",13123333.333,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2919"]]`, nil},
	uint32(2920):  {`NAD83(HARN) / Texas South (ftUS)`, uint32(2920), `EPSG`, uint32(2920), `PROJCS["NAD83(HARN) / Texas South (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",25.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-98.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",27.8333333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",26.1666666666667,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",984250,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",16404166.667,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2920"]]`, nil},
	uint32(2921):  {`NAD83(HARN) / Utah North (ft)`, uint32(2921), `EPSG`, uint32(2921), `PROJCS["NAD83(HARN) / Utah North (ft)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",40.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-111.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",41.7944444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",40.7277777777778,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1640419.948,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",3280839.895,AUTHORITY["EPSG","8827"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2921"]]`, nil},
	uint32(2922):  {`NAD83(HARN) / Utah Central (ft)`, uint32(2922), `EPSG`, uint32(2922), `PROJCS["NAD83(HARN) / Utah Central (ft)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",38.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-111.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",40.65,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",39.0277777777778,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1640419.948,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",6561679.79,AUTHORITY["EPSG","8827"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2922"]]`, nil},
	uint32(2923):  {`NAD83(HARN) / Utah South (ft)`, uint32(2923), `EPSG`, uint32(2923), `PROJCS["NAD83(HARN) / Utah South (ft)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",36.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-111.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",38.35,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",37.2166666666667,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1640419.948,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",9842519.685,AUTHORITY["EPSG","8827"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2923"]]`, nil},
	uint32(2924):  {`NAD83(HARN) / Virginia North (ftUS)`, uint32(2924), `EPSG`, uint32(2924), `PROJCS["NAD83(HARN) / Virginia North (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",37.6777777777778,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-78.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",39.2111111111111,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",38.0333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",11482916.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",6561666.667,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2924"]]`, nil},
	uint32(2925):  {`NAD83(HARN) / Virginia South (ftUS)`, uint32(2925), `EPSG`, uint32(2925), `PROJCS["NAD83(HARN) / Virginia South (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",36.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-78.5111111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",37.9777777777778,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",36.7666666666667,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",11482916.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",3280833.333,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2925"]]`, nil},
	uint32(2926):  {`NAD83(HARN) / Washington North (ftUS)`, uint32(2926), `EPSG`, uint32(2926), `PROJCS["NAD83(HARN) / Washington North (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",47,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-120.833333333333,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",48.7444444444444,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",47.5111111111111,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1640416.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2926"]]`, nil},
	uint32(2927):  {`NAD83(HARN) / Washington South (ftUS)`, uint32(2927), `EPSG`, uint32(2927), `PROJCS["NAD83(HARN) / Washington South (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",45.3333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-120.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",47.3333333333333,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",45.8333333333333,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1640416.667,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2927"]]`, nil},
	uint32(2928):  {`NAD83(HARN) / Wisconsin North (ftUS)`, uint32(2928), `EPSG`, uint32(2928), `PROJCS["NAD83(HARN) / Wisconsin North (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",45.1666666666667,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-90,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",46.7666666666667,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",45.5666666666667,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1968500,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2928"]]`, nil},
	uint32(2929):  {`NAD83(HARN) / Wisconsin Central (ftUS)`, uint32(2929), `EPSG`, uint32(2929), `PROJCS["NAD83(HARN) / Wisconsin Central (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",43.8333333333333,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-90,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",45.5111111111111,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",44.2611111111111,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1968500,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2929"]]`, nil},
	uint32(2930):  {`NAD83(HARN) / Wisconsin South (ftUS)`, uint32(2930), `EPSG`, uint32(2930), `PROJCS["NAD83(HARN) / Wisconsin South (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",42,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-90,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",44.0777777777778,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",42.7444444444444,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1968500,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2930"]]`, nil},
	uint32(2931):  {`Beduaram / TM 13 NE`, uint32(2931), `EPSG`, uint32(2931), `PROJCS["Beduaram / TM 13 NE",GEOGCS["Beduaram",DATUM["Beduaram",SPHEROID["Clarke 1880 (IGN)",6378249.2,293.4660212936269,AUTHORITY["EPSG","7011"]],TOWGS84[-106,-87,188,0,0,0,0],AUTHORITY["EPSG","6213"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4213"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",13,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2931"]]`, nil},
	uint32(2932):  {`QND95 / Qatar National Grid`, uint32(2932), `EPSG`, uint32(2932), `PROJCS["QND95 / Qatar National Grid",GEOGCS["QND95",DATUM["Qatar National Datum 1995",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-119.4248,-303.65872,-11.00061,1.164298,0.174458,1.096259,3.657065],AUTHORITY["EPSG","6614"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4614"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",24.4611111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",51.2166666666667,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.99999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",200000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",300000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2932"]]`, nil},
	uint32(2933):  {`Segara / UTM zone 50S`, uint32(2933), `EPSG`, uint32(2933), `PROJCS["Segara / UTM zone 50S",GEOGCS["Segara",DATUM["Gunung Segara",SPHEROID["Bessel 1841",6377397.155,299.1528128,AUTHORITY["EPSG","7004"]],TOWGS84[-403,684,41,0,0,0,0],AUTHORITY["EPSG","6613"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4613"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",117,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2933"]]`, nil},
	uint32(2935):  {`Pulkovo 1942 / CS63 zone A1`, uint32(2935), `EPSG`, uint32(2935), `PROJCS["Pulkovo 1942 / CS63 zone A1",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0.116666666666667,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",41.5333333333333,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",1300000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2935"]]`, nil},
	uint32(2936):  {`Pulkovo 1942 / CS63 zone A2`, uint32(2936), `EPSG`, uint32(2936), `PROJCS["Pulkovo 1942 / CS63 zone A2",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0.116666666666667,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",44.5333333333333,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",2300000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2936"]]`, nil},
	uint32(2937):  {`Pulkovo 1942 / CS63 zone A3`, uint32(2937), `EPSG`, uint32(2937), `PROJCS["Pulkovo 1942 / CS63 zone A3",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0.116666666666667,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",47.5333333333333,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",3300000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2937"]]`, nil},
	uint32(2938):  {`Pulkovo 1942 / CS63 zone A4`, uint32(2938), `EPSG`, uint32(2938), `PROJCS["Pulkovo 1942 / CS63 zone A4",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0.116666666666667,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",50.5333333333333,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",4300000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2938"]]`, nil},
	uint32(2939):  {`Pulkovo 1942 / CS63 zone K2`, uint32(2939), `EPSG`, uint32(2939), `PROJCS["Pulkovo 1942 / CS63 zone K2",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0.133333333333333,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",50.7666666666667,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",2300000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2939"]]`, nil},
	uint32(2940):  {`Pulkovo 1942 / CS63 zone K3`, uint32(2940), `EPSG`, uint32(2940), `PROJCS["Pulkovo 1942 / CS63 zone K3",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0.133333333333333,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",53.7666666666667,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",3300000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2940"]]`, nil},
	uint32(2941):  {`Pulkovo 1942 / CS63 zone K4`, uint32(2941), `EPSG`, uint32(2941), `PROJCS["Pulkovo 1942 / CS63 zone K4",GEOGCS["Pulkovo 1942",DATUM["Pulkovo 1942",SPHEROID["Krassowsky 1940",6378245,298.3,AUTHORITY["EPSG","7024"]],TOWGS84[25,-141,-78.5,0,0.35,0.736,0],AUTHORITY["EPSG","6284"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4284"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0.133333333333333,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",56.7666666666667,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",4300000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","2941"]]`, nil},
	uint32(2942):  {`Porto Santo / UTM zone 28N`, uint32(2942), `EPSG`, uint32(2942), `PROJCS["Porto Santo / UTM zone 28N",GEOGCS["Porto Santo",DATUM["Porto Santo 1936",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-499,-249,314,0,0,0,0],AUTHORITY["EPSG","6615"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4615"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-15,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2942"]]`, nil},
	uint32(2943):  {`Selvagem Grande / UTM zone 28N`, uint32(2943), `EPSG`, uint32(2943), `PROJCS["Selvagem Grande / UTM zone 28N",GEOGCS["Selvagem Grande",DATUM["Selvagem Grande",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-289,-124,60,0,0,0,0],AUTHORITY["EPSG","6616"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4616"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-15,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2943"]]`, nil},
	uint32(2945):  {`NAD83(CSRS) / MTM zone 3`, uint32(2945), `EPSG`, uint32(2945), `PROJCS["NAD83(CSRS) / MTM zone 3",GEOGCS["NAD83(CSRS)",DATUM["NAD83 Canadian Spatial Reference System",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[-0.991,1.9072,0.5129,0.0257899075194932,0.00965009896027041,0.0116599432323421,0],AUTHORITY["EPSG","6140"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4617"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-58.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",304800,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E(X)",EAST],AXIS["N(Y)",NORTH],AUTHORITY["EPSG","2945"]]`, nil},
	uint32(2946):  {`NAD83(CSRS) / MTM zone 4`, uint32(2946), `EPSG`, uint32(2946), `PROJCS["NAD83(CSRS) / MTM zone 4",GEOGCS["NAD83(CSRS)",DATUM["NAD83 Canadian Spatial Reference System",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[-0.991,1.9072,0.5129,0.0257899075194932,0.00965009896027041,0.0116599432323421,0],AUTHORITY["EPSG","6140"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4617"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-61.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",304800,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E(X)",EAST],AXIS["N(Y)",NORTH],AUTHORITY["EPSG","2946"]]`, nil},
	uint32(2947):  {`NAD83(CSRS) / MTM zone 5`, uint32(2947), `EPSG`, uint32(2947), `PROJCS["NAD83(CSRS) / MTM zone 5",GEOGCS["NAD83(CSRS)",DATUM["NAD83 Canadian Spatial Reference System",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[-0.991,1.9072,0.5129,0.0257899075194932,0.00965009896027041,0.0116599432323421,0],AUTHORITY["EPSG","6140"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4617"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-64.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",304800,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E(X)",EAST],AXIS["N(Y)",NORTH],AUTHORITY["EPSG","2947"]]`, nil},
	uint32(2948):  {`NAD83(CSRS) / MTM zone 6`, uint32(2948), `EPSG`, uint32(2948), `PROJCS["NAD83(CSRS) / MTM zone 6",GEOGCS["NAD83(CSRS)",DATUM["NAD83 Canadian Spatial Reference System",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[-0.991,1.9072,0.5129,0.0257899075194932,0.00965009896027041,0.0116599432323421,0],AUTHORITY["EPSG","6140"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4617"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-67.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",304800,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E(X)",EAST],AXIS["N(Y)",NORTH],AUTHORITY["EPSG","2948"]]`, nil},
	uint32(2949):  {`NAD83(CSRS) / MTM zone 7`, uint32(2949), `EPSG`, uint32(2949), `PROJCS["NAD83(CSRS) / MTM zone 7",GEOGCS["NAD83(CSRS)",DATUM["NAD83 Canadian Spatial Reference System",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[-0.991,1.9072,0.5129,0.0257899075194932,0.00965009896027041,0.0116599432323421,0],AUTHORITY["EPSG","6140"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4617"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-70.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",304800,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E(X)",EAST],AXIS["N(Y)",NORTH],AUTHORITY["EPSG","2949"]]`, nil},
	uint32(2950):  {`NAD83(CSRS) / MTM zone 8`, uint32(2950), `EPSG`, uint32(2950), `PROJCS["NAD83(CSRS) / MTM zone 8",GEOGCS["NAD83(CSRS)",DATUM["NAD83 Canadian Spatial Reference System",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[-0.991,1.9072,0.5129,0.0257899075194932,0.00965009896027041,0.0116599432323421,0],AUTHORITY["EPSG","6140"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4617"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-73.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",304800,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E(X)",EAST],AXIS["N(Y)",NORTH],AUTHORITY["EPSG","2950"]]`, nil},
	uint32(2951):  {`NAD83(CSRS) / MTM zone 9`, uint32(2951), `EPSG`, uint32(2951), `PROJCS["NAD83(CSRS) / MTM zone 9",GEOGCS["NAD83(CSRS)",DATUM["NAD83 Canadian Spatial Reference System",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[-0.991,1.9072,0.5129,0.0257899075194932,0.00965009896027041,0.0116599432323421,0],AUTHORITY["EPSG","6140"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4617"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-76.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",304800,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E(X)",EAST],AXIS["N(Y)",NORTH],AUTHORITY["EPSG","2951"]]`, nil},
	uint32(2952):  {`NAD83(CSRS) / MTM zone 10`, uint32(2952), `EPSG`, uint32(2952), `PROJCS["NAD83(CSRS) / MTM zone 10",GEOGCS["NAD83(CSRS)",DATUM["NAD83 Canadian Spatial Reference System",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[-0.991,1.9072,0.5129,0.0257899075194932,0.00965009896027041,0.0116599432323421,0],AUTHORITY["EPSG","6140"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4617"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-79.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9999,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",304800,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E(X)",EAST],AXIS["N(Y)",NORTH],AUTHORITY["EPSG","2952"]]`, nil},
	uint32(2953):  {`NAD83(CSRS) / New Brunswick Stereographic`, uint32(2953), `EPSG`, uint32(2953), `PROJCS["NAD83(CSRS) / New Brunswick Stereographic",GEOGCS["NAD83(CSRS)",DATUM["NAD83 Canadian Spatial Reference System",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[-0.991,1.9072,0.5129,0.0257899075194932,0.00965009896027041,0.0116599432323421,0],AUTHORITY["EPSG","6140"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4617"]],PROJECTION["Oblique Stereographic",AUTHORITY["EPSG","9809"]],PARAMETER["Latitude of natural origin",46.5111111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-66.5111111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999912,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",2500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",7500000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","2953"]]`, nil},
	uint32(2954):  {`NAD83(CSRS) / Prince Edward Isl. Stereographic (NAD83)`, uint32(2954), `EPSG`, uint32(2954), `PROJCS["NAD83(CSRS) / Prince Edward Isl. Stereographic (NAD83)",GEOGCS["NAD83(CSRS)",DATUM["NAD83 Canadian Spatial Reference System",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[-0.991,1.9072,0.5129,0.0257899075194932,0.00965009896027041,0.0116599432323421,0],AUTHORITY["EPSG","6140"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4617"]],PROJECTION["Oblique Stereographic",AUTHORITY["EPSG","9809"]],PARAMETER["Latitude of natural origin",47.2611111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-63,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999912,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",400000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E(X)",EAST],AXIS["N(Y)",NORTH],AUTHORITY["EPSG","2954"]]`, nil},
	uint32(2955):  {`NAD83(CSRS) / UTM zone 11N`, uint32(2955), `EPSG`, uint32(2955), `PROJCS["NAD83(CSRS) / UTM zone 11N",GEOGCS["NAD83(CSRS)",DATUM["NAD83 Canadian Spatial Reference System",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[-0.991,1.9072,0.5129,0.0257899075194932,0.00965009896027041,0.0116599432323421,0],AUTHORITY["EPSG","6140"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4617"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-117,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2955"]]`, nil},
	uint32(2956):  {`NAD83(CSRS) / UTM zone 12N`, uint32(2956), `EPSG`, uint32(2956), `PROJCS["NAD83(CSRS) / UTM zone 12N",GEOGCS["NAD83(CSRS)",DATUM["NAD83 Canadian Spatial Reference System",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[-0.991,1.9072,0.5129,0.0257899075194932,0.00965009896027041,0.0116599432323421,0],AUTHORITY["EPSG","6140"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4617"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2956"]]`, nil},
	uint32(2957):  {`NAD83(CSRS) / UTM zone 13N`, uint32(2957), `EPSG`, uint32(2957), `PROJCS["NAD83(CSRS) / UTM zone 13N",GEOGCS["NAD83(CSRS)",DATUM["NAD83 Canadian Spatial Reference System",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[-0.991,1.9072,0.5129,0.0257899075194932,0.00965009896027041,0.0116599432323421,0],AUTHORITY["EPSG","6140"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4617"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-105,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2957"]]`, nil},
	uint32(2958):  {`NAD83(CSRS) / UTM zone 17N`, uint32(2958), `EPSG`, uint32(2958), `PROJCS["NAD83(CSRS) / UTM zone 17N",GEOGCS["NAD83(CSRS)",DATUM["NAD83 Canadian Spatial Reference System",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[-0.991,1.9072,0.5129,0.0257899075194932,0.00965009896027041,0.0116599432323421,0],AUTHORITY["EPSG","6140"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4617"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-81,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2958"]]`, nil},
	uint32(2959):  {`NAD83(CSRS) / UTM zone 18N`, uint32(2959), `EPSG`, uint32(2959), `PROJCS["NAD83(CSRS) / UTM zone 18N",GEOGCS["NAD83(CSRS)",DATUM["NAD83 Canadian Spatial Reference System",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[-0.991,1.9072,0.5129,0.0257899075194932,0.00965009896027041,0.0116599432323421,0],AUTHORITY["EPSG","6140"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4617"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-75,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2959"]]`, nil},
	uint32(2960):  {`NAD83(CSRS) / UTM zone 19N`, uint32(2960), `EPSG`, uint32(2960), `PROJCS["NAD83(CSRS) / UTM zone 19N",GEOGCS["NAD83(CSRS)",DATUM["NAD83 Canadian Spatial Reference System",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[-0.991,1.9072,0.5129,0.0257899075194932,0.00965009896027041,0.0116599432323421,0],AUTHORITY["EPSG","6140"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4617"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-69,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2960"]]`, nil},
	uint32(2961):  {`NAD83(CSRS) / UTM zone 20N`, uint32(2961), `EPSG`, uint32(2961), `PROJCS["NAD83(CSRS) / UTM zone 20N",GEOGCS["NAD83(CSRS)",DATUM["NAD83 Canadian Spatial Reference System",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[-0.991,1.9072,0.5129,0.0257899075194932,0.00965009896027041,0.0116599432323421,0],AUTHORITY["EPSG","6140"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4617"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-63,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2961"]]`, nil},
	uint32(2962):  {`NAD83(CSRS) / UTM zone 21N`, uint32(2962), `EPSG`, uint32(2962), `PROJCS["NAD83(CSRS) / UTM zone 21N",GEOGCS["NAD83(CSRS)",DATUM["NAD83 Canadian Spatial Reference System",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[-0.991,1.9072,0.5129,0.0257899075194932,0.00965009896027041,0.0116599432323421,0],AUTHORITY["EPSG","6140"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4617"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-57,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2962"]]`, nil},
	uint32(2963):  {`Lisbon 1890 (Lisbon) / Portugal Bonne`, uint32(2963), `EPSG`, uint32(2963), `PROJCS["Lisbon 1890 (Lisbon) / Portugal Bonne",GEOGCS["Lisbon 1890 (Lisbon)",DATUM["Lisbon 1890 (Lisbon)",SPHEROID["Bessel 1841",6377397.155,299.1528128,AUTHORITY["EPSG","7004"]],AUTHORITY["EPSG","6904"]],PRIMEM["Lisbon",-9.131906111111112,AUTHORITY["EPSG","8902"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4904"]],PROJECTION["Bonne (South Orientated)",AUTHORITY["EPSG","9828"]],PARAMETER["Latitude of natural origin",39.6777777777778,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",1,AUTHORITY["EPSG","8802"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["P",SOUTH],AXIS["M",WEST],AUTHORITY["EPSG","2963"]]`, nil},
	uint32(2964):  {`NAD27 / Alaska Albers`, uint32(2964), `EPSG`, uint32(2964), `PROJCS["NAD27 / Alaska Albers",GEOGCS["NAD27",DATUM["North American Datum 1927",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],TOWGS84[-32.3841359,180.4090461,120.8442577,-2.1545854,-0.1498782,0.5742915,8.1049164],AUTHORITY["EPSG","6267"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4267"]],PROJECTION["Albers Equal Area",AUTHORITY["EPSG","9822"]],PARAMETER["Latitude of false origin",50,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-154,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",55,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",65,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",0,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2964"]]`, nil},
	uint32(2965):  {`NAD83 / Indiana East (ftUS)`, uint32(2965), `EPSG`, uint32(2965), `PROJCS["NAD83 / Indiana East (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",37.5111111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-85.6666666666667,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999966667,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",328083.333,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",820208.333,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2965"]]`, nil},
	uint32(2966):  {`NAD83 / Indiana West (ftUS)`, uint32(2966), `EPSG`, uint32(2966), `PROJCS["NAD83 / Indiana West (ftUS)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",37.5111111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-87.0944444444444,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999966667,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",2952750,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",820208.333,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2966"]]`, nil},
	uint32(2967):  {`NAD83(HARN) / Indiana East (ftUS)`, uint32(2967), `EPSG`, uint32(2967), `PROJCS["NAD83(HARN) / Indiana East (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",37.5111111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-85.6666666666667,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999966667,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",328083.333,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",820208.333,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2967"]]`, nil},
	uint32(2968):  {`NAD83(HARN) / Indiana West (ftUS)`, uint32(2968), `EPSG`, uint32(2968), `PROJCS["NAD83(HARN) / Indiana West (ftUS)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",37.5111111111111,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-87.0944444444444,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.999966667,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",2952750,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",820208.333,AUTHORITY["EPSG","8807"]],UNIT["US survey foot",0.30480060960121924,AUTHORITY["EPSG","9003"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2968"]]`, nil},
	uint32(2969):  {`Fort Marigot / UTM zone 20N`, uint32(2969), `EPSG`, uint32(2969), `PROJCS["Fort Marigot / UTM zone 20N",GEOGCS["Fort Marigot",DATUM["Fort Marigot",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[137,248,-430,0,0,0,0],AUTHORITY["EPSG","6621"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4621"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-63,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2969"]]`, nil},
	uint32(2970):  {`Guadeloupe 1948 / UTM zone 20N`, uint32(2970), `EPSG`, uint32(2970), `PROJCS["Guadeloupe 1948 / UTM zone 20N",GEOGCS["Guadeloupe 1948",DATUM["Guadeloupe 1948",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-472.29,-5.63,-304.12,0.4362,-0.8374,0.2563,1.8984],AUTHORITY["EPSG","6622"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4622"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-63,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2970"]]`, nil},
	uint32(2971):  {`CSG67 / UTM zone 22N`, uint32(2971), `EPSG`, uint32(2971), `PROJCS["CSG67 / UTM zone 22N",GEOGCS["CSG67",DATUM["Centre Spatial Guyanais 1967",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-186,230,110,0,0,0,0],AUTHORITY["EPSG","6623"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4623"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-51,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2971"]]`, nil},
	uint32(2972):  {`RGFG95 / UTM zone 22N`, uint32(2972), `EPSG`, uint32(2972), `PROJCS["RGFG95 / UTM zone 22N",GEOGCS["RGFG95",DATUM["Reseau Geodesique Francais Guyane 1995",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6624"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4624"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-51,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2972"]]`, nil},
	uint32(2973):  {`Martinique 1938 / UTM zone 20N`, uint32(2973), `EPSG`, uint32(2973), `PROJCS["Martinique 1938 / UTM zone 20N",GEOGCS["Martinique 1938",DATUM["Martinique 1938",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[126.93,547.94,130.41,-2.7867,5.1612,-0.8584,13.8227],AUTHORITY["EPSG","6625"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4625"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-63,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2973"]]`, nil},
	uint32(2975):  {`RGR92 / UTM zone 40S`, uint32(2975), `EPSG`, uint32(2975), `PROJCS["RGR92 / UTM zone 40S",GEOGCS["RGR92",DATUM["Reseau Geodesique de la Reunion 1992",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6627"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4627"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",57,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2975"]]`, nil},
	uint32(2976):  {`Tahiti 52 / UTM zone 6S`, uint32(2976), `EPSG`, uint32(2976), `PROJCS["Tahiti 52 / UTM zone 6S",GEOGCS["Tahiti 52",DATUM["Tahiti 52",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[162,117,154,0,0,0,0],AUTHORITY["EPSG","6628"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4628"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-147,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2976"]]`, nil},
	uint32(2977):  {`Tahaa 54 / UTM zone 5S`, uint32(2977), `EPSG`, uint32(2977), `PROJCS["Tahaa 54 / UTM zone 5S",GEOGCS["Tahaa 54",DATUM["Tahaa 54",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[72.438,345.918,79.486,1.6045,0.8823,0.5565,1.3746],AUTHORITY["EPSG","6629"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4629"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-153,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2977"]]`, nil},
	uint32(2978):  {`IGN72 Nuku Hiva / UTM zone 7S`, uint32(2978), `EPSG`, uint32(2978), `PROJCS["IGN72 Nuku Hiva / UTM zone 7S",GEOGCS["IGN72 Nuku Hiva",DATUM["IGN72 Nuku Hiva",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[84,274,65,0,0,0,0],AUTHORITY["EPSG","6630"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4630"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-141,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2978"]]`, nil},
	uint32(2980):  {`Combani 1950 / UTM zone 38S`, uint32(2980), `EPSG`, uint32(2980), `PROJCS["Combani 1950 / UTM zone 38S",GEOGCS["Combani 1950",DATUM["Combani 1950",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-382,-59,-262,0,0,0,0],AUTHORITY["EPSG","6632"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4632"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",45,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2980"]]`, nil},
	uint32(2981):  {`IGN56 Lifou / UTM zone 58S`, uint32(2981), `EPSG`, uint32(2981), `PROJCS["IGN56 Lifou / UTM zone 58S",GEOGCS["IGN56 Lifou",DATUM["IGN56 Lifou",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[137.092,131.66,91.475,-1.9436,-11.5993,-4.3321,-7.4824],AUTHORITY["EPSG","6633"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4633"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",165,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2981"]]`, nil},
	uint32(2985):  {`Petrels 1972 / Terre Adelie Polar Stereographic`, uint32(2985), `EPSG`, uint32(2985), `PROJCS["Petrels 1972 / Terre Adelie Polar Stereographic",GEOGCS["Petrels 1972",DATUM["Petrels 1972",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[365,194,166,0,0,0,0],AUTHORITY["EPSG","6636"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4636"]],PROJECTION["Polar Stereographic (variant C)",AUTHORITY["EPSG","9830"]],PARAMETER["Latitude of standard parallel",-67,AUTHORITY["EPSG","8832"]],PARAMETER["Longitude of origin",140,AUTHORITY["EPSG","8833"]],PARAMETER["Easting at false origin",300000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",200000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",NORTH],AUTHORITY["EPSG","2985"]]`, nil},
	uint32(2986):  {`Perroud 1950 / Terre Adelie Polar Stereographic`, uint32(2986), `EPSG`, uint32(2986), `PROJCS["Perroud 1950 / Terre Adelie Polar Stereographic",GEOGCS["Perroud 1950",DATUM["Pointe Geologie Perroud 1950",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[325,154,172,0,0,0,0],AUTHORITY["EPSG","6637"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4637"]],PROJECTION["Polar Stereographic (variant C)",AUTHORITY["EPSG","9830"]],PARAMETER["Latitude of standard parallel",-67,AUTHORITY["EPSG","8832"]],PARAMETER["Longitude of origin",140,AUTHORITY["EPSG","8833"]],PARAMETER["Easting at false origin",300000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",200000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",NORTH],AUTHORITY["EPSG","2986"]]`, nil},
	uint32(2987):  {`Saint Pierre et Miquelon 1950 / UTM zone 21N`, uint32(2987), `EPSG`, uint32(2987), `PROJCS["Saint Pierre et Miquelon 1950 / UTM zone 21N",GEOGCS["Saint Pierre et Miquelon 1950",DATUM["Saint Pierre et Miquelon 1950",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],TOWGS84[11.363,424.148,373.13,0,0,0,0],AUTHORITY["EPSG","6638"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4638"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-57,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2987"]]`, nil},
	uint32(2988):  {`MOP78 / UTM zone 1S`, uint32(2988), `EPSG`, uint32(2988), `PROJCS["MOP78 / UTM zone 1S",GEOGCS["MOP78",DATUM["MOP78",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[253,-132,-127,0,0,0,0],AUTHORITY["EPSG","6639"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4639"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-177,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2988"]]`, nil},
	uint32(2991):  {`NAD83 / Oregon LCC (m)`, uint32(2991), `EPSG`, uint32(2991), `PROJCS["NAD83 / Oregon LCC (m)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",41.75,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-120.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",43,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",45.5111111111111,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",400000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2991"]]`, nil},
	uint32(2992):  {`NAD83 / Oregon GIC Lambert (ft)`, uint32(2992), `EPSG`, uint32(2992), `PROJCS["NAD83 / Oregon GIC Lambert (ft)",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",41.75,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-120.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",43,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",45.5111111111111,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1312335.958,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2992"]]`, nil},
	uint32(2993):  {`NAD83(HARN) / Oregon LCC (m)`, uint32(2993), `EPSG`, uint32(2993), `PROJCS["NAD83(HARN) / Oregon LCC (m)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",41.75,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-120.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",43,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",45.5111111111111,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",400000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2993"]]`, nil},
	uint32(2994):  {`NAD83(HARN) / Oregon GIC Lambert (ft)`, uint32(2994), `EPSG`, uint32(2994), `PROJCS["NAD83(HARN) / Oregon GIC Lambert (ft)",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",41.75,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-120.511111111111,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",43,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",45.5111111111111,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1312335.958,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["foot",0.3048,AUTHORITY["EPSG","9002"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","2994"]]`, nil},
	uint32(2995):  {`IGN53 Mare / UTM zone 58S`, uint32(2995), `EPSG`, uint32(2995), `PROJCS["IGN53 Mare / UTM zone 58S",GEOGCS["IGN53 Mare",DATUM["IGN53 Mare",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-408.809,366.856,-412.987,1.8842,-0.5308,2.1655,-121.0993],AUTHORITY["EPSG","6641"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4641"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",165,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2995"]]`, nil},
	uint32(2996):  {`ST84 Ile des Pins / UTM zone 58S`, uint32(2996), `EPSG`, uint32(2996), `PROJCS["ST84 Ile des Pins / UTM zone 58S",GEOGCS["ST84 Ile des Pins",DATUM["ST84 Ile des Pins",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-13,-348,292,0,0,0,0],AUTHORITY["EPSG","6642"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4642"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",165,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2996"]]`, nil},
	uint32(2997):  {`ST71 Belep / UTM zone 58S`, uint32(2997), `EPSG`, uint32(2997), `PROJCS["ST71 Belep / UTM zone 58S",GEOGCS["ST71 Belep",DATUM["ST71 Belep",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-480.26,-438.32,-643.429,16.3119,20.1721,-4.0349,-111.7002],AUTHORITY["EPSG","6643"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4643"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",165,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2997"]]`, nil},
	uint32(2998):  {`NEA74 Noumea / UTM zone 58S`, uint32(2998), `EPSG`, uint32(2998), `PROJCS["NEA74 Noumea / UTM zone 58S",GEOGCS["NEA74 Noumea",DATUM["NEA74 Noumea",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-10.18,-350.43,291.37,0,0,0,0],AUTHORITY["EPSG","6644"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4644"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",165,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2998"]]`, nil},
	uint32(2999):  {`Grand Comoros / UTM zone 38S`, uint32(2999), `EPSG`, uint32(2999), `PROJCS["Grand Comoros / UTM zone 38S",GEOGCS["Grand Comoros",DATUM["Grand Comoros",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-963,510,-359,0,0,0,0],AUTHORITY["EPSG","6646"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4646"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",45,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","2999"]]`, nil},
	uint32(3000):  {`Segara / NEIEZ`, uint32(3000), `EPSG`, uint32(3000), `PROJCS["Segara / NEIEZ",GEOGCS["Segara",DATUM["Gunung Segara",SPHEROID["Bessel 1841",6377397.155,299.1528128,AUTHORITY["EPSG","7004"]],TOWGS84[-403,684,41,0,0,0,0],AUTHORITY["EPSG","6613"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4613"]],PROJECTION["Mercator (variant A)",AUTHORITY["EPSG","9804"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",110,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.997,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",3900000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",900000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","3000"]]`, nil},
	uint32(3001):  {`Batavia / NEIEZ`, uint32(3001), `EPSG`, uint32(3001), `PROJCS["Batavia / NEIEZ",GEOGCS["Batavia",DATUM["Batavia",SPHEROID["Bessel 1841",6377397.155,299.1528128,AUTHORITY["EPSG","7004"]],TOWGS84[-377.7,675.1,-52.2,0,0,0,0],AUTHORITY["EPSG","6211"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4211"]],PROJECTION["Mercator (variant A)",AUTHORITY["EPSG","9804"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",110,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.997,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",3900000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",900000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","3001"]]`, nil},
	uint32(3002):  {`Makassar / NEIEZ`, uint32(3002), `EPSG`, uint32(3002), `PROJCS["Makassar / NEIEZ",GEOGCS["Makassar",DATUM["Makassar",SPHEROID["Bessel 1841",6377397.155,299.1528128,AUTHORITY["EPSG","7004"]],TOWGS84[-587.8,519.75,145.76,0,0,0,0],AUTHORITY["EPSG","6257"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4257"]],PROJECTION["Mercator (variant A)",AUTHORITY["EPSG","9804"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",110,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.997,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",3900000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",900000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","3002"]]`, nil},
	uint32(3003):  {`Monte Mario / Italy zone 1`, uint32(3003), `EPSG`, uint32(3003), `PROJCS["Monte Mario / Italy zone 1",GEOGCS["Monte Mario",DATUM["Monte Mario",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-104.1,-49.1,-9.9,0.971,-2.917,0.714,-11.68],AUTHORITY["EPSG","6265"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4265"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",9,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",1500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","3003"]]`, nil},
	uint32(3004):  {`Monte Mario / Italy zone 2`, uint32(3004), `EPSG`, uint32(3004), `PROJCS["Monte Mario / Italy zone 2",GEOGCS["Monte Mario",DATUM["Monte Mario",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-104.1,-49.1,-9.9,0.971,-2.917,0.714,-11.68],AUTHORITY["EPSG","6265"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4265"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",15,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",2520000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","3004"]]`, nil},
	uint32(3005):  {`NAD83 / BC Albers`, uint32(3005), `EPSG`, uint32(3005), `PROJCS["NAD83 / BC Albers",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Albers Equal Area",AUTHORITY["EPSG","9822"]],PARAMETER["Latitude of false origin",45,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-126,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",50,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",58.5111111111111,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",1000000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",0,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","3005"]]`, nil},
	uint32(3006):  {`SWEREF99 TM`, uint32(3006), `EPSG`, uint32(3006), `PROJCS["SWEREF99 TM",GEOGCS["SWEREF99",DATUM["SWEREF99",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6619"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4619"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",15,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","3006"]]`, nil},
	uint32(3007):  {`SWEREF99 12 00`, uint32(3007), `EPSG`, uint32(3007), `PROJCS["SWEREF99 12 00",GEOGCS["SWEREF99",DATUM["SWEREF99",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6619"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4619"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",12,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",150000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","3007"]]`, nil},
	uint32(3008):  {`SWEREF99 13 30`, uint32(3008), `EPSG`, uint32(3008), `PROJCS["SWEREF99 13 30",GEOGCS["SWEREF99",DATUM["SWEREF99",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6619"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4619"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",13.5,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",150000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","3008"]]`, nil},
	uint32(3009):  {`SWEREF99 15 00`, uint32(3009), `EPSG`, uint32(3009), `PROJCS["SWEREF99 15 00",GEOGCS["SWEREF99",DATUM["SWEREF99",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6619"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4619"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",15,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",150000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","3009"]]`, nil},
	uint32(3010):  {`SWEREF99 16 30`, uint32(3010), `EPSG`, uint32(3010), `PROJCS["SWEREF99 16 30",GEOGCS["SWEREF99",DATUM["SWEREF99",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6619"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4619"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",16.5,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",150000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","3010"]]`, nil},
	uint32(3011):  {`SWEREF99 18 00`, uint32(3011), `EPSG`, uint32(3011), `PROJCS["SWEREF99 18 00",GEOGCS["SWEREF99",DATUM["SWEREF99",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6619"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4619"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",18,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",150000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","3011"]]`, nil},
	uint32(3012):  {`SWEREF99 14 15`, uint32(3012), `EPSG`, uint32(3012), `PROJCS["SWEREF99 14 15",GEOGCS["SWEREF99",DATUM["SWEREF99",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6619"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4619"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",14.25,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",150000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","3012"]]`, nil},
	uint32(3013):  {`SWEREF99 15 45`, uint32(3013), `EPSG`, uint32(3013), `PROJCS["SWEREF99 15 45",GEOGCS["SWEREF99",DATUM["SWEREF99",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6619"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4619"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",15.7611111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",150000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","3013"]]`, nil},
	uint32(3014):  {`SWEREF99 17 15`, uint32(3014), `EPSG`, uint32(3014), `PROJCS["SWEREF99 17 15",GEOGCS["SWEREF99",DATUM["SWEREF99",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6619"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4619"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",17.2611111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",150000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","3014"]]`, nil},
	uint32(3015):  {`SWEREF99 18 45`, uint32(3015), `EPSG`, uint32(3015), `PROJCS["SWEREF99 18 45",GEOGCS["SWEREF99",DATUM["SWEREF99",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6619"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4619"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",18.7611111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",150000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","3015"]]`, nil},
	uint32(3016):  {`SWEREF99 20 15`, uint32(3016), `EPSG`, uint32(3016), `PROJCS["SWEREF99 20 15",GEOGCS["SWEREF99",DATUM["SWEREF99",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6619"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4619"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",20.2611111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",150000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","3016"]]`, nil},
	uint32(3017):  {`SWEREF99 21 45`, uint32(3017), `EPSG`, uint32(3017), `PROJCS["SWEREF99 21 45",GEOGCS["SWEREF99",DATUM["SWEREF99",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6619"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4619"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",21.7611111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",150000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","3017"]]`, nil},
	uint32(3018):  {`SWEREF99 23 15`, uint32(3018), `EPSG`, uint32(3018), `PROJCS["SWEREF99 23 15",GEOGCS["SWEREF99",DATUM["SWEREF99",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6619"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4619"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",23.2611111111111,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",150000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","3018"]]`, nil},
	uint32(3019):  {`RT90 7.5 gon V`, uint32(3019), `EPSG`, uint32(3019), `PROJCS["RT90 7.5 gon V",GEOGCS["RT90",DATUM["Rikets koordinatsystem 1990",SPHEROID["Bessel 1841",6377397.155,299.1528128,AUTHORITY["EPSG","7004"]],TOWGS84[414.1,41.3,603.1,-0.855,2.141,-7.023,0],AUTHORITY["EPSG","6124"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4124"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",11.3082777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",1500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","3019"]]`, nil},
	uint32(3020):  {`RT90 5 gon V`, uint32(3020), `EPSG`, uint32(3020), `PROJCS["RT90 5 gon V",GEOGCS["RT90",DATUM["Rikets koordinatsystem 1990",SPHEROID["Bessel 1841",6377397.155,299.1528128,AUTHORITY["EPSG","7004"]],TOWGS84[414.1,41.3,603.1,-0.855,2.141,-7.023,0],AUTHORITY["EPSG","6124"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4124"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",13.5582777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",1500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","3020"]]`, nil},
	uint32(3021):  {`RT90 2.5 gon V`, uint32(3021), `EPSG`, uint32(3021), `PROJCS["RT90 2.5 gon V",GEOGCS["RT90",DATUM["Rikets koordinatsystem 1990",SPHEROID["Bessel 1841",6377397.155,299.1528128,AUTHORITY["EPSG","7004"]],TOWGS84[414.1,41.3,603.1,-0.855,2.141,-7.023,0],AUTHORITY["EPSG","6124"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4124"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",15.8082777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",1500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","3021"]]`, nil},
	uint32(3022):  {`RT90 0 gon`, uint32(3022), `EPSG`, uint32(3022), `PROJCS["RT90 0 gon",GEOGCS["RT90",DATUM["Rikets koordinatsystem 1990",SPHEROID["Bessel 1841",6377397.155,299.1528128,AUTHORITY["EPSG","7004"]],TOWGS84[414.1,41.3,603.1,-0.855,2.141,-7.023,0],AUTHORITY["EPSG","6124"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4124"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",18.0582777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",1500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","3022"]]`, nil},
	uint32(3023):  {`RT90 2.5 gon O`, uint32(3023), `EPSG`, uint32(3023), `PROJCS["RT90 2.5 gon O",GEOGCS["RT90",DATUM["Rikets koordinatsystem 1990",SPHEROID["Bessel 1841",6377397.155,299.1528128,AUTHORITY["EPSG","7004"]],TOWGS84[414.1,41.3,603.1,-0.855,2.141,-7.023,0],AUTHORITY["EPSG","6124"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4124"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",20.3082777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",1500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","3023"]]`, nil},
	uint32(3024):  {`RT90 5 gon O`, uint32(3024), `EPSG`, uint32(3024), `PROJCS["RT90 5 gon O",GEOGCS["RT90",DATUM["Rikets koordinatsystem 1990",SPHEROID["Bessel 1841",6377397.155,299.1528128,AUTHORITY["EPSG","7004"]],TOWGS84[414.1,41.3,603.1,-0.855,2.141,-7.023,0],AUTHORITY["EPSG","6124"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4124"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",22.5582777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",1500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","3024"]]`, nil},
	uint32(3025):  {`RT38 7.5 gon V`, uint32(3025), `EPSG`, uint32(3025), `PROJCS["RT38 7.5 gon V",GEOGCS["RT38",DATUM["Stockholm 1938",SPHEROID["Bessel 1841",6377397.155,299.1528128,AUTHORITY["EPSG","7004"]],AUTHORITY["EPSG","6308"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4308"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",11.3082777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",1500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","3025"]]`, nil},
	uint32(3026):  {`RT38 5 gon V`, uint32(3026), `EPSG`, uint32(3026), `PROJCS["RT38 5 gon V",GEOGCS["RT38",DATUM["Stockholm 1938",SPHEROID["Bessel 1841",6377397.155,299.1528128,AUTHORITY["EPSG","7004"]],AUTHORITY["EPSG","6308"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4308"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",13.5582777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",1500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","3026"]]`, nil},
	uint32(3027):  {`RT38 2.5 gon V`, uint32(3027), `EPSG`, uint32(3027), `PROJCS["RT38 2.5 gon V",GEOGCS["RT38",DATUM["Stockholm 1938",SPHEROID["Bessel 1841",6377397.155,299.1528128,AUTHORITY["EPSG","7004"]],AUTHORITY["EPSG","6308"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4308"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",15.8082777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",1500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","3027"]]`, nil},
	uint32(3028):  {`RT38 0 gon`, uint32(3028), `EPSG`, uint32(3028), `PROJCS["RT38 0 gon",GEOGCS["RT38",DATUM["Stockholm 1938",SPHEROID["Bessel 1841",6377397.155,299.1528128,AUTHORITY["EPSG","7004"]],AUTHORITY["EPSG","6308"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4308"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",18.0582777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",1500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","3028"]]`, nil},
	uint32(3029):  {`RT38 2.5 gon O`, uint32(3029), `EPSG`, uint32(3029), `PROJCS["RT38 2.5 gon O",GEOGCS["RT38",DATUM["Stockholm 1938",SPHEROID["Bessel 1841",6377397.155,299.1528128,AUTHORITY["EPSG","7004"]],AUTHORITY["EPSG","6308"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4308"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",20.3082777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",1500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","3029"]]`, nil},
	uint32(3030):  {`RT38 5 gon O`, uint32(3030), `EPSG`, uint32(3030), `PROJCS["RT38 5 gon O",GEOGCS["RT38",DATUM["Stockholm 1938",SPHEROID["Bessel 1841",6377397.155,299.1528128,AUTHORITY["EPSG","7004"]],AUTHORITY["EPSG","6308"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4308"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",22.5582777777778,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",1500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","3030"]]`, nil},
	uint32(3031):  {`WGS 84 / Antarctic Polar Stereographic`, uint32(3031), `EPSG`, uint32(3031), `PROJCS["WGS 84 / Antarctic Polar Stereographic",GEOGCS["WGS 84",DATUM["World Geodetic System 1984",SPHEROID["WGS 84",6378137,298.257223563,AUTHORITY["EPSG","7030"]],AUTHORITY["EPSG","6326"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4326"]],PROJECTION["Polar Stereographic (variant B)",AUTHORITY["EPSG","9829"]],PARAMETER["Latitude of standard parallel",-71,AUTHORITY["EPSG","8832"]],PARAMETER["Longitude of origin",0,AUTHORITY["EPSG","8833"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",NORTH],AXIS["N",NORTH],AUTHORITY["EPSG","3031"]]`, nil},
	uint32(3032):  {`WGS 84 / Australian Antarctic Polar Stereographic`, uint32(3032), `EPSG`, uint32(3032), `PROJCS["WGS 84 / Australian Antarctic Polar Stereographic",GEOGCS["WGS 84",DATUM["World Geodetic System 1984",SPHEROID["WGS 84",6378137,298.257223563,AUTHORITY["EPSG","7030"]],AUTHORITY["EPSG","6326"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4326"]],PROJECTION["Polar Stereographic (variant B)",AUTHORITY["EPSG","9829"]],PARAMETER["Latitude of standard parallel",-71,AUTHORITY["EPSG","8832"]],PARAMETER["Longitude of origin",70,AUTHORITY["EPSG","8833"]],PARAMETER["False easting",6000000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",6000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",NORTH],AXIS["N",NORTH],AUTHORITY["EPSG","3032"]]`, nil},
	uint32(3033):  {`WGS 84 / Australian Antarctic Lambert`, uint32(3033), `EPSG`, uint32(3033), `PROJCS["WGS 84 / Australian Antarctic Lambert",GEOGCS["WGS 84",DATUM["World Geodetic System 1984",SPHEROID["WGS 84",6378137,298.257223563,AUTHORITY["EPSG","7030"]],AUTHORITY["EPSG","6326"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4326"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",-50,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",70,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",-68.5111111111111,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",-74.5111111111111,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",6000000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",6000000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","3033"]]`, nil},
	uint32(3034):  {`ETRS89 / LCC Europe`, uint32(3034), `EPSG`, uint32(3034), `PROJCS["ETRS89 / LCC Europe",GEOGCS["ETRS89",DATUM["European Terrestrial Reference System 1989",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6258"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4258"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",52,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",10,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",35,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",65,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",4000000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",2800000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","3034"]]`, nil},
	uint32(3035):  {`ETRS89 / LAEA Europe`, uint32(3035), `EPSG`, uint32(3035), `PROJCS["ETRS89 / LAEA Europe",GEOGCS["ETRS89",DATUM["European Terrestrial Reference System 1989",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6258"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4258"]],PROJECTION["Lambert Azimuthal Equal Area",AUTHORITY["EPSG","9820"]],PARAMETER["Latitude of natural origin",52,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",10,AUTHORITY["EPSG","8802"]],PARAMETER["False easting",4321000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",3210000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["Y",NORTH],AXIS["X",EAST],AUTHORITY["EPSG","3035"]]`, nil},
	uint32(3036):  {`Moznet / UTM zone 36S`, uint32(3036), `EPSG`, uint32(3036), `PROJCS["Moznet / UTM zone 36S",GEOGCS["Moznet",DATUM["Moznet (ITRF94)",SPHEROID["WGS 84",6378137,298.257223563,AUTHORITY["EPSG","7030"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6130"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4130"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",33,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","3036"]]`, nil},
	uint32(3037):  {`Moznet / UTM zone 37S`, uint32(3037), `EPSG`, uint32(3037), `PROJCS["Moznet / UTM zone 37S",GEOGCS["Moznet",DATUM["Moznet (ITRF94)",SPHEROID["WGS 84",6378137,298.257223563,AUTHORITY["EPSG","7030"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6130"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4130"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",39,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","3037"]]`, nil},
	uint32(3040):  {`ETRS89 / UTM zone 28N (N-E)`, uint32(3040), `EPSG`, uint32(3040), `PROJCS["ETRS89 / UTM zone 28N (N-E)",GEOGCS["ETRS89",DATUM["European Terrestrial Reference System 1989",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6258"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4258"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-15,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","3040"]]`, nil},
	uint32(3041):  {`ETRS89 / UTM zone 29N (N-E)`, uint32(3041), `EPSG`, uint32(3041), `PROJCS["ETRS89 / UTM zone 29N (N-E)",GEOGCS["ETRS89",DATUM["European Terrestrial Reference System 1989",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6258"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4258"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-9,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","3041"]]`, nil},
	uint32(3042):  {`ETRS89 / UTM zone 30N (N-E)`, uint32(3042), `EPSG`, uint32(3042), `PROJCS["ETRS89 / UTM zone 30N (N-E)",GEOGCS["ETRS89",DATUM["European Terrestrial Reference System 1989",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6258"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4258"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-3,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","3042"]]`, nil},
	uint32(3043):  {`ETRS89 / UTM zone 31N (N-E)`, uint32(3043), `EPSG`, uint32(3043), `PROJCS["ETRS89 / UTM zone 31N (N-E)",GEOGCS["ETRS89",DATUM["European Terrestrial Reference System 1989",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6258"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4258"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",3,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","3043"]]`, nil},
	uint32(3044):  {`ETRS89 / UTM zone 32N (N-E)`, uint32(3044), `EPSG`, uint32(3044), `PROJCS["ETRS89 / UTM zone 32N (N-E)",GEOGCS["ETRS89",DATUM["European Terrestrial Reference System 1989",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6258"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4258"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",9,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","3044"]]`, nil},
	uint32(3045):  {`ETRS89 / UTM zone 33N (N-E)`, uint32(3045), `EPSG`, uint32(3045), `PROJCS["ETRS89 / UTM zone 33N (N-E)",GEOGCS["ETRS89",DATUM["European Terrestrial Reference System 1989",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6258"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4258"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",15,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","3045"]]`, nil},
	uint32(3046):  {`ETRS89 / UTM zone 34N (N-E)`, uint32(3046), `EPSG`, uint32(3046), `PROJCS["ETRS89 / UTM zone 34N (N-E)",GEOGCS["ETRS89",DATUM["European Terrestrial Reference System 1989",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6258"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4258"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",21,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","3046"]]`, nil},
	uint32(3047):  {`ETRS89 / UTM zone 35N (N-E)`, uint32(3047), `EPSG`, uint32(3047), `PROJCS["ETRS89 / UTM zone 35N (N-E)",GEOGCS["ETRS89",DATUM["European Terrestrial Reference System 1989",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6258"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4258"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",27,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","3047"]]`, nil},
	uint32(3048):  {`ETRS89 / UTM zone 36N (N-E)`, uint32(3048), `EPSG`, uint32(3048), `PROJCS["ETRS89 / UTM zone 36N (N-E)",GEOGCS["ETRS89",DATUM["European Terrestrial Reference System 1989",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6258"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4258"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",33,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","3048"]]`, nil},
	uint32(3049):  {`ETRS89 / UTM zone 37N (N-E)`, uint32(3049), `EPSG`, uint32(3049), `PROJCS["ETRS89 / UTM zone 37N (N-E)",GEOGCS["ETRS89",DATUM["European Terrestrial Reference System 1989",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6258"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4258"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",39,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["N",NORTH],AXIS["E",EAST],AUTHORITY["EPSG","3049"]]`, nil},
	uint32(3052):  {`Reykjavik 1900 / Lambert 1900`, uint32(3052), `EPSG`, uint32(3052), `PROJCS["Reykjavik 1900 / Lambert 1900",GEOGCS["Reykjavik 1900",DATUM["Reykjavik 1900",SPHEROID["Danish 1876",6377019.27,300,AUTHORITY["EPSG","7051"]],TOWGS84[-28,199,5,0,0,0,0],AUTHORITY["EPSG","6657"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4657"]],PROJECTION["Lambert Conic Conformal (West Orientated)",AUTHORITY["EPSG","9826"]],PARAMETER["Latitude of natural origin",65,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-19.022125,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",0,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["W",WEST],AXIS["N",NORTH],AUTHORITY["EPSG","3052"]]`, nil},
	uint32(3053):  {`Hjorsey 1955 / Lambert 1955`, uint32(3053), `EPSG`, uint32(3053), `PROJCS["Hjorsey 1955 / Lambert 1955",GEOGCS["Hjorsey 1955",DATUM["Hjorsey 1955",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-73,47,-83,0,0,0,0],AUTHORITY["EPSG","6658"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4658"]],PROJECTION["Lambert Conic Conformal (West Orientated)",AUTHORITY["EPSG","9826"]],PARAMETER["Latitude of natural origin",65,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-18,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",500000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["W",WEST],AXIS["N",NORTH],AUTHORITY["EPSG","3053"]]`, nil},
	uint32(3054):  {`Hjorsey 1955 / UTM zone 26N`, uint32(3054), `EPSG`, uint32(3054), `PROJCS["Hjorsey 1955 / UTM zone 26N",GEOGCS["Hjorsey 1955",DATUM["Hjorsey 1955",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-73,47,-83,0,0,0,0],AUTHORITY["EPSG","6658"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4658"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-27,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","3054"]]`, nil},
	uint32(3055):  {`Hjorsey 1955 / UTM zone 27N`, uint32(3055), `EPSG`, uint32(3055), `PROJCS["Hjorsey 1955 / UTM zone 27N",GEOGCS["Hjorsey 1955",DATUM["Hjorsey 1955",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-73,47,-83,0,0,0,0],AUTHORITY["EPSG","6658"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4658"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-21,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","3055"]]`, nil},
	uint32(3056):  {`Hjorsey 1955 / UTM zone 28N`, uint32(3056), `EPSG`, uint32(3056), `PROJCS["Hjorsey 1955 / UTM zone 28N",GEOGCS["Hjorsey 1955",DATUM["Hjorsey 1955",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-73,47,-83,0,0,0,0],AUTHORITY["EPSG","6658"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4658"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-15,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","3056"]]`, nil},
	uint32(3057):  {`ISN93 / Lambert 1993`, uint32(3057), `EPSG`, uint32(3057), `PROJCS["ISN93 / Lambert 1993",GEOGCS["ISN93",DATUM["Islands Net 1993",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6659"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4659"]],PROJECTION["Lambert Conic Conformal (2SP)",AUTHORITY["EPSG","9802"]],PARAMETER["Latitude of false origin",65,AUTHORITY["EPSG","8821"]],PARAMETER["Longitude of false origin",-19,AUTHORITY["EPSG","8822"]],PARAMETER["Latitude of 1st standard parallel",64.25,AUTHORITY["EPSG","8823"]],PARAMETER["Latitude of 2nd standard parallel",65.75,AUTHORITY["EPSG","8824"]],PARAMETER["Easting at false origin",500000,AUTHORITY["EPSG","8826"]],PARAMETER["Northing at false origin",500000,AUTHORITY["EPSG","8827"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","3057"]]`, nil},
	uint32(3058):  {`Helle 1954 / Jan Mayen Grid`, uint32(3058), `EPSG`, uint32(3058), `PROJCS["Helle 1954 / Jan Mayen Grid",GEOGCS["Helle 1954",DATUM["Helle 1954",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[982.6087,552.753,-540.873,6.68162662527695,-31.6114924086423,-19.8481610048169,16.805],AUTHORITY["EPSG","6660"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4660"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-8.5,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",1,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",50000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",-7800000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["x",NORTH],AXIS["y",EAST],AUTHORITY["EPSG","3058"]]`, nil},
	uint32(3059):  {`LKS92 / Latvia TM`, uint32(3059), `EPSG`, uint32(3059), `PROJCS["LKS92 / Latvia TM",GEOGCS["LKS92",DATUM["Latvia 1992",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6661"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4661"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",24,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",-6000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",NORTH],AXIS["Y",EAST],AUTHORITY["EPSG","3059"]]`, nil},
	uint32(3060):  {`IGN72 Grande Terre / UTM zone 58S`, uint32(3060), `EPSG`, uint32(3060), `PROJCS["IGN72 Grande Terre / UTM zone 58S",GEOGCS["IGN72 Grande Terre",DATUM["IGN72 Grande Terre",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[97.295,-263.247,310.882,-1.5999,0.8386,3.1409,13.3259],AUTHORITY["EPSG","6634"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4662"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",165,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","3060"]]`, nil},
	uint32(3061):  {`Porto Santo 1995 / UTM zone 28N`, uint32(3061), `EPSG`, uint32(3061), `PROJCS["Porto Santo 1995 / UTM zone 28N",GEOGCS["Porto Santo 1995",DATUM["Porto Santo 1995",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-210.502,-66.902,-48.476,2.094,-15.067,-5.817,0.485],AUTHORITY["EPSG","6663"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4663"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-15,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","3061"]]`, nil},
	uint32(3062):  {`Azores Oriental 1995 / UTM zone 26N`, uint32(3062), `EPSG`, uint32(3062), `PROJCS["Azores Oriental 1995 / UTM zone 26N",GEOGCS["Azores Oriental 1995",DATUM["Azores Oriental Islands 1995",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-208.719,129.685,52.092,-0.195,-0.014,0.327,0.198],AUTHORITY["EPSG","6664"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4664"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-27,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","3062"]]`, nil},
	uint32(3063):  {`Azores Central 1995 / UTM zone 26N`, uint32(3063), `EPSG`, uint32(3063), `PROJCS["Azores Central 1995 / UTM zone 26N",GEOGCS["Azores Central 1995",DATUM["Azores Central Islands 1995",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-103.088,162.481,-28.276,0.167,0.082,0.168,-1.504],AUTHORITY["EPSG","6665"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4665"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-27,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","3063"]]`, nil},
	uint32(3064):  {`IGM95 / UTM zone 32N`, uint32(3064), `EPSG`, uint32(3064), `PROJCS["IGM95 / UTM zone 32N",GEOGCS["IGM95",DATUM["Istituto Geografico Militaire 1995",SPHEROID["WGS 84",6378137,298.257223563,AUTHORITY["EPSG","7030"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6670"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4670"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",9,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","3064"]]`, nil},
	uint32(3065):  {`IGM95 / UTM zone 33N`, uint32(3065), `EPSG`, uint32(3065), `PROJCS["IGM95 / UTM zone 33N",GEOGCS["IGM95",DATUM["Istituto Geografico Militaire 1995",SPHEROID["WGS 84",6378137,298.257223563,AUTHORITY["EPSG","7030"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6670"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4670"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",15,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","3065"]]`, nil},
	uint32(3066):  {`ED50 / Jordan TM`, uint32(3066), `EPSG`, uint32(3066), `PROJCS["ED50 / Jordan TM",GEOGCS["ED50",DATUM["European Datum 1950",SPHEROID["International 1924",6378388,297,AUTHORITY["EPSG","7022"]],TOWGS84[-157.89,-17.16,-78.41,2.118,2.697,-1.434,-5.38],AUTHORITY["EPSG","6230"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4230"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",37,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9998,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",-3000000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","3066"]]`, nil},
	uint32(3067):  {`ETRS89 / TM35FIN(E,N)`, uint32(3067), `EPSG`, uint32(3067), `PROJCS["ETRS89 / TM35FIN(E,N)",GEOGCS["ETRS89",DATUM["European Terrestrial Reference System 1989",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6258"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4258"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",27,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",0,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["E",EAST],AXIS["N",NORTH],AUTHORITY["EPSG","3067"]]`, nil},
	uint32(3068):  {`DHDN / Soldner Berlin`, uint32(3068), `EPSG`, uint32(3068), `PROJCS["DHDN / Soldner Berlin",GEOGCS["DHDN",DATUM["Deutsches Hauptdreiecksnetz",SPHEROID["Bessel 1841",6377397.155,299.1528128,AUTHORITY["EPSG","7004"]],TOWGS84[598.1,73.7,418.2,0.202,0.045,-2.455,6.7],AUTHORITY["EPSG","6314"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4314"]],PROJECTION["Cassini-Soldner",AUTHORITY["EPSG","9806"]],PARAMETER["Latitude of natural origin",52.4186482777778,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",13.6272036666667,AUTHORITY["EPSG","8802"]],PARAMETER["False easting",40000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",10000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["x",NORTH],AXIS["y",EAST],AUTHORITY["EPSG","3068"]]`, nil},
	uint32(3069):  {`NAD27 / Wisconsin Transverse Mercator`, uint32(3069), `EPSG`, uint32(3069), `PROJCS["NAD27 / Wisconsin Transverse Mercator",GEOGCS["NAD27",DATUM["North American Datum 1927",SPHEROID["Clarke 1866",6378206.4,294.9786982138982,AUTHORITY["EPSG","7008"]],TOWGS84[-32.3841359,180.4090461,120.8442577,-2.1545854,-0.1498782,0.5742915,8.1049164],AUTHORITY["EPSG","6267"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4267"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-90,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",500000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",-4500000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","3069"]]`, nil},
	uint32(3070):  {`NAD83 / Wisconsin Transverse Mercator`, uint32(3070), `EPSG`, uint32(3070), `PROJCS["NAD83 / Wisconsin Transverse Mercator",GEOGCS["NAD83",DATUM["North American Datum 1983",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[1,1,-1,0,0,0,0],AUTHORITY["EPSG","6269"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4269"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-90,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",520000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",-4480000,AUTHORITY["EPSG","8807"]],UNIT["metre",1,AUTHORITY["EPSG","9001"]],AXIS["X",EAST],AXIS["Y",NORTH],AUTHORITY["EPSG","3070"]]`, nil},
	uint32(3071):  {`NAD83(HARN) / Wisconsin Transverse Mercator`, uint32(3071), `EPSG`, uint32(3071), `PROJCS["NAD83(HARN) / Wisconsin Transverse Mercator",GEOGCS["NAD83(HARN)",DATUM["NAD83 (High Accuracy Reference Network)",SPHEROID["GRS 1980",6378137,298.257222101,AUTHORITY["EPSG","7019"]],TOWGS84[0,0,0,0,0,0,0],AUTHORITY["EPSG","6152"]],PRIMEM["Greenwich",0,AUTHORITY["EPSG","8901"]],UNIT["degree",0.017453292519943278,AUTHORITY["EPSG","9122"]],AXIS["Lat",NORTH],AXIS["Lon",EAST],AUTHORITY["EPSG","4152"]],PROJECTION["Transverse Mercator",AUTHORITY["EPSG","9807"]],PARAMETER["Latitude of natural origin",0,AUTHORITY["EPSG","8801"]],PARAMETER["Longitude of natural origin",-90,AUTHORITY["EPSG","8802"]],PARAMETER["Scale factor at natural origin",0.9996,AUTHORITY["EPSG","8805"]],PARAMETER["False easting",520000,AUTHORITY["EPSG","8806"]],PARAMETER["False northing",