/**
 * This script will generate TypeScript type definitions and a RTKQ clients for the alerting k8s APIs.
 *
 * Run `yarn run codegen` from the "grafana-alerting" package to invoke this script.
 *
 * API clients will be placed in "src/grafana/api/<version>/api.gen.ts"
 */
import type { ConfigFile } from '@rtk-query/codegen-openapi';

// ℹ️ append API groups and versions here to generate additional API clients
const SPECS = [
  ['notifications.alerting.grafana.app', ['v0alpha1']],
  ['rules.alerting.grafana.app', ['v0alpha1']],
  // keep this in Grafana Enterprise
  // ['alertenrichment.grafana.app', ['v1beta1']],
] as const;

type OutputFile = Omit<ConfigFile, 'outputFile'>;
type OutputFiles = Record<string, OutputFile>;

const outputFiles = SPECS.reduce<OutputFiles>((groupAcc, [group, versions]) => {
  return versions.reduce<OutputFiles>((versionAcc, version) => {
    // Create a unique export name based on the group
    const groupName = group.split('.')[0]; // e.g., 'notifications', 'rules', 'alertenrichment'
    const exportName = `${groupName}API`;

    // ℹ️ these snapshots are generated by running "go test pkg/tests/apis/openapi_test.go" and "scripts/process-specs.ts",
    // see the README in the "openapi_snapshots" directory
    const schemaFile = `../../../data/openapi/${group}-${version}.json`;

    // ℹ️ make sure there is a API file in each versioned directory
    const apiFile = `../src/grafana/api/${groupName}/${version}/api.ts`;

    // output each api client into a versioned directory with group-specific naming
    const outputPath = `../src/grafana/api/${groupName}/${version}/${groupName}.api.gen.ts`;

    versionAcc[outputPath] = {
      exportName,
      schemaFile,
      apiFile,
      tag: true, // generate tags for cache invalidation
    } satisfies OutputFile;

    return versionAcc;
  }, groupAcc);
}, {});

export default {
  // these are intentionally empty but will be set for each versioned config file
  exportName: '',
  schemaFile: '',
  apiFile: '',

  outputFiles,
} satisfies ConfigFile;
