import { FieldDisplay } from '@grafana/data';

export function getValueAngleForValue(fieldDisplay: FieldDisplay, startAngle: number, endAngle: number) {
  const angleRange = (360 % (startAngle === 0 ? 1 : startAngle)) + endAngle;
  const min = fieldDisplay.field.min ?? 0;
  const max = fieldDisplay.field.max ?? 100;

  let angle = ((fieldDisplay.display.numeric - min) / (max - min)) * angleRange;

  if (angle > angleRange) {
    angle = angleRange;
  } else if (angle < 0) {
    angle = 0;
  }

  return { angleRange, angle };
}

/**
 * Returns the angle in radians for a given angle in degrees
 * But shifted -90 degrees to make 0 degree angle point upwards
 * @param angle
 * @returns
 */
export function toRad(angle: number) {
  return ((angle - 90) * Math.PI) / 180;
}

export interface GaugeDimensions {
  margin: number;
  radius: number;
  centerX: number;
  centerY: number;
  barWidth: number;
  endAngle?: number;
  barIndex: number;
  thresholdsBarRadius: number;
  thresholdsBarWidth: number;
  thresholdsBarSpacing: number;
  showScaleLabels?: boolean;
  scaleLabelsFontSize: number;
  scaleLabelsSpacing: number;
  scaleLabelsRadius: number;
  gaugeBottomY: number;
}

export function calculateDimensions(
  width: number,
  height: number,
  endAngle: number,
  glow: boolean,
  roundedBars: boolean,
  barWidthFactor: number,
  barIndex: number,
  thresholdBar?: boolean,
  showScaleLabels?: boolean
): GaugeDimensions {
  const yMaxAngle = endAngle > 180 ? 180 : endAngle;
  let margin = 0;

  if (glow) {
    margin = 0.02 * Math.min(width, height);
  }

  // Max radius based on width
  let maxRadiusW = width / 2 - margin;

  // Max radius based on height
  let heightRatioV = Math.sin(toRad(yMaxAngle));
  let maxRadiusH = (height - margin * 2) / (1 + heightRatioV);

  let maxRadius = Math.min(maxRadiusW, maxRadiusH);
  let maxRadiusIsLimitedByHeight = maxRadiusH === maxRadius;
  let outerRadius = maxRadius;

  const barWidth = Math.max(barWidthFactor * (maxRadius / 3), 2);

  // If rounded bars is enabled they need a bit more vertical space
  if (yMaxAngle < 180 && roundedBars) {
    outerRadius -= barWidth;
    maxRadiusH -= barWidth;
    maxRadiusW -= barWidth;
  }

  // Scale labels
  let scaleLabelsFontSize = 0;
  let scaleLabelsSpacing = 0;
  let scaleLabelsRadius = 0;

  if (showScaleLabels) {
    scaleLabelsRadius = outerRadius;
    const radiusToFontSizeFactor = 0.12;
    scaleLabelsFontSize = Math.max(radiusToFontSizeFactor * Math.pow(outerRadius, 0.92), 10);
    scaleLabelsSpacing = scaleLabelsFontSize / 3;
    const labelsSize = scaleLabelsFontSize * 1.2 + scaleLabelsSpacing;
    outerRadius -= labelsSize;
    maxRadiusW -= labelsSize;
    maxRadiusH -= labelsSize;

    // For gauges the max label needs a bit more vertical space so that it does not get clipped
    if (maxRadiusIsLimitedByHeight && endAngle < 180) {
      const amount = outerRadius * 0.07;
      scaleLabelsRadius -= amount;
      maxRadiusH -= amount;
      maxRadiusW -= amount;
      outerRadius -= amount;
    }
  }

  // Thresholds bar
  const thresholdsToBarWidth = 0.2 * Math.pow(barWidth, 0.92);
  const thresholdsBarWidth = thresholdBar ? Math.min(Math.max(thresholdsToBarWidth, 4), 12) : 0;
  const thresholdsBarSpacing = Math.min(Math.max(thresholdsBarWidth / 2, 2), 12);
  let thresholdsBarRadius = 0;

  if (thresholdsBarWidth > 0) {
    thresholdsBarRadius = outerRadius - thresholdsBarWidth / 2;
    maxRadiusW -= thresholdsBarWidth + thresholdsBarSpacing;
    maxRadiusH -= thresholdsBarWidth + thresholdsBarSpacing;
    outerRadius = Math.min(maxRadiusW, maxRadiusH);
  }

  let innerRadius = outerRadius - barWidth / 2;

  const belowCenterY = maxRadius * Math.sin(toRad(yMaxAngle));
  const rest = height - belowCenterY - margin * 2 - maxRadius;
  const centerX = width / 2;
  const centerY = maxRadius + margin + rest / 2;

  if (barIndex > 0) {
    innerRadius = innerRadius - (barWidth + 4) * barIndex;
  }

  return {
    margin,
    gaugeBottomY: centerY + belowCenterY,
    radius: innerRadius,
    centerX,
    centerY,
    barWidth,
    barIndex,
    thresholdsBarWidth,
    thresholdsBarSpacing,
    thresholdsBarRadius,
    scaleLabelsFontSize,
    scaleLabelsSpacing,
    scaleLabelsRadius,
  };
}

export function toCartesian(centerX: number, centerY: number, radius: number, angleInDegrees: number) {
  let radian = ((angleInDegrees - 90) * Math.PI) / 180.0;
  return {
    x: centerX + radius * Math.cos(radian),
    y: centerY + radius * Math.sin(radian),
  };
}
