// Copyright 2022 Dolthub, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package variables

import (
	"fmt"
	"math"
	"os"
	"strings"
	"sync"
	"time"

	"github.com/google/uuid"
	"github.com/sirupsen/logrus"

	"github.com/dolthub/go-mysql-server/sql"
	"github.com/dolthub/go-mysql-server/sql/types"
)

// TODO: Add from the following sources because MySQL likes to not have every variable on a single page:
// https://dev.mysql.com/doc/refman/8.0/en/mysql-cluster-options-variables.html
// There's also this page, which shows that a TON of variables are still missing ):
// https://dev.mysql.com/doc/refman/8.0/en/server-system-variable-reference.html

// ServerStartUpTime is needed by uptime status variable
var ServerStartUpTime = time.Now()

// globalSystemVariables is the underlying type of SystemVariables.
type globalSystemVariables struct {
	mutex      *sync.RWMutex
	sysVarVals map[string]sql.SystemVarValue
}

var _ sql.SystemVariableRegistry = (*globalSystemVariables)(nil)

// AddSystemVariables adds the given system variables to the collection. If a name is already used by an existing
// variable, then it is overwritten with the new one.
func (sv *globalSystemVariables) AddSystemVariables(sysVars []sql.SystemVariable) {
	sv.mutex.Lock()
	defer sv.mutex.Unlock()
	for _, originalSysVar := range sysVars {
		if originalSysVar == nil {
			continue
		}
		sysVar := originalSysVar
		lowerName := strings.ToLower(sysVar.GetName())
		systemVars[lowerName] = sysVar
		sv.sysVarVals[lowerName] = sql.SystemVarValue{
			Var: sysVar,
			Val: sysVar.GetDefault(),
		}
	}
}

// AssignValues sets all of the values in the given map to their respective variables. If a variable cannot be found, or
// the value is invalid, then an error is returned. If the values contain any custom system variables, then make sure
// that they've been added using AddSystemVariables first.
func (sv *globalSystemVariables) AssignValues(vals map[string]interface{}) error {
	// TODO: Add context parameter
	ctx := sql.NewEmptyContext()
	sv.mutex.Lock()
	defer sv.mutex.Unlock()
	for varName, val := range vals {
		varName = strings.ToLower(varName)
		sysVar, ok := systemVars[varName]
		if !ok {
			return sql.ErrUnknownSystemVariable.New(varName)
		}
		svv, err := sysVar.InitValue(ctx, val, true)
		if err != nil {
			return err
		}
		sv.sysVarVals[varName] = svv
	}
	return nil
}

// NewSessionMap returns a new map of system variable values for sessions.
func (sv *globalSystemVariables) NewSessionMap() map[string]sql.SystemVarValue {
	sv.mutex.RLock()
	defer sv.mutex.RUnlock()
	sessionVals := make(map[string]sql.SystemVarValue, len(sv.sysVarVals))
	for key, val := range sv.sysVarVals {
		sessionVals[key] = val
	}
	return sessionVals
}

// GetGlobal returns the system variable definition and value for the given name. If the variable does not exist, returns
// false. Case-insensitive.
func (sv *globalSystemVariables) GetGlobal(name string) (sql.SystemVariable, interface{}, bool) {
	sv.mutex.RLock()
	defer sv.mutex.RUnlock()
	name = strings.ToLower(name)
	v, ok := systemVars[name]
	if !ok {
		return nil, nil, false
	}

	if msv, ok := v.(*sql.MysqlSystemVariable); ok && msv.ValueFunction != nil {
		result, err := msv.ValueFunction()
		if err != nil {
			logrus.StandardLogger().Warnf("unable to get value for system variable %s: %s", msv.GetName(), err.Error())
			return nil, nil, true
		}
		return v, result, true
	}

	// convert any set types to strings
	sysVal := sv.sysVarVals[name]
	if sysType, ok := sysVal.Var.GetType().(sql.SetType); ok {
		if setTypeVal, ok := sysVal.Val.(uint64); ok {
			var err error
			sysVal.Val, err = sysType.BitsToString(setTypeVal)
			if err != nil {
				return nil, nil, false
			}
		}
	}
	return v, sysVal.Val, true
}

// SetGlobal sets the system variable with the given name to the given value. If the system variable does not exist,
// then an error is returned. Additionally, if the value is invalid for the variable's type then an error is returned.
// Only global dynamic variables may be set through this function, as it is intended for use through the SET GLOBAL
// statement. To set session system variables, use the appropriate function on the session context. To set values
// directly (such as when loading persisted values), use AssignValues. Case-insensitive.
func (sv *globalSystemVariables) SetGlobal(ctx *sql.Context, name string, val interface{}) error {
	sv.mutex.Lock()
	defer sv.mutex.Unlock()
	name = strings.ToLower(name)
	sysVar, ok := systemVars[name]
	if !ok {
		return sql.ErrUnknownSystemVariable.New(name)
	}
	svv, err := sysVar.SetValue(ctx, val, true)
	if err != nil {
		return err
	}
	sv.sysVarVals[name] = svv
	return nil
}

// GetAllGlobalVariables returns map of global system variables with their values.
func (sv *globalSystemVariables) GetAllGlobalVariables() map[string]interface{} {
	sv.mutex.RLock()
	defer sv.mutex.RUnlock()

	m := make(map[string]interface{})
	for k, varVal := range sv.sysVarVals {
		m[k] = varVal.Val
	}

	return m
}

// InitSystemVariables resets the systemVars singleton in the sql package
func InitSystemVariables() {
	vars := &globalSystemVariables{
		mutex:      &sync.RWMutex{},
		sysVarVals: make(map[string]sql.SystemVarValue, len(systemVars)),
	}
	for _, sysVar := range systemVars {
		vars.sysVarVals[sysVar.GetName()] = sql.SystemVarValue{
			Var: sysVar,
			Val: sysVar.GetDefault(),
		}
	}
	sql.SystemVariables = vars
}

// init initializes SystemVariables as it functions as a global variable.
// TODO: get rid of me, make this construction the responsibility of the engine
func init() {
	InitSystemVariables()
}

func getHostname() string {
	hostname, _ := os.Hostname()
	return hostname
}

// systemVars is the internal collection of all MySQL system variables according to the following pages:
// https://dev.mysql.com/doc/refman/8.0/en/server-system-variables.html
// https://dev.mysql.com/doc/refman/8.0/en/replication-options-gtids.html
// https://dev.mysql.com/doc/refman/8.0/en/replication-options-source.html
var systemVars = map[string]sql.SystemVariable{
	"activate_all_roles_on_login": &sql.MysqlSystemVariable{
		Name:              "activate_all_roles_on_login",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("activate_all_roles_on_login"),
		Default:           int8(0),
	},
	"admin_address": &sql.MysqlSystemVariable{
		Name:              "admin_address",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("admin_address"),
		Default:           "",
	},
	"admin_port": &sql.MysqlSystemVariable{
		Name:              "admin_port",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("admin_port", 0, 65535, false),
		Default:           int64(33062),
	},
	"admin_ssl_ca": &sql.MysqlSystemVariable{
		Name:              "admin_ssl_ca",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("admin_ssl_ca"),
		Default:           "",
	},
	"admin_ssl_capath": &sql.MysqlSystemVariable{
		Name:              "admin_ssl_capath",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("admin_ssl_capath"),
		Default:           "",
	},
	"admin_ssl_cert": &sql.MysqlSystemVariable{
		Name:              "admin_ssl_cert",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("admin_ssl_cert"),
		Default:           "",
	},
	"admin_ssl_cipher": &sql.MysqlSystemVariable{
		Name:              "admin_ssl_cipher",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("admin_ssl_cipher"),
		Default:           "",
	},
	"admin_ssl_crl": &sql.MysqlSystemVariable{
		Name:              "admin_ssl_crl",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("admin_ssl_crl"),
		Default:           "",
	},
	"admin_ssl_crlpath": &sql.MysqlSystemVariable{
		Name:              "admin_ssl_crlpath",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("admin_ssl_crlpath"),
		Default:           "",
	},
	"admin_ssl_key": &sql.MysqlSystemVariable{
		Name:              "admin_ssl_key",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("admin_ssl_key"),
		Default:           "",
	},
	"admin_tls_ciphersuites": &sql.MysqlSystemVariable{
		Name:              "admin_tls_ciphersuites",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("admin_tls_ciphersuites"),
		Default:           "",
	},
	"admin_tls_version": &sql.MysqlSystemVariable{
		Name:              "admin_tls_version",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("admin_tls_version"),
		Default:           "TLSv1,TLSv1.1,TLSv1.2,TLSv1.3",
	},
	"authentication_windows_log_level": &sql.MysqlSystemVariable{
		Name:              "authentication_windows_log_level",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("authentication_windows_log_level", 0, 4, false),
		Default:           int64(2),
	},
	"authentication_windows_use_principal_name": &sql.MysqlSystemVariable{
		Name:              "authentication_windows_use_principal_name",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("authentication_windows_use_principal_name"),
		Default:           int8(1),
	},
	"autocommit": &sql.MysqlSystemVariable{
		Name:              "autocommit",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("autocommit"),
		Default:           int8(1),
	},
	"automatic_sp_privileges": &sql.MysqlSystemVariable{
		Name:              "automatic_sp_privileges",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("automatic_sp_privileges"),
		Default:           int8(1),
	},
	"auto_generate_certs": &sql.MysqlSystemVariable{
		Name:              "auto_generate_certs",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("auto_generate_certs"),
		Default:           int8(1),
	},
	"auto_increment_increment": &sql.MysqlSystemVariable{
		Name:              "auto_increment_increment",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemIntType("auto_increment_increment", 1, 65535, false),
		Default:           int64(1),
	},
	"auto_increment_offset": &sql.MysqlSystemVariable{
		Name:              "auto_increment_offset",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemIntType("auto_increment_offset", 1, 65535, false),
		Default:           int64(1),
	},
	"avoid_temporal_upgrade": &sql.MysqlSystemVariable{
		Name:              "avoid_temporal_upgrade",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("avoid_temporal_upgrade"),
		Default:           int8(0),
	},
	"back_log": &sql.MysqlSystemVariable{
		Name:              "back_log",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("back_log", 1, 65535, true),
		Default:           int64(-1),
	},
	// TODO: add to dolt
	"basedir": &sql.MysqlSystemVariable{
		Name:              "basedir",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("basedir"),
		Default:           "",
	},
	"big_tables": &sql.MysqlSystemVariable{
		Name:              "big_tables",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("big_tables"),
		Default:           int8(0),
	},
	"bind_address": &sql.MysqlSystemVariable{
		Name:              "bind_address",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("bind_address"),
		Default:           "*",
	},
	"binlog_checksum": &sql.MysqlSystemVariable{
		Name:              "binlog_checksum",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("binlog_checksum"),
		Default:           "CRC32",
	},
	"binlog_expire_logs_seconds": &sql.MysqlSystemVariable{
		Name:              "binlog_expire_logs_seconds",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("binlog_expire_logs_seconds", 0, 4294967295, false),
		Default:           int64(2592000),
	},
	"binlog_format": &sql.MysqlSystemVariable{
		Name:              "binlog_format",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemEnumType("binlog_format", "ROW"),
		Default:           "ROW",
	},
	"binlog_gtid_simple_recovery": &sql.MysqlSystemVariable{
		Name:              "binlog_gtid_simple_recovery",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("binlog_gtid_simple_recovery"),
		Default:           int8(1),
	},
	"binlog_row_image": &sql.MysqlSystemVariable{
		Name:              "binlog_row_image",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemEnumType("binlog_row_image", "MINIMAL", "FULL", "NOBLOB"),
		Default:           "FULL",
	},
	"binlog_row_metadata": &sql.MysqlSystemVariable{
		Name:              "binlog_row_metadata",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemEnumType("binlog_row_metadata", "MINIMAL", "FULL"),
		Default:           "MINIMAL",
	},
	"block_encryption_mode": &sql.MysqlSystemVariable{
		Name:              "block_encryption_mode",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("block_encryption_mode"),
		Default:           "aes-128-ecb",
	},
	"bulk_insert_buffer_size": &sql.MysqlSystemVariable{
		Name:              "bulk_insert_buffer_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemUintType("bulk_insert_buffer_size", 0, 18446744073709551615),
		Default:           uint64(8388608),
	},
	"caching_sha2_password_digest_rounds": &sql.MysqlSystemVariable{
		Name:              "caching_sha2_password_digest_rounds",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("caching_sha2_password_digest_rounds", 5000, 4095000, false),
		Default:           int64(5000),
	},
	"caching_sha2_password_auto_generate_rsa_keys": &sql.MysqlSystemVariable{
		Name:              "caching_sha2_password_auto_generate_rsa_keys",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("caching_sha2_password_auto_generate_rsa_keys"),
		Default:           int8(1),
	},
	"caching_sha2_password_private_key_path": &sql.MysqlSystemVariable{
		Name:              "caching_sha2_password_private_key_path",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("caching_sha2_password_private_key_path"),
		Default:           "private_key.pem",
	},
	"caching_sha2_password_public_key_path": &sql.MysqlSystemVariable{
		Name:              "caching_sha2_password_public_key_path",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("caching_sha2_password_public_key_path"),
		Default:           "public_key.pem",
	},
	"character_set_client": &sql.MysqlSystemVariable{
		Name:              "character_set_client",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("character_set_client"),
		Default:           sql.Collation_Default.CharacterSet().String(),
		NotifyChanged:     validateCharacterSet,
	},
	"character_set_connection": &sql.MysqlSystemVariable{
		Name:              "character_set_connection",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("character_set_connection"),
		Default:           sql.Collation_Default.CharacterSet().String(),
		NotifyChanged:     validateCharacterSet,
	},
	"character_set_database": &sql.MysqlSystemVariable{
		Name:              "character_set_database",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("character_set_database"),
		Default:           sql.Collation_Default.CharacterSet().String(),
		NotifyChanged:     validateCharacterSet,
	},
	"character_set_filesystem": &sql.MysqlSystemVariable{
		Name:              "character_set_filesystem",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("character_set_filesystem"),
		Default:           "binary",
		NotifyChanged:     validateCharacterSet,
	},
	"character_set_results": &sql.MysqlSystemVariable{
		Name:              "character_set_results",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("character_set_results"),
		Default:           sql.Collation_Default.CharacterSet().String(),
		NotifyChanged:     validateCharacterSet,
	},
	"character_set_server": &sql.MysqlSystemVariable{
		Name:              "character_set_server",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("character_set_server"),
		Default:           sql.Collation_Default.CharacterSet().String(),
		NotifyChanged:     validateCharacterSet,
	},
	"character_set_system": &sql.MysqlSystemVariable{
		Name:              "character_set_system",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("character_set_system"),
		Default:           sql.Collation_Default.CharacterSet().String(),
		NotifyChanged:     validateCharacterSet,
	},
	"character_sets_dir": &sql.MysqlSystemVariable{
		Name:              "character_sets_dir",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("character_sets_dir"),
		Default:           "",
	},
	"check_proxy_users": &sql.MysqlSystemVariable{
		Name:              "check_proxy_users",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("check_proxy_users"),
		Default:           int8(0),
	},
	"collation_connection": &sql.MysqlSystemVariable{
		Name:              "collation_connection",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("collation_connection"),
		Default:           sql.Collation_Default.String(),
		NotifyChanged:     validateCollation,
	},
	"collation_database": &sql.MysqlSystemVariable{
		Name:              "collation_database",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("collation_database"),
		Default:           sql.Collation_Default.String(),
		NotifyChanged:     validateCollation,
	},
	"collation_server": &sql.MysqlSystemVariable{
		Name:              "collation_server",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("collation_server"),
		Default:           sql.Collation_Default.String(),
		NotifyChanged:     validateCollation,
	},
	"completion_type": &sql.MysqlSystemVariable{
		Name:              "completion_type",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemEnumType("completion_type", "NO_CHAIN", "CHAIN", "RELEASE"),
		Default:           "NO_CHAIN",
	},
	"concurrent_insert": &sql.MysqlSystemVariable{
		Name:              "concurrent_insert",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemEnumType("concurrent_insert", "NEVER", "AUTO", "ALWAYS"),
		Default:           "AUTO",
	},
	"connect_timeout": &sql.MysqlSystemVariable{
		Name:              "connect_timeout",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("connect_timeout", 2, 31536000, false),
		Default:           int64(10),
	},
	"core_file": &sql.MysqlSystemVariable{
		Name:              "core_file",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("core_file"),
		Default:           int8(0),
	},
	"create_admin_listener_thread": &sql.MysqlSystemVariable{
		Name:              "create_admin_listener_thread",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("create_admin_listener_thread"),
		Default:           int8(0),
	},
	"cte_max_recursion_depth": &sql.MysqlSystemVariable{
		Name:              "cte_max_recursion_depth",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("cte_max_recursion_depth", 0, 4294967295, false),
		Default:           int64(1000),
	},
	"datadir": &sql.MysqlSystemVariable{
		Name:              "datadir",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("datadir"),
		Default:           "",
	},
	"debug_sync": &sql.MysqlSystemVariable{
		Name:              "debug_sync",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Session),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("debug_sync"),
		Default:           "",
	},
	"default_authentication_plugin": &sql.MysqlSystemVariable{
		Name:              "default_authentication_plugin",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemEnumType("default_authentication_plugin", "mysql_native_password", "sha256_password", "caching_sha2_password"),
		Default:           "caching_sha2_password",
	},
	"default_collation_for_utf8mb4": &sql.MysqlSystemVariable{
		Name:              "default_collation_for_utf8mb4",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemEnumType("default_collation_for_utf8mb4", "utf8mb4_0900_ai_ci", "utf8mb4_general_ci"),
		Default:           "utf8mb4_0900_ai_ci",
	},
	"default_password_lifetime": &sql.MysqlSystemVariable{
		Name:              "default_password_lifetime",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("default_password_lifetime", 0, 65535, false),
		Default:           int64(0),
	},
	"default_storage_engine": &sql.MysqlSystemVariable{
		Name:              "default_storage_engine",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemEnumType("default_storage_engine", "MEMORY", "MRG_MYISAM", "CSV", "FEDERATED", "PERFORMANCE_SCHEMA", "MyISAM", "InnoDB", "BLACKHOLE", "ARCHIVE"),
		Default:           "InnoDB",
	},
	"default_table_encryption": &sql.MysqlSystemVariable{
		Name:              "default_table_encryption",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemBoolType("default_table_encryption"),
		Default:           int8(0),
	},
	"default_tmp_storage_engine": &sql.MysqlSystemVariable{
		Name:              "default_tmp_storage_engine",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemEnumType("default_tmp_storage_engine", "MEMORY", "MRG_MYISAM", "CSV", "FEDERATED", "PERFORMANCE_SCHEMA", "MyISAM", "InnoDB", "BLACKHOLE", "ARCHIVE"),
		Default:           "InnoDB",
	},
	"default_week_format": &sql.MysqlSystemVariable{
		Name:              "default_week_format",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("default_week_format", 0, 7, false),
		Default:           int64(0),
	},
	"delay_key_write": &sql.MysqlSystemVariable{
		Name:              "delay_key_write",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemEnumType("delay_key_write", "ON", "OFF", "ALL"),
		Default:           "ON",
	},
	"delayed_insert_limit": &sql.MysqlSystemVariable{
		Name:              "delayed_insert_limit",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemUintType("delayed_insert_limit", 1, 18446744073709551615),
		Default:           uint64(100),
	},
	"delayed_insert_timeout": &sql.MysqlSystemVariable{
		Name:              "delayed_insert_timeout",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("delayed_insert_timeout", -9223372036854775808, 9223372036854775807, false),
		Default:           int64(300),
	},
	"delayed_queue_size": &sql.MysqlSystemVariable{
		Name:              "delayed_queue_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemUintType("delayed_queue_size", 1, 18446744073709551615),
		Default:           uint64(1000),
	},
	"disabled_storage_engines": &sql.MysqlSystemVariable{
		Name:              "disabled_storage_engines",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("disabled_storage_engines"),
		Default:           "",
	},
	"disconnect_on_expired_password": &sql.MysqlSystemVariable{
		Name:              "disconnect_on_expired_password",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("disconnect_on_expired_password"),
		Default:           int8(1),
	},
	"div_precision_increment": &sql.MysqlSystemVariable{
		Name:              "div_precision_increment",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemIntType("div_precision_increment", 0, 30, false),
		Default:           int64(4),
	},
	"dragnet.log_error_filter_rules": &sql.MysqlSystemVariable{
		Name:              "dragnet.log_error_filter_rules",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("dragnet.log_error_filter_rules"),
		Default:           "drop",
	},
	"end_markers_in_json": &sql.MysqlSystemVariable{
		Name:              "end_markers_in_json",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemBoolType("end_markers_in_json"),
		Default:           int8(0),
	},
	"enforce_gtid_consistency": &sql.MysqlSystemVariable{
		Name:              "enforce_gtid_consistency",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemEnumType("enforce_gtid_consistency", "OFF", "ON", "WARN"),
		Default:           "OFF",
	},
	"eq_range_index_dive_limit": &sql.MysqlSystemVariable{
		Name:              "eq_range_index_dive_limit",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemIntType("eq_range_index_dive_limit", 0, 4294967295, false),
		Default:           int64(200),
	},
	"event_scheduler": &sql.MysqlSystemVariable{
		Name:              "event_scheduler",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemEnumType("event_scheduler", "ON", "OFF", "DISABLED"),
		Default:           "ON",
		NotifyChanged: func(ctx *sql.Context, _ sql.SystemVariableScope, value sql.SystemVarValue) error {
			convertedVal, _, err := value.Var.GetType().Convert(ctx, value.Val)
			if err == nil {
				// TODO: need to update EventScheduler state at runtime if applicable
				s := strings.ToLower(convertedVal.(string))
				switch s {
				case "on", "1":
					// need access to valid analyzer and ctx to call eventscheduler.TurnOnEventScheduler()
				case "off", "0":
					// need to call eventscheduler.TurnOffEventScheduler()
				default:
					return fmt.Errorf("variable 'event_scheduler' can't be set to the value '%s'", s)
				}
			}
			return nil
		},
	},
	"explicit_defaults_for_timestamp": &sql.MysqlSystemVariable{
		Name:              "explicit_defaults_for_timestamp",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("explicit_defaults_for_timestamp"),
		Default:           int8(1),
	},
	"external_user": &sql.MysqlSystemVariable{
		Name:              "external_user",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Session),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("external_user"),
		Default:           "",
	},
	"flush": &sql.MysqlSystemVariable{
		Name:              "flush",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("flush"),
		Default:           int8(0),
	},
	"flush_time": &sql.MysqlSystemVariable{
		Name:              "flush_time",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("flush_time", 0, 9223372036854775807, false),
		Default:           int64(0),
	},
	"foreign_key_checks": &sql.MysqlSystemVariable{
		Name:              "foreign_key_checks",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemBoolType("foreign_key_checks"),
		Default:           int8(1),
	},
	"ft_boolean_syntax": &sql.MysqlSystemVariable{
		Name:              "ft_boolean_syntax",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("ft_boolean_syntax"),
		Default:           `+ -><()~*:""&|`,
	},
	"ft_max_word_len": &sql.MysqlSystemVariable{
		Name:              "ft_max_word_len",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("ft_max_word_len", 10, 9223372036854775807, false),
		Default:           int64(0),
	},
	"ft_min_word_len": &sql.MysqlSystemVariable{
		Name:              "ft_min_word_len",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("ft_min_word_len", 1, 9223372036854775807, false),
		Default:           int64(4),
	},
	"ft_query_expansion_limit": &sql.MysqlSystemVariable{
		Name:              "ft_query_expansion_limit",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("ft_query_expansion_limit", 0, 1000, false),
		Default:           int64(20),
	},
	"ft_stopword_file": &sql.MysqlSystemVariable{
		Name:              "ft_stopword_file",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("ft_stopword_file"),
		Default:           "",
	},
	"general_log": &sql.MysqlSystemVariable{
		Name:              "general_log",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("general_log"),
		Default:           int8(0),
	},
	"general_log_file": &sql.MysqlSystemVariable{
		Name:              "general_log_file",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("general_log_file"),
		Default:           "host_name.log",
	},
	"generated_random_password_length": &sql.MysqlSystemVariable{
		Name:              "generated_random_password_length",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("generated_random_password_length", 5, 255, false),
		Default:           int64(20),
	},
	"group_concat_max_len": &sql.MysqlSystemVariable{
		Name:              "group_concat_max_len",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemUintType("group_concat_max_len", 4, 18446744073709551615),
		Default:           uint64(1024),
	},
	"gtid_executed": &sql.MysqlSystemVariable{
		Name:              "gtid_executed",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("gtid_executed"),
		Default:           "",
	},
	"gtid_executed_compression_period": &sql.MysqlSystemVariable{
		Name:              "gtid_executed_compression_period",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("gtid_executed_compression_period", 0, 4294967295, false),
		Default:           int64(0),
	},
	"gtid_mode": &sql.MysqlSystemVariable{
		Name:              "gtid_mode",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemEnumType("gtid_mode", "OFF", "OFF_PERMISSIVE", "ON_PERMISSIVE", "ON"),
		Default:           "OFF",
	},
	"gtid_next": &sql.MysqlSystemVariable{
		Name:              "gtid_next",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Session),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemEnumType("gtid_next", "AUTOMATIC", "ANONYMOUS", "UUID:NUMBER"),
		Default:           "AUTOMATIC",
	},
	"gtid_owned": &sql.MysqlSystemVariable{
		Name:              "gtid_owned",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("gtid_owned"),
		Default:           "",
	},
	"gtid_purged": &sql.MysqlSystemVariable{
		Name:              "gtid_purged",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("gtid_purged"),
		Default:           "",
	},
	"have_statement_timeout": &sql.MysqlSystemVariable{
		Name:              "have_statement_timeout",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("have_statement_timeout"),
		Default:           int8(0),
	},
	"histogram_generation_max_mem_size": &sql.MysqlSystemVariable{
		Name:              "histogram_generation_max_mem_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemUintType("histogram_generation_max_mem_size", 1000000, 18446744073709551615),
		Default:           uint64(20000000),
	},
	"host_cache_size": &sql.MysqlSystemVariable{
		Name:              "host_cache_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("host_cache_size", 0, 65536, true),
		Default:           int64(-1),
	},
	"hostname": &sql.MysqlSystemVariable{
		Name:              "hostname",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("hostname"),
		Default:           getHostname(),
	},
	"immediate_server_version": &sql.MysqlSystemVariable{
		Name:              "immediate_server_version",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Session),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("immediate_server_version", -9223372036854775808, 9223372036854775807, false),
		Default:           int64(80017),
	},
	"init_connect": &sql.MysqlSystemVariable{
		Name:              "init_connect",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("init_connect"),
		Default:           "",
	},
	"information_schema_stats_expiry": &sql.MysqlSystemVariable{
		Name:              "information_schema_stats_expiry",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("information_schema_stats_expiry", 0, 31536000, false),
		Default:           int64(86400),
	},
	"init_file": &sql.MysqlSystemVariable{
		Name:              "init_file",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("init_file"),
		Default:           "",
	},
	"inmemory_joins": &sql.MysqlSystemVariable{
		Name:              "inmemory_joins",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Session),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("inmemory_joins"),
		Default:           int8(0),
	},
	"disable_merge_join": &sql.MysqlSystemVariable{
		Name:              sql.DisableMergeJoin,
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType(sql.DisableMergeJoin),
		Default:           int8(0),
	},
	"innodb_autoinc_lock_mode": &sql.MysqlSystemVariable{
		Name:              "innodb_autoinc_lock_mode",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("innodb_autoinc_lock_mode", 0, 2, false),
		Default:           int64(2),
	},
	"innodb_buffer_pool_size": &sql.MysqlSystemVariable{
		Name:              "innodb_buffer_pool_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("innodb_buffer_pool_size", 5242880, math.MaxInt64, false),
		Default:           int64(134217728),
	},
	// Row locking is currently not supported. This variable is provided for 3p tools, and we always return the
	// Lowest value allowed by MySQL, which is 1. If you attempt to set this value to anything other than 1, errors ensue.
	"innodb_lock_wait_timeout": &sql.MysqlSystemVariable{
		Name:              "innodb_lock_wait_timeout",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("innodb_lock_wait_timeout", 1, 1, false),
		Default:           int64(1),
	},
	"innodb_stats_auto_recalc": &sql.MysqlSystemVariable{
		Name:              "innodb_stats_auto_recalc",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("innodb_stats_auto_recalc"),
		Default:           int8(1),
	},
	"interactive_timeout": &sql.MysqlSystemVariable{
		Name:              "interactive_timeout",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("interactive_timeout", 1, 9223372036854775807, false),
		Default:           int64(28800),
	},
	"internal_tmp_disk_storage_engine": &sql.MysqlSystemVariable{
		Name:              "internal_tmp_disk_storage_engine",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemEnumType("internal_tmp_disk_storage_engine", "MYISAM", "INNODB"),
		Default:           "INNODB",
	},
	"internal_tmp_mem_storage_engine": &sql.MysqlSystemVariable{
		Name:              "internal_tmp_mem_storage_engine",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemEnumType("internal_tmp_mem_storage_engine", "TempTable", "MEMORY"),
		Default:           "TempTable",
	},
	"join_buffer_size": &sql.MysqlSystemVariable{
		Name:              "join_buffer_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemUintType("join_buffer_size", 128, 18446744073709547520),
		Default:           uint64(262144),
	},
	"join_complexity_limit": &sql.MysqlSystemVariable{
		Name:              "join_complexity_limit",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemUintType("join_complexity_limit", 2, 20),
		Default:           uint64(12),
	},
	"keep_files_on_create": &sql.MysqlSystemVariable{
		Name:              "keep_files_on_create",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("keep_files_on_create"),
		Default:           int8(0),
	},
	"key_buffer_size": &sql.MysqlSystemVariable{
		Name:              "key_buffer_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemUintType("key_buffer_size", 8, 18446744073709551615),
		Default:           uint64(8388608),
	},
	"key_cache_age_threshold": &sql.MysqlSystemVariable{
		Name:              "key_cache_age_threshold",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemUintType("key_cache_age_threshold", 100, 18446744073709551615),
		Default:           uint64(300),
	},
	"key_cache_block_size": &sql.MysqlSystemVariable{
		Name:              "key_cache_block_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("key_cache_block_size", 512, 16384, false),
		Default:           int64(1024),
	},
	"key_cache_division_limit": &sql.MysqlSystemVariable{
		Name:              "key_cache_division_limit",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("key_cache_division_limit", 1, 100, false),
		Default:           int64(100),
	},
	"large_files_support": &sql.MysqlSystemVariable{
		Name:              "large_files_support",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("large_files_support"),
		Default:           int8(0),
	},
	"large_pages": &sql.MysqlSystemVariable{
		Name:              "large_pages",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("large_pages"),
		Default:           int8(0),
	},
	"large_page_size": &sql.MysqlSystemVariable{
		Name:              "large_page_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("large_page_size", -9223372036854775808, 9223372036854775807, false),
		Default:           int64(0),
	},
	"last_insert_id": &sql.MysqlSystemVariable{
		Name:              "last_insert_id",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Session),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("last_insert_id", -9223372036854775808, 9223372036854775807, false),
		Default:           int64(0),
	},
	"lc_messages": &sql.MysqlSystemVariable{
		Name:              "lc_messages",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("lc_messages"),
		Default:           "en_US",
	},
	"lc_messages_dir": &sql.MysqlSystemVariable{
		Name:              "lc_messages_dir",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("lc_messages_dir"),
		Default:           "",
	},
	"lc_time_names": &sql.MysqlSystemVariable{
		Name:              "lc_time_names",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("lc_time_names"),
		Default:           "",
	},
	"license": &sql.MysqlSystemVariable{
		Name:              "license",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("license"),
		Default:           "GPL",
	},
	"local_infile": &sql.MysqlSystemVariable{
		Name:              "local_infile",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("local_infile"),
		Default:           int8(0),
	},
	"lock_wait_timeout": &sql.MysqlSystemVariable{
		Name:              "lock_wait_timeout",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemIntType("lock_wait_timeout", 1, 31536000, false),
		Default:           int64(31536000),
	},
	"lock_warnings": &sql.MysqlSystemVariable{
		Name:              "lock_warnings",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Session),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("lock_warnings"),
		Default:           int8(0),
		NotifyChanged: func(ctx *sql.Context, _ sql.SystemVariableScope, value sql.SystemVarValue) error {
			switch value.Val.(int8) {
			case 0:
				ctx.UnlockWarnings()
			case 1:
				ctx.LockWarnings()
			}
			return nil
		},
	},
	"log_bin": &sql.MysqlSystemVariable{
		Name:              "log_bin",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Persist),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("log_bin"),
		// NOTE: MySQL defaults to the binary log being enabled, but GMS defaults
		//       to disabled, since binary log support is not available in GMS.
		//       Integrators who provide binary logging may change this default.
		Default: int8(0),
	},
	"log_replica_updates": &sql.MysqlSystemVariable{
		Name:              "log_replica_updates",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Persist),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("log_replica_updates"),
		Default:           int8(1),
	},
	"log_slave_updates": &sql.MysqlSystemVariable{
		// TODO: This var should be an *alias* for log_replica_updates, but
		//       we don't support system variable aliases yet.
		Name:              "log_slave_updates",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Persist),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("log_slave_updates"),
		Default:           int8(1),
	},
	"log_error": &sql.MysqlSystemVariable{
		Name:              "log_error",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("log_error"),
		Default:           "",
	},
	"log_error_services": &sql.MysqlSystemVariable{
		Name:              "log_error_services",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("log_error_services"),
		Default:           "log_filter_internal; log_sink_internal",
	},
	"log_error_suppression_list": &sql.MysqlSystemVariable{
		Name:              "log_error_suppression_list",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("log_error_suppression_list"),
		Default:           "",
	},
	"log_error_verbosity": &sql.MysqlSystemVariable{
		Name:              "log_error_verbosity",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("log_error_verbosity", 1, 3, false),
		Default:           int64(2),
	},
	"log_output": &sql.MysqlSystemVariable{
		Name:              "log_output",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemSetType("log_output", sql.Collation_utf8mb4_0900_ai_ci, "TABLE", "FILE", "NONE"),
		Default:           "FILE",
	},
	"log_queries_not_using_indexes": &sql.MysqlSystemVariable{
		Name:              "log_queries_not_using_indexes",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("log_queries_not_using_indexes"),
		Default:           int8(0),
	},
	"log_raw": &sql.MysqlSystemVariable{
		Name:              "log_raw",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("log_raw"),
		Default:           int8(0),
	},
	"log_slow_admin_statements": &sql.MysqlSystemVariable{
		Name:              "log_slow_admin_statements",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("log_slow_admin_statements"),
		Default:           int8(0),
	},
	"log_slow_extra": &sql.MysqlSystemVariable{
		Name:              "log_slow_extra",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("log_slow_extra"),
		Default:           int8(0),
	},
	"log_syslog": &sql.MysqlSystemVariable{
		Name:              "log_syslog",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("log_syslog"),
		Default:           int8(1),
	},
	"log_syslog_facility": &sql.MysqlSystemVariable{
		Name:              "log_syslog_facility",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("log_syslog_facility"),
		Default:           "daemon",
	},
	"log_syslog_include_pid": &sql.MysqlSystemVariable{
		Name:              "log_syslog_include_pid",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("log_syslog_include_pid"),
		Default:           int8(1),
	},
	"log_syslog_tag": &sql.MysqlSystemVariable{
		Name:              "log_syslog_tag",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("log_syslog_tag"),
		Default:           "",
	},
	"log_timestamps": &sql.MysqlSystemVariable{
		Name:              "log_timestamps",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemEnumType("log_timestamps", "UTC", "SYSTEM"),
		Default:           "UTC",
	},
	"log_throttle_queries_not_using_indexes": &sql.MysqlSystemVariable{
		Name:              "log_throttle_queries_not_using_indexes",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("log_throttle_queries_not_using_indexes", -9223372036854775808, 9223372036854775807, false),
		Default:           int64(0),
	},
	"long_query_time": &sql.MysqlSystemVariable{
		Name:              "long_query_time",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemDoubleType("long_query_time", 0, math.MaxFloat64),
		Default:           float64(10),
	},
	"low_priority_updates": &sql.MysqlSystemVariable{
		Name:              "low_priority_updates",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("low_priority_updates"),
		Default:           int8(0),
	},
	"lower_case_file_system": &sql.MysqlSystemVariable{
		Name:              "lower_case_file_system",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("lower_case_file_system"),
		Default:           int8(0),
	},
	"lower_case_table_names": &sql.MysqlSystemVariable{
		Name:              "lower_case_table_names",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("lower_case_table_names", 0, 2, false),
		Default:           int64(0),
	},
	"mandatory_roles": &sql.MysqlSystemVariable{
		Name:              "mandatory_roles",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("mandatory_roles"),
		Default:           "",
	},
	"max_allowed_packet": &sql.MysqlSystemVariable{
		Name:              "max_allowed_packet",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemUintType("max_allowed_packet", 1024, 1073741824),
		Default:           int64(1073741824),
	},
	"max_binlog_size": &sql.MysqlSystemVariable{
		Name:              "max_binlog_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemUintType("max_binlog_size", 4096, 1073741824),
		Default:           int64(1073741824),
	},
	"max_connect_errors": &sql.MysqlSystemVariable{
		Name:              "max_connect_errors",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemUintType("max_connect_errors", 1, 18446744073709551615),
		Default:           uint64(100),
	},
	"max_connections": &sql.MysqlSystemVariable{
		Name:              "max_connections",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("max_connections", 1, 100000, false),
		Default:           int64(151),
	},
	"max_delayed_threads": &sql.MysqlSystemVariable{
		Name:              "max_delayed_threads",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("max_delayed_threads", 0, 16384, false),
		Default:           int64(20),
	},
	"max_digest_length": &sql.MysqlSystemVariable{
		Name:              "max_digest_length",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("max_digest_length", 0, 1048576, false),
		Default:           int64(1024),
	},
	"max_error_count": &sql.MysqlSystemVariable{
		Name:              "max_error_count",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemIntType("max_error_count", 0, 65535, false),
		Default:           int64(1024),
	},
	"max_execution_time": &sql.MysqlSystemVariable{
		Name:              "max_execution_time",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemIntType("max_execution_time", -9223372036854775808, 9223372036854775807, false),
		Default:           int64(0),
	},
	"max_heap_table_size": &sql.MysqlSystemVariable{
		Name:              "max_heap_table_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemUintType("max_heap_table_size", 16384, 1844674407370954752),
		Default:           uint64(16777216),
	},
	"max_insert_delayed_threads": &sql.MysqlSystemVariable{
		Name:              "max_insert_delayed_threads",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("max_insert_delayed_threads", -9223372036854775808, 9223372036854775807, false),
		Default:           int64(0),
	},
	"max_join_size": &sql.MysqlSystemVariable{
		Name:              "max_join_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemUintType("max_join_size", 1, 18446744073709551615),
		Default:           uint64(18446744073709551615),
	},
	"max_length_for_sort_data": &sql.MysqlSystemVariable{
		Name:              "max_length_for_sort_data",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemIntType("max_length_for_sort_data", 4, 8388608, false),
		Default:           int64(4096),
	},
	"max_points_in_geometry": &sql.MysqlSystemVariable{
		Name:              "max_points_in_geometry",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemIntType("max_points_in_geometry", 3, 1048576, false),
		Default:           int64(65536),
	},
	"max_prepared_stmt_count": &sql.MysqlSystemVariable{
		Name:              "max_prepared_stmt_count",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("max_prepared_stmt_count", 0, 4194304, false),
		Default:           int64(16382),
	},
	"max_seeks_for_key": &sql.MysqlSystemVariable{
		Name:              "max_seeks_for_key",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemUintType("max_seeks_for_key", 1, 18446744073709551615),
		Default:           uint64(18446744073709551615),
	},
	"max_sort_length": &sql.MysqlSystemVariable{
		Name:              "max_sort_length",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemIntType("max_sort_length", 4, 8388608, false),
		Default:           int64(1024),
	},
	"max_sp_recursion_depth": &sql.MysqlSystemVariable{
		Name:              "max_sp_recursion_depth",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("max_sp_recursion_depth", -9223372036854775808, 255, false),
		Default:           int64(0),
	},
	"max_user_connections": &sql.MysqlSystemVariable{
		Name:              "max_user_connections",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("max_user_connections", 0, 4294967295, false),
		Default:           int64(0),
	},
	"max_write_lock_count": &sql.MysqlSystemVariable{
		Name:              "max_write_lock_count",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemUintType("max_write_lock_count", 1, 18446744073709551615),
		Default:           uint64(18446744073709551615),
	},
	"mecab_rc_file": &sql.MysqlSystemVariable{
		Name:              "mecab_rc_file",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("mecab_rc_file"),
		Default:           "",
	},
	"metadata_locks_cache_size": &sql.MysqlSystemVariable{
		Name:              "metadata_locks_cache_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("metadata_locks_cache_size", 1, 1048576, false),
		Default:           int64(1024),
	},
	"metadata_locks_hash_instances": &sql.MysqlSystemVariable{
		Name:              "metadata_locks_hash_instances",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("metadata_locks_hash_instances", 1, 1024, false),
		Default:           int64(8),
	},
	"min_examined_row_limit": &sql.MysqlSystemVariable{
		Name:              "min_examined_row_limit",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemUintType("min_examined_row_limit", 0, 18446744073709551615),
		Default:           uint64(0),
	},
	"myisam_data_pointer_size": &sql.MysqlSystemVariable{
		Name:              "myisam_data_pointer_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("myisam_data_pointer_size", 2, 7, false),
		Default:           int64(6),
	},
	"myisam_max_sort_file_size": &sql.MysqlSystemVariable{
		Name:              "myisam_max_sort_file_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("myisam_max_sort_file_size", -9223372036854775808, 9223372036853727232, false),
		Default:           int64(9223372036853727232),
	},
	"myisam_mmap_size": &sql.MysqlSystemVariable{
		Name:              "myisam_mmap_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemUintType("myisam_mmap_size", 7, 18446744073709551615),
		Default:           uint64(18446744073709551615),
	},
	"myisam_recover_options": &sql.MysqlSystemVariable{
		Name:              "myisam_recover_options",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemEnumType("myisam_recover_options", "OFF", "DEFAULT", "BACKUP", "FORCE", "QUICK"),
		Default:           "OFF",
	},
	"myisam_repair_threads": &sql.MysqlSystemVariable{
		Name:              "myisam_repair_threads",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemUintType("myisam_repair_threads", 1, 18446744073709551615),
		Default:           uint64(1),
	},
	"myisam_sort_buffer_size": &sql.MysqlSystemVariable{
		Name:              "myisam_sort_buffer_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemUintType("myisam_sort_buffer_size", 4096, 18446744073709551615),
		Default:           uint64(8388608),
	},
	"myisam_stats_method": &sql.MysqlSystemVariable{
		Name:              "myisam_stats_method",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemEnumType("myisam_stats_method", "nulls_equal", "nulls_unequal", "nulls_ignored"),
		Default:           "nulls_unequal",
	},
	"myisam_use_mmap": &sql.MysqlSystemVariable{
		Name:              "myisam_use_mmap",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("myisam_use_mmap"),
		Default:           int8(0),
	},
	"mysql_native_password_proxy_users": &sql.MysqlSystemVariable{
		Name:              "mysql_native_password_proxy_users",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("mysql_native_password_proxy_users"),
		Default:           int8(0),
	},
	"named_pipe": &sql.MysqlSystemVariable{
		Name:              "named_pipe",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("named_pipe"),
		Default:           int8(0),
	},
	"named_pipe_full_access_group": &sql.MysqlSystemVariable{
		Name:              "named_pipe_full_access_group",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("named_pipe_full_access_group"),
		Default:           "",
	},
	"ndbinfo_version": &sql.MysqlSystemVariable{
		Name:              "ndbinfo_version",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("ndbinfo_version"),
		Default:           "",
	},
	"net_buffer_length": &sql.MysqlSystemVariable{
		Name:              "net_buffer_length",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("net_buffer_length", 1024, 1048576, false),
		Default:           int64(16384),
	},
	"net_read_timeout": &sql.MysqlSystemVariable{
		Name:              "net_read_timeout",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("net_read_timeout", 1, 9223372036854775807, false),
		Default:           int64(30),
	},
	"net_retry_count": &sql.MysqlSystemVariable{
		Name:              "net_retry_count",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemUintType("net_retry_count", 1, 18446744073709551615),
		Default:           uint64(10),
	},
	"net_write_timeout": &sql.MysqlSystemVariable{
		Name:              "net_write_timeout",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("net_write_timeout", 1, 9223372036854775807, false),
		Default:           int64(60),
	},
	"new": &sql.MysqlSystemVariable{
		Name:              "new",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("new"),
		Default:           int8(0),
	},
	"ngram_token_size": &sql.MysqlSystemVariable{
		Name:              "ngram_token_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("ngram_token_size", 1, 10, false),
		Default:           int64(2),
	},
	"offline_mode": &sql.MysqlSystemVariable{
		Name:              "offline_mode",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("offline_mode"),
		Default:           int8(0),
	},
	"old": &sql.MysqlSystemVariable{
		Name:              "old",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("old"),
		Default:           int8(0),
	},
	"old_alter_table": &sql.MysqlSystemVariable{
		Name:              "old_alter_table",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("old_alter_table"),
		Default:           int8(0),
	},
	"open_files_limit": &sql.MysqlSystemVariable{
		Name:              "open_files_limit",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemUintType("open_files_limit", 0, 18446744073709551615),
		Default:           uint64(5000),
	},
	"optimizer_prune_level": &sql.MysqlSystemVariable{
		Name:              "optimizer_prune_level",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemIntType("optimizer_prune_level", 0, 1, false),
		Default:           int64(1),
	},
	"optimizer_search_depth": &sql.MysqlSystemVariable{
		Name:              "optimizer_search_depth",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemIntType("optimizer_search_depth", 0, 62, false),
		Default:           int64(62),
	},
	// TODO: add proper support for this
	// "optimizer_switch": &sql.MysqlSystemVariable{
	//	Name: "optimizer_switch",
	//	Scope: sql.GetMysqlScope(SystemVariableScope_Both)
	//	Dynamic: true,
	//	SetVarHintApplies: true,
	//	Type: NewSystemSetType("optimizer_switch"),
	//	Default: "",
	// },
	"optimizer_trace": &sql.MysqlSystemVariable{
		Name:              "optimizer_trace",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("optimizer_trace"),
		Default:           "",
	},
	"optimizer_trace_features": &sql.MysqlSystemVariable{
		Name:              "optimizer_trace_features",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("optimizer_trace_features"),
		Default:           "",
	},
	"optimizer_trace_limit": &sql.MysqlSystemVariable{
		Name:              "optimizer_trace_limit",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("optimizer_trace_limit", -9223372036854775808, 9223372036854775807, false),
		Default:           int64(1),
	},
	"optimizer_trace_max_mem_size": &sql.MysqlSystemVariable{
		Name:              "optimizer_trace_max_mem_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemIntType("optimizer_trace_max_mem_size", -9223372036854775808, 9223372036854775807, false),
		Default:           int64(1048576),
	},
	"optimizer_trace_offset": &sql.MysqlSystemVariable{
		Name:              "optimizer_trace_offset",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("optimizer_trace_offset", -9223372036854775808, 9223372036854775807, true),
		Default:           int64(-1),
	},
	"original_server_version": &sql.MysqlSystemVariable{
		Name:              "original_server_version",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Session),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("original_server_version", -9223372036854775808, 9223372036854775807, false),
		Default:           int64(80017),
	},
	"parser_max_mem_size": &sql.MysqlSystemVariable{
		Name:              "parser_max_mem_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemUintType("parser_max_mem_size", 10000000, 18446744073709551615),
		Default:           uint64(18446744073709551615),
	},
	"partial_revokes": &sql.MysqlSystemVariable{
		Name:              "partial_revokes",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("partial_revokes"),
		Default:           int8(0),
	},
	"password_history": &sql.MysqlSystemVariable{
		Name:              "password_history",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("password_history", 0, 4294967295, false),
		Default:           int64(0),
	},
	"password_require_current": &sql.MysqlSystemVariable{
		Name:              "password_require_current",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("password_require_current"),
		Default:           int8(0),
	},
	"password_reuse_interval": &sql.MysqlSystemVariable{
		Name:              "password_reuse_interval",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("password_reuse_interval", 0, 4294967295, false),
		Default:           int64(0),
	},
	"performance_schema": &sql.MysqlSystemVariable{
		Name:              "performance_schema",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("performance_schema"),
		// NOTE: MySQL defaults performance_schema to enabled, but since GMS does not
		//       provide a performance_schema yet, we default to disabled so that tools
		//       know not to try and query it.
		Default: int8(0),
	},
	"persisted_globals_load": &sql.MysqlSystemVariable{
		Name:              "persisted_globals_load",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("persisted_globals_load"),
		Default:           int8(1),
	},
	"persist_only_admin_x509_subject": &sql.MysqlSystemVariable{
		Name:              "persist_only_admin_x509_subject",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("persist_only_admin_x509_subject"),
		Default:           "",
	},
	"pid_file": &sql.MysqlSystemVariable{
		Name:              "pid_file",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("pid_file"),
		Default:           "",
	},
	"plugin_dir": &sql.MysqlSystemVariable{
		Name:              "plugin_dir",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("plugin_dir"),
		Default:           "",
	},
	"port": &sql.MysqlSystemVariable{
		Name:              "port",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("port", 0, 65535, false),
		Default:           int64(3306),
	},
	"preload_buffer_size": &sql.MysqlSystemVariable{
		Name:              "preload_buffer_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("preload_buffer_size", 1024, 1073741824, false),
		Default:           int64(32768),
	},
	"print_identified_with_as_hex": &sql.MysqlSystemVariable{
		Name:              "print_identified_with_as_hex",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("print_identified_with_as_hex"),
		Default:           int8(0),
	},
	"protocol_compression_algorithms": &sql.MysqlSystemVariable{
		Name:              "protocol_compression_algorithms",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemSetType("protocol_compression_algorithms", sql.Collation_latin1_swedish_ci, "zlib", "zstd", "uncompressed"),
		Default:           "zlib,zstd,uncompressed",
	},
	"protocol_version": &sql.MysqlSystemVariable{
		Name:              "protocol_version",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("protocol_version", -9223372036854775808, 9223372036854775807, false),
		Default:           int64(0),
	},
	"proxy_user": &sql.MysqlSystemVariable{
		Name:              "proxy_user",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Session),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("proxy_user"),
		Default:           "",
	},
	"pseudo_slave_mode": &sql.MysqlSystemVariable{
		Name:              "pseudo_slave_mode",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Session),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("pseudo_slave_mode"),
		Default:           int8(0),
	},
	"pseudo_thread_id": &sql.MysqlSystemVariable{
		Name:              "pseudo_thread_id",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Session),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("pseudo_thread_id", -9223372036854775808, 9223372036854775807, false),
		Default:           int64(0),
	},
	// TODO: implement block sizes
	// "query_alloc_block_size": &sql.MysqlSystemVariable{
	//	Name: "query_alloc_block_size",
	//	Scope: sql.GetMysqlScope(SystemVariableScope_Both),
	//	Dynamic: true,
	//	SetVarHintApplies: false,
	//	Type: NewSystemIntType("query_alloc_block_size", 1024, 4294967295, false),
	//	Default: int64(8192),
	// },
	"query_cache_size": &sql.MysqlSystemVariable{
		Name:              "query_cache_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemUintType("query_cache_size", 0, 18446744073709551615),
		Default:           1,
	},
	"query_cache_type": &sql.MysqlSystemVariable{
		Name:              "query_cache_type",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemEnumType("query_cache_type", "OFF", "ON", "DEMAND"),
		Default:           "OFF",
	},
	// "query_prealloc_size": &sql.MysqlSystemVariable{
	//	Name: "query_prealloc_size",
	//	Scope: sql.GetMysqlScope(SystemVariableScope_Both),
	//	Dynamic: true,
	//	SetVarHintApplies: false,
	//	Type: NewSystemUintType("query_prealloc_size", 8192, 18446744073709551615),
	//	Default: uint64(8192),
	// },
	"rand_seed1": &sql.MysqlSystemVariable{
		Name:              "rand_seed1",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Session),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("rand_seed1", -9223372036854775808, 9223372036854775807, false),
		Default:           int64(0),
	},
	// TODO: implement block sizes
	// "range_alloc_block_size": &sql.MysqlSystemVariable{
	//	Name: "range_alloc_block_size",
	//	Scope: sql.GetMysqlScope(SystemVariableScope_Both),
	//	Dynamic: true,
	//	SetVarHintApplies: true,
	//	Type: NewSystemUintType("range_alloc_block_size", 4096, 18446744073709547520),
	//	Default: uint64(4096),
	// },
	"range_optimizer_max_mem_size": &sql.MysqlSystemVariable{
		Name:              "range_optimizer_max_mem_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemUintType("range_optimizer_max_mem_size", 0, 18446744073709551615),
		Default:           uint64(8388608),
	},
	"rbr_exec_mode": &sql.MysqlSystemVariable{
		Name:              "rbr_exec_mode",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemEnumType("rbr_exec_mode", "IDEMPOTENT", "STRICT"),
		Default:           "STRICT",
	},
	"read_buffer_size": &sql.MysqlSystemVariable{
		Name:              "read_buffer_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemIntType("read_buffer_size", 8192, 2147479552, false),
		Default:           int64(131072),
	},
	"read_only": &sql.MysqlSystemVariable{
		Name:              "read_only",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("read_only"),
		Default:           int8(0),
	},
	"read_rnd_buffer_size": &sql.MysqlSystemVariable{
		Name:              "read_rnd_buffer_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemIntType("read_rnd_buffer_size", 1, 2147483647, false),
		Default:           int64(262144),
	},
	"regexp_buffer_size": &sql.MysqlSystemVariable{
		Name:              "regexp_buffer_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemUintType("regexp_buffer_size", 0, 67108864), // 64MB upperbound
		Default:           uint64(524288),
	},
	"regexp_stack_limit": &sql.MysqlSystemVariable{
		Name:              "regexp_stack_limit",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("regexp_stack_limit", 0, 2147483647, false),
		Default:           int64(8000000),
	},
	"regexp_time_limit": &sql.MysqlSystemVariable{
		Name:              "regexp_time_limit",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("regexp_time_limit", 0, 2147483647, false),
		Default:           int64(32),
	},
	"require_row_format": &sql.MysqlSystemVariable{
		Name:              "require_row_format",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Session),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("require_row_format"),
		Default:           int8(0),
	},
	"require_secure_transport": &sql.MysqlSystemVariable{
		Name:              "require_secure_transport",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("require_secure_transport"),
		Default:           int8(0),
	},
	"resultset_metadata": &sql.MysqlSystemVariable{
		Name:              "resultset_metadata",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Session),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemEnumType("resultset_metadata", "FULL", "NONE"),
		Default:           "FULL",
	},
	"secondary_engine_cost_threshold": &sql.MysqlSystemVariable{
		Name:              "secondary_engine_cost_threshold",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Session),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemDoubleType("secondary_engine_cost_threshold", 0, math.MaxFloat64),
		Default:           float64(100000.000000),
	},
	"schema_definition_cache": &sql.MysqlSystemVariable{
		Name:              "schema_definition_cache",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("schema_definition_cache", 256, 524288, false),
		Default:           int64(256),
	},
	"secure_file_priv": &sql.MysqlSystemVariable{
		Name:              "secure_file_priv",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("secure_file_priv"),
		Default:           "",
	},
	"select_into_buffer_size": &sql.MysqlSystemVariable{
		Name:              "select_into_buffer_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemIntType("select_into_buffer_size", 8192, 2147479552, false),
		Default:           int64(131072),
	},
	"select_into_disk_sync": &sql.MysqlSystemVariable{
		Name:              "select_into_disk_sync",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemBoolType("select_into_disk_sync"),
		Default:           int8(0),
	},
	"select_into_disk_sync_delay": &sql.MysqlSystemVariable{
		Name:              "select_into_disk_sync_delay",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemIntType("select_into_disk_sync_delay", 0, 31536000, false),
		Default:           int64(0),
	},
	"server_id": &sql.MysqlSystemVariable{
		Name:              "server_id",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Persist),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.Uint32,
		Default:           uint32(1),
	},
	"server_uuid": &sql.MysqlSystemVariable{
		Name:  "server_uuid",
		Scope: sql.GetMysqlScope(sql.SystemVariableScope_Persist),
		// TODO: MySQL doesn't allow changing @@server_uuid dynamically, but we allow it
		//       here to make setting it easier. MySQL requires that it be set in a config file.
		//       MySQL will also generate a @@server_uuid automatically and persist it
		//       to data_dir/auto.cnf on startup. We should do something similar, but first
		//       we're just providing a way to persist it to the default ~/.dolt/ dir.
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.Text,
		Default:           uuid.New().String(),
	},
	"session_track_gtids": &sql.MysqlSystemVariable{
		Name:              "session_track_gtids",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemEnumType("session_track_gtids", "OFF", "OWN_GTID", "ALL_GTIDS"),
		Default:           "OFF",
	},
	"session_track_schema": &sql.MysqlSystemVariable{
		Name:              "session_track_schema",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("session_track_schema"),
		Default:           int8(1),
	},
	"session_track_state_change": &sql.MysqlSystemVariable{
		Name:              "session_track_state_change",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("session_track_state_change"),
		Default:           int8(0),
	},
	"session_track_system_variables": &sql.MysqlSystemVariable{
		Name:              "session_track_system_variables",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("session_track_system_variables"),
		Default:           "time_zone, autocommit, character_set_client, character_set_results, character_set_connection",
	},
	"session_track_transaction_info": &sql.MysqlSystemVariable{
		Name:              "session_track_transaction_info",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemEnumType("session_track_transaction_info", "OFF", "STATE", "CHARACTERISTICS"),
		Default:           "OFF",
	},
	"sha256_password_auto_generate_rsa_keys": &sql.MysqlSystemVariable{
		Name:              "sha256_password_auto_generate_rsa_keys",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("sha256_password_auto_generate_rsa_keys"),
		Default:           int8(1),
	},
	"sha256_password_private_key_path": &sql.MysqlSystemVariable{
		Name:              "sha256_password_private_key_path",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("sha256_password_private_key_path"),
		Default:           "private_key.pem",
	},
	"sha256_password_proxy_users": &sql.MysqlSystemVariable{
		Name:              "sha256_password_proxy_users",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("sha256_password_proxy_users"),
		Default:           int8(0),
	},
	"sha256_password_public_key_path": &sql.MysqlSystemVariable{
		Name:              "sha256_password_public_key_path",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("sha256_password_public_key_path"),
		Default:           "public_key.pem",
	},
	"shared_memory": &sql.MysqlSystemVariable{
		Name:              "shared_memory",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("shared_memory"),
		Default:           int8(0),
	},
	"shared_memory_base_name": &sql.MysqlSystemVariable{
		Name:              "shared_memory_base_name",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("shared_memory_base_name"),
		Default:           "MYSQL",
	},
	"show_create_table_skip_secondary_engine": &sql.MysqlSystemVariable{
		Name:              "show_create_table_skip_secondary_engine",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Session),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemBoolType("show_create_table_skip_secondary_engine"),
		Default:           int8(0),
	},
	"show_create_table_verbosity": &sql.MysqlSystemVariable{
		Name:              "show_create_table_verbosity",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("show_create_table_verbosity"),
		Default:           int8(0),
	},
	"show_external_procedures": &sql.MysqlSystemVariable{
		Name:              "show_external_procedures",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemBoolType("show_external_procedures"),
		Default:           int8(1),
	},
	"show_old_temporals": &sql.MysqlSystemVariable{
		Name:              "show_old_temporals",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("show_old_temporals"),
		Default:           int8(0),
	},
	"skip_external_locking": &sql.MysqlSystemVariable{
		Name:              "skip_external_locking",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("skip_external_locking"),
		Default:           int8(1),
	},
	"skip_name_resolve": &sql.MysqlSystemVariable{
		Name:              "skip_name_resolve",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("skip_name_resolve"),
		Default:           int8(0),
	},
	"skip_networking": &sql.MysqlSystemVariable{
		Name:              "skip_networking",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("skip_networking"),
		Default:           int8(0),
	},
	"skip_show_database": &sql.MysqlSystemVariable{
		Name:              "skip_show_database",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("skip_show_database"),
		Default:           int8(0),
	},
	"slow_launch_time": &sql.MysqlSystemVariable{
		Name:              "slow_launch_time",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("slow_launch_time", -9223372036854775808, 9223372036854775807, false),
		Default:           int64(2),
	},
	"slow_query_log": &sql.MysqlSystemVariable{
		Name:              "slow_query_log",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("slow_query_log"),
		Default:           int8(0),
	},
	"slow_query_log_file": &sql.MysqlSystemVariable{
		Name:              "slow_query_log_file",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("slow_query_log_file"),
		Default:           "host_name-slow.log",
	},
	"socket": &sql.MysqlSystemVariable{
		Name:              "socket",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("socket"),
		Default:           "/tmp/mysql.sock",
	},
	"sort_buffer_size": &sql.MysqlSystemVariable{
		Name:              "sort_buffer_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemUintType("sort_buffer_size", 32768, 18446744073709551615),
		Default:           uint64(262144),
	},
	"sql_auto_is_null": &sql.MysqlSystemVariable{
		Name:              "sql_auto_is_null",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemBoolType("sql_auto_is_null"),
		Default:           int8(0),
	},
	"sql_big_selects": &sql.MysqlSystemVariable{
		Name:              "sql_big_selects",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemBoolType("sql_big_selects"),
		Default:           int8(1),
	},
	"sql_buffer_result": &sql.MysqlSystemVariable{
		Name:              "sql_buffer_result",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemBoolType("sql_buffer_result"),
		Default:           int8(0),
	},
	"sql_log_bin": &sql.MysqlSystemVariable{
		Name:              "sql_log_bin",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("sql_log_bin"),
		Default:           int8(0),
	},
	"sql_log_off": &sql.MysqlSystemVariable{
		Name:              "sql_log_off",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("sql_log_off"),
		Default:           int8(0),
	},
	"sql_mode": &sql.MysqlSystemVariable{
		Name:              "sql_mode",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemSetType("sql_mode", sql.Collation_utf8mb4_0900_ai_ci, "ALLOW_INVALID_DATES", "ANSI_QUOTES", "ERROR_FOR_DIVISION_BY_ZERO", "HIGH_NOT_PRECEDENCE", "IGNORE_SPACE", "NO_AUTO_VALUE_ON_ZERO", "NO_AUTO_CREATE_USER", "NO_BACKSLASH_ESCAPES", "NO_DIR_IN_CREATE", "NO_ENGINE_SUBSTITUTION", "NO_UNSIGNED_SUBTRACTION", "NO_ZERO_DATE", "NO_ZERO_IN_DATE", "ONLY_FULL_GROUP_BY", "PAD_CHAR_TO_FULL_LENGTH", "PIPES_AS_CONCAT", "REAL_AS_FLOAT", "STRICT_ALL_TABLES", "STRICT_TRANS_TABLES", "TIME_TRUNCATE_FRACTIONAL", "TRADITIONAL", "ANSI"),
		Default:           sql.DefaultSqlMode,
	},
	"sql_notes": &sql.MysqlSystemVariable{
		Name:              "sql_notes",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("sql_notes"),
		Default:           int8(1),
	},
	"sql_quote_show_create": &sql.MysqlSystemVariable{
		Name:              "sql_quote_show_create",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("sql_quote_show_create"),
		Default:           int8(1),
	},
	"sql_require_primary_key": &sql.MysqlSystemVariable{
		Name:              "sql_require_primary_key",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemBoolType("sql_require_primary_key"),
		Default:           int8(0),
	},
	"sql_safe_updates": &sql.MysqlSystemVariable{
		Name:              "sql_safe_updates",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemBoolType("sql_safe_updates"),
		Default:           int8(0),
	},
	"sql_select_limit": &sql.MysqlSystemVariable{
		Name:              "sql_select_limit",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemIntType("sql_select_limit", -9223372036854775808, 9223372036854775807, false),
		Default:           int64(2147483647),
	},
	"sql_warnings": &sql.MysqlSystemVariable{
		Name:              "sql_warnings",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("sql_warnings"),
		Default:           int8(0),
	},
	"ssl_ca": &sql.MysqlSystemVariable{
		Name:              "ssl_ca",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("ssl_ca"),
		Default:           "",
	},
	"ssl_capath": &sql.MysqlSystemVariable{
		Name:              "ssl_capath",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("ssl_capath"),
		Default:           "",
	},
	"ssl_cert": &sql.MysqlSystemVariable{
		Name:              "ssl_cert",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("ssl_cert"),
		Default:           "",
	},
	"ssl_cipher": &sql.MysqlSystemVariable{
		Name:              "ssl_cipher",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("ssl_cipher"),
		Default:           "",
	},
	"ssl_crl": &sql.MysqlSystemVariable{
		Name:              "ssl_crl",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("ssl_crl"),
		Default:           "",
	},
	"ssl_crlpath": &sql.MysqlSystemVariable{
		Name:              "ssl_crlpath",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("ssl_crlpath"),
		Default:           "",
	},
	"ssl_fips_mode": &sql.MysqlSystemVariable{
		Name:              "ssl_fips_mode",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemEnumType("ssl_fips_mode", "OFF", "ON", "STRICT"),
		Default:           "OFF",
	},
	"ssl_key": &sql.MysqlSystemVariable{
		Name:              "ssl_key",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("ssl_key"),
		Default:           "",
	},
	"stored_program_cache": &sql.MysqlSystemVariable{
		Name:              "stored_program_cache",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("stored_program_cache", 16, 524288, false),
		Default:           int64(256),
	},
	"stored_program_definition_cache": &sql.MysqlSystemVariable{
		Name:              "stored_program_definition_cache",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("stored_program_definition_cache", 256, 524288, false),
		Default:           int64(256),
	},
	"strict_mysql_compatibility": &sql.MysqlSystemVariable{
		Name:              "strict_mysql_compatibility",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("strict_mysql_compatibility"),
		Default:           int8(0),
	},
	"super_read_only": &sql.MysqlSystemVariable{
		Name:              "super_read_only",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("super_read_only"),
		Default:           int8(0),
	},
	"syseventlog.facility": &sql.MysqlSystemVariable{
		Name:              "syseventlog.facility",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("syseventlog.facility"),
		Default:           "daemon",
	},
	"syseventlog.include_pid": &sql.MysqlSystemVariable{
		Name:              "syseventlog.include_pid",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("syseventlog.include_pid"),
		Default:           int8(1),
	},
	"syseventlog.tag": &sql.MysqlSystemVariable{
		Name:              "syseventlog.tag",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("syseventlog.tag"),
		Default:           "",
	},
	"system_time_zone": &sql.MysqlSystemVariable{
		Name:              "system_time_zone",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("system_time_zone"),
		Default:           sql.SystemTimezoneName(),
	},
	"table_definition_cache": &sql.MysqlSystemVariable{
		Name:              "table_definition_cache",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("table_definition_cache", 400, 524288, true),
		Default:           int64(-1),
	},
	"table_encryption_privilege_check": &sql.MysqlSystemVariable{
		Name:              "table_encryption_privilege_check",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("table_encryption_privilege_check"),
		Default:           int8(0),
	},
	"table_open_cache": &sql.MysqlSystemVariable{
		Name:              "table_open_cache",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("table_open_cache", 1, 524288, false),
		Default:           int64(4000),
	},
	"table_open_cache_instances": &sql.MysqlSystemVariable{
		Name:              "table_open_cache_instances",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("table_open_cache_instances", 1, 64, false),
		Default:           int64(16),
	},
	"tablespace_definition_cache": &sql.MysqlSystemVariable{
		Name:              "tablespace_definition_cache",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("tablespace_definition_cache", 256, 524288, false),
		Default:           int64(256),
	},
	"temptable_max_mmap": &sql.MysqlSystemVariable{
		Name:              "temptable_max_mmap",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemUintType("temptable_max_mmap", 0, 18446744073709551615),
		Default:           uint64(1073741824),
	},
	"temptable_max_ram": &sql.MysqlSystemVariable{
		Name:              "temptable_max_ram",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemUintType("temptable_max_ram", 2097152, 18446744073709551615),
		Default:           uint64(1073741824),
	},
	"temptable_use_mmap": &sql.MysqlSystemVariable{
		Name:              "temptable_use_mmap",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("temptable_use_mmap"),
		Default:           int8(1),
	},
	"thread_cache_size": &sql.MysqlSystemVariable{
		Name:              "thread_cache_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("thread_cache_size", 0, 16384, true),
		Default:           int64(-1),
	},
	"thread_handling": &sql.MysqlSystemVariable{
		Name:              "thread_handling",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemEnumType("thread_handling", "no-threads", "one-thread-per-connection", "loaded-dynamically"),
		Default:           "one-thread-per-connection",
	},
	"thread_pool_algorithm": &sql.MysqlSystemVariable{
		Name:              "thread_pool_algorithm",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("thread_pool_algorithm", 0, 1, false),
		Default:           int64(0),
	},
	"thread_pool_high_priority_connection": &sql.MysqlSystemVariable{
		Name:              "thread_pool_high_priority_connection",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("thread_pool_high_priority_connection", 0, 1, false),
		Default:           int64(0),
	},
	"thread_pool_max_active_query_threads": &sql.MysqlSystemVariable{
		Name:              "thread_pool_max_active_query_threads",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("thread_pool_max_active_query_threads", 0, 512, false),
		Default:           int64(0),
	},
	"thread_pool_max_unused_threads": &sql.MysqlSystemVariable{
		Name:              "thread_pool_max_unused_threads",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("thread_pool_max_unused_threads", 0, 4096, false),
		Default:           int64(0),
	},
	"thread_pool_prio_kickup_timer": &sql.MysqlSystemVariable{
		Name:              "thread_pool_prio_kickup_timer",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("thread_pool_prio_kickup_timer", 0, 4294967294, false),
		Default:           int64(1000),
	},
	"thread_pool_size": &sql.MysqlSystemVariable{
		Name:              "thread_pool_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("thread_pool_size", 1, 512, false),
		Default:           int64(16),
	},
	"thread_pool_stall_limit": &sql.MysqlSystemVariable{
		Name:              "thread_pool_stall_limit",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("thread_pool_stall_limit", 4, 600, false),
		Default:           int64(6),
	},
	// TODO: implement block sizes
	// "thread_stack": &sql.MysqlSystemVariable{
	//	Name: "thread_stack",
	//	Scope: sql.GetMysqlScope(SystemVariableScope_Global),
	//	Dynamic: false,
	//	SetVarHintApplies: false,
	//	Type: NewSystemUintType("thread_stack", 131072, 18446744073709551615),
	//	Default: uint64(286720),
	// },
	"time_zone": &sql.MysqlSystemVariable{
		Name:              "time_zone",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemStringType("time_zone"),
		Default:           "SYSTEM",
	},
	// TODO: this needs to utilize a function as the value is not static
	"timestamp": &sql.MysqlSystemVariable{
		Name:              "timestamp",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Session),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemDoubleType("timestamp", 1, 2147483647),
		Default:           float64(1000.0),
	},
	"tls_ciphersuites": &sql.MysqlSystemVariable{
		Name:              "tls_ciphersuites",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("tls_ciphersuites"),
		Default:           "",
	},
	"tls_version": &sql.MysqlSystemVariable{
		Name:              "tls_version",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("tls_version"),
		Default:           "TLSv1,TLSv1.1,TLSv1.2,TLSv1.3",
	},
	"tmp_table_size": &sql.MysqlSystemVariable{
		Name:              "tmp_table_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemUintType("tmp_table_size", 1024, 18446744073709551615),
		Default:           uint64(16777216),
	},
	"tmpdir": &sql.MysqlSystemVariable{
		Name:              "tmpdir",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("tmpdir"),
		Default:           sql.GetTmpdirSessionVar(),
	},
	// TODO: implement block sizes
	// "transaction_alloc_block_size": &sql.MysqlSystemVariable{
	//	Name: "transaction_alloc_block_size",
	//	Scope: sql.GetMysqlScope(SystemVariableScope_Both),
	//	Dynamic: true,
	//	SetVarHintApplies: false,
	//	Type: NewSystemIntType("transaction_alloc_block_size", 1024, 131072, false),
	//	Default: int64(8192),
	// },
	"transaction_isolation": &sql.MysqlSystemVariable{
		Name:              "transaction_isolation",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemEnumType("transaction_isolation", "READ-UNCOMMITTED", "READ-COMMITTED", "REPEATABLE-READ", "SERIALIZABLE"),
		Default:           "REPEATABLE-READ",
	},
	"transaction_prealloc_size": &sql.MysqlSystemVariable{
		Name:              "transaction_prealloc_size",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("transaction_prealloc_size", 1024, 131072, false),
		Default:           int64(4096),
	},
	"transaction_read_only": &sql.MysqlSystemVariable{
		Name:              "transaction_read_only",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("transaction_read_only"),
		Default:           int8(0),
	},
	"tx_isolation": &sql.MysqlSystemVariable{
		Name:              "tx_isolation",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemEnumType("tx_isolation", "READ-UNCOMMITTED", "READ-COMMITTED", "REPEATABLE-READ", "SERIALIZABLE"),
		Default:           "REPEATABLE-READ",
	},
	"tx_read_only": &sql.MysqlSystemVariable{
		Name:              "tx_read_only",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("tx_read_only"),
		Default:           int8(0),
	},
	"unique_checks": &sql.MysqlSystemVariable{
		Name:              "unique_checks",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemBoolType("unique_checks"),
		Default:           int8(1),
	},
	"updatable_views_with_limit": &sql.MysqlSystemVariable{
		Name:              "updatable_views_with_limit",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemBoolType("updatable_views_with_limit"),
		Default:           int8(1),
	},
	"uptime": &sql.MysqlSystemVariable{
		Name:              "uptime",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemBoolType("updatable_views_with_limit"),
		Default:           int8(1),
		ValueFunction: func() (interface{}, error) {
			return int(time.Now().Sub(ServerStartUpTime).Seconds()), nil
		},
	},
	"use_secondary_engine": &sql.MysqlSystemVariable{
		Name:              "use_secondary_engine",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Session),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemEnumType("use_secondary_engine", "OFF", "ON", "FORCED"),
		Default:           "ON",
	},
	"validate_password.length": &sql.MysqlSystemVariable{
		Name:              "validate_password.length",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("validate_password.length", 0, 2147483647, false),
		Default:           8,
	},
	"validate_password.number_count": &sql.MysqlSystemVariable{
		Name:              "validate_password.number_count",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("validate_password.number_count", 0, 2147483647, false),
		Default:           1,
	},
	"validate_password.mixed_case_count": &sql.MysqlSystemVariable{
		Name:              "validate_password.mixed_case_count",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("validate_password.mixed_case_count", 0, 2147483647, false),
		Default:           1,
	},
	"validate_password.special_char_count": &sql.MysqlSystemVariable{
		Name:              "validate_password.special_char_count",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("validate_password.special_char_count", 0, 2147483647, false),
		Default:           1,
	},
	"validate_user_plugins": &sql.MysqlSystemVariable{
		Name:              "validate_user_plugins",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemBoolType("validate_user_plugins"),
		Default:           int8(1),
	},
	"version": &sql.MysqlSystemVariable{
		Name:              "version",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("version"),
		Default:           "8.0.31",
	},
	"version_comment": &sql.MysqlSystemVariable{
		Name:              "version_comment",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("version_comment"),
		Default:           "Dolt",
	},
	"version_compile_machine": &sql.MysqlSystemVariable{
		Name:              "version_compile_machine",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("version_compile_machine"),
		Default:           "",
	},
	"version_compile_os": &sql.MysqlSystemVariable{
		Name:              "version_compile_os",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("version_compile_os"),
		Default:           "",
	},
	"version_compile_zlib": &sql.MysqlSystemVariable{
		Name:              "version_compile_zlib",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Global),
		Dynamic:           false,
		SetVarHintApplies: false,
		Type:              types.NewSystemStringType("version_compile_zlib"),
		Default:           "",
	},
	"wait_timeout": &sql.MysqlSystemVariable{
		Name:              "wait_timeout",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: false,
		Type:              types.NewSystemIntType("wait_timeout", 1, 31536000, false),
		Default:           int64(28800),
	},
	"windowing_use_high_precision": &sql.MysqlSystemVariable{
		Name:              "windowing_use_high_precision",
		Scope:             sql.GetMysqlScope(sql.SystemVariableScope_Both),
		Dynamic:           true,
		SetVarHintApplies: true,
		Type:              types.NewSystemBoolType("windowing_use_high_precision"),
		Default:           int8(1),
	},
}

// TODO: need to implement SystemDateTime type
