/*
 * Copyright (c) 2025 Stefan Sperling <stsp@openbsd.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/queue.h>
#include <sys/tree.h>
#include <sys/stat.h>

#include <err.h>
#include <errno.h>
#include <event.h>
#include <fcntl.h>
#include <imsg.h>
#include <limits.h>
#include <pwd.h>
#include <sha1.h>
#include <sha2.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <stdarg.h>
#include <util.h>
#include <unistd.h>

#include "got_error.h"
#include "got_path.h"
#include "got_opentemp.h"
#include "got_object.h"

#include "gotsysd.h"
#include "gotsys.h"

#define SSHD_CONFIG_PATH "/etc/ssh/sshd_config"
#define SSHD_PID_PATH "/var/run/sshd.pid"

static pid_t sshd_pid = -1;
static char *sshd_config_tmppath;
static int sshd_config_tmpfd = -1;

static void
sighdlr(int sig, short event, void *arg)
{
	/*
	 * Normal signal handler rules don't apply because libevent
	 * decouples for us.
	 */

	switch (sig) {
	case SIGHUP:
		break;
	case SIGUSR1:
		break;
	case SIGTERM:
	case SIGINT:
		event_loopexit(NULL);
		break;
	default:
		break;
	}
}

static const struct got_error *
write_sshd_config(void)
{
	const struct got_error *err = NULL;
	int len;
	ssize_t w;
	char buf[1024];

	len = snprintf(buf, sizeof(buf),
	    "# generated by gotsysd, do not edit\n"
	    "AuthorizedKeysFile .ssh/authorized_keys\n"
	    "DisableForwarding yes\n"
	    "PermitTTY no\n"
	    "Subsystem sftp /usr/libexec/sftp-server\n"
	    "\n"
	    "Match Group wheel\n"
	    "\tPermitTTY yes\n"
	    "\n"
	    "Match User anonymous\n"
	    "\tPermitEmptyPasswords yes\n"
	);
	if (len == -1)
		return got_error_from_errno("snprintf");
	if (len == 0)
		return got_error_msg(GOT_ERR_EOF, "empty sshd config");
	if ((size_t)len >= sizeof(buf))
		return got_error_msg(GOT_ERR_NO_SPACE, "sshd config too long");

	w = write(sshd_config_tmpfd, buf, len);
	if (w == -1)
		return got_error_from_errno2("write", sshd_config_tmppath);
	if (w != len)
		return got_error_msg(GOT_ERR_IO, "short write of sshd config");

	if (rename(sshd_config_tmppath, SSHD_CONFIG_PATH) == -1) {
		err = got_error_from_errno3("rename", sshd_config_tmppath,
		    SSHD_CONFIG_PATH);
		unlink(sshd_config_tmppath);
		free(sshd_config_tmppath);
		sshd_config_tmppath = NULL;
		return err;
	}

	close(sshd_config_tmpfd);
	sshd_config_tmpfd = -1;

	free(sshd_config_tmppath);
	sshd_config_tmppath = NULL;

	return err;
}

static void
dispatch_event(int fd, short event, void *arg)
{
	const struct got_error *err = NULL;
	struct gotsysd_imsgev *iev = arg;
	struct imsgbuf *ibuf = &iev->ibuf;
	struct imsg imsg;
	ssize_t n;
	int shut = 0;
	static int flush_and_exit;

	if (event & EV_READ) {
		if ((n = imsgbuf_read(ibuf)) == -1) {
			warn("imsgbuf_read error");
			goto fatal;
		}
		if (n == 0)	/* Connection closed. */
			shut = 1;
	}

	if (event & EV_WRITE) {
		if (imsgbuf_flush(ibuf) == -1) {
			warn("imsgbuf_flush");
			goto fatal;
		} else if (imsgbuf_queuelen(ibuf) == 0 && flush_and_exit) {
			event_del(&iev->ev);
			return;
		}
	}

	for (;;) {
		if ((n = imsg_get(ibuf, &imsg)) == -1) {
			warn("%s: imsg_get", __func__);
			goto fatal;
		}
		if (n == 0)	/* No more messages. */
			break;

		switch (imsg.hdr.type) {
		case GOTSYSD_IMSG_SYSCONF_INSTALL_SSHD_CONFIG:
			err = write_sshd_config();
			if (err)
				break;
			if (sshd_pid != -1 && kill(sshd_pid, SIGHUP) == -1)
				warn("kill %d", sshd_pid);
			if (gotsysd_imsg_compose_event(iev,
			    GOTSYSD_IMSG_SYSCONF_INSTALL_SSHD_CONFIG_DONE,
			    0, -1, NULL, 0) == -1) {
				err = got_error_from_errno("imsg compose "
				    "INSTALL_AUTHORIZED_KEYS_DONE");
				break;
			}
			flush_and_exit = 1;
			break;
		default:
			err = got_error_fmt(GOT_ERR_PRIVSEP_MSG,
			    "unexpected imsg %d", imsg.hdr.type);
			break;
		}

		if (err) {
			warnx("imsg %d: %s", imsg.hdr.type, err->msg);
			gotsysd_imsg_send_error(&iev->ibuf, 0, 0, err);
			shut = 1;
			break;
		}

		imsg_free(&imsg);
	}

	if (!shut) {
		gotsysd_imsg_event_add(iev);
	} else {
fatal:
		/* This pipe is dead. Remove its event handler */
		event_del(&iev->ev);
		event_loopexit(NULL);
	}
}

static void
load_sshd_pid(void)
{
	int fd;
	char buf[16]; /* more than enough columes for INT_MAX + 1*/
	ssize_t r;
	long long pid;
	const char *errstr;

	/*
	 * Our restarting of sshd is best effort so error cases hare are not
	 * fatal. PID-files are a flawed concept, and On a properly system
	 * the sshd config written by gotsys-sshdconfig will already be
	 * affective since the latest system reboot.
	 */

	fd = open(SSHD_PID_PATH, O_RDONLY | O_NOFOLLOW);
	if (fd == -1)
		return;

	r = read(fd, buf, sizeof(buf));
	if (r == -1 || r == 0 || buf[r - 1] != '\n')
		goto done;
	
	buf[r - 1] = '\0';
	pid = strtonum(buf, 2, INT_MAX, &errstr);
	if (errstr != NULL)
		goto done;

	sshd_pid = pid;
done:
	close(fd);
}

static const struct got_error *
apply_unveil_sshd_config(void)
{
	if (unveil(sshd_config_tmppath, "rwc") == -1)
		return got_error_from_errno_fmt("unveil 'rwc' %s",
		    sshd_config_tmppath);
	
	if (unveil(SSHD_CONFIG_PATH, "rwc") == -1)
		return got_error_from_errno_fmt("unveil 'rwc' %s",
		    SSHD_CONFIG_PATH);

	if (unveil(NULL, NULL) == -1)
		return got_error_from_errno("unveil");

	return NULL;
}

int
main(int argc, char **argv)
{
	const struct got_error *error = NULL;
	struct gotsysd_imsgev iev;
	struct event evsigint, evsigterm, evsighup, evsigusr1;
	mode_t mode;

#if 0
	static int attached;

	while (!attached)
		sleep(1);
#endif

#ifndef PROFILE
	if (pledge("stdio rpath wpath cpath fattr proc unveil", NULL) == -1)
		err(1, "pledge");
#endif
	if (geteuid())
		errx(1, "need root privileges");

	if (argc != 1)
		errx(1, "usage: %s", getprogname());

	event_init();

	signal_set(&evsigint, SIGINT, sighdlr, NULL);
	signal_set(&evsigterm, SIGTERM, sighdlr, NULL);
	signal_set(&evsighup, SIGHUP, sighdlr, NULL);
	signal_set(&evsigusr1, SIGUSR1, sighdlr, NULL);
	signal(SIGPIPE, SIG_IGN);

	signal_add(&evsigint, NULL);
	signal_add(&evsigterm, NULL);
	signal_add(&evsighup, NULL);
	signal_add(&evsigusr1, NULL);
	if (imsgbuf_init(&iev.ibuf, GOTSYSD_FILENO_MSG_PIPE) == -1)
		err(1, "imsgbuf_init");

	load_sshd_pid();

	error = got_opentemp_named_fd(&sshd_config_tmppath,
	    &sshd_config_tmpfd, SSHD_CONFIG_PATH, "");
	if (error)
		goto done;

	mode = S_IRUSR|S_IWUSR | S_IRGRP | S_IROTH; /* 0644 */
	if (fchmod(sshd_config_tmpfd, mode) == -1) {
		error = got_error_from_errno_fmt("chmod %o %s", mode,
		    sshd_config_tmppath);
		goto done;
	}

	error = apply_unveil_sshd_config();
	if (error)
		goto done;

	iev.handler = dispatch_event;
	iev.events = EV_READ;
	iev.handler_arg = NULL;
	event_set(&iev.ev, iev.ibuf.fd, EV_READ, dispatch_event, &iev);

	if (gotsysd_imsg_compose_event(&iev, GOTSYSD_IMSG_PROG_READY, 0,
	    -1, NULL, 0) == -1) {
		error = got_error_from_errno("gotsysd_imsg_compose_event");
		goto done;
	}

	event_dispatch();
done:
	if (sshd_config_tmppath && unlink(sshd_config_tmppath) == -1 &&
	    error == NULL)
		error = got_error_from_errno2("unlink", sshd_config_tmppath);
	if (sshd_config_tmpfd != -1 &&
	    close(sshd_config_tmpfd) == -1 && error == NULL)
		error = got_error_from_errno2("close", sshd_config_tmppath);
	free(sshd_config_tmppath);
	if (error) {
		fprintf(stderr, "%s: %s\n", getprogname(), error->msg);
		gotsysd_imsg_send_error(&iev.ibuf, 0, 0, error);
	}
	if (close(GOTSYSD_FILENO_MSG_PIPE) == -1 && error == NULL) {
		error = got_error_from_errno("close");
		fprintf(stderr, "%s: %s\n", getprogname(), error->msg);
	}
	imsgbuf_clear(&iev.ibuf);
	return error ? 1 : 0;
}
