/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { MessageType } from '../types.js';
import { formatDuration } from '../utils/formatters.js';
import { UserAccountManager } from '@google/gemini-cli-core';
import { CommandKind, } from './types.js';
function getUserIdentity(context) {
    const selectedAuthType = context.services.settings.merged.security.auth.selectedType || '';
    const userAccountManager = new UserAccountManager();
    const cachedAccount = userAccountManager.getCachedGoogleAccount();
    const userEmail = cachedAccount ?? undefined;
    const tier = context.services.config?.getUserTierName();
    return { selectedAuthType, userEmail, tier };
}
async function defaultSessionView(context) {
    const now = new Date();
    const { sessionStartTime } = context.session.stats;
    if (!sessionStartTime) {
        context.ui.addItem({
            type: MessageType.ERROR,
            text: 'Session start time is unavailable, cannot calculate stats.',
        });
        return;
    }
    const wallDuration = now.getTime() - sessionStartTime.getTime();
    const { selectedAuthType, userEmail, tier } = getUserIdentity(context);
    const statsItem = {
        type: MessageType.STATS,
        duration: formatDuration(wallDuration),
        selectedAuthType,
        userEmail,
        tier,
    };
    if (context.services.config) {
        const quota = await context.services.config.refreshUserQuota();
        if (quota) {
            statsItem.quotas = quota;
        }
    }
    context.ui.addItem(statsItem);
}
export const statsCommand = {
    name: 'stats',
    altNames: ['usage'],
    description: 'Check session stats. Usage: /stats [session|model|tools]',
    kind: CommandKind.BUILT_IN,
    autoExecute: false,
    action: async (context) => {
        await defaultSessionView(context);
    },
    subCommands: [
        {
            name: 'session',
            description: 'Show session-specific usage statistics',
            kind: CommandKind.BUILT_IN,
            autoExecute: true,
            action: async (context) => {
                await defaultSessionView(context);
            },
        },
        {
            name: 'model',
            description: 'Show model-specific usage statistics',
            kind: CommandKind.BUILT_IN,
            autoExecute: true,
            action: (context) => {
                const { selectedAuthType, userEmail, tier } = getUserIdentity(context);
                context.ui.addItem({
                    type: MessageType.MODEL_STATS,
                    selectedAuthType,
                    userEmail,
                    tier,
                });
            },
        },
        {
            name: 'tools',
            description: 'Show tool-specific usage statistics',
            kind: CommandKind.BUILT_IN,
            autoExecute: true,
            action: (context) => {
                context.ui.addItem({
                    type: MessageType.TOOL_STATS,
                });
            },
        },
    ],
};
//# sourceMappingURL=statsCommand.js.map