"""A custom rule that wraps a crate called to_wrap."""

load("@rules_cc//cc/common:cc_info.bzl", "CcInfo")

# buildifier: disable=bzl-visibility
load("//rust/private:providers.bzl", "BuildInfo", "CrateInfo", "DepInfo", "DepVariantInfo")

# buildifier: disable=bzl-visibility
load("//rust/private:rustc.bzl", "rustc_compile_action")

# buildifier: disable=bzl-visibility
load("//rust/private:utils.bzl", "can_use_metadata_for_pipelining")

_CONTENT = """\
// crate_name: {}
use to_wrap::to_wrap;

pub fn wrap() {{
    to_wrap();
}}
"""

def _wrap_impl(ctx):
    rs_file = ctx.actions.declare_file(ctx.label.name + "_wrapped.rs")
    crate_name = ctx.attr.crate_name if ctx.attr.crate_name else ctx.label.name
    ctx.actions.write(
        output = rs_file,
        content = _CONTENT.format(crate_name),
    )

    toolchain = ctx.toolchains[Label("//rust:toolchain")]

    # Determine unique hash for this rlib
    output_hash = repr(hash(rs_file.path))
    crate_type = "rlib"

    rust_lib_name = "{prefix}{name}-{lib_hash}{extension}".format(
        prefix = "lib",
        name = crate_name,
        lib_hash = output_hash,
        extension = ".rlib",
    )
    rust_metadata_name = "{prefix}{name}-{lib_hash}{extension}".format(
        prefix = "lib",
        name = crate_name,
        lib_hash = output_hash,
        extension = ".rmeta",
    )

    tgt = ctx.attr.target
    deps = [DepVariantInfo(
        crate_info = tgt[CrateInfo] if CrateInfo in tgt else None,
        dep_info = tgt[DepInfo] if DepInfo in tgt else None,
        build_info = tgt[BuildInfo] if BuildInfo in tgt else None,
        cc_info = tgt[CcInfo] if CcInfo in tgt else None,
    )]

    rust_lib = ctx.actions.declare_file(rust_lib_name)
    rust_metadata = None
    if ctx.attr.generate_metadata:
        rust_metadata = ctx.actions.declare_file(rust_metadata_name)
    return rustc_compile_action(
        ctx = ctx,
        attr = ctx.attr,
        toolchain = toolchain,
        crate_info_dict = dict(
            name = crate_name,
            type = crate_type,
            root = rs_file,
            srcs = [rs_file],
            deps = deps,
            proc_macro_deps = [],
            aliases = {},
            output = rust_lib,
            metadata = rust_metadata,
            metadata_supports_pipelining = can_use_metadata_for_pipelining(toolchain, crate_type) and
                                           ctx.attr.generate_metadata,
            owner = ctx.label,
            edition = "2018",
            compile_data = depset([]),
            compile_data_targets = depset([]),
            rustc_env = {},
            is_test = False,
        ),
        output_hash = output_hash,
    )

wrap = rule(
    implementation = _wrap_impl,
    attrs = {
        "crate_name": attr.string(),
        "generate_metadata": attr.bool(default = False),
        "target": attr.label(),
        "_always_enable_metadata_output_groups": attr.label(
            default = Label("//rust/settings:always_enable_metadata_output_groups"),
        ),
        "_error_format": attr.label(
            default = Label("//rust/settings:error_format"),
        ),
        "_process_wrapper": attr.label(
            default = Label("//util/process_wrapper"),
            executable = True,
            allow_single_file = True,
            cfg = "exec",
        ),
    },
    toolchains = [
        "@rules_rust//rust:toolchain",
        config_common.toolchain_type("@bazel_tools//tools/cpp:toolchain_type", mandatory = False),
    ],
    fragments = ["cpp"],
)
