// Copyright 2018-2026 the Deno authors. MIT license.

// TODO(petamoriken): enable prefer-primordials for node polyfills
// deno-lint-ignore-file prefer-primordials

import { Buffer } from "node:buffer";
import {
  emitAfter,
  emitBefore,
  emitDestroy,
  emitInit,
  executionAsyncId,
  newAsyncId,
} from "ext:deno_node/internal/async_hooks.ts";
import process from "node:process";

export const MAX_RANDOM_VALUES = 65536;
export const MAX_SIZE = 4294967295;

function generateRandomBytes(size: number) {
  if (size > MAX_SIZE) {
    throw new RangeError(
      `The value of "size" is out of range. It must be >= 0 && <= ${MAX_SIZE}. Received ${size}`,
    );
  }

  const bytes = Buffer.allocUnsafe(size);

  //Work around for getRandomValues max generation
  if (size > MAX_RANDOM_VALUES) {
    for (let generated = 0; generated < size; generated += MAX_RANDOM_VALUES) {
      globalThis.crypto.getRandomValues(
        bytes.slice(generated, generated + MAX_RANDOM_VALUES),
      );
    }
  } else {
    globalThis.crypto.getRandomValues(bytes);
  }

  return bytes;
}

/**
 * @param size Buffer length, must be equal or greater than zero
 */
export default function randomBytes(size: number): Buffer;
export default function randomBytes(
  size: number,
  cb?: (err: Error | null, buf?: Buffer) => void,
): void;
export default function randomBytes(
  size: number,
  cb?: (err: Error | null, buf?: Buffer) => void,
): Buffer | void {
  if (typeof cb === "function") {
    let err: Error | null = null, bytes: Buffer;
    try {
      bytes = generateRandomBytes(size);
    } catch (e) {
      //NodeJS nonsense
      //If the size is out of range it will throw sync, otherwise throw async
      if (
        e instanceof RangeError &&
        e.message.includes('The value of "size" is out of range')
      ) {
        throw e;
      } else if (e instanceof Error) {
        err = e;
      } else {
        err = new Error("[non-error thrown]");
      }
    }

    // Set up async_hooks tracking
    const asyncId = newAsyncId();
    const triggerAsyncId = executionAsyncId();
    const resource = {};
    emitInit(asyncId, "RANDOMBYTESREQUEST", triggerAsyncId, resource);

    setTimeout(() => {
      emitBefore(asyncId);
      try {
        if (err) {
          cb(err);
        } else {
          cb(null, bytes);
        }
      } catch (callbackErr) {
        // If there's an active domain, emit error to it
        if (process.domain && process.domain.listenerCount("error") > 0) {
          process.domain.emit("error", callbackErr);
        } else {
          throw callbackErr;
        }
      } finally {
        emitAfter(asyncId);
        emitDestroy(asyncId);
      }
    }, 0);
  } else {
    return generateRandomBytes(size);
  }
}
