/*
  str_ops.c:

  Copyright (C) 2005, 2006 Istvan Varga
  (C) 2005       Matt J. Ingalls, John ffitch
  (C) 2013   V Lazzarini (new string code)

  This file is part of Csound.

  The Csound Library is free software; you can redistribute it
  and/or modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  Csound is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with Csound; if not, write to the Free Software
  Foundation, Inc., 31 Milk Street, #960789, Boston, MA, 02196, USA
*/

#include "csoundCore.h"
#define CSOUND_STR_OPS_C    1
#include "str_ops.h"
#include <ctype.h>
#ifdef HAVE_CURL
#include <curl/curl.h>
#include "corfile.h"
#endif

#define STRSMAX 8

// VL since sprintf() is deprecated in modern C
// we have to assume all SANE systems have snprintf()
// there was a check for HAVE_SNPRINTF in this file
// which was basically checking for C99.
// It was removed as nowhere else in the
// system we check for this.

int32_t is_perf_thread(OPDS *p){
  return p->insdshead->init_done;
}


int32_t eqs(CSOUND *csound, EQS_OP *p) {
  *p->r = strcmp(p->a->data, p->b->data) ? 0 : 1;
  return OK;
}


int32_t s_opcode(CSOUND *csound, STRGET_OP *p){
  if (p->r->size < DEFAULT_STRING_SIZE){
    p->r->data = (char *) csound->ReAlloc(csound, p->r->data, DEFAULT_STRING_SIZE);
    p->r->size = DEFAULT_STRING_SIZE;
  }
  snprintf(p->r->data, p->r->size, "%f", *p->indx);
  return OK;
}

int32_t s_opcode_k(CSOUND *csound, STRGET_OP *p){
  snprintf(p->r->data, p->r->size, "%f", *p->indx);
  p->r->timestamp = p->h.insdshead->kcounter;
  return OK;
}

/* strset by John ffitch */
static void str_set(CSOUND *csound, int32_t ndx, const char *s)
{
  if (UNLIKELY(csound->strsets == NULL)) {
    csound->strsmax = STRSMAX;
    csound->strsets = (char **) csound->Calloc(csound, (csound->strsmax + 1)
					       * sizeof(char*));
  }
  if (UNLIKELY(ndx > (int32_t) csound->strsmax)) {
    int32_t   i, newmax;
    /* assumes power of two STRSMAX */
    newmax = (ndx | (STRSMAX - 1)) + 1;
    csound->strsets = (char**) csound->ReAlloc(csound, csound->strsets,
					       (newmax + 1) * sizeof(char*));
    for (i = (csound->strsmax + 1); i <= newmax; i++)
      csound->strsets[i] = NULL;
    csound->strsmax = newmax;
  }
  if (UNLIKELY(ndx < 0))  {  /* -ve index */
    csound->InitError(csound, Str("illegal strset index"));
    return;
  }

  if (csound->strsets[ndx] != NULL) {
    if (strcmp(s, csound->strsets[ndx]) == 0)
      return;
    if (UNLIKELY(csound->oparms->msglevel & CS_WARNMSG)) {
      csound->Warning(csound, Str("strset index conflict at %d"), ndx);
      csound->Warning(csound, Str("previous value: '%s', replaced with '%s'"),
		      csound->strsets[ndx], s);
    }
    csound->Free(csound, csound->strsets[ndx]);
  }
  csound->strsets[ndx] = (char*) csound->Malloc(csound, strlen(s) + 1);
  strcpy(csound->strsets[ndx], s);
  if ((csound->oparms->msglevel & 7) == 7)
    csound->Message(csound, "Strsets[%d]: '%s'\n", ndx, s);
}

int32_t strset_init(CSOUND *csound, STRSET_OP *p)
{
  str_set(csound, (int32_t) MYFLT2LRND(*p->indx), p->str->data);
  return OK;
}

/* for argdecode.c */

void strset_option(CSOUND *csound, char *s)
{
  int32_t indx = 0;

  if (UNLIKELY(!isdigit(*s))) {
    csound->Warning(csound, Str("--strset: invalid format"));
    return;
  }
  do {
    indx = (indx * 10) + (int32_t) (*s++ - '0');
  } while (isdigit(*s));
  if (UNLIKELY(*s++ != '=')){
    csound->Warning(csound, Str("--strset: invalid format"));
    return;
  }
  str_set(csound, indx, s);
}

int32_t strget_init(CSOUND *csound, STRGET_OP *p)
{
  int32_t   indx;
  p->r->timestamp = 0;
  if (IsStringCode(*(p->indx))) {
    char *ss = csound->init_event->strarg;
    if (ss == NULL)
      return OK;
    ss = csoundGetArgString(csound, *p->indx);
    size_t len = strlen(ss);
    if (len >= p->r->size) {
      char *temp = csound->ReAlloc(csound, p->r->data, len + 1);
      if (UNLIKELY(temp == NULL)) {
        return csoundInitError(csound, "strget_init: allocation failure");
      }
      /* Only update the structure after successful reallocation */
      p->r->data = temp;
      p->r->size = len + 1;
    }
    if (p->r->data != NULL) {
      strNcpy(p->r->data, ss, p->r->size);
      p->r->data[p->r->size - 1] = '\0';
    }
    return OK;
  }
  indx = (int32_t)((double)*(p->indx) + (*(p->indx) >= FL(0.0) ? 0.5 : -0.5));
  if (indx < 0 || indx > (int32_t) csound->strsmax ||
      csound->strsets == NULL || csound->strsets[indx] == NULL)
    return OK;
  if (UNLIKELY(strlen(csound->strsets[indx]) >= p->r->size)){
    int32_t size = (int32_t) strlen(csound->strsets[indx]);
    p->r->data = csound->ReAlloc(csound, p->r->data, size + 1);
    p->r->size = size + 1;
  }
  strcpy((char*) p->r->data, csound->strsets[indx]);
  return OK;
}

static CS_NOINLINE int32_t StrOp_ErrMsg(void *p, const char *msg)
{
  CSOUND      *csound = ((OPDS*) p)->insdshead->csound;
  const char  *opname = GetOpcodeName(p);

  if (UNLIKELY(csound->ids != NULL && csound->ids->insdshead == csound->curip))
    return csound->InitError(csound, "%s: %s", opname, Str(msg));
  else if (UNLIKELY(((OPDS*) p)->insdshead->pds != NULL))
    return csound->PerfError(csound, (OPDS*)p,
			     "%s: %s", opname, Str(msg));
  else
    csound->Warning(csound, "%s: %s", opname, Str(msg));

  return NOTOK;
}

int32_t strassign_k(CSOUND *csound, STRCPY_OP *p) {
  if(p->r != p->str) {
  if((uint64_t)p->str->timestamp == p->h.insdshead->kcounter) {
  CS_TYPE *strType = GetTypeForArg(p->str);
  strType->copyValue(csound, strType, p->r, p->str, p->h.insdshead);
  //printf("copy \n");
  }
  }
  return  OK;
}

int32_t strcpy_opcode_S(CSOUND *csound, STRCPY_OP *p) {
  if(p->r != p->str) {
  CS_TYPE *strType = GetTypeForArg(p->str);
  strType->copyValue(csound, strType, p->r, p->str,  p->h.insdshead);
  }
  return  OK;
}


/* this opcode is i-time only, so no need to make
   any adjustments regarding update counts
*/
extern char* get_strarg(CSOUND *csound, MYFLT p, char *strarg);
int32_t strcpy_opcode_p(CSOUND *csound, STRGET_OP *p)
{
  if (IsStringCode(*p->indx)) {
    char *ss;
    ss = csoundGetArgString(csound, *p->indx);
    if (ss == NULL){
      if (UNLIKELY(((OPDS*) p)->insdshead->pds != NULL))
        return csoundPerfError(csound, (OPDS*)p,
                               Str("NULL string\n"));
      else
        return csoundInitError(csound, Str("NULL string\n"));
    }
    size_t len = strlen(ss);
    if (len >= (size_t) p->r->size) {
      void *newp = csound->ReAlloc(csound, p->r->data, len + 1);
      if (newp == NULL) {
        /* ReAlloc failed, keep the original buffer and return error */
        if (UNLIKELY(((OPDS*) p)->insdshead->pds != NULL)) {
          return csoundPerfError(csound, (OPDS*)p,
                                 Str("strcpy_opcode_p: Memory allocation failed\n"));
        } else {
          return csoundInitError(csound, "strcpy_opcode_p: allocation failure");
        }
      }
      /* Only update the structure after successful reallocation */
      p->r->data = newp;
      p->r->size = (int32_t)(len + 1);
    }

    strNcpy(p->r->data, ss, p->r->size);
    p->r->data[p->r->size - 1] = '\0';

  }
  else {
    p->r->data = csound->StringArg2Name(csound, p->r->data, p->indx, "soundin.", 0);
    p->r->size = (int32_t) strlen(p->r->data) + 1;
  }
  return OK;
}

int32_t str_changed(CSOUND *csound, STRCHGD *p)
{
  if (p->mem != NULL)
    csound->Free(csound, p->mem);
  p->mem = csoundStrdup(csound, p->str->data);
  *p->r = 0;
  return OK;
}

int32_t str_changed_k(CSOUND *csound, STRCHGD *p)
{
    if (p->str->data && ( p->mem == NULL || strcmp(p->str->data, p->mem)!=0)) {
    csound->Free(csound, p->mem);
    p->mem = csoundStrdup(csound, p->str->data);
    *p->r = 1;
    }
    else *p->r = 0;
  return OK;
}

/* strcat */
/* rewritten VL Feb 22 */
int32_t strcat_opcode(CSOUND *csound, STRCAT_OP *p)
{
  int64_t kcnt = p->h.insdshead->kcounter;
  size_t size = (int32_t) strlen(p->str1->data) + strlen(p->str2->data);
  if(size >= MAX_STRINGDAT_SIZE) {
     if(is_perf_thread(&p->h))
     return csound->PerfError(csound, &p->h,
		       "strcatk: requested alloc size exceeds max (%u bytes)",
		       MAX_STRINGDAT_SIZE);
     else
     return csound->InitError(csound,
		       "strcat: requested alloc size exceeds max (%u bytes)",
		       MAX_STRINGDAT_SIZE);
   }
  p->r->timestamp = kcnt;
  if(p->str1 != p->r && p->str2 != p->r) {
    // VL: simple case, inputs are not the output
    if (size >= p->r->size) {
      size_t alloc_size;
      if (size > (SIZE_MAX - 1) / 2) {
        if(is_perf_thread(&p->h))
          return csound->PerfError(csound, &p->h,
                         "strcatk: allocation size overflow");
        else
          return csound->InitError(csound, "strcat: allocation size overflow");
      }
      alloc_size = 2 * size + 1; // +1 for null terminator
      char *temp = csound->ReAlloc(csound, p->r->data, alloc_size);
      if (UNLIKELY(temp == NULL)) {
        if(is_perf_thread(&p->h))
          return csound->PerfError(csound, &p->h,
                         "strcatk: allocation failure");
        else
          return csound->InitError(csound, "strcat: allocation failure");
      }
      p->r->data = temp;
      p->r->size = alloc_size;
    }
    memcpy(p->r->data, p->str1->data, p->str1->size);
    strcat(p->r->data, p->str2->data);
    return OK;
  }
  else if(p->str1 == p->r && p->str2 != p->r) {
     if(size >= p->r->size) {
       size_t alloc_size;
       if (size > SIZE_MAX / 2) {
         if(is_perf_thread(&p->h))
           return csound->PerfError(csound, &p->h,
                          "strcatk: allocation size overflow");
         else
           return csound->InitError(csound, "strcat: allocation size overflow");
       }
       alloc_size = 2 * size;
       char *temp = csound->ReAlloc(csound, p->r->data, alloc_size);
       if (UNLIKELY(temp == NULL)) {
         if(is_perf_thread(&p->h))
           return csound->PerfError(csound, &p->h,
                          "strcatk: allocation failure");
         else
           return csound->InitError(csound, "strcat: allocation failure");
       }
       p->r->data = temp;
       p->r->size = alloc_size;
    }
     strcat((char*) p->r->data, p->str2->data);
     return OK;
    }
  else if(p->str1 != p->r && p->str2 == p->r) {
    // the bad case where str2 == r
    char *ostr = csoundStrdup(csound, p->str2->data);
   if(size >= p->r->size) {
       size_t alloc_size;
       if (size > SIZE_MAX / 2) {
         if(is_perf_thread(&p->h))
           return csound->PerfError(csound, &p->h,
                          "strcatk: allocation size overflow");
         else
           return csound->InitError(csound, "strcat: allocation size overflow");
       }
       alloc_size = 2 * size;
       char *temp = csound->ReAlloc(csound, p->r->data, alloc_size);
       if (UNLIKELY(temp == NULL)) {
         if(is_perf_thread(&p->h))
           return csound->PerfError(csound, &p->h,
                          "strcatk: allocation failure");
         else
           return csound->InitError(csound, "strcat: allocation failure");
       }
       p->r->data = temp;
       p->r->size = alloc_size;
    }
     memcpy(p->r->data, p->str1->data, p->r->size - 1);
     strcat(p->r->data,ostr);
     csound->Free(csound, ostr);
     return OK;
    }
  else {
    // the bad case where (str1 == str2) == r
   char *ostr = csoundStrdup(csound, p->str2->data);
   if (size >= p->r->size) {
        size_t alloc_size;
        if (size > SIZE_MAX / 2) {
          if(is_perf_thread(&p->h))
            return csound->PerfError(csound, &p->h,
                           "strcatk: allocation size overflow");
          else
            return csound->InitError(csound, "strcat: allocation size overflow");
        }
        alloc_size = 2 * size;
        char *temp = csound->ReAlloc(csound, p->r->data, alloc_size);
        if (UNLIKELY(temp == NULL)) {
          if(is_perf_thread(&p->h))
            return csound->PerfError(csound, &p->h,
                           "strcatk: allocation failure");
          else
            return csound->InitError(csound, "strcat: allocation failure");
        }
        p->r->data = temp;
        p->r->size = alloc_size;
    }
   strcat(p->r->data, ostr);
   csound->Free(csound, ostr);
   return OK;
  }
}

/* strcmp */
int32_t strcmp_opcode(CSOUND *csound, STRCMP_OP *p)
{
    int32_t     i;
    if (p->str1->data == NULL || p->str2->data == NULL){
      if (UNLIKELY(((OPDS*) p)->insdshead->pds != NULL))
        return csoundPerfError(csound, (OPDS*)p, Str("NULL string\n"));
      else return csoundInitError(csound, Str("NULL string\n"));
    }

    *(p->r) = FL(0.0);
    if (p->str1 == p->str2)
      return OK;
    i = strcmp((char*) p->str1->data, (char*) p->str2->data);
    if (i < 0)
      *(p->r) = FL(-1.0);
    else if (i > 0)
      *(p->r) = FL(1.0);

  return OK;
}

/* perform a sprintf-style format -- based on code by Matt J. Ingalls */

static CS_NOINLINE int32_t
sprintf_opcode_(CSOUND *csound,
		void *p,          /* opcode data structure pointer       */
		STRINGDAT *str,   /* pointer to space for output string  */
		const char *fmt,  /* format string                       */
		MYFLT **kvals,    /* array of argument pointers          */
		int32_t numVals,      /* number of arguments             */
		int32_t strCode)      /* bit mask for string arguments   */
{
  int32_t     len = 0;
  char    *strseg, *outstring = str->data;
  MYFLT   *parm = NULL;
  int32_t     i = 0, j = 0, n;
  const char  *segwaiting = NULL;
  int32_t     maxChars, siz = (int32_t) strlen(fmt) + numVals*7 + 1;

  for (i = 0; i < numVals; i++) {
    if (UNLIKELY(IS_ASIG_ARG(kvals[i]))) {
      return StrOp_ErrMsg(p, Str("a-rate argument not allowed"));
    }
  }

  if (UNLIKELY((int32_t) ((OPDS*) p)->optext->t.inArgCount > 31)){
    StrOp_ErrMsg(p, Str("too many arguments"));
    return NOTOK;
  }
  if (numVals==0) {
    strcpy(str->data, fmt);
    return OK;
  }

  strseg = csound->Malloc(csound, siz);
  i = 0;

  while (1) {
    if (UNLIKELY(i >= siz)) {
      // return StrOp_ErrMsg(p, "format string too long");
      siz *= 2;
      strseg = csound->ReAlloc(csound, strseg, siz);
    }
    if (*fmt != '%' && *fmt != '\0') {
      strseg[i++] = *fmt++;
      continue;
    }
    if (fmt[0] == '%' && fmt[1] == '%') {
      strseg[i++] = *fmt++;   /* Odd code: %% is usually % and as we
				 know the value of *fmt the loads are
				 unnecessary */
      strseg[i++] = *fmt++;

      continue;
    }

    /* if already a segment waiting, then lets print it */
    if (segwaiting != NULL) {

      maxChars = (int32_t)(str->size - len);
      strseg[i] = '\0';
      if (UNLIKELY(numVals <= 0)) {
	csound->Free(csound, strseg);
	return StrOp_ErrMsg(p, Str("insufficient arguments for format"));
      }
      numVals--;
      /* if (UNLIKELY((*segwaiting == 's' && !(strCode & 1)) || */
      /*              (*segwaiting != 's' && (strCode & 1)))) { */
      /*   return StrOp_ErrMsg(p, "argument type inconsistent with format"); */
      /* } */
      strCode >>= 1;
      parm = kvals[j++];

      switch (*segwaiting) {
      case 'd':
      case 'i':
      case 'o':
      case 'x':
      case 'X':
      case 'u':
      case 'c':
	if (strlen(strseg) + 24 > (size_t)maxChars) {
	  size_t offs = outstring - str->data;
	  str->data = csound->ReAlloc(csound, str->data,
				      str->size  + 24);
	  if(str->data == NULL) {
	    return StrOp_ErrMsg(p, Str("memory allocation failure"));
	  }
	  str->size += 24;
	  maxChars += 24;
	  outstring = str->data + offs;
	}
	n = snprintf(outstring, maxChars, strseg, (int32_t) MYFLT2LRND(*parm));
	break;
      case 'e':
      case 'E':
      case 'f':
      case 'F':
      case 'g':
      case 'G':
	{
	//printf("%d %d \n", str->size, strlen(str->data));
	int32_t max_double_chars = 18;
	if (strlen(strseg) + max_double_chars  > (uint32_t)maxChars) {
	  size_t offs = outstring - str->data;
	  str->data = csound->ReAlloc(csound, str->data, str->size  + max_double_chars );
	  if(str->data == NULL) {
	    return StrOp_ErrMsg(p, Str("memory allocation failure"));
	  }
	  str->size += max_double_chars;
	  maxChars += max_double_chars;
	  outstring = str->data + offs;
	  //printf("maxchars = %d  %s\n", maxChars, strseg);
	}
	//printf("%d %d \n", str->size, strlen(str->data));
	n = snprintf(outstring, maxChars, strseg, (double)*parm);
	}
	break;
      case 's':
	if (LIKELY(IS_STR_ARG(parm))) {
          STRINGDAT *sparm = (STRINGDAT*)parm;
	  if (UNLIKELY(sparm->data == str->data)) {
	    csound->Free(csound, strseg);
	    return StrOp_ErrMsg(p, Str("output argument may not be "
				       "the same as any of the input args"));
	  }
	  if (UNLIKELY(strlen(sparm->data)+strlen(strseg) >= (uint32_t)maxChars)) {
	    size_t offs = outstring - str->data;
	    size_t xtra = strlen(sparm->data) + strlen(strseg) + 1;
	    str->data = csound->ReAlloc(csound, str->data,
					str->size + xtra);
	    if(str->data == NULL){
	      return StrOp_ErrMsg(p, Str("memory allocation failure"));
	    }
	    str->size += xtra;
	    maxChars += xtra;
	    outstring = str->data + offs;
	  }
	  n = snprintf(outstring, maxChars, strseg, sparm->data);
	}
	else return StrOp_ErrMsg(p, Str("Not string type for %%s"));
	break;
      default:
	csound->Free(csound, strseg);
	return StrOp_ErrMsg(p, Str("invalid format string"));
      }
      if (n < 0 || n >= maxChars) {
	/* safely detected excess string length */
	size_t offs = outstring - str->data;
	str->data = csound->ReAlloc(csound, str->data, maxChars*2);
	if (str->data == NULL) {
	  return StrOp_ErrMsg(p, Str("memory allocation failure"));
	}
	outstring = str->data + offs;
	str->size = maxChars*2;
	// VL: Coverity says this is unused. (which is true)
	// maxChars += str->size;

      }
      outstring += n;
      len += n;
      i = 0;
    }

    if (*fmt == '\0')
      break;
    /* copy the '%' */
    strseg[i++] = *fmt++;
    /* find the format code */
    segwaiting = fmt;

    while (!isalpha(*segwaiting) && *segwaiting != '\0')
      segwaiting++;
  }
  if (UNLIKELY(numVals > 0)) {
    csound->Free(csound, strseg);
    return StrOp_ErrMsg(p, Str("too many arguments for format"));
  }
  csound->Free(csound, strseg);
  return OK;
}

int32_t sprintf_opcode(CSOUND *csound, SPRINTF_OP *p)
{
  size_t size = strlen(p->sfmt->data) + 13*((size_t) p->INOCOUNT);
  //printf("%d %d \n", p->r->size, strlen(p->r->data));
  if (p->r->data == NULL || p->r->size < size) {
    /* this 10 is 1n incorrect guess which is OK with numbers*/
    p->r->data = csound->Calloc(csound, size);
    p->r->size = size;
  }
  if (UNLIKELY(sprintf_opcode_(csound, p, p->r,
			       (char*) p->sfmt->data, &(p->args[0]),
			       (int32_t) p->INOCOUNT - 1,0) == NOTOK)) {
    ((char*) p->r->data)[0] = '\0';
    return NOTOK;
  }
  p->r->timestamp = p->h.insdshead->kcounter;
  return OK;
}

static CS_NOINLINE int32_t printf_opcode_(CSOUND *csound, PRINTF_OP *p)
{
  STRINGDAT buf;
  int32_t   err;
  buf.size = /*strlen(p->sfmt->data) +*/ 3072;
  buf.data = csound->Calloc(csound, buf.size);

  err = sprintf_opcode_(csound, p, &buf, (char*) p->sfmt->data, &(p->args[0]),
			(int32_t) p->INOCOUNT - 2,0);
  if (LIKELY(err == OK))
    csound->MessageS(csound, CSOUNDMSG_ORCH, "%s", buf.data);
  csound->Free(csound, buf.data);

  return err;
}

int32_t printf_opcode_init(CSOUND *csound, PRINTF_OP *p)
{
  if (*p->ktrig > FL(0.0))
    return (printf_opcode_(csound, p));
  return OK;
}

int32_t printf_opcode_set(CSOUND *csound, PRINTF_OP *p)
{
  (void) csound;
  p->prv_ktrig = FL(0.0);
  return OK;
}

int32_t printf_opcode_perf(CSOUND *csound, PRINTF_OP *p)
{
  MYFLT ktrig = *p->ktrig;
  if (ktrig == p->prv_ktrig)
    return OK;
  p->prv_ktrig = ktrig;
  if (ktrig > FL(0.0))
    return (printf_opcode_(csound, p));
  return OK;
}

int32_t puts_opcode_init(CSOUND *csound, PUTS_OP *p)
{
    if (*p->ktrig > FL(0.0)) {
        if (!p->noNewLine)
          csound->Message(csound, "%s\n", (char*) p->str->data);
        else
          csound->Message(csound, "%s", (char*) p->str->data);
    }
    p->prv_ktrig = *p->ktrig;

    return OK;
}

int32_t puts_opcode_perf(CSOUND *csound, PUTS_OP *p)
{
  if (*p->ktrig != p->prv_ktrig && *p->ktrig > FL(0.0)) {
    p->prv_ktrig = *p->ktrig;
    if (!p->noNewLine)
      csound->Message(csound, "%s\n", (char*) p->str->data);
    else
      csound->Message(csound, "%s", (char*) p->str->data);
  }

  return OK;
}

int32_t strtod_opcode_p(CSOUND *csound, STRTOD_OP *p)
{
  char    *s = NULL, *tmp;
  double  x;

  if (IsStringCode(*p->str))
    s = csoundGetArgString(csound, *p->str);
  else {
    int32_t ndx = (int32_t) MYFLT2LRND(*p->str);
    if (ndx >= 0 && ndx <= (int32_t) csound->strsmax && csound->strsets != NULL)
      s = csound->strsets[ndx];
  }
  if (UNLIKELY(s == NULL))
    return StrOp_ErrMsg(p, Str("empty string"));
  while (isblank(*s)) s++;
  if (UNLIKELY(*s == '\0'))
    return StrOp_ErrMsg(p, Str("empty string"));
  x = csoundStrtod(s, &tmp);
  if (UNLIKELY(*tmp != '\0'))
    return StrOp_ErrMsg(p, Str("invalid format"));
  *p->indx = (MYFLT) x;

  return OK;
}

int32_t strtod_opcode_S(CSOUND *csound, STRSET_OP *p)
{
  IGN(csound);
  char    *s = NULL, *tmp;
  double  x;
  s = (char*) p->str->data;
  while (isblank(*s)) s++;
  if (UNLIKELY(*s == '\0'))
    return StrOp_ErrMsg(p, Str("empty string"));
  x = csoundStrtod(s, &tmp);
  if (UNLIKELY(*tmp != '\0'))
    return StrOp_ErrMsg(p, Str("invalid format"));
  *p->indx = (MYFLT) x;

  return OK;
}

int32_t strtol_opcode_S(CSOUND *csound, STRSET_OP *p)
{

  IGN(csound);
  char  *s = NULL;
  int32_t   sgn = 0, radix = 10;
  int32_t  x = 0;

  s = (char*) p->str->data;
  while (isblank(*s)) s++;
  if (UNLIKELY(*s == '\0'))
    return StrOp_ErrMsg(p, Str("empty string"));
  if (*s == '+') s++;
  else if (*s == '-') sgn++, s++;
  if (*s == '0') {
    if (s[1] == 'x' || s[1] == 'X')
      radix = 16, s += 2;
    else if (s[1] != '\0')
      radix = 8, s++;
    else {
      *p->indx = FL(0.0);
      return OK;
    }
  }
  if (UNLIKELY(*s == '\0'))
    return StrOp_ErrMsg(p, Str("invalid format"));
  switch (radix) {
  case 8:
    while (*s >= '0' && *s <= '7') x = (x * 8L) + (int32_t) (*s++ - '0');
    break;
  case 10:
    while (isdigit(*s)) x = (x * 10L) + (int32_t) (*s++ - '0');
    break;
  default:
    while (1) {
      if (isdigit(*s))
	x = (x * 16L) + (int32_t) (*s++ - '0');
      else if (*s >= 'A' && *s <= 'F')
	x = (x * 16L) + (int32_t) (*s++ - 'A') + 10L;
      else if (*s >= 'a' && *s <= 'f')
	x = (x * 16L) + (int32_t) (*s++ - 'a') + 10L;
      else
	break;
    }
  }
  if (UNLIKELY(*s != '\0'))
    return StrOp_ErrMsg(p, Str("invalid format"));
  if (sgn) x = -x;
  *p->indx = (MYFLT) x;

  return OK;
}


int32_t strtol_opcode_p(CSOUND *csound, STRTOD_OP *p)
{
  char  *s = NULL;
  int32_t   sgn = 0, radix = 10;
  int32_t   x = 0L;

  if (IsStringCode(*p->str))
    s = csoundGetArgString(csound, *p->str);
  else {
    int32_t ndx = (int32_t) MYFLT2LRND(*p->str);
    if (ndx >= 0 && ndx <= (int32_t) csound->strsmax && csound->strsets != NULL)
      s = csound->strsets[ndx];
  }
  if (UNLIKELY(s == NULL))
    return StrOp_ErrMsg(p, Str("empty string"));

  while (isblank(*s)) s++;
  if (UNLIKELY(*s == '\0'))
    return StrOp_ErrMsg(p, Str("empty string"));
  if (*s == '+') s++;
  else if (*s == '-') sgn++, s++;
  if (*s == '0') {
    if (s[1] == 'x' || s[1] == 'X')
      radix = 16, s += 2;
    else if (s[1] != '\0')
      radix = 8, s++;
    else {
      *p->indx = FL(0.0);
      return OK;
    }
  }
  if (UNLIKELY(*s == '\0'))
    return StrOp_ErrMsg(p, Str("invalid format"));
  switch (radix) {
  case 8:
    while (*s >= '0' && *s <= '7') x = (x * 8L) + (int32_t) (*s++ - '0');
    break;
  case 10:
    while (isdigit(*s)) x = (x * 10L) + (int32_t) (*s++ - '0');
    break;
  default:
    while (1) {
      if (isdigit(*s))
	x = (x * 16L) + (int32_t) (*s++ - '0');
      else if (*s >= 'A' && *s <= 'F')
	x = (x * 16L) + (int32_t) (*s++ - 'A') + 10L;
      else if (*s >= 'a' && *s <= 'f')
	x = (x * 16L) + (int32_t) (*s++ - 'a') + 10L;
      else
	break;
    }
  }
  if (UNLIKELY(*s != '\0'))
    return StrOp_ErrMsg(p, Str("invalid format"));
  if (sgn) x = -x;
  *p->indx = (MYFLT) x;

  return OK;
}

/**
 * Sdst    strsub      Ssrc[, istart[, iend]]
 * Sdst    strsubk     Ssrc, kstart, kend
 *
 * Extract a part of Ssrc, from istart to iend; if istart or iend is
 * less than 0, or greater than the length of the source string, it is
 * interpreted as the end of the source string. istart > iend will
 * reverse the string. The default parameters are istart = 0, iend = -1.
 */

int32_t strsub_opcode(CSOUND *csound, STRSUB_OP *p)
{
  int64_t kcnt = p->h.insdshead->kcounter;
    const char  *src;
    char        *dst;
    int32_t    strt, end;
    size_t  rev = 0;
    size_t       len, i;

    if (p->Ssrc->data == NULL) return NOTOK;
    if (p->Sdst->size < p->Ssrc->size) {
      size_t size = p->Ssrc->size;
      char *temp = csound->ReAlloc(csound, p->Sdst->data, size);
      if (UNLIKELY(temp == NULL)) {
        /* ReAlloc failed, keep the original buffer and return error */
        return StrOp_ErrMsg(p, Str("strsub: memory allocation failure"));
      }
      p->Sdst->data = temp;
      p->Sdst->size = size;
    }

    src = (char*) p->Ssrc->data;
    dst = (char*) p->Sdst->data;
    len = (int32_t) strlen(src);
#if defined(MSVC) || (defined(__GNUC__) && defined(__i386__))
    strt = (int32_t) MYFLT2LRND(*(p->istart));
    end = (int32_t) MYFLT2LRND(*(p->iend));
#else
    strt = (int32_t) (*(p->istart) + FL(1.5)) - 1;
    end = (int32_t) (*(p->iend) + FL(1.5)) - 1;
#endif
    if (strt < 0 || (size_t)strt > len)
      strt = (int32_t) len;
    if (end < 0 || (size_t)end > len)
      end = (int32_t) len;
    if (strt == end) {
      /* trivial case: empty output */
      dst[0] = '\0';
      return OK;
    }
    if (strt > end) {
      size_t   tmp = strt;
      /* reverse output */
      strt = (int32_t) end;
      end = (int32_t) tmp;
      rev = 1;
    }

    src += strt;
    len = end - strt;
    if (UNLIKELY(len >=  p->Sdst->size)) {
      char *temp = csound->ReAlloc(csound, p->Sdst->data, len+1);
      if (UNLIKELY(temp == NULL)) {
        /* ReAlloc failed, keep the original buffer and return error */
        return StrOp_ErrMsg(p, Str("memory allocation failure"));
      }
      p->Sdst->data = temp;
      p->Sdst->size = len+1;
      dst = (char*) p->Sdst->data;
    }
    i = 0;
    if (!rev || p->Sdst->data == p->Ssrc->data) {
      /* copying in forward direction is safe */
      /* even if Ssrc and Sdst are the same */
      do {
        dst[i] = src[i];
      } while (++i < len);
      dst[i] = '\0';
      if (rev) {
        size_t   j;
        /* if the destination string variable is the same as the source, */
        /* reversing needs to be handled in a special way */
        i = 0;
        j = len - 1;
        while (i < j) {
          char  tmp = dst[i];
          dst[i++] = dst[j];
          dst[j--] = tmp;
        }
      }
    }
    else {
      /* reverse string out of place (Ssrc and Sdst are not the same) */
      size_t   j = len;
      do {
        dst[i] = src[--j];
      } while (++i < len);
      dst[i] = '\0';
    }
  p->Sdst->timestamp = kcnt;
  return OK;
}

/**
 * ichr    strchar     Sstr[, ipos]
 * kchr    strchark    Sstr[, kpos]
 *
 * Return the ASCII code of the character in Sstr at ipos (defaults to 0).
 * If ipos is out of range, 0 is returned.
 */

int32_t strchar_opcode(CSOUND *csound, STRCHAR_OP *p)
{
  size_t     len = (int32_t) strlen((char*) p->Ssrc->data);
#if defined(MSVC) || (defined(__GNUC__) && defined(__i386__))
  int32_t     pos = (int32_t) MYFLT2LRND(*(p->ipos));
#else
  int32_t     pos = (int32_t) (*(p->ipos) + FL(1.5)) - 1;
#endif

  (void) csound;
  if (pos < 0 || (size_t)pos >= len)
    *(p->ichr) = FL(0.0);
  else
    *(p->ichr) = (MYFLT) ((int32_t)((unsigned char)((char*) p->Ssrc->data)[pos]));

  return OK;
}

/**
 * ilen    strlen      Sstr
 * klen    strlenk     Sstr
 *
 * Return the length of a string.
 */

int32_t strlen_opcode(CSOUND *csound, STRLEN_OP *p)
{
  (void) csound;
  if (p->Ssrc->size)
    *(p->ilen) = (MYFLT) strlen(p->Ssrc->data);
  else *(p->ilen) = FL(0.0);
  return OK;
}

/**
 * Sdst    strupper    Ssrc
 * Sdst    strupperk   Ssrc
 * Sdst    strlower    Ssrc
 * Sdst    strlowerk   Ssrc
 *
 * Convert a string to upper or lower case.
 */

int32_t strupper_opcode(CSOUND *csound, STRUPPER_OP *p)
{
  int64_t kcnt = p->h.insdshead->kcounter;
    const char  *src;
    char        *dst;
    int32_t         i;
    if (p->Ssrc->data == NULL) return NOTOK;
    if (p->Sdst->size < p->Ssrc->size) {
      size_t size = p->Ssrc->size;
      char *temp = csound->ReAlloc(csound, p->Sdst->data, size);
      if (UNLIKELY(temp == NULL)) {
        /* ReAlloc failed, keep the original buffer and return error */
        return csound->InitError(csound, Str("strupper: memory allocation failure"));
      }
      p->Sdst->data = temp;
      p->Sdst->size = size;
    }

    (void) csound;
    src = (char*) p->Ssrc->data;
    dst = (char*) p->Sdst->data;
    for (i = 0; src[i] != '\0'; i++) {
      unsigned char   tmp;
      tmp = (unsigned char) src[i];
      dst[i] = (char) (islower(tmp) ? (unsigned char) toupper(tmp) : tmp);
    }
    dst[i] = '\0';
    p->Sdst->timestamp = kcnt;

  return OK;
}

int32_t strlower_opcode(CSOUND *csound, STRUPPER_OP *p)
{

  int64_t kcnt = p->h.insdshead->kcounter;
    const char  *src;
    char        *dst;
    int32_t         i;
    if (p->Ssrc->data == NULL) return NOTOK;
    if (p->Sdst->size < p->Ssrc->size) {
      size_t size = p->Ssrc->size;
      char *temp = csound->ReAlloc(csound, p->Sdst->data, size);
      if (UNLIKELY(temp == NULL)) {
        /* ReAlloc failed, keep the original buffer and return error */
        return csound->InitError(csound, Str("strlower: memory allocation failure"));
      }
      p->Sdst->data = temp;
      p->Sdst->size = size;
    }

    (void) csound;
    src = (char*) p->Ssrc->data;
    dst = (char*) p->Sdst->data;
    for (i = 0; src[i] != '\0'; i++) {
      unsigned char   tmp;
      tmp = (unsigned char) src[i];
      dst[i] = (char) (isupper(tmp) ? (unsigned char) tolower(tmp) : tmp);
    }
    dst[i] = '\0';
    p->Sdst->timestamp = kcnt;
  return OK;
}

/**
 * Sval    getcfg      iopt
 *
 * Returns the value of a global setting (e.g. input file name) as a
 * string.
 */

int32_t getcfg_opcode(CSOUND *csound, GETCFG_OP *p)
{
  const char  *s;
#if defined(MSVC) || (defined(__GNUC__) && defined(__i386__))
  int32_t         opt = (int32_t) MYFLT2LRND(*(p->iopt));
#else
  int32_t         opt = (int32_t) (*(p->iopt) + FL(0.5));
#endif
  char        buf[32];

  if (p->Sdst->size < 32){
    char *temp = csound->ReAlloc(csound, p->Sdst->data, 32);
    if (UNLIKELY(temp == NULL)) {
      /* ReAlloc failed, keep the original buffer and return error */
      return csound->InitError(csound, Str("getcfg: memory allocation failure"));
    }
    p->Sdst->data = temp;
    p->Sdst->size = 32;
  }
  //((char*) p->Sdst->data)[0] = '\0';
  buf[0] = '\0';
  s = &(buf[0]);
  switch (opt) {
  case 1:             /* maximum length of variable */
    snprintf(&(buf[0]), 32, "%d", (int32_t) p->Sdst->size - 1);
    break;
  case 2:             /* input sound file name */
    s = (csound->oparms->sfread && !csound->initonly ?
	 csound->oparms->infilename : (char*) NULL);
    break;
  case 3:             /* output sound file name */
    s = (csound->oparms->sfwrite && !csound->initonly ?
	 csound->oparms->outfilename : (char*) NULL);
    break;
  case 4:             /* is real-time audio being used ? (0: no, 1: yes) */
    buf[0] = '0';
    buf[1] = '\0';
    if ((csound->oparms->sfread && !csound->initonly &&
	 check_rtaudio_name(csound->oparms->infilename, NULL, 0) >= 0) ||
	(csound->oparms->sfwrite && !csound->initonly &&
	 check_rtaudio_name(csound->oparms->outfilename, NULL, 1) >= 0))
      buf[0] = '1';
    break;
  case 5:             /* is beat mode being used ? (0: no, 1: yes) */
    buf[0] = (csound->oparms->Beatmode ? '1' : '0');
    buf[1] = '\0';
    break;
  case 6:             /* host OS name */
#ifdef LINUX
    s = "Linux";
#elif defined(WIN32)
    s = "Win32";
#elif defined(MACOSX)
    s = "MacOSX";
#else
    s = "unknown";
#endif
    break;
  case 7:             /* is the channel I/O callback set ? (0: no, 1: yes) */
    buf[0] = (csound->InputChannelCallback_ ||
              csound->OutputChannelCallback_ ? '1' : '0');
    buf[1] = '\0';
    break;
  default:
    return csound->InitError(csound, Str("invalid option code: %g"),
			     *(p->iopt));
  }
  if (s != NULL) {

    if (p->Sdst->data == NULL) {
      size_t size = strlen(s) + 1;
      p->Sdst->data = csound->Calloc(csound, size);
      p->Sdst->size = size;
    }
    else if (UNLIKELY(strlen(s) >=  p->Sdst->size)) {
      size_t len = strlen(s) + 1;
      char *temp = csound->ReAlloc(csound, p->Sdst->data, len);
      if (UNLIKELY(temp == NULL)) {
        /* ReAlloc failed, keep the original buffer and return error */
        return csound->InitError(csound, Str("getcfg: memory allocation failure"));
      }
      p->Sdst->data = temp;
      p->Sdst->size = len;
    }
    strcpy((char*) p->Sdst->data, s);
  }

  return OK;
}

/**
 * ipos    strindex    Sstr1, Sstr2
 * kpos    strindexk   Sstr1, Sstr2
 *
 * Return the position of the first occurence of Sstr2 in Sstr1,
 * or -1 if not found. If Sstr2 is empty, 0 is returned.
 */

int32_t strindex_opcode(CSOUND *csound, STRINDEX_OP *p)
{
  const char  *s1 = (char*) p->Ssrc1->data;
  const char  *s2 = (char*) p->Ssrc2->data;
  int32_t         i, j;

  (void) csound;
  /* search substring from left to right, */
  /* and return position of first match */
  i = j = 0;
  while (s2[j] != '\0') {
    if (s1[i] == '\0') {
      *(p->ipos) = -FL(1.0);
      return OK;
    }
    j = (s1[i] != s2[j] ? 0 : j + 1);
    i++;
  }
  *(p->ipos) = (MYFLT) (i - j);

  return OK;
}

/**
 * ipos    strrindex   Sstr1, Sstr2
 * kpos    strrindexk  Sstr1, Sstr2
 *
 * Return the position of the last occurence of Sstr2 in Sstr1,
 * or -1 if not found. If Sstr2 is empty, the length of Sstr1 is
 * returned.
 */

int32_t strrindex_opcode(CSOUND *csound, STRINDEX_OP *p)
{
  const char  *s1 = (char*) p->Ssrc1->data;
  const char  *s2 = (char*) p->Ssrc2->data;
  int32_t         i, j, k;

  (void) csound;
  /* search substring from left to right, */
  /* and return position of last match */
  i = j = 0;
  k = -1;
  while (1) {
    if (s2[j] == '\0') {
      k = i - j;
      j = 0;
    }
    if (s1[i] == '\0')
      break;
    j = (s1[i] != s2[j] ? 0 : j + 1);
    i++;
  }
  *(p->ipos) = (MYFLT) k;

  return OK;
}

#ifdef HAVE_CURL
int32_t str_from_url(CSOUND *csound, STRCPY_OP *p)
{
  char  *newVal = p->str->data;
  if (strstr(newVal, ":/")==NULL) return strcpy_opcode_S(csound, p);
  {
    CORFIL *mm = copy_url_corefile(csound, newVal,0);
    size_t len = corfile_length(mm);

    if (len >= p->r->size) {
      p->r->data = csound->ReAlloc(csound, p->r->data, len + 1);
      p->r->size = len + 1;
    }
    if (p->r->data != NULL)
      strcpy((char*) p->r->data, corfile_body(mm));

    return OK;
  }
}
#endif

#if !defined(HAVE_STRLCAT) && !defined(strlcat) && !defined(EMSCRIPTEN)
/* Direct from BSD sources */
/*
 * Copyright (c) 1998 Todd C. Miller <Todd.Miller@courtesan.com>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 */
size_t
strlcat(char *dst, const char *src, size_t siz)
{
  char *d = dst;
  const char *s = src;
  size_t n = siz;
  size_t dlen;

  /* Find the end of dst and adjust bytes left but don't go past end */
  while (n-- != 0 && *d != '\0')
    d++;
  dlen = d - dst;
  n = siz - dlen;

  if (n == 0)
    return (dlen + strlen(s));
  while (*s != '\0') {
    if (n != 1) {
      *d++ = *s;
      n--;
    }
    s++;
  }
  *d = '\0';

  return (dlen + (s - src));  /* count does not include NUL */
}
#endif

/* Modified from BSD sources for strlcpy */
/*
 * Copyright (c) 1998 Todd C. Miller <Todd.Miller@courtesan.com>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 */
/* modifed for speed -- JPff */
char *
strNcpy(char *dst, const char *src, size_t siz)
{
  char *d = dst;
  const char *s = src;
  size_t n = siz;

  /* Copy as many bytes as will fit or until NULL */
  if (n != 0) {
    while (--n != 0) {
      if ((*d++ = *s++) == '\0')
	break;
    }
  }

  /* Not enough room in dst, add NUL */
  if (n == 0) {
    if (siz != 0)
      *d = '\0';                /* NUL-terminate dst */

    //while (*s++) ;
  }
  return dst;        /* count does not include NUL */
}

/* Debugging opcode for testing runtime type identification */
int32_t print_type_opcode(CSOUND* csound, PRINT_TYPE_OP* p) {
  char* ptr = (char*)p->inVar;

  CS_TYPE* varType = *(CS_TYPE**)(ptr - CS_VAR_TYPE_OFFSET);
  csound->Message(csound, "Variable Type: %s\n", varType->varTypeName);

  return OK;
}

 const char* csoundGetStringData(CSOUND *csound, STRINGDAT *sdata){
  return sdata->data;
}

 void csoundSetStringData(CSOUND *csound, STRINGDAT *sdata, const char *str){
  size_t bytes = strlen(str);
  if(sdata->size > bytes)
    strcpy(sdata->data, str);
  else {
    sdata->data = (char *) csound->ReAlloc(csound, (char *)
                                           sdata->data, bytes+1);
    strcpy(sdata->data, str);
  }
}
