/*
  rtpa.c:

  Copyright (C) 2004, 2005 John ffitch, Istvan Varga,
  Michael Gogins, Victor Lazzarini

  This file is part of Csound.

  The Csound Library is free software; you can redistribute it
  and/or modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  Csound is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with Csound; if not, write to the Free Software
  Foundation, Inc., 31 Milk Street, #960789, Boston, MA, 02196, USA
*/

/*                                              RTPA.C for PortAudio    */

#include "csdl.h"
#if !defined(WIN32)
#include "soundio.h"
#endif
#include <portaudio.h>

typedef struct PaAlsaStreamInfo {
  unsigned long   size;
  PaHostApiTypeId hostApiType;
  unsigned long   version;
  const char      *deviceString;
} PaAlsaStreamInfo;

typedef struct devparams_ {
  PaStream    *handle;        /* stream handle                    */
  float       *buf;           /* sample conversion buffer         */
  int32_t      nchns;          /* number of channels               */
} DEVPARAMS;

typedef struct PA_BLOCKING_STREAM_ {
  CSOUND      *csound;
  PaStream    *paStream;
  int32_t         mode;                   /* 1: rec, 2: play, 3: full-duplex  */
  int32_t         inBufSamples;
  int32_t         outBufSamples;
  int32_t         currentInputIndex;
  int32_t         currentOutputIndex;
  MYFLT       *inputBuffer;
  MYFLT       *outputBuffer;
  csRtAudioParams inParm;
  csRtAudioParams outParm;
  PaStreamParameters inputPaParameters;
  PaStreamParameters outputPaParameters;
#ifdef WIN32
  int32_t         paused;                 /* VL: to allow for smooth pausing  */
#endif
  int32_t  complete;
  int32_t  ksmps;
  void *incb;
  void *outcb;  
} PA_BLOCKING_STREAM;

static int32_t pa_PrintErrMsg(CSOUND *csound, const char *fmt, ...)
{
  va_list args;
  va_start(args, fmt);
  csound->ErrMsgV(csound, Str(" *** PortAudio: error: "), fmt, args);
  va_end(args);
  return -1;
}

static int32_t init_portaudio(CSOUND *csound)
{
  char  *s;
  int32_t   err;
  /* initialise PortAudio */
  if (!csound->QueryGlobalVariable(csound, "::PortAudio::NeedsTerminate")) {
    if (csound->CreateGlobalVariable(csound,
                                     "::PortAudio::NeedsTerminate", 1) != 0)
      return -1;
    err = (int32_t) Pa_Initialize();
    if (UNLIKELY(err != (int32_t) paNoError)) {
      return pa_PrintErrMsg(csound, "%d: %s",
                            err, Pa_GetErrorText((PaError) err));
    }
    /* print PortAudio version */
    {
      if ((s = (char*) Pa_GetVersionText()) != NULL)
        csound->ErrorMsg(csound, "%s\n", s);
    }
  }
  return 0;
}

/* list available input or output devices; returns the number of devices */
int32_t list_devices(CSOUND *csound, CS_AUDIODEVICE *list, int32_t isOutput){
  PaDeviceInfo  *dev_info;
  PaHostApiInfo *api_info;
  int32_t           i, j, ndev;
  char          tmp[256], *s;

  if (init_portaudio(csound) != 0)
    return 0;

  if ((s = (char*) csound->QueryGlobalVariable(csound, "_RTAUDIO")) == NULL)
    return 0;

  ndev = (int32_t) Pa_GetDeviceCount();
  for (i = j = 0; i < ndev; i++) {
    dev_info = (PaDeviceInfo*) Pa_GetDeviceInfo((PaDeviceIndex) i);
    if ((isOutput && dev_info->maxOutputChannels > 0) ||
        (!isOutput && dev_info->maxInputChannels > 0))
      j++;
  }
  if (!j) return 0;
  if(list!=NULL) {
    for (i = j = 0; i < ndev; i++) {
      dev_info = (PaDeviceInfo*) Pa_GetDeviceInfo((PaDeviceIndex) i);
      api_info = (PaHostApiInfo*) Pa_GetHostApiInfo(dev_info->hostApi);
      if ((isOutput && dev_info->maxOutputChannels > 0) ||
          (!isOutput && dev_info->maxInputChannels > 0)) {
        //strncpy(list[j].device_name, dev_info->name, 63);
        snprintf(list[j].device_name, sizeof(list[j].device_name)-1, "%s [%s, %d in, %d out]",
                 dev_info->name, api_info->name, dev_info->maxInputChannels,
                 dev_info->maxOutputChannels);
        if (isOutput) {
          snprintf(tmp, 256, "dac%d", j);
        } else {
          snprintf(tmp, 256, "adc%d", j);
        }
        strncpy(list[j].device_id, tmp, sizeof(list[j].device_id)-1);
        list[j].device_id[sizeof(list[j].device_id)-1]='\0';

        strncpy(list[j].rt_module, s, sizeof(list[j].rt_module)-1);
        list[j].rt_module[sizeof(list[j].rt_module)-1]='\0';

        list[j].max_nchnls =
          isOutput ?  dev_info->maxOutputChannels : dev_info->maxInputChannels;
        list[j].isOutput = isOutput;
        j++;
      }
    }
  }
  return j;
}


static int32_t list_portaudio_devices(CSOUND *csound,
                                      int32_t print_list, int32_t play)
{
  int32_t i,n = list_devices(csound, NULL, play);
  IGN(print_list);
  CS_AUDIODEVICE *devs =
    (CS_AUDIODEVICE *) csound->Malloc(csound, n*sizeof(CS_AUDIODEVICE));
  list_devices(csound, devs, play);
  {
    for(i=0; i < n; i++)
      csound->ErrorMsg(csound, " %3d: %s (%s)\n",
                       i, devs[i].device_id, devs[i].device_name);

  }
  csound->Free(csound, devs);
  return n;
}

static void DAC_channels(CSOUND *csound, int32_t chans){
  int32_t *dachans = (int32_t *) csound->QueryGlobalVariable(csound, "_DAC_CHANNELS_");
  if (dachans == NULL) {
    if (csound->CreateGlobalVariable(csound, "_DAC_CHANNELS_",
                                     sizeof(int32_t)) != 0)
      return;
    dachans = (int32_t *) csound->QueryGlobalVariable(csound, "_DAC_CHANNELS_");
    *dachans = chans;
  }
}

static void ADC_channels(CSOUND *csound, int32_t chans){
  int32_t *dachans = (int32_t *) csound->QueryGlobalVariable(csound, "_ADC_CHANNELS_");
  if (dachans == NULL) {
    if (csound->CreateGlobalVariable(csound, "_ADC_CHANNELS_",
                                     sizeof(int32_t)) != 0)
      return;
    dachans = (int32_t *) csound->QueryGlobalVariable(csound, "_ADC_CHANNELS_");
    *dachans = chans;
  }
}

/* select PortAudio device; returns the actual device number */
static int32_t select_portaudio_device(CSOUND *csound, int32_t devNum, int32_t play)
{
  PaDeviceInfo  *dev_info;
  int32_t           i, j, maxNum;

  maxNum = list_portaudio_devices(csound, 1, play) - 1;
  if (maxNum < 0)
    return -1;
  if (devNum == 1024) {
    if (play)
      devNum = (int32_t) Pa_GetDefaultOutputDevice();
    else
      devNum = (int32_t) Pa_GetDefaultInputDevice();
  }
  else {
    if (devNum < 0 || devNum > maxNum) {
      /* listPortAudioDevices_blocking(csound, 1, play); */
      pa_PrintErrMsg(csound, Str("%s device number %d is out of range"),
                     (play ? Str("output") : Str("input")), devNum);
      return -1;
    }
    for (i = j = 0; j <= maxNum; i++) {
      dev_info = (PaDeviceInfo*) Pa_GetDeviceInfo((PaDeviceIndex) i);
      if ((play && dev_info->maxOutputChannels > 0) ||
          (!play && dev_info->maxInputChannels > 0)) {
        if (j == devNum)
          break;
        j++;
      }
    }
    devNum = i;
  }
  dev_info = (PaDeviceInfo*) Pa_GetDeviceInfo((PaDeviceIndex) devNum);
  if (dev_info) {
    csound->ErrorMsg(csound, Str("PortAudio: selected %s device '%s'\n"),
                     (play ? Str("output") : Str("input")),
                     dev_info->name);
    if(play) {
      csound->GetSystemSr(csound, (MYFLT) dev_info->defaultSampleRate);
      DAC_channels(csound, dev_info->maxOutputChannels);
    } else ADC_channels(csound, dev_info->maxInputChannels);
  }
  else
    pa_PrintErrMsg(csound, "%s",
                   Str("PortAudio: failed to obtain device info.\n"));
  return devNum;
}

static int32_t set_stream_parameters(CSOUND *csound, PaStreamParameters *sp,
                                     csRtAudioParams *parm,
                                     int32_t is_playback)
{
  int32_t dev;
  memset(sp, 0, sizeof(PaStreamParameters));
  if (UNLIKELY(parm->devName != NULL && parm->devName[0] != '\0')) {
    return pa_PrintErrMsg(csound,
                          Str("Must specify a device number, not a name"));
  }
  dev = select_portaudio_device(csound, parm->devNum, is_playback);
  if(parm->sampleRate < 0) {
    parm->sampleRate = csound->GetSystemSr(csound, 0);
  }
  if (dev < 0)
    return -1;
  sp->device = (PaDeviceIndex) dev;
  // VL this is causing problems to open the microphone input
#ifdef __APPLE__
  sp->channelCount = parm->nChannels; //(parm->nChannels < 2 ? 2 : parm->nChannels);
#else
  sp->channelCount = (parm->nChannels < 2 ? 2 : parm->nChannels);
#endif  
  sp->sampleFormat = (PaSampleFormat) paFloat32;
  sp->suggestedLatency = (PaTime) ((double) parm->bufSamp_SW
                                   / (double) parm->sampleRate);
  sp->hostApiSpecificStreamInfo = NULL;
  return 0;
}



static int32_t audio_callback(const void *input, void *output,unsigned long framecount,
                              const PaStreamCallbackTimeInfo *timeInfo,
                              PaStreamCallbackFlags statusFlags, void *userData){
 
  PA_BLOCKING_STREAM *pabs = (PA_BLOCKING_STREAM*) userData;
  CSOUND  *csound = pabs->csound;
  float   *paInput = (float*) input;
  float   *paOutput = (float*) output;
  int32_t samps;
  memset(output,0,framecount*sizeof(float)*pabs->outParm.nChannels);
   
  if (pabs->complete == 1) return paContinue;
  
#ifdef WIN32
  if (pabs->paStream == NULL || pabs->paused) return paContinue;
#endif
  
  if (pabs->mode & 1){
    samps = (int32_t) (pabs->inParm.nChannels*framecount);
    for(int i = 0; i < samps; i++)
      pabs->inputBuffer[i] = paInput[i];
    csound->WriteCircularBuffer(csound,pabs->incb,pabs->inputBuffer,samps);
  }
  if (pabs->mode & 2) {
    samps = pabs->outBufSamples;
    int32_t n = csound->ReadCircularBuffer(csound,pabs->outcb,
                                           pabs->outputBuffer,samps);
    for(int i = 0; i < n; i++)
      paOutput[i] = pabs->outputBuffer[i];     
  }
  return paContinue;
}

static int32_t rtrecord_noblock(CSOUND *csound, MYFLT *inbuff_, int32_t nbytes)
{
  int32_t n = nbytes/sizeof(MYFLT);
  int32_t m = 0, l;
  PA_BLOCKING_STREAM *pabs = (PA_BLOCKING_STREAM*) *(csound->GetRtRecordUserData(csound));
  do{
    l = csound->ReadCircularBuffer(csound,pabs->incb,&inbuff_[m],n);
    m += l;
    n -= l;
    if(n)
#ifdef WIN32
    csound->Sleep(1); // ms res
#else
    usleep(100); // more responsive
#endif 
  } while(n);
  return nbytes;
}

static void rtplay_noblock(CSOUND *csound, const MYFLT *outbuff_, int32_t nbytes)
{
  int32_t n = nbytes/sizeof(MYFLT);
  int32_t m = 0, l;
  PA_BLOCKING_STREAM *pabs = (PA_BLOCKING_STREAM*) *(csound->GetRtPlayUserData(csound));
  do {
    l = csound->WriteCircularBuffer(csound, pabs->outcb,&outbuff_[m],n);
    m += l;
    n -= l;
    if(n)
#ifdef WIN32
    csound->Sleep(1); // ms res
#else
    usleep(100); // more responsive
#endif  
  } while(n);
}

static int32_t recopen_noblock(CSOUND *csound, const csRtAudioParams *parm)
{
  CSOUND *p = csound;
  PA_BLOCKING_STREAM *pabs;

  pabs = (PA_BLOCKING_STREAM*) p->QueryGlobalVariable(p, "_rtpaGlobals");
  if (pabs == NULL) {
    if (p->CreateGlobalVariable(p, "_rtpaGlobals", sizeof(PA_BLOCKING_STREAM))
        != 0)
      return -1;
    pabs = (PA_BLOCKING_STREAM*) p->QueryGlobalVariable(p, "_rtpaGlobals");
    pabs->csound = p;
  }
  pabs->incb = csound->CreateCircularBuffer(csound,parm->bufSamp_HW*parm->nChannels,
                                            sizeof(MYFLT));
  pabs->mode |= 1;
  memcpy(&(pabs->inParm), parm, sizeof(csRtAudioParams));
  *(p->GetRtRecordUserData(p)) = (void*) pabs;
  pabs->ksmps = parm->ksmps;
  pabs->complete = 0;
  return 0;
}

static void rtclose_noblock(CSOUND *csound);
static int32_t set_device_params_noblock(CSOUND *csound)
{
  PA_BLOCKING_STREAM  *pabs;
  PaStream            *stream = NULL;
  PaError             err;

  pabs = (PA_BLOCKING_STREAM*) csound->QueryGlobalVariable(csound,
                                                           "_rtpaGlobals");
  if (pabs == NULL)
    return -1;
  if (UNLIKELY(init_portaudio(csound) != 0))
    goto err_return;

  if (UNLIKELY((int32_t) Pa_GetDeviceCount() <= 0)) {
    pa_PrintErrMsg(csound, Str("No sound device is available"));
    goto err_return;
  }

  if (UNLIKELY(pabs->mode & 1)) {
    if (set_stream_parameters(csound, &(pabs->inputPaParameters),
                              &(pabs->inParm), 0) != 0)
      goto err_return;
    pabs->inBufSamples = pabs->inParm.bufSamp_SW
      * (int32_t) pabs->inputPaParameters.channelCount;
    pabs->inputBuffer = (MYFLT*) csound->Calloc(csound,
                                                (size_t) pabs->inBufSamples*
                                                sizeof(MYFLT));
    if (UNLIKELY(pabs->inputBuffer == NULL)) {
      pa_PrintErrMsg(csound, Str("Memory allocation failure"));
      goto err_return;
    }
  }
  if (pabs->mode & 2) {
    if (UNLIKELY(set_stream_parameters(csound, &(pabs->outputPaParameters),
                                       &(pabs->outParm), 1) != 0))
      goto err_return;
    pabs->outBufSamples = pabs->outParm.bufSamp_SW
      * (int32_t) pabs->outputPaParameters.channelCount;
    pabs->outputBuffer = (MYFLT*) csound->Calloc(csound,
                                                 (size_t) pabs->outBufSamples*
                                                 sizeof(MYFLT));
    if (UNLIKELY(pabs->outputBuffer == NULL)) {
      pa_PrintErrMsg(csound, Str("Memory allocation failure"));
      goto err_return;
    }
  }
  if ((pabs->mode & 3) == 3) {
    if (UNLIKELY(pabs->inParm.bufSamp_SW != pabs->outParm.bufSamp_SW)) {
      pa_PrintErrMsg(csound, Str("Inconsistent full-duplex buffer sizes"));
      goto err_return;
    }
    if (UNLIKELY(pabs->inParm.sampleRate != pabs->outParm.sampleRate)) {
      pa_PrintErrMsg(csound, Str("Inconsistent full-duplex sample rates"));
      goto err_return;
    }
    if (UNLIKELY(((pabs->inParm.bufSamp_SW / pabs->ksmps) *
                  pabs->ksmps) != pabs->inParm.bufSamp_SW))
      csound->Warning(csound,
                      "%s", Str("WARNING: buffer size should be an integer "
                                "multiple of ksmps in full-duplex mode\n"));
  }
  
  err = Pa_OpenStream(&stream,
                      (pabs->mode & 1 ? &(pabs->inputPaParameters)
                       : (PaStreamParameters*) NULL),
                      (pabs->mode & 2 ? &(pabs->outputPaParameters)
                       : (PaStreamParameters*) NULL),
                      (double) (pabs->mode & 2 ? pabs->outParm.sampleRate
                                : pabs->inParm.sampleRate),
                      (unsigned long) (pabs->mode & 2 ?
                                       pabs->outParm.bufSamp_SW
                                       : pabs->inParm.bufSamp_SW),
                      (csound->GetDitherMode(csound) ?
                       (PaStreamFlags) paNoFlag : (PaStreamFlags) paDitherOff),
                      audio_callback,
                      (void*) pabs);
  if (UNLIKELY(err != paNoError)) {
    pa_PrintErrMsg(csound, "%d: %s", (int32_t) err, Pa_GetErrorText(err));
    goto err_return;
  }

  err = Pa_StartStream(stream);
  if (UNLIKELY(err != paNoError)) {
    Pa_CloseStream(stream);
    pa_PrintErrMsg(csound, "%d: %s", (int32_t) err, Pa_GetErrorText(err));
    goto err_return;
  }
  
  pabs->paStream = stream;
  return 0;

  /* clean up and report error */
 err_return:
  rtclose_noblock(csound);
  return -1;
}

static int32_t playopen_noblock(CSOUND *csound, const csRtAudioParams *parm)
{
  CSOUND *p = csound;
  PA_BLOCKING_STREAM *pabs;

  pabs = (PA_BLOCKING_STREAM*) p->QueryGlobalVariable(p, "_rtpaGlobals");
  if (pabs == NULL) {
    if (p->CreateGlobalVariable(p, "_rtpaGlobals", sizeof(PA_BLOCKING_STREAM))
        != 0)
      return -1;
    pabs = (PA_BLOCKING_STREAM*) p->QueryGlobalVariable(p, "_rtpaGlobals");
    pabs->csound = p;
  }
  pabs->outcb = csound->CreateCircularBuffer(csound,parm->bufSamp_HW*parm->nChannels,
                                             sizeof(MYFLT));
  pabs->mode |= 2;
  memcpy(&(pabs->outParm), parm, sizeof(csRtAudioParams));
  *(p->GetRtPlayUserData(p)) = (void*) pabs;
  pabs->ksmps = parm->ksmps;
  pabs->complete = 0;
  

  return set_device_params_noblock(p);
}


/* close the I/O device entirely */
static void rtclose_noblock(CSOUND *csound)
{
  PA_BLOCKING_STREAM *pabs;
  pabs = (PA_BLOCKING_STREAM*) csound->QueryGlobalVariable(csound,
                                                           "_rtpaGlobals");
  {
    csound->ErrorMsg(csound, "%s", Str("closing device\n"));
  }
  if (pabs == NULL)
    return;

  pabs->complete = 1;

  if (pabs->paStream != NULL) {
    PaStream  *stream = pabs->paStream;
    Pa_StopStream(stream);
    Pa_CloseStream(stream);
  }
  
  if (pabs->outputBuffer != NULL) {
    csound->Free(csound,pabs->outputBuffer);
    pabs->outputBuffer = NULL;
  }
  if (pabs->inputBuffer != NULL) {
    csound->Free(csound,pabs->inputBuffer);
    pabs->inputBuffer = NULL;
  }
  pabs->paStream = NULL;
  *(csound->GetRtRecordUserData(csound)) = NULL;
  *(csound->GetRtPlayUserData(csound)) = NULL;
  csound->DestroyGlobalVariable(csound, "_rtpaGlobals");
}

/**
 *  blocking interface 
 */

/* set up audio device */
static int32_t set_device_params(CSOUND *csound, DEVPARAMS *dev,
                                 const csRtAudioParams *parm, int32_t play)
{
  PaStreamParameters  streamParams;
  CSOUND              *p = csound;
  int32_t                 err;

  /* set parameters */
  memset(dev, 0, sizeof(DEVPARAMS));
  memset(&streamParams, 0, sizeof(PaStreamParameters));
  streamParams.hostApiSpecificStreamInfo = NULL;
  if (UNLIKELY(parm->devName != NULL && parm->devName[0] != '\0')) {
#if !defined(LINUX)
    list_portaudio_devices(p, 1, play);
    pa_PrintErrMsg(p, "%s", Str("Must specify a device number, not a name"));
    return -1;
#else
    PaAlsaStreamInfo info;
    p->Message(p, Str("PortAudio: using ALSA device '%s'\n"), parm->devName);
    memset(&info, 0, sizeof(PaAlsaStreamInfo));
    info.deviceString = parm->devName;
    info.hostApiType = paALSA;
    info.version = 1;
    info.size = sizeof(info);
    streamParams.device = paUseHostApiSpecificDeviceSpecification;
    streamParams.hostApiSpecificStreamInfo = &info;
#endif
  }
  else {
    int32_t devNum = select_portaudio_device(p, parm->devNum, play);
    if (devNum < 0)
      return -1;
    streamParams.device = (PaDeviceIndex) devNum;
  }
  streamParams.channelCount = parm->nChannels;
  streamParams.sampleFormat = paFloat32;
  streamParams.suggestedLatency = (PaTime) ((double) parm->bufSamp_HW
                                            / (double) parm->sampleRate);
  /* open stream */
  if (play) {
    err = (int32_t) Pa_OpenStream(&(dev->handle), NULL, &streamParams,
                                  (double) parm->sampleRate,
                                  (unsigned long) parm->bufSamp_SW,
                                  (csound->GetDitherMode(csound) ?
                                   paNoFlag:paDitherOff),
                                  NULL, NULL);
  }
  else {
    err = (int32_t) Pa_OpenStream(&(dev->handle), &streamParams, NULL,
                                  (double) parm->sampleRate,
                                  (unsigned long) parm->bufSamp_SW,
                                  paNoFlag, NULL, NULL);
  }
  if (UNLIKELY(err != (int32_t) paNoError)) {
    pa_PrintErrMsg(p, "%d: %s", err, Pa_GetErrorText((PaError) err));
    return -1;
  }
  /* set up device parameters */
  dev->nchns = parm->nChannels;
  dev->buf = (float*) p->Calloc(p, (size_t) (parm->bufSamp_SW
                                             * parm->nChannels
                                             * (int32_t) sizeof(float)));

  return 0;
}

/* open for audio input */
static int32_t recopen_blocking(CSOUND *csound, const csRtAudioParams *parm)
{
  DEVPARAMS *dev;
  int32_t       retval;

  if (init_portaudio(csound) != 0)
    return -1;
  /* check if the device is already opened */
  if (*(csound->GetRtRecordUserData(csound)) != NULL)
    return 0;
  /* allocate structure */
  dev = (DEVPARAMS*) csound->Calloc(csound, sizeof(DEVPARAMS));
  *(csound->GetRtRecordUserData(csound)) = (void*) dev;
  /* set up parameters and open stream */
  retval = set_device_params(csound, dev, parm, 0);
  if (retval != 0) {
    csound->Free(csound, dev);
    *(csound->GetRtRecordUserData(csound)) = NULL;
  }
  else
    Pa_StartStream(dev->handle);
  return retval;
}

/* open for audio output */
static int32_t playopen_blocking(CSOUND *csound, const csRtAudioParams *parm)
{
  DEVPARAMS *dev;
  int32_t       retval;

  if (init_portaudio(csound) != 0)
    return -1;
  /* check if the device is already opened */
  if (*(csound->GetRtPlayUserData(csound)) != NULL)
    return 0;
  /* allocate structure */
  dev = (DEVPARAMS*) csound->Calloc(csound, sizeof(DEVPARAMS));
  *(csound->GetRtPlayUserData(csound)) = (void*) dev;
  /* set up parameters and open stream */
  retval = set_device_params(csound, dev, parm, 1);
  if (retval != 0) {
    csound->Free(csound, dev);
    *(csound->GetRtPlayUserData(csound)) = NULL;
  }
  else
    Pa_StartStream(dev->handle);
  return retval;
}

/* get samples from ADC */
static int32_t rtrecord_blocking(CSOUND *csound, MYFLT *inbuf, int32_t nbytes)
{
  DEVPARAMS *dev;
  int32_t       i, n, err;

  dev = (DEVPARAMS*) (*(csound->GetRtRecordUserData(csound)));
  /* calculate the number of samples to record */
  n = nbytes / (dev->nchns * (int32_t) sizeof(MYFLT));
  err = (int32_t) Pa_ReadStream(dev->handle, dev->buf, (unsigned long) n);
  if (UNLIKELY(err != (int32_t) paNoError && (csound->GetMessageLevel(csound) & 4)))
    csound->Warning(csound, "%s", Str("Buffer overrun in real-time audio input"));
  /* convert samples to MYFLT */
  for (i = 0; i < (n * dev->nchns); i++)
    inbuf[i] = (MYFLT) dev->buf[i];

  return nbytes;
}

/* put samples to DAC */
static void rtplay_blocking(CSOUND *csound, const MYFLT *outbuf, int32_t nbytes)
{
  DEVPARAMS *dev;
  int32_t       i, n, err;

  dev = (DEVPARAMS*) (*(csound->GetRtPlayUserData(csound)));
  /* calculate the number of samples to play */
  n = nbytes / (dev->nchns * (int32_t) sizeof(MYFLT));
  /* convert samples from MYFLT */
  for (i = 0; i < (n * dev->nchns); i++)
    dev->buf[i] = (float) outbuf[i];
  err = (int32_t) Pa_WriteStream(dev->handle, dev->buf, (unsigned long) n);
  if (UNLIKELY(err != (int32_t) paNoError && (csound->GetMessageLevel(csound) & 4)))
    csound->Warning(csound, "%s",
                    Str("Buffer underrun in real-time audio output"));
}

/* close the I/O device entirely  */
/* called only when both complete */
static void rtclose_blocking(CSOUND *csound)
{
  DEVPARAMS *dev;
  csound->ErrorMsg(csound, "%s", Str("closing device\n"));
  dev = (DEVPARAMS*) (*(csound->GetRtRecordUserData(csound)));
  if (dev != NULL) {
    *(csound->GetRtRecordUserData(csound)) = NULL;
    if (dev->handle != NULL) {
      Pa_StopStream(dev->handle);
      Pa_CloseStream(dev->handle);
    }
    if (dev->buf != NULL)
      csound->Free(csound, dev->buf);
    csound->Free(csound, dev);
  }
  dev = (DEVPARAMS*) (*(csound->GetRtPlayUserData(csound)));
  if (dev != NULL) {
    *(csound->GetRtPlayUserData(csound)) = NULL;
    if (dev->handle != NULL) {
      Pa_StopStream(dev->handle);
      Pa_CloseStream(dev->handle);
    }
    if (dev->buf != NULL)
      csound->Free(csound, dev->buf);
    csound->Free(csound, dev);
  }
}

#ifdef WIN32
#ifdef __cplusplus
extern "C"
{
#endif
  typedef void (*PaUtilLogCallback ) (const char *log);
  extern  void PaUtil_SetDebugPrintFunction(PaUtilLogCallback  cb);
#ifdef __cplusplus
}
#endif

static void PaNoOpDebugPrint(const char* msg) {
}
#endif

/* module interface functions */

 int32_t csoundModuleCreate(CSOUND *csound)
{
  IGN(csound);
#ifdef WIN32
  PaUtil_SetDebugPrintFunction(PaNoOpDebugPrint);
#endif
  return 0;
}

 int32_t csoundModuleInit(CSOUND *csound)
{
  char    *s = NULL;
  char    drv[12];
  int32_t     i;
  memset(drv, '\0', 12);
  csound->ModuleListAdd(csound, "pa_bl", "audio");
  csound->ModuleListAdd(csound, "pa_cb", "audio");
  if ((s = (char*) csound->QueryGlobalVariable(csound, "_RTAUDIO")) == NULL)
    return 0;
  for (i = 0; s[i] != '\0' && i < 11; i++)
    drv[i] = s[i] & (char) 0xDF;
  drv[i] = '\0';
  if (!(strcmp(drv, "PORTAUDIO") == 0 || strcmp(drv, "PA") == 0 ||
        strcmp(drv, "PA_BL") == 0 || strcmp(drv, "PA_CB") == 0)) {
    return 0;
  }
  csound->DebugMsg(csound, "%s", Str("rtaudio: PortAudio module enabled ...\n"));
  /* set function pointers */
#ifdef LINUX
  if (strcmp(drv, "PA_CB") != 0)
#else
    if (strcmp(drv, "PA_BL") == 0)
#endif
      {
        csound->ErrorMsg(csound, "%s", Str("using blocking interface\n"));
        csound->SetPlayopenCallback(csound, playopen_blocking);
        csound->SetRecopenCallback(csound, recopen_blocking);
        csound->SetRtplayCallback(csound, rtplay_blocking);
        csound->SetRtrecordCallback(csound, rtrecord_blocking);
        csound->SetRtcloseCallback(csound, rtclose_blocking);
        csound->SetAudioDeviceListCallback(csound, list_devices);
      }
    else {
      csound->ErrorMsg(csound, "%s", Str("using callback interface\n"));
      csound->SetPlayopenCallback(csound, playopen_noblock);
      csound->SetRecopenCallback(csound, recopen_noblock);
      csound->SetRtplayCallback(csound, rtplay_noblock);
      csound->SetRtrecordCallback(csound, rtrecord_noblock);
      csound->SetRtcloseCallback(csound, rtclose_noblock);
      csound->SetAudioDeviceListCallback(csound, list_devices);
    }

  csound->ModuleListAdd(csound, s, "audio");
  return 0;
}

 int32_t csoundModuleDestroy(CSOUND *csound)
{
  if (csound->QueryGlobalVariable(csound, "::PortAudio::NeedsTerminate")) {
    csound->DestroyGlobalVariable(csound, "::PortAudio::NeedsTerminate");
    return ((int32_t) Pa_Terminate() == (int32_t) paNoError ? 0 : -1);
  }
  return 0;
}

 int32_t csoundModuleInfo(void)
{
  return ((CS_VERSION << 16) + (CS_SUBVER << 8) + (int32_t) sizeof(MYFLT));
}
