/*
    linevent.c:

    Copyright (C) 1991 Barry Vercoe, John ffitch, matt ingalls

    This file is part of Csound.

    The Csound Library is free software; you can redistribute it
    and/or modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    Csound is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with Csound; if not, write to the Free Software
    Foundation, Inc., 31 Milk Street, #960789, Boston, MA, 02196, USA
*/

#include "csoundCore.h"     /*                              LINEVENT.C      */
#include <ctype.h>

#ifdef MSVC
#include <fcntl.h>
#endif


#ifdef PIPES
# if defined(SGI) || defined(LINUX) || defined(NeXT) || defined(__MACH__)
#  define _popen popen
#  define _pclose pclose
# elif defined(__BEOS__) ||  defined(__HAIKU__) || defined(__MACH__)
#  include <stdio.h>
#  define _popen popen
#  define _pclose pclose
# else
   FILE *_popen(const char *, const char *);
# endif
#endif

#define LBUFSIZ1 32768
#define LF        '\n'

void sense_line(CSOUND *csound, void *userData);
static int32_t set_sense_event_callback(CSOUND *csound, void (*func)(CSOUND *, void *),
                 void *userData)
{
  EVT_CB_FUNC *fp = (EVT_CB_FUNC*) csound->evtFuncChain;
  if (fp == NULL) {
    fp = (EVT_CB_FUNC*) csound->Calloc(csound, sizeof(EVT_CB_FUNC));
    csound->evtFuncChain = (void*) fp;
  }
  else {
    while (fp->nxt != NULL)
      fp = fp->nxt;
    fp->nxt = (EVT_CB_FUNC*) csound->Calloc(csound, sizeof(EVT_CB_FUNC));
    fp = fp->nxt;
  }
  if (UNLIKELY(fp == NULL))
    return CSOUND_MEMORY;
  fp->func = func;
  fp->userData = userData;
  fp->nxt = NULL;
  csound->oparms->RTevents = 1;
  return 0;
}


#define STA(x)   (csound->lineventStatics.x)
#define MAXSTR 1048576 /* 1MB */

#ifndef O_NDELAY
#define O_NDELAY 0
#endif

void linevent_open(CSOUND *csound)
/* set up Linebuf & ready the input files */
{ /*     callable once from musmon.c        */
    OPARMS  *O = csound->oparms;
    /* csound->lineventGlobals = (LINEVENT_GLOBALS*) */
    /*                            csound->Calloc(csound, */
    /*                            sizeof(LINEVENT_GLOBALS)); */

    STA(linebufsiz) = LBUFSIZ1;
    STA(Linebuf) = (char *) csound->Calloc(csound, STA(linebufsiz));
    STA(orchestrab) = (char *) csound->Calloc(csound, MAXSTR);
    STA(orchestra) = STA(orchestrab);
    STA(prve).opcod = ' ';
    STA(Linebufend) = STA(Linebuf) + STA(linebufsiz);
    STA(Linep) = STA(Linebuf);
    if (strcmp(O->Linename, "stdin") == 0) {
#if defined(DOSGCC) || defined(WIN32)
      setvbuf(stdin, NULL, _IONBF, 0);
      /* WARNING("-L stdin:  system has no fcntl function to get stdin"); */
#else
      STA(stdmode) = fcntl(csound->Linefd, F_GETFL, 0);
      if (UNLIKELY(fcntl(csound->Linefd, F_SETFL, STA(stdmode) | O_NDELAY) < 0))
        csoundDie(csound, Str("-L stdin fcntl failed"));
#endif
    }
#ifdef PIPES
    else if (UNLIKELY(O->Linename[0] == '|')) {
      csound->Linepipe = _popen(&(O->Linename[1]), "r");
      if (LIKELY(csound->Linepipe != NULL)) {
        csound->Linefd = fileno(csound->Linepipe);
        setvbuf(csound->Linepipe, NULL, _IONBF, 0);
      }
      else csoundDie(csound, Str("Cannot open %s"), O->Linename);
    }
#endif
#define MODE ,0
    else
      if (UNLIKELY((csound->Linefd=open(O->Linename, O_RDONLY|O_NDELAY MODE)) < 0))
        csoundDie(csound, Str("Cannot open %s"), O->Linename);
    if(csound->oparms->odebug)
    csound->Message(csound, Str("stdmode = %.8x Linefd = %d\n"),
                    STA(stdmode), csound->Linefd);

        // allocate pfield memory
    if(STA(pfields) == NULL) {
     STA(msize) = PMAX;
     STA(pfields) = csound->Calloc(csound, sizeof(MYFLT)*(STA(msize)+1));
    }


    set_sense_event_callback(csound, sense_line, NULL);
}

#ifdef PIPES
int32_t _pclose(FILE*);
#endif

void linevent_close(CSOUND *csound)
{
    if (csound->oparms->Linein == 0)
      return;
    csound->oparms->Linein = 0;
    if(csound->oparms->odebug)
    csound->Message(csound, Str("stdmode = %.8x Linefd = %d\n"),
                    STA(stdmode), csound->Linefd);
#ifdef PIPES
    if (csound->oparms->Linename[0] == '|')
      _pclose(csound->Linepipe);
    else
#endif
      {
        if (strcmp(csound->oparms->Linename, "stdin") != 0)
          close(csound->Linefd);
#if !defined(DOSGCC) && !defined(WIN32)
        else
          if (UNLIKELY(fcntl(csound->Linefd, F_SETFL, STA(stdmode))))
            csoundDie(csound, Str("Failed to set file status\n"));
#endif
      }

//csound->Free(csound, csound->lineventGlobals);
//csound->lineventGlobals = NULL;
}

/* does string segment contain LF? */

static inline int32_t containsLF(char *cp, char *endp)
{
    while (cp < endp) {
      if (UNLIKELY(*cp++ == LF))
        return 1;
    }
    return 0;
}

static CS_NOINLINE int32_t linevent_alloc(CSOUND *csound, int32_t reallocsize)
{
    volatile jmp_buf tmpExitJmp;
    int32_t         err;
    uint32_t tmp;

    if (reallocsize > 0) {
      /* VL 20-11-17 need to record the STA(Linep) offset
         in relation to STA(Linebuf) */
      tmp = (uint32_t) (STA(Linep) - STA(Linebuf));
      STA(Linebuf) = (char *) csound->ReAlloc(csound,
                                              (void *) STA(Linebuf), reallocsize);

      STA(linebufsiz) = reallocsize;
      STA(Linebufend) = STA(Linebuf) + STA(linebufsiz);
      /* VL 20-11-17 so we can place it in the correct position
         after reallocation */
      STA(Linep) =  STA(Linebuf) + tmp;
    } else if (STA(Linebuf)==NULL) {
       STA(linebufsiz) = LBUFSIZ1;
       STA(Linebuf) = (char *) csound->Calloc(csound, STA(linebufsiz));
    }
    if (STA(Linebuf) == NULL) {
       return 1;
    }
    //csound->Message(csound, "1. realloc: %d\n", reallocsize);
    if (STA(Linep)) return 0;
    csound->Linefd = -1;
    memcpy((void*) &tmpExitJmp, (void*) &csound->exitjmp, sizeof(jmp_buf));
    if ((err = setjmp(csound->exitjmp)) != 0) {
      memcpy((void*) &csound->exitjmp, (void*) &tmpExitJmp, sizeof(jmp_buf));
      //csound->lineventGlobals = NULL;
      return -1;
    }

    // allocate pfield memory
    if(STA(pfields) == NULL) {
     STA(msize) = PMAX;
     STA(pfields) = csound->Calloc(csound, sizeof(MYFLT)*(STA(msize)+1));
    }

    memcpy((void*) &csound->exitjmp, (void*) &tmpExitJmp, sizeof(jmp_buf));
    STA(prve).opcod = ' ';
    STA(Linebufend) = STA(Linebuf) + STA(linebufsiz);
    STA(Linep) = STA(Linebuf);
    set_sense_event_callback(csound, sense_line, NULL);

    return 0;
}

/* insert text from an external source,
   to be interpreted as if coming in from stdin/Linefd for -L */

void csoundInputMessage(CSOUND *csound, const char *message) {
    int32  size = (int32) strlen(message);
#if 1
    int32_t n;
#endif

    if ((n=linevent_alloc(csound, 0)) != 0) return;

    if (!size) return;
    if (UNLIKELY((STA(Linep) + size) >= STA(Linebufend))) {
      int32_t extralloc = (int32_t) (STA(Linep) + size - STA(Linebufend));
      csound->Message(csound, "realloc %d\n", extralloc);
      // csound->Message(csound, "extralloc: %d %d %d\n",
      //                 extralloc, size, (int)(STA(Linebufend) - STA(Linep)));
      // FIXME -- Coverity points out that this test is always false
      // and n is never used
#if 1
      if ((n=linevent_alloc(csound, (STA(linebufsiz) + extralloc))) != 0) {
        csoundErrorMsg(csound, Str("LineBuffer Overflow - "
                                   "Input Data has been Lost"));
        return;
      }
#else
      (void) linevent_alloc(csound, (STA(linebufsiz) + extralloc));

#endif
    }
    //csound->Message(csound, "%u = %u\n", (STA(Linep) + size),  STA(Linebufend) );
    memcpy(STA(Linep), message, size);
    if (STA(Linep)[size - 1] != (char) '\n')
      STA(Linep)[size++] = (char) '\n';
    STA(Linep) += size;
}

/* accumlate RT Linein buffer, & place completed events in EVTBLK */
/* does more syntax checking than rdscor, since not preprocessed  */

void sense_line(CSOUND *csound, void *userData)
{
    char    *cp, *Linestart, *Linend;
    int32_t     c, cm1, cpp1, n, pcnt, oflag = STA(oflag);
    IGN(userData);

    while (1) {
      if(STA(oflag) > oflag) break;
      Linend = STA(Linep);
      if (csound->Linefd >= 0) {
        n = (int32_t) read(csound->Linefd, Linend, STA(Linebufend) - Linend);
        Linend += (n > 0 ? n : 0);
      }
      if (Linend <= STA(Linebuf))
        break;
      Linestart = STA(Linebuf);
      cp = Linestart;

      while (containsLF(Linestart, Linend)) {
        EVTBLK  e;
        MYFLT *pfields = STA(pfields);
        char    *sstrp = NULL;
        int32_t     scnt = 0;
        int32_t     strsiz = 0;
        memset(&e, 0, sizeof(EVTBLK));
        memset(pfields, 0, STA(msize)*sizeof(MYFLT));
        e.p = (MYFLT *) pfields;
        e.strarg = NULL; e.scnt = 0;
        c = *cp;
        while (isblank(c))              /* skip initial white space */
          c = *(++cp);
        if (c == LF) {                  /* if null line, bugout     */
          Linestart = (++cp);
          continue;
        }
        cm1 = *(cp-1);
        cpp1 = *(cp+1);

        /* new orchestra input
         */
        if(STA(oflag)) {
          if(c == '}' && cm1 != '}' && cpp1 != '}') {
            STA(oflag) = 0;
            STA(orchestra) = STA(orchestrab);
            csoundCompileOrc(csound, STA(orchestrab), 0);
            csound->Message(csound, "::compiling orchestra::\n");
            Linestart = (++cp);
            continue;
          }
          else {
            char *pc;
            memcpy(STA(orchestra), Linestart, Linend - Linestart);
            STA(orchestra) += (Linend - Linestart);
            *STA(orchestra) = '\0';
            STA(oflag)++;
            if((pc = strrchr(STA(orchestrab), '}')) != NULL) {

              if(*(pc-1) != '}') {
              *pc = '\0';
               cp = strrchr(Linestart, '}');
              } else {
               Linestart = Linend;
              }
              } else {
              Linestart = Linend;
            }
            continue;
          }
        } else if(c == '{') {
          STA(oflag) = 1;
          csound->Message(csound,
                          "::reading orchestra, use '}' to terminate::\n");
          cp++;
          continue;
        }

        switch (c) {                    /* look for legal opcode    */
        case 'e':                       /* Quit realtime            */
        case 'i':
        case 'q':
        case 'f':
        case 'a':
        case 'd':
          e.opcod = c;
          break;
        default:
          csound->ErrorMsg(csound, Str("unknown opcode %c"), c);
          goto Lerr;
        }                                       /* for params that follow:  */
        pcnt = 0;
        do {
          char  *newcp;
          do {                                  /* skip white space */
            c = *(++cp);
          } while (isblank(c));
          if (c == LF)
            break;
          pcnt++;
          if (c == '"') {                       /* if find character string */
            if (e.strarg == NULL)
              e.strarg = csound->Malloc(csound, strsiz=SSTRSIZ);
            sstrp = e.strarg;
            n = scnt;
            while (n-->0) sstrp += strlen(sstrp)+1;
            n = 0;

            while ((c = *(++cp)) != '"') {
              /* VL: allow strings to be multi-line */
              // if (UNLIKELY(c == LF)) {
              //  csound->ErrorMsg(csound, Str("unmatched quotes"));
              //  goto Lerr;
              //}

              if(c == '\\') {
                cp++;
                if(*cp == '"') c = *cp; /* if it is a double quote */
                /* otherwise we ignore it */
                else cp--;
              }
              sstrp[n++] = c;                   /*   save in private strbuf */

              if (UNLIKELY((sstrp-e.strarg)+n >= strsiz-10)) {
                e.strarg = csound->ReAlloc(csound, e.strarg, strsiz+=SSTRSIZ);
                sstrp = e.strarg+n;
              }
            }
            sstrp[n] = '\0';
            {
#ifdef USE_DOUBLE
              int32_t sel = (byte_order()+1)&1;
              union {
                MYFLT d;
                int32 i[2];
              } ch;
              ch.d = SSTRCOD; ch.i[sel] += scnt++;
              /* ensure capacity before writing */
              if (pcnt >= STA(msize)) {
                STA(msize) += PMAX;
                STA(pfields) = e.p = csound->ReAlloc(csound, e.p,
                                  sizeof(MYFLT) * (STA(msize) + 1));
              }
              e.p[pcnt] = ch.d;           /* set as string with count */
#else
              union {
                MYFLT d;
                int32 i;
              } ch;
              ch.d = SSTRCOD; ch.i += scnt++;
              /* ensure capacity before writing */
              if (pcnt >= STA(msize)) {
                STA(msize) += PMAX;
                STA(pfields) = e.p = csound->ReAlloc(csound, e.p,
                                  sizeof(MYFLT) * (STA(msize) + 1));
              }
              e.p[pcnt] = ch.d;           /* set as string with count */
#endif
            }
            e.scnt = scnt;

            // printf("string: %s\n", sstrp);
            continue;
          }
          if (UNLIKELY(!(isdigit(c) || c == '+' || c == '-' || c == '.')))
            goto Lerr;
          if (c == '.' &&                       /*  if lone dot,       */
              (isblank(n = cp[1]) || n == LF)) {
            if (UNLIKELY(e.opcod != 'i' ||
                         STA(prve).opcod != 'i' || pcnt > STA(prve).pcnt)) {
              csound->ErrorMsg(csound, Str("dot carry has no reference"));
              goto Lerr;
            }                                   /*        pfld carry   */
            e.p[pcnt] = STA(prve).p[pcnt];
            if (UNLIKELY(IsStringCode(e.p[pcnt]))) {
              csound->ErrorMsg(csound, Str("cannot carry string p-field"));
              goto Lerr;
            }
            continue;
          }
          {
            MYFLT tmpv = (MYFLT) csoundStrtod(cp, &newcp);
            cp = newcp - 1;
            /* ensure capacity before writing */
            if (pcnt >= STA(msize)) {
              STA(msize) += PMAX;
              STA(pfields) = e.p = csound->ReAlloc(csound, e.p,
                                sizeof(MYFLT) * (STA(msize) + 1));
            }
            e.p[pcnt] = tmpv;
          }

        } while (1);

        if (e.opcod =='f' && e.p[1]<FL(0.0)); /* an OK case */
        else  /* Check for sufficient pfields (0-based, opcode counted already). */
          if (UNLIKELY(pcnt < 2 && e.opcod != 'e')) {
            csound->ErrorMsg(csound, Str("too few pfields (%d)"), pcnt + 1);
            goto Lerr;
          }
        if (UNLIKELY(pcnt > 1 && e.p[2] < FL(0.0))) {
          csound->ErrorMsg(csound, Str("-L with negative p2 illegal"));
          goto Lerr;
        }
        e.pcnt = pcnt;                          /*   &  record pfld count    */
        if (e.opcod == 'i') {                   /* do carries for instr data */
          // free carry p-fields
          if(STA(prve).p != NULL) csound->Free(csound, STA(prve).p);
          // copy evtblk (except p-field pointer at the end)
          memcpy((void*) &STA(prve), (void*) &e, sizeof(EVTBLK) - sizeof(MYFLT*));
          // allocate and copy new carry p-fields
          STA(prve).p = csound->Calloc(csound, sizeof(MYFLT)*(e.pcnt+1));
          memcpy(STA(prve).p, e.p, sizeof(MYFLT)*(e.pcnt+1));

          /* FIXME: how to carry string args ? */
          STA(prve).strarg = NULL;
        }
        if (UNLIKELY(pcnt >= PMAX && c != LF)) {
          csound->ErrorMsg(csound, Str("too many pfields"));
          while (*(++cp) != LF)                 /* flush any excess data     */
            ;
        }
        Linestart = (++cp);
        if (e.opcod =='e' && e.p[1]>FL(0.0)) {
          e.p[2] = e.p[1];
          e.pcnt = 2;
        }
        insert_event_at_sample(csound, &e, e.p+1, csound->icurTimeSamples);
        continue;
      Lerr:
        n = (int32_t) (cp - Linestart);                     /* error position */
        while (*cp != LF)
          cp++;                                 /* go on to LF    */
        *cp = '\0';                             /*  & insert NULL */
        csound->ErrorMsg(csound, Str("illegal RT scoreline:\n%s\n%*s"),
                                 Linestart, n + 1, "^");  /* mark the error */
        Linestart = (++cp);
      }
      if (Linestart != &(STA(Linebuf)[0])) {
        int32_t len = (int32_t) (Linend - Linestart);
        /* move any remaining characters to the beginning of the buffer */
        for (n = 0; n < len; n++)
          STA(Linebuf)[n] = Linestart[n];
        n = (int32_t) (Linestart - &(STA(Linebuf)[0]));
        STA(Linep) -= n;
        Linend -= n;
      }
      if (Linend == STA(Linep))      /* return if no more data is available  */
        break;
      STA(Linep) = Linend;                       /* accum the chars          */
    }

}






