/*
  entry.c:

  Copyright (C) 1991-2025 Barry Vercoe, John ffitch, Victor Lazzarini, Steven Yi

  This file is part of Csound

  The Csound Library is free software; you can redistribute it
  and/or modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  Csound is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with Csound; if not, write to the Free Software
  Foundation, Inc., 31 Milk Street, #960789, Boston, MA, 02196, USA
*/

#include "entry.h"
#include "interlocks.h"

/* inarg types include the following:
   i       irate scalar
   k       krate scalar
   a       arate vector
   f       frequency variable
   w       spectral variable
   x       krate scalar or arate vector
   S       String
   T       String or i-rate
   U       String or i/k-rate
   B       Boolean k-rate
   b       Boolean i-rate; internally generated as required
   l       Label
   .       required arg of any-type
   and codes
   ?       optional arg of any-type
   m       begins an indef list of iargs (any count)
   M       begins an indef list of args (any count/rate i,k,a)
   N       begins an indef list of args (any count/rate i,k,a,S)
   o       optional i-rate, defaulting to  0
   p              "             "          1
   q              "             "         10
   v              "             "          .5
   j              "             "         -1
   h              "             "        127
   O       optional k-rate, defaulting to  0
   J              "             "         -1
   V              "             "          .5
   P              "             "          1
   W       begins indef list of Strings (any count)
   y       begins indef list of aargs (any count)
   z       begins indef list of kargs (any count)
   Z       begins alternating kakaka...list (any count)
*/

/* outarg types include:
   i, k, a, S as  above
   *       multiple out args of any-type
   m       multiple out aargs
   z       multiple out kargs
   I       multiple out irate (not implemented yet)
   s       deprecated (use a or k as required)
   X       multiple args (a, k, or i-rate)     IV - Sep 1 2002
   N       multiple args (a, k, i, or S-rate)
   F       multiple args (f-rate)#
*/

/* inargs and outargs may also be arrays, e.g. "a[b]" is an array of
   arate vectors. Then for polymorphic opcode entries, "opcode.a" is
   for arate vectors, and "opcode.A" is for arrays of arate vectors.
*/

const OENTRY opcodlst_1[] = {
  /* opcode   dspace  flags  thread  outarg  inargs  isub    ksub    asub    */
  { "",     0,      0,          "",     "",     NULL, NULL, NULL, NULL },
  { "instr",  0,    0,            "",     "",   NULL, NULL, NULL, NULL },
  { "endin",  0,    0,            "",     "",   NULL, NULL, NULL, NULL },
  /* IV - Sep 8 2002 */
  { "opcode",  0,    0,            "",     "",   NULL, NULL, NULL, NULL },
  { "endop",   0,    0,            "",     "",   NULL, NULL, NULL, NULL },
  { "declare", 0,    0,            "",     "",   NULL, NULL, NULL, NULL },
  { "$label", S(LBLBLK),  0,      "",     "",   NULL, NULL, NULL, NULL },
  { "pset",   S(PVSET),   0,      "",     "m",  NULL, NULL, NULL, NULL },
  { "##userOpcode", S(UOPCODE),0,  "", "", (SUBR) useropcdset, (SUBR) useropcd, NULL, NULL },
  { "xin",  S(XIN_MAX),0,     "****************", "",  xinset,  NULL, NULL, NULL },
  { "xout", S(XOUT_MAX),0,    "",         "*", xoutset, NULL, NULL, NULL },
  { "setksmps", S(SETKSMPS),0,    "",   "i", setksmpsset, NULL, NULL },
  { "oversample", S(OVSMPLE),0,    "",   "ioj", oversampleset, NULL, NULL },
  { "undersample", S(OVSMPLE),0,   "",   "ioj", undersampleset, NULL, NULL },
  { "ctrlinit",S(CTLINIT),0,      "",  "im", ctrlinit, NULL, NULL, NULL},
  { "ctrlinit.S",S(CTLINITS),0,      "",  "Sm", ctrlnameinit, NULL, NULL, NULL},
  { "ctrlsave",S(SAVECTRL),0,       "k[]","im", savectrl_init, savectrl_perf, NULL, NULL},
  { "ctrlprint.S",S(PRINTCTRL),0, "", "k[]S", printctrl_init1, printctrl, NULL},
  { "ctrlprint",S(PRINTCTRL),0,       "", "k[]", printctrl_init, printctrl, NULL},
  { "ctrlpreset", S(PRESETCTRL1), 0, "k", "kk[]", presetctrl1_init, presetctrl1_perf, NULL},
  { "ctrlpreset", S(PRESETCTRL), 0, "k", "kim", presetctrl_init, presetctrl_perf, NULL},
  { "ctrlselect", S(SELECTCTRL), 0,"",   "k", selectctrl_init, selectctrl_perf, NULL },
  { "ctrlprintpresets", S(PRINTPRESETS), 0, "", "", printpresets_init, printpresets_perf, NULL},
  { "ctrlprintpresets.S", S(PRINTPRESETS), 0, "", "S", printpresets_init1, printpresets_perf, NULL},
  { "massign",S(MASSIGN), 0,      "",  "iip",massign_p, NULL, NULL, NULL},
  { "massign.iS",S(MASSIGNS), 0,  "",  "iSp",massign_S, NULL, NULL, NULL},
  { "turnon", S(TURNON),  0,      "",     "io", turnon, NULL, NULL, NULL},
  { "turnon.S", S(TURNON),  0,    "",     "So", turnon_S, NULL, NULL, NULL},
  { "remoteport", S(REMOTEPORT), 0, "",  "i", remoteport, NULL, NULL, NULL},
  { "insremot",S(INSREMOT),0,     "",     "SSm",insremot, NULL, NULL, NULL},
  { "midremot",S(MIDREMOT),0,     "",     "SSm",midremot, NULL, NULL, NULL},
  { "insglobal",S(INSGLOBAL),0,   "",     "Sm", insglobal, NULL, NULL, NULL},
  { "midglobal",S(MIDGLOBAL),0,   "",     "Sm", midglobal, NULL, NULL, NULL},
  { "ihold",  S(LINK),0,          "",     "",     ihold, NULL, NULL, NULL  },
  { "turnoff",S(LINK),0,           "",     "",     NULL,   turnoff, NULL, NULL },
  { "offsetsmps", S(AOP), 0, "k", "", NULL,(SUBR) sa_offset, NULL},
  { "earlysmps", S(AOP), 0, "k", "", NULL,(SUBR) sa_early, NULL},
  { "str", S(IREF_NUM) ,0,  "S", ":InstrDef;", (SUBR) get_instr_name},
  {  "=.S",   S(STRCPY_OP),0,     "S",    "S",
     (SUBR) strcpy_opcode_S, (SUBR) strassign_k, (SUBR) NULL, NULL    },
  {  "#=.S",   S(STRCPY_OP),0,     "S",    "S",
     (SUBR) strcpy_opcode_S, (SUBR) strassign_k, (SUBR) NULL, NULL    },
  {  "=.T",   S(STRCPY_OP),0,     "S",    "i",
     (SUBR) strcpy_opcode_p, (SUBR) NULL, (SUBR) NULL, NULL                 },
  { "=.r",    S(ASSIGN),0,      "r",    "i",    rassign, NULL, NULL, NULL },
  { "=.Ck",    S(R2CXOP),0,     ":Complex;",    "kkO", (SUBR) complex_assign, (SUBR) complex_assign},
  //{ "=.Ci",    S(R2CXOP),0,     ":Complex;",    "iio", (SUBR) complex_assign},
  { "=.C",    S(CXOP),0,        ":Complex;",    ":Complex;", (SUBR) complex_init,
    (SUBR) complex_init},
  { "=.Ca",    S(COPS1),0,        ":Complex;[]",    "aa", (SUBR) cops_init_a,
    (SUBR) complex_array_assign},
  { "=.C[]k[]k[]",    S(COPS1),0,        ":Complex;[]",    "k[]k[]",
    (SUBR) cops_init_a, (SUBR) complex_array_assign},
  { "=.k[]a", sizeof(A2ARR), 0, "k[]", "a", (SUBR)asig2array_init,
    (SUBR)asig2array_perf},
  { "=.i",    S(ASSIGNM),0,       "IIIIIIIIIIIIIIIIIIIIIIII", "m",
    minit, NULL, NULL, NULL  },
  { "init.b",    S(ASSIGNM),0,       "vvvvvvvvvvvvvvvvvvvvvvvv", "m",
    (SUBR) binit, NULL, NULL, NULL  },
  { "init.B",    S(ASSIGNM),0,       "VVVVVVVVVVVVVVVVVVVVVVVV", "m",
    (SUBR) binit, NULL, NULL, NULL  },
  { "=.bi",    S(RELAT),0,       "b", "i", (SUBR) bassign, NULL, NULL, NULL  },
  { "=.B",    S(RELAT),0,       "B", "b", (SUBR) b2b, NULL, NULL, NULL  },
  { "=.Bk",    S(RELAT),0,       "B", "k", NULL, (SUBR) bassign, NULL, NULL  },
  { "b",    S(RELAT),0,       "b", "k", (SUBR) bassign, NULL, NULL, NULL  },
  { "=.bb",    S(RELAT),0,       "b", "b", (SUBR) b2b, NULL, NULL, NULL  },

  { "B",    S(RELAT),0,       "B", "k", NULL, (SUBR) bassign, NULL, NULL  },
  { "=.ib",    S(ASSIGN),0,       "i", "b", (SUBR) b2s, NULL, NULL, NULL  },
  { "=.kB",    S(ASSIGN),0,       "k", "B", NULL, (SUBR) b2s, NULL, NULL  },
  { "i",    S(ASSIGN),0,       "i", "b", (SUBR) b2s, NULL, NULL, NULL  },
  { "k",    S(ASSIGN),0,       "k", "B", NULL, (SUBR) b2s, NULL, NULL  },
  { "=.k",    S(ASSIGNM),0,        "zzzzzzzzzzzzzzzzzzzzzzzz", "z",
    NULL, minit, NULL, NULL },
  { "=.a",    S(ASSIGN),0,         "a",    "a",    NULL, gaassign, NULL },
  { "=.y",    S(ASSIGNM),0,         "mmmmmmmmmmmmmmmmmmmmmmmm",    "y",    NULL, (SUBR) mainit2, NULL },
  { "=.l",    S(ASSIGN),0,         "a",    "a",    NULL,   laassign, NULL },
  { "=.up",   S(UPSAMP),0,         "a",    "k",  NULL, (SUBR)upsamp, NULL },
  { "=.down",   S(DOWNSAMP),0,    "k",    "ao",   (SUBR)downset,(SUBR)downsamp },
  { "init.C",    S(R2CXOP),0,        ":Complex;",    "iio", (SUBR) complex_assign},
  { "init.C",    S(CXOP),0,        ":Complex;",    ":Complex;", (SUBR) complex_init},
  { "init.S", S(STRCPY_OP),0,       "S", "S", (SUBR) strcpy_opcode_S  },
  { "init.Si", S(STRCPY_OP),0,       "S", "i", (SUBR) strcpy_opcode_p  },
  { "init", S(IREF_INIT),0,     ":InstrDef;", "i", (SUBR) init_instr_ref},
  { "init.i", S(ASSIGNM),0,       "IIIIIIIIIIIIIIIIIIIIIIII", "m", minit  },
  { "init.k", S(ASSIGNM),0,       "zzzzzzzzzzzzzzzzzzzzzzzz", "m", minit  },
  { "init.a", S(ASSIGNM),0,       "mmmmmmmmmmmmmmmmmmmmmmmm", "m", mainit },
  { "unschedule", S(RMEVT), 0, "","*", remove_event},
  { "unscheduleall", S(RMEVT), 0, "","*", remove_all_events},
  { "instrnum",   S(IREF_NUM),0,    "i",    ":InstrDef;o",   (SUBR) get_instr_num },
  { "nstrnum",   S(IREF_NUM),0,    "i",    ":InstrDef;o",   (SUBR) get_instr_num },
  { "opcodeinfo", S(OPINFO) ,0,  "", ":OpcodeDef;", (SUBR) opcode_info},
  { "=.opcd", S(ASSIGN), 0, ":OpcodeDef;", ":OpcodeDef;", (SUBR) copy_var_generic},
  { "init.opcd", S(ASSIGN), 0, ":OpcodeDef;", "S", (SUBR) opcode_ref},
  { "opcoderef", S(ASSIGN), 0, ":OpcodeDef;", "S", (SUBR) opcode_ref},
  { "create", S(AOP),0,  ":Opcode;", ":OpcodeDef;o", (SUBR) create_opcode_simple},
  { "create", S(OPARRAY),0,  ":Opcode;[]", ":OpcodeDef;io", (SUBR) create_opcode_array},
  { "delete", S(AOP),0, "", ":Opcode;", NULL, NULL, (SUBR) opcode_delete},
  { "delete", S(AOP),0, "", ":Opcode;[]", NULL, NULL, (SUBR) opcode_delete_array},
  { "opcodeinfo", S(OPINFO) ,0,  "", ":Opcode;", (SUBR) opcode_object_info},
  { "=.Opcode", S(ASSIGN) ,0,  ":Opcode;", ":Opcode;", (SUBR) copy_opcode_obj,
    (SUBR) copy_opcode_obj},
  { "run", S(OPRUN), 0, "*", ":Opcode;*", (SUBR) opcode_object_init, (SUBR) opcode_run_perf},
  { "run", S(OPRUN), 0, "*", ":Opcode;[]*", (SUBR) opcode_array_init, (SUBR) opcode_array_perf},
  { "init", S(OPRUN), 0, "*", ":Opcode;*", (SUBR) opcode_object_init},
  { "perf", S(OPRUN), 0, "*", ":Opcode;*", NULL, (SUBR) opcode_object_perf},
  { "setp", S(AOP), 0, "", ":Opcode;k.", NULL, (SUBR) set_opcode_param },
  { "getp", S(AOP), 0, ".", ":Opcode;k", NULL, (SUBR) get_opcode_output },
  { "init.instr", S(ASSIGN) ,0,  ":InstrDef;", ":InstrDef;", (SUBR) copy_var_generic_init},
  { "floatsize", S(ASSIGN) ,0, "i", "", myflt_size },
  /* VL 4.4.24 removing thread field:
     These boolean were all marked thread 0, with both init and perf functions.
     At instance(), there was a check for NOT thread 3 (meaning init XOR perf),
     and a check for type 'b' output was made, placing the opcode in the init chain,
     or else perf chain.
     I'm Reinterpreting them as either init or perf by removing the relevant function.
     This should take care of assigning these to the correct chain.
  */
  { ">",      S(RELAT),0,  /* 0,*/      "b",    "ii",   gt,    NULL            },
  { ">.0",      S(RELAT),0,  /* 0,*/     "B",    "kk",   NULL,     gt              },
  { ">=",     S(RELAT),0,  /* 0,*/       "b",    "ii",   ge,     NULL             },
  { ">=.0",     S(RELAT),0,  /* 0,*/      "B",    "kk",   NULL,     ge              },
  { "<",      S(RELAT),0,  /* 0,*/       "b",    "ii",   lt,     NULL              },
  { "<.0",      S(RELAT),0,  /* 0,*/       "B",    "kk",   NULL,     lt              },
  { "<=",     S(RELAT),0,  /* 0,*/      "b",    "ii",   le,     NULL             },
  { "<=.0",     S(RELAT),0,  /* 0,*/      "B",    "kk",   NULL,     le              },
  { "==",     S(RELAT),0,  /* 0,*/       "b",    "ii",   eq,     NULL              },
  { "==.S",     S(EQS_OP),0,  /* 0,*/       "b",    "SS",   eqs,     NULL              },
  { "==",     S(RELAT),0,  /* 0,*/       "b",    "bb",   eq,     NULL              },
  { "==",     S(RELAT),0,  /* 0,*/       "b",    "ib",   eq,     NULL              },
  { "==",     S(RELAT),0,  /* 0,*/       "b",    "bi",   eq,     NULL              },
  { "==.0",     S(RELAT),0,  /* 0,*/       "B",    "kk",   NULL,     eq              },
  { "==.1",     S(RELAT),0,  /* 0,*/       "B",    "BB",   NULL,     eq              },
  { "!=",     S(RELAT),0,  /* 0,*/       "b",    "ii",   ne,     NULL              },
  { "!=.0",     S(RELAT),0,  /* 0,*/       "B",    "kk",  NULL,     ne              },
  { "!",      S(LOGCL),0,  /* 0,*/       "b",    "b",    b_not,    NULL        },
  { "!.0",      S(LOGCL),0, /* 0,*/       "B",    "B",    NULL,    b_not         },
  { "&&",     S(LOGCL),0,   /* 0,*/      "b",    "bb",   and,    NULL             },
  { "&&.0",     S(LOGCL),0,   /* 0,*/       "B",    "BB",   NULL,    and             },
  { "||",     S(LOGCL),0,  /* 0,*/     "b",    "bb",   or,     NULL             },
  { "||.0",     S(LOGCL),0,  /* 0,*/       "B",    "BB",   NULL,     or              },
  /* end boolean */
  { ":cond.i",     S(CONVAL),0,        "i",    "bii",  conval                  },
  { ":cond.k",     S(CONVAL),0,         "k",    "Bkk",  NULL,   conval          },
  { ":cond.a",     S(CONVAL),0,         "a",    "Bxx",  NULL,   aconval },
  { ":cond.s",     S(CONVAL),0,        "S",    "bSS",  conval, NULL         },
  { ":cond.S",     S(CONVAL),0,        "S",    "BSS",  conval, conval       },
  { "##add.ii",  S(AOP),0,          "i",    "ii",   addkk                   },
  { "##sub.ii",  S(AOP),0,          "i",    "ii",   subkk                   },
  { "##mul.ii",  S(AOP),0,          "i",    "ii",   mulkk                   },
  { "##div.ii",  S(AOP),0,          "i",    "ii",   divkk                   },
  { "##add.CC",  S(CXOP),0,          ":Complex;",    ":Complex;:Complex;",
    (SUBR) complex_add, (SUBR) complex_add },
  { "##sub.CC",  S(CXOP),0,          ":Complex;",    ":Complex;:Complex;",
    (SUBR) complex_sub, (SUBR) complex_sub },
  { "##mul.CC",  S(CXOP),0,          ":Complex;",    ":Complex;:Complex;",
    (SUBR) complex_prod, (SUBR) complex_prod },
  { "##div.CC",  S(CXOP),0,          ":Complex;",    ":Complex;:Complex;",
    (SUBR) complex_div, (SUBR) complex_div },

  { "##add.Ci",  S(AOP),0,          ":Complex;",    ":Complex;i",
    (SUBR) complex_add_real},
  { "##add.Ck",  S(AOP),0,          ":Complex;",    ":Complex;k", NULL,
    (SUBR) complex_add_real},
  { "##add.iC",  S(AOP),0,          ":Complex;",    "i:Complex;",
    (SUBR) real_add_complex},
  { "##add.kC",  S(AOP),0,          ":Complex;",    "k:Complex;", NULL,
    (SUBR) real_add_complex},

  { "##sub.Ci",  S(AOP),0,          ":Complex;",    ":Complex;i",
    (SUBR) complex_sub_real},
  { "##sub.Ck",  S(AOP),0,          ":Complex;",    ":Complex;k", NULL,
    (SUBR) complex_sub_real},
  { "##sub.iC",  S(AOP),0,          ":Complex;",    "i:Complex;",
    (SUBR) real_sub_complex},
  { "##sub.kC",  S(AOP),0,          ":Complex;",    "k:Complex;", NULL,
    (SUBR) real_sub_complex},
  { "##mul.Ci",  S(AOP),0,          ":Complex;",    ":Complex;i",
    (SUBR) complex_mul_real},
  { "##mul.Ck",  S(AOP),0,          ":Complex;",    ":Complex;k", NULL,
    (SUBR) complex_mul_real},
  { "##mul.iC",  S(AOP),0,          ":Complex;",    "i:Complex;",
    (SUBR) real_mul_complex},
  { "##mul.kC",  S(AOP),0,          ":Complex;",    "k:Complex;", NULL,
    (SUBR) real_mul_complex},
  { "##div.Ci",  S(AOP),0,          ":Complex;",    ":Complex;i",
    (SUBR) complex_div_real},
  { "##div.Ck",  S(AOP),0,          ":Complex;",    ":Complex;k", NULL,
    (SUBR) complex_div_real},
  { "##div.iC",  S(AOP),0,          ":Complex;",    "i:Complex;",
    (SUBR) real_div_complex},
  { "##div.kC",  S(AOP),0,          ":Complex;",    "k:Complex;", NULL,
    (SUBR) real_div_complex},
  { "##mod.ii",  S(AOP),0,          "i",    "ii",   modkk                   },
  { "##add.kk",  S(AOP),0,           "k",    "kk",   NULL,   addkk           },
  { "##sub.kk",  S(AOP),0,           "k",    "kk",   NULL,   subkk           },
  { "##mul.kk",  S(AOP),0,           "k",    "kk",   NULL,   mulkk           },
  { "##div.kk",  S(AOP),0,           "k",    "kk",   NULL,   divkk           },
  { "##mod.kk",  S(AOP),0,           "k",    "kk",   NULL,   modkk           },
  { "##add.ka",  S(AOP),0,           "a",    "ka",   NULL,   addka   },
  { "##sub.ka",  S(AOP),0,           "a",    "ka",   NULL,   subka   },
  { "##mul.ka",  S(AOP),0,           "a",    "ka",   NULL,   mulka   },
  { "##div.ka",  S(AOP),0,           "a",    "ka",   NULL,   divka   },
  { "##mod.ka",  S(AOP),0,           "a",    "ka",   NULL,   modka   },
  { "##add.ak",  S(AOP),0,           "a",    "ak",   NULL,   addak   },
  { "##sub.ak",  S(AOP),0,           "a",    "ak",   NULL,   subak   },
  { "##mul.ak",  S(AOP),0,           "a",    "ak",   NULL,   mulak   },
  { "##div.ak",  S(AOP),0,           "a",    "ak",   NULL,   divak   },
  { "##mod.ak",  S(AOP),0,           "a",    "ak",   NULL,   modak   },
  { "##add.aa",  S(AOP),0,           "a",    "aa",   NULL,   addaa   },
  { "##sub.aa",  S(AOP),0,           "a",    "aa",   NULL,   subaa   },
  { "##mul.aa",  S(AOP),0,           "a",    "aa",   NULL,   mulaa   },
  { "##div.aa",  S(AOP),0,           "a",    "aa",   NULL,   divaa   },
  { "##mod.aa",  S(AOP),0,           "a",    "aa",   NULL,   modaa   },
  { "##addin.i", S(ASSIGN),0,       "i",    "i",    addin,  NULL    },
  { "##addin.k", S(ASSIGN),0,        "k",    "k",    NULL,   addin   },
  { "##addin.K", S(ASSIGN),0,        "a",    "k",    NULL,   addinak },
  { "##addin.a", S(ASSIGN),0,        "a",    "a",    NULL,   addina  },
  { "##subin.i", S(ASSIGN),0,       "i",    "i",    subin,  NULL    },
  { "##subin.k", S(ASSIGN),0,        "k",    "k",    NULL,   subin   },
  { "##subin.K", S(ASSIGN),0,        "a",    "k",    NULL,   subinak },
  { "##subin.a", S(ASSIGN),0,        "a",    "a",    NULL,   subina  },
  { "##mulin.i", S(ASSIGN),0,       "i",    "i",    mulin,  NULL    },
  { "##mulin.k", S(ASSIGN),0,        "k",    "k",    NULL,   mulin   },
  { "##mulin.K", S(ASSIGN),0,        "a",    "k",    NULL,   mulinak },
  { "##mulin.a", S(ASSIGN),0,        "a",    "a",    NULL,   mulina  },
  { "##divin.i", S(ASSIGN),0,       "i",    "i",    divin,  NULL    },
  { "##divin.k", S(ASSIGN),0,        "k",    "k",    NULL,   divin   },
  { "##divin.K", S(ASSIGN),0,        "a",    "k",    NULL,   divinak },
  { "##divin.a", S(ASSIGN),0,        "a",    "a",    NULL,   divina  },
  { "init.i", sizeof(ARRAYINIT), 0, "i[]", "m", (SUBR)array_init },
  { "init.k", sizeof(ARRAYINIT), 0, "k[]", "m", (SUBR)array_init },
  { "init.a", sizeof(ARRAYINIT), 0, "a[]", "m", (SUBR)array_init },
  { "init.S", sizeof(ARRAYINIT), 0, "S[]", "m", (SUBR)array_init },
  { "init.0", sizeof(ARRAYINIT), 0, ".[]", "m", (SUBR)array_init },
  { "fillarray.i", sizeof(TABFILL), 0, "i[]", "m", (SUBR)tabfill },
  { "fillarray.k", sizeof(TABFILL), 0, "k[]", "m", (SUBR)tabfill },
  { "fillarray.s", sizeof(TABFILL), 0, "S[]", "W", (SUBR)tabfill },
  { "fillarray.K", sizeof(TABFILL), 0, "k[]", "z", (SUBR)tabfill, (SUBR)tabfill },
  { "fillarray.a", sizeof(TABFILL), 0, "a[]", "y", (SUBR)tabfill, (SUBR)tabfill },
  { "fillarray.f", sizeof(TABFILLF), 0, "k[]", "S", (SUBR)tabfillf },
  { "fillarray.F", sizeof(TABFILLF), 0, "i[]", "S", (SUBR)tabfillf },
  { "fillarray.b", sizeof(TABFILL), 0, "b[]", "m", (SUBR)tabfill },
  { "fillarray.B", sizeof(TABFILL), 0, "B[]", "m", (SUBR)tabfill },
  { "string2array.S", sizeof(TABFILLF), 0, "i[]", "S", (SUBR)tabsfill },
  { "string2array.s", sizeof(TABFILLF), 0, "k[]", "S", (SUBR)tabsfill },
  { "array.k", sizeof(TABFILL), _QQ, "k[]", "m", (SUBR)tabfill     },
  { "array.i", sizeof(TABFILL), _QQ, "i[]", "m", (SUBR)tabfill     },
  { "array.a", sizeof(A2ARR), 0, "k[]", "a", (SUBR)asig2array_init,
    (SUBR)asig2array_perf},
  { "a.A", sizeof(ARR2A), 0, "a", "k[]", NULL, (SUBR)array2asig_perf},
  { "##array_init", sizeof(ARRAY_SET), 0, "", ".[].m", (SUBR)array_set },
  { "##array_set.i", sizeof(ARRAY_SET), 0, "", "i[]im", (SUBR)array_set },
  { "##array_set.k", sizeof(ARRAY_SET), 0, "", "k[]km", NULL,(SUBR)array_set},
  { "##array_set.k", sizeof(ARRAY_SET), 0, "", "k[][]km", NULL,(SUBR)array_set},
  { "##array_set.a", sizeof(ARRAY_SET), 0, "", "a[]am", NULL, (SUBR)array_set},
  { "##array_set.S", sizeof(ARRAY_SET), 0, "", "S[]Sm", (SUBR)array_set, (SUBR)array_set },
  { "##array_set.X", sizeof(ARRAY_SET), 0, "", ".[].m", (SUBR)array_set, (SUBR)array_set },
  { "##array_set_struct", sizeof(ARRAY_SET), 0, "", ".[].m", (SUBR)array_set_struct },

  { "##array_set.C", sizeof(ARRAY_SET), 0, "", ":Complex;[]Complex;m", NULL,
    (SUBR)array_set},
  { "##array_set.e", sizeof(ARRAY_SET), 0, "", "i[].z", (SUBR)array_err },
  { "##array_set.x", sizeof(ARRAY_SET), 0, "", ".[].z", NULL, (SUBR)array_set},
  { "##array_get.i", sizeof(ARRAY_GET), 0, "i", "i[]m", (SUBR)array_get },
  { "##array_get.k", sizeof(ARRAY_GET), 0, "k", "k[]m", (SUBR)array_get, (SUBR)array_get },
  { "##array_get.a", sizeof(ARRAY_GET), 0, "a", "a[]m", (SUBR)array_get, (SUBR)array_get },
  { "##array_get.S", sizeof(ARRAY_GET), 0, "S", "S[]m", (SUBR)array_get, (SUBR)array_get },
  { "##array_geti.S", sizeof(ARRAY_GET), 0, "S", "S[]m", (SUBR)array_get},
  { "##array_get.C", sizeof(ARRAY_GET), 0, ":Complex;", ":Complex;[]m", (SUBR)array_get,
    (SUBR)array_get},
  { "##array_get.b", sizeof(ARRAY_GET), 0, "b", "b[]m",(SUBR)array_get },
  { "##array_get.x", sizeof(ARRAY_GET), 0, ".", ".[]m",(SUBR)array_get },
  { "##array_get.K", sizeof(ARRAY_GET), 0, ".", ".[]z",NULL, (SUBR)array_get},
  { "i.Ai", sizeof(ARRAY_GET),0,      "i",    "k[]m", (SUBR)array_get  },
  { "i.Ak", sizeof(ARRAY_GET),0,      "i",    "k[]z", (SUBR)array_get  },
  {"=.X", sizeof(TABCOPY), 0, "k[]", "k", (SUBR) scalarset, (SUBR) scalarset},
  {"=.Z", sizeof(TABCOPY), 0 , "a[]", "a", NULL, (SUBR) arrayass},
  {"##add.[i]", sizeof(TABARITH), 0, "i[]", "i[]i[]",
   (SUBR)tabaddi},
  {"##add.[s]", sizeof(TABARITH), 0, "a[]", "a[]a[]",
   (SUBR)tabarithset, (SUBR)tabaadd},
  {"##add.[]", sizeof(TABARITH), 0, "k[]", "k[]k[]",
   (SUBR)tabarithset, (SUBR)tabadd},
  {"##sub.[i]", sizeof(TABARITH), 0, "i[]", "i[]i[]",
   (SUBR)tabsubi},
  {"##sub.[a]", sizeof(TABARITH), 0, "a[]", "a[]a[]",
   (SUBR)tabarithset, (SUBR)tabasub},
  {"##sub.[]", sizeof(TABARITH), 0, "k[]", "k[]k[]",
   (SUBR)tabarithset, (SUBR)tabsub},
  {"##mul.[i]", sizeof(TABARITH), 0, "i[]", "i[]i[]",
   (SUBR)tabmulti},
  {"##mul.[a]", sizeof(TABARITH), 0, "a[]", "a[]a[]",
   (SUBR)tabarithset,(SUBR)tabamul},
  {"##mul.[]", sizeof(TABARITH), 0, "k[]", "k[]k[]",
   (SUBR)tabarithset,(SUBR)tabmult},
  {"##div.[i]",  sizeof(TABARITH), 0, "i[]", "i[]i[]",
   (SUBR)tabdivi },
  {"##div.[a]", sizeof(TABARITH), 0, "a[]", "a[]a[]",
   (SUBR)tabarithset, (SUBR)tabadiv},
  {"##div.[]",  sizeof(TABARITH), 0, "k[]", "k[]k[]",
   (SUBR)tabarithset,(SUBR)tabdiv },
  {"##rem.[i]",  sizeof(TABARITH), 0, "i[]", "i[]i[]",
   (SUBR)tabremi},
  {"##rem.[]",  sizeof(TABARITH), 0, "k[]", "k[]k[]",
   (SUBR)tabarithset, (SUBR)tabrem},
  {"##add.[p", sizeof(TABARITH1), 0, "i[]", "i[]i",
   (SUBR)tabaiaddi },
  {"##add.p[", sizeof(TABARITH2), 0, "i[]", "ii[]",
   (SUBR)tabiaaddi },
  {"##add.[i", sizeof(TABARITH1), 0, "k[]", "k[]i",
   (SUBR)tabarithset1, (SUBR)tabaiadd },
  {"##add.i[", sizeof(TABARITH2), 0, "k[]", "ik[]",
   (SUBR)tabarithset2, (SUBR)tabiaadd },
  {"##add.k[a[", sizeof(TABARITH), 0, "a[]", "k[]a[]",
   (SUBR)tabarithset, (SUBR)tabkrardd },
  {"##add.a[k[", sizeof(TABARITH), 0, "a[]", "a[]k[]",
   (SUBR)tabarithset, (SUBR)tabarkrdd },
  {"##addin.[i", sizeof(TABARITHIN1), 0, "i[]", "i",
   (SUBR)addinAA, NULL },

  {"##addin.[k", sizeof(TABARITHIN1), 0, "k[]", "k", NULL,
   (SUBR)addinAA },

  {"##addin.[", sizeof(TABARITHIN), 0, "i[]", "i[]",
   (SUBR)tabaddinkk, NULL },
  {"##addin.[K", sizeof(TABARITHIN), 0, "k[]", "k[]", NULL,
   (SUBR)tabaddinkk },
  {"##addin.[a", sizeof(TABARITHIN), 0, "a[]", "a[]", NULL,
   (SUBR)tabaaddin },
  {"##addin.[a", sizeof(TABARITHIN), 0, "a[]", "a[]", NULL,
   (SUBR)tabaaddin },
  {"##addin.[ak", sizeof(TABARITHIN), 0, "a[]", "k[]", NULL,
   (SUBR)tabarkrddin },
  {"##addin.[aks", sizeof(TABARITHIN), 0, "a[]", "k", NULL,
   (SUBR)tabakaddin },
  {"##addin.[aa", sizeof(TABARITHIN1), 0, "a[]", "a", NULL,
   (SUBR)taba1addin },

  {"##subin.[i", sizeof(TABARITHIN1), 0, "i[]", "i",
   (SUBR)subinAA, NULL },
  {"##subin.[", sizeof(TABARITHIN), 0, "i[]", "i[]",
   (SUBR)tabsubinkk, NULL },
  {"##mulin.[i", sizeof(TABARITHIN1), 0, "i[]", "i",
   (SUBR)mulinAA, NULL },
  {"##mulin.[k", sizeof(TABARITHIN1), 0, "k[]", "k", NULL,
   (SUBR)mulinAA },
  {"##mulin.[", sizeof(TABARITHIN), 0, "i[]", "i[]",
   (SUBR)tabmulinkk, NULL },
  {"##mulin.[K", sizeof(TABARITHIN), 0, "k[]", "k[]", NULL,
   (SUBR)tabmulinkk },
  {"##mulin.[a", sizeof(TABARITHIN), 0, "a[]", "a[]", NULL,
   (SUBR)tabamulin },
  {"##mulin.[ak", sizeof(TABARITHIN), 0, "a[]", "k[]", NULL,
   (SUBR)tabarkrmulin },
  {"##mulin.a[a", sizeof(TABARITHIN1), 0, "a[]", "a", NULL,
   (SUBR)taba1mulin },
  {"##mulin.[aks", sizeof(TABARITHIN), 0, "a[]", "k", NULL,
   (SUBR)tabakmulin },
  {"##divin.[i", sizeof(TABARITHIN1), 0, "i[]", "i",
   (SUBR)divinAA, NULL },
  {"##divin.[", sizeof(TABARITHIN), 0, "i[]", "i[]",
   (SUBR)tabdivinkk, NULL },
  {"##divin.[K", sizeof(TABARITHIN), 0, "k[]", "k[]",
   NULL, (SUBR)tabdivinkk },
  {"##divin.[k", sizeof(TABARITHIN1), 0, "k[]", "k",
   NULL, (SUBR)divinAA },
  {"##divin.[a", sizeof(TABARITHIN), 0, "a[]", "a[]",
   NULL, (SUBR)tabaadivin },
  {"##divin.[aa", sizeof(TABARITHIN1), 0, "a[]", "a", NULL,
   (SUBR)taba1divin },
  {"##divin.[ak", sizeof(TABARITHIN), 0, "a[]", "k[]",
   NULL, (SUBR)tabarkrdivin },
  {"##divin.[aks", sizeof(TABARITHIN), 0, "a[]", "k",
   NULL, (SUBR)tabakdivin },
  {"##sub.[p", sizeof(TABARITH1), 0, "i[]", "i[]i",
   (SUBR)tabaisubi },
  {"##sub.p[", sizeof(TABARITH2), 0, "i[]", "ii[]",
   (SUBR)tabiasubi },
  {"##sub.[i", sizeof(TABARITH1), 0, "k[]", "k[]i",
   (SUBR)tabarithset1, (SUBR)tabaisub },
  {"##sub.i[", sizeof(TABARITH2), 0, "k[]", "ik[]",
   (SUBR)tabarithset2, (SUBR)tabiasub },
  {"##subin.[K", sizeof(TABARITHIN), 0, "k[]", "k[]",
   NULL, (SUBR)tabsubinkk },
  {"##subin.[k", sizeof(TABARITHIN1), 0, "k[]", "k",
   NULL, (SUBR)subinAA },
  {"##subin.[a", sizeof(TABARITHIN), 0, "a[]", "a[]",
   NULL, (SUBR)tabaasubin },
  {"##subin.[aa", sizeof(TABARITHIN1), 0, "a[]", "a", NULL,
   (SUBR)taba1subin },

  {"##subin.[ak", sizeof(TABARITHIN), 0, "a[]", "k[]",
   NULL, (SUBR)tabarkrsbin },
  {"##subdin.[aks", sizeof(TABARITHIN), 0, "a[]", "k",
   NULL, (SUBR)tabaksubin },

  {"##mul.[p", sizeof(TABARITH1), 0, "i[]", "i[]i",
   (SUBR)tabaimulti },
  {"##mul.p[", sizeof(TABARITH2), 0, "i[]", "ii[]",
   (SUBR)tabiamulti },
  {"##sub.k[a[", sizeof(TABARITH), 0, "a[]", "k[]a[]",
   (SUBR)tabarithset, (SUBR)tabkrarsb },
  {"##sub.a[k[", sizeof(TABARITH), 0, "a[]", "a[]k[]",
   (SUBR)tabarithset, (SUBR)tabarkrsb },
  {"##mul.[i", sizeof(TABARITH1), 0, "k[]", "k[]i",
   (SUBR)tabarithset1, (SUBR)tabaimult },
  {"##mul.i[", sizeof(TABARITH2), 0, "k[]", "ik[]",
   (SUBR)tabarithset2, (SUBR)tabiamult },
  {"##mul.k[a[", sizeof(TABARITH), 0, "a[]", "k[]a[]",
   (SUBR)tabarithset, (SUBR)tabkrarml },
  {"##mul.a[k[", sizeof(TABARITH), 0, "a[]", "a[]k[]",
   (SUBR)tabarithset, (SUBR)tabarkrml },
  {"##div.[p",  sizeof(TABARITH1), 0, "i[]", "i[]i",
   (SUBR)tabaidivi },
  {"##div.p[",  sizeof(TABARITH2), 0, "i[]", "ii[]",
   (SUBR)tabiadivi },
  {"##div.[i",  sizeof(TABARITH1), 0, "k[]", "k[]i",
   (SUBR)tabarithset1, (SUBR)tabaidiv },
  {"##div.i[",  sizeof(TABARITH2), 0, "k[]", "ik[]",
   (SUBR)tabarithset2, (SUBR)tabiadiv },
  {"##div.k[a[", sizeof(TABARITH), 0, "a[]", "k[]a[]",
   (SUBR)tabarithset, (SUBR)tabkrardv },
  {"##div.a[k[", sizeof(TABARITH), 0, "a[]", "a[]k[]",
   (SUBR)tabarithset, (SUBR)tabarkrdv },
  {"##rem.[p",  sizeof(TABARITH1),0,  "i[]", "i[]i",
   (SUBR)tabairemi },
  {"##rem.p[",  sizeof(TABARITH2),0,  "i[]", "ii[]",
   (SUBR)tabiaremi },
  {"##rem.[i",  sizeof(TABARITH1),0,  "k[]", "k[]i",
   (SUBR)tabarithset1, (SUBR)tabairem },
  {"##rem.i[",  sizeof(TABARITH2),0,  "k[]", "ik[]",
   (SUBR)tabarithset2, (SUBR)tabiarem },
  {"##rem.a[k[", sizeof(TABARITH), 0, "a[]", "a[]k[]",
   (SUBR)tabarithset, (SUBR)tabarkrmd },
  {"##re.k[a[", sizeof(TABARITH), 0, "a[]", "k[]a[]",
   (SUBR)tabarithset, (SUBR)tabkrarmd },
  {"##add.[k", sizeof(TABARITH1), 0, "k[]", "i[]k",
   (SUBR)tabarithset1, (SUBR)tabaiadd },
  {"##add.[k", sizeof(TABARITH1), 0, "k[]", "k[]k",
   (SUBR)tabarithset1, (SUBR)tabaiadd },
  {"##add.k[", sizeof(TABARITH2), 0, "k[]", "ik[]",
   (SUBR)tabarithset2, (SUBR)tabiaadd },
  {"##add.k[", sizeof(TABARITH2), 0, "k[]", "kk[]",
   (SUBR)tabarithset2, (SUBR)tabiaadd },
  {"##add.[ak", sizeof(TABARITH1), 0, "a[]", "a[]k",
   (SUBR)tabarithset1, (SUBR)tabakadd },
  {"##add.k[a", sizeof(TABARITH2), 0, "a[]", "ka[]",
   (SUBR)tabarithset2, (SUBR)tabkaadd },
  {"##add.aa[", sizeof(TABARITH1), 0, "a[]", "aa[]",
   (SUBR)tabarithset1, (SUBR)tabaardd },
  {"##add.a[a", sizeof(TABARITH2), 0, "a[]", "a[]a",
   (SUBR)tabarithset2, (SUBR)tabaradd },
  {"##sub.[k", sizeof(TABARITH1), 0, "k[]", "i[]k",
   (SUBR)tabarithset1, (SUBR)tabaisub },
  {"##sub.[k", sizeof(TABARITH1), 0, "k[]", "k[]k",
   (SUBR)tabarithset1, (SUBR)tabaisub },
  {"##sub.k[", sizeof(TABARITH2), 0, "k[]", "kk[]",
   (SUBR)tabarithset2, (SUBR)tabiasub },
  {"##sub.k[", sizeof(TABARITH2), 0, "k[]", "ki[]",
   (SUBR)tabarithset2, (SUBR)tabiasub },
  {"##sub.[ak", sizeof(TABARITH1), 0, "a[]", "a[]k",
   (SUBR)tabarithset1, (SUBR)tabaksub },
  {"##sub.k[a", sizeof(TABARITH2), 0, "a[]", "ka[]",
   (SUBR)tabarithset2, (SUBR)tabkasub },
  {"##sub.aa[", sizeof(TABARITH2), 0, "a[]", "a[]a",
   (SUBR)tabarithset2, (SUBR)tabaarsb },
  {"##sub.a[a", sizeof(TABARITH1), 0, "a[]", "aa[]",
   (SUBR)tabarithset1, (SUBR)tabarasb },
  {"##mul.[k", sizeof(TABARITH1), 0, "k[]", "k[]k",
   (SUBR)tabarithset1, (SUBR)tabaimult },
  {"##mul.[k", sizeof(TABARITH1), 0, "k[]", "i[]k",
   (SUBR)tabarithset1, (SUBR)tabaimult },
  {"##mul.k[", sizeof(TABARITH2), 0, "k[]", "kk[]",
   (SUBR)tabarithset2, (SUBR)tabiamult },
  {"##mul.k[", sizeof(TABARITH2), 0, "k[]", "ki[]",
   (SUBR)tabarithset2, (SUBR)tabiamult },
  {"##mul.[ak", sizeof(TABARITH1), 0, "a[]", "a[]k",
   (SUBR)tabarithset1, (SUBR)tabakmult },
  {"##mul.k[a", sizeof(TABARITH2), 0, "a[]", "ka[]",
   (SUBR)tabarithset2, (SUBR)tabkamult },
  {"##mul.a[a", sizeof(TABARITH1), 0, "a[]", "a[]a",
   (SUBR)tabarithset1, (SUBR)tabaraml },
  {"##mul.aa[", sizeof(TABARITH2), 0, "a[]", "aa[]",
   (SUBR)tabarithset2, (SUBR)tabaarml },
  {"##div.[k",  sizeof(TABARITH1), 0, "k[]", "k[]k",
   (SUBR)tabarithset1, (SUBR)tabaidiv },
  {"##div.k[",  sizeof(TABARITH2), 0, "k[]", "ki[]",
   (SUBR)tabarithset2, (SUBR)tabiadiv },
  {"##div.[k",  sizeof(TABARITH1), 0, "k[]", "i[]k",
   (SUBR)tabarithset1, (SUBR)tabaidiv },
  {"##div.k[",  sizeof(TABARITH2), 0, "k[]", "kk[]",
   (SUBR)tabarithset2, (SUBR)tabiadiv },
  {"##div.[ak", sizeof(TABARITH1), 0, "a[]", "a[]k",
   (SUBR)tabarithset1, (SUBR)tabakdiv },
  {"##div.k[a", sizeof(TABARITH2), 0, "a[]", "ka[]",
   (SUBR)tabarithset2, (SUBR)tabkadiv },
  {"##div.a[a", sizeof(TABARITH1), 0, "a[]", "a[]a",
   (SUBR)tabarithset1, (SUBR)tabaardv },
  {"##div.a[a", sizeof(TABARITH1), 0, "a[]", "a[]a",
   (SUBR)tabarithset1, (SUBR)tabaradv },
  {"##rem.[k",  sizeof(TABARITH1),0,  "k[]", "k[]k",
   (SUBR)tabarithset1, (SUBR)tabairem },
  {"##rem.[ak",  sizeof(TABARITH1),0,  "a[]", "a[]k",
   (SUBR)tabarithset1, (SUBR)tabarkrem },
  {"##rem.k[",  sizeof(TABARITH2),0,  "k[]", "kk[]",
   (SUBR)tabarithset2, (SUBR)tabiarem },
  {"##pow.[i]",  sizeof(TABARITH), 0, "i[]", "i[]i[]",
   (SUBR)tabpowi },
  {"##pow.[p",  sizeof(TABARITH1), 0, "i[]", "i[]i",
   (SUBR)tabaipowi },
  {"##pow.p[",  sizeof(TABARITH2), 0, "i[]", "ii[]",
   (SUBR)tabiapowi },
  {"##pow.[]",  sizeof(TABARITH), 0, "k[]", "k[]k[]",
   (SUBR)tabarithset,(SUBR)tabpow },
  {"##pow.[i",  sizeof(TABARITH1), 0, "k[]", "k[]i",
   (SUBR)tabarithset1, (SUBR)tabaipow },
  {"##pow.i[",  sizeof(TABARITH2), 0, "k[]", "ik[]",
   (SUBR)tabarithset2, (SUBR)tabiapow },
  {"##pow.[k",  sizeof(TABARITH1), 0, "k[]", "k[]k",
   (SUBR)tabarithset1, (SUBR)tabaipow },
  {"##pow.k[",  sizeof(TABARITH2), 0, "k[]", "kk[]",
   (SUBR)tabarithset2, (SUBR)tabiapow },
  {"##pow.ki[",  sizeof(TABARITH2), 0, "k[]", "ki[]",
   (SUBR)tabarithset2, (SUBR)tabiapow },
  {"##pow.[ak",  sizeof(TABARITH1), 0, "a[]", "a[]k",
   (SUBR)tabarithset1, (SUBR)tabarkpow },
  {"##pow.a[k[", sizeof(TABARITH), 0, "a[]", "a[]k[]",
   (SUBR)tabarithset, (SUBR)tabarkrpw },
  { "taninvx2.Ai", sizeof(TABARITH), 0, "i[]", "i[]i[]",
    (SUBR)taninv2_Ai  },
  { "taninv2.Ak", sizeof(TABARITH), 0, "k[]", "k[]k[]",
    (SUBR)tabarithset,
    (SUBR)taninv2_A  },
  { "taninv2.Aa", sizeof(TABARITH), 0, "a[]", "a[]a[]",
    (SUBR)tabarithset,
    (SUBR)taninv2_Aa  },
  { "clear", sizeof(TABCLEAR), 0, "", "a[]",
    (SUBR)tabclearset, (SUBR)tabclear  },
  { "clear", sizeof(TABCLEAR), 0, "", "k[]",
    (SUBR)tabclearset, (SUBR)tabcleark  },
  { "maxtab.k",sizeof(TABQUERY),_QQ, "kz", "k[]",
    (SUBR) tabqset, (SUBR) tabmax },
  { "maxarray.i", sizeof(TABQUERY), 0, "iI", "i[]",
    (SUBR) tabmax1, NULL  },
  { "maxarray.k", sizeof(TABQUERY), 0, "kz", "k[]",
    (SUBR) tabqset,(SUBR) tabmax },
  { "mintab.k", sizeof(TABQUERY),_QQ, "kz", "k[]",
    (SUBR) tabqset, (SUBR) tabmin },
  { "minarray.i", sizeof(TABQUERY),0, "iI", "i[]",(SUBR) tabmin1 },
  { "minarray.k", sizeof(TABQUERY),0, "kz", "k[]",(SUBR) tabqset,
    (SUBR) tabmin },
  { "sumarray.i", sizeof(TABQUERY1),0, "i", "i[]",
    (SUBR) tabsum1   },
  { "sumtab", sizeof(TABQUERY1),_QQ, "k", "k[]",
    (SUBR) tabqset1, (SUBR) tabsum },
  { "sumarray.k", sizeof(TABQUERY1),0, "k", "k[]",
    (SUBR) tabqset1, (SUBR) tabsum },
  { "sumarray.a", sizeof(TABQUERY1),0, "a", "a[]",
    (SUBR) tabqset1, (SUBR) tabsuma1 },
  { "scalet", sizeof(TABSCALE), _QQ|WI, "",  "k[]kkOJ",
    (SUBR) tabscaleset,(SUBR) tabscale },
  { "scalearray.1", sizeof(TABSCALE), WI, "",  "i[]iiOJ",
    (SUBR) tabscale1 },
  { "scalearray.k", sizeof(TABSCALE), WI, "",  "k[]kkOJ",
    (SUBR) tabscaleset,(SUBR) tabscale },
  { "fillarray", sizeof(TABCPY), 0, "k[]", "i[]", (SUBR)tabcopy1, NULL },
  { "=.I", sizeof(TABCPY), 0, "i[]", "i[]", (SUBR)tabcopy1, NULL },
  { "=.J", sizeof(TABCPY), 0, "i[]", "k[]", (SUBR)tabcopy1, NULL },
  { "=.IK", sizeof(TABCPY), 0, "k[]", "i[]", (SUBR)tabcopy1, (SUBR)tabcopy1 },
  { "=.S", sizeof(TABCPY), 0, "S[]", "S[]", (SUBR)tabcopy, (SUBR)tabcopyk },
  { "=.K", sizeof(TABCPY), 0, "k[]", "k[]", (SUBR)tabcopyk_init, (SUBR)tabcopyk },
  { "=._", sizeof(TABCPY), 0, ".[]", ".[]", (SUBR)tabcopyk_init, (SUBR)tabcopy },
  { "=.L", sizeof(TABCPY), 0, ".[]", ".[]", (SUBR)tabcopy2, (SUBR)tabcopy2 },
  { "tabgen", sizeof(TABGEN), _QQ, "k[]", "iip", (SUBR) tabgen, NULL    },
  { "tabmap_i", sizeof(TABMAP), _QQ, "k[]", "k[]S", (SUBR) tabmap_set   },
  { "tabmapi", sizeof(TABMAP), _QQ, "k[]", "k[]S", (SUBR) tabmap_set   }, /* Alias */
  { "tabmap", sizeof(TABMAP), _QQ, "k[]", "k[]S", (SUBR) tabmap_set,
    (SUBR) tabmap_perf},
  { "tabmap", sizeof(TABMAP), _QQ, "k[]", "k[]S", (SUBR) tabmap_set,
    (SUBR) tabmap_perf},
  { "genarray.i", sizeof(TABGEN),0, "i[]", "iip", (SUBR) tabgen, NULL   },
  { "genarray.ki", sizeof(TABGEN),0, "k[]", "iip", (SUBR)tabgen, NULL, NULL},
  { "genarray.k", sizeof(TABGEN),0, "k[]", "kkp", NULL, (SUBR)tabgen    },
  { "genarray_i", sizeof(TABGEN),0, "k[]", "iip", (SUBR) tabgen, NULL, NULL, NULL, 2},
  { "genarrayi", sizeof(TABGEN),0, "k[]", "iip", (SUBR) tabgen, NULL, NULL}, /* Alias */
  { "maparray.i", sizeof(TABMAP), 0, "i[]", "i[]S", (SUBR) tabmap_set },
  { "maparray_i", sizeof(TABMAP),0, "k[]", "k[]S", (SUBR) tabmap_set, NULL, NULL, NULL, 2},  
  { "maparrayi", sizeof(TABMAP),0, "k[]", "k[]S", (SUBR) tabmap_set    },  /* Alias */
  { "maparray.k", sizeof(TABMAP), 0, "k[]", "k[]S", (SUBR) tabmap_set,
    (SUBR) tabmap_perf          },
  { "tabslice", sizeof(TABSLICE), _QQ, "k[]", "k[]iip",
    NULL, (SUBR) tabslice, NULL },
  { "slicearray.i", sizeof(TABSLICE), 0, "i[]", "i[]iip",
    (SUBR) tabslice, NULL, NULL },
  { "slicearray.k", sizeof(TABSLICE), 0, "k[]", "k[]iip",
    (SUBR) tabslice, (SUBR) tabslice, NULL },
  { "slicearray.a", sizeof(TABSLICE), 0, "a[]", "a[]iip",
    (SUBR) tabslice, (SUBR) tabslice, NULL },
  { "slicearray.S", sizeof(TABSLICE), 0, "S[]", "S[]iip",
    (SUBR) tabslice, (SUBR) tabslice, NULL },
  { "slicearray_i.i", sizeof(TABSLICE), 0, "i[]", "i[]iip",
    (SUBR) tabslice, NULL },
  { "slicearrayi.i", sizeof(TABSLICE), 0, "i[]", "i[]iip",
    (SUBR) tabslice, NULL }, /* Alias */
  { "slicearray_i.k", sizeof(TABSLICE), 0, "k[]", "k[]iip",
    (SUBR) tabslice, NULL },
  { "slicearrayi.k", sizeof(TABSLICE), 0, "k[]", "k[]iip",
    (SUBR) tabslice, NULL }, /* Alias */
  { "slicearray_i.S", sizeof(TABSLICE), 0, "S[]", "S[]iip",
    (SUBR) tabslice, NULL },
  { "slicearrayi.S", sizeof(TABSLICE), 0, "S[]", "S[]iip",
    (SUBR) tabslice, NULL }, /* Alias */
  { "trim.i", sizeof(TRIM), WI, "", "i[]i", (SUBR)trim_i, NULL },
  { "trim.k", sizeof(TRIM), WI, "", ".[]k", NULL, (SUBR)trim },
  { "trim_i", sizeof(TRIM), WI, "", ".[]i", (SUBR)trim_i, NULL, NULL, NULL, 2 },
  { "trimi", sizeof(TRIM), WI, "", ".[]i", (SUBR)trim_i, NULL },  /* alias */
  { "copy2ftab", sizeof(TABCOPY), TW|_QQ, "", "k[]k", NULL, (SUBR) tab2ftab },
  { "copy2ttab", sizeof(TABCOPY), TR|_QQ, "", "k[]k", NULL, (SUBR) ftab2tab },
  { "copya2ftab.ii", sizeof(TABCOPY2), TW, "", "i[]io",
    (SUBR) tab2ftab_offset_i },
  { "copya2ftab.k", sizeof(TABCOPY), TW, "", "k[]k",
    (SUBR) tab2ftabi, (SUBR) tab2ftab },
  { "copyf2array.k", sizeof(TABCOPY), TR, "", "k[]k",
    (SUBR) ftab2tabi, (SUBR) ftab2tab },
  { "copyf2array.kk", sizeof(TABCOPY), TR, "", "k[]kk",
    (SUBR) ftab2tabi, (SUBR) ftab2tab },
  { "copya2ftab.kk", sizeof(TABCOPY2), TW, "", "k[]kO", NULL ,
    (SUBR) tab2ftab_offset },
  { "copyf2array.i", sizeof(TABCOPY), TR, "", "i[]i", (SUBR) ftab2tabi },
  { "lentab.i", sizeof(TABQUERY1), _QQ, "i", "k[]p", (SUBR) tablength },
  { "lentab.k", sizeof(TABQUERY1), _QQ, "k", "k[]p", NULL, (SUBR) tablength },
  { "lenarray.ix", sizeof(TABQUERY1), 0, "i", ".[]p", (SUBR) tablength },
  { "lenarray.kx", sizeof(TABQUERY1), 0, "k", ".[]p", (SUBR)tablength, (SUBR)tablength },
  /* complex arrays */
  {"##add.Complex[]Complex[]", sizeof(COPS1), 0, ":Complex;[]",
   ":Complex;[]:Complex;[]",
   (SUBR) cops_init,(SUBR)complexa_plus_complexa},
  {"##mul.Complex[]Complex[]", sizeof(COPS1), 0, ":Complex;[]",
   ":Complex;[]:Complex;[]",
   (SUBR) cops_init,(SUBR)complexa_x_complexa},
  {"##sub.Complex[]Complex[]", sizeof(COPS1), 0, ":Complex;[]",
   ":Complex;[]:Complex;[]",
   (SUBR) cops_init,(SUBR)complexa_sub_complexa},
  {"##div.Complex[]Complex[]", sizeof(COPS1), 0, ":Complex;[]",
   ":Complex;[]:Complex;[]",
   (SUBR) cops_init,(SUBR)complexa_div_complexa},
  {"##add.Complex[]k", sizeof(COPS1), 0, ":Complex;[]", ":Complex;[]k",
   (SUBR) cops_init,(SUBR)complex_plus_scalar},
  {"##add.kComplex[]", sizeof(COPS1), 0, ":Complex;[]", "k:Complex;[]",
   (SUBR) cops_init,(SUBR)complex_plus_scalar},
  {"##mul.Complex[]k", sizeof(COPS1), 0, ":Complex;[]", ":Complex;[]k",
   (SUBR) cops_init,(SUBR)complex_x_scalar},
  {"##mul.kComplex[]", sizeof(COPS1), 0, ":Complex;[]", "k:Complex;[]",
   (SUBR) cops_init,(SUBR)complex_x_scalar},
  {"##div.Complex[]k", sizeof(COPS1), 0, ":Complex;[]", ":Complex;[]k",
   (SUBR) cops_init,(SUBR)complex_div_scalar},
  {"##sub.Complex[]k", sizeof(COPS1), 0, ":Complex;[]", ":Complex;[]k",
   (SUBR) cops_init,(SUBR)complex_minus_scalar},
  {"##sub.kComplex[]", sizeof(COPS1), 0, ":Complex;[]", "k:Complex;[]",
   (SUBR) cops_init,(SUBR)scalar_minus_complex},
  {"##add.Complex[]Complex", sizeof(COPS1), 0, ":Complex;[]",
   ":Complex;[]:Complex;",
   (SUBR) cops_init,(SUBR)complex_plus_complex},
  {"##add.ComplexComplex[]", sizeof(COPS1), 0, ":Complex;[]",
   ":Complex;:Complex;[]",
   (SUBR) cops_init,(SUBR)complex_plus_complex},
  {"##mul.Complex[]Complex", sizeof(COPS1), 0, ":Complex;[]",
   ":Complex;[]:Complex;",
   (SUBR) cops_init,(SUBR)complex_x_complex},
  {"##mul.ComplexComplex[]", sizeof(COPS1), 0, ":Complex;[]",
   ":Complex;:Complex;[]",
   (SUBR) cops_init,(SUBR)complex_x_complex},
  {"##div.Complex[]Complex", sizeof(COPS1), 0, ":Complex;[]",
   ":Complex;[]:Complex;",
   (SUBR) cops_init,(SUBR)complex_div_complex},
  {"##sub.Complex[]Complex", sizeof(COPS1), 0, ":Complex;[]",
   ":Complex;[]:Complex;",
   (SUBR) cops_init,(SUBR)complexa_minus_complex},
  {"##sub.ComplexComplex[]", sizeof(COPS1), 0, ":Complex;[]",
   ":Complex;:Complex;[]",
   (SUBR) cops_init,(SUBR)complex_minus_complexa},
  { "log", S(CXOP), 0, ":Complex;", ":Complex;", (SUBR) complex_log,  (SUBR) complex_log },
  { "log", S(COPS1), 0, ":Complex;[]", ":Complex;[]", (SUBR) cops_init_r,
    (SUBR) complex_array_log },
  { "exp", S(CXOP), 0, ":Complex;", ":Complex;", (SUBR) complex_exp,  (SUBR) complex_exp },
  { "exp", S(COPS1), 0, ":Complex;[]", ":Complex;[]", (SUBR) cops_init_r,
    (SUBR) complex_array_exp },
  { "conj", S(CXOP), 0, ":Complex;", ":Complex;", (SUBR) complex_conj,
    (SUBR) complex_conj },
  { "conj", S(COPS1), 0, ":Complex;[]", ":Complex;[]",(SUBR) cops_init_r,
     (SUBR) complex_array_conj },
  { "real", S(CXOP2R), 0, "k", ":Complex;", NULL, (SUBR) complex_real},
  { "real", S(CXOP2R), 0, "i", ":Complex;", (SUBR) complex_real },
  { "real", S(COPS1), 0, "a", ":Complex;[]",(SUBR) cops_init_r,
     (SUBR) complex_array_real },
  { "real", S(COPS1), 0, "k[]", ":Complex;[]",(SUBR) cops_init_r,
     (SUBR) complex_array_real },
  { "imag", S(CXOP2R), 0, "k", ":Complex;", NULL, (SUBR) complex_imag},
  { "imag", S(CXOP2R), 0, "i", ":Complex;", (SUBR) complex_imag },
  { "imag", S(COPS1), 0, "a", ":Complex;[]",(SUBR) cops_init_r,
     (SUBR) complex_array_imag },
  { "imag", S(COPS1), 0, "k[]", ":Complex;[]",(SUBR) cops_init_r,
     (SUBR) complex_array_imag },
  { "abs", S(CXOP2R), 0, "k", ":Complex;", NULL, (SUBR) complex_abs},
  { "abs", S(CXOP2R), 0, "i", ":Complex;", (SUBR) complex_abs },
  { "abs", S(COPS1), 0, "a", ":Complex;[]",(SUBR) cops_init_r,
     (SUBR) complex_array_abs },
  { "abs", S(COPS1), 0, "k[]", ":Complex;[]",(SUBR) cops_init_r,
     (SUBR) complex_array_abs },
  { "arg", S(CXOP2R), 0, "k", ":Complex;", NULL, (SUBR) complex_arg},
  { "arg", S(CXOP2R), 0, "i", ":Complex;", (SUBR) complex_arg },
  { "arg", S(COPS1), 0, "a", ":Complex;[]",(SUBR) cops_init_r,
     (SUBR) complex_array_arg },
  { "arg", S(COPS1), 0, "k[]", ":Complex;[]",(SUBR) cops_init_r,
     (SUBR) complex_array_arg },
  { "polar", S(CXOP), 0, ":Complex;", ":Complex;", (SUBR) complex_to_polar,
    (SUBR) complex_to_polar},
  { "polar", S(COPS1), 0, ":Complex;[]", ":Complex;[]",(SUBR) cops_init_r,
     (SUBR) complex_array_polar },
  { "complex", S(R2CXOP), 0, ":Complex;", "iio", (SUBR) complex_assign},
  { "complex", S(R2CXOP), 0, ":Complex;", "kko", (SUBR) complex_assign,
    (SUBR) complex_assign},
  { "complex", S(CXOP), 0, ":Complex;", ":Complex;", (SUBR) polar_to_complex,
    (SUBR) polar_to_complex},
  { "complex", S(COPS1), 0, ":Complex;[]", ":Complex;[]", (SUBR) cops_init_r,
     (SUBR) complex_array_complex },
  { "complex", S(COPS1), 0, ":Complex;[]", "k[]k[]", (SUBR) cops_init_r,
     (SUBR) complex_array_assign },
  // Builtin struct opcodes
  { "##array_get_struct", sizeof(STRUCT_ARRAY_GET), 0, ".", ".[]m", (SUBR)struct_array_get, NULL, NULL },
  { "##member_get", sizeof(STRUCT_GET), 0, ".", ".i", (SUBR)struct_member_get_init_and_perf, (SUBR)struct_member_get, NULL },
  { "##member_get.i", sizeof(STRUCT_GET), 0, "i", ".i", (SUBR)struct_member_get_init_and_perf, (SUBR)struct_member_get, NULL },
  { "##member_get.k", sizeof(STRUCT_GET), 0, "k", ".i", (SUBR)struct_member_get_init, (SUBR)struct_member_get, NULL },
  { "##member_get.S", sizeof(STRUCT_GET), 0, "S", ".i", (SUBR)struct_member_get_init, (SUBR)struct_member_get, NULL },
  { "##member_get.a", sizeof(STRUCT_GET), 0, "a", ".i", (SUBR)struct_member_get_init, (SUBR)struct_member_get, NULL },
  { "##member_get.b", sizeof(STRUCT_GET), 0, "b", ".i", (SUBR)struct_member_get_init, (SUBR)struct_member_get, NULL },
  { "##member_set", sizeof(STRUCT_SET), 0, "", ".i.", (SUBR)struct_member_set_init_and_perf, (SUBR)struct_member_set, NULL },
  { "##member_array_assign", sizeof(STRUCT_MEMBER_ARRAY_ASSIGN), 0, "", ".i.[]", (SUBR)struct_member_array_assign, NULL, NULL },
  // Generic struct initialization opcodes - these will be registered dynamically for each struct type
  // For now, add some common patterns to test
  // { "init", sizeof(STRUCT_INIT), 0, "", "m", (SUBR)struct_init, NULL, NULL },
  // { "init.i", sizeof(STRUCT_INIT), 0, "", "m", (SUBR)struct_init, NULL, NULL },
  // Arthimetic opcodes
  { "divz.ii", S(DIVZ),0,         "i",    "iii",  divzkk, NULL,   NULL    },
  { "divz.kk", S(DIVZ),0,          "k",    "kkk",  NULL,   divzkk, NULL    },
  { "divz.ak", S(DIVZ),0,          "a",    "akk",  NULL,   divzak  },
  { "divz.ka", S(DIVZ),0,          "a",    "kak",  NULL,   divzka  },
  { "divz.aa", S(DIVZ),0,          "a",    "aak",  NULL,   divzaa  },
  { "int.i",  S(EVAL),0,          "i",    "i",    int1                    },
  { "frac.i", S(EVAL),0,          "i",    "i",    frac1                   },
  { "round.i",S(EVAL),0,          "i",    "i",    int1_round              },
  { "floor.i",S(EVAL),0,          "i",    "i",    int1_floor              },
  { "ceil.i", S(EVAL),0,          "i",    "i",    int1_ceil               },
  { "rndseed", S(EVAL),0,          "",    "i",    rnd1seed                },
  { "rnd.i",  S(EVAL),0,          "i",    "i",    rnd1                    },
  { "birnd.i",S(EVAL),0,          "i",    "i",    birnd1                  },
  { "abs.i",  S(EVAL),0,          "i",    "i",    abs1                    },
  { "exp.i",  S(EVAL),0,          "i",    "i",    exp01                   },
  { "log.i",  S(EVAL),0,          "i",    "i",    log01                   },
  { "sqrt.i", S(EVAL),0,          "i",    "i",    sqrt1                   },
  { "sin.i",  S(EVAL),0,          "i",    "i",    sin1                    },
  { "cos.i",  S(EVAL),0,          "i",    "i",    cos1                    },
  { "tan.i",  S(EVAL),0,          "i",    "i",    tan1                    },
  { "qinf.i", S(EVAL),0,          "i",    "i",    is_inf                  },
  { "qnan.i", S(EVAL),0,          "i",    "i",    is_NaN                  },
  { "sininv.i", S(EVAL),0,        "i",    "i",    asin1                   },
  { "cosinv.i", S(EVAL),0,        "i",    "i",    acos1                   },
  { "taninv.i", S(EVAL),0,        "i",    "i",    atan1                   },
  { "taninv2.i",S(AOP),0,         "i",    "ii",   atan21                  },
  { "log10.i",S(EVAL),0,          "i",    "i",    log101                  },
  { "log2.i", S(EVAL),0,          "i",    "i",    log21                   },
  { "sinh.i", S(EVAL),0,          "i",    "i",    sinh1                   },
  { "cosh.i", S(EVAL),0,          "i",    "i",    cosh1                   },
  { "tanh.i", S(EVAL),0,          "i",    "i",    tanh1                   },
  { "int.k",  S(EVAL),0,           "k",    "k",    NULL,   int1            },
  { "frac.k", S(EVAL),0,           "k",    "k",    NULL,   frac1           },
  { "round.k",S(EVAL),0,           "k",    "k",    NULL,   int1_round      },
  { "floor.k",S(EVAL),0,           "k",    "k",    NULL,   int1_floor      },
  { "ceil.k", S(EVAL),0,           "k",    "k",    NULL,   int1_ceil       },
  { "rnd.k",  S(EVAL),0,           "k",    "k",    NULL,   rnd1            },
  { "birnd.k",S(EVAL),0,           "k",    "k",    NULL,   birnd1          },
  { "abs.k",  S(EVAL),0,           "k",    "k",    NULL,   abs1            },
  { "exp.k",  S(EVAL),0,           "k",    "k",    NULL,   exp01           },
  { "log.k",  S(EVAL),0,           "k",    "k",    NULL,   log01           },
  { "sqrt.k", S(EVAL),0,           "k",    "k",    NULL,   sqrt1           },
  { "sin.k",  S(EVAL),0,           "k",    "k",    NULL,   sin1            },
  { "cos.k",  S(EVAL),0,           "k",    "k",    NULL,   cos1            },
  { "tan.k",  S(EVAL),0,           "k",    "k",    NULL,   tan1            },
  { "qinf.k", S(EVAL),0,           "k",    "k",    NULL,   is_inf          },
  { "qnan.k", S(EVAL),0,           "k",    "k",    NULL,   is_NaN          },
  { "sininv.k", S(EVAL),0,         "k",    "k",    NULL,   asin1           },
  { "cosinv.k", S(EVAL),0,         "k",    "k",    NULL,   acos1           },
  { "taninv.k", S(EVAL),0,         "k",    "k",    NULL,   atan1           },
  { "taninv2.k",S(AOP),0,          "k",    "kk",   NULL,   atan21          },
  { "sinh.k", S(EVAL),0,           "k",    "k",    NULL,   sinh1           },
  { "cosh.k", S(EVAL),0,           "k",    "k",    NULL,   cosh1           },
  { "tanh.k", S(EVAL),0,           "k",    "k",    NULL,   tanh1           },
  { "log10.k",S(EVAL),0,           "k",    "k",    NULL,   log101          },
  { "log2.k", S(EVAL),0,           "k",    "k",    NULL,   log21           },
  { "int.a",  S(EVAL),0,           "a",    "a",    NULL, int1a       },
  { "frac.a", S(EVAL),0,           "a",    "a",    NULL, frac1a      },
  { "round.a",S(EVAL),0,           "a",    "a",    NULL, int1a_round },
  { "floor.a",S(EVAL),0,           "a",    "a",    NULL, int1a_floor },
  { "ceil.a", S(EVAL),0,           "a",    "a",    NULL, int1a_ceil  },
  { "abs.a",  S(EVAL),0,           "a",    "a",    NULL,   absa    },
  { "exp.a",  S(EVAL),0,           "a",    "a",    NULL,   expa    },
  { "log.a",  S(EVAL),0,           "a",    "a",    NULL,   loga    },
  { "sqrt.a", S(EVAL),0,           "a",    "a",    NULL,   sqrta   },
  { "sin.a",  S(EVAL),0,           "a",    "a",    NULL,   sina    },
  { "cos.a",  S(EVAL),0,           "a",    "a",    NULL,   cosa    },
  { "tan.a",  S(EVAL),0,           "a",    "a",    NULL,   tana    },
  { "qinf.a", S(EVAL),0,           "a",    "a",    NULL,   is_infa },
  { "qnan.a", S(EVAL),0,           "a",    "a",    NULL,   is_NaNa },
  { "sininv.a", S(EVAL),0,         "a",    "a",    NULL,   asina   },
  { "cosinv.a", S(EVAL),0,         "a",    "a",    NULL,   acosa   },
  { "taninv.a", S(EVAL),0,         "a",    "a",    NULL,   atana   },
  { "taninv2.a",S(AOP),0,          "a",    "aa",   NULL,   atan2aa },
  { "sinh.a", S(EVAL),0,           "a",    "a",    NULL,   sinha   },
  { "cosh.a", S(EVAL),0,           "a",    "a",    NULL,   cosha   },
  { "tanh.a", S(EVAL),0,           "a",    "a",    NULL,   tanha   },
  { "log10.a",S(EVAL),0,           "a",    "a",    NULL,   log10a  },
  { "log2.a", S(EVAL),0,           "a",    "a",    NULL,   log2a   },
  { "ampdb.a",S(EVAL),0,           "a",    "a",    NULL,   aampdb  },
  { "ampdb.i",S(EVAL),0,          "i",    "i",    ampdb                   },
  { "ampdb.k",S(EVAL),0,           "k",    "k",    NULL,   ampdb           },
  { "ampdbfs.a",S(EVAL),0,         "a",    "a",    NULL,   aampdbfs },
  { "ampdbfs.i",S(EVAL),0,        "i",    "i",    ampdbfs                 },
  { "ampdbfs.k",S(EVAL),0,         "k",    "k",    NULL,   ampdbfs         },
  { "dbamp.i",S(EVAL),0,          "i",    "i",    dbamp                   },
  { "dbamp.k",S(EVAL),0,           "k",    "k",    NULL,   dbamp           },
  { "dbfsamp.i",S(EVAL),0,        "i",    "i",    dbfsamp                 },
  { "dbfsamp.k",S(EVAL),0,         "k",    "k",    NULL,   dbfsamp         },
  { "rtclock.i",S(EVAL),0,        "i",    "",     rtclock                 },
  { "rtclock.k",S(EVAL),0,         "k",    "",     NULL,   rtclock         },
  { "ftlen.i",S(EVAL),0,          "i",    "i",    ftlen                   },
  { "ftsr.i",S(EVAL),0,           "i",    "i",    ftsr                    },
  { "ftlptim.i",S(EVAL),0,        "i",    "i",    ftlptim                 },
  { "ftchnls.i",S(EVAL),0,        "i",    "i",    ftchnls                 },
  { "ftcps.i",S(EVAL),0,          "i",    "i",    ftcps                   },
  { "=",     S(ASSIGN),0,         "i",    "i",    assign,  NULL           },
  { "=",     S(ASSIGN),0,         "k",    "k",    NULL,    assign         },
  { "=",     S(ASSIGN),0,         "k",    "i",    NULL,    assign         },
  { "i.i",   S(ASSIGN),0,         "i",    "i",    assign                  },
  { "i.k",   S(ASSIGN),0,         "i",    "k",    assign                  },
  /* this one is stupid, but it's there *only* for backwards
     compatibility */
  { "i.kk",   S(STOREI),0,         "k",    "k",   (SUBR) storei, (SUBR) retrievek },
  { "k.i",   S(ASSIGN),0,         "k",    "i",    assign                  },
  { "k.a",   S(DOWNSAMP),0,       "k",    "ao",   (SUBR)downset,(SUBR)downsamp },
  { "=",     S(ASSIGN),0,         "a",    "i",    NULL,    ainit         },
  { "cpsoct.i",S(EVAL),0,         "i",    "i",    cpsoct                  },
  { "octpch.i",S(EVAL),0,         "i",    "i",    octpch                  },
  { "cpspch.i",S(EVAL),0,         "i",    "i",    cpspch                  },
  { "pchoct.i",S(EVAL),0,         "i",    "i",    pchoct                  },
  { "octcps.i",S(EVAL),0,         "i",    "i",    octcps                  },
  { "cpsoct.k",S(EVAL),0,          "k",    "k",    NULL,   cpsoct          },
  { "octpch.k",S(EVAL),0,          "k",    "k",    NULL,   octpch          },
  { "cpspch.k",S(EVAL),0,          "k",    "k",    NULL,   cpspch          },
  { "pchoct.k",S(EVAL),0,          "k",    "k",    NULL,   pchoct          },
  { "octcps.k",S(EVAL),0,          "k",    "k",    NULL,   octcps          },
  { "cpsoct.a",S(EVAL),0,          "a",    "a",    NULL,   acpsoct },
  { "cpsmidinn.i",S(EVAL),0,      "i",    "i",    cpsmidinn               },
  { "octmidinn.i",S(EVAL),0,      "i",    "i",    octmidinn               },
  { "pchmidinn.i",S(EVAL),0,      "i",    "i",    pchmidinn               },
  { "cpsmidinn.k",S(EVAL),0,       "k",    "k",    NULL,   cpsmidinn       },
  { "octmidinn.k",S(EVAL),0,       "k",    "k",    NULL,   octmidinn       },
  { "pchmidinn.k",S(EVAL),0,       "k",    "k",    NULL,   pchmidinn       },
  { "notnum", S(MIDIKMB),0,       "i",    "",     notnum                  },
  { "veloc",  S(MIDIMAP),0,       "i",    "oh",   veloc                   },
  { "pchmidi",S(MIDIKMB),0,       "i",    "",     pchmidi                 },
  { "octmidi",S(MIDIKMB),0,       "i",    "",     octmidi                 },
  { "cpsmidi",S(MIDIKMB),0,       "i",    "",     cpsmidi                 },
  { "pchmidib.i",S(MIDIKMB),0,    "i",    "o",    pchmidib_i              },
  { "octmidib.i",S(MIDIKMB),0,    "i",    "o",    octmidib_i              },
  { "cpsmidib.i",S(MIDIKMB),0,    "i",    "o",    icpsmidib_i             },
  { "pchmidib.k",S(MIDIKMB),0,    "k",    "o",    midibset, pchmidib      },
  { "octmidib.k",S(MIDIKMB),0,    "k",    "o",    midibset, octmidib      },
  { "cpsmidib.k",S(MIDIKMB),0,    "k",    "o",    midibset, icpsmidib     },
  { "ampmidi",S(MIDIAMP),0,       "i",    "io",   ampmidi                 },
  { "aftouch",S(MIDIKMAP),0,      "k",    "oh",   aftset, aftouch         },
  { "pchbend.i",S(MIDIMAP),0,     "i",    "jp",   ipchbend                },
  { "pchbend.k",S(MIDIKMAP),0,    "k",    "jp",   kbndset,kpchbend        },
  { "midictrl.i",S(MIDICTL),0,    "i",    "ioh",  imidictl                },
  { "midictrl.k",S(MIDICTL),0,    "k",    "ioh",  mctlset, midictl        },
  { "polyaft.i",S(MIDICTL),0,     "i",    "ioh",  imidiaft                },
  { "polyaft.k",S(MIDICTL),0,     "k",    "ioh",  maftset, midiaft        },
  { "chanctrl.i",S(CHANCTL),0,    "i",    "iioh", ichanctl                },
  { "chanctrl.k",S(CHANCTL),0,    "k",    "iioh", chctlset,chanctl        },
  { "line",   S(LINE),0,          "k",    "iii",  linset, kline,  NULL  },
  { "line.a",   S(LINE),0,        "a",    "iii",  linset,  aline   },
  { "expon",  S(EXPON),0,         "k",    "iii",  expset, kexpon, NULL   },
  { "expon.a",  S(EXPON),0,         "a",    "iii",  expset, expon   },
  { "cosseg", S(COSSEG),0,        "k",    "iim",  csgset, kosseg, NULL  },
  { "cosseg.a", S(COSSEG),0,        "a",    "iim",  csgset, cosseg  },
  { "cossegb", S(COSSEG),0,       "k",    "iim",  csgset_bkpt, kosseg, NULL  },
  { "cossegb.a", S(COSSEG),0,       "a",    "iim",  csgset_bkpt, cosseg  },
  { "cossegr", S(COSSEG),0,       "k",    "iim",  csgrset, kcssegr, NULL  },
  { "cossegr.a", S(COSSEG),0,       "a",    "iim",  csgrset, cossegr  },
  { "linseg", S(LINSEG),0,        "k",    "iim",  lsgset, klnseg, NULL },
  { "linseg.a", S(LINSEG),0,        "a",    "iim",  lsgset, linseg  },
  { "linsegb", S(LINSEG),0,       "k",    "iim", lsgset_bkpt, klnseg, NULL  },
  { "linsegb.a", S(LINSEG),0,       "a",    "iim", lsgset_bkpt, linseg  },
  { "linsegr",S(LINSEG),0,        "k",    "iim",  lsgrset,klnsegr,NULL },
  { "linsegr.a",S(LINSEG),0,        "a",    "iim",  lsgrset,linsegr },
  { "expseg", S(EXXPSEG),0,       "k",    "iim",  xsgset, kxpseg, NULL  },
  { "expseg.a", S(EXXPSEG),0,       "a",    "iim",  xsgset, expseg  },
  { "expsegb", S(EXXPSEG),0,       "k",    "iim",  xsgset_bkpt, kxpseg, NULL },
  { "expsegb.a", S(EXXPSEG),0,      "a",    "iim",  xsgset_bkpt, expseg },
  { "expsega",S(EXPSEG2),0,       "a",    "iim",  xsgset2, expseg2  },
  { "expsegba",S(EXPSEG2),0,       "a",    "iim",  xsgset2b, expseg2 },
  { "expsegr",S(EXPSEG),0,        "k",    "iim",  xsgrset,kxpsegr,NULL },
  { "expsegr.a",S(EXPSEG),0,        "a",    "iim",  xsgrset,expsegr },
  { "linen",  S(LINEN),0,         "k",    "kiii", lnnset, klinen, NULL   },
  { "linen.a",  S(LINEN),0,         "a",    "aiii", alnnset, linen   },
  { "linen.x",  S(LINEN),0,         "a",    "kiii", alnnset, linen   },
  { "linenr", S(LINENR),0,        "k",    "kiii", lnrset, klinenr,NULL },
  { "linenr.a", S(LINENR),0,        "a",    "aiii", alnrset,linenr  },
  { "linenr.x", S(LINENR),0,        "a",    "kiii", alnrset,linenr  },
  { "envlpx", S(ENVLPX), TR,      "k","kiiiiiio", evxset, knvlpx, NULL },
  { "envlpxr", S(ENVLPR),TR,      "k","kiiiiioo", evrset, knvlpxr, NULL },
  { "envlpx.a", S(ENVLPX), TR,      "a","aiiiiiio", aevxset,envlpx  },
  { "envlpxr.a", S(ENVLPR),TR,      "a","aiiiiioo", aevrset,envlpxr },
  { "envlpx.x", S(ENVLPX), TR,      "a","kiiiiiio", aevxset,envlpx  },
  { "envlpxr.x", S(ENVLPR),TR,      "a","kiiiiioo", aevrset,envlpxr },
  { "phasor", S(PHSOR),0,          "a",   "xo",   phsset, phsor   },
  { "phasor.k", S(PHSOR),0,        "k",   "ko",   phsset, kphsor, NULL  },
  { "ephasor", S(EPHSOR), 0,       "aa",  "xko",  ephsset, ephsor },
  { "signum.i", S(ASSIGN), 0,      "i",   "i", signum, NULL, NULL       },
  { "signum.k", S(ASSIGN), 0,      "k",   "k", signum, signum, NULL     },
  { "signum.a", S(ASSIGN), 0,       "a",   "a", NULL, asignum      },
  { "table.i",  S(TABL),TR,       "i",    "iiooo",(SUBR)tabler_init       },
  { "table.k",  S(TABL),TR,       "k",    "xiooo",(SUBR)tabl_setup,
    (SUBR)tabler_kontrol        },
  { "table.a",  S(TABL),TR,       "a",    "xiooo",(SUBR)tabl_setup,
    (SUBR)tabler_audio                                                      },
  { "tablei.i", S(TABL),TR,       "i",    "iiooo",(SUBR)tableir_init      },
  { "tablei.k", S(TABL),TR,       "k",    "xiooo",(SUBR)tabl_setup,
    (SUBR)tableir_kontrol                                                   },
  { "tablei.a", S(TABL),TR,       "a",    "xiooo",(SUBR)tabl_setup,
    (SUBR)tableir_audio                                                     },
  { "table3.i", S(TABL),TR,       "i",    "iiooo",(SUBR)table3r_init      },
  { "table3.k", S(TABL),TR,       "k",    "xiooo",(SUBR)tabl_setup,
    (SUBR)table3r_kontrol                                                   },
  { "table3.a", S(TABL),TR,       "a",    "xiooo",(SUBR)tabl_setup,
    (SUBR)table3r_audio                                                     },
  { "ptable.i",  S(TABLE),TR|_QQ, "i",    "iiooo",(SUBR)tabler_init       },
  { "ptable.k",  S(TABLE),TR|_QQ,      "k",    "xiooo",(SUBR)tabl_setup,
    (SUBR)tabler_kontrol                                                    },
  { "ptable.a",  S(TABLE),TR|_QQ,      "a",    "xiooo",(SUBR)tabl_setup,
    (SUBR)tabler_audio                                                      },
  { "ptablei.i", S(TABLE),TR|_QQ, "i",    "iiooo",(SUBR)tableir_init      },
  { "ptablei.k", S(TABLE),TR|_QQ,      "k",    "xiooo",(SUBR)tabl_setup,
    (SUBR)tableir_kontrol                                                   },
  { "ptablei.a", S(TABLE),TR|_QQ,      "a",    "xiooo",(SUBR)tabl_setup,
    (SUBR)tableir_audio                                                     },
  { "ptable3.i", S(TABLE),TR|_QQ, "i",    "iiooo",(SUBR)table3r_init      },
  { "ptable3.k", S(TABLE),TR|_QQ,      "k",    "xiooo",(SUBR)tabl_setup,
    (SUBR)table3r_kontrol                                                   },
  { "ptable3.a", S(TABLE),TR|_QQ,      "a",    "xiooo",(SUBR)tabl_setup,
    (SUBR)table3r_audio         },
  { "oscil1", S(OSCIL1), TR,      "k",    "ikij", ko1set, kosc1          },
  { "oscil1i",S(OSCIL1), TR,      "k",    "ikij", ko1set, kosc1i         },
  { "osciln", S(OSCILN), TR,      "a",    "kiii", oscnset,   osciln },
  { "oscil.a",S(OSC),TR,           "a",    "kkjo", oscset,   osckk  },
  { "oscil.kkk",S(OSC),TR,         "k",    "kkjo", oscset, koscil  },
  { "oscil.kka",S(OSC),TR,         "a",    "kkjo", oscset, osckk  },
  { "oscil.ka",S(OSC),TR,          "a",    "kajo", oscset,   oscka  },
  { "oscil.ak",S(OSC),TR,          "a",    "akjo", oscset,   oscak  },
  { "oscil.aa",S(OSC),TR,          "a",    "aajo", oscset,   oscaa  },
  { "oscil.kkA",S(OSC),0,         "k",    "kki[]o", oscsetA, koscil       },
  { "oscil.kkA",S(OSC),0,         "a",    "kki[]o", oscsetA, osckk },
  { "oscil.kaA",S(OSC),0,         "a",    "kai[]o", oscsetA, oscka },
  { "oscil.akA",S(OSC),0,         "a",    "aki[]o", oscsetA, oscak },
  { "oscil.aaA",S(OSC),0,         "a",    "aai[]o", oscsetA,oscaa  },
  { "oscili.a",S(OSC),TR,         "a",    "kkjo", oscset, osckki  },
  { "oscili.kk",S(OSC),TR,         "k",   "kkjo", oscset, koscli, NULL  },
  { "oscili.ka",S(OSC),TR,         "a",   "kajo", oscset,   osckai  },
  { "oscili.ak",S(OSC),TR,         "a",   "akjo", oscset,   oscaki  },
  { "oscili.aa",S(OSC),TR,         "a",   "aajo", oscset,   oscaai  },
  { "oscili.aA",S(OSC),0,         "a",   "kki[]o", oscsetA, osckki  },
  { "oscili.kkA",S(OSC),0,         "k",  "kki[]o", oscsetA, koscli, NULL  },
  { "oscili.kaA",S(OSC),0,         "a",  "kai[]o", oscsetA,   osckai  },
  { "oscili.akA",S(OSC),0,         "a",  "aki[]o", oscsetA,   oscaki  },
  { "oscili.aaA",S(OSC),0,         "a",  "aai[]o", oscsetA,   oscaai  },
  { "oscil3.a",S(OSC),TR,         "a",    "kkjo", oscset, osckk3  },
  { "oscil3.kk",S(OSC),TR,         "k",   "kkjo", oscset, koscl3, NULL  },
  { "oscil3.ka",S(OSC),TR,         "a",   "kajo", oscset,   oscka3  },
  { "oscil3.ak",S(OSC),TR,         "a",   "akjo", oscset,   oscak3  },
  { "oscil3.aa",S(OSC),TR,         "a",   "aajo", oscset,   oscaa3  },
  { "oscil3.aA",S(OSC),0,         "a",   "kki[]o", oscsetA, osckk3 },
  { "oscil3.kkA",S(OSC),0,         "k",  "kki[]o", oscsetA, koscl3, NULL },
  { "oscil3.kaA",S(OSC),0,         "a",  "kai[]o", oscsetA, oscka3 },
  { "oscil3.akA",S(OSC),0,         "a",  "aki[]o", oscsetA, oscak3 },
  { "oscil3.aaA",S(OSC),0,         "a",  "aai[]o", oscsetA, oscaa3 },
  { "poscil.a", S(OSC), TR, "a", "kkjo", (SUBR)posc_set,(SUBR)posckk },
  { "poscil.kk", S(OSC), TR, "k", "kkjo", (SUBR)posc_set,(SUBR)kposc,NULL },
  { "poscil.ka", S(OSC), TR, "a", "kajo", (SUBR)posc_set,  (SUBR)poscka },
  { "poscil.ak", S(OSC), TR, "a", "akjo", (SUBR)posc_set,  (SUBR)poscak },
  { "poscil.aa", S(OSC), TR, "a", "aajo", (SUBR)posc_set,  (SUBR)poscaa },
  { "lposcil",  S(LPOSC), TR, "a", "kkkkjo", (SUBR)lposc_set, (SUBR)lposc},
  { "poscil3.a",S(OSC), TR, "a", "kkjo",
    (SUBR)posc_set,(SUBR)posc3kk },
  { "poscil3.kk",S(OSC), TR, "k", "kkjo",
    (SUBR)posc_set,(SUBR)kposc3,NULL},
  { "poscil3.ak", S(OSC), TR, "a", "akjo", (SUBR)posc_set, (SUBR)posc3ak },
  { "poscil3.ka", S(OSC), TR, "a", "kajo", (SUBR)posc_set, (SUBR)posc3ka },
  { "poscil3.aa", S(OSC), TR, "a", "aajo", (SUBR)posc_set, (SUBR)posc3aa },
  { "lposcil3", S(LPOSC), TR,  "a", "kkkkjo", (SUBR)lposc_set,(SUBR)lposc3},
  { "lposcila", S(LPOSC),      TR,  "a", "akkkio",
    (SUBR)lposc_set, (SUBR)lposca},
  { "foscil", S(FOSC),TR,        "a",  "xkxxkjo",foscset,   foscil  },
  { "foscili",S(FOSC),TR,        "a",  "xkxxkjo",foscset,   foscili },
  { "loscil", S(LOSC),TR,        "mm","xkjojoojoo",losset, loscil   },
  { "loscilphs", S(LOSCPHS),TR,   "amm","xkjojoojoo",losset_phs, loscil_phs   },
  { "loscil3phs", S(LOSCPHS),TR,  "amm","xkjojoojoo",losset_phs, loscil3_phs  },
  { "loscil3", S(LOSC),TR,        "mm","xkjojoojoo",losset, loscil3  },
  { "adsyn",  S(ADSYN),0,         "a",    "kkkSo", adset_S,   adsyn   },
  { "adsyn.i",  S(ADSYN),0,         "a",    "kkkio", adset,   adsyn   },
  { "buzz",   S(BUZZ),TR,        "a",  "xxkio",  bzzset,   buzz    },
  { "gbuzz",  S(GBUZZ),TR,        "a",  "xxkkkio",gbzset,   gbuzz   },
  { "pluck",  S(PLUCK), TR,       "a",  "kkiiioo",plukset,   pluck   },
  { "rand",   S(RAND),0,          "a",    "xvoo", rndset,  arand   },
  { "rand.k",   S(RAND),0,          "k",    "xvoo", rndset, krand,  NULL  },
  { "randh",  S(RANDH),0,         "a",    "xxvoo", rhset, randh   },
  { "randh.k",  S(RANDH),0,         "k",    "xxvoo", rhset, krandh, NULL   },
  { "randi",  S(RANDI),0,         "a",    "xxvoo", riset, randi     },
  { "randi.k",  S(RANDI),0,       "k",    "xxvoo", riset, krandi    },
  { "randc",  S(RANDC),0,         "a",    "xxvoo", rcset, randc     },
  { "randc.k",  S(RANDC),0,       "k",    "xxvoo", rcset, krandc    },
  { "port",   S(PORT),0,          "k",    "kio",  porset, port      },
  { "tone.k", S(TONE),0,          "a",    "ako",  tonset,   tone    },
  { "tonex.k",S(TONEX),0,         "a",    "akoo", tonsetx,  tonex   },
  { "atone.k",  S(TONE),0,        "a",    "ako",  tonset,   atone   },
  { "atonex.k", S(TONEX),0,       "a",    "akoo", tonsetx,  atonex  },
  { "reson", S(RESON),   0,       "a",    "axxoo", rsnset,  reson   },
  { "resonx", S(RESONX),0,        "a",    "axxooo", rsnsetx, resonx },
  { "areson.kk", S(RESON),0,      "a",    "akkoo",rsnset,   areson  },
  { "lpread", S(LPREAD),0,        "kkkk", "kSoo", lprdset_S,lpread          },
  { "lpread.i", S(LPREAD),0,        "kkkk", "kioo", lprdset,lpread          },
  { "lpform", S(LPFORM),0,        "kk", "k",     lpformantset,lpformant   },
  { "lpreson",S(LPRESON),0,       "a",    "a",    lprsnset,  lpreson },
  { "lpfreson",S(LPFRESON),0,     "a",    "ak",   lpfrsnset, lpfreson},
  { "lpslot"  ,  S(LPSLOT),0,     "",     "i",    lpslotset, NULL, NULL   },
  { "lpinterp", S(LPINTERPOL),0,  "",     "iik",  lpitpset, lpinterpol, NULL},
  { "rms",    S(RMS),0,           "k",    "aqo",  rmsset, rms             },
  { "gain",   S(GAIN),0,          "a",    "akqo", gainset,   gain    },
  { "balance",S(BALANCE),0,       "a",    "aaqo", balnset,   balance },
  { "balance2",S(BALANCE),0,       "a",    "aaqo", balnset,   balance2 },
  { "pan",    S(PAN),0,    "aaaa", "akkioo",(SUBR)panset, (SUBR)pan  },
  { "soundin",S(DISKIN2),0,"mmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm","Soooo",
    sndinset_S, soundin, (SUBR) diskin2_async_deinit   },
  { "soundin.i",S(DISKIN2),0,"mmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm","ioooo",
    sndinset, soundin, (SUBR) diskin2_async_deinit   },
  { "soundout",S(SNDOUT), _QQ,    "",    "aSo",  sndoutset_S, soundout, soundout_deinit},
  { "soundout.i",S(SNDOUT), _QQ,    "",    "aio",  sndoutset, soundout, soundout_deinit },
  { "soundouts",S(SNDOUTS),_QQ,   "",    "aaSo", sndoutset_S, soundouts, soundout_deinit},
  { "soundouts.i",S(SNDOUTS),_QQ,   "",    "aaio", sndoutset, soundouts, soundout_deinit},
  { "in.a",   S(INM),0,            "a",    "",     NULL,   in      },
  { "in.s",   S(INS),0,            "aa",    "",     NULL,   ins      },
  { "in.q",   S(INQ),0,            "aaaa",    "",     NULL,   inq      },
  { "in.A",   S(INA),0,            "a[]",  "",     (SUBR)inarray_set,   inarray },
   /* start deprecated */
  { "ins",    S(INS),0,            "aa",   "",     NULL,   ins  , NULL, NULL, 1},
  { "inq",    S(INQ),0,            "aaaa", "",     NULL,   inq  , NULL, NULL, 1},
  { "outs",   S(OUTX),IR,           "",     "y",    ochn,   outall , NULL, NULL, 1},
  { "outq",   S(OUTX),IR,           "",     "y",    ochn,   outall , NULL, NULL, 1},
  { "outh",   S(OUTX),IR,           "",     "y",    ochn,   outall , NULL, NULL, 1},
  { "outo",   S(OUTX),IR,           "",     "y",    ochn,   outall , NULL, NULL, 1},
  { "outx",   S(OUTX),IR,           "",     "y",    ochn,   outall , NULL, NULL, 1},
  { "out32",  S(OUTX),IR,           "",     "y",    ochn,   outall , NULL, NULL, 1},
  { "outs1",  S(OUTM),IR,           "",     "a",    NULL,   outs1  , NULL, NULL, 1},
  { "outs2",  S(OUTM),IR,          "",     "a",    och2,   outs2   , NULL, NULL, 1},
  { "outq1",  S(OUTM),IR,           "",     "a",    NULL,   outs1  , NULL, NULL, 1},
  { "outq2",  S(OUTM),IR,          "",     "a",    och2,   outs2   , NULL, NULL, 1},
  { "outq3",  S(OUTM),IR,          "",     "a",    och3,   outq3   , NULL, NULL, 1},
  { "outq4",  S(OUTM),IR,          "",     "a",    och2,   outq4   , NULL, NULL, 1},
  /* end deprecated */
  { "out.a",  S(OUTX),IR,           "",     "y",    ochn,   outall },
  { "out.A",  S(OUTARRAY),IR,       "",     "a[]",  outarr_init,  outarr },
  { "outall", S(OUTM),IR,           "",     "a",    NULL,   outrep  },
  { "igoto",  S(GOTO),0,          "",     "l",    igoto                   },
  { "kgoto",  S(GOTO),0,           "",     "l",    NULL,   kgoto           },
  { "goto",   S(GOTO),0,          "",     "l",    igoto,  kgoto           },
  { "cigoto", S(CGOTO),0,         "",     "Bl",   icgoto                  },
  { "ckgoto", S(CGOTO),0,          "",     "Bl",   NULL,   kcgoto          },
  { "cggoto.0", S(CGOTO),0,       "",     "Bl",   icgoto, kcgoto          },
  { "timout", S(TIMOUT),0,        "",     "iil",  timset, timout          },
  { "reinit", S(GOTO),0,           "",     "l",    NULL,   reinit          },
  { "rigoto", S(GOTO),0,          "",     "l",    rigoto                  },
  { "rireturn",S(LINK),0,         "",     "",     rireturn                },
  { "tigoto", S(GOTO),0,          "",     "l",    tigoto                  },
  { "tival",  S(EVAL),0,          "i",    "",     tival                   },
  { "print",  S(PRINTV),WR,       "",     "m",    printv                  },
  { "display.k",S(DSPLAY),0,      "",     "kioo", dspset, kdsplay,NULL    },
  { "display.a",S(DSPLAY),0,      "",     "aioo", dspset ,dsplay    },
  { "pvsdisp",S(FSIGDISP),0,      "",     "foo", fdspset, fdsplay,NULL    },
  { "dispfft.k",S(DSPFFT),0,      "",     "kiiooooo",fftset,kdspfft,NULL  },
  { "dispfft.a",S(DSPFFT),0,      "",     "aiiooooo",fftset,dspfft   },
  { "dumpk",  S(KDUMP),0,         "",     "kSii", kdmpset_S,kdump          },
  { "dumpk2", S(KDUMP2),0,        "",     "kkSii",kdmp2set_S,kdump2        },
  { "dumpk3", S(KDUMP3),0,        "",     "kkkSii",kdmp3set_S,kdump3       },
  { "dumpk4", S(KDUMP4),0,        "",     "kkkkSii",kdmp4set_S,kdump4      },
  { "dumpk.i",  S(KDUMP),0,         "",     "kiii", kdmpset_p,kdump        },
  { "dumpk2.i", S(KDUMP2),0,        "",     "kkiii",kdmp2set_p,kdump2      },
  { "dumpk3.i", S(KDUMP3),0,        "",     "kkkiii",kdmp3set_p,kdump3     },
  { "dumpk4.i", S(KDUMP4),0,        "",     "kkkkiii",kdmp4set_p,kdump4    },
  { "readk",  S(KREAD),0,         "k",    "Sii",   krdset_S, kread         },
  { "readk2", S(KREAD2),0,        "kk",   "Sii",   krd2set_S, kread2       },
  { "readk3", S(KREAD3),0,        "kkk",  "Sii",   krd3set_S, kread3       },
  { "readk4", S(KREAD4),0,        "kkkk", "Sii",   krd4set_S, kread4       },
  { "readk.i",  S(KREAD),0,         "k",    "iii",   krdset_p, kread       },
  { "readk2.i", S(KREAD2),0,        "kk",   "iii",   krd2set_p, kread2     },
  { "readk3.i", S(KREAD3),0,        "kkk",  "iii",   krd3set_p, kread3     },
  { "readk4.i", S(KREAD4),0,        "kkkk", "iii",   krd4set_p, kread4     },
  { "readks", S(KREADS),0,        "S",    "Si",    krdsset_S, kreads       },
  { "readks.i", S(KREADS),0,        "S",    "ii",    krdsset_p, kreads     },
  { "xyin",   S(XYIN), _QQ,       "kk",   "iiiiioo",xyinset,NULL          },
  { "tempest",  S(TEMPEST),0,     "k","kiiiiiiiiiop",tempeset,tempest},
  { "tempo",    S(TEMPO),0,       "",     "ki",   tempo_set,tempo           },
  { "pow.i",    S(POW),0,         "i",    "iip",  ipow,    NULL,  NULL    },
  { "pow.k",    S(POW),0,          "k",    "kkp",  NULL,    ipow,  NULL    },
  { "pow.a",    S(POW),0,          "a",    "akp",  NULL,  apow    },
  { "##pow.i",  S(POW),0,         "i",    "iip",  ipow,    NULL,  NULL    },
  { "##pow.k",  S(POW),0,          "k",    "kkp",  NULL,    ipow,  NULL    },
  { "##pow.a",  S(POW),0,          "a",    "akp",  NULL,  apow    },
  { "oscilx",   S(OSCILN), TR,    "a",    "kiii", oscnset,   osciln  },
  { "linrand.i",S(PRAND),0,       "i",    "k",    iklinear, NULL, NULL    },
  { "linrand.k",S(PRAND),0,        "k",    "k",    NULL, iklinear, NULL    },
  { "linrand.a",S(PRAND),0,        "a",    "k",    NULL,     alinear },
  { "trirand.i",S(PRAND),0,       "i",    "k",    iktrian, NULL,  NULL    },
  { "trirand.k",S(PRAND),0,        "k",    "k",    NULL, iktrian,  NULL    },
  { "trirand.a",S(PRAND),0,        "a",    "k",    NULL,     atrian  },
  { "exprand.i",S(PRAND),0,       "i",    "k",    ikexp, NULL,    NULL    },
  { "exprand.k",S(PRAND),0,        "k",    "k",    NULL,    ikexp, NULL    },
  { "exprand.a",S(PRAND),0,        "a",    "k",    NULL,     aexp    },
  { "bexprnd.i",S(PRAND),0,       "i",    "k",    ikbiexp, NULL,  NULL    },
  { "bexprnd.k",S(PRAND),0,        "k",    "k",    NULL, ikbiexp,  NULL    },
  { "bexprnd.a",S(PRAND),0,        "a",    "k",    NULL,     abiexp  },
  { "cauchy.i", S(PRAND),0,       "i",    "k",    ikcauchy, NULL, NULL    },
  { "cauchy.k", S(PRAND),0,        "k",    "k",    NULL, ikcauchy, NULL    },
  { "cauchy.a", S(PRAND),0,        "a",    "k",    NULL,  acauchy },
  { "pcauchy.i",S(PRAND),0,       "i",    "k",    ikpcauchy, NULL,NULL    },
  { "pcauchy.k",S(PRAND),0,        "k",    "k",    NULL, ikpcauchy,NULL    },
  { "pcauchy.a",S(PRAND),0,        "a",    "k",    NULL,  apcauchy},
  { "poisson.i",S(PRAND),0,       "i",    "k",    ikpoiss, NULL,  NULL    },
  { "poisson.k",S(PRAND),0,        "k",    "k",    NULL, ikpoiss,  NULL    },
  { "poisson.a",S(PRAND),0,        "a",    "k",    NULL,  apoiss  },
  { "gauss.i" , S(PRAND),0,       "i",    "k",    ikgaus,  NULL,  NULL    },
  { "gauss.k" , S(PRAND),0,        "k",    "k",    NULL, ikgaus,   NULL    },
  { "gauss.a" , S(PRAND),0,        "a",    "k",    NULL,  agaus   },
  { "gauss.iii" , S(GAUSS),0,       "i",    "ii",    gauss_scalar,  NULL,  NULL    },
  { "gauss.kkk" , S(GAUSS),0,        "k",    "kk",    NULL, gauss_scalar,   NULL    },
  { "gauss.akk" , S(GAUSS),0,        "a",    "kk",    NULL, gauss_vector   },
  { "weibull.i",S(PRAND),0,       "i",    "kk",   ikweib,  NULL,  NULL    },
  { "weibull.k",S(PRAND),0,        "k",    "kk",   NULL, ikweib,   NULL    },
  { "weibull.a",S(PRAND),0,        "a",    "kk",   NULL,  aweib   },
  { "betarand.i",S(PRAND),0,      "i",    "kkk",  ikbeta, NULL,  NULL     },
  { "betarand.k",S(PRAND),0,       "k",    "kkk",  NULL,   ikbeta,NULL     },
  { "betarand.a",S(PRAND),0,       "a",    "kkk",  NULL,  abeta    },
  { "seed",     S(PRAND),0,       "",     "i",    seedrand, NULL, NULL    },
  { "getseed.i",S(GETSEED),0,     "i",     "",    getseed, NULL, NULL     },
  { "getseed.k",S(GETSEED),0,     "k",     "",    getseed, getseed, NULL  },
  { "unirand.i",S(PRAND),0,      "i",     "k",    ikuniform, NULL,  NULL  },
  { "unirand.k",S(PRAND),0,       "k",     "k",    NULL,    ikuniform, NULL},
  { "unirand.a",S(PRAND),0,       "a",     "k",    NULL, auniform },
  { "diskin",S(DISKIN2_ARRAY),0,     "a[]",
    "SPooooooo",
    (SUBR) diskin_init_array_S,
    (SUBR) diskin2_perf_array, (SUBR) diskin2_async_deinit_array },
  { "diskin2",S(DISKIN2_ARRAY),0,  "a[]",
    "SPooooooo",
    (SUBR) diskin2_init_array_S,
    (SUBR) diskin2_perf_array, (SUBR) diskin2_async_deinit_array},
  { "diskin.i",S(DISKIN2_ARRAY),0,     "a[]",
    "iPooooooo",
    (SUBR) diskin_init_array_I,
    (SUBR) diskin2_perf_array, (SUBR) diskin2_async_deinit_array},
  { "diskin2.i",S(DISKIN2_ARRAY),0,  "a[]",
    "iPooooooo",
    (SUBR) diskin2_init_array_I,
    (SUBR) diskin2_perf_array, (SUBR) diskin2_async_deinit_array},
  { "diskin",S(DISKIN2),0,     "mmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm",
    "SPooooooo",
    (SUBR) diskin_init_S,
    (SUBR) diskin2_perf, (SUBR) diskin2_async_deinit                         },
  { "diskin2",S(DISKIN2),0,  "mmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm",
    "SPooooooo",
    (SUBR) diskin2_init_S,
    (SUBR) diskin2_perf, (SUBR) diskin2_async_deinit                           },
  { "diskin.i",S(DISKIN2),0,     "mmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm",
    "iPooooooo",
    (SUBR) diskin_init,
    (SUBR) diskin2_perf, (SUBR) diskin2_async_deinit },
  { "diskin2.i",S(DISKIN2),0,  "mmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm",
    "iPooooooo",
    (SUBR) diskin2_init,
    (SUBR) diskin2_perf, (SUBR) diskin2_async_deinit  },
  { "noteon", S(OUT_ON),0,        "",     "iii",  iout_on, NULL,   NULL    },
  { "noteoff", S(OUT_ON),0,       "",     "iii",  iout_off, NULL,    NULL  },
  { "noteondur",S(OUT_ON_DUR),0,  "", "iiii", iout_on_dur_set,iout_on_dur,NULL},
  { "noteondur2",S(OUT_ON_DUR),0, "", "iiii", iout_on_dur_set,iout_on_dur2,NULL},
  { "moscil",S(MOSCIL),0,         "",     "kkkkk",moscil_set, moscil, NULL},
  { "midion",S(KOUT_ON),0,        "", "kkk", kvar_out_on_set,kvar_out_on,NULL},
  { "outic",S(OUT_CONTR),0,       "",     "iiiii", out_controller, NULL, NULL},
  { "outkc",S(OUT_CONTR),0,        "",     "kkkkk", NULL, out_controller, NULL},
  { "outic14",S(OUT_CONTR14),0,   "",     "iiiiii",out_controller14, NULL,NULL},
  { "outkc14",S(OUT_CONTR14),0,    "",     "kkkkkk",NULL, out_controller14, NULL},
  { "outipb",S(OUT_PB),0,         "",     "iiii", out_pitch_bend, NULL , NULL},
  { "outkpb",S(OUT_PB),0,          "",     "kkkk", NULL,  out_pitch_bend, NULL},
  { "outiat",S(OUT_ATOUCH),0,     "",     "iiii", out_aftertouch, NULL , NULL},
  { "outkat",S(OUT_ATOUCH),0,      "",     "kkkk", NULL,  out_aftertouch, NULL},
  { "outipc",S(OUT_PCHG),0,       "",     "iiii", out_progchange, NULL , NULL},
  { "outkpc",S(OUT_PCHG),0,        "",     "kkkk", NULL,  out_progchange, NULL},
  { "outipat",S(OUT_POLYATOUCH),0,"",    "iiiii", out_poly_aftertouch, NULL,NULL},
  { "outkpat",S(OUT_POLYATOUCH),0, "",    "kkkkk", NULL, out_poly_aftertouch,NULL},
  { "release",S(REL),0,           "k",    "",     release_set, release, NULL },
  { "xtratim",S(XTRADUR),0,       "",     "i",    xtratim,    NULL,     NULL },
  { "mclock", S(MCLOCK),0,        "",     "i",    mclock_set, mclock,   NULL },
  { "mrtmsg", S(XTRADUR),0,       "",     "i",    mrtmsg,     NULL,     NULL },
  { "midiout",S(MIDIOUT),0,        "",     "kkkk", NULL, midiout,   NULL      },
  { "midiout_i",S(MIDIOUT), 0,       "",     "iiii", midiout,   NULL, NULL     },
  { "midiouti",S(MIDIOUT), 0,       "",     "iiii", midiout,   NULL, NULL     }, /* Alias */
  { "midion2", S(KON2),0,         "",     "kkkk", kon2_set, kon2,   NULL     },
  { "nrpn",   S(NRPN),0,           "",     "kkk",  NULL,  nrpn ,NULL          },
  { "mdelay", S(MDELAY),0,        "",     "kkkkk",mdelay_set, mdelay,   NULL },
  { "nsamp.i", S(EVAL),0,         "i",    "i",    numsamp                    },
  { "powoftwo.i",S(EVAL),0,       "i",    "i",    powoftwo                   },
  { "powoftwo.k",S(EVAL),0,        "k",    "k",    NULL, powoftwo             },
  { "powoftwo.a",S(EVAL),0,        "a",    "a",    NULL, powoftwoa      },
  { "logbtwo.i",S(EVAL),0,        "i",    "i",    ilogbasetwo                },
  { "logbtwo.k",S(EVAL),0,        "k",    "k",    logbasetwo_set, logbasetwo },
  { "logbtwo.a",S(EVAL),0,        "a",    "a",
    logbasetwo_set, logbasetwoa },
  { "filelen", S(SNDINFO),0,      "i",    "Sp",   filelen_S, NULL, NULL        },
  { "filenchnls", S(SNDINFO),0,   "i",    "Sp",   filenchnls_S, NULL, NULL     },
  { "filesr", S(SNDINFO),0,       "i",    "Sp",   filesr_S, NULL, NULL         },
  { "filebit", S(SNDINFO),0,       "i",   "Sp",   filebit_S, NULL, NULL        },
  { "filepeak", S(SNDINFOPEAK),0,  "i",   "So",   filepeak_S, NULL, NULL       },
  { "filevalid", S(FILEVALID),0,   "i",   "S",    filevalid_S, NULL, NULL      },
  { "filelen.i", S(SNDINFO),0,      "i",    "ip",   filelen, NULL, NULL        },
  { "filenchnls.i", S(SNDINFO),0,   "i",    "ip",   filenchnls, NULL, NULL     },
  { "filesr.i", S(SNDINFO),0,       "i",    "ip",   filesr, NULL, NULL         },
  { "filebit.i", S(SNDINFO),0,       "i",   "ip",   filebit, NULL, NULL        },
  { "filepeak.i", S(SNDINFOPEAK),0,  "i",   "io",   filepeak, NULL, NULL       },
  { "filevalid.i", S(FILEVALID),0,   "i",   "i",    filevalid, NULL, NULL      },
  { "filevalid.k", S(FILEVALID),0,    "k",   "S",    NULL, filevalid_S, NULL    },
  { "filevalid.k", S(FILEVALID),0,    "k",   "i",    NULL, filevalid, NULL    },
  /*  { "nlalp", S(NLALP),0,          "a",  "akkoo", nlalp_set, nlalp }, */
  { "ptableiw",  S(TABLEW),TW|_QQ,  "", "iiiooo", (SUBR)tablew_init, NULL, NULL},
  { "ptablew.kk", S(TABLEW),TB,    "", "kkiooo",(SUBR)tabl_setup,
    (SUBR)tablew_kontrol, NULL          },
  { "ptablew.aa", S(TABLEW),TB,    "", "aaiooo",(SUBR)tabl_setup,
    (SUBR)tablew_audio               },
  { "tableiw",  S(TABL),TW|_QQ,  "",   "iiiooo", (SUBR)tablew_init, NULL, NULL},
  { "tablew",  S(TABL),TW,     "",   "iiiooo", (SUBR)tablew_init, NULL, NULL},
  { "tablew.kk", S(TABL),TW,      "", "kkiooo",(SUBR)tabl_setup,
    (SUBR)tablew_kontrol, NULL          },
  { "tablew.aa", S(TABL),TW,      "", "aaiooo",(SUBR)tabl_setup,
    (SUBR)tablew_audio               },
  { "tablewkt.kk", S(TABL),TW, "",  "kkkooo",
    (SUBR)tablkt_setup,(SUBR)tablewkt_kontrol,NULL},
  { "tablewkt.aa", S(TABL),TW, "",  "aakooo",
    (SUBR)tablkt_setup,(SUBR)tablewkt_audio},
  { "tableng.i", S(TLEN),TR,     "i",  "i",    (SUBR)table_length, NULL,  NULL},
  { "tableng.k",  S(TLEN),TR,     "k",  "k",    NULL,  (SUBR)table_length, NULL},
  { "tableigpw",S(TGP), TB,      "",  "i",    (SUBR)table_gpw, NULL,  NULL},
  { "tablegpw", S(TGP), TB,       "",  "k",    NULL,   (SUBR)table_gpw, NULL},
  { "tableimix",S(TABLMIX),TB,   "",  "iiiiiiiii", (SUBR)table_mix, NULL, NULL},
  { "tablemix", S(TABLMIX),TB,    "",  "kkkkkkkkk", NULL, (SUBR)table_mix, NULL},
  { "tableicopy",S(TGP),TB,  "", "ii",   (SUBR)table_copy, NULL, NULL},
  { "tablecopy", S(TGP),TB,   "", "kk", NULL, (SUBR)table_copy, NULL},
  { "tablera", S(TABLRA),TR,    "a",  "kkk",
    (SUBR)table_ra_set, (SUBR)table_ra},
  { "tablewa", S(TABLWA),TW,    "k",  "kakp",
    (SUBR)table_wa_set, (SUBR)table_wa},
  { "tablekt",  S(TABL),TR,    "k",  "xkooo",  (SUBR)tablkt_setup,
    (SUBR)tablerkt_kontrol,
    NULL         },
  { "tablekt.a",  S(TABL),TR,    "a",  "xkooo",  (SUBR)tablkt_setup,
    (SUBR)tablerkt_audio           },
  { "tableikt", S(TABL),TR,     "k",  "xkooo", (SUBR)tablkt_setup,
    (SUBR)tableirkt_kontrol,
    NULL          },
  { "tableikt.a", S(TABL),TR,     "a",  "xkooo", (SUBR)tablkt_setup,
    (SUBR)tableirkt_audio          },
  { "table3kt", S(TABL),TR,   "k",  "xkooo", (SUBR)tablkt_setup,
    (SUBR)table3rkt_kontrol,
    NULL         },
  { "table3kt.a", S(TABL),TR,   "a",  "xkooo", (SUBR)tablkt_setup,
    (SUBR)table3rkt_audio          },
  { "inz",    S(IOZ),    ZW,     "",   "k",  NULL,   (SUBR)inz  },
  { "outz",   S(IOZ),ZR|IR,      "",   "k",    NULL,   (SUBR)outz },

  { "timek.k",  S(RDTIME),0,    "k",  "",     NULL,    (SUBR)timek, NULL },
  { "timek.i", S(RDTIME),0,    "i",  "",     (SUBR)timek,   NULL,  NULL },

  { "times.k",  S(RDTIME),0,    "k",  "",     NULL,    (SUBR)timesr,NULL },
  { "times.i", S(RDTIME),0,    "i",  "",     (SUBR)timesr,  NULL,  NULL },

  { "elapsedcycles.k",  S(RDTIME),0,    "k",  "",     NULL,    (SUBR)elapsedcycles, NULL },
  { "elapsedcycles.i", S(RDTIME),0,    "i",  "",     (SUBR)elapsedcycles,   NULL,  NULL },

  { "elapsedtime.k",  S(RDTIME),0,    "k",  "",     NULL,    (SUBR)elapsedtime,NULL },
  { "elapsedtime.i", S(RDTIME),0,    "i",  "",     (SUBR)elapsedtime,  NULL,  NULL },

  { "timeinstk", S(RDTIME),0,  "k",  "",
    (SUBR)instimset, (SUBR)instimek, NULL },
  { "timeinsts", S(RDTIME),0,  "k",  "",
    (SUBR)instimset, (SUBR)instimes, NULL },

  { "eventcycles", S(RDTIME),0,  "k",  "",
    (SUBR)instimset, (SUBR)eventcycles, NULL },
  { "eventtime", S(RDTIME),0,  "k",  "",
    (SUBR)instimset, (SUBR)eventtime, NULL },

  { "peak.k",  S(PEAK),0,       "k",  "k",    NULL,    (SUBR)peakk,    NULL    },
  { "peak.a",   S(PEAK),0,      "k",  "a",    NULL,     (SUBR)peaka   },
  { "printk", S(PRINTK),WR,  "",     "ikoooo",
    (SUBR)printkset, (SUBR)printk, NULL },
  { "printks",S(PRINTKS),WR,    "",   "SiN",
    (SUBR)printksset_S,(SUBR)printks, NULL },
  { "printks2", sizeof(PRINTK3),0,  "", "Sk", (SUBR)printk3set, (SUBR)printk3 },
  { "printks.i",S(PRINTKS),WR,    "",   "iiN",
    (SUBR)printksset,(SUBR)printks, NULL },
  { "prints",S(PRINTS),0,      "",   "SN",   (SUBR)printsset_S, NULL, NULL },
  { "prints.i",S(PRINTS),0,      "",   "iN",   (SUBR)printsset, NULL, NULL },
  { "printk2", S(PRINTK2), WR,  "",   "koo",
    (SUBR)printk2set, (SUBR)printk2, NULL },
  { "portk",  S(PORT),0,     "k",     "kko",  (SUBR)porset,  (SUBR)kport, NULL },
  { "tonek",  S(TONE),0,     "k",     "kko",  (SUBR)tonset,  (SUBR)ktone, NULL },
  { "atonek", S(TONE),0,     "k",     "kko",  (SUBR)tonset,  (SUBR)katone, NULL},
  { "resonk", S(RESON),0,    "k",     "kkkpo",(SUBR)rsnset,  (SUBR)kreson, NULL},
  { "aresonk",S(RESON),0,    "k",     "kkkpo",(SUBR)rsnset,  (SUBR)kreson, NULL},
  { "aresonk",S(RESON),0,    "k",     "kkkpo",(SUBR)rsnset,  (SUBR)kareson, NULL},
  { "limit.i", S(LIMIT),0,   "i",     "iii",  (SUBR)klimit,  NULL,    NULL      },
  { "limit.k",  S(LIMIT),0,   "k",     "kkk",  NULL,          (SUBR)klimit, NULL },
  { "limit.a",  S(LIMIT),0,   "a",     "akk",  NULL,  (SUBR)limit },
  { "prealloc", S(AOP),0,    "",      "iio",  (SUBR)prealloc, NULL, NULL  },
  { "prealloc.S", S(AOP),0,  "",      "Sio",  (SUBR)prealloc_S, NULL, NULL  },
  /* opcode   dspace      thread  outarg  inargs  isub    ksub    asub    */
  { "inh",    S(INH),0,            "aaaaaa","",    NULL,   inh     },
  { "ino",    S(INO),0,            "aaaaaaaa","",  NULL,   ino     },
  { "inx",    S(INALL),0,          "aaaaaaaaaaaaaaaa","",  NULL,   in16 },
  { "in32",   S(INALL),0,          "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",
    "",     NULL,   in32 },
  { "inch",   S(INCH1),0,          "a",
    "k",    inch1_set,   (SUBR) inch_opcode1 },
  { "inch.m",   S(INCH),0,          "mmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm",
    "z",    inch_set,   inch_opcode },
  { "_in",    S(INALL),0,         "mmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm",
    "",     inch_set,   inall_opcode },
  /* Note that there is code in rdorch.c that assumes that opcodes starting
     with the characters out followed by a s, q, h, o or x are in this group
     ***BEWARE***
     CODE REMOVED 2011-Dec-14
  */
  { "outch",  S(OUTCH),IR,          "",     "Z",    NULL,   outch   },
  { "outc",   S(OUTX),IR,           "",     "y",    ochn,   outall  },
  { "cpsxpch", S(XENH),TR,       "i",    "iiii", cpsxpch, NULL,  NULL    },
  { "cps2pch", S(XENH),TR,       "i",    "ii",   cps2pch, NULL,  NULL    },
  { "cpstun", S(CPSTUN),  TR,        "k",    "kkk",   NULL,   cpstun         },
  { "cpstuni",S(CPSTUNI), TR,       "i",    "ii",   cpstun_i,               },
  { "cpstmid", S(CPSTABLE),0,  "i", "i",    (SUBR)cpstmid                    },
  { "adsr", S(LINSEG),0,          "k",    "iiiio",adsrset,klnseg, NULL },
  { "adsr.a", S(LINSEG),0,          "a",    "iiiio",adsrset, linseg     },
  { "madsr", S(LINSEG),0,         "k",    "iiiioj", madsrset,klnsegr,NULL },
  { "madsr.a", S(LINSEG),0,         "a",    "iiiioj", madsrset, linsegr },
  { "xadsr", S(EXXPSEG),0,        "k",    "iiiio", xdsrset, kxpseg, NULL   },
  { "xadsr.a", S(EXXPSEG),0,        "a",    "iiiio", xdsrset, expseg    },
  { "mxadsr", S(EXPSEG),0,        "k",    "iiiioj", mxdsrset, kxpsegr, NULL},
  { "mxadsr.a", S(EXPSEG),0,        "a",    "iiiioj", mxdsrset, expsegr},
  { "schedule.instr", S(SCHEDO),0,       "",     ":InstrDef;iim",
    schedule, NULL, NULL },
  { "schedule", S(SCHEDO),0,       "",     "iiim",
    schedule, NULL, NULL },
  { "schedule.S", S(SCHEDO),0,       "",     "Siim",
    schedule, NULL, NULL },
  { "schedule.Ninstr", S(SCHED),0,       "",     ":InstrDef;iiN",
    schedule_N, NULL, NULL },
  { "schedule.N", S(SCHED),0,       "",     "iiiN",
    schedule_N, NULL, NULL },
  { "schedule.SN", S(SCHED),0,       "",     "SiiN",
    schedule_SN, NULL, NULL },
  { "schedule.array", S(SCHED),0,       "",     "i[]",
    schedule_array, NULL, NULL },
  /* **** Experimental schedulek opcodes **** */
  { "schedulek.instr",   S(SCHEDO),0,        "",     ":InstrDef;kkM",
    NULL, schedule, NULL },
  { "schedulek",   S(SCHEDO),0,        "",     "kkkM",
    NULL, schedule, NULL },
  { "schedulek.S", S(SCHEDO),0,        "",     "SkkM",
    NULL, schedule, NULL },
  { "schedulek.Ninstr", S(SCHED),0,        "",     ":InstrDef;kkN",
    NULL, schedule_N, NULL },
  { "schedulek.N", S(SCHED),0,        "",     "kkkN",
    NULL, schedule_N, NULL },
  { "schedulek.SN", S(SCHED),0,       "",     "SkkN",
    NULL, schedule_SN, NULL },
  { "schedulek.array",   S(SCHED),0,        "",     "k[]",
    NULL, schedule_array, NULL },

  /* **** End of schedulek **** */
  { "schedwhen.instr", S(WSCHED),0,     "",  ":InstrDef;kkkm",ifschedule, kschedule, NULL },
  { "schedwhen", S(WSCHED),0,     "",     "kkkkm",ifschedule, kschedule, NULL },
  { "schedwhen", S(WSCHED),0,     "",     "kSkkm",ifschedule, kschedule, NULL },

  { "schedkwhen.instr", S(TRIGINSTR),0, "",     ":InstrDef;kkkkz",triginset, ktriginstr, NULL },
  { "schedkwhen", S(TRIGINSTR),0, "",     "kkkkkz",triginset, ktriginstr, NULL },
  { "schedkwhen.S", S(TRIGINSTR),0, "",    "kkkSkz",
    triginset_S, ktriginstr_S, NULL },
  { "schedkwhennamed", S(TRIGINSTR),0, "", "kkkkkz",triginset, ktriginstr, NULL },
  { "schedkwhennamed.S", S(TRIGINSTR),0, "",
    "kkkSkz",triginset_S, ktriginstr_S, NULL },
  { "trigseq", S(TRIGSEQ),0,      "",     "kkkkkz", trigseq_set, trigseq, NULL },
  { "event", S(LINEVENT),0,        "",     "S:InstrDef;z",  NULL, event_opcode_Instr, NULL   },
  { "event", S(LINEVENT),0,        "",     "Sz",  NULL, event_opcode, NULL   },
  { "event.S", S(LINEVENT),0,        "",    "SSz",  NULL, event_opcode_S, NULL   },
  { "event_i.instr", S(LINEVENT),0,     "",     "S:InstrDef;m",  event_opcode_i_Instr, NULL, NULL, NULL, 2  },
  { "event_i", S(LINEVENT),0,     "",     "Sim",  event_opcode_i, NULL, NULL, NULL, 2  },
  { "event_i.S", S(LINEVENT),0,     "",    "SSm",  event_opcode_i_S, NULL, NULL, NULL, 2  },
  /* aliases */
  { "eventi.instr", S(LINEVENT),0,     "",     "S:InstrDef;m",  event_opcode_i_Instr, NULL, NULL  },
  { "eventi", S(LINEVENT),0,     "",     "Sim",  event_opcode_i, NULL, NULL  },
  { "eventi.S", S(LINEVENT),0,     "",    "SSm",  event_opcode_i_S, NULL, NULL  },
  { "play", S(LINEVENT2), 0,  ":Instr;", ":InstrDef;m", (SUBR) play_instr, NULL, NULL },
  { "pause", S(PAUSE_INSTR), 0,  "", ":Instr;k", NULL, (SUBR) pause_instance_opcode,  NULL },
  { "create", S(CREATE_INSTANCE), 0,  ":Instr;", ":InstrDef;", (SUBR) create_instance_opcode, NULL, NULL },
  { "init", S(INIT_INSTANCE), 0,  "i", ":Instr;m", (SUBR) init_instance_opcode, NULL, NULL },
  { "perf", S(PERF_INSTR), 0,  "k", ":Instr;M", NULL, (SUBR) perf_instance_opcode, NULL },
  { "setp", S(PARM_INSTR), 0,  "", ":Instr;kk", NULL, (SUBR) set_instance_parameter, NULL },
  { "delete", S(DEL_INSTR), 0,  "", ":Instr;", NULL, NULL, (SUBR) delete_instance_opcode },
  { "getinstance", S(DEL_INSTR), 0,  ":Instr;", "", (SUBR) get_instance, NULL, NULL },
  { "isactive", S(INSTRACT), 0, "b" , ":Instr;", (SUBR) isactive, NULL, NULL },
  { "isactive", S(INSTRACT), 0, "B" , ":Instr;", NULL, (SUBR) isactive, NULL, },
  { "isreleasing", S(INSTRACT), 0, "b",  ":Instr;", (SUBR) isreleasing, NULL, NULL },
  { "isreleasing", S(INSTRACT), 0, "B",  ":Instr;", NULL, (SUBR) isreleasing, NULL, },
  { "isreleasing",S(REL),0,           "B",    "",     release_set, (SUBR) release_bool, NULL },
   { "isreleasing",S(REL),0,           "k",    "",     release_set, (SUBR) release, NULL },
  { "splice", S(SPLICE_INSTR), 0,  "i", ":Instr;:Instr;i", (SUBR) splice_instance, NULL, NULL },
  { "schedule.instri", S(LINEVENT2),0,   ":Instr;",  ":InstrDef;iim",  instance_opcode_Instr, NULL, NULL  },
  { "schedule.instrk", S(LINEVENT2),0,    ":Instr;",  ":InstrDef;kz",  NULL, instance_opcode_Instr, NULL   },
  { "schedule.i", S(LINEVENT2),0,   ":Instr;",  "iiim",  instance_opcode_, NULL, NULL  },
  { "schedule.k", S(LINEVENT2),0,      ":Instr;",  "kkz",  NULL, instance_opcode_, NULL   },
  { "schedule.S", S(LINEVENT2),0,   ":Instr;",  "Siim",  instance_opcode_S, NULL, NULL},
  { "schedule.kS", S(LINEVENT2),0,   ":Instr;",  "SSz",  NULL, instance_opcode_S, NULL },
  /* START DEPRECATED */
  { "nstance", S(LINEVENT2),0,      "k",  ":InstrDef;kz",  NULL, instance_opcode_Instr, NULL   },
  { "nstance", S(LINEVENT2),0,      "k",  "kkz",  NULL, instance_opcode_, NULL   },
  { "nstance.instr", S(LINEVENT2),0,   "i",  ":InstrDef;iim",  instance_opcode_Instr, NULL, NULL  },
  { "nstance.instr", S(LINEVENT2),0,   ":Instr;",  ":InstrDef;iim",  instance_opcode_Instr, NULL, NULL  },
  { "nstance", S(LINEVENT2),0,      ":Instr;",  ":InstrDef;kz",  NULL, instance_opcode_Instr, NULL   },
  { "nstance.i", S(LINEVENT2),0,   "i",  "iiim",  instance_opcode_, NULL, NULL  },
  { "nstance.i", S(LINEVENT2),0,   ":Instr;",  "iiim",  instance_opcode_, NULL, NULL  },
  { "nstance", S(LINEVENT2),0,      ":Instr;",  "kkz",  NULL, instance_opcode_, NULL   },
  { "nstance.kS", S(LINEVENT2),0,   "k",  "SSz",  NULL, instance_opcode_S, NULL },
  { "nstance.S", S(LINEVENT2),0,   "i",  "Siim",  instance_opcode_S, NULL, NULL},
  { "nstance.S", S(LINEVENT2),0,   ":Instr;",  "Siim",  instance_opcode_S, NULL, NULL},
  { "nstance.kS", S(LINEVENT2),0,   ":Instr;",  "SSz",  NULL, instance_opcode_S, NULL },
  { "turnoff.i", S(KILLOP),0,     "",     "i", kill_instance, NULL, NULL  },
  { "turnoff.k", S(KILLOP),0,      "",     "k", NULL, kill_instance, NULL},
  { "turnoff.k", S(KILLOP),0,      "",     "kk", NULL, (SUBR) kill_instancek, NULL},
  /* END DEPRECATED */
  { "turnoff.inst", S(KILLOP),0,      "",     ":Instr;", kill_instance, NULL, NULL},
  { "turnoff.inst", S(KILLOP),0,      "",     ":Instr;k", NULL, (SUBR) kill_instancek, NULL},
  { "lfo", S(LFO),0,              "k",    "kko",  lfoset,   lfok,   NULL   },
  { "lfo.a", S(LFO),0,              "a",    "kko",  lfoset,   lfoa    },
  { "oscils",   S(OSCILS),0,      "a", "iiio",
    (SUBR)oscils_set, (SUBR)oscils  },
  { "lphasor",  S(LPHASOR),0,     "a", "xooooooo" ,
    (SUBR)lphasor_set, (SUBR)lphasor },
  { "tablexkt", S(TABLEXKT),TR,  "a", "xkkiooo", (SUBR)tablexkt_set,
    (SUBR)tablexkt              },
  { "reverb2",  S(NREV2),0,       "a",    "akkoojoj",
    (SUBR)reverbx_set,(SUBR)reverbx  },
  { "nreverb",  S(NREV2),0,       "a",    "akkoojoj",
    (SUBR)reverbx_set,(SUBR) reverbx },
  { "=.f",      S(FASSIGN),0,     "f",   "f", (SUBR)fassign_set, (SUBR)fassign },
  { "init.f",   S(FASSIGN),0,     "f",   "f", (SUBR)fassign_set, NULL, NULL    },
  { "pvsanal",  S(PVSANAL), 0,    "f",   "aiiiioo", pvsanalset, pvsanal   },
  { "pvsynth",  S(PVSYNTH),0,     "a",   "fo",     pvsynthset, pvsynth },
  { "pvsadsyn", S(PVADS),0,       "a",   "fikopo", pvadsynset, pvadsyn, NULL },
  { "pvscross", S(PVSCROSS),0,    "f",   "ffkk",   pvscrosset, pvscross, NULL },
  { "pvsfread", S(PVSFREAD),0,    "f",   "kSo",    pvsfreadset_S, pvsfread, NULL},
  { "pvsfread.i", S(PVSFREAD),0,  "f",   "kio",    pvsfreadset, pvsfread, NULL},
  { "pvsmaska", S(PVSMASKA),0,    "f",   "fik",    pvsmaskaset, pvsmaska, NULL  },
  { "pvsftw",   S(PVSFTW),  TW,   "k",   "fio",    pvsftwset, pvsftw, NULL  },
  { "pvsftr",   S(PVSFTR),TR,     "",    "fio",    pvsftrset, pvsftr, NULL  },
  { "pvsinfo",  S(PVSINFO),0,     "iiii","f",      pvsinfo, NULL, NULL    },
  { "octave.i", S(EVAL),0,        "i",    "i",     powoftwo               },
  { "octave.k", S(EVAL),0,         "k",    "k",     NULL,  powoftwo        },
  { "octave.a", S(EVAL),0,         "a",    "a",     NULL, powoftwoa  },
  { "semitone.i",S(EVAL),0,       "i",    "i",     semitone               },
  { "semitone.k",S(EVAL),0,        "k",    "k",     NULL,  semitone        },
  { "semitone.a",S(EVAL),0,        "a",    "a",     NULL, asemitone  },
  { "cent.i",   S(EVAL),0,        "i",    "i",     cent                   },
  { "cent.k",   S(EVAL),0,         "k",    "k",     NULL,  cent            },
  { "cent.a",   S(EVAL),0,         "a",    "a",     NULL, acent      },
  { "db.i",     S(EVAL),0,        "i",    "i",     db                     },
  { "db.k",     S(EVAL),0,         "k",    "k",     NULL, db               },
  { "db.a",     S(EVAL),0,         "a",    "a",     NULL, dba        },
  { "midiclockin",  S(MIDIKMB),0,     "k",    "",     NULL, midi_clock_in, NULL    },
  { "midistart",  S(MIDIKMB),0,     "k",    "",     NULL, midi_start, NULL    },
    { "midistop",  S(MIDIKMB),0,     "k",    "",     NULL, midi_stop, NULL    },
    { "midicontinue",  S(MIDIKMB),0,     "k",    "",     NULL, midi_continue, NULL    },
  { "midiclockfreq",  S(MIDIKMB),0,     "k",    "", NULL, midi_clock_freq, NULL    },

  { "midichn",  S(MIDICHN),0,     "i",    "",      midichn, NULL, NULL    },
  { "pgmassign",S(PGMASSIGN),0,   "",     "iio",   pgmassign, NULL, NULL  },
  { "pgmassign.S",S(PGMASSIGN),0,  "",    "iSo",   pgmassign_S, NULL, NULL  },
  { "midiin",   S(MIDIIN),0,      "kkkk", "",      midiin_set, midiin, NULL },
  { "pgmchn",   S(PGMIN),0,       "kk",   "o",     pgmin_set, pgmin, NULL },
  { "ctlchn",   S(CTLIN),0,       "kkk",  "oo",    ctlin_set, ctlin, NULL },
  { "miditempo", S(MIDITEMPO),0,  "i",    "o",
    (SUBR) midiTempoOpcode, NULL, NULL    },
  { "miditempo", S(MIDITEMPO),0,  "k",    "o",
    NULL, (SUBR) midiTempoOpcode, NULL    },
  { "midifilestatus", S(MIDITEMPO),0,   "k",    "o",
    NULL, (SUBR) midiFileStatus, NULL },
  { "midinoteoff", S(MIDINOTEON),0   ,"", "xx",   midinoteoff, midinoteoff, },
  { "midinoteonkey", S(MIDINOTEON),0, "", "xx",   midinoteonkey, midinoteonkey },
  { "midinoteoncps", S(MIDINOTEON),0,  "", "xx",  midinoteoncps,midinoteoncps },
  { "midinoteonoct", S(MIDINOTEON),0,  "", "xx",  midinoteonoct,midinoteonoct },
  { "midinoteonpch", S(MIDINOTEON),0,  "", "xx",  midinoteonpch, midinoteonpch },
  { "midipolyaftertouch", S(MIDIPOLYAFTERTOUCH),0,
    "", "xxoh", midipolyaftertouch, midipolyaftertouch},
  { "midicontrolchange", S(MIDICONTROLCHANGE),0,
    "", "xxoh",midicontrolchange, midicontrolchange    },
  { "midiprogramchange", S(MIDIPROGRAMCHANGE),0,
    "", "x", midiprogramchange, midiprogramchange      },
  { "midichannelaftertouch", S(MIDICHANNELAFTERTOUCH),0,
    "", "xoh",midichannelaftertouch, midichannelaftertouch },
  { "midipitchbend", S(MIDIPITCHBEND),0, "", "xoh",
    midipitchbend, midipitchbend },
  { "mididefault", S(MIDIDEFAULT),0,  "", "xx",   mididefault, mididefault },
  //{ "invalue",   0xFFFF,   _CR,    0,   NULL,   NULL, NULL, NULL },
  { "invalue.k", S(INVAL),_CR,  "k", "i", (SUBR) invalset,(SUBR)kinval, NULL },
  { "invalue.i", S(INVAL),_CR,  "i", "i", (SUBR) invalsetgo, NULL, NULL },
  { "invalue.iS", S(INVAL),_CR,  "i", "S", (SUBR) invalsetSgo, NULL, NULL },
  { "invalue.kS", S(INVAL),_CR,  "k", "S",(SUBR) invalset_S,(SUBR)kinval, NULL },
  { "invalue.S", S(INVAL),_CR,  "S", "i",
    (SUBR) invalset_string, (SUBR)kinvalS, NULL },
  { "invalue.SS", S(INVAL),_CR,  "S", "S",
    (SUBR) invalset_string_S, (SUBR)kinvalS, NULL },
  { "outvalue", S(OUTVAL), _CW,  "", "ik", (SUBR) outvalset, (SUBR)koutval, NULL},
  { "outvalue.i", S(OUTVAL), _CW,  "", "ii", (SUBR) outvalsetgo, NULL, NULL},
  { "outvalue.Si", S(OUTVAL), _CW,  "", "Si",
    (SUBR) outvalsetSgo, NULL, NULL},
  { "outvalue.k", S(OUTVAL), _CW,  "", "Sk",
    (SUBR) outvalset_S, (SUBR)koutval, NULL},
  { "outvalue.S", S(OUTVAL), _CW,  "", "iS",
    (SUBR) outvalset_string, (SUBR)koutvalS, NULL},
  { "outvalue.SS", S(OUTVAL), _CW,  "", "SS",
    (SUBR) outvalset_string_S, (SUBR)koutvalS, NULL},
  /* IV - Oct 20 2002 */
  { "subinstr", S(SUBINST),0,  "mmmmmmmm", "SN",  subinstrset_S, subinstr },
  { "subinstrinit", S(SUBINST),0,  "",    "SN",   subinstrset_S, NULL, NULL     },
  { "subinstr.i", S(SUBINST),0,  "mmmmmmmm", "iN",  subinstrset, subinstr },
  { "subinstrinit.i", S(SUBINST),0,  "",    "iN",   subinstrset, NULL, NULL     },
  { "instrnum", S(NSTRNUM),0,      "i",    "S",    nstrnumset_S, NULL, NULL      },
  { "instrnum.i", S(NSTRNUM),0,      "i",    "i",    nstrnumset, NULL, NULL      },
  { "nstrnum", S(NSTRNUM),0,      "i",    "S",    nstrnumset_S, NULL, NULL      },
  { "nstrnum.i", S(NSTRNUM),0,      "i",    "i",    nstrnumset, NULL, NULL      },
  { "instrstr", S(NSTRSTR),0,        "S",    "i",    nstrstr, NULL, NULL      },
  { "instrstr.k", S(NSTRSTR),0,       "S",    "k",    NULL, nstrstr, NULL      },
  { "nstrstr", S(NSTRSTR),0,        "S",    "i",    nstrstr, NULL, NULL      },
  { "nstrstr.k", S(NSTRSTR),0,       "S",    "k",    NULL, nstrstr, NULL      },
  { "turnoff2_i.instr",S(TURNOFF2),_CW,     "",  ":InstrDef;oo",  turnoff2Instr, NULL, NULL, NULL, 2 },
  { "turnoff2i.instr",S(TURNOFF2),_CW,     "",  ":InstrDef;oo",  turnoff2Instr, NULL     }, /* alias */
  { "turnoff2_i.S",S(TURNOFF2),_CW,     "",     "Soo",  turnoff2S, NULL, NULL, NULL, 2     },
  { "turnoff2i.S",S(TURNOFF2),_CW,     "",     "Soo",  turnoff2S, NULL     }, /* alias */
  { "turnoff2_i.i",S(TURNOFF2),_CW,     "",     "ioo",  turnoff2k, NULL, NULL, NULL, 2     },
  { "turnoff2i.i",S(TURNOFF2),_CW,     "",     "ioo",  turnoff2k, NULL     }, /* alias */
  { "turnoff2.Instr",S(TURNOFF2),_CW,      "",     ":InstrDef;kk",  NULL, turnoff2Instr, NULL     },
  { "turnoff2.S",S(TURNOFF2),_CW,      "",     "SOO",  NULL, turnoff2S, NULL     },
  { "turnoff2.c",S(TURNOFF2),_CW,      "",     "iOO",  NULL, turnoff2k, NULL     },
  { "turnoff2.k",S(TURNOFF2),_CW,      "",     "kOO",  NULL, turnoff2k, NULL     },
  { "turnoff2.i",S(TURNOFF2),_CW,      "",     "iOO",  NULL, turnoff2k, NULL     },
  { "turnoff2.r",S(TURNOFF2),_CW,      "",     "iOO",  NULL, turnoff2k, NULL     },
  { "turnoff3.Instr",S(TURNOFF2),_CW,      "",     ":InstrDef;",  NULL, turnoff3Instr, NULL     },
  { "turnoff3.S",S(TURNOFF2),_CW,      "",     "S",  NULL, turnoff3S, NULL     },
  { "turnoff3.c",S(TURNOFF2),_CW,      "",     "i",  NULL, turnoff3k, NULL     },
  { "turnoff3.k",S(TURNOFF2),_CW,      "",     "k",  NULL, turnoff3k, NULL     },
  { "turnoff3.i",S(TURNOFF2),_CW,      "",     "i",  NULL, turnoff3k, NULL     },
  { "turnoff3.r",S(TURNOFF2),_CW,      "",     "i",  NULL, turnoff3k, NULL     },
  { "cngoto", S(CGOTO),0,         "",     "Bl",   ingoto, kngoto, NULL     },
  { "cnkgoto", S(CGOTO),0,          "",     "Bl",   NULL,  kngoto, NULL     },
  { "cingoto", S(CGOTO),0,         "",     "Bl",   ingoto, NULL, NULL     },
  { "tempoval", S(GTEMPO),0,    "k", "",      NULL, (SUBR)get_tempo, NULL    },
  { "downsamp",S(DOWNSAMP),0, "k", "ao",   (SUBR)downset,(SUBR)downsamp        },
  { "upsamp", S(UPSAMP),0,     "a", "k",    NULL,   (SUBR)upsamp        },
  /* IV - Sep 5 2002 */
  { "interp", S(INTERP),0,    "a", "kooo", (SUBR)interpset, (SUBR)interp  },
  { "a.k",    S(INTERP),0,    "a", "k",    (SUBR)a_k_set,   (SUBR)interp  },
  { "integ", S(INDIFF),  0,   "a", "xo",
    (SUBR)indfset,(SUBR)integrate},
  { "integ.k", S(INDIFF),  0,   "k", "xo",
    (SUBR)indfset,(SUBR)kntegrate, NULL},
  { "diff",   S(INDIFF),0,    "a", "xo", (SUBR)indfset, (SUBR)diff },
  { "diff.k",   S(INDIFF),0,    "k", "xo", (SUBR)indfset,(SUBR)kdiff, NULL },
  { "samphold",S(SAMPHOLD),0, "a", "xxoo",
    (SUBR)samphset,(SUBR)samphold    },
  { "samphold.k",S(SAMPHOLD),0, "k", "xxoo",
    (SUBR)samphset,(SUBR)ksmphold,NULL  },
  { "delay",  S(DELAY),0,     "a", "aio",  (SUBR)delset,   (SUBR)delay   },
  { "delayr", S(DELAYR),0,    "aX","io",   (SUBR)delrset,   (SUBR)delayr  },
  { "delayw", S(DELAYW),0,    "",  "a",    (SUBR)delwset,   (SUBR)delayw  },
  { "delay1", S(DELAY1),0,    "a", "ao",   (SUBR)del1set,   (SUBR)delay1  },
  { "deltap", S(DELTAP),0,    "a", "ko",   (SUBR)tapset,   (SUBR)deltap  },
  { "deltapi",S(DELTAP),0,    "a", "xo",   (SUBR)tapset,   (SUBR)deltapi },
  { "deltapn",S(DELTAP),0,    "a", "xo",   (SUBR)tapset,   (SUBR)deltapn },
  { "deltap3",S(DELTAP),0,    "a", "xo",   (SUBR)tapset,   (SUBR)deltap3 },
  { "reverb", S(REVERB),0,    "a", "ako",  (SUBR)rvbset,   (SUBR)reverb  },
  { "vdelay",   S(VDEL),0,    "a", "axio", (SUBR)vdelset,  (SUBR)vdelay  },
  { "vdelay3",  S(VDEL),0,    "a", "axio", (SUBR)vdelset,  (SUBR)vdelay3 },
  { "vdelayxwq",S(VDELXQ),0,  "aaaa", "aaaaaiio",
    (SUBR)vdelxqset, (SUBR)vdelayxwq},
  { "vdelayxws",S(VDELXS),0,  "aa", "aaaiio", (SUBR)vdelxsset,
    (SUBR)vdelayxws                  },
  { "vdelayxw", S(VDELX),0,   "a",  "aaiio",
    (SUBR)vdelxset, (SUBR)vdelayxw},
  { "vdelayxq", S(VDELXQ),0,  "aaaa", "aaaaaiio",
    (SUBR)vdelxqset, (SUBR)vdelayxq},
  { "vdelayxs", S(VDELXS),0,  "aa", "aaaiio",
    (SUBR)vdelxsset, (SUBR)vdelayxs},
  { "vdelayx",  S(VDELX),0,   "a",  "aaiio", (SUBR)vdelxset, (SUBR)vdelayx},
  { "deltapx",  S(DELTAPX),0, "a",  "aio",  (SUBR)tapxset,  (SUBR)deltapx },
  { "deltapxw", S(DELTAPX),0,  "",  "aaio", (SUBR)tapxset, (SUBR)deltapxw },
  { "multitap", S(MDEL),0,     "a", "am",
    (SUBR)multitap_set,(SUBR)multitap_play},
  { "comb",   S(COMB),0,      "a",  "akioo", (SUBR)cmbset, (SUBR)comb    },
  { "combinv",S(COMB),0,      "a",  "akioo", (SUBR)cmbset, (SUBR)invcomb },
  { "alpass", S(COMB),0,      "a",  "axioo", (SUBR)cmbset, (SUBR)alpass  },
  { "strset",   S(STRSET_OP),0,     "",     "iS",
    (SUBR) strset_init, NULL, NULL                        },
  { "strget",   S(STRGET_OP),0,     "S",    "i",
    (SUBR) strget_init, NULL, NULL                        },
  {  "S",   S(STRGET_OP),0,     "S",    "i",
     (SUBR) s_opcode, NULL, NULL                           },
  {  "S",   S(STRGET_OP),0,     "S",    "k",
     (SUBR) s_opcode,(SUBR) s_opcode_k, NULL                       },
  {  "strcpy",   S(STRCPY_OP),0,     "S",    "S",
     (SUBR) strcpy_opcode_S, NULL, NULL                     },
  {  "strcpy",   S(STRGET_OP),0,     "S",    "i",
     (SUBR) strcpy_opcode_p, NULL, NULL                     },
  {  "strcpyk",  S(STRCPY_OP),0,     "S",    "S",
     (SUBR) strcpy_opcode_S, (SUBR) strcpy_opcode_S, NULL          },
  {  "strcpyk.k",  S(STRGET_OP),0,     "S",    "k",
     (SUBR) strcpy_opcode_p, (SUBR) strcpy_opcode_p, NULL          },
  {  "strcat",   S(STRCAT_OP),0,     "S",    "SS",
     (SUBR) strcat_opcode, NULL, NULL                      },
  {  "strcatk",  S(STRCAT_OP),0,     "S",    "SS",
     (SUBR) strcat_opcode, (SUBR) strcat_opcode, NULL             },
  {  "strcmp",   S(STRCMP_OP),0,     "i",    "SS",
     (SUBR) strcmp_opcode, NULL, NULL                      },
  {  "strcmpk",  S(STRCMP_OP),0,     "k",    "SS",
     (SUBR) strcmp_opcode, (SUBR) strcmp_opcode, NULL             },
  {  "sprintf",  S(SPRINTF_OP),0,    "S",    "STN",
     (SUBR) sprintf_opcode, NULL, NULL                     },
  {  "sprintfk", S(SPRINTF_OP),WR,    "S",    "SUN",
     (SUBR) sprintf_opcode, (SUBR) sprintf_opcode, NULL           },
  {  "printf_i", S(PRINTF_OP),0,     "",     "SiN", /* SiTN */
     (SUBR) printf_opcode_init, NULL, NULL, NULL, 2                 },
  {  "printfi", S(PRINTF_OP),0,     "",     "SiN", /* SiTN */
     (SUBR) printf_opcode_init, NULL, NULL                 }, /* alias */
  {  "printf",   S(PRINTF_OP),WR,     "",     "SkN", /* SkUN */
     (SUBR) printf_opcode_set, (SUBR) printf_opcode_perf, NULL    },
  {  "puts",     S(PUTS_OP),WR,       "",     "Sko",
     (SUBR) puts_opcode_init, (SUBR) puts_opcode_perf, NULL       },
  {  "strtod",   S(STRSET_OP),0,     "i",    "S",
     (SUBR) strtod_opcode_S, NULL, NULL                      },
  {  "strtod",   S(STRTOD_OP),0,     "i",    "i",
     (SUBR) strtod_opcode_p, NULL, NULL                      },
  {  "strtodk",  S(STRSET_OP),0,     "k",    "S",
     (SUBR) strtod_opcode_S, (SUBR) strtod_opcode_S, NULL          },
  {  "strtol",   S(STRSET_OP),0,     "i",    "S",
     (SUBR) strtol_opcode_S, NULL, NULL                      },
  {  "strtol",   S(STRTOD_OP),0,     "i",    "i",
     (SUBR) strtol_opcode_p, NULL, NULL                      },
  {  "strtolk",  S(STRSET_OP),0,     "k",    "S",
     (SUBR) strtol_opcode_S, (SUBR) strtol_opcode_S, NULL         },
  {  "strsub",   S(STRSUB_OP),0,     "S",    "Soj",
     (SUBR) strsub_opcode, NULL, NULL                      },
  {  "strsubk",  S(STRSUB_OP),0,     "S",    "Skk",
     (SUBR) strsub_opcode, (SUBR) strsub_opcode, NULL             },
  {  "strchar",  S(STRCHAR_OP),0,    "i",    "So",
     (SUBR) strchar_opcode, NULL, NULL                     },
  {  "strchark", S(STRCHAR_OP),0,    "k",    "SO",
     (SUBR) strchar_opcode, (SUBR) strchar_opcode, NULL           },
  {  "strlen",   S(STRLEN_OP),0,     "i",    "S",
     (SUBR) strlen_opcode, NULL, NULL                      },
  {  "strlenk",  S(STRLEN_OP),0,     "k",    "S",
     (SUBR) strlen_opcode, (SUBR) strlen_opcode, NULL             },
  {  "strupper", S(STRUPPER_OP),0,   "S",    "S",
     (SUBR) strupper_opcode, NULL, NULL                    },
  {  "strupperk", S(STRUPPER_OP),0,  "S",    "S",
     (SUBR) strupper_opcode, (SUBR) strupper_opcode, NULL         },
  {  "strlower", S(STRUPPER_OP),0,   "S",    "S",
     (SUBR) strlower_opcode, NULL, NULL                    },
  {  "strlowerk", S(STRUPPER_OP),0,  "S",    "S",
     (SUBR) strlower_opcode, (SUBR) strlower_opcode, NULL         },
  {  "getcfg",   S(GETCFG_OP),0,     "S",    "i",
     (SUBR) getcfg_opcode, NULL, NULL                      },
  {  "strindex", S(STRINDEX_OP),0,   "i",    "SS",
     (SUBR) strindex_opcode, NULL, NULL                    },
  {  "strindexk", S(STRINDEX_OP),0,  "k",    "SS",
     (SUBR) strindex_opcode, (SUBR) strindex_opcode, NULL         },
  {  "strrindex", S(STRINDEX_OP),0,  "i",    "SS",
     (SUBR) strrindex_opcode, NULL, NULL                   },
  {  "strrindexk", S(STRINDEX_OP),0,  "k",   "SS",
     (SUBR) strrindex_opcode, (SUBR) strrindex_opcode, NULL       },
  {  "print_type", S(PRINT_TYPE_OP),0,  "",   ".",
     (SUBR) print_type_opcode, NULL, NULL, NULL, 2       },
  {  "printtype", S(PRINT_TYPE_OP),0,  "",   ".",
     (SUBR) print_type_opcode, NULL, NULL       }, /* alias */
  {  "assert_true.i", S(ASSERT_OP),0, "", "b",
     (SUBR) assert_true_opcode, NULL, NULL },
  {  "assert_true.k", S(ASSERT_OP),0, "", "B",
     (SUBR) assert_true_opcode, NULL, NULL },
  {  "assert_true", S(ASSERT_OP_NUMERIC),0, "", "c",
     (SUBR) assert_true_opcode_numeric, NULL, NULL },
  {  "assert_true", S(ASSERT_OP_NUMERIC),0, "", "i",
     (SUBR) assert_true_opcode_numeric, NULL, NULL },
  {  "assert_true", S(ASSERT_OP_NUMERIC),0, "", "k",
     (SUBR) assert_true_opcode_numeric, NULL, NULL },
  {  "assert_true", S(ASSERT_OP_NUMERIC),0, "", "p",
     (SUBR) assert_true_opcode_numeric, NULL, NULL },
  {  "assert_false.i", S(ASSERT_OP),0, "", "b",
     (SUBR) assert_false_opcode, NULL, NULL },
  {  "assert_false.k", S(ASSERT_OP),0, "", "B",
     (SUBR) assert_false_opcode, NULL, NULL },
  {  "assert_false", S(ASSERT_OP_NUMERIC),0, "", "c",
     (SUBR) assert_false_opcode_numeric, NULL, NULL },
  {  "assert_false", S(ASSERT_OP_NUMERIC),0, "", "i",
     (SUBR) assert_false_opcode_numeric, NULL, NULL },
  {  "assert_false", S(ASSERT_OP_NUMERIC),0, "", "k",
     (SUBR) assert_false_opcode_numeric, NULL, NULL },
  {  "assert_false", S(ASSERT_OP_NUMERIC),0, "", "p",
     (SUBR) assert_false_opcode_numeric, NULL, NULL },
     /* alises */
  {  "asserttrue.i", S(ASSERT_OP),0, "", "b",
     (SUBR) assert_true_opcode, NULL, NULL },
  {  "asserttrue.k", S(ASSERT_OP),0, "", "B",
     (SUBR) assert_true_opcode, NULL, NULL },
  {  "asserttrue", S(ASSERT_OP_NUMERIC),0, "", "c",
     (SUBR) assert_true_opcode_numeric, NULL, NULL },
  {  "asserttrue", S(ASSERT_OP_NUMERIC),0, "", "i",
     (SUBR) assert_true_opcode_numeric, NULL, NULL },
  {  "asserttrue", S(ASSERT_OP_NUMERIC),0, "", "k",
     (SUBR) assert_true_opcode_numeric, NULL, NULL },
  {  "asserttrue", S(ASSERT_OP_NUMERIC),0, "", "p",
     (SUBR) assert_true_opcode_numeric, NULL, NULL },
  {  "assertfalse.i", S(ASSERT_OP),0, "", "b",
     (SUBR) assert_false_opcode, NULL, NULL },
  {  "assertfalse.k", S(ASSERT_OP),0, "", "B",
     (SUBR) assert_false_opcode, NULL, NULL },
  {  "assertfalse", S(ASSERT_OP_NUMERIC),0, "", "c",
     (SUBR) assert_false_opcode_numeric, NULL, NULL },
  {  "assertfalse", S(ASSERT_OP_NUMERIC),0, "", "i",
     (SUBR) assert_false_opcode_numeric, NULL, NULL },
  {  "assertfalse", S(ASSERT_OP_NUMERIC),0, "", "k",
     (SUBR) assert_false_opcode_numeric, NULL, NULL },
  {  "assertfalse", S(ASSERT_OP_NUMERIC),0, "", "p",
     (SUBR) assert_false_opcode_numeric, NULL, NULL },

  {  "assert.i", S(ASSERT_OP),0, "", "b",
     (SUBR) assert_true_opcode, NULL, NULL },
  {  "assert.k", S(ASSERT_OP),0, "", "B",
     (SUBR) assert_true_opcode, NULL, NULL },
  {  "assert", S(ASSERT_OP_NUMERIC),0, "", "c",
     (SUBR) assert_true_opcode_numeric, NULL, NULL },
  {  "assert", S(ASSERT_OP_NUMERIC),0, "", "i",
     (SUBR) assert_true_opcode_numeric, NULL, NULL },
  {  "assert", S(ASSERT_OP_NUMERIC),0, "", "k",
     (SUBR) assert_true_opcode_numeric, NULL, NULL },
  {  "assert", S(ASSERT_OP_NUMERIC),0, "", "p",
     (SUBR) assert_true_opcode_numeric, NULL, NULL },
#ifdef HAVE_CURL
  {  "strfromurl", S(STRCPY_OP), 0,  "S", "S", (SUBR) str_from_url     },
#endif
  {  "changed.S", S(STRCHGD),0,  "k",   "S",
     (SUBR) str_changed, (SUBR) str_changed_k, NULL       },
  {  "changed2.S", S(STRCHGD),0,  "k",   "S",
     (SUBR) str_changed, (SUBR) str_changed_k, NULL       },
  { "loop_lt.i", S(LOOP_OPS), 0, "", "iiil", (SUBR) loop_l_i, NULL, NULL, NULL, 2 },
  { "loop_le.i", S(LOOP_OPS), 0, "", "iiil", (SUBR) loop_le_i, NULL, NULL, NULL, 2 },
  { "loop_gt.i", S(LOOP_OPS), 0, "", "iiil", (SUBR) loop_g_i, NULL, NULL, NULL, 2 },
  { "loop_ge.i", S(LOOP_OPS), 0, "", "iiil", (SUBR) loop_ge_i, NULL, NULL, NULL, 2 },
  { "loop_lt.k", S(LOOP_OPS), 0, "", "kkkl", NULL, (SUBR) loop_l_p, NULL, NULL, 2 },
  { "loop_le.k", S(LOOP_OPS), 0, "", "kkkl", NULL, (SUBR) loop_le_p, NULL, NULL, 2 },
  { "loop_gt.k", S(LOOP_OPS), 0, "", "kkkl", NULL, (SUBR) loop_g_p, NULL, NULL, 2 },
  { "loop_ge.k", S(LOOP_OPS), 0, "", "kkkl", NULL, (SUBR) loop_ge_p, NULL, NULL, 2 },
  /* aliases */
  { "looplt.i", S(LOOP_OPS),0,    "", "iiil", (SUBR) loop_l_i, NULL, NULL   },
  { "loople.i", S(LOOP_OPS),0,    "", "iiil", (SUBR) loop_le_i, NULL, NULL  },
  { "loopgt.i", S(LOOP_OPS),0,    "", "iiil", (SUBR) loop_g_i, NULL, NULL   },
  { "loopge.i", S(LOOP_OPS),0,    "", "iiil", (SUBR) loop_ge_i, NULL, NULL  },
  { "looplt.k", S(LOOP_OPS),0,     "", "kkkl", NULL, (SUBR) loop_l_p, NULL   },
  { "loople.k", S(LOOP_OPS),0,     "", "kkkl", NULL, (SUBR) loop_le_p, NULL  },
  { "loopgt.k", S(LOOP_OPS),0,     "", "kkkl", NULL, (SUBR) loop_g_p, NULL   },
  { "loopge.k", S(LOOP_OPS),0,     "", "kkkl", NULL, (SUBR) loop_ge_p, NULL  },
  { "chnget",      0xFFFF,    _CR                                             },
  { "chnget.i",    S(CHNGET),_CR,                 "i",            "S",
    (SUBR) chnget_opcode_init_i, NULL, NULL               },
  { "chnget.k",    S(CHNGET),_CR,                 "k",            "S",
    (SUBR) chnget_opcode_init_k, (SUBR) notinit_opcode_stub, NULL },
  { "chngeti.i",    S(CHNGET),_CR,                 "i[]",            "S[]",
    (SUBR) chnget_array_opcode_init_i, NULL, NULL               },
  { "chngeta.a",    S(CHNGET),_CR,                 "a[]",            "S[]",
    (SUBR) chnget_array_opcode_init, (SUBR) notinit_opcode_stub, NULL },
  { "chngets.s",    S(CHNGET),_CR,                 "S[]",            "S[]",
    (SUBR) chnget_array_opcode_init, (SUBR) notinit_opcode_stub, NULL },
  { "chngetk.k",    S(CHNGET),_CR,                 "k[]",            "S[]",
    (SUBR) chnget_array_opcode_init, (SUBR) notinit_opcode_stub, NULL },
  { "chnget.a",    S(CHNGET),_CR,                 "a",            "S",
    (SUBR) chnget_opcode_init_a, (SUBR) notinit_opcode_stub },
  { "chnget.S",    S(CHNGET),_CR,                 "S",            "S",
    (SUBR) chnget_opcode_init_S, (SUBR) chnget_opcode_perf_S, NULL},
  { "chnget.[]",    S(CHNGET), _CR,                 "i[]",            "S",
    (SUBR) chnget_opcode_init_ARRAY, NULL, NULL},
  { "chnget.[]",    S(CHNGET), _CR,                 "k[]",            "S",
    (SUBR) chnget_opcode_init_ARRAY, (SUBR) notinit_opcode_stub, NULL},
  { "chnget.[]",    S(CHNGET), _CR,                 "a[]",            "S",
    (SUBR) chnget_opcode_init_ARRAY, (SUBR) notinit_opcode_stub, NULL},
  { "chnget.[]",    S(CHNGET), _CR,                 "S[]",            "S",
    (SUBR) chnget_opcode_init_ARRAY, (SUBR) notinit_opcode_stub, NULL},
  { "chngetks",    S(CHNGET),_CR,                  "S",            "S",
    NULL, (SUBR) chnget_opcode_perf_S, NULL},
  { "chnseti.i",    S(CHNGET),_CW,                "",             "i[]S[]",
    (SUBR) chnset_array_opcode_init_i, NULL, NULL               },
  { "chnsetk.k",    S(CHNGET),_CW,                 "",             "k[]S[]",
    (SUBR) chnset_array_opcode_init, (SUBR) notinit_opcode_stub, NULL },
  { "chnseta.a",    S(CHNGET),_CW,                 "",             "a[]S[]",
    (SUBR) chnset_array_opcode_init, (SUBR) notinit_opcode_stub, NULL },
  { "chnsets.s",    S(CHNGET),_CW,                 "",             "S[]S[]",
    (SUBR) chnset_array_opcode_init, (SUBR) notinit_opcode_stub, NULL },
  { "chnset.i",    S(CHNGET),_CW,                "",             "iS",
    (SUBR) chnset_opcode_init_i, NULL, NULL               },
  { "chnset.k",    S(CHNGET),_CW,                 "",             "kS",
    (SUBR) chnset_opcode_init_k, (SUBR) notinit_opcode_stub, NULL },
  { "chnset.a",    S(CHNGET),_CW,                 "",             "aS",
    (SUBR) chnset_opcode_init_a, (SUBR) notinit_opcode_stub },
  { "chnset.S",    S(CHNGET),_CW,                 "",             "SS",
    (SUBR) chnset_opcode_init_S, (SUBR) chnset_opcode_perf_S, NULL },
  { "chnset.[]",    S(CHNGET), _CR,                 "",            "i[]S",
    (SUBR) chnset_opcode_init_ARRAY, NULL, NULL},
  { "chnset.[]",    S(CHNGET), _CR,                 "",            "k[]S",
    (SUBR) chnset_opcode_init_ARRAY, (SUBR) notinit_opcode_stub, NULL},
  { "chnset.[]",    S(CHNGET), _CR,                 "",            "a[]S",
    (SUBR) chnset_opcode_init_ARRAY, (SUBR) notinit_opcode_stub, NULL},
  { "chnset.[]",    S(CHNGET), _CR,                 "",            "S[]S",
    (SUBR) chnset_opcode_init_ARRAY, (SUBR) notinit_opcode_stub, NULL},
  { "chnsetks",    S(CHNGET),_CW,                  "",             "SS",
    NULL, (SUBR) chnset_opcode_perf_S, NULL },
  { "chnmix",      S(CHNGET),           _CB,       "",             "aS",
    (SUBR) chnmix_opcode_init, (SUBR) notinit_opcode_stub  },
  { "chnclear",    S(CHNCLEAR),        _CW,       "",             "W",
    (SUBR) chnclear_opcode_init, (SUBR) notinit_opcode_stub },
  { "chnclear_array",    S(CHNCLEAR),        _CW,       "",             "W",
    (SUBR) chnclear_opcode_init_ARRAY, (SUBR) notinit_opcode_stub, NULL, NULL, 2 },
  { "chn_k",       S(CHN_OPCODE_K),    _CW,       "",             "SiooooooooN",
    (SUBR) chn_k_opcode_init, NULL, NULL,  NULL, 2                   },
  { "chn_k",       S(CHN_OPCODE_K),    _CW,       "",             "SSooooooooN",
    (SUBR) chn_k_opcode_init_S, NULL, NULL,  NULL, 2 },
  { "chn_a",       S(CHN_OPCODE),      _CW,       "",             "Si",
    (SUBR) chn_a_opcode_init, NULL, NULL,  NULL, 2                   },
  { "chn_S",       S(CHN_OPCODE),      _CW,       "",             "Si",
    (SUBR) chn_S_opcode_init, NULL, NULL,  NULL, 2                   },
  { "chn_array",    S(CHN_OPCODE_ARRAY),        _CW,       "",             "SiSi[]",
    (SUBR) chn_opcode_init_ARRAY, NULL, NULL,  NULL, 2  },
  /* Aliases */
  { "chncleararray",    S(CHNCLEAR),        _CW,       "",             "W",
    (SUBR) chnclear_opcode_init_ARRAY, (SUBR) notinit_opcode_stub },
  { "chnk",       S(CHN_OPCODE_K),    _CW,       "",             "SiooooooooN",
    (SUBR) chn_k_opcode_init, NULL, NULL                  },
  { "chnk",       S(CHN_OPCODE_K),    _CW,       "",             "SSooooooooN",
    (SUBR) chn_k_opcode_init_S, NULL, NULL},
  { "chna",       S(CHN_OPCODE),      _CW,       "",             "Si",
    (SUBR) chn_a_opcode_init, NULL, NULL                  },
  { "chnS",       S(CHN_OPCODE),      _CW,       "",             "Si",
    (SUBR) chn_S_opcode_init, NULL, NULL                  },
  { "chnarray",    S(CHN_OPCODE_ARRAY),        _CW,       "",             "SiSi[]",
    (SUBR) chn_opcode_init_ARRAY, NULL, NULL },
  { "chnexport.i", S(CHNEXPORT_OPCODE),0,       "i",            "Sioooo",
    (SUBR) chnexport_opcode_init, NULL, NULL              },
  { "chnexport.k", S(CHNEXPORT_OPCODE),0,       "k",            "Sioooo",
    (SUBR) chnexport_opcode_init, NULL, NULL              },
  { "chnexport.a", S(CHNEXPORT_OPCODE),0,       "a",            "Si",
    (SUBR) chnexport_opcode_init, NULL, NULL              },
  { "chnexport.S", S(CHNEXPORT_OPCODE),0,       "S",            "Si",
    (SUBR) chnexport_opcode_init, NULL, NULL              },
  { "chnparams",   S(CHNPARAMS_OPCODE),_CR,       "iiiiii",       "S",
    (SUBR) chnparams_opcode_init, NULL, NULL              },
  { "chano.k",     S(ASSIGN),_CW,                  "",             "kk",
    NULL, (SUBR) chano_opcode_perf_k, NULL                },
  { "chano.a",     S(ASSIGN),_CW,                  "",             "ak",
    NULL, (SUBR) chano_opcode_perf_a                },
  { "pvsout",     S(FCHAN),0,                 "",             "fk",
    (SUBR) pvsout_init, (SUBR) pvsout_perf, NULL                        },
  { "pvsout",     S(FCHAN),0,                 "",             "fS",
    (SUBR) pvsout_init, (SUBR) pvsout_perf, NULL                        },
  { "chnset.f",     S(FCHAN),0,                 "",             "fk",
    (SUBR) pvsout_init, (SUBR) pvsout_perf, NULL                        },
  { "chani.k",     S(ASSIGN),_CR,                  "k",            "k",
    NULL, (SUBR) chani_opcode_perf_k, NULL                },
  { "chani.a",     S(ASSIGN),_CR,                  "a",            "k",
    NULL, (SUBR) chani_opcode_perf_a                },
  { "pvsin",     S(FCHAN),0,                 "f",            "kooopo",
    (SUBR)  pvsin_init, (SUBR) pvsin_perf, NULL                  },
  { "pvsin",     S(FCHAN),0,                 "f",            "Sooopo",
    (SUBR)  pvsin_init, (SUBR) pvsin_perf, NULL                  },
  { "chnget.f",     S(FCHAN),0,                 "f",            "Sooopo",
    (SUBR)  pvsin_init, (SUBR) pvsin_perf, NULL                  },
  { "sense",       S(KSENSE),0,                  "kz",           "",
    NULL, (SUBR) sensekey_perf, NULL                      },
  { "sensekey",    S(KSENSE),0,                  "kz",           "",
    NULL, (SUBR) sensekey_perf, NULL                      },
  { "remove",      S(DELETEIN),0,               "",             "T",
    (SUBR) delete_instr, NULL, NULL                       },
  //  { "##globallock",   S(GLOBAL_LOCK_UNLOCK),0,  "", "i",
  //    globallock,   globallock,   NULL},
  //  { "##globalunlock", S(GLOBAL_LOCK_UNLOCK),0,  "", "i",
  //    globalunlock, globalunlock, NULL},
  { "##error",S(ERRFN),0,           "i",     "i",   error_fn, NULL,    NULL    },
  { "exprandi.i",S(PRANDI),0,       "i",    "kxx",  iexprndi, NULL,    NULL    },
  { "exprandi.k",S(PRANDI),0,       "k",    "kxx",  exprndiset, kexprndi, NULL },
  { "exprandi.a",S(PRANDI),0,        "a",    "kxx",  exprndiset, aexprndi },
  { "cauchyi.i", S(PRANDI),0,       "i",    "kxx",  icauchyi, NULL,    NULL    },
  { "cauchyi.k", S(PRANDI),0,       "k",    "kxx",  cauchyiset, kcauchyi, NULL },
  { "cauchyi.a", S(PRANDI),0,        "a",    "kxx",  cauchyiset, acauchyi },
  { "gaussi.i", S(PRANDI),0,       "i",    "kxx",  igaussi, NULL,    NULL    },
  { "gaussi.k", S(PRANDI),0,       "k",    "kxx",  gaussiset, kgaussi, NULL },
  { "gaussi.a", S(PRANDI),0,        "a",    "kxx",  gaussiset, agaussi },
  { "ftresizei", S(RESIZE), TB,  "i", "ii", (SUBR) resize_table, NULL, NULL },
  { "ftresize",  S(RESIZE), TB,   "k", "kk", NULL, (SUBR) resize_table, NULL },
  { "compileorc",  S(COMPILE), 0,  "i", "S",  (SUBR) compile_orc_i, NULL, NULL },
  { "compilecsd",  S(COMPILE), 0,  "i", "S",  (SUBR) compile_csd_i, NULL, NULL },
  { "compilestr",  S(COMPILE), 0,  "i", "S",  (SUBR) compile_str_i, NULL, NULL },
  { "createinstr",  S(CINSTR), 0,  ":InstrDef;", "S",  (SUBR) compile_instr, NULL, NULL },
  { "create",  S(CINSTR), 0,  ":InstrDef;", "S",  (SUBR) compile_instr, NULL, NULL },
  { "delete",  S(DELETE_INSTRDEF), 0, "", ":InstrDef;",  NULL, NULL, (SUBR) delete_instrdef_opcode},
  { "remove",  S(DELETE_INSTRDEF), 0, "", ":InstrDef;",  (SUBR) delete_instrdef_opcode, NULL },
  { "evalstr",  S(COMPILE), 0,  "i", "S",  (SUBR) eval_str_i, NULL, NULL },
  { "evalstr",  S(COMPILE), 0,   "k", "Sk",  NULL, (SUBR) eval_str_k, NULL },
  { "readscore",  S(COMPILE), 0,  "i", "S",  (SUBR) read_score_i, NULL, NULL },
  { "return",  S(RETVAL), 0,  "", "i",  (SUBR) retval_i, NULL, NULL },
  { "pcount", S(PFIELD),  0, "i", "",       (SUBR)pcount,    NULL, NULL },
  { "pindex", S(PFIELD),  0,  "i", "i",      (SUBR)pvalue,    NULL, NULL },
  { "pindex.S", S(PFIELDSTR), 0, "S", "i",  (SUBR)pvaluestr, NULL, NULL },
  { "passign", S(PINIT),  0,  "IIIIIIIIIIIIIIIIIIIIIIII", "po",
    (SUBR)pinit,     NULL, NULL },
  { "passign.i", S(PAINIT), 0,  "i[]", "po",  (SUBR)painit,    NULL, NULL },
  { "passign.k", S(PAINIT), 0,  "k[]", "po",  (SUBR)painit,    NULL, NULL },
  /* ----------------------------------------------------------------------- */
  { "=.generic", S(ASSIGN), 0, ".", ".", (SUBR)copy_var_generic_init,
    (SUBR)copy_var_generic, NULL},
  { "monitor",  sizeof(MONITOR_OPCODE), IB,   "mmmmmmmmmmmmmmmmmmmmmmmm", "",
    (SUBR) monitor_opcode_init, (SUBR) notinit_opcode_stub,  NULL },
  { "monitor.A", sizeof(MONITOR_A), IB, "a[]", "",
    (SUBR)monitora_init, (SUBR)monitora_perf},
  { "outrg", S(OUTRANGE), IR, "", "ky",
    (SUBR)outRange_i, (SUBR)outRange},
  { "nchnls_hw", S(ASSIGN), 0, "ii", "", (SUBR)hw_channels, NULL, NULL, NULL, 2 },
  { "nchnlshw", S(ASSIGN), 0, "ii", "", (SUBR)hw_channels}, /* alias */
  { "midic7.i",S(MIDICTL2),0,   "i", "iiio", (SUBR)imidic7, NULL, NULL  },
  { "midic7.k", S(MIDICTL2),0, "k", "ikko", (SUBR)midic7set, (SUBR)midic7, NULL },
  { "midic14.i", S(MIDICTL3), 0, "i", "iiiio",(SUBR)imidic14, NULL, NULL },
  { "midic14.k", S(MIDICTL3), 0,"k", "iikko",(SUBR)midic14set, (SUBR)midic14,NULL},
  { "midic21.i", S(MIDICTL4),0, "i", "iiiiio",(SUBR)imidic21, NULL, NULL },
  { "midic21.k", S(MIDICTL4), 0,"k", "iiikko",(SUBR)midic21set,(SUBR)midic21,NULL},
  { "ctrl7.i", S(CTRL7), 0,     "i", "iiiio", (SUBR)ictrl7,   NULL, NULL },
  { "ctrl7.k", S(CTRL7),  0,   "k", "iikko", (SUBR)ctrl7set, (SUBR)ctrl7, NULL },
  { "ctrl14.i", S(CTRL14),0,    "i", "iiiiio",(SUBR)ictrl14, NULL, NULL },
  { "ctrl14.k", S(CTRL14), 0,  "k", "iiikko",(SUBR)ctrl14set, (SUBR)ctrl14, NULL },
  { "ctrl21.i", S(CTRL21),0,    "i", "iiiiiio", (SUBR)ictrl21, NULL, NULL },
  { "ctrl21.k", S(CTRL21), 0,  "k", "iiiikko", (SUBR)ctrl21set,(SUBR)ctrl21,NULL},
  { "initc7", S(INITC7), 0,      "",  "iii",  (SUBR)initc7,     NULL,     NULL },
  { "initc14", S(INITC14), 0,    "",  "iiii", (SUBR)initc14,    NULL,     NULL },
  { "initc21", S(INITC21), 0,    "",  "iiiii",(SUBR)initc21,    NULL,     NULL },
  { "midipgm", S(MIDIPGM_OP), 0,  "i", "o",   (SUBR)midipgm_opcode, NULL, NULL },
  { "slider8.k", S(SLIDER8), 0, "kkkkkkkk",  "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiii", (SUBR)slider_i8, (SUBR)slider8, NULL },
  { "slider8f", S(SLIDER8f), 0, "kkkkkkkk","iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiii",
    (SUBR)slider_i8f, (SUBR)slider8f, NULL },
  { "slider8.i", S(SLIDER8), 0,  "iiiiiiii", "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii",
    (SUBR)islider8, NULL, NULL },
  { "slider16.k", S(SLIDER16), 0, "kkkkkkkkkkkkkkkk",
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiii",
    (SUBR)slider_i16, (SUBR)slider16, NULL },
  { "slider16f", S(SLIDER16f), 0, "kkkkkkkkkkkkkkkk",
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiii",
    (SUBR)slider_i16f, (SUBR)slider16f, NULL },
  { "slider16.i", S(SLIDER16), 0,  "iiiiiiiiiiiiiiii",
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiii",
    (SUBR)islider16, NULL, NULL       },
  { "slider32.k", S(SLIDER32),  0, "kkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkk",
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiii",
    (SUBR)slider_i32, (SUBR)slider32, NULL  },
  { "slider32f", S(SLIDER32f), 0, "kkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkk",
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiii",
    (SUBR)slider_i32f, (SUBR)slider32f, NULL },
  { "slider32.i", S(SLIDER32), 0,  "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii",
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiii",
    (SUBR)islider32, NULL, NULL  },
  { "slider64.k", S(SLIDER64), 0, "kkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkk"
    "kkkkkkkkkkkkkkkkkkk",
    "iiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiii",
    (SUBR)slider_i64, (SUBR)slider64, NULL  },
  { "slider64f", S(SLIDER64f), 0, "kkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkk"
    "kkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkk",
    "iiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiii",
    (SUBR)slider_i64f, (SUBR)slider64f, NULL },
  { "slider64.i", S(SLIDER64), 0,  "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii",
    "iiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiii",
    (SUBR)islider64, NULL, NULL  },
  { "s16b14.k", S(SLIDER16BIT14), 0, "kkkkkkkkkkkkkkkk",
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii",
    (SUBR)slider_i16bit14, (SUBR)slider16bit14, NULL},
  { "s32b14.k", S(SLIDER32BIT14), 0, "kkkkkkkkkkkkkkkkkkkkkkkkkkkkkkkk",
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii",
    (SUBR)slider_i32bit14, (SUBR)slider32bit14, NULL},
  { "s16b14.i", S(ISLIDER16BIT14), 0,  "iiiiiiiiiiiiiiii",
    "iiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiii",
    (SUBR)islider16bit14, NULL, NULL  },
  { "s32b14.i", S(ISLIDER32BIT14), 0,  "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii",
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii",
    (SUBR)islider32bit14, NULL, NULL  },
  { "slider8table", S(SLIDER8t), 0, "k",  "iii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii",
    (SUBR)sliderTable_i8, (SUBR)sliderTable8, (SUBR)NULL },
  { "slider16table", S(SLIDER16t), 0, "k", "iii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii",
    (SUBR)sliderTable_i16, (SUBR)sliderTable16, (SUBR)NULL },
  { "slider32table", S(SLIDER32t), 0, "k", "iii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii",
    (SUBR)sliderTable_i32, (SUBR)sliderTable32, (SUBR)NULL },
  { "slider64table", S(SLIDER64t), 0, "k", "iii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii",
    (SUBR)sliderTable_i64, (SUBR)sliderTable64, (SUBR)NULL },
  { "slider8tablef", S(SLIDER8tf), 0, "k", "iii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii",
    (SUBR)sliderTable_i8f, (SUBR)sliderTable8f, (SUBR)NULL },
  { "slider16tablef",S(SLIDER16tf), 0, "k", "iii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii",
    (SUBR)sliderTable_i16f, (SUBR)sliderTable16f, (SUBR)NULL },
  { "slider32tablef",S(SLIDER32tf), 0, "k", "iii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii",
    (SUBR)sliderTable_i32f, (SUBR)sliderTable32f, (SUBR)NULL },
  { "slider64tablef",S(SLIDER64tf), 0, "k", "iii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii",
    (SUBR)sliderTable_i64f, (SUBR)sliderTable64f, (SUBR)NULL },
  { "sliderKawai", S(SLIDERKAWAI),  0, "kkkkkkkkkkkkkkkk",
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii",
    (SUBR)sliderKawai_i, (SUBR)sliderKawai, NULL, NULL, 2 },
  { "sliderkawai", S(SLIDERKAWAI),  0, "kkkkkkkkkkkkkkkk",
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii"
    "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiiii",
    (SUBR)sliderKawai_i, (SUBR)sliderKawai, NULL }, /* alias */
  { "ctrl7.a", S(CTRL7a),  0, "a",    "iikkoo",
    (SUBR) ctrl7a_set,    (SUBR) ctrl7a },
  { "cpuprc", S(CPU_PERC),0,      "",     "Si",   (SUBR)cpuperc_S, NULL, NULL   },
  { "maxalloc", S(CPU_MAXALLOC),0,"",     "Sio",  (SUBR)maxalloc_S, NULL, NULL  },
  { "cpuprc", S(CPU_PERC),0,      "",     "ii",   (SUBR)cpuperc, NULL, NULL   },
  { "maxalloc", S(CPU_MAXALLOC),0,"",     "iio",  (SUBR)maxalloc, NULL, NULL  },
  { "active.iS", S(INSTCNT),0,    "i",    "Soo",   (SUBR)instcount_S, NULL, NULL },
  { "active.kS", S(INSTCNT),0,    "k",    "Soo",   NULL, (SUBR)instcount_S, NULL },
  { "active.i", S(INSTCNT),0,     "i",    "ioo",   (SUBR)instcount, NULL, NULL },
  { "active.k", S(INSTCNT),0,     "k",    "koo",   NULL, (SUBR)instcount, NULL },
  { "p.i", S(PFUN),        0,     "i",    "i",     (SUBR)pfun, NULL, NULL     },
  { "p.k", S(PFUNK),       0,     "k",    "k",
    (SUBR)pfunk_init, (SUBR)pfunk, NULL },
  { "mute", S(MUTE), 0,           "",      "So",   (SUBR)mute_inst_S             },
  { "mute.i", S(MUTE), 0,         "",      "io",   (SUBR)mute_inst             },
  { "midiarp",   S(MIDIARP),0,      "kk", "kO",
    midiarp_set, midiarp, NULL },
  {"lpcfilter", S(LPCFIL), 0,  "a", "akkiiio",
   (SUBR) lpfil_init, (SUBR) lpfil_perf},
  {"lpcfilter", S(LPCFIL2), 0,  "a", "aakkiio",
   (SUBR) lpfil2_init, (SUBR) lpfil2_perf},
  {"allpole", S(LPCFIL3), 0,  "a", "ak[]",
   (SUBR) lpfil3_init, (SUBR) lpfil3_perf},
  {"lpcanal", S(LPREDA), 0,  "k[]kkk", "kkiiio",
   (SUBR) lpred_alloc, (SUBR) lpred_run},
  {"lpcanal", S(LPREDA2), 0,  "k[]kkk", "akkiio",
   (SUBR) lpred_alloc2, (SUBR) lpred_run2},
  {"lpcanal", S(LPREDA), 0,  "i[]iii", "iiii",
   (SUBR) lpred_i, NULL},
  {"pvslpc", S(LPCPVS), 0,  "f", "aiiio",
   (SUBR) lpcpvs_init, (SUBR) lpcpvs},
  {"pvscfs", S(PVSCFS), 0,  "k[]kk", "fip",
   (SUBR) pvscoefs_init, (SUBR) pvscoefs},
  {"apoleparams", S(CF2P), 0,  "k[]", "k[]",
   (SUBR) coef2parm_init, (SUBR) coef2parm},
  {"resonbnk", S(RESONB), 0,  "a", "ak[]kkipoo",
   (SUBR) resonbnk_init, (SUBR) resonbnk},
  { "inrg", S(INRANGE), WI, "", "ky", (SUBR)inRange_i, (SUBR)inRange },
  { "OSClisten", S(ROSC), 0, "kNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN",
    "SS", NULL, (SUBR) readOSC_perf, NULL, NULL, 2},
  { "OSClisten", S(ROSCA), 0, "kk[]",
    "SS", (SUBR) readOSCarray_init, (SUBR) readOSCarray_perf, NULL, NULL, 2},
  { "osclisten", S(ROSC), 0, "kNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN",
    "SS", NULL, (SUBR) readOSC_perf},  /* Alias */
  { "osclisten", S(ROSCA), 0, "kk[]",
    "SS", (SUBR) readOSCarray_init, (SUBR) readOSCarray_perf},  /* Alias */
  { "midifileopen", S(MFILE), 0, "i", "So", midi_file_opcode},
  { "midifileplay", S(MFILE), 0, "", "o", midi_file_play},
  { "midifilepause", S(MFILE), 0, "", "o", midi_file_pause},
  { "midifilemute", S(MFILE), 0, "", "o", midi_file_mute},
  { "midifilerewind", S(MFILE), 0, "", "o", midi_file_rewind},
  { "midifilelen", S(MFILE), 0, "i", "o", midi_file_len},
  { "midifiletempo", S(MIDITEMPO), 0, "", "io", midi_set_tempo},
  { "midifiletempo", S(MIDITEMPO), 0, "", "ko", NULL, midi_set_tempo},
  { "midifilepos", S(MIDITEMPO), 0, "", "io", midi_set_pos},
  { "midifilepos", S(MIDITEMPO), 0, "", "ko", NULL, midi_set_pos},
  { "midifilepos", S(MIDITEMPO), 0, "i", "o", midi_get_pos},
  { "midifilepos", S(MIDITEMPO), 0, "k", "o", NULL, midi_get_pos},
  { "midifilevents", S(MIDITEMPO), 0, "i", "o", midi_file_get_number_events},
  { "midifilein", S(MIDIFEVT), 0, "iiiii", "io",midi_file_get_event},
  { "midifilein", S(MIDIFEVT), 0, "kkkkk", "ko", NULL, midi_file_get_event},
  { "eventtype", S(MIDIKMB), 0, "i", "", event_type},
  { "typeof", S(ASSIGN), 0, "S", ".", type_of },
  { "typecheck", S(RELAT), 0, "b", "..", check_type },
  /* terminate list */
  {  NULL, 0, 0, 0, NULL, NULL, NULL, NULL, NULL       }
};
