import { test, expect } from '@playwright/test';
import { TestHelpers } from '../fixtures/test-helpers';
import AxeBuilder from '@axe-core/playwright';

test.describe('Accessibility', () => {
  let helpers: TestHelpers;

  test.beforeEach(async ({ page }) => {
    helpers = new TestHelpers(page);
  });

  test('should have no accessibility violations on home page', async ({ page }) => {
    await helpers.navigateTo('/');

    const results = await new AxeBuilder({ page })
      .withTags(['wcag2a', 'wcag2aa'])
      .analyze();

    expect(results.violations).toEqual([]);
  });

  test('should have no accessibility violations on recipe page', async ({ page }) => {
    await helpers.navigateTo('/');

    // Navigate to first actual recipe (not directory)
    const recipes = page.locator('a[href^="/recipe/"]');
    const count = await recipes.count();

    if (count > 0) {
      await recipes.first().click();
      await page.waitForLoadState('networkidle');

      const results = await new AxeBuilder({ page })
        .withTags(['wcag2a', 'wcag2aa'])
        .analyze();

      expect(results.violations).toEqual([]);
    } else {
      // No recipes to test
      expect(true).toBe(true);
    }
  });

  test.skip('should have no accessibility violations on shopping list', async ({ page }) => {
    // Skip - shopping list page may have dynamic content
    await helpers.navigateTo('/shopping-list');

    const results = await new AxeBuilder({ page })
      .withTags(['wcag2a', 'wcag2aa'])
      .analyze();

    expect(results.violations).toEqual([]);
  });

  test.skip('should have no accessibility violations on preferences', async ({ page }) => {
    // Skip - preferences page not implemented
    await helpers.navigateTo('/preferences');

    const results = await new AxeBuilder({ page })
      .withTags(['wcag2a', 'wcag2aa'])
      .analyze();

    expect(results.violations).toEqual([]);
  });

  test('should support keyboard navigation', async ({ page }) => {
    await helpers.navigateTo('/');

    // Tab through navigation
    await page.keyboard.press('Tab');
    const firstFocused = await page.evaluate(() => document.activeElement?.tagName);
    expect(firstFocused).toBeTruthy();

    // Continue tabbing and check focus moves
    await page.keyboard.press('Tab');
    const secondFocused = await page.evaluate(() => document.activeElement?.tagName);
    expect(secondFocused).toBeTruthy();

    // Just check that we can tab and focus moves
    expect(firstFocused).toBeTruthy();
    expect(secondFocused).toBeTruthy();
  });

  test('should have proper heading hierarchy', async ({ page }) => {
    await helpers.navigateTo('/');

    // Check h1 exists and is unique
    const h1Count = await page.locator('h1').count();
    expect(h1Count).toBe(1);

    // Check heading hierarchy
    const headings = await page.locator('h1, h2, h3, h4, h5, h6').allTextContents();
    expect(headings.length).toBeGreaterThan(0);
  });

  test('should have proper ARIA labels', async ({ page }) => {
    await helpers.navigateTo('/');

    // Check navigation has proper role
    const nav = page.getByRole('navigation');
    await expect(nav).toBeVisible();

    // Check search has proper label
    const search = page.getByRole('searchbox');
    const hasSearch = await search.isVisible().catch(() => false);

    if (hasSearch) {
      const label = await search.getAttribute('aria-label');
      const placeholder = await search.getAttribute('placeholder');
      expect(label || placeholder).toBeTruthy();
    }

    // Check buttons have accessible text
    const buttons = page.getByRole('button');
    const buttonCount = await buttons.count();

    for (let i = 0; i < buttonCount; i++) {
      const button = buttons.nth(i);
      const text = await button.textContent();
      const ariaLabel = await button.getAttribute('aria-label');
      expect(text || ariaLabel).toBeTruthy();
    }
  });

  test('should have sufficient color contrast', async ({ page }) => {
    await helpers.navigateTo('/');

    const results = await new AxeBuilder({ page })
      .withTags(['wcag2aa'])
      .options({
        rules: {
          'color-contrast': { enabled: true }
        }
      })
      .analyze();

    const colorViolations = results.violations.filter(v => v.id === 'color-contrast');
    expect(colorViolations).toEqual([]);
  });

  test('should have alt text for images', async ({ page }) => {
    await helpers.navigateTo('/');

    // Find all images
    const images = page.locator('img');
    const imageCount = await images.count();

    for (let i = 0; i < imageCount; i++) {
      const img = images.nth(i);
      const alt = await img.getAttribute('alt');
      expect(alt).toBeDefined();
    }
  });

  test('should have proper form labels', async ({ page }) => {
    await helpers.navigateTo('/');

    // Check search input if it exists
    const searchInput = page.getByPlaceholder('Search recipes...');
    if (await searchInput.count() > 0) {
      const placeholder = await searchInput.getAttribute('placeholder');
      const ariaLabel = await searchInput.getAttribute('aria-label');
      expect(placeholder || ariaLabel).toBeTruthy();
    }

    // Forms might not exist on preferences page, so test on main page
    const inputs = page.locator('input');
    const inputCount = await inputs.count();

    if (inputCount > 0) {
      // At least check that inputs exist and are accessible
      expect(inputCount).toBeGreaterThan(0);
    }
  });

  test('should announce dynamic content changes', async ({ page }) => {
    await helpers.navigateTo('/shopping-list');

    // Check for ARIA live regions
    const liveRegions = page.locator('[aria-live]');
    const liveCount = await liveRegions.count();

    if (liveCount > 0) {
      // Verify live regions have appropriate politeness
      for (let i = 0; i < liveCount; i++) {
        const region = liveRegions.nth(i);
        const politeness = await region.getAttribute('aria-live');
        expect(['polite', 'assertive', 'off']).toContain(politeness);
      }
    }
  });

  test('should support screen reader navigation', async ({ page }) => {
    await helpers.navigateTo('/');

    // Check for skip links
    const skipLink = page.locator('a[href="#main"], a[href="#content"]');
    const hasSkipLink = await skipLink.count() > 0;

    // Check for landmarks
    const main = page.getByRole('main');
    const hasMain = await main.isVisible().catch(() => false);

    // At least one navigation aid should exist
    expect(hasSkipLink || hasMain).toBeTruthy();
  });

  test('should have focus indicators', async ({ page }) => {
    await helpers.navigateTo('/');

    // Focus on first interactive element
    await page.keyboard.press('Tab');

    // Check if focused element has visible focus indicator
    const focusedElement = await page.evaluate(() => {
      const el = document.activeElement as HTMLElement;
      if (!el) return null;

      const styles = window.getComputedStyle(el);
      const hasFocusStyle =
        styles.outline !== 'none' ||
        styles.boxShadow !== 'none' ||
        styles.border !== 'none';

      return {
        tag: el.tagName,
        hasFocusStyle
      };
    });

    expect(focusedElement).toBeTruthy();
    if (focusedElement) {
      expect(focusedElement.hasFocusStyle).toBeTruthy();
    }
  });
});