/******************************************************************************
* choria
* Copyright (c) 2025 Alan Witkowski
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*******************************************************************************/
#include <hud/trader_screen.h>
#include <ae/assets.h>
#include <ae/clientnetwork.h>
#include <ae/font.h>
#include <ae/graphics.h>
#include <ae/ui.h>
#include <ae/util.h>
#include <hud/hud.h>
#include <objects/components/character.h>
#include <objects/item.h>
#include <objects/object.h>
#include <states/play.h>
#include <packet.h>
#include <stats.h>
#include <SDL_keycode.h>
#include <sstream>

// Initialize
void _TraderScreen::Init() {
	if(!HUD.Player)
		return;

	// Activate element
	Element->SetActive(true);

	// Initialize data
	const _Trader *Trader = HUD.Player->Character->Trader;
	RequiredAmounts.resize(Trader->Items.size());
	RequiredTotals.resize(Trader->Items.size());
	for(size_t i = 0; i < Trader->Items.size(); i++)
		RequiredTotals[i] = HUD.Player->Inventory->CountItem(Trader->Items[i].Item);
	RewardTotal = HUD.Player->Inventory->CountItem(Trader->RewardItem);
	LastAmount = 0;

	// Check if item is unlocked on buy
	CanUse = true;
	if(Trader->RewardItem && Trader->RewardItem->UnlockOnBuy()) {
		_ActionResult ActionResult;
		ActionResult.Source.Object = HUD.Player;
		ActionResult.ActionUsed.Item = Trader->RewardItem;
		ActionResult.Scope = ScopeType::WORLD;
		UsesLeft = Trader->RewardItem->CanUse(HUD.Scripting, ActionResult, false);
		CanUse = UsesLeft > 0;
	}

	// Set dialogue
	ae::_Element *DialogueElement = ae::Assets.Elements["element_trader_dialogue"];
	if(!Trader->Dialogue.empty()) {
		ae::_Element *DialogueLabel = ae::Assets.Elements["label_trader_dialogue"];
		DialogueLabel->Text = Trader->Dialogue;
		DialogueLabel->SetWrap(DialogueLabel->Size.x);
		DialogueElement->SetActive(true);
	}
	else
		DialogueElement->SetActive(false);
}

// Close screen
bool _TraderScreen::Close(bool SendNotify, bool Delay) {
	bool WasOpen = Element->Active;
	Element->SetActive(false);
	HUD.Cursor.Reset();

	if(HUD.Player)
		HUD.Player->Character->Trader = nullptr;

	return WasOpen;
}

// Trade accept
void _TraderScreen::TradeItem() {
	ae::_Buffer Packet;
	Packet.Write<PacketType>(PacketType::TRADER_ACCEPT);
	Packet.WriteBit(ae::Input.ModKeyDown(KMOD_SHIFT));
	PlayState.Network->SendPacket(Packet);
}

// Update
void _TraderScreen::Update(double FrameTime) {

	// Check for trader
	const _Trader *Trader = HUD.Player->Character->Trader;
	if(!Trader)
		return;

	// Get amount modifier
	int Amount = ae::Input.ModKeyDown(KMOD_SHIFT) ? GAME_INCREMENT_MODIFIER : 1;

	// Disable amount for beggar buff
	if(!Trader->RewardItem)
		Amount = 1;

	// Check for change in amount
	if(LastAmount != Amount) {

		// Get components
		_Inventory *Inventory = HUD.Player->Inventory;

		// Check for required items
		_Slot RewardSlot;
		int MaxAmount = Inventory->GetTraderAmount(Trader, RequiredAmounts);

		// Set max amount
		Amount = std::min(Amount, MaxAmount);

		// Set max amount based on uses left
		if(Trader->RewardItem && Trader->RewardItem->UnlockOnBuy())
			Amount = std::min(Amount, UsesLeft);

		// Check requirements
		bool CanAccept = Amount && (!Trader->RewardItem || Inventory->GetRewardSlot(Trader, Amount, RewardSlot));

		// Set accept button state
		ae::_Element *AcceptButton = ae::Assets.Elements["button_trader_accept"];
		AcceptButton->SetEnabled(CanAccept && CanUse);
		AcceptButton->Children[0]->Text = "Accept " + std::to_string(std::max(1, Amount)) + "x";

		// Update last amount
		LastAmount = Amount;
	}
}

// Render
void _TraderScreen::Render(double BlendFactor) {
	const _Trader *Trader = HUD.Player->Character->Trader;

	// Check objects
	if(!HUD.Player || !Trader) {
		Element->Active = false;
		return;
	}

	// Render ui elements
	Element->Render();

	// Get constants
	glm::vec2 CountOffset = glm::vec2(28, 26) * ae::_Element::GetUIScale();
	glm::vec2 TotalOffset = glm::vec2(-30, -15) * ae::_Element::GetUIScale();
	glm::vec4 DimColor = glm::vec4(1.0f, 1.0f, 1.0f, 0.5f);

	// Draw trader items
	int TradeAmount = std::max(1, LastAmount);
	for(size_t i = 0; i < Trader->Items.size(); i++) {

		// Get button position
		std::stringstream Buffer;
		Buffer << "button_trader_bag_" << i;
		ae::_Element *Button = ae::Assets.Elements[Buffer.str()];
		Buffer.str("");
		glm::vec2 DrawPosition = (Button->Bounds.Start + Button->Bounds.End) / 2.0f;

		// Draw item
		const _Item *Item = Trader->Items[i].Item;
		ae::Graphics.SetProgram(ae::Assets.Programs["ortho_pos_uv"]);
		ae::Graphics.DrawScaledImage(DrawPosition, Item->Texture, UI_SLOT_SIZE);

		// Get color
		glm::vec4 Color = RequiredAmounts[i] ? glm::vec4(1.0f) : ae::Assets.Colors["red"];

		// Draw total count
		if(ae::Input.ModKeyDown(KMOD_ALT)) {
			ae::FormatSI(Buffer, (int64_t)RequiredTotals[i]);
			ae::Assets.Fonts["hud_tiny"]->DrawText(Buffer.str(), glm::ivec2(DrawPosition + TotalOffset), ae::LEFT_BASELINE, DimColor);
			Buffer.str("");
		}

		// Draw required count
		ae::Assets.Fonts["hud_tiny"]->DrawText(std::to_string(Trader->Items[i].Count * TradeAmount), glm::ivec2(DrawPosition + CountOffset), ae::RIGHT_BASELINE, Color);
	}

	// Get reward button
	ae::_Element *RewardButton = ae::Assets.Elements["button_trader_bag_reward"];
	glm::vec2 DrawPosition = (RewardButton->Bounds.Start + RewardButton->Bounds.End) / 2.0f;

	// Draw item
	if(Trader->RewardItem) {
		ae::Graphics.SetProgram(ae::Assets.Programs["ortho_pos_uv"]);
		ae::Graphics.DrawScaledImage(DrawPosition,Trader->RewardItem->Texture, UI_SLOT_SIZE);

		// Draw total count
		if(ae::Input.ModKeyDown(KMOD_ALT)) {
			std::stringstream Buffer;
			ae::FormatSI(Buffer, (int64_t)RewardTotal);
			ae::Assets.Fonts["hud_tiny"]->DrawText(Buffer.str(), glm::ivec2(DrawPosition + TotalOffset), ae::LEFT_BASELINE, DimColor);
		}

		// Draw reward count
		ae::Assets.Fonts["hud_tiny"]->DrawText(std::to_string(Trader->Count * TradeAmount), DrawPosition + CountOffset, ae::RIGHT_BASELINE);
	}
}
