//===-- Common constants for math functions ---------------------*- C++ -*-===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_LIBC_SRC_MATH_GENERIC_COMMON_CONSTANTS_H
#define LLVM_LIBC_SRC_MATH_GENERIC_COMMON_CONSTANTS_H

#include "src/__support/FPUtil/triple_double.h"
#include "src/__support/macros/config.h"
#include "src/__support/math/acosh_float_constants.h"
#include "src/__support/math/exp_constants.h"
#include "src/__support/number_pair.h"

namespace LIBC_NAMESPACE_DECL {

// Lookup table for (1/f) where f = 1 + n*2^(-7), n = 0..127,
// computed and stored as float precision constants.
extern const float ONE_OVER_F_FLOAT[128];

// Lookup table for log(f) = log(1 + n*2^(-7)) where n = 0..127,
// computed and stored as float precision constants.
extern const float LOG_F_FLOAT[128];

// Lookup table for range reduction constants r for logarithms.
extern const float R[128];

// Lookup table for range reduction constants r for logarithms.
extern const double RD[128];

// Lookup table for compensated constants for exact range reduction when FMA
// instructions are not available.
extern const double CD[128];

// Lookup table for -log(r)
extern const double LOG_R[128];
extern const NumberPair<double> LOG_R_DD[128];

// Lookup table for -log2(r)
extern const double LOG2_R[128];

// Minimax polynomial for (log(1 + x) - x)/x^2, generated by sollya with:
// > P = fpminimax((log(1 + x) - x)/x^2, 5, [|D...|], [-2^-8, 2^-7]);
constexpr double LOG_COEFFS[6] = {-0x1.fffffffffffffp-2, 0x1.5555555554a9bp-2,
                                  -0x1.0000000094567p-2, 0x1.99999dcc9823cp-3,
                                  -0x1.55550ac2e537ap-3, 0x1.21a02c4e624d7p-3};

// Logarithm Range Reduction - Step 2, 3, and 4.
extern const int S2[193];
extern const int S3[161];
extern const int S4[130];

extern const double R2[193];

// log(2) generated by Sollya with:
// > a = 2^-43 * nearestint(2^43*log(2));
// LSB = 2^-43 is chosen so that e_x * LOG_2_HI is exact for -1075 < e_x < 1024.
constexpr double LOG_2_HI = 0x1.62e42fefa38p-1; // LSB = 2^-43
// > b = round(log10(2) - a, D, RN);
constexpr double LOG_2_LO = 0x1.ef35793c7673p-45; // LSB = 2^-97

// Lookup table for exp(m) with m = -104, ..., 89.
//   -104 = floor(log(single precision's min denormal))
//     89 = ceil(log(single precision's max normal))
// Table is generated with Sollya as follow:
// > display = hexadecimal;
// > for i from -104 to 89 do { D(exp(i)); };
extern const double EXP_M1[195];

// Lookup table for exp(m * 2^(-7)) with m = 0, ..., 127.
// Table is generated with Sollya as follow:
// > display = hexadecimal;
// > for i from 0 to 127 do { D(exp(i / 128)); };
extern const double EXP_M2[128];

} // namespace LIBC_NAMESPACE_DECL

#endif // LLVM_LIBC_SRC_MATH_GENERIC_COMMON_CONSTANTS_H
