// Test case that compares a file generated by rust_xlsxwriter with a file
// created by Excel.
//
// SPDX-License-Identifier: MIT OR Apache-2.0
//
// Copyright 2022-2025, John McNamara, jmcnamara@cpan.org

use crate::common;
use rust_xlsxwriter::{Format, Table, TableColumn, Workbook, XlsxError};

// Set column header text but override with cell string and format.
fn create_new_xlsx_file_1(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();

    let worksheet = workbook.add_worksheet();
    let text_wrap = Format::new().set_text_wrap();

    worksheet.set_column_range_width(2, 5, 10.288)?;
    worksheet.set_row_height(2, 39)?;
    worksheet.write(15, 0, "hello")?;

    let columns = vec![
        TableColumn::default(),
        TableColumn::default(),
        TableColumn::default(),
        TableColumn::new().set_header("Column\n4"),
    ];

    let table = Table::new().set_columns(&columns);

    worksheet.add_table(2, 2, 12, 5, &table)?;

    // Overwrite cell to simulate header formatting.
    worksheet.write_with_format(2, 5, "Column\n4", &text_wrap)?;

    workbook.save(filename)?;

    Ok(())
}

// Write header text and format.
fn create_new_xlsx_file_2(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();

    let worksheet = workbook.add_worksheet();
    let text_wrap = Format::new().set_text_wrap();

    worksheet.set_column_range_width(2, 5, 10.288)?;
    worksheet.set_row_height(2, 39)?;
    worksheet.write(15, 0, "hello")?;

    let columns = vec![
        TableColumn::default(),
        TableColumn::default(),
        TableColumn::default(),
        TableColumn::new()
            .set_header("Column\n4")
            .set_header_format(text_wrap),
    ];

    let table = Table::new().set_columns(&columns);

    worksheet.add_table(2, 2, 12, 5, &table)?;

    workbook.save(filename)?;

    Ok(())
}

// Write header format but take text from cell.
fn create_new_xlsx_file_3(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();

    let worksheet = workbook.add_worksheet();
    let text_wrap = Format::new().set_text_wrap();

    worksheet.set_column_range_width(2, 5, 10.288)?;
    worksheet.set_row_height(2, 39)?;
    worksheet.write(15, 0, "hello")?;

    // Write the header text. This should be read and used for the column.
    worksheet.write(2, 5, "Column\n4")?;

    let columns = vec![
        TableColumn::default(),
        TableColumn::default(),
        TableColumn::default(),
        TableColumn::new().set_header_format(text_wrap),
    ];

    let table = Table::new().set_columns(&columns);

    worksheet.add_table(2, 2, 12, 5, &table)?;

    workbook.save(filename)?;

    Ok(())
}

#[test]
fn test_table18_1() {
    let test_runner = common::TestRunner::new()
        .set_name("table18")
        .set_function(create_new_xlsx_file_1)
        .unique("1")
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}

#[test]
fn test_table18_2() {
    let test_runner = common::TestRunner::new()
        .set_name("table18")
        .set_function(create_new_xlsx_file_2)
        .unique("2")
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}

#[test]
fn test_table18_3() {
    let test_runner = common::TestRunner::new()
        .set_name("table18")
        .set_function(create_new_xlsx_file_3)
        .unique("3")
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}
