// Test case that compares a file generated by rust_xlsxwriter with a file
// created by Excel.
//
// SPDX-License-Identifier: MIT OR Apache-2.0
//
// Copyright 2022-2025, John McNamara, jmcnamara@cpan.org

use crate::common;
use rust_xlsxwriter::{CustomSerializeField, SerializeFieldOptions, Table, Workbook, XlsxError};
use serde::Serialize;

// Test case for Serde serialization. First test isn't serialized.
fn create_new_xlsx_file_1(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();

    let table = Table::new();

    let worksheet1 = workbook.add_worksheet();

    worksheet1.set_column_width(1, 10.288)?;
    worksheet1.set_column_width(2, 10.288)?;
    worksheet1.set_column_width(3, 10.288)?;
    worksheet1.set_column_width(4, 10.288)?;
    worksheet1.set_column_width(5, 10.288)?;
    worksheet1.set_column_width(6, 10.288)?;
    worksheet1.set_column_width(7, 10.288)?;
    worksheet1.set_column_width(8, 10.288)?;
    worksheet1.set_column_width(9, 10.288)?;

    worksheet1.add_table(2, 1, 10, 4, &table)?;
    worksheet1.add_table(9, 6, 15, 9, &table)?;
    worksheet1.add_table(17, 2, 24, 5, &table)?;

    let worksheet2 = workbook.add_worksheet();

    worksheet2.set_column_width(2, 10.288)?;
    worksheet2.set_column_width(3, 10.288)?;
    worksheet2.set_column_width(4, 10.288)?;
    worksheet2.set_column_width(5, 10.288)?;
    worksheet2.set_column_width(6, 10.288)?;
    worksheet2.set_column_width(7, 10.288)?;
    worksheet2.set_column_width(8, 10.288)?;
    worksheet2.set_column_width(9, 10.288)?;
    worksheet2.set_column_width(10, 10.288)?;
    worksheet2.set_column_width(11, 10.288)?;

    worksheet2.add_table(3, 8, 10, 11, &table)?;
    worksheet2.add_table(15, 2, 22, 7, &table)?;

    workbook.save(filename)?;

    Ok(())
}

// Test case for Serde serialization. Test Worksheet table.
fn create_new_xlsx_file_2(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();
    let worksheet1 = workbook.add_worksheet();

    #[derive(Serialize)]
    #[serde(rename_all = "PascalCase")]
    struct MyStruct1 {
        column1: Option<f64>,
        column2: Option<f64>,
        column3: Option<f64>,
        column4: Option<f64>,
    }

    let data = MyStruct1 {
        column1: None,
        column2: None,
        column3: None,
        column4: None,
    };

    let header_options1 = SerializeFieldOptions::new()
        .set_table_default()
        .set_custom_headers(&[
            CustomSerializeField::new("Column1").set_column_width(10.288),
            CustomSerializeField::new("Column2").set_column_width(10.288),
            CustomSerializeField::new("Column3").set_column_width(10.288),
            CustomSerializeField::new("Column4").set_column_width(10.288),
        ]);

    worksheet1.serialize_headers_with_options(2, 1, &data, &header_options1)?;
    for _ in 1..=8 {
        worksheet1.serialize(&data)?;
    }

    worksheet1.serialize_headers_with_options(9, 6, &data, &header_options1)?;
    for _ in 1..=6 {
        worksheet1.serialize(&data)?;
    }

    worksheet1.serialize_headers_with_options(17, 2, &data, &header_options1)?;
    for _ in 1..=7 {
        worksheet1.serialize(&data)?;
    }

    // Second worksheet.
    let worksheet2 = workbook.add_worksheet();

    worksheet2.serialize_headers_with_options(3, 8, &data, &header_options1)?;
    for _ in 1..=7 {
        worksheet2.serialize(&data)?;
    }

    // Set the header a second time (without the table) to write the table and
    // force the same ordering as Excel.
    let header_options1 = SerializeFieldOptions::new().set_custom_headers(&[
        CustomSerializeField::new("Column1").set_column_width(10.288),
        CustomSerializeField::new("Column2").set_column_width(10.288),
        CustomSerializeField::new("Column3").set_column_width(10.288),
        CustomSerializeField::new("Column4").set_column_width(10.288),
    ]);
    worksheet2.serialize_headers_with_options(3, 8, &data, &header_options1)?;

    // Use a second struct with different column numbers.
    #[derive(Serialize)]
    #[serde(rename_all = "PascalCase")]
    struct MyStruct2 {
        column1: Option<f64>,
        column2: Option<f64>,
        column3: Option<f64>,
        column4: Option<f64>,
        column5: Option<f64>,
        column6: Option<f64>,
    }

    let data = MyStruct2 {
        column1: None,
        column2: None,
        column3: None,
        column4: None,
        column5: None,
        column6: None,
    };

    let header_options2 = SerializeFieldOptions::new()
        .set_table_default()
        .set_custom_headers(&[
            CustomSerializeField::new("Column1").set_column_width(10.288),
            CustomSerializeField::new("Column2").set_column_width(10.288),
            CustomSerializeField::new("Column3").set_column_width(10.288),
            CustomSerializeField::new("Column4").set_column_width(10.288),
            CustomSerializeField::new("Column5").set_column_width(10.288),
            CustomSerializeField::new("Column6").set_column_width(10.288),
        ]);

    worksheet2.serialize_headers_with_options(15, 2, &data, &header_options2)?;
    for _ in 1..=7 {
        worksheet2.serialize(&data)?;
    }

    workbook.save(filename)?;

    Ok(())
}

#[test]
fn test_serde19_1() {
    let test_runner = common::TestRunner::new()
        .set_name("table02")
        .set_function(create_new_xlsx_file_1)
        .unique("serde19_1")
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}

#[test]
fn test_serde19_2() {
    let test_runner = common::TestRunner::new()
        .set_name("table02")
        .set_function(create_new_xlsx_file_2)
        .unique("serde19_2")
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}
