// Test case that compares a file generated by rust_xlsxwriter with a file
// created by Excel.
//
// SPDX-License-Identifier: MIT OR Apache-2.0
//
// Copyright 2022-2025, John McNamara, jmcnamara@cpan.org

use crate::common;
use rust_xlsxwriter::{
    CustomSerializeField, Format, SerializeFieldOptions, Workbook, XlsxError, XlsxSerialize,
};
use serde::{Deserialize, Serialize};

// Test case for Serde serialization. First test isn't serialized.
fn create_new_xlsx_file_1(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();
    let worksheet = workbook.add_worksheet();
    let bold = Format::new().set_bold();

    worksheet.set_paper_size(9);

    // Not serialized.
    worksheet.write_with_format(0, 0, "col1", &bold)?;
    worksheet.write(1, 0, 123)?;
    worksheet.write(2, 0, 456)?;
    worksheet.write(3, 0, 789)?;
    worksheet.write_with_format(0, 1, "col2", &bold)?;
    worksheet.write(1, 1, true)?;
    worksheet.write(2, 1, false)?;
    worksheet.write(3, 1, true)?;

    workbook.save(filename)?;

    Ok(())
}

// Test case for Serde serialization.
fn create_new_xlsx_file_2(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();
    let worksheet = workbook.add_worksheet();
    let bold = Format::new().set_bold();

    worksheet.set_paper_size(9);

    // Create a serializable test struct.
    #[derive(Serialize)]
    struct MyStruct {
        col1: u16,
        col2: bool,
    }

    let data = [
        MyStruct {
            col1: 123,
            col2: true,
        },
        MyStruct {
            col1: 456,
            col2: false,
        },
        MyStruct {
            col1: 789,
            col2: true,
        },
    ];

    worksheet.serialize_headers_with_format(0, 0, &data[0], &bold)?;
    worksheet.serialize(&data)?;

    workbook.save(filename)?;

    Ok(())
}

// Test case for serialize_headers_with_options().
fn create_new_xlsx_file_3(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();
    let worksheet = workbook.add_worksheet();
    let bold = Format::new().set_bold();

    worksheet.set_paper_size(9);

    // Create a serializable test struct.
    #[derive(Serialize)]
    struct MyStruct {
        col1: u16,
        col2: bool,
    }

    let data = [
        MyStruct {
            col1: 123,
            col2: true,
        },
        MyStruct {
            col1: 456,
            col2: false,
        },
        MyStruct {
            col1: 789,
            col2: true,
        },
    ];

    let custom_headers = [
        CustomSerializeField::new("col1").set_header_format(&bold),
        CustomSerializeField::new("col2").set_header_format(&bold),
    ];
    let header_options = SerializeFieldOptions::new().set_custom_headers(&custom_headers);

    worksheet.serialize_headers_with_options(0, 0, &data[0], &header_options)?;
    worksheet.serialize(&data)?;

    workbook.save(filename)?;

    Ok(())
}

// Test case for serialize_headers_with_options().
fn create_new_xlsx_file_4(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();
    let worksheet = workbook.add_worksheet();
    let bold = Format::new().set_bold();

    worksheet.set_paper_size(9);

    // Create a serializable test struct.
    #[derive(Serialize)]
    struct MyStruct {
        col1: u16,
        col2: bool,
    }

    let data = [
        MyStruct {
            col1: 123,
            col2: true,
        },
        MyStruct {
            col1: 456,
            col2: false,
        },
        MyStruct {
            col1: 789,
            col2: true,
        },
    ];

    let custom_headers = [
        CustomSerializeField::new("col1"),
        CustomSerializeField::new("col2"),
    ];

    let header_options = SerializeFieldOptions::new()
        .set_custom_headers(&custom_headers)
        .set_header_format(bold);

    worksheet.serialize_headers_with_options(0, 0, &data[0], &header_options)?;
    worksheet.serialize(&data)?;

    workbook.save(filename)?;

    Ok(())
}

// Test case for Serde serialization. Header deserialization.
fn create_new_xlsx_file_5(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();
    let worksheet = workbook.add_worksheet();
    let bold = Format::new().set_bold();

    worksheet.set_paper_size(9);

    // Create a serializable test struct.
    #[derive(Deserialize, Serialize)]
    struct MyStruct {
        col1: u16,
        col2: bool,
    }

    let data = [
        MyStruct {
            col1: 123,
            col2: true,
        },
        MyStruct {
            col1: 456,
            col2: false,
        },
        MyStruct {
            col1: 789,
            col2: true,
        },
    ];

    worksheet.deserialize_headers_with_format::<MyStruct>(0, 0, &bold)?;
    worksheet.serialize(&data)?;

    workbook.save(filename)?;

    Ok(())
}

// Test case for Serde serialization. With XlsxSerialize.
fn create_new_xlsx_file_6(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();
    let worksheet = workbook.add_worksheet();
    worksheet.set_paper_size(9);

    // Create a serializable test struct.
    #[derive(XlsxSerialize, Serialize)]
    #[xlsx(header_format = Format::new().set_bold())]
    struct MyStruct {
        col1: u16,
        col2: bool,
    }

    let data = [
        MyStruct {
            col1: 123,
            col2: true,
        },
        MyStruct {
            col1: 456,
            col2: false,
        },
        MyStruct {
            col1: 789,
            col2: true,
        },
    ];

    worksheet.set_serialize_headers::<MyStruct>(0, 0)?;
    worksheet.serialize(&data)?;

    workbook.save(filename)?;

    Ok(())
}

// Test case for Serde serialization. With XlsxSerialize.
fn create_new_xlsx_file_7(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();
    let worksheet = workbook.add_worksheet();
    worksheet.set_paper_size(9);

    // Create a serializable test struct.
    #[derive(XlsxSerialize, Serialize)]
    struct MyStruct {
        #[xlsx(header_format = Format::new().set_bold())]
        col1: u16,

        #[xlsx(header_format = Format::new().set_bold())]
        col2: bool,
    }

    let data = [
        MyStruct {
            col1: 123,
            col2: true,
        },
        MyStruct {
            col1: 456,
            col2: false,
        },
        MyStruct {
            col1: 789,
            col2: true,
        },
    ];

    worksheet.set_serialize_headers::<MyStruct>(0, 0)?;
    worksheet.serialize(&data)?;

    workbook.save(filename)?;

    Ok(())
}

#[test]
fn test_serde06_1() {
    let test_runner = common::TestRunner::new()
        .set_name("serde06")
        .set_function(create_new_xlsx_file_1)
        .unique("1")
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}

#[test]
fn test_serde06_2() {
    let test_runner = common::TestRunner::new()
        .set_name("serde06")
        .set_function(create_new_xlsx_file_2)
        .unique("2")
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}

#[test]
fn test_serde06_3() {
    let test_runner = common::TestRunner::new()
        .set_name("serde06")
        .set_function(create_new_xlsx_file_3)
        .unique("3")
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}

#[test]
fn test_serde06_4() {
    let test_runner = common::TestRunner::new()
        .set_name("serde06")
        .set_function(create_new_xlsx_file_4)
        .unique("4")
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}

#[test]
fn test_serde06_5() {
    let test_runner = common::TestRunner::new()
        .set_name("serde06")
        .set_function(create_new_xlsx_file_5)
        .unique("5")
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}

#[test]
fn test_serde06_6() {
    let test_runner = common::TestRunner::new()
        .set_name("serde06")
        .set_function(create_new_xlsx_file_6)
        .unique("6")
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}

#[test]
fn test_serde06_7() {
    let test_runner = common::TestRunner::new()
        .set_name("serde06")
        .set_function(create_new_xlsx_file_7)
        .unique("7")
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}
