// Test case that compares a file generated by rust_xlsxwriter with a file
// created by Excel.
//
// SPDX-License-Identifier: MIT OR Apache-2.0
//
// Copyright 2022-2025, John McNamara, jmcnamara@cpan.org

use crate::common;
use rust_xlsxwriter::{Format, Workbook, XlsxError};

// Create a rust_xlsxwriter file to compare against an Excel file.
fn create_new_xlsx_file_1(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();

    let worksheet = workbook.add_worksheet();
    let bold = Format::new().set_bold();

    // Add a formatted cell and then remove the format.
    worksheet.write_with_format(2, 1, 123, &bold)?;
    worksheet.clear_cell_format(2, 1);

    // Write cells that aren't modified.
    worksheet.write_with_format(2, 3, 123, &bold)?;
    worksheet.write_with_format(2, 5, "", &bold)?;

    // Add and remove a cell.
    worksheet.write_with_format(2, 4, 222, &bold)?;
    worksheet.clear_cell(2, 4);

    workbook.save(filename)?;

    Ok(())
}

// Test writing the data and formatting separately.
fn create_new_xlsx_file_2(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();

    let worksheet = workbook.add_worksheet();
    let bold = Format::new().set_bold();

    // Add the data, unformatted.
    worksheet.write(2, 1, 123)?;
    worksheet.write(2, 3, 123)?;

    // Add the formatting separately.
    worksheet.set_cell_format(2, 3, &bold)?;
    worksheet.set_cell_format(2, 5, &bold)?;

    // This default format in an empty cell should be ignored.
    worksheet.set_cell_format(2, 4, &Format::default())?;

    workbook.save(filename)?;

    Ok(())
}

// Test with standard methods and no post write changes.
fn create_new_xlsx_file_3(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();

    let worksheet = workbook.add_worksheet();
    let bold = Format::new().set_bold();

    worksheet.write(2, 1, 123)?;
    worksheet.write_with_format(2, 3, 123, &bold)?;
    worksheet.write_with_format(2, 5, "", &bold)?;

    workbook.save(filename)?;

    Ok(())
}

#[test]
fn test_overlay01_1() {
    let test_runner = common::TestRunner::new()
        .set_name("overlay01")
        .unique("1")
        .set_function(create_new_xlsx_file_1)
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}

#[test]
fn test_overlay01_2() {
    let test_runner = common::TestRunner::new()
        .set_name("overlay01")
        .unique("2")
        .set_function(create_new_xlsx_file_2)
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}

#[test]
fn test_overlay01_3() {
    let test_runner = common::TestRunner::new()
        .set_name("overlay01")
        .unique("3")
        .set_function(create_new_xlsx_file_3)
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}
