// Test case that compares a file generated by rust_xlsxwriter with a file
// created by Excel.
//
// SPDX-License-Identifier: MIT OR Apache-2.0
//
// Copyright 2022-2025, John McNamara, jmcnamara@cpan.org

use crate::common;
use rust_xlsxwriter::{Format, Workbook, XlsxError};

// Create rust_xlsxwriter file to compare against Excel file.
fn create_new_xlsx_file(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();

    // Align the Shared String Table to the input file.
    workbook.populate_string_table("Region", 0);
    workbook.populate_string_table("North", 1);
    workbook.populate_string_table("North Total", 2);

    let bold = Format::new().set_bold();

    let worksheet = workbook.add_worksheet().set_name("Collapsed Rows")?;
    worksheet.set_selection(13, 0, 13, 0)?;

    // Create the a first level grouping as a collapsed group.
    worksheet.group_rows_collapsed(1, 10)?;

    // Create 2 secondary groups non-collapsed within the first range.
    worksheet.group_rows(1, 4)?;
    worksheet.group_rows(6, 9)?;

    worksheet.set_column_width(0, 20)?;

    worksheet.write_with_format(0, 0, "Region", &bold)?;
    worksheet.write(1, 0, "North")?;
    worksheet.write(2, 0, "North")?;
    worksheet.write(3, 0, "North")?;
    worksheet.write(4, 0, "North")?;
    worksheet.write_with_format(5, 0, "North Total", &bold)?;

    worksheet.write_with_format(0, 1, "Sales", &bold)?;
    worksheet.write(1, 1, 1000)?;
    worksheet.write(2, 1, 1200)?;
    worksheet.write(3, 1, 900)?;
    worksheet.write(4, 1, 1200)?;
    worksheet
        .write_formula_with_format(5, 1, "=SUBTOTAL(9,B2:B5)", &bold)?
        .set_formula_result(5, 1, "4300");

    worksheet.write(6, 0, "South")?;
    worksheet.write(7, 0, "South")?;
    worksheet.write(8, 0, "South")?;
    worksheet.write(9, 0, "South")?;
    worksheet.write_with_format(10, 0, "South Total", &bold)?;

    worksheet.write(6, 1, 400)?;
    worksheet.write(7, 1, 600)?;
    worksheet.write(8, 1, 500)?;
    worksheet.write(9, 1, 600)?;
    worksheet
        .write_formula_with_format(10, 1, "=SUBTOTAL(9,B7:B10)", &bold)?
        .set_formula_result(10, 1, "2100");

    worksheet.write_with_format(11, 0, "Grand Total", &bold)?;
    worksheet
        .write_formula_with_format(11, 1, "=SUBTOTAL(9,B2:B10)", &bold)?
        .set_formula_result(11, 1, "6400");

    workbook.save(filename)?;

    Ok(())
}

#[test]
fn test_outline02() {
    let test_runner = common::TestRunner::new()
        .set_name("outline02")
        .set_function(create_new_xlsx_file)
        .ignore_calc_chain()
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}
