// Test case that compares a file generated by rust_xlsxwriter with a file
// created by Excel.
//
// SPDX-License-Identifier: MIT OR Apache-2.0
//
// Copyright 2022-2025, John McNamara, jmcnamara@cpan.org

use crate::common;
use rust_xlsxwriter::{Format, Table, TableColumn, Workbook, XlsxError};

// Test with write_with_format().
fn create_new_xlsx_file_1(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();

    let format1 = Format::new().set_num_format("0.00;[Red]0.00");
    let format2 = Format::new().set_num_format("0.00_ ;\\-0.00\\ ");
    let format3 = Format::new().set_num_format("0.00_ ;[Red]\\-0.00\\ ");

    let worksheet = workbook.add_worksheet_with_low_memory();

    // Manually set the indices to get the same order as the target file.
    worksheet.format_dxf_index(&format3);
    worksheet.format_dxf_index(&format2);
    worksheet.format_dxf_index(&format1);

    worksheet.set_column_range_width(2, 5, 10.288)?;

    let columns = vec![
        TableColumn::default(),
        TableColumn::new().set_format(&format1),
        TableColumn::new().set_format(&format2),
        TableColumn::new().set_format(&format3),
    ];

    let table = Table::new().set_columns(&columns);

    worksheet.add_table(1, 2, 5, 5, &table)?;

    worksheet.write(2, 2, "Foo")?;
    worksheet.write_with_format(2, 3, 1234, &format1)?;
    worksheet.write_with_format(2, 4, 2000, &format2)?;
    worksheet.write_with_format(2, 5, 4321, &format3)?;

    worksheet.write(3, 2, "Bar")?;
    worksheet.write_with_format(3, 3, 1256, &format1)?;
    worksheet.write_with_format(3, 4, 4000, &format2)?;
    worksheet.write_with_format(3, 5, 4320, &format3)?;

    worksheet.write(4, 2, "Baz")?;
    worksheet.write_with_format(4, 3, 2234, &format1)?;
    worksheet.write_with_format(4, 4, 3000, &format2)?;
    worksheet.write_with_format(4, 5, 4332, &format3)?;

    worksheet.write(5, 2, "Bop")?;
    worksheet.write_with_format(5, 3, 1324, &format1)?;
    worksheet.write_with_format(5, 4, 1000, &format2)?;
    worksheet.write_with_format(5, 5, 4333, &format3)?;

    workbook.save(filename)?;

    Ok(())
}

// Test using add_table() to set formats for table data.
fn create_new_xlsx_file_2(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();

    let format1 = Format::new().set_num_format("0.00;[Red]0.00");
    let format2 = Format::new().set_num_format("0.00_ ;\\-0.00\\ ");
    let format3 = Format::new().set_num_format("0.00_ ;[Red]\\-0.00\\ ");

    let worksheet = workbook.add_worksheet_with_low_memory();

    // Manually set the indices to get the same order as the target file.
    worksheet.format_dxf_index(&format3);
    worksheet.format_dxf_index(&format2);
    worksheet.format_dxf_index(&format1);

    worksheet.set_column_range_width(2, 5, 10.288)?;

    let columns = vec![
        TableColumn::default(),
        TableColumn::new().set_format(&format1),
        TableColumn::new().set_format(&format2),
        TableColumn::new().set_format(&format3),
    ];

    let table = Table::new().set_columns(&columns);

    worksheet.add_table(1, 2, 5, 5, &table)?;

    worksheet.write(2, 2, "Foo")?;
    worksheet.write_with_format(2, 3, 1234, &format1)?;
    worksheet.write_with_format(2, 4, 2000, &format2)?;
    worksheet.write_with_format(2, 5, 4321, &format3)?;

    worksheet.write(3, 2, "Bar")?;
    worksheet.write_with_format(3, 3, 1256, &format1)?;
    worksheet.write_with_format(3, 4, 4000, &format2)?;
    worksheet.write_with_format(3, 5, 4320, &format3)?;

    worksheet.write(4, 2, "Baz")?;
    worksheet.write_with_format(4, 3, 2234, &format1)?;
    worksheet.write_with_format(4, 4, 3000, &format2)?;
    worksheet.write_with_format(4, 5, 4332, &format3)?;

    worksheet.write(5, 2, "Bop")?;
    worksheet.write_with_format(5, 3, 1324, &format1)?;
    worksheet.write_with_format(5, 4, 1000, &format2)?;
    worksheet.write_with_format(5, 5, 4333, &format3)?;

    workbook.save(filename)?;

    Ok(())
}

// Test using IntoFormat trait.
fn create_new_xlsx_file_3(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();

    let format1 = "0.00;[Red]0.00".into();
    let format2 = "0.00_ ;\\-0.00\\ ".into();
    let format3 = "0.00_ ;[Red]\\-0.00\\ ".into();

    let worksheet = workbook.add_worksheet_with_low_memory();

    // Manually set the indices to get the same order as the target file.
    worksheet.format_dxf_index(&format3);
    worksheet.format_dxf_index(&format2);
    worksheet.format_dxf_index(&format1);

    worksheet.set_column_range_width(2, 5, 10.288)?;

    let columns = vec![
        TableColumn::default(),
        TableColumn::new().set_format("0.00;[Red]0.00"),
        TableColumn::new().set_format("0.00_ ;\\-0.00\\ "),
        TableColumn::new().set_format("0.00_ ;[Red]\\-0.00\\ "),
    ];

    let table = Table::new().set_columns(&columns);

    worksheet.add_table(1, 2, 5, 5, &table)?;

    worksheet.write(2, 2, "Foo")?;
    worksheet.write_with_format(2, 3, 1234, &format1)?;
    worksheet.write_with_format(2, 4, 2000, &format2)?;
    worksheet.write_with_format(2, 5, 4321, &format3)?;

    worksheet.write(3, 2, "Bar")?;
    worksheet.write_with_format(3, 3, 1256, &format1)?;
    worksheet.write_with_format(3, 4, 4000, &format2)?;
    worksheet.write_with_format(3, 5, 4320, &format3)?;

    worksheet.write(4, 2, "Baz")?;
    worksheet.write_with_format(4, 3, 2234, &format1)?;
    worksheet.write_with_format(4, 4, 3000, &format2)?;
    worksheet.write_with_format(4, 5, 4332, &format3)?;

    worksheet.write(5, 2, "Bop")?;
    worksheet.write_with_format(5, 3, 1324, &format1)?;
    worksheet.write_with_format(5, 4, 1000, &format2)?;
    worksheet.write_with_format(5, 5, 4333, &format3)?;

    workbook.save(filename)?;

    Ok(())
}

#[test]
fn test_optimize_table14_1() {
    let test_runner = common::TestRunner::new()
        .set_name("table14")
        .set_function(create_new_xlsx_file_1)
        .unique("optimize1")
        .ignore_worksheet_spans()
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}

#[test]
fn test_optimize_table14_2() {
    let test_runner = common::TestRunner::new()
        .set_name("table14")
        .set_function(create_new_xlsx_file_2)
        .unique("optimize2")
        .ignore_worksheet_spans()
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}

#[test]
fn test_optimize_table14_3() {
    let test_runner = common::TestRunner::new()
        .set_name("table14")
        .set_function(create_new_xlsx_file_3)
        .unique("optimize3")
        .ignore_worksheet_spans()
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}
