// Test case that compares a file generated by rust_xlsxwriter with a file
// created by Excel.
//
// SPDX-License-Identifier: MIT OR Apache-2.0
//
// Copyright 2022-2025, John McNamara, jmcnamara@cpan.org

use crate::common;
use rust_xlsxwriter::{CustomSerializeField, SerializeFieldOptions, Table, Workbook, XlsxError};
use serde::Serialize;

// Test case for Serde serialization. First test isn't serialized.
fn create_new_xlsx_file_1(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();

    let worksheet = workbook.add_worksheet_with_low_memory();

    worksheet.set_column_range_width(2, 5, 10.288)?;

    let table = Table::new();

    worksheet.add_table(1, 2, 5, 5, &table)?;

    worksheet.write(2, 2, "Foo")?;
    worksheet.write_row(2, 3, [1234, 2000, 4321])?;

    worksheet.write(3, 2, "Bar")?;
    worksheet.write_row(3, 3, [1256, 4000, 4320])?;

    worksheet.write(4, 2, "Baz")?;

    worksheet.write_row(4, 3, [2234, 3000, 4332])?;

    worksheet.write(5, 2, "Bop")?;
    worksheet.write_row(5, 3, [1324, 1000, 4333])?;

    workbook.save(filename)?;

    Ok(())
}

// Test case for Serde serialization. Test Worksheet table.
fn create_new_xlsx_file_2(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();
    let worksheet = workbook.add_worksheet_with_low_memory();

    #[derive(Serialize)]
    #[serde(rename_all = "PascalCase")]
    struct MyStruct<'a> {
        column1: &'a str,
        column2: u16,
        column3: u16,
        column4: u16,
    }

    let data = [
        MyStruct {
            column1: "Foo",
            column2: 1234,
            column3: 2000,
            column4: 4321,
        },
        MyStruct {
            column1: "Bar",
            column2: 1256,
            column3: 4000,
            column4: 4320,
        },
        MyStruct {
            column1: "Baz",
            column2: 2234,
            column3: 3000,
            column4: 4332,
        },
        MyStruct {
            column1: "Bop",
            column2: 1324,
            column3: 1000,
            column4: 4333,
        },
    ];

    let table = Table::new();

    let header_options = SerializeFieldOptions::new()
        .set_table(&table)
        .set_custom_headers(&[
            CustomSerializeField::new("Column1").set_column_width(10.288),
            CustomSerializeField::new("Column2").set_column_width(10.288),
            CustomSerializeField::new("Column3").set_column_width(10.288),
            CustomSerializeField::new("Column4").set_column_width(10.288),
        ]);

    worksheet.serialize_headers_with_options(1, 2, &data[0], &header_options)?;

    worksheet.serialize(&data)?;

    workbook.save(filename)?;

    Ok(())
}

#[test]
fn test_optimize_serde23_1() {
    let test_runner = common::TestRunner::new()
        .set_name("table11")
        .set_function(create_new_xlsx_file_1)
        .unique("optimize_serde23_1")
        .ignore_worksheet_spans()
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}

#[test]
fn test_optimize_serde23_2() {
    let test_runner = common::TestRunner::new()
        .set_name("table11")
        .set_function(create_new_xlsx_file_2)
        .unique("optimize_serde23_2")
        .ignore_worksheet_spans()
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}
