// Test case that compares a file generated by rust_xlsxwriter with a file
// created by Excel.
//
// SPDX-License-Identifier: MIT OR Apache-2.0
//
// Copyright 2022-2025, John McNamara, jmcnamara@cpan.org

use crate::common;
use rust_xlsxwriter::{
    CustomSerializeField, SerializeFieldOptions, Table, TableColumn, TableFunction, Workbook,
    XlsxError,
};
use serde::Serialize;

// Test case for Serde serialization. First test isn't serialized.
fn create_new_xlsx_file_1(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();

    // Pre-populate the string table to get the same order as Excel.
    workbook.populate_string_table("Column1", 0);
    workbook.populate_string_table("Column2", 1);
    workbook.populate_string_table("Column3", 2);
    workbook.populate_string_table("Column4", 3);

    let worksheet = workbook.add_worksheet_with_low_memory();

    worksheet.set_column_width(2, 10.288)?;
    worksheet.set_column_width(3, 10.288)?;
    worksheet.set_column_width(4, 10.288)?;
    worksheet.set_column_width(5, 10.288)?;

    let columns = vec![
        TableColumn::new().set_total_label("Total"),
        TableColumn::default(),
        TableColumn::default(),
        TableColumn::new().set_total_function(TableFunction::Count),
    ];

    let table = Table::new().set_columns(&columns).set_total_row(true);

    worksheet.add_table(2, 2, 13, 5, &table)?;

    workbook.save(filename)?;

    Ok(())
}

// Test case for Serde serialization. Test Worksheet table.
fn create_new_xlsx_file_2(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();

    let worksheet = workbook.add_worksheet_with_low_memory();

    #[derive(Serialize)]
    #[serde(rename_all = "PascalCase")]
    struct MyStruct {
        column1: Option<f64>,
        column2: Option<f64>,
        column3: Option<f64>,
        column4: Option<f64>,
    }

    let data = MyStruct {
        column1: None,
        column2: None,
        column3: None,
        column4: None,
    };

    // Create a user defined table.
    let columns = vec![
        TableColumn::new().set_total_label("Total"),
        TableColumn::default(),
        TableColumn::default(),
        TableColumn::new().set_total_function(TableFunction::Count),
    ];

    let table = Table::new().set_columns(&columns).set_total_row(true);

    let header_options = SerializeFieldOptions::new()
        .set_table(&table)
        .set_custom_headers(&[
            CustomSerializeField::new("Column1").set_column_width(10.288),
            CustomSerializeField::new("Column2").set_column_width(10.288),
            CustomSerializeField::new("Column3").set_column_width(10.288),
            CustomSerializeField::new("Column4").set_column_width(10.288),
        ]);

    worksheet.serialize_headers_with_options(2, 2, &data, &header_options)?;

    for _ in 1..=10 {
        worksheet.serialize(&data)?;
    }

    workbook.save(filename)?;

    Ok(())
}

#[test]
fn test_optimize_serde21_1() {
    let test_runner = common::TestRunner::new()
        .set_name("table08")
        .set_function(create_new_xlsx_file_1)
        .ignore_calc_chain()
        .unique("optimize_serde21_1")
        .ignore_worksheet_spans()
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}

#[test]
fn test_optimize_serde21_2() {
    let test_runner = common::TestRunner::new()
        .set_name("table08")
        .set_function(create_new_xlsx_file_2)
        .ignore_calc_chain()
        .unique("optimize_serde21_2")
        .ignore_worksheet_spans()
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}
