// Test case that compares a file generated by rust_xlsxwriter with a file
// created by Excel.
//
// SPDX-License-Identifier: MIT OR Apache-2.0
//
// Copyright 2022-2025, John McNamara, jmcnamara@cpan.org

use crate::common;
use rust_xlsxwriter::{Workbook, XlsxError};

// Create a rust_xlsxwriter file to compare against an Excel file.
fn create_new_xlsx_file_1(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();

    let worksheet = workbook.add_worksheet_with_constant_memory();

    worksheet.write_string(0, 0, "Hello")?;
    worksheet.write_number(1, 0, 123)?;

    // This should be ignored since later row has already been written.
    worksheet.write_string(0, 0, "Foo")?;

    workbook.save(filename)?;

    Ok(())
}

// Test intermediate saves.
fn create_new_xlsx_file_2(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();

    let mut worksheet = workbook.add_worksheet_with_constant_memory();

    worksheet.write_string(0, 0, "Hello")?;
    workbook.save(filename)?;

    // Row 0 should be ignored now.
    worksheet = workbook.worksheet_from_index(0)?;
    worksheet.write_string(0, 0, "World")?;
    workbook.save(filename)?;

    worksheet = workbook.worksheet_from_index(0)?;
    worksheet.write_number(1, 0, 123)?;
    workbook.save(filename)?;

    Ok(())
}

// Test setting a custom directory for temporary files.
fn create_new_xlsx_file_3(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();

    workbook.set_tempdir(".")?;

    let worksheet = workbook.add_worksheet_with_constant_memory();

    worksheet.write_string(0, 0, "Hello")?;
    worksheet.write_number(1, 0, 123)?;

    workbook.save(filename)?;

    Ok(())
}

#[test]
fn test_optimize02_1() {
    let test_runner = common::TestRunner::new()
        .set_name("optimize02")
        .set_function(create_new_xlsx_file_1)
        .unique("1")
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}

#[test]
fn test_optimize02_2() {
    let test_runner = common::TestRunner::new()
        .set_name("optimize02")
        .set_function(create_new_xlsx_file_2)
        .unique("2")
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}

#[test]
fn test_optimize02_3() {
    let test_runner = common::TestRunner::new()
        .set_name("optimize02")
        .set_function(create_new_xlsx_file_3)
        .unique("3")
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}
