// Test case that compares a file generated by rust_xlsxwriter with a file
// created by Excel.
//
// SPDX-License-Identifier: MIT OR Apache-2.0
//
// Copyright 2022-2025, John McNamara, jmcnamara@cpan.org

use crate::common;
use rust_xlsxwriter::{Chart, ChartType, Workbook, XlsxError};

// Create a rust_xlsxwriter file to compare against an Excel file.
fn create_new_xlsx_file_1(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();

    let worksheet = workbook.add_worksheet();

    // Add some test data for the chart(s).
    let data = [[1, 2, 3], [2, 4, 6], [3, 6, 9], [4, 8, 12], [5, 10, 15]];
    for (row_num, row_data) in data.iter().enumerate() {
        for (col_num, col_data) in row_data.iter().enumerate() {
            worksheet.write_number(row_num as u32, col_num as u16, *col_data)?;
        }
    }

    let mut chart = Chart::new(ChartType::Bar);
    chart.set_axis_ids(64265216, 64447616);
    chart
        .add_series()
        .set_categories(("Sheet1", 0, 0, 4, 0))
        .set_values(("Sheet1", 0, 1, 4, 1));

    chart
        .add_series()
        .set_categories(("Sheet1", 0, 0, 4, 0))
        .set_values(("Sheet1", 0, 2, 4, 2));

    worksheet.insert_chart(8, 4, &chart)?;

    let mut chart = Chart::new(ChartType::Bar);
    chart.set_axis_ids(86048128, 86058112);
    chart
        .add_series()
        .set_categories(("Sheet1", 0, 0, 3, 0))
        .set_values(("Sheet1", 0, 1, 3, 1));

    chart
        .add_series()
        .set_categories(("Sheet1", 0, 0, 3, 0))
        .set_values(("Sheet1", 0, 2, 3, 2));

    worksheet.insert_chart(24, 5, &chart)?;

    workbook.save(filename)?;

    Ok(())
}

// Test inserting objects with offsets from a single cell.
fn create_new_xlsx_file_2(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();

    let worksheet = workbook.add_worksheet();

    // Add some test data for the chart(s).
    let data = [[1, 2, 3], [2, 4, 6], [3, 6, 9], [4, 8, 12], [5, 10, 15]];
    for (row_num, row_data) in data.iter().enumerate() {
        for (col_num, col_data) in row_data.iter().enumerate() {
            worksheet.write_number(row_num as u32, col_num as u16, *col_data)?;
        }
    }

    let mut chart = Chart::new(ChartType::Bar);
    chart.set_axis_ids(64265216, 64447616);
    chart
        .add_series()
        .set_categories(("Sheet1", 0, 0, 4, 0))
        .set_values(("Sheet1", 0, 1, 4, 1));

    chart
        .add_series()
        .set_categories(("Sheet1", 0, 0, 4, 0))
        .set_values(("Sheet1", 0, 2, 4, 2));

    worksheet.insert_chart_with_offset(0, 0, &chart, 256, 160)?;

    let mut chart = Chart::new(ChartType::Bar);
    chart.set_axis_ids(86048128, 86058112);
    chart
        .add_series()
        .set_categories(("Sheet1", 0, 0, 3, 0))
        .set_values(("Sheet1", 0, 1, 3, 1));

    chart
        .add_series()
        .set_categories(("Sheet1", 0, 0, 3, 0))
        .set_values(("Sheet1", 0, 2, 3, 2));

    worksheet.insert_chart_with_offset(0, 0, &chart, 320, 480)?;

    workbook.save(filename)?;

    Ok(())
}

#[test]
fn test_chart_bar03_1() {
    let test_runner = common::TestRunner::new()
        .set_name("chart_bar03")
        .unique("1")
        .set_function(create_new_xlsx_file_1)
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}

#[test]
fn test_chart_bar03_2() {
    let test_runner = common::TestRunner::new()
        .set_name("chart_bar03")
        .unique("2")
        .set_function(create_new_xlsx_file_2)
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}
