// Test case that compares a file generated by rust_xlsxwriter with a file
// created by Excel.
//
// SPDX-License-Identifier: MIT OR Apache-2.0
//
// Copyright 2022-2025, John McNamara, jmcnamara@cpan.org

use crate::common;
use rust_xlsxwriter::{ExcelDateTime, Format, Workbook, XlsxError};

// Test case generic write_with_format().
fn create_new_xlsx_file_1(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();

    let format1 = Format::new().set_num_format("yyyy\\-mm\\-dd\\ hh:mm:ss");
    let format2 = Format::new().set_num_format("yyyy\\-mm\\-dd;@");
    let format3 = Format::new().set_num_format("hh:mm:ss;@");

    let worksheet = workbook.add_worksheet();
    worksheet.set_column_width(0, 30)?;

    let datetime = ExcelDateTime::from_ymd(2023, 1, 25)
        .unwrap()
        .and_hms(18, 0, 0)
        .unwrap();
    let time = ExcelDateTime::from_hms(18, 0, 0)?;

    let date = ExcelDateTime::from_ymd(2023, 1, 25)?;

    worksheet.write_with_format(0, 0, &datetime, &format1)?;
    worksheet.write_with_format(1, 0, &date, &format2)?;
    worksheet.write_with_format(2, 0, &time, &format3)?;

    workbook.save(filename)?;

    Ok(())
}

// Test case generic write_with_format(). With chrono.
#[cfg(feature = "chrono")]
fn create_new_xlsx_file_2(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();

    let format1 = Format::new().set_num_format("yyyy\\-mm\\-dd\\ hh:mm:ss");
    let format2 = Format::new().set_num_format("yyyy\\-mm\\-dd;@");
    let format3 = Format::new().set_num_format("hh:mm:ss;@");

    let worksheet = workbook.add_worksheet();
    worksheet.set_column_width(0, 30)?;

    let datetime = chrono::NaiveDate::from_ymd_opt(2023, 1, 25)
        .unwrap()
        .and_hms_opt(18, 0, 0)
        .unwrap();
    let time = datetime.time();

    let datetime2 = chrono::NaiveDate::from_ymd_opt(2023, 1, 25)
        .unwrap()
        .and_hms_opt(0, 0, 0)
        .unwrap();
    let date = datetime2.date();

    worksheet.write_with_format(0, 0, &datetime, &format1)?;
    worksheet.write_with_format(1, 0, &date, &format2)?;
    worksheet.write_with_format(2, 0, &time, &format3)?;

    workbook.save(filename)?;

    Ok(())
}

// Test case generic write_with_format(). With jiff.
#[cfg(feature = "jiff")]
fn create_new_xlsx_file_3(filename: &str) -> Result<(), XlsxError> {
    let mut workbook = Workbook::new();

    let format1 = Format::new().set_num_format("yyyy\\-mm\\-dd\\ hh:mm:ss");
    let format2 = Format::new().set_num_format("yyyy\\-mm\\-dd;@");
    let format3 = Format::new().set_num_format("hh:mm:ss;@");

    let worksheet = workbook.add_worksheet();
    worksheet.set_column_width(0, 30)?;

    let datetime = jiff::civil::datetime(2023, 1, 25, 18, 0, 0, 0);
    let time = datetime.time();

    let datetime2 = jiff::civil::datetime(2023, 1, 25, 0, 0, 0, 0);
    let date = datetime2.date();

    worksheet.write_with_format(0, 0, &datetime, &format1)?;
    worksheet.write_with_format(1, 0, &date, &format2)?;
    worksheet.write_with_format(2, 0, &time, &format3)?;

    workbook.save(filename)?;

    Ok(())
}

#[test]
fn bootstrap61_date_time_1() {
    let test_runner = common::TestRunner::new()
        .set_name("bootstrap61")
        .set_function(create_new_xlsx_file_1)
        .unique("1")
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}

#[cfg(feature = "chrono")]
#[test]
fn bootstrap61_date_time_2() {
    let test_runner = common::TestRunner::new()
        .set_name("bootstrap61")
        .set_function(create_new_xlsx_file_2)
        .unique("2")
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}

#[cfg(feature = "jiff")]
#[test]
fn bootstrap61_date_time_3() {
    let test_runner = common::TestRunner::new()
        .set_name("bootstrap61")
        .set_function(create_new_xlsx_file_3)
        .unique("3")
        .initialize();

    test_runner.assert_eq();
    test_runner.cleanup();
}
