// excel_datetime unit tests.
//
// SPDX-License-Identifier: MIT OR Apache-2.0
//
// Copyright 2022-2025, John McNamara, jmcnamara@cpan.org

#[cfg(test)]
mod datetime_tests {

    use crate::{ExcelDateTime, XlsxError};
    use pretty_assertions::assert_eq;

    #[test]
    fn check_validations() {
        let result = ExcelDateTime::from_ymd(1899, 12, 30);
        assert!(matches!(result, Err(XlsxError::DateTimeRangeError(_))));

        let result = ExcelDateTime::from_ymd(10000, 1, 1);
        assert!(matches!(result, Err(XlsxError::DateTimeRangeError(_))));

        let result = ExcelDateTime::from_ymd(2000, 0, 1);
        assert!(matches!(result, Err(XlsxError::DateTimeRangeError(_))));

        let result = ExcelDateTime::from_ymd(2000, 13, 1);
        assert!(matches!(result, Err(XlsxError::DateTimeRangeError(_))));

        let result = ExcelDateTime::from_ymd(2000, 1, 0);
        assert!(matches!(result, Err(XlsxError::DateTimeRangeError(_))));

        let result = ExcelDateTime::from_ymd(2000, 1, 32);
        assert!(matches!(result, Err(XlsxError::DateTimeRangeError(_))));

        let result = ExcelDateTime::from_ymd(2000, 4, 31); // Invalid month day.
        assert!(matches!(result, Err(XlsxError::DateTimeRangeError(_))));

        let result = ExcelDateTime::from_hms(0, 61, 0);
        assert!(matches!(result, Err(XlsxError::DateTimeRangeError(_))));

        let result = ExcelDateTime::from_hms(0, 0, 59.9999);
        assert!(matches!(result, Err(XlsxError::DateTimeRangeError(_))));

        let result = ExcelDateTime::from_hms_milli(0, 61, 0, 0);
        assert!(matches!(result, Err(XlsxError::DateTimeRangeError(_))));

        let result = ExcelDateTime::from_hms_milli(0, 0, 61, 0);
        assert!(matches!(result, Err(XlsxError::DateTimeRangeError(_))));

        let result = ExcelDateTime::from_hms_milli(0, 0, 0, 1000);
        assert!(matches!(result, Err(XlsxError::DateTimeRangeError(_))));

        let result = ExcelDateTime::from_serial_datetime(-1);
        assert!(matches!(result, Err(XlsxError::DateTimeRangeError(_))));

        let result = ExcelDateTime::from_serial_datetime(2958466);
        assert!(matches!(result, Err(XlsxError::DateTimeRangeError(_))));

        let result = ExcelDateTime::from_timestamp(-2209075201);
        assert!(matches!(result, Err(XlsxError::DateTimeRangeError(_))));

        let result = ExcelDateTime::from_timestamp(253402300800);
        assert!(matches!(result, Err(XlsxError::DateTimeRangeError(_))));
    }

    #[test]
    fn datetimes_from_str() {
        // Test datetimes.
        #[allow(clippy::excessive_precision)]
        let dates = vec![
            ("1899-12-31T00:00:00.000", 0.0),
            ("1982-08-25T00:15:20.213", 30188.010650613425),
            ("2065-04-19T00:16:48.290", 60376.011670023145),
            ("2147-12-15T00:55:25.446", 90565.038488958337),
            ("2230-08-10T01:02:46.891", 120753.04359827546),
            ("2313-04-06T01:04:15.597", 150942.04462496529),
            ("2395-11-30T01:09:40.889", 181130.04838991899),
            ("2478-07-25T01:11:32.560", 211318.04968240741),
            ("2561-03-21T01:30:19.169", 241507.06272186342),
            ("2643-11-15T01:48:25.580", 271695.07529606484),
            ("2726-07-12T02:03:31.919", 301884.08578609955),
            ("2809-03-06T02:11:11.986", 332072.09111094906),
            ("2891-10-31T02:24:37.095", 362261.10042934027),
            ("2974-06-26T02:35:07.220", 392449.10772245371),
            ("3057-02-19T02:45:12.109", 422637.1147234838),
            ("3139-10-17T03:06:39.990", 452826.12962951389),
            ("3222-06-11T03:08:08.251", 483014.13065105322),
            ("3305-02-05T03:19:12.576", 513203.13834),
            ("3387-10-01T03:29:42.574", 543391.14563164348),
            ("3470-05-27T03:37:30.813", 573579.15105107636),
            ("3553-01-21T04:14:38.231", 603768.17683137732),
            ("3635-09-16T04:16:28.559", 633956.17810832174),
            ("3718-05-13T04:17:58.222", 664145.17914608796),
            ("3801-01-06T04:21:41.794", 694333.18173372687),
            ("3883-09-02T04:56:35.792", 724522.20596981479),
            ("3966-04-28T05:25:14.885", 754710.2258667245),
            ("4048-12-21T05:26:05.724", 784898.22645513888),
            ("4131-08-18T05:46:44.068", 815087.24078782403),
            ("4214-04-13T05:48:01.141", 845275.24167987274),
            ("4296-12-07T05:53:52.315", 875464.24574438657),
            ("4379-08-03T06:14:48.580", 905652.26028449077),
            ("4462-03-28T06:46:15.738", 935840.28212659725),
            ("4544-11-22T07:31:20.407", 966029.31343063654),
            ("4627-07-19T07:58:33.754", 996217.33233511576),
            ("4710-03-15T08:07:43.130", 1026406.3386936343),
            ("4792-11-07T08:29:11.091", 1056594.3536005903),
            ("4875-07-04T09:08:15.328", 1086783.3807329629),
            ("4958-02-27T09:30:41.781", 1116971.3963169097),
            ("5040-10-23T09:34:04.462", 1147159.3986627546),
            ("5123-06-20T09:37:23.945", 1177348.4009715857),
            ("5206-02-12T09:37:56.655", 1207536.4013501736),
            ("5288-10-08T09:45:12.230", 1237725.406391551),
            ("5371-06-04T09:54:14.782", 1267913.412671088),
            ("5454-01-28T09:54:22.108", 1298101.4127558796),
            ("5536-09-24T10:01:36.151", 1328290.4177795255),
            ("5619-05-20T12:09:48.602", 1358478.5068125231),
            ("5702-01-14T12:34:08.549", 1388667.5237100578),
            ("5784-09-08T12:56:06.495", 1418855.5389640625),
            ("5867-05-06T12:58:58.217", 1449044.5409515856),
            ("5949-12-30T12:59:54.263", 1479232.5416002662),
            ("6032-08-24T13:34:41.331", 1509420.5657561459),
            ("6115-04-21T13:58:28.601", 1539609.5822754744),
            ("6197-12-14T14:02:16.899", 1569797.5849178126),
            ("6280-08-10T14:36:17.444", 1599986.6085352316),
            ("6363-04-06T14:37:57.451", 1630174.60969272),
            ("6445-11-30T14:57:42.757", 1660363.6234115392),
            ("6528-07-26T15:10:48.307", 1690551.6325035533),
            ("6611-03-22T15:14:39.890", 1720739.635183912),
            ("6693-11-15T15:19:47.988", 1750928.6387498612),
            ("6776-07-11T16:04:24.344", 1781116.6697262037),
            ("6859-03-07T16:22:23.952", 1811305.6822216667),
            ("6941-10-31T16:29:55.999", 1841493.6874536921),
            ("7024-06-26T16:58:20.259", 1871681.7071789235),
            ("7107-02-21T17:04:02.415", 1901870.7111390624),
            ("7189-10-16T17:18:29.630", 1932058.7211762732),
            ("7272-06-11T17:47:21.323", 1962247.7412190163),
            ("7355-02-05T17:53:29.866", 1992435.7454845603),
            ("7437-10-02T17:53:41.076", 2022624.7456143056),
            ("7520-05-28T17:55:06.044", 2052812.7465977315),
            ("7603-01-21T18:14:49.151", 2083000.7602910995),
            ("7685-09-16T18:17:45.738", 2113189.7623349307),
            ("7768-05-12T18:29:59.700", 2143377.7708298611),
            ("7851-01-07T18:33:21.233", 2173566.773162419),
            ("7933-09-02T19:14:24.673", 2203754.8016744559),
            ("8016-04-27T19:17:12.816", 2233942.8036205554),
            ("8098-12-22T19:23:36.418", 2264131.8080603937),
            ("8181-08-17T19:46:25.908", 2294319.8239109721),
            ("8264-04-13T20:07:47.314", 2324508.8387420601),
            ("8346-12-08T20:31:37.603", 2354696.855296331),
            ("8429-08-03T20:39:57.770", 2384885.8610853008),
            ("8512-03-29T20:50:17.067", 2415073.8682530904),
            ("8594-11-22T21:02:57.827", 2445261.8770581828),
            ("8677-07-19T21:23:05.519", 2475450.8910360998),
            ("8760-03-14T21:34:49.572", 2505638.8991848612),
            ("8842-11-08T21:39:05.944", 2535827.9021521294),
            ("8925-07-04T21:39:18.426", 2566015.9022965971),
            ("9008-02-28T21:46:07.769", 2596203.9070343636),
            ("9090-10-24T21:57:55.662", 2626392.9152275696),
            ("9173-06-19T22:19:11.732", 2656580.9299968979),
            ("9256-02-13T22:23:51.376", 2686769.9332335186),
            ("9338-10-09T22:27:58.771", 2716957.9360968866),
            ("9421-06-05T22:43:30.392", 2747146.9468795368),
            ("9504-01-30T22:48:25.834", 2777334.9502990046),
            ("9586-09-24T22:53:51.727", 2807522.9540709145),
            ("9669-05-20T23:12:56.536", 2837711.9673210187),
            ("9752-01-14T23:15:54.109", 2867899.9693762613),
            ("9834-09-10T23:17:12.632", 2898088.9702850925),
            ("9999-12-31T23:59:59.000", 2958465.999988426),
        ];

        for test_data in dates {
            let (datetime_string, expected) = test_data;
            let datetime = ExcelDateTime::parse_from_str(datetime_string).unwrap();
            assert_eq!(expected, datetime.to_excel(), "{datetime_string}");
        }
    }

    #[test]
    fn datetimes_from_str_with_noise() {
        // Test datetimes with extra whitespace and the Zulu timezone.
        #[allow(clippy::excessive_precision)]
        let dates = vec![
            ("1982-08-25T00:15:20.213", 30188.010650613425),
            ("1982-08-25 00:15:20.213", 30188.010650613425),
            ("1982-08-25T00:15:20.213Z", 30188.010650613425),
            ("   1982-08-25T00:15:20.213", 30188.010650613425),
            ("1982-08-25T00:15:20.213   ", 30188.010650613425),
            ("2400-01-31T", 182653.0),
            ("2400-01-31", 182653.0),
            ("  2400-01-31", 182653.0),
            ("2400-01-31T  ", 182653.0),
            ("18:14:49.151", 0.760291099537037),
            ("18:14:49.151Z", 0.760291099537037),
            ("  18:14:49.151", 0.760291099537037),
            ("18:14:49.151  ", 0.760291099537037),
            ("12:00", 0.5),
            (" 12:00", 0.5),
            ("12:00  ", 0.5),
        ];

        for test_data in dates {
            let (datetime_string, expected) = test_data;
            let datetime = ExcelDateTime::parse_from_str(datetime_string).unwrap();
            assert_eq!(expected, datetime.to_excel(), "{datetime_string}");
        }
    }

    #[test]
    fn dates_from_str() {
        // Test date only.
        let dates = vec![
            ("1899-12-31T", 0.0),
            ("1900-01-01T", 1.0),
            ("1900-02-27T", 58.0),
            ("1900-02-28T", 59.0),
            ("1900-02-29T", 60.0),
            ("1900-03-01T", 61.0),
            ("1900-03-02T", 62.0),
            ("1900-03-11T", 71.0),
            ("1900-04-08T", 99.0),
            ("1900-09-12T", 256.0),
            ("1901-05-03T", 489.0),
            ("1901-10-13T", 652.0),
            ("1902-02-15T", 777.0),
            ("1902-06-06T", 888.0),
            ("1902-09-25T", 999.0),
            ("1902-09-27T", 1001.0),
            ("1903-04-26T", 1212.0),
            ("1903-08-05T", 1313.0),
            ("1903-12-31T", 1461.0),
            ("1904-01-01T", 1462.0),
            ("1904-02-28T", 1520.0),
            ("1904-02-29T", 1521.0),
            ("1904-03-01T", 1522.0),
            ("1907-02-27T", 2615.0),
            ("1907-02-28T", 2616.0),
            ("1907-03-01T", 2617.0),
            ("1907-03-02T", 2618.0),
            ("1907-03-03T", 2619.0),
            ("1907-03-04T", 2620.0),
            ("1907-03-05T", 2621.0),
            ("1907-03-06T", 2622.0),
            ("1999-01-01T", 36161.0),
            ("1999-01-31T", 36191.0),
            ("1999-02-01T", 36192.0),
            ("1999-02-28T", 36219.0),
            ("1999-03-01T", 36220.0),
            ("1999-03-31T", 36250.0),
            ("1999-04-01T", 36251.0),
            ("1999-04-30T", 36280.0),
            ("1999-05-01T", 36281.0),
            ("1999-05-31T", 36311.0),
            ("1999-06-01T", 36312.0),
            ("1999-06-30T", 36341.0),
            ("1999-07-01T", 36342.0),
            ("1999-07-31T", 36372.0),
            ("1999-08-01T", 36373.0),
            ("1999-08-31T", 36403.0),
            ("1999-09-01T", 36404.0),
            ("1999-09-30T", 36433.0),
            ("1999-10-01T", 36434.0),
            ("1999-10-31T", 36464.0),
            ("1999-11-01T", 36465.0),
            ("1999-11-30T", 36494.0),
            ("1999-12-01T", 36495.0),
            ("1999-12-31T", 36525.0),
            ("2000-01-01T", 36526.0),
            ("2000-01-31T", 36556.0),
            ("2000-02-01T", 36557.0),
            ("2000-02-29T", 36585.0),
            ("2000-03-01T", 36586.0),
            ("2000-03-31T", 36616.0),
            ("2000-04-01T", 36617.0),
            ("2000-04-30T", 36646.0),
            ("2000-05-01T", 36647.0),
            ("2000-05-31T", 36677.0),
            ("2000-06-01T", 36678.0),
            ("2000-06-30T", 36707.0),
            ("2000-07-01T", 36708.0),
            ("2000-07-31T", 36738.0),
            ("2000-08-01T", 36739.0),
            ("2000-08-31T", 36769.0),
            ("2000-09-01T", 36770.0),
            ("2000-09-30T", 36799.0),
            ("2000-10-01T", 36800.0),
            ("2000-10-31T", 36830.0),
            ("2000-11-01T", 36831.0),
            ("2000-11-30T", 36860.0),
            ("2000-12-01T", 36861.0),
            ("2000-12-31T", 36891.0),
            ("2001-01-01T", 36892.0),
            ("2001-01-31T", 36922.0),
            ("2001-02-01T", 36923.0),
            ("2001-02-28T", 36950.0),
            ("2001-03-01T", 36951.0),
            ("2001-03-31T", 36981.0),
            ("2001-04-01T", 36982.0),
            ("2001-04-30T", 37011.0),
            ("2001-05-01T", 37012.0),
            ("2001-05-31T", 37042.0),
            ("2001-06-01T", 37043.0),
            ("2001-06-30T", 37072.0),
            ("2001-07-01T", 37073.0),
            ("2001-07-31T", 37103.0),
            ("2001-08-01T", 37104.0),
            ("2001-08-31T", 37134.0),
            ("2001-09-01T", 37135.0),
            ("2001-09-30T", 37164.0),
            ("2001-10-01T", 37165.0),
            ("2001-10-31T", 37195.0),
            ("2001-11-01T", 37196.0),
            ("2001-11-30T", 37225.0),
            ("2001-12-01T", 37226.0),
            ("2001-12-31T", 37256.0),
            ("2400-01-01T", 182623.0),
            ("2400-01-31T", 182653.0),
            ("2400-02-01T", 182654.0),
            ("2400-02-29T", 182682.0),
            ("2400-03-01T", 182683.0),
            ("2400-03-31T", 182713.0),
            ("2400-04-01T", 182714.0),
            ("2400-04-30T", 182743.0),
            ("2400-05-01T", 182744.0),
            ("2400-05-31T", 182774.0),
            ("2400-06-01T", 182775.0),
            ("2400-06-30T", 182804.0),
            ("2400-07-01T", 182805.0),
            ("2400-07-31T", 182835.0),
            ("2400-08-01T", 182836.0),
            ("2400-08-31T", 182866.0),
            ("2400-09-01T", 182867.0),
            ("2400-09-30T", 182896.0),
            ("2400-10-01T", 182897.0),
            ("2400-10-31T", 182927.0),
            ("2400-11-01T", 182928.0),
            ("2400-11-30T", 182957.0),
            ("2400-12-01T", 182958.0),
            ("2400-12-31T", 182988.0),
            ("4000-01-01T", 767011.0),
            ("4000-01-31T", 767041.0),
            ("4000-02-01T", 767042.0),
            ("4000-02-29T", 767070.0),
            ("4000-03-01T", 767071.0),
            ("4000-03-31T", 767101.0),
            ("4000-04-01T", 767102.0),
            ("4000-04-30T", 767131.0),
            ("4000-05-01T", 767132.0),
            ("4000-05-31T", 767162.0),
            ("4000-06-01T", 767163.0),
            ("4000-06-30T", 767192.0),
            ("4000-07-01T", 767193.0),
            ("4000-07-31T", 767223.0),
            ("4000-08-01T", 767224.0),
            ("4000-08-31T", 767254.0),
            ("4000-09-01T", 767255.0),
            ("4000-09-30T", 767284.0),
            ("4000-10-01T", 767285.0),
            ("4000-10-31T", 767315.0),
            ("4000-11-01T", 767316.0),
            ("4000-11-30T", 767345.0),
            ("4000-12-01T", 767346.0),
            ("4000-12-31T", 767376.0),
            ("4321-01-01T", 884254.0),
            ("4321-01-31T", 884284.0),
            ("4321-02-01T", 884285.0),
            ("4321-02-28T", 884312.0),
            ("4321-03-01T", 884313.0),
            ("4321-03-31T", 884343.0),
            ("4321-04-01T", 884344.0),
            ("4321-04-30T", 884373.0),
            ("4321-05-01T", 884374.0),
            ("4321-05-31T", 884404.0),
            ("4321-06-01T", 884405.0),
            ("4321-06-30T", 884434.0),
            ("4321-07-01T", 884435.0),
            ("4321-07-31T", 884465.0),
            ("4321-08-01T", 884466.0),
            ("4321-08-31T", 884496.0),
            ("4321-09-01T", 884497.0),
            ("4321-09-30T", 884526.0),
            ("4321-10-01T", 884527.0),
            ("4321-10-31T", 884557.0),
            ("4321-11-01T", 884558.0),
            ("4321-11-30T", 884587.0),
            ("4321-12-01T", 884588.0),
            ("4321-12-31T", 884618.0),
            ("9999-01-01T", 2958101.0),
            ("9999-01-31T", 2958131.0),
            ("9999-02-01T", 2958132.0),
            ("9999-02-28T", 2958159.0),
            ("9999-03-01T", 2958160.0),
            ("9999-03-31T", 2958190.0),
            ("9999-04-01T", 2958191.0),
            ("9999-04-30T", 2958220.0),
            ("9999-05-01T", 2958221.0),
            ("9999-05-31T", 2958251.0),
            ("9999-06-01T", 2958252.0),
            ("9999-06-30T", 2958281.0),
            ("9999-07-01T", 2958282.0),
            ("9999-07-31T", 2958312.0),
            ("9999-08-01T", 2958313.0),
            ("9999-08-31T", 2958343.0),
            ("9999-09-01T", 2958344.0),
            ("9999-09-30T", 2958373.0),
            ("9999-10-01T", 2958374.0),
            ("9999-10-31T", 2958404.0),
            ("9999-11-01T", 2958405.0),
            ("9999-11-30T", 2958434.0),
            ("9999-12-01T", 2958435.0),
            ("9999-12-31T", 2958465.0),
        ];

        for test_data in dates {
            let (datetime_string, expected) = test_data;
            let datetime = ExcelDateTime::parse_from_str(datetime_string).unwrap();
            assert_eq!(expected, datetime.to_excel(), "{datetime_string}");
        }
    }

    #[test]
    fn times_from_str() {
        // Test time only.
        #[allow(clippy::excessive_precision)]
        let times = vec![
            ("00:00", 0.0),
            ("00:00:00", 0.0),
            ("00:00:00.000", 0.0),
            ("00:15:20.213", 1.0650613425925924e-2),
            ("00:16:48.290", 1.1670023148148148e-2),
            ("00:55:25.446", 3.8488958333333337e-2),
            ("01:02:46.891", 4.3598275462962965e-2),
            ("01:04:15.597", 4.4624965277777782e-2),
            ("01:09:40.889", 4.8389918981481483e-2),
            ("01:11:32.560", 4.9682407407407404e-2),
            ("01:30:19.169", 6.2721863425925936e-2),
            ("01:48:25.580", 7.5296064814814809e-2),
            ("02:03:31.919", 8.5786099537037031e-2),
            ("02:11:11.986", 9.1110949074074077e-2),
            ("02:24:37.095", 0.10042934027777778),
            ("02:35:07.220", 0.1077224537037037),
            ("02:45:12.109", 0.11472348379629631),
            ("03:06:39.990", 0.12962951388888888),
            ("03:08:08.251", 0.13065105324074075),
            ("03:19:12.576", 0.13833999999999999),
            ("03:29:42.574", 0.14563164351851851),
            ("03:37:30.813", 0.1510510763888889),
            ("04:14:38.231", 0.1768313773148148),
            ("04:16:28.559", 0.17810832175925925),
            ("04:17:58.222", 0.17914608796296297),
            ("04:21:41.794", 0.18173372685185185),
            ("04:56:35.792", 0.2059698148148148),
            ("05:25:14.885", 0.22586672453703704),
            ("05:26:05.724", 0.22645513888888891),
            ("05:46:44.068", 0.24078782407407406),
            ("05:48:01.141", 0.2416798726851852),
            ("05:53:52.315", 0.24574438657407408),
            ("06:14:48.580", 0.26028449074074073),
            ("06:46:15.738", 0.28212659722222222),
            ("07:31:20.407", 0.31343063657407405),
            ("07:58:33.754", 0.33233511574074076),
            ("08:07:43.130", 0.33869363425925925),
            ("08:29:11.091", 0.35360059027777774),
            ("09:08:15.328", 0.380732962962963),
            ("09:30:41.781", 0.39631690972222228),
            ("09:34:04.462", 0.39866275462962958),
            ("09:37:23.945", 0.40097158564814817),
            ("09:37:56.655", 0.40135017361111114),
            ("09:45:12.230", 0.40639155092592594),
            ("09:54:14.782", 0.41267108796296298),
            ("09:54:22.108", 0.41275587962962962),
            ("10:01:36.151", 0.41777952546296299),
            ("12:09:48.602", 0.50681252314814818),
            ("12:34:08.549", 0.52371005787037039),
            ("12:56:06.495", 0.53896406249999995),
            ("12:58:58.217", 0.54095158564814816),
            ("12:59:54.263", 0.54160026620370372),
            ("13:34:41.331", 0.56575614583333333),
            ("13:58:28.601", 0.58227547453703699),
            ("14:02:16.899", 0.58491781249999997),
            ("14:36:17.444", 0.60853523148148148),
            ("14:37:57.451", 0.60969271990740748),
            ("14:57:42.757", 0.6234115393518519),
            ("15:10:48.307", 0.6325035532407407),
            ("15:14:39.890", 0.63518391203703706),
            ("15:19:47.988", 0.63874986111111109),
            ("16:04:24.344", 0.66972620370370362),
            ("16:22:23.952", 0.68222166666666662),
            ("16:29:55.999", 0.6874536921296297),
            ("16:58:20.259", 0.70717892361111112),
            ("17:04:02.415", 0.71113906250000003),
            ("17:18:29.630", 0.72117627314814825),
            ("17:47:21.323", 0.74121901620370367),
            ("17:53:29.866", 0.74548456018518516),
            ("17:53:41.076", 0.74561430555555563),
            ("17:55:06.044", 0.74659773148148145),
            ("18:14:49.151", 0.760291099537037),
            ("18:17:45.738", 0.76233493055555546),
            ("18:29:59.700", 0.77082986111111118),
            ("18:33:21.233", 0.77316241898148153),
            ("19:14:24.673", 0.80167445601851861),
            ("19:17:12.816", 0.80362055555555545),
            ("19:23:36.418", 0.80806039351851855),
            ("19:46:25.908", 0.82391097222222232),
            ("20:07:47.314", 0.83874206018518516),
            ("20:31:37.603", 0.85529633101851854),
            ("20:39:57.770", 0.86108530092592594),
            ("20:50:17.067", 0.86825309027777775),
            ("21:02:57.827", 0.87705818287037041),
            ("21:23:05.519", 0.891036099537037),
            ("21:34:49.572", 0.89918486111111118),
            ("21:39:05.944", 0.90215212962962965),
            ("21:39:18.426", 0.90229659722222222),
            ("21:46:07.769", 0.90703436342592603),
            ("21:57:55.662", 0.91522756944444439),
            ("22:19:11.732", 0.92999689814814823),
            ("22:23:51.376", 0.93323351851851843),
            ("22:27:58.771", 0.93609688657407408),
            ("22:43:30.392", 0.94687953703703709),
            ("22:48:25.834", 0.95029900462962968),
            ("22:53:51.727", 0.95407091435185187),
            ("23:12:56.536", 0.96732101851851848),
            ("23:15:54.109", 0.96937626157407408),
            ("23:17:12.632", 0.97028509259259266),
            ("23:59:59.999", 0.99999998842592586),
            ("96:15:30.000", 4.01076388888889),
        ];

        for test_data in times {
            let (datetime_string, expected) = test_data;
            let datetime = ExcelDateTime::parse_from_str(datetime_string).unwrap();
            let mut diff = datetime.to_excel() - expected;
            diff = diff.abs();
            assert!(diff < 0.00000000001, "{datetime_string}");
        }
    }

    #[test]
    fn dates_and_times() {
        // Test date and time
        #[allow(clippy::excessive_precision)]
        let datetimes = vec![
            (1899, 12, 31, 0, 0, 0.0, 0.0),
            (1982, 8, 25, 0, 15, 20.213, 30188.010650613425),
            (2065, 4, 19, 0, 16, 48.290, 60376.011670023145),
            (2147, 12, 15, 0, 55, 25.446, 90565.038488958337),
            (2230, 8, 10, 1, 2, 46.891, 120753.04359827546),
            (2313, 4, 6, 1, 4, 15.597, 150942.04462496529),
            (2395, 11, 30, 1, 9, 40.889, 181130.04838991899),
            (2478, 7, 25, 1, 11, 32.560, 211318.04968240741),
            (2561, 3, 21, 1, 30, 19.169, 241507.06272186342),
            (2643, 11, 15, 1, 48, 25.580, 271695.07529606484),
            (2726, 7, 12, 2, 3, 31.919, 301884.08578609955),
            (2809, 3, 6, 2, 11, 11.986, 332072.09111094906),
            (2891, 10, 31, 2, 24, 37.095, 362261.10042934027),
            (2974, 6, 26, 2, 35, 7.220, 392449.10772245371),
            (3057, 2, 19, 2, 45, 12.109, 422637.1147234838),
            (3139, 10, 17, 3, 6, 39.990, 452826.12962951389),
            (3222, 6, 11, 3, 8, 8.251, 483014.13065105322),
            (3305, 2, 5, 3, 19, 12.576, 513203.13834),
            (3387, 10, 1, 3, 29, 42.574, 543391.14563164348),
            (3470, 5, 27, 3, 37, 30.813, 573579.15105107636),
            (3553, 1, 21, 4, 14, 38.231, 603768.17683137732),
            (3635, 9, 16, 4, 16, 28.559, 633956.17810832174),
            (3718, 5, 13, 4, 17, 58.222, 664145.17914608796),
            (3801, 1, 6, 4, 21, 41.794, 694333.18173372687),
            (3883, 9, 2, 4, 56, 35.792, 724522.20596981479),
            (3966, 4, 28, 5, 25, 14.885, 754710.2258667245),
            (4048, 12, 21, 5, 26, 5.724, 784898.22645513888),
            (4131, 8, 18, 5, 46, 44.068, 815087.24078782403),
            (4214, 4, 13, 5, 48, 1.141, 845275.24167987274),
            (4296, 12, 7, 5, 53, 52.315, 875464.24574438657),
            (4379, 8, 3, 6, 14, 48.580, 905652.26028449077),
            (4462, 3, 28, 6, 46, 15.738, 935840.28212659725),
            (4544, 11, 22, 7, 31, 20.407, 966029.31343063654),
            (4627, 7, 19, 7, 58, 33.754, 996217.33233511576),
            (4710, 3, 15, 8, 7, 43.130, 1026406.3386936343),
            (4792, 11, 7, 8, 29, 11.091, 1056594.3536005903),
            (4875, 7, 4, 9, 8, 15.328, 1086783.3807329629),
            (4958, 2, 27, 9, 30, 41.781, 1116971.3963169097),
            (5040, 10, 23, 9, 34, 4.462, 1147159.3986627546),
            (5123, 6, 20, 9, 37, 23.945, 1177348.4009715857),
            (5206, 2, 12, 9, 37, 56.655, 1207536.4013501736),
            (5288, 10, 8, 9, 45, 12.230, 1237725.406391551),
            (5371, 6, 4, 9, 54, 14.782, 1267913.412671088),
            (5454, 1, 28, 9, 54, 22.108, 1298101.4127558796),
            (5536, 9, 24, 10, 1, 36.151, 1328290.4177795255),
            (5619, 5, 20, 12, 9, 48.602, 1358478.5068125231),
            (5702, 1, 14, 12, 34, 8.549, 1388667.5237100578),
            (5784, 9, 8, 12, 56, 6.495, 1418855.5389640625),
            (5867, 5, 6, 12, 58, 58.217, 1449044.5409515856),
            (5949, 12, 30, 12, 59, 54.263, 1479232.5416002662),
            (6032, 8, 24, 13, 34, 41.331, 1509420.5657561459),
            (6115, 4, 21, 13, 58, 28.601, 1539609.5822754744),
            (6197, 12, 14, 14, 2, 16.899, 1569797.5849178126),
            (6280, 8, 10, 14, 36, 17.444, 1599986.6085352316),
            (6363, 4, 6, 14, 37, 57.451, 1630174.60969272),
            (6445, 11, 30, 14, 57, 42.757, 1660363.6234115392),
            (6528, 7, 26, 15, 10, 48.307, 1690551.6325035533),
            (6611, 3, 22, 15, 14, 39.890, 1720739.635183912),
            (6693, 11, 15, 15, 19, 47.988, 1750928.6387498612),
            (6776, 7, 11, 16, 4, 24.344, 1781116.6697262037),
            (6859, 3, 7, 16, 22, 23.952, 1811305.6822216667),
            (6941, 10, 31, 16, 29, 55.999, 1841493.6874536921),
            (7024, 6, 26, 16, 58, 20.259, 1871681.7071789235),
            (7107, 2, 21, 17, 4, 2.415, 1901870.7111390624),
            (7189, 10, 16, 17, 18, 29.630, 1932058.7211762732),
            (7272, 6, 11, 17, 47, 21.323, 1962247.7412190163),
            (7355, 2, 5, 17, 53, 29.866, 1992435.7454845603),
            (7437, 10, 2, 17, 53, 41.076, 2022624.7456143056),
            (7520, 5, 28, 17, 55, 6.044, 2052812.7465977315),
            (7603, 1, 21, 18, 14, 49.151, 2083000.7602910995),
            (7685, 9, 16, 18, 17, 45.738, 2113189.7623349307),
            (7768, 5, 12, 18, 29, 59.700, 2143377.7708298611),
            (7851, 1, 7, 18, 33, 21.233, 2173566.773162419),
            (7933, 9, 2, 19, 14, 24.673, 2203754.8016744559),
            (8016, 4, 27, 19, 17, 12.816, 2233942.8036205554),
            (8098, 12, 22, 19, 23, 36.418, 2264131.8080603937),
            (8181, 8, 17, 19, 46, 25.908, 2294319.8239109721),
            (8264, 4, 13, 20, 7, 47.314, 2324508.8387420601),
            (8346, 12, 8, 20, 31, 37.603, 2354696.855296331),
            (8429, 8, 3, 20, 39, 57.770, 2384885.8610853008),
            (8512, 3, 29, 20, 50, 17.067, 2415073.8682530904),
            (8594, 11, 22, 21, 2, 57.827, 2445261.8770581828),
            (8677, 7, 19, 21, 23, 5.519, 2475450.8910360998),
            (8760, 3, 14, 21, 34, 49.572, 2505638.8991848612),
            (8842, 11, 8, 21, 39, 5.944, 2535827.9021521294),
            (8925, 7, 4, 21, 39, 18.426, 2566015.9022965971),
            (9008, 2, 28, 21, 46, 7.769, 2596203.9070343636),
            (9090, 10, 24, 21, 57, 55.662, 2626392.9152275696),
            (9173, 6, 19, 22, 19, 11.732, 2656580.9299968979),
            (9256, 2, 13, 22, 23, 51.376, 2686769.9332335186),
            (9338, 10, 9, 22, 27, 58.771, 2716957.9360968866),
            (9421, 6, 5, 22, 43, 30.392, 2747146.9468795368),
            (9504, 1, 30, 22, 48, 25.834, 2777334.9502990046),
            (9586, 9, 24, 22, 53, 51.727, 2807522.9540709145),
            (9669, 5, 20, 23, 12, 56.536, 2837711.9673210187),
            (9752, 1, 14, 23, 15, 54.109, 2867899.9693762613),
            (9834, 9, 10, 23, 17, 12.632, 2898088.9702850925),
            (9999, 12, 31, 23, 59, 59.0, 2958465.999988426),
        ];

        for test_data in datetimes {
            let (year, month, day, hour, min, seconds, expected) = test_data;
            let datetime = ExcelDateTime::from_ymd(year, month, day)
                .unwrap()
                .and_hms(hour, min, seconds)
                .unwrap();
            assert_eq!(expected, datetime.to_excel());
        }
    }

    #[test]
    fn dates_and_times_with_milli() {
        // Test date and time
        #[allow(clippy::excessive_precision)]
        let datetimes = vec![
            (1899, 12, 31, 0, 0, 0, 0, 0.0),
            (1982, 8, 25, 0, 15, 20, 213, 30188.010650613425),
            (2065, 4, 19, 0, 16, 48, 290, 60376.011670023145),
            (2147, 12, 15, 0, 55, 25, 446, 90565.038488958337),
            (2230, 8, 10, 1, 2, 46, 891, 120753.04359827546),
            (2313, 4, 6, 1, 4, 15, 597, 150942.04462496529),
            (2395, 11, 30, 1, 9, 40, 889, 181130.04838991899),
            (2478, 7, 25, 1, 11, 32, 560, 211318.04968240741),
            (2561, 3, 21, 1, 30, 19, 169, 241507.06272186342),
            (2643, 11, 15, 1, 48, 25, 580, 271695.07529606484),
            (2726, 7, 12, 2, 3, 31, 919, 301884.08578609955),
            (2809, 3, 6, 2, 11, 11, 986, 332072.09111094906),
            (2891, 10, 31, 2, 24, 37, 95, 362261.10042934027),
            (2974, 6, 26, 2, 35, 7, 220, 392449.10772245371),
            (3057, 2, 19, 2, 45, 12, 109, 422637.1147234838),
            (3139, 10, 17, 3, 6, 39, 990, 452826.12962951389),
            (3222, 6, 11, 3, 8, 8, 251, 483014.13065105322),
            (3305, 2, 5, 3, 19, 12, 576, 513203.13834),
            (3387, 10, 1, 3, 29, 42, 574, 543391.14563164348),
            (3470, 5, 27, 3, 37, 30, 813, 573579.15105107636),
            (3553, 1, 21, 4, 14, 38, 231, 603768.17683137732),
            (3635, 9, 16, 4, 16, 28, 559, 633956.17810832174),
            (3718, 5, 13, 4, 17, 58, 222, 664145.17914608796),
            (3801, 1, 6, 4, 21, 41, 794, 694333.18173372687),
            (3883, 9, 2, 4, 56, 35, 792, 724522.20596981479),
            (3966, 4, 28, 5, 25, 14, 885, 754710.2258667245),
            (4048, 12, 21, 5, 26, 5, 724, 784898.22645513888),
            (4131, 8, 18, 5, 46, 44, 68, 815087.24078782403),
            (4214, 4, 13, 5, 48, 1, 141, 845275.24167987274),
            (4296, 12, 7, 5, 53, 52, 315, 875464.24574438657),
            (4379, 8, 3, 6, 14, 48, 580, 905652.26028449077),
            (4462, 3, 28, 6, 46, 15, 738, 935840.28212659725),
            (4544, 11, 22, 7, 31, 20, 407, 966029.31343063654),
            (4627, 7, 19, 7, 58, 33, 754, 996217.33233511576),
            (4710, 3, 15, 8, 7, 43, 130, 1026406.3386936343),
            (4792, 11, 7, 8, 29, 11, 91, 1056594.3536005903),
            (4875, 7, 4, 9, 8, 15, 328, 1086783.3807329629),
            (4958, 2, 27, 9, 30, 41, 781, 1116971.3963169097),
            (5040, 10, 23, 9, 34, 4, 462, 1147159.3986627546),
            (5123, 6, 20, 9, 37, 23, 945, 1177348.4009715857),
            (5206, 2, 12, 9, 37, 56, 655, 1207536.4013501736),
            (5288, 10, 8, 9, 45, 12, 230, 1237725.406391551),
            (5371, 6, 4, 9, 54, 14, 782, 1267913.412671088),
            (5454, 1, 28, 9, 54, 22, 108, 1298101.4127558796),
            (5536, 9, 24, 10, 1, 36, 151, 1328290.4177795255),
            (5619, 5, 20, 12, 9, 48, 602, 1358478.5068125231),
            (5702, 1, 14, 12, 34, 8, 549, 1388667.5237100578),
            (5784, 9, 8, 12, 56, 6, 495, 1418855.5389640625),
            (5867, 5, 6, 12, 58, 58, 217, 1449044.5409515856),
            (5949, 12, 30, 12, 59, 54, 263, 1479232.5416002662),
            (6032, 8, 24, 13, 34, 41, 331, 1509420.5657561459),
            (6115, 4, 21, 13, 58, 28, 601, 1539609.5822754744),
            (6197, 12, 14, 14, 2, 16, 899, 1569797.5849178126),
            (6280, 8, 10, 14, 36, 17, 444, 1599986.6085352316),
            (6363, 4, 6, 14, 37, 57, 451, 1630174.60969272),
            (6445, 11, 30, 14, 57, 42, 757, 1660363.6234115392),
            (6528, 7, 26, 15, 10, 48, 307, 1690551.6325035533),
            (6611, 3, 22, 15, 14, 39, 890, 1720739.635183912),
            (6693, 11, 15, 15, 19, 47, 988, 1750928.6387498612),
            (6776, 7, 11, 16, 4, 24, 344, 1781116.6697262037),
            (6859, 3, 7, 16, 22, 23, 952, 1811305.6822216667),
            (6941, 10, 31, 16, 29, 55, 999, 1841493.6874536921),
            (7024, 6, 26, 16, 58, 20, 259, 1871681.7071789235),
            (7107, 2, 21, 17, 4, 2, 415, 1901870.7111390624),
            (7189, 10, 16, 17, 18, 29, 630, 1932058.7211762732),
            (7272, 6, 11, 17, 47, 21, 323, 1962247.7412190163),
            (7355, 2, 5, 17, 53, 29, 866, 1992435.7454845603),
            (7437, 10, 2, 17, 53, 41, 76, 2022624.7456143056),
            (7520, 5, 28, 17, 55, 6, 44, 2052812.7465977315),
            (7603, 1, 21, 18, 14, 49, 151, 2083000.7602910995),
            (7685, 9, 16, 18, 17, 45, 738, 2113189.7623349307),
            (7768, 5, 12, 18, 29, 59, 700, 2143377.7708298611),
            (7851, 1, 7, 18, 33, 21, 233, 2173566.773162419),
            (7933, 9, 2, 19, 14, 24, 673, 2203754.8016744559),
            (8016, 4, 27, 19, 17, 12, 816, 2233942.8036205554),
            (8098, 12, 22, 19, 23, 36, 418, 2264131.8080603937),
            (8181, 8, 17, 19, 46, 25, 908, 2294319.8239109721),
            (8264, 4, 13, 20, 7, 47, 314, 2324508.8387420601),
            (8346, 12, 8, 20, 31, 37, 603, 2354696.855296331),
            (8429, 8, 3, 20, 39, 57, 770, 2384885.8610853008),
            (8512, 3, 29, 20, 50, 17, 67, 2415073.8682530904),
            (8594, 11, 22, 21, 2, 57, 827, 2445261.8770581828),
            (8677, 7, 19, 21, 23, 5, 519, 2475450.8910360998),
            (8760, 3, 14, 21, 34, 49, 572, 2505638.8991848612),
            (8842, 11, 8, 21, 39, 5, 944, 2535827.9021521294),
            (8925, 7, 4, 21, 39, 18, 426, 2566015.9022965971),
            (9008, 2, 28, 21, 46, 7, 769, 2596203.9070343636),
            (9090, 10, 24, 21, 57, 55, 662, 2626392.9152275696),
            (9173, 6, 19, 22, 19, 11, 732, 2656580.9299968979),
            (9256, 2, 13, 22, 23, 51, 376, 2686769.9332335186),
            (9338, 10, 9, 22, 27, 58, 771, 2716957.9360968866),
            (9421, 6, 5, 22, 43, 30, 392, 2747146.9468795368),
            (9504, 1, 30, 22, 48, 25, 834, 2777334.9502990046),
            (9586, 9, 24, 22, 53, 51, 727, 2807522.9540709145),
            (9669, 5, 20, 23, 12, 56, 536, 2837711.9673210187),
            (9752, 1, 14, 23, 15, 54, 109, 2867899.9693762613),
            (9834, 9, 10, 23, 17, 12, 632, 2898088.9702850925),
            (9999, 12, 31, 23, 59, 59, 0, 2958465.999988426),
        ];

        for test_data in datetimes {
            let (year, month, day, hour, min, seconds, millis, expected) = test_data;
            let datetime = ExcelDateTime::from_ymd(year, month, day)
                .unwrap()
                .and_hms_milli(hour, min, seconds, millis)
                .unwrap();
            assert_eq!(expected, datetime.to_excel());
        }
    }

    #[test]
    fn dates_only() {
        // Test date only.
        let dates = vec![
            (1899, 12, 31, 0.0),
            (1900, 1, 1, 1.0),
            (1900, 2, 27, 58.0),
            (1900, 2, 28, 59.0),
            (1900, 3, 1, 61.0),
            (1900, 3, 2, 62.0),
            (1900, 3, 11, 71.0),
            (1900, 4, 8, 99.0),
            (1900, 9, 12, 256.0),
            (1901, 5, 3, 489.0),
            (1901, 10, 13, 652.0),
            (1902, 2, 15, 777.0),
            (1902, 6, 6, 888.0),
            (1902, 9, 25, 999.0),
            (1902, 9, 27, 1001.0),
            (1903, 4, 26, 1212.0),
            (1903, 8, 5, 1313.0),
            (1903, 12, 31, 1461.0),
            (1904, 1, 1, 1462.0),
            (1904, 2, 28, 1520.0),
            (1904, 2, 29, 1521.0),
            (1904, 3, 1, 1522.0),
            (1907, 2, 27, 2615.0),
            (1907, 2, 28, 2616.0),
            (1907, 3, 1, 2617.0),
            (1907, 3, 2, 2618.0),
            (1907, 3, 3, 2619.0),
            (1907, 3, 4, 2620.0),
            (1907, 3, 5, 2621.0),
            (1907, 3, 6, 2622.0),
            (1999, 1, 1, 36161.0),
            (1999, 1, 31, 36191.0),
            (1999, 2, 1, 36192.0),
            (1999, 2, 28, 36219.0),
            (1999, 3, 1, 36220.0),
            (1999, 3, 31, 36250.0),
            (1999, 4, 1, 36251.0),
            (1999, 4, 30, 36280.0),
            (1999, 5, 1, 36281.0),
            (1999, 5, 31, 36311.0),
            (1999, 6, 1, 36312.0),
            (1999, 6, 30, 36341.0),
            (1999, 7, 1, 36342.0),
            (1999, 7, 31, 36372.0),
            (1999, 8, 1, 36373.0),
            (1999, 8, 31, 36403.0),
            (1999, 9, 1, 36404.0),
            (1999, 9, 30, 36433.0),
            (1999, 10, 1, 36434.0),
            (1999, 10, 31, 36464.0),
            (1999, 11, 1, 36465.0),
            (1999, 11, 30, 36494.0),
            (1999, 12, 1, 36495.0),
            (1999, 12, 31, 36525.0),
            (2000, 1, 1, 36526.0),
            (2000, 1, 31, 36556.0),
            (2000, 2, 1, 36557.0),
            (2000, 2, 29, 36585.0),
            (2000, 3, 1, 36586.0),
            (2000, 3, 31, 36616.0),
            (2000, 4, 1, 36617.0),
            (2000, 4, 30, 36646.0),
            (2000, 5, 1, 36647.0),
            (2000, 5, 31, 36677.0),
            (2000, 6, 1, 36678.0),
            (2000, 6, 30, 36707.0),
            (2000, 7, 1, 36708.0),
            (2000, 7, 31, 36738.0),
            (2000, 8, 1, 36739.0),
            (2000, 8, 31, 36769.0),
            (2000, 9, 1, 36770.0),
            (2000, 9, 30, 36799.0),
            (2000, 10, 1, 36800.0),
            (2000, 10, 31, 36830.0),
            (2000, 11, 1, 36831.0),
            (2000, 11, 30, 36860.0),
            (2000, 12, 1, 36861.0),
            (2000, 12, 31, 36891.0),
            (2001, 1, 1, 36892.0),
            (2001, 1, 31, 36922.0),
            (2001, 2, 1, 36923.0),
            (2001, 2, 28, 36950.0),
            (2001, 3, 1, 36951.0),
            (2001, 3, 31, 36981.0),
            (2001, 4, 1, 36982.0),
            (2001, 4, 30, 37011.0),
            (2001, 5, 1, 37012.0),
            (2001, 5, 31, 37042.0),
            (2001, 6, 1, 37043.0),
            (2001, 6, 30, 37072.0),
            (2001, 7, 1, 37073.0),
            (2001, 7, 31, 37103.0),
            (2001, 8, 1, 37104.0),
            (2001, 8, 31, 37134.0),
            (2001, 9, 1, 37135.0),
            (2001, 9, 30, 37164.0),
            (2001, 10, 1, 37165.0),
            (2001, 10, 31, 37195.0),
            (2001, 11, 1, 37196.0),
            (2001, 11, 30, 37225.0),
            (2001, 12, 1, 37226.0),
            (2001, 12, 31, 37256.0),
            (2400, 1, 1, 182623.0),
            (2400, 1, 31, 182653.0),
            (2400, 2, 1, 182654.0),
            (2400, 2, 29, 182682.0),
            (2400, 3, 1, 182683.0),
            (2400, 3, 31, 182713.0),
            (2400, 4, 1, 182714.0),
            (2400, 4, 30, 182743.0),
            (2400, 5, 1, 182744.0),
            (2400, 5, 31, 182774.0),
            (2400, 6, 1, 182775.0),
            (2400, 6, 30, 182804.0),
            (2400, 7, 1, 182805.0),
            (2400, 7, 31, 182835.0),
            (2400, 8, 1, 182836.0),
            (2400, 8, 31, 182866.0),
            (2400, 9, 1, 182867.0),
            (2400, 9, 30, 182896.0),
            (2400, 10, 1, 182897.0),
            (2400, 10, 31, 182927.0),
            (2400, 11, 1, 182928.0),
            (2400, 11, 30, 182957.0),
            (2400, 12, 1, 182958.0),
            (2400, 12, 31, 182988.0),
            (4000, 1, 1, 767011.0),
            (4000, 1, 31, 767041.0),
            (4000, 2, 1, 767042.0),
            (4000, 2, 29, 767070.0),
            (4000, 3, 1, 767071.0),
            (4000, 3, 31, 767101.0),
            (4000, 4, 1, 767102.0),
            (4000, 4, 30, 767131.0),
            (4000, 5, 1, 767132.0),
            (4000, 5, 31, 767162.0),
            (4000, 6, 1, 767163.0),
            (4000, 6, 30, 767192.0),
            (4000, 7, 1, 767193.0),
            (4000, 7, 31, 767223.0),
            (4000, 8, 1, 767224.0),
            (4000, 8, 31, 767254.0),
            (4000, 9, 1, 767255.0),
            (4000, 9, 30, 767284.0),
            (4000, 10, 1, 767285.0),
            (4000, 10, 31, 767315.0),
            (4000, 11, 1, 767316.0),
            (4000, 11, 30, 767345.0),
            (4000, 12, 1, 767346.0),
            (4000, 12, 31, 767376.0),
            (4321, 1, 1, 884254.0),
            (4321, 1, 31, 884284.0),
            (4321, 2, 1, 884285.0),
            (4321, 2, 28, 884312.0),
            (4321, 3, 1, 884313.0),
            (4321, 3, 31, 884343.0),
            (4321, 4, 1, 884344.0),
            (4321, 4, 30, 884373.0),
            (4321, 5, 1, 884374.0),
            (4321, 5, 31, 884404.0),
            (4321, 6, 1, 884405.0),
            (4321, 6, 30, 884434.0),
            (4321, 7, 1, 884435.0),
            (4321, 7, 31, 884465.0),
            (4321, 8, 1, 884466.0),
            (4321, 8, 31, 884496.0),
            (4321, 9, 1, 884497.0),
            (4321, 9, 30, 884526.0),
            (4321, 10, 1, 884527.0),
            (4321, 10, 31, 884557.0),
            (4321, 11, 1, 884558.0),
            (4321, 11, 30, 884587.0),
            (4321, 12, 1, 884588.0),
            (4321, 12, 31, 884618.0),
            (9999, 1, 1, 2958101.0),
            (9999, 1, 31, 2958131.0),
            (9999, 2, 1, 2958132.0),
            (9999, 2, 28, 2958159.0),
            (9999, 3, 1, 2958160.0),
            (9999, 3, 31, 2958190.0),
            (9999, 4, 1, 2958191.0),
            (9999, 4, 30, 2958220.0),
            (9999, 5, 1, 2958221.0),
            (9999, 5, 31, 2958251.0),
            (9999, 6, 1, 2958252.0),
            (9999, 6, 30, 2958281.0),
            (9999, 7, 1, 2958282.0),
            (9999, 7, 31, 2958312.0),
            (9999, 8, 1, 2958313.0),
            (9999, 8, 31, 2958343.0),
            (9999, 9, 1, 2958344.0),
            (9999, 9, 30, 2958373.0),
            (9999, 10, 1, 2958374.0),
            (9999, 10, 31, 2958404.0),
            (9999, 11, 1, 2958405.0),
            (9999, 11, 30, 2958434.0),
            (9999, 12, 1, 2958435.0),
            (9999, 12, 31, 2958465.0),
        ];

        for test_data in dates {
            let (year, month, day, expected) = test_data;
            let datetime = ExcelDateTime::from_ymd(year, month, day).unwrap();
            assert_eq!(expected, datetime.to_excel());
        }
    }

    #[test]
    fn dates_only_1904() {
        // Test date only.
        let dates = vec![
            (1904, 1, 1, 0.0),
            (1904, 1, 31, 30.0),
            (1904, 2, 1, 31.0),
            (1904, 2, 29, 59.0),
            (1904, 3, 1, 60.0),
            (1904, 3, 31, 90.0),
            (1904, 4, 1, 91.0),
            (1904, 4, 30, 120.0),
            (1904, 5, 1, 121.0),
            (1904, 5, 31, 151.0),
            (1904, 6, 1, 152.0),
            (1904, 6, 30, 181.0),
            (1904, 7, 1, 182.0),
            (1904, 7, 31, 212.0),
            (1904, 8, 1, 213.0),
            (1904, 8, 31, 243.0),
            (1904, 9, 1, 244.0),
            (1904, 9, 30, 273.0),
            (1904, 10, 1, 274.0),
            (1904, 10, 31, 304.0),
            (1904, 11, 1, 305.0),
            (1904, 11, 30, 334.0),
            (1904, 12, 1, 335.0),
            (1904, 12, 31, 365.0),
            (1907, 2, 27, 1153.0),
            (1907, 2, 28, 1154.0),
            (1907, 3, 1, 1155.0),
            (1907, 3, 2, 1156.0),
            (1907, 3, 3, 1157.0),
            (1907, 3, 4, 1158.0),
            (1907, 3, 5, 1159.0),
            (1907, 3, 6, 1160.0),
            (1999, 1, 1, 34699.0),
            (1999, 1, 31, 34729.0),
            (1999, 2, 1, 34730.0),
            (1999, 2, 28, 34757.0),
            (1999, 3, 1, 34758.0),
            (1999, 3, 31, 34788.0),
            (1999, 4, 1, 34789.0),
            (1999, 4, 30, 34818.0),
            (1999, 5, 1, 34819.0),
            (1999, 5, 31, 34849.0),
            (1999, 6, 1, 34850.0),
            (1999, 6, 30, 34879.0),
            (1999, 7, 1, 34880.0),
            (1999, 7, 31, 34910.0),
            (1999, 8, 1, 34911.0),
            (1999, 8, 31, 34941.0),
            (1999, 9, 1, 34942.0),
            (1999, 9, 30, 34971.0),
            (1999, 10, 1, 34972.0),
            (1999, 10, 31, 35002.0),
            (1999, 11, 1, 35003.0),
            (1999, 11, 30, 35032.0),
            (1999, 12, 1, 35033.0),
            (1999, 12, 31, 35063.0),
            (2000, 1, 1, 35064.0),
            (2000, 1, 31, 35094.0),
            (2000, 2, 1, 35095.0),
            (2000, 2, 29, 35123.0),
            (2000, 3, 1, 35124.0),
            (2000, 3, 31, 35154.0),
            (2000, 4, 1, 35155.0),
            (2000, 4, 30, 35184.0),
            (2000, 5, 1, 35185.0),
            (2000, 5, 31, 35215.0),
            (2000, 6, 1, 35216.0),
            (2000, 6, 30, 35245.0),
            (2000, 7, 1, 35246.0),
            (2000, 7, 31, 35276.0),
            (2000, 8, 1, 35277.0),
            (2000, 8, 31, 35307.0),
            (2000, 9, 1, 35308.0),
            (2000, 9, 30, 35337.0),
            (2000, 10, 1, 35338.0),
            (2000, 10, 31, 35368.0),
            (2000, 11, 1, 35369.0),
            (2000, 11, 30, 35398.0),
            (2000, 12, 1, 35399.0),
            (2000, 12, 31, 35429.0),
            (2001, 1, 1, 35430.0),
            (2001, 1, 31, 35460.0),
            (2001, 2, 1, 35461.0),
            (2001, 2, 28, 35488.0),
            (2001, 3, 1, 35489.0),
            (2001, 3, 31, 35519.0),
            (2001, 4, 1, 35520.0),
            (2001, 4, 30, 35549.0),
            (2001, 5, 1, 35550.0),
            (2001, 5, 31, 35580.0),
            (2001, 6, 1, 35581.0),
            (2001, 6, 30, 35610.0),
            (2001, 7, 1, 35611.0),
            (2001, 7, 31, 35641.0),
            (2001, 8, 1, 35642.0),
            (2001, 8, 31, 35672.0),
            (2001, 9, 1, 35673.0),
            (2001, 9, 30, 35702.0),
            (2001, 10, 1, 35703.0),
            (2001, 10, 31, 35733.0),
            (2001, 11, 1, 35734.0),
            (2001, 11, 30, 35763.0),
            (2001, 12, 1, 35764.0),
            (2001, 12, 31, 35794.0),
            (2400, 1, 1, 181161.0),
            (2400, 1, 31, 181191.0),
            (2400, 2, 1, 181192.0),
            (2400, 2, 29, 181220.0),
            (2400, 3, 1, 181221.0),
            (2400, 3, 31, 181251.0),
            (2400, 4, 1, 181252.0),
            (2400, 4, 30, 181281.0),
            (2400, 5, 1, 181282.0),
            (2400, 5, 31, 181312.0),
            (2400, 6, 1, 181313.0),
            (2400, 6, 30, 181342.0),
            (2400, 7, 1, 181343.0),
            (2400, 7, 31, 181373.0),
            (2400, 8, 1, 181374.0),
            (2400, 8, 31, 181404.0),
            (2400, 9, 1, 181405.0),
            (2400, 9, 30, 181434.0),
            (2400, 10, 1, 181435.0),
            (2400, 10, 31, 181465.0),
            (2400, 11, 1, 181466.0),
            (2400, 11, 30, 181495.0),
            (2400, 12, 1, 181496.0),
            (2400, 12, 31, 181526.0),
            (4000, 1, 1, 765549.0),
            (4000, 1, 31, 765579.0),
            (4000, 2, 1, 765580.0),
            (4000, 2, 29, 765608.0),
            (4000, 3, 1, 765609.0),
            (4000, 3, 31, 765639.0),
            (4000, 4, 1, 765640.0),
            (4000, 4, 30, 765669.0),
            (4000, 5, 1, 765670.0),
            (4000, 5, 31, 765700.0),
            (4000, 6, 1, 765701.0),
            (4000, 6, 30, 765730.0),
            (4000, 7, 1, 765731.0),
            (4000, 7, 31, 765761.0),
            (4000, 8, 1, 765762.0),
            (4000, 8, 31, 765792.0),
            (4000, 9, 1, 765793.0),
            (4000, 9, 30, 765822.0),
            (4000, 10, 1, 765823.0),
            (4000, 10, 31, 765853.0),
            (4000, 11, 1, 765854.0),
            (4000, 11, 30, 765883.0),
            (4000, 12, 1, 765884.0),
            (4000, 12, 31, 765914.0),
            (4321, 1, 1, 882792.0),
            (4321, 1, 31, 882822.0),
            (4321, 2, 1, 882823.0),
            (4321, 2, 28, 882850.0),
            (4321, 3, 1, 882851.0),
            (4321, 3, 31, 882881.0),
            (4321, 4, 1, 882882.0),
            (4321, 4, 30, 882911.0),
            (4321, 5, 1, 882912.0),
            (4321, 5, 31, 882942.0),
            (4321, 6, 1, 882943.0),
            (4321, 6, 30, 882972.0),
            (4321, 7, 1, 882973.0),
            (4321, 7, 31, 883003.0),
            (4321, 8, 1, 883004.0),
            (4321, 8, 31, 883034.0),
            (4321, 9, 1, 883035.0),
            (4321, 9, 30, 883064.0),
            (4321, 10, 1, 883065.0),
            (4321, 10, 31, 883095.0),
            (4321, 11, 1, 883096.0),
            (4321, 11, 30, 883125.0),
            (4321, 12, 1, 883126.0),
            (4321, 12, 31, 883156.0),
            (9999, 1, 1, 2956639.0),
            (9999, 1, 31, 2956669.0),
            (9999, 2, 1, 2956670.0),
            (9999, 2, 28, 2956697.0),
            (9999, 3, 1, 2956698.0),
            (9999, 3, 31, 2956728.0),
            (9999, 4, 1, 2956729.0),
            (9999, 4, 30, 2956758.0),
            (9999, 5, 1, 2956759.0),
            (9999, 5, 31, 2956789.0),
            (9999, 6, 1, 2956790.0),
            (9999, 6, 30, 2956819.0),
            (9999, 7, 1, 2956820.0),
            (9999, 7, 31, 2956850.0),
            (9999, 8, 1, 2956851.0),
            (9999, 8, 31, 2956881.0),
            (9999, 9, 1, 2956882.0),
            (9999, 9, 30, 2956911.0),
            (9999, 10, 1, 2956912.0),
            (9999, 10, 31, 2956942.0),
            (9999, 11, 1, 2956943.0),
            (9999, 11, 30, 2956972.0),
            (9999, 12, 1, 2956973.0),
            (9999, 12, 31, 2957003.0),
        ];

        for test_data in dates {
            let (year, month, day, expected) = test_data;
            let datetime = ExcelDateTime::from_ymd(year, month, day)
                .unwrap()
                .set_1904_date();
            assert_eq!(expected, datetime.to_excel());
        }
    }

    #[test]
    fn times_only() {
        // Test time only.
        #[allow(clippy::excessive_precision)]
        let times = vec![
            (0, 0, 0, 0, 0.0),
            (0, 15, 20, 213, 1.0650613425925924E-2),
            (0, 16, 48, 290, 1.1670023148148148E-2),
            (0, 55, 25, 446, 3.8488958333333337E-2),
            (1, 2, 46, 891, 4.3598275462962965E-2),
            (1, 4, 15, 597, 4.4624965277777782E-2),
            (1, 9, 40, 889, 4.8389918981481483E-2),
            (1, 11, 32, 560, 4.9682407407407404E-2),
            (1, 30, 19, 169, 6.2721863425925936E-2),
            (1, 48, 25, 580, 7.5296064814814809E-2),
            (2, 3, 31, 919, 8.5786099537037031E-2),
            (2, 11, 11, 986, 9.1110949074074077E-2),
            (2, 24, 37, 95, 0.10042934027777778),
            (2, 35, 7, 220, 0.1077224537037037),
            (2, 45, 12, 109, 0.11472348379629631),
            (3, 6, 39, 990, 0.12962951388888888),
            (3, 8, 8, 251, 0.13065105324074075),
            (3, 19, 12, 576, 0.13833999999999999),
            (3, 29, 42, 574, 0.14563164351851851),
            (3, 37, 30, 813, 0.1510510763888889),
            (4, 14, 38, 231, 0.1768313773148148),
            (4, 16, 28, 559, 0.17810832175925925),
            (4, 17, 58, 222, 0.17914608796296297),
            (4, 21, 41, 794, 0.18173372685185185),
            (4, 56, 35, 792, 0.2059698148148148),
            (5, 25, 14, 885, 0.22586672453703704),
            (5, 26, 5, 724, 0.22645513888888891),
            (5, 46, 44, 68, 0.24078782407407406),
            (5, 48, 1, 141, 0.2416798726851852),
            (5, 53, 52, 315, 0.24574438657407408),
            (6, 14, 48, 580, 0.26028449074074073),
            (6, 46, 15, 738, 0.28212659722222222),
            (7, 31, 20, 407, 0.31343063657407405),
            (7, 58, 33, 754, 0.33233511574074076),
            (8, 7, 43, 130, 0.33869363425925925),
            (8, 29, 11, 91, 0.35360059027777774),
            (9, 8, 15, 328, 0.380732962962963),
            (9, 30, 41, 781, 0.39631690972222228),
            (9, 34, 4, 462, 0.39866275462962958),
            (9, 37, 23, 945, 0.40097158564814817),
            (9, 37, 56, 655, 0.40135017361111114),
            (9, 45, 12, 230, 0.40639155092592594),
            (9, 54, 14, 782, 0.41267108796296298),
            (9, 54, 22, 108, 0.41275587962962962),
            (10, 1, 36, 151, 0.41777952546296299),
            (12, 9, 48, 602, 0.50681252314814818),
            (12, 34, 8, 549, 0.52371005787037039),
            (12, 56, 6, 495, 0.53896406249999995),
            (12, 58, 58, 217, 0.54095158564814816),
            (12, 59, 54, 263, 0.54160026620370372),
            (13, 34, 41, 331, 0.56575614583333333),
            (13, 58, 28, 601, 0.58227547453703699),
            (14, 2, 16, 899, 0.58491781249999997),
            (14, 36, 17, 444, 0.60853523148148148),
            (14, 37, 57, 451, 0.60969271990740748),
            (14, 57, 42, 757, 0.6234115393518519),
            (15, 10, 48, 307, 0.6325035532407407),
            (15, 14, 39, 890, 0.63518391203703706),
            (15, 19, 47, 988, 0.63874986111111109),
            (16, 4, 24, 344, 0.66972620370370362),
            (16, 22, 23, 952, 0.68222166666666662),
            (16, 29, 55, 999, 0.6874536921296297),
            (16, 58, 20, 259, 0.70717892361111112),
            (17, 4, 2, 415, 0.71113906250000003),
            (17, 18, 29, 630, 0.72117627314814825),
            (17, 47, 21, 323, 0.74121901620370367),
            (17, 53, 29, 866, 0.74548456018518516),
            (17, 53, 41, 76, 0.74561430555555563),
            (17, 55, 6, 44, 0.74659773148148145),
            (18, 14, 49, 151, 0.760291099537037),
            (18, 17, 45, 738, 0.76233493055555546),
            (18, 29, 59, 700, 0.77082986111111118),
            (18, 33, 21, 233, 0.77316241898148153),
            (19, 14, 24, 673, 0.80167445601851861),
            (19, 17, 12, 816, 0.80362055555555545),
            (19, 23, 36, 418, 0.80806039351851855),
            (19, 46, 25, 908, 0.82391097222222232),
            (20, 7, 47, 314, 0.83874206018518516),
            (20, 31, 37, 603, 0.85529633101851854),
            (20, 39, 57, 770, 0.86108530092592594),
            (20, 50, 17, 67, 0.86825309027777775),
            (21, 2, 57, 827, 0.87705818287037041),
            (21, 23, 5, 519, 0.891036099537037),
            (21, 34, 49, 572, 0.89918486111111118),
            (21, 39, 5, 944, 0.90215212962962965),
            (21, 39, 18, 426, 0.90229659722222222),
            (21, 46, 7, 769, 0.90703436342592603),
            (21, 57, 55, 662, 0.91522756944444439),
            (22, 19, 11, 732, 0.92999689814814823),
            (22, 23, 51, 376, 0.93323351851851843),
            (22, 27, 58, 771, 0.93609688657407408),
            (22, 43, 30, 392, 0.94687953703703709),
            (22, 48, 25, 834, 0.95029900462962968),
            (22, 53, 51, 727, 0.95407091435185187),
            (23, 12, 56, 536, 0.96732101851851848),
            (23, 15, 54, 109, 0.96937626157407408),
            (23, 17, 12, 632, 0.97028509259259266),
            (23, 59, 59, 999, 0.99999998842592586),
            (24, 0, 0, 0, 1.0),
            (1416, 0, 0, 0, 59.0),
            (1440, 0, 0, 0, 60.0),
            (1464, 0, 0, 0, 61.0),
            (1488, 0, 0, 0, 62.0),
            (1500, 0, 0, 0, 62.5),
        ];

        for test_data in times {
            let (hour, min, seconds, millis, expected) = test_data;
            let datetime = ExcelDateTime::from_hms_milli(hour, min, seconds, millis).unwrap();
            let mut diff = datetime.to_excel() - expected;
            diff = diff.abs();
            assert!(diff < 0.00000000001);
        }
    }

    #[test]
    fn serial_date() {
        let expected = 2000.75;
        let datetime = ExcelDateTime::from_serial_datetime(expected).unwrap();
        assert_eq!(expected, datetime.to_excel());
    }

    #[test]
    fn unix_timestamps() {
        // Test unix timestamps.
        let dates = vec![
            (-2209075200, 0.0),        // 1899-12-31 00:00
            (-2209032000, 0.5),        // 1899-12-31 12:00
            (-2208988800, 1.0),        // 1900-01-01 00:00
            (-2208945600, 1.5),        // 1900-01-01 12:00
            (-2203977600, 59.0),       // 1900-02-28 00:00
            (-2203934400, 59.5),       // 1900-02-28 12:00
            (-2203891200, 61.0),       // 1900-03-01 00:00
            (-2203848000, 61.5),       // 1900-03-01 12:00
            (0, 25569.0),              // 1970-01-01 00:00
            (43200, 25569.5),          // 1970-01-01 12:00
            (946684800, 36526.0),      // 2000-01-01 00:00
            (946728000, 36526.5),      // 2000-01-01 12:00
            (253370764800, 2958101.0), // 9999-01-01 00:00
            (253370808000, 2958101.5), // 9999-01-01 12:00
        ];

        for test_data in dates {
            let (timestamp, expected) = test_data;
            let datetime = ExcelDateTime::from_timestamp(timestamp).unwrap();
            assert_eq!(expected, datetime.to_excel());
        }
    }

    #[test]
    fn timestamp_to_rfc3339_times() {
        let tests = [
            ("2000-01-01T01:00:00Z", 946688400),
            ("2000-01-01T00:01:00Z", 946684860),
            ("2000-01-01T00:00:01Z", 946684801),
            ("2000-01-01T23:00:00Z", 946767600),
            ("2000-01-01T00:59:00Z", 946688340),
            ("2000-01-01T00:00:59Z", 946684859),
            ("2000-01-01T23:59:59Z", 946771199),
        ];

        for (expected, unix_time) in tests {
            let got = ExcelDateTime::unix_time_to_iso8601(unix_time);
            assert_eq!(expected, got);
        }
    }

    #[test]
    fn timestamp_to_rfc3339_dates() {
        let tests = [
            ("1970-01-01T00:00:00Z", 0),
            ("1970-02-28T00:00:00Z", 5011200),
            ("1970-03-01T00:00:00Z", 5097600),
            ("1970-12-01T00:00:00Z", 28857600),
            ("1970-12-31T00:00:00Z", 31449600),
            ("1971-01-01T00:00:00Z", 31536000),
            ("1971-02-28T00:00:00Z", 36547200),
            ("1971-03-01T00:00:00Z", 36633600),
            ("1971-12-01T00:00:00Z", 60393600),
            ("1971-12-31T00:00:00Z", 62985600),
            ("1972-01-01T00:00:00Z", 63072000),
            ("1972-01-02T00:00:00Z", 63158400),
            ("1972-02-26T00:00:00Z", 67910400),
            ("1972-02-27T00:00:00Z", 67996800),
            ("1972-02-28T00:00:00Z", 68083200),
            ("1972-02-29T00:00:00Z", 68169600),
            ("1972-03-01T00:00:00Z", 68256000),
            ("1972-12-01T00:00:00Z", 92016000),
            ("1972-12-30T00:00:00Z", 94521600),
            ("1972-12-31T00:00:00Z", 94608000),
            ("1973-01-01T00:00:00Z", 94694400),
            ("1973-02-28T00:00:00Z", 99705600),
            ("1973-03-01T00:00:00Z", 99792000),
            ("1973-12-01T00:00:00Z", 123552000),
            ("1973-12-31T00:00:00Z", 126144000),
            ("1974-01-01T00:00:00Z", 126230400),
            ("1974-02-28T00:00:00Z", 131241600),
            ("1974-03-01T00:00:00Z", 131328000),
            ("1974-12-01T00:00:00Z", 155088000),
            ("1974-12-31T00:00:00Z", 157680000),
            ("2000-01-01T00:00:00Z", 946684800),
            ("2000-02-28T00:00:00Z", 951696000),
            ("2000-02-29T00:00:00Z", 951782400),
            ("2000-03-01T00:00:00Z", 951868800),
            ("2000-12-01T00:00:00Z", 975628800),
            ("2000-12-31T00:00:00Z", 978220800),
            ("2001-01-01T00:00:00Z", 978307200),
            ("2001-02-28T00:00:00Z", 983318400),
            ("2001-03-01T00:00:00Z", 983404800),
            ("2001-12-01T00:00:00Z", 1007164800),
            ("2001-12-31T00:00:00Z", 1009756800),
            ("2002-01-01T00:00:00Z", 1009843200),
            ("2002-02-28T00:00:00Z", 1014854400),
            ("2002-03-01T00:00:00Z", 1014940800),
            ("2002-12-01T00:00:00Z", 1038700800),
            ("2002-12-31T00:00:00Z", 1041292800),
            ("2003-01-01T00:00:00Z", 1041379200),
            ("2003-02-28T00:00:00Z", 1046390400),
            ("2003-03-01T00:00:00Z", 1046476800),
            ("2003-12-01T00:00:00Z", 1070236800),
            ("2003-12-31T00:00:00Z", 1072828800),
            ("2004-01-01T00:00:00Z", 1072915200),
            ("2004-02-28T00:00:00Z", 1077926400),
            ("2004-02-29T00:00:00Z", 1078012800),
            ("2004-03-01T00:00:00Z", 1078099200),
            ("2004-12-01T00:00:00Z", 1101859200),
            ("2004-12-31T00:00:00Z", 1104451200),
            ("2099-01-01T00:00:00Z", 4070908800),
            ("2099-02-28T00:00:00Z", 4075920000),
            ("2099-03-01T00:00:00Z", 4076006400),
            ("2099-12-01T00:00:00Z", 4099766400),
            ("2099-12-30T00:00:00Z", 4102272000),
            ("2099-12-31T00:00:00Z", 4102358400),
            ("2100-01-01T00:00:00Z", 4102444800),
            ("2100-02-28T00:00:00Z", 4107456000),
            ("2100-03-01T00:00:00Z", 4107542400),
            ("2100-12-01T00:00:00Z", 4131302400),
            ("2100-12-31T00:00:00Z", 4133894400),
            ("2399-01-01T00:00:00Z", 13537929600),
            ("2399-02-28T00:00:00Z", 13542940800),
            ("2399-03-01T00:00:00Z", 13543027200),
            ("2399-12-01T00:00:00Z", 13566787200),
            ("2399-12-31T00:00:00Z", 13569379200),
            ("2400-01-01T00:00:00Z", 13569465600),
            ("2400-02-28T00:00:00Z", 13574476800),
            ("2400-03-01T00:00:00Z", 13574649600),
            ("2400-12-01T00:00:00Z", 13598409600),
            ("2400-12-31T00:00:00Z", 13601001600),
            ("2404-01-01T00:00:00Z", 13695696000),
            ("2404-01-02T00:00:00Z", 13695782400),
            ("2404-01-03T00:00:00Z", 13695868800),
            ("2404-02-28T00:00:00Z", 13700707200),
            ("2404-02-29T00:00:00Z", 13700793600),
            ("2404-03-01T00:00:00Z", 13700880000),
            ("2404-12-01T00:00:00Z", 13724640000),
            ("2404-12-31T00:00:00Z", 13727232000),
            ("4000-01-01T00:00:00Z", 64060588800),
            ("4000-02-28T00:00:00Z", 64065600000),
            ("4000-02-29T00:00:00Z", 64065686400),
            ("4000-03-01T00:00:00Z", 64065772800),
            ("4000-12-01T00:00:00Z", 64089532800),
            ("4000-12-31T00:00:00Z", 64092124800),
            ("9999-01-01T00:00:00Z", 253370764800),
            ("9999-02-28T00:00:00Z", 253375776000),
            ("9999-03-01T00:00:00Z", 253375862400),
            ("9999-12-01T00:00:00Z", 253399622400),
            ("9999-12-31T00:00:00Z", 253402214400),
        ];

        for (expected, unix_time) in tests {
            let got = ExcelDateTime::unix_time_to_iso8601(unix_time);
            assert_eq!(expected, got);
        }
    }

    #[test]
    fn from_timestamp_to_rfc3339_dates() {
        let tests = [
            ("1970-01-01T00:00:00Z", 0),
            ("1970-02-28T00:00:00Z", 5011200),
            ("1970-03-01T00:00:00Z", 5097600),
            ("1970-12-01T00:00:00Z", 28857600),
            ("1970-12-31T00:00:00Z", 31449600),
            ("1971-01-01T00:00:00Z", 31536000),
            ("1971-02-28T00:00:00Z", 36547200),
            ("1971-03-01T00:00:00Z", 36633600),
            ("1971-12-01T00:00:00Z", 60393600),
            ("1971-12-31T00:00:00Z", 62985600),
            ("1972-01-01T00:00:00Z", 63072000),
            ("1972-01-02T00:00:00Z", 63158400),
            ("1972-02-26T00:00:00Z", 67910400),
            ("1972-02-27T00:00:00Z", 67996800),
            ("1972-02-28T00:00:00Z", 68083200),
            ("1972-02-29T00:00:00Z", 68169600),
            ("1972-03-01T00:00:00Z", 68256000),
            ("1972-12-01T00:00:00Z", 92016000),
            ("1972-12-30T00:00:00Z", 94521600),
            ("1972-12-31T00:00:00Z", 94608000),
            ("1973-01-01T00:00:00Z", 94694400),
            ("1973-02-28T00:00:00Z", 99705600),
            ("1973-03-01T00:00:00Z", 99792000),
            ("1973-12-01T00:00:00Z", 123552000),
            ("1973-12-31T00:00:00Z", 126144000),
            ("1974-01-01T00:00:00Z", 126230400),
            ("1974-02-28T00:00:00Z", 131241600),
            ("1974-03-01T00:00:00Z", 131328000),
            ("1974-12-01T00:00:00Z", 155088000),
            ("1974-12-31T00:00:00Z", 157680000),
            ("2000-01-01T00:00:00Z", 946684800),
            ("2000-02-28T00:00:00Z", 951696000),
            ("2000-02-29T00:00:00Z", 951782400),
            ("2000-03-01T00:00:00Z", 951868800),
            ("2000-12-01T00:00:00Z", 975628800),
            ("2000-12-31T00:00:00Z", 978220800),
            ("2001-01-01T00:00:00Z", 978307200),
            ("2001-02-28T00:00:00Z", 983318400),
            ("2001-03-01T00:00:00Z", 983404800),
            ("2001-12-01T00:00:00Z", 1007164800),
            ("2001-12-31T00:00:00Z", 1009756800),
            ("2002-01-01T00:00:00Z", 1009843200),
            ("2002-02-28T00:00:00Z", 1014854400),
            ("2002-03-01T00:00:00Z", 1014940800),
            ("2002-12-01T00:00:00Z", 1038700800),
            ("2002-12-31T00:00:00Z", 1041292800),
            ("2003-01-01T00:00:00Z", 1041379200),
            ("2003-02-28T00:00:00Z", 1046390400),
            ("2003-03-01T00:00:00Z", 1046476800),
            ("2003-12-01T00:00:00Z", 1070236800),
            ("2003-12-31T00:00:00Z", 1072828800),
            ("2004-01-01T00:00:00Z", 1072915200),
            ("2004-02-28T00:00:00Z", 1077926400),
            ("2004-02-29T00:00:00Z", 1078012800),
            ("2004-03-01T00:00:00Z", 1078099200),
            ("2004-12-01T00:00:00Z", 1101859200),
            ("2004-12-31T00:00:00Z", 1104451200),
            ("2099-01-01T00:00:00Z", 4070908800),
            ("2099-02-28T00:00:00Z", 4075920000),
            ("2099-03-01T00:00:00Z", 4076006400),
            ("2099-12-01T00:00:00Z", 4099766400),
            ("2099-12-30T00:00:00Z", 4102272000),
            ("2099-12-31T00:00:00Z", 4102358400),
            ("2100-01-01T00:00:00Z", 4102444800),
            ("2100-02-28T00:00:00Z", 4107456000),
            ("2100-03-01T00:00:00Z", 4107542400),
            ("2100-12-01T00:00:00Z", 4131302400),
            ("2100-12-31T00:00:00Z", 4133894400),
            ("2399-01-01T00:00:00Z", 13537929600),
            ("2399-02-28T00:00:00Z", 13542940800),
            ("2399-03-01T00:00:00Z", 13543027200),
            ("2399-12-01T00:00:00Z", 13566787200),
            ("2399-12-31T00:00:00Z", 13569379200),
            ("2400-01-01T00:00:00Z", 13569465600),
            ("2400-02-28T00:00:00Z", 13574476800),
            ("2400-03-01T00:00:00Z", 13574649600),
            ("2400-12-01T00:00:00Z", 13598409600),
            ("2400-12-31T00:00:00Z", 13601001600),
            ("2404-01-01T00:00:00Z", 13695696000),
            ("2404-01-02T00:00:00Z", 13695782400),
            ("2404-01-03T00:00:00Z", 13695868800),
            ("2404-02-28T00:00:00Z", 13700707200),
            ("2404-02-29T00:00:00Z", 13700793600),
            ("2404-03-01T00:00:00Z", 13700880000),
            ("2404-12-01T00:00:00Z", 13724640000),
            ("2404-12-31T00:00:00Z", 13727232000),
            ("4000-01-01T00:00:00Z", 64060588800),
            ("4000-02-28T00:00:00Z", 64065600000),
            ("4000-02-29T00:00:00Z", 64065686400),
            ("4000-03-01T00:00:00Z", 64065772800),
            ("4000-12-01T00:00:00Z", 64089532800),
            ("4000-12-31T00:00:00Z", 64092124800),
            ("9999-01-01T00:00:00Z", 253370764800),
            ("9999-02-28T00:00:00Z", 253375776000),
            ("9999-03-01T00:00:00Z", 253375862400),
            ("9999-12-01T00:00:00Z", 253399622400),
            ("9999-12-31T00:00:00Z", 253402214400),
        ];

        for (expected, unix_time) in tests {
            let datetime = ExcelDateTime::from_timestamp(unix_time).unwrap();
            let got = datetime.to_rfc3339();
            assert_eq!(expected, got);
        }
    }

    #[cfg(feature = "chrono")]
    #[test]
    fn dates_against_chrono() {
        use chrono::TimeZone;

        for year in 1970..=9999 {
            let dt = chrono::Utc.with_ymd_and_hms(year, 1, 1, 0, 0, 0).unwrap();
            let expected = dt.to_rfc3339_opts(chrono::SecondsFormat::Secs, true);
            let unix_time = dt.timestamp();
            let got = ExcelDateTime::unix_time_to_iso8601(unix_time as u64);
            assert_eq!(expected, got);

            let dt = chrono::Utc.with_ymd_and_hms(year, 1, 2, 0, 0, 0).unwrap();
            let expected = dt.to_rfc3339_opts(chrono::SecondsFormat::Secs, true);
            let unix_time = dt.timestamp();
            let got = ExcelDateTime::unix_time_to_iso8601(unix_time as u64);
            assert_eq!(expected, got);

            let dt = chrono::Utc.with_ymd_and_hms(year, 2, 28, 0, 0, 0).unwrap();
            let expected = dt.to_rfc3339_opts(chrono::SecondsFormat::Secs, true);
            let unix_time = dt.timestamp();
            let got = ExcelDateTime::unix_time_to_iso8601(unix_time as u64);
            assert_eq!(expected, got);

            let dt = chrono::Utc.with_ymd_and_hms(year, 3, 1, 0, 0, 0).unwrap();
            let expected = dt.to_rfc3339_opts(chrono::SecondsFormat::Secs, true);
            let unix_time = dt.timestamp();
            let got = ExcelDateTime::unix_time_to_iso8601(unix_time as u64);
            assert_eq!(expected, got);

            let dt = chrono::Utc.with_ymd_and_hms(year, 12, 31, 0, 0, 0).unwrap();
            let expected = dt.to_rfc3339_opts(chrono::SecondsFormat::Secs, true);
            let unix_time = dt.timestamp();
            let got = ExcelDateTime::unix_time_to_iso8601(unix_time as u64);
            assert_eq!(expected, got);
        }
    }

    #[cfg(feature = "chrono")]
    #[test]
    fn times_against_chrono() {
        use chrono::TimeZone;

        for timestamp in 0..=86_4000 {
            let dt = chrono::Utc.timestamp_opt(timestamp, 0).unwrap();
            let expected = dt.to_rfc3339_opts(chrono::SecondsFormat::Secs, true);
            let unix_time = dt.timestamp();
            let got = ExcelDateTime::unix_time_to_iso8601(unix_time as u64);
            assert_eq!(expected, got);
        }
    }

    #[cfg(feature = "chrono")]
    #[test]
    fn days_against_chrono() {
        use chrono::TimeZone;

        for days in 0..=366 * 4 {
            let timestamp = days * 86_400;
            let dt = chrono::Utc.timestamp_opt(timestamp, 0).unwrap();
            let expected = dt.to_rfc3339_opts(chrono::SecondsFormat::Secs, true);
            let unix_time = dt.timestamp();
            let got = ExcelDateTime::unix_time_to_iso8601(unix_time as u64);
            assert_eq!(expected, got);
        }
    }

    #[cfg(feature = "chrono")]
    #[test]
    fn chrono_dates_and_times() {
        // Test date and time
        #[allow(clippy::excessive_precision)]
        let datetimes = vec![
            (1899, 12, 31, 0, 0, 0, 0, 0.0),
            (1982, 8, 25, 0, 15, 20, 213, 30188.010650613425),
            (2065, 4, 19, 0, 16, 48, 290, 60376.011670023145),
            (2147, 12, 15, 0, 55, 25, 446, 90565.038488958337),
            (2230, 8, 10, 1, 2, 46, 891, 120753.04359827546),
            (2313, 4, 6, 1, 4, 15, 597, 150942.04462496529),
            (2395, 11, 30, 1, 9, 40, 889, 181130.04838991899),
            (2478, 7, 25, 1, 11, 32, 560, 211318.04968240741),
            (2561, 3, 21, 1, 30, 19, 169, 241507.06272186342),
            (2643, 11, 15, 1, 48, 25, 580, 271695.07529606484),
            (2726, 7, 12, 2, 3, 31, 919, 301884.08578609955),
            (2809, 3, 6, 2, 11, 11, 986, 332072.09111094906),
            (2891, 10, 31, 2, 24, 37, 95, 362261.10042934027),
            (2974, 6, 26, 2, 35, 7, 220, 392449.10772245371),
            (3057, 2, 19, 2, 45, 12, 109, 422637.1147234838),
            (3139, 10, 17, 3, 6, 39, 990, 452826.12962951389),
            (3222, 6, 11, 3, 8, 8, 251, 483014.13065105322),
            (3305, 2, 5, 3, 19, 12, 576, 513203.13834),
            (3387, 10, 1, 3, 29, 42, 574, 543391.14563164348),
            (3470, 5, 27, 3, 37, 30, 813, 573579.15105107636),
            (3553, 1, 21, 4, 14, 38, 231, 603768.17683137732),
            (3635, 9, 16, 4, 16, 28, 559, 633956.17810832174),
            (3718, 5, 13, 4, 17, 58, 222, 664145.17914608796),
            (3801, 1, 6, 4, 21, 41, 794, 694333.18173372687),
            (3883, 9, 2, 4, 56, 35, 792, 724522.20596981479),
            (3966, 4, 28, 5, 25, 14, 885, 754710.2258667245),
            (4048, 12, 21, 5, 26, 5, 724, 784898.22645513888),
            (4131, 8, 18, 5, 46, 44, 68, 815087.24078782403),
            (4214, 4, 13, 5, 48, 1, 141, 845275.24167987274),
            (4296, 12, 7, 5, 53, 52, 315, 875464.24574438657),
            (4379, 8, 3, 6, 14, 48, 580, 905652.26028449077),
            (4462, 3, 28, 6, 46, 15, 738, 935840.28212659725),
            (4544, 11, 22, 7, 31, 20, 407, 966029.31343063654),
            (4627, 7, 19, 7, 58, 33, 754, 996217.33233511576),
            (4710, 3, 15, 8, 7, 43, 130, 1026406.3386936343),
            (4792, 11, 7, 8, 29, 11, 91, 1056594.3536005903),
            (4875, 7, 4, 9, 8, 15, 328, 1086783.3807329629),
            (4958, 2, 27, 9, 30, 41, 781, 1116971.3963169097),
            (5040, 10, 23, 9, 34, 4, 462, 1147159.3986627546),
            (5123, 6, 20, 9, 37, 23, 945, 1177348.4009715857),
            (5206, 2, 12, 9, 37, 56, 655, 1207536.4013501736),
            (5288, 10, 8, 9, 45, 12, 230, 1237725.406391551),
            (5371, 6, 4, 9, 54, 14, 782, 1267913.412671088),
            (5454, 1, 28, 9, 54, 22, 108, 1298101.4127558796),
            (5536, 9, 24, 10, 1, 36, 151, 1328290.4177795255),
            (5619, 5, 20, 12, 9, 48, 602, 1358478.5068125231),
            (5702, 1, 14, 12, 34, 8, 549, 1388667.5237100578),
            (5784, 9, 8, 12, 56, 6, 495, 1418855.5389640625),
            (5867, 5, 6, 12, 58, 58, 217, 1449044.5409515856),
            (5949, 12, 30, 12, 59, 54, 263, 1479232.5416002662),
            (6032, 8, 24, 13, 34, 41, 331, 1509420.5657561459),
            (6115, 4, 21, 13, 58, 28, 601, 1539609.5822754744),
            (6197, 12, 14, 14, 2, 16, 899, 1569797.5849178126),
            (6280, 8, 10, 14, 36, 17, 444, 1599986.6085352316),
            (6363, 4, 6, 14, 37, 57, 451, 1630174.60969272),
            (6445, 11, 30, 14, 57, 42, 757, 1660363.6234115392),
            (6528, 7, 26, 15, 10, 48, 307, 1690551.6325035533),
            (6611, 3, 22, 15, 14, 39, 890, 1720739.635183912),
            (6693, 11, 15, 15, 19, 47, 988, 1750928.6387498612),
            (6776, 7, 11, 16, 4, 24, 344, 1781116.6697262037),
            (6859, 3, 7, 16, 22, 23, 952, 1811305.6822216667),
            (6941, 10, 31, 16, 29, 55, 999, 1841493.6874536921),
            (7024, 6, 26, 16, 58, 20, 259, 1871681.7071789235),
            (7107, 2, 21, 17, 4, 2, 415, 1901870.7111390624),
            (7189, 10, 16, 17, 18, 29, 630, 1932058.7211762732),
            (7272, 6, 11, 17, 47, 21, 323, 1962247.7412190163),
            (7355, 2, 5, 17, 53, 29, 866, 1992435.7454845603),
            (7437, 10, 2, 17, 53, 41, 76, 2022624.7456143056),
            (7520, 5, 28, 17, 55, 6, 44, 2052812.7465977315),
            (7603, 1, 21, 18, 14, 49, 151, 2083000.7602910995),
            (7685, 9, 16, 18, 17, 45, 738, 2113189.7623349307),
            (7768, 5, 12, 18, 29, 59, 700, 2143377.7708298611),
            (7851, 1, 7, 18, 33, 21, 233, 2173566.773162419),
            (7933, 9, 2, 19, 14, 24, 673, 2203754.8016744559),
            (8016, 4, 27, 19, 17, 12, 816, 2233942.8036205554),
            (8098, 12, 22, 19, 23, 36, 418, 2264131.8080603937),
            (8181, 8, 17, 19, 46, 25, 908, 2294319.8239109721),
            (8264, 4, 13, 20, 7, 47, 314, 2324508.8387420601),
            (8346, 12, 8, 20, 31, 37, 603, 2354696.855296331),
            (8429, 8, 3, 20, 39, 57, 770, 2384885.8610853008),
            (8512, 3, 29, 20, 50, 17, 67, 2415073.8682530904),
            (8594, 11, 22, 21, 2, 57, 827, 2445261.8770581828),
            (8677, 7, 19, 21, 23, 5, 519, 2475450.8910360998),
            (8760, 3, 14, 21, 34, 49, 572, 2505638.8991848612),
            (8842, 11, 8, 21, 39, 5, 944, 2535827.9021521294),
            (8925, 7, 4, 21, 39, 18, 426, 2566015.9022965971),
            (9008, 2, 28, 21, 46, 7, 769, 2596203.9070343636),
            (9090, 10, 24, 21, 57, 55, 662, 2626392.9152275696),
            (9173, 6, 19, 22, 19, 11, 732, 2656580.9299968979),
            (9256, 2, 13, 22, 23, 51, 376, 2686769.9332335186),
            (9338, 10, 9, 22, 27, 58, 771, 2716957.9360968866),
            (9421, 6, 5, 22, 43, 30, 392, 2747146.9468795368),
            (9504, 1, 30, 22, 48, 25, 834, 2777334.9502990046),
            (9586, 9, 24, 22, 53, 51, 727, 2807522.9540709145),
            (9669, 5, 20, 23, 12, 56, 536, 2837711.9673210187),
            (9752, 1, 14, 23, 15, 54, 109, 2867899.9693762613),
            (9834, 9, 10, 23, 17, 12, 632, 2898088.9702850925),
            (9999, 12, 31, 23, 59, 59, 0, 2958465.999988426),
        ];

        for test_data in datetimes {
            let (year, month, day, hour, min, seconds, millis, expected) = test_data;
            let datetime = chrono::NaiveDate::from_ymd_opt(year, month, day)
                .unwrap()
                .and_hms_milli_opt(hour, min, seconds, millis)
                .unwrap();
            assert_eq!(expected, ExcelDateTime::chrono_datetime_to_excel(&datetime));
        }
    }

    #[cfg(feature = "chrono")]
    #[test]
    fn chrono_dates_only() {
        // Test date only.
        let dates = vec![
            (1899, 12, 31, 0.0),
            (1900, 1, 1, 1.0),
            (1900, 2, 27, 58.0),
            (1900, 2, 28, 59.0),
            (1900, 3, 1, 61.0),
            (1900, 3, 2, 62.0),
            (1900, 3, 11, 71.0),
            (1900, 4, 8, 99.0),
            (1900, 9, 12, 256.0),
            (1901, 5, 3, 489.0),
            (1901, 10, 13, 652.0),
            (1902, 2, 15, 777.0),
            (1902, 6, 6, 888.0),
            (1902, 9, 25, 999.0),
            (1902, 9, 27, 1001.0),
            (1903, 4, 26, 1212.0),
            (1903, 8, 5, 1313.0),
            (1903, 12, 31, 1461.0),
            (1904, 1, 1, 1462.0),
            (1904, 2, 28, 1520.0),
            (1904, 2, 29, 1521.0),
            (1904, 3, 1, 1522.0),
            (1907, 2, 27, 2615.0),
            (1907, 2, 28, 2616.0),
            (1907, 3, 1, 2617.0),
            (1907, 3, 2, 2618.0),
            (1907, 3, 3, 2619.0),
            (1907, 3, 4, 2620.0),
            (1907, 3, 5, 2621.0),
            (1907, 3, 6, 2622.0),
            (1999, 1, 1, 36161.0),
            (1999, 1, 31, 36191.0),
            (1999, 2, 1, 36192.0),
            (1999, 2, 28, 36219.0),
            (1999, 3, 1, 36220.0),
            (1999, 3, 31, 36250.0),
            (1999, 4, 1, 36251.0),
            (1999, 4, 30, 36280.0),
            (1999, 5, 1, 36281.0),
            (1999, 5, 31, 36311.0),
            (1999, 6, 1, 36312.0),
            (1999, 6, 30, 36341.0),
            (1999, 7, 1, 36342.0),
            (1999, 7, 31, 36372.0),
            (1999, 8, 1, 36373.0),
            (1999, 8, 31, 36403.0),
            (1999, 9, 1, 36404.0),
            (1999, 9, 30, 36433.0),
            (1999, 10, 1, 36434.0),
            (1999, 10, 31, 36464.0),
            (1999, 11, 1, 36465.0),
            (1999, 11, 30, 36494.0),
            (1999, 12, 1, 36495.0),
            (1999, 12, 31, 36525.0),
            (2000, 1, 1, 36526.0),
            (2000, 1, 31, 36556.0),
            (2000, 2, 1, 36557.0),
            (2000, 2, 29, 36585.0),
            (2000, 3, 1, 36586.0),
            (2000, 3, 31, 36616.0),
            (2000, 4, 1, 36617.0),
            (2000, 4, 30, 36646.0),
            (2000, 5, 1, 36647.0),
            (2000, 5, 31, 36677.0),
            (2000, 6, 1, 36678.0),
            (2000, 6, 30, 36707.0),
            (2000, 7, 1, 36708.0),
            (2000, 7, 31, 36738.0),
            (2000, 8, 1, 36739.0),
            (2000, 8, 31, 36769.0),
            (2000, 9, 1, 36770.0),
            (2000, 9, 30, 36799.0),
            (2000, 10, 1, 36800.0),
            (2000, 10, 31, 36830.0),
            (2000, 11, 1, 36831.0),
            (2000, 11, 30, 36860.0),
            (2000, 12, 1, 36861.0),
            (2000, 12, 31, 36891.0),
            (2001, 1, 1, 36892.0),
            (2001, 1, 31, 36922.0),
            (2001, 2, 1, 36923.0),
            (2001, 2, 28, 36950.0),
            (2001, 3, 1, 36951.0),
            (2001, 3, 31, 36981.0),
            (2001, 4, 1, 36982.0),
            (2001, 4, 30, 37011.0),
            (2001, 5, 1, 37012.0),
            (2001, 5, 31, 37042.0),
            (2001, 6, 1, 37043.0),
            (2001, 6, 30, 37072.0),
            (2001, 7, 1, 37073.0),
            (2001, 7, 31, 37103.0),
            (2001, 8, 1, 37104.0),
            (2001, 8, 31, 37134.0),
            (2001, 9, 1, 37135.0),
            (2001, 9, 30, 37164.0),
            (2001, 10, 1, 37165.0),
            (2001, 10, 31, 37195.0),
            (2001, 11, 1, 37196.0),
            (2001, 11, 30, 37225.0),
            (2001, 12, 1, 37226.0),
            (2001, 12, 31, 37256.0),
            (2400, 1, 1, 182623.0),
            (2400, 1, 31, 182653.0),
            (2400, 2, 1, 182654.0),
            (2400, 2, 29, 182682.0),
            (2400, 3, 1, 182683.0),
            (2400, 3, 31, 182713.0),
            (2400, 4, 1, 182714.0),
            (2400, 4, 30, 182743.0),
            (2400, 5, 1, 182744.0),
            (2400, 5, 31, 182774.0),
            (2400, 6, 1, 182775.0),
            (2400, 6, 30, 182804.0),
            (2400, 7, 1, 182805.0),
            (2400, 7, 31, 182835.0),
            (2400, 8, 1, 182836.0),
            (2400, 8, 31, 182866.0),
            (2400, 9, 1, 182867.0),
            (2400, 9, 30, 182896.0),
            (2400, 10, 1, 182897.0),
            (2400, 10, 31, 182927.0),
            (2400, 11, 1, 182928.0),
            (2400, 11, 30, 182957.0),
            (2400, 12, 1, 182958.0),
            (2400, 12, 31, 182988.0),
            (4000, 1, 1, 767011.0),
            (4000, 1, 31, 767041.0),
            (4000, 2, 1, 767042.0),
            (4000, 2, 29, 767070.0),
            (4000, 3, 1, 767071.0),
            (4000, 3, 31, 767101.0),
            (4000, 4, 1, 767102.0),
            (4000, 4, 30, 767131.0),
            (4000, 5, 1, 767132.0),
            (4000, 5, 31, 767162.0),
            (4000, 6, 1, 767163.0),
            (4000, 6, 30, 767192.0),
            (4000, 7, 1, 767193.0),
            (4000, 7, 31, 767223.0),
            (4000, 8, 1, 767224.0),
            (4000, 8, 31, 767254.0),
            (4000, 9, 1, 767255.0),
            (4000, 9, 30, 767284.0),
            (4000, 10, 1, 767285.0),
            (4000, 10, 31, 767315.0),
            (4000, 11, 1, 767316.0),
            (4000, 11, 30, 767345.0),
            (4000, 12, 1, 767346.0),
            (4000, 12, 31, 767376.0),
            (4321, 1, 1, 884254.0),
            (4321, 1, 31, 884284.0),
            (4321, 2, 1, 884285.0),
            (4321, 2, 28, 884312.0),
            (4321, 3, 1, 884313.0),
            (4321, 3, 31, 884343.0),
            (4321, 4, 1, 884344.0),
            (4321, 4, 30, 884373.0),
            (4321, 5, 1, 884374.0),
            (4321, 5, 31, 884404.0),
            (4321, 6, 1, 884405.0),
            (4321, 6, 30, 884434.0),
            (4321, 7, 1, 884435.0),
            (4321, 7, 31, 884465.0),
            (4321, 8, 1, 884466.0),
            (4321, 8, 31, 884496.0),
            (4321, 9, 1, 884497.0),
            (4321, 9, 30, 884526.0),
            (4321, 10, 1, 884527.0),
            (4321, 10, 31, 884557.0),
            (4321, 11, 1, 884558.0),
            (4321, 11, 30, 884587.0),
            (4321, 12, 1, 884588.0),
            (4321, 12, 31, 884618.0),
            (9999, 1, 1, 2958101.0),
            (9999, 1, 31, 2958131.0),
            (9999, 2, 1, 2958132.0),
            (9999, 2, 28, 2958159.0),
            (9999, 3, 1, 2958160.0),
            (9999, 3, 31, 2958190.0),
            (9999, 4, 1, 2958191.0),
            (9999, 4, 30, 2958220.0),
            (9999, 5, 1, 2958221.0),
            (9999, 5, 31, 2958251.0),
            (9999, 6, 1, 2958252.0),
            (9999, 6, 30, 2958281.0),
            (9999, 7, 1, 2958282.0),
            (9999, 7, 31, 2958312.0),
            (9999, 8, 1, 2958313.0),
            (9999, 8, 31, 2958343.0),
            (9999, 9, 1, 2958344.0),
            (9999, 9, 30, 2958373.0),
            (9999, 10, 1, 2958374.0),
            (9999, 10, 31, 2958404.0),
            (9999, 11, 1, 2958405.0),
            (9999, 11, 30, 2958434.0),
            (9999, 12, 1, 2958435.0),
            (9999, 12, 31, 2958465.0),
        ];

        for test_data in dates {
            let (year, month, day, expected) = test_data;
            let datetime = chrono::NaiveDate::from_ymd_opt(year, month, day).unwrap();
            assert_eq!(expected, ExcelDateTime::chrono_date_to_excel(&datetime));
        }
    }

    #[cfg(feature = "chrono")]
    #[test]
    fn chrono_times_only() {
        // Test time only.
        #[allow(clippy::excessive_precision)]
        let times = vec![
            (0, 0, 0, 0, 0.0),
            (0, 15, 20, 213, 1.0650613425925924E-2),
            (0, 16, 48, 290, 1.1670023148148148E-2),
            (0, 55, 25, 446, 3.8488958333333337E-2),
            (1, 2, 46, 891, 4.3598275462962965E-2),
            (1, 4, 15, 597, 4.4624965277777782E-2),
            (1, 9, 40, 889, 4.8389918981481483E-2),
            (1, 11, 32, 560, 4.9682407407407404E-2),
            (1, 30, 19, 169, 6.2721863425925936E-2),
            (1, 48, 25, 580, 7.5296064814814809E-2),
            (2, 3, 31, 919, 8.5786099537037031E-2),
            (2, 11, 11, 986, 9.1110949074074077E-2),
            (2, 24, 37, 95, 0.10042934027777778),
            (2, 35, 7, 220, 0.1077224537037037),
            (2, 45, 12, 109, 0.11472348379629631),
            (3, 6, 39, 990, 0.12962951388888888),
            (3, 8, 8, 251, 0.13065105324074075),
            (3, 19, 12, 576, 0.13833999999999999),
            (3, 29, 42, 574, 0.14563164351851851),
            (3, 37, 30, 813, 0.1510510763888889),
            (4, 14, 38, 231, 0.1768313773148148),
            (4, 16, 28, 559, 0.17810832175925925),
            (4, 17, 58, 222, 0.17914608796296297),
            (4, 21, 41, 794, 0.18173372685185185),
            (4, 56, 35, 792, 0.2059698148148148),
            (5, 25, 14, 885, 0.22586672453703704),
            (5, 26, 5, 724, 0.22645513888888891),
            (5, 46, 44, 68, 0.24078782407407406),
            (5, 48, 1, 141, 0.2416798726851852),
            (5, 53, 52, 315, 0.24574438657407408),
            (6, 14, 48, 580, 0.26028449074074073),
            (6, 46, 15, 738, 0.28212659722222222),
            (7, 31, 20, 407, 0.31343063657407405),
            (7, 58, 33, 754, 0.33233511574074076),
            (8, 7, 43, 130, 0.33869363425925925),
            (8, 29, 11, 91, 0.35360059027777774),
            (9, 8, 15, 328, 0.380732962962963),
            (9, 30, 41, 781, 0.39631690972222228),
            (9, 34, 4, 462, 0.39866275462962958),
            (9, 37, 23, 945, 0.40097158564814817),
            (9, 37, 56, 655, 0.40135017361111114),
            (9, 45, 12, 230, 0.40639155092592594),
            (9, 54, 14, 782, 0.41267108796296298),
            (9, 54, 22, 108, 0.41275587962962962),
            (10, 1, 36, 151, 0.41777952546296299),
            (12, 9, 48, 602, 0.50681252314814818),
            (12, 34, 8, 549, 0.52371005787037039),
            (12, 56, 6, 495, 0.53896406249999995),
            (12, 58, 58, 217, 0.54095158564814816),
            (12, 59, 54, 263, 0.54160026620370372),
            (13, 34, 41, 331, 0.56575614583333333),
            (13, 58, 28, 601, 0.58227547453703699),
            (14, 2, 16, 899, 0.58491781249999997),
            (14, 36, 17, 444, 0.60853523148148148),
            (14, 37, 57, 451, 0.60969271990740748),
            (14, 57, 42, 757, 0.6234115393518519),
            (15, 10, 48, 307, 0.6325035532407407),
            (15, 14, 39, 890, 0.63518391203703706),
            (15, 19, 47, 988, 0.63874986111111109),
            (16, 4, 24, 344, 0.66972620370370362),
            (16, 22, 23, 952, 0.68222166666666662),
            (16, 29, 55, 999, 0.6874536921296297),
            (16, 58, 20, 259, 0.70717892361111112),
            (17, 4, 2, 415, 0.71113906250000003),
            (17, 18, 29, 630, 0.72117627314814825),
            (17, 47, 21, 323, 0.74121901620370367),
            (17, 53, 29, 866, 0.74548456018518516),
            (17, 53, 41, 76, 0.74561430555555563),
            (17, 55, 6, 44, 0.74659773148148145),
            (18, 14, 49, 151, 0.760291099537037),
            (18, 17, 45, 738, 0.76233493055555546),
            (18, 29, 59, 700, 0.77082986111111118),
            (18, 33, 21, 233, 0.77316241898148153),
            (19, 14, 24, 673, 0.80167445601851861),
            (19, 17, 12, 816, 0.80362055555555545),
            (19, 23, 36, 418, 0.80806039351851855),
            (19, 46, 25, 908, 0.82391097222222232),
            (20, 7, 47, 314, 0.83874206018518516),
            (20, 31, 37, 603, 0.85529633101851854),
            (20, 39, 57, 770, 0.86108530092592594),
            (20, 50, 17, 67, 0.86825309027777775),
            (21, 2, 57, 827, 0.87705818287037041),
            (21, 23, 5, 519, 0.891036099537037),
            (21, 34, 49, 572, 0.89918486111111118),
            (21, 39, 5, 944, 0.90215212962962965),
            (21, 39, 18, 426, 0.90229659722222222),
            (21, 46, 7, 769, 0.90703436342592603),
            (21, 57, 55, 662, 0.91522756944444439),
            (22, 19, 11, 732, 0.92999689814814823),
            (22, 23, 51, 376, 0.93323351851851843),
            (22, 27, 58, 771, 0.93609688657407408),
            (22, 43, 30, 392, 0.94687953703703709),
            (22, 48, 25, 834, 0.95029900462962968),
            (22, 53, 51, 727, 0.95407091435185187),
            (23, 12, 56, 536, 0.96732101851851848),
            (23, 15, 54, 109, 0.96937626157407408),
            (23, 17, 12, 632, 0.97028509259259266),
            (23, 59, 59, 999, 0.99999998842592586),
        ];

        for test_data in times {
            let (hour, min, seconds, millis, expected) = test_data;
            let datetime =
                chrono::NaiveTime::from_hms_milli_opt(hour, min, seconds, millis).unwrap();
            let mut diff = ExcelDateTime::chrono_time_to_excel(&datetime) - expected;
            diff = diff.abs();
            assert!(diff < 0.00000000001);
        }
    }

    #[cfg(feature = "jiff")]
    #[test]
    fn jiffy_dates_and_times() {
        // Test date and time.
        #[allow(clippy::excessive_precision)]
        let datetimes = vec![
            (1899, 12, 31, 0, 0, 0, 0, 0.0),
            (1982, 8, 25, 0, 15, 20, 213, 30188.010650613425),
            (2065, 4, 19, 0, 16, 48, 290, 60376.011670023145),
            (2147, 12, 15, 0, 55, 25, 446, 90565.038488958337),
            (2230, 8, 10, 1, 2, 46, 891, 120753.04359827546),
            (2313, 4, 6, 1, 4, 15, 597, 150942.04462496529),
            (2395, 11, 30, 1, 9, 40, 889, 181130.04838991899),
            (2478, 7, 25, 1, 11, 32, 560, 211318.04968240741),
            (2561, 3, 21, 1, 30, 19, 169, 241507.06272186342),
            (2643, 11, 15, 1, 48, 25, 580, 271695.07529606484),
            (2726, 7, 12, 2, 3, 31, 919, 301884.08578609955),
            (2809, 3, 6, 2, 11, 11, 986, 332072.09111094906),
            (2891, 10, 31, 2, 24, 37, 95, 362261.10042934027),
            (2974, 6, 26, 2, 35, 7, 220, 392449.10772245371),
            (3057, 2, 19, 2, 45, 12, 109, 422637.1147234838),
            (3139, 10, 17, 3, 6, 39, 990, 452826.12962951389),
            (3222, 6, 11, 3, 8, 8, 251, 483014.13065105322),
            (3305, 2, 5, 3, 19, 12, 576, 513203.13834),
            (3387, 10, 1, 3, 29, 42, 574, 543391.14563164348),
            (3470, 5, 27, 3, 37, 30, 813, 573579.15105107636),
            (3553, 1, 21, 4, 14, 38, 231, 603768.17683137732),
            (3635, 9, 16, 4, 16, 28, 559, 633956.17810832174),
            (3718, 5, 13, 4, 17, 58, 222, 664145.17914608796),
            (3801, 1, 6, 4, 21, 41, 794, 694333.18173372687),
            (3883, 9, 2, 4, 56, 35, 792, 724522.20596981479),
            (3966, 4, 28, 5, 25, 14, 885, 754710.2258667245),
            (4048, 12, 21, 5, 26, 5, 724, 784898.22645513888),
            (4131, 8, 18, 5, 46, 44, 68, 815087.24078782403),
            (4214, 4, 13, 5, 48, 1, 141, 845275.24167987274),
            (4296, 12, 7, 5, 53, 52, 315, 875464.24574438657),
            (4379, 8, 3, 6, 14, 48, 580, 905652.26028449077),
            (4462, 3, 28, 6, 46, 15, 738, 935840.28212659725),
            (4544, 11, 22, 7, 31, 20, 407, 966029.31343063654),
            (4627, 7, 19, 7, 58, 33, 754, 996217.33233511576),
            (4710, 3, 15, 8, 7, 43, 130, 1026406.3386936343),
            (4792, 11, 7, 8, 29, 11, 91, 1056594.3536005903),
            (4875, 7, 4, 9, 8, 15, 328, 1086783.3807329629),
            (4958, 2, 27, 9, 30, 41, 781, 1116971.3963169097),
            (5040, 10, 23, 9, 34, 4, 462, 1147159.3986627546),
            (5123, 6, 20, 9, 37, 23, 945, 1177348.4009715857),
            (5206, 2, 12, 9, 37, 56, 655, 1207536.4013501736),
            (5288, 10, 8, 9, 45, 12, 230, 1237725.406391551),
            (5371, 6, 4, 9, 54, 14, 782, 1267913.412671088),
            (5454, 1, 28, 9, 54, 22, 108, 1298101.4127558796),
            (5536, 9, 24, 10, 1, 36, 151, 1328290.4177795255),
            (5619, 5, 20, 12, 9, 48, 602, 1358478.5068125231),
            (5702, 1, 14, 12, 34, 8, 549, 1388667.5237100578),
            (5784, 9, 8, 12, 56, 6, 495, 1418855.5389640625),
            (5867, 5, 6, 12, 58, 58, 217, 1449044.5409515856),
            (5949, 12, 30, 12, 59, 54, 263, 1479232.5416002662),
            (6032, 8, 24, 13, 34, 41, 331, 1509420.5657561459),
            (6115, 4, 21, 13, 58, 28, 601, 1539609.5822754744),
            (6197, 12, 14, 14, 2, 16, 899, 1569797.5849178126),
            (6280, 8, 10, 14, 36, 17, 444, 1599986.6085352316),
            (6363, 4, 6, 14, 37, 57, 451, 1630174.60969272),
            (6445, 11, 30, 14, 57, 42, 757, 1660363.6234115392),
            (6528, 7, 26, 15, 10, 48, 307, 1690551.6325035533),
            (6611, 3, 22, 15, 14, 39, 890, 1720739.635183912),
            (6693, 11, 15, 15, 19, 47, 988, 1750928.6387498612),
            (6776, 7, 11, 16, 4, 24, 344, 1781116.6697262037),
            (6859, 3, 7, 16, 22, 23, 952, 1811305.6822216667),
            (6941, 10, 31, 16, 29, 55, 999, 1841493.6874536921),
            (7024, 6, 26, 16, 58, 20, 259, 1871681.7071789235),
            (7107, 2, 21, 17, 4, 2, 415, 1901870.7111390624),
            (7189, 10, 16, 17, 18, 29, 630, 1932058.7211762732),
            (7272, 6, 11, 17, 47, 21, 323, 1962247.7412190163),
            (7355, 2, 5, 17, 53, 29, 866, 1992435.7454845603),
            (7437, 10, 2, 17, 53, 41, 76, 2022624.7456143056),
            (7520, 5, 28, 17, 55, 6, 44, 2052812.7465977315),
            (7603, 1, 21, 18, 14, 49, 151, 2083000.7602910995),
            (7685, 9, 16, 18, 17, 45, 738, 2113189.7623349307),
            (7768, 5, 12, 18, 29, 59, 700, 2143377.7708298611),
            (7851, 1, 7, 18, 33, 21, 233, 2173566.773162419),
            (7933, 9, 2, 19, 14, 24, 673, 2203754.8016744559),
            (8016, 4, 27, 19, 17, 12, 816, 2233942.8036205554),
            (8098, 12, 22, 19, 23, 36, 418, 2264131.8080603937),
            (8181, 8, 17, 19, 46, 25, 908, 2294319.8239109721),
            (8264, 4, 13, 20, 7, 47, 314, 2324508.8387420601),
            (8346, 12, 8, 20, 31, 37, 603, 2354696.855296331),
            (8429, 8, 3, 20, 39, 57, 770, 2384885.8610853008),
            (8512, 3, 29, 20, 50, 17, 67, 2415073.8682530904),
            (8594, 11, 22, 21, 2, 57, 827, 2445261.8770581828),
            (8677, 7, 19, 21, 23, 5, 519, 2475450.8910360998),
            (8760, 3, 14, 21, 34, 49, 572, 2505638.8991848612),
            (8842, 11, 8, 21, 39, 5, 944, 2535827.9021521294),
            (8925, 7, 4, 21, 39, 18, 426, 2566015.9022965971),
            (9008, 2, 28, 21, 46, 7, 769, 2596203.9070343636),
            (9090, 10, 24, 21, 57, 55, 662, 2626392.9152275696),
            (9173, 6, 19, 22, 19, 11, 732, 2656580.9299968979),
            (9256, 2, 13, 22, 23, 51, 376, 2686769.9332335186),
            (9338, 10, 9, 22, 27, 58, 771, 2716957.9360968866),
            (9421, 6, 5, 22, 43, 30, 392, 2747146.9468795368),
            (9504, 1, 30, 22, 48, 25, 834, 2777334.9502990046),
            (9586, 9, 24, 22, 53, 51, 727, 2807522.9540709145),
            (9669, 5, 20, 23, 12, 56, 536, 2837711.9673210187),
            (9752, 1, 14, 23, 15, 54, 109, 2867899.9693762613),
            (9834, 9, 10, 23, 17, 12, 632, 2898088.9702850925),
            (9999, 12, 31, 23, 59, 59, 0, 2958465.999988426),
        ];

        for test_data in datetimes {
            let (year, month, day, hour, min, seconds, millis, expected) = test_data;
            let datetime =
                jiff::civil::date(year, month, day).at(hour, min, seconds, millis * 1_000_000);
            assert_eq!(expected, ExcelDateTime::jiff_datetime_to_excel(&datetime));
        }
    }

    #[cfg(feature = "jiff")]
    #[test]
    fn jiff_dates_only() {
        let dates = vec![
            (1899, 12, 31, 0.0),
            (1900, 1, 1, 1.0),
            (1900, 2, 27, 58.0),
            (1900, 2, 28, 59.0),
            (1900, 3, 1, 61.0),
            (1900, 3, 2, 62.0),
            (1900, 3, 11, 71.0),
            (1900, 4, 8, 99.0),
            (1900, 9, 12, 256.0),
            (1901, 5, 3, 489.0),
            (1901, 10, 13, 652.0),
            (1902, 2, 15, 777.0),
            (1902, 6, 6, 888.0),
            (1902, 9, 25, 999.0),
            (1902, 9, 27, 1001.0),
            (1903, 4, 26, 1212.0),
            (1903, 8, 5, 1313.0),
            (1903, 12, 31, 1461.0),
            (1904, 1, 1, 1462.0),
            (1904, 2, 28, 1520.0),
            (1904, 2, 29, 1521.0),
            (1904, 3, 1, 1522.0),
            (1907, 2, 27, 2615.0),
            (1907, 2, 28, 2616.0),
            (1907, 3, 1, 2617.0),
            (1907, 3, 2, 2618.0),
            (1907, 3, 3, 2619.0),
            (1907, 3, 4, 2620.0),
            (1907, 3, 5, 2621.0),
            (1907, 3, 6, 2622.0),
            (1999, 1, 1, 36161.0),
            (1999, 1, 31, 36191.0),
            (1999, 2, 1, 36192.0),
            (1999, 2, 28, 36219.0),
            (1999, 3, 1, 36220.0),
            (1999, 3, 31, 36250.0),
            (1999, 4, 1, 36251.0),
            (1999, 4, 30, 36280.0),
            (1999, 5, 1, 36281.0),
            (1999, 5, 31, 36311.0),
            (1999, 6, 1, 36312.0),
            (1999, 6, 30, 36341.0),
            (1999, 7, 1, 36342.0),
            (1999, 7, 31, 36372.0),
            (1999, 8, 1, 36373.0),
            (1999, 8, 31, 36403.0),
            (1999, 9, 1, 36404.0),
            (1999, 9, 30, 36433.0),
            (1999, 10, 1, 36434.0),
            (1999, 10, 31, 36464.0),
            (1999, 11, 1, 36465.0),
            (1999, 11, 30, 36494.0),
            (1999, 12, 1, 36495.0),
            (1999, 12, 31, 36525.0),
            (2000, 1, 1, 36526.0),
            (2000, 1, 31, 36556.0),
            (2000, 2, 1, 36557.0),
            (2000, 2, 29, 36585.0),
            (2000, 3, 1, 36586.0),
            (2000, 3, 31, 36616.0),
            (2000, 4, 1, 36617.0),
            (2000, 4, 30, 36646.0),
            (2000, 5, 1, 36647.0),
            (2000, 5, 31, 36677.0),
            (2000, 6, 1, 36678.0),
            (2000, 6, 30, 36707.0),
            (2000, 7, 1, 36708.0),
            (2000, 7, 31, 36738.0),
            (2000, 8, 1, 36739.0),
            (2000, 8, 31, 36769.0),
            (2000, 9, 1, 36770.0),
            (2000, 9, 30, 36799.0),
            (2000, 10, 1, 36800.0),
            (2000, 10, 31, 36830.0),
            (2000, 11, 1, 36831.0),
            (2000, 11, 30, 36860.0),
            (2000, 12, 1, 36861.0),
            (2000, 12, 31, 36891.0),
            (2001, 1, 1, 36892.0),
            (2001, 1, 31, 36922.0),
            (2001, 2, 1, 36923.0),
            (2001, 2, 28, 36950.0),
            (2001, 3, 1, 36951.0),
            (2001, 3, 31, 36981.0),
            (2001, 4, 1, 36982.0),
            (2001, 4, 30, 37011.0),
            (2001, 5, 1, 37012.0),
            (2001, 5, 31, 37042.0),
            (2001, 6, 1, 37043.0),
            (2001, 6, 30, 37072.0),
            (2001, 7, 1, 37073.0),
            (2001, 7, 31, 37103.0),
            (2001, 8, 1, 37104.0),
            (2001, 8, 31, 37134.0),
            (2001, 9, 1, 37135.0),
            (2001, 9, 30, 37164.0),
            (2001, 10, 1, 37165.0),
            (2001, 10, 31, 37195.0),
            (2001, 11, 1, 37196.0),
            (2001, 11, 30, 37225.0),
            (2001, 12, 1, 37226.0),
            (2001, 12, 31, 37256.0),
            (2400, 1, 1, 182623.0),
            (2400, 1, 31, 182653.0),
            (2400, 2, 1, 182654.0),
            (2400, 2, 29, 182682.0),
            (2400, 3, 1, 182683.0),
            (2400, 3, 31, 182713.0),
            (2400, 4, 1, 182714.0),
            (2400, 4, 30, 182743.0),
            (2400, 5, 1, 182744.0),
            (2400, 5, 31, 182774.0),
            (2400, 6, 1, 182775.0),
            (2400, 6, 30, 182804.0),
            (2400, 7, 1, 182805.0),
            (2400, 7, 31, 182835.0),
            (2400, 8, 1, 182836.0),
            (2400, 8, 31, 182866.0),
            (2400, 9, 1, 182867.0),
            (2400, 9, 30, 182896.0),
            (2400, 10, 1, 182897.0),
            (2400, 10, 31, 182927.0),
            (2400, 11, 1, 182928.0),
            (2400, 11, 30, 182957.0),
            (2400, 12, 1, 182958.0),
            (2400, 12, 31, 182988.0),
            (4000, 1, 1, 767011.0),
            (4000, 1, 31, 767041.0),
            (4000, 2, 1, 767042.0),
            (4000, 2, 29, 767070.0),
            (4000, 3, 1, 767071.0),
            (4000, 3, 31, 767101.0),
            (4000, 4, 1, 767102.0),
            (4000, 4, 30, 767131.0),
            (4000, 5, 1, 767132.0),
            (4000, 5, 31, 767162.0),
            (4000, 6, 1, 767163.0),
            (4000, 6, 30, 767192.0),
            (4000, 7, 1, 767193.0),
            (4000, 7, 31, 767223.0),
            (4000, 8, 1, 767224.0),
            (4000, 8, 31, 767254.0),
            (4000, 9, 1, 767255.0),
            (4000, 9, 30, 767284.0),
            (4000, 10, 1, 767285.0),
            (4000, 10, 31, 767315.0),
            (4000, 11, 1, 767316.0),
            (4000, 11, 30, 767345.0),
            (4000, 12, 1, 767346.0),
            (4000, 12, 31, 767376.0),
            (4321, 1, 1, 884254.0),
            (4321, 1, 31, 884284.0),
            (4321, 2, 1, 884285.0),
            (4321, 2, 28, 884312.0),
            (4321, 3, 1, 884313.0),
            (4321, 3, 31, 884343.0),
            (4321, 4, 1, 884344.0),
            (4321, 4, 30, 884373.0),
            (4321, 5, 1, 884374.0),
            (4321, 5, 31, 884404.0),
            (4321, 6, 1, 884405.0),
            (4321, 6, 30, 884434.0),
            (4321, 7, 1, 884435.0),
            (4321, 7, 31, 884465.0),
            (4321, 8, 1, 884466.0),
            (4321, 8, 31, 884496.0),
            (4321, 9, 1, 884497.0),
            (4321, 9, 30, 884526.0),
            (4321, 10, 1, 884527.0),
            (4321, 10, 31, 884557.0),
            (4321, 11, 1, 884558.0),
            (4321, 11, 30, 884587.0),
            (4321, 12, 1, 884588.0),
            (4321, 12, 31, 884618.0),
            (9999, 1, 1, 2958101.0),
            (9999, 1, 31, 2958131.0),
            (9999, 2, 1, 2958132.0),
            (9999, 2, 28, 2958159.0),
            (9999, 3, 1, 2958160.0),
            (9999, 3, 31, 2958190.0),
            (9999, 4, 1, 2958191.0),
            (9999, 4, 30, 2958220.0),
            (9999, 5, 1, 2958221.0),
            (9999, 5, 31, 2958251.0),
            (9999, 6, 1, 2958252.0),
            (9999, 6, 30, 2958281.0),
            (9999, 7, 1, 2958282.0),
            (9999, 7, 31, 2958312.0),
            (9999, 8, 1, 2958313.0),
            (9999, 8, 31, 2958343.0),
            (9999, 9, 1, 2958344.0),
            (9999, 9, 30, 2958373.0),
            (9999, 10, 1, 2958374.0),
            (9999, 10, 31, 2958404.0),
            (9999, 11, 1, 2958405.0),
            (9999, 11, 30, 2958434.0),
            (9999, 12, 1, 2958435.0),
            (9999, 12, 31, 2958465.0),
        ];

        for test_data in dates {
            let (year, month, day, expected) = test_data;

            let d = jiff::civil::date(year, month, day);
            assert_eq!(ExcelDateTime::jiff_date_to_excel(&d), expected);
        }
    }

    #[cfg(feature = "jiff")]
    #[test]
    fn jiff_times_only() {
        // Test time only.
        #[allow(clippy::excessive_precision)]
        let times = vec![
            (0, 0, 0, 0, 0.0),
            (0, 15, 20, 213, 1.0650613425925924E-2),
            (0, 16, 48, 290, 1.1670023148148148E-2),
            (0, 55, 25, 446, 3.8488958333333337E-2),
            (1, 2, 46, 891, 4.3598275462962965E-2),
            (1, 4, 15, 597, 4.4624965277777782E-2),
            (1, 9, 40, 889, 4.8389918981481483E-2),
            (1, 11, 32, 560, 4.9682407407407404E-2),
            (1, 30, 19, 169, 6.2721863425925936E-2),
            (1, 48, 25, 580, 7.5296064814814809E-2),
            (2, 3, 31, 919, 8.5786099537037031E-2),
            (2, 11, 11, 986, 9.1110949074074077E-2),
            (2, 24, 37, 95, 0.10042934027777778),
            (2, 35, 7, 220, 0.1077224537037037),
            (2, 45, 12, 109, 0.11472348379629631),
            (3, 6, 39, 990, 0.12962951388888888),
            (3, 8, 8, 251, 0.13065105324074075),
            (3, 19, 12, 576, 0.13833999999999999),
            (3, 29, 42, 574, 0.14563164351851851),
            (3, 37, 30, 813, 0.1510510763888889),
            (4, 14, 38, 231, 0.1768313773148148),
            (4, 16, 28, 559, 0.17810832175925925),
            (4, 17, 58, 222, 0.17914608796296297),
            (4, 21, 41, 794, 0.18173372685185185),
            (4, 56, 35, 792, 0.2059698148148148),
            (5, 25, 14, 885, 0.22586672453703704),
            (5, 26, 5, 724, 0.22645513888888891),
            (5, 46, 44, 68, 0.24078782407407406),
            (5, 48, 1, 141, 0.2416798726851852),
            (5, 53, 52, 315, 0.24574438657407408),
            (6, 14, 48, 580, 0.26028449074074073),
            (6, 46, 15, 738, 0.28212659722222222),
            (7, 31, 20, 407, 0.31343063657407405),
            (7, 58, 33, 754, 0.33233511574074076),
            (8, 7, 43, 130, 0.33869363425925925),
            (8, 29, 11, 91, 0.35360059027777774),
            (9, 8, 15, 328, 0.380732962962963),
            (9, 30, 41, 781, 0.39631690972222228),
            (9, 34, 4, 462, 0.39866275462962958),
            (9, 37, 23, 945, 0.40097158564814817),
            (9, 37, 56, 655, 0.40135017361111114),
            (9, 45, 12, 230, 0.40639155092592594),
            (9, 54, 14, 782, 0.41267108796296298),
            (9, 54, 22, 108, 0.41275587962962962),
            (10, 1, 36, 151, 0.41777952546296299),
            (12, 9, 48, 602, 0.50681252314814818),
            (12, 34, 8, 549, 0.52371005787037039),
            (12, 56, 6, 495, 0.53896406249999995),
            (12, 58, 58, 217, 0.54095158564814816),
            (12, 59, 54, 263, 0.54160026620370372),
            (13, 34, 41, 331, 0.56575614583333333),
            (13, 58, 28, 601, 0.58227547453703699),
            (14, 2, 16, 899, 0.58491781249999997),
            (14, 36, 17, 444, 0.60853523148148148),
            (14, 37, 57, 451, 0.60969271990740748),
            (14, 57, 42, 757, 0.6234115393518519),
            (15, 10, 48, 307, 0.6325035532407407),
            (15, 14, 39, 890, 0.63518391203703706),
            (15, 19, 47, 988, 0.63874986111111109),
            (16, 4, 24, 344, 0.66972620370370362),
            (16, 22, 23, 952, 0.68222166666666662),
            (16, 29, 55, 999, 0.6874536921296297),
            (16, 58, 20, 259, 0.70717892361111112),
            (17, 4, 2, 415, 0.71113906250000003),
            (17, 18, 29, 630, 0.72117627314814825),
            (17, 47, 21, 323, 0.74121901620370367),
            (17, 53, 29, 866, 0.74548456018518516),
            (17, 53, 41, 76, 0.74561430555555563),
            (17, 55, 6, 44, 0.74659773148148145),
            (18, 14, 49, 151, 0.760291099537037),
            (18, 17, 45, 738, 0.76233493055555546),
            (18, 29, 59, 700, 0.77082986111111118),
            (18, 33, 21, 233, 0.77316241898148153),
            (19, 14, 24, 673, 0.80167445601851861),
            (19, 17, 12, 816, 0.80362055555555545),
            (19, 23, 36, 418, 0.80806039351851855),
            (19, 46, 25, 908, 0.82391097222222232),
            (20, 7, 47, 314, 0.83874206018518516),
            (20, 31, 37, 603, 0.85529633101851854),
            (20, 39, 57, 770, 0.86108530092592594),
            (20, 50, 17, 67, 0.86825309027777775),
            (21, 2, 57, 827, 0.87705818287037041),
            (21, 23, 5, 519, 0.891036099537037),
            (21, 34, 49, 572, 0.89918486111111118),
            (21, 39, 5, 944, 0.90215212962962965),
            (21, 39, 18, 426, 0.90229659722222222),
            (21, 46, 7, 769, 0.90703436342592603),
            (21, 57, 55, 662, 0.91522756944444439),
            (22, 19, 11, 732, 0.92999689814814823),
            (22, 23, 51, 376, 0.93323351851851843),
            (22, 27, 58, 771, 0.93609688657407408),
            (22, 43, 30, 392, 0.94687953703703709),
            (22, 48, 25, 834, 0.95029900462962968),
            (22, 53, 51, 727, 0.95407091435185187),
            (23, 12, 56, 536, 0.96732101851851848),
            (23, 15, 54, 109, 0.96937626157407408),
            (23, 17, 12, 632, 0.97028509259259266),
            (23, 59, 59, 999, 0.99999998842592586),
        ];

        for test_data in times {
            let (hour, min, seconds, millis, expected) = test_data;
            let time = jiff::civil::time(hour, min, seconds, millis * 1_000_000);
            let mut diff = ExcelDateTime::jiff_time_to_excel(&time) - expected;
            diff = diff.abs();
            assert!(diff < 0.00000000001);
        }
    }
}
