//! The generic SHA3 implementation that uses portable or platform specific
//! sub-routines.

use crate::traits::{get_ij, set_ij, KeccakStateItem};

#[cfg_attr(hax, hax_lib::opaque)]
#[derive(Clone, Copy)]
pub(crate) struct KeccakState<const N: usize, T: KeccakStateItem<N>> {
    st: [T; 25],
}

impl<const N: usize, T: KeccakStateItem<N>> KeccakState<N, T> {
    fn get(&self, i: usize, j: usize) -> T {
        get_ij(&self.st, i, j)
    }
    fn set(&mut self, i: usize, j: usize, v: T) {
        set_ij(&mut self.st, i, j, v);
    }
}

impl<const N: usize, T: KeccakStateItem<N>> KeccakState<N, T> {
    /// Create a new Shake128 x4 state.
    #[inline(always)]
    pub(crate) fn new() -> Self {
        Self {
            st: [T::zero(); 25],
        }
    }
}

/// The internal keccak state that can also buffer inputs to absorb.
/// This is used in the general xof APIs.
#[cfg_attr(hax, hax_lib::opaque)]
pub(crate) struct KeccakXofState<
    const PARALLEL_LANES: usize,
    const RATE: usize,
    STATE: KeccakStateItem<PARALLEL_LANES>,
> {
    inner: KeccakState<PARALLEL_LANES, STATE>,

    // Buffer inputs on absorb.
    buf: [[u8; RATE]; PARALLEL_LANES],

    // Buffered length.
    buf_len: usize,

    // Needs sponge.
    sponge: bool,
}

impl<const PARALLEL_LANES: usize, const RATE: usize, STATE: KeccakStateItem<PARALLEL_LANES>>
    KeccakXofState<PARALLEL_LANES, RATE, STATE>
{
    /// An all zero block
    pub(crate) const fn zero_block() -> [u8; RATE] {
        [0u8; RATE]
    }

    /// Generate a new keccak xof state.
    pub(crate) fn new() -> Self {
        Self {
            inner: KeccakState::new(),
            buf: [Self::zero_block(); PARALLEL_LANES],
            buf_len: 0,
            sponge: false,
        }
    }

    /// Absorb
    ///
    /// This function takes any number of bytes to absorb and buffers if it's not enough.
    /// The function assumes that all input slices in `blocks` have the same length.
    ///
    /// Only a multiple of `RATE` blocks are absorbed.
    /// For the remaining bytes [`absorb_final`] needs to be called.
    ///
    /// This works best with relatively small `inputs`.
    #[inline(always)]
    pub(crate) fn absorb(&mut self, inputs: &[&[u8]; PARALLEL_LANES]) {
        let input_remainder_len = self.absorb_full(inputs);

        // ... buffer the rest if there's not enough input (left).
        if input_remainder_len > 0 {
            debug_assert!(
                self.buf_len == 0  // We consumed everything (or it was empty all along).
                 || self.buf_len + input_remainder_len <= RATE
            );

            let input_len = inputs[0].len();
            for i in 0..PARALLEL_LANES {
                self.buf[i][self.buf_len..self.buf_len + input_remainder_len]
                    .copy_from_slice(&inputs[i][input_len - input_remainder_len..]);
            }
            self.buf_len += input_remainder_len;
        }
    }

    // Note: consciously not inlining this function to avoid using too much stack
    fn absorb_full(&mut self, inputs: &[&[u8]; PARALLEL_LANES]) -> usize {
        debug_assert!(PARALLEL_LANES > 0);
        debug_assert!(self.buf_len < RATE);
        #[cfg(debug_assertions)]
        {
            for block in inputs {
                debug_assert!(block.len() == inputs[0].len());
            }
        }

        // Check if there are buffered bytes to absorb first and consume them.
        let input_consumed = self.fill_buffer(inputs);

        if input_consumed > 0 {
            let mut borrowed = [[0u8; RATE].as_slice(); PARALLEL_LANES];
            // We have a full block in the local buffer now.
            for i in 0..PARALLEL_LANES {
                borrowed[i] = &self.buf[i];
            }
            STATE::load_block::<RATE>(&mut self.inner.st, &borrowed, 0);
            keccakf1600(&mut self.inner);

            // "empty" the local buffer
            self.buf_len = 0;
        }

        // We only need to consume the rest of the input.
        let input_to_consume = inputs[0].len() - input_consumed;

        // Consume the (rest of the) input ...
        let num_blocks = input_to_consume / RATE;
        let remainder = input_to_consume % RATE;
        for i in 0..num_blocks {
            // We only get in here if `input_len / RATE > 0`.
            STATE::load_block::<RATE>(&mut self.inner.st, &inputs, input_consumed + i * RATE);
            keccakf1600(&mut self.inner);
        }

        remainder
    }

    /// Consume the internal buffer and the required amount of the input to pad to
    /// `RATE`.
    ///
    /// Returns the `consumed` bytes from `inputs` if there's enough buffered
    /// content to consume, and `0` otherwise.
    /// If `consumed > 0` is returned, `self.buf` contains a full block to be
    /// loaded.
    // Note: consciously not inlining this function to avoid using too much stack
    fn fill_buffer(&mut self, inputs: &[&[u8]; PARALLEL_LANES]) -> usize {
        let input_len = inputs[0].len();
        let mut consumed = 0;
        if self.buf_len > 0 {
            // There's something buffered internally to consume.
            if self.buf_len + input_len >= RATE {
                // We have enough data when combining the internal buffer and
                // the input.
                consumed = RATE - self.buf_len;
                for i in 0..PARALLEL_LANES {
                    self.buf[i][self.buf_len..].copy_from_slice(&inputs[i][..consumed]);
                }
                self.buf_len += consumed;
            }
        }
        consumed
    }

    /// Absorb a final block.
    ///
    /// The `inputs` block may be empty. Everything in the `inputs` block beyond
    /// `RATE` bytes is ignored.
    #[inline(always)]
    pub(crate) fn absorb_final<const DELIMITER: u8>(&mut self, inputs: &[&[u8]; PARALLEL_LANES]) {
        let input_remainder_len = self.absorb_full(inputs);

        // Consume the remaining bytes.
        // This may be in the local buffer or in the input.
        let input_len = inputs[0].len();
        let mut blocks = [[0u8; 200]; PARALLEL_LANES];
        for i in 0..PARALLEL_LANES {
            if self.buf_len > 0 {
                blocks[i][0..self.buf_len].copy_from_slice(&self.buf[i][0..self.buf_len]);
            }
            if input_remainder_len > 0 {
                blocks[i][self.buf_len..self.buf_len + input_remainder_len]
                    .copy_from_slice(&inputs[i][input_len - input_remainder_len..]);
            }
            blocks[i][self.buf_len + input_remainder_len] = DELIMITER;
            blocks[i][RATE - 1] |= 0x80;
        }

        STATE::load_block_full::<RATE>(&mut self.inner.st, &blocks, 0);
        keccakf1600(&mut self.inner);
    }

    /// Squeeze `N` x `LEN` bytes.
    #[inline(always)]
    pub(crate) fn squeeze(&mut self, out: [&mut [u8]; PARALLEL_LANES]) {
        if self.sponge {
            // If we called `squeeze` before, call f1600 first.
            // We do it this way around so that we don't call f1600 at the end
            // when we don't need it.
            keccakf1600(&mut self.inner);
        }

        // How many blocks do we need to squeeze out?
        let out_len = out[0].len();
        let blocks = out_len / RATE;
        let last = out_len - (out_len % RATE);

        // Squeeze out one to start with.
        // XXX: Eurydice does not extract `core::cmp::min`, so we do
        // this instead. (cf. https://github.com/AeneasVerif/eurydice/issues/49)
        let mid = if RATE >= out_len { out_len } else { RATE };
        let (out0, mut out_rest) = STATE::split_at_mut_n(out, mid);
        STATE::store::<RATE>(&self.inner.st, out0);

        // If we got asked for more than one block, squeeze out more.
        for _ in 1..blocks {
            // Here we know that we always have full blocks to write out.
            let (out0, tmp) = STATE::split_at_mut_n(out_rest, RATE);
            keccakf1600(&mut self.inner);
            STATE::store::<RATE>(&self.inner.st, out0);
            out_rest = tmp;
        }

        if last < out_len {
            // Squeeze out the last partial block
            keccakf1600(&mut self.inner);
            STATE::store::<RATE>(&self.inner.st, out_rest);
        }

        self.sponge = true;
    }
}

/// From here, everything is generic
///
const _ROTC: [usize; 24] = [
    1, 62, 28, 27, 36, 44, 6, 55, 20, 3, 10, 43, 25, 39, 41, 45, 15, 21, 8, 18, 2, 61, 56, 14,
];

#[inline(always)]
pub(crate) fn theta_rho<const N: usize, T: KeccakStateItem<N>>(s: &mut KeccakState<N, T>) {
    let c: [T; 5] = [
        T::xor5(
            s.get(0, 0),
            s.get(1, 0),
            s.get(2, 0),
            s.get(3, 0),
            s.get(4, 0),
        ),
        T::xor5(
            s.get(0, 1),
            s.get(1, 1),
            s.get(2, 1),
            s.get(3, 1),
            s.get(4, 1),
        ),
        T::xor5(
            s.get(0, 2),
            s.get(1, 2),
            s.get(2, 2),
            s.get(3, 2),
            s.get(4, 2),
        ),
        T::xor5(
            s.get(0, 3),
            s.get(1, 3),
            s.get(2, 3),
            s.get(3, 3),
            s.get(4, 3),
        ),
        T::xor5(
            s.get(0, 4),
            s.get(1, 4),
            s.get(2, 4),
            s.get(3, 4),
            s.get(4, 4),
        ),
    ];
    #[allow(clippy::identity_op)]
    let t: [T; 5] = [
        T::rotate_left1_and_xor(c[(0 + 4) % 5], c[(0 + 1) % 5]),
        T::rotate_left1_and_xor(c[(1 + 4) % 5], c[(1 + 1) % 5]),
        T::rotate_left1_and_xor(c[(2 + 4) % 5], c[(2 + 1) % 5]),
        T::rotate_left1_and_xor(c[(3 + 4) % 5], c[(3 + 1) % 5]),
        T::rotate_left1_and_xor(c[(4 + 4) % 5], c[(4 + 1) % 5]),
    ];

    s.set(0, 0, T::xor(s.get(0, 0), t[0]));
    s.set(1, 0, T::xor_and_rotate::<36, 28>(s.get(1, 0), t[0]));
    s.set(2, 0, T::xor_and_rotate::<3, 61>(s.get(2, 0), t[0]));
    s.set(3, 0, T::xor_and_rotate::<41, 23>(s.get(3, 0), t[0]));
    s.set(4, 0, T::xor_and_rotate::<18, 46>(s.get(4, 0), t[0]));

    s.set(0, 1, T::xor_and_rotate::<1, 63>(s.get(0, 1), t[1]));
    s.set(1, 1, T::xor_and_rotate::<44, 20>(s.get(1, 1), t[1]));
    s.set(2, 1, T::xor_and_rotate::<10, 54>(s.get(2, 1), t[1]));
    s.set(3, 1, T::xor_and_rotate::<45, 19>(s.get(3, 1), t[1]));
    s.set(4, 1, T::xor_and_rotate::<2, 62>(s.get(4, 1), t[1]));

    s.set(0, 2, T::xor_and_rotate::<62, 2>(s.get(0, 2), t[2]));
    s.set(1, 2, T::xor_and_rotate::<6, 58>(s.get(1, 2), t[2]));
    s.set(2, 2, T::xor_and_rotate::<43, 21>(s.get(2, 2), t[2]));
    s.set(3, 2, T::xor_and_rotate::<15, 49>(s.get(3, 2), t[2]));
    s.set(4, 2, T::xor_and_rotate::<61, 3>(s.get(4, 2), t[2]));

    s.set(0, 3, T::xor_and_rotate::<28, 36>(s.get(0, 3), t[3]));
    s.set(1, 3, T::xor_and_rotate::<55, 9>(s.get(1, 3), t[3]));
    s.set(2, 3, T::xor_and_rotate::<25, 39>(s.get(2, 3), t[3]));
    s.set(3, 3, T::xor_and_rotate::<21, 43>(s.get(3, 3), t[3]));
    s.set(4, 3, T::xor_and_rotate::<56, 8>(s.get(4, 3), t[3]));

    s.set(0, 4, T::xor_and_rotate::<27, 37>(s.get(0, 4), t[4]));
    s.set(1, 4, T::xor_and_rotate::<20, 44>(s.get(1, 4), t[4]));
    s.set(2, 4, T::xor_and_rotate::<39, 25>(s.get(2, 4), t[4]));
    s.set(3, 4, T::xor_and_rotate::<8, 56>(s.get(3, 4), t[4]));
    s.set(4, 4, T::xor_and_rotate::<14, 50>(s.get(4, 4), t[4]));
}

const _PI: [usize; 24] = [
    6, 12, 18, 24, 3, 9, 10, 16, 22, 1, 7, 13, 19, 20, 4, 5, 11, 17, 23, 2, 8, 14, 15, 21,
];

#[inline(always)]
pub(crate) fn pi<const N: usize, T: KeccakStateItem<N>>(s: &mut KeccakState<N, T>) {
    let old = s.clone();
    s.set(1, 0, old.get(0, 3));
    s.set(2, 0, old.get(0, 1));
    s.set(3, 0, old.get(0, 4));
    s.set(4, 0, old.get(0, 2));
    s.set(0, 1, old.get(1, 1));
    s.set(1, 1, old.get(1, 4));
    s.set(2, 1, old.get(1, 2));
    s.set(3, 1, old.get(1, 0));
    s.set(4, 1, old.get(1, 3));
    s.set(0, 2, old.get(2, 2));
    s.set(1, 2, old.get(2, 0));
    s.set(2, 2, old.get(2, 3));
    s.set(3, 2, old.get(2, 1));
    s.set(4, 2, old.get(2, 4));
    s.set(0, 3, old.get(3, 3));
    s.set(1, 3, old.get(3, 1));
    s.set(2, 3, old.get(3, 4));
    s.set(3, 3, old.get(3, 2));
    s.set(4, 3, old.get(3, 0));
    s.set(0, 4, old.get(4, 4));
    s.set(1, 4, old.get(4, 2));
    s.set(2, 4, old.get(4, 0));
    s.set(3, 4, old.get(4, 3));
    s.set(4, 4, old.get(4, 1));
}

#[inline(always)]
pub(crate) fn chi<const N: usize, T: KeccakStateItem<N>>(s: &mut KeccakState<N, T>) {
    let old = s.clone();

    #[allow(clippy::needless_range_loop)]
    for i in 0..5 {
        for j in 0..5 {
            s.set(
                i,
                j,
                T::and_not_xor(
                    s.get(i, j),
                    old.get(i, (j + 2) % 5),
                    old.get(i, (j + 1) % 5),
                ),
            );
        }
    }
}

const ROUNDCONSTANTS: [u64; 24] = [
    0x0000_0000_0000_0001u64,
    0x0000_0000_0000_8082u64,
    0x8000_0000_0000_808au64,
    0x8000_0000_8000_8000u64,
    0x0000_0000_0000_808bu64,
    0x0000_0000_8000_0001u64,
    0x8000_0000_8000_8081u64,
    0x8000_0000_0000_8009u64,
    0x0000_0000_0000_008au64,
    0x0000_0000_0000_0088u64,
    0x0000_0000_8000_8009u64,
    0x0000_0000_8000_000au64,
    0x0000_0000_8000_808bu64,
    0x8000_0000_0000_008bu64,
    0x8000_0000_0000_8089u64,
    0x8000_0000_0000_8003u64,
    0x8000_0000_0000_8002u64,
    0x8000_0000_0000_0080u64,
    0x0000_0000_0000_800au64,
    0x8000_0000_8000_000au64,
    0x8000_0000_8000_8081u64,
    0x8000_0000_0000_8080u64,
    0x0000_0000_8000_0001u64,
    0x8000_0000_8000_8008u64,
];

#[inline(always)]
pub(crate) fn iota<const N: usize, T: KeccakStateItem<N>>(s: &mut KeccakState<N, T>, i: usize) {
    s.set(0, 0, T::xor_constant(s.get(0, 0), ROUNDCONSTANTS[i]));
}

#[inline(always)]
pub(crate) fn keccakf1600<const N: usize, T: KeccakStateItem<N>>(s: &mut KeccakState<N, T>) {
    for i in 0..24 {
        theta_rho(s);
        pi(s);
        chi(s);
        iota(s, i);
    }
}

#[inline(always)]
pub(crate) fn absorb_block<const N: usize, T: KeccakStateItem<N>, const RATE: usize>(
    s: &mut KeccakState<N, T>,
    blocks: &[&[u8]; N],
    start: usize,
) {
    T::load_block::<RATE>(&mut s.st, blocks, start);
    keccakf1600(s)
}

#[inline(always)]
pub(crate) fn absorb_final<
    const N: usize,
    T: KeccakStateItem<N>,
    const RATE: usize,
    const DELIM: u8,
>(
    s: &mut KeccakState<N, T>,
    last: &[&[u8]; N],
    start: usize,
    len: usize,
) {
    debug_assert!(N > 0 && len < RATE); // && last[0].len() < RATE

    let mut blocks = [[0u8; 200]; N];
    for i in 0..N {
        if len > 0 {
            blocks[i][0..len].copy_from_slice(&last[i][start..start + len]);
        }
        blocks[i][len] = DELIM;
        blocks[i][RATE - 1] |= 0x80;
    }
    T::load_block_full::<RATE>(&mut s.st, &blocks, 0);
    keccakf1600(s)
}

#[inline(always)]
pub(crate) fn squeeze_first_block<const N: usize, T: KeccakStateItem<N>, const RATE: usize>(
    s: &KeccakState<N, T>,
    out: &mut [&mut [u8]; N],
) {
    T::store_block::<RATE>(&s.st, out)
}

#[inline(always)]
pub(crate) fn squeeze_next_block<const N: usize, T: KeccakStateItem<N>, const RATE: usize>(
    s: &mut KeccakState<N, T>,
    out: &mut [&mut [u8]; N],
) {
    keccakf1600(s);
    T::store_block::<RATE>(&s.st, out)
}

#[inline(always)]
pub(crate) fn squeeze_first_three_blocks<
    const N: usize,
    T: KeccakStateItem<N>,
    const RATE: usize,
>(
    s: &mut KeccakState<N, T>,
    out: [&mut [u8]; N],
) {
    let (mut o0, o1) = T::split_at_mut_n(out, RATE);
    squeeze_first_block::<N, T, RATE>(s, &mut o0);
    let (mut o1, mut o2) = T::split_at_mut_n(o1, RATE);
    squeeze_next_block::<N, T, RATE>(s, &mut o1);
    squeeze_next_block::<N, T, RATE>(s, &mut o2);
}

#[inline(always)]
pub(crate) fn squeeze_first_five_blocks<
    const N: usize,
    T: KeccakStateItem<N>,
    const RATE: usize,
>(
    s: &mut KeccakState<N, T>,
    out: [&mut [u8]; N],
) {
    let (mut o0, o1) = T::split_at_mut_n(out, RATE);
    squeeze_first_block::<N, T, RATE>(s, &mut o0);
    let (mut o1, o2) = T::split_at_mut_n(o1, RATE);

    squeeze_next_block::<N, T, RATE>(s, &mut o1);
    let (mut o2, o3) = T::split_at_mut_n(o2, RATE);

    squeeze_next_block::<N, T, RATE>(s, &mut o2);
    let (mut o3, mut o4) = T::split_at_mut_n(o3, RATE);

    squeeze_next_block::<N, T, RATE>(s, &mut o3);
    squeeze_next_block::<N, T, RATE>(s, &mut o4);
}

#[inline(always)]
pub(crate) fn squeeze_last<const N: usize, T: KeccakStateItem<N>, const RATE: usize>(
    mut s: KeccakState<N, T>,
    out: [&mut [u8]; N],
) {
    keccakf1600(&mut s);
    let mut b = [[0u8; 200]; N];
    T::store_block_full::<RATE>(&s.st, &mut b);
    for i in 0..N {
        out[i].copy_from_slice(&b[i][0..out[i].len()]);
    }
}

#[inline(always)]
pub(crate) fn squeeze_first_and_last<const N: usize, T: KeccakStateItem<N>, const RATE: usize>(
    s: &KeccakState<N, T>,
    out: [&mut [u8]; N],
) {
    let mut b = [[0u8; 200]; N];
    T::store_block_full::<RATE>(&s.st, &mut b);
    for i in 0..N {
        out[i].copy_from_slice(&b[i][0..out[i].len()]);
    }
}

#[inline(always)]
pub(crate) fn keccak<const N: usize, T: KeccakStateItem<N>, const RATE: usize, const DELIM: u8>(
    data: &[&[u8]; N],
    out: [&mut [u8]; N],
) {
    let mut s = KeccakState::<N, T>::new();
    for i in 0..data[0].len() / RATE {
        // T::slice_n(data, i * RATE, RATE)
        absorb_block::<N, T, RATE>(&mut s, &data, i * RATE);
    }
    let rem = data[0].len() % RATE;
    // T::slice_n(data, data[0].len() - rem, rem)
    absorb_final::<N, T, RATE, DELIM>(&mut s, data, data[0].len() - rem, rem);

    let outlen = out[0].len();
    let blocks = outlen / RATE;
    let last = outlen - (outlen % RATE);

    if blocks == 0 {
        squeeze_first_and_last::<N, T, RATE>(&s, out)
    } else {
        let (mut o0, mut o1) = T::split_at_mut_n(out, RATE);
        squeeze_first_block::<N, T, RATE>(&s, &mut o0);
        for _i in 1..blocks {
            let (mut o, orest) = T::split_at_mut_n(o1, RATE);
            squeeze_next_block::<N, T, RATE>(&mut s, &mut o);
            o1 = orest;
        }
        if last < outlen {
            squeeze_last::<N, T, RATE>(s, o1)
        }
    }
}
