module Exfetch
  module Resource
    extend self

    # Class variables
    @@platform : String? = nil
    @@release : String? = nil
    ## Cache to use later
    @@uptime_info : String? = nil
    @@swap_info : Tuple(Int64, Int64)? = nil
    @@memory_info : Tuple(Int64, Int64)? = nil

    def get_platform
      @@platform ||=
        begin
          pretty =
            if File.exists?("/etc/os-release")
              Manip.scrape_file("PRETTY_NAME", "/etc/os-release").strip.presence
            end
          sysname =
            begin
              {% if flag?(:linux) %}
                "Linux"
              {% elsif flag?(:freebsd) %}
                "FreeBSD"
              {% elsif flag?(:openbsd) %}
                "OpenBSD"
              {% elsif flag?(:netbsd) %}
                "NetBSD"
              {% end %}
            end
          pretty ? "#{pretty} (#{sysname})" : sysname
        end
    end

    def get_release
      @@release ||=
        begin
          kernel =
            if get_platform.includes?("BSD") && Manip.has_command?("sysctl")
              Manip.run_command("sysctl", ["-n", "kern.osrelease"])
            elsif Manip.has_command?("uname")
              Manip.run_command("uname", ["-r"])
            else
              "unknown"
            end
          version = File.exists?("/etc/os-release") ? Manip.scrape_file("VERSION_ID", "/etc/os-release") : ""
          "#{kernel.strip} #{version}".strip
        end
    end

    private def get_load_avg_info
      if get_uptime_info =~ /load averages?: (\d+\.\d+), (\d+\.\d+), (\d+\.\d+)/
        [ $1.to_f, $2.to_f, $3.to_f ]
      else
        [ 0.0, 0.0, 0.0 ]
      end
    rescue
      [ 0.0, 0.0, 0.0 ]
    end

    def get_load_avg
      get_load_avg_info
        .zip(["1m", "5m", "15m"])
        .map { |(avg, lbl)| "#{lbl}: #{avg.to_s.ljust(4, '0')}" }
        .join(", ")
    end

    def get_user
      ENV["USER"]? || "unknown"
    end

    def get_host
      System.hostname
    end

    def get_shell
      File.basename(ENV["SHELL"]? || "")
    end

    def get_terminal
      ENV["TERM"]? || "unknown"
    end

    def get_de : String?
      ENV["XDG_CURRENT_DESKTOP"]? || ENV["DESKTOP_SESSION"]?
    end

    WM_PROCESS_PATTERNS = {
      "catwm"     => "catwm",
      "fvwm"      => "fvwm",
      "dwm"       => "dwm",
      "2bwm"      => "2bwm",
      "monsterwm" => "monsterwm",
      "wmaker"    => "Window Maker",
      "sowm"      => "sowm",
      "penrose"   => "penrose",
      "cluncwm"   => "cluncwm"
    }

    def get_wm : String?
      if Manip.has_command?("ps")
        begin
          Manip.run_command("ps", ["x"]).each_line do |line|
            WM_PROCESS_PATTERNS.each do |pattern, name|
              return name if line.includes?(pattern)
            end
          end
        rescue
          # Continue to xprop fallback if ps fails
        end
      end

      # fallback
      return nil unless Manip.has_command?("xprop") && ENV.has_key?("DISPLAY")

      id = nil
      begin
        id_output = Manip.run_command("xprop", ["-root", "-notype", "_NET_SUPPORTING_WM_CHECK"]).strip
        return nil if id_output.blank?

        id = id_output.split.last
        return nil unless id =~ /^0x[0-9a-fA-F]+$/
      rescue
        return nil
      end

      wm = nil
      begin
        wm_output = Manip.run_command(
          "xprop",
          [
            "-id", id,
            "-notype",
            "-len", "25",
            "-f", "_NET_WM_NAME", "8t"
          ]
        )

        wm = $1 if wm_output =~ /_NET_WM_NAME.*= "(.*?)"/
      rescue
      end

      wm
    end

    def get_session
      de = get_de.presence
      wm = get_wm.presence

      if de && wm && de.downcase != wm.downcase
        "#{de} (#{wm})"
      elsif de
        de
      elsif wm
        wm
      else
        "unknown"
      end
    end

    private def get_swap_info
      @@swap_info ||=
        begin
          total = used = 0_i64

          if get_platform.includes?("Linux")
            meminfo = File.read_lines("/proc/meminfo").to_h do |line|
              k, v = line.split(":", 2)
              {k.strip, v.strip.split.first.to_i64 * 1024} # convert from kB to bytes
            end

            total = meminfo["SwapTotal"]? || 0_i64
            free  = meminfo["SwapFree"]?  || 0_i64
            used  = total - free
          else
            if Manip.has_command?("swapctl")
              Manip.run_command("swapctl", ["-lk"]).each_line do |line|
                next if line.starts_with?("Device")
                parts = line.split
                next if parts.size < 3
                total += parts[1].to_i64? || 0_i64
                used  += parts[2].to_i64? || 0_i64
              end
            else
              total = 0_i64
              used  = 0_i64
            end
            total *= 1024_i64
            used  *= 1024_i64
          end

          {total, used}
        rescue
          {0_i64, 0_i64}
        end
    end

    def get_swap
      total, _ = get_swap_info
      Manip.bytes_to_mebibytes(total)
    end

    def get_swap_usage
      _, used = get_swap_info
      Manip.bytes_to_mebibytes(used)
    end

    private def get_memory_info
      @@memory_info ||=
        begin
          total = used = 0_i64

          if get_platform.includes?("Linux")
            meminfo = File.read_lines("/proc/meminfo").to_h do |line|
              k, v = line.split(":", 2)
              {k.strip, v.strip.split.first.to_i64 * 1024} # convert from kB to bytes
            end

            total   = meminfo["MemTotal"]?
            free    = meminfo["MemFree"]?
            buffers = meminfo["Buffers"]?
            cached  = meminfo["Cached"]?

            if total && free
              used = total - free - (buffers || 0_i64) - (cached || 0_i64)
            end
          else
            pagesize =
              if Manip.has_command?("sysctl")
                Manip.run_command("sysctl", ["-n", "hw.pagesize"]).to_i64? || 4096_i64
              else
                4096_i64
              end

            total =
              if Manip.has_command?("sysctl")
                Manip.run_command("sysctl", ["-n", "hw.physmem"]).to_i64? || 0_i64
              else
                0_i64
              end

            active_pages =
              if Manip.has_command?("vmstat")
                Manip.run_command("vmstat", ["-s"])
                  .each_line
                  .find(&.includes?("pages active"))
                  .try { |line| line.split.first.to_i64 } || 0_i64
              else
                0_i64
              end

            used = active_pages * pagesize
          end

          {total || 0_i64, used || 0_i64}
        rescue
          {0_i64, 0_i64}
        end
    end

    def get_memory
      total, _ = get_memory_info
      Manip.bytes_to_mebibytes(total)
    end

    def get_memory_usage
      _, used = get_memory_info
      Manip.bytes_to_mebibytes(used)
    end

    def get_cpu : String
      model =
        begin
          if get_platform.includes?("Linux")
            File.read_lines("/proc/cpuinfo")
              .find(&.starts_with?("model name"))
              .try(&.split(":", 2).last.strip)
          elsif Manip.has_command?("sysctl")
            Manip.run_command("sysctl", ["-n", "hw.model"]).strip.presence
          else
            nil
          end
        rescue
          nil
        end

      (model || "unknown").gsub(/\s*\(.*?\)/, "")
    end

    private def get_gpu_info
      gpus = [] of String

      if Manip.has_command?("nvidia-smi")
        nvidia_output = Manip.run_command("nvidia-smi", ["-q"])
        nvidia_lines = nvidia_output
                       .lines
                       .select(&.includes?("Product Name"))
                       .map { |l| l.split(":", 2)[1]?.to_s.strip }
        gpus.concat(nvidia_lines)
      end

      if Manip.has_command?("glxinfo")
        glx_output = Manip.run_command("glxinfo", ["-B"])
        if line = glx_output.lines.find(&.includes?("OpenGL renderer string"))
          renderer = line.split(":", 2)[1]?.to_s.strip
          gpus << renderer if renderer && !renderer.empty?
        end
      end

      gpus
        .compact
        .map { |g| g.gsub("Mesa", "").gsub(/\s*\([^)]*\)/, "").strip }
        .uniq
    end

    def get_gpu
      gpus = get_gpu_info
      gpus.empty? ? "unknown" : gpus.join(", ")
    end

    private def get_uptime_info
      @@uptime_info ||=
        if Manip.has_command?("uptime")
          Manip.run_command("uptime").strip
        else
          "uptime unavailable"
        end
    end

    def get_uptime
      if get_uptime_info =~ /\bup\b\s+(.*?),\s+\d\s+users?/
        $1.strip
      else
        "unknown"
      end
    end

    def get_pkgs
      platform = get_platform
      if platform.includes?("Linux")
        get_linux_pkgs.to_s
      elsif platform.includes?("BSD")
        get_bsd_pkgs.to_s
      else
        "unsupported"
      end
    end

    private def get_linux_pkgs
      count = 0

      count += Manip.command_lines_len("dpkg-query", ["-f", ".\n", "-W"]) if Manip.has_command?("dpkg-query")
      count += Manip.command_lines_len("pacman", ["-Qq"]) if Manip.has_command?("pacman")
      count += Manip.command_lines_len("rpm", ["-qa"]) if Manip.has_command?("rpm")
      count += Manip.dir_size("/var/db/pkg/", recursive = true) if Manip.has_command?("emerge")
      count += Manip.command_lines_len("apk", ["info"]) if Manip.has_command?("apk")
      if Manip.has_command?("nix-store")
        count += Manip.command_lines_len("nix-store", ["-q", "--requisites", "/run/current-system/sw"])
        count += Manip.command_lines_len("nix-store", ["-q", "--requisites", "#{ENV["HOME"]}/.nix-profile"])
      end

      count.to_s
    end

    private def get_bsd_pkgs
      platform = get_platform
      if platform.includes?("FreeBSD")
        Manip.command_lines_len("pkg", ["info"])
      elsif platform.includes?("OpenBSD")
        Manip.dir_size("/var/db/pkg/", recursive = false)
      elsif platform.includes?("NetBSD")
        Manip.command_lines_len("pkg_info")
      else
        "unsupported"
      end
    end

  end
end
