//
// Copyright (c) 1993-2022 Robert McNeel & Associates. All rights reserved.
// OpenNURBS, Rhinoceros, and Rhino3D are registered trademarks of Robert
// McNeel & Associates.
//
// THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
// ALL IMPLIED WARRANTIES OF FITNESS FOR ANY PARTICULAR PURPOSE AND OF
// MERCHANTABILITY ARE HEREBY DISCLAIMED.
//
// For complete openNURBS copyright information see <http://www.opennurbs.org>.
//
////////////////////////////////////////////////////////////////

#include "opennurbs.h"

#if !defined(ON_COMPILING_OPENNURBS)
// This check is included in all opennurbs source .c and .cpp files to insure
// ON_COMPILING_OPENNURBS is defined when opennurbs source is compiled.
// When opennurbs source is being compiled, ON_COMPILING_OPENNURBS is defined
// and the opennurbs .h files alter what is declared and how it is declared.
#error ON_COMPILING_OPENNURBS must be defined when compiling opennurbs
#endif

int ON_IsValidUnicodeCodePoint(ON__UINT32 u)
{
  if (u < 0xD800)
    return true;

  if (u < 0xE000)
  {
    // 0xD800 <= u < 0xDC00 is used for the first value in a UTF-16 surrogate pair.
    // 0xDC00 <= u < 0xE000 is used for the second value in a UTF-16 surrogate pair.
    return false;
  }

  if (u < 0xFFFE)
    return true;

  if (u == 0xFFFE || u == 0xFFFF)
  {
    // The Unicode specification states these values are <not a character>.
    // 0xFFFE is a swapped Unicode byte order mark (U+FEFF).
    // https://www.unicode.org/charts/PDF/UFFF0.pdf
    return false;
  }

  if (u < 0xFFFFE)
    return true;

  if (u == 0xFFFFE || u == 0xFFFFF)
  {
    // The Unicode specification states these values are <not a character>.
    // https://www.unicode.org/charts/PDF/UFFF80.pdf
    return false;
  }

  if (u < 0x10FFFE)
    return true;

  if (u == 0x10FFFE || u == 0x10FFFF)
  {
    // The Unicode specification states these values are <not a character>.
    // https://www.unicode.org/charts/PDF/U10FF80.pdf
    return false;
  }

  // All valid Unicode code points are < 0x110000
  return false;
}

bool ON_IsPrivateUseUnicodeCodePoint(ON__UINT32 unicode_code_point)
{
  // Unicode code points in the ranges U+E000–U+F8FF, U+F0000–U+FFFFD, 
  // and U+100000–U+10FFFD are for private use (user defined characters).

  if (unicode_code_point < 0xE000U)
    return false;

  if (unicode_code_point <= 0xF8FFU)
    return true; // Private use area E000-F8FF

  if (unicode_code_point < 0xF0000U)
    return false;

  if (unicode_code_point <= 0xFFFFDU)
    return true; // Supplementary Private Use Area-A F0000-FFFFD

  // FFFFE and FFFFF are <not a character>

  if (unicode_code_point < 0x100000U)
    return false;

  if (unicode_code_point <= 0x10FFFDU)
    return true; // Supplementary Private Use Area-B 100000-10FFFD

  // 10FFFE and 10FFFF are <not a character>
   
  return false;
}

bool ON_IsStandardUnicodeCodePoint(ON__UINT32 unicode_code_point)
{
  return 
    0 != ON_IsValidUnicodeCodePoint(unicode_code_point) 
    && ON_UnicodeCodePoint::ON_ByteOrderMark != unicode_code_point
    && false == ON_IsPrivateUseUnicodeCodePoint(unicode_code_point)
    ;
}

int ON_IsUnicodeSpaceCodePoint(
  ON__UINT32 u
)
{
  // Additional code points may be added in the future.
  // The goal is to detect code points that typically separate words 
  // and which should not be at the beginning or end of a word.
  return
    ON_UnicodeCodePoint::ON_Space == u
    || ON_UnicodeCodePoint::ON_NoBreakSpace == u
    || ON_UnicodeCodePoint::ON_NarrowNoBreakSpace == u
    || (u >= ON_UnicodeCodePoint::ON_EnQuad && u <= ON_UnicodeCodePoint::ON_ZeroWidthSpace)
    ;
}

int ON_IsUnicodeSpaceOrControlCodePoint(
  ON__UINT32 u
)
{
  // Additional space code points may be added in the future.
  // The goal is to detect code points that should be trimmed
  // when isolating text tokens in a string.
  
  if (u > 0 && u <= ON_UnicodeCodePoint::ON_Space)
    return true;

  if (u < ON_UnicodeCodePoint::ON_Delete)
    return false;

  if (u == ON_UnicodeCodePoint::ON_Delete)
    return true;
  
  if (ON_IsUnicodeSpaceCodePoint(u))
    return true;
  
  if (ON_IsUnicodeC1ControlCodePoint(u))
    return true;

  if (u < 0x2000)
    return false;

  // Bidirectional text control
  if (u >= 0x200E && u <= 0x200F)
    return true;

  // Separators, Format characters, and NNBSP
  if (u >= 0x2028 && u <= 0x202F)
    return true;

  // Bidirectional text control
  if (u >= 0x202A && u <= 0x202E)
    return true;

  // Bidirectional text control
  if (u >= 0x2066 && u <= 0x2069)
    return true;

  return false;
}

int ON_IsUnicodeC1ControlCodePoint(
  ON__UINT32 u
)
{
  return (u >= 0x0080 && u <= 0x009F);
}

int ON_IsValidUTF32Value(
  ON__UINT32 c
  )
{
  return (c < 0xD800 || (c >= 0xE000 && c <= 0x10FFFF));
}

int ON_IsValidSingleElementUTF16Value(ON__UINT32 c)
{
  return ((c <= 0xD7FF) || (c >= 0xE000 && c <= 0xFFFF));
}

int ON_IsValidUTF16Singleton(ON__UINT32 c)
{
  return ((c <= 0xD7FF) || (c >= 0xE000 && c <= 0xFFFF));
}

enum ON_UnicodeEncoding ON_UnicodeNativeCPU_UTF16()
{
  return (ON::endian::little_endian== ON::Endian()) ? ON_UTF_16LE : ON_UTF_16BE;
}

enum ON_UnicodeEncoding ON_UnicodeNativeCPU_UTF32()
{
  return (ON::endian::little_endian== ON::Endian()) ? ON_UTF_32LE : ON_UTF_32BE;
}

int ON_IsValidSingleByteUTF8CharValue(
  char c
  )
{
  return (c >= 0 && c <= 0x7F);
}

int ON_IsValidUTF8SingletonChar(
  char c
  )
{
  return (c >= 0 && c <= 0x7F);
}

int ON_IsValidSingleElementUTF8Value(
  ON__UINT32 c
  )
{
  return (c <= 0x7F);
}

int ON_IsValidUTF8Singleton(
  ON__UINT32 c
  )
{
  return (c <= 0x7FU);
}

int ON_IsValidUTF16SurrogatePair(
  unsigned int w1,
  unsigned int w2
  )
{
  return ( w1 >= 0xD800U && w1 < 0xDC00 && w2 >= 0xDC00 && w2 < 0xE000 );
}

unsigned int ON_DecodeUTF16SurrogatePair(
  unsigned int u1,
  unsigned int u2,
  unsigned int error_code_point
)
{
  if (u1 >= 0xD800U && u1 < 0xDC00 && u2 >= 0xDC00 && u2 < 0xE000)
  {
    return ((u1-0xD800)*0x400 + (u2-0xDC00) + 0x10000);
  }
  return error_code_point;
}


int ON_IsValidSingleElementWideCharValue(
  wchar_t w
  )
{
#pragma ON_PRAGMA_WARNING_PUSH
// warning C4127: conditional expression is constant
#pragma ON_PRAGMA_WARNING_DISABLE_MSC( 4127 )
  if (1 == sizeof(w))
    return ON_IsValidSingleElementUTF8Value((ON__UINT32)w);
  if (2 == sizeof(w))
    return ON_IsValidSingleElementUTF16Value((ON__UINT32)w);
  return ON_IsValidUTF32Value((ON__UINT32)w);
#pragma ON_PRAGMA_WARNING_POP
}

enum ON_UnicodeEncoding ON_IsUTFByteOrderMark(
  const void* buffer,
  size_t sizeof_buffer
  )
{
  if ( 0 != buffer && sizeof_buffer >= 2 )
  {
    const unsigned char* b = static_cast<const unsigned char*>(buffer);

    if ( 0 == b[0] )
    {
      if ( sizeof_buffer >= 4 && 0 == b[1] && 0xFE == b[2] && 0xFF == b[3] )
        return ON_UTF_32BE;
    }
    else if ( 0xEF == b[0] )
    {
      if ( sizeof_buffer >= 3 && 0xBB == b[1] && 0xBF == b[2] )
        return ON_UTF_8;
    }
    else if ( 0xFE == b[0] )
    {
      if ( 0xFF == b[1] )
        return ON_UTF_16BE;
    }
    else if ( 0xFF == b[0] && 0xFE == b[1] )
    {
      return ( sizeof_buffer >= 4 && 0 == b[2] && 0 == b[3] )
        ? ON_UTF_32LE
        : ON_UTF_16LE;
    }

  }

  return ON_UTF_unset;
}

unsigned int ON_UTFSizeofByteOrderMark(
  ON_UnicodeEncoding e
  )
{
  unsigned int sizeof_bom;
  switch (e)
  {
  case ON_UTF_8:
    sizeof_bom = 3;
    break;

  case ON_UTF_16:
  case ON_UTF_16BE:
  case ON_UTF_16LE:
    sizeof_bom = 2;
    break;

  case ON_UTF_32:
  case ON_UTF_32BE:
  case ON_UTF_32LE:
    sizeof_bom = 4;
    break;

  default:
    sizeof_bom = 0;
    break;
  }

  return sizeof_bom;
}

static int ON_IsUTF8ByteOrderMark(
  const char* sUTF8,
  int sUTF8_count
  )
{
  if ( 0 == sUTF8 )
    return 0;
  
  if (sUTF8_count < 3 )
    return 0;
  
  return (0xEF == (unsigned char)(sUTF8[0]) && 0xBB == (unsigned char)(sUTF8[1]) && 0xBF == (unsigned char)(sUTF8[2]));
}

bool ON_IsUnicodeControlCodePoint(
  ON__UINT32 code_point,
  bool bNullReturnValue
)
{
  if (0 == code_point)
    return bNullReturnValue ? true : false;
  if (code_point < 0x0020)
    return true; // below space
  if (code_point < 0x007f)
    return false;
  if (code_point <= 0x00A0)
    return true; // del to 0xA0
  if (code_point < 0x00AD)
    return false;
  if (code_point == 0x00AD)
    return true; // soft hyphen

  return false;
}

int ON_EncodeUTF8( ON__UINT32 u, char sUTF8[6] )
{
  ON__UINT32 c;

  if ( u <= 0x7F )
  {
    // 1 byte UTF8 encoding: 0xxxxxxx (7 bits of u)
    sUTF8[0] = (char)u;
    return 1;
  }

  if ( u <= 0x7FF )
  {
    // 2 byte UTF8 encoding: 110xxxxx, 10xxxxxx (11 bits of u)
    c = (u / 0x40);  // c  = 000xxxxx
    c |= 0xC0;                      //   |= 11000000
    sUTF8[0] = (char)c;
    c = (u & 0x3F);
    c |= 0x80;
    sUTF8[1] = (char)c;
    return 2;
  }

  if ( u <= 0xFFFF )
  {
    // 3 byte UTF8 encoding: 1110xxxx, 10xxxxxx, 10xxxxxx (16 bits of u)
    c = (u / 0x1000); // c  = 0000xxxx
    c |= 0xE0;                       //   |= 11100000
    sUTF8[0] = (char)c;
    c = ((u & 0xFFF) / 0x40);
    c |= 0x80;
    sUTF8[1] = (char)c;
    c = u & 0x3F;
    c |= 0x80;
    sUTF8[2] = (char)c;
    return 3;
  }

  if ( u <= 0x1FFFFF )
  {
    // (maximum valid unicode codepoint is 0x10FFFF)
    // 4 byte UTF8 encoding: 11110xxx, 10xxxxxx, 10xxxxxx, 10xxxxxx (21 bits of u)
    // Note: 0x10FFFF is the maximum valid unicode code point.
    // For u > 0x10FFFF and u <= 0x1FFFFF, this calculation encodes the low 21 bits of u.
    c = (u / 0x40000);  // c  = 00000xxx
    c |= 0xF0;                         //   |= 11110000
    sUTF8[0] = (char)c;
    c = ((u & 0x3FFFF)/0x1000);
    c |= 0x80;
    sUTF8[1] = (char)c;
    c = ((u & 0xFFF) / 0x40);
    c |= 0x80;
    sUTF8[2] = (char)c;
    c = u & 0x3F;
    c |= 0x80;
    sUTF8[3] = (char)c;
    return 4;
  }

  if ( u <= 0x3FFFFFF )
  {
    // 5 byte encoding: 111110xx, 10xxxxxx, 10xxxxxx, 10xxxxxx, 10xxxxxx (26 bits of u)
    // Note: 0x10FFFF is the maximum valid unicode code point.
    c = (u / 0x1000000); // c  = 000000xx
    c |= 0xF8;                         //   |= 11111000
    sUTF8[0] = (char)c;
    c = ((u & 0xFFFFFF)/0x40000);
    c |= 0x80;
    sUTF8[1] = (char)c;
    c = ((u & 0x3FFFF)/0x1000);
    c |= 0x80;
    sUTF8[2] = (char)c;
    c = ((u & 0xFFF) / 0x40);
    c |= 0x80;
    sUTF8[3] = (char)c;
    c = u & 0x3F;
    c |= 0x80;
    sUTF8[4] = (char)c;
    return 5;
  }

  if ( u <= 0x7FFFFFFF )
  {
    // 6 byte encoding: 1111110x, 10xxxxxx, 10xxxxxx, 10xxxxxx, 10xxxxxx, 10xxxxxx (31 bits of u)
    // Note: 0x10FFFF is the maximum valid unicode code point.
    c = (u / 0x40000000); // c  = 00000000x
    c |= 0xFC;                           //   |= 11111100
    sUTF8[0] = (char)c;
    c = ((u & 0x3FFFFFFF)/0x1000000);
    c |= 0x80;
    sUTF8[1] = (char)c;
    c = ((u & 0xFFFFFF)/0x40000);
    c |= 0x80;
    sUTF8[2] = (char)c;
    c = ((u & 0x3FFFF)/0x1000);
    c |= 0x80;
    sUTF8[3] = (char)c;
    c = ((u & 0xFFF) / 0x40);
    c |= 0x80;
    sUTF8[4] = (char)c;
    c = u & 0x3F;
    c |= 0x80;
    sUTF8[5] = (char)c;
    return 6;
  }

  return 0;
}

static int ON_DecodeUTF8Helper(
    const char* sUTF8,
    int sUTF8_count,
    ON__UINT32* value,
    unsigned int* error_status
    )
{
#define INPUT_BUFFER_TOO_SHORT 16
#define INVALID_CONTINUATION_VALUE 16
#define OVERLONG_ENCODING 8

  ON__UINT32 u;
  char c;

  c = sUTF8[0];

  if ( 0 == (0x80 & c) )
  {
    // 1 byte ASCII encoding: 0xxxxxxx
    *value = c;
    return 1;
  }

  if ( 0xC0 == ( 0xE0 & c) )
  {
    // 2 byte character encoding: 10xxxxxx, 10xxxxxx
    if ( sUTF8_count < 2 )
    {
      *error_status |= INPUT_BUFFER_TOO_SHORT; // input buffer too short
      return 0;
    }
    u = (0x1F & c);
    c = sUTF8[1];
    if (  0x80 != ( 0xC0 & c) )
    {
      *error_status |= INVALID_CONTINUATION_VALUE; // invalid UTF=8 continuation value
      return 0;
    }
    u *= 64;
    u |= (0x3F & c);
    if ( u <= 0x7F )
    {
      *error_status |= OVERLONG_ENCODING; // overlong 2 byte character encoding
    }
    *value = u;
    return 2;
  }

  if ( 0xE0 == ( 0xF0 & c) )
  {
    // 3 byte character encoding: 110xxxxx, 10xxxxxx, 10xxxxxx
    if ( sUTF8_count < 3 )
    {
      *error_status |= INPUT_BUFFER_TOO_SHORT; // input buffer too short
      return 0;
    }
    u = (0x0F & c);
    c = sUTF8[1];
    if (  0x80 != ( 0xC0 & c) )
    {
      *error_status |= INVALID_CONTINUATION_VALUE; // invalid UTF=8 continuation value
      return 0;
    }
    u *= 64;
    u |= (0x3F & c);
    c = sUTF8[2];
    if (  0x80 != ( 0xC0 & c) )
    {
      *error_status |= INVALID_CONTINUATION_VALUE; // invalid UTF=8 continuation value
      return 0;
    }
    u *= 64;
    u |= (0x3F & c);
    if ( u <= 0x7FF )
    {
      *error_status |= OVERLONG_ENCODING; // overlong 3 byte character encoding
    }
    *value = u;
    return 3;
  }

  if ( 0xF0 == ( 0xF8 & c) )
  {
    // 4 byte character encoding: 11110xxx, 10xxxxxx, 10xxxxxx, 10xxxxxx
    if ( sUTF8_count < 4 )
    {
      *error_status |= INPUT_BUFFER_TOO_SHORT; // input buffer too short
      return 0;
    }

    u = (0x07 & c);
    c = sUTF8[1];
    if (  0x80 != ( 0xC0 & c) )
    {
      *error_status |= INVALID_CONTINUATION_VALUE; // invalid UTF=8 continuation value
      return 0;
    }
    u *= 64;
    u |= (0x3F & c);
    c = sUTF8[2];
    if (  0x80 != ( 0xC0 & c) )
    {
      *error_status |= INVALID_CONTINUATION_VALUE; // invalid UTF=8 continuation value
      return 0;
    }
    u *= 64;
    u |= (0x3F & c);
    c = sUTF8[3];
    if (  0x80 != ( 0xC0 & c) )
    {
      *error_status |= INVALID_CONTINUATION_VALUE; // invalid UTF=8 continuation value
      return 0;
    }
    u *= 64;
    u |= (0x3F & c);
    if ( u <= 0xFFFF )
    {
      *error_status |= OVERLONG_ENCODING; // overlong 4 byte character encoding
    }
    *value = u;
    return 4;
  }

  if ( 0xF8 == ( 0xFC & c) )
  {
    // 5 byte character encoding: 111110xx, 10xxxxxx, 10xxxxxx, 10xxxxxx, 10xxxxxx
    if ( sUTF8_count < 5 )
    {
      *error_status |= INPUT_BUFFER_TOO_SHORT; // input buffer too short
      return 0;
    }

    u = (0x03 & c);
    c = sUTF8[1];
    if (  0x80 != ( 0xC0 & c) )
    {
      *error_status |= INVALID_CONTINUATION_VALUE; // invalid UTF=8 continuation value
      return 0;
    }
    u *= 64;
    u |= (0x3F & c);
    c = sUTF8[2];
    if (  0x80 != ( 0xC0 & c) )
    {
      *error_status |= INVALID_CONTINUATION_VALUE; // invalid UTF=8 continuation value
      return 0;
    }
    u *= 64;
    u |= (0x3F & c);
    c = sUTF8[3];
    if (  0x80 != ( 0xC0 & c) )
    {
      *error_status |= INVALID_CONTINUATION_VALUE; // invalid UTF=8 continuation value
      return 0;
    }
    u *= 64;
    u |= (0x3F & c);
    c = sUTF8[4];
    if (  0x80 != ( 0xC0 & c) )
    {
      *error_status |= INVALID_CONTINUATION_VALUE; // invalid UTF=8 continuation value
      return 0;
    }
    u *= 64;
    u |= (0x3F & c);
    if ( u <= 0x1FFFFF )
    {
      *error_status |= OVERLONG_ENCODING; // overlong 5 byte character encoding
    }
    *value = u;
    return 5;
  }

  if ( 0xFC == ( 0xFE & c) )
  {
    // 6 byte character encoding: 110xxxxx, 10xxxxxx, 10xxxxxx, 10xxxxxx, 10xxxxxx, 10xxxxxx
    if ( sUTF8_count < 6 )
    {
      *error_status |= INPUT_BUFFER_TOO_SHORT; // input buffer too short
      return 0;
    }

    u = (0x01 & c);
    c = sUTF8[1];
    if (  0x80 != ( 0xC0 & c) )
    {
      *error_status |= INVALID_CONTINUATION_VALUE; // invalid UTF=8 continuation value
      return 0;
    }
    u *= 64;
    u |= (0x3F & c);
    c = sUTF8[2];
    if (  0x80 != ( 0xC0 & c) )
    {
      *error_status |= INVALID_CONTINUATION_VALUE; // invalid UTF=8 continuation value
      return 0;
    }
    u *= 64;
    u |= (0x3F & c);
    c = sUTF8[3];
    if (  0x80 != ( 0xC0 & c) )
    {
      *error_status |= INVALID_CONTINUATION_VALUE; // invalid UTF=8 continuation value
      return 0;
    }
    u *= 64;
    u |= (0x3F & c);
    c = sUTF8[4];
    if (  0x80 != ( 0xC0 & c) )
    {
      *error_status |= INVALID_CONTINUATION_VALUE; // invalid UTF=8 continuation value
      return 0;
    }
    u *= 64;
    u |= (0x3F & c);
    c = sUTF8[5];
    if (  0x80 != ( 0xC0 & c) )
    {
      *error_status |= INVALID_CONTINUATION_VALUE; // invalid UTF=8 continuation value
      return 0;
    }
    u *= 64;
    u |= (0x3F & c);
    if ( u <= 0x3FFFFFF )
    {
      *error_status |= OVERLONG_ENCODING; // overlong 6 byte character encoding
    }
    *value = u;
    return 6;
  }

  *error_status |= INVALID_CONTINUATION_VALUE; // invalid UTF=8 start value
  return 0;

#undef INPUT_BUFFER_TOO_SHORT
#undef INVALID_CONTINUATION_VALUE
#undef OVERLONG_ENCODING
}

int ON_DecodeUTF8(
    const char* sUTF8,
    int sUTF8_count,
    struct ON_UnicodeErrorParameters* e,
    ON__UINT32* unicode_code_point
    )
{
  ON__UINT32 u0, u1;
  int i0, i1;
  unsigned int error_status;
  ON__UINT16 sUTF16[2];
  char c;

  ON_UnicodeErrorParameters local_e = ON_UnicodeErrorParameters::MaskErrors;
  if (nullptr == e)
    e = &local_e;

  if (  0 == sUTF8 || sUTF8_count <= 0 || 0 == unicode_code_point )
  {
    if ( e )
      e->m_error_status |= 1;
    return 0;
  }

  // special cases for most common unicode values
  // If any error conditions exist, then ON_DecodeUTF8Helper()
  // is used.
  if ( 0 == (0x80 & sUTF8[0]) )
  {
    *unicode_code_point = sUTF8[0];
    return 1;
  }

  c = sUTF8[0];
  if ( 0xC0 == ( 0xE0 & c) && sUTF8_count >= 2 )
  {
    // 2 byte character encoding: 10xxxxxx, 10xxxxxx
    u0 = (0x1F & c);
    c = sUTF8[1];
    if (  0x80 == ( 0xC0 & c) )
    {
      u0 *= 64;
      u0 |= (0x3F & c);
      if ( u0 > 0x7F )
      {
        *unicode_code_point = u0;
        return 2;
      }
    }
  }
  else if ( 0xE0 == ( 0xF0 & c) && sUTF8_count >= 3 )
  {
    // 3 byte character encoding: 110xxxxx, 10xxxxxx, 10xxxxxx
    u0 = (0x0F & c);
    c = sUTF8[1];
    if (  0x80 == ( 0xC0 & c) )
    {
      u0 *= 64;
      u0 |= (0x3F & c);
      c = sUTF8[2];
      if (  0x80 == ( 0xC0 & c) )
      {
        u0 *= 64;
        u0 |= (0x3F & c);
        if ( u0 >= 0x0800 && (u0 <= 0xD800 || u0 >= 0xE000) )
        {
          *unicode_code_point = u0;
          return 3;
        }
      }
    }
  }
  else if ( 0xF0 == ( 0xF8 & c) && sUTF8_count >= 4 )
  {
    // 4 byte character encoding: 11110xxx, 10xxxxxx, 10xxxxxx, 10xxxxxx
    u0 = (0x07 & c);
    c = sUTF8[1];
    if (  0x80 == ( 0xC0 & c) )
    {
      u0 *= 64;
      u0 |= (0x3F & c);
      c = sUTF8[2];
      if (  0x80 == ( 0xC0 & c) )
      {
        u0 *= 64;
        u0 |= (0x3F & c);
        c = sUTF8[3];
        if (  0x80 == ( 0xC0 & c) )
        {
          u0 *= 64;
          u0 |= (0x3F & c);
          if ( u0 >= 0x010000 && u0 <= 0x10FFFF )
          {
            *unicode_code_point = u0;
            return 4;
          }
        }
      }
    }
  }


  error_status = 0;
  u0 = 0xFFFFFFFF;
  i0 = ON_DecodeUTF8Helper(sUTF8,sUTF8_count,&u0,&error_status);
  if ( i0 > 0 && 0 == error_status && (u0 < 0xD800 || (u0 >= 0xE000 && u0 <= 0x10FFFF) ) )
  {
    // valid UTF-8 multibyte encoding parsed
    *unicode_code_point = u0;
    return i0;
  }

  // handle errors
  if ( 0 == e )
  {
    // no errors are masked.
    return 0;
  }

  // report error condition
  e->m_error_status |= error_status;

  if ( error_status != (error_status & e->m_error_mask) )
  {
    // this error is not masked
    return 0;
  }

  if ( i0 <= 0 )
  {
    i0 = 1;
    if ( ON_IsValidUnicodeCodePoint(e->m_error_code_point) )
    {
      // skip to next UTF-8 start element
      for ( /*empty for initializer*/; i0 < sUTF8_count; i0++ )
      {
        // Search for the next element of sUTF8[] that is the
        // start of a UTF-8 encoding sequence.
        c = sUTF8[i0];
        if (    0 == (0x80 & c)     // ASCII 0 - 127
              || 0xC0 == ( 0xE0 & c) // 2 byte encoding first character
              || 0xE0 == ( 0xF0 & c) // 3 byte encoding first character
              || 0xF0 == ( 0xF8 & c) // 4 byte encoding first character
              || 0xF8 == ( 0xFC & c) // 5 byte encoding first character
              || 0xFC == ( 0xFE & c) // 6 byte encoding first character
            )
        {
          // resume parsing at this character
          break;
        }
      }
      *unicode_code_point = e->m_error_code_point;
    }
    return i0;
  }

  if ( ON_IsValidUnicodeCodePoint(u0) && 8 == error_status )
  {
    // overlong UTF-8 multibyte encoding of valid unicode code point
    *unicode_code_point = u0;
    return i0;
  }

  if ( i0 < sUTF8_count
       && u0 >= 0xD800 && u0 <= 0xDBFF
       && (0 == error_status || 8 == error_status)
       && 0 != (4 & e->m_error_mask)
     )
  {
    // See if a UFT-16 surrogate pair was incorrectly encoded
    // as two consecutive UTF-8 sequences.
    u1 = 0xFFFFFFFF;
    i1 = ON_DecodeUTF8Helper(sUTF8+i0,sUTF8_count-i0,&u1,&error_status);
    if ( i1 > 0 && (0 == error_status || 8 == error_status) )
    {
      error_status = 0;
      sUTF16[0] = (ON__UINT16)u0;
      sUTF16[1] = (ON__UINT16)u1;
      u0 = 0xFFFFFFFF;
      if ( 2 == ON_ConvertUTF16ToUTF32(false,sUTF16,2,&u0,1,&error_status,0,0,0)
           && 0 == error_status
           && ON_IsValidUnicodeCodePoint(u0)
         )
      {
        *unicode_code_point = u0;
        e->m_error_status |= 4;
        return i0+i1;
      }
    }
  }

  if ( ON_IsValidUnicodeCodePoint(e->m_error_code_point) )
  {
    *unicode_code_point = e->m_error_code_point;
    return i0;
  }

  return 0;
}

int ON_EncodeUTF16( ON__UINT32 unicode_code_point, ON__UINT16 sUTF16[2] )
{
  // put the most common case first
  if ( unicode_code_point < 0xD800 )
  {
    // code point values U+0000 ... U+D7FF
    // = UTF-16 values
    sUTF16[0] = (ON__UINT16)unicode_code_point;
    return 1;
  }

  if ( unicode_code_point < 0xE000 )
  {
    // 0xD800 ... 0xDFFF are invalid unicode code point values
    return 0;
  }

  if ( unicode_code_point <= 0xFFFF )
  {
    // code point values U+E000 ... U+FFFF
    // = UTF-16 values
    sUTF16[0] = (ON__UINT16)unicode_code_point;
    return 1;
  }

  if ( unicode_code_point <= 0x10FFFF )
  {
    // code point values U+10000 ... U+10FFFF
    // = surrogate pair UTF-16 values
    unicode_code_point -= 0x10000;
    sUTF16[0] = (ON__UINT16)(0xD800 + (unicode_code_point / 0x400)); // high surrogate value (0xD800 ... 0xDBFF)
    sUTF16[1] = (ON__UINT16)(0xDC00 + (unicode_code_point & 0x3FF)); // low surrogate value (0xDC00 ... 0xDFFF)
    return 2;
  }

  // 0x110000 ... 0xFFFFFFFF are invalid unicode code point values
  return 0;
}

int ON_DecodeUTF16(
    const ON__UINT16* sUTF16,
    int sUTF16_count,
    struct ON_UnicodeErrorParameters* e,
    ON__UINT32* unicode_code_point
    )
{
  ON__UINT32 uhi, ulo;

  ON_UnicodeErrorParameters local_e = ON_UnicodeErrorParameters::MaskErrors;
  if (nullptr == e)
    e = &local_e;

  if ( 0 == sUTF16 || sUTF16_count <= 0 || 0 == unicode_code_point )
  {
    if ( e )
      e->m_error_status |= 1;
    return 0;
  }

  // special case for most common UTF-16 single element values
  if ( ( sUTF16[0] < 0xD800 ) || ( sUTF16[0] >= 0xE000 ) )
  {
    *unicode_code_point = sUTF16[0];
    return 1;
  }

  if ( sUTF16_count >= 2 && sUTF16[0] < 0xDC00 && sUTF16[1] >=  0xDC00 && sUTF16[1] < 0xE000 )
  {
    // UTF-16 surrogate pair
    uhi = sUTF16[0];
    ulo = sUTF16[1];
    *unicode_code_point = (uhi-0xD800)*0x400 + (ulo-0xDC00) + 0x10000;
    return 2;
  }

  // handle errors
  if ( 0 == e )
  {
    // no errors are masked.
    return 0;
  }

  // report error condition
  e->m_error_status |= 16;

  if ( 16 != (16 & e->m_error_mask) || !ON_IsValidUnicodeCodePoint(e->m_error_code_point) )
  {
    // this error is not masked
    return 0;
  }

  // Search for the next element of sUTF16[] that is a
  // valid UTF-16 encoding sequence.
  int i;
  for ( i = 1; i < sUTF16_count; i++ )
  {
    if ( ( sUTF16[i] < 0xD800 ) || ( sUTF16[i] >= 0xE000 ) )
    {
      // valid single UTF-16 code unit
      break;
    }
    if ( i+1 < sUTF16_count
         && sUTF16[i] >= 0xD800 && sUTF16[i] < 0xDC00
         && sUTF16[i+1] >= 0xDC00 && sUTF16[i+1] < 0xE000
       )
    {
      // valid UTF-16 surrogate pair
      break;
    }
  }

  *unicode_code_point = e->m_error_code_point;

  return i;
}

int ON_DecodeUTF16LE(
  const ON__UINT16* sUTF16,
  int sUTF16_count,
  struct ON_UnicodeErrorParameters* e,
  ON__UINT32* unicode_code_point
  )
{
#if defined(ON_LITTLE_ENDIAN)
  return ON_DecodeUTF16(sUTF16,sUTF16_count,e,unicode_code_point);
#else
  return ON_DecodeSwapByteUTF16(sUTF16,sUTF16_count,e,unicode_code_point);
#endif
}

int ON_DecodeUTF16BE(
  const ON__UINT16* sUTF16,
  int sUTF16_count,
  struct ON_UnicodeErrorParameters* e,
  ON__UINT32* unicode_code_point
  )
{
#if defined(ON_BIG_ENDIAN)
  return ON_DecodeUTF16(sUTF16,sUTF16_count,e,unicode_code_point);
#else
  return ON_DecodeSwapByteUTF16(sUTF16,sUTF16_count,e,unicode_code_point);
#endif
}

int ON_DecodeUTF32LE(
  const ON__UINT32* sUTF32,
  int sUTF32_count,
  struct ON_UnicodeErrorParameters* e,
  ON__UINT32* unicode_code_point
  )
{
#if defined(ON_LITTLE_ENDIAN)
  return ON_DecodeUTF32(sUTF32,sUTF32_count,e,unicode_code_point);
#else
  return ON_DecodeSwapByteUTF32(sUTF32,sUTF32_count,e,unicode_code_point);
#endif
}

int ON_DecodeUTF32BE(
  const ON__UINT32* sUTF32,
  int sUTF32_count,
  struct ON_UnicodeErrorParameters* e,
  ON__UINT32* unicode_code_point
  )
{
#if defined(ON_BIG_ENDIAN)
  return ON_DecodeUTF32(sUTF32,sUTF32_count,e,unicode_code_point);
#else
  return ON_DecodeSwapByteUTF32(sUTF32,sUTF32_count,e,unicode_code_point);
#endif
}


int ON_EncodeWideChar(
  ON__UINT32 code_point,
  size_t w_capacity,
  wchar_t* w
  )
{
  int rc = 0;
  if (nullptr != w && w_capacity > 0)
  {
    if (ON_IsValidUnicodeCodePoint(code_point))
    {
#if 1 == ON_SIZEOF_WCHAR_T
      char sUTF8[6];
      rc = ON_EncodeUTF8(code_point, sUTF8);
      if (rc > (int)w_capacity)
        rc = 0;
      switch (rc)
      {
      case 1:
        w[0] = (wchar_t)sUTF8[0];
        break;
      case 2:
        w[0] = (wchar_t)sUTF8[0];
        w[1] = (wchar_t)sUTF8[1];
        break;
      case 3:
        w[0] = (wchar_t)sUTF8[0];
        w[1] = (wchar_t)sUTF8[1];
        w[2] = (wchar_t)sUTF8[2];
        break;
      case 4:
        w[0] = (wchar_t)sUTF8[0];
        w[1] = (wchar_t)sUTF8[1];
        w[2] = (wchar_t)sUTF8[2];
        w[3] = (wchar_t)sUTF8[3];
        break;
      default:
        rc = 0; break;
      }
#elif 2 == ON_SIZEOF_WCHAR_T
      ON__UINT16 sUTF16[2];
      rc = ON_EncodeUTF16(code_point, sUTF16);
      if (rc > (int)w_capacity)
        rc = 0;
      switch (rc)
      {
      case 1:
        w[0] = (wchar_t)sUTF16[0];
        break;
      case 2:
        w[0] = (wchar_t)sUTF16[0];
        w[1] = (wchar_t)sUTF16[1];
        break;
      default:
        rc = 0; break;
      }
#elif 4 == ON_SIZEOF_WCHAR_T
      if (w_capacity > 0)
      {
        w[0] = (wchar_t)code_point;
        rc = 1;
      }
#endif
    }
    if (rc >= 0 && rc < (int)w_capacity)
      w[rc] = 0;
  }
  return rc;
}

int ON_DecodeWideChar(
  const wchar_t* sWideChar,
  int sWideChar_count,
  struct ON_UnicodeErrorParameters* e,
  ON__UINT32* unicode_code_point
  )
{
  const ON_UnicodeEncoding widechar_encoding = ON_WCHAR_T_ENCODING;
  int rc;

  switch (widechar_encoding)
  {
#if 1 == ON_SIZEOF_WCHAR_T
  case ON_UTF_8:
    rc = ON_DecodeUTF8((const char*)sWideChar,sWideChar_count,e,unicode_code_point);
    break;

#elif 2 == ON_SIZEOF_WCHAR_T
  case ON_UTF_16:
    return ON_DecodeUTF16((const ON__UINT16*)sWideChar,sWideChar_count,e,unicode_code_point);
    break;

  case ON_UTF_16BE:
    rc = ON_DecodeUTF16BE((const ON__UINT16*)sWideChar,sWideChar_count,e,unicode_code_point);
    break;

  case ON_UTF_16LE:
    rc = ON_DecodeUTF16LE((const ON__UINT16*)sWideChar,sWideChar_count,e,unicode_code_point);
    break;

#elif 4 == ON_SIZEOF_WCHAR_T
  case ON_UTF_32:
    rc = ON_DecodeUTF32((const ON__UINT32*)sWideChar,sWideChar_count,e,unicode_code_point);
    break;

  case ON_UTF_32BE:
    rc = ON_DecodeUTF32BE((const ON__UINT32*)sWideChar,sWideChar_count,e,unicode_code_point);
    break;

  case ON_UTF_32LE:
    rc = ON_DecodeUTF32LE((const ON__UINT32*)sWideChar,sWideChar_count,e,unicode_code_point);
    break;
#endif

  default:
    rc = 0;
    if ( e )
      e->m_error_status |= 1;
    break;
  }

  return rc;
}


int ON_DecodeSwapByteUTF16(
    const ON__UINT16* sUTF16,
    int sUTF16_count,
    struct ON_UnicodeErrorParameters* e,
    ON__UINT32* unicode_code_point
    )
{
  int i;
  ON__UINT32 uhi, ulo;
  ON__UINT16 w0, w1;
  const ON__UINT8* p;
  ON__UINT8* p0;
  ON__UINT8* p1;

  ON_UnicodeErrorParameters local_e = ON_UnicodeErrorParameters::MaskErrors;
  if (nullptr == e)
    e = &local_e;

  if ( 0 == sUTF16 || sUTF16_count <= 0 || 0 == unicode_code_point )
  {
    if ( e )
      e->m_error_status |= 1;
    return 0;
  }

  // special case for most common UTF-16 single element values
  // w0 = byte swapped sUTF16[0]
  p = (const ON__UINT8*)sUTF16;
  p0 = (ON__UINT8*)&w0;
  p0[1] = p[0];
  p0[0] = p[1];
  if ( ( w0 < 0xD800 ) || (w0 >= 0xE000 ) )
  {
    *unicode_code_point = w0;
    return 1;
  }

  if ( sUTF16_count >= 2 && w0 < 0xDC00 )
  {
    // w1 = byte swapped sUTF16[1]
    p1 = (ON__UINT8*)&w1;
    p1[1] = p[2];
    p1[0] = p[3];
    if ( w1 >=  0xDC00 && w1 < 0xE000 )
    {
      // UTF-16 surrogate pair
      uhi = w0;
      ulo = w1;
      *unicode_code_point = (uhi-0xD800)*0x400 + (ulo-0xDC00) + 0x10000;
      return 2;
    }
  }

  // handle errors
  if ( 0 == e )
  {
    // no errors are masked.
    return 0;
  }

  // report error condition
  e->m_error_status |= 16;

  if ( 16 != (16 & e->m_error_mask) || !ON_IsValidUnicodeCodePoint(e->m_error_code_point) )
  {
    // this error is not masked
    return 0;
  }

  // Search for the next element of sUTF16[] that is a
  // valid UTF-16 encoding sequence.
  p1 = (ON__UINT8*)&w1;
  p += sizeof(sUTF16[0]);
  for ( i = 1; i < sUTF16_count; i++, p += sizeof(sUTF16[0]) )
  {
    // w0 = byte swapped sUTF16[i]
    p0[1] = p[0];
    p0[0] = p[1];
    if ( ( w0 < 0xD800 ) || ( w0 >= 0xE000 ) )
    {
      // valid single UTF-16 code unit
      break;
    }
    if ( i+1 < sUTF16_count && w0 >= 0xD800 && w0 < 0xDC00 )
    {
      // w1 = byte swapped sUTF16[i+1]
      p1[1] = p[sizeof(sUTF16[0])];
      p1[0] = p[sizeof(sUTF16[0])+1];
      if ( w1 >= 0xDC00 && w1 < 0xE000 )
      {
        // valid UTF-16 surrogate pair
        break;
      }
    }
  }

  *unicode_code_point = e->m_error_code_point;

  return i;
}

int ON_ConvertUTF8ToUTF8(
    int bTestByteOrder,
    const char* sInputUTF8,
    int sInputUTF8_count,
    char* sOutputUTF8,
    int sOutputUTF8_count,
    unsigned int* error_status,
    unsigned int error_mask,
    ON__UINT32 error_code_point,
    const char** sNextInputUTF8
    )
{
  int i, j, k, output_count;
  ON__UINT32 u;
  char s[6];
  struct ON_UnicodeErrorParameters e;

  if ( 0 != error_status )
    *error_status = 0;

  if ( -1 == sInputUTF8_count && 0 != sInputUTF8 )
  {
    for ( sInputUTF8_count = 0; 0 != sInputUTF8[sInputUTF8_count]; sInputUTF8_count++)
    {
      // empty for body
    }
  }

  if ( 0 == sInputUTF8 || sInputUTF8_count < 0 )
  {
    if ( 0 != error_status )
      *error_status |= 1;
    if ( sNextInputUTF8 )
      *sNextInputUTF8 = sInputUTF8;
    return 0;
  }

  if ( 0 == sOutputUTF8_count )
  {
    sOutputUTF8 = 0;
    sOutputUTF8_count = 2147483647; // maximum value of a 32-bit signed int
  }
  else if ( 0 == sOutputUTF8 )
  {
    if ( 0 != error_status )
      *error_status |= 1;
    if ( sNextInputUTF8 )
      *sNextInputUTF8 = sInputUTF8;
    return 0;
  }

  if ( bTestByteOrder && ON_IsUTF8ByteOrderMark(sInputUTF8,sInputUTF8_count) )
  {
    // skip UTF-8 byte order element
    sInputUTF8_count -= 3;
    sInputUTF8 += 3;
  }

  e.m_error_status = 0;
  e.m_error_mask = error_mask;
  e.m_error_code_point = error_code_point;

  output_count = 0;

  for ( i = 0; i < sInputUTF8_count; i += j )
  {
    j = ON_DecodeUTF8(sInputUTF8+i,sInputUTF8_count-i,&e,&u);
    if ( j <= 0 )
      break;
    k = ON_EncodeUTF8(u,s);
    if ( 0 != sOutputUTF8 )
    {
      if ( output_count + k > sOutputUTF8_count )
      {
        e.m_error_status |= 2;
        break;
      }
      memcpy(sOutputUTF8+output_count,s,k*sizeof(sOutputUTF8[0]));
    }
    output_count += k;
  }

  if ( 0 != sOutputUTF8 && output_count < sOutputUTF8_count)
    sOutputUTF8[output_count] = 0;
  if ( sNextInputUTF8 )
    *sNextInputUTF8 = sInputUTF8+i;
  if ( error_status )
    *error_status = e.m_error_status;

  return output_count;
}


int ON_ConvertUTF8ToUTF16(
    int bTestByteOrder,
     const char* sUTF8,
    int sUTF8_count,
    ON__UINT16* sUTF16,
    int sUTF16_count,
    unsigned int* error_status,
    unsigned int error_mask,
    ON__UINT32 error_code_point,
    const char** sNextUTF8
    )
{
  int i, j, k, output_count;
  ON__UINT32 u;
  ON__UINT16 w[2];
  struct ON_UnicodeErrorParameters e;

  if ( 0 != error_status )
    *error_status = 0;

  if ( -1 == sUTF8_count && 0 != sUTF8 )
  {
    for ( sUTF8_count = 0; 0 != sUTF8[sUTF8_count]; sUTF8_count++)
    {
      // empty for body
    }
  }

  if ( 0 == sUTF8 || sUTF8_count < 0 )
  {
    if ( 0 != error_status )
      *error_status |= 1;
    if ( sNextUTF8 )
      *sNextUTF8 = sUTF8;
    return 0;
  }

  if ( bTestByteOrder && ON_IsUTF8ByteOrderMark(sUTF8,sUTF8_count) )
  {
    // skip UTF-8 byte order element
    sUTF8_count -= 3;
    sUTF8 += 3;
  }

  if ( 0 == sUTF16_count )
  {
    sUTF16 = 0;
    sUTF16_count = 2147483647; // maximum value of a 32-bit signed int
  }
  else if ( 0 == sUTF16 )
  {
    if ( 0 != error_status )
      *error_status |= 1;
    if ( sNextUTF8 )
      *sNextUTF8 = sUTF8;
    return 0;
  }

  e.m_error_status = 0;
  e.m_error_mask = error_mask;
  e.m_error_code_point = error_code_point;

  output_count = 0;

  for ( i = 0; i < sUTF8_count; i += j )
  {
    j = ON_DecodeUTF8(sUTF8+i,sUTF8_count-i,&e,&u);
    if ( j <= 0 )
      break;
    k = ON_EncodeUTF16(u,w);
    if ( 0 != sUTF16 )
    {
      if ( output_count + k > sUTF16_count )
      {
        e.m_error_status |= 2;
        break;
      }
      sUTF16[output_count] = w[0];
      if ( 2 == k )
        sUTF16[output_count+1] = w[1];
    }
    output_count += k;
  }

  if ( 0 != sUTF16 && output_count < sUTF16_count)
    sUTF16[output_count] = 0;
  if ( sNextUTF8 )
    *sNextUTF8 = sUTF8+i;
  if ( error_status )
    *error_status = e.m_error_status;

  return output_count;
}

int ON_ConvertUTF8ToUTF32(
    int bTestByteOrder,
    const char* sUTF8,
    int sUTF8_count,
    ON__UINT32* sUTF32,
    int sUTF32_count,
    unsigned int* error_status,
    unsigned int error_mask,
    ON__UINT32 error_code_point,
    const char** sNextUTF8
    )
{
  int i, j, output_count;
  ON__UINT32 u;
  struct ON_UnicodeErrorParameters e;

  if ( 0 != error_status )
    *error_status = 0;

  if ( -1 == sUTF8_count && 0 != sUTF8 )
  {
    for ( sUTF8_count = 0; 0 != sUTF8[sUTF8_count]; sUTF8_count++)
    {
      // empty for body
    }
  }

  if ( 0 == sUTF8 || sUTF8_count < 0 )
  {
    if ( 0 != error_status )
      *error_status |= 1;
    if ( sNextUTF8 )
      *sNextUTF8 = sUTF8;
    return 0;
  }

  if ( bTestByteOrder && ON_IsUTF8ByteOrderMark(sUTF8,sUTF8_count) )
  {
    // skip UTF-8 byte order element
    sUTF8_count -= 3;
    sUTF8 += 3;
  }

  if ( 0 == sUTF32_count )
  {
    sUTF32 = 0;
    sUTF32_count = 2147483647; // maximum value of a 32-bit signed int
  }
  else if ( 0 == sUTF32 )
  {
    if ( 0 != error_status )
      *error_status |= 1;
    if ( sNextUTF8 )
      *sNextUTF8 = sUTF8;
    return 0;
  }

  e.m_error_status = 0;
  e.m_error_mask = error_mask;
  e.m_error_code_point = error_code_point;

  output_count = 0;

  for ( i = 0; i < sUTF8_count; i += j )
  {
    j = ON_DecodeUTF8(sUTF8+i,sUTF8_count-i,&e,&u);
    if ( j <= 0 )
      break;
    if ( 0 != sUTF32 )
    {
      if ( output_count >= sUTF32_count )
      {
        e.m_error_status |= 2;
        break;
      }
      sUTF32[output_count] = u;
    }
    output_count++;
  }

  if ( 0 != sUTF32 && output_count < sUTF32_count)
    sUTF32[output_count] = 0;
  if ( sNextUTF8 )
    *sNextUTF8 = sUTF8+i;
  if ( error_status )
    *error_status = e.m_error_status;

  return output_count;
}

int ON_ConvertUTF16ToUTF8(
    int bTestByteOrder,
    const ON__UINT16* sUTF16,
    int sUTF16_count,
    char* sUTF8,
    int sUTF8_count,
    unsigned int* error_status,
    unsigned int error_mask,
    ON__UINT32 error_code_point,
    const ON__UINT16** sNextUTF16
    )
{
  int i, j, k, output_count, bSwapBytes;
  ON__UINT32 u;
  char s[6];
  struct ON_UnicodeErrorParameters e;

  if ( 0 != error_status )
    *error_status = 0;

  if ( -1 == sUTF16_count && 0 != sUTF16 )
  {
    for ( sUTF16_count = 0; 0 != sUTF16[sUTF16_count]; sUTF16_count++)
    {
      // empty for body
    }
  }

  if ( 0 == sUTF16 || sUTF16_count < 0 )
  {
    if ( 0 != error_status )
      *error_status |= 1;
    if ( sNextUTF16 )
      *sNextUTF16 = sUTF16;
    return 0;
  }

  if ( 0 == sUTF8_count )
  {
    sUTF8 = 0;
    sUTF8_count = 2147483647; // maximum value of a 32-bit signed int
  }
  else if ( 0 == sUTF8 )
  {
    if ( 0 != error_status )
      *error_status |= 1;
    if ( sNextUTF16 )
      *sNextUTF16 = sUTF16;
    return 0;
  }

  bSwapBytes = false;
  if ( bTestByteOrder && sUTF16_count > 0 )
  {
    if ( 0xFEFF == sUTF16[0] )
    {
      // skip BOM
      sUTF16_count--;
      sUTF16++;
    }
    else if ( 0xFFFE == sUTF16[0] )
    {
      // skip BOM and swap bytes in rest of sUTF16
      bSwapBytes = true;
      sUTF16_count--;
      sUTF16++;
    }
  }

  e.m_error_status = 0;
  e.m_error_mask = error_mask;
  e.m_error_code_point = error_code_point;

  output_count = 0;

  if ( bSwapBytes )
  {
    for ( i = 0; i < sUTF16_count; i += j )
    {
      j = ON_DecodeSwapByteUTF16(sUTF16+i,sUTF16_count-i,&e,&u);
      if ( j <= 0 )
        break;
      k = ON_EncodeUTF8(u,s);
      if ( 0 != sUTF8 )
      {
        if ( output_count + k > sUTF8_count )
        {
          e.m_error_status |= 2;
          break;
        }
        memcpy(sUTF8+output_count,s,k*sizeof(sUTF8[0]));
      }
      output_count += k;
    }
  }
  else
  {
    for ( i = 0; i < sUTF16_count; i += j )
    {
      j = ON_DecodeUTF16(sUTF16+i,sUTF16_count-i,&e,&u);
      if ( j <= 0 )
        break;
      k = ON_EncodeUTF8(u,s);
      if ( 0 != sUTF8 )
      {
        if ( output_count + k > sUTF8_count )
        {
          e.m_error_status |= 2;
          break;
        }
        memcpy(sUTF8+output_count,s,k*sizeof(sUTF8[0]));
      }
      output_count += k;
    }
  }
  if ( 0 != sUTF8 && output_count < sUTF8_count)
    sUTF8[output_count] = 0;
  if ( sNextUTF16 )
    *sNextUTF16 = sUTF16+i;
  if ( error_status )
    *error_status = e.m_error_status;

  return output_count;
}

int ON_ConvertUTF16ToUTF16(
    int bTestByteOrder,
    const ON__UINT16* sInputUTF16,
    int sInputUTF16_count,
    ON__UINT16* sOutputUTF16,
    int sOutputUTF16_count,
    unsigned int* error_status,
    unsigned int error_mask,
    ON__UINT32 error_code_point,
    const ON__UINT16** sNextInputUTF16
    )
{
  int i, j, k, output_count, bSwapBytes;
  ON__UINT32 u;
  ON__UINT16 s[2];
  struct ON_UnicodeErrorParameters e;

  if ( 0 != error_status )
    *error_status = 0;

  if ( -1 == sInputUTF16_count && 0 != sInputUTF16 )
  {
    for ( sInputUTF16_count = 0; 0 != sInputUTF16[sInputUTF16_count]; sInputUTF16_count++)
    {
      // empty for body
    }
  }

  if ( 0 == sInputUTF16 || sInputUTF16_count < 0 )
  {
    if ( 0 != error_status )
      *error_status |= 1;
    if ( sNextInputUTF16 )
      *sNextInputUTF16 = sInputUTF16;
    return 0;
  }

  if ( 0 == sOutputUTF16_count )
  {
    sOutputUTF16 = 0;
    sOutputUTF16_count = 2147483647; // maximum value of a 32-bit signed int
  }
  else if ( 0 == sOutputUTF16 )
  {
    if ( 0 != error_status )
      *error_status |= 1;
    if ( sNextInputUTF16 )
      *sNextInputUTF16 = sInputUTF16;
    return 0;
  }

  bSwapBytes = false;
  if ( bTestByteOrder && sInputUTF16_count > 0 )
  {
    if ( 0xFEFF == sInputUTF16[0] )
    {
      // skip BOM
      sInputUTF16_count--;
      sInputUTF16++;
    }
    else if ( 0xFFFE == sInputUTF16[0] )
    {
      // skip BOM and swap bytes in rest of sInputUTF16
      bSwapBytes = true;
      sInputUTF16_count--;
      sInputUTF16++;
    }
  }

  e.m_error_status = 0;
  e.m_error_mask = error_mask;
  e.m_error_code_point = error_code_point;

  output_count = 0;

  if ( bSwapBytes )
  {
    for ( i = 0; i < sInputUTF16_count; i += j )
    {
      j = ON_DecodeSwapByteUTF16(sInputUTF16+i,sInputUTF16_count-i,&e,&u);
      if ( j <= 0 )
        break;
      k = ON_EncodeUTF16(u,s);
      if ( 0 != sOutputUTF16 )
      {
        if ( output_count + k > sOutputUTF16_count )
        {
          e.m_error_status |= 2;
          break;
        }
        memcpy(sOutputUTF16+output_count,s,k*sizeof(sOutputUTF16[0]));
      }
      output_count += k;
    }
  }
  else
  {
    for ( i = 0; i < sInputUTF16_count; i += j )
    {
      j = ON_DecodeUTF16(sInputUTF16+i,sInputUTF16_count-i,&e,&u);
      if ( j <= 0 )
        break;
      k = ON_EncodeUTF16(u,s);
      if ( 0 != sOutputUTF16 )
      {
        if ( output_count + k > sOutputUTF16_count )
        {
          e.m_error_status |= 2;
          break;
        }
        memcpy(sOutputUTF16+output_count,s,k*sizeof(sOutputUTF16[0]));
      }
      output_count += k;
    }
  }
  if ( 0 != sOutputUTF16 && output_count < sOutputUTF16_count)
    sOutputUTF16[output_count] = 0;
  if ( sNextInputUTF16 )
    *sNextInputUTF16 = sInputUTF16+i;
  if ( error_status )
    *error_status = e.m_error_status;

  return output_count;
}

int ON_ConvertUTF16ToUTF32(
    int bTestByteOrder,
    const ON__UINT16* sUTF16,
    int sUTF16_count,
    unsigned int* sUTF32,
    int sUTF32_count,
    unsigned int* error_status,
    unsigned int error_mask,
    ON__UINT32 error_code_point,
    const ON__UINT16** sNextUTF16
    )
{
  int i, j, output_count, bSwapBytes;
  ON__UINT32 u;
  struct ON_UnicodeErrorParameters e;

  if ( 0 != error_status )
    *error_status = 0;

  if ( -1 == sUTF16_count && 0 != sUTF16 )
  {
    for ( sUTF16_count = 0; 0 != sUTF16[sUTF16_count]; sUTF16_count++)
    {
      // empty for body
    }
  }

  if ( 0 == sUTF16 || sUTF16_count < 0 )
  {
    if ( 0 != error_status )
      *error_status |= 1;
    if ( sNextUTF16 )
      *sNextUTF16 = sUTF16;
    return 0;
  }

  if ( 0 == sUTF32_count )
  {
    sUTF32 = 0;
    sUTF32_count = 2147483647; // maximum value of a 32-bit signed int
  }
  else if ( 0 == sUTF32 )
  {
    if ( 0 != error_status )
      *error_status |= 1;
    if ( sNextUTF16 )
      *sNextUTF16 = sUTF16;
    return 0;
  }

  bSwapBytes = false;
  if ( bTestByteOrder && sUTF16_count > 0 )
  {
    if ( 0xFEFF == sUTF16[0] )
    {
      // skip BOM
      sUTF16_count--;
      sUTF16++;
    }
    else if ( 0xFFFE == sUTF16[0] )
    {
      // skip BOM and swap bytes in rest of sUTF16
      bSwapBytes = true;
      sUTF16_count--;
      sUTF16++;
    }
  }

  e.m_error_status = 0;
  e.m_error_mask = error_mask;
  e.m_error_code_point = error_code_point;

  output_count = 0;

  if ( bSwapBytes )
  {
    for ( i = 0; i < sUTF16_count; i += j )
    {
      j = ON_DecodeSwapByteUTF16(sUTF16+i,sUTF16_count-i,&e,&u);
      if ( j <= 0 )
        break;
      if ( 0 != sUTF32 )
      {
        if ( output_count >= sUTF32_count )
        {
          e.m_error_status |= 2;
          break;
        }
        sUTF32[output_count] = u;
      }
      output_count++;
    }
  }
  else
  {
    for ( i = 0; i < sUTF16_count; i += j )
    {
      j = ON_DecodeUTF16(sUTF16+i,sUTF16_count-i,&e,&u);
      if ( j <= 0 )
        break;
      if ( 0 != sUTF32 )
      {
        if ( output_count >= sUTF32_count )
        {
          e.m_error_status |= 2;
          break;
        }
        sUTF32[output_count] = u;
      }
      output_count++;
    }
  }

  if ( 0 != sUTF32 && output_count < sUTF32_count)
    sUTF32[output_count] = 0;
  if ( sNextUTF16 )
    *sNextUTF16 = sUTF16+i;
  if ( error_status )
    *error_status = e.m_error_status;

  return output_count;
}

static ON__UINT32 SwapBytes32(ON__UINT32 u)
{
  ON__UINT8 b;
  ON__UINT8* p = (ON__UINT8*)&u;
  b = p[0]; p[0] = p[3]; p[3] = b;
  b = p[1]; p[1] = p[2]; p[2] = b;
  return u;
}

int ON_DecodeUTF32(
    const ON__UINT32* sUTF32,
    int sUTF32_count,
    struct ON_UnicodeErrorParameters* e,
    ON__UINT32* unicode_code_point
    )
{
  ON__UINT32 uhi, ulo;

  ON_UnicodeErrorParameters local_e = ON_UnicodeErrorParameters::MaskErrors;
  if (nullptr == e)
    e = &local_e;

  if ( 0 == sUTF32 || sUTF32_count <= 0 || 0 == unicode_code_point )
  {
    e->m_error_status |= 1;
    return 0;
  }

  // special case for most common UTF-16 single element values
  if ( ( sUTF32[0] < 0xD800 ) || ( sUTF32[0] >= 0xE000 && sUTF32[0] <= 0x10FFFF) )
  {
    // valid UTF-32 encoding.
    *unicode_code_point = sUTF32[0];
    return 1;
  }

  // handle errors
  if ( 0 == e )
    return 0;

  if ( sUTF32_count >= 2 && sUTF32[0] < 0xDC00 && sUTF32[1] >=  0xDC00 && sUTF32[1] < 0xE000 )
  {
    // UTF-16 surrogate pair appears in UTF-32 array
    e->m_error_status |= 4;
    if ( 0 == (4 & e->m_error_mask) )
      return 0; // this error is not masked.

    uhi = sUTF32[0];
    ulo = sUTF32[1];
    *unicode_code_point = (uhi-0xD800)*0x400 + (ulo-0xDC00) + 0x10000;

    return 2; // error masked and reasonable value returned.
  }

  // bogus value
  e->m_error_status |= 16;
  if ( 16 != (16 & e->m_error_mask) || !ON_IsValidUnicodeCodePoint(e->m_error_code_point) )
  {
    // this error is not masked
    return 0;
  }

  *unicode_code_point = e->m_error_code_point;
  return 1; // error masked and e->m_error_code_point returned.
}

int ON_DecodeSwapByteUTF32(
    const ON__UINT32* sUTF32,
    int sUTF32_count,
    struct ON_UnicodeErrorParameters* e,
    ON__UINT32* unicode_code_point
    )
{
  ON__UINT32 sUTF32swap[2];

  ON_UnicodeErrorParameters local_e = ON_UnicodeErrorParameters::MaskErrors;
  if (nullptr == e)
    e = &local_e;

  if ( 0 != sUTF32 && sUTF32_count > 0 )
  {
    sUTF32swap[0] = SwapBytes32(sUTF32[0]);
    if ( sUTF32_count > 1 )
    {
      // Get up to 2 elements to pass to the unswapped
      // decoder so that masked errors are uniformly
      // handled.
      sUTF32swap[1] = SwapBytes32(sUTF32[1]);
      sUTF32_count = 2;
    }
    sUTF32 = sUTF32swap;
  }

  return ON_DecodeUTF32(sUTF32,sUTF32_count,e,unicode_code_point);
}

int ON_ConvertUTF32ToUTF8(
    int bTestByteOrder,
    const ON__UINT32* sUTF32,
    int sUTF32_count,
    char* sUTF8,
    int sUTF8_count,
    unsigned int* error_status,
    unsigned int error_mask,
    ON__UINT32 error_code_point,
    const ON__UINT32** sNextUTF32
    )
{
  int i, k, output_count, bSwapBytes;
  ON__UINT32 u;
  char s[6];
  struct ON_UnicodeErrorParameters e;

  if ( 0 != error_status )
    *error_status = 0;

  if ( -1 == sUTF32_count && 0 != sUTF32 )
  {
    for ( sUTF32_count = 0; 0 != sUTF32[sUTF32_count]; sUTF32_count++)
    {
      // empty for body
    }
  }

  if ( 0 == sUTF32 || sUTF32_count < 0 )
  {
    if ( 0 != error_status )
      *error_status |= 1;
    if ( sNextUTF32 )
      *sNextUTF32 = sUTF32;
    return 0;
  }

  if ( 0 == sUTF8_count )
  {
    sUTF8 = 0;
    sUTF8_count = 2147483647; // maximum value of a 32-bit signed int
  }
  else if ( 0 == sUTF8 )
  {
    if ( 0 != error_status )
      *error_status |= 1;
    if ( sNextUTF32 )
      *sNextUTF32 = sUTF32;
    return 0;
  }

  bSwapBytes = false;
  if ( bTestByteOrder && sUTF32_count > 0 )
  {
    if ( 0x0000FEFF == sUTF32[0] )
    {
      // skip BOM
      sUTF32_count--;
      sUTF32++;
    }
    else if ( 0xFFFE0000 == sUTF32[0] )
    {
      // skip BOM and swap bytes in rest of sUTF32
      bSwapBytes = true;
      sUTF32_count--;
      sUTF32++;
    }
  }

  e.m_error_status = 0;
  e.m_error_mask = error_mask;
  e.m_error_code_point = error_code_point;

  output_count = 0;

  for ( i = 0; i < sUTF32_count; i++ )
  {
    u = bSwapBytes ? SwapBytes32(sUTF32[i]) : sUTF32[i];
    if ( !ON_IsValidUnicodeCodePoint(u) )
    {
      e.m_error_status |= 16;
      if ( 16 != (16 & e.m_error_mask) )
        break;
      if ( !ON_IsValidUnicodeCodePoint(e.m_error_code_point) )
        break;
      u = e.m_error_code_point;
    }
    k = ON_EncodeUTF8(u,s);
    if ( 0 != sUTF8 )
    {
      if ( output_count + k > sUTF8_count )
      {
        e.m_error_status |= 2;
        break;
      }
      memcpy(sUTF8+output_count,s,k*sizeof(sUTF8[0]));
    }
    output_count += k;
  }

  if ( 0 != sUTF8 && output_count < sUTF8_count)
    sUTF8[output_count] = 0;
  if ( sNextUTF32 )
    *sNextUTF32 = sUTF32+i;
  if ( error_status )
    *error_status = e.m_error_status;

  return output_count;
}

int ON_ConvertUTF32ToUTF16(
    int bTestByteOrder,
    const ON__UINT32* sUTF32,
    int sUTF32_count,
    ON__UINT16* sUTF16,
    int sUTF16_count,
    unsigned int* error_status,
    unsigned int error_mask,
    ON__UINT32 error_code_point,
    const ON__UINT32** sNextUTF32
    )
{
  int i, k, output_count, bSwapBytes;
  ON__UINT32 u;
  ON__UINT16 w[2];
  struct ON_UnicodeErrorParameters e;

  if ( 0 != error_status )
    *error_status = 0;

  if ( -1 == sUTF32_count && 0 != sUTF32 )
  {
    for ( sUTF32_count = 0; 0 != sUTF32[sUTF32_count]; sUTF32_count++)
    {
      // empty for body
    }
  }

  if ( 0 == sUTF32 || sUTF32_count < 0 )
  {
    if ( 0 != error_status )
      *error_status |= 1;
    if ( sNextUTF32 )
      *sNextUTF32 = sUTF32;
    return 0;
  }

  if ( 0 == sUTF16_count )
  {
    sUTF16 = 0;
    sUTF16_count = 2147483647; // maximum value of a 32-bit signed int
  }
  else if ( 0 == sUTF16 )
  {
    if ( 0 != error_status )
      *error_status |= 1;
    if ( sNextUTF32 )
      *sNextUTF32 = sUTF32;
    return 0;
  }

  bSwapBytes = false;
  if ( bTestByteOrder && sUTF32_count > 0 )
  {
    if ( 0x0000FEFF == sUTF32[0] )
    {
      // skip BOM
      sUTF32_count--;
      sUTF32++;
    }
    else if ( 0xFFFE0000 == sUTF32[0] )
    {
      // skip BOM and swap bytes in rest of sUTF32
      bSwapBytes = true;
      sUTF32_count--;
      sUTF32++;
    }
  }

  e.m_error_status = 0;
  e.m_error_mask = error_mask;
  e.m_error_code_point = error_code_point;

  output_count = 0;

  for ( i = 0; i < sUTF32_count; i++ )
  {
    u = bSwapBytes ? SwapBytes32(sUTF32[i]) : sUTF32[i];
    if ( !ON_IsValidUnicodeCodePoint(u) )
    {
      e.m_error_status |= 16;
      if ( 16 != (16 & e.m_error_mask) )
        break;
      if ( !ON_IsValidUnicodeCodePoint(e.m_error_code_point) )
        break;
      u = e.m_error_code_point;
    }
    k = ON_EncodeUTF16(u,w);
    if ( 0 != sUTF16 )
    {
      if ( output_count + k > sUTF16_count )
      {
        e.m_error_status |= 2;
        break;
      }
      sUTF16[output_count] = w[0];
      if ( 2 == k )
        sUTF16[output_count+1] = w[1];
    }
    output_count += k;
  }

  if ( 0 != sUTF16 && output_count < sUTF16_count)
    sUTF16[output_count] = 0;
  if ( sNextUTF32 )
    *sNextUTF32 = sUTF32+i;
  if ( error_status )
    *error_status = e.m_error_status;

  return output_count;
}


int ON_ConvertUTF32ToUTF32(
    int bTestByteOrder,
    const ON__UINT32* sUTF16,
    int sUTF16_count,
    unsigned int* sUTF32,
    int sUTF32_count,
    unsigned int* error_status,
    unsigned int error_mask,
    ON__UINT32 error_code_point,
    const ON__UINT32** sNextUTF16
    )
{
  int i, j, output_count, bSwapBytes;
  ON__UINT32 u;
  struct ON_UnicodeErrorParameters e;

  if ( 0 != error_status )
    *error_status = 0;

  if ( -1 == sUTF16_count && 0 != sUTF16 )
  {
    for ( sUTF16_count = 0; 0 != sUTF16[sUTF16_count]; sUTF16_count++)
    {
      // empty for body
    }
  }

  if ( 0 == sUTF16 || sUTF16_count < 0 )
  {
    if ( 0 != error_status )
      *error_status |= 1;
    if ( sNextUTF16 )
      *sNextUTF16 = sUTF16;
    return 0;
  }

  if ( 0 == sUTF32_count )
  {
    sUTF32 = 0;
    sUTF32_count = 2147483647; // maximum value of a 32-bit signed int
  }
  else if ( 0 == sUTF32 )
  {
    if ( 0 != error_status )
      *error_status |= 1;
    if ( sNextUTF16 )
      *sNextUTF16 = sUTF16;
    return 0;
  }

  bSwapBytes = false;
  if ( bTestByteOrder && sUTF16_count > 0 )
  {
    if ( 0x0000FEFF == sUTF16[0] )
    {
      // skip BOM
      sUTF16_count--;
      sUTF16++;
    }
    else if ( 0xFFFE0000 == sUTF16[0])
    {
      // skip BOM and swap bytes in rest of sUTF16
      bSwapBytes = true;
      sUTF16_count--;
      sUTF16++;
    }
  }

  e.m_error_status = 0;
  e.m_error_mask = error_mask;
  e.m_error_code_point = error_code_point;

  output_count = 0;

  if ( bSwapBytes )
  {
    for ( i = 0; i < sUTF16_count; i += j )
    {
      j = ON_DecodeSwapByteUTF32(sUTF16+i,sUTF16_count-i,&e,&u);
      if ( j <= 0 )
        break;
      if ( 0 != sUTF32 )
      {
        if ( output_count >= sUTF32_count )
        {
          e.m_error_status |= 2;
          break;
        }
        sUTF32[output_count] = u;
      }
      output_count++;
    }
  }
  else
  {
    for ( i = 0; i < sUTF16_count; i += j )
    {
      j = ON_DecodeUTF32(sUTF16+i,sUTF16_count-i,&e,&u);
      if ( j <= 0 )
        break;
      if ( 0 != sUTF32 )
      {
        if ( output_count >= sUTF32_count )
        {
          e.m_error_status |= 2;
          break;
        }
        sUTF32[output_count] = u;
      }
      output_count++;
    }
  }

  if ( 0 != sUTF32 && output_count < sUTF32_count)
    sUTF32[output_count] = 0;
  if ( sNextUTF16 )
    *sNextUTF16 = sUTF16+i;
  if ( error_status )
    *error_status = e.m_error_status;

  return output_count;
}

int ON_ConvertWideCharToUTF8(
    int bTestByteOrder,
    const wchar_t* sWideChar,
    int sWideChar_count,
    char* sUTF8,
    int sUTF8_count,
    unsigned int* error_status,
    unsigned int error_mask,
    ON__UINT32 error_code_point,
    const wchar_t** sNextWideChar
    )
{
  int rc;

  switch(sizeof(sWideChar[0]))
  {
  case sizeof(char):
    // assume wchar_t strings are UTF-8 encoded
    rc = ON_ConvertUTF8ToUTF8(
            bTestByteOrder,
            (const char*)sWideChar,sWideChar_count,
            sUTF8,sUTF8_count,
            error_status,error_mask,error_code_point,
            (const char**)sNextWideChar
            );
    break;

  case sizeof(ON__UINT16):
    // assume wchar_t strings are UTF-16 encoded
    rc = ON_ConvertUTF16ToUTF8(
            bTestByteOrder,
            (const ON__UINT16*)sWideChar,sWideChar_count,
            sUTF8,sUTF8_count,
            error_status,error_mask,error_code_point,
            (const ON__UINT16**)sNextWideChar
            );
    break;

  case sizeof(ON__UINT32):
    // assume wchar_t strings are UTF-32 encoded
    rc = ON_ConvertUTF32ToUTF8(
            bTestByteOrder,
            (const ON__UINT32*)sWideChar,sWideChar_count,
            sUTF8,sUTF8_count,
            error_status,error_mask,error_code_point,
            (const ON__UINT32**)sNextWideChar
            );
    break;

  default:
    rc = 0;
  }

  return rc;
}


int ON_ConvertWideCharToUTF16(
    int bTestByteOrder,
    const wchar_t* sWideChar,
    int sWideChar_count,
    char* sUTF16,
    int sUTF16_count,
    unsigned int* error_status,
    unsigned int error_mask,
    ON__UINT32 error_code_point,
    const wchar_t** sNextWideChar
    )
{
  int rc;

  switch(sizeof(sWideChar[0]))
  {
  case sizeof(char):
    // assume wchar_t strings are UTF-8 encoded
    rc = ON_ConvertUTF8ToUTF16(
            bTestByteOrder,
            (const char*)sWideChar,sWideChar_count,
            (ON__UINT16*)sUTF16,sUTF16_count,
            error_status,error_mask,error_code_point,
            (const char**)sNextWideChar
            );
    break;

  case sizeof(ON__UINT16):
    // assume wchar_t strings are UTF-16 encoded
    rc = ON_ConvertUTF16ToUTF16(
            bTestByteOrder,
            (const ON__UINT16*)sWideChar,sWideChar_count,
            (ON__UINT16*)sUTF16,sUTF16_count,
            error_status,error_mask,error_code_point,
            (const ON__UINT16**)sNextWideChar
            );
    break;

  case sizeof(ON__UINT32):
    // assume wchar_t strings are UTF-32 encoded
    rc = ON_ConvertUTF32ToUTF16(
            bTestByteOrder,
            (const ON__UINT32*)sWideChar,sWideChar_count,
            (ON__UINT16*)sUTF16,sUTF16_count,
            error_status,error_mask,error_code_point,
            (const ON__UINT32**)sNextWideChar
            );
    break;

  default:
    rc = 0;
  }

  return rc;
}


int ON_ConvertWideCharToUTF32(
    int bTestByteOrder,
    const wchar_t* sWideChar,
    int sWideChar_count,
    ON__UINT32* sUTF32,
    int sUTF32_count,
    unsigned int* error_status,
    unsigned int error_mask,
    ON__UINT32 error_code_point,
    const wchar_t** sNextWideChar
    )
{
  int rc;

  switch(sizeof(sWideChar[0]))
  {
  case sizeof(char):
    // assume wchar_t strings are UTF-8 encoded
    rc = ON_ConvertUTF8ToUTF32(
            bTestByteOrder,
            (const char*)sWideChar,sWideChar_count,
            sUTF32,sUTF32_count,
            error_status,error_mask,error_code_point,
            (const char**)sNextWideChar
            );
    break;

  case sizeof(ON__UINT16):
    // assume wchar_t strings are UTF-16 encoded
    rc = ON_ConvertUTF16ToUTF32(
            bTestByteOrder,
            (const ON__UINT16*)sWideChar,sWideChar_count,
            sUTF32,sUTF32_count,
            error_status,error_mask,error_code_point,
            (const ON__UINT16**)sNextWideChar
            );
    break;

  case sizeof(ON__UINT32):
    // assume wchar_t strings are UTF-32 encoded
    rc = ON_ConvertUTF32ToUTF32(
            bTestByteOrder,
            (const ON__UINT32*)sWideChar,sWideChar_count,
            sUTF32,sUTF32_count,
            error_status,error_mask,error_code_point,
            (const ON__UINT32**)sNextWideChar
            );
     break;

  default:
    rc = 0;
  }

  return rc;
}


int ON_ConvertUTF8ToWideChar(
    int bTestByteOrder,
    const char* sUTF8,
    int sUTF8_count,
    wchar_t* sWideChar,
    int sWideChar_count,
    unsigned int* error_status,
    unsigned int error_mask,
    ON__UINT32 error_code_point,
    const char** sNextUTF8
    )
{
  int rc;

  switch(sizeof(sWideChar[0]))
  {
  case sizeof(char):
    // assume wchar_t strings are UTF-8 encoded
    rc = ON_ConvertUTF8ToUTF8(
            bTestByteOrder,
            sUTF8,sUTF8_count,
            (char*)sWideChar,sWideChar_count,
            error_status,error_mask,error_code_point,
            sNextUTF8
            );
    break;

  case sizeof(ON__UINT16):
    // assume wchar_t strings are UTF-16 encoded
    rc = ON_ConvertUTF8ToUTF16(
            bTestByteOrder,
            sUTF8,sUTF8_count,
            (ON__UINT16*)sWideChar,sWideChar_count,
            error_status,error_mask,error_code_point,
            sNextUTF8
            );
    break;

  case sizeof(ON__UINT32):
    // assume wchar_t strings are UTF-32 encoded
    rc = ON_ConvertUTF8ToUTF32(
            bTestByteOrder,
            sUTF8,sUTF8_count,
            (ON__UINT32*)sWideChar,sWideChar_count,
            error_status,error_mask,error_code_point,
            sNextUTF8
            );
    break;

  default:
    if (error_status)
      *error_status = 1;
    if (sNextUTF8)
      *sNextUTF8 = sUTF8;
    rc = 0;
  }

  return rc;
}


int ON_ConvertUTF16ToWideChar(
    int bTestByteOrder,
    const ON__UINT16* sUTF16,
    int sUTF16_count,
    wchar_t* sWideChar,
    int sWideChar_count,
    unsigned int* error_status,
    unsigned int error_mask,
    ON__UINT32 error_code_point,
    const ON__UINT16** sNextUTF16
    )
{
  int rc;

  switch(sizeof(sWideChar[0]))
  {
  case sizeof(char):
    // assume wchar_t strings are UTF-8 encoded
    rc = ON_ConvertUTF16ToUTF8(
            bTestByteOrder,
            (const ON__UINT16*)sUTF16,sUTF16_count,
            (char*)sWideChar,sWideChar_count,
            error_status,error_mask,error_code_point,
            (const ON__UINT16**)sNextUTF16
            );
    break;

  case sizeof(ON__UINT16):
    // assume wchar_t strings are UTF-16 encoded
    rc = ON_ConvertUTF16ToUTF16(
            bTestByteOrder,
            (const ON__UINT16*)sUTF16,sUTF16_count,
            (ON__UINT16*)sWideChar,sWideChar_count,
            error_status,error_mask,error_code_point,
            (const ON__UINT16**)sNextUTF16
            );
    break;

  case sizeof(ON__UINT32):
    // assume wchar_t strings are UTF-32 encoded
    rc = ON_ConvertUTF16ToUTF32(
            bTestByteOrder,
            (const ON__UINT16*)sUTF16,sUTF16_count,
            (ON__UINT32*)sWideChar,sWideChar_count,
            error_status,error_mask,error_code_point,
            (const ON__UINT16**)sNextUTF16
            );
    break;

  default:
    if (error_status)
      *error_status = 1;
    if (sNextUTF16)
      *sNextUTF16 = sUTF16;
    rc = 0;
  }

  return rc;
}

int ON_ConvertUTF32ToWideChar(
    int bTestByteOrder,
    const ON__UINT32* sUTF32,
    int sUTF32_count,
    wchar_t* sWideChar,
    int sWideChar_count,
    unsigned int* error_status,
    unsigned int error_mask,
    ON__UINT32 error_code_point,
    const ON__UINT32** sNextUTF32
    )
{
  int rc;

  switch(sizeof(sWideChar[0]))
  {
  case sizeof(char):
    // assume wchar_t strings are UTF-8 encoded
    rc = ON_ConvertUTF32ToUTF8(
            bTestByteOrder,
            (const ON__UINT32*)sUTF32,sUTF32_count,
            (char*)sWideChar,sWideChar_count,
            error_status,error_mask,error_code_point,
            (const ON__UINT32**)sNextUTF32
            );
    break;

  case sizeof(ON__UINT16):
    // assume wchar_t strings are UTF-16 encoded
    rc = ON_ConvertUTF32ToUTF16(
            bTestByteOrder,
            (const ON__UINT32*)sUTF32,sUTF32_count,
            (ON__UINT16*)sWideChar,sWideChar_count,
            error_status,error_mask,error_code_point,
            (const ON__UINT32**)sNextUTF32
            );
    break;

  case sizeof(ON__UINT32):
    // assume wchar_t strings are UTF-32 encoded
    rc = ON_ConvertUTF32ToUTF32(
            bTestByteOrder,
            (const ON__UINT32*)sUTF32,sUTF32_count,
            (ON__UINT32*)sWideChar,sWideChar_count,
            error_status,error_mask,error_code_point,
            (const ON__UINT32**)sNextUTF32
            );
    break;

  default:
    if (error_status)
      *error_status = 1;
    if (sNextUTF32)
      *sNextUTF32 = sUTF32;
    rc = 0;
  }

  return rc;
}

const ON_wString ON_wString::FromUnicodeCodePoint(
  ON__UINT32 code_point
)
{
  return ON_wString::FromUnicodeCodePoints(&code_point, 1, ON_UnicodeCodePoint::ON_ReplacementCharacter);
}


const ON_wString ON_wString::FromUnicodeCodePoints(
  const ON__UINT32* code_points,
  int code_point_count,
  ON__UINT32 error_code_point
)
{
  const bool bErrorCodePointIsValid = ON_IsValidUnicodeCodePoint(error_code_point);

  if (nullptr == code_points)
    return ON_wString::EmptyString;

  if (-1 == code_point_count)
  {
    code_point_count = 0;
    while (0 != code_points[code_point_count])
    {
      if (
        false == bErrorCodePointIsValid
        && false == ON_IsValidUnicodeCodePoint(code_points[code_point_count])
        )
      {
        break;
      }
      code_point_count++;
    }
  }

  if ( code_point_count <= 0 )
    return ON_wString::EmptyString;

  const int bTestByteOrder = false;
  unsigned int error_status = 0;
  const unsigned int error_mask = bErrorCodePointIsValid ? 0xFFFFFFFF : 0;
  int wchar_count = ON_ConvertUTF32ToWideChar(
    bTestByteOrder,
    code_points,
    code_point_count,
    nullptr,
    0,
    &error_status,
    error_mask,
    error_code_point,
    nullptr
  );

  if (wchar_count <= 0)
    return ON_wString::EmptyString;

  ON_wString s;
  const int s_capacity = (wchar_count + 1);
  wchar_t* a = s.ReserveArray((size_t)s_capacity);
  error_status = 0;
  wchar_count = ON_ConvertUTF32ToWideChar(
    bTestByteOrder,
    code_points,
    code_point_count,
    a,
    s_capacity,
    &error_status,
    error_mask,
    error_code_point,
    nullptr
  );

  if (wchar_count <= 0)
    return ON_wString::EmptyString;

  s.SetLength(wchar_count);
  return s;
}


////int ON_ConvertWindowsCodePageValueToWideChar(
////  int windows_code_page,
////  ON__UINT32 code_page_character_value,
////  size_t w_capacity,
////  wchar_t* w
////)
////{
////  ON__UINT32 unicode_code_point = ON_UnicodeCodePoint::ON_ReplacementCharacter;
////  ON_UnicodeErrorParameters e;
////  memset(&e, 0, sizeof(e));
////  e.m_error_mask = 0xFF;
////  e.m_error_code_point = ON_UnicodeCodePoint::ON_ReplacementCharacter;
////  ON_DecodeWindowsCodePageValue( windows_code_page, code_page_character_value,  &e,  &unicode_code_point);
////  return ON_EncodeWideChar(unicode_code_point, w_capacity, w);
////}

ON__UINT32 ON_MapRTFcharsetToWindowsCodePage(
  ON__UINT32 rtf_charset,
  ON__UINT32 default_code_page
)
{
  //  From the Microsoft version of the RTF ver 1.9 spec available on MSDN
  //
  //  \fcharsetN:  	Specifies the character set of a font in the font table.If this appears, it implies that bytes in runs
  //    tagged with the associated \fN are character codes in the codepage corresponding to the charset N.
  //    Use this codepage to convert the codes to Unicode using a function like the Windows MultiByteToWideChar().
  //    See also the \cpgN control word, which, if it appears, supersedes the codepage given by \fcharsetN.Values for N are defined,
  //    for example, in the Windows header file wingdi.h(e.g., see ANSI_CHARSET) and are repeated here together with the corresponding
  //    Windows or Mac codepages for convenience:charset	codepage	Windows / Mac name

  //  A font may have a different character set from the character set of the document. For example, the Symbol font has the
  //    same characters in the same code positions both on the Macintosh and in Windows. Typically, RTF fonts use the code page
  //    corresponding to the \fcharsetN control word in their \fonttbl description. If the charset doesn’t exist, the codepage
  //    may be given by the \cpgN control word, for which the code page is N. If the \cpgN does appear, it supersedes the code
  //    page corresponding to the \fcharsetN.
  //    For such cases, codepage conversions can be avoided altogether by using the Unicode \uN notation for characters.
  //    In addition, file names (used in field instructions and in embedded fonts) may not necessarily be the same as the character
  //    set of the document; the \cpgN control word can change the character set for these file names as well.
  //

  ON__UINT32 cp;
  switch (rtf_charset)
  {
  case 0:   cp = 1252; break;         // ANSI
  case 1:   cp = 0; break;            // Default
  case 2:   cp = 42; break;           // Symbol
  case 77:  cp = 10000; break;        // Mac Roman
  case 78:  cp = 10001; break;        // Mac Shift Jis
  case 79:  cp = 10003; break;        // Mac Hangul
  case 80:  cp = 10008; break;        // Mac GB2312
  case 81:  cp = 10002; break;        // Mac Big5
  case 82:  cp = default_code_page; break; // Mac Johab (old)
  case 83:  cp = 10005; break;        // Mac Hebrew
  case 84:  cp = 10004; break;        // Mac Arabic
  case 85:  cp = 10006; break;        // Mac Greek
  case 86:  cp = 10081; break;        // Mac Turkish
  case 87:  cp = 10021; break;        // Mac Thai
  case 88:  cp = 10029; break;        // Mac East Europe
  case 89:  cp = 10007; break;        // Mac Russian
  case 128: cp = 932; break;          // Shift JIS
  case 129: cp = 949; break;          // Hangul (Korean)
  case 130: cp = 1361; break;         // Johab
  case 134: cp = 936; break;          // GB2312
  case 136: cp = 950; break;          // Big5
  case 161: cp = 1253; break;         // Greek
  case 162: cp = 1254; break;         // Turkish
  case 163: cp = 1258; break;         // Vietnamese
  case 177: cp = 1255; break;         // Hebrew
  case 178: cp = 1256; break;         // Arabic
  case 179: cp = default_code_page; break; // Arabic Traditional (old)
  case 180: cp = default_code_page; break; // Arabic user (old)
  case 181: cp = default_code_page; break; // Hebrew user (old)
  case 186: cp = 1257; break;         // Baltic
  case 204: cp = 1251; break;         // Russian
  case 222: cp = 874; break;          // Thai
  case 238: cp = 1250; break;         // Eastern European
  case 254: cp = 437; break;          // PC 437
  case 255: cp = 850; break;          // OEM
  default: cp = default_code_page; break;
  }
  return cp;
}

static int ON_Internal_ConvertMSSBCPToWideChar(
  const ON__UINT32* sb_code_page_0x80_to_0xFF_to_unicode,
  const char* sMBCS,
  int sMBCS_count,
  wchar_t* sWideChar,
  int sWideChar_capacity,
  unsigned int* error_status
)
{
  wchar_t* sWideCharMax
    = (sWideChar_capacity > 0 && nullptr != sWideChar)
    ? sWideChar + sWideChar_capacity
    : nullptr;
  if (nullptr == sWideCharMax)
  {
    sWideChar = nullptr;
    sWideChar_capacity = 0;
  }
  else
  {
    sWideChar[0] = 0;
  }
  if (nullptr != error_status)
    *error_status = 0;

  unsigned int e = 0;
  if (nullptr == sMBCS || sMBCS_count < 0)
    sMBCS_count = 0;
  wchar_t* s = sWideChar;
  wchar_t w_buffer[8];
  int rc = 0;

  for (int i = 0; i < sMBCS_count; i++)
  {
    const ON__UINT32 c = (unsigned char)sMBCS[i];
    ON__UINT32 unicode_code_point;
    if (c < 0x80)
      unicode_code_point = c;
    else
    {
      if (c <= 0xFF && nullptr != sb_code_page_0x80_to_0xFF_to_unicode )
      {
        unicode_code_point = sb_code_page_0x80_to_0xFF_to_unicode[c - 0x80];
        if (0 == ON_IsValidUnicodeCodePoint(unicode_code_point))
          unicode_code_point = ON_UnicodeCodePoint::ON_ReplacementCharacter;
      }
      else
        unicode_code_point = ON_UnicodeCodePoint::ON_ReplacementCharacter;
      if ( ON_UnicodeCodePoint::ON_ReplacementCharacter == unicode_code_point )
        e |= 16;
    }
    const int w_count = ON_EncodeWideChar(unicode_code_point, sizeof(w_buffer)/sizeof(w_buffer[0]), w_buffer);
    if (w_count <= 0)
    {
      e |= 16;
      continue;
    }
    rc += w_count;
    if (s == nullptr)
      continue;
    wchar_t* s1 = s + w_count;
    if (s1 > sWideCharMax)
    {
      e |= 2;
      continue;
    }
    const wchar_t* w = w_buffer;
    while (s < s1)
      *s++ = *w++;
  }

  while (s < sWideCharMax)
  {
    *s++ = 0;
  }

  if (nullptr != error_status)
    *error_status = e;

  return rc;
}

int ON_ConvertMSMBCPToWideChar(
    ON__UINT32 windows_code_page,
    const char* sMBCS,
    int sMBCS_count,
    wchar_t* sWideChar,
    int sWideChar_capacity,
    unsigned int* error_status
    )
{
  if ( 0 != error_status )
    *error_status = 0;

  if (nullptr != sWideChar && sWideChar_capacity > 0)
    sWideChar[0] = 0;

  bool bNullTerminated = false; // Mac OS code needs bNullTerminated

  if ( -1 == sMBCS_count && nullptr != sMBCS )
  {
    for ( sMBCS_count = 0; true; sMBCS_count++)
    {
      if (0 == sMBCS[sMBCS_count])
      {
        bNullTerminated = true;
        break;
      }
    }
  }

  if ( nullptr == sMBCS || sMBCS_count < 0 )
  {
    if ( 0 != error_status )
      *error_status |= 1;
    return 0;
  }

  if ( 0 == sMBCS_count )
  {
    return 0;
  }

  if (sWideChar_capacity <= 0)
  {
    sWideChar_capacity = 0;
    sWideChar = nullptr;
  }
  else if (nullptr == sWideChar)
  {
    sWideChar_capacity = 0;
  }

  // ASCII check
  const char* c = sMBCS;
  const char* c1 = c + sMBCS_count;
  wchar_t* w = sWideChar;
  wchar_t* w1 = w + sWideChar_capacity;
  while (c < c1 && *c >= 0 && *c <= 127)
  {
    if (nullptr != w)
    {
      if (w >= w1)
        break;
      *w++ = (wchar_t)*c;
    }
    c++;
  }
  if (c == c1)
  {
    if (w < w1)
      *w = 0;
    return sMBCS_count;
  }

  const ON__UINT32* sb_code_page_0x80_to_0xFF_to_unicode = ON_MSSBCP_0x80_0xFF_Unicode(windows_code_page);
  if (nullptr != sb_code_page_0x80_to_0xFF_to_unicode)
  {
    // fast platform independent single byte code page conversion built into opennurbs
    return ON_Internal_ConvertMSSBCPToWideChar(
      sb_code_page_0x80_to_0xFF_to_unicode,
      sMBCS,
      sMBCS_count,
      sWideChar,
      sWideChar_capacity,
      error_status
    );
  }

#if defined(ON_RUNTIME_WIN)
  // Starting with Windows Vista, the function does not drop illegal code points when dwFlags=0.
  // It replaces illegal sequences with U+FFFD (encoded as appropriate for the specified codepage).
  DWORD dwFlags = 0;
  int sWideChar_count = ::MultiByteToWideChar(windows_code_page, dwFlags, sMBCS, sMBCS_count, sWideChar, sWideChar_capacity);
  if (sWideChar_count < 0)
    sWideChar_count = 0;
  if (nullptr == sWideChar)
    return sWideChar_count;

  for (int i = 0; i < sWideChar_count; i++)
  {
    if (0 == sWideChar[i])
    {
      sWideChar_count = i;
      break;
    }
    if ( ON_wString::ReplacementCharacter == sWideChar[i] )
    {
      if ( nullptr != error_status)
        *error_status |= 16;
    }
  }
  if (sWideChar_count < sWideChar_capacity)
    sWideChar[sWideChar_count] = 0;
  return sWideChar_count;

#elif defined (ON_RUNTIME_COCOA_AVAILABLE)
  CFStringEncoding cfEncoding = CFStringConvertWindowsCodepageToEncoding(windows_code_page);
  if (cfEncoding == kCFStringEncodingInvalidId)
  {
    ON_ERROR("No Apple CFStringEncoding support for this value of windows_code_page");
    return 0;
  }

  char* szMBCS = nullptr;
  if (false == bNullTerminated)
  {
    szMBCS = (char*)onmalloc((sMBCS_count + 1) * sizeof(szMBCS[0]));
    memcpy(szMBCS, sMBCS, sMBCS_count * sizeof(szMBCS[0]));
    szMBCS[sMBCS_count] = 0;
    sMBCS = szMBCS;
  }

  int sWideChar_count = 0;

  for (;;)
  {
    NSStringEncoding nsEncoding = CFStringConvertEncodingToNSStringEncoding(cfEncoding);
    NSString* str = [NSString stringWithCString : sMBCS encoding : nsEncoding];
    if (nullptr == str)
    {
      ON_ERROR("[NSString stringWithCString: sMBCS encoding: nsEncoding] failed.");
      break;
    }
    const int len = (int)(str.length);
    if (len <= 0)
    {
      break;
    }

    for (int i = 0; i < len; i++)
    {
      ON__UINT32 unicode_code_point = 0;
      const int u1 = [str characterAtIndex : i];
      if (u1 >= 0xD800U && u1 < 0xDC00 && i+1 < len)
      {
        const int u2 = [str characterAtIndex : (i+1)];
        unicode_code_point = ON_DecodeUTF16SurrogatePair((unsigned int)u1, (unsigned int)u2, ON_wString::ReplacementCharacter);
        if (ON_wString::ReplacementCharacter != unicode_code_point)
          i++;
      }
      else
      {
        unicode_code_point = (unsigned int)u1;
      }
      if (
        false == ON_IsValidUnicodeCodePoint(unicode_code_point)
        || ON_wString::ReplacementCharacter == unicode_code_point
        )
      {
        unicode_code_point = ON_wString::ReplacementCharacter;
        if (nullptr != error_status)
          *error_status |= 16;
      }
      if (nullptr != sWideChar && sWideChar_capacity > 0)
      {
        if (sWideChar_count < sWideChar_capacity)
          sWideChar[sWideChar_count] = (wchar_t)unicode_code_point;
        else
        {
          // continue counting but no more output to sWideChar[]
          sWideChar[sWideChar_capacity-1] = 0;
          sWideChar = nullptr;
          sWideChar_capacity = 0;
          if (nullptr != error_status)
            *error_status |= 2;
        }
      }
      sWideChar_count++;
    }

    break;
  }

  if (nullptr != szMBCS)
    onfree(szMBCS);

  if (nullptr != sWideChar && sWideChar_count < sWideChar_capacity)
  {
    sWideChar[sWideChar_count] = 0;
    sWideChar[sWideChar_capacity-1] = 0;
  }

  return sWideChar_count;

#else
  // Add support for other platforms as needed.
  return 0;
#endif

}

int ON_wString::ByteOrder(wchar_t bom_candidate)
{
  if (ON_UnicodeCodePoint::ON_ByteOrderMark == bom_candidate)
    return 1;
  if (ON_UnicodeCodePoint::ON_ByteOrderMark == ON_wString::SwapByteOrder(bom_candidate))
    return -1;
  return 0;
}

wchar_t ON_wString::SwapByteOrder(wchar_t w)
{
  unsigned char* b = (unsigned char*)(&w);
  unsigned char tmp;
#if (2 == ON_SIZEOF_WCHAR_T)
  tmp = b[0]; b[0] = b[1]; b[1] = tmp;
#elif (4 == ON_SIZEOF_WCHAR_T)
  tmp = b[0]; b[0] = b[3]; b[3] = tmp;
  tmp = b[1]; b[1] = b[2]; b[2] = tmp;
#endif
  return w;
}

int ON_ConvertWideCharToMSMBCP(
  const wchar_t* sWideChar,
  int sWideChar_count,
  ON__UINT32 windows_code_page,
  char* sMBCS,
  int sMBCS_capacity,
  unsigned int* error_status
)
{
  if (0 != error_status)
    *error_status = 0;

  if (nullptr != sMBCS && sMBCS_capacity > 0)
    sMBCS[0] = 0;

  if (-1 == sWideChar_count && nullptr != sWideChar)
  {
    for (sWideChar_count = 0; true; sWideChar_count++)
    {
      if (0 == sWideChar[sWideChar_count])
        break;
    }
  }  

  if (nullptr == sWideChar || sWideChar_count < 0)
  {
    if (0 != error_status)
      *error_status |= 1;
    return 0;
  }

  const int byte_order_mark = (nullptr != sWideChar && sWideChar_count > 0) ? ON_wString::ByteOrder(sWideChar[0]) : 0;
  if (0 != byte_order_mark)
  {
    // handle leading byte order mark.
    ++sWideChar;
    --sWideChar_count;

    while (sWideChar_count > 0 && 0 != ON_wString::ByteOrder(sWideChar[0]))
    {
      // Ignore additional byte order marks. If this is causing trouble,
      // you need to clean up your input before calling this function.
      ++sWideChar;
      --sWideChar_count;
    }

    if (sWideChar_count <= 0)
    {
      // The entire input string was byte order marks. Nothing to convert
      return 0;
    }

    if (-1 == byte_order_mark)
    {
      // Recursively call this function 1 time with swapped bytes.
      wchar_t* swappedWideChar = (wchar_t*)onmalloc((sWideChar_count + 1) * sizeof(swappedWideChar[0]));
      for (int i = 0; i < sWideChar_count; ++i)
        swappedWideChar[i] = ON_wString::SwapByteOrder(sWideChar[i]);
      swappedWideChar[sWideChar_count] = 0;
      if (0 == ON_wString::ByteOrder(swappedWideChar[0]))
      {
        // The 0 == ON_wString::ByteOrder(swappedWideChar[0]) condition insures we will not recurse again.
        const int rc = ON_ConvertWideCharToMSMBCP(swappedWideChar, sWideChar_count, windows_code_page, sMBCS, sMBCS_capacity, error_status);
        onfree(swappedWideChar);
        return rc;
      }
      else
      {
        // A modification to this code made after May 2021 introduced a bug. 
        // Depending on the bug and the string length, a possible stack overflow recursion crash was prevented by this check.
        ON_ERROR("A bug was added after May 2021.");
        onfree(swappedWideChar);
      }
    }
  }

  if (0 == sWideChar_count)
  {
    return 0;
  }

  if (sMBCS_capacity <= 0)
  {
    sMBCS_capacity = 0;
    sMBCS = nullptr;
  }
  else if (nullptr == sMBCS)
  {
    sMBCS_capacity = 0;
  }

  // ASCII check
  const wchar_t* w = sWideChar;
  const wchar_t* w1 = w + sWideChar_count;
  char* c = sMBCS;
  char* c1 = c + sMBCS_capacity;
  while (w < w1 && *w >= 0 && *w <= 127)
  {
    if (nullptr != c)
    {
      if (c >= c1)
        break;
      *c++ = (char)*w;
    }
    c++;
  }
  if (w == w1)
  {
    if (c < c1)
      *c = 0;
    return sWideChar_count;
  }

  // UTF wide char to UTF-8 - use opennurbs converter
  const int utf8_code_page = 65001; // UTF-8
  if (utf8_code_page == windows_code_page)
  {
    const unsigned int error_mask = 0xFFFFFFFFU;
    return ON_ConvertWideCharToUTF8(false, sWideChar, sWideChar_count, sMBCS, sMBCS_capacity, error_status, error_mask, ON_UnicodeCodePoint::ON_ReplacementCharacter, nullptr);
  }

#if defined(ON_RUNTIME_WIN)
  // Starting with Windows Vista, the function does not drop illegal code points when dwFlags=0.
  // It replaces illegal sequences with U+FFFD (encoded as appropriate for the specified codepage).
  const char default_char = '?';
  BOOL bUsedDefaultChar = 0;

  const char* pdefault_char = nullptr;
  BOOL* pbUsedDefaultChar = nullptr;
  DWORD dwFlags = 0;

  switch (windows_code_page)
  {
  case 42: // Symbol
  case 50220:
  case 50221:
  case 50222:
  case 50225:
  case 50227:
  case 50229:
  case 57002:
  case 57003:
  case 57004:
  case 57005:
  case 57006:
  case 57007:
  case 57008:
  case 57009:
  case 57010:
  case 57011:
  case 65000: // UTF-7
    // For these code pages listed below, dwFlags must be 0. Otherwise, WideCharToMultiByte() fails with ERROR_INVALID_FLAGS.
    break;

  case 54936: // GB18030
  case 65001: // UTF-8
    // dwFlags must be set to either 0 or WC_ERR_INVALID_CHARS. Otherwise, the function fails with ERROR_INVALID_FLAGS.
    break;

  default:
    pdefault_char = &default_char;
    pbUsedDefaultChar = &bUsedDefaultChar;
    dwFlags |= WC_NO_BEST_FIT_CHARS;
    dwFlags |= WC_COMPOSITECHECK;
    dwFlags |= WC_DEFAULTCHAR;
    break;
  }

  int sMBCS_count = ::WideCharToMultiByte(windows_code_page, dwFlags, sWideChar, sWideChar_count, sMBCS, sMBCS_capacity, pdefault_char, pbUsedDefaultChar);
  if (sMBCS_count < 0)
    sMBCS_count = 0;

  if (nullptr == sMBCS)
    return sMBCS_count;

  for (int i = 0; i < sMBCS_count; i++)
  {
    if (0 == sMBCS[i])
    {
      sMBCS_count = i;
      break;
    }
  }
  if (bUsedDefaultChar)
  {
    if (nullptr != error_status)
      *error_status |= 16;
  }

  if (sMBCS_count < sMBCS_capacity)
    sMBCS[sMBCS_count] = 0;
  return sMBCS_count;

#elif defined (ON_RUNTIME_APPLE_OBJECTIVE_C_AVAILABLE)
  // Add support for other platforms as needed.
  ON_ERROR("Time to figure out how to create Windows code page encodings in MacOS");
  return 0;
#else
  // Add support for other platforms as needed.
  return 0;
#endif
}

const ON_String ON_wString::MultiByteEncode(int windows_code_page) const
{
  unsigned int error_status0 = 0;
  const int sChar_count0 = ON_ConvertWideCharToMSMBCP(
    this->Array(),
    this->Length(),
    windows_code_page,
    nullptr, // wchar_t* sWideChar,
    0, //int sWideChar_capacity,
    &error_status0
  );
  if (sChar_count0 <= 0)
    return ON_String::EmptyString;

  ON_String s;
  s.ReserveArray(((size_t)sChar_count0) + 1);
  unsigned int error_status1 = 0;
  const int sChar_count1 = ON_ConvertWideCharToMSMBCP(
    this->Array(),
    this->Length(),
    windows_code_page,
    s.Array(),
    sChar_count0 + 1,
    &error_status1
  );
  if (sChar_count1 <= 0)
    return ON_String::EmptyString;

  s.SetLength(sChar_count1);
  return s;
}

const ON_wString ON_String::MultiByteDecode(int windows_code_page) const
{
  unsigned int error_status0 = 0;
  const int sWideChar_count0 = ON_ConvertMSMBCPToWideChar(
    windows_code_page,
    this->Array(),
    this->Length(),
    nullptr, // wchar_t* sWideChar,
    0, //int sWideChar_capacity,
    &error_status0
  );
  if (sWideChar_count0 <= 0)
    return ON_wString::EmptyString;

  ON_wString s;
  s.ReserveArray(((size_t)sWideChar_count0) + 1);
  unsigned int error_status1 = 0;
  const int sWideChar_count1 = ON_ConvertMSMBCPToWideChar(
    windows_code_page,
    this->Array(),
    this->Length(),
    s.Array(),
    sWideChar_count0+1,
    &error_status1
  );
  if (sWideChar_count1 <= 0)
    return ON_wString::EmptyString;

  s.SetLength(sWideChar_count1);
  return s;
}


unsigned ON_UnicodeSuperscriptFromCodePoint(
  unsigned cp,
  unsigned no_superscript_cp
)
{
  if (cp >= '0' && cp <= '9')
  {
    static const unsigned digit_cp[10]
    {
      0x2070,
      0x00B9,
      0x00B2,
      0x00B3,
      0x2074,
      0x2075,
      0x2076,
      0x2077,
      0x2078,
      0x2079
    };
    return digit_cp[cp - '0'];
  }
  else if (cp >= 'a' && cp <= 'z')
  {
    // a-z
    static const unsigned atoz_cp[26]
    {
      0x1D43, // a
      0x1d47, // b
      0x1d9c, // c
      0x1d48, // d
      0x1d49, // e
      0x1da0, // f
      0x1d4d, // g
      0x20b0, // h
      0x2071, // i
      0x02b2, // j
      0x1d4f, // k
      0x02e1, // l
      0x1d40, // m
      0x207f, // n
      0x1d52, // o
      0x1d56, // p
      0, // q NONE AVAILABLE
      0x02b3, // r
      0x02e2, // s
      0x1d57, // t
      0x1d58, // u
      0x1d5b, // v
      0x02b7, // w
      0x02e3, // x
      0x02b8, // y
      0x1dbb  // z
    };
    const unsigned sup_cp = atoz_cp[cp - 'a'];
    if (0 != sup_cp)
      return sup_cp;
  }
  else if (cp >= 'A' && cp <= 'Z')
  {
    // a-z
    static const unsigned atoz_cp[26]
    {
      0x1DC2, // A
      0x1D2D, // B
      0, // C NOT AVAILABLE
      0x1D30, // D
      0x1D31, // E
      0, // F NOT AVAILABLE
      0x1D33, // G
      0x1D34, // H
      0x1D35, // I
      0x1D36, // J
      0x1D37, // K
      0x1D38, // L
      0x1D39, // M
      0x1D3A, // N
      0x1D3C, // O
      0x1D3E, // P
      0, // Q NOT AVAILABLE
      0x1D3F, // R
      0, // S NOT AVAILABLE
      0x1D40, // T
      0x2C7D, // V
      0x1D42, // W
      0, // X NOT AVAILABLE
      0, // Y NOT AVAILABLE
      0  // Z NOT AVAILABLE
    };
    const unsigned sup_cp = atoz_cp[cp - 'a'];
    if (0 != sup_cp)
      return sup_cp;
  }
  else
  {
    switch (cp)
    {
    case '+':
      return 0x207A; // +
      break;
    case '-':
      return 0x207B; // -
      break;
    case '=':
      return 0x207C; // =
      break;
    case '(':
      return 0x207C; // =
      break;
    case ')':
      return 0x207E; // )
      break;
    }
  }

  // either cp is already a superscript or none is available.
  return no_superscript_cp;
}


unsigned ON_UnicodeSubscriptFromDigit(unsigned decimal_digit)
{
  if (decimal_digit >= 0 && decimal_digit <= 9)
    return 0x2080U + decimal_digit;
  return 0;
}

unsigned ON_UnicodeSuperscriptFromDigit(unsigned decimal_digit)
{
  switch (decimal_digit)
  {
  case 1:
    return 0x00B9U;
    break;
  case 2:
    return 0x00B2U;
    break;
  case 3:
    return 0x00B3U;
    break;
  default:
    if (decimal_digit >= 4 && decimal_digit <= 9)
      return (0x2070U + decimal_digit);
    break;
  }
  return 0;
}

unsigned ON_UnicodeSubcriptFromCodePoint(
  unsigned cp,
  unsigned no_subscript_cp
)
{
  if (cp >= '0' && cp <= '9')
  {
    static const unsigned digit_cp[10]
    {
      0x2080,
      0x2081,
      0x2082,
      0x2083,
      0x2084,
      0x2085,
      0x2086,
      0x2087,
      0x2088,
      0x2089
    };
    return digit_cp[cp - '0'];
  }
  else
  {
    switch (cp)
    {
    case '+':
      return 0x208A; // +
      break;
    case '-':
      return 0x208B; // -
      break;
    case '=':
      return 0x208C; // =
      break;
    case '(':
      return 0x208C; // =
      break;
    case ')':
      return 0x208E; // )
      break;
    }
  }

  // either cp is already a subscript or none is available.
  return cp;
}

const char* ON_Big5CodePoint::Decode(
  const char* buffer,
  size_t buffer_count,
  bool bParseNull,
  bool bParseASCII,
  ON_Big5CodePoint* big5_code_point
)
{
  ON_Big5CodePoint local_cp;
  if (nullptr == big5_code_point)
    big5_code_point = &local_cp;
  for (;;)
  {
    if (nullptr == buffer)
      break;
    if (-1 == buffer_count)
      buffer_count = 2; // buffer[] is null terminated
    if (buffer_count <= 0)
      break;

    if (0 == buffer[0])
    {
      if (bParseNull)
      {
        *big5_code_point = ON_Big5CodePoint::Null;
        return buffer + 1;
      }
      break;
    }

    if (buffer[0] > 0 && buffer[0] <= 0x7F)
    {
      if (bParseASCII)
      {
        *big5_code_point = ON_Big5CodePoint::Create((ON__UINT16)buffer[0]);
        return buffer + 1;
      }
      break;
    }

    if (buffer_count < 2)
      break;
    const ON__UINT16 db[2] = { ((const unsigned char*)buffer)[0], ((const unsigned char*)buffer)[1] };
    if (
      (db[0] >= 0x81 && db[0] <= 0xFE) 
      &&
      ((db[1] >= 0x40 && db[1] <= 0x7E) || (db[1] >= 0xA1 && db[1] <= 0xFE))
      )
    {
      // valid BIG5 double double byte encoding.
      *big5_code_point = ON_Big5CodePoint::Create(db[0] * 0x100 + db[1]);
      return buffer + 2;
    }

    break;
  }
  
  *big5_code_point = ON_Big5CodePoint::Error;
  return nullptr;
}

int ON_Big5CodePoint::Encode(
  char* buffer,
  size_t buffer_capacity
) const
{
  if (nullptr == buffer || buffer_capacity <= 0)
    return 0;

  if (IsValid(true, true))
  {
    if (m_big5_code_point <= 0x7F)
    {
      buffer[0] = (char)m_big5_code_point;
      if (buffer_capacity > 1)
        buffer[1] = 0;
      return 1;
    }

    if (buffer_capacity < 2)
      return 0;

    const unsigned db[2] = { m_big5_code_point / 0x100U, m_big5_code_point % 0x100U };
    if (
      (db[0] >= 0x81 && db[0] <= 0xFE)
      &&
      ((db[1] >= 0x40 && db[1] <= 0x7E) || (db[1] >= 0xA1 && db[1] <= 0xFE))
      )
    {
      ((unsigned char*)buffer)[0] = (unsigned char)(db[0]);
      ((unsigned char*)buffer)[1] = (unsigned char)(db[1]);
      if (buffer_capacity > 2)
        buffer[2] = 0;
      return 2;
    }

    ON_ERROR("Bug in tis if() scope.");    
  }

  return 0;

}

const ON_Big5CodePoint ON_Big5CodePoint::Create(unsigned int big5_code_point)
{
  if (big5_code_point > 0xFFFF)
    return ON_Big5CodePoint::Error;
  ON_Big5CodePoint cp;
  cp.m_big5_code_point = (big5_code_point <= 0xFFFF) ? ((ON__UINT16)big5_code_point) : ((ON__UINT16)0xFFFF);
  return (cp.IsValid(true, true) || 0xFFFF==cp.m_big5_code_point) ? cp : ON_Big5CodePoint::Error;
}

bool ON_Big5CodePoint::IsNull() const
{
  return 0 == m_big5_code_point;
}

bool ON_Big5CodePoint::IsASCII(bool bNullIsASCII) const
{
  if (0 == m_big5_code_point)
    return bNullIsASCII ? true : false;
  return (m_big5_code_point > 0 && m_big5_code_point <= 0x007F);
}

bool ON_Big5CodePoint::IsValid(bool bNullIsValid, bool bASCIICodePointIsValid) const
{
  if (0 == m_big5_code_point)
    return bNullIsValid ? true : false;

  if (m_big5_code_point <= 0x007F)
    return bASCIICodePointIsValid ? true : false;

  if (m_big5_code_point >= 0xA3C0 && m_big5_code_point <= 0xA3FE)
  {
    // Some older versions of Windows use the reserved BIG5 code point 0xA3E1 for the EURO CURRENCY SYMBOL
    // The primary use of ON_Big5CodePoint is to convert legacy BIG5 encoded information
    // into Unicode encoded information and ON_Big5CodePoint supports this legacy Windows mapping. 
    if (0xA3E1 != m_big5_code_point)
      return false; // reserved for future use
  }

  const ON__UINT16 db[2] = { (ON__UINT16)(m_big5_code_point / 0x100U), (ON__UINT16)(m_big5_code_point % 0x100U) };
  return
    (db[0] >= 0x81U && db[0] <= 0xFEU)
    &&
    ((db[1] >= 0x40U && db[1] <= 0x7EU) || (db[1] >= 0xA1U && db[1] <= 0xFEU))
    ;
}

bool ON_Big5CodePoint::IsStandard(bool bNullIsValid, bool bASCIICodePointIsValid) const
{
  if (IsValid(bNullIsValid, bASCIICodePointIsValid))
  {
    // 0xa140 to 0xa3bf	
    if (m_big5_code_point < 0xA140)
      return false;
    if (m_big5_code_point <= 0xA3BF)
      return true;

    // 0xa3c0 to 0xa3fe - reserved - not user defined and not standard
    if (0xA3E1 == m_big5_code_point)
      return true; // Old Windows Euro currency symbol

    // 0xa440 to 0xc67e
    if (m_big5_code_point < 0xA440)
      return false;
    if (m_big5_code_point <= 0xC67E)
      return true;

    // 0xc940 to 0xf9d5
    if (m_big5_code_point < 0xC940)
      return false;
    if (m_big5_code_point <= 0xF9D5)
      return true;
  }

  return false;
}


bool ON_Big5CodePoint::IsPrivateUse() const
{
  if (IsValid(false,false))
  {
    // 0x8140 to 0xa0fe
    if (m_big5_code_point < 0x8140)
      return false;
    if (m_big5_code_point < 0xA140)
      return true;

    // 0xc6a1 to 0xc8fe
    if (m_big5_code_point < 0xC6A1)
      return false;
    if (m_big5_code_point <= 0xC8FE)
      return true;

    // 0xf9d6 to 0xfefe
    if (m_big5_code_point < 0xF9D6)
      return false;
    if (m_big5_code_point <= 0xFEFE)
      return true;
  }

  return false;
}

unsigned int ON_Big5CodePoint::Big5CodePoint() const
{
  return m_big5_code_point;
}

int ON_Big5CodePoint::Compare(const ON_Big5CodePoint* lhs, const ON_Big5CodePoint* rhs)
{
  const unsigned lhs_cp = (nullptr != lhs) ? ((unsigned)lhs->m_big5_code_point) : 0xFFFFFFFFU;
  const unsigned rhs_cp = (nullptr != rhs) ? ((unsigned)rhs->m_big5_code_point) : 0xFFFFFFFFU;
  if (lhs_cp < rhs_cp)
    return -1;
  if (lhs_cp > rhs_cp)
    return 1;
  return 0;
};

bool operator==(ON_Big5CodePoint lhs, ON_Big5CodePoint rhs)
{
  return 0 == ON_Big5CodePoint::Compare(&lhs, &rhs);
}

bool operator!=(ON_Big5CodePoint lhs, ON_Big5CodePoint rhs)
{
  return 0 != ON_Big5CodePoint::Compare(&lhs, &rhs);
}

bool operator<=(ON_Big5CodePoint lhs, ON_Big5CodePoint rhs)
{
  return ON_Big5CodePoint::Compare(&lhs, &rhs) <= 0;
}

bool operator>=(ON_Big5CodePoint lhs, ON_Big5CodePoint rhs)
{
  return ON_Big5CodePoint::Compare(&lhs, &rhs) >= 0;
}

bool operator<(ON_Big5CodePoint lhs, ON_Big5CodePoint rhs)
{
  return ON_Big5CodePoint::Compare(&lhs, &rhs) < 0;
}

bool operator>(ON_Big5CodePoint lhs, ON_Big5CodePoint rhs)
{
  return ON_Big5CodePoint::Compare(&lhs, &rhs) > 0;
}

const ON_UnicodeShortCodePoint ON_UnicodeShortCodePoint::Create(unsigned int unicode_code_point)
{
  ON_UnicodeShortCodePoint cp;
  cp.m_unicode_code_point
    = (0xFFFE == unicode_code_point || ON_IsValidUnicodeCodePoint(unicode_code_point))
      ? ((ON__UINT16)unicode_code_point)
      : ((ON__UINT16)0xFFFFU)
    ;
  return cp;
}

bool ON_UnicodeShortCodePoint::IsNull() const
{
  return 0 == m_unicode_code_point;
}

bool ON_UnicodeShortCodePoint::IsASCII(bool bNullIsASCII) const
{
  if (0 == m_unicode_code_point)
    return bNullIsASCII ? true : false;

  return (m_unicode_code_point > 0 && m_unicode_code_point <= 0x007F);
}

bool ON_UnicodeShortCodePoint::IsValid(bool bNullIsValid, bool bByteOrderMarkIsValid) const
{
  if (0 == m_unicode_code_point)
    return bNullIsValid ? true : false;

  if (0xFFFE == m_unicode_code_point)
    return bByteOrderMarkIsValid ? true : false;

  return ON_IsValidUnicodeCodePoint(m_unicode_code_point);
}

bool ON_UnicodeShortCodePoint::IsStandard(bool bNullIsValid) const
{
  if (IsValid(bNullIsValid,false))
    return ON_IsStandardUnicodeCodePoint(m_unicode_code_point);

  return false;
}

bool ON_UnicodeShortCodePoint::IsPrivateUse() const
{
  return ON_IsPrivateUseUnicodeCodePoint(m_unicode_code_point);
}

unsigned int ON_UnicodeShortCodePoint::UnicodeCodePoint() const
{
  return m_unicode_code_point;
}

bool ON_UnicodeShortCodePoint::IsReplacementCharacter() const
{
  return 0xFFFD == m_unicode_code_point;
}

bool ON_UnicodeShortCodePoint::IsByteOrderMark() const
{
  return 0xFFFE == m_unicode_code_point;
}


int ON_UnicodeShortCodePoint::Compare(const ON_UnicodeShortCodePoint* lhs, const ON_UnicodeShortCodePoint* rhs)
{
  const unsigned lhs_cp = (nullptr != lhs) ? ((unsigned)lhs->m_unicode_code_point) : 0xFFFFFFFFU;
  const unsigned rhs_cp = (nullptr != rhs) ? ((unsigned)rhs->m_unicode_code_point) : 0xFFFFFFFFU;
  if (lhs_cp < rhs_cp)
    return -1;
  if (lhs_cp > rhs_cp)
    return 1;
  return 0;
};

bool operator==(ON_UnicodeShortCodePoint lhs, ON_UnicodeShortCodePoint rhs)
{
  return 0 == ON_UnicodeShortCodePoint::Compare(&lhs, &rhs);
}

bool operator!=(ON_UnicodeShortCodePoint lhs, ON_UnicodeShortCodePoint rhs)
{
  return 0 != ON_UnicodeShortCodePoint::Compare(&lhs, &rhs);
}

bool operator<=(ON_UnicodeShortCodePoint lhs, ON_UnicodeShortCodePoint rhs)
{
  return ON_UnicodeShortCodePoint::Compare(&lhs, &rhs) <= 0;
}

bool operator>=(ON_UnicodeShortCodePoint lhs, ON_UnicodeShortCodePoint rhs)
{
  return ON_UnicodeShortCodePoint::Compare(&lhs, &rhs) >= 0;
}

bool operator<(ON_UnicodeShortCodePoint lhs, ON_UnicodeShortCodePoint rhs)
{
  return ON_UnicodeShortCodePoint::Compare(&lhs, &rhs) < 0;
}

bool operator>(ON_UnicodeShortCodePoint lhs, ON_UnicodeShortCodePoint rhs)
{
  return ON_UnicodeShortCodePoint::Compare(&lhs, &rhs) > 0;
}

const ON_Big5UnicodePair ON_Big5UnicodePair::Create(
  ON_Big5CodePoint big5_code_point,
  ON_UnicodeShortCodePoint unicode_code_point
)
{
  ON_Big5UnicodePair p;
  p.m_big5 = big5_code_point;
  p.m_unicode = unicode_code_point;
  return p;
}

const ON_Big5UnicodePair ON_Big5UnicodePair::Create(
  unsigned int big5_code_point,
  unsigned int unicode_code_point
)
{
  return ON_Big5UnicodePair::Create(ON_Big5CodePoint::Create(big5_code_point), ON_UnicodeShortCodePoint::Create(unicode_code_point));
}


bool ON_Big5UnicodePair::IsNull() const
{
  return m_big5.IsNull() && m_unicode.IsNull();
}

bool ON_Big5UnicodePair::IsASCII(bool bNullIsASCII) const
{
  return m_big5.IsASCII(bNullIsASCII) && m_big5.Big5CodePoint() == m_unicode.UnicodeCodePoint();
}

bool ON_Big5UnicodePair::IsValid(bool bNullIsValid, bool bASCIICodePointIsValid) const
{
  const unsigned big5_code_point = m_big5.Big5CodePoint();
  if (0 == big5_code_point)
    return bNullIsValid ? IsNull() : false;

  if (big5_code_point <= 0x7F)
    return bASCIICodePointIsValid ? IsASCII(false) : false;

  return m_big5.IsValid(false, false) && m_unicode.IsValid(false,false);
}

const ON_Big5CodePoint ON_Big5UnicodePair::Big5() const
{
  return m_big5;
}

const ON_UnicodeShortCodePoint ON_Big5UnicodePair::Unicode() const
{
  return m_unicode;
}

unsigned int ON_Big5UnicodePair::Big5CodePoint() const
{
  return m_big5.Big5CodePoint();
}

unsigned int ON_Big5UnicodePair::UnicodeCodePoint() const
{
  return m_unicode.UnicodeCodePoint();
}

bool ON_Big5UnicodePair::IsStandard(bool bNullIsValid, bool bASCIICodePointIsStandard) const
{
  const unsigned big5_code_point = m_big5.Big5CodePoint();
  if (0 == big5_code_point)
    return bNullIsValid ? IsNull() : false;

  if (big5_code_point <= 0x7F)
    return bASCIICodePointIsStandard ? IsASCII(false) : false;

  return m_big5.IsStandard(false, false) && m_unicode.IsStandard(false);
}

bool ON_Big5UnicodePair::IsPrivateUse() const
{
  return IsValid(false, false) && (m_big5.IsPrivateUse() || m_unicode.IsPrivateUse());
}


int ON_Big5UnicodePair::CompareBig5CodePoint(const ON_Big5UnicodePair* lhs, const ON_Big5UnicodePair* rhs)
{
  const unsigned lhs_cp = (nullptr != lhs) ? lhs->Big5CodePoint() : 0xFFFFFFFFU;
  const unsigned rhs_cp = (nullptr != rhs) ? rhs->Big5CodePoint() : 0xFFFFFFFFU;
  if (lhs_cp < rhs_cp)
    return -1;
  if (lhs_cp > rhs_cp)
    return 1;
  return 0;
}

int ON_Big5UnicodePair::CompareUnicodeCodePoint(const ON_Big5UnicodePair* lhs, const ON_Big5UnicodePair* rhs)
{
  const unsigned lhs_cp = (nullptr != lhs) ? lhs->UnicodeCodePoint() : 0xFFFFFFFFU;
  const unsigned rhs_cp = (nullptr != rhs) ? rhs->UnicodeCodePoint() : 0xFFFFFFFFU;
  if (lhs_cp < rhs_cp)
    return -1;
  if (lhs_cp > rhs_cp)
    return 1;
  return 0;
}

int ON_Big5UnicodePair::CompareBig5AndUnicodeCodePoints(const ON_Big5UnicodePair* lhs, const ON_Big5UnicodePair* rhs)
{
  int rc = ON_Big5UnicodePair::CompareBig5CodePoint(lhs, rhs);
  if (0 == rc)
    rc = ON_Big5UnicodePair::CompareUnicodeCodePoint(lhs, rhs);
  return rc;
}

int ON_Big5UnicodePair::CompareUnicodeAndBig5CodePoints(const ON_Big5UnicodePair* lhs, const ON_Big5UnicodePair* rhs)
{
  int rc = ON_Big5UnicodePair::CompareUnicodeCodePoint(lhs, rhs);
  if (0 == rc)
    rc = ON_Big5UnicodePair::CompareBig5CodePoint(lhs, rhs);
  return rc;
}

bool operator==(ON_Big5UnicodePair lhs, ON_Big5UnicodePair rhs)
{
  return 0 == ON_Big5UnicodePair::CompareBig5AndUnicodeCodePoints(&lhs, &rhs);
}

bool operator!=(ON_Big5UnicodePair lhs, ON_Big5UnicodePair rhs)
{
  return 0 != ON_Big5UnicodePair::CompareBig5AndUnicodeCodePoints(&lhs, &rhs);
}

const ON_Big5CodePoint ON_Big5CodePoint::CreateFromUnicode(
  unsigned int unicode_code_point,
  ON_Big5CodePoint not_avilable
)
{
  if (unicode_code_point > 0xFFFF)
    return ON_IsValidUnicodeCodePoint(unicode_code_point) ? not_avilable : ON_Big5CodePoint::Error;

  return ON_Big5CodePoint::CreateFromUnicode(ON_UnicodeShortCodePoint::Create(unicode_code_point), not_avilable);
}

const ON_Big5CodePoint ON_Big5CodePoint::CreateFromUnicode(
  const ON_UnicodeShortCodePoint& unicode_code_point,
  ON_Big5CodePoint not_avilable
)
{
  if (unicode_code_point.IsASCII(true))
    return ON_Big5CodePoint::Create(unicode_code_point.UnicodeCodePoint()); // ASCII code point

  if (false == unicode_code_point.IsValid(false, false))
    return ON_Big5CodePoint::Error;

  const ON_SimpleArray<ON_Big5UnicodePair>& unicode_to_big5 = ON_Big5UnicodePair::UnicodeToBig5();

  const ON_Big5UnicodePair key = ON_Big5UnicodePair::Create(ON_Big5CodePoint::Null, unicode_code_point);
  const int i = unicode_to_big5.BinarySearch(&key, ON_Big5UnicodePair::CompareUnicodeCodePoint);
  if (i >= 0)
  {
    const ON_Big5UnicodePair pair = unicode_to_big5[i];
    if (unicode_code_point == pair.Unicode() && pair.IsValid(false, false))
      return pair.Big5();
  }

  // No BIG5 code point available for this Unicode code point.
  return not_avilable;
}

const ON_UnicodeShortCodePoint ON_UnicodeShortCodePoint::CreateFromBig5(
  unsigned int big5_code_point,
  ON_UnicodeShortCodePoint not_avilable
)
{
  return ON_UnicodeShortCodePoint::CreateFromBig5(ON_Big5CodePoint::Create(big5_code_point), not_avilable);
}

const ON_UnicodeShortCodePoint ON_UnicodeShortCodePoint::CreateFromBig5(
  const ON_Big5CodePoint& big5_code_point,
  ON_UnicodeShortCodePoint not_avilable
)
{
  if (big5_code_point.IsASCII(true))
    return ON_UnicodeShortCodePoint::Create(big5_code_point.Big5CodePoint()); // ASCII code point

  if (false == big5_code_point.IsValid(false, false))
    return ON_UnicodeShortCodePoint::Error;

  const ON_SimpleArray<ON_Big5UnicodePair>& big5_to_unicode = ON_Big5UnicodePair::Big5ToUnicode();

  const ON_Big5UnicodePair key = ON_Big5UnicodePair::Create(big5_code_point, ON_UnicodeShortCodePoint::Null);
  const int i = big5_to_unicode.BinarySearch(&key, ON_Big5UnicodePair::CompareBig5CodePoint);
  if (i >= 0)
  {
    const ON_Big5UnicodePair pair = big5_to_unicode[i];
    if (big5_code_point == pair.Big5() && pair.IsValid(false, false))
      return pair.Unicode();
  }

  // No Unicode code point available for this BIG5 code point.
  return not_avilable;
}

////////////////////////

const ON_SimpleArray< ON_Big5UnicodePair >& ON_Big5UnicodePair::Big5ToUnicode()
{
  // The pair[][2] array was created by combining information from 
  // http://www.unicode.org/Public/MAPPINGS/OBSOLETE/EASTASIA/OTHER/BIG5.TXT, retrieved: May 17, 2021,
  // and Windows 10 MultiByteToWideChar() using code page 950.
  //
  // pairs[][2] = {...{BIG5 code point, Unicode code point}...}
  // sorted by BIG5 code point. All Unicode code points must be valid and < 0xFFFF.
  //
  // The primary use of this code is to convert legacy BIG5 encoded information into
  // Unicode encoded information and modifications to the original sources
  // (marked with [MODyyymmdd]) were made to make thos conversions as robust as possible.
  //

  static const ON__UINT16 pairs[][2] = {
  {0xA140,0x3000}, // IDEOGRAPHIC SPACE
  {0xA141,0xff0c}, // FULLWIDTH COMMA
  {0xA142,0x3001}, // IDEOGRAPHIC COMMA
  {0xA143,0x3002}, // IDEOGRAPHIC FULL STOP
  {0xA144,0xff0e}, // FULLWIDTH FULL STOP
  {0xA145,0x2027}, // HYPHENATION POINT [MOD20210517] changed from 0x2022 # BULLET
  {0xA146,0xff1b}, // FULLWIDTH SEMICOLON
  {0xA147,0xff1a}, // FULLWIDTH COLON
  {0xA148,0xff1f}, // FULLWIDTH QUESTION MARK
  {0xA149,0xff01}, // FULLWIDTH EXCLAMATION MARK
  {0xA14A,0xfe30}, // PRESENTATION FORM FOR VERTICAL TWO DOT LEADER
  {0xA14B,0x2026}, // HORIZONTAL ELLIPSIS
  {0xA14C,0x2025}, // TWO DOT LEADER
  {0xA14D,0xfe50}, // SMALL COMMA
  {0xA14E,0xfe51}, // SMALL IDEOGRAPHIC COMMA [MOD20210517] changed from 0xFF64 # HALFWIDTH IDEOGRAPHIC COMMA
  {0xA14F,0xfe52}, // SMALL FULL STOP
  {0xA150,0x00b7}, // MIDDLE DOT
  {0xA151,0xfe54}, // SMALL SEMICOLON
  {0xA152,0xfe55}, // SMALL COLON
  {0xA153,0xfe56}, // SMALL QUESTION MARK
  {0xA154,0xfe57}, // SMALL EXCLAMATION MARK
  {0xA155,0xff5c}, // FULLWIDTH VERTICAL BAR
  {0xA156,0x2013}, // EN DASH
  {0xA157,0xfe31}, // PRESENTATION FORM FOR VERTICAL EM DASH
  {0xA158,0x2014}, // EM DASH
  {0xA159,0xfe33}, // PRESENTATION FORM FOR VERTICAL LOW LINE
  {0xA15A,0x2574}, // BOX DRAWINGS LIGHT LEFT [MOD20210517] changed from 0xFFFD # *** NO MAPPING ***
  {0xA15B,0xfe34}, // PRESENTATION FORM FOR VERTICAL WAVY LOW LINE
  {0xA15C,0xfe4f}, // WAVY LOW LINE
  {0xA15D,0xff08}, // FULLWIDTH LEFT PARENTHESIS
  {0xA15E,0xff09}, // FULLWIDTH RIGHT PARENTHESIS
  {0xA15F,0xfe35}, // PRESENTATION FORM FOR VERTICAL LEFT PARENTHESIS
  {0xA160,0xfe36}, // PRESENTATION FORM FOR VERTICAL RIGHT PARENTHESIS
  {0xA161,0xff5b}, // FULLWIDTH LEFT CURLY BRACKET
  {0xA162,0xff5d}, // FULLWIDTH RIGHT CURLY BRACKET
  {0xA163,0xfe37}, // PRESENTATION FORM FOR VERTICAL LEFT CURLY BRACKET
  {0xA164,0xfe38}, // PRESENTATION FORM FOR VERTICAL RIGHT CURLY BRACKET
  {0xA165,0x3014}, // LEFT TORTOISE SHELL BRACKET
  {0xA166,0x3015}, // RIGHT TORTOISE SHELL BRACKET
  {0xA167,0xfe39}, // PRESENTATION FORM FOR VERTICAL LEFT TORTOISE SHELL BRACKET
  {0xA168,0xfe3a}, // PRESENTATION FORM FOR VERTICAL RIGHT TORTOISE SHELL BRACKET
  {0xA169,0x3010}, // LEFT BLACK LENTICULAR BRACKET
  {0xA16A,0x3011}, // RIGHT BLACK LENTICULAR BRACKET
  {0xA16B,0xfe3b}, // PRESENTATION FORM FOR VERTICAL LEFT BLACK LENTICULAR BRACKET
  {0xA16C,0xfe3c}, // PRESENTATION FORM FOR VERTICAL RIGHT BLACK LENTICULAR BRACKET
  {0xA16D,0x300a}, // LEFT DOUBLE ANGLE BRACKET
  {0xA16E,0x300b}, // RIGHT DOUBLE ANGLE BRACKET
  {0xA16F,0xfe3d}, // PRESENTATION FORM FOR VERTICAL LEFT DOUBLE ANGLE BRACKET
  {0xA170,0xfe3e}, // PRESENTATION FORM FOR VERTICAL RIGHT DOUBLE ANGLE BRACKET
  {0xA171,0x3008}, // LEFT ANGLE BRACKET
  {0xA172,0x3009}, // RIGHT ANGLE BRACKET
  {0xA173,0xfe3f}, // PRESENTATION FORM FOR VERTICAL LEFT ANGLE BRACKET
  {0xA174,0xfe40}, // PRESENTATION FORM FOR VERTICAL RIGHT ANGLE BRACKET
  {0xA175,0x300c}, // LEFT CORNER BRACKET
  {0xA176,0x300d}, // RIGHT CORNER BRACKET
  {0xA177,0xfe41}, // PRESENTATION FORM FOR VERTICAL LEFT CORNER BRACKET
  {0xA178,0xfe42}, // PRESENTATION FORM FOR VERTICAL RIGHT CORNER BRACKET
  {0xA179,0x300e}, // LEFT WHITE CORNER BRACKET
  {0xA17A,0x300f}, // RIGHT WHITE CORNER BRACKET
  {0xA17B,0xfe43}, // PRESENTATION FORM FOR VERTICAL LEFT WHITE CORNER BRACKET
  {0xA17C,0xfe44}, // PRESENTATION FORM FOR VERTICAL RIGHT WHITE CORNER BRACKET
  {0xA17D,0xfe59}, // SMALL LEFT PARENTHESIS
  {0xA17E,0xfe5a}, // SMALL RIGHT PARENTHESIS
  {0xA1A1,0xfe5b}, // SMALL LEFT CURLY BRACKET
  {0xA1A2,0xfe5c}, // SMALL RIGHT CURLY BRACKET
  {0xA1A3,0xfe5d}, // SMALL LEFT TORTOISE SHELL BRACKET
  {0xA1A4,0xfe5e}, // SMALL RIGHT TORTOISE SHELL BRACKET
  {0xA1A5,0x2018}, // LEFT SINGLE QUOTATION MARK
  {0xA1A6,0x2019}, // RIGHT SINGLE QUOTATION MARK
  {0xA1A7,0x201c}, // LEFT DOUBLE QUOTATION MARK
  {0xA1A8,0x201d}, // RIGHT DOUBLE QUOTATION MARK
  {0xA1A9,0x301d}, // REVERSED DOUBLE PRIME QUOTATION MARK
  {0xA1AA,0x301e}, // DOUBLE PRIME QUOTATION MARK
  {0xA1AB,0x2035}, // REVERSED PRIME
  {0xA1AC,0x2032}, // PRIME
  {0xA1AD,0xff03}, // FULLWIDTH NUMBER SIGN
  {0xA1AE,0xff06}, // FULLWIDTH AMPERSAND
  {0xA1AF,0xff0a}, // FULLWIDTH ASTERISK
  {0xA1B0,0x203b}, // REFERENCE MARK
  {0xA1B1,0x00a7}, // SECTION SIGN
  {0xA1B2,0x3003}, // DITTO MARK
  {0xA1B3,0x25cb}, // WHITE CIRCLE
  {0xA1B4,0x25cf}, // BLACK CIRCLE
  {0xA1B5,0x25b3}, // WHITE UP-POINTING TRIANGLE
  {0xA1B6,0x25b2}, // BLACK UP-POINTING TRIANGLE
  {0xA1B7,0x25ce}, // BULLSEYE
  {0xA1B8,0x2606}, // WHITE STAR
  {0xA1B9,0x2605}, // BLACK STAR
  {0xA1BA,0x25c7}, // WHITE DIAMOND
  {0xA1BB,0x25c6}, // BLACK DIAMOND
  {0xA1BC,0x25a1}, // WHITE SQUARE
  {0xA1BD,0x25a0}, // BLACK SQUARE
  {0xA1BE,0x25bd}, // WHITE DOWN-POINTING TRIANGLE
  {0xA1BF,0x25bc}, // BLACK DOWN-POINTING TRIANGLE
  {0xA1C0,0x32a3}, // CIRCLED IDEOGRAPH CORRECT
  {0xA1C1,0x2105}, // CARE OF
  {0xA1C2,0x00af}, // MACRON [MOD20210517] changed from 0x203E # OVERLINE
  {0xA1C3,0xffe3}, // FULLWIDTH MACRON [MOD20210517] changed from 0xFFFD # *** NO MAPPING ***
  {0xA1C4,0xff3f}, // FULLWIDTH LOW LINE
  {0xA1C5,0x02cd}, // MODIFIER LETTER LOW MACRON [MOD20210517] changed from 0xFFFD # *** NO MAPPING ***
  {0xA1C6,0xfe49}, // DASHED OVERLINE
  {0xA1C7,0xfe4a}, // CENTRELINE OVERLINE
  {0xA1C8,0xfe4d}, // DASHED LOW LINE
  {0xA1C9,0xfe4e}, // CENTRELINE LOW LINE
  {0xA1CA,0xfe4b}, // WAVY OVERLINE
  {0xA1CB,0xfe4c}, // DOUBLE WAVY OVERLINE
  {0xA1CC,0xfe5f}, // SMALL NUMBER SIGN
  {0xA1CD,0xfe60}, // SMALL AMPERSAND
  {0xA1CE,0xfe61}, // SMALL ASTERISK
  {0xA1CF,0xff0b}, // FULLWIDTH PLUS SIGN
  {0xA1D0,0xff0d}, // FULLWIDTH HYPHEN-MINUS
  {0xA1D1,0x00d7}, // MULTIPLICATION SIGN
  {0xA1D2,0x00f7}, // DIVISION SIGN
  {0xA1D3,0x00b1}, // PLUS-MINUS SIGN
  {0xA1D4,0x221a}, // SQUARE ROOT
  {0xA1D5,0xff1c}, // FULLWIDTH LESS-THAN SIGN
  {0xA1D6,0xff1e}, // FULLWIDTH GREATER-THAN SIGN
  {0xA1D7,0xff1d}, // FULLWIDTH EQUALS SIGN
  {0xA1D8,0x2266}, // LESS THAN OVER EQUAL TO
  {0xA1D9,0x2267}, // GREATER THAN OVER EQUAL TO
  {0xA1DA,0x2260}, // NOT EQUAL TO
  {0xA1DB,0x221e}, // INFINITY
  {0xA1DC,0x2252}, // APPROXIMATELY EQUAL TO OR THE IMAGE OF
  {0xA1DD,0x2261}, // IDENTICAL TO
  {0xA1DE,0xfe62}, // SMALL PLUS SIGN
  {0xA1DF,0xfe63}, // SMALL HYPHEN-MINUS
  {0xA1E0,0xfe64}, // SMALL LESS-THAN SIGN
  {0xA1E1,0xfe65}, // SMALL GREATER-THAN SIGN
  {0xA1E2,0xfe66}, // SMALL EQUALS SIGN
  {0xA1E3,0xff5e}, // YI SYLLABLE GAP [MOD20210517] changed from 0x223C # TILDE OPERATOR
  {0xA1E4,0x2229}, // INTERSECTION
  {0xA1E5,0x222a}, // UNION
  {0xA1E6,0x22a5}, // UP TACK
  {0xA1E7,0x2220}, // ANGLE
  {0xA1E8,0x221f}, // RIGHT ANGLE
  {0xA1E9,0x22bf}, // RIGHT TRIANGLE
  {0xA1EA,0x33d2}, // SQUARE LOG
  {0xA1EB,0x33d1}, // SQUARE LN
  {0xA1EC,0x222b}, // INTEGRAL
  {0xA1ED,0x222e}, // CONTOUR INTEGRAL
  {0xA1EE,0x2235}, // BECAUSE
  {0xA1EF,0x2234}, // THEREFORE
  {0xA1F0,0x2640}, // FEMALE SIGN
  {0xA1F1,0x2642}, // MALE SIGN
  {0xA1F2,0x2295}, // CIRCLED PLUS [MOD20210517] changed from 0x2641	# EARTH
  {0xA1F3,0x2299}, // CIRCLED DOT OPERATOR [MOD20210517] changed from 0x2609	# SUN
  {0xA1F4,0x2191}, // UPWARDS ARROW
  {0xA1F5,0x2193}, // DOWNWARDS ARROW
  {0xA1F6,0x2190}, // LEFTWARDS ARROW
  {0xA1F7,0x2192}, // RIGHTWARDS ARROW
  {0xA1F8,0x2196}, // NORTH WEST ARROW
  {0xA1F9,0x2197}, // NORTH EAST ARROW
  {0xA1FA,0x2199}, // SOUTH WEST ARROW
  {0xA1FB,0x2198}, // SOUTH EAST ARROW
  {0xA1FC,0x2225}, // PARALLEL TO
  {0xA1FD,0x2223}, // DIVIDES
  {0xA1FE,0xff0f}, // FULLWIDTH SOLIDUS [MOD20210517] changed from 0xFFFD # *** NO MAPPING ***
  {0xA240,0xff3c}, // FULLWIDTH REVERSE SOLIDUS [MOD20210517] changed from 0xFFFD # *** NO MAPPING ***
  {0xA241,0x2215}, // DIVISION SLASH [MOD20210517] changed from 0xFF0F # FULLWIDTH SOLIDUS
  {0xA242,0xfe68}, // SMALL REVERSE SOLIDUS [MOD20210517] changed from 0xFF3C # FULLWIDTH REVERSE SOLIDUS
  {0xA243,0xff04}, // FULLWIDTH DOLLAR SIGN
  {0xA244,0xffe5}, // FULLWIDTH YEN SIGN [MOD20210517] changed from 0x00A5 # YEN SIGN
  {0xA245,0x3012}, // POSTAL MARK
  {0xA246,0xffe0}, // FULLWIDTH CENT SIGN [MOD20210517] changed from 0x00A2 # CENT SIGN
  {0xA247,0xffe1}, // FULLWIDTH POUND SIGN [MOD20210517] changed from 0x00A3 # POUND SIGN
  {0xA248,0xff05}, // FULLWIDTH PERCENT SIGN
  {0xA249,0xff20}, // FULLWIDTH COMMERCIAL AT
  {0xA24A,0x2103}, // DEGREE CELSIUS
  {0xA24B,0x2109}, // DEGREE FAHRENHEIT
  {0xA24C,0xfe69}, // SMALL DOLLAR SIGN
  {0xA24D,0xfe6a}, // SMALL PERCENT SIGN
  {0xA24E,0xfe6b}, // SMALL COMMERCIAL AT
  {0xA24F,0x33d5}, // SQUARE MIL
  {0xA250,0x339c}, // SQUARE MM
  {0xA251,0x339d}, // SQUARE CM
  {0xA252,0x339e}, // SQUARE KM
  {0xA253,0x33ce}, // SQUARE KM CAPITAL
  {0xA254,0x33a1}, // SQUARE M SQUARED
  {0xA255,0x338e}, // SQUARE MG
  {0xA256,0x338f}, // SQUARE KG
  {0xA257,0x33c4}, // SQUARE CC
  {0xA258,0x00b0}, // DEGREE SIGN
  {0xA259,0x5159}, // <CJK>
  {0xA25A,0x515b}, // <CJK>
  {0xA25B,0x515e}, // <CJK>
  {0xA25C,0x515d}, // <CJK>
  {0xA25D,0x5161}, // <CJK>
  {0xA25E,0x5163}, // <CJK>
  {0xA25F,0x55e7}, // <CJK>
  {0xA260,0x74e9}, // <CJK>
  {0xA261,0x7cce}, // <CJK>
  {0xA262,0x2581}, // LOWER ONE EIGHTH BLOCK
  {0xA263,0x2582}, // LOWER ONE QUARTER BLOCK
  {0xA264,0x2583}, // LOWER THREE EIGHTHS BLOCK
  {0xA265,0x2584}, // LOWER HALF BLOCK
  {0xA266,0x2585}, // LOWER FIVE EIGHTHS BLOCK
  {0xA267,0x2586}, // LOWER THREE QUARTERS BLOCK
  {0xA268,0x2587}, // LOWER SEVEN EIGHTHS BLOCK
  {0xA269,0x2588}, // FULL BLOCK
  {0xA26A,0x258f}, // LEFT ONE EIGHTH BLOCK
  {0xA26B,0x258e}, // LEFT ONE QUARTER BLOCK
  {0xA26C,0x258d}, // LEFT THREE EIGHTHS BLOCK
  {0xA26D,0x258c}, // LEFT HALF BLOCK
  {0xA26E,0x258b}, // LEFT FIVE EIGHTHS BLOCK
  {0xA26F,0x258a}, // LEFT THREE QUARTERS BLOCK
  {0xA270,0x2589}, // LEFT SEVEN EIGHTHS BLOCK
  {0xA271,0x253c}, // BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL
  {0xA272,0x2534}, // BOX DRAWINGS LIGHT UP AND HORIZONTAL
  {0xA273,0x252c}, // BOX DRAWINGS LIGHT DOWN AND HORIZONTAL
  {0xA274,0x2524}, // BOX DRAWINGS LIGHT VERTICAL AND LEFT
  {0xA275,0x251c}, // BOX DRAWINGS LIGHT VERTICAL AND RIGHT
  {0xA276,0x2594}, // UPPER ONE EIGHTH BLOCK
  {0xA277,0x2500}, // BOX DRAWINGS LIGHT HORIZONTAL
  {0xA278,0x2502}, // BOX DRAWINGS LIGHT VERTICAL
  {0xA279,0x2595}, // RIGHT ONE EIGHTH BLOCK
  {0xA27A,0x250c}, // BOX DRAWINGS LIGHT DOWN AND RIGHT
  {0xA27B,0x2510}, // BOX DRAWINGS LIGHT DOWN AND LEFT
  {0xA27C,0x2514}, // BOX DRAWINGS LIGHT UP AND RIGHT
  {0xA27D,0x2518}, // BOX DRAWINGS LIGHT UP AND LEFT
  {0xA27E,0x256d}, // BOX DRAWINGS LIGHT ARC DOWN AND RIGHT
  {0xA2A1,0x256e}, // BOX DRAWINGS LIGHT ARC DOWN AND LEFT
  {0xA2A2,0x2570}, // BOX DRAWINGS LIGHT ARC UP AND RIGHT
  {0xA2A3,0x256f}, // BOX DRAWINGS LIGHT ARC UP AND LEFT
  {0xA2A4,0x2550}, // BOX DRAWINGS DOUBLE HORIZONTAL
  {0xA2A5,0x255e}, // BOX DRAWINGS VERTICAL SINGLE AND RIGHT DOUBLE
  {0xA2A6,0x256a}, // BOX DRAWINGS VERTICAL SINGLE AND HORIZONTAL DOUBLE
  {0xA2A7,0x2561}, // BOX DRAWINGS VERTICAL SINGLE AND LEFT DOUBLE
  {0xA2A8,0x25e2}, // BLACK LOWER RIGHT TRIANGLE
  {0xA2A9,0x25e3}, // BLACK LOWER LEFT TRIANGLE
  {0xA2AA,0x25e5}, // BLACK UPPER RIGHT TRIANGLE
  {0xA2AB,0x25e4}, // BLACK UPPER LEFT TRIANGLE
  {0xA2AC,0x2571}, // BOX DRAWINGS LIGHT DIAGONAL UPPER RIGHT TO LOWER LEFT
  {0xA2AD,0x2572}, // BOX DRAWINGS LIGHT DIAGONAL UPPER LEFT TO LOWER RIGHT
  {0xA2AE,0x2573}, // BOX DRAWINGS LIGHT DIAGONAL CROSS
  {0xA2AF,0xff10}, // FULLWIDTH DIGIT ZERO
  {0xA2B0,0xff11}, // FULLWIDTH DIGIT ONE
  {0xA2B1,0xff12}, // FULLWIDTH DIGIT TWO
  {0xA2B2,0xff13}, // FULLWIDTH DIGIT THREE
  {0xA2B3,0xff14}, // FULLWIDTH DIGIT FOUR
  {0xA2B4,0xff15}, // FULLWIDTH DIGIT FIVE
  {0xA2B5,0xff16}, // FULLWIDTH DIGIT SIX
  {0xA2B6,0xff17}, // FULLWIDTH DIGIT SEVEN
  {0xA2B7,0xff18}, // FULLWIDTH DIGIT EIGHT
  {0xA2B8,0xff19}, // FULLWIDTH DIGIT NINE
  {0xA2B9,0x2160}, // ROMAN NUMERAL ONE
  {0xA2BA,0x2161}, // ROMAN NUMERAL TWO
  {0xA2BB,0x2162}, // ROMAN NUMERAL THREE
  {0xA2BC,0x2163}, // ROMAN NUMERAL FOUR
  {0xA2BD,0x2164}, // ROMAN NUMERAL FIVE
  {0xA2BE,0x2165}, // ROMAN NUMERAL SIX
  {0xA2BF,0x2166}, // ROMAN NUMERAL SEVEN
  {0xA2C0,0x2167}, // ROMAN NUMERAL EIGHT
  {0xA2C1,0x2168}, // ROMAN NUMERAL NINE
  {0xA2C2,0x2169}, // ROMAN NUMERAL TEN
  {0xA2C3,0x3021}, // HANGZHOU NUMERAL ONE
  {0xA2C4,0x3022}, // HANGZHOU NUMERAL TWO
  {0xA2C5,0x3023}, // HANGZHOU NUMERAL THREE
  {0xA2C6,0x3024}, // HANGZHOU NUMERAL FOUR
  {0xA2C7,0x3025}, // HANGZHOU NUMERAL FIVE
  {0xA2C8,0x3026}, // HANGZHOU NUMERAL SIX
  {0xA2C9,0x3027}, // HANGZHOU NUMERAL SEVEN
  {0xA2CA,0x3028}, // HANGZHOU NUMERAL EIGHT
  {0xA2CB,0x3029}, // HANGZHOU NUMERAL NINE
  {0xA2CC,0x5341}, // <CJK> [MOD20210517] changed from 0xFFFD # *** NO MAPPING ***
  {0xA2CD,0x5344}, // <CJK>
  {0xA2CE,0x5345}, // <CJK> [MOD20210517] changed from 0xFFFD # *** NO MAPPING ***
  {0xA2CF,0xff21}, // FULLWIDTH LATIN CAPITAL LETTER A
  {0xA2D0,0xff22}, // FULLWIDTH LATIN CAPITAL LETTER B
  {0xA2D1,0xff23}, // FULLWIDTH LATIN CAPITAL LETTER C
  {0xA2D2,0xff24}, // FULLWIDTH LATIN CAPITAL LETTER D
  {0xA2D3,0xff25}, // FULLWIDTH LATIN CAPITAL LETTER E
  {0xA2D4,0xff26}, // FULLWIDTH LATIN CAPITAL LETTER F
  {0xA2D5,0xff27}, // FULLWIDTH LATIN CAPITAL LETTER G
  {0xA2D6,0xff28}, // FULLWIDTH LATIN CAPITAL LETTER H
  {0xA2D7,0xff29}, // FULLWIDTH LATIN CAPITAL LETTER I
  {0xA2D8,0xff2a}, // FULLWIDTH LATIN CAPITAL LETTER J
  {0xA2D9,0xff2b}, // FULLWIDTH LATIN CAPITAL LETTER K
  {0xA2DA,0xff2c}, // FULLWIDTH LATIN CAPITAL LETTER L
  {0xA2DB,0xff2d}, // FULLWIDTH LATIN CAPITAL LETTER M
  {0xA2DC,0xff2e}, // FULLWIDTH LATIN CAPITAL LETTER N
  {0xA2DD,0xff2f}, // FULLWIDTH LATIN CAPITAL LETTER O
  {0xA2DE,0xff30}, // FULLWIDTH LATIN CAPITAL LETTER P
  {0xA2DF,0xff31}, // FULLWIDTH LATIN CAPITAL LETTER Q
  {0xA2E0,0xff32}, // FULLWIDTH LATIN CAPITAL LETTER R
  {0xA2E1,0xff33}, // FULLWIDTH LATIN CAPITAL LETTER S
  {0xA2E2,0xff34}, // FULLWIDTH LATIN CAPITAL LETTER T
  {0xA2E3,0xff35}, // FULLWIDTH LATIN CAPITAL LETTER U
  {0xA2E4,0xff36}, // FULLWIDTH LATIN CAPITAL LETTER V
  {0xA2E5,0xff37}, // FULLWIDTH LATIN CAPITAL LETTER W
  {0xA2E6,0xff38}, // FULLWIDTH LATIN CAPITAL LETTER X
  {0xA2E7,0xff39}, // FULLWIDTH LATIN CAPITAL LETTER Y
  {0xA2E8,0xff3a}, // FULLWIDTH LATIN CAPITAL LETTER Z
  {0xA2E9,0xff41}, // FULLWIDTH LATIN SMALL LETTER A
  {0xA2EA,0xff42}, // FULLWIDTH LATIN SMALL LETTER B
  {0xA2EB,0xff43}, // FULLWIDTH LATIN SMALL LETTER C
  {0xA2EC,0xff44}, // FULLWIDTH LATIN SMALL LETTER D
  {0xA2ED,0xff45}, // FULLWIDTH LATIN SMALL LETTER E
  {0xA2EE,0xff46}, // FULLWIDTH LATIN SMALL LETTER F
  {0xA2EF,0xff47}, // FULLWIDTH LATIN SMALL LETTER G
  {0xA2F0,0xff48}, // FULLWIDTH LATIN SMALL LETTER H
  {0xA2F1,0xff49}, // FULLWIDTH LATIN SMALL LETTER I
  {0xA2F2,0xff4a}, // FULLWIDTH LATIN SMALL LETTER J
  {0xA2F3,0xff4b}, // FULLWIDTH LATIN SMALL LETTER K
  {0xA2F4,0xff4c}, // FULLWIDTH LATIN SMALL LETTER L
  {0xA2F5,0xff4d}, // FULLWIDTH LATIN SMALL LETTER M
  {0xA2F6,0xff4e}, // FULLWIDTH LATIN SMALL LETTER N
  {0xA2F7,0xff4f}, // FULLWIDTH LATIN SMALL LETTER O
  {0xA2F8,0xff50}, // FULLWIDTH LATIN SMALL LETTER P
  {0xA2F9,0xff51}, // FULLWIDTH LATIN SMALL LETTER Q
  {0xA2FA,0xff52}, // FULLWIDTH LATIN SMALL LETTER R
  {0xA2FB,0xff53}, // FULLWIDTH LATIN SMALL LETTER S
  {0xA2FC,0xff54}, // FULLWIDTH LATIN SMALL LETTER T
  {0xA2FD,0xff55}, // FULLWIDTH LATIN SMALL LETTER U
  {0xA2FE,0xff56}, // FULLWIDTH LATIN SMALL LETTER V
  {0xA340,0xff57}, // FULLWIDTH LATIN SMALL LETTER W
  {0xA341,0xff58}, // FULLWIDTH LATIN SMALL LETTER X
  {0xA342,0xff59}, // FULLWIDTH LATIN SMALL LETTER Y
  {0xA343,0xff5a}, // FULLWIDTH LATIN SMALL LETTER Z
  {0xA344,0x0391}, // GREEK CAPITAL LETTER ALPHA
  {0xA345,0x0392}, // GREEK CAPITAL LETTER BETA
  {0xA346,0x0393}, // GREEK CAPITAL LETTER GAMMA
  {0xA347,0x0394}, // GREEK CAPITAL LETTER DELTA
  {0xA348,0x0395}, // GREEK CAPITAL LETTER EPSILON
  {0xA349,0x0396}, // GREEK CAPITAL LETTER ZETA
  {0xA34A,0x0397}, // GREEK CAPITAL LETTER ETA
  {0xA34B,0x0398}, // GREEK CAPITAL LETTER THETA
  {0xA34C,0x0399}, // GREEK CAPITAL LETTER IOTA
  {0xA34D,0x039a}, // GREEK CAPITAL LETTER KAPPA
  {0xA34E,0x039b}, // GREEK CAPITAL LETTER LAMBDA
  {0xA34F,0x039c}, // GREEK CAPITAL LETTER MU
  {0xA350,0x039d}, // GREEK CAPITAL LETTER NU
  {0xA351,0x039e}, // GREEK CAPITAL LETTER XI
  {0xA352,0x039f}, // GREEK CAPITAL LETTER OMICRON
  {0xA353,0x03a0}, // GREEK CAPITAL LETTER PI
  {0xA354,0x03a1}, // GREEK CAPITAL LETTER RHO
  {0xA355,0x03a3}, // GREEK CAPITAL LETTER SIGMA
  {0xA356,0x03a4}, // GREEK CAPITAL LETTER TAU
  {0xA357,0x03a5}, // GREEK CAPITAL LETTER UPSILON
  {0xA358,0x03a6}, // GREEK CAPITAL LETTER PHI
  {0xA359,0x03a7}, // GREEK CAPITAL LETTER CHI
  {0xA35A,0x03a8}, // GREEK CAPITAL LETTER PSI
  {0xA35B,0x03a9}, // GREEK CAPITAL LETTER OMEGA
  {0xA35C,0x03b1}, // GREEK SMALL LETTER ALPHA
  {0xA35D,0x03b2}, // GREEK SMALL LETTER BETA
  {0xA35E,0x03b3}, // GREEK SMALL LETTER GAMMA
  {0xA35F,0x03b4}, // GREEK SMALL LETTER DELTA
  {0xA360,0x03b5}, // GREEK SMALL LETTER EPSILON
  {0xA361,0x03b6}, // GREEK SMALL LETTER ZETA
  {0xA362,0x03b7}, // GREEK SMALL LETTER ETA
  {0xA363,0x03b8}, // GREEK SMALL LETTER THETA
  {0xA364,0x03b9}, // GREEK SMALL LETTER IOTA
  {0xA365,0x03ba}, // GREEK SMALL LETTER KAPPA
  {0xA366,0x03bb}, // GREEK SMALL LETTER LAMBDA
  {0xA367,0x03bc}, // GREEK SMALL LETTER MU
  {0xA368,0x03bd}, // GREEK SMALL LETTER NU
  {0xA369,0x03be}, // GREEK SMALL LETTER XI
  {0xA36A,0x03bf}, // GREEK SMALL LETTER OMICRON
  {0xA36B,0x03c0}, // GREEK SMALL LETTER PI
  {0xA36C,0x03c1}, // GREEK SMALL LETTER RHO
  {0xA36D,0x03c3}, // GREEK SMALL LETTER SIGMA
  {0xA36E,0x03c4}, // GREEK SMALL LETTER TAU
  {0xA36F,0x03c5}, // GREEK SMALL LETTER UPSILON
  {0xA370,0x03c6}, // GREEK SMALL LETTER PHI
  {0xA371,0x03c7}, // GREEK SMALL LETTER CHI
  {0xA372,0x03c8}, // GREEK SMALL LETTER PSI
  {0xA373,0x03c9}, // GREEK SMALL LETTER OMEGA
  {0xA374,0x3105}, // BOPOMOFO LETTER B
  {0xA375,0x3106}, // BOPOMOFO LETTER P
  {0xA376,0x3107}, // BOPOMOFO LETTER M
  {0xA377,0x3108}, // BOPOMOFO LETTER F
  {0xA378,0x3109}, // BOPOMOFO LETTER D
  {0xA379,0x310a}, // BOPOMOFO LETTER T
  {0xA37A,0x310b}, // BOPOMOFO LETTER N
  {0xA37B,0x310c}, // BOPOMOFO LETTER L
  {0xA37C,0x310d}, // BOPOMOFO LETTER G
  {0xA37D,0x310e}, // BOPOMOFO LETTER K
  {0xA37E,0x310f}, // BOPOMOFO LETTER H
  {0xA3A1,0x3110}, // BOPOMOFO LETTER J
  {0xA3A2,0x3111}, // BOPOMOFO LETTER Q
  {0xA3A3,0x3112}, // BOPOMOFO LETTER X
  {0xA3A4,0x3113}, // BOPOMOFO LETTER ZH
  {0xA3A5,0x3114}, // BOPOMOFO LETTER CH
  {0xA3A6,0x3115}, // BOPOMOFO LETTER SH
  {0xA3A7,0x3116}, // BOPOMOFO LETTER R
  {0xA3A8,0x3117}, // BOPOMOFO LETTER Z
  {0xA3A9,0x3118}, // BOPOMOFO LETTER C
  {0xA3AA,0x3119}, // BOPOMOFO LETTER S
  {0xA3AB,0x311a}, // BOPOMOFO LETTER A
  {0xA3AC,0x311b}, // BOPOMOFO LETTER O
  {0xA3AD,0x311c}, // BOPOMOFO LETTER E
  {0xA3AE,0x311d}, // BOPOMOFO LETTER EH
  {0xA3AF,0x311e}, // BOPOMOFO LETTER AI
  {0xA3B0,0x311f}, // BOPOMOFO LETTER EI
  {0xA3B1,0x3120}, // BOPOMOFO LETTER AU
  {0xA3B2,0x3121}, // BOPOMOFO LETTER OU
  {0xA3B3,0x3122}, // BOPOMOFO LETTER AN
  {0xA3B4,0x3123}, // BOPOMOFO LETTER EN
  {0xA3B5,0x3124}, // BOPOMOFO LETTER ANG
  {0xA3B6,0x3125}, // BOPOMOFO LETTER ENG
  {0xA3B7,0x3126}, // BOPOMOFO LETTER ER
  {0xA3B8,0x3127}, // BOPOMOFO LETTER I
  {0xA3B9,0x3128}, // BOPOMOFO LETTER U
  {0xA3BA,0x3129}, // BOPOMOFO LETTER IU
  {0xA3BB,0x02d9}, // DOT ABOVE (Mandarin Chinese light tone)
  {0xA3BC,0x02c9}, // MODIFIER LETTER MACRON (Mandarin Chinese first tone)
  {0xA3BD,0x02ca}, // MODIFIER LETTER ACUTE ACCENT (Mandarin Chinese second tone)
  {0xA3BE,0x02c7}, // CARON (Mandarin Chinese third tone)
  {0xA3BF,0x02cb}, // MODIFIER LETTER GRAVE ACCENT (Mandarin Chinese fourth tone)
  {0xA3E1,0x20ac}, // EURO SIGN [MOD20210517] Old versions of Windows code page 950 addition
  {0xA440,0x4e00}, // <CJK>
  {0xA441,0x4e59}, // <CJK>
  {0xA442,0x4e01}, // <CJK>
  {0xA443,0x4e03}, // <CJK>
  {0xA444,0x4e43}, // <CJK>
  {0xA445,0x4e5d}, // <CJK>
  {0xA446,0x4e86}, // <CJK>
  {0xA447,0x4e8c}, // <CJK>
  {0xA448,0x4eba}, // <CJK>
  {0xA449,0x513f}, // <CJK>
  {0xA44A,0x5165}, // <CJK>
  {0xA44B,0x516b}, // <CJK>
  {0xA44C,0x51e0}, // <CJK>
  {0xA44D,0x5200}, // <CJK>
  {0xA44E,0x5201}, // <CJK>
  {0xA44F,0x529b}, // <CJK>
  {0xA450,0x5315}, // <CJK>
  {0xA451,0x5341}, // <CJK>
  {0xA452,0x535c}, // <CJK>
  {0xA453,0x53c8}, // <CJK>
  {0xA454,0x4e09}, // <CJK>
  {0xA455,0x4e0b}, // <CJK>
  {0xA456,0x4e08}, // <CJK>
  {0xA457,0x4e0a}, // <CJK>
  {0xA458,0x4e2b}, // <CJK>
  {0xA459,0x4e38}, // <CJK>
  {0xA45A,0x51e1}, // <CJK>
  {0xA45B,0x4e45}, // <CJK>
  {0xA45C,0x4e48}, // <CJK>
  {0xA45D,0x4e5f}, // <CJK>
  {0xA45E,0x4e5e}, // <CJK>
  {0xA45F,0x4e8e}, // <CJK>
  {0xA460,0x4ea1}, // <CJK>
  {0xA461,0x5140}, // <CJK>
  {0xA462,0x5203}, // <CJK>
  {0xA463,0x52fa}, // <CJK>
  {0xA464,0x5343}, // <CJK>
  {0xA465,0x53c9}, // <CJK>
  {0xA466,0x53e3}, // <CJK>
  {0xA467,0x571f}, // <CJK>
  {0xA468,0x58eb}, // <CJK>
  {0xA469,0x5915}, // <CJK>
  {0xA46A,0x5927}, // <CJK>
  {0xA46B,0x5973}, // <CJK>
  {0xA46C,0x5b50}, // <CJK>
  {0xA46D,0x5b51}, // <CJK>
  {0xA46E,0x5b53}, // <CJK>
  {0xA46F,0x5bf8}, // <CJK>
  {0xA470,0x5c0f}, // <CJK>
  {0xA471,0x5c22}, // <CJK>
  {0xA472,0x5c38}, // <CJK>
  {0xA473,0x5c71}, // <CJK>
  {0xA474,0x5ddd}, // <CJK>
  {0xA475,0x5de5}, // <CJK>
  {0xA476,0x5df1}, // <CJK>
  {0xA477,0x5df2}, // <CJK>
  {0xA478,0x5df3}, // <CJK>
  {0xA479,0x5dfe}, // <CJK>
  {0xA47A,0x5e72}, // <CJK>
  {0xA47B,0x5efe}, // <CJK>
  {0xA47C,0x5f0b}, // <CJK>
  {0xA47D,0x5f13}, // <CJK>
  {0xA47E,0x624d}, // <CJK>
  {0xA4A1,0x4e11}, // <CJK>
  {0xA4A2,0x4e10}, // <CJK>
  {0xA4A3,0x4e0d}, // <CJK>
  {0xA4A4,0x4e2d}, // <CJK>
  {0xA4A5,0x4e30}, // <CJK>
  {0xA4A6,0x4e39}, // <CJK>
  {0xA4A7,0x4e4b}, // <CJK>
  {0xA4A8,0x5c39}, // <CJK>
  {0xA4A9,0x4e88}, // <CJK>
  {0xA4AA,0x4e91}, // <CJK>
  {0xA4AB,0x4e95}, // <CJK>
  {0xA4AC,0x4e92}, // <CJK>
  {0xA4AD,0x4e94}, // <CJK>
  {0xA4AE,0x4ea2}, // <CJK>
  {0xA4AF,0x4ec1}, // <CJK>
  {0xA4B0,0x4ec0}, // <CJK>
  {0xA4B1,0x4ec3}, // <CJK>
  {0xA4B2,0x4ec6}, // <CJK>
  {0xA4B3,0x4ec7}, // <CJK>
  {0xA4B4,0x4ecd}, // <CJK>
  {0xA4B5,0x4eca}, // <CJK>
  {0xA4B6,0x4ecb}, // <CJK>
  {0xA4B7,0x4ec4}, // <CJK>
  {0xA4B8,0x5143}, // <CJK>
  {0xA4B9,0x5141}, // <CJK>
  {0xA4BA,0x5167}, // <CJK>
  {0xA4BB,0x516d}, // <CJK>
  {0xA4BC,0x516e}, // <CJK>
  {0xA4BD,0x516c}, // <CJK>
  {0xA4BE,0x5197}, // <CJK>
  {0xA4BF,0x51f6}, // <CJK>
  {0xA4C0,0x5206}, // <CJK>
  {0xA4C1,0x5207}, // <CJK>
  {0xA4C2,0x5208}, // <CJK>
  {0xA4C3,0x52fb}, // <CJK>
  {0xA4C4,0x52fe}, // <CJK>
  {0xA4C5,0x52ff}, // <CJK>
  {0xA4C6,0x5316}, // <CJK>
  {0xA4C7,0x5339}, // <CJK>
  {0xA4C8,0x5348}, // <CJK>
  {0xA4C9,0x5347}, // <CJK>
  {0xA4CA,0x5345}, // <CJK>
  {0xA4CB,0x535e}, // <CJK>
  {0xA4CC,0x5384}, // <CJK>
  {0xA4CD,0x53cb}, // <CJK>
  {0xA4CE,0x53ca}, // <CJK>
  {0xA4CF,0x53cd}, // <CJK>
  {0xA4D0,0x58ec}, // <CJK>
  {0xA4D1,0x5929}, // <CJK>
  {0xA4D2,0x592b}, // <CJK>
  {0xA4D3,0x592a}, // <CJK>
  {0xA4D4,0x592d}, // <CJK>
  {0xA4D5,0x5b54}, // <CJK>
  {0xA4D6,0x5c11}, // <CJK>
  {0xA4D7,0x5c24}, // <CJK>
  {0xA4D8,0x5c3a}, // <CJK>
  {0xA4D9,0x5c6f}, // <CJK>
  {0xA4DA,0x5df4}, // <CJK>
  {0xA4DB,0x5e7b}, // <CJK>
  {0xA4DC,0x5eff}, // <CJK>
  {0xA4DD,0x5f14}, // <CJK>
  {0xA4DE,0x5f15}, // <CJK>
  {0xA4DF,0x5fc3}, // <CJK>
  {0xA4E0,0x6208}, // <CJK>
  {0xA4E1,0x6236}, // <CJK>
  {0xA4E2,0x624b}, // <CJK>
  {0xA4E3,0x624e}, // <CJK>
  {0xA4E4,0x652f}, // <CJK>
  {0xA4E5,0x6587}, // <CJK>
  {0xA4E6,0x6597}, // <CJK>
  {0xA4E7,0x65a4}, // <CJK>
  {0xA4E8,0x65b9}, // <CJK>
  {0xA4E9,0x65e5}, // <CJK>
  {0xA4EA,0x66f0}, // <CJK>
  {0xA4EB,0x6708}, // <CJK>
  {0xA4EC,0x6728}, // <CJK>
  {0xA4ED,0x6b20}, // <CJK>
  {0xA4EE,0x6b62}, // <CJK>
  {0xA4EF,0x6b79}, // <CJK>
  {0xA4F0,0x6bcb}, // <CJK>
  {0xA4F1,0x6bd4}, // <CJK>
  {0xA4F2,0x6bdb}, // <CJK>
  {0xA4F3,0x6c0f}, // <CJK>
  {0xA4F4,0x6c34}, // <CJK>
  {0xA4F5,0x706b}, // <CJK>
  {0xA4F6,0x722a}, // <CJK>
  {0xA4F7,0x7236}, // <CJK>
  {0xA4F8,0x723b}, // <CJK>
  {0xA4F9,0x7247}, // <CJK>
  {0xA4FA,0x7259}, // <CJK>
  {0xA4FB,0x725b}, // <CJK>
  {0xA4FC,0x72ac}, // <CJK>
  {0xA4FD,0x738b}, // <CJK>
  {0xA4FE,0x4e19}, // <CJK>
  {0xA540,0x4e16}, // <CJK>
  {0xA541,0x4e15}, // <CJK>
  {0xA542,0x4e14}, // <CJK>
  {0xA543,0x4e18}, // <CJK>
  {0xA544,0x4e3b}, // <CJK>
  {0xA545,0x4e4d}, // <CJK>
  {0xA546,0x4e4f}, // <CJK>
  {0xA547,0x4e4e}, // <CJK>
  {0xA548,0x4ee5}, // <CJK>
  {0xA549,0x4ed8}, // <CJK>
  {0xA54A,0x4ed4}, // <CJK>
  {0xA54B,0x4ed5}, // <CJK>
  {0xA54C,0x4ed6}, // <CJK>
  {0xA54D,0x4ed7}, // <CJK>
  {0xA54E,0x4ee3}, // <CJK>
  {0xA54F,0x4ee4}, // <CJK>
  {0xA550,0x4ed9}, // <CJK>
  {0xA551,0x4ede}, // <CJK>
  {0xA552,0x5145}, // <CJK>
  {0xA553,0x5144}, // <CJK>
  {0xA554,0x5189}, // <CJK>
  {0xA555,0x518a}, // <CJK>
  {0xA556,0x51ac}, // <CJK>
  {0xA557,0x51f9}, // <CJK>
  {0xA558,0x51fa}, // <CJK>
  {0xA559,0x51f8}, // <CJK>
  {0xA55A,0x520a}, // <CJK>
  {0xA55B,0x52a0}, // <CJK>
  {0xA55C,0x529f}, // <CJK>
  {0xA55D,0x5305}, // <CJK>
  {0xA55E,0x5306}, // <CJK>
  {0xA55F,0x5317}, // <CJK>
  {0xA560,0x531d}, // <CJK>
  {0xA561,0x4edf}, // <CJK>
  {0xA562,0x534a}, // <CJK>
  {0xA563,0x5349}, // <CJK>
  {0xA564,0x5361}, // <CJK>
  {0xA565,0x5360}, // <CJK>
  {0xA566,0x536f}, // <CJK>
  {0xA567,0x536e}, // <CJK>
  {0xA568,0x53bb}, // <CJK>
  {0xA569,0x53ef}, // <CJK>
  {0xA56A,0x53e4}, // <CJK>
  {0xA56B,0x53f3}, // <CJK>
  {0xA56C,0x53ec}, // <CJK>
  {0xA56D,0x53ee}, // <CJK>
  {0xA56E,0x53e9}, // <CJK>
  {0xA56F,0x53e8}, // <CJK>
  {0xA570,0x53fc}, // <CJK>
  {0xA571,0x53f8}, // <CJK>
  {0xA572,0x53f5}, // <CJK>
  {0xA573,0x53eb}, // <CJK>
  {0xA574,0x53e6}, // <CJK>
  {0xA575,0x53ea}, // <CJK>
  {0xA576,0x53f2}, // <CJK>
  {0xA577,0x53f1}, // <CJK>
  {0xA578,0x53f0}, // <CJK>
  {0xA579,0x53e5}, // <CJK>
  {0xA57A,0x53ed}, // <CJK>
  {0xA57B,0x53fb}, // <CJK>
  {0xA57C,0x56db}, // <CJK>
  {0xA57D,0x56da}, // <CJK>
  {0xA57E,0x5916}, // <CJK>
  {0xA5A1,0x592e}, // <CJK>
  {0xA5A2,0x5931}, // <CJK>
  {0xA5A3,0x5974}, // <CJK>
  {0xA5A4,0x5976}, // <CJK>
  {0xA5A5,0x5b55}, // <CJK>
  {0xA5A6,0x5b83}, // <CJK>
  {0xA5A7,0x5c3c}, // <CJK>
  {0xA5A8,0x5de8}, // <CJK>
  {0xA5A9,0x5de7}, // <CJK>
  {0xA5AA,0x5de6}, // <CJK>
  {0xA5AB,0x5e02}, // <CJK>
  {0xA5AC,0x5e03}, // <CJK>
  {0xA5AD,0x5e73}, // <CJK>
  {0xA5AE,0x5e7c}, // <CJK>
  {0xA5AF,0x5f01}, // <CJK>
  {0xA5B0,0x5f18}, // <CJK>
  {0xA5B1,0x5f17}, // <CJK>
  {0xA5B2,0x5fc5}, // <CJK>
  {0xA5B3,0x620a}, // <CJK>
  {0xA5B4,0x6253}, // <CJK>
  {0xA5B5,0x6254}, // <CJK>
  {0xA5B6,0x6252}, // <CJK>
  {0xA5B7,0x6251}, // <CJK>
  {0xA5B8,0x65a5}, // <CJK>
  {0xA5B9,0x65e6}, // <CJK>
  {0xA5BA,0x672e}, // <CJK>
  {0xA5BB,0x672c}, // <CJK>
  {0xA5BC,0x672a}, // <CJK>
  {0xA5BD,0x672b}, // <CJK>
  {0xA5BE,0x672d}, // <CJK>
  {0xA5BF,0x6b63}, // <CJK>
  {0xA5C0,0x6bcd}, // <CJK>
  {0xA5C1,0x6c11}, // <CJK>
  {0xA5C2,0x6c10}, // <CJK>
  {0xA5C3,0x6c38}, // <CJK>
  {0xA5C4,0x6c41}, // <CJK>
  {0xA5C5,0x6c40}, // <CJK>
  {0xA5C6,0x6c3e}, // <CJK>
  {0xA5C7,0x72af}, // <CJK>
  {0xA5C8,0x7384}, // <CJK>
  {0xA5C9,0x7389}, // <CJK>
  {0xA5CA,0x74dc}, // <CJK>
  {0xA5CB,0x74e6}, // <CJK>
  {0xA5CC,0x7518}, // <CJK>
  {0xA5CD,0x751f}, // <CJK>
  {0xA5CE,0x7528}, // <CJK>
  {0xA5CF,0x7529}, // <CJK>
  {0xA5D0,0x7530}, // <CJK>
  {0xA5D1,0x7531}, // <CJK>
  {0xA5D2,0x7532}, // <CJK>
  {0xA5D3,0x7533}, // <CJK>
  {0xA5D4,0x758b}, // <CJK>
  {0xA5D5,0x767d}, // <CJK>
  {0xA5D6,0x76ae}, // <CJK>
  {0xA5D7,0x76bf}, // <CJK>
  {0xA5D8,0x76ee}, // <CJK>
  {0xA5D9,0x77db}, // <CJK>
  {0xA5DA,0x77e2}, // <CJK>
  {0xA5DB,0x77f3}, // <CJK>
  {0xA5DC,0x793a}, // <CJK>
  {0xA5DD,0x79be}, // <CJK>
  {0xA5DE,0x7a74}, // <CJK>
  {0xA5DF,0x7acb}, // <CJK>
  {0xA5E0,0x4e1e}, // <CJK>
  {0xA5E1,0x4e1f}, // <CJK>
  {0xA5E2,0x4e52}, // <CJK>
  {0xA5E3,0x4e53}, // <CJK>
  {0xA5E4,0x4e69}, // <CJK>
  {0xA5E5,0x4e99}, // <CJK>
  {0xA5E6,0x4ea4}, // <CJK>
  {0xA5E7,0x4ea6}, // <CJK>
  {0xA5E8,0x4ea5}, // <CJK>
  {0xA5E9,0x4eff}, // <CJK>
  {0xA5EA,0x4f09}, // <CJK>
  {0xA5EB,0x4f19}, // <CJK>
  {0xA5EC,0x4f0a}, // <CJK>
  {0xA5ED,0x4f15}, // <CJK>
  {0xA5EE,0x4f0d}, // <CJK>
  {0xA5EF,0x4f10}, // <CJK>
  {0xA5F0,0x4f11}, // <CJK>
  {0xA5F1,0x4f0f}, // <CJK>
  {0xA5F2,0x4ef2}, // <CJK>
  {0xA5F3,0x4ef6}, // <CJK>
  {0xA5F4,0x4efb}, // <CJK>
  {0xA5F5,0x4ef0}, // <CJK>
  {0xA5F6,0x4ef3}, // <CJK>
  {0xA5F7,0x4efd}, // <CJK>
  {0xA5F8,0x4f01}, // <CJK>
  {0xA5F9,0x4f0b}, // <CJK>
  {0xA5FA,0x5149}, // <CJK>
  {0xA5FB,0x5147}, // <CJK>
  {0xA5FC,0x5146}, // <CJK>
  {0xA5FD,0x5148}, // <CJK>
  {0xA5FE,0x5168}, // <CJK>
  {0xA640,0x5171}, // <CJK>
  {0xA641,0x518d}, // <CJK>
  {0xA642,0x51b0}, // <CJK>
  {0xA643,0x5217}, // <CJK>
  {0xA644,0x5211}, // <CJK>
  {0xA645,0x5212}, // <CJK>
  {0xA646,0x520e}, // <CJK>
  {0xA647,0x5216}, // <CJK>
  {0xA648,0x52a3}, // <CJK>
  {0xA649,0x5308}, // <CJK>
  {0xA64A,0x5321}, // <CJK>
  {0xA64B,0x5320}, // <CJK>
  {0xA64C,0x5370}, // <CJK>
  {0xA64D,0x5371}, // <CJK>
  {0xA64E,0x5409}, // <CJK>
  {0xA64F,0x540f}, // <CJK>
  {0xA650,0x540c}, // <CJK>
  {0xA651,0x540a}, // <CJK>
  {0xA652,0x5410}, // <CJK>
  {0xA653,0x5401}, // <CJK>
  {0xA654,0x540b}, // <CJK>
  {0xA655,0x5404}, // <CJK>
  {0xA656,0x5411}, // <CJK>
  {0xA657,0x540d}, // <CJK>
  {0xA658,0x5408}, // <CJK>
  {0xA659,0x5403}, // <CJK>
  {0xA65A,0x540e}, // <CJK>
  {0xA65B,0x5406}, // <CJK>
  {0xA65C,0x5412}, // <CJK>
  {0xA65D,0x56e0}, // <CJK>
  {0xA65E,0x56de}, // <CJK>
  {0xA65F,0x56dd}, // <CJK>
  {0xA660,0x5733}, // <CJK>
  {0xA661,0x5730}, // <CJK>
  {0xA662,0x5728}, // <CJK>
  {0xA663,0x572d}, // <CJK>
  {0xA664,0x572c}, // <CJK>
  {0xA665,0x572f}, // <CJK>
  {0xA666,0x5729}, // <CJK>
  {0xA667,0x5919}, // <CJK>
  {0xA668,0x591a}, // <CJK>
  {0xA669,0x5937}, // <CJK>
  {0xA66A,0x5938}, // <CJK>
  {0xA66B,0x5984}, // <CJK>
  {0xA66C,0x5978}, // <CJK>
  {0xA66D,0x5983}, // <CJK>
  {0xA66E,0x597d}, // <CJK>
  {0xA66F,0x5979}, // <CJK>
  {0xA670,0x5982}, // <CJK>
  {0xA671,0x5981}, // <CJK>
  {0xA672,0x5b57}, // <CJK>
  {0xA673,0x5b58}, // <CJK>
  {0xA674,0x5b87}, // <CJK>
  {0xA675,0x5b88}, // <CJK>
  {0xA676,0x5b85}, // <CJK>
  {0xA677,0x5b89}, // <CJK>
  {0xA678,0x5bfa}, // <CJK>
  {0xA679,0x5c16}, // <CJK>
  {0xA67A,0x5c79}, // <CJK>
  {0xA67B,0x5dde}, // <CJK>
  {0xA67C,0x5e06}, // <CJK>
  {0xA67D,0x5e76}, // <CJK>
  {0xA67E,0x5e74}, // <CJK>
  {0xA6A1,0x5f0f}, // <CJK>
  {0xA6A2,0x5f1b}, // <CJK>
  {0xA6A3,0x5fd9}, // <CJK>
  {0xA6A4,0x5fd6}, // <CJK>
  {0xA6A5,0x620e}, // <CJK>
  {0xA6A6,0x620c}, // <CJK>
  {0xA6A7,0x620d}, // <CJK>
  {0xA6A8,0x6210}, // <CJK>
  {0xA6A9,0x6263}, // <CJK>
  {0xA6AA,0x625b}, // <CJK>
  {0xA6AB,0x6258}, // <CJK>
  {0xA6AC,0x6536}, // <CJK>
  {0xA6AD,0x65e9}, // <CJK>
  {0xA6AE,0x65e8}, // <CJK>
  {0xA6AF,0x65ec}, // <CJK>
  {0xA6B0,0x65ed}, // <CJK>
  {0xA6B1,0x66f2}, // <CJK>
  {0xA6B2,0x66f3}, // <CJK>
  {0xA6B3,0x6709}, // <CJK>
  {0xA6B4,0x673d}, // <CJK>
  {0xA6B5,0x6734}, // <CJK>
  {0xA6B6,0x6731}, // <CJK>
  {0xA6B7,0x6735}, // <CJK>
  {0xA6B8,0x6b21}, // <CJK>
  {0xA6B9,0x6b64}, // <CJK>
  {0xA6BA,0x6b7b}, // <CJK>
  {0xA6BB,0x6c16}, // <CJK>
  {0xA6BC,0x6c5d}, // <CJK>
  {0xA6BD,0x6c57}, // <CJK>
  {0xA6BE,0x6c59}, // <CJK>
  {0xA6BF,0x6c5f}, // <CJK>
  {0xA6C0,0x6c60}, // <CJK>
  {0xA6C1,0x6c50}, // <CJK>
  {0xA6C2,0x6c55}, // <CJK>
  {0xA6C3,0x6c61}, // <CJK>
  {0xA6C4,0x6c5b}, // <CJK>
  {0xA6C5,0x6c4d}, // <CJK>
  {0xA6C6,0x6c4e}, // <CJK>
  {0xA6C7,0x7070}, // <CJK>
  {0xA6C8,0x725f}, // <CJK>
  {0xA6C9,0x725d}, // <CJK>
  {0xA6CA,0x767e}, // <CJK>
  {0xA6CB,0x7af9}, // <CJK>
  {0xA6CC,0x7c73}, // <CJK>
  {0xA6CD,0x7cf8}, // <CJK>
  {0xA6CE,0x7f36}, // <CJK>
  {0xA6CF,0x7f8a}, // <CJK>
  {0xA6D0,0x7fbd}, // <CJK>
  {0xA6D1,0x8001}, // <CJK>
  {0xA6D2,0x8003}, // <CJK>
  {0xA6D3,0x800c}, // <CJK>
  {0xA6D4,0x8012}, // <CJK>
  {0xA6D5,0x8033}, // <CJK>
  {0xA6D6,0x807f}, // <CJK>
  {0xA6D7,0x8089}, // <CJK>
  {0xA6D8,0x808b}, // <CJK>
  {0xA6D9,0x808c}, // <CJK>
  {0xA6DA,0x81e3}, // <CJK>
  {0xA6DB,0x81ea}, // <CJK>
  {0xA6DC,0x81f3}, // <CJK>
  {0xA6DD,0x81fc}, // <CJK>
  {0xA6DE,0x820c}, // <CJK>
  {0xA6DF,0x821b}, // <CJK>
  {0xA6E0,0x821f}, // <CJK>
  {0xA6E1,0x826e}, // <CJK>
  {0xA6E2,0x8272}, // <CJK>
  {0xA6E3,0x827e}, // <CJK>
  {0xA6E4,0x866b}, // <CJK>
  {0xA6E5,0x8840}, // <CJK>
  {0xA6E6,0x884c}, // <CJK>
  {0xA6E7,0x8863}, // <CJK>
  {0xA6E8,0x897f}, // <CJK>
  {0xA6E9,0x9621}, // <CJK>
  {0xA6EA,0x4e32}, // <CJK>
  {0xA6EB,0x4ea8}, // <CJK>
  {0xA6EC,0x4f4d}, // <CJK>
  {0xA6ED,0x4f4f}, // <CJK>
  {0xA6EE,0x4f47}, // <CJK>
  {0xA6EF,0x4f57}, // <CJK>
  {0xA6F0,0x4f5e}, // <CJK>
  {0xA6F1,0x4f34}, // <CJK>
  {0xA6F2,0x4f5b}, // <CJK>
  {0xA6F3,0x4f55}, // <CJK>
  {0xA6F4,0x4f30}, // <CJK>
  {0xA6F5,0x4f50}, // <CJK>
  {0xA6F6,0x4f51}, // <CJK>
  {0xA6F7,0x4f3d}, // <CJK>
  {0xA6F8,0x4f3a}, // <CJK>
  {0xA6F9,0x4f38}, // <CJK>
  {0xA6FA,0x4f43}, // <CJK>
  {0xA6FB,0x4f54}, // <CJK>
  {0xA6FC,0x4f3c}, // <CJK>
  {0xA6FD,0x4f46}, // <CJK>
  {0xA6FE,0x4f63}, // <CJK>
  {0xA740,0x4f5c}, // <CJK>
  {0xA741,0x4f60}, // <CJK>
  {0xA742,0x4f2f}, // <CJK>
  {0xA743,0x4f4e}, // <CJK>
  {0xA744,0x4f36}, // <CJK>
  {0xA745,0x4f59}, // <CJK>
  {0xA746,0x4f5d}, // <CJK>
  {0xA747,0x4f48}, // <CJK>
  {0xA748,0x4f5a}, // <CJK>
  {0xA749,0x514c}, // <CJK>
  {0xA74A,0x514b}, // <CJK>
  {0xA74B,0x514d}, // <CJK>
  {0xA74C,0x5175}, // <CJK>
  {0xA74D,0x51b6}, // <CJK>
  {0xA74E,0x51b7}, // <CJK>
  {0xA74F,0x5225}, // <CJK>
  {0xA750,0x5224}, // <CJK>
  {0xA751,0x5229}, // <CJK>
  {0xA752,0x522a}, // <CJK>
  {0xA753,0x5228}, // <CJK>
  {0xA754,0x52ab}, // <CJK>
  {0xA755,0x52a9}, // <CJK>
  {0xA756,0x52aa}, // <CJK>
  {0xA757,0x52ac}, // <CJK>
  {0xA758,0x5323}, // <CJK>
  {0xA759,0x5373}, // <CJK>
  {0xA75A,0x5375}, // <CJK>
  {0xA75B,0x541d}, // <CJK>
  {0xA75C,0x542d}, // <CJK>
  {0xA75D,0x541e}, // <CJK>
  {0xA75E,0x543e}, // <CJK>
  {0xA75F,0x5426}, // <CJK>
  {0xA760,0x544e}, // <CJK>
  {0xA761,0x5427}, // <CJK>
  {0xA762,0x5446}, // <CJK>
  {0xA763,0x5443}, // <CJK>
  {0xA764,0x5433}, // <CJK>
  {0xA765,0x5448}, // <CJK>
  {0xA766,0x5442}, // <CJK>
  {0xA767,0x541b}, // <CJK>
  {0xA768,0x5429}, // <CJK>
  {0xA769,0x544a}, // <CJK>
  {0xA76A,0x5439}, // <CJK>
  {0xA76B,0x543b}, // <CJK>
  {0xA76C,0x5438}, // <CJK>
  {0xA76D,0x542e}, // <CJK>
  {0xA76E,0x5435}, // <CJK>
  {0xA76F,0x5436}, // <CJK>
  {0xA770,0x5420}, // <CJK>
  {0xA771,0x543c}, // <CJK>
  {0xA772,0x5440}, // <CJK>
  {0xA773,0x5431}, // <CJK>
  {0xA774,0x542b}, // <CJK>
  {0xA775,0x541f}, // <CJK>
  {0xA776,0x542c}, // <CJK>
  {0xA777,0x56ea}, // <CJK>
  {0xA778,0x56f0}, // <CJK>
  {0xA779,0x56e4}, // <CJK>
  {0xA77A,0x56eb}, // <CJK>
  {0xA77B,0x574a}, // <CJK>
  {0xA77C,0x5751}, // <CJK>
  {0xA77D,0x5740}, // <CJK>
  {0xA77E,0x574d}, // <CJK>
  {0xA7A1,0x5747}, // <CJK>
  {0xA7A2,0x574e}, // <CJK>
  {0xA7A3,0x573e}, // <CJK>
  {0xA7A4,0x5750}, // <CJK>
  {0xA7A5,0x574f}, // <CJK>
  {0xA7A6,0x573b}, // <CJK>
  {0xA7A7,0x58ef}, // <CJK>
  {0xA7A8,0x593e}, // <CJK>
  {0xA7A9,0x599d}, // <CJK>
  {0xA7AA,0x5992}, // <CJK>
  {0xA7AB,0x59a8}, // <CJK>
  {0xA7AC,0x599e}, // <CJK>
  {0xA7AD,0x59a3}, // <CJK>
  {0xA7AE,0x5999}, // <CJK>
  {0xA7AF,0x5996}, // <CJK>
  {0xA7B0,0x598d}, // <CJK>
  {0xA7B1,0x59a4}, // <CJK>
  {0xA7B2,0x5993}, // <CJK>
  {0xA7B3,0x598a}, // <CJK>
  {0xA7B4,0x59a5}, // <CJK>
  {0xA7B5,0x5b5d}, // <CJK>
  {0xA7B6,0x5b5c}, // <CJK>
  {0xA7B7,0x5b5a}, // <CJK>
  {0xA7B8,0x5b5b}, // <CJK>
  {0xA7B9,0x5b8c}, // <CJK>
  {0xA7BA,0x5b8b}, // <CJK>
  {0xA7BB,0x5b8f}, // <CJK>
  {0xA7BC,0x5c2c}, // <CJK>
  {0xA7BD,0x5c40}, // <CJK>
  {0xA7BE,0x5c41}, // <CJK>
  {0xA7BF,0x5c3f}, // <CJK>
  {0xA7C0,0x5c3e}, // <CJK>
  {0xA7C1,0x5c90}, // <CJK>
  {0xA7C2,0x5c91}, // <CJK>
  {0xA7C3,0x5c94}, // <CJK>
  {0xA7C4,0x5c8c}, // <CJK>
  {0xA7C5,0x5deb}, // <CJK>
  {0xA7C6,0x5e0c}, // <CJK>
  {0xA7C7,0x5e8f}, // <CJK>
  {0xA7C8,0x5e87}, // <CJK>
  {0xA7C9,0x5e8a}, // <CJK>
  {0xA7CA,0x5ef7}, // <CJK>
  {0xA7CB,0x5f04}, // <CJK>
  {0xA7CC,0x5f1f}, // <CJK>
  {0xA7CD,0x5f64}, // <CJK>
  {0xA7CE,0x5f62}, // <CJK>
  {0xA7CF,0x5f77}, // <CJK>
  {0xA7D0,0x5f79}, // <CJK>
  {0xA7D1,0x5fd8}, // <CJK>
  {0xA7D2,0x5fcc}, // <CJK>
  {0xA7D3,0x5fd7}, // <CJK>
  {0xA7D4,0x5fcd}, // <CJK>
  {0xA7D5,0x5ff1}, // <CJK>
  {0xA7D6,0x5feb}, // <CJK>
  {0xA7D7,0x5ff8}, // <CJK>
  {0xA7D8,0x5fea}, // <CJK>
  {0xA7D9,0x6212}, // <CJK>
  {0xA7DA,0x6211}, // <CJK>
  {0xA7DB,0x6284}, // <CJK>
  {0xA7DC,0x6297}, // <CJK>
  {0xA7DD,0x6296}, // <CJK>
  {0xA7DE,0x6280}, // <CJK>
  {0xA7DF,0x6276}, // <CJK>
  {0xA7E0,0x6289}, // <CJK>
  {0xA7E1,0x626d}, // <CJK>
  {0xA7E2,0x628a}, // <CJK>
  {0xA7E3,0x627c}, // <CJK>
  {0xA7E4,0x627e}, // <CJK>
  {0xA7E5,0x6279}, // <CJK>
  {0xA7E6,0x6273}, // <CJK>
  {0xA7E7,0x6292}, // <CJK>
  {0xA7E8,0x626f}, // <CJK>
  {0xA7E9,0x6298}, // <CJK>
  {0xA7EA,0x626e}, // <CJK>
  {0xA7EB,0x6295}, // <CJK>
  {0xA7EC,0x6293}, // <CJK>
  {0xA7ED,0x6291}, // <CJK>
  {0xA7EE,0x6286}, // <CJK>
  {0xA7EF,0x6539}, // <CJK>
  {0xA7F0,0x653b}, // <CJK>
  {0xA7F1,0x6538}, // <CJK>
  {0xA7F2,0x65f1}, // <CJK>
  {0xA7F3,0x66f4}, // <CJK>
  {0xA7F4,0x675f}, // <CJK>
  {0xA7F5,0x674e}, // <CJK>
  {0xA7F6,0x674f}, // <CJK>
  {0xA7F7,0x6750}, // <CJK>
  {0xA7F8,0x6751}, // <CJK>
  {0xA7F9,0x675c}, // <CJK>
  {0xA7FA,0x6756}, // <CJK>
  {0xA7FB,0x675e}, // <CJK>
  {0xA7FC,0x6749}, // <CJK>
  {0xA7FD,0x6746}, // <CJK>
  {0xA7FE,0x6760}, // <CJK>
  {0xA840,0x6753}, // <CJK>
  {0xA841,0x6757}, // <CJK>
  {0xA842,0x6b65}, // <CJK>
  {0xA843,0x6bcf}, // <CJK>
  {0xA844,0x6c42}, // <CJK>
  {0xA845,0x6c5e}, // <CJK>
  {0xA846,0x6c99}, // <CJK>
  {0xA847,0x6c81}, // <CJK>
  {0xA848,0x6c88}, // <CJK>
  {0xA849,0x6c89}, // <CJK>
  {0xA84A,0x6c85}, // <CJK>
  {0xA84B,0x6c9b}, // <CJK>
  {0xA84C,0x6c6a}, // <CJK>
  {0xA84D,0x6c7a}, // <CJK>
  {0xA84E,0x6c90}, // <CJK>
  {0xA84F,0x6c70}, // <CJK>
  {0xA850,0x6c8c}, // <CJK>
  {0xA851,0x6c68}, // <CJK>
  {0xA852,0x6c96}, // <CJK>
  {0xA853,0x6c92}, // <CJK>
  {0xA854,0x6c7d}, // <CJK>
  {0xA855,0x6c83}, // <CJK>
  {0xA856,0x6c72}, // <CJK>
  {0xA857,0x6c7e}, // <CJK>
  {0xA858,0x6c74}, // <CJK>
  {0xA859,0x6c86}, // <CJK>
  {0xA85A,0x6c76}, // <CJK>
  {0xA85B,0x6c8d}, // <CJK>
  {0xA85C,0x6c94}, // <CJK>
  {0xA85D,0x6c98}, // <CJK>
  {0xA85E,0x6c82}, // <CJK>
  {0xA85F,0x7076}, // <CJK>
  {0xA860,0x707c}, // <CJK>
  {0xA861,0x707d}, // <CJK>
  {0xA862,0x7078}, // <CJK>
  {0xA863,0x7262}, // <CJK>
  {0xA864,0x7261}, // <CJK>
  {0xA865,0x7260}, // <CJK>
  {0xA866,0x72c4}, // <CJK>
  {0xA867,0x72c2}, // <CJK>
  {0xA868,0x7396}, // <CJK>
  {0xA869,0x752c}, // <CJK>
  {0xA86A,0x752b}, // <CJK>
  {0xA86B,0x7537}, // <CJK>
  {0xA86C,0x7538}, // <CJK>
  {0xA86D,0x7682}, // <CJK>
  {0xA86E,0x76ef}, // <CJK>
  {0xA86F,0x77e3}, // <CJK>
  {0xA870,0x79c1}, // <CJK>
  {0xA871,0x79c0}, // <CJK>
  {0xA872,0x79bf}, // <CJK>
  {0xA873,0x7a76}, // <CJK>
  {0xA874,0x7cfb}, // <CJK>
  {0xA875,0x7f55}, // <CJK>
  {0xA876,0x8096}, // <CJK>
  {0xA877,0x8093}, // <CJK>
  {0xA878,0x809d}, // <CJK>
  {0xA879,0x8098}, // <CJK>
  {0xA87A,0x809b}, // <CJK>
  {0xA87B,0x809a}, // <CJK>
  {0xA87C,0x80b2}, // <CJK>
  {0xA87D,0x826f}, // <CJK>
  {0xA87E,0x8292}, // <CJK>
  {0xA8A1,0x828b}, // <CJK>
  {0xA8A2,0x828d}, // <CJK>
  {0xA8A3,0x898b}, // <CJK>
  {0xA8A4,0x89d2}, // <CJK>
  {0xA8A5,0x8a00}, // <CJK>
  {0xA8A6,0x8c37}, // <CJK>
  {0xA8A7,0x8c46}, // <CJK>
  {0xA8A8,0x8c55}, // <CJK>
  {0xA8A9,0x8c9d}, // <CJK>
  {0xA8AA,0x8d64}, // <CJK>
  {0xA8AB,0x8d70}, // <CJK>
  {0xA8AC,0x8db3}, // <CJK>
  {0xA8AD,0x8eab}, // <CJK>
  {0xA8AE,0x8eca}, // <CJK>
  {0xA8AF,0x8f9b}, // <CJK>
  {0xA8B0,0x8fb0}, // <CJK>
  {0xA8B1,0x8fc2}, // <CJK>
  {0xA8B2,0x8fc6}, // <CJK>
  {0xA8B3,0x8fc5}, // <CJK>
  {0xA8B4,0x8fc4}, // <CJK>
  {0xA8B5,0x5de1}, // <CJK>
  {0xA8B6,0x9091}, // <CJK>
  {0xA8B7,0x90a2}, // <CJK>
  {0xA8B8,0x90aa}, // <CJK>
  {0xA8B9,0x90a6}, // <CJK>
  {0xA8BA,0x90a3}, // <CJK>
  {0xA8BB,0x9149}, // <CJK>
  {0xA8BC,0x91c6}, // <CJK>
  {0xA8BD,0x91cc}, // <CJK>
  {0xA8BE,0x9632}, // <CJK>
  {0xA8BF,0x962e}, // <CJK>
  {0xA8C0,0x9631}, // <CJK>
  {0xA8C1,0x962a}, // <CJK>
  {0xA8C2,0x962c}, // <CJK>
  {0xA8C3,0x4e26}, // <CJK>
  {0xA8C4,0x4e56}, // <CJK>
  {0xA8C5,0x4e73}, // <CJK>
  {0xA8C6,0x4e8b}, // <CJK>
  {0xA8C7,0x4e9b}, // <CJK>
  {0xA8C8,0x4e9e}, // <CJK>
  {0xA8C9,0x4eab}, // <CJK>
  {0xA8CA,0x4eac}, // <CJK>
  {0xA8CB,0x4f6f}, // <CJK>
  {0xA8CC,0x4f9d}, // <CJK>
  {0xA8CD,0x4f8d}, // <CJK>
  {0xA8CE,0x4f73}, // <CJK>
  {0xA8CF,0x4f7f}, // <CJK>
  {0xA8D0,0x4f6c}, // <CJK>
  {0xA8D1,0x4f9b}, // <CJK>
  {0xA8D2,0x4f8b}, // <CJK>
  {0xA8D3,0x4f86}, // <CJK>
  {0xA8D4,0x4f83}, // <CJK>
  {0xA8D5,0x4f70}, // <CJK>
  {0xA8D6,0x4f75}, // <CJK>
  {0xA8D7,0x4f88}, // <CJK>
  {0xA8D8,0x4f69}, // <CJK>
  {0xA8D9,0x4f7b}, // <CJK>
  {0xA8DA,0x4f96}, // <CJK>
  {0xA8DB,0x4f7e}, // <CJK>
  {0xA8DC,0x4f8f}, // <CJK>
  {0xA8DD,0x4f91}, // <CJK>
  {0xA8DE,0x4f7a}, // <CJK>
  {0xA8DF,0x5154}, // <CJK>
  {0xA8E0,0x5152}, // <CJK>
  {0xA8E1,0x5155}, // <CJK>
  {0xA8E2,0x5169}, // <CJK>
  {0xA8E3,0x5177}, // <CJK>
  {0xA8E4,0x5176}, // <CJK>
  {0xA8E5,0x5178}, // <CJK>
  {0xA8E6,0x51bd}, // <CJK>
  {0xA8E7,0x51fd}, // <CJK>
  {0xA8E8,0x523b}, // <CJK>
  {0xA8E9,0x5238}, // <CJK>
  {0xA8EA,0x5237}, // <CJK>
  {0xA8EB,0x523a}, // <CJK>
  {0xA8EC,0x5230}, // <CJK>
  {0xA8ED,0x522e}, // <CJK>
  {0xA8EE,0x5236}, // <CJK>
  {0xA8EF,0x5241}, // <CJK>
  {0xA8F0,0x52be}, // <CJK>
  {0xA8F1,0x52bb}, // <CJK>
  {0xA8F2,0x5352}, // <CJK>
  {0xA8F3,0x5354}, // <CJK>
  {0xA8F4,0x5353}, // <CJK>
  {0xA8F5,0x5351}, // <CJK>
  {0xA8F6,0x5366}, // <CJK>
  {0xA8F7,0x5377}, // <CJK>
  {0xA8F8,0x5378}, // <CJK>
  {0xA8F9,0x5379}, // <CJK>
  {0xA8FA,0x53d6}, // <CJK>
  {0xA8FB,0x53d4}, // <CJK>
  {0xA8FC,0x53d7}, // <CJK>
  {0xA8FD,0x5473}, // <CJK>
  {0xA8FE,0x5475}, // <CJK>
  {0xA940,0x5496}, // <CJK>
  {0xA941,0x5478}, // <CJK>
  {0xA942,0x5495}, // <CJK>
  {0xA943,0x5480}, // <CJK>
  {0xA944,0x547b}, // <CJK>
  {0xA945,0x5477}, // <CJK>
  {0xA946,0x5484}, // <CJK>
  {0xA947,0x5492}, // <CJK>
  {0xA948,0x5486}, // <CJK>
  {0xA949,0x547c}, // <CJK>
  {0xA94A,0x5490}, // <CJK>
  {0xA94B,0x5471}, // <CJK>
  {0xA94C,0x5476}, // <CJK>
  {0xA94D,0x548c}, // <CJK>
  {0xA94E,0x549a}, // <CJK>
  {0xA94F,0x5462}, // <CJK>
  {0xA950,0x5468}, // <CJK>
  {0xA951,0x548b}, // <CJK>
  {0xA952,0x547d}, // <CJK>
  {0xA953,0x548e}, // <CJK>
  {0xA954,0x56fa}, // <CJK>
  {0xA955,0x5783}, // <CJK>
  {0xA956,0x5777}, // <CJK>
  {0xA957,0x576a}, // <CJK>
  {0xA958,0x5769}, // <CJK>
  {0xA959,0x5761}, // <CJK>
  {0xA95A,0x5766}, // <CJK>
  {0xA95B,0x5764}, // <CJK>
  {0xA95C,0x577c}, // <CJK>
  {0xA95D,0x591c}, // <CJK>
  {0xA95E,0x5949}, // <CJK>
  {0xA95F,0x5947}, // <CJK>
  {0xA960,0x5948}, // <CJK>
  {0xA961,0x5944}, // <CJK>
  {0xA962,0x5954}, // <CJK>
  {0xA963,0x59be}, // <CJK>
  {0xA964,0x59bb}, // <CJK>
  {0xA965,0x59d4}, // <CJK>
  {0xA966,0x59b9}, // <CJK>
  {0xA967,0x59ae}, // <CJK>
  {0xA968,0x59d1}, // <CJK>
  {0xA969,0x59c6}, // <CJK>
  {0xA96A,0x59d0}, // <CJK>
  {0xA96B,0x59cd}, // <CJK>
  {0xA96C,0x59cb}, // <CJK>
  {0xA96D,0x59d3}, // <CJK>
  {0xA96E,0x59ca}, // <CJK>
  {0xA96F,0x59af}, // <CJK>
  {0xA970,0x59b3}, // <CJK>
  {0xA971,0x59d2}, // <CJK>
  {0xA972,0x59c5}, // <CJK>
  {0xA973,0x5b5f}, // <CJK>
  {0xA974,0x5b64}, // <CJK>
  {0xA975,0x5b63}, // <CJK>
  {0xA976,0x5b97}, // <CJK>
  {0xA977,0x5b9a}, // <CJK>
  {0xA978,0x5b98}, // <CJK>
  {0xA979,0x5b9c}, // <CJK>
  {0xA97A,0x5b99}, // <CJK>
  {0xA97B,0x5b9b}, // <CJK>
  {0xA97C,0x5c1a}, // <CJK>
  {0xA97D,0x5c48}, // <CJK>
  {0xA97E,0x5c45}, // <CJK>
  {0xA9A1,0x5c46}, // <CJK>
  {0xA9A2,0x5cb7}, // <CJK>
  {0xA9A3,0x5ca1}, // <CJK>
  {0xA9A4,0x5cb8}, // <CJK>
  {0xA9A5,0x5ca9}, // <CJK>
  {0xA9A6,0x5cab}, // <CJK>
  {0xA9A7,0x5cb1}, // <CJK>
  {0xA9A8,0x5cb3}, // <CJK>
  {0xA9A9,0x5e18}, // <CJK>
  {0xA9AA,0x5e1a}, // <CJK>
  {0xA9AB,0x5e16}, // <CJK>
  {0xA9AC,0x5e15}, // <CJK>
  {0xA9AD,0x5e1b}, // <CJK>
  {0xA9AE,0x5e11}, // <CJK>
  {0xA9AF,0x5e78}, // <CJK>
  {0xA9B0,0x5e9a}, // <CJK>
  {0xA9B1,0x5e97}, // <CJK>
  {0xA9B2,0x5e9c}, // <CJK>
  {0xA9B3,0x5e95}, // <CJK>
  {0xA9B4,0x5e96}, // <CJK>
  {0xA9B5,0x5ef6}, // <CJK>
  {0xA9B6,0x5f26}, // <CJK>
  {0xA9B7,0x5f27}, // <CJK>
  {0xA9B8,0x5f29}, // <CJK>
  {0xA9B9,0x5f80}, // <CJK>
  {0xA9BA,0x5f81}, // <CJK>
  {0xA9BB,0x5f7f}, // <CJK>
  {0xA9BC,0x5f7c}, // <CJK>
  {0xA9BD,0x5fdd}, // <CJK>
  {0xA9BE,0x5fe0}, // <CJK>
  {0xA9BF,0x5ffd}, // <CJK>
  {0xA9C0,0x5ff5}, // <CJK>
  {0xA9C1,0x5fff}, // <CJK>
  {0xA9C2,0x600f}, // <CJK>
  {0xA9C3,0x6014}, // <CJK>
  {0xA9C4,0x602f}, // <CJK>
  {0xA9C5,0x6035}, // <CJK>
  {0xA9C6,0x6016}, // <CJK>
  {0xA9C7,0x602a}, // <CJK>
  {0xA9C8,0x6015}, // <CJK>
  {0xA9C9,0x6021}, // <CJK>
  {0xA9CA,0x6027}, // <CJK>
  {0xA9CB,0x6029}, // <CJK>
  {0xA9CC,0x602b}, // <CJK>
  {0xA9CD,0x601b}, // <CJK>
  {0xA9CE,0x6216}, // <CJK>
  {0xA9CF,0x6215}, // <CJK>
  {0xA9D0,0x623f}, // <CJK>
  {0xA9D1,0x623e}, // <CJK>
  {0xA9D2,0x6240}, // <CJK>
  {0xA9D3,0x627f}, // <CJK>
  {0xA9D4,0x62c9}, // <CJK>
  {0xA9D5,0x62cc}, // <CJK>
  {0xA9D6,0x62c4}, // <CJK>
  {0xA9D7,0x62bf}, // <CJK>
  {0xA9D8,0x62c2}, // <CJK>
  {0xA9D9,0x62b9}, // <CJK>
  {0xA9DA,0x62d2}, // <CJK>
  {0xA9DB,0x62db}, // <CJK>
  {0xA9DC,0x62ab}, // <CJK>
  {0xA9DD,0x62d3}, // <CJK>
  {0xA9DE,0x62d4}, // <CJK>
  {0xA9DF,0x62cb}, // <CJK>
  {0xA9E0,0x62c8}, // <CJK>
  {0xA9E1,0x62a8}, // <CJK>
  {0xA9E2,0x62bd}, // <CJK>
  {0xA9E3,0x62bc}, // <CJK>
  {0xA9E4,0x62d0}, // <CJK>
  {0xA9E5,0x62d9}, // <CJK>
  {0xA9E6,0x62c7}, // <CJK>
  {0xA9E7,0x62cd}, // <CJK>
  {0xA9E8,0x62b5}, // <CJK>
  {0xA9E9,0x62da}, // <CJK>
  {0xA9EA,0x62b1}, // <CJK>
  {0xA9EB,0x62d8}, // <CJK>
  {0xA9EC,0x62d6}, // <CJK>
  {0xA9ED,0x62d7}, // <CJK>
  {0xA9EE,0x62c6}, // <CJK>
  {0xA9EF,0x62ac}, // <CJK>
  {0xA9F0,0x62ce}, // <CJK>
  {0xA9F1,0x653e}, // <CJK>
  {0xA9F2,0x65a7}, // <CJK>
  {0xA9F3,0x65bc}, // <CJK>
  {0xA9F4,0x65fa}, // <CJK>
  {0xA9F5,0x6614}, // <CJK>
  {0xA9F6,0x6613}, // <CJK>
  {0xA9F7,0x660c}, // <CJK>
  {0xA9F8,0x6606}, // <CJK>
  {0xA9F9,0x6602}, // <CJK>
  {0xA9FA,0x660e}, // <CJK>
  {0xA9FB,0x6600}, // <CJK>
  {0xA9FC,0x660f}, // <CJK>
  {0xA9FD,0x6615}, // <CJK>
  {0xA9FE,0x660a}, // <CJK>
  {0xAA40,0x6607}, // <CJK>
  {0xAA41,0x670d}, // <CJK>
  {0xAA42,0x670b}, // <CJK>
  {0xAA43,0x676d}, // <CJK>
  {0xAA44,0x678b}, // <CJK>
  {0xAA45,0x6795}, // <CJK>
  {0xAA46,0x6771}, // <CJK>
  {0xAA47,0x679c}, // <CJK>
  {0xAA48,0x6773}, // <CJK>
  {0xAA49,0x6777}, // <CJK>
  {0xAA4A,0x6787}, // <CJK>
  {0xAA4B,0x679d}, // <CJK>
  {0xAA4C,0x6797}, // <CJK>
  {0xAA4D,0x676f}, // <CJK>
  {0xAA4E,0x6770}, // <CJK>
  {0xAA4F,0x677f}, // <CJK>
  {0xAA50,0x6789}, // <CJK>
  {0xAA51,0x677e}, // <CJK>
  {0xAA52,0x6790}, // <CJK>
  {0xAA53,0x6775}, // <CJK>
  {0xAA54,0x679a}, // <CJK>
  {0xAA55,0x6793}, // <CJK>
  {0xAA56,0x677c}, // <CJK>
  {0xAA57,0x676a}, // <CJK>
  {0xAA58,0x6772}, // <CJK>
  {0xAA59,0x6b23}, // <CJK>
  {0xAA5A,0x6b66}, // <CJK>
  {0xAA5B,0x6b67}, // <CJK>
  {0xAA5C,0x6b7f}, // <CJK>
  {0xAA5D,0x6c13}, // <CJK>
  {0xAA5E,0x6c1b}, // <CJK>
  {0xAA5F,0x6ce3}, // <CJK>
  {0xAA60,0x6ce8}, // <CJK>
  {0xAA61,0x6cf3}, // <CJK>
  {0xAA62,0x6cb1}, // <CJK>
  {0xAA63,0x6ccc}, // <CJK>
  {0xAA64,0x6ce5}, // <CJK>
  {0xAA65,0x6cb3}, // <CJK>
  {0xAA66,0x6cbd}, // <CJK>
  {0xAA67,0x6cbe}, // <CJK>
  {0xAA68,0x6cbc}, // <CJK>
  {0xAA69,0x6ce2}, // <CJK>
  {0xAA6A,0x6cab}, // <CJK>
  {0xAA6B,0x6cd5}, // <CJK>
  {0xAA6C,0x6cd3}, // <CJK>
  {0xAA6D,0x6cb8}, // <CJK>
  {0xAA6E,0x6cc4}, // <CJK>
  {0xAA6F,0x6cb9}, // <CJK>
  {0xAA70,0x6cc1}, // <CJK>
  {0xAA71,0x6cae}, // <CJK>
  {0xAA72,0x6cd7}, // <CJK>
  {0xAA73,0x6cc5}, // <CJK>
  {0xAA74,0x6cf1}, // <CJK>
  {0xAA75,0x6cbf}, // <CJK>
  {0xAA76,0x6cbb}, // <CJK>
  {0xAA77,0x6ce1}, // <CJK>
  {0xAA78,0x6cdb}, // <CJK>
  {0xAA79,0x6cca}, // <CJK>
  {0xAA7A,0x6cac}, // <CJK>
  {0xAA7B,0x6cef}, // <CJK>
  {0xAA7C,0x6cdc}, // <CJK>
  {0xAA7D,0x6cd6}, // <CJK>
  {0xAA7E,0x6ce0}, // <CJK>
  {0xAAA1,0x7095}, // <CJK>
  {0xAAA2,0x708e}, // <CJK>
  {0xAAA3,0x7092}, // <CJK>
  {0xAAA4,0x708a}, // <CJK>
  {0xAAA5,0x7099}, // <CJK>
  {0xAAA6,0x722c}, // <CJK>
  {0xAAA7,0x722d}, // <CJK>
  {0xAAA8,0x7238}, // <CJK>
  {0xAAA9,0x7248}, // <CJK>
  {0xAAAA,0x7267}, // <CJK>
  {0xAAAB,0x7269}, // <CJK>
  {0xAAAC,0x72c0}, // <CJK>
  {0xAAAD,0x72ce}, // <CJK>
  {0xAAAE,0x72d9}, // <CJK>
  {0xAAAF,0x72d7}, // <CJK>
  {0xAAB0,0x72d0}, // <CJK>
  {0xAAB1,0x73a9}, // <CJK>
  {0xAAB2,0x73a8}, // <CJK>
  {0xAAB3,0x739f}, // <CJK>
  {0xAAB4,0x73ab}, // <CJK>
  {0xAAB5,0x73a5}, // <CJK>
  {0xAAB6,0x753d}, // <CJK>
  {0xAAB7,0x759d}, // <CJK>
  {0xAAB8,0x7599}, // <CJK>
  {0xAAB9,0x759a}, // <CJK>
  {0xAABA,0x7684}, // <CJK>
  {0xAABB,0x76c2}, // <CJK>
  {0xAABC,0x76f2}, // <CJK>
  {0xAABD,0x76f4}, // <CJK>
  {0xAABE,0x77e5}, // <CJK>
  {0xAABF,0x77fd}, // <CJK>
  {0xAAC0,0x793e}, // <CJK>
  {0xAAC1,0x7940}, // <CJK>
  {0xAAC2,0x7941}, // <CJK>
  {0xAAC3,0x79c9}, // <CJK>
  {0xAAC4,0x79c8}, // <CJK>
  {0xAAC5,0x7a7a}, // <CJK>
  {0xAAC6,0x7a79}, // <CJK>
  {0xAAC7,0x7afa}, // <CJK>
  {0xAAC8,0x7cfe}, // <CJK>
  {0xAAC9,0x7f54}, // <CJK>
  {0xAACA,0x7f8c}, // <CJK>
  {0xAACB,0x7f8b}, // <CJK>
  {0xAACC,0x8005}, // <CJK>
  {0xAACD,0x80ba}, // <CJK>
  {0xAACE,0x80a5}, // <CJK>
  {0xAACF,0x80a2}, // <CJK>
  {0xAAD0,0x80b1}, // <CJK>
  {0xAAD1,0x80a1}, // <CJK>
  {0xAAD2,0x80ab}, // <CJK>
  {0xAAD3,0x80a9}, // <CJK>
  {0xAAD4,0x80b4}, // <CJK>
  {0xAAD5,0x80aa}, // <CJK>
  {0xAAD6,0x80af}, // <CJK>
  {0xAAD7,0x81e5}, // <CJK>
  {0xAAD8,0x81fe}, // <CJK>
  {0xAAD9,0x820d}, // <CJK>
  {0xAADA,0x82b3}, // <CJK>
  {0xAADB,0x829d}, // <CJK>
  {0xAADC,0x8299}, // <CJK>
  {0xAADD,0x82ad}, // <CJK>
  {0xAADE,0x82bd}, // <CJK>
  {0xAADF,0x829f}, // <CJK>
  {0xAAE0,0x82b9}, // <CJK>
  {0xAAE1,0x82b1}, // <CJK>
  {0xAAE2,0x82ac}, // <CJK>
  {0xAAE3,0x82a5}, // <CJK>
  {0xAAE4,0x82af}, // <CJK>
  {0xAAE5,0x82b8}, // <CJK>
  {0xAAE6,0x82a3}, // <CJK>
  {0xAAE7,0x82b0}, // <CJK>
  {0xAAE8,0x82be}, // <CJK>
  {0xAAE9,0x82b7}, // <CJK>
  {0xAAEA,0x864e}, // <CJK>
  {0xAAEB,0x8671}, // <CJK>
  {0xAAEC,0x521d}, // <CJK>
  {0xAAED,0x8868}, // <CJK>
  {0xAAEE,0x8ecb}, // <CJK>
  {0xAAEF,0x8fce}, // <CJK>
  {0xAAF0,0x8fd4}, // <CJK>
  {0xAAF1,0x8fd1}, // <CJK>
  {0xAAF2,0x90b5}, // <CJK>
  {0xAAF3,0x90b8}, // <CJK>
  {0xAAF4,0x90b1}, // <CJK>
  {0xAAF5,0x90b6}, // <CJK>
  {0xAAF6,0x91c7}, // <CJK>
  {0xAAF7,0x91d1}, // <CJK>
  {0xAAF8,0x9577}, // <CJK>
  {0xAAF9,0x9580}, // <CJK>
  {0xAAFA,0x961c}, // <CJK>
  {0xAAFB,0x9640}, // <CJK>
  {0xAAFC,0x963f}, // <CJK>
  {0xAAFD,0x963b}, // <CJK>
  {0xAAFE,0x9644}, // <CJK>
  {0xAB40,0x9642}, // <CJK>
  {0xAB41,0x96b9}, // <CJK>
  {0xAB42,0x96e8}, // <CJK>
  {0xAB43,0x9752}, // <CJK>
  {0xAB44,0x975e}, // <CJK>
  {0xAB45,0x4e9f}, // <CJK>
  {0xAB46,0x4ead}, // <CJK>
  {0xAB47,0x4eae}, // <CJK>
  {0xAB48,0x4fe1}, // <CJK>
  {0xAB49,0x4fb5}, // <CJK>
  {0xAB4A,0x4faf}, // <CJK>
  {0xAB4B,0x4fbf}, // <CJK>
  {0xAB4C,0x4fe0}, // <CJK>
  {0xAB4D,0x4fd1}, // <CJK>
  {0xAB4E,0x4fcf}, // <CJK>
  {0xAB4F,0x4fdd}, // <CJK>
  {0xAB50,0x4fc3}, // <CJK>
  {0xAB51,0x4fb6}, // <CJK>
  {0xAB52,0x4fd8}, // <CJK>
  {0xAB53,0x4fdf}, // <CJK>
  {0xAB54,0x4fca}, // <CJK>
  {0xAB55,0x4fd7}, // <CJK>
  {0xAB56,0x4fae}, // <CJK>
  {0xAB57,0x4fd0}, // <CJK>
  {0xAB58,0x4fc4}, // <CJK>
  {0xAB59,0x4fc2}, // <CJK>
  {0xAB5A,0x4fda}, // <CJK>
  {0xAB5B,0x4fce}, // <CJK>
  {0xAB5C,0x4fde}, // <CJK>
  {0xAB5D,0x4fb7}, // <CJK>
  {0xAB5E,0x5157}, // <CJK>
  {0xAB5F,0x5192}, // <CJK>
  {0xAB60,0x5191}, // <CJK>
  {0xAB61,0x51a0}, // <CJK>
  {0xAB62,0x524e}, // <CJK>
  {0xAB63,0x5243}, // <CJK>
  {0xAB64,0x524a}, // <CJK>
  {0xAB65,0x524d}, // <CJK>
  {0xAB66,0x524c}, // <CJK>
  {0xAB67,0x524b}, // <CJK>
  {0xAB68,0x5247}, // <CJK>
  {0xAB69,0x52c7}, // <CJK>
  {0xAB6A,0x52c9}, // <CJK>
  {0xAB6B,0x52c3}, // <CJK>
  {0xAB6C,0x52c1}, // <CJK>
  {0xAB6D,0x530d}, // <CJK>
  {0xAB6E,0x5357}, // <CJK>
  {0xAB6F,0x537b}, // <CJK>
  {0xAB70,0x539a}, // <CJK>
  {0xAB71,0x53db}, // <CJK>
  {0xAB72,0x54ac}, // <CJK>
  {0xAB73,0x54c0}, // <CJK>
  {0xAB74,0x54a8}, // <CJK>
  {0xAB75,0x54ce}, // <CJK>
  {0xAB76,0x54c9}, // <CJK>
  {0xAB77,0x54b8}, // <CJK>
  {0xAB78,0x54a6}, // <CJK>
  {0xAB79,0x54b3}, // <CJK>
  {0xAB7A,0x54c7}, // <CJK>
  {0xAB7B,0x54c2}, // <CJK>
  {0xAB7C,0x54bd}, // <CJK>
  {0xAB7D,0x54aa}, // <CJK>
  {0xAB7E,0x54c1}, // <CJK>
  {0xABA1,0x54c4}, // <CJK>
  {0xABA2,0x54c8}, // <CJK>
  {0xABA3,0x54af}, // <CJK>
  {0xABA4,0x54ab}, // <CJK>
  {0xABA5,0x54b1}, // <CJK>
  {0xABA6,0x54bb}, // <CJK>
  {0xABA7,0x54a9}, // <CJK>
  {0xABA8,0x54a7}, // <CJK>
  {0xABA9,0x54bf}, // <CJK>
  {0xABAA,0x56ff}, // <CJK>
  {0xABAB,0x5782}, // <CJK>
  {0xABAC,0x578b}, // <CJK>
  {0xABAD,0x57a0}, // <CJK>
  {0xABAE,0x57a3}, // <CJK>
  {0xABAF,0x57a2}, // <CJK>
  {0xABB0,0x57ce}, // <CJK>
  {0xABB1,0x57ae}, // <CJK>
  {0xABB2,0x5793}, // <CJK>
  {0xABB3,0x5955}, // <CJK>
  {0xABB4,0x5951}, // <CJK>
  {0xABB5,0x594f}, // <CJK>
  {0xABB6,0x594e}, // <CJK>
  {0xABB7,0x5950}, // <CJK>
  {0xABB8,0x59dc}, // <CJK>
  {0xABB9,0x59d8}, // <CJK>
  {0xABBA,0x59ff}, // <CJK>
  {0xABBB,0x59e3}, // <CJK>
  {0xABBC,0x59e8}, // <CJK>
  {0xABBD,0x5a03}, // <CJK>
  {0xABBE,0x59e5}, // <CJK>
  {0xABBF,0x59ea}, // <CJK>
  {0xABC0,0x59da}, // <CJK>
  {0xABC1,0x59e6}, // <CJK>
  {0xABC2,0x5a01}, // <CJK>
  {0xABC3,0x59fb}, // <CJK>
  {0xABC4,0x5b69}, // <CJK>
  {0xABC5,0x5ba3}, // <CJK>
  {0xABC6,0x5ba6}, // <CJK>
  {0xABC7,0x5ba4}, // <CJK>
  {0xABC8,0x5ba2}, // <CJK>
  {0xABC9,0x5ba5}, // <CJK>
  {0xABCA,0x5c01}, // <CJK>
  {0xABCB,0x5c4e}, // <CJK>
  {0xABCC,0x5c4f}, // <CJK>
  {0xABCD,0x5c4d}, // <CJK>
  {0xABCE,0x5c4b}, // <CJK>
  {0xABCF,0x5cd9}, // <CJK>
  {0xABD0,0x5cd2}, // <CJK>
  {0xABD1,0x5df7}, // <CJK>
  {0xABD2,0x5e1d}, // <CJK>
  {0xABD3,0x5e25}, // <CJK>
  {0xABD4,0x5e1f}, // <CJK>
  {0xABD5,0x5e7d}, // <CJK>
  {0xABD6,0x5ea0}, // <CJK>
  {0xABD7,0x5ea6}, // <CJK>
  {0xABD8,0x5efa}, // <CJK>
  {0xABD9,0x5f08}, // <CJK>
  {0xABDA,0x5f2d}, // <CJK>
  {0xABDB,0x5f65}, // <CJK>
  {0xABDC,0x5f88}, // <CJK>
  {0xABDD,0x5f85}, // <CJK>
  {0xABDE,0x5f8a}, // <CJK>
  {0xABDF,0x5f8b}, // <CJK>
  {0xABE0,0x5f87}, // <CJK>
  {0xABE1,0x5f8c}, // <CJK>
  {0xABE2,0x5f89}, // <CJK>
  {0xABE3,0x6012}, // <CJK>
  {0xABE4,0x601d}, // <CJK>
  {0xABE5,0x6020}, // <CJK>
  {0xABE6,0x6025}, // <CJK>
  {0xABE7,0x600e}, // <CJK>
  {0xABE8,0x6028}, // <CJK>
  {0xABE9,0x604d}, // <CJK>
  {0xABEA,0x6070}, // <CJK>
  {0xABEB,0x6068}, // <CJK>
  {0xABEC,0x6062}, // <CJK>
  {0xABED,0x6046}, // <CJK>
  {0xABEE,0x6043}, // <CJK>
  {0xABEF,0x606c}, // <CJK>
  {0xABF0,0x606b}, // <CJK>
  {0xABF1,0x606a}, // <CJK>
  {0xABF2,0x6064}, // <CJK>
  {0xABF3,0x6241}, // <CJK>
  {0xABF4,0x62dc}, // <CJK>
  {0xABF5,0x6316}, // <CJK>
  {0xABF6,0x6309}, // <CJK>
  {0xABF7,0x62fc}, // <CJK>
  {0xABF8,0x62ed}, // <CJK>
  {0xABF9,0x6301}, // <CJK>
  {0xABFA,0x62ee}, // <CJK>
  {0xABFB,0x62fd}, // <CJK>
  {0xABFC,0x6307}, // <CJK>
  {0xABFD,0x62f1}, // <CJK>
  {0xABFE,0x62f7}, // <CJK>
  {0xAC40,0x62ef}, // <CJK>
  {0xAC41,0x62ec}, // <CJK>
  {0xAC42,0x62fe}, // <CJK>
  {0xAC43,0x62f4}, // <CJK>
  {0xAC44,0x6311}, // <CJK>
  {0xAC45,0x6302}, // <CJK>
  {0xAC46,0x653f}, // <CJK>
  {0xAC47,0x6545}, // <CJK>
  {0xAC48,0x65ab}, // <CJK>
  {0xAC49,0x65bd}, // <CJK>
  {0xAC4A,0x65e2}, // <CJK>
  {0xAC4B,0x6625}, // <CJK>
  {0xAC4C,0x662d}, // <CJK>
  {0xAC4D,0x6620}, // <CJK>
  {0xAC4E,0x6627}, // <CJK>
  {0xAC4F,0x662f}, // <CJK>
  {0xAC50,0x661f}, // <CJK>
  {0xAC51,0x6628}, // <CJK>
  {0xAC52,0x6631}, // <CJK>
  {0xAC53,0x6624}, // <CJK>
  {0xAC54,0x66f7}, // <CJK>
  {0xAC55,0x67ff}, // <CJK>
  {0xAC56,0x67d3}, // <CJK>
  {0xAC57,0x67f1}, // <CJK>
  {0xAC58,0x67d4}, // <CJK>
  {0xAC59,0x67d0}, // <CJK>
  {0xAC5A,0x67ec}, // <CJK>
  {0xAC5B,0x67b6}, // <CJK>
  {0xAC5C,0x67af}, // <CJK>
  {0xAC5D,0x67f5}, // <CJK>
  {0xAC5E,0x67e9}, // <CJK>
  {0xAC5F,0x67ef}, // <CJK>
  {0xAC60,0x67c4}, // <CJK>
  {0xAC61,0x67d1}, // <CJK>
  {0xAC62,0x67b4}, // <CJK>
  {0xAC63,0x67da}, // <CJK>
  {0xAC64,0x67e5}, // <CJK>
  {0xAC65,0x67b8}, // <CJK>
  {0xAC66,0x67cf}, // <CJK>
  {0xAC67,0x67de}, // <CJK>
  {0xAC68,0x67f3}, // <CJK>
  {0xAC69,0x67b0}, // <CJK>
  {0xAC6A,0x67d9}, // <CJK>
  {0xAC6B,0x67e2}, // <CJK>
  {0xAC6C,0x67dd}, // <CJK>
  {0xAC6D,0x67d2}, // <CJK>
  {0xAC6E,0x6b6a}, // <CJK>
  {0xAC6F,0x6b83}, // <CJK>
  {0xAC70,0x6b86}, // <CJK>
  {0xAC71,0x6bb5}, // <CJK>
  {0xAC72,0x6bd2}, // <CJK>
  {0xAC73,0x6bd7}, // <CJK>
  {0xAC74,0x6c1f}, // <CJK>
  {0xAC75,0x6cc9}, // <CJK>
  {0xAC76,0x6d0b}, // <CJK>
  {0xAC77,0x6d32}, // <CJK>
  {0xAC78,0x6d2a}, // <CJK>
  {0xAC79,0x6d41}, // <CJK>
  {0xAC7A,0x6d25}, // <CJK>
  {0xAC7B,0x6d0c}, // <CJK>
  {0xAC7C,0x6d31}, // <CJK>
  {0xAC7D,0x6d1e}, // <CJK>
  {0xAC7E,0x6d17}, // <CJK>
  {0xACA1,0x6d3b}, // <CJK>
  {0xACA2,0x6d3d}, // <CJK>
  {0xACA3,0x6d3e}, // <CJK>
  {0xACA4,0x6d36}, // <CJK>
  {0xACA5,0x6d1b}, // <CJK>
  {0xACA6,0x6cf5}, // <CJK>
  {0xACA7,0x6d39}, // <CJK>
  {0xACA8,0x6d27}, // <CJK>
  {0xACA9,0x6d38}, // <CJK>
  {0xACAA,0x6d29}, // <CJK>
  {0xACAB,0x6d2e}, // <CJK>
  {0xACAC,0x6d35}, // <CJK>
  {0xACAD,0x6d0e}, // <CJK>
  {0xACAE,0x6d2b}, // <CJK>
  {0xACAF,0x70ab}, // <CJK>
  {0xACB0,0x70ba}, // <CJK>
  {0xACB1,0x70b3}, // <CJK>
  {0xACB2,0x70ac}, // <CJK>
  {0xACB3,0x70af}, // <CJK>
  {0xACB4,0x70ad}, // <CJK>
  {0xACB5,0x70b8}, // <CJK>
  {0xACB6,0x70ae}, // <CJK>
  {0xACB7,0x70a4}, // <CJK>
  {0xACB8,0x7230}, // <CJK>
  {0xACB9,0x7272}, // <CJK>
  {0xACBA,0x726f}, // <CJK>
  {0xACBB,0x7274}, // <CJK>
  {0xACBC,0x72e9}, // <CJK>
  {0xACBD,0x72e0}, // <CJK>
  {0xACBE,0x72e1}, // <CJK>
  {0xACBF,0x73b7}, // <CJK>
  {0xACC0,0x73ca}, // <CJK>
  {0xACC1,0x73bb}, // <CJK>
  {0xACC2,0x73b2}, // <CJK>
  {0xACC3,0x73cd}, // <CJK>
  {0xACC4,0x73c0}, // <CJK>
  {0xACC5,0x73b3}, // <CJK>
  {0xACC6,0x751a}, // <CJK>
  {0xACC7,0x752d}, // <CJK>
  {0xACC8,0x754f}, // <CJK>
  {0xACC9,0x754c}, // <CJK>
  {0xACCA,0x754e}, // <CJK>
  {0xACCB,0x754b}, // <CJK>
  {0xACCC,0x75ab}, // <CJK>
  {0xACCD,0x75a4}, // <CJK>
  {0xACCE,0x75a5}, // <CJK>
  {0xACCF,0x75a2}, // <CJK>
  {0xACD0,0x75a3}, // <CJK>
  {0xACD1,0x7678}, // <CJK>
  {0xACD2,0x7686}, // <CJK>
  {0xACD3,0x7687}, // <CJK>
  {0xACD4,0x7688}, // <CJK>
  {0xACD5,0x76c8}, // <CJK>
  {0xACD6,0x76c6}, // <CJK>
  {0xACD7,0x76c3}, // <CJK>
  {0xACD8,0x76c5}, // <CJK>
  {0xACD9,0x7701}, // <CJK>
  {0xACDA,0x76f9}, // <CJK>
  {0xACDB,0x76f8}, // <CJK>
  {0xACDC,0x7709}, // <CJK>
  {0xACDD,0x770b}, // <CJK>
  {0xACDE,0x76fe}, // <CJK>
  {0xACDF,0x76fc}, // <CJK>
  {0xACE0,0x7707}, // <CJK>
  {0xACE1,0x77dc}, // <CJK>
  {0xACE2,0x7802}, // <CJK>
  {0xACE3,0x7814}, // <CJK>
  {0xACE4,0x780c}, // <CJK>
  {0xACE5,0x780d}, // <CJK>
  {0xACE6,0x7946}, // <CJK>
  {0xACE7,0x7949}, // <CJK>
  {0xACE8,0x7948}, // <CJK>
  {0xACE9,0x7947}, // <CJK>
  {0xACEA,0x79b9}, // <CJK>
  {0xACEB,0x79ba}, // <CJK>
  {0xACEC,0x79d1}, // <CJK>
  {0xACED,0x79d2}, // <CJK>
  {0xACEE,0x79cb}, // <CJK>
  {0xACEF,0x7a7f}, // <CJK>
  {0xACF0,0x7a81}, // <CJK>
  {0xACF1,0x7aff}, // <CJK>
  {0xACF2,0x7afd}, // <CJK>
  {0xACF3,0x7c7d}, // <CJK>
  {0xACF4,0x7d02}, // <CJK>
  {0xACF5,0x7d05}, // <CJK>
  {0xACF6,0x7d00}, // <CJK>
  {0xACF7,0x7d09}, // <CJK>
  {0xACF8,0x7d07}, // <CJK>
  {0xACF9,0x7d04}, // <CJK>
  {0xACFA,0x7d06}, // <CJK>
  {0xACFB,0x7f38}, // <CJK>
  {0xACFC,0x7f8e}, // <CJK>
  {0xACFD,0x7fbf}, // <CJK>
  {0xACFE,0x8004}, // <CJK>
  {0xAD40,0x8010}, // <CJK>
  {0xAD41,0x800d}, // <CJK>
  {0xAD42,0x8011}, // <CJK>
  {0xAD43,0x8036}, // <CJK>
  {0xAD44,0x80d6}, // <CJK>
  {0xAD45,0x80e5}, // <CJK>
  {0xAD46,0x80da}, // <CJK>
  {0xAD47,0x80c3}, // <CJK>
  {0xAD48,0x80c4}, // <CJK>
  {0xAD49,0x80cc}, // <CJK>
  {0xAD4A,0x80e1}, // <CJK>
  {0xAD4B,0x80db}, // <CJK>
  {0xAD4C,0x80ce}, // <CJK>
  {0xAD4D,0x80de}, // <CJK>
  {0xAD4E,0x80e4}, // <CJK>
  {0xAD4F,0x80dd}, // <CJK>
  {0xAD50,0x81f4}, // <CJK>
  {0xAD51,0x8222}, // <CJK>
  {0xAD52,0x82e7}, // <CJK>
  {0xAD53,0x8303}, // <CJK>
  {0xAD54,0x8305}, // <CJK>
  {0xAD55,0x82e3}, // <CJK>
  {0xAD56,0x82db}, // <CJK>
  {0xAD57,0x82e6}, // <CJK>
  {0xAD58,0x8304}, // <CJK>
  {0xAD59,0x82e5}, // <CJK>
  {0xAD5A,0x8302}, // <CJK>
  {0xAD5B,0x8309}, // <CJK>
  {0xAD5C,0x82d2}, // <CJK>
  {0xAD5D,0x82d7}, // <CJK>
  {0xAD5E,0x82f1}, // <CJK>
  {0xAD5F,0x8301}, // <CJK>
  {0xAD60,0x82dc}, // <CJK>
  {0xAD61,0x82d4}, // <CJK>
  {0xAD62,0x82d1}, // <CJK>
  {0xAD63,0x82de}, // <CJK>
  {0xAD64,0x82d3}, // <CJK>
  {0xAD65,0x82df}, // <CJK>
  {0xAD66,0x82ef}, // <CJK>
  {0xAD67,0x8306}, // <CJK>
  {0xAD68,0x8650}, // <CJK>
  {0xAD69,0x8679}, // <CJK>
  {0xAD6A,0x867b}, // <CJK>
  {0xAD6B,0x867a}, // <CJK>
  {0xAD6C,0x884d}, // <CJK>
  {0xAD6D,0x886b}, // <CJK>
  {0xAD6E,0x8981}, // <CJK>
  {0xAD6F,0x89d4}, // <CJK>
  {0xAD70,0x8a08}, // <CJK>
  {0xAD71,0x8a02}, // <CJK>
  {0xAD72,0x8a03}, // <CJK>
  {0xAD73,0x8c9e}, // <CJK>
  {0xAD74,0x8ca0}, // <CJK>
  {0xAD75,0x8d74}, // <CJK>
  {0xAD76,0x8d73}, // <CJK>
  {0xAD77,0x8db4}, // <CJK>
  {0xAD78,0x8ecd}, // <CJK>
  {0xAD79,0x8ecc}, // <CJK>
  {0xAD7A,0x8ff0}, // <CJK>
  {0xAD7B,0x8fe6}, // <CJK>
  {0xAD7C,0x8fe2}, // <CJK>
  {0xAD7D,0x8fea}, // <CJK>
  {0xAD7E,0x8fe5}, // <CJK>
  {0xADA1,0x8fed}, // <CJK>
  {0xADA2,0x8feb}, // <CJK>
  {0xADA3,0x8fe4}, // <CJK>
  {0xADA4,0x8fe8}, // <CJK>
  {0xADA5,0x90ca}, // <CJK>
  {0xADA6,0x90ce}, // <CJK>
  {0xADA7,0x90c1}, // <CJK>
  {0xADA8,0x90c3}, // <CJK>
  {0xADA9,0x914b}, // <CJK>
  {0xADAA,0x914a}, // <CJK>
  {0xADAB,0x91cd}, // <CJK>
  {0xADAC,0x9582}, // <CJK>
  {0xADAD,0x9650}, // <CJK>
  {0xADAE,0x964b}, // <CJK>
  {0xADAF,0x964c}, // <CJK>
  {0xADB0,0x964d}, // <CJK>
  {0xADB1,0x9762}, // <CJK>
  {0xADB2,0x9769}, // <CJK>
  {0xADB3,0x97cb}, // <CJK>
  {0xADB4,0x97ed}, // <CJK>
  {0xADB5,0x97f3}, // <CJK>
  {0xADB6,0x9801}, // <CJK>
  {0xADB7,0x98a8}, // <CJK>
  {0xADB8,0x98db}, // <CJK>
  {0xADB9,0x98df}, // <CJK>
  {0xADBA,0x9996}, // <CJK>
  {0xADBB,0x9999}, // <CJK>
  {0xADBC,0x4e58}, // <CJK>
  {0xADBD,0x4eb3}, // <CJK>
  {0xADBE,0x500c}, // <CJK>
  {0xADBF,0x500d}, // <CJK>
  {0xADC0,0x5023}, // <CJK>
  {0xADC1,0x4fef}, // <CJK>
  {0xADC2,0x5026}, // <CJK>
  {0xADC3,0x5025}, // <CJK>
  {0xADC4,0x4ff8}, // <CJK>
  {0xADC5,0x5029}, // <CJK>
  {0xADC6,0x5016}, // <CJK>
  {0xADC7,0x5006}, // <CJK>
  {0xADC8,0x503c}, // <CJK>
  {0xADC9,0x501f}, // <CJK>
  {0xADCA,0x501a}, // <CJK>
  {0xADCB,0x5012}, // <CJK>
  {0xADCC,0x5011}, // <CJK>
  {0xADCD,0x4ffa}, // <CJK>
  {0xADCE,0x5000}, // <CJK>
  {0xADCF,0x5014}, // <CJK>
  {0xADD0,0x5028}, // <CJK>
  {0xADD1,0x4ff1}, // <CJK>
  {0xADD2,0x5021}, // <CJK>
  {0xADD3,0x500b}, // <CJK>
  {0xADD4,0x5019}, // <CJK>
  {0xADD5,0x5018}, // <CJK>
  {0xADD6,0x4ff3}, // <CJK>
  {0xADD7,0x4fee}, // <CJK>
  {0xADD8,0x502d}, // <CJK>
  {0xADD9,0x502a}, // <CJK>
  {0xADDA,0x4ffe}, // <CJK>
  {0xADDB,0x502b}, // <CJK>
  {0xADDC,0x5009}, // <CJK>
  {0xADDD,0x517c}, // <CJK>
  {0xADDE,0x51a4}, // <CJK>
  {0xADDF,0x51a5}, // <CJK>
  {0xADE0,0x51a2}, // <CJK>
  {0xADE1,0x51cd}, // <CJK>
  {0xADE2,0x51cc}, // <CJK>
  {0xADE3,0x51c6}, // <CJK>
  {0xADE4,0x51cb}, // <CJK>
  {0xADE5,0x5256}, // <CJK>
  {0xADE6,0x525c}, // <CJK>
  {0xADE7,0x5254}, // <CJK>
  {0xADE8,0x525b}, // <CJK>
  {0xADE9,0x525d}, // <CJK>
  {0xADEA,0x532a}, // <CJK>
  {0xADEB,0x537f}, // <CJK>
  {0xADEC,0x539f}, // <CJK>
  {0xADED,0x539d}, // <CJK>
  {0xADEE,0x53df}, // <CJK>
  {0xADEF,0x54e8}, // <CJK>
  {0xADF0,0x5510}, // <CJK>
  {0xADF1,0x5501}, // <CJK>
  {0xADF2,0x5537}, // <CJK>
  {0xADF3,0x54fc}, // <CJK>
  {0xADF4,0x54e5}, // <CJK>
  {0xADF5,0x54f2}, // <CJK>
  {0xADF6,0x5506}, // <CJK>
  {0xADF7,0x54fa}, // <CJK>
  {0xADF8,0x5514}, // <CJK>
  {0xADF9,0x54e9}, // <CJK>
  {0xADFA,0x54ed}, // <CJK>
  {0xADFB,0x54e1}, // <CJK>
  {0xADFC,0x5509}, // <CJK>
  {0xADFD,0x54ee}, // <CJK>
  {0xADFE,0x54ea}, // <CJK>
  {0xAE40,0x54e6}, // <CJK>
  {0xAE41,0x5527}, // <CJK>
  {0xAE42,0x5507}, // <CJK>
  {0xAE43,0x54fd}, // <CJK>
  {0xAE44,0x550f}, // <CJK>
  {0xAE45,0x5703}, // <CJK>
  {0xAE46,0x5704}, // <CJK>
  {0xAE47,0x57c2}, // <CJK>
  {0xAE48,0x57d4}, // <CJK>
  {0xAE49,0x57cb}, // <CJK>
  {0xAE4A,0x57c3}, // <CJK>
  {0xAE4B,0x5809}, // <CJK>
  {0xAE4C,0x590f}, // <CJK>
  {0xAE4D,0x5957}, // <CJK>
  {0xAE4E,0x5958}, // <CJK>
  {0xAE4F,0x595a}, // <CJK>
  {0xAE50,0x5a11}, // <CJK>
  {0xAE51,0x5a18}, // <CJK>
  {0xAE52,0x5a1c}, // <CJK>
  {0xAE53,0x5a1f}, // <CJK>
  {0xAE54,0x5a1b}, // <CJK>
  {0xAE55,0x5a13}, // <CJK>
  {0xAE56,0x59ec}, // <CJK>
  {0xAE57,0x5a20}, // <CJK>
  {0xAE58,0x5a23}, // <CJK>
  {0xAE59,0x5a29}, // <CJK>
  {0xAE5A,0x5a25}, // <CJK>
  {0xAE5B,0x5a0c}, // <CJK>
  {0xAE5C,0x5a09}, // <CJK>
  {0xAE5D,0x5b6b}, // <CJK>
  {0xAE5E,0x5c58}, // <CJK>
  {0xAE5F,0x5bb0}, // <CJK>
  {0xAE60,0x5bb3}, // <CJK>
  {0xAE61,0x5bb6}, // <CJK>
  {0xAE62,0x5bb4}, // <CJK>
  {0xAE63,0x5bae}, // <CJK>
  {0xAE64,0x5bb5}, // <CJK>
  {0xAE65,0x5bb9}, // <CJK>
  {0xAE66,0x5bb8}, // <CJK>
  {0xAE67,0x5c04}, // <CJK>
  {0xAE68,0x5c51}, // <CJK>
  {0xAE69,0x5c55}, // <CJK>
  {0xAE6A,0x5c50}, // <CJK>
  {0xAE6B,0x5ced}, // <CJK>
  {0xAE6C,0x5cfd}, // <CJK>
  {0xAE6D,0x5cfb}, // <CJK>
  {0xAE6E,0x5cea}, // <CJK>
  {0xAE6F,0x5ce8}, // <CJK>
  {0xAE70,0x5cf0}, // <CJK>
  {0xAE71,0x5cf6}, // <CJK>
  {0xAE72,0x5d01}, // <CJK>
  {0xAE73,0x5cf4}, // <CJK>
  {0xAE74,0x5dee}, // <CJK>
  {0xAE75,0x5e2d}, // <CJK>
  {0xAE76,0x5e2b}, // <CJK>
  {0xAE77,0x5eab}, // <CJK>
  {0xAE78,0x5ead}, // <CJK>
  {0xAE79,0x5ea7}, // <CJK>
  {0xAE7A,0x5f31}, // <CJK>
  {0xAE7B,0x5f92}, // <CJK>
  {0xAE7C,0x5f91}, // <CJK>
  {0xAE7D,0x5f90}, // <CJK>
  {0xAE7E,0x6059}, // <CJK>
  {0xAEA1,0x6063}, // <CJK>
  {0xAEA2,0x6065}, // <CJK>
  {0xAEA3,0x6050}, // <CJK>
  {0xAEA4,0x6055}, // <CJK>
  {0xAEA5,0x606d}, // <CJK>
  {0xAEA6,0x6069}, // <CJK>
  {0xAEA7,0x606f}, // <CJK>
  {0xAEA8,0x6084}, // <CJK>
  {0xAEA9,0x609f}, // <CJK>
  {0xAEAA,0x609a}, // <CJK>
  {0xAEAB,0x608d}, // <CJK>
  {0xAEAC,0x6094}, // <CJK>
  {0xAEAD,0x608c}, // <CJK>
  {0xAEAE,0x6085}, // <CJK>
  {0xAEAF,0x6096}, // <CJK>
  {0xAEB0,0x6247}, // <CJK>
  {0xAEB1,0x62f3}, // <CJK>
  {0xAEB2,0x6308}, // <CJK>
  {0xAEB3,0x62ff}, // <CJK>
  {0xAEB4,0x634e}, // <CJK>
  {0xAEB5,0x633e}, // <CJK>
  {0xAEB6,0x632f}, // <CJK>
  {0xAEB7,0x6355}, // <CJK>
  {0xAEB8,0x6342}, // <CJK>
  {0xAEB9,0x6346}, // <CJK>
  {0xAEBA,0x634f}, // <CJK>
  {0xAEBB,0x6349}, // <CJK>
  {0xAEBC,0x633a}, // <CJK>
  {0xAEBD,0x6350}, // <CJK>
  {0xAEBE,0x633d}, // <CJK>
  {0xAEBF,0x632a}, // <CJK>
  {0xAEC0,0x632b}, // <CJK>
  {0xAEC1,0x6328}, // <CJK>
  {0xAEC2,0x634d}, // <CJK>
  {0xAEC3,0x634c}, // <CJK>
  {0xAEC4,0x6548}, // <CJK>
  {0xAEC5,0x6549}, // <CJK>
  {0xAEC6,0x6599}, // <CJK>
  {0xAEC7,0x65c1}, // <CJK>
  {0xAEC8,0x65c5}, // <CJK>
  {0xAEC9,0x6642}, // <CJK>
  {0xAECA,0x6649}, // <CJK>
  {0xAECB,0x664f}, // <CJK>
  {0xAECC,0x6643}, // <CJK>
  {0xAECD,0x6652}, // <CJK>
  {0xAECE,0x664c}, // <CJK>
  {0xAECF,0x6645}, // <CJK>
  {0xAED0,0x6641}, // <CJK>
  {0xAED1,0x66f8}, // <CJK>
  {0xAED2,0x6714}, // <CJK>
  {0xAED3,0x6715}, // <CJK>
  {0xAED4,0x6717}, // <CJK>
  {0xAED5,0x6821}, // <CJK>
  {0xAED6,0x6838}, // <CJK>
  {0xAED7,0x6848}, // <CJK>
  {0xAED8,0x6846}, // <CJK>
  {0xAED9,0x6853}, // <CJK>
  {0xAEDA,0x6839}, // <CJK>
  {0xAEDB,0x6842}, // <CJK>
  {0xAEDC,0x6854}, // <CJK>
  {0xAEDD,0x6829}, // <CJK>
  {0xAEDE,0x68b3}, // <CJK>
  {0xAEDF,0x6817}, // <CJK>
  {0xAEE0,0x684c}, // <CJK>
  {0xAEE1,0x6851}, // <CJK>
  {0xAEE2,0x683d}, // <CJK>
  {0xAEE3,0x67f4}, // <CJK>
  {0xAEE4,0x6850}, // <CJK>
  {0xAEE5,0x6840}, // <CJK>
  {0xAEE6,0x683c}, // <CJK>
  {0xAEE7,0x6843}, // <CJK>
  {0xAEE8,0x682a}, // <CJK>
  {0xAEE9,0x6845}, // <CJK>
  {0xAEEA,0x6813}, // <CJK>
  {0xAEEB,0x6818}, // <CJK>
  {0xAEEC,0x6841}, // <CJK>
  {0xAEED,0x6b8a}, // <CJK>
  {0xAEEE,0x6b89}, // <CJK>
  {0xAEEF,0x6bb7}, // <CJK>
  {0xAEF0,0x6c23}, // <CJK>
  {0xAEF1,0x6c27}, // <CJK>
  {0xAEF2,0x6c28}, // <CJK>
  {0xAEF3,0x6c26}, // <CJK>
  {0xAEF4,0x6c24}, // <CJK>
  {0xAEF5,0x6cf0}, // <CJK>
  {0xAEF6,0x6d6a}, // <CJK>
  {0xAEF7,0x6d95}, // <CJK>
  {0xAEF8,0x6d88}, // <CJK>
  {0xAEF9,0x6d87}, // <CJK>
  {0xAEFA,0x6d66}, // <CJK>
  {0xAEFB,0x6d78}, // <CJK>
  {0xAEFC,0x6d77}, // <CJK>
  {0xAEFD,0x6d59}, // <CJK>
  {0xAEFE,0x6d93}, // <CJK>
  {0xAF40,0x6d6c}, // <CJK>
  {0xAF41,0x6d89}, // <CJK>
  {0xAF42,0x6d6e}, // <CJK>
  {0xAF43,0x6d5a}, // <CJK>
  {0xAF44,0x6d74}, // <CJK>
  {0xAF45,0x6d69}, // <CJK>
  {0xAF46,0x6d8c}, // <CJK>
  {0xAF47,0x6d8a}, // <CJK>
  {0xAF48,0x6d79}, // <CJK>
  {0xAF49,0x6d85}, // <CJK>
  {0xAF4A,0x6d65}, // <CJK>
  {0xAF4B,0x6d94}, // <CJK>
  {0xAF4C,0x70ca}, // <CJK>
  {0xAF4D,0x70d8}, // <CJK>
  {0xAF4E,0x70e4}, // <CJK>
  {0xAF4F,0x70d9}, // <CJK>
  {0xAF50,0x70c8}, // <CJK>
  {0xAF51,0x70cf}, // <CJK>
  {0xAF52,0x7239}, // <CJK>
  {0xAF53,0x7279}, // <CJK>
  {0xAF54,0x72fc}, // <CJK>
  {0xAF55,0x72f9}, // <CJK>
  {0xAF56,0x72fd}, // <CJK>
  {0xAF57,0x72f8}, // <CJK>
  {0xAF58,0x72f7}, // <CJK>
  {0xAF59,0x7386}, // <CJK>
  {0xAF5A,0x73ed}, // <CJK>
  {0xAF5B,0x7409}, // <CJK>
  {0xAF5C,0x73ee}, // <CJK>
  {0xAF5D,0x73e0}, // <CJK>
  {0xAF5E,0x73ea}, // <CJK>
  {0xAF5F,0x73de}, // <CJK>
  {0xAF60,0x7554}, // <CJK>
  {0xAF61,0x755d}, // <CJK>
  {0xAF62,0x755c}, // <CJK>
  {0xAF63,0x755a}, // <CJK>
  {0xAF64,0x7559}, // <CJK>
  {0xAF65,0x75be}, // <CJK>
  {0xAF66,0x75c5}, // <CJK>
  {0xAF67,0x75c7}, // <CJK>
  {0xAF68,0x75b2}, // <CJK>
  {0xAF69,0x75b3}, // <CJK>
  {0xAF6A,0x75bd}, // <CJK>
  {0xAF6B,0x75bc}, // <CJK>
  {0xAF6C,0x75b9}, // <CJK>
  {0xAF6D,0x75c2}, // <CJK>
  {0xAF6E,0x75b8}, // <CJK>
  {0xAF6F,0x768b}, // <CJK>
  {0xAF70,0x76b0}, // <CJK>
  {0xAF71,0x76ca}, // <CJK>
  {0xAF72,0x76cd}, // <CJK>
  {0xAF73,0x76ce}, // <CJK>
  {0xAF74,0x7729}, // <CJK>
  {0xAF75,0x771f}, // <CJK>
  {0xAF76,0x7720}, // <CJK>
  {0xAF77,0x7728}, // <CJK>
  {0xAF78,0x77e9}, // <CJK>
  {0xAF79,0x7830}, // <CJK>
  {0xAF7A,0x7827}, // <CJK>
  {0xAF7B,0x7838}, // <CJK>
  {0xAF7C,0x781d}, // <CJK>
  {0xAF7D,0x7834}, // <CJK>
  {0xAF7E,0x7837}, // <CJK>
  {0xAFA1,0x7825}, // <CJK>
  {0xAFA2,0x782d}, // <CJK>
  {0xAFA3,0x7820}, // <CJK>
  {0xAFA4,0x781f}, // <CJK>
  {0xAFA5,0x7832}, // <CJK>
  {0xAFA6,0x7955}, // <CJK>
  {0xAFA7,0x7950}, // <CJK>
  {0xAFA8,0x7960}, // <CJK>
  {0xAFA9,0x795f}, // <CJK>
  {0xAFAA,0x7956}, // <CJK>
  {0xAFAB,0x795e}, // <CJK>
  {0xAFAC,0x795d}, // <CJK>
  {0xAFAD,0x7957}, // <CJK>
  {0xAFAE,0x795a}, // <CJK>
  {0xAFAF,0x79e4}, // <CJK>
  {0xAFB0,0x79e3}, // <CJK>
  {0xAFB1,0x79e7}, // <CJK>
  {0xAFB2,0x79df}, // <CJK>
  {0xAFB3,0x79e6}, // <CJK>
  {0xAFB4,0x79e9}, // <CJK>
  {0xAFB5,0x79d8}, // <CJK>
  {0xAFB6,0x7a84}, // <CJK>
  {0xAFB7,0x7a88}, // <CJK>
  {0xAFB8,0x7ad9}, // <CJK>
  {0xAFB9,0x7b06}, // <CJK>
  {0xAFBA,0x7b11}, // <CJK>
  {0xAFBB,0x7c89}, // <CJK>
  {0xAFBC,0x7d21}, // <CJK>
  {0xAFBD,0x7d17}, // <CJK>
  {0xAFBE,0x7d0b}, // <CJK>
  {0xAFBF,0x7d0a}, // <CJK>
  {0xAFC0,0x7d20}, // <CJK>
  {0xAFC1,0x7d22}, // <CJK>
  {0xAFC2,0x7d14}, // <CJK>
  {0xAFC3,0x7d10}, // <CJK>
  {0xAFC4,0x7d15}, // <CJK>
  {0xAFC5,0x7d1a}, // <CJK>
  {0xAFC6,0x7d1c}, // <CJK>
  {0xAFC7,0x7d0d}, // <CJK>
  {0xAFC8,0x7d19}, // <CJK>
  {0xAFC9,0x7d1b}, // <CJK>
  {0xAFCA,0x7f3a}, // <CJK>
  {0xAFCB,0x7f5f}, // <CJK>
  {0xAFCC,0x7f94}, // <CJK>
  {0xAFCD,0x7fc5}, // <CJK>
  {0xAFCE,0x7fc1}, // <CJK>
  {0xAFCF,0x8006}, // <CJK>
  {0xAFD0,0x8018}, // <CJK>
  {0xAFD1,0x8015}, // <CJK>
  {0xAFD2,0x8019}, // <CJK>
  {0xAFD3,0x8017}, // <CJK>
  {0xAFD4,0x803d}, // <CJK>
  {0xAFD5,0x803f}, // <CJK>
  {0xAFD6,0x80f1}, // <CJK>
  {0xAFD7,0x8102}, // <CJK>
  {0xAFD8,0x80f0}, // <CJK>
  {0xAFD9,0x8105}, // <CJK>
  {0xAFDA,0x80ed}, // <CJK>
  {0xAFDB,0x80f4}, // <CJK>
  {0xAFDC,0x8106}, // <CJK>
  {0xAFDD,0x80f8}, // <CJK>
  {0xAFDE,0x80f3}, // <CJK>
  {0xAFDF,0x8108}, // <CJK>
  {0xAFE0,0x80fd}, // <CJK>
  {0xAFE1,0x810a}, // <CJK>
  {0xAFE2,0x80fc}, // <CJK>
  {0xAFE3,0x80ef}, // <CJK>
  {0xAFE4,0x81ed}, // <CJK>
  {0xAFE5,0x81ec}, // <CJK>
  {0xAFE6,0x8200}, // <CJK>
  {0xAFE7,0x8210}, // <CJK>
  {0xAFE8,0x822a}, // <CJK>
  {0xAFE9,0x822b}, // <CJK>
  {0xAFEA,0x8228}, // <CJK>
  {0xAFEB,0x822c}, // <CJK>
  {0xAFEC,0x82bb}, // <CJK>
  {0xAFED,0x832b}, // <CJK>
  {0xAFEE,0x8352}, // <CJK>
  {0xAFEF,0x8354}, // <CJK>
  {0xAFF0,0x834a}, // <CJK>
  {0xAFF1,0x8338}, // <CJK>
  {0xAFF2,0x8350}, // <CJK>
  {0xAFF3,0x8349}, // <CJK>
  {0xAFF4,0x8335}, // <CJK>
  {0xAFF5,0x8334}, // <CJK>
  {0xAFF6,0x834f}, // <CJK>
  {0xAFF7,0x8332}, // <CJK>
  {0xAFF8,0x8339}, // <CJK>
  {0xAFF9,0x8336}, // <CJK>
  {0xAFFA,0x8317}, // <CJK>
  {0xAFFB,0x8340}, // <CJK>
  {0xAFFC,0x8331}, // <CJK>
  {0xAFFD,0x8328}, // <CJK>
  {0xAFFE,0x8343}, // <CJK>
  {0xB040,0x8654}, // <CJK>
  {0xB041,0x868a}, // <CJK>
  {0xB042,0x86aa}, // <CJK>
  {0xB043,0x8693}, // <CJK>
  {0xB044,0x86a4}, // <CJK>
  {0xB045,0x86a9}, // <CJK>
  {0xB046,0x868c}, // <CJK>
  {0xB047,0x86a3}, // <CJK>
  {0xB048,0x869c}, // <CJK>
  {0xB049,0x8870}, // <CJK>
  {0xB04A,0x8877}, // <CJK>
  {0xB04B,0x8881}, // <CJK>
  {0xB04C,0x8882}, // <CJK>
  {0xB04D,0x887d}, // <CJK>
  {0xB04E,0x8879}, // <CJK>
  {0xB04F,0x8a18}, // <CJK>
  {0xB050,0x8a10}, // <CJK>
  {0xB051,0x8a0e}, // <CJK>
  {0xB052,0x8a0c}, // <CJK>
  {0xB053,0x8a15}, // <CJK>
  {0xB054,0x8a0a}, // <CJK>
  {0xB055,0x8a17}, // <CJK>
  {0xB056,0x8a13}, // <CJK>
  {0xB057,0x8a16}, // <CJK>
  {0xB058,0x8a0f}, // <CJK>
  {0xB059,0x8a11}, // <CJK>
  {0xB05A,0x8c48}, // <CJK>
  {0xB05B,0x8c7a}, // <CJK>
  {0xB05C,0x8c79}, // <CJK>
  {0xB05D,0x8ca1}, // <CJK>
  {0xB05E,0x8ca2}, // <CJK>
  {0xB05F,0x8d77}, // <CJK>
  {0xB060,0x8eac}, // <CJK>
  {0xB061,0x8ed2}, // <CJK>
  {0xB062,0x8ed4}, // <CJK>
  {0xB063,0x8ecf}, // <CJK>
  {0xB064,0x8fb1}, // <CJK>
  {0xB065,0x9001}, // <CJK>
  {0xB066,0x9006}, // <CJK>
  {0xB067,0x8ff7}, // <CJK>
  {0xB068,0x9000}, // <CJK>
  {0xB069,0x8ffa}, // <CJK>
  {0xB06A,0x8ff4}, // <CJK>
  {0xB06B,0x9003}, // <CJK>
  {0xB06C,0x8ffd}, // <CJK>
  {0xB06D,0x9005}, // <CJK>
  {0xB06E,0x8ff8}, // <CJK>
  {0xB06F,0x9095}, // <CJK>
  {0xB070,0x90e1}, // <CJK>
  {0xB071,0x90dd}, // <CJK>
  {0xB072,0x90e2}, // <CJK>
  {0xB073,0x9152}, // <CJK>
  {0xB074,0x914d}, // <CJK>
  {0xB075,0x914c}, // <CJK>
  {0xB076,0x91d8}, // <CJK>
  {0xB077,0x91dd}, // <CJK>
  {0xB078,0x91d7}, // <CJK>
  {0xB079,0x91dc}, // <CJK>
  {0xB07A,0x91d9}, // <CJK>
  {0xB07B,0x9583}, // <CJK>
  {0xB07C,0x9662}, // <CJK>
  {0xB07D,0x9663}, // <CJK>
  {0xB07E,0x9661}, // <CJK>
  {0xB0A1,0x965b}, // <CJK>
  {0xB0A2,0x965d}, // <CJK>
  {0xB0A3,0x9664}, // <CJK>
  {0xB0A4,0x9658}, // <CJK>
  {0xB0A5,0x965e}, // <CJK>
  {0xB0A6,0x96bb}, // <CJK>
  {0xB0A7,0x98e2}, // <CJK>
  {0xB0A8,0x99ac}, // <CJK>
  {0xB0A9,0x9aa8}, // <CJK>
  {0xB0AA,0x9ad8}, // <CJK>
  {0xB0AB,0x9b25}, // <CJK>
  {0xB0AC,0x9b32}, // <CJK>
  {0xB0AD,0x9b3c}, // <CJK>
  {0xB0AE,0x4e7e}, // <CJK>
  {0xB0AF,0x507a}, // <CJK>
  {0xB0B0,0x507d}, // <CJK>
  {0xB0B1,0x505c}, // <CJK>
  {0xB0B2,0x5047}, // <CJK>
  {0xB0B3,0x5043}, // <CJK>
  {0xB0B4,0x504c}, // <CJK>
  {0xB0B5,0x505a}, // <CJK>
  {0xB0B6,0x5049}, // <CJK>
  {0xB0B7,0x5065}, // <CJK>
  {0xB0B8,0x5076}, // <CJK>
  {0xB0B9,0x504e}, // <CJK>
  {0xB0BA,0x5055}, // <CJK>
  {0xB0BB,0x5075}, // <CJK>
  {0xB0BC,0x5074}, // <CJK>
  {0xB0BD,0x5077}, // <CJK>
  {0xB0BE,0x504f}, // <CJK>
  {0xB0BF,0x500f}, // <CJK>
  {0xB0C0,0x506f}, // <CJK>
  {0xB0C1,0x506d}, // <CJK>
  {0xB0C2,0x515c}, // <CJK>
  {0xB0C3,0x5195}, // <CJK>
  {0xB0C4,0x51f0}, // <CJK>
  {0xB0C5,0x526a}, // <CJK>
  {0xB0C6,0x526f}, // <CJK>
  {0xB0C7,0x52d2}, // <CJK>
  {0xB0C8,0x52d9}, // <CJK>
  {0xB0C9,0x52d8}, // <CJK>
  {0xB0CA,0x52d5}, // <CJK>
  {0xB0CB,0x5310}, // <CJK>
  {0xB0CC,0x530f}, // <CJK>
  {0xB0CD,0x5319}, // <CJK>
  {0xB0CE,0x533f}, // <CJK>
  {0xB0CF,0x5340}, // <CJK>
  {0xB0D0,0x533e}, // <CJK>
  {0xB0D1,0x53c3}, // <CJK>
  {0xB0D2,0x66fc}, // <CJK>
  {0xB0D3,0x5546}, // <CJK>
  {0xB0D4,0x556a}, // <CJK>
  {0xB0D5,0x5566}, // <CJK>
  {0xB0D6,0x5544}, // <CJK>
  {0xB0D7,0x555e}, // <CJK>
  {0xB0D8,0x5561}, // <CJK>
  {0xB0D9,0x5543}, // <CJK>
  {0xB0DA,0x554a}, // <CJK>
  {0xB0DB,0x5531}, // <CJK>
  {0xB0DC,0x5556}, // <CJK>
  {0xB0DD,0x554f}, // <CJK>
  {0xB0DE,0x5555}, // <CJK>
  {0xB0DF,0x552f}, // <CJK>
  {0xB0E0,0x5564}, // <CJK>
  {0xB0E1,0x5538}, // <CJK>
  {0xB0E2,0x552e}, // <CJK>
  {0xB0E3,0x555c}, // <CJK>
  {0xB0E4,0x552c}, // <CJK>
  {0xB0E5,0x5563}, // <CJK>
  {0xB0E6,0x5533}, // <CJK>
  {0xB0E7,0x5541}, // <CJK>
  {0xB0E8,0x5557}, // <CJK>
  {0xB0E9,0x5708}, // <CJK>
  {0xB0EA,0x570b}, // <CJK>
  {0xB0EB,0x5709}, // <CJK>
  {0xB0EC,0x57df}, // <CJK>
  {0xB0ED,0x5805}, // <CJK>
  {0xB0EE,0x580a}, // <CJK>
  {0xB0EF,0x5806}, // <CJK>
  {0xB0F0,0x57e0}, // <CJK>
  {0xB0F1,0x57e4}, // <CJK>
  {0xB0F2,0x57fa}, // <CJK>
  {0xB0F3,0x5802}, // <CJK>
  {0xB0F4,0x5835}, // <CJK>
  {0xB0F5,0x57f7}, // <CJK>
  {0xB0F6,0x57f9}, // <CJK>
  {0xB0F7,0x5920}, // <CJK>
  {0xB0F8,0x5962}, // <CJK>
  {0xB0F9,0x5a36}, // <CJK>
  {0xB0FA,0x5a41}, // <CJK>
  {0xB0FB,0x5a49}, // <CJK>
  {0xB0FC,0x5a66}, // <CJK>
  {0xB0FD,0x5a6a}, // <CJK>
  {0xB0FE,0x5a40}, // <CJK>
  {0xB140,0x5a3c}, // <CJK>
  {0xB141,0x5a62}, // <CJK>
  {0xB142,0x5a5a}, // <CJK>
  {0xB143,0x5a46}, // <CJK>
  {0xB144,0x5a4a}, // <CJK>
  {0xB145,0x5b70}, // <CJK>
  {0xB146,0x5bc7}, // <CJK>
  {0xB147,0x5bc5}, // <CJK>
  {0xB148,0x5bc4}, // <CJK>
  {0xB149,0x5bc2}, // <CJK>
  {0xB14A,0x5bbf}, // <CJK>
  {0xB14B,0x5bc6}, // <CJK>
  {0xB14C,0x5c09}, // <CJK>
  {0xB14D,0x5c08}, // <CJK>
  {0xB14E,0x5c07}, // <CJK>
  {0xB14F,0x5c60}, // <CJK>
  {0xB150,0x5c5c}, // <CJK>
  {0xB151,0x5c5d}, // <CJK>
  {0xB152,0x5d07}, // <CJK>
  {0xB153,0x5d06}, // <CJK>
  {0xB154,0x5d0e}, // <CJK>
  {0xB155,0x5d1b}, // <CJK>
  {0xB156,0x5d16}, // <CJK>
  {0xB157,0x5d22}, // <CJK>
  {0xB158,0x5d11}, // <CJK>
  {0xB159,0x5d29}, // <CJK>
  {0xB15A,0x5d14}, // <CJK>
  {0xB15B,0x5d19}, // <CJK>
  {0xB15C,0x5d24}, // <CJK>
  {0xB15D,0x5d27}, // <CJK>
  {0xB15E,0x5d17}, // <CJK>
  {0xB15F,0x5de2}, // <CJK>
  {0xB160,0x5e38}, // <CJK>
  {0xB161,0x5e36}, // <CJK>
  {0xB162,0x5e33}, // <CJK>
  {0xB163,0x5e37}, // <CJK>
  {0xB164,0x5eb7}, // <CJK>
  {0xB165,0x5eb8}, // <CJK>
  {0xB166,0x5eb6}, // <CJK>
  {0xB167,0x5eb5}, // <CJK>
  {0xB168,0x5ebe}, // <CJK>
  {0xB169,0x5f35}, // <CJK>
  {0xB16A,0x5f37}, // <CJK>
  {0xB16B,0x5f57}, // <CJK>
  {0xB16C,0x5f6c}, // <CJK>
  {0xB16D,0x5f69}, // <CJK>
  {0xB16E,0x5f6b}, // <CJK>
  {0xB16F,0x5f97}, // <CJK>
  {0xB170,0x5f99}, // <CJK>
  {0xB171,0x5f9e}, // <CJK>
  {0xB172,0x5f98}, // <CJK>
  {0xB173,0x5fa1}, // <CJK>
  {0xB174,0x5fa0}, // <CJK>
  {0xB175,0x5f9c}, // <CJK>
  {0xB176,0x607f}, // <CJK>
  {0xB177,0x60a3}, // <CJK>
  {0xB178,0x6089}, // <CJK>
  {0xB179,0x60a0}, // <CJK>
  {0xB17A,0x60a8}, // <CJK>
  {0xB17B,0x60cb}, // <CJK>
  {0xB17C,0x60b4}, // <CJK>
  {0xB17D,0x60e6}, // <CJK>
  {0xB17E,0x60bd}, // <CJK>
  {0xB1A1,0x60c5}, // <CJK>
  {0xB1A2,0x60bb}, // <CJK>
  {0xB1A3,0x60b5}, // <CJK>
  {0xB1A4,0x60dc}, // <CJK>
  {0xB1A5,0x60bc}, // <CJK>
  {0xB1A6,0x60d8}, // <CJK>
  {0xB1A7,0x60d5}, // <CJK>
  {0xB1A8,0x60c6}, // <CJK>
  {0xB1A9,0x60df}, // <CJK>
  {0xB1AA,0x60b8}, // <CJK>
  {0xB1AB,0x60da}, // <CJK>
  {0xB1AC,0x60c7}, // <CJK>
  {0xB1AD,0x621a}, // <CJK>
  {0xB1AE,0x621b}, // <CJK>
  {0xB1AF,0x6248}, // <CJK>
  {0xB1B0,0x63a0}, // <CJK>
  {0xB1B1,0x63a7}, // <CJK>
  {0xB1B2,0x6372}, // <CJK>
  {0xB1B3,0x6396}, // <CJK>
  {0xB1B4,0x63a2}, // <CJK>
  {0xB1B5,0x63a5}, // <CJK>
  {0xB1B6,0x6377}, // <CJK>
  {0xB1B7,0x6367}, // <CJK>
  {0xB1B8,0x6398}, // <CJK>
  {0xB1B9,0x63aa}, // <CJK>
  {0xB1BA,0x6371}, // <CJK>
  {0xB1BB,0x63a9}, // <CJK>
  {0xB1BC,0x6389}, // <CJK>
  {0xB1BD,0x6383}, // <CJK>
  {0xB1BE,0x639b}, // <CJK>
  {0xB1BF,0x636b}, // <CJK>
  {0xB1C0,0x63a8}, // <CJK>
  {0xB1C1,0x6384}, // <CJK>
  {0xB1C2,0x6388}, // <CJK>
  {0xB1C3,0x6399}, // <CJK>
  {0xB1C4,0x63a1}, // <CJK>
  {0xB1C5,0x63ac}, // <CJK>
  {0xB1C6,0x6392}, // <CJK>
  {0xB1C7,0x638f}, // <CJK>
  {0xB1C8,0x6380}, // <CJK>
  {0xB1C9,0x637b}, // <CJK>
  {0xB1CA,0x6369}, // <CJK>
  {0xB1CB,0x6368}, // <CJK>
  {0xB1CC,0x637a}, // <CJK>
  {0xB1CD,0x655d}, // <CJK>
  {0xB1CE,0x6556}, // <CJK>
  {0xB1CF,0x6551}, // <CJK>
  {0xB1D0,0x6559}, // <CJK>
  {0xB1D1,0x6557}, // <CJK>
  {0xB1D2,0x555f}, // <CJK>
  {0xB1D3,0x654f}, // <CJK>
  {0xB1D4,0x6558}, // <CJK>
  {0xB1D5,0x6555}, // <CJK>
  {0xB1D6,0x6554}, // <CJK>
  {0xB1D7,0x659c}, // <CJK>
  {0xB1D8,0x659b}, // <CJK>
  {0xB1D9,0x65ac}, // <CJK>
  {0xB1DA,0x65cf}, // <CJK>
  {0xB1DB,0x65cb}, // <CJK>
  {0xB1DC,0x65cc}, // <CJK>
  {0xB1DD,0x65ce}, // <CJK>
  {0xB1DE,0x665d}, // <CJK>
  {0xB1DF,0x665a}, // <CJK>
  {0xB1E0,0x6664}, // <CJK>
  {0xB1E1,0x6668}, // <CJK>
  {0xB1E2,0x6666}, // <CJK>
  {0xB1E3,0x665e}, // <CJK>
  {0xB1E4,0x66f9}, // <CJK>
  {0xB1E5,0x52d7}, // <CJK>
  {0xB1E6,0x671b}, // <CJK>
  {0xB1E7,0x6881}, // <CJK>
  {0xB1E8,0x68af}, // <CJK>
  {0xB1E9,0x68a2}, // <CJK>
  {0xB1EA,0x6893}, // <CJK>
  {0xB1EB,0x68b5}, // <CJK>
  {0xB1EC,0x687f}, // <CJK>
  {0xB1ED,0x6876}, // <CJK>
  {0xB1EE,0x68b1}, // <CJK>
  {0xB1EF,0x68a7}, // <CJK>
  {0xB1F0,0x6897}, // <CJK>
  {0xB1F1,0x68b0}, // <CJK>
  {0xB1F2,0x6883}, // <CJK>
  {0xB1F3,0x68c4}, // <CJK>
  {0xB1F4,0x68ad}, // <CJK>
  {0xB1F5,0x6886}, // <CJK>
  {0xB1F6,0x6885}, // <CJK>
  {0xB1F7,0x6894}, // <CJK>
  {0xB1F8,0x689d}, // <CJK>
  {0xB1F9,0x68a8}, // <CJK>
  {0xB1FA,0x689f}, // <CJK>
  {0xB1FB,0x68a1}, // <CJK>
  {0xB1FC,0x6882}, // <CJK>
  {0xB1FD,0x6b32}, // <CJK>
  {0xB1FE,0x6bba}, // <CJK>
  {0xB240,0x6beb}, // <CJK>
  {0xB241,0x6bec}, // <CJK>
  {0xB242,0x6c2b}, // <CJK>
  {0xB243,0x6d8e}, // <CJK>
  {0xB244,0x6dbc}, // <CJK>
  {0xB245,0x6df3}, // <CJK>
  {0xB246,0x6dd9}, // <CJK>
  {0xB247,0x6db2}, // <CJK>
  {0xB248,0x6de1}, // <CJK>
  {0xB249,0x6dcc}, // <CJK>
  {0xB24A,0x6de4}, // <CJK>
  {0xB24B,0x6dfb}, // <CJK>
  {0xB24C,0x6dfa}, // <CJK>
  {0xB24D,0x6e05}, // <CJK>
  {0xB24E,0x6dc7}, // <CJK>
  {0xB24F,0x6dcb}, // <CJK>
  {0xB250,0x6daf}, // <CJK>
  {0xB251,0x6dd1}, // <CJK>
  {0xB252,0x6dae}, // <CJK>
  {0xB253,0x6dde}, // <CJK>
  {0xB254,0x6df9}, // <CJK>
  {0xB255,0x6db8}, // <CJK>
  {0xB256,0x6df7}, // <CJK>
  {0xB257,0x6df5}, // <CJK>
  {0xB258,0x6dc5}, // <CJK>
  {0xB259,0x6dd2}, // <CJK>
  {0xB25A,0x6e1a}, // <CJK>
  {0xB25B,0x6db5}, // <CJK>
  {0xB25C,0x6dda}, // <CJK>
  {0xB25D,0x6deb}, // <CJK>
  {0xB25E,0x6dd8}, // <CJK>
  {0xB25F,0x6dea}, // <CJK>
  {0xB260,0x6df1}, // <CJK>
  {0xB261,0x6dee}, // <CJK>
  {0xB262,0x6de8}, // <CJK>
  {0xB263,0x6dc6}, // <CJK>
  {0xB264,0x6dc4}, // <CJK>
  {0xB265,0x6daa}, // <CJK>
  {0xB266,0x6dec}, // <CJK>
  {0xB267,0x6dbf}, // <CJK>
  {0xB268,0x6de6}, // <CJK>
  {0xB269,0x70f9}, // <CJK>
  {0xB26A,0x7109}, // <CJK>
  {0xB26B,0x710a}, // <CJK>
  {0xB26C,0x70fd}, // <CJK>
  {0xB26D,0x70ef}, // <CJK>
  {0xB26E,0x723d}, // <CJK>
  {0xB26F,0x727d}, // <CJK>
  {0xB270,0x7281}, // <CJK>
  {0xB271,0x731c}, // <CJK>
  {0xB272,0x731b}, // <CJK>
  {0xB273,0x7316}, // <CJK>
  {0xB274,0x7313}, // <CJK>
  {0xB275,0x7319}, // <CJK>
  {0xB276,0x7387}, // <CJK>
  {0xB277,0x7405}, // <CJK>
  {0xB278,0x740a}, // <CJK>
  {0xB279,0x7403}, // <CJK>
  {0xB27A,0x7406}, // <CJK>
  {0xB27B,0x73fe}, // <CJK>
  {0xB27C,0x740d}, // <CJK>
  {0xB27D,0x74e0}, // <CJK>
  {0xB27E,0x74f6}, // <CJK>
  {0xB2A1,0x74f7}, // <CJK>
  {0xB2A2,0x751c}, // <CJK>
  {0xB2A3,0x7522}, // <CJK>
  {0xB2A4,0x7565}, // <CJK>
  {0xB2A5,0x7566}, // <CJK>
  {0xB2A6,0x7562}, // <CJK>
  {0xB2A7,0x7570}, // <CJK>
  {0xB2A8,0x758f}, // <CJK>
  {0xB2A9,0x75d4}, // <CJK>
  {0xB2AA,0x75d5}, // <CJK>
  {0xB2AB,0x75b5}, // <CJK>
  {0xB2AC,0x75ca}, // <CJK>
  {0xB2AD,0x75cd}, // <CJK>
  {0xB2AE,0x768e}, // <CJK>
  {0xB2AF,0x76d4}, // <CJK>
  {0xB2B0,0x76d2}, // <CJK>
  {0xB2B1,0x76db}, // <CJK>
  {0xB2B2,0x7737}, // <CJK>
  {0xB2B3,0x773e}, // <CJK>
  {0xB2B4,0x773c}, // <CJK>
  {0xB2B5,0x7736}, // <CJK>
  {0xB2B6,0x7738}, // <CJK>
  {0xB2B7,0x773a}, // <CJK>
  {0xB2B8,0x786b}, // <CJK>
  {0xB2B9,0x7843}, // <CJK>
  {0xB2BA,0x784e}, // <CJK>
  {0xB2BB,0x7965}, // <CJK>
  {0xB2BC,0x7968}, // <CJK>
  {0xB2BD,0x796d}, // <CJK>
  {0xB2BE,0x79fb}, // <CJK>
  {0xB2BF,0x7a92}, // <CJK>
  {0xB2C0,0x7a95}, // <CJK>
  {0xB2C1,0x7b20}, // <CJK>
  {0xB2C2,0x7b28}, // <CJK>
  {0xB2C3,0x7b1b}, // <CJK>
  {0xB2C4,0x7b2c}, // <CJK>
  {0xB2C5,0x7b26}, // <CJK>
  {0xB2C6,0x7b19}, // <CJK>
  {0xB2C7,0x7b1e}, // <CJK>
  {0xB2C8,0x7b2e}, // <CJK>
  {0xB2C9,0x7c92}, // <CJK>
  {0xB2CA,0x7c97}, // <CJK>
  {0xB2CB,0x7c95}, // <CJK>
  {0xB2CC,0x7d46}, // <CJK>
  {0xB2CD,0x7d43}, // <CJK>
  {0xB2CE,0x7d71}, // <CJK>
  {0xB2CF,0x7d2e}, // <CJK>
  {0xB2D0,0x7d39}, // <CJK>
  {0xB2D1,0x7d3c}, // <CJK>
  {0xB2D2,0x7d40}, // <CJK>
  {0xB2D3,0x7d30}, // <CJK>
  {0xB2D4,0x7d33}, // <CJK>
  {0xB2D5,0x7d44}, // <CJK>
  {0xB2D6,0x7d2f}, // <CJK>
  {0xB2D7,0x7d42}, // <CJK>
  {0xB2D8,0x7d32}, // <CJK>
  {0xB2D9,0x7d31}, // <CJK>
  {0xB2DA,0x7f3d}, // <CJK>
  {0xB2DB,0x7f9e}, // <CJK>
  {0xB2DC,0x7f9a}, // <CJK>
  {0xB2DD,0x7fcc}, // <CJK>
  {0xB2DE,0x7fce}, // <CJK>
  {0xB2DF,0x7fd2}, // <CJK>
  {0xB2E0,0x801c}, // <CJK>
  {0xB2E1,0x804a}, // <CJK>
  {0xB2E2,0x8046}, // <CJK>
  {0xB2E3,0x812f}, // <CJK>
  {0xB2E4,0x8116}, // <CJK>
  {0xB2E5,0x8123}, // <CJK>
  {0xB2E6,0x812b}, // <CJK>
  {0xB2E7,0x8129}, // <CJK>
  {0xB2E8,0x8130}, // <CJK>
  {0xB2E9,0x8124}, // <CJK>
  {0xB2EA,0x8202}, // <CJK>
  {0xB2EB,0x8235}, // <CJK>
  {0xB2EC,0x8237}, // <CJK>
  {0xB2ED,0x8236}, // <CJK>
  {0xB2EE,0x8239}, // <CJK>
  {0xB2EF,0x838e}, // <CJK>
  {0xB2F0,0x839e}, // <CJK>
  {0xB2F1,0x8398}, // <CJK>
  {0xB2F2,0x8378}, // <CJK>
  {0xB2F3,0x83a2}, // <CJK>
  {0xB2F4,0x8396}, // <CJK>
  {0xB2F5,0x83bd}, // <CJK>
  {0xB2F6,0x83ab}, // <CJK>
  {0xB2F7,0x8392}, // <CJK>
  {0xB2F8,0x838a}, // <CJK>
  {0xB2F9,0x8393}, // <CJK>
  {0xB2FA,0x8389}, // <CJK>
  {0xB2FB,0x83a0}, // <CJK>
  {0xB2FC,0x8377}, // <CJK>
  {0xB2FD,0x837b}, // <CJK>
  {0xB2FE,0x837c}, // <CJK>
  {0xB340,0x8386}, // <CJK>
  {0xB341,0x83a7}, // <CJK>
  {0xB342,0x8655}, // <CJK>
  {0xB343,0x5f6a}, // <CJK>
  {0xB344,0x86c7}, // <CJK>
  {0xB345,0x86c0}, // <CJK>
  {0xB346,0x86b6}, // <CJK>
  {0xB347,0x86c4}, // <CJK>
  {0xB348,0x86b5}, // <CJK>
  {0xB349,0x86c6}, // <CJK>
  {0xB34A,0x86cb}, // <CJK>
  {0xB34B,0x86b1}, // <CJK>
  {0xB34C,0x86af}, // <CJK>
  {0xB34D,0x86c9}, // <CJK>
  {0xB34E,0x8853}, // <CJK>
  {0xB34F,0x889e}, // <CJK>
  {0xB350,0x8888}, // <CJK>
  {0xB351,0x88ab}, // <CJK>
  {0xB352,0x8892}, // <CJK>
  {0xB353,0x8896}, // <CJK>
  {0xB354,0x888d}, // <CJK>
  {0xB355,0x888b}, // <CJK>
  {0xB356,0x8993}, // <CJK>
  {0xB357,0x898f}, // <CJK>
  {0xB358,0x8a2a}, // <CJK>
  {0xB359,0x8a1d}, // <CJK>
  {0xB35A,0x8a23}, // <CJK>
  {0xB35B,0x8a25}, // <CJK>
  {0xB35C,0x8a31}, // <CJK>
  {0xB35D,0x8a2d}, // <CJK>
  {0xB35E,0x8a1f}, // <CJK>
  {0xB35F,0x8a1b}, // <CJK>
  {0xB360,0x8a22}, // <CJK>
  {0xB361,0x8c49}, // <CJK>
  {0xB362,0x8c5a}, // <CJK>
  {0xB363,0x8ca9}, // <CJK>
  {0xB364,0x8cac}, // <CJK>
  {0xB365,0x8cab}, // <CJK>
  {0xB366,0x8ca8}, // <CJK>
  {0xB367,0x8caa}, // <CJK>
  {0xB368,0x8ca7}, // <CJK>
  {0xB369,0x8d67}, // <CJK>
  {0xB36A,0x8d66}, // <CJK>
  {0xB36B,0x8dbe}, // <CJK>
  {0xB36C,0x8dba}, // <CJK>
  {0xB36D,0x8edb}, // <CJK>
  {0xB36E,0x8edf}, // <CJK>
  {0xB36F,0x9019}, // <CJK>
  {0xB370,0x900d}, // <CJK>
  {0xB371,0x901a}, // <CJK>
  {0xB372,0x9017}, // <CJK>
  {0xB373,0x9023}, // <CJK>
  {0xB374,0x901f}, // <CJK>
  {0xB375,0x901d}, // <CJK>
  {0xB376,0x9010}, // <CJK>
  {0xB377,0x9015}, // <CJK>
  {0xB378,0x901e}, // <CJK>
  {0xB379,0x9020}, // <CJK>
  {0xB37A,0x900f}, // <CJK>
  {0xB37B,0x9022}, // <CJK>
  {0xB37C,0x9016}, // <CJK>
  {0xB37D,0x901b}, // <CJK>
  {0xB37E,0x9014}, // <CJK>
  {0xB3A1,0x90e8}, // <CJK>
  {0xB3A2,0x90ed}, // <CJK>
  {0xB3A3,0x90fd}, // <CJK>
  {0xB3A4,0x9157}, // <CJK>
  {0xB3A5,0x91ce}, // <CJK>
  {0xB3A6,0x91f5}, // <CJK>
  {0xB3A7,0x91e6}, // <CJK>
  {0xB3A8,0x91e3}, // <CJK>
  {0xB3A9,0x91e7}, // <CJK>
  {0xB3AA,0x91ed}, // <CJK>
  {0xB3AB,0x91e9}, // <CJK>
  {0xB3AC,0x9589}, // <CJK>
  {0xB3AD,0x966a}, // <CJK>
  {0xB3AE,0x9675}, // <CJK>
  {0xB3AF,0x9673}, // <CJK>
  {0xB3B0,0x9678}, // <CJK>
  {0xB3B1,0x9670}, // <CJK>
  {0xB3B2,0x9674}, // <CJK>
  {0xB3B3,0x9676}, // <CJK>
  {0xB3B4,0x9677}, // <CJK>
  {0xB3B5,0x966c}, // <CJK>
  {0xB3B6,0x96c0}, // <CJK>
  {0xB3B7,0x96ea}, // <CJK>
  {0xB3B8,0x96e9}, // <CJK>
  {0xB3B9,0x7ae0}, // <CJK>
  {0xB3BA,0x7adf}, // <CJK>
  {0xB3BB,0x9802}, // <CJK>
  {0xB3BC,0x9803}, // <CJK>
  {0xB3BD,0x9b5a}, // <CJK>
  {0xB3BE,0x9ce5}, // <CJK>
  {0xB3BF,0x9e75}, // <CJK>
  {0xB3C0,0x9e7f}, // <CJK>
  {0xB3C1,0x9ea5}, // <CJK>
  {0xB3C2,0x9ebb}, // <CJK>
  {0xB3C3,0x50a2}, // <CJK>
  {0xB3C4,0x508d}, // <CJK>
  {0xB3C5,0x5085}, // <CJK>
  {0xB3C6,0x5099}, // <CJK>
  {0xB3C7,0x5091}, // <CJK>
  {0xB3C8,0x5080}, // <CJK>
  {0xB3C9,0x5096}, // <CJK>
  {0xB3CA,0x5098}, // <CJK>
  {0xB3CB,0x509a}, // <CJK>
  {0xB3CC,0x6700}, // <CJK>
  {0xB3CD,0x51f1}, // <CJK>
  {0xB3CE,0x5272}, // <CJK>
  {0xB3CF,0x5274}, // <CJK>
  {0xB3D0,0x5275}, // <CJK>
  {0xB3D1,0x5269}, // <CJK>
  {0xB3D2,0x52de}, // <CJK>
  {0xB3D3,0x52dd}, // <CJK>
  {0xB3D4,0x52db}, // <CJK>
  {0xB3D5,0x535a}, // <CJK>
  {0xB3D6,0x53a5}, // <CJK>
  {0xB3D7,0x557b}, // <CJK>
  {0xB3D8,0x5580}, // <CJK>
  {0xB3D9,0x55a7}, // <CJK>
  {0xB3DA,0x557c}, // <CJK>
  {0xB3DB,0x558a}, // <CJK>
  {0xB3DC,0x559d}, // <CJK>
  {0xB3DD,0x5598}, // <CJK>
  {0xB3DE,0x5582}, // <CJK>
  {0xB3DF,0x559c}, // <CJK>
  {0xB3E0,0x55aa}, // <CJK>
  {0xB3E1,0x5594}, // <CJK>
  {0xB3E2,0x5587}, // <CJK>
  {0xB3E3,0x558b}, // <CJK>
  {0xB3E4,0x5583}, // <CJK>
  {0xB3E5,0x55b3}, // <CJK>
  {0xB3E6,0x55ae}, // <CJK>
  {0xB3E7,0x559f}, // <CJK>
  {0xB3E8,0x553e}, // <CJK>
  {0xB3E9,0x55b2}, // <CJK>
  {0xB3EA,0x559a}, // <CJK>
  {0xB3EB,0x55bb}, // <CJK>
  {0xB3EC,0x55ac}, // <CJK>
  {0xB3ED,0x55b1}, // <CJK>
  {0xB3EE,0x557e}, // <CJK>
  {0xB3EF,0x5589}, // <CJK>
  {0xB3F0,0x55ab}, // <CJK>
  {0xB3F1,0x5599}, // <CJK>
  {0xB3F2,0x570d}, // <CJK>
  {0xB3F3,0x582f}, // <CJK>
  {0xB3F4,0x582a}, // <CJK>
  {0xB3F5,0x5834}, // <CJK>
  {0xB3F6,0x5824}, // <CJK>
  {0xB3F7,0x5830}, // <CJK>
  {0xB3F8,0x5831}, // <CJK>
  {0xB3F9,0x5821}, // <CJK>
  {0xB3FA,0x581d}, // <CJK>
  {0xB3FB,0x5820}, // <CJK>
  {0xB3FC,0x58f9}, // <CJK>
  {0xB3FD,0x58fa}, // <CJK>
  {0xB3FE,0x5960}, // <CJK>
  {0xB440,0x5a77}, // <CJK>
  {0xB441,0x5a9a}, // <CJK>
  {0xB442,0x5a7f}, // <CJK>
  {0xB443,0x5a92}, // <CJK>
  {0xB444,0x5a9b}, // <CJK>
  {0xB445,0x5aa7}, // <CJK>
  {0xB446,0x5b73}, // <CJK>
  {0xB447,0x5b71}, // <CJK>
  {0xB448,0x5bd2}, // <CJK>
  {0xB449,0x5bcc}, // <CJK>
  {0xB44A,0x5bd3}, // <CJK>
  {0xB44B,0x5bd0}, // <CJK>
  {0xB44C,0x5c0a}, // <CJK>
  {0xB44D,0x5c0b}, // <CJK>
  {0xB44E,0x5c31}, // <CJK>
  {0xB44F,0x5d4c}, // <CJK>
  {0xB450,0x5d50}, // <CJK>
  {0xB451,0x5d34}, // <CJK>
  {0xB452,0x5d47}, // <CJK>
  {0xB453,0x5dfd}, // <CJK>
  {0xB454,0x5e45}, // <CJK>
  {0xB455,0x5e3d}, // <CJK>
  {0xB456,0x5e40}, // <CJK>
  {0xB457,0x5e43}, // <CJK>
  {0xB458,0x5e7e}, // <CJK>
  {0xB459,0x5eca}, // <CJK>
  {0xB45A,0x5ec1}, // <CJK>
  {0xB45B,0x5ec2}, // <CJK>
  {0xB45C,0x5ec4}, // <CJK>
  {0xB45D,0x5f3c}, // <CJK>
  {0xB45E,0x5f6d}, // <CJK>
  {0xB45F,0x5fa9}, // <CJK>
  {0xB460,0x5faa}, // <CJK>
  {0xB461,0x5fa8}, // <CJK>
  {0xB462,0x60d1}, // <CJK>
  {0xB463,0x60e1}, // <CJK>
  {0xB464,0x60b2}, // <CJK>
  {0xB465,0x60b6}, // <CJK>
  {0xB466,0x60e0}, // <CJK>
  {0xB467,0x611c}, // <CJK>
  {0xB468,0x6123}, // <CJK>
  {0xB469,0x60fa}, // <CJK>
  {0xB46A,0x6115}, // <CJK>
  {0xB46B,0x60f0}, // <CJK>
  {0xB46C,0x60fb}, // <CJK>
  {0xB46D,0x60f4}, // <CJK>
  {0xB46E,0x6168}, // <CJK>
  {0xB46F,0x60f1}, // <CJK>
  {0xB470,0x610e}, // <CJK>
  {0xB471,0x60f6}, // <CJK>
  {0xB472,0x6109}, // <CJK>
  {0xB473,0x6100}, // <CJK>
  {0xB474,0x6112}, // <CJK>
  {0xB475,0x621f}, // <CJK>
  {0xB476,0x6249}, // <CJK>
  {0xB477,0x63a3}, // <CJK>
  {0xB478,0x638c}, // <CJK>
  {0xB479,0x63cf}, // <CJK>
  {0xB47A,0x63c0}, // <CJK>
  {0xB47B,0x63e9}, // <CJK>
  {0xB47C,0x63c9}, // <CJK>
  {0xB47D,0x63c6}, // <CJK>
  {0xB47E,0x63cd}, // <CJK>
  {0xB4A1,0x63d2}, // <CJK>
  {0xB4A2,0x63e3}, // <CJK>
  {0xB4A3,0x63d0}, // <CJK>
  {0xB4A4,0x63e1}, // <CJK>
  {0xB4A5,0x63d6}, // <CJK>
  {0xB4A6,0x63ed}, // <CJK>
  {0xB4A7,0x63ee}, // <CJK>
  {0xB4A8,0x6376}, // <CJK>
  {0xB4A9,0x63f4}, // <CJK>
  {0xB4AA,0x63ea}, // <CJK>
  {0xB4AB,0x63db}, // <CJK>
  {0xB4AC,0x6452}, // <CJK>
  {0xB4AD,0x63da}, // <CJK>
  {0xB4AE,0x63f9}, // <CJK>
  {0xB4AF,0x655e}, // <CJK>
  {0xB4B0,0x6566}, // <CJK>
  {0xB4B1,0x6562}, // <CJK>
  {0xB4B2,0x6563}, // <CJK>
  {0xB4B3,0x6591}, // <CJK>
  {0xB4B4,0x6590}, // <CJK>
  {0xB4B5,0x65af}, // <CJK>
  {0xB4B6,0x666e}, // <CJK>
  {0xB4B7,0x6670}, // <CJK>
  {0xB4B8,0x6674}, // <CJK>
  {0xB4B9,0x6676}, // <CJK>
  {0xB4BA,0x666f}, // <CJK>
  {0xB4BB,0x6691}, // <CJK>
  {0xB4BC,0x667a}, // <CJK>
  {0xB4BD,0x667e}, // <CJK>
  {0xB4BE,0x6677}, // <CJK>
  {0xB4BF,0x66fe}, // <CJK>
  {0xB4C0,0x66ff}, // <CJK>
  {0xB4C1,0x671f}, // <CJK>
  {0xB4C2,0x671d}, // <CJK>
  {0xB4C3,0x68fa}, // <CJK>
  {0xB4C4,0x68d5}, // <CJK>
  {0xB4C5,0x68e0}, // <CJK>
  {0xB4C6,0x68d8}, // <CJK>
  {0xB4C7,0x68d7}, // <CJK>
  {0xB4C8,0x6905}, // <CJK>
  {0xB4C9,0x68df}, // <CJK>
  {0xB4CA,0x68f5}, // <CJK>
  {0xB4CB,0x68ee}, // <CJK>
  {0xB4CC,0x68e7}, // <CJK>
  {0xB4CD,0x68f9}, // <CJK>
  {0xB4CE,0x68d2}, // <CJK>
  {0xB4CF,0x68f2}, // <CJK>
  {0xB4D0,0x68e3}, // <CJK>
  {0xB4D1,0x68cb}, // <CJK>
  {0xB4D2,0x68cd}, // <CJK>
  {0xB4D3,0x690d}, // <CJK>
  {0xB4D4,0x6912}, // <CJK>
  {0xB4D5,0x690e}, // <CJK>
  {0xB4D6,0x68c9}, // <CJK>
  {0xB4D7,0x68da}, // <CJK>
  {0xB4D8,0x696e}, // <CJK>
  {0xB4D9,0x68fb}, // <CJK>
  {0xB4DA,0x6b3e}, // <CJK>
  {0xB4DB,0x6b3a}, // <CJK>
  {0xB4DC,0x6b3d}, // <CJK>
  {0xB4DD,0x6b98}, // <CJK>
  {0xB4DE,0x6b96}, // <CJK>
  {0xB4DF,0x6bbc}, // <CJK>
  {0xB4E0,0x6bef}, // <CJK>
  {0xB4E1,0x6c2e}, // <CJK>
  {0xB4E2,0x6c2f}, // <CJK>
  {0xB4E3,0x6c2c}, // <CJK>
  {0xB4E4,0x6e2f}, // <CJK>
  {0xB4E5,0x6e38}, // <CJK>
  {0xB4E6,0x6e54}, // <CJK>
  {0xB4E7,0x6e21}, // <CJK>
  {0xB4E8,0x6e32}, // <CJK>
  {0xB4E9,0x6e67}, // <CJK>
  {0xB4EA,0x6e4a}, // <CJK>
  {0xB4EB,0x6e20}, // <CJK>
  {0xB4EC,0x6e25}, // <CJK>
  {0xB4ED,0x6e23}, // <CJK>
  {0xB4EE,0x6e1b}, // <CJK>
  {0xB4EF,0x6e5b}, // <CJK>
  {0xB4F0,0x6e58}, // <CJK>
  {0xB4F1,0x6e24}, // <CJK>
  {0xB4F2,0x6e56}, // <CJK>
  {0xB4F3,0x6e6e}, // <CJK>
  {0xB4F4,0x6e2d}, // <CJK>
  {0xB4F5,0x6e26}, // <CJK>
  {0xB4F6,0x6e6f}, // <CJK>
  {0xB4F7,0x6e34}, // <CJK>
  {0xB4F8,0x6e4d}, // <CJK>
  {0xB4F9,0x6e3a}, // <CJK>
  {0xB4FA,0x6e2c}, // <CJK>
  {0xB4FB,0x6e43}, // <CJK>
  {0xB4FC,0x6e1d}, // <CJK>
  {0xB4FD,0x6e3e}, // <CJK>
  {0xB4FE,0x6ecb}, // <CJK>
  {0xB540,0x6e89}, // <CJK>
  {0xB541,0x6e19}, // <CJK>
  {0xB542,0x6e4e}, // <CJK>
  {0xB543,0x6e63}, // <CJK>
  {0xB544,0x6e44}, // <CJK>
  {0xB545,0x6e72}, // <CJK>
  {0xB546,0x6e69}, // <CJK>
  {0xB547,0x6e5f}, // <CJK>
  {0xB548,0x7119}, // <CJK>
  {0xB549,0x711a}, // <CJK>
  {0xB54A,0x7126}, // <CJK>
  {0xB54B,0x7130}, // <CJK>
  {0xB54C,0x7121}, // <CJK>
  {0xB54D,0x7136}, // <CJK>
  {0xB54E,0x716e}, // <CJK>
  {0xB54F,0x711c}, // <CJK>
  {0xB550,0x724c}, // <CJK>
  {0xB551,0x7284}, // <CJK>
  {0xB552,0x7280}, // <CJK>
  {0xB553,0x7336}, // <CJK>
  {0xB554,0x7325}, // <CJK>
  {0xB555,0x7334}, // <CJK>
  {0xB556,0x7329}, // <CJK>
  {0xB557,0x743a}, // <CJK>
  {0xB558,0x742a}, // <CJK>
  {0xB559,0x7433}, // <CJK>
  {0xB55A,0x7422}, // <CJK>
  {0xB55B,0x7425}, // <CJK>
  {0xB55C,0x7435}, // <CJK>
  {0xB55D,0x7436}, // <CJK>
  {0xB55E,0x7434}, // <CJK>
  {0xB55F,0x742f}, // <CJK>
  {0xB560,0x741b}, // <CJK>
  {0xB561,0x7426}, // <CJK>
  {0xB562,0x7428}, // <CJK>
  {0xB563,0x7525}, // <CJK>
  {0xB564,0x7526}, // <CJK>
  {0xB565,0x756b}, // <CJK>
  {0xB566,0x756a}, // <CJK>
  {0xB567,0x75e2}, // <CJK>
  {0xB568,0x75db}, // <CJK>
  {0xB569,0x75e3}, // <CJK>
  {0xB56A,0x75d9}, // <CJK>
  {0xB56B,0x75d8}, // <CJK>
  {0xB56C,0x75de}, // <CJK>
  {0xB56D,0x75e0}, // <CJK>
  {0xB56E,0x767b}, // <CJK>
  {0xB56F,0x767c}, // <CJK>
  {0xB570,0x7696}, // <CJK>
  {0xB571,0x7693}, // <CJK>
  {0xB572,0x76b4}, // <CJK>
  {0xB573,0x76dc}, // <CJK>
  {0xB574,0x774f}, // <CJK>
  {0xB575,0x77ed}, // <CJK>
  {0xB576,0x785d}, // <CJK>
  {0xB577,0x786c}, // <CJK>
  {0xB578,0x786f}, // <CJK>
  {0xB579,0x7a0d}, // <CJK>
  {0xB57A,0x7a08}, // <CJK>
  {0xB57B,0x7a0b}, // <CJK>
  {0xB57C,0x7a05}, // <CJK>
  {0xB57D,0x7a00}, // <CJK>
  {0xB57E,0x7a98}, // <CJK>
  {0xB5A1,0x7a97}, // <CJK>
  {0xB5A2,0x7a96}, // <CJK>
  {0xB5A3,0x7ae5}, // <CJK>
  {0xB5A4,0x7ae3}, // <CJK>
  {0xB5A5,0x7b49}, // <CJK>
  {0xB5A6,0x7b56}, // <CJK>
  {0xB5A7,0x7b46}, // <CJK>
  {0xB5A8,0x7b50}, // <CJK>
  {0xB5A9,0x7b52}, // <CJK>
  {0xB5AA,0x7b54}, // <CJK>
  {0xB5AB,0x7b4d}, // <CJK>
  {0xB5AC,0x7b4b}, // <CJK>
  {0xB5AD,0x7b4f}, // <CJK>
  {0xB5AE,0x7b51}, // <CJK>
  {0xB5AF,0x7c9f}, // <CJK>
  {0xB5B0,0x7ca5}, // <CJK>
  {0xB5B1,0x7d5e}, // <CJK>
  {0xB5B2,0x7d50}, // <CJK>
  {0xB5B3,0x7d68}, // <CJK>
  {0xB5B4,0x7d55}, // <CJK>
  {0xB5B5,0x7d2b}, // <CJK>
  {0xB5B6,0x7d6e}, // <CJK>
  {0xB5B7,0x7d72}, // <CJK>
  {0xB5B8,0x7d61}, // <CJK>
  {0xB5B9,0x7d66}, // <CJK>
  {0xB5BA,0x7d62}, // <CJK>
  {0xB5BB,0x7d70}, // <CJK>
  {0xB5BC,0x7d73}, // <CJK>
  {0xB5BD,0x5584}, // <CJK>
  {0xB5BE,0x7fd4}, // <CJK>
  {0xB5BF,0x7fd5}, // <CJK>
  {0xB5C0,0x800b}, // <CJK>
  {0xB5C1,0x8052}, // <CJK>
  {0xB5C2,0x8085}, // <CJK>
  {0xB5C3,0x8155}, // <CJK>
  {0xB5C4,0x8154}, // <CJK>
  {0xB5C5,0x814b}, // <CJK>
  {0xB5C6,0x8151}, // <CJK>
  {0xB5C7,0x814e}, // <CJK>
  {0xB5C8,0x8139}, // <CJK>
  {0xB5C9,0x8146}, // <CJK>
  {0xB5CA,0x813e}, // <CJK>
  {0xB5CB,0x814c}, // <CJK>
  {0xB5CC,0x8153}, // <CJK>
  {0xB5CD,0x8174}, // <CJK>
  {0xB5CE,0x8212}, // <CJK>
  {0xB5CF,0x821c}, // <CJK>
  {0xB5D0,0x83e9}, // <CJK>
  {0xB5D1,0x8403}, // <CJK>
  {0xB5D2,0x83f8}, // <CJK>
  {0xB5D3,0x840d}, // <CJK>
  {0xB5D4,0x83e0}, // <CJK>
  {0xB5D5,0x83c5}, // <CJK>
  {0xB5D6,0x840b}, // <CJK>
  {0xB5D7,0x83c1}, // <CJK>
  {0xB5D8,0x83ef}, // <CJK>
  {0xB5D9,0x83f1}, // <CJK>
  {0xB5DA,0x83f4}, // <CJK>
  {0xB5DB,0x8457}, // <CJK>
  {0xB5DC,0x840a}, // <CJK>
  {0xB5DD,0x83f0}, // <CJK>
  {0xB5DE,0x840c}, // <CJK>
  {0xB5DF,0x83cc}, // <CJK>
  {0xB5E0,0x83fd}, // <CJK>
  {0xB5E1,0x83f2}, // <CJK>
  {0xB5E2,0x83ca}, // <CJK>
  {0xB5E3,0x8438}, // <CJK>
  {0xB5E4,0x840e}, // <CJK>
  {0xB5E5,0x8404}, // <CJK>
  {0xB5E6,0x83dc}, // <CJK>
  {0xB5E7,0x8407}, // <CJK>
  {0xB5E8,0x83d4}, // <CJK>
  {0xB5E9,0x83df}, // <CJK>
  {0xB5EA,0x865b}, // <CJK>
  {0xB5EB,0x86df}, // <CJK>
  {0xB5EC,0x86d9}, // <CJK>
  {0xB5ED,0x86ed}, // <CJK>
  {0xB5EE,0x86d4}, // <CJK>
  {0xB5EF,0x86db}, // <CJK>
  {0xB5F0,0x86e4}, // <CJK>
  {0xB5F1,0x86d0}, // <CJK>
  {0xB5F2,0x86de}, // <CJK>
  {0xB5F3,0x8857}, // <CJK>
  {0xB5F4,0x88c1}, // <CJK>
  {0xB5F5,0x88c2}, // <CJK>
  {0xB5F6,0x88b1}, // <CJK>
  {0xB5F7,0x8983}, // <CJK>
  {0xB5F8,0x8996}, // <CJK>
  {0xB5F9,0x8a3b}, // <CJK>
  {0xB5FA,0x8a60}, // <CJK>
  {0xB5FB,0x8a55}, // <CJK>
  {0xB5FC,0x8a5e}, // <CJK>
  {0xB5FD,0x8a3c}, // <CJK>
  {0xB5FE,0x8a41}, // <CJK>
  {0xB640,0x8a54}, // <CJK>
  {0xB641,0x8a5b}, // <CJK>
  {0xB642,0x8a50}, // <CJK>
  {0xB643,0x8a46}, // <CJK>
  {0xB644,0x8a34}, // <CJK>
  {0xB645,0x8a3a}, // <CJK>
  {0xB646,0x8a36}, // <CJK>
  {0xB647,0x8a56}, // <CJK>
  {0xB648,0x8c61}, // <CJK>
  {0xB649,0x8c82}, // <CJK>
  {0xB64A,0x8caf}, // <CJK>
  {0xB64B,0x8cbc}, // <CJK>
  {0xB64C,0x8cb3}, // <CJK>
  {0xB64D,0x8cbd}, // <CJK>
  {0xB64E,0x8cc1}, // <CJK>
  {0xB64F,0x8cbb}, // <CJK>
  {0xB650,0x8cc0}, // <CJK>
  {0xB651,0x8cb4}, // <CJK>
  {0xB652,0x8cb7}, // <CJK>
  {0xB653,0x8cb6}, // <CJK>
  {0xB654,0x8cbf}, // <CJK>
  {0xB655,0x8cb8}, // <CJK>
  {0xB656,0x8d8a}, // <CJK>
  {0xB657,0x8d85}, // <CJK>
  {0xB658,0x8d81}, // <CJK>
  {0xB659,0x8dce}, // <CJK>
  {0xB65A,0x8ddd}, // <CJK>
  {0xB65B,0x8dcb}, // <CJK>
  {0xB65C,0x8dda}, // <CJK>
  {0xB65D,0x8dd1}, // <CJK>
  {0xB65E,0x8dcc}, // <CJK>
  {0xB65F,0x8ddb}, // <CJK>
  {0xB660,0x8dc6}, // <CJK>
  {0xB661,0x8efb}, // <CJK>
  {0xB662,0x8ef8}, // <CJK>
  {0xB663,0x8efc}, // <CJK>
  {0xB664,0x8f9c}, // <CJK>
  {0xB665,0x902e}, // <CJK>
  {0xB666,0x9035}, // <CJK>
  {0xB667,0x9031}, // <CJK>
  {0xB668,0x9038}, // <CJK>
  {0xB669,0x9032}, // <CJK>
  {0xB66A,0x9036}, // <CJK>
  {0xB66B,0x9102}, // <CJK>
  {0xB66C,0x90f5}, // <CJK>
  {0xB66D,0x9109}, // <CJK>
  {0xB66E,0x90fe}, // <CJK>
  {0xB66F,0x9163}, // <CJK>
  {0xB670,0x9165}, // <CJK>
  {0xB671,0x91cf}, // <CJK>
  {0xB672,0x9214}, // <CJK>
  {0xB673,0x9215}, // <CJK>
  {0xB674,0x9223}, // <CJK>
  {0xB675,0x9209}, // <CJK>
  {0xB676,0x921e}, // <CJK>
  {0xB677,0x920d}, // <CJK>
  {0xB678,0x9210}, // <CJK>
  {0xB679,0x9207}, // <CJK>
  {0xB67A,0x9211}, // <CJK>
  {0xB67B,0x9594}, // <CJK>
  {0xB67C,0x958f}, // <CJK>
  {0xB67D,0x958b}, // <CJK>
  {0xB67E,0x9591}, // <CJK>
  {0xB6A1,0x9593}, // <CJK>
  {0xB6A2,0x9592}, // <CJK>
  {0xB6A3,0x958e}, // <CJK>
  {0xB6A4,0x968a}, // <CJK>
  {0xB6A5,0x968e}, // <CJK>
  {0xB6A6,0x968b}, // <CJK>
  {0xB6A7,0x967d}, // <CJK>
  {0xB6A8,0x9685}, // <CJK>
  {0xB6A9,0x9686}, // <CJK>
  {0xB6AA,0x968d}, // <CJK>
  {0xB6AB,0x9672}, // <CJK>
  {0xB6AC,0x9684}, // <CJK>
  {0xB6AD,0x96c1}, // <CJK>
  {0xB6AE,0x96c5}, // <CJK>
  {0xB6AF,0x96c4}, // <CJK>
  {0xB6B0,0x96c6}, // <CJK>
  {0xB6B1,0x96c7}, // <CJK>
  {0xB6B2,0x96ef}, // <CJK>
  {0xB6B3,0x96f2}, // <CJK>
  {0xB6B4,0x97cc}, // <CJK>
  {0xB6B5,0x9805}, // <CJK>
  {0xB6B6,0x9806}, // <CJK>
  {0xB6B7,0x9808}, // <CJK>
  {0xB6B8,0x98e7}, // <CJK>
  {0xB6B9,0x98ea}, // <CJK>
  {0xB6BA,0x98ef}, // <CJK>
  {0xB6BB,0x98e9}, // <CJK>
  {0xB6BC,0x98f2}, // <CJK>
  {0xB6BD,0x98ed}, // <CJK>
  {0xB6BE,0x99ae}, // <CJK>
  {0xB6BF,0x99ad}, // <CJK>
  {0xB6C0,0x9ec3}, // <CJK>
  {0xB6C1,0x9ecd}, // <CJK>
  {0xB6C2,0x9ed1}, // <CJK>
  {0xB6C3,0x4e82}, // <CJK>
  {0xB6C4,0x50ad}, // <CJK>
  {0xB6C5,0x50b5}, // <CJK>
  {0xB6C6,0x50b2}, // <CJK>
  {0xB6C7,0x50b3}, // <CJK>
  {0xB6C8,0x50c5}, // <CJK>
  {0xB6C9,0x50be}, // <CJK>
  {0xB6CA,0x50ac}, // <CJK>
  {0xB6CB,0x50b7}, // <CJK>
  {0xB6CC,0x50bb}, // <CJK>
  {0xB6CD,0x50af}, // <CJK>
  {0xB6CE,0x50c7}, // <CJK>
  {0xB6CF,0x527f}, // <CJK>
  {0xB6D0,0x5277}, // <CJK>
  {0xB6D1,0x527d}, // <CJK>
  {0xB6D2,0x52df}, // <CJK>
  {0xB6D3,0x52e6}, // <CJK>
  {0xB6D4,0x52e4}, // <CJK>
  {0xB6D5,0x52e2}, // <CJK>
  {0xB6D6,0x52e3}, // <CJK>
  {0xB6D7,0x532f}, // <CJK>
  {0xB6D8,0x55df}, // <CJK>
  {0xB6D9,0x55e8}, // <CJK>
  {0xB6DA,0x55d3}, // <CJK>
  {0xB6DB,0x55e6}, // <CJK>
  {0xB6DC,0x55ce}, // <CJK>
  {0xB6DD,0x55dc}, // <CJK>
  {0xB6DE,0x55c7}, // <CJK>
  {0xB6DF,0x55d1}, // <CJK>
  {0xB6E0,0x55e3}, // <CJK>
  {0xB6E1,0x55e4}, // <CJK>
  {0xB6E2,0x55ef}, // <CJK>
  {0xB6E3,0x55da}, // <CJK>
  {0xB6E4,0x55e1}, // <CJK>
  {0xB6E5,0x55c5}, // <CJK>
  {0xB6E6,0x55c6}, // <CJK>
  {0xB6E7,0x55e5}, // <CJK>
  {0xB6E8,0x55c9}, // <CJK>
  {0xB6E9,0x5712}, // <CJK>
  {0xB6EA,0x5713}, // <CJK>
  {0xB6EB,0x585e}, // <CJK>
  {0xB6EC,0x5851}, // <CJK>
  {0xB6ED,0x5858}, // <CJK>
  {0xB6EE,0x5857}, // <CJK>
  {0xB6EF,0x585a}, // <CJK>
  {0xB6F0,0x5854}, // <CJK>
  {0xB6F1,0x586b}, // <CJK>
  {0xB6F2,0x584c}, // <CJK>
  {0xB6F3,0x586d}, // <CJK>
  {0xB6F4,0x584a}, // <CJK>
  {0xB6F5,0x5862}, // <CJK>
  {0xB6F6,0x5852}, // <CJK>
  {0xB6F7,0x584b}, // <CJK>
  {0xB6F8,0x5967}, // <CJK>
  {0xB6F9,0x5ac1}, // <CJK>
  {0xB6FA,0x5ac9}, // <CJK>
  {0xB6FB,0x5acc}, // <CJK>
  {0xB6FC,0x5abe}, // <CJK>
  {0xB6FD,0x5abd}, // <CJK>
  {0xB6FE,0x5abc}, // <CJK>
  {0xB740,0x5ab3}, // <CJK>
  {0xB741,0x5ac2}, // <CJK>
  {0xB742,0x5ab2}, // <CJK>
  {0xB743,0x5d69}, // <CJK>
  {0xB744,0x5d6f}, // <CJK>
  {0xB745,0x5e4c}, // <CJK>
  {0xB746,0x5e79}, // <CJK>
  {0xB747,0x5ec9}, // <CJK>
  {0xB748,0x5ec8}, // <CJK>
  {0xB749,0x5f12}, // <CJK>
  {0xB74A,0x5f59}, // <CJK>
  {0xB74B,0x5fac}, // <CJK>
  {0xB74C,0x5fae}, // <CJK>
  {0xB74D,0x611a}, // <CJK>
  {0xB74E,0x610f}, // <CJK>
  {0xB74F,0x6148}, // <CJK>
  {0xB750,0x611f}, // <CJK>
  {0xB751,0x60f3}, // <CJK>
  {0xB752,0x611b}, // <CJK>
  {0xB753,0x60f9}, // <CJK>
  {0xB754,0x6101}, // <CJK>
  {0xB755,0x6108}, // <CJK>
  {0xB756,0x614e}, // <CJK>
  {0xB757,0x614c}, // <CJK>
  {0xB758,0x6144}, // <CJK>
  {0xB759,0x614d}, // <CJK>
  {0xB75A,0x613e}, // <CJK>
  {0xB75B,0x6134}, // <CJK>
  {0xB75C,0x6127}, // <CJK>
  {0xB75D,0x610d}, // <CJK>
  {0xB75E,0x6106}, // <CJK>
  {0xB75F,0x6137}, // <CJK>
  {0xB760,0x6221}, // <CJK>
  {0xB761,0x6222}, // <CJK>
  {0xB762,0x6413}, // <CJK>
  {0xB763,0x643e}, // <CJK>
  {0xB764,0x641e}, // <CJK>
  {0xB765,0x642a}, // <CJK>
  {0xB766,0x642d}, // <CJK>
  {0xB767,0x643d}, // <CJK>
  {0xB768,0x642c}, // <CJK>
  {0xB769,0x640f}, // <CJK>
  {0xB76A,0x641c}, // <CJK>
  {0xB76B,0x6414}, // <CJK>
  {0xB76C,0x640d}, // <CJK>
  {0xB76D,0x6436}, // <CJK>
  {0xB76E,0x6416}, // <CJK>
  {0xB76F,0x6417}, // <CJK>
  {0xB770,0x6406}, // <CJK>
  {0xB771,0x656c}, // <CJK>
  {0xB772,0x659f}, // <CJK>
  {0xB773,0x65b0}, // <CJK>
  {0xB774,0x6697}, // <CJK>
  {0xB775,0x6689}, // <CJK>
  {0xB776,0x6687}, // <CJK>
  {0xB777,0x6688}, // <CJK>
  {0xB778,0x6696}, // <CJK>
  {0xB779,0x6684}, // <CJK>
  {0xB77A,0x6698}, // <CJK>
  {0xB77B,0x668d}, // <CJK>
  {0xB77C,0x6703}, // <CJK>
  {0xB77D,0x6994}, // <CJK>
  {0xB77E,0x696d}, // <CJK>
  {0xB7A1,0x695a}, // <CJK>
  {0xB7A2,0x6977}, // <CJK>
  {0xB7A3,0x6960}, // <CJK>
  {0xB7A4,0x6954}, // <CJK>
  {0xB7A5,0x6975}, // <CJK>
  {0xB7A6,0x6930}, // <CJK>
  {0xB7A7,0x6982}, // <CJK>
  {0xB7A8,0x694a}, // <CJK>
  {0xB7A9,0x6968}, // <CJK>
  {0xB7AA,0x696b}, // <CJK>
  {0xB7AB,0x695e}, // <CJK>
  {0xB7AC,0x6953}, // <CJK>
  {0xB7AD,0x6979}, // <CJK>
  {0xB7AE,0x6986}, // <CJK>
  {0xB7AF,0x695d}, // <CJK>
  {0xB7B0,0x6963}, // <CJK>
  {0xB7B1,0x695b}, // <CJK>
  {0xB7B2,0x6b47}, // <CJK>
  {0xB7B3,0x6b72}, // <CJK>
  {0xB7B4,0x6bc0}, // <CJK>
  {0xB7B5,0x6bbf}, // <CJK>
  {0xB7B6,0x6bd3}, // <CJK>
  {0xB7B7,0x6bfd}, // <CJK>
  {0xB7B8,0x6ea2}, // <CJK>
  {0xB7B9,0x6eaf}, // <CJK>
  {0xB7BA,0x6ed3}, // <CJK>
  {0xB7BB,0x6eb6}, // <CJK>
  {0xB7BC,0x6ec2}, // <CJK>
  {0xB7BD,0x6e90}, // <CJK>
  {0xB7BE,0x6e9d}, // <CJK>
  {0xB7BF,0x6ec7}, // <CJK>
  {0xB7C0,0x6ec5}, // <CJK>
  {0xB7C1,0x6ea5}, // <CJK>
  {0xB7C2,0x6e98}, // <CJK>
  {0xB7C3,0x6ebc}, // <CJK>
  {0xB7C4,0x6eba}, // <CJK>
  {0xB7C5,0x6eab}, // <CJK>
  {0xB7C6,0x6ed1}, // <CJK>
  {0xB7C7,0x6e96}, // <CJK>
  {0xB7C8,0x6e9c}, // <CJK>
  {0xB7C9,0x6ec4}, // <CJK>
  {0xB7CA,0x6ed4}, // <CJK>
  {0xB7CB,0x6eaa}, // <CJK>
  {0xB7CC,0x6ea7}, // <CJK>
  {0xB7CD,0x6eb4}, // <CJK>
  {0xB7CE,0x714e}, // <CJK>
  {0xB7CF,0x7159}, // <CJK>
  {0xB7D0,0x7169}, // <CJK>
  {0xB7D1,0x7164}, // <CJK>
  {0xB7D2,0x7149}, // <CJK>
  {0xB7D3,0x7167}, // <CJK>
  {0xB7D4,0x715c}, // <CJK>
  {0xB7D5,0x716c}, // <CJK>
  {0xB7D6,0x7166}, // <CJK>
  {0xB7D7,0x714c}, // <CJK>
  {0xB7D8,0x7165}, // <CJK>
  {0xB7D9,0x715e}, // <CJK>
  {0xB7DA,0x7146}, // <CJK>
  {0xB7DB,0x7168}, // <CJK>
  {0xB7DC,0x7156}, // <CJK>
  {0xB7DD,0x723a}, // <CJK>
  {0xB7DE,0x7252}, // <CJK>
  {0xB7DF,0x7337}, // <CJK>
  {0xB7E0,0x7345}, // <CJK>
  {0xB7E1,0x733f}, // <CJK>
  {0xB7E2,0x733e}, // <CJK>
  {0xB7E3,0x746f}, // <CJK>
  {0xB7E4,0x745a}, // <CJK>
  {0xB7E5,0x7455}, // <CJK>
  {0xB7E6,0x745f}, // <CJK>
  {0xB7E7,0x745e}, // <CJK>
  {0xB7E8,0x7441}, // <CJK>
  {0xB7E9,0x743f}, // <CJK>
  {0xB7EA,0x7459}, // <CJK>
  {0xB7EB,0x745b}, // <CJK>
  {0xB7EC,0x745c}, // <CJK>
  {0xB7ED,0x7576}, // <CJK>
  {0xB7EE,0x7578}, // <CJK>
  {0xB7EF,0x7600}, // <CJK>
  {0xB7F0,0x75f0}, // <CJK>
  {0xB7F1,0x7601}, // <CJK>
  {0xB7F2,0x75f2}, // <CJK>
  {0xB7F3,0x75f1}, // <CJK>
  {0xB7F4,0x75fa}, // <CJK>
  {0xB7F5,0x75ff}, // <CJK>
  {0xB7F6,0x75f4}, // <CJK>
  {0xB7F7,0x75f3}, // <CJK>
  {0xB7F8,0x76de}, // <CJK>
  {0xB7F9,0x76df}, // <CJK>
  {0xB7FA,0x775b}, // <CJK>
  {0xB7FB,0x776b}, // <CJK>
  {0xB7FC,0x7766}, // <CJK>
  {0xB7FD,0x775e}, // <CJK>
  {0xB7FE,0x7763}, // <CJK>
  {0xB840,0x7779}, // <CJK>
  {0xB841,0x776a}, // <CJK>
  {0xB842,0x776c}, // <CJK>
  {0xB843,0x775c}, // <CJK>
  {0xB844,0x7765}, // <CJK>
  {0xB845,0x7768}, // <CJK>
  {0xB846,0x7762}, // <CJK>
  {0xB847,0x77ee}, // <CJK>
  {0xB848,0x788e}, // <CJK>
  {0xB849,0x78b0}, // <CJK>
  {0xB84A,0x7897}, // <CJK>
  {0xB84B,0x7898}, // <CJK>
  {0xB84C,0x788c}, // <CJK>
  {0xB84D,0x7889}, // <CJK>
  {0xB84E,0x787c}, // <CJK>
  {0xB84F,0x7891}, // <CJK>
  {0xB850,0x7893}, // <CJK>
  {0xB851,0x787f}, // <CJK>
  {0xB852,0x797a}, // <CJK>
  {0xB853,0x797f}, // <CJK>
  {0xB854,0x7981}, // <CJK>
  {0xB855,0x842c}, // <CJK>
  {0xB856,0x79bd}, // <CJK>
  {0xB857,0x7a1c}, // <CJK>
  {0xB858,0x7a1a}, // <CJK>
  {0xB859,0x7a20}, // <CJK>
  {0xB85A,0x7a14}, // <CJK>
  {0xB85B,0x7a1f}, // <CJK>
  {0xB85C,0x7a1e}, // <CJK>
  {0xB85D,0x7a9f}, // <CJK>
  {0xB85E,0x7aa0}, // <CJK>
  {0xB85F,0x7b77}, // <CJK>
  {0xB860,0x7bc0}, // <CJK>
  {0xB861,0x7b60}, // <CJK>
  {0xB862,0x7b6e}, // <CJK>
  {0xB863,0x7b67}, // <CJK>
  {0xB864,0x7cb1}, // <CJK>
  {0xB865,0x7cb3}, // <CJK>
  {0xB866,0x7cb5}, // <CJK>
  {0xB867,0x7d93}, // <CJK>
  {0xB868,0x7d79}, // <CJK>
  {0xB869,0x7d91}, // <CJK>
  {0xB86A,0x7d81}, // <CJK>
  {0xB86B,0x7d8f}, // <CJK>
  {0xB86C,0x7d5b}, // <CJK>
  {0xB86D,0x7f6e}, // <CJK>
  {0xB86E,0x7f69}, // <CJK>
  {0xB86F,0x7f6a}, // <CJK>
  {0xB870,0x7f72}, // <CJK>
  {0xB871,0x7fa9}, // <CJK>
  {0xB872,0x7fa8}, // <CJK>
  {0xB873,0x7fa4}, // <CJK>
  {0xB874,0x8056}, // <CJK>
  {0xB875,0x8058}, // <CJK>
  {0xB876,0x8086}, // <CJK>
  {0xB877,0x8084}, // <CJK>
  {0xB878,0x8171}, // <CJK>
  {0xB879,0x8170}, // <CJK>
  {0xB87A,0x8178}, // <CJK>
  {0xB87B,0x8165}, // <CJK>
  {0xB87C,0x816e}, // <CJK>
  {0xB87D,0x8173}, // <CJK>
  {0xB87E,0x816b}, // <CJK>
  {0xB8A1,0x8179}, // <CJK>
  {0xB8A2,0x817a}, // <CJK>
  {0xB8A3,0x8166}, // <CJK>
  {0xB8A4,0x8205}, // <CJK>
  {0xB8A5,0x8247}, // <CJK>
  {0xB8A6,0x8482}, // <CJK>
  {0xB8A7,0x8477}, // <CJK>
  {0xB8A8,0x843d}, // <CJK>
  {0xB8A9,0x8431}, // <CJK>
  {0xB8AA,0x8475}, // <CJK>
  {0xB8AB,0x8466}, // <CJK>
  {0xB8AC,0x846b}, // <CJK>
  {0xB8AD,0x8449}, // <CJK>
  {0xB8AE,0x846c}, // <CJK>
  {0xB8AF,0x845b}, // <CJK>
  {0xB8B0,0x843c}, // <CJK>
  {0xB8B1,0x8435}, // <CJK>
  {0xB8B2,0x8461}, // <CJK>
  {0xB8B3,0x8463}, // <CJK>
  {0xB8B4,0x8469}, // <CJK>
  {0xB8B5,0x846d}, // <CJK>
  {0xB8B6,0x8446}, // <CJK>
  {0xB8B7,0x865e}, // <CJK>
  {0xB8B8,0x865c}, // <CJK>
  {0xB8B9,0x865f}, // <CJK>
  {0xB8BA,0x86f9}, // <CJK>
  {0xB8BB,0x8713}, // <CJK>
  {0xB8BC,0x8708}, // <CJK>
  {0xB8BD,0x8707}, // <CJK>
  {0xB8BE,0x8700}, // <CJK>
  {0xB8BF,0x86fe}, // <CJK>
  {0xB8C0,0x86fb}, // <CJK>
  {0xB8C1,0x8702}, // <CJK>
  {0xB8C2,0x8703}, // <CJK>
  {0xB8C3,0x8706}, // <CJK>
  {0xB8C4,0x870a}, // <CJK>
  {0xB8C5,0x8859}, // <CJK>
  {0xB8C6,0x88df}, // <CJK>
  {0xB8C7,0x88d4}, // <CJK>
  {0xB8C8,0x88d9}, // <CJK>
  {0xB8C9,0x88dc}, // <CJK>
  {0xB8CA,0x88d8}, // <CJK>
  {0xB8CB,0x88dd}, // <CJK>
  {0xB8CC,0x88e1}, // <CJK>
  {0xB8CD,0x88ca}, // <CJK>
  {0xB8CE,0x88d5}, // <CJK>
  {0xB8CF,0x88d2}, // <CJK>
  {0xB8D0,0x899c}, // <CJK>
  {0xB8D1,0x89e3}, // <CJK>
  {0xB8D2,0x8a6b}, // <CJK>
  {0xB8D3,0x8a72}, // <CJK>
  {0xB8D4,0x8a73}, // <CJK>
  {0xB8D5,0x8a66}, // <CJK>
  {0xB8D6,0x8a69}, // <CJK>
  {0xB8D7,0x8a70}, // <CJK>
  {0xB8D8,0x8a87}, // <CJK>
  {0xB8D9,0x8a7c}, // <CJK>
  {0xB8DA,0x8a63}, // <CJK>
  {0xB8DB,0x8aa0}, // <CJK>
  {0xB8DC,0x8a71}, // <CJK>
  {0xB8DD,0x8a85}, // <CJK>
  {0xB8DE,0x8a6d}, // <CJK>
  {0xB8DF,0x8a62}, // <CJK>
  {0xB8E0,0x8a6e}, // <CJK>
  {0xB8E1,0x8a6c}, // <CJK>
  {0xB8E2,0x8a79}, // <CJK>
  {0xB8E3,0x8a7b}, // <CJK>
  {0xB8E4,0x8a3e}, // <CJK>
  {0xB8E5,0x8a68}, // <CJK>
  {0xB8E6,0x8c62}, // <CJK>
  {0xB8E7,0x8c8a}, // <CJK>
  {0xB8E8,0x8c89}, // <CJK>
  {0xB8E9,0x8cca}, // <CJK>
  {0xB8EA,0x8cc7}, // <CJK>
  {0xB8EB,0x8cc8}, // <CJK>
  {0xB8EC,0x8cc4}, // <CJK>
  {0xB8ED,0x8cb2}, // <CJK>
  {0xB8EE,0x8cc3}, // <CJK>
  {0xB8EF,0x8cc2}, // <CJK>
  {0xB8F0,0x8cc5}, // <CJK>
  {0xB8F1,0x8de1}, // <CJK>
  {0xB8F2,0x8ddf}, // <CJK>
  {0xB8F3,0x8de8}, // <CJK>
  {0xB8F4,0x8def}, // <CJK>
  {0xB8F5,0x8df3}, // <CJK>
  {0xB8F6,0x8dfa}, // <CJK>
  {0xB8F7,0x8dea}, // <CJK>
  {0xB8F8,0x8de4}, // <CJK>
  {0xB8F9,0x8de6}, // <CJK>
  {0xB8FA,0x8eb2}, // <CJK>
  {0xB8FB,0x8f03}, // <CJK>
  {0xB8FC,0x8f09}, // <CJK>
  {0xB8FD,0x8efe}, // <CJK>
  {0xB8FE,0x8f0a}, // <CJK>
  {0xB940,0x8f9f}, // <CJK>
  {0xB941,0x8fb2}, // <CJK>
  {0xB942,0x904b}, // <CJK>
  {0xB943,0x904a}, // <CJK>
  {0xB944,0x9053}, // <CJK>
  {0xB945,0x9042}, // <CJK>
  {0xB946,0x9054}, // <CJK>
  {0xB947,0x903c}, // <CJK>
  {0xB948,0x9055}, // <CJK>
  {0xB949,0x9050}, // <CJK>
  {0xB94A,0x9047}, // <CJK>
  {0xB94B,0x904f}, // <CJK>
  {0xB94C,0x904e}, // <CJK>
  {0xB94D,0x904d}, // <CJK>
  {0xB94E,0x9051}, // <CJK>
  {0xB94F,0x903e}, // <CJK>
  {0xB950,0x9041}, // <CJK>
  {0xB951,0x9112}, // <CJK>
  {0xB952,0x9117}, // <CJK>
  {0xB953,0x916c}, // <CJK>
  {0xB954,0x916a}, // <CJK>
  {0xB955,0x9169}, // <CJK>
  {0xB956,0x91c9}, // <CJK>
  {0xB957,0x9237}, // <CJK>
  {0xB958,0x9257}, // <CJK>
  {0xB959,0x9238}, // <CJK>
  {0xB95A,0x923d}, // <CJK>
  {0xB95B,0x9240}, // <CJK>
  {0xB95C,0x923e}, // <CJK>
  {0xB95D,0x925b}, // <CJK>
  {0xB95E,0x924b}, // <CJK>
  {0xB95F,0x9264}, // <CJK>
  {0xB960,0x9251}, // <CJK>
  {0xB961,0x9234}, // <CJK>
  {0xB962,0x9249}, // <CJK>
  {0xB963,0x924d}, // <CJK>
  {0xB964,0x9245}, // <CJK>
  {0xB965,0x9239}, // <CJK>
  {0xB966,0x923f}, // <CJK>
  {0xB967,0x925a}, // <CJK>
  {0xB968,0x9598}, // <CJK>
  {0xB969,0x9698}, // <CJK>
  {0xB96A,0x9694}, // <CJK>
  {0xB96B,0x9695}, // <CJK>
  {0xB96C,0x96cd}, // <CJK>
  {0xB96D,0x96cb}, // <CJK>
  {0xB96E,0x96c9}, // <CJK>
  {0xB96F,0x96ca}, // <CJK>
  {0xB970,0x96f7}, // <CJK>
  {0xB971,0x96fb}, // <CJK>
  {0xB972,0x96f9}, // <CJK>
  {0xB973,0x96f6}, // <CJK>
  {0xB974,0x9756}, // <CJK>
  {0xB975,0x9774}, // <CJK>
  {0xB976,0x9776}, // <CJK>
  {0xB977,0x9810}, // <CJK>
  {0xB978,0x9811}, // <CJK>
  {0xB979,0x9813}, // <CJK>
  {0xB97A,0x980a}, // <CJK>
  {0xB97B,0x9812}, // <CJK>
  {0xB97C,0x980c}, // <CJK>
  {0xB97D,0x98fc}, // <CJK>
  {0xB97E,0x98f4}, // <CJK>
  {0xB9A1,0x98fd}, // <CJK>
  {0xB9A2,0x98fe}, // <CJK>
  {0xB9A3,0x99b3}, // <CJK>
  {0xB9A4,0x99b1}, // <CJK>
  {0xB9A5,0x99b4}, // <CJK>
  {0xB9A6,0x9ae1}, // <CJK>
  {0xB9A7,0x9ce9}, // <CJK>
  {0xB9A8,0x9e82}, // <CJK>
  {0xB9A9,0x9f0e}, // <CJK>
  {0xB9AA,0x9f13}, // <CJK>
  {0xB9AB,0x9f20}, // <CJK>
  {0xB9AC,0x50e7}, // <CJK>
  {0xB9AD,0x50ee}, // <CJK>
  {0xB9AE,0x50e5}, // <CJK>
  {0xB9AF,0x50d6}, // <CJK>
  {0xB9B0,0x50ed}, // <CJK>
  {0xB9B1,0x50da}, // <CJK>
  {0xB9B2,0x50d5}, // <CJK>
  {0xB9B3,0x50cf}, // <CJK>
  {0xB9B4,0x50d1}, // <CJK>
  {0xB9B5,0x50f1}, // <CJK>
  {0xB9B6,0x50ce}, // <CJK>
  {0xB9B7,0x50e9}, // <CJK>
  {0xB9B8,0x5162}, // <CJK>
  {0xB9B9,0x51f3}, // <CJK>
  {0xB9BA,0x5283}, // <CJK>
  {0xB9BB,0x5282}, // <CJK>
  {0xB9BC,0x5331}, // <CJK>
  {0xB9BD,0x53ad}, // <CJK>
  {0xB9BE,0x55fe}, // <CJK>
  {0xB9BF,0x5600}, // <CJK>
  {0xB9C0,0x561b}, // <CJK>
  {0xB9C1,0x5617}, // <CJK>
  {0xB9C2,0x55fd}, // <CJK>
  {0xB9C3,0x5614}, // <CJK>
  {0xB9C4,0x5606}, // <CJK>
  {0xB9C5,0x5609}, // <CJK>
  {0xB9C6,0x560d}, // <CJK>
  {0xB9C7,0x560e}, // <CJK>
  {0xB9C8,0x55f7}, // <CJK>
  {0xB9C9,0x5616}, // <CJK>
  {0xB9CA,0x561f}, // <CJK>
  {0xB9CB,0x5608}, // <CJK>
  {0xB9CC,0x5610}, // <CJK>
  {0xB9CD,0x55f6}, // <CJK>
  {0xB9CE,0x5718}, // <CJK>
  {0xB9CF,0x5716}, // <CJK>
  {0xB9D0,0x5875}, // <CJK>
  {0xB9D1,0x587e}, // <CJK>
  {0xB9D2,0x5883}, // <CJK>
  {0xB9D3,0x5893}, // <CJK>
  {0xB9D4,0x588a}, // <CJK>
  {0xB9D5,0x5879}, // <CJK>
  {0xB9D6,0x5885}, // <CJK>
  {0xB9D7,0x587d}, // <CJK>
  {0xB9D8,0x58fd}, // <CJK>
  {0xB9D9,0x5925}, // <CJK>
  {0xB9DA,0x5922}, // <CJK>
  {0xB9DB,0x5924}, // <CJK>
  {0xB9DC,0x596a}, // <CJK>
  {0xB9DD,0x5969}, // <CJK>
  {0xB9DE,0x5ae1}, // <CJK>
  {0xB9DF,0x5ae6}, // <CJK>
  {0xB9E0,0x5ae9}, // <CJK>
  {0xB9E1,0x5ad7}, // <CJK>
  {0xB9E2,0x5ad6}, // <CJK>
  {0xB9E3,0x5ad8}, // <CJK>
  {0xB9E4,0x5ae3}, // <CJK>
  {0xB9E5,0x5b75}, // <CJK>
  {0xB9E6,0x5bde}, // <CJK>
  {0xB9E7,0x5be7}, // <CJK>
  {0xB9E8,0x5be1}, // <CJK>
  {0xB9E9,0x5be5}, // <CJK>
  {0xB9EA,0x5be6}, // <CJK>
  {0xB9EB,0x5be8}, // <CJK>
  {0xB9EC,0x5be2}, // <CJK>
  {0xB9ED,0x5be4}, // <CJK>
  {0xB9EE,0x5bdf}, // <CJK>
  {0xB9EF,0x5c0d}, // <CJK>
  {0xB9F0,0x5c62}, // <CJK>
  {0xB9F1,0x5d84}, // <CJK>
  {0xB9F2,0x5d87}, // <CJK>
  {0xB9F3,0x5e5b}, // <CJK>
  {0xB9F4,0x5e63}, // <CJK>
  {0xB9F5,0x5e55}, // <CJK>
  {0xB9F6,0x5e57}, // <CJK>
  {0xB9F7,0x5e54}, // <CJK>
  {0xB9F8,0x5ed3}, // <CJK>
  {0xB9F9,0x5ed6}, // <CJK>
  {0xB9FA,0x5f0a}, // <CJK>
  {0xB9FB,0x5f46}, // <CJK>
  {0xB9FC,0x5f70}, // <CJK>
  {0xB9FD,0x5fb9}, // <CJK>
  {0xB9FE,0x6147}, // <CJK>
  {0xBA40,0x613f}, // <CJK>
  {0xBA41,0x614b}, // <CJK>
  {0xBA42,0x6177}, // <CJK>
  {0xBA43,0x6162}, // <CJK>
  {0xBA44,0x6163}, // <CJK>
  {0xBA45,0x615f}, // <CJK>
  {0xBA46,0x615a}, // <CJK>
  {0xBA47,0x6158}, // <CJK>
  {0xBA48,0x6175}, // <CJK>
  {0xBA49,0x622a}, // <CJK>
  {0xBA4A,0x6487}, // <CJK>
  {0xBA4B,0x6458}, // <CJK>
  {0xBA4C,0x6454}, // <CJK>
  {0xBA4D,0x64a4}, // <CJK>
  {0xBA4E,0x6478}, // <CJK>
  {0xBA4F,0x645f}, // <CJK>
  {0xBA50,0x647a}, // <CJK>
  {0xBA51,0x6451}, // <CJK>
  {0xBA52,0x6467}, // <CJK>
  {0xBA53,0x6434}, // <CJK>
  {0xBA54,0x646d}, // <CJK>
  {0xBA55,0x647b}, // <CJK>
  {0xBA56,0x6572}, // <CJK>
  {0xBA57,0x65a1}, // <CJK>
  {0xBA58,0x65d7}, // <CJK>
  {0xBA59,0x65d6}, // <CJK>
  {0xBA5A,0x66a2}, // <CJK>
  {0xBA5B,0x66a8}, // <CJK>
  {0xBA5C,0x669d}, // <CJK>
  {0xBA5D,0x699c}, // <CJK>
  {0xBA5E,0x69a8}, // <CJK>
  {0xBA5F,0x6995}, // <CJK>
  {0xBA60,0x69c1}, // <CJK>
  {0xBA61,0x69ae}, // <CJK>
  {0xBA62,0x69d3}, // <CJK>
  {0xBA63,0x69cb}, // <CJK>
  {0xBA64,0x699b}, // <CJK>
  {0xBA65,0x69b7}, // <CJK>
  {0xBA66,0x69bb}, // <CJK>
  {0xBA67,0x69ab}, // <CJK>
  {0xBA68,0x69b4}, // <CJK>
  {0xBA69,0x69d0}, // <CJK>
  {0xBA6A,0x69cd}, // <CJK>
  {0xBA6B,0x69ad}, // <CJK>
  {0xBA6C,0x69cc}, // <CJK>
  {0xBA6D,0x69a6}, // <CJK>
  {0xBA6E,0x69c3}, // <CJK>
  {0xBA6F,0x69a3}, // <CJK>
  {0xBA70,0x6b49}, // <CJK>
  {0xBA71,0x6b4c}, // <CJK>
  {0xBA72,0x6c33}, // <CJK>
  {0xBA73,0x6f33}, // <CJK>
  {0xBA74,0x6f14}, // <CJK>
  {0xBA75,0x6efe}, // <CJK>
  {0xBA76,0x6f13}, // <CJK>
  {0xBA77,0x6ef4}, // <CJK>
  {0xBA78,0x6f29}, // <CJK>
  {0xBA79,0x6f3e}, // <CJK>
  {0xBA7A,0x6f20}, // <CJK>
  {0xBA7B,0x6f2c}, // <CJK>
  {0xBA7C,0x6f0f}, // <CJK>
  {0xBA7D,0x6f02}, // <CJK>
  {0xBA7E,0x6f22}, // <CJK>
  {0xBAA1,0x6eff}, // <CJK>
  {0xBAA2,0x6eef}, // <CJK>
  {0xBAA3,0x6f06}, // <CJK>
  {0xBAA4,0x6f31}, // <CJK>
  {0xBAA5,0x6f38}, // <CJK>
  {0xBAA6,0x6f32}, // <CJK>
  {0xBAA7,0x6f23}, // <CJK>
  {0xBAA8,0x6f15}, // <CJK>
  {0xBAA9,0x6f2b}, // <CJK>
  {0xBAAA,0x6f2f}, // <CJK>
  {0xBAAB,0x6f88}, // <CJK>
  {0xBAAC,0x6f2a}, // <CJK>
  {0xBAAD,0x6eec}, // <CJK>
  {0xBAAE,0x6f01}, // <CJK>
  {0xBAAF,0x6ef2}, // <CJK>
  {0xBAB0,0x6ecc}, // <CJK>
  {0xBAB1,0x6ef7}, // <CJK>
  {0xBAB2,0x7194}, // <CJK>
  {0xBAB3,0x7199}, // <CJK>
  {0xBAB4,0x717d}, // <CJK>
  {0xBAB5,0x718a}, // <CJK>
  {0xBAB6,0x7184}, // <CJK>
  {0xBAB7,0x7192}, // <CJK>
  {0xBAB8,0x723e}, // <CJK>
  {0xBAB9,0x7292}, // <CJK>
  {0xBABA,0x7296}, // <CJK>
  {0xBABB,0x7344}, // <CJK>
  {0xBABC,0x7350}, // <CJK>
  {0xBABD,0x7464}, // <CJK>
  {0xBABE,0x7463}, // <CJK>
  {0xBABF,0x746a}, // <CJK>
  {0xBAC0,0x7470}, // <CJK>
  {0xBAC1,0x746d}, // <CJK>
  {0xBAC2,0x7504}, // <CJK>
  {0xBAC3,0x7591}, // <CJK>
  {0xBAC4,0x7627}, // <CJK>
  {0xBAC5,0x760d}, // <CJK>
  {0xBAC6,0x760b}, // <CJK>
  {0xBAC7,0x7609}, // <CJK>
  {0xBAC8,0x7613}, // <CJK>
  {0xBAC9,0x76e1}, // <CJK>
  {0xBACA,0x76e3}, // <CJK>
  {0xBACB,0x7784}, // <CJK>
  {0xBACC,0x777d}, // <CJK>
  {0xBACD,0x777f}, // <CJK>
  {0xBACE,0x7761}, // <CJK>
  {0xBACF,0x78c1}, // <CJK>
  {0xBAD0,0x789f}, // <CJK>
  {0xBAD1,0x78a7}, // <CJK>
  {0xBAD2,0x78b3}, // <CJK>
  {0xBAD3,0x78a9}, // <CJK>
  {0xBAD4,0x78a3}, // <CJK>
  {0xBAD5,0x798e}, // <CJK>
  {0xBAD6,0x798f}, // <CJK>
  {0xBAD7,0x798d}, // <CJK>
  {0xBAD8,0x7a2e}, // <CJK>
  {0xBAD9,0x7a31}, // <CJK>
  {0xBADA,0x7aaa}, // <CJK>
  {0xBADB,0x7aa9}, // <CJK>
  {0xBADC,0x7aed}, // <CJK>
  {0xBADD,0x7aef}, // <CJK>
  {0xBADE,0x7ba1}, // <CJK>
  {0xBADF,0x7b95}, // <CJK>
  {0xBAE0,0x7b8b}, // <CJK>
  {0xBAE1,0x7b75}, // <CJK>
  {0xBAE2,0x7b97}, // <CJK>
  {0xBAE3,0x7b9d}, // <CJK>
  {0xBAE4,0x7b94}, // <CJK>
  {0xBAE5,0x7b8f}, // <CJK>
  {0xBAE6,0x7bb8}, // <CJK>
  {0xBAE7,0x7b87}, // <CJK>
  {0xBAE8,0x7b84}, // <CJK>
  {0xBAE9,0x7cb9}, // <CJK>
  {0xBAEA,0x7cbd}, // <CJK>
  {0xBAEB,0x7cbe}, // <CJK>
  {0xBAEC,0x7dbb}, // <CJK>
  {0xBAED,0x7db0}, // <CJK>
  {0xBAEE,0x7d9c}, // <CJK>
  {0xBAEF,0x7dbd}, // <CJK>
  {0xBAF0,0x7dbe}, // <CJK>
  {0xBAF1,0x7da0}, // <CJK>
  {0xBAF2,0x7dca}, // <CJK>
  {0xBAF3,0x7db4}, // <CJK>
  {0xBAF4,0x7db2}, // <CJK>
  {0xBAF5,0x7db1}, // <CJK>
  {0xBAF6,0x7dba}, // <CJK>
  {0xBAF7,0x7da2}, // <CJK>
  {0xBAF8,0x7dbf}, // <CJK>
  {0xBAF9,0x7db5}, // <CJK>
  {0xBAFA,0x7db8}, // <CJK>
  {0xBAFB,0x7dad}, // <CJK>
  {0xBAFC,0x7dd2}, // <CJK>
  {0xBAFD,0x7dc7}, // <CJK>
  {0xBAFE,0x7dac}, // <CJK>
  {0xBB40,0x7f70}, // <CJK>
  {0xBB41,0x7fe0}, // <CJK>
  {0xBB42,0x7fe1}, // <CJK>
  {0xBB43,0x7fdf}, // <CJK>
  {0xBB44,0x805e}, // <CJK>
  {0xBB45,0x805a}, // <CJK>
  {0xBB46,0x8087}, // <CJK>
  {0xBB47,0x8150}, // <CJK>
  {0xBB48,0x8180}, // <CJK>
  {0xBB49,0x818f}, // <CJK>
  {0xBB4A,0x8188}, // <CJK>
  {0xBB4B,0x818a}, // <CJK>
  {0xBB4C,0x817f}, // <CJK>
  {0xBB4D,0x8182}, // <CJK>
  {0xBB4E,0x81e7}, // <CJK>
  {0xBB4F,0x81fa}, // <CJK>
  {0xBB50,0x8207}, // <CJK>
  {0xBB51,0x8214}, // <CJK>
  {0xBB52,0x821e}, // <CJK>
  {0xBB53,0x824b}, // <CJK>
  {0xBB54,0x84c9}, // <CJK>
  {0xBB55,0x84bf}, // <CJK>
  {0xBB56,0x84c6}, // <CJK>
  {0xBB57,0x84c4}, // <CJK>
  {0xBB58,0x8499}, // <CJK>
  {0xBB59,0x849e}, // <CJK>
  {0xBB5A,0x84b2}, // <CJK>
  {0xBB5B,0x849c}, // <CJK>
  {0xBB5C,0x84cb}, // <CJK>
  {0xBB5D,0x84b8}, // <CJK>
  {0xBB5E,0x84c0}, // <CJK>
  {0xBB5F,0x84d3}, // <CJK>
  {0xBB60,0x8490}, // <CJK>
  {0xBB61,0x84bc}, // <CJK>
  {0xBB62,0x84d1}, // <CJK>
  {0xBB63,0x84ca}, // <CJK>
  {0xBB64,0x873f}, // <CJK>
  {0xBB65,0x871c}, // <CJK>
  {0xBB66,0x873b}, // <CJK>
  {0xBB67,0x8722}, // <CJK>
  {0xBB68,0x8725}, // <CJK>
  {0xBB69,0x8734}, // <CJK>
  {0xBB6A,0x8718}, // <CJK>
  {0xBB6B,0x8755}, // <CJK>
  {0xBB6C,0x8737}, // <CJK>
  {0xBB6D,0x8729}, // <CJK>
  {0xBB6E,0x88f3}, // <CJK>
  {0xBB6F,0x8902}, // <CJK>
  {0xBB70,0x88f4}, // <CJK>
  {0xBB71,0x88f9}, // <CJK>
  {0xBB72,0x88f8}, // <CJK>
  {0xBB73,0x88fd}, // <CJK>
  {0xBB74,0x88e8}, // <CJK>
  {0xBB75,0x891a}, // <CJK>
  {0xBB76,0x88ef}, // <CJK>
  {0xBB77,0x8aa6}, // <CJK>
  {0xBB78,0x8a8c}, // <CJK>
  {0xBB79,0x8a9e}, // <CJK>
  {0xBB7A,0x8aa3}, // <CJK>
  {0xBB7B,0x8a8d}, // <CJK>
  {0xBB7C,0x8aa1}, // <CJK>
  {0xBB7D,0x8a93}, // <CJK>
  {0xBB7E,0x8aa4}, // <CJK>
  {0xBBA1,0x8aaa}, // <CJK>
  {0xBBA2,0x8aa5}, // <CJK>
  {0xBBA3,0x8aa8}, // <CJK>
  {0xBBA4,0x8a98}, // <CJK>
  {0xBBA5,0x8a91}, // <CJK>
  {0xBBA6,0x8a9a}, // <CJK>
  {0xBBA7,0x8aa7}, // <CJK>
  {0xBBA8,0x8c6a}, // <CJK>
  {0xBBA9,0x8c8d}, // <CJK>
  {0xBBAA,0x8c8c}, // <CJK>
  {0xBBAB,0x8cd3}, // <CJK>
  {0xBBAC,0x8cd1}, // <CJK>
  {0xBBAD,0x8cd2}, // <CJK>
  {0xBBAE,0x8d6b}, // <CJK>
  {0xBBAF,0x8d99}, // <CJK>
  {0xBBB0,0x8d95}, // <CJK>
  {0xBBB1,0x8dfc}, // <CJK>
  {0xBBB2,0x8f14}, // <CJK>
  {0xBBB3,0x8f12}, // <CJK>
  {0xBBB4,0x8f15}, // <CJK>
  {0xBBB5,0x8f13}, // <CJK>
  {0xBBB6,0x8fa3}, // <CJK>
  {0xBBB7,0x9060}, // <CJK>
  {0xBBB8,0x9058}, // <CJK>
  {0xBBB9,0x905c}, // <CJK>
  {0xBBBA,0x9063}, // <CJK>
  {0xBBBB,0x9059}, // <CJK>
  {0xBBBC,0x905e}, // <CJK>
  {0xBBBD,0x9062}, // <CJK>
  {0xBBBE,0x905d}, // <CJK>
  {0xBBBF,0x905b}, // <CJK>
  {0xBBC0,0x9119}, // <CJK>
  {0xBBC1,0x9118}, // <CJK>
  {0xBBC2,0x911e}, // <CJK>
  {0xBBC3,0x9175}, // <CJK>
  {0xBBC4,0x9178}, // <CJK>
  {0xBBC5,0x9177}, // <CJK>
  {0xBBC6,0x9174}, // <CJK>
  {0xBBC7,0x9278}, // <CJK>
  {0xBBC8,0x9280}, // <CJK>
  {0xBBC9,0x9285}, // <CJK>
  {0xBBCA,0x9298}, // <CJK>
  {0xBBCB,0x9296}, // <CJK>
  {0xBBCC,0x927b}, // <CJK>
  {0xBBCD,0x9293}, // <CJK>
  {0xBBCE,0x929c}, // <CJK>
  {0xBBCF,0x92a8}, // <CJK>
  {0xBBD0,0x927c}, // <CJK>
  {0xBBD1,0x9291}, // <CJK>
  {0xBBD2,0x95a1}, // <CJK>
  {0xBBD3,0x95a8}, // <CJK>
  {0xBBD4,0x95a9}, // <CJK>
  {0xBBD5,0x95a3}, // <CJK>
  {0xBBD6,0x95a5}, // <CJK>
  {0xBBD7,0x95a4}, // <CJK>
  {0xBBD8,0x9699}, // <CJK>
  {0xBBD9,0x969c}, // <CJK>
  {0xBBDA,0x969b}, // <CJK>
  {0xBBDB,0x96cc}, // <CJK>
  {0xBBDC,0x96d2}, // <CJK>
  {0xBBDD,0x9700}, // <CJK>
  {0xBBDE,0x977c}, // <CJK>
  {0xBBDF,0x9785}, // <CJK>
  {0xBBE0,0x97f6}, // <CJK>
  {0xBBE1,0x9817}, // <CJK>
  {0xBBE2,0x9818}, // <CJK>
  {0xBBE3,0x98af}, // <CJK>
  {0xBBE4,0x98b1}, // <CJK>
  {0xBBE5,0x9903}, // <CJK>
  {0xBBE6,0x9905}, // <CJK>
  {0xBBE7,0x990c}, // <CJK>
  {0xBBE8,0x9909}, // <CJK>
  {0xBBE9,0x99c1}, // <CJK>
  {0xBBEA,0x9aaf}, // <CJK>
  {0xBBEB,0x9ab0}, // <CJK>
  {0xBBEC,0x9ae6}, // <CJK>
  {0xBBED,0x9b41}, // <CJK>
  {0xBBEE,0x9b42}, // <CJK>
  {0xBBEF,0x9cf4}, // <CJK>
  {0xBBF0,0x9cf6}, // <CJK>
  {0xBBF1,0x9cf3}, // <CJK>
  {0xBBF2,0x9ebc}, // <CJK>
  {0xBBF3,0x9f3b}, // <CJK>
  {0xBBF4,0x9f4a}, // <CJK>
  {0xBBF5,0x5104}, // <CJK>
  {0xBBF6,0x5100}, // <CJK>
  {0xBBF7,0x50fb}, // <CJK>
  {0xBBF8,0x50f5}, // <CJK>
  {0xBBF9,0x50f9}, // <CJK>
  {0xBBFA,0x5102}, // <CJK>
  {0xBBFB,0x5108}, // <CJK>
  {0xBBFC,0x5109}, // <CJK>
  {0xBBFD,0x5105}, // <CJK>
  {0xBBFE,0x51dc}, // <CJK>
  {0xBC40,0x5287}, // <CJK>
  {0xBC41,0x5288}, // <CJK>
  {0xBC42,0x5289}, // <CJK>
  {0xBC43,0x528d}, // <CJK>
  {0xBC44,0x528a}, // <CJK>
  {0xBC45,0x52f0}, // <CJK>
  {0xBC46,0x53b2}, // <CJK>
  {0xBC47,0x562e}, // <CJK>
  {0xBC48,0x563b}, // <CJK>
  {0xBC49,0x5639}, // <CJK>
  {0xBC4A,0x5632}, // <CJK>
  {0xBC4B,0x563f}, // <CJK>
  {0xBC4C,0x5634}, // <CJK>
  {0xBC4D,0x5629}, // <CJK>
  {0xBC4E,0x5653}, // <CJK>
  {0xBC4F,0x564e}, // <CJK>
  {0xBC50,0x5657}, // <CJK>
  {0xBC51,0x5674}, // <CJK>
  {0xBC52,0x5636}, // <CJK>
  {0xBC53,0x562f}, // <CJK>
  {0xBC54,0x5630}, // <CJK>
  {0xBC55,0x5880}, // <CJK>
  {0xBC56,0x589f}, // <CJK>
  {0xBC57,0x589e}, // <CJK>
  {0xBC58,0x58b3}, // <CJK>
  {0xBC59,0x589c}, // <CJK>
  {0xBC5A,0x58ae}, // <CJK>
  {0xBC5B,0x58a9}, // <CJK>
  {0xBC5C,0x58a6}, // <CJK>
  {0xBC5D,0x596d}, // <CJK>
  {0xBC5E,0x5b09}, // <CJK>
  {0xBC5F,0x5afb}, // <CJK>
  {0xBC60,0x5b0b}, // <CJK>
  {0xBC61,0x5af5}, // <CJK>
  {0xBC62,0x5b0c}, // <CJK>
  {0xBC63,0x5b08}, // <CJK>
  {0xBC64,0x5bee}, // <CJK>
  {0xBC65,0x5bec}, // <CJK>
  {0xBC66,0x5be9}, // <CJK>
  {0xBC67,0x5beb}, // <CJK>
  {0xBC68,0x5c64}, // <CJK>
  {0xBC69,0x5c65}, // <CJK>
  {0xBC6A,0x5d9d}, // <CJK>
  {0xBC6B,0x5d94}, // <CJK>
  {0xBC6C,0x5e62}, // <CJK>
  {0xBC6D,0x5e5f}, // <CJK>
  {0xBC6E,0x5e61}, // <CJK>
  {0xBC6F,0x5ee2}, // <CJK>
  {0xBC70,0x5eda}, // <CJK>
  {0xBC71,0x5edf}, // <CJK>
  {0xBC72,0x5edd}, // <CJK>
  {0xBC73,0x5ee3}, // <CJK>
  {0xBC74,0x5ee0}, // <CJK>
  {0xBC75,0x5f48}, // <CJK>
  {0xBC76,0x5f71}, // <CJK>
  {0xBC77,0x5fb7}, // <CJK>
  {0xBC78,0x5fb5}, // <CJK>
  {0xBC79,0x6176}, // <CJK>
  {0xBC7A,0x6167}, // <CJK>
  {0xBC7B,0x616e}, // <CJK>
  {0xBC7C,0x615d}, // <CJK>
  {0xBC7D,0x6155}, // <CJK>
  {0xBC7E,0x6182}, // <CJK>
  {0xBCA1,0x617c}, // <CJK>
  {0xBCA2,0x6170}, // <CJK>
  {0xBCA3,0x616b}, // <CJK>
  {0xBCA4,0x617e}, // <CJK>
  {0xBCA5,0x61a7}, // <CJK>
  {0xBCA6,0x6190}, // <CJK>
  {0xBCA7,0x61ab}, // <CJK>
  {0xBCA8,0x618e}, // <CJK>
  {0xBCA9,0x61ac}, // <CJK>
  {0xBCAA,0x619a}, // <CJK>
  {0xBCAB,0x61a4}, // <CJK>
  {0xBCAC,0x6194}, // <CJK>
  {0xBCAD,0x61ae}, // <CJK>
  {0xBCAE,0x622e}, // <CJK>
  {0xBCAF,0x6469}, // <CJK>
  {0xBCB0,0x646f}, // <CJK>
  {0xBCB1,0x6479}, // <CJK>
  {0xBCB2,0x649e}, // <CJK>
  {0xBCB3,0x64b2}, // <CJK>
  {0xBCB4,0x6488}, // <CJK>
  {0xBCB5,0x6490}, // <CJK>
  {0xBCB6,0x64b0}, // <CJK>
  {0xBCB7,0x64a5}, // <CJK>
  {0xBCB8,0x6493}, // <CJK>
  {0xBCB9,0x6495}, // <CJK>
  {0xBCBA,0x64a9}, // <CJK>
  {0xBCBB,0x6492}, // <CJK>
  {0xBCBC,0x64ae}, // <CJK>
  {0xBCBD,0x64ad}, // <CJK>
  {0xBCBE,0x64ab}, // <CJK>
  {0xBCBF,0x649a}, // <CJK>
  {0xBCC0,0x64ac}, // <CJK>
  {0xBCC1,0x6499}, // <CJK>
  {0xBCC2,0x64a2}, // <CJK>
  {0xBCC3,0x64b3}, // <CJK>
  {0xBCC4,0x6575}, // <CJK>
  {0xBCC5,0x6577}, // <CJK>
  {0xBCC6,0x6578}, // <CJK>
  {0xBCC7,0x66ae}, // <CJK>
  {0xBCC8,0x66ab}, // <CJK>
  {0xBCC9,0x66b4}, // <CJK>
  {0xBCCA,0x66b1}, // <CJK>
  {0xBCCB,0x6a23}, // <CJK>
  {0xBCCC,0x6a1f}, // <CJK>
  {0xBCCD,0x69e8}, // <CJK>
  {0xBCCE,0x6a01}, // <CJK>
  {0xBCCF,0x6a1e}, // <CJK>
  {0xBCD0,0x6a19}, // <CJK>
  {0xBCD1,0x69fd}, // <CJK>
  {0xBCD2,0x6a21}, // <CJK>
  {0xBCD3,0x6a13}, // <CJK>
  {0xBCD4,0x6a0a}, // <CJK>
  {0xBCD5,0x69f3}, // <CJK>
  {0xBCD6,0x6a02}, // <CJK>
  {0xBCD7,0x6a05}, // <CJK>
  {0xBCD8,0x69ed}, // <CJK>
  {0xBCD9,0x6a11}, // <CJK>
  {0xBCDA,0x6b50}, // <CJK>
  {0xBCDB,0x6b4e}, // <CJK>
  {0xBCDC,0x6ba4}, // <CJK>
  {0xBCDD,0x6bc5}, // <CJK>
  {0xBCDE,0x6bc6}, // <CJK>
  {0xBCDF,0x6f3f}, // <CJK>
  {0xBCE0,0x6f7c}, // <CJK>
  {0xBCE1,0x6f84}, // <CJK>
  {0xBCE2,0x6f51}, // <CJK>
  {0xBCE3,0x6f66}, // <CJK>
  {0xBCE4,0x6f54}, // <CJK>
  {0xBCE5,0x6f86}, // <CJK>
  {0xBCE6,0x6f6d}, // <CJK>
  {0xBCE7,0x6f5b}, // <CJK>
  {0xBCE8,0x6f78}, // <CJK>
  {0xBCE9,0x6f6e}, // <CJK>
  {0xBCEA,0x6f8e}, // <CJK>
  {0xBCEB,0x6f7a}, // <CJK>
  {0xBCEC,0x6f70}, // <CJK>
  {0xBCED,0x6f64}, // <CJK>
  {0xBCEE,0x6f97}, // <CJK>
  {0xBCEF,0x6f58}, // <CJK>
  {0xBCF0,0x6ed5}, // <CJK>
  {0xBCF1,0x6f6f}, // <CJK>
  {0xBCF2,0x6f60}, // <CJK>
  {0xBCF3,0x6f5f}, // <CJK>
  {0xBCF4,0x719f}, // <CJK>
  {0xBCF5,0x71ac}, // <CJK>
  {0xBCF6,0x71b1}, // <CJK>
  {0xBCF7,0x71a8}, // <CJK>
  {0xBCF8,0x7256}, // <CJK>
  {0xBCF9,0x729b}, // <CJK>
  {0xBCFA,0x734e}, // <CJK>
  {0xBCFB,0x7357}, // <CJK>
  {0xBCFC,0x7469}, // <CJK>
  {0xBCFD,0x748b}, // <CJK>
  {0xBCFE,0x7483}, // <CJK>
  {0xBD40,0x747e}, // <CJK>
  {0xBD41,0x7480}, // <CJK>
  {0xBD42,0x757f}, // <CJK>
  {0xBD43,0x7620}, // <CJK>
  {0xBD44,0x7629}, // <CJK>
  {0xBD45,0x761f}, // <CJK>
  {0xBD46,0x7624}, // <CJK>
  {0xBD47,0x7626}, // <CJK>
  {0xBD48,0x7621}, // <CJK>
  {0xBD49,0x7622}, // <CJK>
  {0xBD4A,0x769a}, // <CJK>
  {0xBD4B,0x76ba}, // <CJK>
  {0xBD4C,0x76e4}, // <CJK>
  {0xBD4D,0x778e}, // <CJK>
  {0xBD4E,0x7787}, // <CJK>
  {0xBD4F,0x778c}, // <CJK>
  {0xBD50,0x7791}, // <CJK>
  {0xBD51,0x778b}, // <CJK>
  {0xBD52,0x78cb}, // <CJK>
  {0xBD53,0x78c5}, // <CJK>
  {0xBD54,0x78ba}, // <CJK>
  {0xBD55,0x78ca}, // <CJK>
  {0xBD56,0x78be}, // <CJK>
  {0xBD57,0x78d5}, // <CJK>
  {0xBD58,0x78bc}, // <CJK>
  {0xBD59,0x78d0}, // <CJK>
  {0xBD5A,0x7a3f}, // <CJK>
  {0xBD5B,0x7a3c}, // <CJK>
  {0xBD5C,0x7a40}, // <CJK>
  {0xBD5D,0x7a3d}, // <CJK>
  {0xBD5E,0x7a37}, // <CJK>
  {0xBD5F,0x7a3b}, // <CJK>
  {0xBD60,0x7aaf}, // <CJK>
  {0xBD61,0x7aae}, // <CJK>
  {0xBD62,0x7bad}, // <CJK>
  {0xBD63,0x7bb1}, // <CJK>
  {0xBD64,0x7bc4}, // <CJK>
  {0xBD65,0x7bb4}, // <CJK>
  {0xBD66,0x7bc6}, // <CJK>
  {0xBD67,0x7bc7}, // <CJK>
  {0xBD68,0x7bc1}, // <CJK>
  {0xBD69,0x7ba0}, // <CJK>
  {0xBD6A,0x7bcc}, // <CJK>
  {0xBD6B,0x7cca}, // <CJK>
  {0xBD6C,0x7de0}, // <CJK>
  {0xBD6D,0x7df4}, // <CJK>
  {0xBD6E,0x7def}, // <CJK>
  {0xBD6F,0x7dfb}, // <CJK>
  {0xBD70,0x7dd8}, // <CJK>
  {0xBD71,0x7dec}, // <CJK>
  {0xBD72,0x7ddd}, // <CJK>
  {0xBD73,0x7de8}, // <CJK>
  {0xBD74,0x7de3}, // <CJK>
  {0xBD75,0x7dda}, // <CJK>
  {0xBD76,0x7dde}, // <CJK>
  {0xBD77,0x7de9}, // <CJK>
  {0xBD78,0x7d9e}, // <CJK>
  {0xBD79,0x7dd9}, // <CJK>
  {0xBD7A,0x7df2}, // <CJK>
  {0xBD7B,0x7df9}, // <CJK>
  {0xBD7C,0x7f75}, // <CJK>
  {0xBD7D,0x7f77}, // <CJK>
  {0xBD7E,0x7faf}, // <CJK>
  {0xBDA1,0x7fe9}, // <CJK>
  {0xBDA2,0x8026}, // <CJK>
  {0xBDA3,0x819b}, // <CJK>
  {0xBDA4,0x819c}, // <CJK>
  {0xBDA5,0x819d}, // <CJK>
  {0xBDA6,0x81a0}, // <CJK>
  {0xBDA7,0x819a}, // <CJK>
  {0xBDA8,0x8198}, // <CJK>
  {0xBDA9,0x8517}, // <CJK>
  {0xBDAA,0x853d}, // <CJK>
  {0xBDAB,0x851a}, // <CJK>
  {0xBDAC,0x84ee}, // <CJK>
  {0xBDAD,0x852c}, // <CJK>
  {0xBDAE,0x852d}, // <CJK>
  {0xBDAF,0x8513}, // <CJK>
  {0xBDB0,0x8511}, // <CJK>
  {0xBDB1,0x8523}, // <CJK>
  {0xBDB2,0x8521}, // <CJK>
  {0xBDB3,0x8514}, // <CJK>
  {0xBDB4,0x84ec}, // <CJK>
  {0xBDB5,0x8525}, // <CJK>
  {0xBDB6,0x84ff}, // <CJK>
  {0xBDB7,0x8506}, // <CJK>
  {0xBDB8,0x8782}, // <CJK>
  {0xBDB9,0x8774}, // <CJK>
  {0xBDBA,0x8776}, // <CJK>
  {0xBDBB,0x8760}, // <CJK>
  {0xBDBC,0x8766}, // <CJK>
  {0xBDBD,0x8778}, // <CJK>
  {0xBDBE,0x8768}, // <CJK>
  {0xBDBF,0x8759}, // <CJK>
  {0xBDC0,0x8757}, // <CJK>
  {0xBDC1,0x874c}, // <CJK>
  {0xBDC2,0x8753}, // <CJK>
  {0xBDC3,0x885b}, // <CJK>
  {0xBDC4,0x885d}, // <CJK>
  {0xBDC5,0x8910}, // <CJK>
  {0xBDC6,0x8907}, // <CJK>
  {0xBDC7,0x8912}, // <CJK>
  {0xBDC8,0x8913}, // <CJK>
  {0xBDC9,0x8915}, // <CJK>
  {0xBDCA,0x890a}, // <CJK>
  {0xBDCB,0x8abc}, // <CJK>
  {0xBDCC,0x8ad2}, // <CJK>
  {0xBDCD,0x8ac7}, // <CJK>
  {0xBDCE,0x8ac4}, // <CJK>
  {0xBDCF,0x8a95}, // <CJK>
  {0xBDD0,0x8acb}, // <CJK>
  {0xBDD1,0x8af8}, // <CJK>
  {0xBDD2,0x8ab2}, // <CJK>
  {0xBDD3,0x8ac9}, // <CJK>
  {0xBDD4,0x8ac2}, // <CJK>
  {0xBDD5,0x8abf}, // <CJK>
  {0xBDD6,0x8ab0}, // <CJK>
  {0xBDD7,0x8ad6}, // <CJK>
  {0xBDD8,0x8acd}, // <CJK>
  {0xBDD9,0x8ab6}, // <CJK>
  {0xBDDA,0x8ab9}, // <CJK>
  {0xBDDB,0x8adb}, // <CJK>
  {0xBDDC,0x8c4c}, // <CJK>
  {0xBDDD,0x8c4e}, // <CJK>
  {0xBDDE,0x8c6c}, // <CJK>
  {0xBDDF,0x8ce0}, // <CJK>
  {0xBDE0,0x8cde}, // <CJK>
  {0xBDE1,0x8ce6}, // <CJK>
  {0xBDE2,0x8ce4}, // <CJK>
  {0xBDE3,0x8cec}, // <CJK>
  {0xBDE4,0x8ced}, // <CJK>
  {0xBDE5,0x8ce2}, // <CJK>
  {0xBDE6,0x8ce3}, // <CJK>
  {0xBDE7,0x8cdc}, // <CJK>
  {0xBDE8,0x8cea}, // <CJK>
  {0xBDE9,0x8ce1}, // <CJK>
  {0xBDEA,0x8d6d}, // <CJK>
  {0xBDEB,0x8d9f}, // <CJK>
  {0xBDEC,0x8da3}, // <CJK>
  {0xBDED,0x8e2b}, // <CJK>
  {0xBDEE,0x8e10}, // <CJK>
  {0xBDEF,0x8e1d}, // <CJK>
  {0xBDF0,0x8e22}, // <CJK>
  {0xBDF1,0x8e0f}, // <CJK>
  {0xBDF2,0x8e29}, // <CJK>
  {0xBDF3,0x8e1f}, // <CJK>
  {0xBDF4,0x8e21}, // <CJK>
  {0xBDF5,0x8e1e}, // <CJK>
  {0xBDF6,0x8eba}, // <CJK>
  {0xBDF7,0x8f1d}, // <CJK>
  {0xBDF8,0x8f1b}, // <CJK>
  {0xBDF9,0x8f1f}, // <CJK>
  {0xBDFA,0x8f29}, // <CJK>
  {0xBDFB,0x8f26}, // <CJK>
  {0xBDFC,0x8f2a}, // <CJK>
  {0xBDFD,0x8f1c}, // <CJK>
  {0xBDFE,0x8f1e}, // <CJK>
  {0xBE40,0x8f25}, // <CJK>
  {0xBE41,0x9069}, // <CJK>
  {0xBE42,0x906e}, // <CJK>
  {0xBE43,0x9068}, // <CJK>
  {0xBE44,0x906d}, // <CJK>
  {0xBE45,0x9077}, // <CJK>
  {0xBE46,0x9130}, // <CJK>
  {0xBE47,0x912d}, // <CJK>
  {0xBE48,0x9127}, // <CJK>
  {0xBE49,0x9131}, // <CJK>
  {0xBE4A,0x9187}, // <CJK>
  {0xBE4B,0x9189}, // <CJK>
  {0xBE4C,0x918b}, // <CJK>
  {0xBE4D,0x9183}, // <CJK>
  {0xBE4E,0x92c5}, // <CJK>
  {0xBE4F,0x92bb}, // <CJK>
  {0xBE50,0x92b7}, // <CJK>
  {0xBE51,0x92ea}, // <CJK>
  {0xBE52,0x92ac}, // <CJK>
  {0xBE53,0x92e4}, // <CJK>
  {0xBE54,0x92c1}, // <CJK>
  {0xBE55,0x92b3}, // <CJK>
  {0xBE56,0x92bc}, // <CJK>
  {0xBE57,0x92d2}, // <CJK>
  {0xBE58,0x92c7}, // <CJK>
  {0xBE59,0x92f0}, // <CJK>
  {0xBE5A,0x92b2}, // <CJK>
  {0xBE5B,0x95ad}, // <CJK>
  {0xBE5C,0x95b1}, // <CJK>
  {0xBE5D,0x9704}, // <CJK>
  {0xBE5E,0x9706}, // <CJK>
  {0xBE5F,0x9707}, // <CJK>
  {0xBE60,0x9709}, // <CJK>
  {0xBE61,0x9760}, // <CJK>
  {0xBE62,0x978d}, // <CJK>
  {0xBE63,0x978b}, // <CJK>
  {0xBE64,0x978f}, // <CJK>
  {0xBE65,0x9821}, // <CJK>
  {0xBE66,0x982b}, // <CJK>
  {0xBE67,0x981c}, // <CJK>
  {0xBE68,0x98b3}, // <CJK>
  {0xBE69,0x990a}, // <CJK>
  {0xBE6A,0x9913}, // <CJK>
  {0xBE6B,0x9912}, // <CJK>
  {0xBE6C,0x9918}, // <CJK>
  {0xBE6D,0x99dd}, // <CJK>
  {0xBE6E,0x99d0}, // <CJK>
  {0xBE6F,0x99df}, // <CJK>
  {0xBE70,0x99db}, // <CJK>
  {0xBE71,0x99d1}, // <CJK>
  {0xBE72,0x99d5}, // <CJK>
  {0xBE73,0x99d2}, // <CJK>
  {0xBE74,0x99d9}, // <CJK>
  {0xBE75,0x9ab7}, // <CJK>
  {0xBE76,0x9aee}, // <CJK>
  {0xBE77,0x9aef}, // <CJK>
  {0xBE78,0x9b27}, // <CJK>
  {0xBE79,0x9b45}, // <CJK>
  {0xBE7A,0x9b44}, // <CJK>
  {0xBE7B,0x9b77}, // <CJK>
  {0xBE7C,0x9b6f}, // <CJK>
  {0xBE7D,0x9d06}, // <CJK>
  {0xBE7E,0x9d09}, // <CJK>
  {0xBEA1,0x9d03}, // <CJK>
  {0xBEA2,0x9ea9}, // <CJK>
  {0xBEA3,0x9ebe}, // <CJK>
  {0xBEA4,0x9ece}, // <CJK>
  {0xBEA5,0x58a8}, // <CJK>
  {0xBEA6,0x9f52}, // <CJK>
  {0xBEA7,0x5112}, // <CJK>
  {0xBEA8,0x5118}, // <CJK>
  {0xBEA9,0x5114}, // <CJK>
  {0xBEAA,0x5110}, // <CJK>
  {0xBEAB,0x5115}, // <CJK>
  {0xBEAC,0x5180}, // <CJK>
  {0xBEAD,0x51aa}, // <CJK>
  {0xBEAE,0x51dd}, // <CJK>
  {0xBEAF,0x5291}, // <CJK>
  {0xBEB0,0x5293}, // <CJK>
  {0xBEB1,0x52f3}, // <CJK>
  {0xBEB2,0x5659}, // <CJK>
  {0xBEB3,0x566b}, // <CJK>
  {0xBEB4,0x5679}, // <CJK>
  {0xBEB5,0x5669}, // <CJK>
  {0xBEB6,0x5664}, // <CJK>
  {0xBEB7,0x5678}, // <CJK>
  {0xBEB8,0x566a}, // <CJK>
  {0xBEB9,0x5668}, // <CJK>
  {0xBEBA,0x5665}, // <CJK>
  {0xBEBB,0x5671}, // <CJK>
  {0xBEBC,0x566f}, // <CJK>
  {0xBEBD,0x566c}, // <CJK>
  {0xBEBE,0x5662}, // <CJK>
  {0xBEBF,0x5676}, // <CJK>
  {0xBEC0,0x58c1}, // <CJK>
  {0xBEC1,0x58be}, // <CJK>
  {0xBEC2,0x58c7}, // <CJK>
  {0xBEC3,0x58c5}, // <CJK>
  {0xBEC4,0x596e}, // <CJK>
  {0xBEC5,0x5b1d}, // <CJK>
  {0xBEC6,0x5b34}, // <CJK>
  {0xBEC7,0x5b78}, // <CJK>
  {0xBEC8,0x5bf0}, // <CJK>
  {0xBEC9,0x5c0e}, // <CJK>
  {0xBECA,0x5f4a}, // <CJK>
  {0xBECB,0x61b2}, // <CJK>
  {0xBECC,0x6191}, // <CJK>
  {0xBECD,0x61a9}, // <CJK>
  {0xBECE,0x618a}, // <CJK>
  {0xBECF,0x61cd}, // <CJK>
  {0xBED0,0x61b6}, // <CJK>
  {0xBED1,0x61be}, // <CJK>
  {0xBED2,0x61ca}, // <CJK>
  {0xBED3,0x61c8}, // <CJK>
  {0xBED4,0x6230}, // <CJK>
  {0xBED5,0x64c5}, // <CJK>
  {0xBED6,0x64c1}, // <CJK>
  {0xBED7,0x64cb}, // <CJK>
  {0xBED8,0x64bb}, // <CJK>
  {0xBED9,0x64bc}, // <CJK>
  {0xBEDA,0x64da}, // <CJK>
  {0xBEDB,0x64c4}, // <CJK>
  {0xBEDC,0x64c7}, // <CJK>
  {0xBEDD,0x64c2}, // <CJK>
  {0xBEDE,0x64cd}, // <CJK>
  {0xBEDF,0x64bf}, // <CJK>
  {0xBEE0,0x64d2}, // <CJK>
  {0xBEE1,0x64d4}, // <CJK>
  {0xBEE2,0x64be}, // <CJK>
  {0xBEE3,0x6574}, // <CJK>
  {0xBEE4,0x66c6}, // <CJK>
  {0xBEE5,0x66c9}, // <CJK>
  {0xBEE6,0x66b9}, // <CJK>
  {0xBEE7,0x66c4}, // <CJK>
  {0xBEE8,0x66c7}, // <CJK>
  {0xBEE9,0x66b8}, // <CJK>
  {0xBEEA,0x6a3d}, // <CJK>
  {0xBEEB,0x6a38}, // <CJK>
  {0xBEEC,0x6a3a}, // <CJK>
  {0xBEED,0x6a59}, // <CJK>
  {0xBEEE,0x6a6b}, // <CJK>
  {0xBEEF,0x6a58}, // <CJK>
  {0xBEF0,0x6a39}, // <CJK>
  {0xBEF1,0x6a44}, // <CJK>
  {0xBEF2,0x6a62}, // <CJK>
  {0xBEF3,0x6a61}, // <CJK>
  {0xBEF4,0x6a4b}, // <CJK>
  {0xBEF5,0x6a47}, // <CJK>
  {0xBEF6,0x6a35}, // <CJK>
  {0xBEF7,0x6a5f}, // <CJK>
  {0xBEF8,0x6a48}, // <CJK>
  {0xBEF9,0x6b59}, // <CJK>
  {0xBEFA,0x6b77}, // <CJK>
  {0xBEFB,0x6c05}, // <CJK>
  {0xBEFC,0x6fc2}, // <CJK>
  {0xBEFD,0x6fb1}, // <CJK>
  {0xBEFE,0x6fa1}, // <CJK>
  {0xBF40,0x6fc3}, // <CJK>
  {0xBF41,0x6fa4}, // <CJK>
  {0xBF42,0x6fc1}, // <CJK>
  {0xBF43,0x6fa7}, // <CJK>
  {0xBF44,0x6fb3}, // <CJK>
  {0xBF45,0x6fc0}, // <CJK>
  {0xBF46,0x6fb9}, // <CJK>
  {0xBF47,0x6fb6}, // <CJK>
  {0xBF48,0x6fa6}, // <CJK>
  {0xBF49,0x6fa0}, // <CJK>
  {0xBF4A,0x6fb4}, // <CJK>
  {0xBF4B,0x71be}, // <CJK>
  {0xBF4C,0x71c9}, // <CJK>
  {0xBF4D,0x71d0}, // <CJK>
  {0xBF4E,0x71d2}, // <CJK>
  {0xBF4F,0x71c8}, // <CJK>
  {0xBF50,0x71d5}, // <CJK>
  {0xBF51,0x71b9}, // <CJK>
  {0xBF52,0x71ce}, // <CJK>
  {0xBF53,0x71d9}, // <CJK>
  {0xBF54,0x71dc}, // <CJK>
  {0xBF55,0x71c3}, // <CJK>
  {0xBF56,0x71c4}, // <CJK>
  {0xBF57,0x7368}, // <CJK>
  {0xBF58,0x749c}, // <CJK>
  {0xBF59,0x74a3}, // <CJK>
  {0xBF5A,0x7498}, // <CJK>
  {0xBF5B,0x749f}, // <CJK>
  {0xBF5C,0x749e}, // <CJK>
  {0xBF5D,0x74e2}, // <CJK>
  {0xBF5E,0x750c}, // <CJK>
  {0xBF5F,0x750d}, // <CJK>
  {0xBF60,0x7634}, // <CJK>
  {0xBF61,0x7638}, // <CJK>
  {0xBF62,0x763a}, // <CJK>
  {0xBF63,0x76e7}, // <CJK>
  {0xBF64,0x76e5}, // <CJK>
  {0xBF65,0x77a0}, // <CJK>
  {0xBF66,0x779e}, // <CJK>
  {0xBF67,0x779f}, // <CJK>
  {0xBF68,0x77a5}, // <CJK>
  {0xBF69,0x78e8}, // <CJK>
  {0xBF6A,0x78da}, // <CJK>
  {0xBF6B,0x78ec}, // <CJK>
  {0xBF6C,0x78e7}, // <CJK>
  {0xBF6D,0x79a6}, // <CJK>
  {0xBF6E,0x7a4d}, // <CJK>
  {0xBF6F,0x7a4e}, // <CJK>
  {0xBF70,0x7a46}, // <CJK>
  {0xBF71,0x7a4c}, // <CJK>
  {0xBF72,0x7a4b}, // <CJK>
  {0xBF73,0x7aba}, // <CJK>
  {0xBF74,0x7bd9}, // <CJK>
  {0xBF75,0x7c11}, // <CJK>
  {0xBF76,0x7bc9}, // <CJK>
  {0xBF77,0x7be4}, // <CJK>
  {0xBF78,0x7bdb}, // <CJK>
  {0xBF79,0x7be1}, // <CJK>
  {0xBF7A,0x7be9}, // <CJK>
  {0xBF7B,0x7be6}, // <CJK>
  {0xBF7C,0x7cd5}, // <CJK>
  {0xBF7D,0x7cd6}, // <CJK>
  {0xBF7E,0x7e0a}, // <CJK>
  {0xBFA1,0x7e11}, // <CJK>
  {0xBFA2,0x7e08}, // <CJK>
  {0xBFA3,0x7e1b}, // <CJK>
  {0xBFA4,0x7e23}, // <CJK>
  {0xBFA5,0x7e1e}, // <CJK>
  {0xBFA6,0x7e1d}, // <CJK>
  {0xBFA7,0x7e09}, // <CJK>
  {0xBFA8,0x7e10}, // <CJK>
  {0xBFA9,0x7f79}, // <CJK>
  {0xBFAA,0x7fb2}, // <CJK>
  {0xBFAB,0x7ff0}, // <CJK>
  {0xBFAC,0x7ff1}, // <CJK>
  {0xBFAD,0x7fee}, // <CJK>
  {0xBFAE,0x8028}, // <CJK>
  {0xBFAF,0x81b3}, // <CJK>
  {0xBFB0,0x81a9}, // <CJK>
  {0xBFB1,0x81a8}, // <CJK>
  {0xBFB2,0x81fb}, // <CJK>
  {0xBFB3,0x8208}, // <CJK>
  {0xBFB4,0x8258}, // <CJK>
  {0xBFB5,0x8259}, // <CJK>
  {0xBFB6,0x854a}, // <CJK>
  {0xBFB7,0x8559}, // <CJK>
  {0xBFB8,0x8548}, // <CJK>
  {0xBFB9,0x8568}, // <CJK>
  {0xBFBA,0x8569}, // <CJK>
  {0xBFBB,0x8543}, // <CJK>
  {0xBFBC,0x8549}, // <CJK>
  {0xBFBD,0x856d}, // <CJK>
  {0xBFBE,0x856a}, // <CJK>
  {0xBFBF,0x855e}, // <CJK>
  {0xBFC0,0x8783}, // <CJK>
  {0xBFC1,0x879f}, // <CJK>
  {0xBFC2,0x879e}, // <CJK>
  {0xBFC3,0x87a2}, // <CJK>
  {0xBFC4,0x878d}, // <CJK>
  {0xBFC5,0x8861}, // <CJK>
  {0xBFC6,0x892a}, // <CJK>
  {0xBFC7,0x8932}, // <CJK>
  {0xBFC8,0x8925}, // <CJK>
  {0xBFC9,0x892b}, // <CJK>
  {0xBFCA,0x8921}, // <CJK>
  {0xBFCB,0x89aa}, // <CJK>
  {0xBFCC,0x89a6}, // <CJK>
  {0xBFCD,0x8ae6}, // <CJK>
  {0xBFCE,0x8afa}, // <CJK>
  {0xBFCF,0x8aeb}, // <CJK>
  {0xBFD0,0x8af1}, // <CJK>
  {0xBFD1,0x8b00}, // <CJK>
  {0xBFD2,0x8adc}, // <CJK>
  {0xBFD3,0x8ae7}, // <CJK>
  {0xBFD4,0x8aee}, // <CJK>
  {0xBFD5,0x8afe}, // <CJK>
  {0xBFD6,0x8b01}, // <CJK>
  {0xBFD7,0x8b02}, // <CJK>
  {0xBFD8,0x8af7}, // <CJK>
  {0xBFD9,0x8aed}, // <CJK>
  {0xBFDA,0x8af3}, // <CJK>
  {0xBFDB,0x8af6}, // <CJK>
  {0xBFDC,0x8afc}, // <CJK>
  {0xBFDD,0x8c6b}, // <CJK>
  {0xBFDE,0x8c6d}, // <CJK>
  {0xBFDF,0x8c93}, // <CJK>
  {0xBFE0,0x8cf4}, // <CJK>
  {0xBFE1,0x8e44}, // <CJK>
  {0xBFE2,0x8e31}, // <CJK>
  {0xBFE3,0x8e34}, // <CJK>
  {0xBFE4,0x8e42}, // <CJK>
  {0xBFE5,0x8e39}, // <CJK>
  {0xBFE6,0x8e35}, // <CJK>
  {0xBFE7,0x8f3b}, // <CJK>
  {0xBFE8,0x8f2f}, // <CJK>
  {0xBFE9,0x8f38}, // <CJK>
  {0xBFEA,0x8f33}, // <CJK>
  {0xBFEB,0x8fa8}, // <CJK>
  {0xBFEC,0x8fa6}, // <CJK>
  {0xBFED,0x9075}, // <CJK>
  {0xBFEE,0x9074}, // <CJK>
  {0xBFEF,0x9078}, // <CJK>
  {0xBFF0,0x9072}, // <CJK>
  {0xBFF1,0x907c}, // <CJK>
  {0xBFF2,0x907a}, // <CJK>
  {0xBFF3,0x9134}, // <CJK>
  {0xBFF4,0x9192}, // <CJK>
  {0xBFF5,0x9320}, // <CJK>
  {0xBFF6,0x9336}, // <CJK>
  {0xBFF7,0x92f8}, // <CJK>
  {0xBFF8,0x9333}, // <CJK>
  {0xBFF9,0x932f}, // <CJK>
  {0xBFFA,0x9322}, // <CJK>
  {0xBFFB,0x92fc}, // <CJK>
  {0xBFFC,0x932b}, // <CJK>
  {0xBFFD,0x9304}, // <CJK>
  {0xBFFE,0x931a}, // <CJK>
  {0xC040,0x9310}, // <CJK>
  {0xC041,0x9326}, // <CJK>
  {0xC042,0x9321}, // <CJK>
  {0xC043,0x9315}, // <CJK>
  {0xC044,0x932e}, // <CJK>
  {0xC045,0x9319}, // <CJK>
  {0xC046,0x95bb}, // <CJK>
  {0xC047,0x96a7}, // <CJK>
  {0xC048,0x96a8}, // <CJK>
  {0xC049,0x96aa}, // <CJK>
  {0xC04A,0x96d5}, // <CJK>
  {0xC04B,0x970e}, // <CJK>
  {0xC04C,0x9711}, // <CJK>
  {0xC04D,0x9716}, // <CJK>
  {0xC04E,0x970d}, // <CJK>
  {0xC04F,0x9713}, // <CJK>
  {0xC050,0x970f}, // <CJK>
  {0xC051,0x975b}, // <CJK>
  {0xC052,0x975c}, // <CJK>
  {0xC053,0x9766}, // <CJK>
  {0xC054,0x9798}, // <CJK>
  {0xC055,0x9830}, // <CJK>
  {0xC056,0x9838}, // <CJK>
  {0xC057,0x983b}, // <CJK>
  {0xC058,0x9837}, // <CJK>
  {0xC059,0x982d}, // <CJK>
  {0xC05A,0x9839}, // <CJK>
  {0xC05B,0x9824}, // <CJK>
  {0xC05C,0x9910}, // <CJK>
  {0xC05D,0x9928}, // <CJK>
  {0xC05E,0x991e}, // <CJK>
  {0xC05F,0x991b}, // <CJK>
  {0xC060,0x9921}, // <CJK>
  {0xC061,0x991a}, // <CJK>
  {0xC062,0x99ed}, // <CJK>
  {0xC063,0x99e2}, // <CJK>
  {0xC064,0x99f1}, // <CJK>
  {0xC065,0x9ab8}, // <CJK>
  {0xC066,0x9abc}, // <CJK>
  {0xC067,0x9afb}, // <CJK>
  {0xC068,0x9aed}, // <CJK>
  {0xC069,0x9b28}, // <CJK>
  {0xC06A,0x9b91}, // <CJK>
  {0xC06B,0x9d15}, // <CJK>
  {0xC06C,0x9d23}, // <CJK>
  {0xC06D,0x9d26}, // <CJK>
  {0xC06E,0x9d28}, // <CJK>
  {0xC06F,0x9d12}, // <CJK>
  {0xC070,0x9d1b}, // <CJK>
  {0xC071,0x9ed8}, // <CJK>
  {0xC072,0x9ed4}, // <CJK>
  {0xC073,0x9f8d}, // <CJK>
  {0xC074,0x9f9c}, // <CJK>
  {0xC075,0x512a}, // <CJK>
  {0xC076,0x511f}, // <CJK>
  {0xC077,0x5121}, // <CJK>
  {0xC078,0x5132}, // <CJK>
  {0xC079,0x52f5}, // <CJK>
  {0xC07A,0x568e}, // <CJK>
  {0xC07B,0x5680}, // <CJK>
  {0xC07C,0x5690}, // <CJK>
  {0xC07D,0x5685}, // <CJK>
  {0xC07E,0x5687}, // <CJK>
  {0xC0A1,0x568f}, // <CJK>
  {0xC0A2,0x58d5}, // <CJK>
  {0xC0A3,0x58d3}, // <CJK>
  {0xC0A4,0x58d1}, // <CJK>
  {0xC0A5,0x58ce}, // <CJK>
  {0xC0A6,0x5b30}, // <CJK>
  {0xC0A7,0x5b2a}, // <CJK>
  {0xC0A8,0x5b24}, // <CJK>
  {0xC0A9,0x5b7a}, // <CJK>
  {0xC0AA,0x5c37}, // <CJK>
  {0xC0AB,0x5c68}, // <CJK>
  {0xC0AC,0x5dbc}, // <CJK>
  {0xC0AD,0x5dba}, // <CJK>
  {0xC0AE,0x5dbd}, // <CJK>
  {0xC0AF,0x5db8}, // <CJK>
  {0xC0B0,0x5e6b}, // <CJK>
  {0xC0B1,0x5f4c}, // <CJK>
  {0xC0B2,0x5fbd}, // <CJK>
  {0xC0B3,0x61c9}, // <CJK>
  {0xC0B4,0x61c2}, // <CJK>
  {0xC0B5,0x61c7}, // <CJK>
  {0xC0B6,0x61e6}, // <CJK>
  {0xC0B7,0x61cb}, // <CJK>
  {0xC0B8,0x6232}, // <CJK>
  {0xC0B9,0x6234}, // <CJK>
  {0xC0BA,0x64ce}, // <CJK>
  {0xC0BB,0x64ca}, // <CJK>
  {0xC0BC,0x64d8}, // <CJK>
  {0xC0BD,0x64e0}, // <CJK>
  {0xC0BE,0x64f0}, // <CJK>
  {0xC0BF,0x64e6}, // <CJK>
  {0xC0C0,0x64ec}, // <CJK>
  {0xC0C1,0x64f1}, // <CJK>
  {0xC0C2,0x64e2}, // <CJK>
  {0xC0C3,0x64ed}, // <CJK>
  {0xC0C4,0x6582}, // <CJK>
  {0xC0C5,0x6583}, // <CJK>
  {0xC0C6,0x66d9}, // <CJK>
  {0xC0C7,0x66d6}, // <CJK>
  {0xC0C8,0x6a80}, // <CJK>
  {0xC0C9,0x6a94}, // <CJK>
  {0xC0CA,0x6a84}, // <CJK>
  {0xC0CB,0x6aa2}, // <CJK>
  {0xC0CC,0x6a9c}, // <CJK>
  {0xC0CD,0x6adb}, // <CJK>
  {0xC0CE,0x6aa3}, // <CJK>
  {0xC0CF,0x6a7e}, // <CJK>
  {0xC0D0,0x6a97}, // <CJK>
  {0xC0D1,0x6a90}, // <CJK>
  {0xC0D2,0x6aa0}, // <CJK>
  {0xC0D3,0x6b5c}, // <CJK>
  {0xC0D4,0x6bae}, // <CJK>
  {0xC0D5,0x6bda}, // <CJK>
  {0xC0D6,0x6c08}, // <CJK>
  {0xC0D7,0x6fd8}, // <CJK>
  {0xC0D8,0x6ff1}, // <CJK>
  {0xC0D9,0x6fdf}, // <CJK>
  {0xC0DA,0x6fe0}, // <CJK>
  {0xC0DB,0x6fdb}, // <CJK>
  {0xC0DC,0x6fe4}, // <CJK>
  {0xC0DD,0x6feb}, // <CJK>
  {0xC0DE,0x6fef}, // <CJK>
  {0xC0DF,0x6f80}, // <CJK>
  {0xC0E0,0x6fec}, // <CJK>
  {0xC0E1,0x6fe1}, // <CJK>
  {0xC0E2,0x6fe9}, // <CJK>
  {0xC0E3,0x6fd5}, // <CJK>
  {0xC0E4,0x6fee}, // <CJK>
  {0xC0E5,0x6ff0}, // <CJK>
  {0xC0E6,0x71e7}, // <CJK>
  {0xC0E7,0x71df}, // <CJK>
  {0xC0E8,0x71ee}, // <CJK>
  {0xC0E9,0x71e6}, // <CJK>
  {0xC0EA,0x71e5}, // <CJK>
  {0xC0EB,0x71ed}, // <CJK>
  {0xC0EC,0x71ec}, // <CJK>
  {0xC0ED,0x71f4}, // <CJK>
  {0xC0EE,0x71e0}, // <CJK>
  {0xC0EF,0x7235}, // <CJK>
  {0xC0F0,0x7246}, // <CJK>
  {0xC0F1,0x7370}, // <CJK>
  {0xC0F2,0x7372}, // <CJK>
  {0xC0F3,0x74a9}, // <CJK>
  {0xC0F4,0x74b0}, // <CJK>
  {0xC0F5,0x74a6}, // <CJK>
  {0xC0F6,0x74a8}, // <CJK>
  {0xC0F7,0x7646}, // <CJK>
  {0xC0F8,0x7642}, // <CJK>
  {0xC0F9,0x764c}, // <CJK>
  {0xC0FA,0x76ea}, // <CJK>
  {0xC0FB,0x77b3}, // <CJK>
  {0xC0FC,0x77aa}, // <CJK>
  {0xC0FD,0x77b0}, // <CJK>
  {0xC0FE,0x77ac}, // <CJK>
  {0xC140,0x77a7}, // <CJK>
  {0xC141,0x77ad}, // <CJK>
  {0xC142,0x77ef}, // <CJK>
  {0xC143,0x78f7}, // <CJK>
  {0xC144,0x78fa}, // <CJK>
  {0xC145,0x78f4}, // <CJK>
  {0xC146,0x78ef}, // <CJK>
  {0xC147,0x7901}, // <CJK>
  {0xC148,0x79a7}, // <CJK>
  {0xC149,0x79aa}, // <CJK>
  {0xC14A,0x7a57}, // <CJK>
  {0xC14B,0x7abf}, // <CJK>
  {0xC14C,0x7c07}, // <CJK>
  {0xC14D,0x7c0d}, // <CJK>
  {0xC14E,0x7bfe}, // <CJK>
  {0xC14F,0x7bf7}, // <CJK>
  {0xC150,0x7c0c}, // <CJK>
  {0xC151,0x7be0}, // <CJK>
  {0xC152,0x7ce0}, // <CJK>
  {0xC153,0x7cdc}, // <CJK>
  {0xC154,0x7cde}, // <CJK>
  {0xC155,0x7ce2}, // <CJK>
  {0xC156,0x7cdf}, // <CJK>
  {0xC157,0x7cd9}, // <CJK>
  {0xC158,0x7cdd}, // <CJK>
  {0xC159,0x7e2e}, // <CJK>
  {0xC15A,0x7e3e}, // <CJK>
  {0xC15B,0x7e46}, // <CJK>
  {0xC15C,0x7e37}, // <CJK>
  {0xC15D,0x7e32}, // <CJK>
  {0xC15E,0x7e43}, // <CJK>
  {0xC15F,0x7e2b}, // <CJK>
  {0xC160,0x7e3d}, // <CJK>
  {0xC161,0x7e31}, // <CJK>
  {0xC162,0x7e45}, // <CJK>
  {0xC163,0x7e41}, // <CJK>
  {0xC164,0x7e34}, // <CJK>
  {0xC165,0x7e39}, // <CJK>
  {0xC166,0x7e48}, // <CJK>
  {0xC167,0x7e35}, // <CJK>
  {0xC168,0x7e3f}, // <CJK>
  {0xC169,0x7e2f}, // <CJK>
  {0xC16A,0x7f44}, // <CJK>
  {0xC16B,0x7ff3}, // <CJK>
  {0xC16C,0x7ffc}, // <CJK>
  {0xC16D,0x8071}, // <CJK>
  {0xC16E,0x8072}, // <CJK>
  {0xC16F,0x8070}, // <CJK>
  {0xC170,0x806f}, // <CJK>
  {0xC171,0x8073}, // <CJK>
  {0xC172,0x81c6}, // <CJK>
  {0xC173,0x81c3}, // <CJK>
  {0xC174,0x81ba}, // <CJK>
  {0xC175,0x81c2}, // <CJK>
  {0xC176,0x81c0}, // <CJK>
  {0xC177,0x81bf}, // <CJK>
  {0xC178,0x81bd}, // <CJK>
  {0xC179,0x81c9}, // <CJK>
  {0xC17A,0x81be}, // <CJK>
  {0xC17B,0x81e8}, // <CJK>
  {0xC17C,0x8209}, // <CJK>
  {0xC17D,0x8271}, // <CJK>
  {0xC17E,0x85aa}, // <CJK>
  {0xC1A1,0x8584}, // <CJK>
  {0xC1A2,0x857e}, // <CJK>
  {0xC1A3,0x859c}, // <CJK>
  {0xC1A4,0x8591}, // <CJK>
  {0xC1A5,0x8594}, // <CJK>
  {0xC1A6,0x85af}, // <CJK>
  {0xC1A7,0x859b}, // <CJK>
  {0xC1A8,0x8587}, // <CJK>
  {0xC1A9,0x85a8}, // <CJK>
  {0xC1AA,0x858a}, // <CJK>
  {0xC1AB,0x8667}, // <CJK>
  {0xC1AC,0x87c0}, // <CJK>
  {0xC1AD,0x87d1}, // <CJK>
  {0xC1AE,0x87b3}, // <CJK>
  {0xC1AF,0x87d2}, // <CJK>
  {0xC1B0,0x87c6}, // <CJK>
  {0xC1B1,0x87ab}, // <CJK>
  {0xC1B2,0x87bb}, // <CJK>
  {0xC1B3,0x87ba}, // <CJK>
  {0xC1B4,0x87c8}, // <CJK>
  {0xC1B5,0x87cb}, // <CJK>
  {0xC1B6,0x893b}, // <CJK>
  {0xC1B7,0x8936}, // <CJK>
  {0xC1B8,0x8944}, // <CJK>
  {0xC1B9,0x8938}, // <CJK>
  {0xC1BA,0x893d}, // <CJK>
  {0xC1BB,0x89ac}, // <CJK>
  {0xC1BC,0x8b0e}, // <CJK>
  {0xC1BD,0x8b17}, // <CJK>
  {0xC1BE,0x8b19}, // <CJK>
  {0xC1BF,0x8b1b}, // <CJK>
  {0xC1C0,0x8b0a}, // <CJK>
  {0xC1C1,0x8b20}, // <CJK>
  {0xC1C2,0x8b1d}, // <CJK>
  {0xC1C3,0x8b04}, // <CJK>
  {0xC1C4,0x8b10}, // <CJK>
  {0xC1C5,0x8c41}, // <CJK>
  {0xC1C6,0x8c3f}, // <CJK>
  {0xC1C7,0x8c73}, // <CJK>
  {0xC1C8,0x8cfa}, // <CJK>
  {0xC1C9,0x8cfd}, // <CJK>
  {0xC1CA,0x8cfc}, // <CJK>
  {0xC1CB,0x8cf8}, // <CJK>
  {0xC1CC,0x8cfb}, // <CJK>
  {0xC1CD,0x8da8}, // <CJK>
  {0xC1CE,0x8e49}, // <CJK>
  {0xC1CF,0x8e4b}, // <CJK>
  {0xC1D0,0x8e48}, // <CJK>
  {0xC1D1,0x8e4a}, // <CJK>
  {0xC1D2,0x8f44}, // <CJK>
  {0xC1D3,0x8f3e}, // <CJK>
  {0xC1D4,0x8f42}, // <CJK>
  {0xC1D5,0x8f45}, // <CJK>
  {0xC1D6,0x8f3f}, // <CJK>
  {0xC1D7,0x907f}, // <CJK>
  {0xC1D8,0x907d}, // <CJK>
  {0xC1D9,0x9084}, // <CJK>
  {0xC1DA,0x9081}, // <CJK>
  {0xC1DB,0x9082}, // <CJK>
  {0xC1DC,0x9080}, // <CJK>
  {0xC1DD,0x9139}, // <CJK>
  {0xC1DE,0x91a3}, // <CJK>
  {0xC1DF,0x919e}, // <CJK>
  {0xC1E0,0x919c}, // <CJK>
  {0xC1E1,0x934d}, // <CJK>
  {0xC1E2,0x9382}, // <CJK>
  {0xC1E3,0x9328}, // <CJK>
  {0xC1E4,0x9375}, // <CJK>
  {0xC1E5,0x934a}, // <CJK>
  {0xC1E6,0x9365}, // <CJK>
  {0xC1E7,0x934b}, // <CJK>
  {0xC1E8,0x9318}, // <CJK>
  {0xC1E9,0x937e}, // <CJK>
  {0xC1EA,0x936c}, // <CJK>
  {0xC1EB,0x935b}, // <CJK>
  {0xC1EC,0x9370}, // <CJK>
  {0xC1ED,0x935a}, // <CJK>
  {0xC1EE,0x9354}, // <CJK>
  {0xC1EF,0x95ca}, // <CJK>
  {0xC1F0,0x95cb}, // <CJK>
  {0xC1F1,0x95cc}, // <CJK>
  {0xC1F2,0x95c8}, // <CJK>
  {0xC1F3,0x95c6}, // <CJK>
  {0xC1F4,0x96b1}, // <CJK>
  {0xC1F5,0x96b8}, // <CJK>
  {0xC1F6,0x96d6}, // <CJK>
  {0xC1F7,0x971c}, // <CJK>
  {0xC1F8,0x971e}, // <CJK>
  {0xC1F9,0x97a0}, // <CJK>
  {0xC1FA,0x97d3}, // <CJK>
  {0xC1FB,0x9846}, // <CJK>
  {0xC1FC,0x98b6}, // <CJK>
  {0xC1FD,0x9935}, // <CJK>
  {0xC1FE,0x9a01}, // <CJK>
  {0xC240,0x99ff}, // <CJK>
  {0xC241,0x9bae}, // <CJK>
  {0xC242,0x9bab}, // <CJK>
  {0xC243,0x9baa}, // <CJK>
  {0xC244,0x9bad}, // <CJK>
  {0xC245,0x9d3b}, // <CJK>
  {0xC246,0x9d3f}, // <CJK>
  {0xC247,0x9e8b}, // <CJK>
  {0xC248,0x9ecf}, // <CJK>
  {0xC249,0x9ede}, // <CJK>
  {0xC24A,0x9edc}, // <CJK>
  {0xC24B,0x9edd}, // <CJK>
  {0xC24C,0x9edb}, // <CJK>
  {0xC24D,0x9f3e}, // <CJK>
  {0xC24E,0x9f4b}, // <CJK>
  {0xC24F,0x53e2}, // <CJK>
  {0xC250,0x5695}, // <CJK>
  {0xC251,0x56ae}, // <CJK>
  {0xC252,0x58d9}, // <CJK>
  {0xC253,0x58d8}, // <CJK>
  {0xC254,0x5b38}, // <CJK>
  {0xC255,0x5f5d}, // <CJK>
  {0xC256,0x61e3}, // <CJK>
  {0xC257,0x6233}, // <CJK>
  {0xC258,0x64f4}, // <CJK>
  {0xC259,0x64f2}, // <CJK>
  {0xC25A,0x64fe}, // <CJK>
  {0xC25B,0x6506}, // <CJK>
  {0xC25C,0x64fa}, // <CJK>
  {0xC25D,0x64fb}, // <CJK>
  {0xC25E,0x64f7}, // <CJK>
  {0xC25F,0x65b7}, // <CJK>
  {0xC260,0x66dc}, // <CJK>
  {0xC261,0x6726}, // <CJK>
  {0xC262,0x6ab3}, // <CJK>
  {0xC263,0x6aac}, // <CJK>
  {0xC264,0x6ac3}, // <CJK>
  {0xC265,0x6abb}, // <CJK>
  {0xC266,0x6ab8}, // <CJK>
  {0xC267,0x6ac2}, // <CJK>
  {0xC268,0x6aae}, // <CJK>
  {0xC269,0x6aaf}, // <CJK>
  {0xC26A,0x6b5f}, // <CJK>
  {0xC26B,0x6b78}, // <CJK>
  {0xC26C,0x6baf}, // <CJK>
  {0xC26D,0x7009}, // <CJK>
  {0xC26E,0x700b}, // <CJK>
  {0xC26F,0x6ffe}, // <CJK>
  {0xC270,0x7006}, // <CJK>
  {0xC271,0x6ffa}, // <CJK>
  {0xC272,0x7011}, // <CJK>
  {0xC273,0x700f}, // <CJK>
  {0xC274,0x71fb}, // <CJK>
  {0xC275,0x71fc}, // <CJK>
  {0xC276,0x71fe}, // <CJK>
  {0xC277,0x71f8}, // <CJK>
  {0xC278,0x7377}, // <CJK>
  {0xC279,0x7375}, // <CJK>
  {0xC27A,0x74a7}, // <CJK>
  {0xC27B,0x74bf}, // <CJK>
  {0xC27C,0x7515}, // <CJK>
  {0xC27D,0x7656}, // <CJK>
  {0xC27E,0x7658}, // <CJK>
  {0xC2A1,0x7652}, // <CJK>
  {0xC2A2,0x77bd}, // <CJK>
  {0xC2A3,0x77bf}, // <CJK>
  {0xC2A4,0x77bb}, // <CJK>
  {0xC2A5,0x77bc}, // <CJK>
  {0xC2A6,0x790e}, // <CJK>
  {0xC2A7,0x79ae}, // <CJK>
  {0xC2A8,0x7a61}, // <CJK>
  {0xC2A9,0x7a62}, // <CJK>
  {0xC2AA,0x7a60}, // <CJK>
  {0xC2AB,0x7ac4}, // <CJK>
  {0xC2AC,0x7ac5}, // <CJK>
  {0xC2AD,0x7c2b}, // <CJK>
  {0xC2AE,0x7c27}, // <CJK>
  {0xC2AF,0x7c2a}, // <CJK>
  {0xC2B0,0x7c1e}, // <CJK>
  {0xC2B1,0x7c23}, // <CJK>
  {0xC2B2,0x7c21}, // <CJK>
  {0xC2B3,0x7ce7}, // <CJK>
  {0xC2B4,0x7e54}, // <CJK>
  {0xC2B5,0x7e55}, // <CJK>
  {0xC2B6,0x7e5e}, // <CJK>
  {0xC2B7,0x7e5a}, // <CJK>
  {0xC2B8,0x7e61}, // <CJK>
  {0xC2B9,0x7e52}, // <CJK>
  {0xC2BA,0x7e59}, // <CJK>
  {0xC2BB,0x7f48}, // <CJK>
  {0xC2BC,0x7ff9}, // <CJK>
  {0xC2BD,0x7ffb}, // <CJK>
  {0xC2BE,0x8077}, // <CJK>
  {0xC2BF,0x8076}, // <CJK>
  {0xC2C0,0x81cd}, // <CJK>
  {0xC2C1,0x81cf}, // <CJK>
  {0xC2C2,0x820a}, // <CJK>
  {0xC2C3,0x85cf}, // <CJK>
  {0xC2C4,0x85a9}, // <CJK>
  {0xC2C5,0x85cd}, // <CJK>
  {0xC2C6,0x85d0}, // <CJK>
  {0xC2C7,0x85c9}, // <CJK>
  {0xC2C8,0x85b0}, // <CJK>
  {0xC2C9,0x85ba}, // <CJK>
  {0xC2CA,0x85b9}, // <CJK>
  {0xC2CB,0x85a6}, // <CJK>
  {0xC2CC,0x87ef}, // <CJK>
  {0xC2CD,0x87ec}, // <CJK>
  {0xC2CE,0x87f2}, // <CJK>
  {0xC2CF,0x87e0}, // <CJK>
  {0xC2D0,0x8986}, // <CJK>
  {0xC2D1,0x89b2}, // <CJK>
  {0xC2D2,0x89f4}, // <CJK>
  {0xC2D3,0x8b28}, // <CJK>
  {0xC2D4,0x8b39}, // <CJK>
  {0xC2D5,0x8b2c}, // <CJK>
  {0xC2D6,0x8b2b}, // <CJK>
  {0xC2D7,0x8c50}, // <CJK>
  {0xC2D8,0x8d05}, // <CJK>
  {0xC2D9,0x8e59}, // <CJK>
  {0xC2DA,0x8e63}, // <CJK>
  {0xC2DB,0x8e66}, // <CJK>
  {0xC2DC,0x8e64}, // <CJK>
  {0xC2DD,0x8e5f}, // <CJK>
  {0xC2DE,0x8e55}, // <CJK>
  {0xC2DF,0x8ec0}, // <CJK>
  {0xC2E0,0x8f49}, // <CJK>
  {0xC2E1,0x8f4d}, // <CJK>
  {0xC2E2,0x9087}, // <CJK>
  {0xC2E3,0x9083}, // <CJK>
  {0xC2E4,0x9088}, // <CJK>
  {0xC2E5,0x91ab}, // <CJK>
  {0xC2E6,0x91ac}, // <CJK>
  {0xC2E7,0x91d0}, // <CJK>
  {0xC2E8,0x9394}, // <CJK>
  {0xC2E9,0x938a}, // <CJK>
  {0xC2EA,0x9396}, // <CJK>
  {0xC2EB,0x93a2}, // <CJK>
  {0xC2EC,0x93b3}, // <CJK>
  {0xC2ED,0x93ae}, // <CJK>
  {0xC2EE,0x93ac}, // <CJK>
  {0xC2EF,0x93b0}, // <CJK>
  {0xC2F0,0x9398}, // <CJK>
  {0xC2F1,0x939a}, // <CJK>
  {0xC2F2,0x9397}, // <CJK>
  {0xC2F3,0x95d4}, // <CJK>
  {0xC2F4,0x95d6}, // <CJK>
  {0xC2F5,0x95d0}, // <CJK>
  {0xC2F6,0x95d5}, // <CJK>
  {0xC2F7,0x96e2}, // <CJK>
  {0xC2F8,0x96dc}, // <CJK>
  {0xC2F9,0x96d9}, // <CJK>
  {0xC2FA,0x96db}, // <CJK>
  {0xC2FB,0x96de}, // <CJK>
  {0xC2FC,0x9724}, // <CJK>
  {0xC2FD,0x97a3}, // <CJK>
  {0xC2FE,0x97a6}, // <CJK>
  {0xC340,0x97ad}, // <CJK>
  {0xC341,0x97f9}, // <CJK>
  {0xC342,0x984d}, // <CJK>
  {0xC343,0x984f}, // <CJK>
  {0xC344,0x984c}, // <CJK>
  {0xC345,0x984e}, // <CJK>
  {0xC346,0x9853}, // <CJK>
  {0xC347,0x98ba}, // <CJK>
  {0xC348,0x993e}, // <CJK>
  {0xC349,0x993f}, // <CJK>
  {0xC34A,0x993d}, // <CJK>
  {0xC34B,0x992e}, // <CJK>
  {0xC34C,0x99a5}, // <CJK>
  {0xC34D,0x9a0e}, // <CJK>
  {0xC34E,0x9ac1}, // <CJK>
  {0xC34F,0x9b03}, // <CJK>
  {0xC350,0x9b06}, // <CJK>
  {0xC351,0x9b4f}, // <CJK>
  {0xC352,0x9b4e}, // <CJK>
  {0xC353,0x9b4d}, // <CJK>
  {0xC354,0x9bca}, // <CJK>
  {0xC355,0x9bc9}, // <CJK>
  {0xC356,0x9bfd}, // <CJK>
  {0xC357,0x9bc8}, // <CJK>
  {0xC358,0x9bc0}, // <CJK>
  {0xC359,0x9d51}, // <CJK>
  {0xC35A,0x9d5d}, // <CJK>
  {0xC35B,0x9d60}, // <CJK>
  {0xC35C,0x9ee0}, // <CJK>
  {0xC35D,0x9f15}, // <CJK>
  {0xC35E,0x9f2c}, // <CJK>
  {0xC35F,0x5133}, // <CJK>
  {0xC360,0x56a5}, // <CJK>
  {0xC361,0x58de}, // <CJK>
  {0xC362,0x58df}, // <CJK>
  {0xC363,0x58e2}, // <CJK>
  {0xC364,0x5bf5}, // <CJK>
  {0xC365,0x9f90}, // <CJK>
  {0xC366,0x5eec}, // <CJK>
  {0xC367,0x61f2}, // <CJK>
  {0xC368,0x61f7}, // <CJK>
  {0xC369,0x61f6}, // <CJK>
  {0xC36A,0x61f5}, // <CJK>
  {0xC36B,0x6500}, // <CJK>
  {0xC36C,0x650f}, // <CJK>
  {0xC36D,0x66e0}, // <CJK>
  {0xC36E,0x66dd}, // <CJK>
  {0xC36F,0x6ae5}, // <CJK>
  {0xC370,0x6add}, // <CJK>
  {0xC371,0x6ada}, // <CJK>
  {0xC372,0x6ad3}, // <CJK>
  {0xC373,0x701b}, // <CJK>
  {0xC374,0x701f}, // <CJK>
  {0xC375,0x7028}, // <CJK>
  {0xC376,0x701a}, // <CJK>
  {0xC377,0x701d}, // <CJK>
  {0xC378,0x7015}, // <CJK>
  {0xC379,0x7018}, // <CJK>
  {0xC37A,0x7206}, // <CJK>
  {0xC37B,0x720d}, // <CJK>
  {0xC37C,0x7258}, // <CJK>
  {0xC37D,0x72a2}, // <CJK>
  {0xC37E,0x7378}, // <CJK>
  {0xC3A1,0x737a}, // <CJK>
  {0xC3A2,0x74bd}, // <CJK>
  {0xC3A3,0x74ca}, // <CJK>
  {0xC3A4,0x74e3}, // <CJK>
  {0xC3A5,0x7587}, // <CJK>
  {0xC3A6,0x7586}, // <CJK>
  {0xC3A7,0x765f}, // <CJK>
  {0xC3A8,0x7661}, // <CJK>
  {0xC3A9,0x77c7}, // <CJK>
  {0xC3AA,0x7919}, // <CJK>
  {0xC3AB,0x79b1}, // <CJK>
  {0xC3AC,0x7a6b}, // <CJK>
  {0xC3AD,0x7a69}, // <CJK>
  {0xC3AE,0x7c3e}, // <CJK>
  {0xC3AF,0x7c3f}, // <CJK>
  {0xC3B0,0x7c38}, // <CJK>
  {0xC3B1,0x7c3d}, // <CJK>
  {0xC3B2,0x7c37}, // <CJK>
  {0xC3B3,0x7c40}, // <CJK>
  {0xC3B4,0x7e6b}, // <CJK>
  {0xC3B5,0x7e6d}, // <CJK>
  {0xC3B6,0x7e79}, // <CJK>
  {0xC3B7,0x7e69}, // <CJK>
  {0xC3B8,0x7e6a}, // <CJK>
  {0xC3B9,0x7f85}, // <CJK>
  {0xC3BA,0x7e73}, // <CJK>
  {0xC3BB,0x7fb6}, // <CJK>
  {0xC3BC,0x7fb9}, // <CJK>
  {0xC3BD,0x7fb8}, // <CJK>
  {0xC3BE,0x81d8}, // <CJK>
  {0xC3BF,0x85e9}, // <CJK>
  {0xC3C0,0x85dd}, // <CJK>
  {0xC3C1,0x85ea}, // <CJK>
  {0xC3C2,0x85d5}, // <CJK>
  {0xC3C3,0x85e4}, // <CJK>
  {0xC3C4,0x85e5}, // <CJK>
  {0xC3C5,0x85f7}, // <CJK>
  {0xC3C6,0x87fb}, // <CJK>
  {0xC3C7,0x8805}, // <CJK>
  {0xC3C8,0x880d}, // <CJK>
  {0xC3C9,0x87f9}, // <CJK>
  {0xC3CA,0x87fe}, // <CJK>
  {0xC3CB,0x8960}, // <CJK>
  {0xC3CC,0x895f}, // <CJK>
  {0xC3CD,0x8956}, // <CJK>
  {0xC3CE,0x895e}, // <CJK>
  {0xC3CF,0x8b41}, // <CJK>
  {0xC3D0,0x8b5c}, // <CJK>
  {0xC3D1,0x8b58}, // <CJK>
  {0xC3D2,0x8b49}, // <CJK>
  {0xC3D3,0x8b5a}, // <CJK>
  {0xC3D4,0x8b4e}, // <CJK>
  {0xC3D5,0x8b4f}, // <CJK>
  {0xC3D6,0x8b46}, // <CJK>
  {0xC3D7,0x8b59}, // <CJK>
  {0xC3D8,0x8d08}, // <CJK>
  {0xC3D9,0x8d0a}, // <CJK>
  {0xC3DA,0x8e7c}, // <CJK>
  {0xC3DB,0x8e72}, // <CJK>
  {0xC3DC,0x8e87}, // <CJK>
  {0xC3DD,0x8e76}, // <CJK>
  {0xC3DE,0x8e6c}, // <CJK>
  {0xC3DF,0x8e7a}, // <CJK>
  {0xC3E0,0x8e74}, // <CJK>
  {0xC3E1,0x8f54}, // <CJK>
  {0xC3E2,0x8f4e}, // <CJK>
  {0xC3E3,0x8fad}, // <CJK>
  {0xC3E4,0x908a}, // <CJK>
  {0xC3E5,0x908b}, // <CJK>
  {0xC3E6,0x91b1}, // <CJK>
  {0xC3E7,0x91ae}, // <CJK>
  {0xC3E8,0x93e1}, // <CJK>
  {0xC3E9,0x93d1}, // <CJK>
  {0xC3EA,0x93df}, // <CJK>
  {0xC3EB,0x93c3}, // <CJK>
  {0xC3EC,0x93c8}, // <CJK>
  {0xC3ED,0x93dc}, // <CJK>
  {0xC3EE,0x93dd}, // <CJK>
  {0xC3EF,0x93d6}, // <CJK>
  {0xC3F0,0x93e2}, // <CJK>
  {0xC3F1,0x93cd}, // <CJK>
  {0xC3F2,0x93d8}, // <CJK>
  {0xC3F3,0x93e4}, // <CJK>
  {0xC3F4,0x93d7}, // <CJK>
  {0xC3F5,0x93e8}, // <CJK>
  {0xC3F6,0x95dc}, // <CJK>
  {0xC3F7,0x96b4}, // <CJK>
  {0xC3F8,0x96e3}, // <CJK>
  {0xC3F9,0x972a}, // <CJK>
  {0xC3FA,0x9727}, // <CJK>
  {0xC3FB,0x9761}, // <CJK>
  {0xC3FC,0x97dc}, // <CJK>
  {0xC3FD,0x97fb}, // <CJK>
  {0xC3FE,0x985e}, // <CJK>
  {0xC440,0x9858}, // <CJK>
  {0xC441,0x985b}, // <CJK>
  {0xC442,0x98bc}, // <CJK>
  {0xC443,0x9945}, // <CJK>
  {0xC444,0x9949}, // <CJK>
  {0xC445,0x9a16}, // <CJK>
  {0xC446,0x9a19}, // <CJK>
  {0xC447,0x9b0d}, // <CJK>
  {0xC448,0x9be8}, // <CJK>
  {0xC449,0x9be7}, // <CJK>
  {0xC44A,0x9bd6}, // <CJK>
  {0xC44B,0x9bdb}, // <CJK>
  {0xC44C,0x9d89}, // <CJK>
  {0xC44D,0x9d61}, // <CJK>
  {0xC44E,0x9d72}, // <CJK>
  {0xC44F,0x9d6a}, // <CJK>
  {0xC450,0x9d6c}, // <CJK>
  {0xC451,0x9e92}, // <CJK>
  {0xC452,0x9e97}, // <CJK>
  {0xC453,0x9e93}, // <CJK>
  {0xC454,0x9eb4}, // <CJK>
  {0xC455,0x52f8}, // <CJK>
  {0xC456,0x56a8}, // <CJK>
  {0xC457,0x56b7}, // <CJK>
  {0xC458,0x56b6}, // <CJK>
  {0xC459,0x56b4}, // <CJK>
  {0xC45A,0x56bc}, // <CJK>
  {0xC45B,0x58e4}, // <CJK>
  {0xC45C,0x5b40}, // <CJK>
  {0xC45D,0x5b43}, // <CJK>
  {0xC45E,0x5b7d}, // <CJK>
  {0xC45F,0x5bf6}, // <CJK>
  {0xC460,0x5dc9}, // <CJK>
  {0xC461,0x61f8}, // <CJK>
  {0xC462,0x61fa}, // <CJK>
  {0xC463,0x6518}, // <CJK>
  {0xC464,0x6514}, // <CJK>
  {0xC465,0x6519}, // <CJK>
  {0xC466,0x66e6}, // <CJK>
  {0xC467,0x6727}, // <CJK>
  {0xC468,0x6aec}, // <CJK>
  {0xC469,0x703e}, // <CJK>
  {0xC46A,0x7030}, // <CJK>
  {0xC46B,0x7032}, // <CJK>
  {0xC46C,0x7210}, // <CJK>
  {0xC46D,0x737b}, // <CJK>
  {0xC46E,0x74cf}, // <CJK>
  {0xC46F,0x7662}, // <CJK>
  {0xC470,0x7665}, // <CJK>
  {0xC471,0x7926}, // <CJK>
  {0xC472,0x792a}, // <CJK>
  {0xC473,0x792c}, // <CJK>
  {0xC474,0x792b}, // <CJK>
  {0xC475,0x7ac7}, // <CJK>
  {0xC476,0x7af6}, // <CJK>
  {0xC477,0x7c4c}, // <CJK>
  {0xC478,0x7c43}, // <CJK>
  {0xC479,0x7c4d}, // <CJK>
  {0xC47A,0x7cef}, // <CJK>
  {0xC47B,0x7cf0}, // <CJK>
  {0xC47C,0x8fae}, // <CJK>
  {0xC47D,0x7e7d}, // <CJK>
  {0xC47E,0x7e7c}, // <CJK>
  {0xC4A1,0x7e82}, // <CJK>
  {0xC4A2,0x7f4c}, // <CJK>
  {0xC4A3,0x8000}, // <CJK>
  {0xC4A4,0x81da}, // <CJK>
  {0xC4A5,0x8266}, // <CJK>
  {0xC4A6,0x85fb}, // <CJK>
  {0xC4A7,0x85f9}, // <CJK>
  {0xC4A8,0x8611}, // <CJK>
  {0xC4A9,0x85fa}, // <CJK>
  {0xC4AA,0x8606}, // <CJK>
  {0xC4AB,0x860b}, // <CJK>
  {0xC4AC,0x8607}, // <CJK>
  {0xC4AD,0x860a}, // <CJK>
  {0xC4AE,0x8814}, // <CJK>
  {0xC4AF,0x8815}, // <CJK>
  {0xC4B0,0x8964}, // <CJK>
  {0xC4B1,0x89ba}, // <CJK>
  {0xC4B2,0x89f8}, // <CJK>
  {0xC4B3,0x8b70}, // <CJK>
  {0xC4B4,0x8b6c}, // <CJK>
  {0xC4B5,0x8b66}, // <CJK>
  {0xC4B6,0x8b6f}, // <CJK>
  {0xC4B7,0x8b5f}, // <CJK>
  {0xC4B8,0x8b6b}, // <CJK>
  {0xC4B9,0x8d0f}, // <CJK>
  {0xC4BA,0x8d0d}, // <CJK>
  {0xC4BB,0x8e89}, // <CJK>
  {0xC4BC,0x8e81}, // <CJK>
  {0xC4BD,0x8e85}, // <CJK>
  {0xC4BE,0x8e82}, // <CJK>
  {0xC4BF,0x91b4}, // <CJK>
  {0xC4C0,0x91cb}, // <CJK>
  {0xC4C1,0x9418}, // <CJK>
  {0xC4C2,0x9403}, // <CJK>
  {0xC4C3,0x93fd}, // <CJK>
  {0xC4C4,0x95e1}, // <CJK>
  {0xC4C5,0x9730}, // <CJK>
  {0xC4C6,0x98c4}, // <CJK>
  {0xC4C7,0x9952}, // <CJK>
  {0xC4C8,0x9951}, // <CJK>
  {0xC4C9,0x99a8}, // <CJK>
  {0xC4CA,0x9a2b}, // <CJK>
  {0xC4CB,0x9a30}, // <CJK>
  {0xC4CC,0x9a37}, // <CJK>
  {0xC4CD,0x9a35}, // <CJK>
  {0xC4CE,0x9c13}, // <CJK>
  {0xC4CF,0x9c0d}, // <CJK>
  {0xC4D0,0x9e79}, // <CJK>
  {0xC4D1,0x9eb5}, // <CJK>
  {0xC4D2,0x9ee8}, // <CJK>
  {0xC4D3,0x9f2f}, // <CJK>
  {0xC4D4,0x9f5f}, // <CJK>
  {0xC4D5,0x9f63}, // <CJK>
  {0xC4D6,0x9f61}, // <CJK>
  {0xC4D7,0x5137}, // <CJK>
  {0xC4D8,0x5138}, // <CJK>
  {0xC4D9,0x56c1}, // <CJK>
  {0xC4DA,0x56c0}, // <CJK>
  {0xC4DB,0x56c2}, // <CJK>
  {0xC4DC,0x5914}, // <CJK>
  {0xC4DD,0x5c6c}, // <CJK>
  {0xC4DE,0x5dcd}, // <CJK>
  {0xC4DF,0x61fc}, // <CJK>
  {0xC4E0,0x61fe}, // <CJK>
  {0xC4E1,0x651d}, // <CJK>
  {0xC4E2,0x651c}, // <CJK>
  {0xC4E3,0x6595}, // <CJK>
  {0xC4E4,0x66e9}, // <CJK>
  {0xC4E5,0x6afb}, // <CJK>
  {0xC4E6,0x6b04}, // <CJK>
  {0xC4E7,0x6afa}, // <CJK>
  {0xC4E8,0x6bb2}, // <CJK>
  {0xC4E9,0x704c}, // <CJK>
  {0xC4EA,0x721b}, // <CJK>
  {0xC4EB,0x72a7}, // <CJK>
  {0xC4EC,0x74d6}, // <CJK>
  {0xC4ED,0x74d4}, // <CJK>
  {0xC4EE,0x7669}, // <CJK>
  {0xC4EF,0x77d3}, // <CJK>
  {0xC4F0,0x7c50}, // <CJK>
  {0xC4F1,0x7e8f}, // <CJK>
  {0xC4F2,0x7e8c}, // <CJK>
  {0xC4F3,0x7fbc}, // <CJK>
  {0xC4F4,0x8617}, // <CJK>
  {0xC4F5,0x862d}, // <CJK>
  {0xC4F6,0x861a}, // <CJK>
  {0xC4F7,0x8823}, // <CJK>
  {0xC4F8,0x8822}, // <CJK>
  {0xC4F9,0x8821}, // <CJK>
  {0xC4FA,0x881f}, // <CJK>
  {0xC4FB,0x896a}, // <CJK>
  {0xC4FC,0x896c}, // <CJK>
  {0xC4FD,0x89bd}, // <CJK>
  {0xC4FE,0x8b74}, // <CJK>
  {0xC540,0x8b77}, // <CJK>
  {0xC541,0x8b7d}, // <CJK>
  {0xC542,0x8d13}, // <CJK>
  {0xC543,0x8e8a}, // <CJK>
  {0xC544,0x8e8d}, // <CJK>
  {0xC545,0x8e8b}, // <CJK>
  {0xC546,0x8f5f}, // <CJK>
  {0xC547,0x8faf}, // <CJK>
  {0xC548,0x91ba}, // <CJK>
  {0xC549,0x942e}, // <CJK>
  {0xC54A,0x9433}, // <CJK>
  {0xC54B,0x9435}, // <CJK>
  {0xC54C,0x943a}, // <CJK>
  {0xC54D,0x9438}, // <CJK>
  {0xC54E,0x9432}, // <CJK>
  {0xC54F,0x942b}, // <CJK>
  {0xC550,0x95e2}, // <CJK>
  {0xC551,0x9738}, // <CJK>
  {0xC552,0x9739}, // <CJK>
  {0xC553,0x9732}, // <CJK>
  {0xC554,0x97ff}, // <CJK>
  {0xC555,0x9867}, // <CJK>
  {0xC556,0x9865}, // <CJK>
  {0xC557,0x9957}, // <CJK>
  {0xC558,0x9a45}, // <CJK>
  {0xC559,0x9a43}, // <CJK>
  {0xC55A,0x9a40}, // <CJK>
  {0xC55B,0x9a3e}, // <CJK>
  {0xC55C,0x9acf}, // <CJK>
  {0xC55D,0x9b54}, // <CJK>
  {0xC55E,0x9b51}, // <CJK>
  {0xC55F,0x9c2d}, // <CJK>
  {0xC560,0x9c25}, // <CJK>
  {0xC561,0x9daf}, // <CJK>
  {0xC562,0x9db4}, // <CJK>
  {0xC563,0x9dc2}, // <CJK>
  {0xC564,0x9db8}, // <CJK>
  {0xC565,0x9e9d}, // <CJK>
  {0xC566,0x9eef}, // <CJK>
  {0xC567,0x9f19}, // <CJK>
  {0xC568,0x9f5c}, // <CJK>
  {0xC569,0x9f66}, // <CJK>
  {0xC56A,0x9f67}, // <CJK>
  {0xC56B,0x513c}, // <CJK>
  {0xC56C,0x513b}, // <CJK>
  {0xC56D,0x56c8}, // <CJK>
  {0xC56E,0x56ca}, // <CJK>
  {0xC56F,0x56c9}, // <CJK>
  {0xC570,0x5b7f}, // <CJK>
  {0xC571,0x5dd4}, // <CJK>
  {0xC572,0x5dd2}, // <CJK>
  {0xC573,0x5f4e}, // <CJK>
  {0xC574,0x61ff}, // <CJK>
  {0xC575,0x6524}, // <CJK>
  {0xC576,0x6b0a}, // <CJK>
  {0xC577,0x6b61}, // <CJK>
  {0xC578,0x7051}, // <CJK>
  {0xC579,0x7058}, // <CJK>
  {0xC57A,0x7380}, // <CJK>
  {0xC57B,0x74e4}, // <CJK>
  {0xC57C,0x758a}, // <CJK>
  {0xC57D,0x766e}, // <CJK>
  {0xC57E,0x766c}, // <CJK>
  {0xC5A1,0x79b3}, // <CJK>
  {0xC5A2,0x7c60}, // <CJK>
  {0xC5A3,0x7c5f}, // <CJK>
  {0xC5A4,0x807e}, // <CJK>
  {0xC5A5,0x807d}, // <CJK>
  {0xC5A6,0x81df}, // <CJK>
  {0xC5A7,0x8972}, // <CJK>
  {0xC5A8,0x896f}, // <CJK>
  {0xC5A9,0x89fc}, // <CJK>
  {0xC5AA,0x8b80}, // <CJK>
  {0xC5AB,0x8d16}, // <CJK>
  {0xC5AC,0x8d17}, // <CJK>
  {0xC5AD,0x8e91}, // <CJK>
  {0xC5AE,0x8e93}, // <CJK>
  {0xC5AF,0x8f61}, // <CJK>
  {0xC5B0,0x9148}, // <CJK>
  {0xC5B1,0x9444}, // <CJK>
  {0xC5B2,0x9451}, // <CJK>
  {0xC5B3,0x9452}, // <CJK>
  {0xC5B4,0x973d}, // <CJK>
  {0xC5B5,0x973e}, // <CJK>
  {0xC5B6,0x97c3}, // <CJK>
  {0xC5B7,0x97c1}, // <CJK>
  {0xC5B8,0x986b}, // <CJK>
  {0xC5B9,0x9955}, // <CJK>
  {0xC5BA,0x9a55}, // <CJK>
  {0xC5BB,0x9a4d}, // <CJK>
  {0xC5BC,0x9ad2}, // <CJK>
  {0xC5BD,0x9b1a}, // <CJK>
  {0xC5BE,0x9c49}, // <CJK>
  {0xC5BF,0x9c31}, // <CJK>
  {0xC5C0,0x9c3e}, // <CJK>
  {0xC5C1,0x9c3b}, // <CJK>
  {0xC5C2,0x9dd3}, // <CJK>
  {0xC5C3,0x9dd7}, // <CJK>
  {0xC5C4,0x9f34}, // <CJK>
  {0xC5C5,0x9f6c}, // <CJK>
  {0xC5C6,0x9f6a}, // <CJK>
  {0xC5C7,0x9f94}, // <CJK>
  {0xC5C8,0x56cc}, // <CJK>
  {0xC5C9,0x5dd6}, // <CJK>
  {0xC5CA,0x6200}, // <CJK>
  {0xC5CB,0x6523}, // <CJK>
  {0xC5CC,0x652b}, // <CJK>
  {0xC5CD,0x652a}, // <CJK>
  {0xC5CE,0x66ec}, // <CJK>
  {0xC5CF,0x6b10}, // <CJK>
  {0xC5D0,0x74da}, // <CJK>
  {0xC5D1,0x7aca}, // <CJK>
  {0xC5D2,0x7c64}, // <CJK>
  {0xC5D3,0x7c63}, // <CJK>
  {0xC5D4,0x7c65}, // <CJK>
  {0xC5D5,0x7e93}, // <CJK>
  {0xC5D6,0x7e96}, // <CJK>
  {0xC5D7,0x7e94}, // <CJK>
  {0xC5D8,0x81e2}, // <CJK>
  {0xC5D9,0x8638}, // <CJK>
  {0xC5DA,0x863f}, // <CJK>
  {0xC5DB,0x8831}, // <CJK>
  {0xC5DC,0x8b8a}, // <CJK>
  {0xC5DD,0x9090}, // <CJK>
  {0xC5DE,0x908f}, // <CJK>
  {0xC5DF,0x9463}, // <CJK>
  {0xC5E0,0x9460}, // <CJK>
  {0xC5E1,0x9464}, // <CJK>
  {0xC5E2,0x9768}, // <CJK>
  {0xC5E3,0x986f}, // <CJK>
  {0xC5E4,0x995c}, // <CJK>
  {0xC5E5,0x9a5a}, // <CJK>
  {0xC5E6,0x9a5b}, // <CJK>
  {0xC5E7,0x9a57}, // <CJK>
  {0xC5E8,0x9ad3}, // <CJK>
  {0xC5E9,0x9ad4}, // <CJK>
  {0xC5EA,0x9ad1}, // <CJK>
  {0xC5EB,0x9c54}, // <CJK>
  {0xC5EC,0x9c57}, // <CJK>
  {0xC5ED,0x9c56}, // <CJK>
  {0xC5EE,0x9de5}, // <CJK>
  {0xC5EF,0x9e9f}, // <CJK>
  {0xC5F0,0x9ef4}, // <CJK>
  {0xC5F1,0x56d1}, // <CJK>
  {0xC5F2,0x58e9}, // <CJK>
  {0xC5F3,0x652c}, // <CJK>
  {0xC5F4,0x705e}, // <CJK>
  {0xC5F5,0x7671}, // <CJK>
  {0xC5F6,0x7672}, // <CJK>
  {0xC5F7,0x77d7}, // <CJK>
  {0xC5F8,0x7f50}, // <CJK>
  {0xC5F9,0x7f88}, // <CJK>
  {0xC5FA,0x8836}, // <CJK>
  {0xC5FB,0x8839}, // <CJK>
  {0xC5FC,0x8862}, // <CJK>
  {0xC5FD,0x8b93}, // <CJK>
  {0xC5FE,0x8b92}, // <CJK>
  {0xC640,0x8b96}, // <CJK>
  {0xC641,0x8277}, // <CJK>
  {0xC642,0x8d1b}, // <CJK>
  {0xC643,0x91c0}, // <CJK>
  {0xC644,0x946a}, // <CJK>
  {0xC645,0x9742}, // <CJK>
  {0xC646,0x9748}, // <CJK>
  {0xC647,0x9744}, // <CJK>
  {0xC648,0x97c6}, // <CJK>
  {0xC649,0x9870}, // <CJK>
  {0xC64A,0x9a5f}, // <CJK>
  {0xC64B,0x9b22}, // <CJK>
  {0xC64C,0x9b58}, // <CJK>
  {0xC64D,0x9c5f}, // <CJK>
  {0xC64E,0x9df9}, // <CJK>
  {0xC64F,0x9dfa}, // <CJK>
  {0xC650,0x9e7c}, // <CJK>
  {0xC651,0x9e7d}, // <CJK>
  {0xC652,0x9f07}, // <CJK>
  {0xC653,0x9f77}, // <CJK>
  {0xC654,0x9f72}, // <CJK>
  {0xC655,0x5ef3}, // <CJK>
  {0xC656,0x6b16}, // <CJK>
  {0xC657,0x7063}, // <CJK>
  {0xC658,0x7c6c}, // <CJK>
  {0xC659,0x7c6e}, // <CJK>
  {0xC65A,0x883b}, // <CJK>
  {0xC65B,0x89c0}, // <CJK>
  {0xC65C,0x8ea1}, // <CJK>
  {0xC65D,0x91c1}, // <CJK>
  {0xC65E,0x9472}, // <CJK>
  {0xC65F,0x9470}, // <CJK>
  {0xC660,0x9871}, // <CJK>
  {0xC661,0x995e}, // <CJK>
  {0xC662,0x9ad6}, // <CJK>
  {0xC663,0x9b23}, // <CJK>
  {0xC664,0x9ecc}, // <CJK>
  {0xC665,0x7064}, // <CJK>
  {0xC666,0x77da}, // <CJK>
  {0xC667,0x8b9a}, // <CJK>
  {0xC668,0x9477}, // <CJK>
  {0xC669,0x97c9}, // <CJK>
  {0xC66A,0x9a62}, // <CJK>
  {0xC66B,0x9a65}, // <CJK>
  {0xC66C,0x7e9c}, // <CJK>
  {0xC66D,0x8b9c}, // <CJK>
  {0xC66E,0x8eaa}, // <CJK>
  {0xC66F,0x91c5}, // <CJK>
  {0xC670,0x947d}, // <CJK>
  {0xC671,0x947e}, // <CJK>
  {0xC672,0x947c}, // <CJK>
  {0xC673,0x9c77}, // <CJK>
  {0xC674,0x9c78}, // <CJK>
  {0xC675,0x9ef7}, // <CJK>
  {0xC676,0x8c54}, // <CJK>
  {0xC677,0x947f}, // <CJK>
  {0xC678,0x9e1a}, // <CJK>
  {0xC679,0x7228}, // <CJK>
  {0xC67A,0x9a6a}, // <CJK>
  {0xC67B,0x9b31}, // <CJK>
  {0xC67C,0x9e1b}, // <CJK>
  {0xC67D,0x9e1e}, // <CJK>
  {0xC67E,0x7c72}, // <CJK>
  {0xC6A1,0x30fe}, // KATAKANA VOICED ITERATION MARK
  {0xC6A2,0x309d}, // HIRAGANA ITERATION MARK
  {0xC6A3,0x309e}, // HIRAGANA VOICED ITERATION MARK
  {0xC6A4,0x3005}, // IDEOGRAPHIC ITERATION MARK
  {0xC6A5,0x3041}, // HIRAGANA LETTER SMALL A
  {0xC6A6,0x3042}, // HIRAGANA LETTER A
  {0xC6A7,0x3043}, // HIRAGANA LETTER SMALL I
  {0xC6A8,0x3044}, // HIRAGANA LETTER I
  {0xC6A9,0x3045}, // HIRAGANA LETTER SMALL U
  {0xC6AA,0x3046}, // HIRAGANA LETTER U
  {0xC6AB,0x3047}, // HIRAGANA LETTER SMALL E
  {0xC6AC,0x3048}, // HIRAGANA LETTER E
  {0xC6AD,0x3049}, // HIRAGANA LETTER SMALL O
  {0xC6AE,0x304a}, // HIRAGANA LETTER O
  {0xC6AF,0x304b}, // HIRAGANA LETTER KA
  {0xC6B0,0x304c}, // HIRAGANA LETTER GA
  {0xC6B1,0x304d}, // HIRAGANA LETTER KI
  {0xC6B2,0x304e}, // HIRAGANA LETTER GI
  {0xC6B3,0x304f}, // HIRAGANA LETTER KU
  {0xC6B4,0x3050}, // HIRAGANA LETTER GU
  {0xC6B5,0x3051}, // HIRAGANA LETTER KE
  {0xC6B6,0x3052}, // HIRAGANA LETTER GE
  {0xC6B7,0x3053}, // HIRAGANA LETTER KO
  {0xC6B8,0x3054}, // HIRAGANA LETTER GO
  {0xC6B9,0x3055}, // HIRAGANA LETTER SA
  {0xC6BA,0x3056}, // HIRAGANA LETTER ZA
  {0xC6BB,0x3057}, // HIRAGANA LETTER SI
  {0xC6BC,0x3058}, // HIRAGANA LETTER ZI
  {0xC6BD,0x3059}, // HIRAGANA LETTER SU
  {0xC6BE,0x305a}, // HIRAGANA LETTER ZU
  {0xC6BF,0x305b}, // HIRAGANA LETTER SE
  {0xC6C0,0x305c}, // HIRAGANA LETTER ZE
  {0xC6C1,0x305d}, // HIRAGANA LETTER SO
  {0xC6C2,0x305e}, // HIRAGANA LETTER ZO
  {0xC6C3,0x305f}, // HIRAGANA LETTER TA
  {0xC6C4,0x3060}, // HIRAGANA LETTER DA
  {0xC6C5,0x3061}, // HIRAGANA LETTER TI
  {0xC6C6,0x3062}, // HIRAGANA LETTER DI
  {0xC6C7,0x3063}, // HIRAGANA LETTER SMALL TU
  {0xC6C8,0x3064}, // HIRAGANA LETTER TU
  {0xC6C9,0x3065}, // HIRAGANA LETTER DU
  {0xC6CA,0x3066}, // HIRAGANA LETTER TE
  {0xC6CB,0x3067}, // HIRAGANA LETTER DE
  {0xC6CC,0x3068}, // HIRAGANA LETTER TO
  {0xC6CD,0x3069}, // HIRAGANA LETTER DO
  {0xC6CE,0x306a}, // HIRAGANA LETTER NA
  {0xC6CF,0x306b}, // HIRAGANA LETTER NI
  {0xC6D0,0x306c}, // HIRAGANA LETTER NU
  {0xC6D1,0x306d}, // HIRAGANA LETTER NE
  {0xC6D2,0x306e}, // HIRAGANA LETTER NO
  {0xC6D3,0x306f}, // HIRAGANA LETTER HA
  {0xC6D4,0x3070}, // HIRAGANA LETTER BA
  {0xC6D5,0x3071}, // HIRAGANA LETTER PA
  {0xC6D6,0x3072}, // HIRAGANA LETTER HI
  {0xC6D7,0x3073}, // HIRAGANA LETTER BI
  {0xC6D8,0x3074}, // HIRAGANA LETTER PI
  {0xC6D9,0x3075}, // HIRAGANA LETTER HU
  {0xC6DA,0x3076}, // HIRAGANA LETTER BU
  {0xC6DB,0x3077}, // HIRAGANA LETTER PU
  {0xC6DC,0x3078}, // HIRAGANA LETTER HE
  {0xC6DD,0x3079}, // HIRAGANA LETTER BE
  {0xC6DE,0x307a}, // HIRAGANA LETTER PE
  {0xC6DF,0x307b}, // HIRAGANA LETTER HO
  {0xC6E0,0x307c}, // HIRAGANA LETTER BO
  {0xC6E1,0x307d}, // HIRAGANA LETTER PO
  {0xC6E2,0x307e}, // HIRAGANA LETTER MA
  {0xC6E3,0x307f}, // HIRAGANA LETTER MI
  {0xC6E4,0x3080}, // HIRAGANA LETTER MU
  {0xC6E5,0x3081}, // HIRAGANA LETTER ME
  {0xC6E6,0x3082}, // HIRAGANA LETTER MO
  {0xC6E7,0x3083}, // HIRAGANA LETTER SMALL YA
  {0xC6E8,0x3084}, // HIRAGANA LETTER YA
  {0xC6E9,0x3085}, // HIRAGANA LETTER SMALL YU
  {0xC6EA,0x3086}, // HIRAGANA LETTER YU
  {0xC6EB,0x3087}, // HIRAGANA LETTER SMALL YO
  {0xC6EC,0x3088}, // HIRAGANA LETTER YO
  {0xC6ED,0x3089}, // HIRAGANA LETTER RA
  {0xC6EE,0x308a}, // HIRAGANA LETTER RI
  {0xC6EF,0x308b}, // HIRAGANA LETTER RU
  {0xC6F0,0x308c}, // HIRAGANA LETTER RE
  {0xC6F1,0x308d}, // HIRAGANA LETTER RO
  {0xC6F2,0x308e}, // HIRAGANA LETTER SMALL WA
  {0xC6F3,0x308f}, // HIRAGANA LETTER WA
  {0xC6F4,0x3090}, // HIRAGANA LETTER WI
  {0xC6F5,0x3091}, // HIRAGANA LETTER WE
  {0xC6F6,0x3092}, // HIRAGANA LETTER WO
  {0xC6F7,0x3093}, // HIRAGANA LETTER N
  {0xC6F8,0x30a1}, // KATAKANA LETTER SMALL A
  {0xC6F9,0x30a2}, // KATAKANA LETTER A
  {0xC6FA,0x30a3}, // KATAKANA LETTER SMALL I
  {0xC6FB,0x30a4}, // KATAKANA LETTER I
  {0xC6FC,0x30a5}, // KATAKANA LETTER SMALL U
  {0xC6FD,0x30a6}, // KATAKANA LETTER U
  {0xC6FE,0x30a7}, // KATAKANA LETTER SMALL E
  {0xC740,0x30a8}, // KATAKANA LETTER E
  {0xC741,0x30a9}, // KATAKANA LETTER SMALL O
  {0xC742,0x30aa}, // KATAKANA LETTER O
  {0xC743,0x30ab}, // KATAKANA LETTER KA
  {0xC744,0x30ac}, // KATAKANA LETTER GA
  {0xC745,0x30ad}, // KATAKANA LETTER KI
  {0xC746,0x30ae}, // KATAKANA LETTER GI
  {0xC747,0x30af}, // KATAKANA LETTER KU
  {0xC748,0x30b0}, // KATAKANA LETTER GU
  {0xC749,0x30b1}, // KATAKANA LETTER KE
  {0xC74A,0x30b2}, // KATAKANA LETTER GE
  {0xC74B,0x30b3}, // KATAKANA LETTER KO
  {0xC74C,0x30b4}, // KATAKANA LETTER GO
  {0xC74D,0x30b5}, // KATAKANA LETTER SA
  {0xC74E,0x30b6}, // KATAKANA LETTER ZA
  {0xC74F,0x30b7}, // KATAKANA LETTER SI
  {0xC750,0x30b8}, // KATAKANA LETTER ZI
  {0xC751,0x30b9}, // KATAKANA LETTER SU
  {0xC752,0x30ba}, // KATAKANA LETTER ZU
  {0xC753,0x30bb}, // KATAKANA LETTER SE
  {0xC754,0x30bc}, // KATAKANA LETTER ZE
  {0xC755,0x30bd}, // KATAKANA LETTER SO
  {0xC756,0x30be}, // KATAKANA LETTER ZO
  {0xC757,0x30bf}, // KATAKANA LETTER TA
  {0xC758,0x30c0}, // KATAKANA LETTER DA
  {0xC759,0x30c1}, // KATAKANA LETTER TI
  {0xC75A,0x30c2}, // KATAKANA LETTER DI
  {0xC75B,0x30c3}, // KATAKANA LETTER SMALL TU
  {0xC75C,0x30c4}, // KATAKANA LETTER TU
  {0xC75D,0x30c5}, // KATAKANA LETTER DU
  {0xC75E,0x30c6}, // KATAKANA LETTER TE
  {0xC75F,0x30c7}, // KATAKANA LETTER DE
  {0xC760,0x30c8}, // KATAKANA LETTER TO
  {0xC761,0x30c9}, // KATAKANA LETTER DO
  {0xC762,0x30ca}, // KATAKANA LETTER NA
  {0xC763,0x30cb}, // KATAKANA LETTER NI
  {0xC764,0x30cc}, // KATAKANA LETTER NU
  {0xC765,0x30cd}, // KATAKANA LETTER NE
  {0xC766,0x30ce}, // KATAKANA LETTER NO
  {0xC767,0x30cf}, // KATAKANA LETTER HA
  {0xC768,0x30d0}, // KATAKANA LETTER BA
  {0xC769,0x30d1}, // KATAKANA LETTER PA
  {0xC76A,0x30d2}, // KATAKANA LETTER HI
  {0xC76B,0x30d3}, // KATAKANA LETTER BI
  {0xC76C,0x30d4}, // KATAKANA LETTER PI
  {0xC76D,0x30d5}, // KATAKANA LETTER HU
  {0xC76E,0x30d6}, // KATAKANA LETTER BU
  {0xC76F,0x30d7}, // KATAKANA LETTER PU
  {0xC770,0x30d8}, // KATAKANA LETTER HE
  {0xC771,0x30d9}, // KATAKANA LETTER BE
  {0xC772,0x30da}, // KATAKANA LETTER PE
  {0xC773,0x30db}, // KATAKANA LETTER HO
  {0xC774,0x30dc}, // KATAKANA LETTER BO
  {0xC775,0x30dd}, // KATAKANA LETTER PO
  {0xC776,0x30de}, // KATAKANA LETTER MA
  {0xC777,0x30df}, // KATAKANA LETTER MI
  {0xC778,0x30e0}, // KATAKANA LETTER MU
  {0xC779,0x30e1}, // KATAKANA LETTER ME
  {0xC77A,0x30e2}, // KATAKANA LETTER MO
  {0xC77B,0x30e3}, // KATAKANA LETTER SMALL YA
  {0xC77C,0x30e4}, // KATAKANA LETTER YA
  {0xC77D,0x30e5}, // KATAKANA LETTER SMALL YU
  {0xC77E,0x30e6}, // KATAKANA LETTER YU
  {0xC7A1,0x30e7}, // KATAKANA LETTER SMALL YO
  {0xC7A2,0x30e8}, // KATAKANA LETTER YO
  {0xC7A3,0x30e9}, // KATAKANA LETTER RA
  {0xC7A4,0x30ea}, // KATAKANA LETTER RI
  {0xC7A5,0x30eb}, // KATAKANA LETTER RU
  {0xC7A6,0x30ec}, // KATAKANA LETTER RE
  {0xC7A7,0x30ed}, // KATAKANA LETTER RO
  {0xC7A8,0x30ee}, // KATAKANA LETTER SMALL WA
  {0xC7A9,0x30ef}, // KATAKANA LETTER WA
  {0xC7AA,0x30f0}, // KATAKANA LETTER WI
  {0xC7AB,0x30f1}, // KATAKANA LETTER WE
  {0xC7AC,0x30f2}, // KATAKANA LETTER WO
  {0xC7AD,0x30f3}, // KATAKANA LETTER N
  {0xC7AE,0x30f4}, // KATAKANA LETTER VU
  {0xC7AF,0x30f5}, // KATAKANA LETTER SMALL KA
  {0xC7B0,0x30f6}, // KATAKANA LETTER SMALL KE
  {0xC7B1,0x0414}, // CYRILLIC CAPITAL LETTER DE
  {0xC7B2,0x0415}, // CYRILLIC CAPITAL LETTER IE
  {0xC7B3,0x0401}, // CYRILLIC CAPITAL LETTER IO
  {0xC7B4,0x0416}, // CYRILLIC CAPITAL LETTER ZHE
  {0xC7B5,0x0417}, // CYRILLIC CAPITAL LETTER ZE
  {0xC7B6,0x0418}, // CYRILLIC CAPITAL LETTER I
  {0xC7B7,0x0419}, // CYRILLIC CAPITAL LETTER SHORT I
  {0xC7B8,0x041a}, // CYRILLIC CAPITAL LETTER KA
  {0xC7B9,0x041b}, // CYRILLIC CAPITAL LETTER EL
  {0xC7BA,0x041c}, // CYRILLIC CAPITAL LETTER EM
  {0xC7BB,0x0423}, // CYRILLIC CAPITAL LETTER U
  {0xC7BC,0x0424}, // CYRILLIC CAPITAL LETTER EF
  {0xC7BD,0x0425}, // CYRILLIC CAPITAL LETTER HA
  {0xC7BE,0x0426}, // CYRILLIC CAPITAL LETTER TSE
  {0xC7BF,0x0427}, // CYRILLIC CAPITAL LETTER CHE
  {0xC7C0,0x0428}, // CYRILLIC CAPITAL LETTER SHA
  {0xC7C1,0x0429}, // CYRILLIC CAPITAL LETTER SHCHA
  {0xC7C2,0x042a}, // CYRILLIC CAPITAL LETTER HARD SIGN
  {0xC7C3,0x042b}, // CYRILLIC CAPITAL LETTER YERU
  {0xC7C4,0x042c}, // CYRILLIC CAPITAL LETTER SOFT SIGN
  {0xC7C5,0x042d}, // CYRILLIC CAPITAL LETTER E
  {0xC7C6,0x042e}, // CYRILLIC CAPITAL LETTER YU
  {0xC7C7,0x042f}, // CYRILLIC CAPITAL LETTER YA
  {0xC7C8,0x0430}, // CYRILLIC SMALL LETTER A
  {0xC7C9,0x0431}, // CYRILLIC SMALL LETTER BE
  {0xC7CA,0x0432}, // CYRILLIC SMALL LETTER VE
  {0xC7CB,0x0433}, // CYRILLIC SMALL LETTER GHE
  {0xC7CC,0x0434}, // CYRILLIC SMALL LETTER DE
  {0xC7CD,0x0435}, // CYRILLIC SMALL LETTER IE
  {0xC7CE,0x0451}, // CYRILLIC SMALL LETTER IO
  {0xC7CF,0x0436}, // CYRILLIC SMALL LETTER ZHE
  {0xC7D0,0x0437}, // CYRILLIC SMALL LETTER ZE
  {0xC7D1,0x0438}, // CYRILLIC SMALL LETTER I
  {0xC7D2,0x0439}, // CYRILLIC SMALL LETTER SHORT I
  {0xC7D3,0x043a}, // CYRILLIC SMALL LETTER KA
  {0xC7D4,0x043b}, // CYRILLIC SMALL LETTER EL
  {0xC7D5,0x043c}, // CYRILLIC SMALL LETTER EM
  {0xC7D6,0x043d}, // CYRILLIC SMALL LETTER EN
  {0xC7D7,0x043e}, // CYRILLIC SMALL LETTER O
  {0xC7D8,0x043f}, // CYRILLIC SMALL LETTER PE
  {0xC7D9,0x0440}, // CYRILLIC SMALL LETTER ER
  {0xC7DA,0x0441}, // CYRILLIC SMALL LETTER ES
  {0xC7DB,0x0442}, // CYRILLIC SMALL LETTER TE
  {0xC7DC,0x0443}, // CYRILLIC SMALL LETTER U
  {0xC7DD,0x0444}, // CYRILLIC SMALL LETTER EF
  {0xC7DE,0x0445}, // CYRILLIC SMALL LETTER HA
  {0xC7DF,0x0446}, // CYRILLIC SMALL LETTER TSE
  {0xC7E0,0x0447}, // CYRILLIC SMALL LETTER CHE
  {0xC7E1,0x0448}, // CYRILLIC SMALL LETTER SHA
  {0xC7E2,0x0449}, // CYRILLIC SMALL LETTER SHCHA
  {0xC7E3,0x044a}, // CYRILLIC SMALL LETTER HARD SIGN
  {0xC7E4,0x044b}, // CYRILLIC SMALL LETTER YERU
  {0xC7E5,0x044c}, // CYRILLIC SMALL LETTER SOFT SIGN
  {0xC7E6,0x044d}, // CYRILLIC SMALL LETTER E
  {0xC7E7,0x044e}, // CYRILLIC SMALL LETTER YU
  {0xC7E8,0x044f}, // CYRILLIC SMALL LETTER YA
  {0xC7E9,0x2460}, // CIRCLED DIGIT ONE
  {0xC7EA,0x2461}, // CIRCLED DIGIT TWO
  {0xC7EB,0x2462}, // CIRCLED DIGIT THREE
  {0xC7EC,0x2463}, // CIRCLED DIGIT FOUR
  {0xC7ED,0x2464}, // CIRCLED DIGIT FIVE
  {0xC7EE,0x2465}, // CIRCLED DIGIT SIX
  {0xC7EF,0x2466}, // CIRCLED DIGIT SEVEN
  {0xC7F0,0x2467}, // CIRCLED DIGIT EIGHT
  {0xC7F1,0x2468}, // CIRCLED DIGIT NINE
  {0xC7F2,0x2469}, // CIRCLED NUMBER TEN
  {0xC7F3,0x2474}, // PARENTHESIZED DIGIT ONE
  {0xC7F4,0x2475}, // PARENTHESIZED DIGIT TWO
  {0xC7F5,0x2476}, // PARENTHESIZED DIGIT THREE
  {0xC7F6,0x2477}, // PARENTHESIZED DIGIT FOUR
  {0xC7F7,0x2478}, // PARENTHESIZED DIGIT FIVE
  {0xC7F8,0x2479}, // PARENTHESIZED DIGIT SIX
  {0xC7F9,0x247a}, // PARENTHESIZED DIGIT SEVEN
  {0xC7FA,0x247b}, // PARENTHESIZED DIGIT EIGHT
  {0xC7FB,0x247c}, // PARENTHESIZED DIGIT NINE
  {0xC7FC,0x247d}, // PARENTHESIZED NUMBER TEN
  {0xC940,0x4e42}, // <CJK>
  {0xC941,0x4e5c}, // <CJK>
  {0xC942,0x51f5}, // <CJK>
  {0xC943,0x531a}, // <CJK>
  {0xC944,0x5382}, // <CJK>
  {0xC945,0x4e07}, // <CJK>
  {0xC946,0x4e0c}, // <CJK>
  {0xC947,0x4e47}, // <CJK>
  {0xC948,0x4e8d}, // <CJK>
  {0xC949,0x56d7}, // <CJK>
  {0xC94A,0xfa0c}, // CJK COMPATIBILITY IDEOGRAPH-FA0C
  {0xC94B,0x5c6e}, // <CJK>
  {0xC94C,0x5f73}, // <CJK>
  {0xC94D,0x4e0f}, // <CJK>
  {0xC94E,0x5187}, // <CJK>
  {0xC94F,0x4e0e}, // <CJK>
  {0xC950,0x4e2e}, // <CJK>
  {0xC951,0x4e93}, // <CJK>
  {0xC952,0x4ec2}, // <CJK>
  {0xC953,0x4ec9}, // <CJK>
  {0xC954,0x4ec8}, // <CJK>
  {0xC955,0x5198}, // <CJK>
  {0xC956,0x52fc}, // <CJK>
  {0xC957,0x536c}, // <CJK>
  {0xC958,0x53b9}, // <CJK>
  {0xC959,0x5720}, // <CJK>
  {0xC95A,0x5903}, // <CJK>
  {0xC95B,0x592c}, // <CJK>
  {0xC95C,0x5c10}, // <CJK>
  {0xC95D,0x5dff}, // <CJK>
  {0xC95E,0x65e1}, // <CJK>
  {0xC95F,0x6bb3}, // <CJK>
  {0xC960,0x6bcc}, // <CJK>
  {0xC961,0x6c14}, // <CJK>
  {0xC962,0x723f}, // <CJK>
  {0xC963,0x4e31}, // <CJK>
  {0xC964,0x4e3c}, // <CJK>
  {0xC965,0x4ee8}, // <CJK>
  {0xC966,0x4edc}, // <CJK>
  {0xC967,0x4ee9}, // <CJK>
  {0xC968,0x4ee1}, // <CJK>
  {0xC969,0x4edd}, // <CJK>
  {0xC96A,0x4eda}, // <CJK>
  {0xC96B,0x520c}, // <CJK>
  {0xC96C,0x531c}, // <CJK>
  {0xC96D,0x534c}, // <CJK>
  {0xC96E,0x5722}, // <CJK>
  {0xC96F,0x5723}, // <CJK>
  {0xC970,0x5917}, // <CJK>
  {0xC971,0x592f}, // <CJK>
  {0xC972,0x5b81}, // <CJK>
  {0xC973,0x5b84}, // <CJK>
  {0xC974,0x5c12}, // <CJK>
  {0xC975,0x5c3b}, // <CJK>
  {0xC976,0x5c74}, // <CJK>
  {0xC977,0x5c73}, // <CJK>
  {0xC978,0x5e04}, // <CJK>
  {0xC979,0x5e80}, // <CJK>
  {0xC97A,0x5e82}, // <CJK>
  {0xC97B,0x5fc9}, // <CJK>
  {0xC97C,0x6209}, // <CJK>
  {0xC97D,0x6250}, // <CJK>
  {0xC97E,0x6c15}, // <CJK>
  {0xC9A1,0x6c36}, // <CJK>
  {0xC9A2,0x6c43}, // <CJK>
  {0xC9A3,0x6c3f}, // <CJK>
  {0xC9A4,0x6c3b}, // <CJK>
  {0xC9A5,0x72ae}, // <CJK>
  {0xC9A6,0x72b0}, // <CJK>
  {0xC9A7,0x738a}, // <CJK>
  {0xC9A8,0x79b8}, // <CJK>
  {0xC9A9,0x808a}, // <CJK>
  {0xC9AA,0x961e}, // <CJK>
  {0xC9AB,0x4f0e}, // <CJK>
  {0xC9AC,0x4f18}, // <CJK>
  {0xC9AD,0x4f2c}, // <CJK>
  {0xC9AE,0x4ef5}, // <CJK>
  {0xC9AF,0x4f14}, // <CJK>
  {0xC9B0,0x4ef1}, // <CJK>
  {0xC9B1,0x4f00}, // <CJK>
  {0xC9B2,0x4ef7}, // <CJK>
  {0xC9B3,0x4f08}, // <CJK>
  {0xC9B4,0x4f1d}, // <CJK>
  {0xC9B5,0x4f02}, // <CJK>
  {0xC9B6,0x4f05}, // <CJK>
  {0xC9B7,0x4f22}, // <CJK>
  {0xC9B8,0x4f13}, // <CJK>
  {0xC9B9,0x4f04}, // <CJK>
  {0xC9BA,0x4ef4}, // <CJK>
  {0xC9BB,0x4f12}, // <CJK>
  {0xC9BC,0x51b1}, // <CJK>
  {0xC9BD,0x5213}, // <CJK>
  {0xC9BE,0x5209}, // <CJK>
  {0xC9BF,0x5210}, // <CJK>
  {0xC9C0,0x52a6}, // <CJK>
  {0xC9C1,0x5322}, // <CJK>
  {0xC9C2,0x531f}, // <CJK>
  {0xC9C3,0x534d}, // <CJK>
  {0xC9C4,0x538a}, // <CJK>
  {0xC9C5,0x5407}, // <CJK>
  {0xC9C6,0x56e1}, // <CJK>
  {0xC9C7,0x56df}, // <CJK>
  {0xC9C8,0x572e}, // <CJK>
  {0xC9C9,0x572a}, // <CJK>
  {0xC9CA,0x5734}, // <CJK>
  {0xC9CB,0x593c}, // <CJK>
  {0xC9CC,0x5980}, // <CJK>
  {0xC9CD,0x597c}, // <CJK>
  {0xC9CE,0x5985}, // <CJK>
  {0xC9CF,0x597b}, // <CJK>
  {0xC9D0,0x597e}, // <CJK>
  {0xC9D1,0x5977}, // <CJK>
  {0xC9D2,0x597f}, // <CJK>
  {0xC9D3,0x5b56}, // <CJK>
  {0xC9D4,0x5c15}, // <CJK>
  {0xC9D5,0x5c25}, // <CJK>
  {0xC9D6,0x5c7c}, // <CJK>
  {0xC9D7,0x5c7a}, // <CJK>
  {0xC9D8,0x5c7b}, // <CJK>
  {0xC9D9,0x5c7e}, // <CJK>
  {0xC9DA,0x5ddf}, // <CJK>
  {0xC9DB,0x5e75}, // <CJK>
  {0xC9DC,0x5e84}, // <CJK>
  {0xC9DD,0x5f02}, // <CJK>
  {0xC9DE,0x5f1a}, // <CJK>
  {0xC9DF,0x5f74}, // <CJK>
  {0xC9E0,0x5fd5}, // <CJK>
  {0xC9E1,0x5fd4}, // <CJK>
  {0xC9E2,0x5fcf}, // <CJK>
  {0xC9E3,0x625c}, // <CJK>
  {0xC9E4,0x625e}, // <CJK>
  {0xC9E5,0x6264}, // <CJK>
  {0xC9E6,0x6261}, // <CJK>
  {0xC9E7,0x6266}, // <CJK>
  {0xC9E8,0x6262}, // <CJK>
  {0xC9E9,0x6259}, // <CJK>
  {0xC9EA,0x6260}, // <CJK>
  {0xC9EB,0x625a}, // <CJK>
  {0xC9EC,0x6265}, // <CJK>
  {0xC9ED,0x65ef}, // <CJK>
  {0xC9EE,0x65ee}, // <CJK>
  {0xC9EF,0x673e}, // <CJK>
  {0xC9F0,0x6739}, // <CJK>
  {0xC9F1,0x6738}, // <CJK>
  {0xC9F2,0x673b}, // <CJK>
  {0xC9F3,0x673a}, // <CJK>
  {0xC9F4,0x673f}, // <CJK>
  {0xC9F5,0x673c}, // <CJK>
  {0xC9F6,0x6733}, // <CJK>
  {0xC9F7,0x6c18}, // <CJK>
  {0xC9F8,0x6c46}, // <CJK>
  {0xC9F9,0x6c52}, // <CJK>
  {0xC9FA,0x6c5c}, // <CJK>
  {0xC9FB,0x6c4f}, // <CJK>
  {0xC9FC,0x6c4a}, // <CJK>
  {0xC9FD,0x6c54}, // <CJK>
  {0xC9FE,0x6c4b}, // <CJK>
  {0xCA40,0x6c4c}, // <CJK>
  {0xCA41,0x7071}, // <CJK>
  {0xCA42,0x725e}, // <CJK>
  {0xCA43,0x72b4}, // <CJK>
  {0xCA44,0x72b5}, // <CJK>
  {0xCA45,0x738e}, // <CJK>
  {0xCA46,0x752a}, // <CJK>
  {0xCA47,0x767f}, // <CJK>
  {0xCA48,0x7a75}, // <CJK>
  {0xCA49,0x7f51}, // <CJK>
  {0xCA4A,0x8278}, // <CJK>
  {0xCA4B,0x827c}, // <CJK>
  {0xCA4C,0x8280}, // <CJK>
  {0xCA4D,0x827d}, // <CJK>
  {0xCA4E,0x827f}, // <CJK>
  {0xCA4F,0x864d}, // <CJK>
  {0xCA50,0x897e}, // <CJK>
  {0xCA51,0x9099}, // <CJK>
  {0xCA52,0x9097}, // <CJK>
  {0xCA53,0x9098}, // <CJK>
  {0xCA54,0x909b}, // <CJK>
  {0xCA55,0x9094}, // <CJK>
  {0xCA56,0x9622}, // <CJK>
  {0xCA57,0x9624}, // <CJK>
  {0xCA58,0x9620}, // <CJK>
  {0xCA59,0x9623}, // <CJK>
  {0xCA5A,0x4f56}, // <CJK>
  {0xCA5B,0x4f3b}, // <CJK>
  {0xCA5C,0x4f62}, // <CJK>
  {0xCA5D,0x4f49}, // <CJK>
  {0xCA5E,0x4f53}, // <CJK>
  {0xCA5F,0x4f64}, // <CJK>
  {0xCA60,0x4f3e}, // <CJK>
  {0xCA61,0x4f67}, // <CJK>
  {0xCA62,0x4f52}, // <CJK>
  {0xCA63,0x4f5f}, // <CJK>
  {0xCA64,0x4f41}, // <CJK>
  {0xCA65,0x4f58}, // <CJK>
  {0xCA66,0x4f2d}, // <CJK>
  {0xCA67,0x4f33}, // <CJK>
  {0xCA68,0x4f3f}, // <CJK>
  {0xCA69,0x4f61}, // <CJK>
  {0xCA6A,0x518f}, // <CJK>
  {0xCA6B,0x51b9}, // <CJK>
  {0xCA6C,0x521c}, // <CJK>
  {0xCA6D,0x521e}, // <CJK>
  {0xCA6E,0x5221}, // <CJK>
  {0xCA6F,0x52ad}, // <CJK>
  {0xCA70,0x52ae}, // <CJK>
  {0xCA71,0x5309}, // <CJK>
  {0xCA72,0x5363}, // <CJK>
  {0xCA73,0x5372}, // <CJK>
  {0xCA74,0x538e}, // <CJK>
  {0xCA75,0x538f}, // <CJK>
  {0xCA76,0x5430}, // <CJK>
  {0xCA77,0x5437}, // <CJK>
  {0xCA78,0x542a}, // <CJK>
  {0xCA79,0x5454}, // <CJK>
  {0xCA7A,0x5445}, // <CJK>
  {0xCA7B,0x5419}, // <CJK>
  {0xCA7C,0x541c}, // <CJK>
  {0xCA7D,0x5425}, // <CJK>
  {0xCA7E,0x5418}, // <CJK>
  {0xCAA1,0x543d}, // <CJK>
  {0xCAA2,0x544f}, // <CJK>
  {0xCAA3,0x5441}, // <CJK>
  {0xCAA4,0x5428}, // <CJK>
  {0xCAA5,0x5424}, // <CJK>
  {0xCAA6,0x5447}, // <CJK>
  {0xCAA7,0x56ee}, // <CJK>
  {0xCAA8,0x56e7}, // <CJK>
  {0xCAA9,0x56e5}, // <CJK>
  {0xCAAA,0x5741}, // <CJK>
  {0xCAAB,0x5745}, // <CJK>
  {0xCAAC,0x574c}, // <CJK>
  {0xCAAD,0x5749}, // <CJK>
  {0xCAAE,0x574b}, // <CJK>
  {0xCAAF,0x5752}, // <CJK>
  {0xCAB0,0x5906}, // <CJK>
  {0xCAB1,0x5940}, // <CJK>
  {0xCAB2,0x59a6}, // <CJK>
  {0xCAB3,0x5998}, // <CJK>
  {0xCAB4,0x59a0}, // <CJK>
  {0xCAB5,0x5997}, // <CJK>
  {0xCAB6,0x598e}, // <CJK>
  {0xCAB7,0x59a2}, // <CJK>
  {0xCAB8,0x5990}, // <CJK>
  {0xCAB9,0x598f}, // <CJK>
  {0xCABA,0x59a7}, // <CJK>
  {0xCABB,0x59a1}, // <CJK>
  {0xCABC,0x5b8e}, // <CJK>
  {0xCABD,0x5b92}, // <CJK>
  {0xCABE,0x5c28}, // <CJK>
  {0xCABF,0x5c2a}, // <CJK>
  {0xCAC0,0x5c8d}, // <CJK>
  {0xCAC1,0x5c8f}, // <CJK>
  {0xCAC2,0x5c88}, // <CJK>
  {0xCAC3,0x5c8b}, // <CJK>
  {0xCAC4,0x5c89}, // <CJK>
  {0xCAC5,0x5c92}, // <CJK>
  {0xCAC6,0x5c8a}, // <CJK>
  {0xCAC7,0x5c86}, // <CJK>
  {0xCAC8,0x5c93}, // <CJK>
  {0xCAC9,0x5c95}, // <CJK>
  {0xCACA,0x5de0}, // <CJK>
  {0xCACB,0x5e0a}, // <CJK>
  {0xCACC,0x5e0e}, // <CJK>
  {0xCACD,0x5e8b}, // <CJK>
  {0xCACE,0x5e89}, // <CJK>
  {0xCACF,0x5e8c}, // <CJK>
  {0xCAD0,0x5e88}, // <CJK>
  {0xCAD1,0x5e8d}, // <CJK>
  {0xCAD2,0x5f05}, // <CJK>
  {0xCAD3,0x5f1d}, // <CJK>
  {0xCAD4,0x5f78}, // <CJK>
  {0xCAD5,0x5f76}, // <CJK>
  {0xCAD6,0x5fd2}, // <CJK>
  {0xCAD7,0x5fd1}, // <CJK>
  {0xCAD8,0x5fd0}, // <CJK>
  {0xCAD9,0x5fed}, // <CJK>
  {0xCADA,0x5fe8}, // <CJK>
  {0xCADB,0x5fee}, // <CJK>
  {0xCADC,0x5ff3}, // <CJK>
  {0xCADD,0x5fe1}, // <CJK>
  {0xCADE,0x5fe4}, // <CJK>
  {0xCADF,0x5fe3}, // <CJK>
  {0xCAE0,0x5ffa}, // <CJK>
  {0xCAE1,0x5fef}, // <CJK>
  {0xCAE2,0x5ff7}, // <CJK>
  {0xCAE3,0x5ffb}, // <CJK>
  {0xCAE4,0x6000}, // <CJK>
  {0xCAE5,0x5ff4}, // <CJK>
  {0xCAE6,0x623a}, // <CJK>
  {0xCAE7,0x6283}, // <CJK>
  {0xCAE8,0x628c}, // <CJK>
  {0xCAE9,0x628e}, // <CJK>
  {0xCAEA,0x628f}, // <CJK>
  {0xCAEB,0x6294}, // <CJK>
  {0xCAEC,0x6287}, // <CJK>
  {0xCAED,0x6271}, // <CJK>
  {0xCAEE,0x627b}, // <CJK>
  {0xCAEF,0x627a}, // <CJK>
  {0xCAF0,0x6270}, // <CJK>
  {0xCAF1,0x6281}, // <CJK>
  {0xCAF2,0x6288}, // <CJK>
  {0xCAF3,0x6277}, // <CJK>
  {0xCAF4,0x627d}, // <CJK>
  {0xCAF5,0x6272}, // <CJK>
  {0xCAF6,0x6274}, // <CJK>
  {0xCAF7,0x6537}, // <CJK>
  {0xCAF8,0x65f0}, // <CJK>
  {0xCAF9,0x65f4}, // <CJK>
  {0xCAFA,0x65f3}, // <CJK>
  {0xCAFB,0x65f2}, // <CJK>
  {0xCAFC,0x65f5}, // <CJK>
  {0xCAFD,0x6745}, // <CJK>
  {0xCAFE,0x6747}, // <CJK>
  {0xCB40,0x6759}, // <CJK>
  {0xCB41,0x6755}, // <CJK>
  {0xCB42,0x674c}, // <CJK>
  {0xCB43,0x6748}, // <CJK>
  {0xCB44,0x675d}, // <CJK>
  {0xCB45,0x674d}, // <CJK>
  {0xCB46,0x675a}, // <CJK>
  {0xCB47,0x674b}, // <CJK>
  {0xCB48,0x6bd0}, // <CJK>
  {0xCB49,0x6c19}, // <CJK>
  {0xCB4A,0x6c1a}, // <CJK>
  {0xCB4B,0x6c78}, // <CJK>
  {0xCB4C,0x6c67}, // <CJK>
  {0xCB4D,0x6c6b}, // <CJK>
  {0xCB4E,0x6c84}, // <CJK>
  {0xCB4F,0x6c8b}, // <CJK>
  {0xCB50,0x6c8f}, // <CJK>
  {0xCB51,0x6c71}, // <CJK>
  {0xCB52,0x6c6f}, // <CJK>
  {0xCB53,0x6c69}, // <CJK>
  {0xCB54,0x6c9a}, // <CJK>
  {0xCB55,0x6c6d}, // <CJK>
  {0xCB56,0x6c87}, // <CJK>
  {0xCB57,0x6c95}, // <CJK>
  {0xCB58,0x6c9c}, // <CJK>
  {0xCB59,0x6c66}, // <CJK>
  {0xCB5A,0x6c73}, // <CJK>
  {0xCB5B,0x6c65}, // <CJK>
  {0xCB5C,0x6c7b}, // <CJK>
  {0xCB5D,0x6c8e}, // <CJK>
  {0xCB5E,0x7074}, // <CJK>
  {0xCB5F,0x707a}, // <CJK>
  {0xCB60,0x7263}, // <CJK>
  {0xCB61,0x72bf}, // <CJK>
  {0xCB62,0x72bd}, // <CJK>
  {0xCB63,0x72c3}, // <CJK>
  {0xCB64,0x72c6}, // <CJK>
  {0xCB65,0x72c1}, // <CJK>
  {0xCB66,0x72ba}, // <CJK>
  {0xCB67,0x72c5}, // <CJK>
  {0xCB68,0x7395}, // <CJK>
  {0xCB69,0x7397}, // <CJK>
  {0xCB6A,0x7393}, // <CJK>
  {0xCB6B,0x7394}, // <CJK>
  {0xCB6C,0x7392}, // <CJK>
  {0xCB6D,0x753a}, // <CJK>
  {0xCB6E,0x7539}, // <CJK>
  {0xCB6F,0x7594}, // <CJK>
  {0xCB70,0x7595}, // <CJK>
  {0xCB71,0x7681}, // <CJK>
  {0xCB72,0x793d}, // <CJK>
  {0xCB73,0x8034}, // <CJK>
  {0xCB74,0x8095}, // <CJK>
  {0xCB75,0x8099}, // <CJK>
  {0xCB76,0x8090}, // <CJK>
  {0xCB77,0x8092}, // <CJK>
  {0xCB78,0x809c}, // <CJK>
  {0xCB79,0x8290}, // <CJK>
  {0xCB7A,0x828f}, // <CJK>
  {0xCB7B,0x8285}, // <CJK>
  {0xCB7C,0x828e}, // <CJK>
  {0xCB7D,0x8291}, // <CJK>
  {0xCB7E,0x8293}, // <CJK>
  {0xCBA1,0x828a}, // <CJK>
  {0xCBA2,0x8283}, // <CJK>
  {0xCBA3,0x8284}, // <CJK>
  {0xCBA4,0x8c78}, // <CJK>
  {0xCBA5,0x8fc9}, // <CJK>
  {0xCBA6,0x8fbf}, // <CJK>
  {0xCBA7,0x909f}, // <CJK>
  {0xCBA8,0x90a1}, // <CJK>
  {0xCBA9,0x90a5}, // <CJK>
  {0xCBAA,0x909e}, // <CJK>
  {0xCBAB,0x90a7}, // <CJK>
  {0xCBAC,0x90a0}, // <CJK>
  {0xCBAD,0x9630}, // <CJK>
  {0xCBAE,0x9628}, // <CJK>
  {0xCBAF,0x962f}, // <CJK>
  {0xCBB0,0x962d}, // <CJK>
  {0xCBB1,0x4e33}, // <CJK>
  {0xCBB2,0x4f98}, // <CJK>
  {0xCBB3,0x4f7c}, // <CJK>
  {0xCBB4,0x4f85}, // <CJK>
  {0xCBB5,0x4f7d}, // <CJK>
  {0xCBB6,0x4f80}, // <CJK>
  {0xCBB7,0x4f87}, // <CJK>
  {0xCBB8,0x4f76}, // <CJK>
  {0xCBB9,0x4f74}, // <CJK>
  {0xCBBA,0x4f89}, // <CJK>
  {0xCBBB,0x4f84}, // <CJK>
  {0xCBBC,0x4f77}, // <CJK>
  {0xCBBD,0x4f4c}, // <CJK>
  {0xCBBE,0x4f97}, // <CJK>
  {0xCBBF,0x4f6a}, // <CJK>
  {0xCBC0,0x4f9a}, // <CJK>
  {0xCBC1,0x4f79}, // <CJK>
  {0xCBC2,0x4f81}, // <CJK>
  {0xCBC3,0x4f78}, // <CJK>
  {0xCBC4,0x4f90}, // <CJK>
  {0xCBC5,0x4f9c}, // <CJK>
  {0xCBC6,0x4f94}, // <CJK>
  {0xCBC7,0x4f9e}, // <CJK>
  {0xCBC8,0x4f92}, // <CJK>
  {0xCBC9,0x4f82}, // <CJK>
  {0xCBCA,0x4f95}, // <CJK>
  {0xCBCB,0x4f6b}, // <CJK>
  {0xCBCC,0x4f6e}, // <CJK>
  {0xCBCD,0x519e}, // <CJK>
  {0xCBCE,0x51bc}, // <CJK>
  {0xCBCF,0x51be}, // <CJK>
  {0xCBD0,0x5235}, // <CJK>
  {0xCBD1,0x5232}, // <CJK>
  {0xCBD2,0x5233}, // <CJK>
  {0xCBD3,0x5246}, // <CJK>
  {0xCBD4,0x5231}, // <CJK>
  {0xCBD5,0x52bc}, // <CJK>
  {0xCBD6,0x530a}, // <CJK>
  {0xCBD7,0x530b}, // <CJK>
  {0xCBD8,0x533c}, // <CJK>
  {0xCBD9,0x5392}, // <CJK>
  {0xCBDA,0x5394}, // <CJK>
  {0xCBDB,0x5487}, // <CJK>
  {0xCBDC,0x547f}, // <CJK>
  {0xCBDD,0x5481}, // <CJK>
  {0xCBDE,0x5491}, // <CJK>
  {0xCBDF,0x5482}, // <CJK>
  {0xCBE0,0x5488}, // <CJK>
  {0xCBE1,0x546b}, // <CJK>
  {0xCBE2,0x547a}, // <CJK>
  {0xCBE3,0x547e}, // <CJK>
  {0xCBE4,0x5465}, // <CJK>
  {0xCBE5,0x546c}, // <CJK>
  {0xCBE6,0x5474}, // <CJK>
  {0xCBE7,0x5466}, // <CJK>
  {0xCBE8,0x548d}, // <CJK>
  {0xCBE9,0x546f}, // <CJK>
  {0xCBEA,0x5461}, // <CJK>
  {0xCBEB,0x5460}, // <CJK>
  {0xCBEC,0x5498}, // <CJK>
  {0xCBED,0x5463}, // <CJK>
  {0xCBEE,0x5467}, // <CJK>
  {0xCBEF,0x5464}, // <CJK>
  {0xCBF0,0x56f7}, // <CJK>
  {0xCBF1,0x56f9}, // <CJK>
  {0xCBF2,0x576f}, // <CJK>
  {0xCBF3,0x5772}, // <CJK>
  {0xCBF4,0x576d}, // <CJK>
  {0xCBF5,0x576b}, // <CJK>
  {0xCBF6,0x5771}, // <CJK>
  {0xCBF7,0x5770}, // <CJK>
  {0xCBF8,0x5776}, // <CJK>
  {0xCBF9,0x5780}, // <CJK>
  {0xCBFA,0x5775}, // <CJK>
  {0xCBFB,0x577b}, // <CJK>
  {0xCBFC,0x5773}, // <CJK>
  {0xCBFD,0x5774}, // <CJK>
  {0xCBFE,0x5762}, // <CJK>
  {0xCC40,0x5768}, // <CJK>
  {0xCC41,0x577d}, // <CJK>
  {0xCC42,0x590c}, // <CJK>
  {0xCC43,0x5945}, // <CJK>
  {0xCC44,0x59b5}, // <CJK>
  {0xCC45,0x59ba}, // <CJK>
  {0xCC46,0x59cf}, // <CJK>
  {0xCC47,0x59ce}, // <CJK>
  {0xCC48,0x59b2}, // <CJK>
  {0xCC49,0x59cc}, // <CJK>
  {0xCC4A,0x59c1}, // <CJK>
  {0xCC4B,0x59b6}, // <CJK>
  {0xCC4C,0x59bc}, // <CJK>
  {0xCC4D,0x59c3}, // <CJK>
  {0xCC4E,0x59d6}, // <CJK>
  {0xCC4F,0x59b1}, // <CJK>
  {0xCC50,0x59bd}, // <CJK>
  {0xCC51,0x59c0}, // <CJK>
  {0xCC52,0x59c8}, // <CJK>
  {0xCC53,0x59b4}, // <CJK>
  {0xCC54,0x59c7}, // <CJK>
  {0xCC55,0x5b62}, // <CJK>
  {0xCC56,0x5b65}, // <CJK>
  {0xCC57,0x5b93}, // <CJK>
  {0xCC58,0x5b95}, // <CJK>
  {0xCC59,0x5c44}, // <CJK>
  {0xCC5A,0x5c47}, // <CJK>
  {0xCC5B,0x5cae}, // <CJK>
  {0xCC5C,0x5ca4}, // <CJK>
  {0xCC5D,0x5ca0}, // <CJK>
  {0xCC5E,0x5cb5}, // <CJK>
  {0xCC5F,0x5caf}, // <CJK>
  {0xCC60,0x5ca8}, // <CJK>
  {0xCC61,0x5cac}, // <CJK>
  {0xCC62,0x5c9f}, // <CJK>
  {0xCC63,0x5ca3}, // <CJK>
  {0xCC64,0x5cad}, // <CJK>
  {0xCC65,0x5ca2}, // <CJK>
  {0xCC66,0x5caa}, // <CJK>
  {0xCC67,0x5ca7}, // <CJK>
  {0xCC68,0x5c9d}, // <CJK>
  {0xCC69,0x5ca5}, // <CJK>
  {0xCC6A,0x5cb6}, // <CJK>
  {0xCC6B,0x5cb0}, // <CJK>
  {0xCC6C,0x5ca6}, // <CJK>
  {0xCC6D,0x5e17}, // <CJK>
  {0xCC6E,0x5e14}, // <CJK>
  {0xCC6F,0x5e19}, // <CJK>
  {0xCC70,0x5f28}, // <CJK>
  {0xCC71,0x5f22}, // <CJK>
  {0xCC72,0x5f23}, // <CJK>
  {0xCC73,0x5f24}, // <CJK>
  {0xCC74,0x5f54}, // <CJK>
  {0xCC75,0x5f82}, // <CJK>
  {0xCC76,0x5f7e}, // <CJK>
  {0xCC77,0x5f7d}, // <CJK>
  {0xCC78,0x5fde}, // <CJK>
  {0xCC79,0x5fe5}, // <CJK>
  {0xCC7A,0x602d}, // <CJK>
  {0xCC7B,0x6026}, // <CJK>
  {0xCC7C,0x6019}, // <CJK>
  {0xCC7D,0x6032}, // <CJK>
  {0xCC7E,0x600b}, // <CJK>
  {0xCCA1,0x6034}, // <CJK>
  {0xCCA2,0x600a}, // <CJK>
  {0xCCA3,0x6017}, // <CJK>
  {0xCCA4,0x6033}, // <CJK>
  {0xCCA5,0x601a}, // <CJK>
  {0xCCA6,0x601e}, // <CJK>
  {0xCCA7,0x602c}, // <CJK>
  {0xCCA8,0x6022}, // <CJK>
  {0xCCA9,0x600d}, // <CJK>
  {0xCCAA,0x6010}, // <CJK>
  {0xCCAB,0x602e}, // <CJK>
  {0xCCAC,0x6013}, // <CJK>
  {0xCCAD,0x6011}, // <CJK>
  {0xCCAE,0x600c}, // <CJK>
  {0xCCAF,0x6009}, // <CJK>
  {0xCCB0,0x601c}, // <CJK>
  {0xCCB1,0x6214}, // <CJK>
  {0xCCB2,0x623d}, // <CJK>
  {0xCCB3,0x62ad}, // <CJK>
  {0xCCB4,0x62b4}, // <CJK>
  {0xCCB5,0x62d1}, // <CJK>
  {0xCCB6,0x62be}, // <CJK>
  {0xCCB7,0x62aa}, // <CJK>
  {0xCCB8,0x62b6}, // <CJK>
  {0xCCB9,0x62ca}, // <CJK>
  {0xCCBA,0x62ae}, // <CJK>
  {0xCCBB,0x62b3}, // <CJK>
  {0xCCBC,0x62af}, // <CJK>
  {0xCCBD,0x62bb}, // <CJK>
  {0xCCBE,0x62a9}, // <CJK>
  {0xCCBF,0x62b0}, // <CJK>
  {0xCCC0,0x62b8}, // <CJK>
  {0xCCC1,0x653d}, // <CJK>
  {0xCCC2,0x65a8}, // <CJK>
  {0xCCC3,0x65bb}, // <CJK>
  {0xCCC4,0x6609}, // <CJK>
  {0xCCC5,0x65fc}, // <CJK>
  {0xCCC6,0x6604}, // <CJK>
  {0xCCC7,0x6612}, // <CJK>
  {0xCCC8,0x6608}, // <CJK>
  {0xCCC9,0x65fb}, // <CJK>
  {0xCCCA,0x6603}, // <CJK>
  {0xCCCB,0x660b}, // <CJK>
  {0xCCCC,0x660d}, // <CJK>
  {0xCCCD,0x6605}, // <CJK>
  {0xCCCE,0x65fd}, // <CJK>
  {0xCCCF,0x6611}, // <CJK>
  {0xCCD0,0x6610}, // <CJK>
  {0xCCD1,0x66f6}, // <CJK>
  {0xCCD2,0x670a}, // <CJK>
  {0xCCD3,0x6785}, // <CJK>
  {0xCCD4,0x676c}, // <CJK>
  {0xCCD5,0x678e}, // <CJK>
  {0xCCD6,0x6792}, // <CJK>
  {0xCCD7,0x6776}, // <CJK>
  {0xCCD8,0x677b}, // <CJK>
  {0xCCD9,0x6798}, // <CJK>
  {0xCCDA,0x6786}, // <CJK>
  {0xCCDB,0x6784}, // <CJK>
  {0xCCDC,0x6774}, // <CJK>
  {0xCCDD,0x678d}, // <CJK>
  {0xCCDE,0x678c}, // <CJK>
  {0xCCDF,0x677a}, // <CJK>
  {0xCCE0,0x679f}, // <CJK>
  {0xCCE1,0x6791}, // <CJK>
  {0xCCE2,0x6799}, // <CJK>
  {0xCCE3,0x6783}, // <CJK>
  {0xCCE4,0x677d}, // <CJK>
  {0xCCE5,0x6781}, // <CJK>
  {0xCCE6,0x6778}, // <CJK>
  {0xCCE7,0x6779}, // <CJK>
  {0xCCE8,0x6794}, // <CJK>
  {0xCCE9,0x6b25}, // <CJK>
  {0xCCEA,0x6b80}, // <CJK>
  {0xCCEB,0x6b7e}, // <CJK>
  {0xCCEC,0x6bde}, // <CJK>
  {0xCCED,0x6c1d}, // <CJK>
  {0xCCEE,0x6c93}, // <CJK>
  {0xCCEF,0x6cec}, // <CJK>
  {0xCCF0,0x6ceb}, // <CJK>
  {0xCCF1,0x6cee}, // <CJK>
  {0xCCF2,0x6cd9}, // <CJK>
  {0xCCF3,0x6cb6}, // <CJK>
  {0xCCF4,0x6cd4}, // <CJK>
  {0xCCF5,0x6cad}, // <CJK>
  {0xCCF6,0x6ce7}, // <CJK>
  {0xCCF7,0x6cb7}, // <CJK>
  {0xCCF8,0x6cd0}, // <CJK>
  {0xCCF9,0x6cc2}, // <CJK>
  {0xCCFA,0x6cba}, // <CJK>
  {0xCCFB,0x6cc3}, // <CJK>
  {0xCCFC,0x6cc6}, // <CJK>
  {0xCCFD,0x6ced}, // <CJK>
  {0xCCFE,0x6cf2}, // <CJK>
  {0xCD40,0x6cd2}, // <CJK>
  {0xCD41,0x6cdd}, // <CJK>
  {0xCD42,0x6cb4}, // <CJK>
  {0xCD43,0x6c8a}, // <CJK>
  {0xCD44,0x6c9d}, // <CJK>
  {0xCD45,0x6c80}, // <CJK>
  {0xCD46,0x6cde}, // <CJK>
  {0xCD47,0x6cc0}, // <CJK>
  {0xCD48,0x6d30}, // <CJK>
  {0xCD49,0x6ccd}, // <CJK>
  {0xCD4A,0x6cc7}, // <CJK>
  {0xCD4B,0x6cb0}, // <CJK>
  {0xCD4C,0x6cf9}, // <CJK>
  {0xCD4D,0x6ccf}, // <CJK>
  {0xCD4E,0x6ce9}, // <CJK>
  {0xCD4F,0x6cd1}, // <CJK>
  {0xCD50,0x7094}, // <CJK>
  {0xCD51,0x7098}, // <CJK>
  {0xCD52,0x7085}, // <CJK>
  {0xCD53,0x7093}, // <CJK>
  {0xCD54,0x7086}, // <CJK>
  {0xCD55,0x7084}, // <CJK>
  {0xCD56,0x7091}, // <CJK>
  {0xCD57,0x7096}, // <CJK>
  {0xCD58,0x7082}, // <CJK>
  {0xCD59,0x709a}, // <CJK>
  {0xCD5A,0x7083}, // <CJK>
  {0xCD5B,0x726a}, // <CJK>
  {0xCD5C,0x72d6}, // <CJK>
  {0xCD5D,0x72cb}, // <CJK>
  {0xCD5E,0x72d8}, // <CJK>
  {0xCD5F,0x72c9}, // <CJK>
  {0xCD60,0x72dc}, // <CJK>
  {0xCD61,0x72d2}, // <CJK>
  {0xCD62,0x72d4}, // <CJK>
  {0xCD63,0x72da}, // <CJK>
  {0xCD64,0x72cc}, // <CJK>
  {0xCD65,0x72d1}, // <CJK>
  {0xCD66,0x73a4}, // <CJK>
  {0xCD67,0x73a1}, // <CJK>
  {0xCD68,0x73ad}, // <CJK>
  {0xCD69,0x73a6}, // <CJK>
  {0xCD6A,0x73a2}, // <CJK>
  {0xCD6B,0x73a0}, // <CJK>
  {0xCD6C,0x73ac}, // <CJK>
  {0xCD6D,0x739d}, // <CJK>
  {0xCD6E,0x74dd}, // <CJK>
  {0xCD6F,0x74e8}, // <CJK>
  {0xCD70,0x753f}, // <CJK>
  {0xCD71,0x7540}, // <CJK>
  {0xCD72,0x753e}, // <CJK>
  {0xCD73,0x758c}, // <CJK>
  {0xCD74,0x7598}, // <CJK>
  {0xCD75,0x76af}, // <CJK>
  {0xCD76,0x76f3}, // <CJK>
  {0xCD77,0x76f1}, // <CJK>
  {0xCD78,0x76f0}, // <CJK>
  {0xCD79,0x76f5}, // <CJK>
  {0xCD7A,0x77f8}, // <CJK>
  {0xCD7B,0x77fc}, // <CJK>
  {0xCD7C,0x77f9}, // <CJK>
  {0xCD7D,0x77fb}, // <CJK>
  {0xCD7E,0x77fa}, // <CJK>
  {0xCDA1,0x77f7}, // <CJK>
  {0xCDA2,0x7942}, // <CJK>
  {0xCDA3,0x793f}, // <CJK>
  {0xCDA4,0x79c5}, // <CJK>
  {0xCDA5,0x7a78}, // <CJK>
  {0xCDA6,0x7a7b}, // <CJK>
  {0xCDA7,0x7afb}, // <CJK>
  {0xCDA8,0x7c75}, // <CJK>
  {0xCDA9,0x7cfd}, // <CJK>
  {0xCDAA,0x8035}, // <CJK>
  {0xCDAB,0x808f}, // <CJK>
  {0xCDAC,0x80ae}, // <CJK>
  {0xCDAD,0x80a3}, // <CJK>
  {0xCDAE,0x80b8}, // <CJK>
  {0xCDAF,0x80b5}, // <CJK>
  {0xCDB0,0x80ad}, // <CJK>
  {0xCDB1,0x8220}, // <CJK>
  {0xCDB2,0x82a0}, // <CJK>
  {0xCDB3,0x82c0}, // <CJK>
  {0xCDB4,0x82ab}, // <CJK>
  {0xCDB5,0x829a}, // <CJK>
  {0xCDB6,0x8298}, // <CJK>
  {0xCDB7,0x829b}, // <CJK>
  {0xCDB8,0x82b5}, // <CJK>
  {0xCDB9,0x82a7}, // <CJK>
  {0xCDBA,0x82ae}, // <CJK>
  {0xCDBB,0x82bc}, // <CJK>
  {0xCDBC,0x829e}, // <CJK>
  {0xCDBD,0x82ba}, // <CJK>
  {0xCDBE,0x82b4}, // <CJK>
  {0xCDBF,0x82a8}, // <CJK>
  {0xCDC0,0x82a1}, // <CJK>
  {0xCDC1,0x82a9}, // <CJK>
  {0xCDC2,0x82c2}, // <CJK>
  {0xCDC3,0x82a4}, // <CJK>
  {0xCDC4,0x82c3}, // <CJK>
  {0xCDC5,0x82b6}, // <CJK>
  {0xCDC6,0x82a2}, // <CJK>
  {0xCDC7,0x8670}, // <CJK>
  {0xCDC8,0x866f}, // <CJK>
  {0xCDC9,0x866d}, // <CJK>
  {0xCDCA,0x866e}, // <CJK>
  {0xCDCB,0x8c56}, // <CJK>
  {0xCDCC,0x8fd2}, // <CJK>
  {0xCDCD,0x8fcb}, // <CJK>
  {0xCDCE,0x8fd3}, // <CJK>
  {0xCDCF,0x8fcd}, // <CJK>
  {0xCDD0,0x8fd6}, // <CJK>
  {0xCDD1,0x8fd5}, // <CJK>
  {0xCDD2,0x8fd7}, // <CJK>
  {0xCDD3,0x90b2}, // <CJK>
  {0xCDD4,0x90b4}, // <CJK>
  {0xCDD5,0x90af}, // <CJK>
  {0xCDD6,0x90b3}, // <CJK>
  {0xCDD7,0x90b0}, // <CJK>
  {0xCDD8,0x9639}, // <CJK>
  {0xCDD9,0x963d}, // <CJK>
  {0xCDDA,0x963c}, // <CJK>
  {0xCDDB,0x963a}, // <CJK>
  {0xCDDC,0x9643}, // <CJK>
  {0xCDDD,0x4fcd}, // <CJK>
  {0xCDDE,0x4fc5}, // <CJK>
  {0xCDDF,0x4fd3}, // <CJK>
  {0xCDE0,0x4fb2}, // <CJK>
  {0xCDE1,0x4fc9}, // <CJK>
  {0xCDE2,0x4fcb}, // <CJK>
  {0xCDE3,0x4fc1}, // <CJK>
  {0xCDE4,0x4fd4}, // <CJK>
  {0xCDE5,0x4fdc}, // <CJK>
  {0xCDE6,0x4fd9}, // <CJK>
  {0xCDE7,0x4fbb}, // <CJK>
  {0xCDE8,0x4fb3}, // <CJK>
  {0xCDE9,0x4fdb}, // <CJK>
  {0xCDEA,0x4fc7}, // <CJK>
  {0xCDEB,0x4fd6}, // <CJK>
  {0xCDEC,0x4fba}, // <CJK>
  {0xCDED,0x4fc0}, // <CJK>
  {0xCDEE,0x4fb9}, // <CJK>
  {0xCDEF,0x4fec}, // <CJK>
  {0xCDF0,0x5244}, // <CJK>
  {0xCDF1,0x5249}, // <CJK>
  {0xCDF2,0x52c0}, // <CJK>
  {0xCDF3,0x52c2}, // <CJK>
  {0xCDF4,0x533d}, // <CJK>
  {0xCDF5,0x537c}, // <CJK>
  {0xCDF6,0x5397}, // <CJK>
  {0xCDF7,0x5396}, // <CJK>
  {0xCDF8,0x5399}, // <CJK>
  {0xCDF9,0x5398}, // <CJK>
  {0xCDFA,0x54ba}, // <CJK>
  {0xCDFB,0x54a1}, // <CJK>
  {0xCDFC,0x54ad}, // <CJK>
  {0xCDFD,0x54a5}, // <CJK>
  {0xCDFE,0x54cf}, // <CJK>
  {0xCE40,0x54c3}, // <CJK>
  {0xCE41,0x830d}, // <CJK>
  {0xCE42,0x54b7}, // <CJK>
  {0xCE43,0x54ae}, // <CJK>
  {0xCE44,0x54d6}, // <CJK>
  {0xCE45,0x54b6}, // <CJK>
  {0xCE46,0x54c5}, // <CJK>
  {0xCE47,0x54c6}, // <CJK>
  {0xCE48,0x54a0}, // <CJK>
  {0xCE49,0x5470}, // <CJK>
  {0xCE4A,0x54bc}, // <CJK>
  {0xCE4B,0x54a2}, // <CJK>
  {0xCE4C,0x54be}, // <CJK>
  {0xCE4D,0x5472}, // <CJK>
  {0xCE4E,0x54de}, // <CJK>
  {0xCE4F,0x54b0}, // <CJK>
  {0xCE50,0x57b5}, // <CJK>
  {0xCE51,0x579e}, // <CJK>
  {0xCE52,0x579f}, // <CJK>
  {0xCE53,0x57a4}, // <CJK>
  {0xCE54,0x578c}, // <CJK>
  {0xCE55,0x5797}, // <CJK>
  {0xCE56,0x579d}, // <CJK>
  {0xCE57,0x579b}, // <CJK>
  {0xCE58,0x5794}, // <CJK>
  {0xCE59,0x5798}, // <CJK>
  {0xCE5A,0x578f}, // <CJK>
  {0xCE5B,0x5799}, // <CJK>
  {0xCE5C,0x57a5}, // <CJK>
  {0xCE5D,0x579a}, // <CJK>
  {0xCE5E,0x5795}, // <CJK>
  {0xCE5F,0x58f4}, // <CJK>
  {0xCE60,0x590d}, // <CJK>
  {0xCE61,0x5953}, // <CJK>
  {0xCE62,0x59e1}, // <CJK>
  {0xCE63,0x59de}, // <CJK>
  {0xCE64,0x59ee}, // <CJK>
  {0xCE65,0x5a00}, // <CJK>
  {0xCE66,0x59f1}, // <CJK>
  {0xCE67,0x59dd}, // <CJK>
  {0xCE68,0x59fa}, // <CJK>
  {0xCE69,0x59fd}, // <CJK>
  {0xCE6A,0x59fc}, // <CJK>
  {0xCE6B,0x59f6}, // <CJK>
  {0xCE6C,0x59e4}, // <CJK>
  {0xCE6D,0x59f2}, // <CJK>
  {0xCE6E,0x59f7}, // <CJK>
  {0xCE6F,0x59db}, // <CJK>
  {0xCE70,0x59e9}, // <CJK>
  {0xCE71,0x59f3}, // <CJK>
  {0xCE72,0x59f5}, // <CJK>
  {0xCE73,0x59e0}, // <CJK>
  {0xCE74,0x59fe}, // <CJK>
  {0xCE75,0x59f4}, // <CJK>
  {0xCE76,0x59ed}, // <CJK>
  {0xCE77,0x5ba8}, // <CJK>
  {0xCE78,0x5c4c}, // <CJK>
  {0xCE79,0x5cd0}, // <CJK>
  {0xCE7A,0x5cd8}, // <CJK>
  {0xCE7B,0x5ccc}, // <CJK>
  {0xCE7C,0x5cd7}, // <CJK>
  {0xCE7D,0x5ccb}, // <CJK>
  {0xCE7E,0x5cdb}, // <CJK>
  {0xCEA1,0x5cde}, // <CJK>
  {0xCEA2,0x5cda}, // <CJK>
  {0xCEA3,0x5cc9}, // <CJK>
  {0xCEA4,0x5cc7}, // <CJK>
  {0xCEA5,0x5cca}, // <CJK>
  {0xCEA6,0x5cd6}, // <CJK>
  {0xCEA7,0x5cd3}, // <CJK>
  {0xCEA8,0x5cd4}, // <CJK>
  {0xCEA9,0x5ccf}, // <CJK>
  {0xCEAA,0x5cc8}, // <CJK>
  {0xCEAB,0x5cc6}, // <CJK>
  {0xCEAC,0x5cce}, // <CJK>
  {0xCEAD,0x5cdf}, // <CJK>
  {0xCEAE,0x5cf8}, // <CJK>
  {0xCEAF,0x5df9}, // <CJK>
  {0xCEB0,0x5e21}, // <CJK>
  {0xCEB1,0x5e22}, // <CJK>
  {0xCEB2,0x5e23}, // <CJK>
  {0xCEB3,0x5e20}, // <CJK>
  {0xCEB4,0x5e24}, // <CJK>
  {0xCEB5,0x5eb0}, // <CJK>
  {0xCEB6,0x5ea4}, // <CJK>
  {0xCEB7,0x5ea2}, // <CJK>
  {0xCEB8,0x5e9b}, // <CJK>
  {0xCEB9,0x5ea3}, // <CJK>
  {0xCEBA,0x5ea5}, // <CJK>
  {0xCEBB,0x5f07}, // <CJK>
  {0xCEBC,0x5f2e}, // <CJK>
  {0xCEBD,0x5f56}, // <CJK>
  {0xCEBE,0x5f86}, // <CJK>
  {0xCEBF,0x6037}, // <CJK>
  {0xCEC0,0x6039}, // <CJK>
  {0xCEC1,0x6054}, // <CJK>
  {0xCEC2,0x6072}, // <CJK>
  {0xCEC3,0x605e}, // <CJK>
  {0xCEC4,0x6045}, // <CJK>
  {0xCEC5,0x6053}, // <CJK>
  {0xCEC6,0x6047}, // <CJK>
  {0xCEC7,0x6049}, // <CJK>
  {0xCEC8,0x605b}, // <CJK>
  {0xCEC9,0x604c}, // <CJK>
  {0xCECA,0x6040}, // <CJK>
  {0xCECB,0x6042}, // <CJK>
  {0xCECC,0x605f}, // <CJK>
  {0xCECD,0x6024}, // <CJK>
  {0xCECE,0x6044}, // <CJK>
  {0xCECF,0x6058}, // <CJK>
  {0xCED0,0x6066}, // <CJK>
  {0xCED1,0x606e}, // <CJK>
  {0xCED2,0x6242}, // <CJK>
  {0xCED3,0x6243}, // <CJK>
  {0xCED4,0x62cf}, // <CJK>
  {0xCED5,0x630d}, // <CJK>
  {0xCED6,0x630b}, // <CJK>
  {0xCED7,0x62f5}, // <CJK>
  {0xCED8,0x630e}, // <CJK>
  {0xCED9,0x6303}, // <CJK>
  {0xCEDA,0x62eb}, // <CJK>
  {0xCEDB,0x62f9}, // <CJK>
  {0xCEDC,0x630f}, // <CJK>
  {0xCEDD,0x630c}, // <CJK>
  {0xCEDE,0x62f8}, // <CJK>
  {0xCEDF,0x62f6}, // <CJK>
  {0xCEE0,0x6300}, // <CJK>
  {0xCEE1,0x6313}, // <CJK>
  {0xCEE2,0x6314}, // <CJK>
  {0xCEE3,0x62fa}, // <CJK>
  {0xCEE4,0x6315}, // <CJK>
  {0xCEE5,0x62fb}, // <CJK>
  {0xCEE6,0x62f0}, // <CJK>
  {0xCEE7,0x6541}, // <CJK>
  {0xCEE8,0x6543}, // <CJK>
  {0xCEE9,0x65aa}, // <CJK>
  {0xCEEA,0x65bf}, // <CJK>
  {0xCEEB,0x6636}, // <CJK>
  {0xCEEC,0x6621}, // <CJK>
  {0xCEED,0x6632}, // <CJK>
  {0xCEEE,0x6635}, // <CJK>
  {0xCEEF,0x661c}, // <CJK>
  {0xCEF0,0x6626}, // <CJK>
  {0xCEF1,0x6622}, // <CJK>
  {0xCEF2,0x6633}, // <CJK>
  {0xCEF3,0x662b}, // <CJK>
  {0xCEF4,0x663a}, // <CJK>
  {0xCEF5,0x661d}, // <CJK>
  {0xCEF6,0x6634}, // <CJK>
  {0xCEF7,0x6639}, // <CJK>
  {0xCEF8,0x662e}, // <CJK>
  {0xCEF9,0x670f}, // <CJK>
  {0xCEFA,0x6710}, // <CJK>
  {0xCEFB,0x67c1}, // <CJK>
  {0xCEFC,0x67f2}, // <CJK>
  {0xCEFD,0x67c8}, // <CJK>
  {0xCEFE,0x67ba}, // <CJK>
  {0xCF40,0x67dc}, // <CJK>
  {0xCF41,0x67bb}, // <CJK>
  {0xCF42,0x67f8}, // <CJK>
  {0xCF43,0x67d8}, // <CJK>
  {0xCF44,0x67c0}, // <CJK>
  {0xCF45,0x67b7}, // <CJK>
  {0xCF46,0x67c5}, // <CJK>
  {0xCF47,0x67eb}, // <CJK>
  {0xCF48,0x67e4}, // <CJK>
  {0xCF49,0x67df}, // <CJK>
  {0xCF4A,0x67b5}, // <CJK>
  {0xCF4B,0x67cd}, // <CJK>
  {0xCF4C,0x67b3}, // <CJK>
  {0xCF4D,0x67f7}, // <CJK>
  {0xCF4E,0x67f6}, // <CJK>
  {0xCF4F,0x67ee}, // <CJK>
  {0xCF50,0x67e3}, // <CJK>
  {0xCF51,0x67c2}, // <CJK>
  {0xCF52,0x67b9}, // <CJK>
  {0xCF53,0x67ce}, // <CJK>
  {0xCF54,0x67e7}, // <CJK>
  {0xCF55,0x67f0}, // <CJK>
  {0xCF56,0x67b2}, // <CJK>
  {0xCF57,0x67fc}, // <CJK>
  {0xCF58,0x67c6}, // <CJK>
  {0xCF59,0x67ed}, // <CJK>
  {0xCF5A,0x67cc}, // <CJK>
  {0xCF5B,0x67ae}, // <CJK>
  {0xCF5C,0x67e6}, // <CJK>
  {0xCF5D,0x67db}, // <CJK>
  {0xCF5E,0x67fa}, // <CJK>
  {0xCF5F,0x67c9}, // <CJK>
  {0xCF60,0x67ca}, // <CJK>
  {0xCF61,0x67c3}, // <CJK>
  {0xCF62,0x67ea}, // <CJK>
  {0xCF63,0x67cb}, // <CJK>
  {0xCF64,0x6b28}, // <CJK>
  {0xCF65,0x6b82}, // <CJK>
  {0xCF66,0x6b84}, // <CJK>
  {0xCF67,0x6bb6}, // <CJK>
  {0xCF68,0x6bd6}, // <CJK>
  {0xCF69,0x6bd8}, // <CJK>
  {0xCF6A,0x6be0}, // <CJK>
  {0xCF6B,0x6c20}, // <CJK>
  {0xCF6C,0x6c21}, // <CJK>
  {0xCF6D,0x6d28}, // <CJK>
  {0xCF6E,0x6d34}, // <CJK>
  {0xCF6F,0x6d2d}, // <CJK>
  {0xCF70,0x6d1f}, // <CJK>
  {0xCF71,0x6d3c}, // <CJK>
  {0xCF72,0x6d3f}, // <CJK>
  {0xCF73,0x6d12}, // <CJK>
  {0xCF74,0x6d0a}, // <CJK>
  {0xCF75,0x6cda}, // <CJK>
  {0xCF76,0x6d33}, // <CJK>
  {0xCF77,0x6d04}, // <CJK>
  {0xCF78,0x6d19}, // <CJK>
  {0xCF79,0x6d3a}, // <CJK>
  {0xCF7A,0x6d1a}, // <CJK>
  {0xCF7B,0x6d11}, // <CJK>
  {0xCF7C,0x6d00}, // <CJK>
  {0xCF7D,0x6d1d}, // <CJK>
  {0xCF7E,0x6d42}, // <CJK>
  {0xCFA1,0x6d01}, // <CJK>
  {0xCFA2,0x6d18}, // <CJK>
  {0xCFA3,0x6d37}, // <CJK>
  {0xCFA4,0x6d03}, // <CJK>
  {0xCFA5,0x6d0f}, // <CJK>
  {0xCFA6,0x6d40}, // <CJK>
  {0xCFA7,0x6d07}, // <CJK>
  {0xCFA8,0x6d20}, // <CJK>
  {0xCFA9,0x6d2c}, // <CJK>
  {0xCFAA,0x6d08}, // <CJK>
  {0xCFAB,0x6d22}, // <CJK>
  {0xCFAC,0x6d09}, // <CJK>
  {0xCFAD,0x6d10}, // <CJK>
  {0xCFAE,0x70b7}, // <CJK>
  {0xCFAF,0x709f}, // <CJK>
  {0xCFB0,0x70be}, // <CJK>
  {0xCFB1,0x70b1}, // <CJK>
  {0xCFB2,0x70b0}, // <CJK>
  {0xCFB3,0x70a1}, // <CJK>
  {0xCFB4,0x70b4}, // <CJK>
  {0xCFB5,0x70b5}, // <CJK>
  {0xCFB6,0x70a9}, // <CJK>
  {0xCFB7,0x7241}, // <CJK>
  {0xCFB8,0x7249}, // <CJK>
  {0xCFB9,0x724a}, // <CJK>
  {0xCFBA,0x726c}, // <CJK>
  {0xCFBB,0x7270}, // <CJK>
  {0xCFBC,0x7273}, // <CJK>
  {0xCFBD,0x726e}, // <CJK>
  {0xCFBE,0x72ca}, // <CJK>
  {0xCFBF,0x72e4}, // <CJK>
  {0xCFC0,0x72e8}, // <CJK>
  {0xCFC1,0x72eb}, // <CJK>
  {0xCFC2,0x72df}, // <CJK>
  {0xCFC3,0x72ea}, // <CJK>
  {0xCFC4,0x72e6}, // <CJK>
  {0xCFC5,0x72e3}, // <CJK>
  {0xCFC6,0x7385}, // <CJK>
  {0xCFC7,0x73cc}, // <CJK>
  {0xCFC8,0x73c2}, // <CJK>
  {0xCFC9,0x73c8}, // <CJK>
  {0xCFCA,0x73c5}, // <CJK>
  {0xCFCB,0x73b9}, // <CJK>
  {0xCFCC,0x73b6}, // <CJK>
  {0xCFCD,0x73b5}, // <CJK>
  {0xCFCE,0x73b4}, // <CJK>
  {0xCFCF,0x73eb}, // <CJK>
  {0xCFD0,0x73bf}, // <CJK>
  {0xCFD1,0x73c7}, // <CJK>
  {0xCFD2,0x73be}, // <CJK>
  {0xCFD3,0x73c3}, // <CJK>
  {0xCFD4,0x73c6}, // <CJK>
  {0xCFD5,0x73b8}, // <CJK>
  {0xCFD6,0x73cb}, // <CJK>
  {0xCFD7,0x74ec}, // <CJK>
  {0xCFD8,0x74ee}, // <CJK>
  {0xCFD9,0x752e}, // <CJK>
  {0xCFDA,0x7547}, // <CJK>
  {0xCFDB,0x7548}, // <CJK>
  {0xCFDC,0x75a7}, // <CJK>
  {0xCFDD,0x75aa}, // <CJK>
  {0xCFDE,0x7679}, // <CJK>
  {0xCFDF,0x76c4}, // <CJK>
  {0xCFE0,0x7708}, // <CJK>
  {0xCFE1,0x7703}, // <CJK>
  {0xCFE2,0x7704}, // <CJK>
  {0xCFE3,0x7705}, // <CJK>
  {0xCFE4,0x770a}, // <CJK>
  {0xCFE5,0x76f7}, // <CJK>
  {0xCFE6,0x76fb}, // <CJK>
  {0xCFE7,0x76fa}, // <CJK>
  {0xCFE8,0x77e7}, // <CJK>
  {0xCFE9,0x77e8}, // <CJK>
  {0xCFEA,0x7806}, // <CJK>
  {0xCFEB,0x7811}, // <CJK>
  {0xCFEC,0x7812}, // <CJK>
  {0xCFED,0x7805}, // <CJK>
  {0xCFEE,0x7810}, // <CJK>
  {0xCFEF,0x780f}, // <CJK>
  {0xCFF0,0x780e}, // <CJK>
  {0xCFF1,0x7809}, // <CJK>
  {0xCFF2,0x7803}, // <CJK>
  {0xCFF3,0x7813}, // <CJK>
  {0xCFF4,0x794a}, // <CJK>
  {0xCFF5,0x794c}, // <CJK>
  {0xCFF6,0x794b}, // <CJK>
  {0xCFF7,0x7945}, // <CJK>
  {0xCFF8,0x7944}, // <CJK>
  {0xCFF9,0x79d5}, // <CJK>
  {0xCFFA,0x79cd}, // <CJK>
  {0xCFFB,0x79cf}, // <CJK>
  {0xCFFC,0x79d6}, // <CJK>
  {0xCFFD,0x79ce}, // <CJK>
  {0xCFFE,0x7a80}, // <CJK>
  {0xD040,0x7a7e}, // <CJK>
  {0xD041,0x7ad1}, // <CJK>
  {0xD042,0x7b00}, // <CJK>
  {0xD043,0x7b01}, // <CJK>
  {0xD044,0x7c7a}, // <CJK>
  {0xD045,0x7c78}, // <CJK>
  {0xD046,0x7c79}, // <CJK>
  {0xD047,0x7c7f}, // <CJK>
  {0xD048,0x7c80}, // <CJK>
  {0xD049,0x7c81}, // <CJK>
  {0xD04A,0x7d03}, // <CJK>
  {0xD04B,0x7d08}, // <CJK>
  {0xD04C,0x7d01}, // <CJK>
  {0xD04D,0x7f58}, // <CJK>
  {0xD04E,0x7f91}, // <CJK>
  {0xD04F,0x7f8d}, // <CJK>
  {0xD050,0x7fbe}, // <CJK>
  {0xD051,0x8007}, // <CJK>
  {0xD052,0x800e}, // <CJK>
  {0xD053,0x800f}, // <CJK>
  {0xD054,0x8014}, // <CJK>
  {0xD055,0x8037}, // <CJK>
  {0xD056,0x80d8}, // <CJK>
  {0xD057,0x80c7}, // <CJK>
  {0xD058,0x80e0}, // <CJK>
  {0xD059,0x80d1}, // <CJK>
  {0xD05A,0x80c8}, // <CJK>
  {0xD05B,0x80c2}, // <CJK>
  {0xD05C,0x80d0}, // <CJK>
  {0xD05D,0x80c5}, // <CJK>
  {0xD05E,0x80e3}, // <CJK>
  {0xD05F,0x80d9}, // <CJK>
  {0xD060,0x80dc}, // <CJK>
  {0xD061,0x80ca}, // <CJK>
  {0xD062,0x80d5}, // <CJK>
  {0xD063,0x80c9}, // <CJK>
  {0xD064,0x80cf}, // <CJK>
  {0xD065,0x80d7}, // <CJK>
  {0xD066,0x80e6}, // <CJK>
  {0xD067,0x80cd}, // <CJK>
  {0xD068,0x81ff}, // <CJK>
  {0xD069,0x8221}, // <CJK>
  {0xD06A,0x8294}, // <CJK>
  {0xD06B,0x82d9}, // <CJK>
  {0xD06C,0x82fe}, // <CJK>
  {0xD06D,0x82f9}, // <CJK>
  {0xD06E,0x8307}, // <CJK>
  {0xD06F,0x82e8}, // <CJK>
  {0xD070,0x8300}, // <CJK>
  {0xD071,0x82d5}, // <CJK>
  {0xD072,0x833a}, // <CJK>
  {0xD073,0x82eb}, // <CJK>
  {0xD074,0x82d6}, // <CJK>
  {0xD075,0x82f4}, // <CJK>
  {0xD076,0x82ec}, // <CJK>
  {0xD077,0x82e1}, // <CJK>
  {0xD078,0x82f2}, // <CJK>
  {0xD079,0x82f5}, // <CJK>
  {0xD07A,0x830c}, // <CJK>
  {0xD07B,0x82fb}, // <CJK>
  {0xD07C,0x82f6}, // <CJK>
  {0xD07D,0x82f0}, // <CJK>
  {0xD07E,0x82ea}, // <CJK>
  {0xD0A1,0x82e4}, // <CJK>
  {0xD0A2,0x82e0}, // <CJK>
  {0xD0A3,0x82fa}, // <CJK>
  {0xD0A4,0x82f3}, // <CJK>
  {0xD0A5,0x82ed}, // <CJK>
  {0xD0A6,0x8677}, // <CJK>
  {0xD0A7,0x8674}, // <CJK>
  {0xD0A8,0x867c}, // <CJK>
  {0xD0A9,0x8673}, // <CJK>
  {0xD0AA,0x8841}, // <CJK>
  {0xD0AB,0x884e}, // <CJK>
  {0xD0AC,0x8867}, // <CJK>
  {0xD0AD,0x886a}, // <CJK>
  {0xD0AE,0x8869}, // <CJK>
  {0xD0AF,0x89d3}, // <CJK>
  {0xD0B0,0x8a04}, // <CJK>
  {0xD0B1,0x8a07}, // <CJK>
  {0xD0B2,0x8d72}, // <CJK>
  {0xD0B3,0x8fe3}, // <CJK>
  {0xD0B4,0x8fe1}, // <CJK>
  {0xD0B5,0x8fee}, // <CJK>
  {0xD0B6,0x8fe0}, // <CJK>
  {0xD0B7,0x90f1}, // <CJK>
  {0xD0B8,0x90bd}, // <CJK>
  {0xD0B9,0x90bf}, // <CJK>
  {0xD0BA,0x90d5}, // <CJK>
  {0xD0BB,0x90c5}, // <CJK>
  {0xD0BC,0x90be}, // <CJK>
  {0xD0BD,0x90c7}, // <CJK>
  {0xD0BE,0x90cb}, // <CJK>
  {0xD0BF,0x90c8}, // <CJK>
  {0xD0C0,0x91d4}, // <CJK>
  {0xD0C1,0x91d3}, // <CJK>
  {0xD0C2,0x9654}, // <CJK>
  {0xD0C3,0x964f}, // <CJK>
  {0xD0C4,0x9651}, // <CJK>
  {0xD0C5,0x9653}, // <CJK>
  {0xD0C6,0x964a}, // <CJK>
  {0xD0C7,0x964e}, // <CJK>
  {0xD0C8,0x501e}, // <CJK>
  {0xD0C9,0x5005}, // <CJK>
  {0xD0CA,0x5007}, // <CJK>
  {0xD0CB,0x5013}, // <CJK>
  {0xD0CC,0x5022}, // <CJK>
  {0xD0CD,0x5030}, // <CJK>
  {0xD0CE,0x501b}, // <CJK>
  {0xD0CF,0x4ff5}, // <CJK>
  {0xD0D0,0x4ff4}, // <CJK>
  {0xD0D1,0x5033}, // <CJK>
  {0xD0D2,0x5037}, // <CJK>
  {0xD0D3,0x502c}, // <CJK>
  {0xD0D4,0x4ff6}, // <CJK>
  {0xD0D5,0x4ff7}, // <CJK>
  {0xD0D6,0x5017}, // <CJK>
  {0xD0D7,0x501c}, // <CJK>
  {0xD0D8,0x5020}, // <CJK>
  {0xD0D9,0x5027}, // <CJK>
  {0xD0DA,0x5035}, // <CJK>
  {0xD0DB,0x502f}, // <CJK>
  {0xD0DC,0x5031}, // <CJK>
  {0xD0DD,0x500e}, // <CJK>
  {0xD0DE,0x515a}, // <CJK>
  {0xD0DF,0x5194}, // <CJK>
  {0xD0E0,0x5193}, // <CJK>
  {0xD0E1,0x51ca}, // <CJK>
  {0xD0E2,0x51c4}, // <CJK>
  {0xD0E3,0x51c5}, // <CJK>
  {0xD0E4,0x51c8}, // <CJK>
  {0xD0E5,0x51ce}, // <CJK>
  {0xD0E6,0x5261}, // <CJK>
  {0xD0E7,0x525a}, // <CJK>
  {0xD0E8,0x5252}, // <CJK>
  {0xD0E9,0x525e}, // <CJK>
  {0xD0EA,0x525f}, // <CJK>
  {0xD0EB,0x5255}, // <CJK>
  {0xD0EC,0x5262}, // <CJK>
  {0xD0ED,0x52cd}, // <CJK>
  {0xD0EE,0x530e}, // <CJK>
  {0xD0EF,0x539e}, // <CJK>
  {0xD0F0,0x5526}, // <CJK>
  {0xD0F1,0x54e2}, // <CJK>
  {0xD0F2,0x5517}, // <CJK>
  {0xD0F3,0x5512}, // <CJK>
  {0xD0F4,0x54e7}, // <CJK>
  {0xD0F5,0x54f3}, // <CJK>
  {0xD0F6,0x54e4}, // <CJK>
  {0xD0F7,0x551a}, // <CJK>
  {0xD0F8,0x54ff}, // <CJK>
  {0xD0F9,0x5504}, // <CJK>
  {0xD0FA,0x5508}, // <CJK>
  {0xD0FB,0x54eb}, // <CJK>
  {0xD0FC,0x5511}, // <CJK>
  {0xD0FD,0x5505}, // <CJK>
  {0xD0FE,0x54f1}, // <CJK>
  {0xD140,0x550a}, // <CJK>
  {0xD141,0x54fb}, // <CJK>
  {0xD142,0x54f7}, // <CJK>
  {0xD143,0x54f8}, // <CJK>
  {0xD144,0x54e0}, // <CJK>
  {0xD145,0x550e}, // <CJK>
  {0xD146,0x5503}, // <CJK>
  {0xD147,0x550b}, // <CJK>
  {0xD148,0x5701}, // <CJK>
  {0xD149,0x5702}, // <CJK>
  {0xD14A,0x57cc}, // <CJK>
  {0xD14B,0x5832}, // <CJK>
  {0xD14C,0x57d5}, // <CJK>
  {0xD14D,0x57d2}, // <CJK>
  {0xD14E,0x57ba}, // <CJK>
  {0xD14F,0x57c6}, // <CJK>
  {0xD150,0x57bd}, // <CJK>
  {0xD151,0x57bc}, // <CJK>
  {0xD152,0x57b8}, // <CJK>
  {0xD153,0x57b6}, // <CJK>
  {0xD154,0x57bf}, // <CJK>
  {0xD155,0x57c7}, // <CJK>
  {0xD156,0x57d0}, // <CJK>
  {0xD157,0x57b9}, // <CJK>
  {0xD158,0x57c1}, // <CJK>
  {0xD159,0x590e}, // <CJK>
  {0xD15A,0x594a}, // <CJK>
  {0xD15B,0x5a19}, // <CJK>
  {0xD15C,0x5a16}, // <CJK>
  {0xD15D,0x5a2d}, // <CJK>
  {0xD15E,0x5a2e}, // <CJK>
  {0xD15F,0x5a15}, // <CJK>
  {0xD160,0x5a0f}, // <CJK>
  {0xD161,0x5a17}, // <CJK>
  {0xD162,0x5a0a}, // <CJK>
  {0xD163,0x5a1e}, // <CJK>
  {0xD164,0x5a33}, // <CJK>
  {0xD165,0x5b6c}, // <CJK>
  {0xD166,0x5ba7}, // <CJK>
  {0xD167,0x5bad}, // <CJK>
  {0xD168,0x5bac}, // <CJK>
  {0xD169,0x5c03}, // <CJK>
  {0xD16A,0x5c56}, // <CJK>
  {0xD16B,0x5c54}, // <CJK>
  {0xD16C,0x5cec}, // <CJK>
  {0xD16D,0x5cff}, // <CJK>
  {0xD16E,0x5cee}, // <CJK>
  {0xD16F,0x5cf1}, // <CJK>
  {0xD170,0x5cf7}, // <CJK>
  {0xD171,0x5d00}, // <CJK>
  {0xD172,0x5cf9}, // <CJK>
  {0xD173,0x5e29}, // <CJK>
  {0xD174,0x5e28}, // <CJK>
  {0xD175,0x5ea8}, // <CJK>
  {0xD176,0x5eae}, // <CJK>
  {0xD177,0x5eaa}, // <CJK>
  {0xD178,0x5eac}, // <CJK>
  {0xD179,0x5f33}, // <CJK>
  {0xD17A,0x5f30}, // <CJK>
  {0xD17B,0x5f67}, // <CJK>
  {0xD17C,0x605d}, // <CJK>
  {0xD17D,0x605a}, // <CJK>
  {0xD17E,0x6067}, // <CJK>
  {0xD1A1,0x6041}, // <CJK>
  {0xD1A2,0x60a2}, // <CJK>
  {0xD1A3,0x6088}, // <CJK>
  {0xD1A4,0x6080}, // <CJK>
  {0xD1A5,0x6092}, // <CJK>
  {0xD1A6,0x6081}, // <CJK>
  {0xD1A7,0x609d}, // <CJK>
  {0xD1A8,0x6083}, // <CJK>
  {0xD1A9,0x6095}, // <CJK>
  {0xD1AA,0x609b}, // <CJK>
  {0xD1AB,0x6097}, // <CJK>
  {0xD1AC,0x6087}, // <CJK>
  {0xD1AD,0x609c}, // <CJK>
  {0xD1AE,0x608e}, // <CJK>
  {0xD1AF,0x6219}, // <CJK>
  {0xD1B0,0x6246}, // <CJK>
  {0xD1B1,0x62f2}, // <CJK>
  {0xD1B2,0x6310}, // <CJK>
  {0xD1B3,0x6356}, // <CJK>
  {0xD1B4,0x632c}, // <CJK>
  {0xD1B5,0x6344}, // <CJK>
  {0xD1B6,0x6345}, // <CJK>
  {0xD1B7,0x6336}, // <CJK>
  {0xD1B8,0x6343}, // <CJK>
  {0xD1B9,0x63e4}, // <CJK>
  {0xD1BA,0x6339}, // <CJK>
  {0xD1BB,0x634b}, // <CJK>
  {0xD1BC,0x634a}, // <CJK>
  {0xD1BD,0x633c}, // <CJK>
  {0xD1BE,0x6329}, // <CJK>
  {0xD1BF,0x6341}, // <CJK>
  {0xD1C0,0x6334}, // <CJK>
  {0xD1C1,0x6358}, // <CJK>
  {0xD1C2,0x6354}, // <CJK>
  {0xD1C3,0x6359}, // <CJK>
  {0xD1C4,0x632d}, // <CJK>
  {0xD1C5,0x6347}, // <CJK>
  {0xD1C6,0x6333}, // <CJK>
  {0xD1C7,0x635a}, // <CJK>
  {0xD1C8,0x6351}, // <CJK>
  {0xD1C9,0x6338}, // <CJK>
  {0xD1CA,0x6357}, // <CJK>
  {0xD1CB,0x6340}, // <CJK>
  {0xD1CC,0x6348}, // <CJK>
  {0xD1CD,0x654a}, // <CJK>
  {0xD1CE,0x6546}, // <CJK>
  {0xD1CF,0x65c6}, // <CJK>
  {0xD1D0,0x65c3}, // <CJK>
  {0xD1D1,0x65c4}, // <CJK>
  {0xD1D2,0x65c2}, // <CJK>
  {0xD1D3,0x664a}, // <CJK>
  {0xD1D4,0x665f}, // <CJK>
  {0xD1D5,0x6647}, // <CJK>
  {0xD1D6,0x6651}, // <CJK>
  {0xD1D7,0x6712}, // <CJK>
  {0xD1D8,0x6713}, // <CJK>
  {0xD1D9,0x681f}, // <CJK>
  {0xD1DA,0x681a}, // <CJK>
  {0xD1DB,0x6849}, // <CJK>
  {0xD1DC,0x6832}, // <CJK>
  {0xD1DD,0x6833}, // <CJK>
  {0xD1DE,0x683b}, // <CJK>
  {0xD1DF,0x684b}, // <CJK>
  {0xD1E0,0x684f}, // <CJK>
  {0xD1E1,0x6816}, // <CJK>
  {0xD1E2,0x6831}, // <CJK>
  {0xD1E3,0x681c}, // <CJK>
  {0xD1E4,0x6835}, // <CJK>
  {0xD1E5,0x682b}, // <CJK>
  {0xD1E6,0x682d}, // <CJK>
  {0xD1E7,0x682f}, // <CJK>
  {0xD1E8,0x684e}, // <CJK>
  {0xD1E9,0x6844}, // <CJK>
  {0xD1EA,0x6834}, // <CJK>
  {0xD1EB,0x681d}, // <CJK>
  {0xD1EC,0x6812}, // <CJK>
  {0xD1ED,0x6814}, // <CJK>
  {0xD1EE,0x6826}, // <CJK>
  {0xD1EF,0x6828}, // <CJK>
  {0xD1F0,0x682e}, // <CJK>
  {0xD1F1,0x684d}, // <CJK>
  {0xD1F2,0x683a}, // <CJK>
  {0xD1F3,0x6825}, // <CJK>
  {0xD1F4,0x6820}, // <CJK>
  {0xD1F5,0x6b2c}, // <CJK>
  {0xD1F6,0x6b2f}, // <CJK>
  {0xD1F7,0x6b2d}, // <CJK>
  {0xD1F8,0x6b31}, // <CJK>
  {0xD1F9,0x6b34}, // <CJK>
  {0xD1FA,0x6b6d}, // <CJK>
  {0xD1FB,0x8082}, // <CJK>
  {0xD1FC,0x6b88}, // <CJK>
  {0xD1FD,0x6be6}, // <CJK>
  {0xD1FE,0x6be4}, // <CJK>
  {0xD240,0x6be8}, // <CJK>
  {0xD241,0x6be3}, // <CJK>
  {0xD242,0x6be2}, // <CJK>
  {0xD243,0x6be7}, // <CJK>
  {0xD244,0x6c25}, // <CJK>
  {0xD245,0x6d7a}, // <CJK>
  {0xD246,0x6d63}, // <CJK>
  {0xD247,0x6d64}, // <CJK>
  {0xD248,0x6d76}, // <CJK>
  {0xD249,0x6d0d}, // <CJK>
  {0xD24A,0x6d61}, // <CJK>
  {0xD24B,0x6d92}, // <CJK>
  {0xD24C,0x6d58}, // <CJK>
  {0xD24D,0x6d62}, // <CJK>
  {0xD24E,0x6d6d}, // <CJK>
  {0xD24F,0x6d6f}, // <CJK>
  {0xD250,0x6d91}, // <CJK>
  {0xD251,0x6d8d}, // <CJK>
  {0xD252,0x6def}, // <CJK>
  {0xD253,0x6d7f}, // <CJK>
  {0xD254,0x6d86}, // <CJK>
  {0xD255,0x6d5e}, // <CJK>
  {0xD256,0x6d67}, // <CJK>
  {0xD257,0x6d60}, // <CJK>
  {0xD258,0x6d97}, // <CJK>
  {0xD259,0x6d70}, // <CJK>
  {0xD25A,0x6d7c}, // <CJK>
  {0xD25B,0x6d5f}, // <CJK>
  {0xD25C,0x6d82}, // <CJK>
  {0xD25D,0x6d98}, // <CJK>
  {0xD25E,0x6d2f}, // <CJK>
  {0xD25F,0x6d68}, // <CJK>
  {0xD260,0x6d8b}, // <CJK>
  {0xD261,0x6d7e}, // <CJK>
  {0xD262,0x6d80}, // <CJK>
  {0xD263,0x6d84}, // <CJK>
  {0xD264,0x6d16}, // <CJK>
  {0xD265,0x6d83}, // <CJK>
  {0xD266,0x6d7b}, // <CJK>
  {0xD267,0x6d7d}, // <CJK>
  {0xD268,0x6d75}, // <CJK>
  {0xD269,0x6d90}, // <CJK>
  {0xD26A,0x70dc}, // <CJK>
  {0xD26B,0x70d3}, // <CJK>
  {0xD26C,0x70d1}, // <CJK>
  {0xD26D,0x70dd}, // <CJK>
  {0xD26E,0x70cb}, // <CJK>
  {0xD26F,0x7f39}, // <CJK>
  {0xD270,0x70e2}, // <CJK>
  {0xD271,0x70d7}, // <CJK>
  {0xD272,0x70d2}, // <CJK>
  {0xD273,0x70de}, // <CJK>
  {0xD274,0x70e0}, // <CJK>
  {0xD275,0x70d4}, // <CJK>
  {0xD276,0x70cd}, // <CJK>
  {0xD277,0x70c5}, // <CJK>
  {0xD278,0x70c6}, // <CJK>
  {0xD279,0x70c7}, // <CJK>
  {0xD27A,0x70da}, // <CJK>
  {0xD27B,0x70ce}, // <CJK>
  {0xD27C,0x70e1}, // <CJK>
  {0xD27D,0x7242}, // <CJK>
  {0xD27E,0x7278}, // <CJK>
  {0xD2A1,0x7277}, // <CJK>
  {0xD2A2,0x7276}, // <CJK>
  {0xD2A3,0x7300}, // <CJK>
  {0xD2A4,0x72fa}, // <CJK>
  {0xD2A5,0x72f4}, // <CJK>
  {0xD2A6,0x72fe}, // <CJK>
  {0xD2A7,0x72f6}, // <CJK>
  {0xD2A8,0x72f3}, // <CJK>
  {0xD2A9,0x72fb}, // <CJK>
  {0xD2AA,0x7301}, // <CJK>
  {0xD2AB,0x73d3}, // <CJK>
  {0xD2AC,0x73d9}, // <CJK>
  {0xD2AD,0x73e5}, // <CJK>
  {0xD2AE,0x73d6}, // <CJK>
  {0xD2AF,0x73bc}, // <CJK>
  {0xD2B0,0x73e7}, // <CJK>
  {0xD2B1,0x73e3}, // <CJK>
  {0xD2B2,0x73e9}, // <CJK>
  {0xD2B3,0x73dc}, // <CJK>
  {0xD2B4,0x73d2}, // <CJK>
  {0xD2B5,0x73db}, // <CJK>
  {0xD2B6,0x73d4}, // <CJK>
  {0xD2B7,0x73dd}, // <CJK>
  {0xD2B8,0x73da}, // <CJK>
  {0xD2B9,0x73d7}, // <CJK>
  {0xD2BA,0x73d8}, // <CJK>
  {0xD2BB,0x73e8}, // <CJK>
  {0xD2BC,0x74de}, // <CJK>
  {0xD2BD,0x74df}, // <CJK>
  {0xD2BE,0x74f4}, // <CJK>
  {0xD2BF,0x74f5}, // <CJK>
  {0xD2C0,0x7521}, // <CJK>
  {0xD2C1,0x755b}, // <CJK>
  {0xD2C2,0x755f}, // <CJK>
  {0xD2C3,0x75b0}, // <CJK>
  {0xD2C4,0x75c1}, // <CJK>
  {0xD2C5,0x75bb}, // <CJK>
  {0xD2C6,0x75c4}, // <CJK>
  {0xD2C7,0x75c0}, // <CJK>
  {0xD2C8,0x75bf}, // <CJK>
  {0xD2C9,0x75b6}, // <CJK>
  {0xD2CA,0x75ba}, // <CJK>
  {0xD2CB,0x768a}, // <CJK>
  {0xD2CC,0x76c9}, // <CJK>
  {0xD2CD,0x771d}, // <CJK>
  {0xD2CE,0x771b}, // <CJK>
  {0xD2CF,0x7710}, // <CJK>
  {0xD2D0,0x7713}, // <CJK>
  {0xD2D1,0x7712}, // <CJK>
  {0xD2D2,0x7723}, // <CJK>
  {0xD2D3,0x7711}, // <CJK>
  {0xD2D4,0x7715}, // <CJK>
  {0xD2D5,0x7719}, // <CJK>
  {0xD2D6,0x771a}, // <CJK>
  {0xD2D7,0x7722}, // <CJK>
  {0xD2D8,0x7727}, // <CJK>
  {0xD2D9,0x7823}, // <CJK>
  {0xD2DA,0x782c}, // <CJK>
  {0xD2DB,0x7822}, // <CJK>
  {0xD2DC,0x7835}, // <CJK>
  {0xD2DD,0x782f}, // <CJK>
  {0xD2DE,0x7828}, // <CJK>
  {0xD2DF,0x782e}, // <CJK>
  {0xD2E0,0x782b}, // <CJK>
  {0xD2E1,0x7821}, // <CJK>
  {0xD2E2,0x7829}, // <CJK>
  {0xD2E3,0x7833}, // <CJK>
  {0xD2E4,0x782a}, // <CJK>
  {0xD2E5,0x7831}, // <CJK>
  {0xD2E6,0x7954}, // <CJK>
  {0xD2E7,0x795b}, // <CJK>
  {0xD2E8,0x794f}, // <CJK>
  {0xD2E9,0x795c}, // <CJK>
  {0xD2EA,0x7953}, // <CJK>
  {0xD2EB,0x7952}, // <CJK>
  {0xD2EC,0x7951}, // <CJK>
  {0xD2ED,0x79eb}, // <CJK>
  {0xD2EE,0x79ec}, // <CJK>
  {0xD2EF,0x79e0}, // <CJK>
  {0xD2F0,0x79ee}, // <CJK>
  {0xD2F1,0x79ed}, // <CJK>
  {0xD2F2,0x79ea}, // <CJK>
  {0xD2F3,0x79dc}, // <CJK>
  {0xD2F4,0x79de}, // <CJK>
  {0xD2F5,0x79dd}, // <CJK>
  {0xD2F6,0x7a86}, // <CJK>
  {0xD2F7,0x7a89}, // <CJK>
  {0xD2F8,0x7a85}, // <CJK>
  {0xD2F9,0x7a8b}, // <CJK>
  {0xD2FA,0x7a8c}, // <CJK>
  {0xD2FB,0x7a8a}, // <CJK>
  {0xD2FC,0x7a87}, // <CJK>
  {0xD2FD,0x7ad8}, // <CJK>
  {0xD2FE,0x7b10}, // <CJK>
  {0xD340,0x7b04}, // <CJK>
  {0xD341,0x7b13}, // <CJK>
  {0xD342,0x7b05}, // <CJK>
  {0xD343,0x7b0f}, // <CJK>
  {0xD344,0x7b08}, // <CJK>
  {0xD345,0x7b0a}, // <CJK>
  {0xD346,0x7b0e}, // <CJK>
  {0xD347,0x7b09}, // <CJK>
  {0xD348,0x7b12}, // <CJK>
  {0xD349,0x7c84}, // <CJK>
  {0xD34A,0x7c91}, // <CJK>
  {0xD34B,0x7c8a}, // <CJK>
  {0xD34C,0x7c8c}, // <CJK>
  {0xD34D,0x7c88}, // <CJK>
  {0xD34E,0x7c8d}, // <CJK>
  {0xD34F,0x7c85}, // <CJK>
  {0xD350,0x7d1e}, // <CJK>
  {0xD351,0x7d1d}, // <CJK>
  {0xD352,0x7d11}, // <CJK>
  {0xD353,0x7d0e}, // <CJK>
  {0xD354,0x7d18}, // <CJK>
  {0xD355,0x7d16}, // <CJK>
  {0xD356,0x7d13}, // <CJK>
  {0xD357,0x7d1f}, // <CJK>
  {0xD358,0x7d12}, // <CJK>
  {0xD359,0x7d0f}, // <CJK>
  {0xD35A,0x7d0c}, // <CJK>
  {0xD35B,0x7f5c}, // <CJK>
  {0xD35C,0x7f61}, // <CJK>
  {0xD35D,0x7f5e}, // <CJK>
  {0xD35E,0x7f60}, // <CJK>
  {0xD35F,0x7f5d}, // <CJK>
  {0xD360,0x7f5b}, // <CJK>
  {0xD361,0x7f96}, // <CJK>
  {0xD362,0x7f92}, // <CJK>
  {0xD363,0x7fc3}, // <CJK>
  {0xD364,0x7fc2}, // <CJK>
  {0xD365,0x7fc0}, // <CJK>
  {0xD366,0x8016}, // <CJK>
  {0xD367,0x803e}, // <CJK>
  {0xD368,0x8039}, // <CJK>
  {0xD369,0x80fa}, // <CJK>
  {0xD36A,0x80f2}, // <CJK>
  {0xD36B,0x80f9}, // <CJK>
  {0xD36C,0x80f5}, // <CJK>
  {0xD36D,0x8101}, // <CJK>
  {0xD36E,0x80fb}, // <CJK>
  {0xD36F,0x8100}, // <CJK>
  {0xD370,0x8201}, // <CJK>
  {0xD371,0x822f}, // <CJK>
  {0xD372,0x8225}, // <CJK>
  {0xD373,0x8333}, // <CJK>
  {0xD374,0x832d}, // <CJK>
  {0xD375,0x8344}, // <CJK>
  {0xD376,0x8319}, // <CJK>
  {0xD377,0x8351}, // <CJK>
  {0xD378,0x8325}, // <CJK>
  {0xD379,0x8356}, // <CJK>
  {0xD37A,0x833f}, // <CJK>
  {0xD37B,0x8341}, // <CJK>
  {0xD37C,0x8326}, // <CJK>
  {0xD37D,0x831c}, // <CJK>
  {0xD37E,0x8322}, // <CJK>
  {0xD3A1,0x8342}, // <CJK>
  {0xD3A2,0x834e}, // <CJK>
  {0xD3A3,0x831b}, // <CJK>
  {0xD3A4,0x832a}, // <CJK>
  {0xD3A5,0x8308}, // <CJK>
  {0xD3A6,0x833c}, // <CJK>
  {0xD3A7,0x834d}, // <CJK>
  {0xD3A8,0x8316}, // <CJK>
  {0xD3A9,0x8324}, // <CJK>
  {0xD3AA,0x8320}, // <CJK>
  {0xD3AB,0x8337}, // <CJK>
  {0xD3AC,0x832f}, // <CJK>
  {0xD3AD,0x8329}, // <CJK>
  {0xD3AE,0x8347}, // <CJK>
  {0xD3AF,0x8345}, // <CJK>
  {0xD3B0,0x834c}, // <CJK>
  {0xD3B1,0x8353}, // <CJK>
  {0xD3B2,0x831e}, // <CJK>
  {0xD3B3,0x832c}, // <CJK>
  {0xD3B4,0x834b}, // <CJK>
  {0xD3B5,0x8327}, // <CJK>
  {0xD3B6,0x8348}, // <CJK>
  {0xD3B7,0x8653}, // <CJK>
  {0xD3B8,0x8652}, // <CJK>
  {0xD3B9,0x86a2}, // <CJK>
  {0xD3BA,0x86a8}, // <CJK>
  {0xD3BB,0x8696}, // <CJK>
  {0xD3BC,0x868d}, // <CJK>
  {0xD3BD,0x8691}, // <CJK>
  {0xD3BE,0x869e}, // <CJK>
  {0xD3BF,0x8687}, // <CJK>
  {0xD3C0,0x8697}, // <CJK>
  {0xD3C1,0x8686}, // <CJK>
  {0xD3C2,0x868b}, // <CJK>
  {0xD3C3,0x869a}, // <CJK>
  {0xD3C4,0x8685}, // <CJK>
  {0xD3C5,0x86a5}, // <CJK>
  {0xD3C6,0x8699}, // <CJK>
  {0xD3C7,0x86a1}, // <CJK>
  {0xD3C8,0x86a7}, // <CJK>
  {0xD3C9,0x8695}, // <CJK>
  {0xD3CA,0x8698}, // <CJK>
  {0xD3CB,0x868e}, // <CJK>
  {0xD3CC,0x869d}, // <CJK>
  {0xD3CD,0x8690}, // <CJK>
  {0xD3CE,0x8694}, // <CJK>
  {0xD3CF,0x8843}, // <CJK>
  {0xD3D0,0x8844}, // <CJK>
  {0xD3D1,0x886d}, // <CJK>
  {0xD3D2,0x8875}, // <CJK>
  {0xD3D3,0x8876}, // <CJK>
  {0xD3D4,0x8872}, // <CJK>
  {0xD3D5,0x8880}, // <CJK>
  {0xD3D6,0x8871}, // <CJK>
  {0xD3D7,0x887f}, // <CJK>
  {0xD3D8,0x886f}, // <CJK>
  {0xD3D9,0x8883}, // <CJK>
  {0xD3DA,0x887e}, // <CJK>
  {0xD3DB,0x8874}, // <CJK>
  {0xD3DC,0x887c}, // <CJK>
  {0xD3DD,0x8a12}, // <CJK>
  {0xD3DE,0x8c47}, // <CJK>
  {0xD3DF,0x8c57}, // <CJK>
  {0xD3E0,0x8c7b}, // <CJK>
  {0xD3E1,0x8ca4}, // <CJK>
  {0xD3E2,0x8ca3}, // <CJK>
  {0xD3E3,0x8d76}, // <CJK>
  {0xD3E4,0x8d78}, // <CJK>
  {0xD3E5,0x8db5}, // <CJK>
  {0xD3E6,0x8db7}, // <CJK>
  {0xD3E7,0x8db6}, // <CJK>
  {0xD3E8,0x8ed1}, // <CJK>
  {0xD3E9,0x8ed3}, // <CJK>
  {0xD3EA,0x8ffe}, // <CJK>
  {0xD3EB,0x8ff5}, // <CJK>
  {0xD3EC,0x9002}, // <CJK>
  {0xD3ED,0x8fff}, // <CJK>
  {0xD3EE,0x8ffb}, // <CJK>
  {0xD3EF,0x9004}, // <CJK>
  {0xD3F0,0x8ffc}, // <CJK>
  {0xD3F1,0x8ff6}, // <CJK>
  {0xD3F2,0x90d6}, // <CJK>
  {0xD3F3,0x90e0}, // <CJK>
  {0xD3F4,0x90d9}, // <CJK>
  {0xD3F5,0x90da}, // <CJK>
  {0xD3F6,0x90e3}, // <CJK>
  {0xD3F7,0x90df}, // <CJK>
  {0xD3F8,0x90e5}, // <CJK>
  {0xD3F9,0x90d8}, // <CJK>
  {0xD3FA,0x90db}, // <CJK>
  {0xD3FB,0x90d7}, // <CJK>
  {0xD3FC,0x90dc}, // <CJK>
  {0xD3FD,0x90e4}, // <CJK>
  {0xD3FE,0x9150}, // <CJK>
  {0xD440,0x914e}, // <CJK>
  {0xD441,0x914f}, // <CJK>
  {0xD442,0x91d5}, // <CJK>
  {0xD443,0x91e2}, // <CJK>
  {0xD444,0x91da}, // <CJK>
  {0xD445,0x965c}, // <CJK>
  {0xD446,0x965f}, // <CJK>
  {0xD447,0x96bc}, // <CJK>
  {0xD448,0x98e3}, // <CJK>
  {0xD449,0x9adf}, // <CJK>
  {0xD44A,0x9b2f}, // <CJK>
  {0xD44B,0x4e7f}, // <CJK>
  {0xD44C,0x5070}, // <CJK>
  {0xD44D,0x506a}, // <CJK>
  {0xD44E,0x5061}, // <CJK>
  {0xD44F,0x505e}, // <CJK>
  {0xD450,0x5060}, // <CJK>
  {0xD451,0x5053}, // <CJK>
  {0xD452,0x504b}, // <CJK>
  {0xD453,0x505d}, // <CJK>
  {0xD454,0x5072}, // <CJK>
  {0xD455,0x5048}, // <CJK>
  {0xD456,0x504d}, // <CJK>
  {0xD457,0x5041}, // <CJK>
  {0xD458,0x505b}, // <CJK>
  {0xD459,0x504a}, // <CJK>
  {0xD45A,0x5062}, // <CJK>
  {0xD45B,0x5015}, // <CJK>
  {0xD45C,0x5045}, // <CJK>
  {0xD45D,0x505f}, // <CJK>
  {0xD45E,0x5069}, // <CJK>
  {0xD45F,0x506b}, // <CJK>
  {0xD460,0x5063}, // <CJK>
  {0xD461,0x5064}, // <CJK>
  {0xD462,0x5046}, // <CJK>
  {0xD463,0x5040}, // <CJK>
  {0xD464,0x506e}, // <CJK>
  {0xD465,0x5073}, // <CJK>
  {0xD466,0x5057}, // <CJK>
  {0xD467,0x5051}, // <CJK>
  {0xD468,0x51d0}, // <CJK>
  {0xD469,0x526b}, // <CJK>
  {0xD46A,0x526d}, // <CJK>
  {0xD46B,0x526c}, // <CJK>
  {0xD46C,0x526e}, // <CJK>
  {0xD46D,0x52d6}, // <CJK>
  {0xD46E,0x52d3}, // <CJK>
  {0xD46F,0x532d}, // <CJK>
  {0xD470,0x539c}, // <CJK>
  {0xD471,0x5575}, // <CJK>
  {0xD472,0x5576}, // <CJK>
  {0xD473,0x553c}, // <CJK>
  {0xD474,0x554d}, // <CJK>
  {0xD475,0x5550}, // <CJK>
  {0xD476,0x5534}, // <CJK>
  {0xD477,0x552a}, // <CJK>
  {0xD478,0x5551}, // <CJK>
  {0xD479,0x5562}, // <CJK>
  {0xD47A,0x5536}, // <CJK>
  {0xD47B,0x5535}, // <CJK>
  {0xD47C,0x5530}, // <CJK>
  {0xD47D,0x5552}, // <CJK>
  {0xD47E,0x5545}, // <CJK>
  {0xD4A1,0x550c}, // <CJK>
  {0xD4A2,0x5532}, // <CJK>
  {0xD4A3,0x5565}, // <CJK>
  {0xD4A4,0x554e}, // <CJK>
  {0xD4A5,0x5539}, // <CJK>
  {0xD4A6,0x5548}, // <CJK>
  {0xD4A7,0x552d}, // <CJK>
  {0xD4A8,0x553b}, // <CJK>
  {0xD4A9,0x5540}, // <CJK>
  {0xD4AA,0x554b}, // <CJK>
  {0xD4AB,0x570a}, // <CJK>
  {0xD4AC,0x5707}, // <CJK>
  {0xD4AD,0x57fb}, // <CJK>
  {0xD4AE,0x5814}, // <CJK>
  {0xD4AF,0x57e2}, // <CJK>
  {0xD4B0,0x57f6}, // <CJK>
  {0xD4B1,0x57dc}, // <CJK>
  {0xD4B2,0x57f4}, // <CJK>
  {0xD4B3,0x5800}, // <CJK>
  {0xD4B4,0x57ed}, // <CJK>
  {0xD4B5,0x57fd}, // <CJK>
  {0xD4B6,0x5808}, // <CJK>
  {0xD4B7,0x57f8}, // <CJK>
  {0xD4B8,0x580b}, // <CJK>
  {0xD4B9,0x57f3}, // <CJK>
  {0xD4BA,0x57cf}, // <CJK>
  {0xD4BB,0x5807}, // <CJK>
  {0xD4BC,0x57ee}, // <CJK>
  {0xD4BD,0x57e3}, // <CJK>
  {0xD4BE,0x57f2}, // <CJK>
  {0xD4BF,0x57e5}, // <CJK>
  {0xD4C0,0x57ec}, // <CJK>
  {0xD4C1,0x57e1}, // <CJK>
  {0xD4C2,0x580e}, // <CJK>
  {0xD4C3,0x57fc}, // <CJK>
  {0xD4C4,0x5810}, // <CJK>
  {0xD4C5,0x57e7}, // <CJK>
  {0xD4C6,0x5801}, // <CJK>
  {0xD4C7,0x580c}, // <CJK>
  {0xD4C8,0x57f1}, // <CJK>
  {0xD4C9,0x57e9}, // <CJK>
  {0xD4CA,0x57f0}, // <CJK>
  {0xD4CB,0x580d}, // <CJK>
  {0xD4CC,0x5804}, // <CJK>
  {0xD4CD,0x595c}, // <CJK>
  {0xD4CE,0x5a60}, // <CJK>
  {0xD4CF,0x5a58}, // <CJK>
  {0xD4D0,0x5a55}, // <CJK>
  {0xD4D1,0x5a67}, // <CJK>
  {0xD4D2,0x5a5e}, // <CJK>
  {0xD4D3,0x5a38}, // <CJK>
  {0xD4D4,0x5a35}, // <CJK>
  {0xD4D5,0x5a6d}, // <CJK>
  {0xD4D6,0x5a50}, // <CJK>
  {0xD4D7,0x5a5f}, // <CJK>
  {0xD4D8,0x5a65}, // <CJK>
  {0xD4D9,0x5a6c}, // <CJK>
  {0xD4DA,0x5a53}, // <CJK>
  {0xD4DB,0x5a64}, // <CJK>
  {0xD4DC,0x5a57}, // <CJK>
  {0xD4DD,0x5a43}, // <CJK>
  {0xD4DE,0x5a5d}, // <CJK>
  {0xD4DF,0x5a52}, // <CJK>
  {0xD4E0,0x5a44}, // <CJK>
  {0xD4E1,0x5a5b}, // <CJK>
  {0xD4E2,0x5a48}, // <CJK>
  {0xD4E3,0x5a8e}, // <CJK>
  {0xD4E4,0x5a3e}, // <CJK>
  {0xD4E5,0x5a4d}, // <CJK>
  {0xD4E6,0x5a39}, // <CJK>
  {0xD4E7,0x5a4c}, // <CJK>
  {0xD4E8,0x5a70}, // <CJK>
  {0xD4E9,0x5a69}, // <CJK>
  {0xD4EA,0x5a47}, // <CJK>
  {0xD4EB,0x5a51}, // <CJK>
  {0xD4EC,0x5a56}, // <CJK>
  {0xD4ED,0x5a42}, // <CJK>
  {0xD4EE,0x5a5c}, // <CJK>
  {0xD4EF,0x5b72}, // <CJK>
  {0xD4F0,0x5b6e}, // <CJK>
  {0xD4F1,0x5bc1}, // <CJK>
  {0xD4F2,0x5bc0}, // <CJK>
  {0xD4F3,0x5c59}, // <CJK>
  {0xD4F4,0x5d1e}, // <CJK>
  {0xD4F5,0x5d0b}, // <CJK>
  {0xD4F6,0x5d1d}, // <CJK>
  {0xD4F7,0x5d1a}, // <CJK>
  {0xD4F8,0x5d20}, // <CJK>
  {0xD4F9,0x5d0c}, // <CJK>
  {0xD4FA,0x5d28}, // <CJK>
  {0xD4FB,0x5d0d}, // <CJK>
  {0xD4FC,0x5d26}, // <CJK>
  {0xD4FD,0x5d25}, // <CJK>
  {0xD4FE,0x5d0f}, // <CJK>
  {0xD540,0x5d30}, // <CJK>
  {0xD541,0x5d12}, // <CJK>
  {0xD542,0x5d23}, // <CJK>
  {0xD543,0x5d1f}, // <CJK>
  {0xD544,0x5d2e}, // <CJK>
  {0xD545,0x5e3e}, // <CJK>
  {0xD546,0x5e34}, // <CJK>
  {0xD547,0x5eb1}, // <CJK>
  {0xD548,0x5eb4}, // <CJK>
  {0xD549,0x5eb9}, // <CJK>
  {0xD54A,0x5eb2}, // <CJK>
  {0xD54B,0x5eb3}, // <CJK>
  {0xD54C,0x5f36}, // <CJK>
  {0xD54D,0x5f38}, // <CJK>
  {0xD54E,0x5f9b}, // <CJK>
  {0xD54F,0x5f96}, // <CJK>
  {0xD550,0x5f9f}, // <CJK>
  {0xD551,0x608a}, // <CJK>
  {0xD552,0x6090}, // <CJK>
  {0xD553,0x6086}, // <CJK>
  {0xD554,0x60be}, // <CJK>
  {0xD555,0x60b0}, // <CJK>
  {0xD556,0x60ba}, // <CJK>
  {0xD557,0x60d3}, // <CJK>
  {0xD558,0x60d4}, // <CJK>
  {0xD559,0x60cf}, // <CJK>
  {0xD55A,0x60e4}, // <CJK>
  {0xD55B,0x60d9}, // <CJK>
  {0xD55C,0x60dd}, // <CJK>
  {0xD55D,0x60c8}, // <CJK>
  {0xD55E,0x60b1}, // <CJK>
  {0xD55F,0x60db}, // <CJK>
  {0xD560,0x60b7}, // <CJK>
  {0xD561,0x60ca}, // <CJK>
  {0xD562,0x60bf}, // <CJK>
  {0xD563,0x60c3}, // <CJK>
  {0xD564,0x60cd}, // <CJK>
  {0xD565,0x60c0}, // <CJK>
  {0xD566,0x6332}, // <CJK>
  {0xD567,0x6365}, // <CJK>
  {0xD568,0x638a}, // <CJK>
  {0xD569,0x6382}, // <CJK>
  {0xD56A,0x637d}, // <CJK>
  {0xD56B,0x63bd}, // <CJK>
  {0xD56C,0x639e}, // <CJK>
  {0xD56D,0x63ad}, // <CJK>
  {0xD56E,0x639d}, // <CJK>
  {0xD56F,0x6397}, // <CJK>
  {0xD570,0x63ab}, // <CJK>
  {0xD571,0x638e}, // <CJK>
  {0xD572,0x636f}, // <CJK>
  {0xD573,0x6387}, // <CJK>
  {0xD574,0x6390}, // <CJK>
  {0xD575,0x636e}, // <CJK>
  {0xD576,0x63af}, // <CJK>
  {0xD577,0x6375}, // <CJK>
  {0xD578,0x639c}, // <CJK>
  {0xD579,0x636d}, // <CJK>
  {0xD57A,0x63ae}, // <CJK>
  {0xD57B,0x637c}, // <CJK>
  {0xD57C,0x63a4}, // <CJK>
  {0xD57D,0x633b}, // <CJK>
  {0xD57E,0x639f}, // <CJK>
  {0xD5A1,0x6378}, // <CJK>
  {0xD5A2,0x6385}, // <CJK>
  {0xD5A3,0x6381}, // <CJK>
  {0xD5A4,0x6391}, // <CJK>
  {0xD5A5,0x638d}, // <CJK>
  {0xD5A6,0x6370}, // <CJK>
  {0xD5A7,0x6553}, // <CJK>
  {0xD5A8,0x65cd}, // <CJK>
  {0xD5A9,0x6665}, // <CJK>
  {0xD5AA,0x6661}, // <CJK>
  {0xD5AB,0x665b}, // <CJK>
  {0xD5AC,0x6659}, // <CJK>
  {0xD5AD,0x665c}, // <CJK>
  {0xD5AE,0x6662}, // <CJK>
  {0xD5AF,0x6718}, // <CJK>
  {0xD5B0,0x6879}, // <CJK>
  {0xD5B1,0x6887}, // <CJK>
  {0xD5B2,0x6890}, // <CJK>
  {0xD5B3,0x689c}, // <CJK>
  {0xD5B4,0x686d}, // <CJK>
  {0xD5B5,0x686e}, // <CJK>
  {0xD5B6,0x68ae}, // <CJK>
  {0xD5B7,0x68ab}, // <CJK>
  {0xD5B8,0x6956}, // <CJK>
  {0xD5B9,0x686f}, // <CJK>
  {0xD5BA,0x68a3}, // <CJK>
  {0xD5BB,0x68ac}, // <CJK>
  {0xD5BC,0x68a9}, // <CJK>
  {0xD5BD,0x6875}, // <CJK>
  {0xD5BE,0x6874}, // <CJK>
  {0xD5BF,0x68b2}, // <CJK>
  {0xD5C0,0x688f}, // <CJK>
  {0xD5C1,0x6877}, // <CJK>
  {0xD5C2,0x6892}, // <CJK>
  {0xD5C3,0x687c}, // <CJK>
  {0xD5C4,0x686b}, // <CJK>
  {0xD5C5,0x6872}, // <CJK>
  {0xD5C6,0x68aa}, // <CJK>
  {0xD5C7,0x6880}, // <CJK>
  {0xD5C8,0x6871}, // <CJK>
  {0xD5C9,0x687e}, // <CJK>
  {0xD5CA,0x689b}, // <CJK>
  {0xD5CB,0x6896}, // <CJK>
  {0xD5CC,0x688b}, // <CJK>
  {0xD5CD,0x68a0}, // <CJK>
  {0xD5CE,0x6889}, // <CJK>
  {0xD5CF,0x68a4}, // <CJK>
  {0xD5D0,0x6878}, // <CJK>
  {0xD5D1,0x687b}, // <CJK>
  {0xD5D2,0x6891}, // <CJK>
  {0xD5D3,0x688c}, // <CJK>
  {0xD5D4,0x688a}, // <CJK>
  {0xD5D5,0x687d}, // <CJK>
  {0xD5D6,0x6b36}, // <CJK>
  {0xD5D7,0x6b33}, // <CJK>
  {0xD5D8,0x6b37}, // <CJK>
  {0xD5D9,0x6b38}, // <CJK>
  {0xD5DA,0x6b91}, // <CJK>
  {0xD5DB,0x6b8f}, // <CJK>
  {0xD5DC,0x6b8d}, // <CJK>
  {0xD5DD,0x6b8e}, // <CJK>
  {0xD5DE,0x6b8c}, // <CJK>
  {0xD5DF,0x6c2a}, // <CJK>
  {0xD5E0,0x6dc0}, // <CJK>
  {0xD5E1,0x6dab}, // <CJK>
  {0xD5E2,0x6db4}, // <CJK>
  {0xD5E3,0x6db3}, // <CJK>
  {0xD5E4,0x6e74}, // <CJK>
  {0xD5E5,0x6dac}, // <CJK>
  {0xD5E6,0x6de9}, // <CJK>
  {0xD5E7,0x6de2}, // <CJK>
  {0xD5E8,0x6db7}, // <CJK>
  {0xD5E9,0x6df6}, // <CJK>
  {0xD5EA,0x6dd4}, // <CJK>
  {0xD5EB,0x6e00}, // <CJK>
  {0xD5EC,0x6dc8}, // <CJK>
  {0xD5ED,0x6de0}, // <CJK>
  {0xD5EE,0x6ddf}, // <CJK>
  {0xD5EF,0x6dd6}, // <CJK>
  {0xD5F0,0x6dbe}, // <CJK>
  {0xD5F1,0x6de5}, // <CJK>
  {0xD5F2,0x6ddc}, // <CJK>
  {0xD5F3,0x6ddd}, // <CJK>
  {0xD5F4,0x6ddb}, // <CJK>
  {0xD5F5,0x6df4}, // <CJK>
  {0xD5F6,0x6dca}, // <CJK>
  {0xD5F7,0x6dbd}, // <CJK>
  {0xD5F8,0x6ded}, // <CJK>
  {0xD5F9,0x6df0}, // <CJK>
  {0xD5FA,0x6dba}, // <CJK>
  {0xD5FB,0x6dd5}, // <CJK>
  {0xD5FC,0x6dc2}, // <CJK>
  {0xD5FD,0x6dcf}, // <CJK>
  {0xD5FE,0x6dc9}, // <CJK>
  {0xD640,0x6dd0}, // <CJK>
  {0xD641,0x6df2}, // <CJK>
  {0xD642,0x6dd3}, // <CJK>
  {0xD643,0x6dfd}, // <CJK>
  {0xD644,0x6dd7}, // <CJK>
  {0xD645,0x6dcd}, // <CJK>
  {0xD646,0x6de3}, // <CJK>
  {0xD647,0x6dbb}, // <CJK>
  {0xD648,0x70fa}, // <CJK>
  {0xD649,0x710d}, // <CJK>
  {0xD64A,0x70f7}, // <CJK>
  {0xD64B,0x7117}, // <CJK>
  {0xD64C,0x70f4}, // <CJK>
  {0xD64D,0x710c}, // <CJK>
  {0xD64E,0x70f0}, // <CJK>
  {0xD64F,0x7104}, // <CJK>
  {0xD650,0x70f3}, // <CJK>
  {0xD651,0x7110}, // <CJK>
  {0xD652,0x70fc}, // <CJK>
  {0xD653,0x70ff}, // <CJK>
  {0xD654,0x7106}, // <CJK>
  {0xD655,0x7113}, // <CJK>
  {0xD656,0x7100}, // <CJK>
  {0xD657,0x70f8}, // <CJK>
  {0xD658,0x70f6}, // <CJK>
  {0xD659,0x710b}, // <CJK>
  {0xD65A,0x7102}, // <CJK>
  {0xD65B,0x710e}, // <CJK>
  {0xD65C,0x727e}, // <CJK>
  {0xD65D,0x727b}, // <CJK>
  {0xD65E,0x727c}, // <CJK>
  {0xD65F,0x727f}, // <CJK>
  {0xD660,0x731d}, // <CJK>
  {0xD661,0x7317}, // <CJK>
  {0xD662,0x7307}, // <CJK>
  {0xD663,0x7311}, // <CJK>
  {0xD664,0x7318}, // <CJK>
  {0xD665,0x730a}, // <CJK>
  {0xD666,0x7308}, // <CJK>
  {0xD667,0x72ff}, // <CJK>
  {0xD668,0x730f}, // <CJK>
  {0xD669,0x731e}, // <CJK>
  {0xD66A,0x7388}, // <CJK>
  {0xD66B,0x73f6}, // <CJK>
  {0xD66C,0x73f8}, // <CJK>
  {0xD66D,0x73f5}, // <CJK>
  {0xD66E,0x7404}, // <CJK>
  {0xD66F,0x7401}, // <CJK>
  {0xD670,0x73fd}, // <CJK>
  {0xD671,0x7407}, // <CJK>
  {0xD672,0x7400}, // <CJK>
  {0xD673,0x73fa}, // <CJK>
  {0xD674,0x73fc}, // <CJK>
  {0xD675,0x73ff}, // <CJK>
  {0xD676,0x740c}, // <CJK>
  {0xD677,0x740b}, // <CJK>
  {0xD678,0x73f4}, // <CJK>
  {0xD679,0x7408}, // <CJK>
  {0xD67A,0x7564}, // <CJK>
  {0xD67B,0x7563}, // <CJK>
  {0xD67C,0x75ce}, // <CJK>
  {0xD67D,0x75d2}, // <CJK>
  {0xD67E,0x75cf}, // <CJK>
  {0xD6A1,0x75cb}, // <CJK>
  {0xD6A2,0x75cc}, // <CJK>
  {0xD6A3,0x75d1}, // <CJK>
  {0xD6A4,0x75d0}, // <CJK>
  {0xD6A5,0x768f}, // <CJK>
  {0xD6A6,0x7689}, // <CJK>
  {0xD6A7,0x76d3}, // <CJK>
  {0xD6A8,0x7739}, // <CJK>
  {0xD6A9,0x772f}, // <CJK>
  {0xD6AA,0x772d}, // <CJK>
  {0xD6AB,0x7731}, // <CJK>
  {0xD6AC,0x7732}, // <CJK>
  {0xD6AD,0x7734}, // <CJK>
  {0xD6AE,0x7733}, // <CJK>
  {0xD6AF,0x773d}, // <CJK>
  {0xD6B0,0x7725}, // <CJK>
  {0xD6B1,0x773b}, // <CJK>
  {0xD6B2,0x7735}, // <CJK>
  {0xD6B3,0x7848}, // <CJK>
  {0xD6B4,0x7852}, // <CJK>
  {0xD6B5,0x7849}, // <CJK>
  {0xD6B6,0x784d}, // <CJK>
  {0xD6B7,0x784a}, // <CJK>
  {0xD6B8,0x784c}, // <CJK>
  {0xD6B9,0x7826}, // <CJK>
  {0xD6BA,0x7845}, // <CJK>
  {0xD6BB,0x7850}, // <CJK>
  {0xD6BC,0x7964}, // <CJK>
  {0xD6BD,0x7967}, // <CJK>
  {0xD6BE,0x7969}, // <CJK>
  {0xD6BF,0x796a}, // <CJK>
  {0xD6C0,0x7963}, // <CJK>
  {0xD6C1,0x796b}, // <CJK>
  {0xD6C2,0x7961}, // <CJK>
  {0xD6C3,0x79bb}, // <CJK>
  {0xD6C4,0x79fa}, // <CJK>
  {0xD6C5,0x79f8}, // <CJK>
  {0xD6C6,0x79f6}, // <CJK>
  {0xD6C7,0x79f7}, // <CJK>
  {0xD6C8,0x7a8f}, // <CJK>
  {0xD6C9,0x7a94}, // <CJK>
  {0xD6CA,0x7a90}, // <CJK>
  {0xD6CB,0x7b35}, // <CJK>
  {0xD6CC,0x7b47}, // <CJK>
  {0xD6CD,0x7b34}, // <CJK>
  {0xD6CE,0x7b25}, // <CJK>
  {0xD6CF,0x7b30}, // <CJK>
  {0xD6D0,0x7b22}, // <CJK>
  {0xD6D1,0x7b24}, // <CJK>
  {0xD6D2,0x7b33}, // <CJK>
  {0xD6D3,0x7b18}, // <CJK>
  {0xD6D4,0x7b2a}, // <CJK>
  {0xD6D5,0x7b1d}, // <CJK>
  {0xD6D6,0x7b31}, // <CJK>
  {0xD6D7,0x7b2b}, // <CJK>
  {0xD6D8,0x7b2d}, // <CJK>
  {0xD6D9,0x7b2f}, // <CJK>
  {0xD6DA,0x7b32}, // <CJK>
  {0xD6DB,0x7b38}, // <CJK>
  {0xD6DC,0x7b1a}, // <CJK>
  {0xD6DD,0x7b23}, // <CJK>
  {0xD6DE,0x7c94}, // <CJK>
  {0xD6DF,0x7c98}, // <CJK>
  {0xD6E0,0x7c96}, // <CJK>
  {0xD6E1,0x7ca3}, // <CJK>
  {0xD6E2,0x7d35}, // <CJK>
  {0xD6E3,0x7d3d}, // <CJK>
  {0xD6E4,0x7d38}, // <CJK>
  {0xD6E5,0x7d36}, // <CJK>
  {0xD6E6,0x7d3a}, // <CJK>
  {0xD6E7,0x7d45}, // <CJK>
  {0xD6E8,0x7d2c}, // <CJK>
  {0xD6E9,0x7d29}, // <CJK>
  {0xD6EA,0x7d41}, // <CJK>
  {0xD6EB,0x7d47}, // <CJK>
  {0xD6EC,0x7d3e}, // <CJK>
  {0xD6ED,0x7d3f}, // <CJK>
  {0xD6EE,0x7d4a}, // <CJK>
  {0xD6EF,0x7d3b}, // <CJK>
  {0xD6F0,0x7d28}, // <CJK>
  {0xD6F1,0x7f63}, // <CJK>
  {0xD6F2,0x7f95}, // <CJK>
  {0xD6F3,0x7f9c}, // <CJK>
  {0xD6F4,0x7f9d}, // <CJK>
  {0xD6F5,0x7f9b}, // <CJK>
  {0xD6F6,0x7fca}, // <CJK>
  {0xD6F7,0x7fcb}, // <CJK>
  {0xD6F8,0x7fcd}, // <CJK>
  {0xD6F9,0x7fd0}, // <CJK>
  {0xD6FA,0x7fd1}, // <CJK>
  {0xD6FB,0x7fc7}, // <CJK>
  {0xD6FC,0x7fcf}, // <CJK>
  {0xD6FD,0x7fc9}, // <CJK>
  {0xD6FE,0x801f}, // <CJK>
  {0xD740,0x801e}, // <CJK>
  {0xD741,0x801b}, // <CJK>
  {0xD742,0x8047}, // <CJK>
  {0xD743,0x8043}, // <CJK>
  {0xD744,0x8048}, // <CJK>
  {0xD745,0x8118}, // <CJK>
  {0xD746,0x8125}, // <CJK>
  {0xD747,0x8119}, // <CJK>
  {0xD748,0x811b}, // <CJK>
  {0xD749,0x812d}, // <CJK>
  {0xD74A,0x811f}, // <CJK>
  {0xD74B,0x812c}, // <CJK>
  {0xD74C,0x811e}, // <CJK>
  {0xD74D,0x8121}, // <CJK>
  {0xD74E,0x8115}, // <CJK>
  {0xD74F,0x8127}, // <CJK>
  {0xD750,0x811d}, // <CJK>
  {0xD751,0x8122}, // <CJK>
  {0xD752,0x8211}, // <CJK>
  {0xD753,0x8238}, // <CJK>
  {0xD754,0x8233}, // <CJK>
  {0xD755,0x823a}, // <CJK>
  {0xD756,0x8234}, // <CJK>
  {0xD757,0x8232}, // <CJK>
  {0xD758,0x8274}, // <CJK>
  {0xD759,0x8390}, // <CJK>
  {0xD75A,0x83a3}, // <CJK>
  {0xD75B,0x83a8}, // <CJK>
  {0xD75C,0x838d}, // <CJK>
  {0xD75D,0x837a}, // <CJK>
  {0xD75E,0x8373}, // <CJK>
  {0xD75F,0x83a4}, // <CJK>
  {0xD760,0x8374}, // <CJK>
  {0xD761,0x838f}, // <CJK>
  {0xD762,0x8381}, // <CJK>
  {0xD763,0x8395}, // <CJK>
  {0xD764,0x8399}, // <CJK>
  {0xD765,0x8375}, // <CJK>
  {0xD766,0x8394}, // <CJK>
  {0xD767,0x83a9}, // <CJK>
  {0xD768,0x837d}, // <CJK>
  {0xD769,0x8383}, // <CJK>
  {0xD76A,0x838c}, // <CJK>
  {0xD76B,0x839d}, // <CJK>
  {0xD76C,0x839b}, // <CJK>
  {0xD76D,0x83aa}, // <CJK>
  {0xD76E,0x838b}, // <CJK>
  {0xD76F,0x837e}, // <CJK>
  {0xD770,0x83a5}, // <CJK>
  {0xD771,0x83af}, // <CJK>
  {0xD772,0x8388}, // <CJK>
  {0xD773,0x8397}, // <CJK>
  {0xD774,0x83b0}, // <CJK>
  {0xD775,0x837f}, // <CJK>
  {0xD776,0x83a6}, // <CJK>
  {0xD777,0x8387}, // <CJK>
  {0xD778,0x83ae}, // <CJK>
  {0xD779,0x8376}, // <CJK>
  {0xD77A,0x839a}, // <CJK>
  {0xD77B,0x8659}, // <CJK>
  {0xD77C,0x8656}, // <CJK>
  {0xD77D,0x86bf}, // <CJK>
  {0xD77E,0x86b7}, // <CJK>
  {0xD7A1,0x86c2}, // <CJK>
  {0xD7A2,0x86c1}, // <CJK>
  {0xD7A3,0x86c5}, // <CJK>
  {0xD7A4,0x86ba}, // <CJK>
  {0xD7A5,0x86b0}, // <CJK>
  {0xD7A6,0x86c8}, // <CJK>
  {0xD7A7,0x86b9}, // <CJK>
  {0xD7A8,0x86b3}, // <CJK>
  {0xD7A9,0x86b8}, // <CJK>
  {0xD7AA,0x86cc}, // <CJK>
  {0xD7AB,0x86b4}, // <CJK>
  {0xD7AC,0x86bb}, // <CJK>
  {0xD7AD,0x86bc}, // <CJK>
  {0xD7AE,0x86c3}, // <CJK>
  {0xD7AF,0x86bd}, // <CJK>
  {0xD7B0,0x86be}, // <CJK>
  {0xD7B1,0x8852}, // <CJK>
  {0xD7B2,0x8889}, // <CJK>
  {0xD7B3,0x8895}, // <CJK>
  {0xD7B4,0x88a8}, // <CJK>
  {0xD7B5,0x88a2}, // <CJK>
  {0xD7B6,0x88aa}, // <CJK>
  {0xD7B7,0x889a}, // <CJK>
  {0xD7B8,0x8891}, // <CJK>
  {0xD7B9,0x88a1}, // <CJK>
  {0xD7BA,0x889f}, // <CJK>
  {0xD7BB,0x8898}, // <CJK>
  {0xD7BC,0x88a7}, // <CJK>
  {0xD7BD,0x8899}, // <CJK>
  {0xD7BE,0x889b}, // <CJK>
  {0xD7BF,0x8897}, // <CJK>
  {0xD7C0,0x88a4}, // <CJK>
  {0xD7C1,0x88ac}, // <CJK>
  {0xD7C2,0x888c}, // <CJK>
  {0xD7C3,0x8893}, // <CJK>
  {0xD7C4,0x888e}, // <CJK>
  {0xD7C5,0x8982}, // <CJK>
  {0xD7C6,0x89d6}, // <CJK>
  {0xD7C7,0x89d9}, // <CJK>
  {0xD7C8,0x89d5}, // <CJK>
  {0xD7C9,0x8a30}, // <CJK>
  {0xD7CA,0x8a27}, // <CJK>
  {0xD7CB,0x8a2c}, // <CJK>
  {0xD7CC,0x8a1e}, // <CJK>
  {0xD7CD,0x8c39}, // <CJK>
  {0xD7CE,0x8c3b}, // <CJK>
  {0xD7CF,0x8c5c}, // <CJK>
  {0xD7D0,0x8c5d}, // <CJK>
  {0xD7D1,0x8c7d}, // <CJK>
  {0xD7D2,0x8ca5}, // <CJK>
  {0xD7D3,0x8d7d}, // <CJK>
  {0xD7D4,0x8d7b}, // <CJK>
  {0xD7D5,0x8d79}, // <CJK>
  {0xD7D6,0x8dbc}, // <CJK>
  {0xD7D7,0x8dc2}, // <CJK>
  {0xD7D8,0x8db9}, // <CJK>
  {0xD7D9,0x8dbf}, // <CJK>
  {0xD7DA,0x8dc1}, // <CJK>
  {0xD7DB,0x8ed8}, // <CJK>
  {0xD7DC,0x8ede}, // <CJK>
  {0xD7DD,0x8edd}, // <CJK>
  {0xD7DE,0x8edc}, // <CJK>
  {0xD7DF,0x8ed7}, // <CJK>
  {0xD7E0,0x8ee0}, // <CJK>
  {0xD7E1,0x8ee1}, // <CJK>
  {0xD7E2,0x9024}, // <CJK>
  {0xD7E3,0x900b}, // <CJK>
  {0xD7E4,0x9011}, // <CJK>
  {0xD7E5,0x901c}, // <CJK>
  {0xD7E6,0x900c}, // <CJK>
  {0xD7E7,0x9021}, // <CJK>
  {0xD7E8,0x90ef}, // <CJK>
  {0xD7E9,0x90ea}, // <CJK>
  {0xD7EA,0x90f0}, // <CJK>
  {0xD7EB,0x90f4}, // <CJK>
  {0xD7EC,0x90f2}, // <CJK>
  {0xD7ED,0x90f3}, // <CJK>
  {0xD7EE,0x90d4}, // <CJK>
  {0xD7EF,0x90eb}, // <CJK>
  {0xD7F0,0x90ec}, // <CJK>
  {0xD7F1,0x90e9}, // <CJK>
  {0xD7F2,0x9156}, // <CJK>
  {0xD7F3,0x9158}, // <CJK>
  {0xD7F4,0x915a}, // <CJK>
  {0xD7F5,0x9153}, // <CJK>
  {0xD7F6,0x9155}, // <CJK>
  {0xD7F7,0x91ec}, // <CJK>
  {0xD7F8,0x91f4}, // <CJK>
  {0xD7F9,0x91f1}, // <CJK>
  {0xD7FA,0x91f3}, // <CJK>
  {0xD7FB,0x91f8}, // <CJK>
  {0xD7FC,0x91e4}, // <CJK>
  {0xD7FD,0x91f9}, // <CJK>
  {0xD7FE,0x91ea}, // <CJK>
  {0xD840,0x91eb}, // <CJK>
  {0xD841,0x91f7}, // <CJK>
  {0xD842,0x91e8}, // <CJK>
  {0xD843,0x91ee}, // <CJK>
  {0xD844,0x957a}, // <CJK>
  {0xD845,0x9586}, // <CJK>
  {0xD846,0x9588}, // <CJK>
  {0xD847,0x967c}, // <CJK>
  {0xD848,0x966d}, // <CJK>
  {0xD849,0x966b}, // <CJK>
  {0xD84A,0x9671}, // <CJK>
  {0xD84B,0x966f}, // <CJK>
  {0xD84C,0x96bf}, // <CJK>
  {0xD84D,0x976a}, // <CJK>
  {0xD84E,0x9804}, // <CJK>
  {0xD84F,0x98e5}, // <CJK>
  {0xD850,0x9997}, // <CJK>
  {0xD851,0x509b}, // <CJK>
  {0xD852,0x5095}, // <CJK>
  {0xD853,0x5094}, // <CJK>
  {0xD854,0x509e}, // <CJK>
  {0xD855,0x508b}, // <CJK>
  {0xD856,0x50a3}, // <CJK>
  {0xD857,0x5083}, // <CJK>
  {0xD858,0x508c}, // <CJK>
  {0xD859,0x508e}, // <CJK>
  {0xD85A,0x509d}, // <CJK>
  {0xD85B,0x5068}, // <CJK>
  {0xD85C,0x509c}, // <CJK>
  {0xD85D,0x5092}, // <CJK>
  {0xD85E,0x5082}, // <CJK>
  {0xD85F,0x5087}, // <CJK>
  {0xD860,0x515f}, // <CJK>
  {0xD861,0x51d4}, // <CJK>
  {0xD862,0x5312}, // <CJK>
  {0xD863,0x5311}, // <CJK>
  {0xD864,0x53a4}, // <CJK>
  {0xD865,0x53a7}, // <CJK>
  {0xD866,0x5591}, // <CJK>
  {0xD867,0x55a8}, // <CJK>
  {0xD868,0x55a5}, // <CJK>
  {0xD869,0x55ad}, // <CJK>
  {0xD86A,0x5577}, // <CJK>
  {0xD86B,0x5645}, // <CJK>
  {0xD86C,0x55a2}, // <CJK>
  {0xD86D,0x5593}, // <CJK>
  {0xD86E,0x5588}, // <CJK>
  {0xD86F,0x558f}, // <CJK>
  {0xD870,0x55b5}, // <CJK>
  {0xD871,0x5581}, // <CJK>
  {0xD872,0x55a3}, // <CJK>
  {0xD873,0x5592}, // <CJK>
  {0xD874,0x55a4}, // <CJK>
  {0xD875,0x557d}, // <CJK>
  {0xD876,0x558c}, // <CJK>
  {0xD877,0x55a6}, // <CJK>
  {0xD878,0x557f}, // <CJK>
  {0xD879,0x5595}, // <CJK>
  {0xD87A,0x55a1}, // <CJK>
  {0xD87B,0x558e}, // <CJK>
  {0xD87C,0x570c}, // <CJK>
  {0xD87D,0x5829}, // <CJK>
  {0xD87E,0x5837}, // <CJK>
  {0xD8A1,0x5819}, // <CJK>
  {0xD8A2,0x581e}, // <CJK>
  {0xD8A3,0x5827}, // <CJK>
  {0xD8A4,0x5823}, // <CJK>
  {0xD8A5,0x5828}, // <CJK>
  {0xD8A6,0x57f5}, // <CJK>
  {0xD8A7,0x5848}, // <CJK>
  {0xD8A8,0x5825}, // <CJK>
  {0xD8A9,0x581c}, // <CJK>
  {0xD8AA,0x581b}, // <CJK>
  {0xD8AB,0x5833}, // <CJK>
  {0xD8AC,0x583f}, // <CJK>
  {0xD8AD,0x5836}, // <CJK>
  {0xD8AE,0x582e}, // <CJK>
  {0xD8AF,0x5839}, // <CJK>
  {0xD8B0,0x5838}, // <CJK>
  {0xD8B1,0x582d}, // <CJK>
  {0xD8B2,0x582c}, // <CJK>
  {0xD8B3,0x583b}, // <CJK>
  {0xD8B4,0x5961}, // <CJK>
  {0xD8B5,0x5aaf}, // <CJK>
  {0xD8B6,0x5a94}, // <CJK>
  {0xD8B7,0x5a9f}, // <CJK>
  {0xD8B8,0x5a7a}, // <CJK>
  {0xD8B9,0x5aa2}, // <CJK>
  {0xD8BA,0x5a9e}, // <CJK>
  {0xD8BB,0x5a78}, // <CJK>
  {0xD8BC,0x5aa6}, // <CJK>
  {0xD8BD,0x5a7c}, // <CJK>
  {0xD8BE,0x5aa5}, // <CJK>
  {0xD8BF,0x5aac}, // <CJK>
  {0xD8C0,0x5a95}, // <CJK>
  {0xD8C1,0x5aae}, // <CJK>
  {0xD8C2,0x5a37}, // <CJK>
  {0xD8C3,0x5a84}, // <CJK>
  {0xD8C4,0x5a8a}, // <CJK>
  {0xD8C5,0x5a97}, // <CJK>
  {0xD8C6,0x5a83}, // <CJK>
  {0xD8C7,0x5a8b}, // <CJK>
  {0xD8C8,0x5aa9}, // <CJK>
  {0xD8C9,0x5a7b}, // <CJK>
  {0xD8CA,0x5a7d}, // <CJK>
  {0xD8CB,0x5a8c}, // <CJK>
  {0xD8CC,0x5a9c}, // <CJK>
  {0xD8CD,0x5a8f}, // <CJK>
  {0xD8CE,0x5a93}, // <CJK>
  {0xD8CF,0x5a9d}, // <CJK>
  {0xD8D0,0x5bea}, // <CJK>
  {0xD8D1,0x5bcd}, // <CJK>
  {0xD8D2,0x5bcb}, // <CJK>
  {0xD8D3,0x5bd4}, // <CJK>
  {0xD8D4,0x5bd1}, // <CJK>
  {0xD8D5,0x5bca}, // <CJK>
  {0xD8D6,0x5bce}, // <CJK>
  {0xD8D7,0x5c0c}, // <CJK>
  {0xD8D8,0x5c30}, // <CJK>
  {0xD8D9,0x5d37}, // <CJK>
  {0xD8DA,0x5d43}, // <CJK>
  {0xD8DB,0x5d6b}, // <CJK>
  {0xD8DC,0x5d41}, // <CJK>
  {0xD8DD,0x5d4b}, // <CJK>
  {0xD8DE,0x5d3f}, // <CJK>
  {0xD8DF,0x5d35}, // <CJK>
  {0xD8E0,0x5d51}, // <CJK>
  {0xD8E1,0x5d4e}, // <CJK>
  {0xD8E2,0x5d55}, // <CJK>
  {0xD8E3,0x5d33}, // <CJK>
  {0xD8E4,0x5d3a}, // <CJK>
  {0xD8E5,0x5d52}, // <CJK>
  {0xD8E6,0x5d3d}, // <CJK>
  {0xD8E7,0x5d31}, // <CJK>
  {0xD8E8,0x5d59}, // <CJK>
  {0xD8E9,0x5d42}, // <CJK>
  {0xD8EA,0x5d39}, // <CJK>
  {0xD8EB,0x5d49}, // <CJK>
  {0xD8EC,0x5d38}, // <CJK>
  {0xD8ED,0x5d3c}, // <CJK>
  {0xD8EE,0x5d32}, // <CJK>
  {0xD8EF,0x5d36}, // <CJK>
  {0xD8F0,0x5d40}, // <CJK>
  {0xD8F1,0x5d45}, // <CJK>
  {0xD8F2,0x5e44}, // <CJK>
  {0xD8F3,0x5e41}, // <CJK>
  {0xD8F4,0x5f58}, // <CJK>
  {0xD8F5,0x5fa6}, // <CJK>
  {0xD8F6,0x5fa5}, // <CJK>
  {0xD8F7,0x5fab}, // <CJK>
  {0xD8F8,0x60c9}, // <CJK>
  {0xD8F9,0x60b9}, // <CJK>
  {0xD8FA,0x60cc}, // <CJK>
  {0xD8FB,0x60e2}, // <CJK>
  {0xD8FC,0x60ce}, // <CJK>
  {0xD8FD,0x60c4}, // <CJK>
  {0xD8FE,0x6114}, // <CJK>
  {0xD940,0x60f2}, // <CJK>
  {0xD941,0x610a}, // <CJK>
  {0xD942,0x6116}, // <CJK>
  {0xD943,0x6105}, // <CJK>
  {0xD944,0x60f5}, // <CJK>
  {0xD945,0x6113}, // <CJK>
  {0xD946,0x60f8}, // <CJK>
  {0xD947,0x60fc}, // <CJK>
  {0xD948,0x60fe}, // <CJK>
  {0xD949,0x60c1}, // <CJK>
  {0xD94A,0x6103}, // <CJK>
  {0xD94B,0x6118}, // <CJK>
  {0xD94C,0x611d}, // <CJK>
  {0xD94D,0x6110}, // <CJK>
  {0xD94E,0x60ff}, // <CJK>
  {0xD94F,0x6104}, // <CJK>
  {0xD950,0x610b}, // <CJK>
  {0xD951,0x624a}, // <CJK>
  {0xD952,0x6394}, // <CJK>
  {0xD953,0x63b1}, // <CJK>
  {0xD954,0x63b0}, // <CJK>
  {0xD955,0x63ce}, // <CJK>
  {0xD956,0x63e5}, // <CJK>
  {0xD957,0x63e8}, // <CJK>
  {0xD958,0x63ef}, // <CJK>
  {0xD959,0x63c3}, // <CJK>
  {0xD95A,0x649d}, // <CJK>
  {0xD95B,0x63f3}, // <CJK>
  {0xD95C,0x63ca}, // <CJK>
  {0xD95D,0x63e0}, // <CJK>
  {0xD95E,0x63f6}, // <CJK>
  {0xD95F,0x63d5}, // <CJK>
  {0xD960,0x63f2}, // <CJK>
  {0xD961,0x63f5}, // <CJK>
  {0xD962,0x6461}, // <CJK>
  {0xD963,0x63df}, // <CJK>
  {0xD964,0x63be}, // <CJK>
  {0xD965,0x63dd}, // <CJK>
  {0xD966,0x63dc}, // <CJK>
  {0xD967,0x63c4}, // <CJK>
  {0xD968,0x63d8}, // <CJK>
  {0xD969,0x63d3}, // <CJK>
  {0xD96A,0x63c2}, // <CJK>
  {0xD96B,0x63c7}, // <CJK>
  {0xD96C,0x63cc}, // <CJK>
  {0xD96D,0x63cb}, // <CJK>
  {0xD96E,0x63c8}, // <CJK>
  {0xD96F,0x63f0}, // <CJK>
  {0xD970,0x63d7}, // <CJK>
  {0xD971,0x63d9}, // <CJK>
  {0xD972,0x6532}, // <CJK>
  {0xD973,0x6567}, // <CJK>
  {0xD974,0x656a}, // <CJK>
  {0xD975,0x6564}, // <CJK>
  {0xD976,0x655c}, // <CJK>
  {0xD977,0x6568}, // <CJK>
  {0xD978,0x6565}, // <CJK>
  {0xD979,0x658c}, // <CJK>
  {0xD97A,0x659d}, // <CJK>
  {0xD97B,0x659e}, // <CJK>
  {0xD97C,0x65ae}, // <CJK>
  {0xD97D,0x65d0}, // <CJK>
  {0xD97E,0x65d2}, // <CJK>
  {0xD9A1,0x667c}, // <CJK>
  {0xD9A2,0x666c}, // <CJK>
  {0xD9A3,0x667b}, // <CJK>
  {0xD9A4,0x6680}, // <CJK>
  {0xD9A5,0x6671}, // <CJK>
  {0xD9A6,0x6679}, // <CJK>
  {0xD9A7,0x666a}, // <CJK>
  {0xD9A8,0x6672}, // <CJK>
  {0xD9A9,0x6701}, // <CJK>
  {0xD9AA,0x690c}, // <CJK>
  {0xD9AB,0x68d3}, // <CJK>
  {0xD9AC,0x6904}, // <CJK>
  {0xD9AD,0x68dc}, // <CJK>
  {0xD9AE,0x692a}, // <CJK>
  {0xD9AF,0x68ec}, // <CJK>
  {0xD9B0,0x68ea}, // <CJK>
  {0xD9B1,0x68f1}, // <CJK>
  {0xD9B2,0x690f}, // <CJK>
  {0xD9B3,0x68d6}, // <CJK>
  {0xD9B4,0x68f7}, // <CJK>
  {0xD9B5,0x68eb}, // <CJK>
  {0xD9B6,0x68e4}, // <CJK>
  {0xD9B7,0x68f6}, // <CJK>
  {0xD9B8,0x6913}, // <CJK>
  {0xD9B9,0x6910}, // <CJK>
  {0xD9BA,0x68f3}, // <CJK>
  {0xD9BB,0x68e1}, // <CJK>
  {0xD9BC,0x6907}, // <CJK>
  {0xD9BD,0x68cc}, // <CJK>
  {0xD9BE,0x6908}, // <CJK>
  {0xD9BF,0x6970}, // <CJK>
  {0xD9C0,0x68b4}, // <CJK>
  {0xD9C1,0x6911}, // <CJK>
  {0xD9C2,0x68ef}, // <CJK>
  {0xD9C3,0x68c6}, // <CJK>
  {0xD9C4,0x6914}, // <CJK>
  {0xD9C5,0x68f8}, // <CJK>
  {0xD9C6,0x68d0}, // <CJK>
  {0xD9C7,0x68fd}, // <CJK>
  {0xD9C8,0x68fc}, // <CJK>
  {0xD9C9,0x68e8}, // <CJK>
  {0xD9CA,0x690b}, // <CJK>
  {0xD9CB,0x690a}, // <CJK>
  {0xD9CC,0x6917}, // <CJK>
  {0xD9CD,0x68ce}, // <CJK>
  {0xD9CE,0x68c8}, // <CJK>
  {0xD9CF,0x68dd}, // <CJK>
  {0xD9D0,0x68de}, // <CJK>
  {0xD9D1,0x68e6}, // <CJK>
  {0xD9D2,0x68f4}, // <CJK>
  {0xD9D3,0x68d1}, // <CJK>
  {0xD9D4,0x6906}, // <CJK>
  {0xD9D5,0x68d4}, // <CJK>
  {0xD9D6,0x68e9}, // <CJK>
  {0xD9D7,0x6915}, // <CJK>
  {0xD9D8,0x6925}, // <CJK>
  {0xD9D9,0x68c7}, // <CJK>
  {0xD9DA,0x6b39}, // <CJK>
  {0xD9DB,0x6b3b}, // <CJK>
  {0xD9DC,0x6b3f}, // <CJK>
  {0xD9DD,0x6b3c}, // <CJK>
  {0xD9DE,0x6b94}, // <CJK>
  {0xD9DF,0x6b97}, // <CJK>
  {0xD9E0,0x6b99}, // <CJK>
  {0xD9E1,0x6b95}, // <CJK>
  {0xD9E2,0x6bbd}, // <CJK>
  {0xD9E3,0x6bf0}, // <CJK>
  {0xD9E4,0x6bf2}, // <CJK>
  {0xD9E5,0x6bf3}, // <CJK>
  {0xD9E6,0x6c30}, // <CJK>
  {0xD9E7,0x6dfc}, // <CJK>
  {0xD9E8,0x6e46}, // <CJK>
  {0xD9E9,0x6e47}, // <CJK>
  {0xD9EA,0x6e1f}, // <CJK>
  {0xD9EB,0x6e49}, // <CJK>
  {0xD9EC,0x6e88}, // <CJK>
  {0xD9ED,0x6e3c}, // <CJK>
  {0xD9EE,0x6e3d}, // <CJK>
  {0xD9EF,0x6e45}, // <CJK>
  {0xD9F0,0x6e62}, // <CJK>
  {0xD9F1,0x6e2b}, // <CJK>
  {0xD9F2,0x6e3f}, // <CJK>
  {0xD9F3,0x6e41}, // <CJK>
  {0xD9F4,0x6e5d}, // <CJK>
  {0xD9F5,0x6e73}, // <CJK>
  {0xD9F6,0x6e1c}, // <CJK>
  {0xD9F7,0x6e33}, // <CJK>
  {0xD9F8,0x6e4b}, // <CJK>
  {0xD9F9,0x6e40}, // <CJK>
  {0xD9FA,0x6e51}, // <CJK>
  {0xD9FB,0x6e3b}, // <CJK>
  {0xD9FC,0x6e03}, // <CJK>
  {0xD9FD,0x6e2e}, // <CJK>
  {0xD9FE,0x6e5e}, // <CJK>
  {0xDA40,0x6e68}, // <CJK>
  {0xDA41,0x6e5c}, // <CJK>
  {0xDA42,0x6e61}, // <CJK>
  {0xDA43,0x6e31}, // <CJK>
  {0xDA44,0x6e28}, // <CJK>
  {0xDA45,0x6e60}, // <CJK>
  {0xDA46,0x6e71}, // <CJK>
  {0xDA47,0x6e6b}, // <CJK>
  {0xDA48,0x6e39}, // <CJK>
  {0xDA49,0x6e22}, // <CJK>
  {0xDA4A,0x6e30}, // <CJK>
  {0xDA4B,0x6e53}, // <CJK>
  {0xDA4C,0x6e65}, // <CJK>
  {0xDA4D,0x6e27}, // <CJK>
  {0xDA4E,0x6e78}, // <CJK>
  {0xDA4F,0x6e64}, // <CJK>
  {0xDA50,0x6e77}, // <CJK>
  {0xDA51,0x6e55}, // <CJK>
  {0xDA52,0x6e79}, // <CJK>
  {0xDA53,0x6e52}, // <CJK>
  {0xDA54,0x6e66}, // <CJK>
  {0xDA55,0x6e35}, // <CJK>
  {0xDA56,0x6e36}, // <CJK>
  {0xDA57,0x6e5a}, // <CJK>
  {0xDA58,0x7120}, // <CJK>
  {0xDA59,0x711e}, // <CJK>
  {0xDA5A,0x712f}, // <CJK>
  {0xDA5B,0x70fb}, // <CJK>
  {0xDA5C,0x712e}, // <CJK>
  {0xDA5D,0x7131}, // <CJK>
  {0xDA5E,0x7123}, // <CJK>
  {0xDA5F,0x7125}, // <CJK>
  {0xDA60,0x7122}, // <CJK>
  {0xDA61,0x7132}, // <CJK>
  {0xDA62,0x711f}, // <CJK>
  {0xDA63,0x7128}, // <CJK>
  {0xDA64,0x713a}, // <CJK>
  {0xDA65,0x711b}, // <CJK>
  {0xDA66,0x724b}, // <CJK>
  {0xDA67,0x725a}, // <CJK>
  {0xDA68,0x7288}, // <CJK>
  {0xDA69,0x7289}, // <CJK>
  {0xDA6A,0x7286}, // <CJK>
  {0xDA6B,0x7285}, // <CJK>
  {0xDA6C,0x728b}, // <CJK>
  {0xDA6D,0x7312}, // <CJK>
  {0xDA6E,0x730b}, // <CJK>
  {0xDA6F,0x7330}, // <CJK>
  {0xDA70,0x7322}, // <CJK>
  {0xDA71,0x7331}, // <CJK>
  {0xDA72,0x7333}, // <CJK>
  {0xDA73,0x7327}, // <CJK>
  {0xDA74,0x7332}, // <CJK>
  {0xDA75,0x732d}, // <CJK>
  {0xDA76,0x7326}, // <CJK>
  {0xDA77,0x7323}, // <CJK>
  {0xDA78,0x7335}, // <CJK>
  {0xDA79,0x730c}, // <CJK>
  {0xDA7A,0x742e}, // <CJK>
  {0xDA7B,0x742c}, // <CJK>
  {0xDA7C,0x7430}, // <CJK>
  {0xDA7D,0x742b}, // <CJK>
  {0xDA7E,0x7416}, // <CJK>
  {0xDAA1,0x741a}, // <CJK>
  {0xDAA2,0x7421}, // <CJK>
  {0xDAA3,0x742d}, // <CJK>
  {0xDAA4,0x7431}, // <CJK>
  {0xDAA5,0x7424}, // <CJK>
  {0xDAA6,0x7423}, // <CJK>
  {0xDAA7,0x741d}, // <CJK>
  {0xDAA8,0x7429}, // <CJK>
  {0xDAA9,0x7420}, // <CJK>
  {0xDAAA,0x7432}, // <CJK>
  {0xDAAB,0x74fb}, // <CJK>
  {0xDAAC,0x752f}, // <CJK>
  {0xDAAD,0x756f}, // <CJK>
  {0xDAAE,0x756c}, // <CJK>
  {0xDAAF,0x75e7}, // <CJK>
  {0xDAB0,0x75da}, // <CJK>
  {0xDAB1,0x75e1}, // <CJK>
  {0xDAB2,0x75e6}, // <CJK>
  {0xDAB3,0x75dd}, // <CJK>
  {0xDAB4,0x75df}, // <CJK>
  {0xDAB5,0x75e4}, // <CJK>
  {0xDAB6,0x75d7}, // <CJK>
  {0xDAB7,0x7695}, // <CJK>
  {0xDAB8,0x7692}, // <CJK>
  {0xDAB9,0x76da}, // <CJK>
  {0xDABA,0x7746}, // <CJK>
  {0xDABB,0x7747}, // <CJK>
  {0xDABC,0x7744}, // <CJK>
  {0xDABD,0x774d}, // <CJK>
  {0xDABE,0x7745}, // <CJK>
  {0xDABF,0x774a}, // <CJK>
  {0xDAC0,0x774e}, // <CJK>
  {0xDAC1,0x774b}, // <CJK>
  {0xDAC2,0x774c}, // <CJK>
  {0xDAC3,0x77de}, // <CJK>
  {0xDAC4,0x77ec}, // <CJK>
  {0xDAC5,0x7860}, // <CJK>
  {0xDAC6,0x7864}, // <CJK>
  {0xDAC7,0x7865}, // <CJK>
  {0xDAC8,0x785c}, // <CJK>
  {0xDAC9,0x786d}, // <CJK>
  {0xDACA,0x7871}, // <CJK>
  {0xDACB,0x786a}, // <CJK>
  {0xDACC,0x786e}, // <CJK>
  {0xDACD,0x7870}, // <CJK>
  {0xDACE,0x7869}, // <CJK>
  {0xDACF,0x7868}, // <CJK>
  {0xDAD0,0x785e}, // <CJK>
  {0xDAD1,0x7862}, // <CJK>
  {0xDAD2,0x7974}, // <CJK>
  {0xDAD3,0x7973}, // <CJK>
  {0xDAD4,0x7972}, // <CJK>
  {0xDAD5,0x7970}, // <CJK>
  {0xDAD6,0x7a02}, // <CJK>
  {0xDAD7,0x7a0a}, // <CJK>
  {0xDAD8,0x7a03}, // <CJK>
  {0xDAD9,0x7a0c}, // <CJK>
  {0xDADA,0x7a04}, // <CJK>
  {0xDADB,0x7a99}, // <CJK>
  {0xDADC,0x7ae6}, // <CJK>
  {0xDADD,0x7ae4}, // <CJK>
  {0xDADE,0x7b4a}, // <CJK>
  {0xDADF,0x7b3b}, // <CJK>
  {0xDAE0,0x7b44}, // <CJK>
  {0xDAE1,0x7b48}, // <CJK>
  {0xDAE2,0x7b4c}, // <CJK>
  {0xDAE3,0x7b4e}, // <CJK>
  {0xDAE4,0x7b40}, // <CJK>
  {0xDAE5,0x7b58}, // <CJK>
  {0xDAE6,0x7b45}, // <CJK>
  {0xDAE7,0x7ca2}, // <CJK>
  {0xDAE8,0x7c9e}, // <CJK>
  {0xDAE9,0x7ca8}, // <CJK>
  {0xDAEA,0x7ca1}, // <CJK>
  {0xDAEB,0x7d58}, // <CJK>
  {0xDAEC,0x7d6f}, // <CJK>
  {0xDAED,0x7d63}, // <CJK>
  {0xDAEE,0x7d53}, // <CJK>
  {0xDAEF,0x7d56}, // <CJK>
  {0xDAF0,0x7d67}, // <CJK>
  {0xDAF1,0x7d6a}, // <CJK>
  {0xDAF2,0x7d4f}, // <CJK>
  {0xDAF3,0x7d6d}, // <CJK>
  {0xDAF4,0x7d5c}, // <CJK>
  {0xDAF5,0x7d6b}, // <CJK>
  {0xDAF6,0x7d52}, // <CJK>
  {0xDAF7,0x7d54}, // <CJK>
  {0xDAF8,0x7d69}, // <CJK>
  {0xDAF9,0x7d51}, // <CJK>
  {0xDAFA,0x7d5f}, // <CJK>
  {0xDAFB,0x7d4e}, // <CJK>
  {0xDAFC,0x7f3e}, // <CJK>
  {0xDAFD,0x7f3f}, // <CJK>
  {0xDAFE,0x7f65}, // <CJK>
  {0xDB40,0x7f66}, // <CJK>
  {0xDB41,0x7fa2}, // <CJK>
  {0xDB42,0x7fa0}, // <CJK>
  {0xDB43,0x7fa1}, // <CJK>
  {0xDB44,0x7fd7}, // <CJK>
  {0xDB45,0x8051}, // <CJK>
  {0xDB46,0x804f}, // <CJK>
  {0xDB47,0x8050}, // <CJK>
  {0xDB48,0x80fe}, // <CJK>
  {0xDB49,0x80d4}, // <CJK>
  {0xDB4A,0x8143}, // <CJK>
  {0xDB4B,0x814a}, // <CJK>
  {0xDB4C,0x8152}, // <CJK>
  {0xDB4D,0x814f}, // <CJK>
  {0xDB4E,0x8147}, // <CJK>
  {0xDB4F,0x813d}, // <CJK>
  {0xDB50,0x814d}, // <CJK>
  {0xDB51,0x813a}, // <CJK>
  {0xDB52,0x81e6}, // <CJK>
  {0xDB53,0x81ee}, // <CJK>
  {0xDB54,0x81f7}, // <CJK>
  {0xDB55,0x81f8}, // <CJK>
  {0xDB56,0x81f9}, // <CJK>
  {0xDB57,0x8204}, // <CJK>
  {0xDB58,0x823c}, // <CJK>
  {0xDB59,0x823d}, // <CJK>
  {0xDB5A,0x823f}, // <CJK>
  {0xDB5B,0x8275}, // <CJK>
  {0xDB5C,0x833b}, // <CJK>
  {0xDB5D,0x83cf}, // <CJK>
  {0xDB5E,0x83f9}, // <CJK>
  {0xDB5F,0x8423}, // <CJK>
  {0xDB60,0x83c0}, // <CJK>
  {0xDB61,0x83e8}, // <CJK>
  {0xDB62,0x8412}, // <CJK>
  {0xDB63,0x83e7}, // <CJK>
  {0xDB64,0x83e4}, // <CJK>
  {0xDB65,0x83fc}, // <CJK>
  {0xDB66,0x83f6}, // <CJK>
  {0xDB67,0x8410}, // <CJK>
  {0xDB68,0x83c6}, // <CJK>
  {0xDB69,0x83c8}, // <CJK>
  {0xDB6A,0x83eb}, // <CJK>
  {0xDB6B,0x83e3}, // <CJK>
  {0xDB6C,0x83bf}, // <CJK>
  {0xDB6D,0x8401}, // <CJK>
  {0xDB6E,0x83dd}, // <CJK>
  {0xDB6F,0x83e5}, // <CJK>
  {0xDB70,0x83d8}, // <CJK>
  {0xDB71,0x83ff}, // <CJK>
  {0xDB72,0x83e1}, // <CJK>
  {0xDB73,0x83cb}, // <CJK>
  {0xDB74,0x83ce}, // <CJK>
  {0xDB75,0x83d6}, // <CJK>
  {0xDB76,0x83f5}, // <CJK>
  {0xDB77,0x83c9}, // <CJK>
  {0xDB78,0x8409}, // <CJK>
  {0xDB79,0x840f}, // <CJK>
  {0xDB7A,0x83de}, // <CJK>
  {0xDB7B,0x8411}, // <CJK>
  {0xDB7C,0x8406}, // <CJK>
  {0xDB7D,0x83c2}, // <CJK>
  {0xDB7E,0x83f3}, // <CJK>
  {0xDBA1,0x83d5}, // <CJK>
  {0xDBA2,0x83fa}, // <CJK>
  {0xDBA3,0x83c7}, // <CJK>
  {0xDBA4,0x83d1}, // <CJK>
  {0xDBA5,0x83ea}, // <CJK>
  {0xDBA6,0x8413}, // <CJK>
  {0xDBA7,0x83c3}, // <CJK>
  {0xDBA8,0x83ec}, // <CJK>
  {0xDBA9,0x83ee}, // <CJK>
  {0xDBAA,0x83c4}, // <CJK>
  {0xDBAB,0x83fb}, // <CJK>
  {0xDBAC,0x83d7}, // <CJK>
  {0xDBAD,0x83e2}, // <CJK>
  {0xDBAE,0x841b}, // <CJK>
  {0xDBAF,0x83db}, // <CJK>
  {0xDBB0,0x83fe}, // <CJK>
  {0xDBB1,0x86d8}, // <CJK>
  {0xDBB2,0x86e2}, // <CJK>
  {0xDBB3,0x86e6}, // <CJK>
  {0xDBB4,0x86d3}, // <CJK>
  {0xDBB5,0x86e3}, // <CJK>
  {0xDBB6,0x86da}, // <CJK>
  {0xDBB7,0x86ea}, // <CJK>
  {0xDBB8,0x86dd}, // <CJK>
  {0xDBB9,0x86eb}, // <CJK>
  {0xDBBA,0x86dc}, // <CJK>
  {0xDBBB,0x86ec}, // <CJK>
  {0xDBBC,0x86e9}, // <CJK>
  {0xDBBD,0x86d7}, // <CJK>
  {0xDBBE,0x86e8}, // <CJK>
  {0xDBBF,0x86d1}, // <CJK>
  {0xDBC0,0x8848}, // <CJK>
  {0xDBC1,0x8856}, // <CJK>
  {0xDBC2,0x8855}, // <CJK>
  {0xDBC3,0x88ba}, // <CJK>
  {0xDBC4,0x88d7}, // <CJK>
  {0xDBC5,0x88b9}, // <CJK>
  {0xDBC6,0x88b8}, // <CJK>
  {0xDBC7,0x88c0}, // <CJK>
  {0xDBC8,0x88be}, // <CJK>
  {0xDBC9,0x88b6}, // <CJK>
  {0xDBCA,0x88bc}, // <CJK>
  {0xDBCB,0x88b7}, // <CJK>
  {0xDBCC,0x88bd}, // <CJK>
  {0xDBCD,0x88b2}, // <CJK>
  {0xDBCE,0x8901}, // <CJK>
  {0xDBCF,0x88c9}, // <CJK>
  {0xDBD0,0x8995}, // <CJK>
  {0xDBD1,0x8998}, // <CJK>
  {0xDBD2,0x8997}, // <CJK>
  {0xDBD3,0x89dd}, // <CJK>
  {0xDBD4,0x89da}, // <CJK>
  {0xDBD5,0x89db}, // <CJK>
  {0xDBD6,0x8a4e}, // <CJK>
  {0xDBD7,0x8a4d}, // <CJK>
  {0xDBD8,0x8a39}, // <CJK>
  {0xDBD9,0x8a59}, // <CJK>
  {0xDBDA,0x8a40}, // <CJK>
  {0xDBDB,0x8a57}, // <CJK>
  {0xDBDC,0x8a58}, // <CJK>
  {0xDBDD,0x8a44}, // <CJK>
  {0xDBDE,0x8a45}, // <CJK>
  {0xDBDF,0x8a52}, // <CJK>
  {0xDBE0,0x8a48}, // <CJK>
  {0xDBE1,0x8a51}, // <CJK>
  {0xDBE2,0x8a4a}, // <CJK>
  {0xDBE3,0x8a4c}, // <CJK>
  {0xDBE4,0x8a4f}, // <CJK>
  {0xDBE5,0x8c5f}, // <CJK>
  {0xDBE6,0x8c81}, // <CJK>
  {0xDBE7,0x8c80}, // <CJK>
  {0xDBE8,0x8cba}, // <CJK>
  {0xDBE9,0x8cbe}, // <CJK>
  {0xDBEA,0x8cb0}, // <CJK>
  {0xDBEB,0x8cb9}, // <CJK>
  {0xDBEC,0x8cb5}, // <CJK>
  {0xDBED,0x8d84}, // <CJK>
  {0xDBEE,0x8d80}, // <CJK>
  {0xDBEF,0x8d89}, // <CJK>
  {0xDBF0,0x8dd8}, // <CJK>
  {0xDBF1,0x8dd3}, // <CJK>
  {0xDBF2,0x8dcd}, // <CJK>
  {0xDBF3,0x8dc7}, // <CJK>
  {0xDBF4,0x8dd6}, // <CJK>
  {0xDBF5,0x8ddc}, // <CJK>
  {0xDBF6,0x8dcf}, // <CJK>
  {0xDBF7,0x8dd5}, // <CJK>
  {0xDBF8,0x8dd9}, // <CJK>
  {0xDBF9,0x8dc8}, // <CJK>
  {0xDBFA,0x8dd7}, // <CJK>
  {0xDBFB,0x8dc5}, // <CJK>
  {0xDBFC,0x8eef}, // <CJK>
  {0xDBFD,0x8ef7}, // <CJK>
  {0xDBFE,0x8efa}, // <CJK>
  {0xDC40,0x8ef9}, // <CJK>
  {0xDC41,0x8ee6}, // <CJK>
  {0xDC42,0x8eee}, // <CJK>
  {0xDC43,0x8ee5}, // <CJK>
  {0xDC44,0x8ef5}, // <CJK>
  {0xDC45,0x8ee7}, // <CJK>
  {0xDC46,0x8ee8}, // <CJK>
  {0xDC47,0x8ef6}, // <CJK>
  {0xDC48,0x8eeb}, // <CJK>
  {0xDC49,0x8ef1}, // <CJK>
  {0xDC4A,0x8eec}, // <CJK>
  {0xDC4B,0x8ef4}, // <CJK>
  {0xDC4C,0x8ee9}, // <CJK>
  {0xDC4D,0x902d}, // <CJK>
  {0xDC4E,0x9034}, // <CJK>
  {0xDC4F,0x902f}, // <CJK>
  {0xDC50,0x9106}, // <CJK>
  {0xDC51,0x912c}, // <CJK>
  {0xDC52,0x9104}, // <CJK>
  {0xDC53,0x90ff}, // <CJK>
  {0xDC54,0x90fc}, // <CJK>
  {0xDC55,0x9108}, // <CJK>
  {0xDC56,0x90f9}, // <CJK>
  {0xDC57,0x90fb}, // <CJK>
  {0xDC58,0x9101}, // <CJK>
  {0xDC59,0x9100}, // <CJK>
  {0xDC5A,0x9107}, // <CJK>
  {0xDC5B,0x9105}, // <CJK>
  {0xDC5C,0x9103}, // <CJK>
  {0xDC5D,0x9161}, // <CJK>
  {0xDC5E,0x9164}, // <CJK>
  {0xDC5F,0x915f}, // <CJK>
  {0xDC60,0x9162}, // <CJK>
  {0xDC61,0x9160}, // <CJK>
  {0xDC62,0x9201}, // <CJK>
  {0xDC63,0x920a}, // <CJK>
  {0xDC64,0x9225}, // <CJK>
  {0xDC65,0x9203}, // <CJK>
  {0xDC66,0x921a}, // <CJK>
  {0xDC67,0x9226}, // <CJK>
  {0xDC68,0x920f}, // <CJK>
  {0xDC69,0x920c}, // <CJK>
  {0xDC6A,0x9200}, // <CJK>
  {0xDC6B,0x9212}, // <CJK>
  {0xDC6C,0x91ff}, // <CJK>
  {0xDC6D,0x91fd}, // <CJK>
  {0xDC6E,0x9206}, // <CJK>
  {0xDC6F,0x9204}, // <CJK>
  {0xDC70,0x9227}, // <CJK>
  {0xDC71,0x9202}, // <CJK>
  {0xDC72,0x921c}, // <CJK>
  {0xDC73,0x9224}, // <CJK>
  {0xDC74,0x9219}, // <CJK>
  {0xDC75,0x9217}, // <CJK>
  {0xDC76,0x9205}, // <CJK>
  {0xDC77,0x9216}, // <CJK>
  {0xDC78,0x957b}, // <CJK>
  {0xDC79,0x958d}, // <CJK>
  {0xDC7A,0x958c}, // <CJK>
  {0xDC7B,0x9590}, // <CJK>
  {0xDC7C,0x9687}, // <CJK>
  {0xDC7D,0x967e}, // <CJK>
  {0xDC7E,0x9688}, // <CJK>
  {0xDCA1,0x9689}, // <CJK>
  {0xDCA2,0x9683}, // <CJK>
  {0xDCA3,0x9680}, // <CJK>
  {0xDCA4,0x96c2}, // <CJK>
  {0xDCA5,0x96c8}, // <CJK>
  {0xDCA6,0x96c3}, // <CJK>
  {0xDCA7,0x96f1}, // <CJK>
  {0xDCA8,0x96f0}, // <CJK>
  {0xDCA9,0x976c}, // <CJK>
  {0xDCAA,0x9770}, // <CJK>
  {0xDCAB,0x976e}, // <CJK>
  {0xDCAC,0x9807}, // <CJK>
  {0xDCAD,0x98a9}, // <CJK>
  {0xDCAE,0x98eb}, // <CJK>
  {0xDCAF,0x9ce6}, // <CJK>
  {0xDCB0,0x9ef9}, // <CJK>
  {0xDCB1,0x4e83}, // <CJK>
  {0xDCB2,0x4e84}, // <CJK>
  {0xDCB3,0x4eb6}, // <CJK>
  {0xDCB4,0x50bd}, // <CJK>
  {0xDCB5,0x50bf}, // <CJK>
  {0xDCB6,0x50c6}, // <CJK>
  {0xDCB7,0x50ae}, // <CJK>
  {0xDCB8,0x50c4}, // <CJK>
  {0xDCB9,0x50ca}, // <CJK>
  {0xDCBA,0x50b4}, // <CJK>
  {0xDCBB,0x50c8}, // <CJK>
  {0xDCBC,0x50c2}, // <CJK>
  {0xDCBD,0x50b0}, // <CJK>
  {0xDCBE,0x50c1}, // <CJK>
  {0xDCBF,0x50ba}, // <CJK>
  {0xDCC0,0x50b1}, // <CJK>
  {0xDCC1,0x50cb}, // <CJK>
  {0xDCC2,0x50c9}, // <CJK>
  {0xDCC3,0x50b6}, // <CJK>
  {0xDCC4,0x50b8}, // <CJK>
  {0xDCC5,0x51d7}, // <CJK>
  {0xDCC6,0x527a}, // <CJK>
  {0xDCC7,0x5278}, // <CJK>
  {0xDCC8,0x527b}, // <CJK>
  {0xDCC9,0x527c}, // <CJK>
  {0xDCCA,0x55c3}, // <CJK>
  {0xDCCB,0x55db}, // <CJK>
  {0xDCCC,0x55cc}, // <CJK>
  {0xDCCD,0x55d0}, // <CJK>
  {0xDCCE,0x55cb}, // <CJK>
  {0xDCCF,0x55ca}, // <CJK>
  {0xDCD0,0x55dd}, // <CJK>
  {0xDCD1,0x55c0}, // <CJK>
  {0xDCD2,0x55d4}, // <CJK>
  {0xDCD3,0x55c4}, // <CJK>
  {0xDCD4,0x55e9}, // <CJK>
  {0xDCD5,0x55bf}, // <CJK>
  {0xDCD6,0x55d2}, // <CJK>
  {0xDCD7,0x558d}, // <CJK>
  {0xDCD8,0x55cf}, // <CJK>
  {0xDCD9,0x55d5}, // <CJK>
  {0xDCDA,0x55e2}, // <CJK>
  {0xDCDB,0x55d6}, // <CJK>
  {0xDCDC,0x55c8}, // <CJK>
  {0xDCDD,0x55f2}, // <CJK>
  {0xDCDE,0x55cd}, // <CJK>
  {0xDCDF,0x55d9}, // <CJK>
  {0xDCE0,0x55c2}, // <CJK>
  {0xDCE1,0x5714}, // <CJK>
  {0xDCE2,0x5853}, // <CJK>
  {0xDCE3,0x5868}, // <CJK>
  {0xDCE4,0x5864}, // <CJK>
  {0xDCE5,0x584f}, // <CJK>
  {0xDCE6,0x584d}, // <CJK>
  {0xDCE7,0x5849}, // <CJK>
  {0xDCE8,0x586f}, // <CJK>
  {0xDCE9,0x5855}, // <CJK>
  {0xDCEA,0x584e}, // <CJK>
  {0xDCEB,0x585d}, // <CJK>
  {0xDCEC,0x5859}, // <CJK>
  {0xDCED,0x5865}, // <CJK>
  {0xDCEE,0x585b}, // <CJK>
  {0xDCEF,0x583d}, // <CJK>
  {0xDCF0,0x5863}, // <CJK>
  {0xDCF1,0x5871}, // <CJK>
  {0xDCF2,0x58fc}, // <CJK>
  {0xDCF3,0x5ac7}, // <CJK>
  {0xDCF4,0x5ac4}, // <CJK>
  {0xDCF5,0x5acb}, // <CJK>
  {0xDCF6,0x5aba}, // <CJK>
  {0xDCF7,0x5ab8}, // <CJK>
  {0xDCF8,0x5ab1}, // <CJK>
  {0xDCF9,0x5ab5}, // <CJK>
  {0xDCFA,0x5ab0}, // <CJK>
  {0xDCFB,0x5abf}, // <CJK>
  {0xDCFC,0x5ac8}, // <CJK>
  {0xDCFD,0x5abb}, // <CJK>
  {0xDCFE,0x5ac6}, // <CJK>
  {0xDD40,0x5ab7}, // <CJK>
  {0xDD41,0x5ac0}, // <CJK>
  {0xDD42,0x5aca}, // <CJK>
  {0xDD43,0x5ab4}, // <CJK>
  {0xDD44,0x5ab6}, // <CJK>
  {0xDD45,0x5acd}, // <CJK>
  {0xDD46,0x5ab9}, // <CJK>
  {0xDD47,0x5a90}, // <CJK>
  {0xDD48,0x5bd6}, // <CJK>
  {0xDD49,0x5bd8}, // <CJK>
  {0xDD4A,0x5bd9}, // <CJK>
  {0xDD4B,0x5c1f}, // <CJK>
  {0xDD4C,0x5c33}, // <CJK>
  {0xDD4D,0x5d71}, // <CJK>
  {0xDD4E,0x5d63}, // <CJK>
  {0xDD4F,0x5d4a}, // <CJK>
  {0xDD50,0x5d65}, // <CJK>
  {0xDD51,0x5d72}, // <CJK>
  {0xDD52,0x5d6c}, // <CJK>
  {0xDD53,0x5d5e}, // <CJK>
  {0xDD54,0x5d68}, // <CJK>
  {0xDD55,0x5d67}, // <CJK>
  {0xDD56,0x5d62}, // <CJK>
  {0xDD57,0x5df0}, // <CJK>
  {0xDD58,0x5e4f}, // <CJK>
  {0xDD59,0x5e4e}, // <CJK>
  {0xDD5A,0x5e4a}, // <CJK>
  {0xDD5B,0x5e4d}, // <CJK>
  {0xDD5C,0x5e4b}, // <CJK>
  {0xDD5D,0x5ec5}, // <CJK>
  {0xDD5E,0x5ecc}, // <CJK>
  {0xDD5F,0x5ec6}, // <CJK>
  {0xDD60,0x5ecb}, // <CJK>
  {0xDD61,0x5ec7}, // <CJK>
  {0xDD62,0x5f40}, // <CJK>
  {0xDD63,0x5faf}, // <CJK>
  {0xDD64,0x5fad}, // <CJK>
  {0xDD65,0x60f7}, // <CJK>
  {0xDD66,0x6149}, // <CJK>
  {0xDD67,0x614a}, // <CJK>
  {0xDD68,0x612b}, // <CJK>
  {0xDD69,0x6145}, // <CJK>
  {0xDD6A,0x6136}, // <CJK>
  {0xDD6B,0x6132}, // <CJK>
  {0xDD6C,0x612e}, // <CJK>
  {0xDD6D,0x6146}, // <CJK>
  {0xDD6E,0x612f}, // <CJK>
  {0xDD6F,0x614f}, // <CJK>
  {0xDD70,0x6129}, // <CJK>
  {0xDD71,0x6140}, // <CJK>
  {0xDD72,0x6220}, // <CJK>
  {0xDD73,0x9168}, // <CJK>
  {0xDD74,0x6223}, // <CJK>
  {0xDD75,0x6225}, // <CJK>
  {0xDD76,0x6224}, // <CJK>
  {0xDD77,0x63c5}, // <CJK>
  {0xDD78,0x63f1}, // <CJK>
  {0xDD79,0x63eb}, // <CJK>
  {0xDD7A,0x6410}, // <CJK>
  {0xDD7B,0x6412}, // <CJK>
  {0xDD7C,0x6409}, // <CJK>
  {0xDD7D,0x6420}, // <CJK>
  {0xDD7E,0x6424}, // <CJK>
  {0xDDA1,0x6433}, // <CJK>
  {0xDDA2,0x6443}, // <CJK>
  {0xDDA3,0x641f}, // <CJK>
  {0xDDA4,0x6415}, // <CJK>
  {0xDDA5,0x6418}, // <CJK>
  {0xDDA6,0x6439}, // <CJK>
  {0xDDA7,0x6437}, // <CJK>
  {0xDDA8,0x6422}, // <CJK>
  {0xDDA9,0x6423}, // <CJK>
  {0xDDAA,0x640c}, // <CJK>
  {0xDDAB,0x6426}, // <CJK>
  {0xDDAC,0x6430}, // <CJK>
  {0xDDAD,0x6428}, // <CJK>
  {0xDDAE,0x6441}, // <CJK>
  {0xDDAF,0x6435}, // <CJK>
  {0xDDB0,0x642f}, // <CJK>
  {0xDDB1,0x640a}, // <CJK>
  {0xDDB2,0x641a}, // <CJK>
  {0xDDB3,0x6440}, // <CJK>
  {0xDDB4,0x6425}, // <CJK>
  {0xDDB5,0x6427}, // <CJK>
  {0xDDB6,0x640b}, // <CJK>
  {0xDDB7,0x63e7}, // <CJK>
  {0xDDB8,0x641b}, // <CJK>
  {0xDDB9,0x642e}, // <CJK>
  {0xDDBA,0x6421}, // <CJK>
  {0xDDBB,0x640e}, // <CJK>
  {0xDDBC,0x656f}, // <CJK>
  {0xDDBD,0x6592}, // <CJK>
  {0xDDBE,0x65d3}, // <CJK>
  {0xDDBF,0x6686}, // <CJK>
  {0xDDC0,0x668c}, // <CJK>
  {0xDDC1,0x6695}, // <CJK>
  {0xDDC2,0x6690}, // <CJK>
  {0xDDC3,0x668b}, // <CJK>
  {0xDDC4,0x668a}, // <CJK>
  {0xDDC5,0x6699}, // <CJK>
  {0xDDC6,0x6694}, // <CJK>
  {0xDDC7,0x6678}, // <CJK>
  {0xDDC8,0x6720}, // <CJK>
  {0xDDC9,0x6966}, // <CJK>
  {0xDDCA,0x695f}, // <CJK>
  {0xDDCB,0x6938}, // <CJK>
  {0xDDCC,0x694e}, // <CJK>
  {0xDDCD,0x6962}, // <CJK>
  {0xDDCE,0x6971}, // <CJK>
  {0xDDCF,0x693f}, // <CJK>
  {0xDDD0,0x6945}, // <CJK>
  {0xDDD1,0x696a}, // <CJK>
  {0xDDD2,0x6939}, // <CJK>
  {0xDDD3,0x6942}, // <CJK>
  {0xDDD4,0x6957}, // <CJK>
  {0xDDD5,0x6959}, // <CJK>
  {0xDDD6,0x697a}, // <CJK>
  {0xDDD7,0x6948}, // <CJK>
  {0xDDD8,0x6949}, // <CJK>
  {0xDDD9,0x6935}, // <CJK>
  {0xDDDA,0x696c}, // <CJK>
  {0xDDDB,0x6933}, // <CJK>
  {0xDDDC,0x693d}, // <CJK>
  {0xDDDD,0x6965}, // <CJK>
  {0xDDDE,0x68f0}, // <CJK>
  {0xDDDF,0x6978}, // <CJK>
  {0xDDE0,0x6934}, // <CJK>
  {0xDDE1,0x6969}, // <CJK>
  {0xDDE2,0x6940}, // <CJK>
  {0xDDE3,0x696f}, // <CJK>
  {0xDDE4,0x6944}, // <CJK>
  {0xDDE5,0x6976}, // <CJK>
  {0xDDE6,0x6958}, // <CJK>
  {0xDDE7,0x6941}, // <CJK>
  {0xDDE8,0x6974}, // <CJK>
  {0xDDE9,0x694c}, // <CJK>
  {0xDDEA,0x693b}, // <CJK>
  {0xDDEB,0x694b}, // <CJK>
  {0xDDEC,0x6937}, // <CJK>
  {0xDDED,0x695c}, // <CJK>
  {0xDDEE,0x694f}, // <CJK>
  {0xDDEF,0x6951}, // <CJK>
  {0xDDF0,0x6932}, // <CJK>
  {0xDDF1,0x6952}, // <CJK>
  {0xDDF2,0x692f}, // <CJK>
  {0xDDF3,0x697b}, // <CJK>
  {0xDDF4,0x693c}, // <CJK>
  {0xDDF5,0x6b46}, // <CJK>
  {0xDDF6,0x6b45}, // <CJK>
  {0xDDF7,0x6b43}, // <CJK>
  {0xDDF8,0x6b42}, // <CJK>
  {0xDDF9,0x6b48}, // <CJK>
  {0xDDFA,0x6b41}, // <CJK>
  {0xDDFB,0x6b9b}, // <CJK>
  {0xDDFC,0xfa0d}, // CJK COMPATIBILITY IDEOGRAPH-FA0D
  {0xDDFD,0x6bfb}, // <CJK>
  {0xDDFE,0x6bfc}, // <CJK>
  {0xDE40,0x6bf9}, // <CJK>
  {0xDE41,0x6bf7}, // <CJK>
  {0xDE42,0x6bf8}, // <CJK>
  {0xDE43,0x6e9b}, // <CJK>
  {0xDE44,0x6ed6}, // <CJK>
  {0xDE45,0x6ec8}, // <CJK>
  {0xDE46,0x6e8f}, // <CJK>
  {0xDE47,0x6ec0}, // <CJK>
  {0xDE48,0x6e9f}, // <CJK>
  {0xDE49,0x6e93}, // <CJK>
  {0xDE4A,0x6e94}, // <CJK>
  {0xDE4B,0x6ea0}, // <CJK>
  {0xDE4C,0x6eb1}, // <CJK>
  {0xDE4D,0x6eb9}, // <CJK>
  {0xDE4E,0x6ec6}, // <CJK>
  {0xDE4F,0x6ed2}, // <CJK>
  {0xDE50,0x6ebd}, // <CJK>
  {0xDE51,0x6ec1}, // <CJK>
  {0xDE52,0x6e9e}, // <CJK>
  {0xDE53,0x6ec9}, // <CJK>
  {0xDE54,0x6eb7}, // <CJK>
  {0xDE55,0x6eb0}, // <CJK>
  {0xDE56,0x6ecd}, // <CJK>
  {0xDE57,0x6ea6}, // <CJK>
  {0xDE58,0x6ecf}, // <CJK>
  {0xDE59,0x6eb2}, // <CJK>
  {0xDE5A,0x6ebe}, // <CJK>
  {0xDE5B,0x6ec3}, // <CJK>
  {0xDE5C,0x6edc}, // <CJK>
  {0xDE5D,0x6ed8}, // <CJK>
  {0xDE5E,0x6e99}, // <CJK>
  {0xDE5F,0x6e92}, // <CJK>
  {0xDE60,0x6e8e}, // <CJK>
  {0xDE61,0x6e8d}, // <CJK>
  {0xDE62,0x6ea4}, // <CJK>
  {0xDE63,0x6ea1}, // <CJK>
  {0xDE64,0x6ebf}, // <CJK>
  {0xDE65,0x6eb3}, // <CJK>
  {0xDE66,0x6ed0}, // <CJK>
  {0xDE67,0x6eca}, // <CJK>
  {0xDE68,0x6e97}, // <CJK>
  {0xDE69,0x6eae}, // <CJK>
  {0xDE6A,0x6ea3}, // <CJK>
  {0xDE6B,0x7147}, // <CJK>
  {0xDE6C,0x7154}, // <CJK>
  {0xDE6D,0x7152}, // <CJK>
  {0xDE6E,0x7163}, // <CJK>
  {0xDE6F,0x7160}, // <CJK>
  {0xDE70,0x7141}, // <CJK>
  {0xDE71,0x715d}, // <CJK>
  {0xDE72,0x7162}, // <CJK>
  {0xDE73,0x7172}, // <CJK>
  {0xDE74,0x7178}, // <CJK>
  {0xDE75,0x716a}, // <CJK>
  {0xDE76,0x7161}, // <CJK>
  {0xDE77,0x7142}, // <CJK>
  {0xDE78,0x7158}, // <CJK>
  {0xDE79,0x7143}, // <CJK>
  {0xDE7A,0x714b}, // <CJK>
  {0xDE7B,0x7170}, // <CJK>
  {0xDE7C,0x715f}, // <CJK>
  {0xDE7D,0x7150}, // <CJK>
  {0xDE7E,0x7153}, // <CJK>
  {0xDEA1,0x7144}, // <CJK>
  {0xDEA2,0x714d}, // <CJK>
  {0xDEA3,0x715a}, // <CJK>
  {0xDEA4,0x724f}, // <CJK>
  {0xDEA5,0x728d}, // <CJK>
  {0xDEA6,0x728c}, // <CJK>
  {0xDEA7,0x7291}, // <CJK>
  {0xDEA8,0x7290}, // <CJK>
  {0xDEA9,0x728e}, // <CJK>
  {0xDEAA,0x733c}, // <CJK>
  {0xDEAB,0x7342}, // <CJK>
  {0xDEAC,0x733b}, // <CJK>
  {0xDEAD,0x733a}, // <CJK>
  {0xDEAE,0x7340}, // <CJK>
  {0xDEAF,0x734a}, // <CJK>
  {0xDEB0,0x7349}, // <CJK>
  {0xDEB1,0x7444}, // <CJK>
  {0xDEB2,0x744a}, // <CJK>
  {0xDEB3,0x744b}, // <CJK>
  {0xDEB4,0x7452}, // <CJK>
  {0xDEB5,0x7451}, // <CJK>
  {0xDEB6,0x7457}, // <CJK>
  {0xDEB7,0x7440}, // <CJK>
  {0xDEB8,0x744f}, // <CJK>
  {0xDEB9,0x7450}, // <CJK>
  {0xDEBA,0x744e}, // <CJK>
  {0xDEBB,0x7442}, // <CJK>
  {0xDEBC,0x7446}, // <CJK>
  {0xDEBD,0x744d}, // <CJK>
  {0xDEBE,0x7454}, // <CJK>
  {0xDEBF,0x74e1}, // <CJK>
  {0xDEC0,0x74ff}, // <CJK>
  {0xDEC1,0x74fe}, // <CJK>
  {0xDEC2,0x74fd}, // <CJK>
  {0xDEC3,0x751d}, // <CJK>
  {0xDEC4,0x7579}, // <CJK>
  {0xDEC5,0x7577}, // <CJK>
  {0xDEC6,0x6983}, // <CJK>
  {0xDEC7,0x75ef}, // <CJK>
  {0xDEC8,0x760f}, // <CJK>
  {0xDEC9,0x7603}, // <CJK>
  {0xDECA,0x75f7}, // <CJK>
  {0xDECB,0x75fe}, // <CJK>
  {0xDECC,0x75fc}, // <CJK>
  {0xDECD,0x75f9}, // <CJK>
  {0xDECE,0x75f8}, // <CJK>
  {0xDECF,0x7610}, // <CJK>
  {0xDED0,0x75fb}, // <CJK>
  {0xDED1,0x75f6}, // <CJK>
  {0xDED2,0x75ed}, // <CJK>
  {0xDED3,0x75f5}, // <CJK>
  {0xDED4,0x75fd}, // <CJK>
  {0xDED5,0x7699}, // <CJK>
  {0xDED6,0x76b5}, // <CJK>
  {0xDED7,0x76dd}, // <CJK>
  {0xDED8,0x7755}, // <CJK>
  {0xDED9,0x775f}, // <CJK>
  {0xDEDA,0x7760}, // <CJK>
  {0xDEDB,0x7752}, // <CJK>
  {0xDEDC,0x7756}, // <CJK>
  {0xDEDD,0x775a}, // <CJK>
  {0xDEDE,0x7769}, // <CJK>
  {0xDEDF,0x7767}, // <CJK>
  {0xDEE0,0x7754}, // <CJK>
  {0xDEE1,0x7759}, // <CJK>
  {0xDEE2,0x776d}, // <CJK>
  {0xDEE3,0x77e0}, // <CJK>
  {0xDEE4,0x7887}, // <CJK>
  {0xDEE5,0x789a}, // <CJK>
  {0xDEE6,0x7894}, // <CJK>
  {0xDEE7,0x788f}, // <CJK>
  {0xDEE8,0x7884}, // <CJK>
  {0xDEE9,0x7895}, // <CJK>
  {0xDEEA,0x7885}, // <CJK>
  {0xDEEB,0x7886}, // <CJK>
  {0xDEEC,0x78a1}, // <CJK>
  {0xDEED,0x7883}, // <CJK>
  {0xDEEE,0x7879}, // <CJK>
  {0xDEEF,0x7899}, // <CJK>
  {0xDEF0,0x7880}, // <CJK>
  {0xDEF1,0x7896}, // <CJK>
  {0xDEF2,0x787b}, // <CJK>
  {0xDEF3,0x797c}, // <CJK>
  {0xDEF4,0x7982}, // <CJK>
  {0xDEF5,0x797d}, // <CJK>
  {0xDEF6,0x7979}, // <CJK>
  {0xDEF7,0x7a11}, // <CJK>
  {0xDEF8,0x7a18}, // <CJK>
  {0xDEF9,0x7a19}, // <CJK>
  {0xDEFA,0x7a12}, // <CJK>
  {0xDEFB,0x7a17}, // <CJK>
  {0xDEFC,0x7a15}, // <CJK>
  {0xDEFD,0x7a22}, // <CJK>
  {0xDEFE,0x7a13}, // <CJK>
  {0xDF40,0x7a1b}, // <CJK>
  {0xDF41,0x7a10}, // <CJK>
  {0xDF42,0x7aa3}, // <CJK>
  {0xDF43,0x7aa2}, // <CJK>
  {0xDF44,0x7a9e}, // <CJK>
  {0xDF45,0x7aeb}, // <CJK>
  {0xDF46,0x7b66}, // <CJK>
  {0xDF47,0x7b64}, // <CJK>
  {0xDF48,0x7b6d}, // <CJK>
  {0xDF49,0x7b74}, // <CJK>
  {0xDF4A,0x7b69}, // <CJK>
  {0xDF4B,0x7b72}, // <CJK>
  {0xDF4C,0x7b65}, // <CJK>
  {0xDF4D,0x7b73}, // <CJK>
  {0xDF4E,0x7b71}, // <CJK>
  {0xDF4F,0x7b70}, // <CJK>
  {0xDF50,0x7b61}, // <CJK>
  {0xDF51,0x7b78}, // <CJK>
  {0xDF52,0x7b76}, // <CJK>
  {0xDF53,0x7b63}, // <CJK>
  {0xDF54,0x7cb2}, // <CJK>
  {0xDF55,0x7cb4}, // <CJK>
  {0xDF56,0x7caf}, // <CJK>
  {0xDF57,0x7d88}, // <CJK>
  {0xDF58,0x7d86}, // <CJK>
  {0xDF59,0x7d80}, // <CJK>
  {0xDF5A,0x7d8d}, // <CJK>
  {0xDF5B,0x7d7f}, // <CJK>
  {0xDF5C,0x7d85}, // <CJK>
  {0xDF5D,0x7d7a}, // <CJK>
  {0xDF5E,0x7d8e}, // <CJK>
  {0xDF5F,0x7d7b}, // <CJK>
  {0xDF60,0x7d83}, // <CJK>
  {0xDF61,0x7d7c}, // <CJK>
  {0xDF62,0x7d8c}, // <CJK>
  {0xDF63,0x7d94}, // <CJK>
  {0xDF64,0x7d84}, // <CJK>
  {0xDF65,0x7d7d}, // <CJK>
  {0xDF66,0x7d92}, // <CJK>
  {0xDF67,0x7f6d}, // <CJK>
  {0xDF68,0x7f6b}, // <CJK>
  {0xDF69,0x7f67}, // <CJK>
  {0xDF6A,0x7f68}, // <CJK>
  {0xDF6B,0x7f6c}, // <CJK>
  {0xDF6C,0x7fa6}, // <CJK>
  {0xDF6D,0x7fa5}, // <CJK>
  {0xDF6E,0x7fa7}, // <CJK>
  {0xDF6F,0x7fdb}, // <CJK>
  {0xDF70,0x7fdc}, // <CJK>
  {0xDF71,0x8021}, // <CJK>
  {0xDF72,0x8164}, // <CJK>
  {0xDF73,0x8160}, // <CJK>
  {0xDF74,0x8177}, // <CJK>
  {0xDF75,0x815c}, // <CJK>
  {0xDF76,0x8169}, // <CJK>
  {0xDF77,0x815b}, // <CJK>
  {0xDF78,0x8162}, // <CJK>
  {0xDF79,0x8172}, // <CJK>
  {0xDF7A,0x6721}, // <CJK>
  {0xDF7B,0x815e}, // <CJK>
  {0xDF7C,0x8176}, // <CJK>
  {0xDF7D,0x8167}, // <CJK>
  {0xDF7E,0x816f}, // <CJK>
  {0xDFA1,0x8144}, // <CJK>
  {0xDFA2,0x8161}, // <CJK>
  {0xDFA3,0x821d}, // <CJK>
  {0xDFA4,0x8249}, // <CJK>
  {0xDFA5,0x8244}, // <CJK>
  {0xDFA6,0x8240}, // <CJK>
  {0xDFA7,0x8242}, // <CJK>
  {0xDFA8,0x8245}, // <CJK>
  {0xDFA9,0x84f1}, // <CJK>
  {0xDFAA,0x843f}, // <CJK>
  {0xDFAB,0x8456}, // <CJK>
  {0xDFAC,0x8476}, // <CJK>
  {0xDFAD,0x8479}, // <CJK>
  {0xDFAE,0x848f}, // <CJK>
  {0xDFAF,0x848d}, // <CJK>
  {0xDFB0,0x8465}, // <CJK>
  {0xDFB1,0x8451}, // <CJK>
  {0xDFB2,0x8440}, // <CJK>
  {0xDFB3,0x8486}, // <CJK>
  {0xDFB4,0x8467}, // <CJK>
  {0xDFB5,0x8430}, // <CJK>
  {0xDFB6,0x844d}, // <CJK>
  {0xDFB7,0x847d}, // <CJK>
  {0xDFB8,0x845a}, // <CJK>
  {0xDFB9,0x8459}, // <CJK>
  {0xDFBA,0x8474}, // <CJK>
  {0xDFBB,0x8473}, // <CJK>
  {0xDFBC,0x845d}, // <CJK>
  {0xDFBD,0x8507}, // <CJK>
  {0xDFBE,0x845e}, // <CJK>
  {0xDFBF,0x8437}, // <CJK>
  {0xDFC0,0x843a}, // <CJK>
  {0xDFC1,0x8434}, // <CJK>
  {0xDFC2,0x847a}, // <CJK>
  {0xDFC3,0x8443}, // <CJK>
  {0xDFC4,0x8478}, // <CJK>
  {0xDFC5,0x8432}, // <CJK>
  {0xDFC6,0x8445}, // <CJK>
  {0xDFC7,0x8429}, // <CJK>
  {0xDFC8,0x83d9}, // <CJK>
  {0xDFC9,0x844b}, // <CJK>
  {0xDFCA,0x842f}, // <CJK>
  {0xDFCB,0x8442}, // <CJK>
  {0xDFCC,0x842d}, // <CJK>
  {0xDFCD,0x845f}, // <CJK>
  {0xDFCE,0x8470}, // <CJK>
  {0xDFCF,0x8439}, // <CJK>
  {0xDFD0,0x844e}, // <CJK>
  {0xDFD1,0x844c}, // <CJK>
  {0xDFD2,0x8452}, // <CJK>
  {0xDFD3,0x846f}, // <CJK>
  {0xDFD4,0x84c5}, // <CJK>
  {0xDFD5,0x848e}, // <CJK>
  {0xDFD6,0x843b}, // <CJK>
  {0xDFD7,0x8447}, // <CJK>
  {0xDFD8,0x8436}, // <CJK>
  {0xDFD9,0x8433}, // <CJK>
  {0xDFDA,0x8468}, // <CJK>
  {0xDFDB,0x847e}, // <CJK>
  {0xDFDC,0x8444}, // <CJK>
  {0xDFDD,0x842b}, // <CJK>
  {0xDFDE,0x8460}, // <CJK>
  {0xDFDF,0x8454}, // <CJK>
  {0xDFE0,0x846e}, // <CJK>
  {0xDFE1,0x8450}, // <CJK>
  {0xDFE2,0x870b}, // <CJK>
  {0xDFE3,0x8704}, // <CJK>
  {0xDFE4,0x86f7}, // <CJK>
  {0xDFE5,0x870c}, // <CJK>
  {0xDFE6,0x86fa}, // <CJK>
  {0xDFE7,0x86d6}, // <CJK>
  {0xDFE8,0x86f5}, // <CJK>
  {0xDFE9,0x874d}, // <CJK>
  {0xDFEA,0x86f8}, // <CJK>
  {0xDFEB,0x870e}, // <CJK>
  {0xDFEC,0x8709}, // <CJK>
  {0xDFED,0x8701}, // <CJK>
  {0xDFEE,0x86f6}, // <CJK>
  {0xDFEF,0x870d}, // <CJK>
  {0xDFF0,0x8705}, // <CJK>
  {0xDFF1,0x88d6}, // <CJK>
  {0xDFF2,0x88cb}, // <CJK>
  {0xDFF3,0x88cd}, // <CJK>
  {0xDFF4,0x88ce}, // <CJK>
  {0xDFF5,0x88de}, // <CJK>
  {0xDFF6,0x88db}, // <CJK>
  {0xDFF7,0x88da}, // <CJK>
  {0xDFF8,0x88cc}, // <CJK>
  {0xDFF9,0x88d0}, // <CJK>
  {0xDFFA,0x8985}, // <CJK>
  {0xDFFB,0x899b}, // <CJK>
  {0xDFFC,0x89df}, // <CJK>
  {0xDFFD,0x89e5}, // <CJK>
  {0xDFFE,0x89e4}, // <CJK>
  {0xE040,0x89e1}, // <CJK>
  {0xE041,0x89e0}, // <CJK>
  {0xE042,0x89e2}, // <CJK>
  {0xE043,0x89dc}, // <CJK>
  {0xE044,0x89e6}, // <CJK>
  {0xE045,0x8a76}, // <CJK>
  {0xE046,0x8a86}, // <CJK>
  {0xE047,0x8a7f}, // <CJK>
  {0xE048,0x8a61}, // <CJK>
  {0xE049,0x8a3f}, // <CJK>
  {0xE04A,0x8a77}, // <CJK>
  {0xE04B,0x8a82}, // <CJK>
  {0xE04C,0x8a84}, // <CJK>
  {0xE04D,0x8a75}, // <CJK>
  {0xE04E,0x8a83}, // <CJK>
  {0xE04F,0x8a81}, // <CJK>
  {0xE050,0x8a74}, // <CJK>
  {0xE051,0x8a7a}, // <CJK>
  {0xE052,0x8c3c}, // <CJK>
  {0xE053,0x8c4b}, // <CJK>
  {0xE054,0x8c4a}, // <CJK>
  {0xE055,0x8c65}, // <CJK>
  {0xE056,0x8c64}, // <CJK>
  {0xE057,0x8c66}, // <CJK>
  {0xE058,0x8c86}, // <CJK>
  {0xE059,0x8c84}, // <CJK>
  {0xE05A,0x8c85}, // <CJK>
  {0xE05B,0x8ccc}, // <CJK>
  {0xE05C,0x8d68}, // <CJK>
  {0xE05D,0x8d69}, // <CJK>
  {0xE05E,0x8d91}, // <CJK>
  {0xE05F,0x8d8c}, // <CJK>
  {0xE060,0x8d8e}, // <CJK>
  {0xE061,0x8d8f}, // <CJK>
  {0xE062,0x8d8d}, // <CJK>
  {0xE063,0x8d93}, // <CJK>
  {0xE064,0x8d94}, // <CJK>
  {0xE065,0x8d90}, // <CJK>
  {0xE066,0x8d92}, // <CJK>
  {0xE067,0x8df0}, // <CJK>
  {0xE068,0x8de0}, // <CJK>
  {0xE069,0x8dec}, // <CJK>
  {0xE06A,0x8df1}, // <CJK>
  {0xE06B,0x8dee}, // <CJK>
  {0xE06C,0x8dd0}, // <CJK>
  {0xE06D,0x8de9}, // <CJK>
  {0xE06E,0x8de3}, // <CJK>
  {0xE06F,0x8de2}, // <CJK>
  {0xE070,0x8de7}, // <CJK>
  {0xE071,0x8df2}, // <CJK>
  {0xE072,0x8deb}, // <CJK>
  {0xE073,0x8df4}, // <CJK>
  {0xE074,0x8f06}, // <CJK>
  {0xE075,0x8eff}, // <CJK>
  {0xE076,0x8f01}, // <CJK>
  {0xE077,0x8f00}, // <CJK>
  {0xE078,0x8f05}, // <CJK>
  {0xE079,0x8f07}, // <CJK>
  {0xE07A,0x8f08}, // <CJK>
  {0xE07B,0x8f02}, // <CJK>
  {0xE07C,0x8f0b}, // <CJK>
  {0xE07D,0x9052}, // <CJK>
  {0xE07E,0x903f}, // <CJK>
  {0xE0A1,0x9044}, // <CJK>
  {0xE0A2,0x9049}, // <CJK>
  {0xE0A3,0x903d}, // <CJK>
  {0xE0A4,0x9110}, // <CJK>
  {0xE0A5,0x910d}, // <CJK>
  {0xE0A6,0x910f}, // <CJK>
  {0xE0A7,0x9111}, // <CJK>
  {0xE0A8,0x9116}, // <CJK>
  {0xE0A9,0x9114}, // <CJK>
  {0xE0AA,0x910b}, // <CJK>
  {0xE0AB,0x910e}, // <CJK>
  {0xE0AC,0x916e}, // <CJK>
  {0xE0AD,0x916f}, // <CJK>
  {0xE0AE,0x9248}, // <CJK>
  {0xE0AF,0x9252}, // <CJK>
  {0xE0B0,0x9230}, // <CJK>
  {0xE0B1,0x923a}, // <CJK>
  {0xE0B2,0x9266}, // <CJK>
  {0xE0B3,0x9233}, // <CJK>
  {0xE0B4,0x9265}, // <CJK>
  {0xE0B5,0x925e}, // <CJK>
  {0xE0B6,0x9283}, // <CJK>
  {0xE0B7,0x922e}, // <CJK>
  {0xE0B8,0x924a}, // <CJK>
  {0xE0B9,0x9246}, // <CJK>
  {0xE0BA,0x926d}, // <CJK>
  {0xE0BB,0x926c}, // <CJK>
  {0xE0BC,0x924f}, // <CJK>
  {0xE0BD,0x9260}, // <CJK>
  {0xE0BE,0x9267}, // <CJK>
  {0xE0BF,0x926f}, // <CJK>
  {0xE0C0,0x9236}, // <CJK>
  {0xE0C1,0x9261}, // <CJK>
  {0xE0C2,0x9270}, // <CJK>
  {0xE0C3,0x9231}, // <CJK>
  {0xE0C4,0x9254}, // <CJK>
  {0xE0C5,0x9263}, // <CJK>
  {0xE0C6,0x9250}, // <CJK>
  {0xE0C7,0x9272}, // <CJK>
  {0xE0C8,0x924e}, // <CJK>
  {0xE0C9,0x9253}, // <CJK>
  {0xE0CA,0x924c}, // <CJK>
  {0xE0CB,0x9256}, // <CJK>
  {0xE0CC,0x9232}, // <CJK>
  {0xE0CD,0x959f}, // <CJK>
  {0xE0CE,0x959c}, // <CJK>
  {0xE0CF,0x959e}, // <CJK>
  {0xE0D0,0x959b}, // <CJK>
  {0xE0D1,0x9692}, // <CJK>
  {0xE0D2,0x9693}, // <CJK>
  {0xE0D3,0x9691}, // <CJK>
  {0xE0D4,0x9697}, // <CJK>
  {0xE0D5,0x96ce}, // <CJK>
  {0xE0D6,0x96fa}, // <CJK>
  {0xE0D7,0x96fd}, // <CJK>
  {0xE0D8,0x96f8}, // <CJK>
  {0xE0D9,0x96f5}, // <CJK>
  {0xE0DA,0x9773}, // <CJK>
  {0xE0DB,0x9777}, // <CJK>
  {0xE0DC,0x9778}, // <CJK>
  {0xE0DD,0x9772}, // <CJK>
  {0xE0DE,0x980f}, // <CJK>
  {0xE0DF,0x980d}, // <CJK>
  {0xE0E0,0x980e}, // <CJK>
  {0xE0E1,0x98ac}, // <CJK>
  {0xE0E2,0x98f6}, // <CJK>
  {0xE0E3,0x98f9}, // <CJK>
  {0xE0E4,0x99af}, // <CJK>
  {0xE0E5,0x99b2}, // <CJK>
  {0xE0E6,0x99b0}, // <CJK>
  {0xE0E7,0x99b5}, // <CJK>
  {0xE0E8,0x9aad}, // <CJK>
  {0xE0E9,0x9aab}, // <CJK>
  {0xE0EA,0x9b5b}, // <CJK>
  {0xE0EB,0x9cea}, // <CJK>
  {0xE0EC,0x9ced}, // <CJK>
  {0xE0ED,0x9ce7}, // <CJK>
  {0xE0EE,0x9e80}, // <CJK>
  {0xE0EF,0x9efd}, // <CJK>
  {0xE0F0,0x50e6}, // <CJK>
  {0xE0F1,0x50d4}, // <CJK>
  {0xE0F2,0x50d7}, // <CJK>
  {0xE0F3,0x50e8}, // <CJK>
  {0xE0F4,0x50f3}, // <CJK>
  {0xE0F5,0x50db}, // <CJK>
  {0xE0F6,0x50ea}, // <CJK>
  {0xE0F7,0x50dd}, // <CJK>
  {0xE0F8,0x50e4}, // <CJK>
  {0xE0F9,0x50d3}, // <CJK>
  {0xE0FA,0x50ec}, // <CJK>
  {0xE0FB,0x50f0}, // <CJK>
  {0xE0FC,0x50ef}, // <CJK>
  {0xE0FD,0x50e3}, // <CJK>
  {0xE0FE,0x50e0}, // <CJK>
  {0xE140,0x51d8}, // <CJK>
  {0xE141,0x5280}, // <CJK>
  {0xE142,0x5281}, // <CJK>
  {0xE143,0x52e9}, // <CJK>
  {0xE144,0x52eb}, // <CJK>
  {0xE145,0x5330}, // <CJK>
  {0xE146,0x53ac}, // <CJK>
  {0xE147,0x5627}, // <CJK>
  {0xE148,0x5615}, // <CJK>
  {0xE149,0x560c}, // <CJK>
  {0xE14A,0x5612}, // <CJK>
  {0xE14B,0x55fc}, // <CJK>
  {0xE14C,0x560f}, // <CJK>
  {0xE14D,0x561c}, // <CJK>
  {0xE14E,0x5601}, // <CJK>
  {0xE14F,0x5613}, // <CJK>
  {0xE150,0x5602}, // <CJK>
  {0xE151,0x55fa}, // <CJK>
  {0xE152,0x561d}, // <CJK>
  {0xE153,0x5604}, // <CJK>
  {0xE154,0x55ff}, // <CJK>
  {0xE155,0x55f9}, // <CJK>
  {0xE156,0x5889}, // <CJK>
  {0xE157,0x587c}, // <CJK>
  {0xE158,0x5890}, // <CJK>
  {0xE159,0x5898}, // <CJK>
  {0xE15A,0x5886}, // <CJK>
  {0xE15B,0x5881}, // <CJK>
  {0xE15C,0x587f}, // <CJK>
  {0xE15D,0x5874}, // <CJK>
  {0xE15E,0x588b}, // <CJK>
  {0xE15F,0x587a}, // <CJK>
  {0xE160,0x5887}, // <CJK>
  {0xE161,0x5891}, // <CJK>
  {0xE162,0x588e}, // <CJK>
  {0xE163,0x5876}, // <CJK>
  {0xE164,0x5882}, // <CJK>
  {0xE165,0x5888}, // <CJK>
  {0xE166,0x587b}, // <CJK>
  {0xE167,0x5894}, // <CJK>
  {0xE168,0x588f}, // <CJK>
  {0xE169,0x58fe}, // <CJK>
  {0xE16A,0x596b}, // <CJK>
  {0xE16B,0x5adc}, // <CJK>
  {0xE16C,0x5aee}, // <CJK>
  {0xE16D,0x5ae5}, // <CJK>
  {0xE16E,0x5ad5}, // <CJK>
  {0xE16F,0x5aea}, // <CJK>
  {0xE170,0x5ada}, // <CJK>
  {0xE171,0x5aed}, // <CJK>
  {0xE172,0x5aeb}, // <CJK>
  {0xE173,0x5af3}, // <CJK>
  {0xE174,0x5ae2}, // <CJK>
  {0xE175,0x5ae0}, // <CJK>
  {0xE176,0x5adb}, // <CJK>
  {0xE177,0x5aec}, // <CJK>
  {0xE178,0x5ade}, // <CJK>
  {0xE179,0x5add}, // <CJK>
  {0xE17A,0x5ad9}, // <CJK>
  {0xE17B,0x5ae8}, // <CJK>
  {0xE17C,0x5adf}, // <CJK>
  {0xE17D,0x5b77}, // <CJK>
  {0xE17E,0x5be0}, // <CJK>
  {0xE1A1,0x5be3}, // <CJK>
  {0xE1A2,0x5c63}, // <CJK>
  {0xE1A3,0x5d82}, // <CJK>
  {0xE1A4,0x5d80}, // <CJK>
  {0xE1A5,0x5d7d}, // <CJK>
  {0xE1A6,0x5d86}, // <CJK>
  {0xE1A7,0x5d7a}, // <CJK>
  {0xE1A8,0x5d81}, // <CJK>
  {0xE1A9,0x5d77}, // <CJK>
  {0xE1AA,0x5d8a}, // <CJK>
  {0xE1AB,0x5d89}, // <CJK>
  {0xE1AC,0x5d88}, // <CJK>
  {0xE1AD,0x5d7e}, // <CJK>
  {0xE1AE,0x5d7c}, // <CJK>
  {0xE1AF,0x5d8d}, // <CJK>
  {0xE1B0,0x5d79}, // <CJK>
  {0xE1B1,0x5d7f}, // <CJK>
  {0xE1B2,0x5e58}, // <CJK>
  {0xE1B3,0x5e59}, // <CJK>
  {0xE1B4,0x5e53}, // <CJK>
  {0xE1B5,0x5ed8}, // <CJK>
  {0xE1B6,0x5ed1}, // <CJK>
  {0xE1B7,0x5ed7}, // <CJK>
  {0xE1B8,0x5ece}, // <CJK>
  {0xE1B9,0x5edc}, // <CJK>
  {0xE1BA,0x5ed5}, // <CJK>
  {0xE1BB,0x5ed9}, // <CJK>
  {0xE1BC,0x5ed2}, // <CJK>
  {0xE1BD,0x5ed4}, // <CJK>
  {0xE1BE,0x5f44}, // <CJK>
  {0xE1BF,0x5f43}, // <CJK>
  {0xE1C0,0x5f6f}, // <CJK>
  {0xE1C1,0x5fb6}, // <CJK>
  {0xE1C2,0x612c}, // <CJK>
  {0xE1C3,0x6128}, // <CJK>
  {0xE1C4,0x6141}, // <CJK>
  {0xE1C5,0x615e}, // <CJK>
  {0xE1C6,0x6171}, // <CJK>
  {0xE1C7,0x6173}, // <CJK>
  {0xE1C8,0x6152}, // <CJK>
  {0xE1C9,0x6153}, // <CJK>
  {0xE1CA,0x6172}, // <CJK>
  {0xE1CB,0x616c}, // <CJK>
  {0xE1CC,0x6180}, // <CJK>
  {0xE1CD,0x6174}, // <CJK>
  {0xE1CE,0x6154}, // <CJK>
  {0xE1CF,0x617a}, // <CJK>
  {0xE1D0,0x615b}, // <CJK>
  {0xE1D1,0x6165}, // <CJK>
  {0xE1D2,0x613b}, // <CJK>
  {0xE1D3,0x616a}, // <CJK>
  {0xE1D4,0x6161}, // <CJK>
  {0xE1D5,0x6156}, // <CJK>
  {0xE1D6,0x6229}, // <CJK>
  {0xE1D7,0x6227}, // <CJK>
  {0xE1D8,0x622b}, // <CJK>
  {0xE1D9,0x642b}, // <CJK>
  {0xE1DA,0x644d}, // <CJK>
  {0xE1DB,0x645b}, // <CJK>
  {0xE1DC,0x645d}, // <CJK>
  {0xE1DD,0x6474}, // <CJK>
  {0xE1DE,0x6476}, // <CJK>
  {0xE1DF,0x6472}, // <CJK>
  {0xE1E0,0x6473}, // <CJK>
  {0xE1E1,0x647d}, // <CJK>
  {0xE1E2,0x6475}, // <CJK>
  {0xE1E3,0x6466}, // <CJK>
  {0xE1E4,0x64a6}, // <CJK>
  {0xE1E5,0x644e}, // <CJK>
  {0xE1E6,0x6482}, // <CJK>
  {0xE1E7,0x645e}, // <CJK>
  {0xE1E8,0x645c}, // <CJK>
  {0xE1E9,0x644b}, // <CJK>
  {0xE1EA,0x6453}, // <CJK>
  {0xE1EB,0x6460}, // <CJK>
  {0xE1EC,0x6450}, // <CJK>
  {0xE1ED,0x647f}, // <CJK>
  {0xE1EE,0x643f}, // <CJK>
  {0xE1EF,0x646c}, // <CJK>
  {0xE1F0,0x646b}, // <CJK>
  {0xE1F1,0x6459}, // <CJK>
  {0xE1F2,0x6465}, // <CJK>
  {0xE1F3,0x6477}, // <CJK>
  {0xE1F4,0x6573}, // <CJK>
  {0xE1F5,0x65a0}, // <CJK>
  {0xE1F6,0x66a1}, // <CJK>
  {0xE1F7,0x66a0}, // <CJK>
  {0xE1F8,0x669f}, // <CJK>
  {0xE1F9,0x6705}, // <CJK>
  {0xE1FA,0x6704}, // <CJK>
  {0xE1FB,0x6722}, // <CJK>
  {0xE1FC,0x69b1}, // <CJK>
  {0xE1FD,0x69b6}, // <CJK>
  {0xE1FE,0x69c9}, // <CJK>
  {0xE240,0x69a0}, // <CJK>
  {0xE241,0x69ce}, // <CJK>
  {0xE242,0x6996}, // <CJK>
  {0xE243,0x69b0}, // <CJK>
  {0xE244,0x69ac}, // <CJK>
  {0xE245,0x69bc}, // <CJK>
  {0xE246,0x6991}, // <CJK>
  {0xE247,0x6999}, // <CJK>
  {0xE248,0x698e}, // <CJK>
  {0xE249,0x69a7}, // <CJK>
  {0xE24A,0x698d}, // <CJK>
  {0xE24B,0x69a9}, // <CJK>
  {0xE24C,0x69be}, // <CJK>
  {0xE24D,0x69af}, // <CJK>
  {0xE24E,0x69bf}, // <CJK>
  {0xE24F,0x69c4}, // <CJK>
  {0xE250,0x69bd}, // <CJK>
  {0xE251,0x69a4}, // <CJK>
  {0xE252,0x69d4}, // <CJK>
  {0xE253,0x69b9}, // <CJK>
  {0xE254,0x69ca}, // <CJK>
  {0xE255,0x699a}, // <CJK>
  {0xE256,0x69cf}, // <CJK>
  {0xE257,0x69b3}, // <CJK>
  {0xE258,0x6993}, // <CJK>
  {0xE259,0x69aa}, // <CJK>
  {0xE25A,0x69a1}, // <CJK>
  {0xE25B,0x699e}, // <CJK>
  {0xE25C,0x69d9}, // <CJK>
  {0xE25D,0x6997}, // <CJK>
  {0xE25E,0x6990}, // <CJK>
  {0xE25F,0x69c2}, // <CJK>
  {0xE260,0x69b5}, // <CJK>
  {0xE261,0x69a5}, // <CJK>
  {0xE262,0x69c6}, // <CJK>
  {0xE263,0x6b4a}, // <CJK>
  {0xE264,0x6b4d}, // <CJK>
  {0xE265,0x6b4b}, // <CJK>
  {0xE266,0x6b9e}, // <CJK>
  {0xE267,0x6b9f}, // <CJK>
  {0xE268,0x6ba0}, // <CJK>
  {0xE269,0x6bc3}, // <CJK>
  {0xE26A,0x6bc4}, // <CJK>
  {0xE26B,0x6bfe}, // <CJK>
  {0xE26C,0x6ece}, // <CJK>
  {0xE26D,0x6ef5}, // <CJK>
  {0xE26E,0x6ef1}, // <CJK>
  {0xE26F,0x6f03}, // <CJK>
  {0xE270,0x6f25}, // <CJK>
  {0xE271,0x6ef8}, // <CJK>
  {0xE272,0x6f37}, // <CJK>
  {0xE273,0x6efb}, // <CJK>
  {0xE274,0x6f2e}, // <CJK>
  {0xE275,0x6f09}, // <CJK>
  {0xE276,0x6f4e}, // <CJK>
  {0xE277,0x6f19}, // <CJK>
  {0xE278,0x6f1a}, // <CJK>
  {0xE279,0x6f27}, // <CJK>
  {0xE27A,0x6f18}, // <CJK>
  {0xE27B,0x6f3b}, // <CJK>
  {0xE27C,0x6f12}, // <CJK>
  {0xE27D,0x6eed}, // <CJK>
  {0xE27E,0x6f0a}, // <CJK>
  {0xE2A1,0x6f36}, // <CJK>
  {0xE2A2,0x6f73}, // <CJK>
  {0xE2A3,0x6ef9}, // <CJK>
  {0xE2A4,0x6eee}, // <CJK>
  {0xE2A5,0x6f2d}, // <CJK>
  {0xE2A6,0x6f40}, // <CJK>
  {0xE2A7,0x6f30}, // <CJK>
  {0xE2A8,0x6f3c}, // <CJK>
  {0xE2A9,0x6f35}, // <CJK>
  {0xE2AA,0x6eeb}, // <CJK>
  {0xE2AB,0x6f07}, // <CJK>
  {0xE2AC,0x6f0e}, // <CJK>
  {0xE2AD,0x6f43}, // <CJK>
  {0xE2AE,0x6f05}, // <CJK>
  {0xE2AF,0x6efd}, // <CJK>
  {0xE2B0,0x6ef6}, // <CJK>
  {0xE2B1,0x6f39}, // <CJK>
  {0xE2B2,0x6f1c}, // <CJK>
  {0xE2B3,0x6efc}, // <CJK>
  {0xE2B4,0x6f3a}, // <CJK>
  {0xE2B5,0x6f1f}, // <CJK>
  {0xE2B6,0x6f0d}, // <CJK>
  {0xE2B7,0x6f1e}, // <CJK>
  {0xE2B8,0x6f08}, // <CJK>
  {0xE2B9,0x6f21}, // <CJK>
  {0xE2BA,0x7187}, // <CJK>
  {0xE2BB,0x7190}, // <CJK>
  {0xE2BC,0x7189}, // <CJK>
  {0xE2BD,0x7180}, // <CJK>
  {0xE2BE,0x7185}, // <CJK>
  {0xE2BF,0x7182}, // <CJK>
  {0xE2C0,0x718f}, // <CJK>
  {0xE2C1,0x717b}, // <CJK>
  {0xE2C2,0x7186}, // <CJK>
  {0xE2C3,0x7181}, // <CJK>
  {0xE2C4,0x7197}, // <CJK>
  {0xE2C5,0x7244}, // <CJK>
  {0xE2C6,0x7253}, // <CJK>
  {0xE2C7,0x7297}, // <CJK>
  {0xE2C8,0x7295}, // <CJK>
  {0xE2C9,0x7293}, // <CJK>
  {0xE2CA,0x7343}, // <CJK>
  {0xE2CB,0x734d}, // <CJK>
  {0xE2CC,0x7351}, // <CJK>
  {0xE2CD,0x734c}, // <CJK>
  {0xE2CE,0x7462}, // <CJK>
  {0xE2CF,0x7473}, // <CJK>
  {0xE2D0,0x7471}, // <CJK>
  {0xE2D1,0x7475}, // <CJK>
  {0xE2D2,0x7472}, // <CJK>
  {0xE2D3,0x7467}, // <CJK>
  {0xE2D4,0x746e}, // <CJK>
  {0xE2D5,0x7500}, // <CJK>
  {0xE2D6,0x7502}, // <CJK>
  {0xE2D7,0x7503}, // <CJK>
  {0xE2D8,0x757d}, // <CJK>
  {0xE2D9,0x7590}, // <CJK>
  {0xE2DA,0x7616}, // <CJK>
  {0xE2DB,0x7608}, // <CJK>
  {0xE2DC,0x760c}, // <CJK>
  {0xE2DD,0x7615}, // <CJK>
  {0xE2DE,0x7611}, // <CJK>
  {0xE2DF,0x760a}, // <CJK>
  {0xE2E0,0x7614}, // <CJK>
  {0xE2E1,0x76b8}, // <CJK>
  {0xE2E2,0x7781}, // <CJK>
  {0xE2E3,0x777c}, // <CJK>
  {0xE2E4,0x7785}, // <CJK>
  {0xE2E5,0x7782}, // <CJK>
  {0xE2E6,0x776e}, // <CJK>
  {0xE2E7,0x7780}, // <CJK>
  {0xE2E8,0x776f}, // <CJK>
  {0xE2E9,0x777e}, // <CJK>
  {0xE2EA,0x7783}, // <CJK>
  {0xE2EB,0x78b2}, // <CJK>
  {0xE2EC,0x78aa}, // <CJK>
  {0xE2ED,0x78b4}, // <CJK>
  {0xE2EE,0x78ad}, // <CJK>
  {0xE2EF,0x78a8}, // <CJK>
  {0xE2F0,0x787e}, // <CJK>
  {0xE2F1,0x78ab}, // <CJK>
  {0xE2F2,0x789e}, // <CJK>
  {0xE2F3,0x78a5}, // <CJK>
  {0xE2F4,0x78a0}, // <CJK>
  {0xE2F5,0x78ac}, // <CJK>
  {0xE2F6,0x78a2}, // <CJK>
  {0xE2F7,0x78a4}, // <CJK>
  {0xE2F8,0x7998}, // <CJK>
  {0xE2F9,0x798a}, // <CJK>
  {0xE2FA,0x798b}, // <CJK>
  {0xE2FB,0x7996}, // <CJK>
  {0xE2FC,0x7995}, // <CJK>
  {0xE2FD,0x7994}, // <CJK>
  {0xE2FE,0x7993}, // <CJK>
  {0xE340,0x7997}, // <CJK>
  {0xE341,0x7988}, // <CJK>
  {0xE342,0x7992}, // <CJK>
  {0xE343,0x7990}, // <CJK>
  {0xE344,0x7a2b}, // <CJK>
  {0xE345,0x7a4a}, // <CJK>
  {0xE346,0x7a30}, // <CJK>
  {0xE347,0x7a2f}, // <CJK>
  {0xE348,0x7a28}, // <CJK>
  {0xE349,0x7a26}, // <CJK>
  {0xE34A,0x7aa8}, // <CJK>
  {0xE34B,0x7aab}, // <CJK>
  {0xE34C,0x7aac}, // <CJK>
  {0xE34D,0x7aee}, // <CJK>
  {0xE34E,0x7b88}, // <CJK>
  {0xE34F,0x7b9c}, // <CJK>
  {0xE350,0x7b8a}, // <CJK>
  {0xE351,0x7b91}, // <CJK>
  {0xE352,0x7b90}, // <CJK>
  {0xE353,0x7b96}, // <CJK>
  {0xE354,0x7b8d}, // <CJK>
  {0xE355,0x7b8c}, // <CJK>
  {0xE356,0x7b9b}, // <CJK>
  {0xE357,0x7b8e}, // <CJK>
  {0xE358,0x7b85}, // <CJK>
  {0xE359,0x7b98}, // <CJK>
  {0xE35A,0x5284}, // <CJK>
  {0xE35B,0x7b99}, // <CJK>
  {0xE35C,0x7ba4}, // <CJK>
  {0xE35D,0x7b82}, // <CJK>
  {0xE35E,0x7cbb}, // <CJK>
  {0xE35F,0x7cbf}, // <CJK>
  {0xE360,0x7cbc}, // <CJK>
  {0xE361,0x7cba}, // <CJK>
  {0xE362,0x7da7}, // <CJK>
  {0xE363,0x7db7}, // <CJK>
  {0xE364,0x7dc2}, // <CJK>
  {0xE365,0x7da3}, // <CJK>
  {0xE366,0x7daa}, // <CJK>
  {0xE367,0x7dc1}, // <CJK>
  {0xE368,0x7dc0}, // <CJK>
  {0xE369,0x7dc5}, // <CJK>
  {0xE36A,0x7d9d}, // <CJK>
  {0xE36B,0x7dce}, // <CJK>
  {0xE36C,0x7dc4}, // <CJK>
  {0xE36D,0x7dc6}, // <CJK>
  {0xE36E,0x7dcb}, // <CJK>
  {0xE36F,0x7dcc}, // <CJK>
  {0xE370,0x7daf}, // <CJK>
  {0xE371,0x7db9}, // <CJK>
  {0xE372,0x7d96}, // <CJK>
  {0xE373,0x7dbc}, // <CJK>
  {0xE374,0x7d9f}, // <CJK>
  {0xE375,0x7da6}, // <CJK>
  {0xE376,0x7dae}, // <CJK>
  {0xE377,0x7da9}, // <CJK>
  {0xE378,0x7da1}, // <CJK>
  {0xE379,0x7dc9}, // <CJK>
  {0xE37A,0x7f73}, // <CJK>
  {0xE37B,0x7fe2}, // <CJK>
  {0xE37C,0x7fe3}, // <CJK>
  {0xE37D,0x7fe5}, // <CJK>
  {0xE37E,0x7fde}, // <CJK>
  {0xE3A1,0x8024}, // <CJK>
  {0xE3A2,0x805d}, // <CJK>
  {0xE3A3,0x805c}, // <CJK>
  {0xE3A4,0x8189}, // <CJK>
  {0xE3A5,0x8186}, // <CJK>
  {0xE3A6,0x8183}, // <CJK>
  {0xE3A7,0x8187}, // <CJK>
  {0xE3A8,0x818d}, // <CJK>
  {0xE3A9,0x818c}, // <CJK>
  {0xE3AA,0x818b}, // <CJK>
  {0xE3AB,0x8215}, // <CJK>
  {0xE3AC,0x8497}, // <CJK>
  {0xE3AD,0x84a4}, // <CJK>
  {0xE3AE,0x84a1}, // <CJK>
  {0xE3AF,0x849f}, // <CJK>
  {0xE3B0,0x84ba}, // <CJK>
  {0xE3B1,0x84ce}, // <CJK>
  {0xE3B2,0x84c2}, // <CJK>
  {0xE3B3,0x84ac}, // <CJK>
  {0xE3B4,0x84ae}, // <CJK>
  {0xE3B5,0x84ab}, // <CJK>
  {0xE3B6,0x84b9}, // <CJK>
  {0xE3B7,0x84b4}, // <CJK>
  {0xE3B8,0x84c1}, // <CJK>
  {0xE3B9,0x84cd}, // <CJK>
  {0xE3BA,0x84aa}, // <CJK>
  {0xE3BB,0x849a}, // <CJK>
  {0xE3BC,0x84b1}, // <CJK>
  {0xE3BD,0x84d0}, // <CJK>
  {0xE3BE,0x849d}, // <CJK>
  {0xE3BF,0x84a7}, // <CJK>
  {0xE3C0,0x84bb}, // <CJK>
  {0xE3C1,0x84a2}, // <CJK>
  {0xE3C2,0x8494}, // <CJK>
  {0xE3C3,0x84c7}, // <CJK>
  {0xE3C4,0x84cc}, // <CJK>
  {0xE3C5,0x849b}, // <CJK>
  {0xE3C6,0x84a9}, // <CJK>
  {0xE3C7,0x84af}, // <CJK>
  {0xE3C8,0x84a8}, // <CJK>
  {0xE3C9,0x84d6}, // <CJK>
  {0xE3CA,0x8498}, // <CJK>
  {0xE3CB,0x84b6}, // <CJK>
  {0xE3CC,0x84cf}, // <CJK>
  {0xE3CD,0x84a0}, // <CJK>
  {0xE3CE,0x84d7}, // <CJK>
  {0xE3CF,0x84d4}, // <CJK>
  {0xE3D0,0x84d2}, // <CJK>
  {0xE3D1,0x84db}, // <CJK>
  {0xE3D2,0x84b0}, // <CJK>
  {0xE3D3,0x8491}, // <CJK>
  {0xE3D4,0x8661}, // <CJK>
  {0xE3D5,0x8733}, // <CJK>
  {0xE3D6,0x8723}, // <CJK>
  {0xE3D7,0x8728}, // <CJK>
  {0xE3D8,0x876b}, // <CJK>
  {0xE3D9,0x8740}, // <CJK>
  {0xE3DA,0x872e}, // <CJK>
  {0xE3DB,0x871e}, // <CJK>
  {0xE3DC,0x8721}, // <CJK>
  {0xE3DD,0x8719}, // <CJK>
  {0xE3DE,0x871b}, // <CJK>
  {0xE3DF,0x8743}, // <CJK>
  {0xE3E0,0x872c}, // <CJK>
  {0xE3E1,0x8741}, // <CJK>
  {0xE3E2,0x873e}, // <CJK>
  {0xE3E3,0x8746}, // <CJK>
  {0xE3E4,0x8720}, // <CJK>
  {0xE3E5,0x8732}, // <CJK>
  {0xE3E6,0x872a}, // <CJK>
  {0xE3E7,0x872d}, // <CJK>
  {0xE3E8,0x873c}, // <CJK>
  {0xE3E9,0x8712}, // <CJK>
  {0xE3EA,0x873a}, // <CJK>
  {0xE3EB,0x8731}, // <CJK>
  {0xE3EC,0x8735}, // <CJK>
  {0xE3ED,0x8742}, // <CJK>
  {0xE3EE,0x8726}, // <CJK>
  {0xE3EF,0x8727}, // <CJK>
  {0xE3F0,0x8738}, // <CJK>
  {0xE3F1,0x8724}, // <CJK>
  {0xE3F2,0x871a}, // <CJK>
  {0xE3F3,0x8730}, // <CJK>
  {0xE3F4,0x8711}, // <CJK>
  {0xE3F5,0x88f7}, // <CJK>
  {0xE3F6,0x88e7}, // <CJK>
  {0xE3F7,0x88f1}, // <CJK>
  {0xE3F8,0x88f2}, // <CJK>
  {0xE3F9,0x88fa}, // <CJK>
  {0xE3FA,0x88fe}, // <CJK>
  {0xE3FB,0x88ee}, // <CJK>
  {0xE3FC,0x88fc}, // <CJK>
  {0xE3FD,0x88f6}, // <CJK>
  {0xE3FE,0x88fb}, // <CJK>
  {0xE440,0x88f0}, // <CJK>
  {0xE441,0x88ec}, // <CJK>
  {0xE442,0x88eb}, // <CJK>
  {0xE443,0x899d}, // <CJK>
  {0xE444,0x89a1}, // <CJK>
  {0xE445,0x899f}, // <CJK>
  {0xE446,0x899e}, // <CJK>
  {0xE447,0x89e9}, // <CJK>
  {0xE448,0x89eb}, // <CJK>
  {0xE449,0x89e8}, // <CJK>
  {0xE44A,0x8aab}, // <CJK>
  {0xE44B,0x8a99}, // <CJK>
  {0xE44C,0x8a8b}, // <CJK>
  {0xE44D,0x8a92}, // <CJK>
  {0xE44E,0x8a8f}, // <CJK>
  {0xE44F,0x8a96}, // <CJK>
  {0xE450,0x8c3d}, // <CJK>
  {0xE451,0x8c68}, // <CJK>
  {0xE452,0x8c69}, // <CJK>
  {0xE453,0x8cd5}, // <CJK>
  {0xE454,0x8ccf}, // <CJK>
  {0xE455,0x8cd7}, // <CJK>
  {0xE456,0x8d96}, // <CJK>
  {0xE457,0x8e09}, // <CJK>
  {0xE458,0x8e02}, // <CJK>
  {0xE459,0x8dff}, // <CJK>
  {0xE45A,0x8e0d}, // <CJK>
  {0xE45B,0x8dfd}, // <CJK>
  {0xE45C,0x8e0a}, // <CJK>
  {0xE45D,0x8e03}, // <CJK>
  {0xE45E,0x8e07}, // <CJK>
  {0xE45F,0x8e06}, // <CJK>
  {0xE460,0x8e05}, // <CJK>
  {0xE461,0x8dfe}, // <CJK>
  {0xE462,0x8e00}, // <CJK>
  {0xE463,0x8e04}, // <CJK>
  {0xE464,0x8f10}, // <CJK>
  {0xE465,0x8f11}, // <CJK>
  {0xE466,0x8f0e}, // <CJK>
  {0xE467,0x8f0d}, // <CJK>
  {0xE468,0x9123}, // <CJK>
  {0xE469,0x911c}, // <CJK>
  {0xE46A,0x9120}, // <CJK>
  {0xE46B,0x9122}, // <CJK>
  {0xE46C,0x911f}, // <CJK>
  {0xE46D,0x911d}, // <CJK>
  {0xE46E,0x911a}, // <CJK>
  {0xE46F,0x9124}, // <CJK>
  {0xE470,0x9121}, // <CJK>
  {0xE471,0x911b}, // <CJK>
  {0xE472,0x917a}, // <CJK>
  {0xE473,0x9172}, // <CJK>
  {0xE474,0x9179}, // <CJK>
  {0xE475,0x9173}, // <CJK>
  {0xE476,0x92a5}, // <CJK>
  {0xE477,0x92a4}, // <CJK>
  {0xE478,0x9276}, // <CJK>
  {0xE479,0x929b}, // <CJK>
  {0xE47A,0x927a}, // <CJK>
  {0xE47B,0x92a0}, // <CJK>
  {0xE47C,0x9294}, // <CJK>
  {0xE47D,0x92aa}, // <CJK>
  {0xE47E,0x928d}, // <CJK>
  {0xE4A1,0x92a6}, // <CJK>
  {0xE4A2,0x929a}, // <CJK>
  {0xE4A3,0x92ab}, // <CJK>
  {0xE4A4,0x9279}, // <CJK>
  {0xE4A5,0x9297}, // <CJK>
  {0xE4A6,0x927f}, // <CJK>
  {0xE4A7,0x92a3}, // <CJK>
  {0xE4A8,0x92ee}, // <CJK>
  {0xE4A9,0x928e}, // <CJK>
  {0xE4AA,0x9282}, // <CJK>
  {0xE4AB,0x9295}, // <CJK>
  {0xE4AC,0x92a2}, // <CJK>
  {0xE4AD,0x927d}, // <CJK>
  {0xE4AE,0x9288}, // <CJK>
  {0xE4AF,0x92a1}, // <CJK>
  {0xE4B0,0x928a}, // <CJK>
  {0xE4B1,0x9286}, // <CJK>
  {0xE4B2,0x928c}, // <CJK>
  {0xE4B3,0x9299}, // <CJK>
  {0xE4B4,0x92a7}, // <CJK>
  {0xE4B5,0x927e}, // <CJK>
  {0xE4B6,0x9287}, // <CJK>
  {0xE4B7,0x92a9}, // <CJK>
  {0xE4B8,0x929d}, // <CJK>
  {0xE4B9,0x928b}, // <CJK>
  {0xE4BA,0x922d}, // <CJK>
  {0xE4BB,0x969e}, // <CJK>
  {0xE4BC,0x96a1}, // <CJK>
  {0xE4BD,0x96ff}, // <CJK>
  {0xE4BE,0x9758}, // <CJK>
  {0xE4BF,0x977d}, // <CJK>
  {0xE4C0,0x977a}, // <CJK>
  {0xE4C1,0x977e}, // <CJK>
  {0xE4C2,0x9783}, // <CJK>
  {0xE4C3,0x9780}, // <CJK>
  {0xE4C4,0x9782}, // <CJK>
  {0xE4C5,0x977b}, // <CJK>
  {0xE4C6,0x9784}, // <CJK>
  {0xE4C7,0x9781}, // <CJK>
  {0xE4C8,0x977f}, // <CJK>
  {0xE4C9,0x97ce}, // <CJK>
  {0xE4CA,0x97cd}, // <CJK>
  {0xE4CB,0x9816}, // <CJK>
  {0xE4CC,0x98ad}, // <CJK>
  {0xE4CD,0x98ae}, // <CJK>
  {0xE4CE,0x9902}, // <CJK>
  {0xE4CF,0x9900}, // <CJK>
  {0xE4D0,0x9907}, // <CJK>
  {0xE4D1,0x999d}, // <CJK>
  {0xE4D2,0x999c}, // <CJK>
  {0xE4D3,0x99c3}, // <CJK>
  {0xE4D4,0x99b9}, // <CJK>
  {0xE4D5,0x99bb}, // <CJK>
  {0xE4D6,0x99ba}, // <CJK>
  {0xE4D7,0x99c2}, // <CJK>
  {0xE4D8,0x99bd}, // <CJK>
  {0xE4D9,0x99c7}, // <CJK>
  {0xE4DA,0x9ab1}, // <CJK>
  {0xE4DB,0x9ae3}, // <CJK>
  {0xE4DC,0x9ae7}, // <CJK>
  {0xE4DD,0x9b3e}, // <CJK>
  {0xE4DE,0x9b3f}, // <CJK>
  {0xE4DF,0x9b60}, // <CJK>
  {0xE4E0,0x9b61}, // <CJK>
  {0xE4E1,0x9b5f}, // <CJK>
  {0xE4E2,0x9cf1}, // <CJK>
  {0xE4E3,0x9cf2}, // <CJK>
  {0xE4E4,0x9cf5}, // <CJK>
  {0xE4E5,0x9ea7}, // <CJK>
  {0xE4E6,0x50ff}, // <CJK>
  {0xE4E7,0x5103}, // <CJK>
  {0xE4E8,0x5130}, // <CJK>
  {0xE4E9,0x50f8}, // <CJK>
  {0xE4EA,0x5106}, // <CJK>
  {0xE4EB,0x5107}, // <CJK>
  {0xE4EC,0x50f6}, // <CJK>
  {0xE4ED,0x50fe}, // <CJK>
  {0xE4EE,0x510b}, // <CJK>
  {0xE4EF,0x510c}, // <CJK>
  {0xE4F0,0x50fd}, // <CJK>
  {0xE4F1,0x510a}, // <CJK>
  {0xE4F2,0x528b}, // <CJK>
  {0xE4F3,0x528c}, // <CJK>
  {0xE4F4,0x52f1}, // <CJK>
  {0xE4F5,0x52ef}, // <CJK>
  {0xE4F6,0x5648}, // <CJK>
  {0xE4F7,0x5642}, // <CJK>
  {0xE4F8,0x564c}, // <CJK>
  {0xE4F9,0x5635}, // <CJK>
  {0xE4FA,0x5641}, // <CJK>
  {0xE4FB,0x564a}, // <CJK>
  {0xE4FC,0x5649}, // <CJK>
  {0xE4FD,0x5646}, // <CJK>
  {0xE4FE,0x5658}, // <CJK>
  {0xE540,0x565a}, // <CJK>
  {0xE541,0x5640}, // <CJK>
  {0xE542,0x5633}, // <CJK>
  {0xE543,0x563d}, // <CJK>
  {0xE544,0x562c}, // <CJK>
  {0xE545,0x563e}, // <CJK>
  {0xE546,0x5638}, // <CJK>
  {0xE547,0x562a}, // <CJK>
  {0xE548,0x563a}, // <CJK>
  {0xE549,0x571a}, // <CJK>
  {0xE54A,0x58ab}, // <CJK>
  {0xE54B,0x589d}, // <CJK>
  {0xE54C,0x58b1}, // <CJK>
  {0xE54D,0x58a0}, // <CJK>
  {0xE54E,0x58a3}, // <CJK>
  {0xE54F,0x58af}, // <CJK>
  {0xE550,0x58ac}, // <CJK>
  {0xE551,0x58a5}, // <CJK>
  {0xE552,0x58a1}, // <CJK>
  {0xE553,0x58ff}, // <CJK>
  {0xE554,0x5aff}, // <CJK>
  {0xE555,0x5af4}, // <CJK>
  {0xE556,0x5afd}, // <CJK>
  {0xE557,0x5af7}, // <CJK>
  {0xE558,0x5af6}, // <CJK>
  {0xE559,0x5b03}, // <CJK>
  {0xE55A,0x5af8}, // <CJK>
  {0xE55B,0x5b02}, // <CJK>
  {0xE55C,0x5af9}, // <CJK>
  {0xE55D,0x5b01}, // <CJK>
  {0xE55E,0x5b07}, // <CJK>
  {0xE55F,0x5b05}, // <CJK>
  {0xE560,0x5b0f}, // <CJK>
  {0xE561,0x5c67}, // <CJK>
  {0xE562,0x5d99}, // <CJK>
  {0xE563,0x5d97}, // <CJK>
  {0xE564,0x5d9f}, // <CJK>
  {0xE565,0x5d92}, // <CJK>
  {0xE566,0x5da2}, // <CJK>
  {0xE567,0x5d93}, // <CJK>
  {0xE568,0x5d95}, // <CJK>
  {0xE569,0x5da0}, // <CJK>
  {0xE56A,0x5d9c}, // <CJK>
  {0xE56B,0x5da1}, // <CJK>
  {0xE56C,0x5d9a}, // <CJK>
  {0xE56D,0x5d9e}, // <CJK>
  {0xE56E,0x5e69}, // <CJK>
  {0xE56F,0x5e5d}, // <CJK>
  {0xE570,0x5e60}, // <CJK>
  {0xE571,0x5e5c}, // <CJK>
  {0xE572,0x7df3}, // <CJK>
  {0xE573,0x5edb}, // <CJK>
  {0xE574,0x5ede}, // <CJK>
  {0xE575,0x5ee1}, // <CJK>
  {0xE576,0x5f49}, // <CJK>
  {0xE577,0x5fb2}, // <CJK>
  {0xE578,0x618b}, // <CJK>
  {0xE579,0x6183}, // <CJK>
  {0xE57A,0x6179}, // <CJK>
  {0xE57B,0x61b1}, // <CJK>
  {0xE57C,0x61b0}, // <CJK>
  {0xE57D,0x61a2}, // <CJK>
  {0xE57E,0x6189}, // <CJK>
  {0xE5A1,0x619b}, // <CJK>
  {0xE5A2,0x6193}, // <CJK>
  {0xE5A3,0x61af}, // <CJK>
  {0xE5A4,0x61ad}, // <CJK>
  {0xE5A5,0x619f}, // <CJK>
  {0xE5A6,0x6192}, // <CJK>
  {0xE5A7,0x61aa}, // <CJK>
  {0xE5A8,0x61a1}, // <CJK>
  {0xE5A9,0x618d}, // <CJK>
  {0xE5AA,0x6166}, // <CJK>
  {0xE5AB,0x61b3}, // <CJK>
  {0xE5AC,0x622d}, // <CJK>
  {0xE5AD,0x646e}, // <CJK>
  {0xE5AE,0x6470}, // <CJK>
  {0xE5AF,0x6496}, // <CJK>
  {0xE5B0,0x64a0}, // <CJK>
  {0xE5B1,0x6485}, // <CJK>
  {0xE5B2,0x6497}, // <CJK>
  {0xE5B3,0x649c}, // <CJK>
  {0xE5B4,0x648f}, // <CJK>
  {0xE5B5,0x648b}, // <CJK>
  {0xE5B6,0x648a}, // <CJK>
  {0xE5B7,0x648c}, // <CJK>
  {0xE5B8,0x64a3}, // <CJK>
  {0xE5B9,0x649f}, // <CJK>
  {0xE5BA,0x6468}, // <CJK>
  {0xE5BB,0x64b1}, // <CJK>
  {0xE5BC,0x6498}, // <CJK>
  {0xE5BD,0x6576}, // <CJK>
  {0xE5BE,0x657a}, // <CJK>
  {0xE5BF,0x6579}, // <CJK>
  {0xE5C0,0x657b}, // <CJK>
  {0xE5C1,0x65b2}, // <CJK>
  {0xE5C2,0x65b3}, // <CJK>
  {0xE5C3,0x66b5}, // <CJK>
  {0xE5C4,0x66b0}, // <CJK>
  {0xE5C5,0x66a9}, // <CJK>
  {0xE5C6,0x66b2}, // <CJK>
  {0xE5C7,0x66b7}, // <CJK>
  {0xE5C8,0x66aa}, // <CJK>
  {0xE5C9,0x66af}, // <CJK>
  {0xE5CA,0x6a00}, // <CJK>
  {0xE5CB,0x6a06}, // <CJK>
  {0xE5CC,0x6a17}, // <CJK>
  {0xE5CD,0x69e5}, // <CJK>
  {0xE5CE,0x69f8}, // <CJK>
  {0xE5CF,0x6a15}, // <CJK>
  {0xE5D0,0x69f1}, // <CJK>
  {0xE5D1,0x69e4}, // <CJK>
  {0xE5D2,0x6a20}, // <CJK>
  {0xE5D3,0x69ff}, // <CJK>
  {0xE5D4,0x69ec}, // <CJK>
  {0xE5D5,0x69e2}, // <CJK>
  {0xE5D6,0x6a1b}, // <CJK>
  {0xE5D7,0x6a1d}, // <CJK>
  {0xE5D8,0x69fe}, // <CJK>
  {0xE5D9,0x6a27}, // <CJK>
  {0xE5DA,0x69f2}, // <CJK>
  {0xE5DB,0x69ee}, // <CJK>
  {0xE5DC,0x6a14}, // <CJK>
  {0xE5DD,0x69f7}, // <CJK>
  {0xE5DE,0x69e7}, // <CJK>
  {0xE5DF,0x6a40}, // <CJK>
  {0xE5E0,0x6a08}, // <CJK>
  {0xE5E1,0x69e6}, // <CJK>
  {0xE5E2,0x69fb}, // <CJK>
  {0xE5E3,0x6a0d}, // <CJK>
  {0xE5E4,0x69fc}, // <CJK>
  {0xE5E5,0x69eb}, // <CJK>
  {0xE5E6,0x6a09}, // <CJK>
  {0xE5E7,0x6a04}, // <CJK>
  {0xE5E8,0x6a18}, // <CJK>
  {0xE5E9,0x6a25}, // <CJK>
  {0xE5EA,0x6a0f}, // <CJK>
  {0xE5EB,0x69f6}, // <CJK>
  {0xE5EC,0x6a26}, // <CJK>
  {0xE5ED,0x6a07}, // <CJK>
  {0xE5EE,0x69f4}, // <CJK>
  {0xE5EF,0x6a16}, // <CJK>
  {0xE5F0,0x6b51}, // <CJK>
  {0xE5F1,0x6ba5}, // <CJK>
  {0xE5F2,0x6ba3}, // <CJK>
  {0xE5F3,0x6ba2}, // <CJK>
  {0xE5F4,0x6ba6}, // <CJK>
  {0xE5F5,0x6c01}, // <CJK>
  {0xE5F6,0x6c00}, // <CJK>
  {0xE5F7,0x6bff}, // <CJK>
  {0xE5F8,0x6c02}, // <CJK>
  {0xE5F9,0x6f41}, // <CJK>
  {0xE5FA,0x6f26}, // <CJK>
  {0xE5FB,0x6f7e}, // <CJK>
  {0xE5FC,0x6f87}, // <CJK>
  {0xE5FD,0x6fc6}, // <CJK>
  {0xE5FE,0x6f92}, // <CJK>
  {0xE640,0x6f8d}, // <CJK>
  {0xE641,0x6f89}, // <CJK>
  {0xE642,0x6f8c}, // <CJK>
  {0xE643,0x6f62}, // <CJK>
  {0xE644,0x6f4f}, // <CJK>
  {0xE645,0x6f85}, // <CJK>
  {0xE646,0x6f5a}, // <CJK>
  {0xE647,0x6f96}, // <CJK>
  {0xE648,0x6f76}, // <CJK>
  {0xE649,0x6f6c}, // <CJK>
  {0xE64A,0x6f82}, // <CJK>
  {0xE64B,0x6f55}, // <CJK>
  {0xE64C,0x6f72}, // <CJK>
  {0xE64D,0x6f52}, // <CJK>
  {0xE64E,0x6f50}, // <CJK>
  {0xE64F,0x6f57}, // <CJK>
  {0xE650,0x6f94}, // <CJK>
  {0xE651,0x6f93}, // <CJK>
  {0xE652,0x6f5d}, // <CJK>
  {0xE653,0x6f00}, // <CJK>
  {0xE654,0x6f61}, // <CJK>
  {0xE655,0x6f6b}, // <CJK>
  {0xE656,0x6f7d}, // <CJK>
  {0xE657,0x6f67}, // <CJK>
  {0xE658,0x6f90}, // <CJK>
  {0xE659,0x6f53}, // <CJK>
  {0xE65A,0x6f8b}, // <CJK>
  {0xE65B,0x6f69}, // <CJK>
  {0xE65C,0x6f7f}, // <CJK>
  {0xE65D,0x6f95}, // <CJK>
  {0xE65E,0x6f63}, // <CJK>
  {0xE65F,0x6f77}, // <CJK>
  {0xE660,0x6f6a}, // <CJK>
  {0xE661,0x6f7b}, // <CJK>
  {0xE662,0x71b2}, // <CJK>
  {0xE663,0x71af}, // <CJK>
  {0xE664,0x719b}, // <CJK>
  {0xE665,0x71b0}, // <CJK>
  {0xE666,0x71a0}, // <CJK>
  {0xE667,0x719a}, // <CJK>
  {0xE668,0x71a9}, // <CJK>
  {0xE669,0x71b5}, // <CJK>
  {0xE66A,0x719d}, // <CJK>
  {0xE66B,0x71a5}, // <CJK>
  {0xE66C,0x719e}, // <CJK>
  {0xE66D,0x71a4}, // <CJK>
  {0xE66E,0x71a1}, // <CJK>
  {0xE66F,0x71aa}, // <CJK>
  {0xE670,0x719c}, // <CJK>
  {0xE671,0x71a7}, // <CJK>
  {0xE672,0x71b3}, // <CJK>
  {0xE673,0x7298}, // <CJK>
  {0xE674,0x729a}, // <CJK>
  {0xE675,0x7358}, // <CJK>
  {0xE676,0x7352}, // <CJK>
  {0xE677,0x735e}, // <CJK>
  {0xE678,0x735f}, // <CJK>
  {0xE679,0x7360}, // <CJK>
  {0xE67A,0x735d}, // <CJK>
  {0xE67B,0x735b}, // <CJK>
  {0xE67C,0x7361}, // <CJK>
  {0xE67D,0x735a}, // <CJK>
  {0xE67E,0x7359}, // <CJK>
  {0xE6A1,0x7362}, // <CJK>
  {0xE6A2,0x7487}, // <CJK>
  {0xE6A3,0x7489}, // <CJK>
  {0xE6A4,0x748a}, // <CJK>
  {0xE6A5,0x7486}, // <CJK>
  {0xE6A6,0x7481}, // <CJK>
  {0xE6A7,0x747d}, // <CJK>
  {0xE6A8,0x7485}, // <CJK>
  {0xE6A9,0x7488}, // <CJK>
  {0xE6AA,0x747c}, // <CJK>
  {0xE6AB,0x7479}, // <CJK>
  {0xE6AC,0x7508}, // <CJK>
  {0xE6AD,0x7507}, // <CJK>
  {0xE6AE,0x757e}, // <CJK>
  {0xE6AF,0x7625}, // <CJK>
  {0xE6B0,0x761e}, // <CJK>
  {0xE6B1,0x7619}, // <CJK>
  {0xE6B2,0x761d}, // <CJK>
  {0xE6B3,0x761c}, // <CJK>
  {0xE6B4,0x7623}, // <CJK>
  {0xE6B5,0x761a}, // <CJK>
  {0xE6B6,0x7628}, // <CJK>
  {0xE6B7,0x761b}, // <CJK>
  {0xE6B8,0x769c}, // <CJK>
  {0xE6B9,0x769d}, // <CJK>
  {0xE6BA,0x769e}, // <CJK>
  {0xE6BB,0x769b}, // <CJK>
  {0xE6BC,0x778d}, // <CJK>
  {0xE6BD,0x778f}, // <CJK>
  {0xE6BE,0x7789}, // <CJK>
  {0xE6BF,0x7788}, // <CJK>
  {0xE6C0,0x78cd}, // <CJK>
  {0xE6C1,0x78bb}, // <CJK>
  {0xE6C2,0x78cf}, // <CJK>
  {0xE6C3,0x78cc}, // <CJK>
  {0xE6C4,0x78d1}, // <CJK>
  {0xE6C5,0x78ce}, // <CJK>
  {0xE6C6,0x78d4}, // <CJK>
  {0xE6C7,0x78c8}, // <CJK>
  {0xE6C8,0x78c3}, // <CJK>
  {0xE6C9,0x78c4}, // <CJK>
  {0xE6CA,0x78c9}, // <CJK>
  {0xE6CB,0x799a}, // <CJK>
  {0xE6CC,0x79a1}, // <CJK>
  {0xE6CD,0x79a0}, // <CJK>
  {0xE6CE,0x799c}, // <CJK>
  {0xE6CF,0x79a2}, // <CJK>
  {0xE6D0,0x799b}, // <CJK>
  {0xE6D1,0x6b76}, // <CJK>
  {0xE6D2,0x7a39}, // <CJK>
  {0xE6D3,0x7ab2}, // <CJK>
  {0xE6D4,0x7ab4}, // <CJK>
  {0xE6D5,0x7ab3}, // <CJK>
  {0xE6D6,0x7bb7}, // <CJK>
  {0xE6D7,0x7bcb}, // <CJK>
  {0xE6D8,0x7bbe}, // <CJK>
  {0xE6D9,0x7bac}, // <CJK>
  {0xE6DA,0x7bce}, // <CJK>
  {0xE6DB,0x7baf}, // <CJK>
  {0xE6DC,0x7bb9}, // <CJK>
  {0xE6DD,0x7bca}, // <CJK>
  {0xE6DE,0x7bb5}, // <CJK>
  {0xE6DF,0x7cc5}, // <CJK>
  {0xE6E0,0x7cc8}, // <CJK>
  {0xE6E1,0x7ccc}, // <CJK>
  {0xE6E2,0x7ccb}, // <CJK>
  {0xE6E3,0x7df7}, // <CJK>
  {0xE6E4,0x7ddb}, // <CJK>
  {0xE6E5,0x7dea}, // <CJK>
  {0xE6E6,0x7de7}, // <CJK>
  {0xE6E7,0x7dd7}, // <CJK>
  {0xE6E8,0x7de1}, // <CJK>
  {0xE6E9,0x7e03}, // <CJK>
  {0xE6EA,0x7dfa}, // <CJK>
  {0xE6EB,0x7de6}, // <CJK>
  {0xE6EC,0x7df6}, // <CJK>
  {0xE6ED,0x7df1}, // <CJK>
  {0xE6EE,0x7df0}, // <CJK>
  {0xE6EF,0x7dee}, // <CJK>
  {0xE6F0,0x7ddf}, // <CJK>
  {0xE6F1,0x7f76}, // <CJK>
  {0xE6F2,0x7fac}, // <CJK>
  {0xE6F3,0x7fb0}, // <CJK>
  {0xE6F4,0x7fad}, // <CJK>
  {0xE6F5,0x7fed}, // <CJK>
  {0xE6F6,0x7feb}, // <CJK>
  {0xE6F7,0x7fea}, // <CJK>
  {0xE6F8,0x7fec}, // <CJK>
  {0xE6F9,0x7fe6}, // <CJK>
  {0xE6FA,0x7fe8}, // <CJK>
  {0xE6FB,0x8064}, // <CJK>
  {0xE6FC,0x8067}, // <CJK>
  {0xE6FD,0x81a3}, // <CJK>
  {0xE6FE,0x819f}, // <CJK>
  {0xE740,0x819e}, // <CJK>
  {0xE741,0x8195}, // <CJK>
  {0xE742,0x81a2}, // <CJK>
  {0xE743,0x8199}, // <CJK>
  {0xE744,0x8197}, // <CJK>
  {0xE745,0x8216}, // <CJK>
  {0xE746,0x824f}, // <CJK>
  {0xE747,0x8253}, // <CJK>
  {0xE748,0x8252}, // <CJK>
  {0xE749,0x8250}, // <CJK>
  {0xE74A,0x824e}, // <CJK>
  {0xE74B,0x8251}, // <CJK>
  {0xE74C,0x8524}, // <CJK>
  {0xE74D,0x853b}, // <CJK>
  {0xE74E,0x850f}, // <CJK>
  {0xE74F,0x8500}, // <CJK>
  {0xE750,0x8529}, // <CJK>
  {0xE751,0x850e}, // <CJK>
  {0xE752,0x8509}, // <CJK>
  {0xE753,0x850d}, // <CJK>
  {0xE754,0x851f}, // <CJK>
  {0xE755,0x850a}, // <CJK>
  {0xE756,0x8527}, // <CJK>
  {0xE757,0x851c}, // <CJK>
  {0xE758,0x84fb}, // <CJK>
  {0xE759,0x852b}, // <CJK>
  {0xE75A,0x84fa}, // <CJK>
  {0xE75B,0x8508}, // <CJK>
  {0xE75C,0x850c}, // <CJK>
  {0xE75D,0x84f4}, // <CJK>
  {0xE75E,0x852a}, // <CJK>
  {0xE75F,0x84f2}, // <CJK>
  {0xE760,0x8515}, // <CJK>
  {0xE761,0x84f7}, // <CJK>
  {0xE762,0x84eb}, // <CJK>
  {0xE763,0x84f3}, // <CJK>
  {0xE764,0x84fc}, // <CJK>
  {0xE765,0x8512}, // <CJK>
  {0xE766,0x84ea}, // <CJK>
  {0xE767,0x84e9}, // <CJK>
  {0xE768,0x8516}, // <CJK>
  {0xE769,0x84fe}, // <CJK>
  {0xE76A,0x8528}, // <CJK>
  {0xE76B,0x851d}, // <CJK>
  {0xE76C,0x852e}, // <CJK>
  {0xE76D,0x8502}, // <CJK>
  {0xE76E,0x84fd}, // <CJK>
  {0xE76F,0x851e}, // <CJK>
  {0xE770,0x84f6}, // <CJK>
  {0xE771,0x8531}, // <CJK>
  {0xE772,0x8526}, // <CJK>
  {0xE773,0x84e7}, // <CJK>
  {0xE774,0x84e8}, // <CJK>
  {0xE775,0x84f0}, // <CJK>
  {0xE776,0x84ef}, // <CJK>
  {0xE777,0x84f9}, // <CJK>
  {0xE778,0x8518}, // <CJK>
  {0xE779,0x8520}, // <CJK>
  {0xE77A,0x8530}, // <CJK>
  {0xE77B,0x850b}, // <CJK>
  {0xE77C,0x8519}, // <CJK>
  {0xE77D,0x852f}, // <CJK>
  {0xE77E,0x8662}, // <CJK>
  {0xE7A1,0x8756}, // <CJK>
  {0xE7A2,0x8763}, // <CJK>
  {0xE7A3,0x8764}, // <CJK>
  {0xE7A4,0x8777}, // <CJK>
  {0xE7A5,0x87e1}, // <CJK>
  {0xE7A6,0x8773}, // <CJK>
  {0xE7A7,0x8758}, // <CJK>
  {0xE7A8,0x8754}, // <CJK>
  {0xE7A9,0x875b}, // <CJK>
  {0xE7AA,0x8752}, // <CJK>
  {0xE7AB,0x8761}, // <CJK>
  {0xE7AC,0x875a}, // <CJK>
  {0xE7AD,0x8751}, // <CJK>
  {0xE7AE,0x875e}, // <CJK>
  {0xE7AF,0x876d}, // <CJK>
  {0xE7B0,0x876a}, // <CJK>
  {0xE7B1,0x8750}, // <CJK>
  {0xE7B2,0x874e}, // <CJK>
  {0xE7B3,0x875f}, // <CJK>
  {0xE7B4,0x875d}, // <CJK>
  {0xE7B5,0x876f}, // <CJK>
  {0xE7B6,0x876c}, // <CJK>
  {0xE7B7,0x877a}, // <CJK>
  {0xE7B8,0x876e}, // <CJK>
  {0xE7B9,0x875c}, // <CJK>
  {0xE7BA,0x8765}, // <CJK>
  {0xE7BB,0x874f}, // <CJK>
  {0xE7BC,0x877b}, // <CJK>
  {0xE7BD,0x8775}, // <CJK>
  {0xE7BE,0x8762}, // <CJK>
  {0xE7BF,0x8767}, // <CJK>
  {0xE7C0,0x8769}, // <CJK>
  {0xE7C1,0x885a}, // <CJK>
  {0xE7C2,0x8905}, // <CJK>
  {0xE7C3,0x890c}, // <CJK>
  {0xE7C4,0x8914}, // <CJK>
  {0xE7C5,0x890b}, // <CJK>
  {0xE7C6,0x8917}, // <CJK>
  {0xE7C7,0x8918}, // <CJK>
  {0xE7C8,0x8919}, // <CJK>
  {0xE7C9,0x8906}, // <CJK>
  {0xE7CA,0x8916}, // <CJK>
  {0xE7CB,0x8911}, // <CJK>
  {0xE7CC,0x890e}, // <CJK>
  {0xE7CD,0x8909}, // <CJK>
  {0xE7CE,0x89a2}, // <CJK>
  {0xE7CF,0x89a4}, // <CJK>
  {0xE7D0,0x89a3}, // <CJK>
  {0xE7D1,0x89ed}, // <CJK>
  {0xE7D2,0x89f0}, // <CJK>
  {0xE7D3,0x89ec}, // <CJK>
  {0xE7D4,0x8acf}, // <CJK>
  {0xE7D5,0x8ac6}, // <CJK>
  {0xE7D6,0x8ab8}, // <CJK>
  {0xE7D7,0x8ad3}, // <CJK>
  {0xE7D8,0x8ad1}, // <CJK>
  {0xE7D9,0x8ad4}, // <CJK>
  {0xE7DA,0x8ad5}, // <CJK>
  {0xE7DB,0x8abb}, // <CJK>
  {0xE7DC,0x8ad7}, // <CJK>
  {0xE7DD,0x8abe}, // <CJK>
  {0xE7DE,0x8ac0}, // <CJK>
  {0xE7DF,0x8ac5}, // <CJK>
  {0xE7E0,0x8ad8}, // <CJK>
  {0xE7E1,0x8ac3}, // <CJK>
  {0xE7E2,0x8aba}, // <CJK>
  {0xE7E3,0x8abd}, // <CJK>
  {0xE7E4,0x8ad9}, // <CJK>
  {0xE7E5,0x8c3e}, // <CJK>
  {0xE7E6,0x8c4d}, // <CJK>
  {0xE7E7,0x8c8f}, // <CJK>
  {0xE7E8,0x8ce5}, // <CJK>
  {0xE7E9,0x8cdf}, // <CJK>
  {0xE7EA,0x8cd9}, // <CJK>
  {0xE7EB,0x8ce8}, // <CJK>
  {0xE7EC,0x8cda}, // <CJK>
  {0xE7ED,0x8cdd}, // <CJK>
  {0xE7EE,0x8ce7}, // <CJK>
  {0xE7EF,0x8da0}, // <CJK>
  {0xE7F0,0x8d9c}, // <CJK>
  {0xE7F1,0x8da1}, // <CJK>
  {0xE7F2,0x8d9b}, // <CJK>
  {0xE7F3,0x8e20}, // <CJK>
  {0xE7F4,0x8e23}, // <CJK>
  {0xE7F5,0x8e25}, // <CJK>
  {0xE7F6,0x8e24}, // <CJK>
  {0xE7F7,0x8e2e}, // <CJK>
  {0xE7F8,0x8e15}, // <CJK>
  {0xE7F9,0x8e1b}, // <CJK>
  {0xE7FA,0x8e16}, // <CJK>
  {0xE7FB,0x8e11}, // <CJK>
  {0xE7FC,0x8e19}, // <CJK>
  {0xE7FD,0x8e26}, // <CJK>
  {0xE7FE,0x8e27}, // <CJK>
  {0xE840,0x8e14}, // <CJK>
  {0xE841,0x8e12}, // <CJK>
  {0xE842,0x8e18}, // <CJK>
  {0xE843,0x8e13}, // <CJK>
  {0xE844,0x8e1c}, // <CJK>
  {0xE845,0x8e17}, // <CJK>
  {0xE846,0x8e1a}, // <CJK>
  {0xE847,0x8f2c}, // <CJK>
  {0xE848,0x8f24}, // <CJK>
  {0xE849,0x8f18}, // <CJK>
  {0xE84A,0x8f1a}, // <CJK>
  {0xE84B,0x8f20}, // <CJK>
  {0xE84C,0x8f23}, // <CJK>
  {0xE84D,0x8f16}, // <CJK>
  {0xE84E,0x8f17}, // <CJK>
  {0xE84F,0x9073}, // <CJK>
  {0xE850,0x9070}, // <CJK>
  {0xE851,0x906f}, // <CJK>
  {0xE852,0x9067}, // <CJK>
  {0xE853,0x906b}, // <CJK>
  {0xE854,0x912f}, // <CJK>
  {0xE855,0x912b}, // <CJK>
  {0xE856,0x9129}, // <CJK>
  {0xE857,0x912a}, // <CJK>
  {0xE858,0x9132}, // <CJK>
  {0xE859,0x9126}, // <CJK>
  {0xE85A,0x912e}, // <CJK>
  {0xE85B,0x9185}, // <CJK>
  {0xE85C,0x9186}, // <CJK>
  {0xE85D,0x918a}, // <CJK>
  {0xE85E,0x9181}, // <CJK>
  {0xE85F,0x9182}, // <CJK>
  {0xE860,0x9184}, // <CJK>
  {0xE861,0x9180}, // <CJK>
  {0xE862,0x92d0}, // <CJK>
  {0xE863,0x92c3}, // <CJK>
  {0xE864,0x92c4}, // <CJK>
  {0xE865,0x92c0}, // <CJK>
  {0xE866,0x92d9}, // <CJK>
  {0xE867,0x92b6}, // <CJK>
  {0xE868,0x92cf}, // <CJK>
  {0xE869,0x92f1}, // <CJK>
  {0xE86A,0x92df}, // <CJK>
  {0xE86B,0x92d8}, // <CJK>
  {0xE86C,0x92e9}, // <CJK>
  {0xE86D,0x92d7}, // <CJK>
  {0xE86E,0x92dd}, // <CJK>
  {0xE86F,0x92cc}, // <CJK>
  {0xE870,0x92ef}, // <CJK>
  {0xE871,0x92c2}, // <CJK>
  {0xE872,0x92e8}, // <CJK>
  {0xE873,0x92ca}, // <CJK>
  {0xE874,0x92c8}, // <CJK>
  {0xE875,0x92ce}, // <CJK>
  {0xE876,0x92e6}, // <CJK>
  {0xE877,0x92cd}, // <CJK>
  {0xE878,0x92d5}, // <CJK>
  {0xE879,0x92c9}, // <CJK>
  {0xE87A,0x92e0}, // <CJK>
  {0xE87B,0x92de}, // <CJK>
  {0xE87C,0x92e7}, // <CJK>
  {0xE87D,0x92d1}, // <CJK>
  {0xE87E,0x92d3}, // <CJK>
  {0xE8A1,0x92b5}, // <CJK>
  {0xE8A2,0x92e1}, // <CJK>
  {0xE8A3,0x92c6}, // <CJK>
  {0xE8A4,0x92b4}, // <CJK>
  {0xE8A5,0x957c}, // <CJK>
  {0xE8A6,0x95ac}, // <CJK>
  {0xE8A7,0x95ab}, // <CJK>
  {0xE8A8,0x95ae}, // <CJK>
  {0xE8A9,0x95b0}, // <CJK>
  {0xE8AA,0x96a4}, // <CJK>
  {0xE8AB,0x96a2}, // <CJK>
  {0xE8AC,0x96d3}, // <CJK>
  {0xE8AD,0x9705}, // <CJK>
  {0xE8AE,0x9708}, // <CJK>
  {0xE8AF,0x9702}, // <CJK>
  {0xE8B0,0x975a}, // <CJK>
  {0xE8B1,0x978a}, // <CJK>
  {0xE8B2,0x978e}, // <CJK>
  {0xE8B3,0x9788}, // <CJK>
  {0xE8B4,0x97d0}, // <CJK>
  {0xE8B5,0x97cf}, // <CJK>
  {0xE8B6,0x981e}, // <CJK>
  {0xE8B7,0x981d}, // <CJK>
  {0xE8B8,0x9826}, // <CJK>
  {0xE8B9,0x9829}, // <CJK>
  {0xE8BA,0x9828}, // <CJK>
  {0xE8BB,0x9820}, // <CJK>
  {0xE8BC,0x981b}, // <CJK>
  {0xE8BD,0x9827}, // <CJK>
  {0xE8BE,0x98b2}, // <CJK>
  {0xE8BF,0x9908}, // <CJK>
  {0xE8C0,0x98fa}, // <CJK>
  {0xE8C1,0x9911}, // <CJK>
  {0xE8C2,0x9914}, // <CJK>
  {0xE8C3,0x9916}, // <CJK>
  {0xE8C4,0x9917}, // <CJK>
  {0xE8C5,0x9915}, // <CJK>
  {0xE8C6,0x99dc}, // <CJK>
  {0xE8C7,0x99cd}, // <CJK>
  {0xE8C8,0x99cf}, // <CJK>
  {0xE8C9,0x99d3}, // <CJK>
  {0xE8CA,0x99d4}, // <CJK>
  {0xE8CB,0x99ce}, // <CJK>
  {0xE8CC,0x99c9}, // <CJK>
  {0xE8CD,0x99d6}, // <CJK>
  {0xE8CE,0x99d8}, // <CJK>
  {0xE8CF,0x99cb}, // <CJK>
  {0xE8D0,0x99d7}, // <CJK>
  {0xE8D1,0x99cc}, // <CJK>
  {0xE8D2,0x9ab3}, // <CJK>
  {0xE8D3,0x9aec}, // <CJK>
  {0xE8D4,0x9aeb}, // <CJK>
  {0xE8D5,0x9af3}, // <CJK>
  {0xE8D6,0x9af2}, // <CJK>
  {0xE8D7,0x9af1}, // <CJK>
  {0xE8D8,0x9b46}, // <CJK>
  {0xE8D9,0x9b43}, // <CJK>
  {0xE8DA,0x9b67}, // <CJK>
  {0xE8DB,0x9b74}, // <CJK>
  {0xE8DC,0x9b71}, // <CJK>
  {0xE8DD,0x9b66}, // <CJK>
  {0xE8DE,0x9b76}, // <CJK>
  {0xE8DF,0x9b75}, // <CJK>
  {0xE8E0,0x9b70}, // <CJK>
  {0xE8E1,0x9b68}, // <CJK>
  {0xE8E2,0x9b64}, // <CJK>
  {0xE8E3,0x9b6c}, // <CJK>
  {0xE8E4,0x9cfc}, // <CJK>
  {0xE8E5,0x9cfa}, // <CJK>
  {0xE8E6,0x9cfd}, // <CJK>
  {0xE8E7,0x9cff}, // <CJK>
  {0xE8E8,0x9cf7}, // <CJK>
  {0xE8E9,0x9d07}, // <CJK>
  {0xE8EA,0x9d00}, // <CJK>
  {0xE8EB,0x9cf9}, // <CJK>
  {0xE8EC,0x9cfb}, // <CJK>
  {0xE8ED,0x9d08}, // <CJK>
  {0xE8EE,0x9d05}, // <CJK>
  {0xE8EF,0x9d04}, // <CJK>
  {0xE8F0,0x9e83}, // <CJK>
  {0xE8F1,0x9ed3}, // <CJK>
  {0xE8F2,0x9f0f}, // <CJK>
  {0xE8F3,0x9f10}, // <CJK>
  {0xE8F4,0x511c}, // <CJK>
  {0xE8F5,0x5113}, // <CJK>
  {0xE8F6,0x5117}, // <CJK>
  {0xE8F7,0x511a}, // <CJK>
  {0xE8F8,0x5111}, // <CJK>
  {0xE8F9,0x51de}, // <CJK>
  {0xE8FA,0x5334}, // <CJK>
  {0xE8FB,0x53e1}, // <CJK>
  {0xE8FC,0x5670}, // <CJK>
  {0xE8FD,0x5660}, // <CJK>
  {0xE8FE,0x566e}, // <CJK>
  {0xE940,0x5673}, // <CJK>
  {0xE941,0x5666}, // <CJK>
  {0xE942,0x5663}, // <CJK>
  {0xE943,0x566d}, // <CJK>
  {0xE944,0x5672}, // <CJK>
  {0xE945,0x565e}, // <CJK>
  {0xE946,0x5677}, // <CJK>
  {0xE947,0x571c}, // <CJK>
  {0xE948,0x571b}, // <CJK>
  {0xE949,0x58c8}, // <CJK>
  {0xE94A,0x58bd}, // <CJK>
  {0xE94B,0x58c9}, // <CJK>
  {0xE94C,0x58bf}, // <CJK>
  {0xE94D,0x58ba}, // <CJK>
  {0xE94E,0x58c2}, // <CJK>
  {0xE94F,0x58bc}, // <CJK>
  {0xE950,0x58c6}, // <CJK>
  {0xE951,0x5b17}, // <CJK>
  {0xE952,0x5b19}, // <CJK>
  {0xE953,0x5b1b}, // <CJK>
  {0xE954,0x5b21}, // <CJK>
  {0xE955,0x5b14}, // <CJK>
  {0xE956,0x5b13}, // <CJK>
  {0xE957,0x5b10}, // <CJK>
  {0xE958,0x5b16}, // <CJK>
  {0xE959,0x5b28}, // <CJK>
  {0xE95A,0x5b1a}, // <CJK>
  {0xE95B,0x5b20}, // <CJK>
  {0xE95C,0x5b1e}, // <CJK>
  {0xE95D,0x5bef}, // <CJK>
  {0xE95E,0x5dac}, // <CJK>
  {0xE95F,0x5db1}, // <CJK>
  {0xE960,0x5da9}, // <CJK>
  {0xE961,0x5da7}, // <CJK>
  {0xE962,0x5db5}, // <CJK>
  {0xE963,0x5db0}, // <CJK>
  {0xE964,0x5dae}, // <CJK>
  {0xE965,0x5daa}, // <CJK>
  {0xE966,0x5da8}, // <CJK>
  {0xE967,0x5db2}, // <CJK>
  {0xE968,0x5dad}, // <CJK>
  {0xE969,0x5daf}, // <CJK>
  {0xE96A,0x5db4}, // <CJK>
  {0xE96B,0x5e67}, // <CJK>
  {0xE96C,0x5e68}, // <CJK>
  {0xE96D,0x5e66}, // <CJK>
  {0xE96E,0x5e6f}, // <CJK>
  {0xE96F,0x5ee9}, // <CJK>
  {0xE970,0x5ee7}, // <CJK>
  {0xE971,0x5ee6}, // <CJK>
  {0xE972,0x5ee8}, // <CJK>
  {0xE973,0x5ee5}, // <CJK>
  {0xE974,0x5f4b}, // <CJK>
  {0xE975,0x5fbc}, // <CJK>
  {0xE976,0x619d}, // <CJK>
  {0xE977,0x61a8}, // <CJK>
  {0xE978,0x6196}, // <CJK>
  {0xE979,0x61c5}, // <CJK>
  {0xE97A,0x61b4}, // <CJK>
  {0xE97B,0x61c6}, // <CJK>
  {0xE97C,0x61c1}, // <CJK>
  {0xE97D,0x61cc}, // <CJK>
  {0xE97E,0x61ba}, // <CJK>
  {0xE9A1,0x61bf}, // <CJK>
  {0xE9A2,0x61b8}, // <CJK>
  {0xE9A3,0x618c}, // <CJK>
  {0xE9A4,0x64d7}, // <CJK>
  {0xE9A5,0x64d6}, // <CJK>
  {0xE9A6,0x64d0}, // <CJK>
  {0xE9A7,0x64cf}, // <CJK>
  {0xE9A8,0x64c9}, // <CJK>
  {0xE9A9,0x64bd}, // <CJK>
  {0xE9AA,0x6489}, // <CJK>
  {0xE9AB,0x64c3}, // <CJK>
  {0xE9AC,0x64db}, // <CJK>
  {0xE9AD,0x64f3}, // <CJK>
  {0xE9AE,0x64d9}, // <CJK>
  {0xE9AF,0x6533}, // <CJK>
  {0xE9B0,0x657f}, // <CJK>
  {0xE9B1,0x657c}, // <CJK>
  {0xE9B2,0x65a2}, // <CJK>
  {0xE9B3,0x66c8}, // <CJK>
  {0xE9B4,0x66be}, // <CJK>
  {0xE9B5,0x66c0}, // <CJK>
  {0xE9B6,0x66ca}, // <CJK>
  {0xE9B7,0x66cb}, // <CJK>
  {0xE9B8,0x66cf}, // <CJK>
  {0xE9B9,0x66bd}, // <CJK>
  {0xE9BA,0x66bb}, // <CJK>
  {0xE9BB,0x66ba}, // <CJK>
  {0xE9BC,0x66cc}, // <CJK>
  {0xE9BD,0x6723}, // <CJK>
  {0xE9BE,0x6a34}, // <CJK>
  {0xE9BF,0x6a66}, // <CJK>
  {0xE9C0,0x6a49}, // <CJK>
  {0xE9C1,0x6a67}, // <CJK>
  {0xE9C2,0x6a32}, // <CJK>
  {0xE9C3,0x6a68}, // <CJK>
  {0xE9C4,0x6a3e}, // <CJK>
  {0xE9C5,0x6a5d}, // <CJK>
  {0xE9C6,0x6a6d}, // <CJK>
  {0xE9C7,0x6a76}, // <CJK>
  {0xE9C8,0x6a5b}, // <CJK>
  {0xE9C9,0x6a51}, // <CJK>
  {0xE9CA,0x6a28}, // <CJK>
  {0xE9CB,0x6a5a}, // <CJK>
  {0xE9CC,0x6a3b}, // <CJK>
  {0xE9CD,0x6a3f}, // <CJK>
  {0xE9CE,0x6a41}, // <CJK>
  {0xE9CF,0x6a6a}, // <CJK>
  {0xE9D0,0x6a64}, // <CJK>
  {0xE9D1,0x6a50}, // <CJK>
  {0xE9D2,0x6a4f}, // <CJK>
  {0xE9D3,0x6a54}, // <CJK>
  {0xE9D4,0x6a6f}, // <CJK>
  {0xE9D5,0x6a69}, // <CJK>
  {0xE9D6,0x6a60}, // <CJK>
  {0xE9D7,0x6a3c}, // <CJK>
  {0xE9D8,0x6a5e}, // <CJK>
  {0xE9D9,0x6a56}, // <CJK>
  {0xE9DA,0x6a55}, // <CJK>
  {0xE9DB,0x6a4d}, // <CJK>
  {0xE9DC,0x6a4e}, // <CJK>
  {0xE9DD,0x6a46}, // <CJK>
  {0xE9DE,0x6b55}, // <CJK>
  {0xE9DF,0x6b54}, // <CJK>
  {0xE9E0,0x6b56}, // <CJK>
  {0xE9E1,0x6ba7}, // <CJK>
  {0xE9E2,0x6baa}, // <CJK>
  {0xE9E3,0x6bab}, // <CJK>
  {0xE9E4,0x6bc8}, // <CJK>
  {0xE9E5,0x6bc7}, // <CJK>
  {0xE9E6,0x6c04}, // <CJK>
  {0xE9E7,0x6c03}, // <CJK>
  {0xE9E8,0x6c06}, // <CJK>
  {0xE9E9,0x6fad}, // <CJK>
  {0xE9EA,0x6fcb}, // <CJK>
  {0xE9EB,0x6fa3}, // <CJK>
  {0xE9EC,0x6fc7}, // <CJK>
  {0xE9ED,0x6fbc}, // <CJK>
  {0xE9EE,0x6fce}, // <CJK>
  {0xE9EF,0x6fc8}, // <CJK>
  {0xE9F0,0x6f5e}, // <CJK>
  {0xE9F1,0x6fc4}, // <CJK>
  {0xE9F2,0x6fbd}, // <CJK>
  {0xE9F3,0x6f9e}, // <CJK>
  {0xE9F4,0x6fca}, // <CJK>
  {0xE9F5,0x6fa8}, // <CJK>
  {0xE9F6,0x7004}, // <CJK>
  {0xE9F7,0x6fa5}, // <CJK>
  {0xE9F8,0x6fae}, // <CJK>
  {0xE9F9,0x6fba}, // <CJK>
  {0xE9FA,0x6fac}, // <CJK>
  {0xE9FB,0x6faa}, // <CJK>
  {0xE9FC,0x6fcf}, // <CJK>
  {0xE9FD,0x6fbf}, // <CJK>
  {0xE9FE,0x6fb8}, // <CJK>
  {0xEA40,0x6fa2}, // <CJK>
  {0xEA41,0x6fc9}, // <CJK>
  {0xEA42,0x6fab}, // <CJK>
  {0xEA43,0x6fcd}, // <CJK>
  {0xEA44,0x6faf}, // <CJK>
  {0xEA45,0x6fb2}, // <CJK>
  {0xEA46,0x6fb0}, // <CJK>
  {0xEA47,0x71c5}, // <CJK>
  {0xEA48,0x71c2}, // <CJK>
  {0xEA49,0x71bf}, // <CJK>
  {0xEA4A,0x71b8}, // <CJK>
  {0xEA4B,0x71d6}, // <CJK>
  {0xEA4C,0x71c0}, // <CJK>
  {0xEA4D,0x71c1}, // <CJK>
  {0xEA4E,0x71cb}, // <CJK>
  {0xEA4F,0x71d4}, // <CJK>
  {0xEA50,0x71ca}, // <CJK>
  {0xEA51,0x71c7}, // <CJK>
  {0xEA52,0x71cf}, // <CJK>
  {0xEA53,0x71bd}, // <CJK>
  {0xEA54,0x71d8}, // <CJK>
  {0xEA55,0x71bc}, // <CJK>
  {0xEA56,0x71c6}, // <CJK>
  {0xEA57,0x71da}, // <CJK>
  {0xEA58,0x71db}, // <CJK>
  {0xEA59,0x729d}, // <CJK>
  {0xEA5A,0x729e}, // <CJK>
  {0xEA5B,0x7369}, // <CJK>
  {0xEA5C,0x7366}, // <CJK>
  {0xEA5D,0x7367}, // <CJK>
  {0xEA5E,0x736c}, // <CJK>
  {0xEA5F,0x7365}, // <CJK>
  {0xEA60,0x736b}, // <CJK>
  {0xEA61,0x736a}, // <CJK>
  {0xEA62,0x747f}, // <CJK>
  {0xEA63,0x749a}, // <CJK>
  {0xEA64,0x74a0}, // <CJK>
  {0xEA65,0x7494}, // <CJK>
  {0xEA66,0x7492}, // <CJK>
  {0xEA67,0x7495}, // <CJK>
  {0xEA68,0x74a1}, // <CJK>
  {0xEA69,0x750b}, // <CJK>
  {0xEA6A,0x7580}, // <CJK>
  {0xEA6B,0x762f}, // <CJK>
  {0xEA6C,0x762d}, // <CJK>
  {0xEA6D,0x7631}, // <CJK>
  {0xEA6E,0x763d}, // <CJK>
  {0xEA6F,0x7633}, // <CJK>
  {0xEA70,0x763c}, // <CJK>
  {0xEA71,0x7635}, // <CJK>
  {0xEA72,0x7632}, // <CJK>
  {0xEA73,0x7630}, // <CJK>
  {0xEA74,0x76bb}, // <CJK>
  {0xEA75,0x76e6}, // <CJK>
  {0xEA76,0x779a}, // <CJK>
  {0xEA77,0x779d}, // <CJK>
  {0xEA78,0x77a1}, // <CJK>
  {0xEA79,0x779c}, // <CJK>
  {0xEA7A,0x779b}, // <CJK>
  {0xEA7B,0x77a2}, // <CJK>
  {0xEA7C,0x77a3}, // <CJK>
  {0xEA7D,0x7795}, // <CJK>
  {0xEA7E,0x7799}, // <CJK>
  {0xEAA1,0x7797}, // <CJK>
  {0xEAA2,0x78dd}, // <CJK>
  {0xEAA3,0x78e9}, // <CJK>
  {0xEAA4,0x78e5}, // <CJK>
  {0xEAA5,0x78ea}, // <CJK>
  {0xEAA6,0x78de}, // <CJK>
  {0xEAA7,0x78e3}, // <CJK>
  {0xEAA8,0x78db}, // <CJK>
  {0xEAA9,0x78e1}, // <CJK>
  {0xEAAA,0x78e2}, // <CJK>
  {0xEAAB,0x78ed}, // <CJK>
  {0xEAAC,0x78df}, // <CJK>
  {0xEAAD,0x78e0}, // <CJK>
  {0xEAAE,0x79a4}, // <CJK>
  {0xEAAF,0x7a44}, // <CJK>
  {0xEAB0,0x7a48}, // <CJK>
  {0xEAB1,0x7a47}, // <CJK>
  {0xEAB2,0x7ab6}, // <CJK>
  {0xEAB3,0x7ab8}, // <CJK>
  {0xEAB4,0x7ab5}, // <CJK>
  {0xEAB5,0x7ab1}, // <CJK>
  {0xEAB6,0x7ab7}, // <CJK>
  {0xEAB7,0x7bde}, // <CJK>
  {0xEAB8,0x7be3}, // <CJK>
  {0xEAB9,0x7be7}, // <CJK>
  {0xEABA,0x7bdd}, // <CJK>
  {0xEABB,0x7bd5}, // <CJK>
  {0xEABC,0x7be5}, // <CJK>
  {0xEABD,0x7bda}, // <CJK>
  {0xEABE,0x7be8}, // <CJK>
  {0xEABF,0x7bf9}, // <CJK>
  {0xEAC0,0x7bd4}, // <CJK>
  {0xEAC1,0x7bea}, // <CJK>
  {0xEAC2,0x7be2}, // <CJK>
  {0xEAC3,0x7bdc}, // <CJK>
  {0xEAC4,0x7beb}, // <CJK>
  {0xEAC5,0x7bd8}, // <CJK>
  {0xEAC6,0x7bdf}, // <CJK>
  {0xEAC7,0x7cd2}, // <CJK>
  {0xEAC8,0x7cd4}, // <CJK>
  {0xEAC9,0x7cd7}, // <CJK>
  {0xEACA,0x7cd0}, // <CJK>
  {0xEACB,0x7cd1}, // <CJK>
  {0xEACC,0x7e12}, // <CJK>
  {0xEACD,0x7e21}, // <CJK>
  {0xEACE,0x7e17}, // <CJK>
  {0xEACF,0x7e0c}, // <CJK>
  {0xEAD0,0x7e1f}, // <CJK>
  {0xEAD1,0x7e20}, // <CJK>
  {0xEAD2,0x7e13}, // <CJK>
  {0xEAD3,0x7e0e}, // <CJK>
  {0xEAD4,0x7e1c}, // <CJK>
  {0xEAD5,0x7e15}, // <CJK>
  {0xEAD6,0x7e1a}, // <CJK>
  {0xEAD7,0x7e22}, // <CJK>
  {0xEAD8,0x7e0b}, // <CJK>
  {0xEAD9,0x7e0f}, // <CJK>
  {0xEADA,0x7e16}, // <CJK>
  {0xEADB,0x7e0d}, // <CJK>
  {0xEADC,0x7e14}, // <CJK>
  {0xEADD,0x7e25}, // <CJK>
  {0xEADE,0x7e24}, // <CJK>
  {0xEADF,0x7f43}, // <CJK>
  {0xEAE0,0x7f7b}, // <CJK>
  {0xEAE1,0x7f7c}, // <CJK>
  {0xEAE2,0x7f7a}, // <CJK>
  {0xEAE3,0x7fb1}, // <CJK>
  {0xEAE4,0x7fef}, // <CJK>
  {0xEAE5,0x802a}, // <CJK>
  {0xEAE6,0x8029}, // <CJK>
  {0xEAE7,0x806c}, // <CJK>
  {0xEAE8,0x81b1}, // <CJK>
  {0xEAE9,0x81a6}, // <CJK>
  {0xEAEA,0x81ae}, // <CJK>
  {0xEAEB,0x81b9}, // <CJK>
  {0xEAEC,0x81b5}, // <CJK>
  {0xEAED,0x81ab}, // <CJK>
  {0xEAEE,0x81b0}, // <CJK>
  {0xEAEF,0x81ac}, // <CJK>
  {0xEAF0,0x81b4}, // <CJK>
  {0xEAF1,0x81b2}, // <CJK>
  {0xEAF2,0x81b7}, // <CJK>
  {0xEAF3,0x81a7}, // <CJK>
  {0xEAF4,0x81f2}, // <CJK>
  {0xEAF5,0x8255}, // <CJK>
  {0xEAF6,0x8256}, // <CJK>
  {0xEAF7,0x8257}, // <CJK>
  {0xEAF8,0x8556}, // <CJK>
  {0xEAF9,0x8545}, // <CJK>
  {0xEAFA,0x856b}, // <CJK>
  {0xEAFB,0x854d}, // <CJK>
  {0xEAFC,0x8553}, // <CJK>
  {0xEAFD,0x8561}, // <CJK>
  {0xEAFE,0x8558}, // <CJK>
  {0xEB40,0x8540}, // <CJK>
  {0xEB41,0x8546}, // <CJK>
  {0xEB42,0x8564}, // <CJK>
  {0xEB43,0x8541}, // <CJK>
  {0xEB44,0x8562}, // <CJK>
  {0xEB45,0x8544}, // <CJK>
  {0xEB46,0x8551}, // <CJK>
  {0xEB47,0x8547}, // <CJK>
  {0xEB48,0x8563}, // <CJK>
  {0xEB49,0x853e}, // <CJK>
  {0xEB4A,0x855b}, // <CJK>
  {0xEB4B,0x8571}, // <CJK>
  {0xEB4C,0x854e}, // <CJK>
  {0xEB4D,0x856e}, // <CJK>
  {0xEB4E,0x8575}, // <CJK>
  {0xEB4F,0x8555}, // <CJK>
  {0xEB50,0x8567}, // <CJK>
  {0xEB51,0x8560}, // <CJK>
  {0xEB52,0x858c}, // <CJK>
  {0xEB53,0x8566}, // <CJK>
  {0xEB54,0x855d}, // <CJK>
  {0xEB55,0x8554}, // <CJK>
  {0xEB56,0x8565}, // <CJK>
  {0xEB57,0x856c}, // <CJK>
  {0xEB58,0x8663}, // <CJK>
  {0xEB59,0x8665}, // <CJK>
  {0xEB5A,0x8664}, // <CJK>
  {0xEB5B,0x879b}, // <CJK>
  {0xEB5C,0x878f}, // <CJK>
  {0xEB5D,0x8797}, // <CJK>
  {0xEB5E,0x8793}, // <CJK>
  {0xEB5F,0x8792}, // <CJK>
  {0xEB60,0x8788}, // <CJK>
  {0xEB61,0x8781}, // <CJK>
  {0xEB62,0x8796}, // <CJK>
  {0xEB63,0x8798}, // <CJK>
  {0xEB64,0x8779}, // <CJK>
  {0xEB65,0x8787}, // <CJK>
  {0xEB66,0x87a3}, // <CJK>
  {0xEB67,0x8785}, // <CJK>
  {0xEB68,0x8790}, // <CJK>
  {0xEB69,0x8791}, // <CJK>
  {0xEB6A,0x879d}, // <CJK>
  {0xEB6B,0x8784}, // <CJK>
  {0xEB6C,0x8794}, // <CJK>
  {0xEB6D,0x879c}, // <CJK>
  {0xEB6E,0x879a}, // <CJK>
  {0xEB6F,0x8789}, // <CJK>
  {0xEB70,0x891e}, // <CJK>
  {0xEB71,0x8926}, // <CJK>
  {0xEB72,0x8930}, // <CJK>
  {0xEB73,0x892d}, // <CJK>
  {0xEB74,0x892e}, // <CJK>
  {0xEB75,0x8927}, // <CJK>
  {0xEB76,0x8931}, // <CJK>
  {0xEB77,0x8922}, // <CJK>
  {0xEB78,0x8929}, // <CJK>
  {0xEB79,0x8923}, // <CJK>
  {0xEB7A,0x892f}, // <CJK>
  {0xEB7B,0x892c}, // <CJK>
  {0xEB7C,0x891f}, // <CJK>
  {0xEB7D,0x89f1}, // <CJK>
  {0xEB7E,0x8ae0}, // <CJK>
  {0xEBA1,0x8ae2}, // <CJK>
  {0xEBA2,0x8af2}, // <CJK>
  {0xEBA3,0x8af4}, // <CJK>
  {0xEBA4,0x8af5}, // <CJK>
  {0xEBA5,0x8add}, // <CJK>
  {0xEBA6,0x8b14}, // <CJK>
  {0xEBA7,0x8ae4}, // <CJK>
  {0xEBA8,0x8adf}, // <CJK>
  {0xEBA9,0x8af0}, // <CJK>
  {0xEBAA,0x8ac8}, // <CJK>
  {0xEBAB,0x8ade}, // <CJK>
  {0xEBAC,0x8ae1}, // <CJK>
  {0xEBAD,0x8ae8}, // <CJK>
  {0xEBAE,0x8aff}, // <CJK>
  {0xEBAF,0x8aef}, // <CJK>
  {0xEBB0,0x8afb}, // <CJK>
  {0xEBB1,0x8c91}, // <CJK>
  {0xEBB2,0x8c92}, // <CJK>
  {0xEBB3,0x8c90}, // <CJK>
  {0xEBB4,0x8cf5}, // <CJK>
  {0xEBB5,0x8cee}, // <CJK>
  {0xEBB6,0x8cf1}, // <CJK>
  {0xEBB7,0x8cf0}, // <CJK>
  {0xEBB8,0x8cf3}, // <CJK>
  {0xEBB9,0x8d6c}, // <CJK>
  {0xEBBA,0x8d6e}, // <CJK>
  {0xEBBB,0x8da5}, // <CJK>
  {0xEBBC,0x8da7}, // <CJK>
  {0xEBBD,0x8e33}, // <CJK>
  {0xEBBE,0x8e3e}, // <CJK>
  {0xEBBF,0x8e38}, // <CJK>
  {0xEBC0,0x8e40}, // <CJK>
  {0xEBC1,0x8e45}, // <CJK>
  {0xEBC2,0x8e36}, // <CJK>
  {0xEBC3,0x8e3c}, // <CJK>
  {0xEBC4,0x8e3d}, // <CJK>
  {0xEBC5,0x8e41}, // <CJK>
  {0xEBC6,0x8e30}, // <CJK>
  {0xEBC7,0x8e3f}, // <CJK>
  {0xEBC8,0x8ebd}, // <CJK>
  {0xEBC9,0x8f36}, // <CJK>
  {0xEBCA,0x8f2e}, // <CJK>
  {0xEBCB,0x8f35}, // <CJK>
  {0xEBCC,0x8f32}, // <CJK>
  {0xEBCD,0x8f39}, // <CJK>
  {0xEBCE,0x8f37}, // <CJK>
  {0xEBCF,0x8f34}, // <CJK>
  {0xEBD0,0x9076}, // <CJK>
  {0xEBD1,0x9079}, // <CJK>
  {0xEBD2,0x907b}, // <CJK>
  {0xEBD3,0x9086}, // <CJK>
  {0xEBD4,0x90fa}, // <CJK>
  {0xEBD5,0x9133}, // <CJK>
  {0xEBD6,0x9135}, // <CJK>
  {0xEBD7,0x9136}, // <CJK>
  {0xEBD8,0x9193}, // <CJK>
  {0xEBD9,0x9190}, // <CJK>
  {0xEBDA,0x9191}, // <CJK>
  {0xEBDB,0x918d}, // <CJK>
  {0xEBDC,0x918f}, // <CJK>
  {0xEBDD,0x9327}, // <CJK>
  {0xEBDE,0x931e}, // <CJK>
  {0xEBDF,0x9308}, // <CJK>
  {0xEBE0,0x931f}, // <CJK>
  {0xEBE1,0x9306}, // <CJK>
  {0xEBE2,0x930f}, // <CJK>
  {0xEBE3,0x937a}, // <CJK>
  {0xEBE4,0x9338}, // <CJK>
  {0xEBE5,0x933c}, // <CJK>
  {0xEBE6,0x931b}, // <CJK>
  {0xEBE7,0x9323}, // <CJK>
  {0xEBE8,0x9312}, // <CJK>
  {0xEBE9,0x9301}, // <CJK>
  {0xEBEA,0x9346}, // <CJK>
  {0xEBEB,0x932d}, // <CJK>
  {0xEBEC,0x930e}, // <CJK>
  {0xEBED,0x930d}, // <CJK>
  {0xEBEE,0x92cb}, // <CJK>
  {0xEBEF,0x931d}, // <CJK>
  {0xEBF0,0x92fa}, // <CJK>
  {0xEBF1,0x9325}, // <CJK>
  {0xEBF2,0x9313}, // <CJK>
  {0xEBF3,0x92f9}, // <CJK>
  {0xEBF4,0x92f7}, // <CJK>
  {0xEBF5,0x9334}, // <CJK>
  {0xEBF6,0x9302}, // <CJK>
  {0xEBF7,0x9324}, // <CJK>
  {0xEBF8,0x92ff}, // <CJK>
  {0xEBF9,0x9329}, // <CJK>
  {0xEBFA,0x9339}, // <CJK>
  {0xEBFB,0x9335}, // <CJK>
  {0xEBFC,0x932a}, // <CJK>
  {0xEBFD,0x9314}, // <CJK>
  {0xEBFE,0x930c}, // <CJK>
  {0xEC40,0x930b}, // <CJK>
  {0xEC41,0x92fe}, // <CJK>
  {0xEC42,0x9309}, // <CJK>
  {0xEC43,0x9300}, // <CJK>
  {0xEC44,0x92fb}, // <CJK>
  {0xEC45,0x9316}, // <CJK>
  {0xEC46,0x95bc}, // <CJK>
  {0xEC47,0x95cd}, // <CJK>
  {0xEC48,0x95be}, // <CJK>
  {0xEC49,0x95b9}, // <CJK>
  {0xEC4A,0x95ba}, // <CJK>
  {0xEC4B,0x95b6}, // <CJK>
  {0xEC4C,0x95bf}, // <CJK>
  {0xEC4D,0x95b5}, // <CJK>
  {0xEC4E,0x95bd}, // <CJK>
  {0xEC4F,0x96a9}, // <CJK>
  {0xEC50,0x96d4}, // <CJK>
  {0xEC51,0x970b}, // <CJK>
  {0xEC52,0x9712}, // <CJK>
  {0xEC53,0x9710}, // <CJK>
  {0xEC54,0x9799}, // <CJK>
  {0xEC55,0x9797}, // <CJK>
  {0xEC56,0x9794}, // <CJK>
  {0xEC57,0x97f0}, // <CJK>
  {0xEC58,0x97f8}, // <CJK>
  {0xEC59,0x9835}, // <CJK>
  {0xEC5A,0x982f}, // <CJK>
  {0xEC5B,0x9832}, // <CJK>
  {0xEC5C,0x9924}, // <CJK>
  {0xEC5D,0x991f}, // <CJK>
  {0xEC5E,0x9927}, // <CJK>
  {0xEC5F,0x9929}, // <CJK>
  {0xEC60,0x999e}, // <CJK>
  {0xEC61,0x99ee}, // <CJK>
  {0xEC62,0x99ec}, // <CJK>
  {0xEC63,0x99e5}, // <CJK>
  {0xEC64,0x99e4}, // <CJK>
  {0xEC65,0x99f0}, // <CJK>
  {0xEC66,0x99e3}, // <CJK>
  {0xEC67,0x99ea}, // <CJK>
  {0xEC68,0x99e9}, // <CJK>
  {0xEC69,0x99e7}, // <CJK>
  {0xEC6A,0x9ab9}, // <CJK>
  {0xEC6B,0x9abf}, // <CJK>
  {0xEC6C,0x9ab4}, // <CJK>
  {0xEC6D,0x9abb}, // <CJK>
  {0xEC6E,0x9af6}, // <CJK>
  {0xEC6F,0x9afa}, // <CJK>
  {0xEC70,0x9af9}, // <CJK>
  {0xEC71,0x9af7}, // <CJK>
  {0xEC72,0x9b33}, // <CJK>
  {0xEC73,0x9b80}, // <CJK>
  {0xEC74,0x9b85}, // <CJK>
  {0xEC75,0x9b87}, // <CJK>
  {0xEC76,0x9b7c}, // <CJK>
  {0xEC77,0x9b7e}, // <CJK>
  {0xEC78,0x9b7b}, // <CJK>
  {0xEC79,0x9b82}, // <CJK>
  {0xEC7A,0x9b93}, // <CJK>
  {0xEC7B,0x9b92}, // <CJK>
  {0xEC7C,0x9b90}, // <CJK>
  {0xEC7D,0x9b7a}, // <CJK>
  {0xEC7E,0x9b95}, // <CJK>
  {0xECA1,0x9b7d}, // <CJK>
  {0xECA2,0x9b88}, // <CJK>
  {0xECA3,0x9d25}, // <CJK>
  {0xECA4,0x9d17}, // <CJK>
  {0xECA5,0x9d20}, // <CJK>
  {0xECA6,0x9d1e}, // <CJK>
  {0xECA7,0x9d14}, // <CJK>
  {0xECA8,0x9d29}, // <CJK>
  {0xECA9,0x9d1d}, // <CJK>
  {0xECAA,0x9d18}, // <CJK>
  {0xECAB,0x9d22}, // <CJK>
  {0xECAC,0x9d10}, // <CJK>
  {0xECAD,0x9d19}, // <CJK>
  {0xECAE,0x9d1f}, // <CJK>
  {0xECAF,0x9e88}, // <CJK>
  {0xECB0,0x9e86}, // <CJK>
  {0xECB1,0x9e87}, // <CJK>
  {0xECB2,0x9eae}, // <CJK>
  {0xECB3,0x9ead}, // <CJK>
  {0xECB4,0x9ed5}, // <CJK>
  {0xECB5,0x9ed6}, // <CJK>
  {0xECB6,0x9efa}, // <CJK>
  {0xECB7,0x9f12}, // <CJK>
  {0xECB8,0x9f3d}, // <CJK>
  {0xECB9,0x5126}, // <CJK>
  {0xECBA,0x5125}, // <CJK>
  {0xECBB,0x5122}, // <CJK>
  {0xECBC,0x5124}, // <CJK>
  {0xECBD,0x5120}, // <CJK>
  {0xECBE,0x5129}, // <CJK>
  {0xECBF,0x52f4}, // <CJK>
  {0xECC0,0x5693}, // <CJK>
  {0xECC1,0x568c}, // <CJK>
  {0xECC2,0x568d}, // <CJK>
  {0xECC3,0x5686}, // <CJK>
  {0xECC4,0x5684}, // <CJK>
  {0xECC5,0x5683}, // <CJK>
  {0xECC6,0x567e}, // <CJK>
  {0xECC7,0x5682}, // <CJK>
  {0xECC8,0x567f}, // <CJK>
  {0xECC9,0x5681}, // <CJK>
  {0xECCA,0x58d6}, // <CJK>
  {0xECCB,0x58d4}, // <CJK>
  {0xECCC,0x58cf}, // <CJK>
  {0xECCD,0x58d2}, // <CJK>
  {0xECCE,0x5b2d}, // <CJK>
  {0xECCF,0x5b25}, // <CJK>
  {0xECD0,0x5b32}, // <CJK>
  {0xECD1,0x5b23}, // <CJK>
  {0xECD2,0x5b2c}, // <CJK>
  {0xECD3,0x5b27}, // <CJK>
  {0xECD4,0x5b26}, // <CJK>
  {0xECD5,0x5b2f}, // <CJK>
  {0xECD6,0x5b2e}, // <CJK>
  {0xECD7,0x5b7b}, // <CJK>
  {0xECD8,0x5bf1}, // <CJK>
  {0xECD9,0x5bf2}, // <CJK>
  {0xECDA,0x5db7}, // <CJK>
  {0xECDB,0x5e6c}, // <CJK>
  {0xECDC,0x5e6a}, // <CJK>
  {0xECDD,0x5fbe}, // <CJK>
  {0xECDE,0x5fbb}, // <CJK>
  {0xECDF,0x61c3}, // <CJK>
  {0xECE0,0x61b5}, // <CJK>
  {0xECE1,0x61bc}, // <CJK>
  {0xECE2,0x61e7}, // <CJK>
  {0xECE3,0x61e0}, // <CJK>
  {0xECE4,0x61e5}, // <CJK>
  {0xECE5,0x61e4}, // <CJK>
  {0xECE6,0x61e8}, // <CJK>
  {0xECE7,0x61de}, // <CJK>
  {0xECE8,0x64ef}, // <CJK>
  {0xECE9,0x64e9}, // <CJK>
  {0xECEA,0x64e3}, // <CJK>
  {0xECEB,0x64eb}, // <CJK>
  {0xECEC,0x64e4}, // <CJK>
  {0xECED,0x64e8}, // <CJK>
  {0xECEE,0x6581}, // <CJK>
  {0xECEF,0x6580}, // <CJK>
  {0xECF0,0x65b6}, // <CJK>
  {0xECF1,0x65da}, // <CJK>
  {0xECF2,0x66d2}, // <CJK>
  {0xECF3,0x6a8d}, // <CJK>
  {0xECF4,0x6a96}, // <CJK>
  {0xECF5,0x6a81}, // <CJK>
  {0xECF6,0x6aa5}, // <CJK>
  {0xECF7,0x6a89}, // <CJK>
  {0xECF8,0x6a9f}, // <CJK>
  {0xECF9,0x6a9b}, // <CJK>
  {0xECFA,0x6aa1}, // <CJK>
  {0xECFB,0x6a9e}, // <CJK>
  {0xECFC,0x6a87}, // <CJK>
  {0xECFD,0x6a93}, // <CJK>
  {0xECFE,0x6a8e}, // <CJK>
  {0xED40,0x6a95}, // <CJK>
  {0xED41,0x6a83}, // <CJK>
  {0xED42,0x6aa8}, // <CJK>
  {0xED43,0x6aa4}, // <CJK>
  {0xED44,0x6a91}, // <CJK>
  {0xED45,0x6a7f}, // <CJK>
  {0xED46,0x6aa6}, // <CJK>
  {0xED47,0x6a9a}, // <CJK>
  {0xED48,0x6a85}, // <CJK>
  {0xED49,0x6a8c}, // <CJK>
  {0xED4A,0x6a92}, // <CJK>
  {0xED4B,0x6b5b}, // <CJK>
  {0xED4C,0x6bad}, // <CJK>
  {0xED4D,0x6c09}, // <CJK>
  {0xED4E,0x6fcc}, // <CJK>
  {0xED4F,0x6fa9}, // <CJK>
  {0xED50,0x6ff4}, // <CJK>
  {0xED51,0x6fd4}, // <CJK>
  {0xED52,0x6fe3}, // <CJK>
  {0xED53,0x6fdc}, // <CJK>
  {0xED54,0x6fed}, // <CJK>
  {0xED55,0x6fe7}, // <CJK>
  {0xED56,0x6fe6}, // <CJK>
  {0xED57,0x6fde}, // <CJK>
  {0xED58,0x6ff2}, // <CJK>
  {0xED59,0x6fdd}, // <CJK>
  {0xED5A,0x6fe2}, // <CJK>
  {0xED5B,0x6fe8}, // <CJK>
  {0xED5C,0x71e1}, // <CJK>
  {0xED5D,0x71f1}, // <CJK>
  {0xED5E,0x71e8}, // <CJK>
  {0xED5F,0x71f2}, // <CJK>
  {0xED60,0x71e4}, // <CJK>
  {0xED61,0x71f0}, // <CJK>
  {0xED62,0x71e2}, // <CJK>
  {0xED63,0x7373}, // <CJK>
  {0xED64,0x736e}, // <CJK>
  {0xED65,0x736f}, // <CJK>
  {0xED66,0x7497}, // <CJK>
  {0xED67,0x74b2}, // <CJK>
  {0xED68,0x74ab}, // <CJK>
  {0xED69,0x7490}, // <CJK>
  {0xED6A,0x74aa}, // <CJK>
  {0xED6B,0x74ad}, // <CJK>
  {0xED6C,0x74b1}, // <CJK>
  {0xED6D,0x74a5}, // <CJK>
  {0xED6E,0x74af}, // <CJK>
  {0xED6F,0x7510}, // <CJK>
  {0xED70,0x7511}, // <CJK>
  {0xED71,0x7512}, // <CJK>
  {0xED72,0x750f}, // <CJK>
  {0xED73,0x7584}, // <CJK>
  {0xED74,0x7643}, // <CJK>
  {0xED75,0x7648}, // <CJK>
  {0xED76,0x7649}, // <CJK>
  {0xED77,0x7647}, // <CJK>
  {0xED78,0x76a4}, // <CJK>
  {0xED79,0x76e9}, // <CJK>
  {0xED7A,0x77b5}, // <CJK>
  {0xED7B,0x77ab}, // <CJK>
  {0xED7C,0x77b2}, // <CJK>
  {0xED7D,0x77b7}, // <CJK>
  {0xED7E,0x77b6}, // <CJK>
  {0xEDA1,0x77b4}, // <CJK>
  {0xEDA2,0x77b1}, // <CJK>
  {0xEDA3,0x77a8}, // <CJK>
  {0xEDA4,0x77f0}, // <CJK>
  {0xEDA5,0x78f3}, // <CJK>
  {0xEDA6,0x78fd}, // <CJK>
  {0xEDA7,0x7902}, // <CJK>
  {0xEDA8,0x78fb}, // <CJK>
  {0xEDA9,0x78fc}, // <CJK>
  {0xEDAA,0x78f2}, // <CJK>
  {0xEDAB,0x7905}, // <CJK>
  {0xEDAC,0x78f9}, // <CJK>
  {0xEDAD,0x78fe}, // <CJK>
  {0xEDAE,0x7904}, // <CJK>
  {0xEDAF,0x79ab}, // <CJK>
  {0xEDB0,0x79a8}, // <CJK>
  {0xEDB1,0x7a5c}, // <CJK>
  {0xEDB2,0x7a5b}, // <CJK>
  {0xEDB3,0x7a56}, // <CJK>
  {0xEDB4,0x7a58}, // <CJK>
  {0xEDB5,0x7a54}, // <CJK>
  {0xEDB6,0x7a5a}, // <CJK>
  {0xEDB7,0x7abe}, // <CJK>
  {0xEDB8,0x7ac0}, // <CJK>
  {0xEDB9,0x7ac1}, // <CJK>
  {0xEDBA,0x7c05}, // <CJK>
  {0xEDBB,0x7c0f}, // <CJK>
  {0xEDBC,0x7bf2}, // <CJK>
  {0xEDBD,0x7c00}, // <CJK>
  {0xEDBE,0x7bff}, // <CJK>
  {0xEDBF,0x7bfb}, // <CJK>
  {0xEDC0,0x7c0e}, // <CJK>
  {0xEDC1,0x7bf4}, // <CJK>
  {0xEDC2,0x7c0b}, // <CJK>
  {0xEDC3,0x7bf3}, // <CJK>
  {0xEDC4,0x7c02}, // <CJK>
  {0xEDC5,0x7c09}, // <CJK>
  {0xEDC6,0x7c03}, // <CJK>
  {0xEDC7,0x7c01}, // <CJK>
  {0xEDC8,0x7bf8}, // <CJK>
  {0xEDC9,0x7bfd}, // <CJK>
  {0xEDCA,0x7c06}, // <CJK>
  {0xEDCB,0x7bf0}, // <CJK>
  {0xEDCC,0x7bf1}, // <CJK>
  {0xEDCD,0x7c10}, // <CJK>
  {0xEDCE,0x7c0a}, // <CJK>
  {0xEDCF,0x7ce8}, // <CJK>
  {0xEDD0,0x7e2d}, // <CJK>
  {0xEDD1,0x7e3c}, // <CJK>
  {0xEDD2,0x7e42}, // <CJK>
  {0xEDD3,0x7e33}, // <CJK>
  {0xEDD4,0x9848}, // <CJK>
  {0xEDD5,0x7e38}, // <CJK>
  {0xEDD6,0x7e2a}, // <CJK>
  {0xEDD7,0x7e49}, // <CJK>
  {0xEDD8,0x7e40}, // <CJK>
  {0xEDD9,0x7e47}, // <CJK>
  {0xEDDA,0x7e29}, // <CJK>
  {0xEDDB,0x7e4c}, // <CJK>
  {0xEDDC,0x7e30}, // <CJK>
  {0xEDDD,0x7e3b}, // <CJK>
  {0xEDDE,0x7e36}, // <CJK>
  {0xEDDF,0x7e44}, // <CJK>
  {0xEDE0,0x7e3a}, // <CJK>
  {0xEDE1,0x7f45}, // <CJK>
  {0xEDE2,0x7f7f}, // <CJK>
  {0xEDE3,0x7f7e}, // <CJK>
  {0xEDE4,0x7f7d}, // <CJK>
  {0xEDE5,0x7ff4}, // <CJK>
  {0xEDE6,0x7ff2}, // <CJK>
  {0xEDE7,0x802c}, // <CJK>
  {0xEDE8,0x81bb}, // <CJK>
  {0xEDE9,0x81c4}, // <CJK>
  {0xEDEA,0x81cc}, // <CJK>
  {0xEDEB,0x81ca}, // <CJK>
  {0xEDEC,0x81c5}, // <CJK>
  {0xEDED,0x81c7}, // <CJK>
  {0xEDEE,0x81bc}, // <CJK>
  {0xEDEF,0x81e9}, // <CJK>
  {0xEDF0,0x825b}, // <CJK>
  {0xEDF1,0x825a}, // <CJK>
  {0xEDF2,0x825c}, // <CJK>
  {0xEDF3,0x8583}, // <CJK>
  {0xEDF4,0x8580}, // <CJK>
  {0xEDF5,0x858f}, // <CJK>
  {0xEDF6,0x85a7}, // <CJK>
  {0xEDF7,0x8595}, // <CJK>
  {0xEDF8,0x85a0}, // <CJK>
  {0xEDF9,0x858b}, // <CJK>
  {0xEDFA,0x85a3}, // <CJK>
  {0xEDFB,0x857b}, // <CJK>
  {0xEDFC,0x85a4}, // <CJK>
  {0xEDFD,0x859a}, // <CJK>
  {0xEDFE,0x859e}, // <CJK>
  {0xEE40,0x8577}, // <CJK>
  {0xEE41,0x857c}, // <CJK>
  {0xEE42,0x8589}, // <CJK>
  {0xEE43,0x85a1}, // <CJK>
  {0xEE44,0x857a}, // <CJK>
  {0xEE45,0x8578}, // <CJK>
  {0xEE46,0x8557}, // <CJK>
  {0xEE47,0x858e}, // <CJK>
  {0xEE48,0x8596}, // <CJK>
  {0xEE49,0x8586}, // <CJK>
  {0xEE4A,0x858d}, // <CJK>
  {0xEE4B,0x8599}, // <CJK>
  {0xEE4C,0x859d}, // <CJK>
  {0xEE4D,0x8581}, // <CJK>
  {0xEE4E,0x85a2}, // <CJK>
  {0xEE4F,0x8582}, // <CJK>
  {0xEE50,0x8588}, // <CJK>
  {0xEE51,0x8585}, // <CJK>
  {0xEE52,0x8579}, // <CJK>
  {0xEE53,0x8576}, // <CJK>
  {0xEE54,0x8598}, // <CJK>
  {0xEE55,0x8590}, // <CJK>
  {0xEE56,0x859f}, // <CJK>
  {0xEE57,0x8668}, // <CJK>
  {0xEE58,0x87be}, // <CJK>
  {0xEE59,0x87aa}, // <CJK>
  {0xEE5A,0x87ad}, // <CJK>
  {0xEE5B,0x87c5}, // <CJK>
  {0xEE5C,0x87b0}, // <CJK>
  {0xEE5D,0x87ac}, // <CJK>
  {0xEE5E,0x87b9}, // <CJK>
  {0xEE5F,0x87b5}, // <CJK>
  {0xEE60,0x87bc}, // <CJK>
  {0xEE61,0x87ae}, // <CJK>
  {0xEE62,0x87c9}, // <CJK>
  {0xEE63,0x87c3}, // <CJK>
  {0xEE64,0x87c2}, // <CJK>
  {0xEE65,0x87cc}, // <CJK>
  {0xEE66,0x87b7}, // <CJK>
  {0xEE67,0x87af}, // <CJK>
  {0xEE68,0x87c4}, // <CJK>
  {0xEE69,0x87ca}, // <CJK>
  {0xEE6A,0x87b4}, // <CJK>
  {0xEE6B,0x87b6}, // <CJK>
  {0xEE6C,0x87bf}, // <CJK>
  {0xEE6D,0x87b8}, // <CJK>
  {0xEE6E,0x87bd}, // <CJK>
  {0xEE6F,0x87de}, // <CJK>
  {0xEE70,0x87b2}, // <CJK>
  {0xEE71,0x8935}, // <CJK>
  {0xEE72,0x8933}, // <CJK>
  {0xEE73,0x893c}, // <CJK>
  {0xEE74,0x893e}, // <CJK>
  {0xEE75,0x8941}, // <CJK>
  {0xEE76,0x8952}, // <CJK>
  {0xEE77,0x8937}, // <CJK>
  {0xEE78,0x8942}, // <CJK>
  {0xEE79,0x89ad}, // <CJK>
  {0xEE7A,0x89af}, // <CJK>
  {0xEE7B,0x89ae}, // <CJK>
  {0xEE7C,0x89f2}, // <CJK>
  {0xEE7D,0x89f3}, // <CJK>
  {0xEE7E,0x8b1e}, // <CJK>
  {0xEEA1,0x8b18}, // <CJK>
  {0xEEA2,0x8b16}, // <CJK>
  {0xEEA3,0x8b11}, // <CJK>
  {0xEEA4,0x8b05}, // <CJK>
  {0xEEA5,0x8b0b}, // <CJK>
  {0xEEA6,0x8b22}, // <CJK>
  {0xEEA7,0x8b0f}, // <CJK>
  {0xEEA8,0x8b12}, // <CJK>
  {0xEEA9,0x8b15}, // <CJK>
  {0xEEAA,0x8b07}, // <CJK>
  {0xEEAB,0x8b0d}, // <CJK>
  {0xEEAC,0x8b08}, // <CJK>
  {0xEEAD,0x8b06}, // <CJK>
  {0xEEAE,0x8b1c}, // <CJK>
  {0xEEAF,0x8b13}, // <CJK>
  {0xEEB0,0x8b1a}, // <CJK>
  {0xEEB1,0x8c4f}, // <CJK>
  {0xEEB2,0x8c70}, // <CJK>
  {0xEEB3,0x8c72}, // <CJK>
  {0xEEB4,0x8c71}, // <CJK>
  {0xEEB5,0x8c6f}, // <CJK>
  {0xEEB6,0x8c95}, // <CJK>
  {0xEEB7,0x8c94}, // <CJK>
  {0xEEB8,0x8cf9}, // <CJK>
  {0xEEB9,0x8d6f}, // <CJK>
  {0xEEBA,0x8e4e}, // <CJK>
  {0xEEBB,0x8e4d}, // <CJK>
  {0xEEBC,0x8e53}, // <CJK>
  {0xEEBD,0x8e50}, // <CJK>
  {0xEEBE,0x8e4c}, // <CJK>
  {0xEEBF,0x8e47}, // <CJK>
  {0xEEC0,0x8f43}, // <CJK>
  {0xEEC1,0x8f40}, // <CJK>
  {0xEEC2,0x9085}, // <CJK>
  {0xEEC3,0x907e}, // <CJK>
  {0xEEC4,0x9138}, // <CJK>
  {0xEEC5,0x919a}, // <CJK>
  {0xEEC6,0x91a2}, // <CJK>
  {0xEEC7,0x919b}, // <CJK>
  {0xEEC8,0x9199}, // <CJK>
  {0xEEC9,0x919f}, // <CJK>
  {0xEECA,0x91a1}, // <CJK>
  {0xEECB,0x919d}, // <CJK>
  {0xEECC,0x91a0}, // <CJK>
  {0xEECD,0x93a1}, // <CJK>
  {0xEECE,0x9383}, // <CJK>
  {0xEECF,0x93af}, // <CJK>
  {0xEED0,0x9364}, // <CJK>
  {0xEED1,0x9356}, // <CJK>
  {0xEED2,0x9347}, // <CJK>
  {0xEED3,0x937c}, // <CJK>
  {0xEED4,0x9358}, // <CJK>
  {0xEED5,0x935c}, // <CJK>
  {0xEED6,0x9376}, // <CJK>
  {0xEED7,0x9349}, // <CJK>
  {0xEED8,0x9350}, // <CJK>
  {0xEED9,0x9351}, // <CJK>
  {0xEEDA,0x9360}, // <CJK>
  {0xEEDB,0x936d}, // <CJK>
  {0xEEDC,0x938f}, // <CJK>
  {0xEEDD,0x934c}, // <CJK>
  {0xEEDE,0x936a}, // <CJK>
  {0xEEDF,0x9379}, // <CJK>
  {0xEEE0,0x9357}, // <CJK>
  {0xEEE1,0x9355}, // <CJK>
  {0xEEE2,0x9352}, // <CJK>
  {0xEEE3,0x934f}, // <CJK>
  {0xEEE4,0x9371}, // <CJK>
  {0xEEE5,0x9377}, // <CJK>
  {0xEEE6,0x937b}, // <CJK>
  {0xEEE7,0x9361}, // <CJK>
  {0xEEE8,0x935e}, // <CJK>
  {0xEEE9,0x9363}, // <CJK>
  {0xEEEA,0x9367}, // <CJK>
  {0xEEEB,0x9380}, // <CJK>
  {0xEEEC,0x934e}, // <CJK>
  {0xEEED,0x9359}, // <CJK>
  {0xEEEE,0x95c7}, // <CJK>
  {0xEEEF,0x95c0}, // <CJK>
  {0xEEF0,0x95c9}, // <CJK>
  {0xEEF1,0x95c3}, // <CJK>
  {0xEEF2,0x95c5}, // <CJK>
  {0xEEF3,0x95b7}, // <CJK>
  {0xEEF4,0x96ae}, // <CJK>
  {0xEEF5,0x96b0}, // <CJK>
  {0xEEF6,0x96ac}, // <CJK>
  {0xEEF7,0x9720}, // <CJK>
  {0xEEF8,0x971f}, // <CJK>
  {0xEEF9,0x9718}, // <CJK>
  {0xEEFA,0x971d}, // <CJK>
  {0xEEFB,0x9719}, // <CJK>
  {0xEEFC,0x979a}, // <CJK>
  {0xEEFD,0x97a1}, // <CJK>
  {0xEEFE,0x979c}, // <CJK>
  {0xEF40,0x979e}, // <CJK>
  {0xEF41,0x979d}, // <CJK>
  {0xEF42,0x97d5}, // <CJK>
  {0xEF43,0x97d4}, // <CJK>
  {0xEF44,0x97f1}, // <CJK>
  {0xEF45,0x9841}, // <CJK>
  {0xEF46,0x9844}, // <CJK>
  {0xEF47,0x984a}, // <CJK>
  {0xEF48,0x9849}, // <CJK>
  {0xEF49,0x9845}, // <CJK>
  {0xEF4A,0x9843}, // <CJK>
  {0xEF4B,0x9925}, // <CJK>
  {0xEF4C,0x992b}, // <CJK>
  {0xEF4D,0x992c}, // <CJK>
  {0xEF4E,0x992a}, // <CJK>
  {0xEF4F,0x9933}, // <CJK>
  {0xEF50,0x9932}, // <CJK>
  {0xEF51,0x992f}, // <CJK>
  {0xEF52,0x992d}, // <CJK>
  {0xEF53,0x9931}, // <CJK>
  {0xEF54,0x9930}, // <CJK>
  {0xEF55,0x9998}, // <CJK>
  {0xEF56,0x99a3}, // <CJK>
  {0xEF57,0x99a1}, // <CJK>
  {0xEF58,0x9a02}, // <CJK>
  {0xEF59,0x99fa}, // <CJK>
  {0xEF5A,0x99f4}, // <CJK>
  {0xEF5B,0x99f7}, // <CJK>
  {0xEF5C,0x99f9}, // <CJK>
  {0xEF5D,0x99f8}, // <CJK>
  {0xEF5E,0x99f6}, // <CJK>
  {0xEF5F,0x99fb}, // <CJK>
  {0xEF60,0x99fd}, // <CJK>
  {0xEF61,0x99fe}, // <CJK>
  {0xEF62,0x99fc}, // <CJK>
  {0xEF63,0x9a03}, // <CJK>
  {0xEF64,0x9abe}, // <CJK>
  {0xEF65,0x9afe}, // <CJK>
  {0xEF66,0x9afd}, // <CJK>
  {0xEF67,0x9b01}, // <CJK>
  {0xEF68,0x9afc}, // <CJK>
  {0xEF69,0x9b48}, // <CJK>
  {0xEF6A,0x9b9a}, // <CJK>
  {0xEF6B,0x9ba8}, // <CJK>
  {0xEF6C,0x9b9e}, // <CJK>
  {0xEF6D,0x9b9b}, // <CJK>
  {0xEF6E,0x9ba6}, // <CJK>
  {0xEF6F,0x9ba1}, // <CJK>
  {0xEF70,0x9ba5}, // <CJK>
  {0xEF71,0x9ba4}, // <CJK>
  {0xEF72,0x9b86}, // <CJK>
  {0xEF73,0x9ba2}, // <CJK>
  {0xEF74,0x9ba0}, // <CJK>
  {0xEF75,0x9baf}, // <CJK>
  {0xEF76,0x9d33}, // <CJK>
  {0xEF77,0x9d41}, // <CJK>
  {0xEF78,0x9d67}, // <CJK>
  {0xEF79,0x9d36}, // <CJK>
  {0xEF7A,0x9d2e}, // <CJK>
  {0xEF7B,0x9d2f}, // <CJK>
  {0xEF7C,0x9d31}, // <CJK>
  {0xEF7D,0x9d38}, // <CJK>
  {0xEF7E,0x9d30}, // <CJK>
  {0xEFA1,0x9d45}, // <CJK>
  {0xEFA2,0x9d42}, // <CJK>
  {0xEFA3,0x9d43}, // <CJK>
  {0xEFA4,0x9d3e}, // <CJK>
  {0xEFA5,0x9d37}, // <CJK>
  {0xEFA6,0x9d40}, // <CJK>
  {0xEFA7,0x9d3d}, // <CJK>
  {0xEFA8,0x7ff5}, // <CJK>
  {0xEFA9,0x9d2d}, // <CJK>
  {0xEFAA,0x9e8a}, // <CJK>
  {0xEFAB,0x9e89}, // <CJK>
  {0xEFAC,0x9e8d}, // <CJK>
  {0xEFAD,0x9eb0}, // <CJK>
  {0xEFAE,0x9ec8}, // <CJK>
  {0xEFAF,0x9eda}, // <CJK>
  {0xEFB0,0x9efb}, // <CJK>
  {0xEFB1,0x9eff}, // <CJK>
  {0xEFB2,0x9f24}, // <CJK>
  {0xEFB3,0x9f23}, // <CJK>
  {0xEFB4,0x9f22}, // <CJK>
  {0xEFB5,0x9f54}, // <CJK>
  {0xEFB6,0x9fa0}, // <CJK>
  {0xEFB7,0x5131}, // <CJK>
  {0xEFB8,0x512d}, // <CJK>
  {0xEFB9,0x512e}, // <CJK>
  {0xEFBA,0x5698}, // <CJK>
  {0xEFBB,0x569c}, // <CJK>
  {0xEFBC,0x5697}, // <CJK>
  {0xEFBD,0x569a}, // <CJK>
  {0xEFBE,0x569d}, // <CJK>
  {0xEFBF,0x5699}, // <CJK>
  {0xEFC0,0x5970}, // <CJK>
  {0xEFC1,0x5b3c}, // <CJK>
  {0xEFC2,0x5c69}, // <CJK>
  {0xEFC3,0x5c6a}, // <CJK>
  {0xEFC4,0x5dc0}, // <CJK>
  {0xEFC5,0x5e6d}, // <CJK>
  {0xEFC6,0x5e6e}, // <CJK>
  {0xEFC7,0x61d8}, // <CJK>
  {0xEFC8,0x61df}, // <CJK>
  {0xEFC9,0x61ed}, // <CJK>
  {0xEFCA,0x61ee}, // <CJK>
  {0xEFCB,0x61f1}, // <CJK>
  {0xEFCC,0x61ea}, // <CJK>
  {0xEFCD,0x61f0}, // <CJK>
  {0xEFCE,0x61eb}, // <CJK>
  {0xEFCF,0x61d6}, // <CJK>
  {0xEFD0,0x61e9}, // <CJK>
  {0xEFD1,0x64ff}, // <CJK>
  {0xEFD2,0x6504}, // <CJK>
  {0xEFD3,0x64fd}, // <CJK>
  {0xEFD4,0x64f8}, // <CJK>
  {0xEFD5,0x6501}, // <CJK>
  {0xEFD6,0x6503}, // <CJK>
  {0xEFD7,0x64fc}, // <CJK>
  {0xEFD8,0x6594}, // <CJK>
  {0xEFD9,0x65db}, // <CJK>
  {0xEFDA,0x66da}, // <CJK>
  {0xEFDB,0x66db}, // <CJK>
  {0xEFDC,0x66d8}, // <CJK>
  {0xEFDD,0x6ac5}, // <CJK>
  {0xEFDE,0x6ab9}, // <CJK>
  {0xEFDF,0x6abd}, // <CJK>
  {0xEFE0,0x6ae1}, // <CJK>
  {0xEFE1,0x6ac6}, // <CJK>
  {0xEFE2,0x6aba}, // <CJK>
  {0xEFE3,0x6ab6}, // <CJK>
  {0xEFE4,0x6ab7}, // <CJK>
  {0xEFE5,0x6ac7}, // <CJK>
  {0xEFE6,0x6ab4}, // <CJK>
  {0xEFE7,0x6aad}, // <CJK>
  {0xEFE8,0x6b5e}, // <CJK>
  {0xEFE9,0x6bc9}, // <CJK>
  {0xEFEA,0x6c0b}, // <CJK>
  {0xEFEB,0x7007}, // <CJK>
  {0xEFEC,0x700c}, // <CJK>
  {0xEFED,0x700d}, // <CJK>
  {0xEFEE,0x7001}, // <CJK>
  {0xEFEF,0x7005}, // <CJK>
  {0xEFF0,0x7014}, // <CJK>
  {0xEFF1,0x700e}, // <CJK>
  {0xEFF2,0x6fff}, // <CJK>
  {0xEFF3,0x7000}, // <CJK>
  {0xEFF4,0x6ffb}, // <CJK>
  {0xEFF5,0x7026}, // <CJK>
  {0xEFF6,0x6ffc}, // <CJK>
  {0xEFF7,0x6ff7}, // <CJK>
  {0xEFF8,0x700a}, // <CJK>
  {0xEFF9,0x7201}, // <CJK>
  {0xEFFA,0x71ff}, // <CJK>
  {0xEFFB,0x71f9}, // <CJK>
  {0xEFFC,0x7203}, // <CJK>
  {0xEFFD,0x71fd}, // <CJK>
  {0xEFFE,0x7376}, // <CJK>
  {0xF040,0x74b8}, // <CJK>
  {0xF041,0x74c0}, // <CJK>
  {0xF042,0x74b5}, // <CJK>
  {0xF043,0x74c1}, // <CJK>
  {0xF044,0x74be}, // <CJK>
  {0xF045,0x74b6}, // <CJK>
  {0xF046,0x74bb}, // <CJK>
  {0xF047,0x74c2}, // <CJK>
  {0xF048,0x7514}, // <CJK>
  {0xF049,0x7513}, // <CJK>
  {0xF04A,0x765c}, // <CJK>
  {0xF04B,0x7664}, // <CJK>
  {0xF04C,0x7659}, // <CJK>
  {0xF04D,0x7650}, // <CJK>
  {0xF04E,0x7653}, // <CJK>
  {0xF04F,0x7657}, // <CJK>
  {0xF050,0x765a}, // <CJK>
  {0xF051,0x76a6}, // <CJK>
  {0xF052,0x76bd}, // <CJK>
  {0xF053,0x76ec}, // <CJK>
  {0xF054,0x77c2}, // <CJK>
  {0xF055,0x77ba}, // <CJK>
  {0xF056,0x78ff}, // <CJK>
  {0xF057,0x790c}, // <CJK>
  {0xF058,0x7913}, // <CJK>
  {0xF059,0x7914}, // <CJK>
  {0xF05A,0x7909}, // <CJK>
  {0xF05B,0x7910}, // <CJK>
  {0xF05C,0x7912}, // <CJK>
  {0xF05D,0x7911}, // <CJK>
  {0xF05E,0x79ad}, // <CJK>
  {0xF05F,0x79ac}, // <CJK>
  {0xF060,0x7a5f}, // <CJK>
  {0xF061,0x7c1c}, // <CJK>
  {0xF062,0x7c29}, // <CJK>
  {0xF063,0x7c19}, // <CJK>
  {0xF064,0x7c20}, // <CJK>
  {0xF065,0x7c1f}, // <CJK>
  {0xF066,0x7c2d}, // <CJK>
  {0xF067,0x7c1d}, // <CJK>
  {0xF068,0x7c26}, // <CJK>
  {0xF069,0x7c28}, // <CJK>
  {0xF06A,0x7c22}, // <CJK>
  {0xF06B,0x7c25}, // <CJK>
  {0xF06C,0x7c30}, // <CJK>
  {0xF06D,0x7e5c}, // <CJK>
  {0xF06E,0x7e50}, // <CJK>
  {0xF06F,0x7e56}, // <CJK>
  {0xF070,0x7e63}, // <CJK>
  {0xF071,0x7e58}, // <CJK>
  {0xF072,0x7e62}, // <CJK>
  {0xF073,0x7e5f}, // <CJK>
  {0xF074,0x7e51}, // <CJK>
  {0xF075,0x7e60}, // <CJK>
  {0xF076,0x7e57}, // <CJK>
  {0xF077,0x7e53}, // <CJK>
  {0xF078,0x7fb5}, // <CJK>
  {0xF079,0x7fb3}, // <CJK>
  {0xF07A,0x7ff7}, // <CJK>
  {0xF07B,0x7ff8}, // <CJK>
  {0xF07C,0x8075}, // <CJK>
  {0xF07D,0x81d1}, // <CJK>
  {0xF07E,0x81d2}, // <CJK>
  {0xF0A1,0x81d0}, // <CJK>
  {0xF0A2,0x825f}, // <CJK>
  {0xF0A3,0x825e}, // <CJK>
  {0xF0A4,0x85b4}, // <CJK>
  {0xF0A5,0x85c6}, // <CJK>
  {0xF0A6,0x85c0}, // <CJK>
  {0xF0A7,0x85c3}, // <CJK>
  {0xF0A8,0x85c2}, // <CJK>
  {0xF0A9,0x85b3}, // <CJK>
  {0xF0AA,0x85b5}, // <CJK>
  {0xF0AB,0x85bd}, // <CJK>
  {0xF0AC,0x85c7}, // <CJK>
  {0xF0AD,0x85c4}, // <CJK>
  {0xF0AE,0x85bf}, // <CJK>
  {0xF0AF,0x85cb}, // <CJK>
  {0xF0B0,0x85ce}, // <CJK>
  {0xF0B1,0x85c8}, // <CJK>
  {0xF0B2,0x85c5}, // <CJK>
  {0xF0B3,0x85b1}, // <CJK>
  {0xF0B4,0x85b6}, // <CJK>
  {0xF0B5,0x85d2}, // <CJK>
  {0xF0B6,0x8624}, // <CJK>
  {0xF0B7,0x85b8}, // <CJK>
  {0xF0B8,0x85b7}, // <CJK>
  {0xF0B9,0x85be}, // <CJK>
  {0xF0BA,0x8669}, // <CJK>
  {0xF0BB,0x87e7}, // <CJK>
  {0xF0BC,0x87e6}, // <CJK>
  {0xF0BD,0x87e2}, // <CJK>
  {0xF0BE,0x87db}, // <CJK>
  {0xF0BF,0x87eb}, // <CJK>
  {0xF0C0,0x87ea}, // <CJK>
  {0xF0C1,0x87e5}, // <CJK>
  {0xF0C2,0x87df}, // <CJK>
  {0xF0C3,0x87f3}, // <CJK>
  {0xF0C4,0x87e4}, // <CJK>
  {0xF0C5,0x87d4}, // <CJK>
  {0xF0C6,0x87dc}, // <CJK>
  {0xF0C7,0x87d3}, // <CJK>
  {0xF0C8,0x87ed}, // <CJK>
  {0xF0C9,0x87d8}, // <CJK>
  {0xF0CA,0x87e3}, // <CJK>
  {0xF0CB,0x87a4}, // <CJK>
  {0xF0CC,0x87d7}, // <CJK>
  {0xF0CD,0x87d9}, // <CJK>
  {0xF0CE,0x8801}, // <CJK>
  {0xF0CF,0x87f4}, // <CJK>
  {0xF0D0,0x87e8}, // <CJK>
  {0xF0D1,0x87dd}, // <CJK>
  {0xF0D2,0x8953}, // <CJK>
  {0xF0D3,0x894b}, // <CJK>
  {0xF0D4,0x894f}, // <CJK>
  {0xF0D5,0x894c}, // <CJK>
  {0xF0D6,0x8946}, // <CJK>
  {0xF0D7,0x8950}, // <CJK>
  {0xF0D8,0x8951}, // <CJK>
  {0xF0D9,0x8949}, // <CJK>
  {0xF0DA,0x8b2a}, // <CJK>
  {0xF0DB,0x8b27}, // <CJK>
  {0xF0DC,0x8b23}, // <CJK>
  {0xF0DD,0x8b33}, // <CJK>
  {0xF0DE,0x8b30}, // <CJK>
  {0xF0DF,0x8b35}, // <CJK>
  {0xF0E0,0x8b47}, // <CJK>
  {0xF0E1,0x8b2f}, // <CJK>
  {0xF0E2,0x8b3c}, // <CJK>
  {0xF0E3,0x8b3e}, // <CJK>
  {0xF0E4,0x8b31}, // <CJK>
  {0xF0E5,0x8b25}, // <CJK>
  {0xF0E6,0x8b37}, // <CJK>
  {0xF0E7,0x8b26}, // <CJK>
  {0xF0E8,0x8b36}, // <CJK>
  {0xF0E9,0x8b2e}, // <CJK>
  {0xF0EA,0x8b24}, // <CJK>
  {0xF0EB,0x8b3b}, // <CJK>
  {0xF0EC,0x8b3d}, // <CJK>
  {0xF0ED,0x8b3a}, // <CJK>
  {0xF0EE,0x8c42}, // <CJK>
  {0xF0EF,0x8c75}, // <CJK>
  {0xF0F0,0x8c99}, // <CJK>
  {0xF0F1,0x8c98}, // <CJK>
  {0xF0F2,0x8c97}, // <CJK>
  {0xF0F3,0x8cfe}, // <CJK>
  {0xF0F4,0x8d04}, // <CJK>
  {0xF0F5,0x8d02}, // <CJK>
  {0xF0F6,0x8d00}, // <CJK>
  {0xF0F7,0x8e5c}, // <CJK>
  {0xF0F8,0x8e62}, // <CJK>
  {0xF0F9,0x8e60}, // <CJK>
  {0xF0FA,0x8e57}, // <CJK>
  {0xF0FB,0x8e56}, // <CJK>
  {0xF0FC,0x8e5e}, // <CJK>
  {0xF0FD,0x8e65}, // <CJK>
  {0xF0FE,0x8e67}, // <CJK>
  {0xF140,0x8e5b}, // <CJK>
  {0xF141,0x8e5a}, // <CJK>
  {0xF142,0x8e61}, // <CJK>
  {0xF143,0x8e5d}, // <CJK>
  {0xF144,0x8e69}, // <CJK>
  {0xF145,0x8e54}, // <CJK>
  {0xF146,0x8f46}, // <CJK>
  {0xF147,0x8f47}, // <CJK>
  {0xF148,0x8f48}, // <CJK>
  {0xF149,0x8f4b}, // <CJK>
  {0xF14A,0x9128}, // <CJK>
  {0xF14B,0x913a}, // <CJK>
  {0xF14C,0x913b}, // <CJK>
  {0xF14D,0x913e}, // <CJK>
  {0xF14E,0x91a8}, // <CJK>
  {0xF14F,0x91a5}, // <CJK>
  {0xF150,0x91a7}, // <CJK>
  {0xF151,0x91af}, // <CJK>
  {0xF152,0x91aa}, // <CJK>
  {0xF153,0x93b5}, // <CJK>
  {0xF154,0x938c}, // <CJK>
  {0xF155,0x9392}, // <CJK>
  {0xF156,0x93b7}, // <CJK>
  {0xF157,0x939b}, // <CJK>
  {0xF158,0x939d}, // <CJK>
  {0xF159,0x9389}, // <CJK>
  {0xF15A,0x93a7}, // <CJK>
  {0xF15B,0x938e}, // <CJK>
  {0xF15C,0x93aa}, // <CJK>
  {0xF15D,0x939e}, // <CJK>
  {0xF15E,0x93a6}, // <CJK>
  {0xF15F,0x9395}, // <CJK>
  {0xF160,0x9388}, // <CJK>
  {0xF161,0x9399}, // <CJK>
  {0xF162,0x939f}, // <CJK>
  {0xF163,0x938d}, // <CJK>
  {0xF164,0x93b1}, // <CJK>
  {0xF165,0x9391}, // <CJK>
  {0xF166,0x93b2}, // <CJK>
  {0xF167,0x93a4}, // <CJK>
  {0xF168,0x93a8}, // <CJK>
  {0xF169,0x93b4}, // <CJK>
  {0xF16A,0x93a3}, // <CJK>
  {0xF16B,0x93a5}, // <CJK>
  {0xF16C,0x95d2}, // <CJK>
  {0xF16D,0x95d3}, // <CJK>
  {0xF16E,0x95d1}, // <CJK>
  {0xF16F,0x96b3}, // <CJK>
  {0xF170,0x96d7}, // <CJK>
  {0xF171,0x96da}, // <CJK>
  {0xF172,0x5dc2}, // <CJK>
  {0xF173,0x96df}, // <CJK>
  {0xF174,0x96d8}, // <CJK>
  {0xF175,0x96dd}, // <CJK>
  {0xF176,0x9723}, // <CJK>
  {0xF177,0x9722}, // <CJK>
  {0xF178,0x9725}, // <CJK>
  {0xF179,0x97ac}, // <CJK>
  {0xF17A,0x97ae}, // <CJK>
  {0xF17B,0x97a8}, // <CJK>
  {0xF17C,0x97ab}, // <CJK>
  {0xF17D,0x97a4}, // <CJK>
  {0xF17E,0x97aa}, // <CJK>
  {0xF1A1,0x97a2}, // <CJK>
  {0xF1A2,0x97a5}, // <CJK>
  {0xF1A3,0x97d7}, // <CJK>
  {0xF1A4,0x97d9}, // <CJK>
  {0xF1A5,0x97d6}, // <CJK>
  {0xF1A6,0x97d8}, // <CJK>
  {0xF1A7,0x97fa}, // <CJK>
  {0xF1A8,0x9850}, // <CJK>
  {0xF1A9,0x9851}, // <CJK>
  {0xF1AA,0x9852}, // <CJK>
  {0xF1AB,0x98b8}, // <CJK>
  {0xF1AC,0x9941}, // <CJK>
  {0xF1AD,0x993c}, // <CJK>
  {0xF1AE,0x993a}, // <CJK>
  {0xF1AF,0x9a0f}, // <CJK>
  {0xF1B0,0x9a0b}, // <CJK>
  {0xF1B1,0x9a09}, // <CJK>
  {0xF1B2,0x9a0d}, // <CJK>
  {0xF1B3,0x9a04}, // <CJK>
  {0xF1B4,0x9a11}, // <CJK>
  {0xF1B5,0x9a0a}, // <CJK>
  {0xF1B6,0x9a05}, // <CJK>
  {0xF1B7,0x9a07}, // <CJK>
  {0xF1B8,0x9a06}, // <CJK>
  {0xF1B9,0x9ac0}, // <CJK>
  {0xF1BA,0x9adc}, // <CJK>
  {0xF1BB,0x9b08}, // <CJK>
  {0xF1BC,0x9b04}, // <CJK>
  {0xF1BD,0x9b05}, // <CJK>
  {0xF1BE,0x9b29}, // <CJK>
  {0xF1BF,0x9b35}, // <CJK>
  {0xF1C0,0x9b4a}, // <CJK>
  {0xF1C1,0x9b4c}, // <CJK>
  {0xF1C2,0x9b4b}, // <CJK>
  {0xF1C3,0x9bc7}, // <CJK>
  {0xF1C4,0x9bc6}, // <CJK>
  {0xF1C5,0x9bc3}, // <CJK>
  {0xF1C6,0x9bbf}, // <CJK>
  {0xF1C7,0x9bc1}, // <CJK>
  {0xF1C8,0x9bb5}, // <CJK>
  {0xF1C9,0x9bb8}, // <CJK>
  {0xF1CA,0x9bd3}, // <CJK>
  {0xF1CB,0x9bb6}, // <CJK>
  {0xF1CC,0x9bc4}, // <CJK>
  {0xF1CD,0x9bb9}, // <CJK>
  {0xF1CE,0x9bbd}, // <CJK>
  {0xF1CF,0x9d5c}, // <CJK>
  {0xF1D0,0x9d53}, // <CJK>
  {0xF1D1,0x9d4f}, // <CJK>
  {0xF1D2,0x9d4a}, // <CJK>
  {0xF1D3,0x9d5b}, // <CJK>
  {0xF1D4,0x9d4b}, // <CJK>
  {0xF1D5,0x9d59}, // <CJK>
  {0xF1D6,0x9d56}, // <CJK>
  {0xF1D7,0x9d4c}, // <CJK>
  {0xF1D8,0x9d57}, // <CJK>
  {0xF1D9,0x9d52}, // <CJK>
  {0xF1DA,0x9d54}, // <CJK>
  {0xF1DB,0x9d5f}, // <CJK>
  {0xF1DC,0x9d58}, // <CJK>
  {0xF1DD,0x9d5a}, // <CJK>
  {0xF1DE,0x9e8e}, // <CJK>
  {0xF1DF,0x9e8c}, // <CJK>
  {0xF1E0,0x9edf}, // <CJK>
  {0xF1E1,0x9f01}, // <CJK>
  {0xF1E2,0x9f00}, // <CJK>
  {0xF1E3,0x9f16}, // <CJK>
  {0xF1E4,0x9f25}, // <CJK>
  {0xF1E5,0x9f2b}, // <CJK>
  {0xF1E6,0x9f2a}, // <CJK>
  {0xF1E7,0x9f29}, // <CJK>
  {0xF1E8,0x9f28}, // <CJK>
  {0xF1E9,0x9f4c}, // <CJK>
  {0xF1EA,0x9f55}, // <CJK>
  {0xF1EB,0x5134}, // <CJK>
  {0xF1EC,0x5135}, // <CJK>
  {0xF1ED,0x5296}, // <CJK>
  {0xF1EE,0x52f7}, // <CJK>
  {0xF1EF,0x53b4}, // <CJK>
  {0xF1F0,0x56ab}, // <CJK>
  {0xF1F1,0x56ad}, // <CJK>
  {0xF1F2,0x56a6}, // <CJK>
  {0xF1F3,0x56a7}, // <CJK>
  {0xF1F4,0x56aa}, // <CJK>
  {0xF1F5,0x56ac}, // <CJK>
  {0xF1F6,0x58da}, // <CJK>
  {0xF1F7,0x58dd}, // <CJK>
  {0xF1F8,0x58db}, // <CJK>
  {0xF1F9,0x5912}, // <CJK>
  {0xF1FA,0x5b3d}, // <CJK>
  {0xF1FB,0x5b3e}, // <CJK>
  {0xF1FC,0x5b3f}, // <CJK>
  {0xF1FD,0x5dc3}, // <CJK>
  {0xF1FE,0x5e70}, // <CJK>
  {0xF240,0x5fbf}, // <CJK>
  {0xF241,0x61fb}, // <CJK>
  {0xF242,0x6507}, // <CJK>
  {0xF243,0x6510}, // <CJK>
  {0xF244,0x650d}, // <CJK>
  {0xF245,0x6509}, // <CJK>
  {0xF246,0x650c}, // <CJK>
  {0xF247,0x650e}, // <CJK>
  {0xF248,0x6584}, // <CJK>
  {0xF249,0x65de}, // <CJK>
  {0xF24A,0x65dd}, // <CJK>
  {0xF24B,0x66de}, // <CJK>
  {0xF24C,0x6ae7}, // <CJK>
  {0xF24D,0x6ae0}, // <CJK>
  {0xF24E,0x6acc}, // <CJK>
  {0xF24F,0x6ad1}, // <CJK>
  {0xF250,0x6ad9}, // <CJK>
  {0xF251,0x6acb}, // <CJK>
  {0xF252,0x6adf}, // <CJK>
  {0xF253,0x6adc}, // <CJK>
  {0xF254,0x6ad0}, // <CJK>
  {0xF255,0x6aeb}, // <CJK>
  {0xF256,0x6acf}, // <CJK>
  {0xF257,0x6acd}, // <CJK>
  {0xF258,0x6ade}, // <CJK>
  {0xF259,0x6b60}, // <CJK>
  {0xF25A,0x6bb0}, // <CJK>
  {0xF25B,0x6c0c}, // <CJK>
  {0xF25C,0x7019}, // <CJK>
  {0xF25D,0x7027}, // <CJK>
  {0xF25E,0x7020}, // <CJK>
  {0xF25F,0x7016}, // <CJK>
  {0xF260,0x702b}, // <CJK>
  {0xF261,0x7021}, // <CJK>
  {0xF262,0x7022}, // <CJK>
  {0xF263,0x7023}, // <CJK>
  {0xF264,0x7029}, // <CJK>
  {0xF265,0x7017}, // <CJK>
  {0xF266,0x7024}, // <CJK>
  {0xF267,0x701c}, // <CJK>
  {0xF268,0x702a}, // <CJK>
  {0xF269,0x720c}, // <CJK>
  {0xF26A,0x720a}, // <CJK>
  {0xF26B,0x7207}, // <CJK>
  {0xF26C,0x7202}, // <CJK>
  {0xF26D,0x7205}, // <CJK>
  {0xF26E,0x72a5}, // <CJK>
  {0xF26F,0x72a6}, // <CJK>
  {0xF270,0x72a4}, // <CJK>
  {0xF271,0x72a3}, // <CJK>
  {0xF272,0x72a1}, // <CJK>
  {0xF273,0x74cb}, // <CJK>
  {0xF274,0x74c5}, // <CJK>
  {0xF275,0x74b7}, // <CJK>
  {0xF276,0x74c3}, // <CJK>
  {0xF277,0x7516}, // <CJK>
  {0xF278,0x7660}, // <CJK>
  {0xF279,0x77c9}, // <CJK>
  {0xF27A,0x77ca}, // <CJK>
  {0xF27B,0x77c4}, // <CJK>
  {0xF27C,0x77f1}, // <CJK>
  {0xF27D,0x791d}, // <CJK>
  {0xF27E,0x791b}, // <CJK>
  {0xF2A1,0x7921}, // <CJK>
  {0xF2A2,0x791c}, // <CJK>
  {0xF2A3,0x7917}, // <CJK>
  {0xF2A4,0x791e}, // <CJK>
  {0xF2A5,0x79b0}, // <CJK>
  {0xF2A6,0x7a67}, // <CJK>
  {0xF2A7,0x7a68}, // <CJK>
  {0xF2A8,0x7c33}, // <CJK>
  {0xF2A9,0x7c3c}, // <CJK>
  {0xF2AA,0x7c39}, // <CJK>
  {0xF2AB,0x7c2c}, // <CJK>
  {0xF2AC,0x7c3b}, // <CJK>
  {0xF2AD,0x7cec}, // <CJK>
  {0xF2AE,0x7cea}, // <CJK>
  {0xF2AF,0x7e76}, // <CJK>
  {0xF2B0,0x7e75}, // <CJK>
  {0xF2B1,0x7e78}, // <CJK>
  {0xF2B2,0x7e70}, // <CJK>
  {0xF2B3,0x7e77}, // <CJK>
  {0xF2B4,0x7e6f}, // <CJK>
  {0xF2B5,0x7e7a}, // <CJK>
  {0xF2B6,0x7e72}, // <CJK>
  {0xF2B7,0x7e74}, // <CJK>
  {0xF2B8,0x7e68}, // <CJK>
  {0xF2B9,0x7f4b}, // <CJK>
  {0xF2BA,0x7f4a}, // <CJK>
  {0xF2BB,0x7f83}, // <CJK>
  {0xF2BC,0x7f86}, // <CJK>
  {0xF2BD,0x7fb7}, // <CJK>
  {0xF2BE,0x7ffd}, // <CJK>
  {0xF2BF,0x7ffe}, // <CJK>
  {0xF2C0,0x8078}, // <CJK>
  {0xF2C1,0x81d7}, // <CJK>
  {0xF2C2,0x81d5}, // <CJK>
  {0xF2C3,0x8264}, // <CJK>
  {0xF2C4,0x8261}, // <CJK>
  {0xF2C5,0x8263}, // <CJK>
  {0xF2C6,0x85eb}, // <CJK>
  {0xF2C7,0x85f1}, // <CJK>
  {0xF2C8,0x85ed}, // <CJK>
  {0xF2C9,0x85d9}, // <CJK>
  {0xF2CA,0x85e1}, // <CJK>
  {0xF2CB,0x85e8}, // <CJK>
  {0xF2CC,0x85da}, // <CJK>
  {0xF2CD,0x85d7}, // <CJK>
  {0xF2CE,0x85ec}, // <CJK>
  {0xF2CF,0x85f2}, // <CJK>
  {0xF2D0,0x85f8}, // <CJK>
  {0xF2D1,0x85d8}, // <CJK>
  {0xF2D2,0x85df}, // <CJK>
  {0xF2D3,0x85e3}, // <CJK>
  {0xF2D4,0x85dc}, // <CJK>
  {0xF2D5,0x85d1}, // <CJK>
  {0xF2D6,0x85f0}, // <CJK>
  {0xF2D7,0x85e6}, // <CJK>
  {0xF2D8,0x85ef}, // <CJK>
  {0xF2D9,0x85de}, // <CJK>
  {0xF2DA,0x85e2}, // <CJK>
  {0xF2DB,0x8800}, // <CJK>
  {0xF2DC,0x87fa}, // <CJK>
  {0xF2DD,0x8803}, // <CJK>
  {0xF2DE,0x87f6}, // <CJK>
  {0xF2DF,0x87f7}, // <CJK>
  {0xF2E0,0x8809}, // <CJK>
  {0xF2E1,0x880c}, // <CJK>
  {0xF2E2,0x880b}, // <CJK>
  {0xF2E3,0x8806}, // <CJK>
  {0xF2E4,0x87fc}, // <CJK>
  {0xF2E5,0x8808}, // <CJK>
  {0xF2E6,0x87ff}, // <CJK>
  {0xF2E7,0x880a}, // <CJK>
  {0xF2E8,0x8802}, // <CJK>
  {0xF2E9,0x8962}, // <CJK>
  {0xF2EA,0x895a}, // <CJK>
  {0xF2EB,0x895b}, // <CJK>
  {0xF2EC,0x8957}, // <CJK>
  {0xF2ED,0x8961}, // <CJK>
  {0xF2EE,0x895c}, // <CJK>
  {0xF2EF,0x8958}, // <CJK>
  {0xF2F0,0x895d}, // <CJK>
  {0xF2F1,0x8959}, // <CJK>
  {0xF2F2,0x8988}, // <CJK>
  {0xF2F3,0x89b7}, // <CJK>
  {0xF2F4,0x89b6}, // <CJK>
  {0xF2F5,0x89f6}, // <CJK>
  {0xF2F6,0x8b50}, // <CJK>
  {0xF2F7,0x8b48}, // <CJK>
  {0xF2F8,0x8b4a}, // <CJK>
  {0xF2F9,0x8b40}, // <CJK>
  {0xF2FA,0x8b53}, // <CJK>
  {0xF2FB,0x8b56}, // <CJK>
  {0xF2FC,0x8b54}, // <CJK>
  {0xF2FD,0x8b4b}, // <CJK>
  {0xF2FE,0x8b55}, // <CJK>
  {0xF340,0x8b51}, // <CJK>
  {0xF341,0x8b42}, // <CJK>
  {0xF342,0x8b52}, // <CJK>
  {0xF343,0x8b57}, // <CJK>
  {0xF344,0x8c43}, // <CJK>
  {0xF345,0x8c77}, // <CJK>
  {0xF346,0x8c76}, // <CJK>
  {0xF347,0x8c9a}, // <CJK>
  {0xF348,0x8d06}, // <CJK>
  {0xF349,0x8d07}, // <CJK>
  {0xF34A,0x8d09}, // <CJK>
  {0xF34B,0x8dac}, // <CJK>
  {0xF34C,0x8daa}, // <CJK>
  {0xF34D,0x8dad}, // <CJK>
  {0xF34E,0x8dab}, // <CJK>
  {0xF34F,0x8e6d}, // <CJK>
  {0xF350,0x8e78}, // <CJK>
  {0xF351,0x8e73}, // <CJK>
  {0xF352,0x8e6a}, // <CJK>
  {0xF353,0x8e6f}, // <CJK>
  {0xF354,0x8e7b}, // <CJK>
  {0xF355,0x8ec2}, // <CJK>
  {0xF356,0x8f52}, // <CJK>
  {0xF357,0x8f51}, // <CJK>
  {0xF358,0x8f4f}, // <CJK>
  {0xF359,0x8f50}, // <CJK>
  {0xF35A,0x8f53}, // <CJK>
  {0xF35B,0x8fb4}, // <CJK>
  {0xF35C,0x9140}, // <CJK>
  {0xF35D,0x913f}, // <CJK>
  {0xF35E,0x91b0}, // <CJK>
  {0xF35F,0x91ad}, // <CJK>
  {0xF360,0x93de}, // <CJK>
  {0xF361,0x93c7}, // <CJK>
  {0xF362,0x93cf}, // <CJK>
  {0xF363,0x93c2}, // <CJK>
  {0xF364,0x93da}, // <CJK>
  {0xF365,0x93d0}, // <CJK>
  {0xF366,0x93f9}, // <CJK>
  {0xF367,0x93ec}, // <CJK>
  {0xF368,0x93cc}, // <CJK>
  {0xF369,0x93d9}, // <CJK>
  {0xF36A,0x93a9}, // <CJK>
  {0xF36B,0x93e6}, // <CJK>
  {0xF36C,0x93ca}, // <CJK>
  {0xF36D,0x93d4}, // <CJK>
  {0xF36E,0x93ee}, // <CJK>
  {0xF36F,0x93e3}, // <CJK>
  {0xF370,0x93d5}, // <CJK>
  {0xF371,0x93c4}, // <CJK>
  {0xF372,0x93ce}, // <CJK>
  {0xF373,0x93c0}, // <CJK>
  {0xF374,0x93d2}, // <CJK>
  {0xF375,0x93e7}, // <CJK>
  {0xF376,0x957d}, // <CJK>
  {0xF377,0x95da}, // <CJK>
  {0xF378,0x95db}, // <CJK>
  {0xF379,0x96e1}, // <CJK>
  {0xF37A,0x9729}, // <CJK>
  {0xF37B,0x972b}, // <CJK>
  {0xF37C,0x972c}, // <CJK>
  {0xF37D,0x9728}, // <CJK>
  {0xF37E,0x9726}, // <CJK>
  {0xF3A1,0x97b3}, // <CJK>
  {0xF3A2,0x97b7}, // <CJK>
  {0xF3A3,0x97b6}, // <CJK>
  {0xF3A4,0x97dd}, // <CJK>
  {0xF3A5,0x97de}, // <CJK>
  {0xF3A6,0x97df}, // <CJK>
  {0xF3A7,0x985c}, // <CJK>
  {0xF3A8,0x9859}, // <CJK>
  {0xF3A9,0x985d}, // <CJK>
  {0xF3AA,0x9857}, // <CJK>
  {0xF3AB,0x98bf}, // <CJK>
  {0xF3AC,0x98bd}, // <CJK>
  {0xF3AD,0x98bb}, // <CJK>
  {0xF3AE,0x98be}, // <CJK>
  {0xF3AF,0x9948}, // <CJK>
  {0xF3B0,0x9947}, // <CJK>
  {0xF3B1,0x9943}, // <CJK>
  {0xF3B2,0x99a6}, // <CJK>
  {0xF3B3,0x99a7}, // <CJK>
  {0xF3B4,0x9a1a}, // <CJK>
  {0xF3B5,0x9a15}, // <CJK>
  {0xF3B6,0x9a25}, // <CJK>
  {0xF3B7,0x9a1d}, // <CJK>
  {0xF3B8,0x9a24}, // <CJK>
  {0xF3B9,0x9a1b}, // <CJK>
  {0xF3BA,0x9a22}, // <CJK>
  {0xF3BB,0x9a20}, // <CJK>
  {0xF3BC,0x9a27}, // <CJK>
  {0xF3BD,0x9a23}, // <CJK>
  {0xF3BE,0x9a1e}, // <CJK>
  {0xF3BF,0x9a1c}, // <CJK>
  {0xF3C0,0x9a14}, // <CJK>
  {0xF3C1,0x9ac2}, // <CJK>
  {0xF3C2,0x9b0b}, // <CJK>
  {0xF3C3,0x9b0a}, // <CJK>
  {0xF3C4,0x9b0e}, // <CJK>
  {0xF3C5,0x9b0c}, // <CJK>
  {0xF3C6,0x9b37}, // <CJK>
  {0xF3C7,0x9bea}, // <CJK>
  {0xF3C8,0x9beb}, // <CJK>
  {0xF3C9,0x9be0}, // <CJK>
  {0xF3CA,0x9bde}, // <CJK>
  {0xF3CB,0x9be4}, // <CJK>
  {0xF3CC,0x9be6}, // <CJK>
  {0xF3CD,0x9be2}, // <CJK>
  {0xF3CE,0x9bf0}, // <CJK>
  {0xF3CF,0x9bd4}, // <CJK>
  {0xF3D0,0x9bd7}, // <CJK>
  {0xF3D1,0x9bec}, // <CJK>
  {0xF3D2,0x9bdc}, // <CJK>
  {0xF3D3,0x9bd9}, // <CJK>
  {0xF3D4,0x9be5}, // <CJK>
  {0xF3D5,0x9bd5}, // <CJK>
  {0xF3D6,0x9be1}, // <CJK>
  {0xF3D7,0x9bda}, // <CJK>
  {0xF3D8,0x9d77}, // <CJK>
  {0xF3D9,0x9d81}, // <CJK>
  {0xF3DA,0x9d8a}, // <CJK>
  {0xF3DB,0x9d84}, // <CJK>
  {0xF3DC,0x9d88}, // <CJK>
  {0xF3DD,0x9d71}, // <CJK>
  {0xF3DE,0x9d80}, // <CJK>
  {0xF3DF,0x9d78}, // <CJK>
  {0xF3E0,0x9d86}, // <CJK>
  {0xF3E1,0x9d8b}, // <CJK>
  {0xF3E2,0x9d8c}, // <CJK>
  {0xF3E3,0x9d7d}, // <CJK>
  {0xF3E4,0x9d6b}, // <CJK>
  {0xF3E5,0x9d74}, // <CJK>
  {0xF3E6,0x9d75}, // <CJK>
  {0xF3E7,0x9d70}, // <CJK>
  {0xF3E8,0x9d69}, // <CJK>
  {0xF3E9,0x9d85}, // <CJK>
  {0xF3EA,0x9d73}, // <CJK>
  {0xF3EB,0x9d7b}, // <CJK>
  {0xF3EC,0x9d82}, // <CJK>
  {0xF3ED,0x9d6f}, // <CJK>
  {0xF3EE,0x9d79}, // <CJK>
  {0xF3EF,0x9d7f}, // <CJK>
  {0xF3F0,0x9d87}, // <CJK>
  {0xF3F1,0x9d68}, // <CJK>
  {0xF3F2,0x9e94}, // <CJK>
  {0xF3F3,0x9e91}, // <CJK>
  {0xF3F4,0x9ec0}, // <CJK>
  {0xF3F5,0x9efc}, // <CJK>
  {0xF3F6,0x9f2d}, // <CJK>
  {0xF3F7,0x9f40}, // <CJK>
  {0xF3F8,0x9f41}, // <CJK>
  {0xF3F9,0x9f4d}, // <CJK>
  {0xF3FA,0x9f56}, // <CJK>
  {0xF3FB,0x9f57}, // <CJK>
  {0xF3FC,0x9f58}, // <CJK>
  {0xF3FD,0x5337}, // <CJK>
  {0xF3FE,0x56b2}, // <CJK>
  {0xF440,0x56b5}, // <CJK>
  {0xF441,0x56b3}, // <CJK>
  {0xF442,0x58e3}, // <CJK>
  {0xF443,0x5b45}, // <CJK>
  {0xF444,0x5dc6}, // <CJK>
  {0xF445,0x5dc7}, // <CJK>
  {0xF446,0x5eee}, // <CJK>
  {0xF447,0x5eef}, // <CJK>
  {0xF448,0x5fc0}, // <CJK>
  {0xF449,0x5fc1}, // <CJK>
  {0xF44A,0x61f9}, // <CJK>
  {0xF44B,0x6517}, // <CJK>
  {0xF44C,0x6516}, // <CJK>
  {0xF44D,0x6515}, // <CJK>
  {0xF44E,0x6513}, // <CJK>
  {0xF44F,0x65df}, // <CJK>
  {0xF450,0x66e8}, // <CJK>
  {0xF451,0x66e3}, // <CJK>
  {0xF452,0x66e4}, // <CJK>
  {0xF453,0x6af3}, // <CJK>
  {0xF454,0x6af0}, // <CJK>
  {0xF455,0x6aea}, // <CJK>
  {0xF456,0x6ae8}, // <CJK>
  {0xF457,0x6af9}, // <CJK>
  {0xF458,0x6af1}, // <CJK>
  {0xF459,0x6aee}, // <CJK>
  {0xF45A,0x6aef}, // <CJK>
  {0xF45B,0x703c}, // <CJK>
  {0xF45C,0x7035}, // <CJK>
  {0xF45D,0x702f}, // <CJK>
  {0xF45E,0x7037}, // <CJK>
  {0xF45F,0x7034}, // <CJK>
  {0xF460,0x7031}, // <CJK>
  {0xF461,0x7042}, // <CJK>
  {0xF462,0x7038}, // <CJK>
  {0xF463,0x703f}, // <CJK>
  {0xF464,0x703a}, // <CJK>
  {0xF465,0x7039}, // <CJK>
  {0xF466,0x7040}, // <CJK>
  {0xF467,0x703b}, // <CJK>
  {0xF468,0x7033}, // <CJK>
  {0xF469,0x7041}, // <CJK>
  {0xF46A,0x7213}, // <CJK>
  {0xF46B,0x7214}, // <CJK>
  {0xF46C,0x72a8}, // <CJK>
  {0xF46D,0x737d}, // <CJK>
  {0xF46E,0x737c}, // <CJK>
  {0xF46F,0x74ba}, // <CJK>
  {0xF470,0x76ab}, // <CJK>
  {0xF471,0x76aa}, // <CJK>
  {0xF472,0x76be}, // <CJK>
  {0xF473,0x76ed}, // <CJK>
  {0xF474,0x77cc}, // <CJK>
  {0xF475,0x77ce}, // <CJK>
  {0xF476,0x77cf}, // <CJK>
  {0xF477,0x77cd}, // <CJK>
  {0xF478,0x77f2}, // <CJK>
  {0xF479,0x7925}, // <CJK>
  {0xF47A,0x7923}, // <CJK>
  {0xF47B,0x7927}, // <CJK>
  {0xF47C,0x7928}, // <CJK>
  {0xF47D,0x7924}, // <CJK>
  {0xF47E,0x7929}, // <CJK>
  {0xF4A1,0x79b2}, // <CJK>
  {0xF4A2,0x7a6e}, // <CJK>
  {0xF4A3,0x7a6c}, // <CJK>
  {0xF4A4,0x7a6d}, // <CJK>
  {0xF4A5,0x7af7}, // <CJK>
  {0xF4A6,0x7c49}, // <CJK>
  {0xF4A7,0x7c48}, // <CJK>
  {0xF4A8,0x7c4a}, // <CJK>
  {0xF4A9,0x7c47}, // <CJK>
  {0xF4AA,0x7c45}, // <CJK>
  {0xF4AB,0x7cee}, // <CJK>
  {0xF4AC,0x7e7b}, // <CJK>
  {0xF4AD,0x7e7e}, // <CJK>
  {0xF4AE,0x7e81}, // <CJK>
  {0xF4AF,0x7e80}, // <CJK>
  {0xF4B0,0x7fba}, // <CJK>
  {0xF4B1,0x7fff}, // <CJK>
  {0xF4B2,0x8079}, // <CJK>
  {0xF4B3,0x81db}, // <CJK>
  {0xF4B4,0x81d9}, // <CJK>
  {0xF4B5,0x820b}, // <CJK>
  {0xF4B6,0x8268}, // <CJK>
  {0xF4B7,0x8269}, // <CJK>
  {0xF4B8,0x8622}, // <CJK>
  {0xF4B9,0x85ff}, // <CJK>
  {0xF4BA,0x8601}, // <CJK>
  {0xF4BB,0x85fe}, // <CJK>
  {0xF4BC,0x861b}, // <CJK>
  {0xF4BD,0x8600}, // <CJK>
  {0xF4BE,0x85f6}, // <CJK>
  {0xF4BF,0x8604}, // <CJK>
  {0xF4C0,0x8609}, // <CJK>
  {0xF4C1,0x8605}, // <CJK>
  {0xF4C2,0x860c}, // <CJK>
  {0xF4C3,0x85fd}, // <CJK>
  {0xF4C4,0x8819}, // <CJK>
  {0xF4C5,0x8810}, // <CJK>
  {0xF4C6,0x8811}, // <CJK>
  {0xF4C7,0x8817}, // <CJK>
  {0xF4C8,0x8813}, // <CJK>
  {0xF4C9,0x8816}, // <CJK>
  {0xF4CA,0x8963}, // <CJK>
  {0xF4CB,0x8966}, // <CJK>
  {0xF4CC,0x89b9}, // <CJK>
  {0xF4CD,0x89f7}, // <CJK>
  {0xF4CE,0x8b60}, // <CJK>
  {0xF4CF,0x8b6a}, // <CJK>
  {0xF4D0,0x8b5d}, // <CJK>
  {0xF4D1,0x8b68}, // <CJK>
  {0xF4D2,0x8b63}, // <CJK>
  {0xF4D3,0x8b65}, // <CJK>
  {0xF4D4,0x8b67}, // <CJK>
  {0xF4D5,0x8b6d}, // <CJK>
  {0xF4D6,0x8dae}, // <CJK>
  {0xF4D7,0x8e86}, // <CJK>
  {0xF4D8,0x8e88}, // <CJK>
  {0xF4D9,0x8e84}, // <CJK>
  {0xF4DA,0x8f59}, // <CJK>
  {0xF4DB,0x8f56}, // <CJK>
  {0xF4DC,0x8f57}, // <CJK>
  {0xF4DD,0x8f55}, // <CJK>
  {0xF4DE,0x8f58}, // <CJK>
  {0xF4DF,0x8f5a}, // <CJK>
  {0xF4E0,0x908d}, // <CJK>
  {0xF4E1,0x9143}, // <CJK>
  {0xF4E2,0x9141}, // <CJK>
  {0xF4E3,0x91b7}, // <CJK>
  {0xF4E4,0x91b5}, // <CJK>
  {0xF4E5,0x91b2}, // <CJK>
  {0xF4E6,0x91b3}, // <CJK>
  {0xF4E7,0x940b}, // <CJK>
  {0xF4E8,0x9413}, // <CJK>
  {0xF4E9,0x93fb}, // <CJK>
  {0xF4EA,0x9420}, // <CJK>
  {0xF4EB,0x940f}, // <CJK>
  {0xF4EC,0x9414}, // <CJK>
  {0xF4ED,0x93fe}, // <CJK>
  {0xF4EE,0x9415}, // <CJK>
  {0xF4EF,0x9410}, // <CJK>
  {0xF4F0,0x9428}, // <CJK>
  {0xF4F1,0x9419}, // <CJK>
  {0xF4F2,0x940d}, // <CJK>
  {0xF4F3,0x93f5}, // <CJK>
  {0xF4F4,0x9400}, // <CJK>
  {0xF4F5,0x93f7}, // <CJK>
  {0xF4F6,0x9407}, // <CJK>
  {0xF4F7,0x940e}, // <CJK>
  {0xF4F8,0x9416}, // <CJK>
  {0xF4F9,0x9412}, // <CJK>
  {0xF4FA,0x93fa}, // <CJK>
  {0xF4FB,0x9409}, // <CJK>
  {0xF4FC,0x93f8}, // <CJK>
  {0xF4FD,0x940a}, // <CJK>
  {0xF4FE,0x93ff}, // <CJK>
  {0xF540,0x93fc}, // <CJK>
  {0xF541,0x940c}, // <CJK>
  {0xF542,0x93f6}, // <CJK>
  {0xF543,0x9411}, // <CJK>
  {0xF544,0x9406}, // <CJK>
  {0xF545,0x95de}, // <CJK>
  {0xF546,0x95e0}, // <CJK>
  {0xF547,0x95df}, // <CJK>
  {0xF548,0x972e}, // <CJK>
  {0xF549,0x972f}, // <CJK>
  {0xF54A,0x97b9}, // <CJK>
  {0xF54B,0x97bb}, // <CJK>
  {0xF54C,0x97fd}, // <CJK>
  {0xF54D,0x97fe}, // <CJK>
  {0xF54E,0x9860}, // <CJK>
  {0xF54F,0x9862}, // <CJK>
  {0xF550,0x9863}, // <CJK>
  {0xF551,0x985f}, // <CJK>
  {0xF552,0x98c1}, // <CJK>
  {0xF553,0x98c2}, // <CJK>
  {0xF554,0x9950}, // <CJK>
  {0xF555,0x994e}, // <CJK>
  {0xF556,0x9959}, // <CJK>
  {0xF557,0x994c}, // <CJK>
  {0xF558,0x994b}, // <CJK>
  {0xF559,0x9953}, // <CJK>
  {0xF55A,0x9a32}, // <CJK>
  {0xF55B,0x9a34}, // <CJK>
  {0xF55C,0x9a31}, // <CJK>
  {0xF55D,0x9a2c}, // <CJK>
  {0xF55E,0x9a2a}, // <CJK>
  {0xF55F,0x9a36}, // <CJK>
  {0xF560,0x9a29}, // <CJK>
  {0xF561,0x9a2e}, // <CJK>
  {0xF562,0x9a38}, // <CJK>
  {0xF563,0x9a2d}, // <CJK>
  {0xF564,0x9ac7}, // <CJK>
  {0xF565,0x9aca}, // <CJK>
  {0xF566,0x9ac6}, // <CJK>
  {0xF567,0x9b10}, // <CJK>
  {0xF568,0x9b12}, // <CJK>
  {0xF569,0x9b11}, // <CJK>
  {0xF56A,0x9c0b}, // <CJK>
  {0xF56B,0x9c08}, // <CJK>
  {0xF56C,0x9bf7}, // <CJK>
  {0xF56D,0x9c05}, // <CJK>
  {0xF56E,0x9c12}, // <CJK>
  {0xF56F,0x9bf8}, // <CJK>
  {0xF570,0x9c40}, // <CJK>
  {0xF571,0x9c07}, // <CJK>
  {0xF572,0x9c0e}, // <CJK>
  {0xF573,0x9c06}, // <CJK>
  {0xF574,0x9c17}, // <CJK>
  {0xF575,0x9c14}, // <CJK>
  {0xF576,0x9c09}, // <CJK>
  {0xF577,0x9d9f}, // <CJK>
  {0xF578,0x9d99}, // <CJK>
  {0xF579,0x9da4}, // <CJK>
  {0xF57A,0x9d9d}, // <CJK>
  {0xF57B,0x9d92}, // <CJK>
  {0xF57C,0x9d98}, // <CJK>
  {0xF57D,0x9d90}, // <CJK>
  {0xF57E,0x9d9b}, // <CJK>
  {0xF5A1,0x9da0}, // <CJK>
  {0xF5A2,0x9d94}, // <CJK>
  {0xF5A3,0x9d9c}, // <CJK>
  {0xF5A4,0x9daa}, // <CJK>
  {0xF5A5,0x9d97}, // <CJK>
  {0xF5A6,0x9da1}, // <CJK>
  {0xF5A7,0x9d9a}, // <CJK>
  {0xF5A8,0x9da2}, // <CJK>
  {0xF5A9,0x9da8}, // <CJK>
  {0xF5AA,0x9d9e}, // <CJK>
  {0xF5AB,0x9da3}, // <CJK>
  {0xF5AC,0x9dbf}, // <CJK>
  {0xF5AD,0x9da9}, // <CJK>
  {0xF5AE,0x9d96}, // <CJK>
  {0xF5AF,0x9da6}, // <CJK>
  {0xF5B0,0x9da7}, // <CJK>
  {0xF5B1,0x9e99}, // <CJK>
  {0xF5B2,0x9e9b}, // <CJK>
  {0xF5B3,0x9e9a}, // <CJK>
  {0xF5B4,0x9ee5}, // <CJK>
  {0xF5B5,0x9ee4}, // <CJK>
  {0xF5B6,0x9ee7}, // <CJK>
  {0xF5B7,0x9ee6}, // <CJK>
  {0xF5B8,0x9f30}, // <CJK>
  {0xF5B9,0x9f2e}, // <CJK>
  {0xF5BA,0x9f5b}, // <CJK>
  {0xF5BB,0x9f60}, // <CJK>
  {0xF5BC,0x9f5e}, // <CJK>
  {0xF5BD,0x9f5d}, // <CJK>
  {0xF5BE,0x9f59}, // <CJK>
  {0xF5BF,0x9f91}, // <CJK>
  {0xF5C0,0x513a}, // <CJK>
  {0xF5C1,0x5139}, // <CJK>
  {0xF5C2,0x5298}, // <CJK>
  {0xF5C3,0x5297}, // <CJK>
  {0xF5C4,0x56c3}, // <CJK>
  {0xF5C5,0x56bd}, // <CJK>
  {0xF5C6,0x56be}, // <CJK>
  {0xF5C7,0x5b48}, // <CJK>
  {0xF5C8,0x5b47}, // <CJK>
  {0xF5C9,0x5dcb}, // <CJK>
  {0xF5CA,0x5dcf}, // <CJK>
  {0xF5CB,0x5ef1}, // <CJK>
  {0xF5CC,0x61fd}, // <CJK>
  {0xF5CD,0x651b}, // <CJK>
  {0xF5CE,0x6b02}, // <CJK>
  {0xF5CF,0x6afc}, // <CJK>
  {0xF5D0,0x6b03}, // <CJK>
  {0xF5D1,0x6af8}, // <CJK>
  {0xF5D2,0x6b00}, // <CJK>
  {0xF5D3,0x7043}, // <CJK>
  {0xF5D4,0x7044}, // <CJK>
  {0xF5D5,0x704a}, // <CJK>
  {0xF5D6,0x7048}, // <CJK>
  {0xF5D7,0x7049}, // <CJK>
  {0xF5D8,0x7045}, // <CJK>
  {0xF5D9,0x7046}, // <CJK>
  {0xF5DA,0x721d}, // <CJK>
  {0xF5DB,0x721a}, // <CJK>
  {0xF5DC,0x7219}, // <CJK>
  {0xF5DD,0x737e}, // <CJK>
  {0xF5DE,0x7517}, // <CJK>
  {0xF5DF,0x766a}, // <CJK>
  {0xF5E0,0x77d0}, // <CJK>
  {0xF5E1,0x792d}, // <CJK>
  {0xF5E2,0x7931}, // <CJK>
  {0xF5E3,0x792f}, // <CJK>
  {0xF5E4,0x7c54}, // <CJK>
  {0xF5E5,0x7c53}, // <CJK>
  {0xF5E6,0x7cf2}, // <CJK>
  {0xF5E7,0x7e8a}, // <CJK>
  {0xF5E8,0x7e87}, // <CJK>
  {0xF5E9,0x7e88}, // <CJK>
  {0xF5EA,0x7e8b}, // <CJK>
  {0xF5EB,0x7e86}, // <CJK>
  {0xF5EC,0x7e8d}, // <CJK>
  {0xF5ED,0x7f4d}, // <CJK>
  {0xF5EE,0x7fbb}, // <CJK>
  {0xF5EF,0x8030}, // <CJK>
  {0xF5F0,0x81dd}, // <CJK>
  {0xF5F1,0x8618}, // <CJK>
  {0xF5F2,0x862a}, // <CJK>
  {0xF5F3,0x8626}, // <CJK>
  {0xF5F4,0x861f}, // <CJK>
  {0xF5F5,0x8623}, // <CJK>
  {0xF5F6,0x861c}, // <CJK>
  {0xF5F7,0x8619}, // <CJK>
  {0xF5F8,0x8627}, // <CJK>
  {0xF5F9,0x862e}, // <CJK>
  {0xF5FA,0x8621}, // <CJK>
  {0xF5FB,0x8620}, // <CJK>
  {0xF5FC,0x8629}, // <CJK>
  {0xF5FD,0x861e}, // <CJK>
  {0xF5FE,0x8625}, // <CJK>
  {0xF640,0x8829}, // <CJK>
  {0xF641,0x881d}, // <CJK>
  {0xF642,0x881b}, // <CJK>
  {0xF643,0x8820}, // <CJK>
  {0xF644,0x8824}, // <CJK>
  {0xF645,0x881c}, // <CJK>
  {0xF646,0x882b}, // <CJK>
  {0xF647,0x884a}, // <CJK>
  {0xF648,0x896d}, // <CJK>
  {0xF649,0x8969}, // <CJK>
  {0xF64A,0x896e}, // <CJK>
  {0xF64B,0x896b}, // <CJK>
  {0xF64C,0x89fa}, // <CJK>
  {0xF64D,0x8b79}, // <CJK>
  {0xF64E,0x8b78}, // <CJK>
  {0xF64F,0x8b45}, // <CJK>
  {0xF650,0x8b7a}, // <CJK>
  {0xF651,0x8b7b}, // <CJK>
  {0xF652,0x8d10}, // <CJK>
  {0xF653,0x8d14}, // <CJK>
  {0xF654,0x8daf}, // <CJK>
  {0xF655,0x8e8e}, // <CJK>
  {0xF656,0x8e8c}, // <CJK>
  {0xF657,0x8f5e}, // <CJK>
  {0xF658,0x8f5b}, // <CJK>
  {0xF659,0x8f5d}, // <CJK>
  {0xF65A,0x9146}, // <CJK>
  {0xF65B,0x9144}, // <CJK>
  {0xF65C,0x9145}, // <CJK>
  {0xF65D,0x91b9}, // <CJK>
  {0xF65E,0x943f}, // <CJK>
  {0xF65F,0x943b}, // <CJK>
  {0xF660,0x9436}, // <CJK>
  {0xF661,0x9429}, // <CJK>
  {0xF662,0x943d}, // <CJK>
  {0xF663,0x943c}, // <CJK>
  {0xF664,0x9430}, // <CJK>
  {0xF665,0x9439}, // <CJK>
  {0xF666,0x942a}, // <CJK>
  {0xF667,0x9437}, // <CJK>
  {0xF668,0x942c}, // <CJK>
  {0xF669,0x9440}, // <CJK>
  {0xF66A,0x9431}, // <CJK>
  {0xF66B,0x95e5}, // <CJK>
  {0xF66C,0x95e4}, // <CJK>
  {0xF66D,0x95e3}, // <CJK>
  {0xF66E,0x9735}, // <CJK>
  {0xF66F,0x973a}, // <CJK>
  {0xF670,0x97bf}, // <CJK>
  {0xF671,0x97e1}, // <CJK>
  {0xF672,0x9864}, // <CJK>
  {0xF673,0x98c9}, // <CJK>
  {0xF674,0x98c6}, // <CJK>
  {0xF675,0x98c0}, // <CJK>
  {0xF676,0x9958}, // <CJK>
  {0xF677,0x9956}, // <CJK>
  {0xF678,0x9a39}, // <CJK>
  {0xF679,0x9a3d}, // <CJK>
  {0xF67A,0x9a46}, // <CJK>
  {0xF67B,0x9a44}, // <CJK>
  {0xF67C,0x9a42}, // <CJK>
  {0xF67D,0x9a41}, // <CJK>
  {0xF67E,0x9a3a}, // <CJK>
  {0xF6A1,0x9a3f}, // <CJK>
  {0xF6A2,0x9acd}, // <CJK>
  {0xF6A3,0x9b15}, // <CJK>
  {0xF6A4,0x9b17}, // <CJK>
  {0xF6A5,0x9b18}, // <CJK>
  {0xF6A6,0x9b16}, // <CJK>
  {0xF6A7,0x9b3a}, // <CJK>
  {0xF6A8,0x9b52}, // <CJK>
  {0xF6A9,0x9c2b}, // <CJK>
  {0xF6AA,0x9c1d}, // <CJK>
  {0xF6AB,0x9c1c}, // <CJK>
  {0xF6AC,0x9c2c}, // <CJK>
  {0xF6AD,0x9c23}, // <CJK>
  {0xF6AE,0x9c28}, // <CJK>
  {0xF6AF,0x9c29}, // <CJK>
  {0xF6B0,0x9c24}, // <CJK>
  {0xF6B1,0x9c21}, // <CJK>
  {0xF6B2,0x9db7}, // <CJK>
  {0xF6B3,0x9db6}, // <CJK>
  {0xF6B4,0x9dbc}, // <CJK>
  {0xF6B5,0x9dc1}, // <CJK>
  {0xF6B6,0x9dc7}, // <CJK>
  {0xF6B7,0x9dca}, // <CJK>
  {0xF6B8,0x9dcf}, // <CJK>
  {0xF6B9,0x9dbe}, // <CJK>
  {0xF6BA,0x9dc5}, // <CJK>
  {0xF6BB,0x9dc3}, // <CJK>
  {0xF6BC,0x9dbb}, // <CJK>
  {0xF6BD,0x9db5}, // <CJK>
  {0xF6BE,0x9dce}, // <CJK>
  {0xF6BF,0x9db9}, // <CJK>
  {0xF6C0,0x9dba}, // <CJK>
  {0xF6C1,0x9dac}, // <CJK>
  {0xF6C2,0x9dc8}, // <CJK>
  {0xF6C3,0x9db1}, // <CJK>
  {0xF6C4,0x9dad}, // <CJK>
  {0xF6C5,0x9dcc}, // <CJK>
  {0xF6C6,0x9db3}, // <CJK>
  {0xF6C7,0x9dcd}, // <CJK>
  {0xF6C8,0x9db2}, // <CJK>
  {0xF6C9,0x9e7a}, // <CJK>
  {0xF6CA,0x9e9c}, // <CJK>
  {0xF6CB,0x9eeb}, // <CJK>
  {0xF6CC,0x9eee}, // <CJK>
  {0xF6CD,0x9eed}, // <CJK>
  {0xF6CE,0x9f1b}, // <CJK>
  {0xF6CF,0x9f18}, // <CJK>
  {0xF6D0,0x9f1a}, // <CJK>
  {0xF6D1,0x9f31}, // <CJK>
  {0xF6D2,0x9f4e}, // <CJK>
  {0xF6D3,0x9f65}, // <CJK>
  {0xF6D4,0x9f64}, // <CJK>
  {0xF6D5,0x9f92}, // <CJK>
  {0xF6D6,0x4eb9}, // <CJK>
  {0xF6D7,0x56c6}, // <CJK>
  {0xF6D8,0x56c5}, // <CJK>
  {0xF6D9,0x56cb}, // <CJK>
  {0xF6DA,0x5971}, // <CJK>
  {0xF6DB,0x5b4b}, // <CJK>
  {0xF6DC,0x5b4c}, // <CJK>
  {0xF6DD,0x5dd5}, // <CJK>
  {0xF6DE,0x5dd1}, // <CJK>
  {0xF6DF,0x5ef2}, // <CJK>
  {0xF6E0,0x6521}, // <CJK>
  {0xF6E1,0x6520}, // <CJK>
  {0xF6E2,0x6526}, // <CJK>
  {0xF6E3,0x6522}, // <CJK>
  {0xF6E4,0x6b0b}, // <CJK>
  {0xF6E5,0x6b08}, // <CJK>
  {0xF6E6,0x6b09}, // <CJK>
  {0xF6E7,0x6c0d}, // <CJK>
  {0xF6E8,0x7055}, // <CJK>
  {0xF6E9,0x7056}, // <CJK>
  {0xF6EA,0x7057}, // <CJK>
  {0xF6EB,0x7052}, // <CJK>
  {0xF6EC,0x721e}, // <CJK>
  {0xF6ED,0x721f}, // <CJK>
  {0xF6EE,0x72a9}, // <CJK>
  {0xF6EF,0x737f}, // <CJK>
  {0xF6F0,0x74d8}, // <CJK>
  {0xF6F1,0x74d5}, // <CJK>
  {0xF6F2,0x74d9}, // <CJK>
  {0xF6F3,0x74d7}, // <CJK>
  {0xF6F4,0x766d}, // <CJK>
  {0xF6F5,0x76ad}, // <CJK>
  {0xF6F6,0x7935}, // <CJK>
  {0xF6F7,0x79b4}, // <CJK>
  {0xF6F8,0x7a70}, // <CJK>
  {0xF6F9,0x7a71}, // <CJK>
  {0xF6FA,0x7c57}, // <CJK>
  {0xF6FB,0x7c5c}, // <CJK>
  {0xF6FC,0x7c59}, // <CJK>
  {0xF6FD,0x7c5b}, // <CJK>
  {0xF6FE,0x7c5a}, // <CJK>
  {0xF740,0x7cf4}, // <CJK>
  {0xF741,0x7cf1}, // <CJK>
  {0xF742,0x7e91}, // <CJK>
  {0xF743,0x7f4f}, // <CJK>
  {0xF744,0x7f87}, // <CJK>
  {0xF745,0x81de}, // <CJK>
  {0xF746,0x826b}, // <CJK>
  {0xF747,0x8634}, // <CJK>
  {0xF748,0x8635}, // <CJK>
  {0xF749,0x8633}, // <CJK>
  {0xF74A,0x862c}, // <CJK>
  {0xF74B,0x8632}, // <CJK>
  {0xF74C,0x8636}, // <CJK>
  {0xF74D,0x882c}, // <CJK>
  {0xF74E,0x8828}, // <CJK>
  {0xF74F,0x8826}, // <CJK>
  {0xF750,0x882a}, // <CJK>
  {0xF751,0x8825}, // <CJK>
  {0xF752,0x8971}, // <CJK>
  {0xF753,0x89bf}, // <CJK>
  {0xF754,0x89be}, // <CJK>
  {0xF755,0x89fb}, // <CJK>
  {0xF756,0x8b7e}, // <CJK>
  {0xF757,0x8b84}, // <CJK>
  {0xF758,0x8b82}, // <CJK>
  {0xF759,0x8b86}, // <CJK>
  {0xF75A,0x8b85}, // <CJK>
  {0xF75B,0x8b7f}, // <CJK>
  {0xF75C,0x8d15}, // <CJK>
  {0xF75D,0x8e95}, // <CJK>
  {0xF75E,0x8e94}, // <CJK>
  {0xF75F,0x8e9a}, // <CJK>
  {0xF760,0x8e92}, // <CJK>
  {0xF761,0x8e90}, // <CJK>
  {0xF762,0x8e96}, // <CJK>
  {0xF763,0x8e97}, // <CJK>
  {0xF764,0x8f60}, // <CJK>
  {0xF765,0x8f62}, // <CJK>
  {0xF766,0x9147}, // <CJK>
  {0xF767,0x944c}, // <CJK>
  {0xF768,0x9450}, // <CJK>
  {0xF769,0x944a}, // <CJK>
  {0xF76A,0x944b}, // <CJK>
  {0xF76B,0x944f}, // <CJK>
  {0xF76C,0x9447}, // <CJK>
  {0xF76D,0x9445}, // <CJK>
  {0xF76E,0x9448}, // <CJK>
  {0xF76F,0x9449}, // <CJK>
  {0xF770,0x9446}, // <CJK>
  {0xF771,0x973f}, // <CJK>
  {0xF772,0x97e3}, // <CJK>
  {0xF773,0x986a}, // <CJK>
  {0xF774,0x9869}, // <CJK>
  {0xF775,0x98cb}, // <CJK>
  {0xF776,0x9954}, // <CJK>
  {0xF777,0x995b}, // <CJK>
  {0xF778,0x9a4e}, // <CJK>
  {0xF779,0x9a53}, // <CJK>
  {0xF77A,0x9a54}, // <CJK>
  {0xF77B,0x9a4c}, // <CJK>
  {0xF77C,0x9a4f}, // <CJK>
  {0xF77D,0x9a48}, // <CJK>
  {0xF77E,0x9a4a}, // <CJK>
  {0xF7A1,0x9a49}, // <CJK>
  {0xF7A2,0x9a52}, // <CJK>
  {0xF7A3,0x9a50}, // <CJK>
  {0xF7A4,0x9ad0}, // <CJK>
  {0xF7A5,0x9b19}, // <CJK>
  {0xF7A6,0x9b2b}, // <CJK>
  {0xF7A7,0x9b3b}, // <CJK>
  {0xF7A8,0x9b56}, // <CJK>
  {0xF7A9,0x9b55}, // <CJK>
  {0xF7AA,0x9c46}, // <CJK>
  {0xF7AB,0x9c48}, // <CJK>
  {0xF7AC,0x9c3f}, // <CJK>
  {0xF7AD,0x9c44}, // <CJK>
  {0xF7AE,0x9c39}, // <CJK>
  {0xF7AF,0x9c33}, // <CJK>
  {0xF7B0,0x9c41}, // <CJK>
  {0xF7B1,0x9c3c}, // <CJK>
  {0xF7B2,0x9c37}, // <CJK>
  {0xF7B3,0x9c34}, // <CJK>
  {0xF7B4,0x9c32}, // <CJK>
  {0xF7B5,0x9c3d}, // <CJK>
  {0xF7B6,0x9c36}, // <CJK>
  {0xF7B7,0x9ddb}, // <CJK>
  {0xF7B8,0x9dd2}, // <CJK>
  {0xF7B9,0x9dde}, // <CJK>
  {0xF7BA,0x9dda}, // <CJK>
  {0xF7BB,0x9dcb}, // <CJK>
  {0xF7BC,0x9dd0}, // <CJK>
  {0xF7BD,0x9ddc}, // <CJK>
  {0xF7BE,0x9dd1}, // <CJK>
  {0xF7BF,0x9ddf}, // <CJK>
  {0xF7C0,0x9de9}, // <CJK>
  {0xF7C1,0x9dd9}, // <CJK>
  {0xF7C2,0x9dd8}, // <CJK>
  {0xF7C3,0x9dd6}, // <CJK>
  {0xF7C4,0x9df5}, // <CJK>
  {0xF7C5,0x9dd5}, // <CJK>
  {0xF7C6,0x9ddd}, // <CJK>
  {0xF7C7,0x9eb6}, // <CJK>
  {0xF7C8,0x9ef0}, // <CJK>
  {0xF7C9,0x9f35}, // <CJK>
  {0xF7CA,0x9f33}, // <CJK>
  {0xF7CB,0x9f32}, // <CJK>
  {0xF7CC,0x9f42}, // <CJK>
  {0xF7CD,0x9f6b}, // <CJK>
  {0xF7CE,0x9f95}, // <CJK>
  {0xF7CF,0x9fa2}, // <CJK>
  {0xF7D0,0x513d}, // <CJK>
  {0xF7D1,0x5299}, // <CJK>
  {0xF7D2,0x58e8}, // <CJK>
  {0xF7D3,0x58e7}, // <CJK>
  {0xF7D4,0x5972}, // <CJK>
  {0xF7D5,0x5b4d}, // <CJK>
  {0xF7D6,0x5dd8}, // <CJK>
  {0xF7D7,0x882f}, // <CJK>
  {0xF7D8,0x5f4f}, // <CJK>
  {0xF7D9,0x6201}, // <CJK>
  {0xF7DA,0x6203}, // <CJK>
  {0xF7DB,0x6204}, // <CJK>
  {0xF7DC,0x6529}, // <CJK>
  {0xF7DD,0x6525}, // <CJK>
  {0xF7DE,0x6596}, // <CJK>
  {0xF7DF,0x66eb}, // <CJK>
  {0xF7E0,0x6b11}, // <CJK>
  {0xF7E1,0x6b12}, // <CJK>
  {0xF7E2,0x6b0f}, // <CJK>
  {0xF7E3,0x6bca}, // <CJK>
  {0xF7E4,0x705b}, // <CJK>
  {0xF7E5,0x705a}, // <CJK>
  {0xF7E6,0x7222}, // <CJK>
  {0xF7E7,0x7382}, // <CJK>
  {0xF7E8,0x7381}, // <CJK>
  {0xF7E9,0x7383}, // <CJK>
  {0xF7EA,0x7670}, // <CJK>
  {0xF7EB,0x77d4}, // <CJK>
  {0xF7EC,0x7c67}, // <CJK>
  {0xF7ED,0x7c66}, // <CJK>
  {0xF7EE,0x7e95}, // <CJK>
  {0xF7EF,0x826c}, // <CJK>
  {0xF7F0,0x863a}, // <CJK>
  {0xF7F1,0x8640}, // <CJK>
  {0xF7F2,0x8639}, // <CJK>
  {0xF7F3,0x863c}, // <CJK>
  {0xF7F4,0x8631}, // <CJK>
  {0xF7F5,0x863b}, // <CJK>
  {0xF7F6,0x863e}, // <CJK>
  {0xF7F7,0x8830}, // <CJK>
  {0xF7F8,0x8832}, // <CJK>
  {0xF7F9,0x882e}, // <CJK>
  {0xF7FA,0x8833}, // <CJK>
  {0xF7FB,0x8976}, // <CJK>
  {0xF7FC,0x8974}, // <CJK>
  {0xF7FD,0x8973}, // <CJK>
  {0xF7FE,0x89fe}, // <CJK>
  {0xF840,0x8b8c}, // <CJK>
  {0xF841,0x8b8e}, // <CJK>
  {0xF842,0x8b8b}, // <CJK>
  {0xF843,0x8b88}, // <CJK>
  {0xF844,0x8c45}, // <CJK>
  {0xF845,0x8d19}, // <CJK>
  {0xF846,0x8e98}, // <CJK>
  {0xF847,0x8f64}, // <CJK>
  {0xF848,0x8f63}, // <CJK>
  {0xF849,0x91bc}, // <CJK>
  {0xF84A,0x9462}, // <CJK>
  {0xF84B,0x9455}, // <CJK>
  {0xF84C,0x945d}, // <CJK>
  {0xF84D,0x9457}, // <CJK>
  {0xF84E,0x945e}, // <CJK>
  {0xF84F,0x97c4}, // <CJK>
  {0xF850,0x97c5}, // <CJK>
  {0xF851,0x9800}, // <CJK>
  {0xF852,0x9a56}, // <CJK>
  {0xF853,0x9a59}, // <CJK>
  {0xF854,0x9b1e}, // <CJK>
  {0xF855,0x9b1f}, // <CJK>
  {0xF856,0x9b20}, // <CJK>
  {0xF857,0x9c52}, // <CJK>
  {0xF858,0x9c58}, // <CJK>
  {0xF859,0x9c50}, // <CJK>
  {0xF85A,0x9c4a}, // <CJK>
  {0xF85B,0x9c4d}, // <CJK>
  {0xF85C,0x9c4b}, // <CJK>
  {0xF85D,0x9c55}, // <CJK>
  {0xF85E,0x9c59}, // <CJK>
  {0xF85F,0x9c4c}, // <CJK>
  {0xF860,0x9c4e}, // <CJK>
  {0xF861,0x9dfb}, // <CJK>
  {0xF862,0x9df7}, // <CJK>
  {0xF863,0x9def}, // <CJK>
  {0xF864,0x9de3}, // <CJK>
  {0xF865,0x9deb}, // <CJK>
  {0xF866,0x9df8}, // <CJK>
  {0xF867,0x9de4}, // <CJK>
  {0xF868,0x9df6}, // <CJK>
  {0xF869,0x9de1}, // <CJK>
  {0xF86A,0x9dee}, // <CJK>
  {0xF86B,0x9de6}, // <CJK>
  {0xF86C,0x9df2}, // <CJK>
  {0xF86D,0x9df0}, // <CJK>
  {0xF86E,0x9de2}, // <CJK>
  {0xF86F,0x9dec}, // <CJK>
  {0xF870,0x9df4}, // <CJK>
  {0xF871,0x9df3}, // <CJK>
  {0xF872,0x9de8}, // <CJK>
  {0xF873,0x9ded}, // <CJK>
  {0xF874,0x9ec2}, // <CJK>
  {0xF875,0x9ed0}, // <CJK>
  {0xF876,0x9ef2}, // <CJK>
  {0xF877,0x9ef3}, // <CJK>
  {0xF878,0x9f06}, // <CJK>
  {0xF879,0x9f1c}, // <CJK>
  {0xF87A,0x9f38}, // <CJK>
  {0xF87B,0x9f37}, // <CJK>
  {0xF87C,0x9f36}, // <CJK>
  {0xF87D,0x9f43}, // <CJK>
  {0xF87E,0x9f4f}, // <CJK>
  {0xF8A1,0x9f71}, // <CJK>
  {0xF8A2,0x9f70}, // <CJK>
  {0xF8A3,0x9f6e}, // <CJK>
  {0xF8A4,0x9f6f}, // <CJK>
  {0xF8A5,0x56d3}, // <CJK>
  {0xF8A6,0x56cd}, // <CJK>
  {0xF8A7,0x5b4e}, // <CJK>
  {0xF8A8,0x5c6d}, // <CJK>
  {0xF8A9,0x652d}, // <CJK>
  {0xF8AA,0x66ed}, // <CJK>
  {0xF8AB,0x66ee}, // <CJK>
  {0xF8AC,0x6b13}, // <CJK>
  {0xF8AD,0x705f}, // <CJK>
  {0xF8AE,0x7061}, // <CJK>
  {0xF8AF,0x705d}, // <CJK>
  {0xF8B0,0x7060}, // <CJK>
  {0xF8B1,0x7223}, // <CJK>
  {0xF8B2,0x74db}, // <CJK>
  {0xF8B3,0x74e5}, // <CJK>
  {0xF8B4,0x77d5}, // <CJK>
  {0xF8B5,0x7938}, // <CJK>
  {0xF8B6,0x79b7}, // <CJK>
  {0xF8B7,0x79b6}, // <CJK>
  {0xF8B8,0x7c6a}, // <CJK>
  {0xF8B9,0x7e97}, // <CJK>
  {0xF8BA,0x7f89}, // <CJK>
  {0xF8BB,0x826d}, // <CJK>
  {0xF8BC,0x8643}, // <CJK>
  {0xF8BD,0x8838}, // <CJK>
  {0xF8BE,0x8837}, // <CJK>
  {0xF8BF,0x8835}, // <CJK>
  {0xF8C0,0x884b}, // <CJK>
  {0xF8C1,0x8b94}, // <CJK>
  {0xF8C2,0x8b95}, // <CJK>
  {0xF8C3,0x8e9e}, // <CJK>
  {0xF8C4,0x8e9f}, // <CJK>
  {0xF8C5,0x8ea0}, // <CJK>
  {0xF8C6,0x8e9d}, // <CJK>
  {0xF8C7,0x91be}, // <CJK>
  {0xF8C8,0x91bd}, // <CJK>
  {0xF8C9,0x91c2}, // <CJK>
  {0xF8CA,0x946b}, // <CJK>
  {0xF8CB,0x9468}, // <CJK>
  {0xF8CC,0x9469}, // <CJK>
  {0xF8CD,0x96e5}, // <CJK>
  {0xF8CE,0x9746}, // <CJK>
  {0xF8CF,0x9743}, // <CJK>
  {0xF8D0,0x9747}, // <CJK>
  {0xF8D1,0x97c7}, // <CJK>
  {0xF8D2,0x97e5}, // <CJK>
  {0xF8D3,0x9a5e}, // <CJK>
  {0xF8D4,0x9ad5}, // <CJK>
  {0xF8D5,0x9b59}, // <CJK>
  {0xF8D6,0x9c63}, // <CJK>
  {0xF8D7,0x9c67}, // <CJK>
  {0xF8D8,0x9c66}, // <CJK>
  {0xF8D9,0x9c62}, // <CJK>
  {0xF8DA,0x9c5e}, // <CJK>
  {0xF8DB,0x9c60}, // <CJK>
  {0xF8DC,0x9e02}, // <CJK>
  {0xF8DD,0x9dfe}, // <CJK>
  {0xF8DE,0x9e07}, // <CJK>
  {0xF8DF,0x9e03}, // <CJK>
  {0xF8E0,0x9e06}, // <CJK>
  {0xF8E1,0x9e05}, // <CJK>
  {0xF8E2,0x9e00}, // <CJK>
  {0xF8E3,0x9e01}, // <CJK>
  {0xF8E4,0x9e09}, // <CJK>
  {0xF8E5,0x9dff}, // <CJK>
  {0xF8E6,0x9dfd}, // <CJK>
  {0xF8E7,0x9e04}, // <CJK>
  {0xF8E8,0x9ea0}, // <CJK>
  {0xF8E9,0x9f1e}, // <CJK>
  {0xF8EA,0x9f46}, // <CJK>
  {0xF8EB,0x9f74}, // <CJK>
  {0xF8EC,0x9f75}, // <CJK>
  {0xF8ED,0x9f76}, // <CJK>
  {0xF8EE,0x56d4}, // <CJK>
  {0xF8EF,0x652e}, // <CJK>
  {0xF8F0,0x65b8}, // <CJK>
  {0xF8F1,0x6b18}, // <CJK>
  {0xF8F2,0x6b19}, // <CJK>
  {0xF8F3,0x6b17}, // <CJK>
  {0xF8F4,0x6b1a}, // <CJK>
  {0xF8F5,0x7062}, // <CJK>
  {0xF8F6,0x7226}, // <CJK>
  {0xF8F7,0x72aa}, // <CJK>
  {0xF8F8,0x77d8}, // <CJK>
  {0xF8F9,0x77d9}, // <CJK>
  {0xF8FA,0x7939}, // <CJK>
  {0xF8FB,0x7c69}, // <CJK>
  {0xF8FC,0x7c6b}, // <CJK>
  {0xF8FD,0x7cf6}, // <CJK>
  {0xF8FE,0x7e9a}, // <CJK>
  {0xF940,0x7e98}, // <CJK>
  {0xF941,0x7e9b}, // <CJK>
  {0xF942,0x7e99}, // <CJK>
  {0xF943,0x81e0}, // <CJK>
  {0xF944,0x81e1}, // <CJK>
  {0xF945,0x8646}, // <CJK>
  {0xF946,0x8647}, // <CJK>
  {0xF947,0x8648}, // <CJK>
  {0xF948,0x8979}, // <CJK>
  {0xF949,0x897a}, // <CJK>
  {0xF94A,0x897c}, // <CJK>
  {0xF94B,0x897b}, // <CJK>
  {0xF94C,0x89ff}, // <CJK>
  {0xF94D,0x8b98}, // <CJK>
  {0xF94E,0x8b99}, // <CJK>
  {0xF94F,0x8ea5}, // <CJK>
  {0xF950,0x8ea4}, // <CJK>
  {0xF951,0x8ea3}, // <CJK>
  {0xF952,0x946e}, // <CJK>
  {0xF953,0x946d}, // <CJK>
  {0xF954,0x946f}, // <CJK>
  {0xF955,0x9471}, // <CJK>
  {0xF956,0x9473}, // <CJK>
  {0xF957,0x9749}, // <CJK>
  {0xF958,0x9872}, // <CJK>
  {0xF959,0x995f}, // <CJK>
  {0xF95A,0x9c68}, // <CJK>
  {0xF95B,0x9c6e}, // <CJK>
  {0xF95C,0x9c6d}, // <CJK>
  {0xF95D,0x9e0b}, // <CJK>
  {0xF95E,0x9e0d}, // <CJK>
  {0xF95F,0x9e10}, // <CJK>
  {0xF960,0x9e0f}, // <CJK>
  {0xF961,0x9e12}, // <CJK>
  {0xF962,0x9e11}, // <CJK>
  {0xF963,0x9ea1}, // <CJK>
  {0xF964,0x9ef5}, // <CJK>
  {0xF965,0x9f09}, // <CJK>
  {0xF966,0x9f47}, // <CJK>
  {0xF967,0x9f78}, // <CJK>
  {0xF968,0x9f7b}, // <CJK>
  {0xF969,0x9f7a}, // <CJK>
  {0xF96A,0x9f79}, // <CJK>
  {0xF96B,0x571e}, // <CJK>
  {0xF96C,0x7066}, // <CJK>
  {0xF96D,0x7c6f}, // <CJK>
  {0xF96E,0x883c}, // <CJK>
  {0xF96F,0x8db2}, // <CJK>
  {0xF970,0x8ea6}, // <CJK>
  {0xF971,0x91c3}, // <CJK>
  {0xF972,0x9474}, // <CJK>
  {0xF973,0x9478}, // <CJK>
  {0xF974,0x9476}, // <CJK>
  {0xF975,0x9475}, // <CJK>
  {0xF976,0x9a60}, // <CJK>
  {0xF977,0x9c74}, // <CJK>
  {0xF978,0x9c73}, // <CJK>
  {0xF979,0x9c71}, // <CJK>
  {0xF97A,0x9c75}, // <CJK>
  {0xF97B,0x9e14}, // <CJK>
  {0xF97C,0x9e13}, // <CJK>
  {0xF97D,0x9ef6}, // <CJK>
  {0xF97E,0x9f0a}, // <CJK>
  {0xF9A1,0x9fa4}, // <CJK>
  {0xF9A2,0x7068}, // <CJK>
  {0xF9A3,0x7065}, // <CJK>
  {0xF9A4,0x7cf7}, // <CJK>
  {0xF9A5,0x866a}, // <CJK>
  {0xF9A6,0x883e}, // <CJK>
  {0xF9A7,0x883d}, // <CJK>
  {0xF9A8,0x883f}, // <CJK>
  {0xF9A9,0x8b9e}, // <CJK>
  {0xF9AA,0x8c9c}, // <CJK>
  {0xF9AB,0x8ea9}, // <CJK>
  {0xF9AC,0x8ec9}, // <CJK>
  {0xF9AD,0x974b}, // <CJK>
  {0xF9AE,0x9873}, // <CJK>
  {0xF9AF,0x9874}, // <CJK>
  {0xF9B0,0x98cc}, // <CJK>
  {0xF9B1,0x9961}, // <CJK>
  {0xF9B2,0x99ab}, // <CJK>
  {0xF9B3,0x9a64}, // <CJK>
  {0xF9B4,0x9a66}, // <CJK>
  {0xF9B5,0x9a67}, // <CJK>
  {0xF9B6,0x9b24}, // <CJK>
  {0xF9B7,0x9e15}, // <CJK>
  {0xF9B8,0x9e17}, // <CJK>
  {0xF9B9,0x9f48}, // <CJK>
  {0xF9BA,0x6207}, // <CJK>
  {0xF9BB,0x6b1e}, // <CJK>
  {0xF9BC,0x7227}, // <CJK>
  {0xF9BD,0x864c}, // <CJK>
  {0xF9BE,0x8ea8}, // <CJK>
  {0xF9BF,0x9482}, // <CJK>
  {0xF9C0,0x9480}, // <CJK>
  {0xF9C1,0x9481}, // <CJK>
  {0xF9C2,0x9a69}, // <CJK>
  {0xF9C3,0x9a68}, // <CJK>
  {0xF9C4,0x9b2e}, // <CJK>
  {0xF9C5,0x9e19}, // <CJK>
  {0xF9C6,0x7229}, // <CJK>
  {0xF9C7,0x864b}, // <CJK>
  {0xF9C8,0x8b9f}, // <CJK>
  {0xF9C9,0x9483}, // <CJK>
  {0xF9CA,0x9c79}, // <CJK>
  {0xF9CB,0x9eb7}, // <CJK>
  {0xF9CC,0x7675}, // <CJK>
  {0xF9CD,0x9a6b}, // <CJK>
  {0xF9CE,0x9c7a}, // <CJK>
  {0xF9CF,0x9e1d}, // <CJK>
  {0xF9D0,0x7069}, // <CJK>
  {0xF9D1,0x706a}, // <CJK>
  {0xF9D2,0x9ea4}, // <CJK>
  {0xF9D3,0x9f7e}, // <CJK>
  {0xF9D4,0x9f49}, // <CJK>
  {0xF9D5,0x9f98}, // <CJK>
  {0xF9D6,0x7881}, // <CJK> [MOD20210517] Windows code page 950 addition
  {0xF9D7,0x92b9}, // <CJK> [MOD20210517] Windows code page 950 addition
  {0xF9D8,0x88cf}, // <CJK> [MOD20210517] Windows code page 950 addition
  {0xF9D9,0x58bb}, // <CJK> [MOD20210517] Windows code page 950 addition
  {0xF9DA,0x6052}, // <CJK> [MOD20210517] Windows code page 950 addition
  {0xF9DB,0x7ca7}, // <CJK> [MOD20210517] Windows code page 950 addition
  {0xF9DC,0x5afa}  // <CJK> [MOD20210517] Windows code page 950 addition
  };

  static ON_SleepLock lock;
  static ON_SimpleArray< ON_Big5UnicodePair > big5_to_unicode;

  if (big5_to_unicode.Capacity() > 0)
  {
    // big5_to_unicode[] is initialized.
    return big5_to_unicode;
  }

  const ON_SleepLockGuard getlock(lock);
  if (big5_to_unicode.Capacity() > 0)
  {
    // by the time we got the lock another thread had initialized big5_to_unicode[].
    return big5_to_unicode; 
  }

  // thread safe one time initialization of big5_to_unicode[]
  const int pairs_count = (int)(sizeof(pairs) / (2 * sizeof(pairs[0][0])));
  big5_to_unicode.Reserve(pairs_count);
  unsigned prev_big5_code_point = 0;
  for (int i = 0; i < pairs_count; ++i)
  {
    const ON_Big5UnicodePair pair = ON_Big5UnicodePair::Create(ON_Big5CodePoint::Create(pairs[i][0]), ON_UnicodeShortCodePoint::Create(pairs[i][1]));
    const unsigned big5_code_point = pair.Big5CodePoint();
    if (pair.IsValid(false, false) && big5_code_point > prev_big5_code_point)
    {
      prev_big5_code_point = big5_code_point;
      big5_to_unicode.Append(pair);
    }
    else
    {
      // Bug introduced after May 2021.
      ON_ERROR("pairs[][2] is no longer valid.");
    }
  }

  return big5_to_unicode;
}

const ON_SimpleArray< ON_Big5UnicodePair >& ON_Big5UnicodePair::UnicodeToBig5()
{
  static ON_SleepLock lock;
  static ON_SimpleArray< ON_Big5UnicodePair > unicode_to_big5;

  if (unicode_to_big5.Capacity() > 0)
    return unicode_to_big5;


  const ON_SleepLockGuard getlock(lock);
  if (unicode_to_big5.Capacity() > 0)
    return unicode_to_big5;

  const ON_SimpleArray< ON_Big5UnicodePair >& big5_to_unicode = ON_Big5UnicodePair::Big5ToUnicode();
  const int count0 = big5_to_unicode.Count();
  if (count0 <= 0)
    return unicode_to_big5;

  ON_MemoryAllocationTracking debug_memory_tracking(false);


  const ON_Big5UnicodePair u_big5_pair_fix[][2] =
  {
    // big5_to_unicode[] maps BIG5 code points 0xA2CC and 0xA451 to U+5341.
    // U+5341 is a <CJK> character that looks like U+3038 HANGZHOU NUMERAL TEN
    // To make unicode_to_big5[] a well defined Unicode to BIG5 map,
    // (0xA2CC,U+5341) is replaced with (0xA2CC,U+3038).
    {ON_Big5UnicodePair::Create(0xA2CC,0x5341),ON_Big5UnicodePair::Create(0xA2CC,0x3038)},
    
    // big5_to_unicode[] maps BIG5 code points 0xA2CE and 0xA4CA to U+5345.
    // U+5345 is a <CJK> character that looks like U+303A HANGZHOU NUMERAL THIRTY
    // To make unicode_to_big5[] a well defined Unicode to BIG5 map,
    // (0xA2CE,U+5345) is replaced with (0xA2CE,U+303A).
    {ON_Big5UnicodePair::Create(0xA2CE,0x5345),ON_Big5UnicodePair::Create(0xA2CE,0x303A)},
  };

  const ON_Big5UnicodePair u_big5_pair_add[] =
  {
    // big5_to_unicode[] maps BIG5 code point 0xA2CD to U+5344.
    // U+5344 is a <CJK> character that looks like U+3039 HANGZHOU NUMERAL TWENTY
    // (0xA2CD,U+3039) is added to unicode_to_big5[].
    ON_Big5UnicodePair::Create(0xA2CD,0x3039),
  };

  unicode_to_big5.Reserve(count0 + ((int)(sizeof(u_big5_pair_add)/sizeof(u_big5_pair_add[0]))));
  unicode_to_big5.Append(count0, big5_to_unicode.Array());

  // Dictionary sort unicode_to_big5[] by Unicode then BIG5 code point.
  // (There are typically duplicate Unicode code points.)
  unicode_to_big5.QuickSortAndRemoveDuplicates(ON_Big5UnicodePair::CompareUnicodeAndBig5CodePoints);
  if (count0 != unicode_to_big5.Count())
  {
    ON_ERROR("Input parameter ON_Big5UnicodePair::Big5ToUnicode() returned corrupt information.");
  }

  // Locate the elements in unicode_to_big5[] where the fixes will be made.
  int fixdex[sizeof(u_big5_pair_fix) / sizeof(u_big5_pair_fix[0])] = {};
  for (size_t i = 0; i < sizeof(u_big5_pair_fix) / sizeof(u_big5_pair_fix[0]); ++i)
  {
    const ON_Big5UnicodePair key = u_big5_pair_fix[i][0];
    const ON_Big5UnicodePair fix = u_big5_pair_fix[i][1];
    fixdex[i]
      = (key.Unicode() != fix.Unicode() && key.Big5() == fix.Big5())
      ? unicode_to_big5.BinarySearch(&key, ON_Big5UnicodePair::CompareUnicodeAndBig5CodePoints)
      : -1;
  }

  // Fix those elements (after the first fix unicode_to_big5[] is no longer sorted).
  for (size_t i = 0; i < sizeof(u_big5_pair_fix) / sizeof(u_big5_pair_fix[0]); ++i)
  {
    const ON_Big5UnicodePair key = u_big5_pair_fix[i][0];
    const ON_Big5UnicodePair fix = u_big5_pair_fix[i][1];
    const int j = fixdex[i];
    if (j >= 0 && key == unicode_to_big5[j])
      unicode_to_big5[j] = fix;
    else
    {
      ON_ERROR("skipping a fix.");
    }
  }

  // Add additional UNICODE to BIG5 mappings.
  unicode_to_big5.Append((int)(sizeof(u_big5_pair_add) / sizeof(u_big5_pair_add[0])), u_big5_pair_add);

  // Dictionary sort the fixed and expanded unicodeto_big5[]
  const unsigned count1 = unicode_to_big5.UnsignedCount();
  unicode_to_big5.QuickSortAndRemoveDuplicates(ON_Big5UnicodePair::CompareUnicodeCodePoint);
  if (count1 != unicode_to_big5.UnsignedCount())
  {
    ON_ERROR("Invalid fixes or additions to unicodeto_big5[].");
  }

  return unicode_to_big5;
}
