import Component from '@glimmer/component';
import type { FilterFieldControl } from '../filter-bar';
import type { Filter } from '../../../utils/filters';
export interface FilterControlGroupSignature {
    Args: {
        controls: FilterFieldControl[];
        appliedFilters: Record<string, Filter>;
        onUpdateFilters: (filters: Record<string, Filter>) => void;
    };
    Blocks: {
        default: [];
    };
    Element: HTMLElement;
}
/**
 * Component for rendering a group of filter controls in the filter bar.
 * Handles staging and applying filters, and supports dynamic control visibility
 * based on dependencies.
 *
 * @class FilterControlGroup
 * @extends Component
 * @typedef {object} Args
 * @property {FilterFieldControl[]} controls - List of filter controls to render.
 * @property {Record<string, Filter>} appliedFilters - Currently applied filters.
 * @property {(filters: Record<string, Filter>) => void} onUpdateFilters - Callback when filters are updated.
 */
export default class FilterControlGroup extends Component<FilterControlGroupSignature> {
    /**
     * Filters staged by the user via form changes, but not yet applied.
     * Used to determine which controls are visible and to preview filter changes.
     */
    stagedFilters?: Record<string, Filter>;
    /**
     * Builds a new filters object from the provided form data.
     * Handles different filter types (list, lookback, single-select, boolean)
     * and merges with the currently applied filters.
     *
     * @param {FormData} [formData] - Form data from the filter controls form.
     * @returns {Record<string, Filter>} - The reconciled filters object.
     */
    reconcileControlGroupFormData(formData?: FormData): Record<string, Filter<Record<string, unknown>>>;
    /**
     * Handles form submission for the filter controls.
     * Prevents default form behavior, reconciles filters from form data,
     * and invokes the onUpdateFilters callback with the new filters.
     *
     * @param {Event} event - The form submit event.
     */
    handleSubmit: (event: Event) => void;
    /**
     * Handles form change events for the filter controls.
     * Updates the stagedFilters property to reflect the current form state,
     * which can be used to check if conditional fields can be rendered.
     *
     * @param {Event} event - The form change event.
     */
    handleChange: (event: Event) => void;
    /**
     * Returns the list of filter controls to render, filtered by their dependencies.
     * If stagedFilters are present, uses them to determine visibility; otherwise uses appliedFilters
     * to get initial state.
     *
     * @returns {FilterFieldControl[]} - The visible filter controls.
     */
    get controls(): FilterFieldControl<Record<string, unknown>>[];
}
