/*	$Id: login_totp.c,v 1.6 2012/11/28 22:53:00 sthen Exp $ */
/*
 * Copyright (c) 2011, 2012 Stuart Henderson <sthen@openbsd.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/types.h>
#include <sys/resource.h>
#include <sys/param.h>
#include <sys/stat.h>

#include <errno.h>
#include <signal.h>
#include <syslog.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <login_cap.h>
#include <bsd_auth.h>
#include <stdio.h>
#include <string.h>
#include <pwd.h>
#include <err.h>
#include <util.h>

#include <oath.h>

#include "common.h"

/* login.conf strings */
#define CAP_OATHDIR		"x-oath-dir"
#define CAP_START		"x-start-offset"
#define CAP_STEP		"x-step-size"
#define CAP_WINDOW		"x-window"

#define SECRET_MAX		64444

extern int errno;

int
check_file_secrecy(int fd, const char *fname, uid_t pw_uid)
{
	struct stat	 st;

	if (fstat(fd, &st)) {
		syslog(LOG_ERR, "cannot stat %s: %s", fname, strerror(errno));
		return (-1);
	}
	if (st.st_uid != 0 && st.st_uid != pw_uid) {
		syslog(LOG_ERR, "%s: owner not root or current user", fname);
		return (-1);
	}
	if (st.st_mode & (S_IRWXG | S_IRWXO)) {
		syslog(LOG_ERR, "%s: group/world readable/writeable", fname);
		return (-1);
	}
	return (0);
}

int
oath_login(char *username, char *class, char *otp)
{
	login_cap_t	*lc;
	struct passwd	*pe;

	FILE		*fp;
	char		 fname[PATH_MAX];

	int		 ret, r = 1;
	char		 secret_bin[SECRET_MAX], *s;
	char		 secret[SECRET_MAX*2]; /* hex encoded */
	size_t		 secret_length=SECRET_MAX;
	char		 invokinguser[MAXLOGNAME];
	time_t		 start_offset = OATH_TOTP_DEFAULT_START_TIME;
	unsigned	 time_step_size = OATH_TOTP_DEFAULT_TIME_STEP_SIZE;
	const char	*errstr;
	size_t		 window = 0;

	invokinguser[0] = '\0';

	setpriority(PRIO_PROCESS, 0, 0);

	openlog(NULL, LOG_ODELAY, LOG_AUTH);

	if (username == NULL)
		return (AUTH_FAILED);

	ret = AUTH_FAILED;

	lc = login_getclass(class);
	if (lc == NULL) {
		syslog(LOG_ERR, "couldn't find %s in login.conf", class);
		exit(1);
	}

	s = login_getcapstr(lc, CAP_START, NULL, NULL);
	if (s != NULL) {
	        start_offset = strtonum(s, 0, INT_MAX, &errstr);
		if (errstr) {
			errx(1, "The start offset is %s: %s", errstr, s);
		}
	}

	s = login_getcapstr(lc, CAP_STEP, NULL, NULL);
	if (s != NULL) {
	        time_step_size = strtonum(s, 0, UINT_MAX, &errstr);
		if (errstr)
			errx(1, "The step size is %s: %s", errstr, s);
	}

	s = login_getcapstr(lc, CAP_WINDOW, NULL, NULL);
	if (s != NULL) {
	        window = strtonum(s, 0, UINT_MAX, &errstr);
		if (errstr)
			errx(1, "The window size is %s: %s", errstr, s);
	}

	pe = getpwnam(username);
	if (pe == NULL) {
		syslog(LOG_ERR, "unknown user: %s", username);
		exit(1);
	}

	if (oath_check_version(OATH_VERSION) == NULL) {
		syslog(LOG_ERR, "liboath version mismatch\n");
		exit(1);
	}

	r = oath_init();
	if (r != 0) {
		syslog(LOG_ERR, "failed to initialize: %d\n", r);
		exit(1);
	}

	s = login_getcapstr(lc, CAP_OATHDIR, NULL, NULL);
	if (s == NULL) {
		/* oathdir not set, use ~/.totp-key */
		snprintf(fname, sizeof fname,
		    "%s/%s", pe->pw_dir, ".totp-key");
	} else {
		/* oathdir set, use <oathdir>/username.totp-key */
		snprintf(fname, sizeof fname,
		    "%s/%s%s", s, pe->pw_name, ".totp-key");
	}
	login_close(lc);

	if ((fp = fopen(fname, "r")) == NULL) {
		syslog(LOG_ERR, "failed to open %s: %s\n",
		    fname, strerror(errno));
		exit(1);
	}
	if (check_file_secrecy(fileno(fp), fname, pe->pw_uid)) {
		fclose(fp);
		exit(1);
	}
	while ((ret == AUTH_FAILED) &&
	    (fgets(secret, sizeof secret, fp) != NULL)) {

		secret[strcspn(secret, "\n")] = '\0';

		if ((r = oath_hex2bin(secret, secret_bin,
		    &secret_length)) < 0) {
			switch (r) {
			case OATH_INVALID_HEX:
				syslog(LOG_ERR,
				    "invalid hex in key: %s", username);
				break;
			case OATH_TOO_SMALL_BUFFER:
				syslog(LOG_ERR, "key too long: %s", username);
				break;
			}
			exit(1);
		}

		memset(secret, 0, strlen(secret));

		if ((r = oath_totp_validate(secret_bin, secret_length, time(NULL),
		    time_step_size, start_offset, window, otp) >= 0))
			ret = AUTH_OK;

		memset(secret_bin, 0, secret_length);
	}

	fclose(fp);
	oath_done();

	return (ret);
}
